// This file defines templates for transforming Modelica/MetaModelica code to C
// code. They are used in the code generator phase of the compiler to write
// target code.
//
// There are two root templates intended to be called from the code generator:
// translateModel and translateFunctions. These templates do not return any
// result but instead write the result to files. All other templates return
// text and are used by the root templates (most of them indirectly).
//
// To future maintainers of this file:
//
// - A line like this
//     # var = ""
//   declares a text buffer that you can later append text to. It can also be
//   passed to other templates that in turn can append text to it. In the new
//   version of Susan it should be written like this instead:
//     let &var = buffer ""
//
// - A line like this
//     ..., Text var, ...
//   declares that a template takes a tmext buffer as input parameter. In the
//   new version of Susan it should be written like this instead:
//     ..., Text &var, ...
//
// - A line like this:
//     ..., var, ...
//   passes a text buffer to a template. In the new version of Susan it should
//   be written like this instead:
//     ..., &var, ...
//
// - Style guidelines:
//
//   - Try (hard) to limit each row to 80 characters
//
//   - Code for a template should be indented with 2 spaces
//
//     - Exception to this rule is if you have only a single case, then that
//       single case can be written using no indentation
//
//       This single case can be seen as a clarification of the input to the
//       template
//
//   - Code after a case should be indented with 2 spaces if not written on the
//     same line

package CodegenC

import interface SimCodeTV;
import interface SimCodeBackendTV;
import CodegenUtil.*;
import CodegenCFunctions.*;

/* public */ template translateModel(SimCode simCode, String guid)
  "Generates C code and Makefile for compiling and running a simulation of a
  Modelica model.
  used in Compiler/SimCode/SimCodeMain.mo"
::=
  match simCode
  case sc as SIMCODE(modelInfo=modelInfo as MODELINFO(__)) then
    let target  = simulationCodeTarget()
    let &dummy = buffer ""
    let()= System.tmpTickResetIndex(0,2) /* auxFunction index */
    let()= System.tmpTickResetIndex(0,20)  /*parfor index*/
    let()= textFile(simulationMakefile(target, simCode), '<%fileNamePrefix%>.makefile') // write the makefile first!
    let()= textFile(simulationLiteralsFile(fileNamePrefix, literals), '<%fileNamePrefix%>_literals.h')
    let()= textFile(simulationFunctionsHeaderFile(fileNamePrefix, modelInfo.functions, recordDecls), '<%fileNamePrefix%>_functions.h')

    let()= textFileConvertLines(simulationFunctionsFile(fileNamePrefix, modelInfo.functions, dummy), '<%fileNamePrefix%>_functions.c')
    let()= textFile(externalFunctionIncludes(sc.externalFunctionIncludes), '<%fileNamePrefix%>_includes.h')

    let()= textFile(recordsFile(fileNamePrefix, recordDecls), '<%fileNamePrefix%>_records.c')

    let()= textFile(simulationHeaderFile(simCode,guid), '<%fileNamePrefix%>_model.h')
    // adpro: write the main .c file last! Make on windows doesn't seem to realize that
    //        the .c file is newer than the .o file if we have succesive simulate commands
    //        for the same model (i.e. see testsuite/linearize/simextfunction.mos).
    let _ = generateSimulationFiles(simCode,guid,fileNamePrefix,false)

    // If ParModelica generate the kernels file too.
    if acceptParModelicaGrammar() then
      let()= textFile(simulationParModelicaKernelsFile(fileNamePrefix, modelInfo.functions), '<%fileNamePrefix%>_kernels.cl')

    //this top-level template always returns an empty result
    //since generated texts are written to files directly
    ""
  end match
end translateModel;

/* public */ template translateInitFile(SimCode simCode, String guid) "used in Compiler/SimCode/SimCodeMain.mo"
::=
  match simCode
  case sc as SIMCODE(__) then
    let _ = if simulationSettingsOpt then //tests the Option<> for SOME()
              textFile(simulationInitFile(simCode,guid), '<%fileNamePrefix%>_init.xml')
    let _ = (if stringEq(Config.simCodeTarget(),"JavaScript") then
              covertTextFileToCLiteral('<%fileNamePrefix%>_init.xml','<%fileNamePrefix%>_init.c'))
    ""
end translateInitFile;

/* public */ template simulationHeaderFile(SimCode simCode, String guid)
  "Generates code for main C file for simulation target.
   used in Compiler/Template/CodegenFMU.tpl"
::=
  let modelNamePrefixStr = modelNamePrefix(simCode)
  match simCode
  case simCode as SIMCODE(modelInfo=MODELINFO(__)) then
    <<
    /* Simulation code for <%dotPath(modelInfo.name)%> generated by the OpenModelica Compiler <%getVersionNr()%>. */
    <%variableDefinitions(modelInfo, timeEvents)%>

    extern void <%symbolName(modelNamePrefixStr,"callExternalObjectConstructors")%>(DATA *data, threadData_t *threadData);
    extern void <%symbolName(modelNamePrefixStr,"callExternalObjectDestructors")%>(DATA *_data, threadData_t *threadData);
    extern void <%symbolName(modelNamePrefixStr,"initialNonLinearSystem")%>(int nNonLinearSystems, NONLINEAR_SYSTEM_DATA *data);
    extern void <%symbolName(modelNamePrefixStr,"initialLinearSystem")%>(int nLinearSystems, LINEAR_SYSTEM_DATA *data);
    extern void <%symbolName(modelNamePrefixStr,"initialMixedSystem")%>(int nMixedSystems, MIXED_SYSTEM_DATA *data);
    extern void <%symbolName(modelNamePrefixStr,"initializeStateSets")%>(int nStateSets, STATE_SET_DATA* statesetData, DATA *data);
    extern int <%symbolName(modelNamePrefixStr,"functionAlgebraics")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"function_storeDelayed")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"updateBoundVariableAttributes")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionInitialEquations")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionInitialEquations_lambda0")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionRemovedInitialEquations")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"updateBoundParameters")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"checkForAsserts")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"function_ZeroCrossingsEquations")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"function_ZeroCrossings")%>(DATA *data, threadData_t *threadData, double* gout);
    extern int <%symbolName(modelNamePrefixStr,"function_updateRelations")%>(DATA *data, threadData_t *threadData, int evalZeroCross);
    extern int <%symbolName(modelNamePrefixStr,"checkForDiscreteChanges")%>(DATA *data, threadData_t *threadData);
    extern const char* <%symbolName(modelNamePrefixStr,"zeroCrossingDescription")%>(int i, int **out_EquationIndexes);
    extern const char* <%symbolName(modelNamePrefixStr,"relationDescription")%>(int i);
    extern void <%symbolName(modelNamePrefixStr,"function_initSample")%>(DATA *data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianG")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianA")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianB")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianC")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianD")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionJacG_column")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionJacA_column")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionJacB_column")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionJacC_column")%>(void* data, threadData_t *threadData);
    extern int <%symbolName(modelNamePrefixStr,"functionJacD_column")%>(void* data, threadData_t *threadData);
    extern const char* <%symbolName(modelNamePrefixStr,"linear_model_frame")%>(void);
    extern int <%symbolName(modelNamePrefixStr,"mayer")%>(DATA* data, modelica_real** res, short *);
    extern int <%symbolName(modelNamePrefixStr,"lagrange")%>(DATA* data, modelica_real** res, short *, short *);
    extern int <%symbolName(modelNamePrefixStr,"pickUpBoundsForInputsInOptimization")%>(DATA* data, modelica_real* min, modelica_real* max, modelica_real*nominal, modelica_boolean *useNominal, char ** name, modelica_real * start, modelica_real * startTimeOpt);
    extern int <%symbolName(modelNamePrefixStr,"setInputData")%>(DATA *data, const modelica_boolean file);
    extern int <%symbolName(modelNamePrefixStr,"getTimeGrid")%>(DATA *data, modelica_integer * nsi, modelica_real**t);
    extern void <%symbolName(modelNamePrefixStr,"function_initSynchronous")%>(DATA * data, threadData_t *threadData);
    extern void <%symbolName(modelNamePrefixStr,"function_updateSynchronous")%>(DATA * data, threadData_t *threadData, long i);
    extern int <%symbolName(modelNamePrefixStr,"function_equationsSynchronous")%>(DATA * data, threadData_t *threadData, long i);
    extern void <%symbolName(modelNamePrefixStr,"read_input_fmu")%>(MODEL_DATA* modelData, SIMULATION_INFO* simulationData);
    extern void <%symbolName(modelNamePrefixStr,"function_savePreSynchronous")%>(DATA *data, threadData_t *threadData);
    <%\n%>
    >>
  end match
end simulationHeaderFile;

/* public */ template generateSimulationFiles(SimCode simCode, String guid, String modelNamePrefix, Boolean isModelExchangeFMU)
 "Generates code in different C files for the simulation target.
  To make the compilation faster we split the simulation files into several
  used in Compiler/Template/CodegenFMU.tpl"
::=
  match simCode
    case simCode as SIMCODE(__) then
     // external objects
     let()= textFileConvertLines(simulationFile_exo(simCode,guid), '<%modelNamePrefix%>_01exo.c')
     // non-linear systems
     let()= textFileConvertLines(simulationFile_nls(simCode,guid), '<%modelNamePrefix%>_02nls.c')
     // linear systems
     let()= textFileConvertLines(simulationFile_lsy(simCode,guid), '<%modelNamePrefix%>_03lsy.c')
     // state set
     let()= textFileConvertLines(simulationFile_set(simCode,guid), '<%modelNamePrefix%>_04set.c')
     // events: sample, zero crossings, relations
     let()= textFileConvertLines(simulationFile_evt(simCode,guid), '<%modelNamePrefix%>_05evt.c')
     // initialization
     let()= textFileConvertLines(simulationFile_inz(simCode,guid), '<%modelNamePrefix%>_06inz.c')
     // delay
     let()= textFileConvertLines(simulationFile_dly(simCode,guid), '<%modelNamePrefix%>_07dly.c')
     // update bound start values, update bound parameters
     let()= textFileConvertLines(simulationFile_bnd(simCode,guid), '<%modelNamePrefix%>_08bnd.c')
     // algebraic
     let()= textFileConvertLines(simulationFile_alg(simCode,guid), '<%modelNamePrefix%>_09alg.c')
     // asserts
     let()= textFileConvertLines(simulationFile_asr(simCode,guid), '<%modelNamePrefix%>_10asr.c')
     // mixed systems
     let &mixheader = buffer ""
     let()= textFileConvertLines(simulationFile_mix(simCode,guid,&mixheader), '<%modelNamePrefix%>_11mix.c')
     let()= textFile(&mixheader, '<%modelNamePrefix%>_11mix.h')
     // jacobians
     let()= textFileConvertLines(simulationFile_jac(simCode,guid), '<%modelNamePrefix%>_12jac.c')
     let()= textFile(simulationFile_jac_header(simCode,guid), '<%modelNamePrefix%>_12jac.h')
     // optimization
     let()= textFileConvertLines(simulationFile_opt(simCode,guid), '<%modelNamePrefix%>_13opt.c')
     let()= textFile(simulationFile_opt_header(simCode,guid), '<%modelNamePrefix%>_13opt.h')
     // linearization
     let()= textFileConvertLines(simulationFile_lnz(simCode,guid), '<%modelNamePrefix%>_14lnz.c')
     // synchronous
     let()= textFileConvertLines(simulationFile_syn(simCode,guid), '<%modelNamePrefix%>_15syn.c')
     // main file
     let()= textFileConvertLines(simulationFile(simCode,guid,isModelExchangeFMU), '<%modelNamePrefix%>.c')
     ""
  end match
end generateSimulationFiles;

template simulationFile_syn(SimCode simCode, String guid)
"Synchonous features"
::= match simCode
    case simCode as SIMCODE(__) then
      <<
      /* Synchronous systems */
      <%simulationFileHeader(simCode)%>
      #if defined(__cplusplus)
      extern "C" {
      #endif

      <%functionInitSynchronous(clockedPartitions, modelNamePrefix(simCode))%>

      <%functionUpdateSynchronous(clockedPartitions, modelNamePrefix(simCode))%>

      <%functionSystemsSynchronous(getSubPartitions(clockedPartitions), modelNamePrefix(simCode))%>

      <%functionSavePreSynchronous(getSubPartitions(clockedPartitions), modelNamePrefix(simCode))%>

      #if defined(__cplusplus)
      }
      #endif
      <%\n%>
      >>
  end match
end simulationFile_syn;

template functionSavePreSynchronous(list<SimCode.SubPartition> subPartitions, String modelNamePrefix)
::=
  let preVars = subPartitions |> subPartition =>
    functionSavePreSynchronous1(subPartition); separator="\n"
  <<
  /* $P$PRE%v% = %v% */
  void <%symbolName(modelNamePrefix,"function_savePreSynchronous")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH

    <%preVars%>

    TRACE_POP
  }
  >>
end functionSavePreSynchronous;

template functionSavePreSynchronous1(SimCode.SubPartition subPartition)
::=
match subPartition
  case SUBPARTITION(__) then functionSavePreSynchronous2(vars)
end functionSavePreSynchronous1;

template functionSavePreSynchronous2(list<tuple<SimCodeVar.SimVar, Boolean>> vars)
::=
  vars |> var => functionSavePreSynchronous3(var); separator = "\n"
end functionSavePreSynchronous2;

template functionSavePreSynchronous3(tuple<SimCodeVar.SimVar, Boolean> var)
::=
match var
  case (simVar, previous) then
    match simVar
      case SIMVAR(arrayCref=SOME(c), aliasvar=NOALIAS()) then
        '<%cref(c)%> = $P$PRE<%cref(c)%>;'
      case SIMVAR(aliasvar=NOALIAS()) then
        '<%cref(name)%> = $P$PRE<%cref(name)%>;'
end functionSavePreSynchronous3;

template isBoolClock(DAE.ClockKind clock)
::=
match clock
  case BOOLEAN_CLOCK(__) then boolStrC(true)
  else boolStrC(false)
end isBoolClock;

template functionInitSynchronous(list<ClockedPartition> clockedPartitions, String modelNamePrefix)
"Synchonous features"
::=
  let body = clockedPartitions |> partition =>
    match partition
      case CLOCKED_PARTITION(__) then
        let boolClock = isBoolClock(baseClock)
        let subClocksInfo = subPartitions |> subPartition =>
                            subPartitionStr(subPartition); separator="\n"
        <<
        data->modelData->clocksInfo[i].nSubClocks = <%listLength(subPartitions)%>;
        data->modelData->clocksInfo[i].subClocks = data->modelData->subClocksInfo + j;
        data->modelData->clocksInfo[i].isBoolClock = <%boolClock%>;
        i++;
        <%subClocksInfo%>

        >>
  <<
  /* Initializes the clocks of model. */
  void <%symbolName(modelNamePrefix,"function_initSynchronous")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    long i=0, j=0;
    <%body%>
    TRACE_POP
  }
  >>
end functionInitSynchronous;

template subPartitionStr(SubPartition subPartition)
::=
match subPartition
  case SUBPARTITION(subClock = SUBCLOCK(__), holdEvents=holdEvents) then
    let methodStr = match subClock.solver
      case NONE() then ""
      else "External"
    <<
    <%rationalStr("data->modelData->subClocksInfo[j].shift", subClock.shift)%>
    <%rationalStr("data->modelData->subClocksInfo[j].factor", subClock.factor)%>
    data->modelData->subClocksInfo[j].solverMethod = "<%methodStr%>";
    data->modelData->subClocksInfo[j].holdEvents = <%boolStrC(holdEvents)%>;
    j++;
    >>
end subPartitionStr;

template rationalStr(String prefix, Rational rational)
::=
match rational
  case RATIONAL(__) then
    <<
    <%prefix%>.m = <%nom%>;
    <%prefix%>.n = <%denom%>;
    >>
end rationalStr;

template functionUpdateSynchronous(list<ClockedPartition> clockedPartitions, String modelNamePrefix)
::=
  let &varDecls = buffer ""
  let &auxFunction = buffer ""
  let body = clockedPartitions |> partition hasindex i =>
    match partition
      case CLOCKED_PARTITION(__) then
        let caseBody = updatePartition(i, baseClock, &varDecls, &auxFunction)
        <<
        case <%i%>:
          <%caseBody%>
          break;
        >>; separator = "\n"
  <<
  <%auxFunction%>
  /* Update the base clock. */
  void <%symbolName(modelNamePrefix,"function_updateSynchronous")%>(DATA *data, threadData_t *threadData, long i)
  {
    TRACE_PUSH
    <%varDecls%>
    modelica_boolean ret;
    switch (i) {
      <%body%>
      default:
        throwStreamPrint(NULL, "Internal Error: unknown base partition %ld", i);
        break;
    }
    TRACE_POP
  }
  >>
end functionUpdateSynchronous;

template updatePartition(Integer i, DAE.ClockKind baseClock, Text &varDecls, Text &auxFunction)
::=
match baseClock
  case DAE.BOOLEAN_CLOCK(__) then
    let cond = cref(expCref(condition))
    <<
    if (data->simulationInfo->clocksData[i].cnt > 0)
      data->simulationInfo->clocksData[i].interval = data->localData[0]->timeValue - data->simulationInfo->clocksData[i].timepoint;
    else
      data->simulationInfo->clocksData[i].interval = <%startInterval%>;
    >>
  else
    let &preExp = buffer ""
    let intvl = daeExp(getClockInterval(baseClock), contextOther, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    data->simulationInfo->clocksData[i].interval = <%intvl%>;
    >>
end updatePartition;

template functionSystemsSynchronous(list<SubPartition> subPartitions, String modelNamePrefix)
::=
  let systs = subPartitions |> subPartition hasindex i =>
    match subPartition
      case SUBPARTITION(__) then
        functionEquationsSynchronous(i, vars, listAppend(equations, removedEquations), modelNamePrefix)
    ; separator = "\n"
  let cases = subPartitions |> subPartition hasindex i =>
    let name = 'functionEquationsSynchronous_system<%i%>'
    <<
    case <%i%>:
      ret = <%symbolName(modelNamePrefix, name)%>(data, threadData);
      break;
    >>; separator = "\n"
  <<

  <%systs%>

  /*Clocked systems equations */
  int <%symbolName(modelNamePrefix,"function_equationsSynchronous")%>(DATA *data, threadData_t *threadData, long i)
  {
    TRACE_PUSH
    int ret;

    switch (i) {
      <%cases%>
      default:
        throwStreamPrint(NULL, "Internal Error: unknown sub partition %ld", i);
        ret = 1;
        break;
    }

    TRACE_POP
    return ret;
  }
  >>

end functionSystemsSynchronous;

template functionEquationsSynchronous(Integer i, list<tuple<SimCodeVar.SimVar, Boolean>> vars, list<SimEqSystem> equations, String modelNamePrefix)
::=
  let &varDecls = buffer ""
  let &eqfuncs = buffer ""
  let fncalls = equations |> eq => equation_(i, eq, contextOther, &varDecls, &eqfuncs, modelNamePrefix); separator="\n"
  <<
  <%&eqfuncs%>

  int <%symbolName(modelNamePrefix, 'functionEquationsSynchronous_system<%i%>')%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    int i;

    <%addRootsTempArray()%>

    <%fncalls%>

    TRACE_POP
    return 0;
  }
  >>
end functionEquationsSynchronous;

template simulationFile_exo(SimCode simCode, String guid)
"External Objects"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* External objects file */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionCallExternalObjectConstructors(extObjInfo, modelNamePrefix(simCode))%>

    <%functionCallExternalObjectDestructors(extObjInfo, modelNamePrefix(simCode))%>
    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_exo;

template simulationFile_nls(SimCode simCode, String guid)
"Non Linear Systems"
::=
  match simCode
    case simCode as SIMCODE(__) then
    let modelNamePrefixStr = modelNamePrefix(simCode)
    let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionNonLinearResiduals(jacobianEquations,modelNamePrefixStr);separator="\n\n")
    <<
    /* Non Linear Systems */
    <%simulationFileHeader(simCode)%>
    #include "<%simCode.fileNamePrefix%>_12jac.h"
    #if defined(__cplusplus)
    extern "C" {
    #endif
    <%functionNonLinearResiduals(initialEquations, modelNamePrefixStr)%>
    <%functionNonLinearResiduals(initialEquations_lambda0, modelNamePrefixStr)%>
    <%functionNonLinearResiduals(parameterEquations,modelNamePrefixStr)%>
    <%functionNonLinearResiduals(allEquations,modelNamePrefixStr)%>
    <%jacobianbody%>

    <%functionInitialNonLinearSystems(initialEquations, initialEquations_lambda0, parameterEquations, allEquations, jacobianMatrixes, modelNamePrefixStr)%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_nls;

template simulationFile_lsy(SimCode simCode, String guid)
"Linear Systems"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Linear Systems */
    <%simulationFileHeader(simCode)%>
    #include "<%simCode.fileNamePrefix%>_12jac.h"
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionSetupLinearSystems(initialEquations, initialEquations_lambda0, parameterEquations, allEquations, jacobianMatrixes, modelNamePrefix(simCode))%>

    <%functionInitialLinearSystems(initialEquations, initialEquations_lambda0, parameterEquations, allEquations, jacobianMatrixes, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_lsy;

template simulationFile_set(SimCode simCode, String guid)
"Initial State Set"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Initial State Set */
    <%simulationFileHeader(simCode)%>
    #include "<%simCode.fileNamePrefix%>_11mix.h"
    #include "<%simCode.fileNamePrefix%>_12jac.h"
    #if defined(__cplusplus)
    extern "C" {
    #endif
    <%functionInitialStateSets(stateSets, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_set;

template simulationFile_evt(SimCode simCode, String guid)
"Events: Sample, Zero Crossings, Relations, Discrete Changes"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Events: Sample, Zero Crossings, Relations, Discrete Changes */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionInitSample(timeEvents, modelNamePrefix(simCode))%>

    <%functionZeroCrossing(zeroCrossings, equationsForZeroCrossings, modelNamePrefix(simCode))%>

    <%functionRelations(relations, modelNamePrefix(simCode))%>

    <%functionCheckForDiscreteChanges(discreteModelVars, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_evt;

template simulationFile_inz(SimCode simCode, String guid)
"Initialization"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Initialization */
    <%simulationFileHeader(simCode)%>
    #include "<%simCode.fileNamePrefix%>_11mix.h"
    #include "<%simCode.fileNamePrefix%>_12jac.h"
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionInitialEquations(initialEquations, modelNamePrefix(simCode))%>
    <%functionInitialEquations_lambda0(initialEquations_lambda0, modelNamePrefix(simCode))%>
    <%functionRemovedInitialEquations(removedInitialEquations, modelNamePrefix(simCode))%>

    <%functionInitialMixedSystems(initialEquations, initialEquations_lambda0, parameterEquations, allEquations, jacobianMatrixes, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_inz;

template simulationFile_dly(SimCode simCode, String guid)
"Delay"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Delay */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionStoreDelayed(delayedExps, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_dly;

template simulationFile_bnd(SimCode simCode, String guid)
"update bound parameters and variable attributes (start, nominal, min, max)"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* update bound parameters and variable attributes (start, nominal, min, max) */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionUpdateBoundVariableAttributes(startValueEquations, nominalValueEquations, minValueEquations, maxValueEquations, modelNamePrefix(simCode))%>

    <%functionUpdateBoundParameters(parameterEquations, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_bnd;

template simulationFile_alg(SimCode simCode, String guid)
"Algebraic"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Algebraic */
    <%simulationFileHeader(simCode)%>

    #ifdef __cplusplus
    extern "C" {
    #endif

    <%functionAlgebraic(algebraicEquations, modelNamePrefix(simCode))%>

    #ifdef __cplusplus
    }
    #endif<%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_alg;

template simulationFile_asr(SimCode simCode, String guid)
"Asserts"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Asserts */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionAssertsforCheck(algorithmAndEquationAsserts, modelNamePrefix(simCode))%>

    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_asr;

template simulationFile_mix(SimCode simCode, String guid, Text &header)
"Mixed Systems"
::=
  match simCode
    case simCode as SIMCODE(__) then
    let modelNamePrefixStr = modelNamePrefix(simCode)
    <<
    /* Mixed Systems */
    <%simulationFileHeader(simCode)%>
    #include "<%simCode.fileNamePrefix%>_11mix.h"
    <%functionSetupMixedSystems(initialEquations, initialEquations_lambda0, parameterEquations, allEquations, jacobianMatrixes, &header, modelNamePrefixStr)%>

    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_mix;

template simulationFile_jac(SimCode simCode, String guid)
"Jacobians"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Jacobians */
    <%simulationFileHeader(simCode)%>
    #include "<%fileNamePrefix%>_12jac.h"
    <%functionAnalyticJacobians(jacobianMatrixes, modelNamePrefix(simCode))%>

    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_jac;

template simulationFile_jac_header(SimCode simCode, String guid)
"Jacobians"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Jacobians */
    static const REAL_ATTRIBUTE dummyREAL_ATTRIBUTE = omc_dummyRealAttribute;
    <%variableDefinitionsJacobians(jacobianMatrixes, modelNamePrefix(simCode))%>
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_jac_header;

template simulationFile_opt(SimCode simCode, String guid)
"Optimization"
::=
  match simCode
    case simCode as SIMCODE(__) then
    let modelNamePrefixStr = modelNamePrefix(simCode)
    <<
    /* Optimization */
    <%simulationFileHeader(simCode)%>
    #include "<%fileNamePrefix%>_12jac.h"
    #if defined(__cplusplus)
    extern "C" {
    #endif
    <%optimizationComponents(classAttributes, simCode, modelNamePrefixStr)%>
    #if defined(__cplusplus)
    }
    #endif
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_opt;

template simulationFile_opt_header(SimCode simCode, String guid)
"Jacobians"
::=
  match simCode
    case simCode as SIMCODE(__) then
    let modelNamePrefixStr = modelNamePrefix(simCode)
    <<
    #if defined(__cplusplus)
      extern "C" {
    #endif
      int <%symbolName(modelNamePrefixStr,"mayer")%>(DATA* data, modelica_real** res, short*);
      int <%symbolName(modelNamePrefixStr,"lagrange")%>(DATA* data, modelica_real** res, short *, short *);
      int <%symbolName(modelNamePrefixStr,"pickUpBoundsForInputsInOptimization")%>(DATA* data, modelica_real* min, modelica_real* max, modelica_real*nominal, modelica_boolean *useNominal, char ** name, modelica_real * start, modelica_real * startTimeOpt);
      int <%symbolName(modelNamePrefixStr,"setInputData")%>(DATA *data, const modelica_boolean file);
      int <%symbolName(modelNamePrefixStr,"getTimeGrid")%>(DATA *data, modelica_integer * nsi, modelica_real**t);
    #if defined(__cplusplus)
    }
    #endif
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_opt_header;

template simulationFile_lnz(SimCode simCode, String guid)
"Linearization"
::=
  match simCode
    case simCode as SIMCODE(__) then
    <<
    /* Linearization */
    <%simulationFileHeader(simCode)%>
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionlinearmodel(modelInfo, modelNamePrefix(simCode))%>
    #if defined(__cplusplus)
    }
    #endif
    <%\n%>
    >>
    /* adrpo: leave a newline at the end of file to get rid of the warning */
  end match
end simulationFile_lnz;

template simulationFile(SimCode simCode, String guid, Boolean isModelExchangeFMU)
  "Generates code for main C file for simulation target."
::=
  match simCode
    case simCode as SIMCODE(hpcomData=HPCOMDATA(__)) then
    let modelNamePrefixStr = modelNamePrefix(simCode)
    let mainInit = if boolOr(isModelExchangeFMU, boolOr(Flags.isSet(Flags.PARMODAUTO), Flags.isSet(HPCOM))) then
                     <<
                     mmc_init_nogc();
                     omc_alloc_interface = omc_alloc_interface_pooled;
                     >>
                   else if stringEq(Config.simCodeTarget(),"JavaScript") then
                     <<
                     mmc_init_nogc();
                     omc_alloc_interface = omc_alloc_interface_pooled;
                     >>
                   else
                     <<
                     MMC_INIT();
                     >>
    let &mainInit += 'omc_alloc_interface.init();'
    let pminit = if Flags.isSet(Flags.PARMODAUTO) then 'PM_Model_init("<%fileNamePrefix%>", &data, threadData, functionODE_systems);' else ''
    let mainBody =
      <<
      <%symbolName(modelNamePrefixStr,"setupDataStruc")%>(&data, threadData);
      <%pminit%>
      res = _main_SimulationRuntime(argc, argv, &data, threadData);
      >>
    <<
    /* Main Simulation File */
    <%simulationFileHeader(simCode)%>

    <% if boolNot(isModelExchangeFMU) then
    <<
    #define prefixedName_performSimulation <%symbolName(modelNamePrefixStr,"performSimulation")%>
    #define prefixedName_updateContinuousSystem <%symbolName(modelNamePrefixStr,"updateContinuousSystem")%>
    #include <simulation/solver/perform_simulation.c>

    #define prefixedName_performQSSSimulation <%symbolName(modelNamePrefixStr,"performQSSSimulation")%>
    #include <simulation/solver/perform_qss_simulation.c>
    >>
    %>

    /* dummy VARINFO and FILEINFO */
    const FILE_INFO dummyFILE_INFO = omc_dummyFileInfo;
    const VAR_INFO dummyVAR_INFO = omc_dummyVarInfo;
    #if defined(__cplusplus)
    extern "C" {
    #endif

    <%functionInput(modelInfo, modelNamePrefixStr)%>

    <%functionOutput(modelInfo, modelNamePrefixStr)%>

    <%functionDAE(allEquations, modelNamePrefixStr)%>

    <%functionSymEuler(modelInfo, modelNamePrefixStr)%>

    <%functionODE(odeEquations,(match simulationSettingsOpt case SOME(settings as SIMULATION_SETTINGS(__)) then settings.method else ""), hpcomData.schedules, modelNamePrefixStr)%>

    #ifdef FMU_EXPERIMENTAL
    <% if Flags.isSet(Flags.FMU_EXPERIMENTAL) then functionODEPartial(odeEquations,(match simulationSettingsOpt case SOME(settings as SIMULATION_SETTINGS(__)) then settings.method else ""), hpcomData.schedules, modelNamePrefixStr, modelInfo)%>
    <% if Flags.isSet(Flags.FMU_EXPERIMENTAL) then
    <<
    void <%symbolName(modelNamePrefixStr,"functionFMIJacobian")%>(DATA *data, threadData_t *threadData, const unsigned *unknown, int nUnk, const unsigned *ders, int nKnown, double *dvKnown, double *out) {
        int i;
        /* TODO: Use the literal names instead of the data-> structure
         * Beware! This code assumes that the FMI variables are sorted putting
         * states first (0 to nStates-1) and state derivatives (nStates to 2*nStates-1) second. */
        for (i=0;i<data->modelData->nStates; i++) {
          // Clear out the seeds
          data->simulationInfo->analyticJacobians[0].seedVars[i]=0;
        }
        for (i=0;i<nUnk; i++) {
          /* Put the supplied value in the seeds */
          data->simulationInfo->analyticJacobians[0].seedVars[unknown[i]]=dvKnown[i];
        }
        /* Call the Jacobian evaluation function. This function evaluates the whole column of the Jacobian.
         * More efficient code could only evaluate the equations needed for the
         * known variables only */
        <%symbolName(modelNamePrefixStr,"functionJacA_column")%>(data,threadData);

        // Write the results back to the array
        for (i=0;i<nKnown; i++) {
          out[ders[i]-data->modelData->nStates] = data->simulationInfo->analyticJacobians[0].resultVars[ders[i]-data->modelData->nStates];
        }
    }
    >> %>
    #endif
    /* forward the main in the simulation runtime */
    extern int _main_SimulationRuntime(int argc, char**argv, DATA *data, threadData_t *threadData);

    #include "<%simCode.fileNamePrefix%>_12jac.h"
    #include "<%simCode.fileNamePrefix%>_13opt.h"

    struct OpenModelicaGeneratedFunctionCallbacks <%symbolName(modelNamePrefixStr,"callback")%> = {
       <% if isModelExchangeFMU then "NULL" else '(int (*)(DATA *, threadData_t *, void *)) <%symbolName(modelNamePrefixStr,"performSimulation")%>'%>,
       <% if isModelExchangeFMU then "NULL" else '(int (*)(DATA *, threadData_t *, void *)) <%symbolName(modelNamePrefixStr,"performQSSSimulation")%>'%>,
       <% if isModelExchangeFMU then "NULL" else '<%symbolName(modelNamePrefixStr,"updateContinuousSystem")%>'%>,
       <%symbolName(modelNamePrefixStr,"callExternalObjectConstructors")%>,
       <%symbolName(modelNamePrefixStr,"callExternalObjectDestructors")%>,
       <%symbolName(modelNamePrefixStr,"initialNonLinearSystem")%>,
       <%symbolName(modelNamePrefixStr,"initialLinearSystem")%>,
       <%symbolName(modelNamePrefixStr,"initialMixedSystem")%>,
       <%symbolName(modelNamePrefixStr,"initializeStateSets")%>,
       <%symbolName(modelNamePrefixStr,"functionODE")%>,
       <%symbolName(modelNamePrefixStr,"functionAlgebraics")%>,
       <%symbolName(modelNamePrefixStr,"functionDAE")%>,
       <%symbolName(modelNamePrefixStr,"input_function")%>,
       <%symbolName(modelNamePrefixStr,"input_function_init")%>,
       <%symbolName(modelNamePrefixStr,"input_function_updateStartValues")%>,
       <%symbolName(modelNamePrefixStr,"output_function")%>,
       <%symbolName(modelNamePrefixStr,"function_storeDelayed")%>,
       <%symbolName(modelNamePrefixStr,"updateBoundVariableAttributes")%>,
       <%boolStrC(useHomotopy)%> /* useHomotopy */,
       <%symbolName(modelNamePrefixStr,"functionInitialEquations")%>,
       <%symbolName(modelNamePrefixStr,"functionInitialEquations_lambda0")%>,
       <%symbolName(modelNamePrefixStr,"functionRemovedInitialEquations")%>,
       <%symbolName(modelNamePrefixStr,"updateBoundParameters")%>,
       <%symbolName(modelNamePrefixStr,"checkForAsserts")%>,
       <%symbolName(modelNamePrefixStr,"function_ZeroCrossingsEquations")%>,
       <%symbolName(modelNamePrefixStr,"function_ZeroCrossings")%>,
       <%symbolName(modelNamePrefixStr,"function_updateRelations")%>,
       <%symbolName(modelNamePrefixStr,"checkForDiscreteChanges")%>,
       <%symbolName(modelNamePrefixStr,"zeroCrossingDescription")%>,
       <%symbolName(modelNamePrefixStr,"relationDescription")%>,
       <%symbolName(modelNamePrefixStr,"function_initSample")%>,
       <%symbolName(modelNamePrefixStr,"INDEX_JAC_A")%>,
       <%symbolName(modelNamePrefixStr,"INDEX_JAC_B")%>,
       <%symbolName(modelNamePrefixStr,"INDEX_JAC_C")%>,
       <%symbolName(modelNamePrefixStr,"INDEX_JAC_D")%>,
       <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianA")%>,
       <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianB")%>,
       <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianC")%>,
       <%symbolName(modelNamePrefixStr,"initialAnalyticJacobianD")%>,
       <%symbolName(modelNamePrefixStr,"functionJacA_column")%>,
       <%symbolName(modelNamePrefixStr,"functionJacB_column")%>,
       <%symbolName(modelNamePrefixStr,"functionJacC_column")%>,
       <%symbolName(modelNamePrefixStr,"functionJacD_column")%>,
       <%symbolName(modelNamePrefixStr,"linear_model_frame")%>,
       <%symbolName(modelNamePrefixStr,"mayer")%>,
       <%symbolName(modelNamePrefixStr,"lagrange")%>,
       <%symbolName(modelNamePrefixStr,"pickUpBoundsForInputsInOptimization")%>,
       <%symbolName(modelNamePrefixStr,"setInputData")%>,
       <%symbolName(modelNamePrefixStr,"getTimeGrid")%>,
       <%symbolName(modelNamePrefixStr,"symEulerUpdate")%>,
       <%symbolName(modelNamePrefixStr,"function_initSynchronous")%>,
       <%symbolName(modelNamePrefixStr,"function_updateSynchronous")%>,
       <%symbolName(modelNamePrefixStr,"function_equationsSynchronous")%>,
       <% if isModelExchangeFMU then symbolName(modelNamePrefixStr,"read_input_fmu") else "NULL" %>
       #ifdef FMU_EXPERIMENTAL
       ,<%symbolName(modelNamePrefixStr,"functionODE_Partial")%>
       ,<%symbolName(modelNamePrefixStr,"functionFMIJacobian")%>
       #endif

    <%\n%>
    };

    <%functionInitializeDataStruc(modelInfo, fileNamePrefix, guid, allEquations, jacobianMatrixes, delayedExps, modelNamePrefixStr, isModelExchangeFMU)%>

    #ifdef __cplusplus
    }
    #endif

    static int rml_execution_failed()
    {
      fflush(NULL);
      fprintf(stderr, "Execution failed!\n");
      fflush(NULL);
      return 1;
    }

    <% if boolNot(isModelExchangeFMU) then
    <<
    #if defined(threadData)
    #undef threadData
    #endif
    /* call the simulation runtime main from our main! */
    int main(int argc, char**argv)
    {
      int res;
      DATA data;
      MODEL_DATA modelData;
      SIMULATION_INFO simInfo;
      data.modelData = &modelData;
      data.simulationInfo = &simInfo;
      measure_time_flag = <% if profileHtml() then "5" else if profileSome() then "1" else if profileAll() then "2" else "0" /* Would be good if this was not a global variable...*/ %>;
      <%mainInit%>
      <%mainTop(mainBody,"https://trac.openmodelica.org/OpenModelica/newticket")%>

      <%if Flags.isSet(HPCOM) then "terminateHpcOmThreads();" %>
      <%if Flags.isSet(Flags.PARMODAUTO) then "dump_times();" %>
      fflush(NULL);
      EXIT(res);
      return res;
    }
    <%\n%>
    >>
    %>
    >>
    /* adrpo: leave a newline at the end of file to get ridsymbolName(String fileNamePrefix of the warning */
  end match
end simulationFile;

template simulationFileHeader(SimCode simCode)
  "Generates header part of simulation file."
::=
  match simCode
  case SIMCODE(modelInfo=MODELINFO(__), extObjInfo=EXTOBJINFO(__)) then
    <<
    /* Simulation code for <%dotPath(modelInfo.name)%> generated by the OpenModelica Compiler <%getVersionNr()%>. */

    #include "openmodelica.h"
    #include "openmodelica_func.h"
    #include "simulation_data.h"
    #include "simulation/simulation_info_json.h"
    #include "simulation/simulation_runtime.h"
    #include "util/omc_error.h"
    #include "simulation/solver/model_help.h"
    #include "simulation/solver/delay.h"
    #include "simulation/solver/linearSystem.h"
    #include "simulation/solver/nonlinearSystem.h"
    #include "simulation/solver/mixedSystem.h"

    #include <string.h>

    #include "<%fileNamePrefix%>_functions.h"
    #include "<%fileNamePrefix%>_model.h"
    #include "<%fileNamePrefix%>_literals.h"

    <%if Flags.isSet(Flags.PARMODAUTO) then "#include \"ParModelica/auto/om_pm_interface.hpp\""%>

    <%if stringEq(getConfigString(HPCOM_CODE),"pthreads_spin") then "#include \"util/omc_spinlock.h\""%>

    <%if Flags.isSet(HPCOM) then "#define HPCOM"%>

    #if defined(HPCOM) && !defined(_OPENMP)
      #error "HPCOM requires OpenMP or the results are wrong"
    #endif
    #if defined(_OPENMP)
      #include <omp.h>
    #else
      /* dummy omp defines */
      #define omp_get_max_threads() 1
    #endif
    >>
  end match
end simulationFileHeader;

template populateModelInfo(ModelInfo modelInfo, String fileNamePrefix, String guid, list<SimEqSystem> allEquations, list<SimCode.JacobianMatrix> symJacs, DelayedExpression delayed, Boolean isModelExchangeFMU)
  "Generates information for data.modelInfo struct."
::=
  match modelInfo
  case MODELINFO(varInfo=VARINFO(__)) then
    <<
    data->modelData->modelName = "<%dotPath(name)%>";
    data->modelData->modelFilePrefix = "<%fileNamePrefix%>";
    data->modelData->resultFileName = NULL;
    data->modelData->modelDir = "<%directory%>";
    data->modelData->modelGUID = "{<%guid%>}";
    <% if isModelExchangeFMU then
    <<
    data->modelData->initXMLData = NULL;
    data->modelData->modelDataXml.infoXMLData =
    #include "<%fileNamePrefix%>_info.c"
    ;
    >>
    else
    <<
    #if defined(OPENMODELICA_XML_FROM_FILE_AT_RUNTIME)
    data->modelData->initXMLData = NULL;
    data->modelData->modelDataXml.infoXMLData = NULL;
    #else
    data->modelData->initXMLData =
    #include "<%fileNamePrefix%>_init.c"
    ;
    data->modelData->modelDataXml.infoXMLData =
    #include "<%fileNamePrefix%>_info.c"
    ;
    #endif
    >>
    %>

    data->modelData->nStates = <%varInfo.numStateVars%>;
    data->modelData->nVariablesReal = 2*<%varInfo.numStateVars%>+<%varInfo.numAlgVars%>+<%varInfo.numDiscreteReal%>+<%varInfo.numOptimizeConstraints%> + <%varInfo.numOptimizeFinalConstraints%>;
    data->modelData->nDiscreteReal = <%varInfo.numDiscreteReal%>;
    data->modelData->nVariablesInteger = <%varInfo.numIntAlgVars%>;
    data->modelData->nVariablesBoolean = <%varInfo.numBoolAlgVars%>;
    data->modelData->nVariablesString = <%varInfo.numStringAlgVars%>;
    data->modelData->nParametersReal = <%varInfo.numParams%>;
    data->modelData->nParametersInteger = <%varInfo.numIntParams%>;
    data->modelData->nParametersBoolean = <%varInfo.numBoolParams%>;
    data->modelData->nParametersString = <%varInfo.numStringParamVars%>;
    data->modelData->nInputVars = <%varInfo.numInVars%>;
    data->modelData->nOutputVars = <%varInfo.numOutVars%>;

    data->modelData->nAliasReal = <%varInfo.numAlgAliasVars%>;
    data->modelData->nAliasInteger = <%varInfo.numIntAliasVars%>;
    data->modelData->nAliasBoolean = <%varInfo.numBoolAliasVars%>;
    data->modelData->nAliasString = <%varInfo.numStringAliasVars%>;

    data->modelData->nZeroCrossings = <%varInfo.numZeroCrossings%>;
    data->modelData->nSamples = <%varInfo.numTimeEvents%>;
    data->modelData->nRelations = <%varInfo.numRelations%>;
    data->modelData->nMathEvents = <%varInfo.numMathEventFunctions%>;
    data->modelData->nExtObjs = <%varInfo.numExternalObjects%>;
    data->modelData->modelDataXml.fileName = "<%fileNamePrefix%>_info.<%if Flags.isSet(Flags.MODEL_INFO_JSON) then "json" else "xml"%>";
    data->modelData->modelDataXml.modelInfoXmlLength = 0;
    data->modelData->modelDataXml.nFunctions = <%listLength(functions)%>;
    data->modelData->modelDataXml.nProfileBlocks = 0;
    data->modelData->modelDataXml.nEquations = <%varInfo.numEquations%>;
    data->modelData->nMixedSystems = <%varInfo.numMixedSystems%>;
    data->modelData->nLinearSystems = <%varInfo.numLinearSystems%>;
    data->modelData->nNonLinearSystems = <%varInfo.numNonLinearSystems%>;
    data->modelData->nStateSets = <%varInfo.numStateSets%>;
    data->modelData->nJacobians = <%varInfo.numJacobians%>;
    data->modelData->nOptimizeConstraints = <%varInfo.numOptimizeConstraints%>;
    data->modelData->nOptimizeFinalConstraints = <%varInfo.numOptimizeFinalConstraints%>;

    data->modelData->nDelayExpressions = <%match delayed case DELAYED_EXPRESSIONS(__) then maxDelayedIndex%>;

    data->modelData->nClocks = <%nClocks%>;
    data->modelData->nSubClocks = <%nSubClocks%>;

    >>
  end match
end populateModelInfo;

template functionInitializeDataStruc(ModelInfo modelInfo, String fileNamePrefix, String guid, list<SimEqSystem> allEquations, list<SimCode.JacobianMatrix> symJacs, DelayedExpression delayed, String modelNamePrefix, Boolean isModelExchangeFMU)
  "Generates function in simulation file."
::=
  <<
  void <%symbolName(modelNamePrefix,"setupDataStruc")%>(DATA *data, threadData_t *threadData)
  {
    assertStreamPrint(threadData,0!=data, "Error while initialize Data");
    data->callback = &<%symbolName(modelNamePrefix,"callback")%>;
    <%populateModelInfo(modelInfo, fileNamePrefix, guid, allEquations, symJacs, delayed, isModelExchangeFMU)%>
  }
  >>
end functionInitializeDataStruc;

template functionSimProfDef(SimEqSystem eq, Integer value, Text &reverseProf)
  "Generates function in simulation file."
::=
  match eq
  case SES_MIXED(__) then
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%index%>;<%\n%>'
    <<
    #define SIM_PROF_EQ_<%index%> <%value%><%\n%>
    >>
  // no dynamic tearing
  case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%ls.index%>;<%\n%>'
    <<
    #define SIM_PROF_EQ_<%ls.index%> <%value%><%\n%>
    >>
  case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%nls.index%>;<%\n%>'
    <<
    #define SIM_PROF_EQ_<%nls.index%> <%value%><%\n%>
    >>
  // dynamic tearing
  case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(at as LINEARSYSTEM(__))) then
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%ls.index%>;<%\n%>'
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%at.index%>;<%\n%>'
    <<
    #define SIM_PROF_EQ_<%ls.index%> <%value%><%\n%>
    #define SIM_PROF_EQ_<%at.index%> <%value%><%\n%>
    >>
  case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%nls.index%>;<%\n%>'
    let &reverseProf += 'data->modelData->equationInfo_reverse_prof_index[<%value%>] = <%at.index%>;<%\n%>'
    <<
    #define SIM_PROF_EQ_<%nls.index%> <%value%><%\n%>
    #define SIM_PROF_EQ_<%at.index%> <%value%><%\n%>
    >>
  end match
end functionSimProfDef;

template variableDefinitions(ModelInfo modelInfo, list<BackendDAE.TimeEvent> timeEvents)
  "Generates global data in simulation file."
::=
  let () = System.tmpTickReset(1000)
  match modelInfo
    case MODELINFO(varInfo=VARINFO(numStateVars=numStateVars, numAlgVars= numAlgVars, numDiscreteReal=numDiscreteReal, numOptimizeConstraints=numOptimizeConstraints, numOptimizeFinalConstraints=numOptimizeFinalConstraints), vars=SIMVARS(__)) then
      <<
      /* States */
      <%vars.stateVars |> var =>
        globalDataVarDefine(var, "realVars", 0)
      ;separator="\n"%>

      /* StatesDerivatives */
      <%vars.derivativeVars |> var =>
        globalDataVarDefine(var, "realVars", numStateVars)
      ;separator="\n"%>

      /* Non Discrete Real Algebraic Vars */
      <%vars.algVars |> var =>
        globalDataVarDefine(var, "realVars", intMul(2, numStateVars) )
      ;separator="\n"%>

      /* Discrete Real Algebraic Vars */
      <%vars.discreteAlgVars |> var =>
        globalDataVarDefine(var, "realVars", intAdd(intMul(2, numStateVars),numAlgVars))
      ;separator="\n"%>

      /* Nonlinear Constraints For Dyn. Optimization */
      <%vars.realOptimizeConstraintsVars |> var =>
        globalDataVarDefine(var, "realVars", intAdd(intAdd(intMul(2, numStateVars),numAlgVars), numDiscreteReal))
      ;separator="\n"%>

      /* Nonlinear Final Constraints For Dyn. Optimization */
      <%vars.realOptimizeFinalConstraintsVars |> var =>
        globalDataVarDefine(var, "realVars", intAdd(intAdd(intMul(2, numStateVars),numAlgVars), intAdd(numDiscreteReal,numOptimizeConstraints)))
      ;separator="\n"%>

      /* Algebraic Parameter */
      <%vars.paramVars |> var =>
        globalDataParDefine(var, "realParameter")
      ;separator="\n"%>

      /* External Objects */
      <%vars.extObjVars |> var =>
        globalDataParDefine(var, "extObjs")
      ;separator="\n"%>

      /* Algebraic Integer Vars */
      <%vars.intAlgVars |> var =>
        globalDataVarDefine(var, "integerVars",0)
      ;separator="\n"%>

      /* Algebraic Integer Parameter */
      <%vars.intParamVars |> var =>
        globalDataParDefine(var, "integerParameter")
      ;separator="\n"%>

      /* Algebraic Boolean Vars */
      <%vars.boolAlgVars |> var =>
        globalDataVarDefine(var, "booleanVars",0)
      ;separator="\n"%>

      /* Algebraic Boolean Parameters */
      <%vars.boolParamVars |> var =>
        globalDataParDefine(var, "booleanParameter")
      ;separator="\n"%>

      /* Algebraic String Variables */
      <%vars.stringAlgVars |> var =>
        globalDataVarDefine(var, "stringVars",0)
      ;separator="\n"%>

      /* Algebraic String Parameter */
      <%vars.stringParamVars |> var =>
        globalDataParDefine(var, "stringParameter")
      ;separator="\n"%>

      /* sample */
      <%(timeEvents |> timeEvent =>
        match timeEvent
          case SAMPLE_TIME_EVENT(__) then '#define $P$sample<%index%> data->simulationInfo->samples[<%intSub(index, 1)%>]'
          else ''
        ;separator="\n")%>

      <%functions |> fn hasindex i0 => '#define <%functionName(fn,false)%>_index <%i0%>'; separator="\n"%>
      >>
  end match
end variableDefinitions;

template globalDataParDefine(SimVar simVar, String arrayName)
  "Generates a define statement for a parameter."
::=
 match simVar
  case SIMVAR(arrayCref=SOME(c),aliasvar=NOALIAS()) then
    <<
    /* <%crefStrNoUnderscore(c)%> */
    #define <%cref(c)%> data->simulationInfo-><%arrayName%>[<%index%>]

    <%crefMacroSubsAtEndParNew(c)%>

    /* <%crefStrNoUnderscore(name)%> */
    #define <%cref(name)%> data->simulationInfo-><%arrayName%>[<%index%>]
    #define $P$ATTRIBUTE<%cref(name)%> data->modelData-><%arrayName%>Data[<%index%>].attribute
    #define $P$ATTRIBUTE$P$PRE<%cref(name)%> $P$ATTRIBUTE<%cref(name)%>
    #define _<%cref(name)%>(i) <%cref(name)%>
    #define <%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%index%>].info

    >>
  case SIMVAR(aliasvar=NOALIAS()) then
    <<
    /* <%crefStrNoUnderscore(name)%> */
    #define <%cref(name)%> data->simulationInfo-><%arrayName%>[<%index%>]
    #define _<%cref(name)%>(i) <%cref(name)%>
    #define $P$ATTRIBUTE<%cref(name)%> data->modelData-><%arrayName%>Data[<%index%>].attribute
    #define $P$ATTRIBUTE$P$PRE<%cref(name)%> $P$ATTRIBUTE<%cref(name)%>
    #define <%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%index%>].info

    >>
  end match
end globalDataParDefine;

template globalDataVarDefine(SimVar simVar, String arrayName, Integer offset) "template globalDataVarDefine
  Generates a define statement for a varable in the global data section."
::=
  match simVar
  case SIMVAR(arrayCref=SOME(c),aliasvar=NOALIAS()) then
  let tmp = System.tmpTick()
    <<
    /* <%crefStrNoUnderscore(c)%> */
    #define _<%cref(c)%>(i) data->localData[i]-><%arrayName%>[<%intAdd(offset,index)%>]
    #define <%cref(c)%> _<%cref(c)%>(0)
    #define $P$PRE<%cref(c)%> data->simulationInfo-><%arrayName%>Pre[<%intAdd(offset,index)%>]

    <%crefMacroSubsAtEndVarNew(c)%>

    /* <%crefStrNoUnderscore(name)%> */
    #define _<%cref(name)%>(i) data->localData[i]-><%arrayName%>[<%intAdd(offset,index)%>]
    #define <%cref(name)%> _<%cref(name)%>(0)
    #define $P$PRE<%cref(name)%> data->simulationInfo-><%arrayName%>Pre[<%intAdd(offset,index)%>]
    #define $P$ATTRIBUTE<%cref(name)%> data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].attribute
    #define $P$ATTRIBUTE$P$PRE<%cref(name)%> $P$ATTRIBUTE<%cref(name)%>
    #define <%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].info
    #define $P$PRE<%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].info

    >>
  case SIMVAR(aliasvar=NOALIAS()) then
  let tmp = System.tmpTick()
    <<
    /* <%crefStrNoUnderscore(name)%> */
    #define _<%cref(name)%>(i) data->localData[i]-><%arrayName%>[<%intAdd(offset,index)%>]
    #define <%cref(name)%> _<%cref(name)%>(0)
    #define $P$PRE<%cref(name)%> data->simulationInfo-><%arrayName%>Pre[<%intAdd(offset,index)%>]
    #define $P$ATTRIBUTE<%cref(name)%> data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].attribute
    #define $P$ATTRIBUTE$P$PRE<%cref(name)%> $P$ATTRIBUTE<%cref(name)%>
    #define <%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].info
    #define $P$PRE<%cref(name)%>__varInfo data->modelData-><%arrayName%>Data[<%intAdd(offset,index)%>].info
    #define _$P$PRE<%cref(name)%>(i) $P$PRE<%cref(name)%>

    >>
  end match
end globalDataVarDefine;

template crefMacroSubsAtEndParNew(ComponentRef cr)
::=
  let &auxFunction = buffer ""
  let fullpath = contextCref(cr,contextSimulationNonDiscrete, &auxFunction)
  let nosubfullpath = contextCref(crefStripSubs(cr),contextSimulationNonDiscrete, &auxFunction)
  let totnrdims = listLength(crefDims(cr))
  let dimstr = crefDims(cr) |> dim => dimension(dim) ;separator=", "
  let substr = generateSubPalceholders(cr)
  let &subsDimThread = buffer "" /*BUFD*/
  <<
  #define <%nosubfullpath%>_index(<%substr%>)    (&<%fullpath%>)[calc_base_index_dims_subs(<%totnrdims%>, <%dimstr%>, <%substr%>)]
  >>
end crefMacroSubsAtEndParNew;

template crefMacroSubsAtEndVarNew(ComponentRef cr)
::=
  let &auxFunction = buffer ""
  let fullpath = contextCref(cr,contextSimulationNonDiscrete, &auxFunction)
  let nosubfullpath = contextCref(crefStripSubs(cr),contextSimulationNonDiscrete, &auxFunction)
  let totnrdims = listLength(crefDims(cr))
  let dimstr = crefDims(cr) |> dim => dimension(dim) ;separator=", "
  let substr = generateSubPalceholders(cr)
  let &subsDimThread = buffer "" /*BUFD*/
  <<
  #define <%nosubfullpath%>_index(<%substr%>)    (&<%fullpath%>)[calc_base_index_dims_subs(<%totnrdims%>, <%dimstr%>, <%substr%>)]
  #define $P$PRE<%nosubfullpath%>_index(<%substr%>)    (&$P$PRE<%fullpath%>)[calc_base_index_dims_subs(<%totnrdims%>, <%dimstr%>, <%substr%>)]
  >>
end crefMacroSubsAtEndVarNew;

template globalDataAliasVarArray(String _type, String _name, list<SimVar> items)
  "Generates array with variable names in global data section."
::=
  match items
  case {} then
    <<
      <%_type%> <%_name%>[1] = {{0,0,-1}};
    >>
  case items then
    <<
      <%_type%> <%_name%>[<%listLength(items)%>] = {
        <%items |> var as SIMVAR(__) => '{<%aliasVarNameType(aliasvar)%>,<%index%>}'; separator=",\n"%>
      };
    >>
  end match
end globalDataAliasVarArray;

template variableDefinitionsJacobians(list<JacobianMatrix> JacobianMatrixes, String modelNamePrefix) "template variableDefinitionsJacobians
  Generates defines for jacobian vars."
::=
  let analyticVars = (JacobianMatrixes |> (jacColumn, seedVars, name, (_,_), _, _, indexJacobian)  =>
    let varsDef = variableDefinitionsJacobians2(indexJacobian, jacColumn, seedVars, name)
    <<
    #if defined(__cplusplus)
    extern "C" {
    #endif
      #define <%symbolName(modelNamePrefix,"INDEX_JAC_")%><%name%> <%indexJacobian%>
      int <%symbolName(modelNamePrefix,"functionJac")%><%name%>_column(void* data, threadData_t *threadData);
      int <%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%name%>(void* data, threadData_t *threadData);
    #if defined(__cplusplus)
    }
    #endif
    <%varsDef%>
    >>
    ;separator="\n";empty)

  <<
  /* Jacobian Variables */
  <%analyticVars%>

  >>
end variableDefinitionsJacobians;

template variableDefinitionsJacobians2(Integer indexJacobian, list<JacobianColumn> jacobianColumn, list<SimVar> seedVars, String name) "template variableDefinitionsJacobians2
  Generates Matrixes for Linear Model."
::=
  let seedVarsResult = (seedVars |> var hasindex index0 =>
    jacobianVarDefine(var, "jacobianVarsSeed", indexJacobian, index0, name)
    ;separator="\n")
  let columnVarsResult = (jacobianColumn |> (_,vars,_) =>
    (vars |> var hasindex index0 => jacobianVarDefine(var, "jacobianVars", indexJacobian, index0, name);separator="\n")
    ;separator="\n\n")
  /* generate at least one print command to have the same index and avoid the strange side effect */
  <<
  /* <%name%> */
  <%seedVarsResult%>
  <%columnVarsResult%>
  >>
end variableDefinitionsJacobians2;

template jacobianVarDefine(SimVar simVar, String array, Integer indexJac, Integer index0, String matrixName) "template jacobianVarDefine
  "
::=
  match array
  case "jacobianVars" then
    match simVar
    case SIMVAR(aliasvar=NOALIAS(),name=name) then
      let crefName = cref(name)
      let arrayName = match index case -1 then 'tmpVars[<%index0%>]' else 'resultVars[<%index%>]'
      let optDefineMayer = if stringEq('<%crefName%>', '$P<%BackendDAE.optimizationMayerTermName%>$P$pDERC$PdummyVarC') then "\n"+'#define <%crefName%>$indexdiffed <%index%>' else ''
      let optDefineLangrangeB = if stringEq('<%crefName%>', '$P<%BackendDAE.optimizationLagrangeTermName%>$P$pDERB$PdummyVarB') then "\n"+'#define <%crefName%>$indexdiffed <%index%>' else ''
      let optDefineLangrangeC = if stringEq('<%crefName%>', '$P<%BackendDAE.optimizationLagrangeTermName%>$P$pDERC$PdummyVarC') then "\n"+'#define <%crefName%>$indexdiffed <%index%>' else ''
        <<
        #define _<%crefName%>(i) data->simulationInfo->analyticJacobians[<%indexJac%>].<%arrayName%>
        #define <%crefName%> _<%crefName%>(0)
        #define <%crefName%>__varInfo dummyVAR_INFO
        #define $P$ATTRIBUTE<%crefName%> dummyREAL_ATTRIBUTE<%optDefineMayer%><%optDefineLangrangeB%><%optDefineLangrangeC%>
        >>
    end match
  case "jacobianVarsSeed" then
    match simVar
    case SIMVAR(aliasvar=NOALIAS()) then
      let tmp = System.tmpTick()
      let crefName = cref(name)
      <<
      #define <%crefName%> data->simulationInfo->analyticJacobians[<%indexJac%>].seedVars[<%index0%>]
      #define <%crefName%>__varInfo dummyVAR_INFO
      >>
    end match
  end match
end jacobianVarDefine;

template aliasVarNameType(AliasVariable var)
  "Generates type of alias."
::=
  match var
  case NOALIAS() then
    <<
    0,0
    >>
  case ALIAS(__) then
    <<
    &<%cref(varName)%>,0
    >>
  case NEGATEDALIAS(__) then
    <<
    &<%cref(varName)%>,1
    >>
  end match
end aliasVarNameType;

template functionCallExternalObjectConstructors(ExtObjInfo extObjInfo, String modelNamePrefix)
  "Generates function in simulation file."
::=
  match extObjInfo
  case EXTOBJINFO(__) then
    let &auxFunction = buffer ""
    let &varDecls = buffer ""
    let ctorCalls = (vars |> var as SIMVAR(initialValue=SOME(exp)) =>
        let &preExp = buffer ""
        let arg = daeExp(exp, contextOther, &preExp, &varDecls, &auxFunction)
        /* Restore the memory state after each object has been initialized. Then we can
         * initalize a really large number of external objects that play with strings :)
         */
        <<
        <%preExp%>
        <%cref(var.name)%> = <%arg%>;
        >>
      ;separator="\n")

    <<
    <%auxFunction%>
    /* Has to be performed after _init.xml file has been read */
    void <%symbolName(modelNamePrefix,"callExternalObjectConstructors")%>(DATA *data, threadData_t *threadData)
    {
      <%varDecls%>
      /* data->simulationInfo->extObjs = NULL; */
      infoStreamPrint(LOG_DEBUG, 0, "call external Object Constructors");
      <%ctorCalls%>
      <%aliases |> (var1, var2) => '<%cref(var1)%> = <%cref(var2)%>;' ;separator="\n"%>
      infoStreamPrint(LOG_DEBUG, 0, "call external Object Constructors finished");
    }
    >>
  end match
end functionCallExternalObjectConstructors;

template functionCallExternalObjectDestructors(ExtObjInfo extObjInfo, String modelNamePrefix)
  "Generates function in simulation file."
::=
  match extObjInfo
  case extObjInfo as EXTOBJINFO(__) then
    <<
    void <%symbolName(modelNamePrefix,"callExternalObjectDestructors")%>(DATA *data, threadData_t *threadData)
    {
      if(data->simulationInfo->extObjs)
      {
        <%extObjInfo.vars |> var as SIMVAR(varKind=ext as EXTOBJ(__)) => 'omc_<%underscorePath(ext.fullClassName)%>_destructor(threadData,<%cref(var.name)%>);' ;separator="\n"%>
        free(data->simulationInfo->extObjs);
        data->simulationInfo->extObjs = 0;
      }
    }
    >>
  end match
end functionCallExternalObjectDestructors;

template functionInput(ModelInfo modelInfo, String modelNamePrefix)
  "Generates function in simulation file."
::=
  match modelInfo
  case MODELINFO(vars=SIMVARS(__)) then
    <<
    int <%symbolName(modelNamePrefix,"input_function")%>(DATA *data, threadData_t *threadData)
    {
      TRACE_PUSH

      <%vars.inputVars |> SIMVAR(__) hasindex i0 =>
        '<%cref(name)%> = data->simulationInfo->inputVars[<%i0%>];'
        ;separator="\n"
      %>

      TRACE_POP
      return 0;
    }

    int <%symbolName(modelNamePrefix,"input_function_init")%>(DATA *data, threadData_t *threadData)
    {
      TRACE_PUSH

      <%vars.inputVars |> SIMVAR(__) hasindex i0 =>
        'data->simulationInfo->inputVars[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.start;'
        ;separator="\n"
      %>

      TRACE_POP
      return 0;
    }

    int <%symbolName(modelNamePrefix,"input_function_updateStartValues")%>(DATA *data, threadData_t *threadData)
    {
      TRACE_PUSH

      <%vars.inputVars |> SIMVAR(__) hasindex i0 =>
        '$P$ATTRIBUTE<%cref(name)%>.start = data->simulationInfo->inputVars[<%i0%>];'
        ;separator="\n"
      %>

      TRACE_POP
      return 0;
    }
    >>
  end match
end functionInput;

template functionSymEuler(ModelInfo modelInfo, String modelNamePrefix)
  "Generates function in simulation file."
::=
  match modelInfo
  case MODELINFO(vars=SIMVARS(__)) then
    <<
    int <%symbolName(modelNamePrefix,"symEulerUpdate")%>(DATA *data, modelica_real dt)
    {
      TRACE_PUSH
      #ifdef $P<%BackendDAE.symEulerDT%>
        $P<%BackendDAE.symEulerDT%> = dt;
      #else
        return -1;
      #endif

      TRACE_POP
      return 0;
    }

    >>
  end match
end functionSymEuler;

template functionOutput(ModelInfo modelInfo, String modelNamePrefix)
  "Generates function in simulation file."
::=
  match modelInfo
  case MODELINFO(vars=SIMVARS(__)) then
    <<
    int <%symbolName(modelNamePrefix,"output_function")%>(DATA *data, threadData_t *threadData)
    {
      TRACE_PUSH

      <%vars.outputVars |> SIMVAR(__) hasindex i0 =>
        'data->simulationInfo->outputVars[<%i0%>] = <%cref(name)%>;'
        ;separator="\n"
      %>

      TRACE_POP
      return 0;
    }
    >>
  end match
end functionOutput;

template functionInitSample(list<BackendDAE.TimeEvent> timeEvents, String modelNamePrefix)
  "Generates function initSample() in simulation file."
::=
  let &varDecls = buffer ""
  let &auxFunction = buffer ""
  let body = (timeEvents |> timeEvent =>
      match timeEvent
        case SAMPLE_TIME_EVENT(__) then
          let &preExp = buffer ""
          let e1 = daeExp(startExp, contextOther, &preExp, &varDecls, &auxFunction)
          let e2 = daeExp(intervalExp, contextOther, &preExp, &varDecls, &auxFunction)
          <<
          <%preExp%>
          /* $P$sample<%index%> */
          data->modelData->samplesInfo[i].index = <%index%>;
          data->modelData->samplesInfo[i].start = <%e1%>;
          data->modelData->samplesInfo[i].interval = <%e2%>;
          assertStreamPrint(threadData,data->modelData->samplesInfo[i].interval > 0.0, "sample-interval <= 0.0");
          i++;
          >>
        else '')
  let res = <<
  /* Initializes the raw time events of the simulation using the now
     calcualted parameters. */
  void <%symbolName(modelNamePrefix,"function_initSample")%>(DATA *data, threadData_t *threadData)
  {
    long i=0;
    <%varDecls%>
    <%body%>
  }
  >>
  <<
  <%auxFunction%>
  <%res%>
  >>
end functionInitSample;


template functionInitialMixedSystems(list<SimEqSystem> initialEquations, list<SimEqSystem> initialEquations_lambda0, list<SimEqSystem> parameterEquations, list<SimEqSystem> allEquations, list<JacobianMatrix> jacobianMatrixes, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  let initbody = functionInitialMixedSystemsTemp(initialEquations)
  let initbody_lambda0 = functionInitialMixedSystemsTemp(initialEquations_lambda0)
  let parambody = functionInitialMixedSystemsTemp(parameterEquations)
  let body = functionInitialMixedSystemsTemp(allEquations)
  let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionInitialMixedSystemsTemp(jacobianEquations) ;separator="\n\n")
  <<
  /* funtion initialize mixed systems */
  void <%symbolName(modelNamePrefix,"initialMixedSystem")%>(int nMixedSystems, MIXED_SYSTEM_DATA* mixedSystemData)
  {
    /* initial mixed systems */
    <%initbody%>
    /* initial_lambda0 mixed systems */
    <%initbody_lambda0%>
    /* parameter mixed systems */
    <%parambody%>
    /* model mixed systems */
    <%body%>
    /* jacobians mixed systems */
    <%jacobianbody%>
  }
  >>
end functionInitialMixedSystems;

template functionInitialMixedSystemsTemp(list<SimEqSystem> allEquations)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
     case eq as SES_MIXED(__) then
     let size = listLength(discVars)
     <<
     assertStreamPrint(NULL, nMixedSystems > <%indexMixedSystem%>, "Internal Error: nMixedSystems mismatch!");
     mixedSystemData[<%indexMixedSystem%>].equationIndex = <%index%>;
     mixedSystemData[<%indexMixedSystem%>].size = <%size%>;
     mixedSystemData[<%indexMixedSystem%>].solveContinuousPart = updateContinuousPart<%index%>;
     mixedSystemData[<%indexMixedSystem%>].updateIterationExps = updateIterationExpMixedSystem<%index%>;
     >>
   )
   ;separator="\n\n")
end functionInitialMixedSystemsTemp;


template functionSetupMixedSystems(list<SimEqSystem> initialEquations, list<SimEqSystem> initialEquations_lambda0, list<SimEqSystem> parameterEquations, list<SimEqSystem> allEquations, list<JacobianMatrix> jacobianMatrixes, Text &header, String modelNamePrefixStr)
  "Generates functions in simulation file."
::=
  let initbody = functionSetupMixedSystemsTemp(initialEquations, &header,modelNamePrefixStr)
  let initbody_lambda0 = functionSetupMixedSystemsTemp(initialEquations_lambda0, &header,modelNamePrefixStr)
  let parambody = functionSetupMixedSystemsTemp(parameterEquations,&header,modelNamePrefixStr)
  let body = functionSetupMixedSystemsTemp(allEquations,&header,modelNamePrefixStr)
  let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionSetupMixedSystemsTemp(jacobianEquations,&header,modelNamePrefixStr);separator="\n\n")
  <<
  /* initial mixed systems */
  <%initbody%>
  /* initial_lambda0 mixed systems */
  <%initbody_lambda0%>
  /* parameter mixed systems */
  <%parambody%>
  /* model mixed systems */
  <%body%>
  /* jacobians mixed systems */
  <%jacobianbody%>
  >>
end functionSetupMixedSystems;

template functionSetupMixedSystemsTemp(list<SimEqSystem> allEquations, Text &header, String modelNamePrefixStr)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
     case eq as SES_MIXED(__) then
       let contEqsIndex = equationIndex(cont)
       let solvedContinuous =
         match cont
           // no dynamic tearing
           case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
             'data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].solved'
           case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
             'data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].solved'
           // dynamic tearing
           case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(at as LINEARSYSTEM(__))) then
             <<
               data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].solved'
               data->simulationInfo->linearSystemData[<%at.indexLinearSystem%>].solved'
             >>
           case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
             <<
               data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].solved
               data->simulationInfo->nonlinearSystemData[<%at.indexNonLinearSystem%>].solved
             >>
       let &preDisc = buffer ""
       let &varDecls = buffer ""
       let &auxFunction = buffer ""
       let discExp = (discEqs |> SES_SIMPLE_ASSIGN(__) hasindex i0 =>
          let expPart = daeExp(exp, contextSimulationDiscrete, &preDisc, &varDecls, &auxFunction)
          <<
          <%cref(cref)%> = <%expPart%>;
          >>
        ;separator="\n")
       let &header += 'void updateContinuousPart<%index%>(void *);<%\n%>void updateIterationExpMixedSystem<%index%>(void *);<%\n%>'
       <<
       <%auxFunction%>
       void updateContinuousPart<%index%>(void *inData)
       {
         DATA* data = (DATA*) inData;
         <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%contEqsIndex%>(data, threadData);
         data->simulationInfo->mixedSystemData[<%indexMixedSystem%>].continuous_solution = <%solvedContinuous%>;
       }

       void updateIterationExpMixedSystem<%index%>(void *inData)
       {
         DATA* data = (DATA*) inData;
         <%varDecls%>

         <%preDisc%>
         <%discExp%>
       }
       >>
   )
   ;separator="\n\n")
end functionSetupMixedSystemsTemp;


template functionInitialLinearSystems(list<SimEqSystem> initialEquations, list<SimEqSystem> initialEquations_lambda0, list<SimEqSystem> parameterEquations, list<SimEqSystem> allEquations, list<JacobianMatrix> jacobianMatrixes, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  let initbody = functionInitialLinearSystemsTemp(initialEquations, modelNamePrefix)
  let initbody_lambda0 = functionInitialLinearSystemsTemp(initialEquations_lambda0, modelNamePrefix)
  let parambody = functionInitialLinearSystemsTemp(parameterEquations, modelNamePrefix)
  let body = functionInitialLinearSystemsTemp(allEquations, modelNamePrefix)
  let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionInitialLinearSystemsTemp(jacobianEquations, modelNamePrefix);separator="\n\n")
  <<
  /* funtion initialize linear systems */
  void <%symbolName(modelNamePrefix,"initialLinearSystem")%>(int nLinearSystems, LINEAR_SYSTEM_DATA* linearSystemData)
  {
    /* initial linear systems */
    <%initbody%>
    /* initial_lambda0 linear systems */
    <%initbody_lambda0%>
    /* parameter linear systems */
    <%parambody%>
    /* model linear systems */
    <%body%>
    /* jacobians linear systems */
    <%jacobianbody%>
  }
  >>
end functionInitialLinearSystems;

template functionInitialLinearSystemsTemp(list<SimEqSystem> allEquations, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
     case eq as SES_MIXED(__) then functionInitialLinearSystemsTemp(fill(eq.cont,1), modelNamePrefix)
     // no dynamic tearing
     case eq as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
       match ls.jacobianMatrix
         case NONE() then
           let size = listLength(ls.vars)
           let nnz = listLength(ls.simJac)
           <<
           assertStreamPrint(NULL, nLinearSystems > <%ls.indexLinearSystem%>, "Internal Error: nLinearSystems mismatch!");
           linearSystemData[<%ls.indexLinearSystem%>].equationIndex = <%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].size = <%size%>;
           linearSystemData[<%ls.indexLinearSystem%>].nnz = <%nnz%>;
           linearSystemData[<%ls.indexLinearSystem%>].method = 0;
           linearSystemData[<%ls.indexLinearSystem%>].setA = setLinearMatrixA<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].setb = setLinearVectorb<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%ls.index%>;
           >>
         case SOME(__) then
           let size = listLength(ls.vars)
           let nnz = listLength(ls.simJac)
           let generatedJac = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"functionJac")%><%name%>_column' case NONE() then 'NULL'
           let initialJac = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
           let jacIndex = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex)) then '<%jacindex%>' case NONE() then '-1'
           <<
           assertStreamPrint(NULL, nLinearSystems > <%ls.indexLinearSystem%>, "Internal Error: indexlinearSystem mismatch!");
           linearSystemData[<%ls.indexLinearSystem%>].equationIndex = <%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].size = <%size%>;
           linearSystemData[<%ls.indexLinearSystem%>].nnz = <%nnz%>;
           linearSystemData[<%ls.indexLinearSystem%>].method = 1;
           linearSystemData[<%ls.indexLinearSystem%>].residualFunc = residualFunc<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].analyticalJacobianColumn = <%generatedJac%>;
           linearSystemData[<%ls.indexLinearSystem%>].initialAnalyticalJacobian = <%initialJac%>;
           linearSystemData[<%ls.indexLinearSystem%>].jacobianIndex = <%jacIndex%>;
           linearSystemData[<%ls.indexLinearSystem%>].setA = NULL;//setLinearMatrixA<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].setb = NULL; //setLinearVectorb<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%ls.index%>;
           >>
         else
         error(sourceInfo(), ' No jacobian create for linear system <%ls.index%>.')
       end match

     // dynamic tearing
     case eq as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(at as LINEARSYSTEM(__))) then
       match ls.jacobianMatrix
         case NONE() then
           // for strict tearing set
           let size = listLength(ls.vars)
           let nnz = listLength(ls.simJac)
           // for casual tearing set
           let size2 = listLength(at.vars)
           let nnz2 = listLength(at.simJac)
           <<
           assertStreamPrint(NULL, nLinearSystems > <%ls.indexLinearSystem%>, "Internal Error: nLinearSystems mismatch!");
           linearSystemData[<%ls.indexLinearSystem%>].equationIndex = <%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].size = <%size%>;
           linearSystemData[<%ls.indexLinearSystem%>].nnz = <%nnz%>;
           linearSystemData[<%ls.indexLinearSystem%>].method = 0;
           linearSystemData[<%ls.indexLinearSystem%>].setA = setLinearMatrixA<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].setb = setLinearVectorb<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%ls.index%>;

           assertStreamPrint(NULL, nLinearSystems > <%at.indexLinearSystem%>, "Internal Error: nLinearSystems mismatch!");
           linearSystemData[<%at.indexLinearSystem%>].equationIndex = <%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].size = <%size2%>;
           linearSystemData[<%at.indexLinearSystem%>].nnz = <%nnz2%>;
           linearSystemData[<%at.indexLinearSystem%>].method = 0;
           linearSystemData[<%at.indexLinearSystem%>].setA = setLinearMatrixA<%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].setb = setLinearVectorb<%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%at.index%>;
           >>
         case SOME(__) then
           let size = listLength(ls.vars)
           let nnz = listLength(ls.simJac)
           let generatedJac = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"functionJac")%><%name%>_column' case NONE() then 'NULL'
           let initialJac = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
           let jacIndex = match ls.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex)) then '<%jacindex%>' case NONE() then '-1'
           let size2 = listLength(at.vars)
           let nnz2 = listLength(at.simJac)
           let generatedJac2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"functionJac")%><%name%>_column' case NONE() then 'NULL'
           let initialJac2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
           let jacIndex2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex)) then '<%jacindex%>' case NONE() then '-1'
           <<
           assertStreamPrint(NULL, nLinearSystems > <%ls.indexLinearSystem%>, "Internal Error: indexlinearSystem mismatch!");
           linearSystemData[<%ls.indexLinearSystem%>].equationIndex = <%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].size = <%size%>;
           linearSystemData[<%ls.indexLinearSystem%>].nnz = <%nnz%>;
           linearSystemData[<%ls.indexLinearSystem%>].method = 1;
           linearSystemData[<%ls.indexLinearSystem%>].residualFunc = residualFunc<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].analyticalJacobianColumn = <%generatedJac%>;
           linearSystemData[<%ls.indexLinearSystem%>].initialAnalyticalJacobian = <%initialJac%>;
           linearSystemData[<%ls.indexLinearSystem%>].jacobianIndex = <%jacIndex%>;
           linearSystemData[<%ls.indexLinearSystem%>].setA = NULL;//setLinearMatrixA<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].setb = NULL; //setLinearVectorb<%ls.index%>;
           linearSystemData[<%ls.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%ls.index%>;

           assertStreamPrint(NULL, nLinearSystems > <%at.indexLinearSystem%>, "Internal Error: indexlinearSystem mismatch!");
           linearSystemData[<%at.indexLinearSystem%>].equationIndex = <%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].size = <%size2%>;
           linearSystemData[<%at.indexLinearSystem%>].nnz = <%nnz2%>;
           linearSystemData[<%at.indexLinearSystem%>].method = 1;
           linearSystemData[<%at.indexLinearSystem%>].residualFunc = residualFunc<%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].analyticalJacobianColumn = <%generatedJac2%>;
           linearSystemData[<%at.indexLinearSystem%>].initialAnalyticalJacobian = <%initialJac2%>;
           linearSystemData[<%at.indexLinearSystem%>].jacobianIndex = <%jacIndex2%>;
           linearSystemData[<%at.indexLinearSystem%>].setA = NULL;//setLinearMatrixA<%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].setb = NULL; //setLinearVectorb<%at.index%>;
           linearSystemData[<%at.indexLinearSystem%>].initializeStaticLSData = initializeStaticLSData<%at.index%>;
           >>
         else
         error(sourceInfo(), ' No jacobian create for linear system <%ls.index%> or <%at.index%>.')
       end match
   )
   ;separator="\n\n")
end functionInitialLinearSystemsTemp;

template functionSetupLinearSystems(list<SimEqSystem> initialEquations, list<SimEqSystem> initialEquations_lambda0, list<SimEqSystem> parameterEquations, list<SimEqSystem> allEquations, list<JacobianMatrix> jacobianMatrixes, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  let initbody = functionSetupLinearSystemsTemp(initialEquations, modelNamePrefix)
  let initbody_lambda0 = functionSetupLinearSystemsTemp(initialEquations_lambda0, modelNamePrefix)
  let parambody = functionSetupLinearSystemsTemp(parameterEquations, modelNamePrefix)
  let body = functionSetupLinearSystemsTemp(allEquations, modelNamePrefix)
  let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionSetupLinearSystemsTemp(jacobianEquations, modelNamePrefix);separator="\n\n")
  <<
  /* initial linear systems */
  <%initbody%>
  /* initial_lambda0 linear systems */
  <%initbody_lambda0%>
  /* parameter linear systems */
  <%parambody%>
  /* model linear systems */
  <%body%>
  /* jacobians linear systems */
  <%jacobianbody%>
  >>
end functionSetupLinearSystems;

template functionSetupLinearSystemsTemp(list<SimEqSystem> allEquations, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
     case eq as SES_MIXED(__) then functionSetupLinearSystemsTemp(fill(eq.cont,1), modelNamePrefix)
     // no dynamic tearing
     case eq as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
     match ls.jacobianMatrix
       case SOME(__) then
         let &varDeclsRes = buffer "" /*BUFD*/
         let &auxFunction = buffer ""
         let &tmp = buffer ""
         let xlocs = (ls.vars |> var hasindex i0 => '<%cref(varName(var))%> = xloc[<%i0%>];' ;separator="\n")
         let prebody = (ls.residual |> eq2 =>
               functionExtraResidualsPreBody(eq2, &varDeclsRes /*BUFD*/, &tmp, modelNamePrefix)
          ;separator="\n")
         let body = (ls.residual |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
         let &preExp = buffer "" /*BUFD*/
         let expPart = daeExp(eq2.exp, contextSimulationDiscrete,
                              &preExp /*BUFC*/, &varDeclsRes, &auxFunction)
           <<
           <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
           <%preExp%>res[<%i0%>] = <%expPart%>;
           <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
           >> ;separator="\n")
         let body_initializeStaticLSData = (ls.vars |> var hasindex i0 =>
           <<
           /* static ls data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")
       <<
       <%auxFunction%>
       <%tmp%>

       void residualFunc<%ls.index%>(void** dataIn, const double* xloc, double* res, const int* iflag)
       {
         DATA *data = (DATA*) ((void**)dataIn[0]);
         threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
         const int equationIndexes[2] = {1,<%ls.index%>};
         <%varDeclsRes%>
         <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ls.index%>);' %>
         <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex,1);' %>
         <%xlocs%>
         <%prebody%>
         <%body%>
         <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ls.index%>);' %>
       }
       void initializeStaticLSData<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData%>
       }
       >>
       else
         let &varDecls = buffer "" /*BUFD*/
         let &auxFunction = buffer ""
         let MatrixA = (ls.simJac |> (row, col, eq as SES_RESIDUAL(__)) hasindex i0 =>
           let &preExp = buffer "" /*BUFD*/
           let expPart = daeExp(eq.exp, contextSimulationDiscrete, &preExp,  &varDecls, &auxFunction)
             '<%preExp%>linearSystemData->setAElement(<%row%>, <%col%>, <%expPart%>, <%i0%>, linearSystemData, threadData);'
          ;separator="\n")

         let &varDecls2 = buffer "" /*BUFD*/
         let vectorb = (ls.beqs |> exp hasindex i0 =>
           let &preExp = buffer "" /*BUFD*/
           let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, &varDecls2, &auxFunction)
             '<%preExp%>linearSystemData->setBElement(<%i0%>, <%expPart%>, linearSystemData, threadData);'
          ;separator="\n")
         let body_initializeStaticLSData = (ls.vars |> var hasindex i0 =>
           <<
           /* static ls data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")
       <<
       <%auxFunction%>
       void setLinearMatrixA<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%ls.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls%>
         <%MatrixA%>
       }
       void setLinearVectorb<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%ls.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls2%>
         <%vectorb%>
       }
       void initializeStaticLSData<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData%>
       }
       >>
     end match

     // dynamic tearing
     case eq as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(at as LINEARSYSTEM(__))) then
     match ls.jacobianMatrix
       case SOME(__) then
         // for strict tearing set
         let &varDeclsRes = buffer "" /*BUFD*/
         let &auxFunction = buffer ""
         let &tmp = buffer ""
         let xlocs = (ls.vars |> var hasindex i0 => '<%cref(varName(var))%> = xloc[<%i0%>];' ;separator="\n")
         let prebody = (ls.residual |> eq2 =>
               functionExtraResidualsPreBody(eq2, &varDeclsRes /*BUFD*/, &tmp, modelNamePrefix)
          ;separator="\n")
         let body = (ls.residual |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
         let &preExp = buffer "" /*BUFD*/
         let expPart = daeExp(eq2.exp, contextSimulationDiscrete,
                              &preExp /*BUFC*/, &varDeclsRes, &auxFunction)
           <<
           <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
           <%preExp%>res[<%i0%>] = <%expPart%>;
           <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
           >> ;separator="\n")
         let body_initializeStaticLSData = (ls.vars |> var hasindex i0 =>
           <<
           /* static ls data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")
         // for casual tearing set
         let &varDeclsRes2 = buffer "" /*BUFD*/
         let &auxFunction2 = buffer ""
         let &tmp2 = buffer ""
         let xlocs2 = (at.vars |> var hasindex i0 => '<%cref(varName(var))%> = xloc[<%i0%>];' ;separator="\n")
         let prebody2 = (at.residual |> eq2 =>
               functionExtraResidualsPreBody(eq2, &varDeclsRes2 /*BUFD*/, &tmp2, modelNamePrefix)
          ;separator="\n")
         let body2 = (at.residual |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
         let &preExp2 = buffer "" /*BUFD*/
         let expPart2 = daeExp(eq2.exp, contextSimulationDiscrete,
                              &preExp2 /*BUFC*/, &varDeclsRes2, &auxFunction2)
           <<
           <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
           <%preExp2%>res[<%i0%>] = <%expPart2%>;
           <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
           >> ;separator="\n")
         let body_initializeStaticLSData2 = (at.vars |> var hasindex i0 =>
           <<
           /* static at data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")

       <<
       <%auxFunction%>
       <%tmp%>

       void residualFunc<%ls.index%>(void** dataIn, const double* xloc, double* res, const int* iflag)
       {
         DATA *data = (DATA*) ((void**)dataIn[0]);
         threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
         const int equationIndexes[2] = {1,<%ls.index%>};
         <%varDeclsRes%>
         <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ls.index%>);' %>
         <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex,1);' %>
         <%xlocs%>
         <%prebody%>
         <%body%>
         <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ls.index%>);' %>
       }
       void initializeStaticLSData<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData%>
       }

       <%auxFunction2%>
       <%tmp2%>

       void residualFunc<%at.index%>(void** dataIn, const double* xloc, double* res, const int* iflag)
       {
         DATA *data = (DATA*) ((void**)dataIn[0]);
         threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
         const int equationIndexes[2] = {1,<%at.index%>};
         <%varDeclsRes2%>
         <% if profileAll() then 'SIM_PROF_TICK_EQ(<%at.index%>);' %>
         <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%at.index%>).profileBlockIndex,1);' %>
         <%xlocs2%>
         <%prebody2%>
         <%body2%>
         <% if profileAll() then 'SIM_PROF_ACC_EQ(<%at.index%>);' %>
       }
       void initializeStaticLSData<%at.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData2%>
       }
       >>
       else
         // for strict tearing set
         let &varDecls = buffer "" /*BUFD*/
         let &auxFunction = buffer ""
         let MatrixA = (ls.simJac |> (row, col, eq as SES_RESIDUAL(__)) hasindex i0 =>
           let &preExp = buffer "" /*BUFD*/
           let expPart = daeExp(eq.exp, contextSimulationDiscrete, &preExp,  &varDecls, &auxFunction)
             '<%preExp%>linearSystemData->setAElement(<%row%>, <%col%>, <%expPart%>, <%i0%>, linearSystemData, threadData);'
          ;separator="\n")

         let &varDecls2 = buffer "" /*BUFD*/
         let vectorb = (ls.beqs |> exp hasindex i0 =>
           let &preExp = buffer "" /*BUFD*/
           let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, &varDecls2, &auxFunction)
             '<%preExp%>linearSystemData->setBElement(<%i0%>, <%expPart%>, linearSystemData, threadData);'
          ;separator="\n")
         let body_initializeStaticLSData = (ls.vars |> var hasindex i0 =>
           <<
           /* static ls data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")
         // for casual tearing set
         let &varDecls3 = buffer "" /*BUFD*/
         let &auxFunction2 = buffer ""
         let MatrixA2 = (at.simJac |> (row, col, eq as SES_RESIDUAL(__)) hasindex i0 =>
           let &preExp3 = buffer "" /*BUFD*/
           let expPart3 = daeExp(eq.exp, contextSimulationDiscrete, &preExp3,  &varDecls3, &auxFunction2)
             '<%preExp3%>linearSystemData->setAElement(<%row%>, <%col%>, <%expPart3%>, <%i0%>, linearSystemData, threadData);'
          ;separator="\n")

         let &varDecls4 = buffer "" /*BUFD*/
         let vectorb2 = (at.beqs |> exp hasindex i0 =>
           let &preExp4 = buffer "" /*BUFD*/
           let expPart4 = daeExp(exp, contextSimulationDiscrete, &preExp4, &varDecls4, &auxFunction2)
             '<%preExp4%>linearSystemData->setBElement(<%i0%>, <%expPart4%>, linearSystemData, threadData);'
          ;separator="\n")
         let body_initializeStaticLSData2 = (at.vars |> var hasindex i0 =>
           <<
           /* static at data for <%cref(varName(var))%> */
           linearSystemData->nominal[i] = $P$ATTRIBUTE<%cref(varName(var))%>.nominal;
           linearSystemData->min[i]     = $P$ATTRIBUTE<%cref(varName(var))%>.min;
           linearSystemData->max[i++]   = $P$ATTRIBUTE<%cref(varName(var))%>.max;
           >> ;separator="\n")

       <<
       <%auxFunction%>
       void setLinearMatrixA<%ls.index%>(void *inData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%ls.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls%>
         <%MatrixA%>
       }
       void setLinearVectorb<%ls.index%>(void *inData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%ls.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls2%>
         <%vectorb%>
       }
       void initializeStaticLSData<%ls.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData%>
       }

       <%auxFunction2%>
       void setLinearMatrixA<%at.index%>(void *inData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%at.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls3%>
         <%MatrixA2%>
       }
       void setLinearVectorb<%at.index%>(void *inData, void *systemData)
       {
         const int equationIndexes[2] = {1,<%at.index%>};
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         <%varDecls4%>
         <%vectorb2%>
       }
       void initializeStaticLSData<%at.index%>(void *inData, threadData_t *threadData, void *systemData)
       {
         DATA* data = (DATA*) inData;
         LINEAR_SYSTEM_DATA* linearSystemData = (LINEAR_SYSTEM_DATA*) systemData;
         int i=0;
         <%body_initializeStaticLSData2%>
       }
       >>
     end match
   )
   ;separator="\n\n")
end functionSetupLinearSystemsTemp;

template functionInitialNonLinearSystems(list<SimEqSystem> initialEquations, list<SimEqSystem> initialEquations_lambda0, list<SimEqSystem> parameterEquations, list<SimEqSystem> allEquations, list<JacobianMatrix> jacobianMatrixes, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  let &tempeqns = buffer ""
  let &tempeqns += (allEquations |> eq => match eq case eq as SES_NONLINEAR(alternativeTearing = SOME(__)) then 'int <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*, threadData_t*);' ; separator = "\n")
  let initbody = functionInitialNonLinearSystemsTemp(initialEquations, modelNamePrefix)
  let initbody_lambda0 = functionInitialNonLinearSystemsTemp(initialEquations_lambda0, modelNamePrefix)
  let parambody = functionInitialNonLinearSystemsTemp(parameterEquations,modelNamePrefix)
  let equationbody = functionInitialNonLinearSystemsTemp(allEquations,modelNamePrefix)
  let jacobianbody = (jacobianMatrixes |> ({(jacobianEquations,_,_)}, _, _, _, _, _, _) => functionInitialNonLinearSystemsTemp(jacobianEquations, modelNamePrefix) ;separator="\n\n")
  <<
  /* function initialize non-linear systems */
  <%tempeqns%>

  void <%symbolName(modelNamePrefix,"initialNonLinearSystem")%>(int nNonLinearSystems, NONLINEAR_SYSTEM_DATA* nonLinearSystemData)
  {
    <%initbody%>
    <%initbody_lambda0%>
    <%parambody%>
    <%equationbody%>
    <%jacobianbody%>
  }
  >>
end functionInitialNonLinearSystems;

template functionInitialNonLinearSystemsTemp(list<SimEqSystem> allEquations, String modelPrefixName)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
     case eq as SES_MIXED(__) then functionInitialNonLinearSystemsTemp(fill(eq.cont,1), modelPrefixName)
     // no dynamic tearing
     case eq as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
       let size = listLength(nls.crefs)
       let generatedJac = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"functionJac")%><%name%>_column' case NONE() then 'NULL'
       let initialJac = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
       let jacIndex = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex)) then '<%jacindex%>' case NONE() then '-1'
       let innerEqs = functionInitialNonLinearSystemsTemp(nls.eqs, modelPrefixName)
       <<
       assertStreamPrint(NULL, nNonLinearSystems > <%nls.indexNonLinearSystem%>, "Internal Error: nNonLinearSystems mismatch!");
       <%innerEqs%>
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].equationIndex = <%nls.index%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].size = <%size%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].method = 0;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].homotopySupport = <%boolStrC(nls.homotopySupport)%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].mixedSystem = <%boolStrC(nls.mixedSystem)%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].residualFunc = residualFunc<%nls.index%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].strictTearingFunctionCall = NULL;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].analyticalJacobianColumn = <%generatedJac%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].initialAnalyticalJacobian = <%initialJac%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].jacobianIndex = <%jacIndex%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].initializeStaticNLSData = initializeStaticNLSData<%nls.index%>;
       >>

     // dynamic tearing
     case eq as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
       let size = listLength(nls.crefs)
       let generatedJac = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"functionJac")%><%name%>_column' case NONE() then 'NULL'
       let initialJac = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
       let jacIndex = match nls.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex)) then '<%jacindex%>' case NONE() then '-1'
       let innerEqs = functionInitialNonLinearSystemsTemp(nls.eqs, modelPrefixName)
       let size2 = listLength(at.crefs)
       let generatedJac2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"functionJac")%><%name%>_column' case NONE() then 'NULL'
       let initialJac2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,_)) then '<%symbolName(modelPrefixName,"initialAnalyticJacobian")%><%name%>' case NONE() then 'NULL'
       let jacIndex2 = match at.jacobianMatrix case SOME((_,_,name,_,_,_,jacindex2)) then '<%jacindex2%>' case NONE() then '-1'
       let innerEqs2 = functionInitialNonLinearSystemsTemp(at.eqs, modelPrefixName)
       <<
       assertStreamPrint(NULL, nNonLinearSystems > <%nls.indexNonLinearSystem%>, "Internal Error: nNonLinearSystems mismatch!");
       <%innerEqs%>
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].equationIndex = <%nls.index%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].size = <%size%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].method = 0;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].homotopySupport = <%if nls.homotopySupport then '1' else '0'%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].mixedSystem = <%if nls.mixedSystem then '1' else '0'%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].residualFunc = residualFunc<%nls.index%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].strictTearingFunctionCall = NULL;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].analyticalJacobianColumn = <%generatedJac%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].initialAnalyticalJacobian = <%initialJac%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].jacobianIndex = <%jacIndex%>;
       nonLinearSystemData[<%nls.indexNonLinearSystem%>].initializeStaticNLSData = initializeStaticNLSData<%nls.index%>;

       assertStreamPrint(NULL, nNonLinearSystems > <%nls.indexNonLinearSystem%>, "Internal Error: nNonLinearSystems mismatch!");
       <%innerEqs2%>
       nonLinearSystemData[<%at.indexNonLinearSystem%>].equationIndex = <%at.index%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].size = <%size2%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].method = 0;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].homotopySupport = <%if at.homotopySupport then '1' else '0'%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].mixedSystem = <%if at.mixedSystem then '1' else '0'%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].residualFunc = residualFunc<%at.index%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].strictTearingFunctionCall = <%symbolName(modelPrefixName,"eqFunction")%>_<%nls.index%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].analyticalJacobianColumn = <%generatedJac2%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].initialAnalyticalJacobian = <%initialJac2%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].jacobianIndex = <%jacIndex2%>;
       nonLinearSystemData[<%at.indexNonLinearSystem%>].initializeStaticNLSData = initializeStaticNLSData<%at.index%>;
       >>
     )
   ;separator="\n\n")
end functionInitialNonLinearSystemsTemp;

template functionExtraResidualsPreBody(SimEqSystem eq, Text &varDecls, Text &eqs, String modelNamePrefixStr)
 "Generates an equation."
::=
  match eq
  case e as SES_RESIDUAL(__)
  then ""
  else
  equation_(-1, eq, contextSimulationDiscrete, &varDecls, &eqs, modelNamePrefixStr)
  end match
end functionExtraResidualsPreBody;

template functionNonLinearResiduals(list<SimEqSystem> allEquations, String modelNamePrefix)
  "Generates functions in simulation file."
::=
  (allEquations |> eqn => (match eqn
    case eq as SES_MIXED(__) then functionNonLinearResiduals(fill(eq.cont,1),modelNamePrefix)
    // no dynamic tearing
    case eq as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
      let &varDecls = buffer ""
      let &tmp = buffer ""
      let innerEqs = functionNonLinearResiduals(nls.eqs,modelNamePrefix)
      let backupOutputs = match nls.eqs
        case (alg as SES_INVERSE_ALGORITHM(__))::{} then
          let body = (alg.knownOutputCrefs |> cr =>
            let &varDecls += '<%crefType(cr)%> $OLD_<%cref(cr)%>;<%\n%>'
            '$OLD_<%cref(cr)%> = <%cref(cr)%>;'
          ;separator="\n")
          <<
          /* backup outputs of the algorithm */
          <%body%>
          >>
      let restoreKnownOutputs = match nls.eqs
        case (alg as SES_INVERSE_ALGORITHM(__))::{} then
          let body = (alg.knownOutputCrefs |> cr hasindex i0 => '<%cref(cr)%> = $OLD_<%cref(cr)%>;' ;separator="\n")
          <<
          /* restore previously known outputs of the algorithm */
          <%body%>
          >>
      let xlocs = (nls.crefs |> cr hasindex i0 => '<%cref(cr)%> = xloc[<%i0%>];' ;separator="\n")
      let body_initializeStaticNLSData = (nls.crefs |> cr hasindex i0 =>
        <<
        /* static nls data for <%cref(cr)%> */
        nlsData->nominal[i] = $P$ATTRIBUTE<%cref(cr)%>.nominal;
        nlsData->min[i]     = $P$ATTRIBUTE<%cref(cr)%>.min;
        nlsData->max[i++]   = $P$ATTRIBUTE<%cref(cr)%>.max;
        >>
      ;separator="\n")
      let prebody = (nls.eqs |> eq2 =>
        functionExtraResidualsPreBody(eq2, &varDecls, &tmp, modelNamePrefix)
      ;separator="\n")
      let body = match nls.eqs
        case (alg as SES_INVERSE_ALGORITHM(__))::{} then
          (alg.knownOutputCrefs |> cr hasindex i0 => 'res[<%i0%>] = $OLD_<%cref(cr)%> - <%cref(cr)%>;' ;separator="\n")
        else
          (nls.eqs |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
            let &preExp = buffer ""
            let expPart = daeExp(eq2.exp, contextSimulationDiscrete, &preExp, &varDecls, &tmp)
            <<
            <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
            <%preExp%>res[<%i0%>] = <%expPart%>;
            <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
            >>
          ;separator="\n")
      <<
      <%innerEqs%>
      <%&tmp%>
      void initializeStaticNLSData<%nls.index%>(void *inData, threadData_t *threadData, void *inNlsData)
      {
        DATA* data = (DATA*) inData;
        NONLINEAR_SYSTEM_DATA* nlsData = (NONLINEAR_SYSTEM_DATA*) inNlsData;
        int i=0;
        <%body_initializeStaticNLSData%>
      }

      void residualFunc<%nls.index%>(void** dataIn, const double* xloc, double* res, const int* iflag)
      {
        DATA *data = (DATA*) ((void**)dataIn[0]);
        threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
        const int equationIndexes[2] = {1,<%nls.index%>};
        <%varDecls%>
        <% if profileAll() then 'SIM_PROF_TICK_EQ(<%nls.index%>);' %>
        <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%nls.index%>).profileBlockIndex,1);' %>
        <%xlocs%>
        <%backupOutputs%>

        <%prebody%>

        <%body%>
        <%restoreKnownOutputs%>
        <% if profileAll() then 'SIM_PROF_ACC_EQ(<%nls.index%>);' %>
      }
      >>
    // dynamic tearing
    case eq as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
      // for strict tearing set
      let &varDecls = buffer ""
      let &tmp = buffer ""
      let innerEqs = functionNonLinearResiduals(nls.eqs,modelNamePrefix)
      let xlocs = (nls.crefs |> cr hasindex i0 => '<%cref(cr)%> = xloc[<%i0%>];' ;separator="\n")
      let body_initializeStaticNLSData = (nls.crefs |> cr hasindex i0 =>
        <<
          /* static nls data for <%cref(cr)%> */
          nlsData->nominal[i] = $P$ATTRIBUTE<%cref(cr)%>.nominal;
          nlsData->min[i]     = $P$ATTRIBUTE<%cref(cr)%>.min;
          nlsData->max[i++]   = $P$ATTRIBUTE<%cref(cr)%>.max;
        >>
      ;separator="\n")
      let prebody = (nls.eqs |> eq2 =>
        functionExtraResidualsPreBody(eq2, &varDecls, &tmp, modelNamePrefix)
      ;separator="\n")
      let body = (nls.eqs |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
        let &preExp = buffer ""
        let expPart = daeExp(eq2.exp, contextSimulationDiscrete, &preExp, &varDecls, &tmp)
        <<
          <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
          <%preExp%>res[<%i0%>] = <%expPart%>;
          <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
        >>
      ;separator="\n")

      // for casual tearing set
      let &varDecls2 = buffer ""
      let &tmp2 = buffer ""
      let innerEqs2 = functionNonLinearResiduals(at.eqs,modelNamePrefix)
      let xlocs2 = (at.crefs |> cr hasindex i0 => '<%cref(cr)%> = xloc[<%i0%>];' ;separator="\n")
      let body_initializeStaticNLSData2 = (at.crefs |> cr hasindex i0 =>
        <<
          /* static nls data for <%cref(cr)%> */
          nlsData->nominal[i] = $P$ATTRIBUTE<%cref(cr)%>.nominal;
          nlsData->min[i]     = $P$ATTRIBUTE<%cref(cr)%>.min;
          nlsData->max[i++]   = $P$ATTRIBUTE<%cref(cr)%>.max;
        >>
      ;separator="\n")
      let prebody2 = (at.eqs |> eq2 =>
        functionExtraResidualsPreBody(eq2, &varDecls2, &tmp2, modelNamePrefix)
      ;separator="\n")
      let body2 = (at.eqs |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
        let &preExp2 = buffer ""
        let expPart2 = daeExp(eq2.exp, contextSimulationDiscrete, &preExp2, &varDecls2, &tmp2)
        <<
          <% if profileAll() then 'SIM_PROF_TICK_EQ(<%eq2.index%>);' %>
          <%preExp2%>res[<%i0%>] = <%expPart2%>;
          <% if profileAll() then 'SIM_PROF_ACC_EQ(<%eq2.index%>);' %>
        >>
      ;separator="\n")
      <<
        <%innerEqs%>
        <%&tmp%>
        void initializeStaticNLSData<%nls.index%>(void *inData, threadData_t *threadData, void *inNlsData)
        {
          DATA* data = (DATA*) inData;
          NONLINEAR_SYSTEM_DATA* nlsData = (NONLINEAR_SYSTEM_DATA*) inNlsData;
          int i=0;
          <%body_initializeStaticNLSData%>
        }

        void residualFunc<%nls.index%>(void **dataIn, const double* xloc, double* res, const int* iflag)
        {
          DATA *data = (DATA*) ((void**)dataIn[0]);
          threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
          const int equationIndexes[2] = {1,<%nls.index%>};
          <%varDecls%>
          <% if profileAll() then 'SIM_PROF_TICK_EQ(<%nls.index%>);' %>
          <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%nls.index%>).profileBlockIndex,1);' %>
          <%xlocs%>
          <%prebody%>
          <%body%>
          <% if profileAll() then 'SIM_PROF_ACC_EQ(<%nls.index%>);' %>
        }

        <%innerEqs2%>
        <%&tmp2%>
        void initializeStaticNLSData<%at.index%>(void *inData, threadData_t *threadData, void *inNlsData)
        {
          DATA* data = (DATA*) inData;
          NONLINEAR_SYSTEM_DATA* nlsData = (NONLINEAR_SYSTEM_DATA*) inNlsData;
          int i=0;
          <%body_initializeStaticNLSData2%>
        }

        void residualFunc<%at.index%>(void **dataIn, const double* xloc, double* res, const int* iflag)
        {
          DATA *data = (DATA*) ((void**)dataIn[0]);
          threadData_t *threadData = (threadData_t*) ((void**)dataIn[1]);
          const int equationIndexes[2] = {1,<%at.index%>};
          <%varDecls2%>
          <% if profileAll() then 'SIM_PROF_TICK_EQ(<%at.index%>);' %>
          <% if profileSome() then 'SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%at.index%>).profileBlockIndex,1);' %>
          <%xlocs2%>
          <%prebody2%>
          <%body2%>
          <% if profileAll() then 'SIM_PROF_ACC_EQ(<%at.index%>);' %>
        }
      >>
  )
  ;separator="\n\n")
end functionNonLinearResiduals;

// =============================================================================
// section for State Sets
//
// This section generates the followng c functions:
//   - void initializeStateSets(int nStateSets, STATE_SET_DATA* statesetData, DATA *data)
// =============================================================================

template functionInitialStateSets(list<StateSet> stateSets, String modelNamePrefix)
  "Generates functions in simulation file to initialize the stateset data."
::=
     let body = (stateSets |> set hasindex i1 fromindex 0 => (match set
       case set as SES_STATESET(__) then
       let generatedJac = match jacobianMatrix case (_,_,name,_,_,_,_) then '<%symbolName(modelNamePrefix,"functionJac")%><%name%>_column'
       let initialJac =  match jacobianMatrix case (_,_,name,_,_,_,_) then '<%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%name%>'
       let jacIndex = match jacobianMatrix case (_,_,_,_,_,_,jacindex) then '<%jacindex%>'
       let statesvars = (states |> s hasindex i2 fromindex 0 => 'statesetData[<%i1%>].states[<%i2%>] = &<%cref(s)%>__varInfo;' ;separator="\n")
       let statescandidatesvars = (statescandidates |> cstate hasindex i2 fromindex 0 => 'statesetData[<%i1%>].statescandidates[<%i2%>] = &<%cref(cstate)%>__varInfo;' ;separator="\n")
       <<
       assertStreamPrint(NULL, nStateSets > <%i1%>, "Internal Error: nStateSets mismatch!");
       statesetData[<%i1%>].nCandidates = <%nCandidates%>;
       statesetData[<%i1%>].nStates = <%nStates%>;
       statesetData[<%i1%>].nDummyStates = <%nCandidates%>-<%nStates%>;
       statesetData[<%i1%>].states = (VAR_INFO**) calloc(<%nStates%>,sizeof(VAR_INFO));
       <%statesvars%>
       statesetData[<%i1%>].statescandidates = (VAR_INFO**) calloc(<%nCandidates%>,sizeof(VAR_INFO));
       <%statescandidatesvars%>
       statesetData[<%i1%>].A = &<%cref(crA)%>__varInfo;
       statesetData[<%i1%>].rowPivot = (modelica_integer*) calloc(<%nCandidates%>-<%nStates%>,sizeof(modelica_integer));
       statesetData[<%i1%>].colPivot = (modelica_integer*) calloc(<%nCandidates%>,sizeof(modelica_integer));
       statesetData[<%i1%>].J = (modelica_real*) calloc(<%nCandidates%>*(<%nCandidates%>-<%nStates%>),sizeof(modelica_real));
       statesetData[<%i1%>].analyticalJacobianColumn = <%generatedJac%>;
       statesetData[<%i1%>].initialAnalyticalJacobian = <%initialJac%>;
       statesetData[<%i1%>].jacobianIndex = <%jacIndex%>;

       >>
   )
   ;separator="\n\n")
  <<
  /* funtion initialize state sets */
  void <%symbolName(modelNamePrefix,"initializeStateSets")%>(int nStateSets, STATE_SET_DATA* statesetData, DATA *data)
  {
    <%body%>
  }
  >>
end functionInitialStateSets;

// =============================================================================
// section for initialization
//
// This section generates the followng c functions:
//   - int updateBoundParameters(DATA *data)
//   - int updateBoundVariableAttributes(DATA *data)
//   - int functionInitialEquations(DATA *data)
//   - int functionInitialEquations_lambda0(DATA *data)
//   - int functionRemovedInitialEquations(DATA *data)
// =============================================================================

template functionUpdateBoundVariableAttributes(list<SimEqSystem> startValueEquations, list<SimEqSystem> nominalValueEquations, list<SimEqSystem> minValueEquations, list<SimEqSystem> maxValueEquations, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let &varDecls = buffer ""
  let &tmp = buffer ""
  let startEqPart = (startValueEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(-1, eq, contextOther, &varDecls, &tmp, modelNamePrefix)
    ;separator="\n")
  let nominalEqPart = (nominalValueEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(-1, eq, contextOther, &varDecls, &tmp, modelNamePrefix)
    ;separator="\n")
  let minEqPart = (minValueEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(-1, eq, contextOther, &varDecls, &tmp, modelNamePrefix)
    ;separator="\n")
  let maxEqPart = (maxValueEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(-1, eq, contextOther, &varDecls, &tmp, modelNamePrefix)
    ;separator="\n")

  <<
  <%&tmp%>
  int <%symbolName(modelNamePrefix,"updateBoundVariableAttributes")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    /* min ******************************************************** */
    <%minEqPart%>

    infoStreamPrint(LOG_INIT, 1, "updating min-values");
    <%minValueEquations |> SES_SIMPLE_ASSIGN(__) =>
      <<
      $P$ATTRIBUTE<%cref(cref)%>.min = <%cref(cref)%>;
        infoStreamPrint(LOG_INIT, 0, "%s(min=<%crefToPrintfArg(cref)%>)", <%cref(cref)%>__varInfo.name, (<%crefType(cref)%>) $P$ATTRIBUTE<%cref(cref)%>.min);
      >>
      ;separator="\n"
    %>
    if (ACTIVE_STREAM(LOG_INIT)) messageClose(LOG_INIT);

    /* max ******************************************************** */
    <%maxEqPart%>

    infoStreamPrint(LOG_INIT, 1, "updating max-values");
    <%maxValueEquations |> SES_SIMPLE_ASSIGN(__) =>
      <<
      $P$ATTRIBUTE<%cref(cref)%>.max = <%cref(cref)%>;
        infoStreamPrint(LOG_INIT, 0, "%s(max=<%crefToPrintfArg(cref)%>)", <%cref(cref)%>__varInfo.name, (<%crefType(cref)%>) $P$ATTRIBUTE<%cref(cref)%>.max);
      >>
      ;separator="\n"
    %>
    if (ACTIVE_STREAM(LOG_INIT)) messageClose(LOG_INIT);

    /* nominal **************************************************** */
    <%nominalEqPart%>

    infoStreamPrint(LOG_INIT, 1, "updating nominal-values");
    <%nominalValueEquations |> SES_SIMPLE_ASSIGN(__) =>
      <<
      $P$ATTRIBUTE<%cref(cref)%>.nominal = <%cref(cref)%>;
        infoStreamPrint(LOG_INIT, 0, "%s(nominal=<%crefToPrintfArg(cref)%>)", <%cref(cref)%>__varInfo.name, (<%crefType(cref)%>) $P$ATTRIBUTE<%cref(cref)%>.nominal);
      >>
      ;separator="\n"
    %>
    if (ACTIVE_STREAM(LOG_INIT)) messageClose(LOG_INIT);

    /* start ****************************************************** */
    <%startEqPart%>

    infoStreamPrint(LOG_INIT, 1, "updating start-values");
    <%startValueEquations |> SES_SIMPLE_ASSIGN(__) =>
      <<
      $P$ATTRIBUTE<%cref(cref)%>.start = <%cref(cref)%>;
        infoStreamPrint(LOG_INIT, 0, "%s(start=<%crefToPrintfArg(cref)%>)", <%cref(cref)%>__varInfo.name, (<%crefType(cref)%>)  $P$ATTRIBUTE<%cref(cref)%>.start);
      >>
      ;separator="\n"
    %>
    if (ACTIVE_STREAM(LOG_INIT)) messageClose(LOG_INIT);

    TRACE_POP
    return 0;
  }
  >>
end functionUpdateBoundVariableAttributes;

template functionUpdateBoundParameters(list<SimEqSystem> parameterEquations, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let () = System.tmpTickReset(0)
  let &varDecls = buffer ""
  let &tmp = buffer ""
  let body = (parameterEquations |> eq  =>
    '<%equation_(-1, eq, contextSimulationDiscrete, &varDecls, &tmp, modelNamePrefix)%>'
    ;separator="\n")

  <<
  <%&tmp%>
  int <%symbolName(modelNamePrefix,"updateBoundParameters")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>
    <%body%>

    TRACE_POP
    return 0;
  }
  >>
end functionUpdateBoundParameters;

template functionInitialEquations(list<SimEqSystem> initalEquations, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let () = System.tmpTickReset(0)
  let &varDecls = buffer ""
  let nrfuncs = listLength(initalEquations)
  let &eqfuncs = buffer ""
  let &eqArray = buffer ""
  let fncalls = if Flags.isSet(Flags.PARMODAUTO) then
                (initalEquations |> eq hasindex i0 =>
                    equation_arrayFormat(eq, "InitialEquations", contextSimulationDiscrete, i0, &varDecls, &eqArray, &eqfuncs, modelNamePrefix)
                    ;separator="\n")
              else
                (initalEquations |> eq hasindex i0 =>
                    equation_(-1, eq, contextSimulationDiscrete, &varDecls, &eqfuncs, modelNamePrefix)
                    ;separator="\n")

  let eqArrayDecl = if Flags.isSet(Flags.PARMODAUTO) then
                <<
                static void (*functionInitialEquations_systems[<%listLength(initalEquations)%>])(DATA *, threadData_t*) = {
                    <%eqArray%>
                };
                >>
              else
                ""

  <<
  <%eqfuncs%>

  <%eqArrayDecl%>

  int <%symbolName(modelNamePrefix,"functionInitialEquations")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    data->simulationInfo->discreteCall = 1;
    <%if Flags.isSet(Flags.PARMODAUTO) then 'PM_functionInitialEquations(<%nrfuncs%>, data, threadData, functionInitialEquations_systems);'
    else '<%fncalls%>' %>
    data->simulationInfo->discreteCall = 0;

    TRACE_POP
    return 0;
  }
  >>
end functionInitialEquations;

template functionInitialEquations_lambda0(list<SimEqSystem> initalEquations_lambda0, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let () = System.tmpTickReset(0)
  let &varDecls = buffer ""
  let nrfuncs = listLength(initalEquations_lambda0)
  let &eqfuncs = buffer ""
  let &eqArray = buffer ""
  let fncalls = if Flags.isSet(Flags.PARMODAUTO) then
                (initalEquations_lambda0 |> eq hasindex i0 =>
                    equation_arrayFormat(eq, "InitialEquations", contextSimulationDiscrete, i0, &varDecls, &eqArray, &eqfuncs, modelNamePrefix)
                    ;separator="\n")
              else
                (initalEquations_lambda0 |> eq hasindex i0 =>
                    equation_(-1, eq, contextSimulationDiscrete, &varDecls, &eqfuncs, modelNamePrefix)
                    ;separator="\n")

  let eqArrayDecl = if Flags.isSet(Flags.PARMODAUTO) then
                <<
                static void (*functionInitialEquations_systems[<%listLength(initalEquations_lambda0)%>])(DATA *, threadData_t*) = {
                    <%eqArray%>
                };
                >>
              else
                ""

  <<
  <%eqfuncs%>

  <%eqArrayDecl%>

  int <%symbolName(modelNamePrefix,"functionInitialEquations_lambda0")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    data->simulationInfo->discreteCall = 1;
    <%if Flags.isSet(Flags.PARMODAUTO) then 'PM_functionInitialEquations_lambda0(<%nrfuncs%>, data, threadData, functionInitialEquations_lambda0_systems);'
    else '<%fncalls%>' %>
    data->simulationInfo->discreteCall = 0;

    TRACE_POP
    return 0;
  }
  >>
end functionInitialEquations_lambda0;

template functionRemovedInitialEquationsBody(SimEqSystem eq, Text &varDecls, Text &eqs, String modelNamePrefix)
 "Generates an equation."
::=
  match eq
  case e as SES_RESIDUAL(__) then
    match exp
    case DAE.SCONST(__) then
      'res = 0;'
    else
      let &preExp = buffer ""
      let expPart = daeExp(exp, contextOther, &preExp, &varDecls, &eqs)
      <<
      <% if profileAll() then 'SIM_PROF_TICK_EQ(<%e.index%>);' %>
      <%preExp%>res = <%expPart%>;
      <% if profileAll() then 'SIM_PROF_ACC_EQ(<%e.index%>);' %>
      if(fabs(res) > 1e-5)
      {
        errorStreamPrint(LOG_INIT, 0, "The initialization problem is inconsistent due to the following equation: 0 != %g = <%ExpressionDump.printExpStr(exp)%>", res);
        return 1;
      }
      >>
    end match
  else
  equation_(-1, eq, contextSimulationDiscrete, &varDecls, &eqs, modelNamePrefix)
  end match
end functionRemovedInitialEquationsBody;

template functionRemovedInitialEquations(list<SimEqSystem> removedInitalEquations, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let &varDecls = buffer ""
  let &tmp = buffer ""

  let body = (removedInitalEquations |> eq2 =>
       functionRemovedInitialEquationsBody(eq2, &varDecls, &tmp, modelNamePrefix)
     ;separator="\n")

  <<
  <%tmp%>
  int <%symbolName(modelNamePrefix,"functionRemovedInitialEquations")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    const int *equationIndexes = NULL;
    double res = 0.0;
    <%varDecls%>

    <%body%>

    TRACE_POP
    return 0;
  }
  >>
end functionRemovedInitialEquations;

template functionStoreDelayed(DelayedExpression delayed, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let &varDecls = buffer ""
  let &auxFunction = buffer ""
  let storePart = (match delayed case DELAYED_EXPRESSIONS(__) then (delayedExps |> (id, (e, d, delayMax)) =>
      let &preExp = buffer ""
      let eRes = daeExp(e, contextSimulationNonDiscrete, &preExp, &varDecls, &auxFunction)
      let delayExp = daeExp(d, contextSimulationNonDiscrete, &preExp, &varDecls, &auxFunction)
      let delayExpMax = daeExp(delayMax, contextSimulationNonDiscrete, &preExp, &varDecls, &auxFunction)
      <<
      <%preExp%>
      storeDelayedExpression(data, threadData, <%id%>, <%eRes%>, data->localData[0]->timeValue, <%delayExp%>, <%delayExpMax%>);<%\n%>
      >>
    ))
  <<
  <%auxFunction%>
  int <%symbolName(modelNamePrefix,"function_storeDelayed")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH

    <%varDecls%>
    <%storePart%>

    TRACE_POP
    return 0;
  }
  >>
end functionStoreDelayed;


//------------------------------------
// Begin: Modified functions for HpcOm
//------------------------------------

template functionXXX_systems_HPCOM(list<list<SimEqSystem>> eqs, String name, Text &loop, Text &varDecls, Option<tuple<Schedule,Schedule,Schedule>> hpcOmSchedulesOpt, String modelNamePrefixStr)
::=
 let funcs = (eqs |> eq hasindex i0 fromindex 0 => functionXXX_system_HPCOM(eq,name,i0,hpcOmSchedulesOpt, modelNamePrefixStr) ; separator="\n")
 match listLength(eqs)
     case 0 then //empty case
       let &loop +=
           <<
           /* no <%name%> systems */
           >>
       <<
       void terminateHpcOmThreads()
       {

       }
       >>
     case 1 then //1 function
       let &loop +=
           <<
           function<%name%>_system0(data, threadData);
           >>
       funcs //just the one function
     case nFuncs then //2 and more
       let funcNames = eqs |> e hasindex i0 fromindex 0 => 'function<%name%>_system<%i0%>' ; separator=",\n"
       let &varDecls += 'int id;<%\n%>'
       let &loop +=
         if Flags.isSet(Flags.PARMODAUTO) then /* Text for the loop body that calls the equations */
         <<
         #pragma omp parallel for private(id) schedule(<%match noProc() case 0 then "dynamic" else "static"%>)
         for(id=0; id<<%nFuncs%>; id++) {
           function<%name%>_systems[id](data, threadData);
         }
         >>
         else
         <<
         for(id=0; id<<%nFuncs%>; id++) {
           function<%name%>_systems[id](data, threadData);
         }
         >>
       /* Text before the function head */
       <<
       <%funcs%>
       static void (*function<%name%>_systems[<%nFuncs%>])(DATA *, threadData_t *) = {
         <%funcNames%>
       };
       >>
end functionXXX_systems_HPCOM;

template functionXXX_system_HPCOM(list<SimEqSystem> derivativEquations, String name, Integer n, Option<tuple<Schedule,Schedule,Schedule>> hpcOmSchedulesOpt, String modelNamePrefixStr)
::=
  let type = getConfigString(HPCOM_CODE)
  match hpcOmSchedulesOpt
    case SOME((hpcOmSchedule as EMPTYSCHEDULE(__),_,_)) then
      <<
      void terminateHpcOmThreads()
      {
      }

      <%functionXXX_system(derivativEquations,name,n,modelNamePrefixStr)%>
      >>
    case SOME((hpcOmSchedule as TASKDEPSCHEDULE(__),_,_)) then
      let taskEqs = functionXXX_system0_HPCOM_TaskDep(hpcOmSchedule.tasks, derivativEquations, type, name, modelNamePrefixStr); separator="\n"
      <<
      void terminateHpcOmThreads()
      {
      }

      //using type: <%type%>
      void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
      {
        <%taskEqs%>
      }
      >>
    case SOME((hpcOmSchedule as LEVELSCHEDULE(__),_,_)) then
      let odeEqs = hpcOmSchedule.tasksOfLevels |> tasks => functionXXX_system0_HPCOM_Level(derivativEquations,name,tasks,type,modelNamePrefixStr); separator="\n"
      <<
      void terminateHpcOmThreads()
      {
      }

      //using type: <%type%>
      void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
      {
        int fail=0;
        omp_set_dynamic(0);
        #pragma omp parallel num_threads(<%getConfigInt(NUM_PROC)%>)
        {
            MMC_TRY_TOP()
            <%odeEqs%>
            MMC_CATCH_TOP(fail=1)
        }
        if (fail) {
          MMC_THROW_INTERNAL()
        }
      }
      >>
   case SOME((hpcOmSchedule as THREADSCHEDULE(__),_,_)) then
      let locks = hpcOmSchedule.outgoingDepTasks |> task => function_HPCOM_createLockByDepTask(task, "lock", type); separator="\n"
      let initlocks = hpcOmSchedule.outgoingDepTasks |> task => function_HPCOM_initializeLockByDepTask(task, "lock", type); separator="\n"
      let assignLocks = hpcOmSchedule.outgoingDepTasks |> task => function_HPCOM_assignLockByDepTask(task, "lock", type); separator="\n"
      match type
        case ("openmp") then
          let taskEqs = functionXXX_system0_HPCOM_Thread(derivativEquations,name,hpcOmSchedule.threadTasks, type, modelNamePrefixStr); separator="\n"
          <<
          void terminateHpcOmThreads()
          {
          }

          //using type: <%type%>
          static int initialized = 0;
          void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
          {
            omp_set_dynamic(0);
            //create locks
            <%locks%>
            if(!initialized)
            {
                <%initlocks%>

                //set locks
                <%assignLocks%>

                initialized = 1;
            }

            <%taskEqs%>
          }
          >>
        case ("pthreads") then
          let threadDecl = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_decl(i0); separator="\n"
          let threadFuncs = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_func(derivativEquations, name, n, hpcOmSchedule.threadTasks, type, i0, modelNamePrefixStr); separator="\n"
          let threadFuncCalls = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_call(name, n, i0); separator="\n"
          let threadStart = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_start(i0); separator="\n"

          let threadLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_createLockByLockName(i0, "th_lock", type); separator="\n"
          let threadLocks1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_createLockByLockName(i0, "th_lock1", type); separator="\n"
          let threadLocksInit = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_initializeLockByLockName(i0, "th_lock", type); separator="\n"
          let threadLocksInit1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_initializeLockByLockName(i0, "th_lock1", type); separator="\n"
          let threadAssignLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_assignLockByLockName(i0, "th_lock", type); separator="\n"
          let threadAssignLocks1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_assignLockByLockName(i0, "th_lock1", type); separator="\n"
          let threadReleaseLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_releaseLockByLockName(i0, "th_lock", type); separator="\n"

          <<
          // number of threads: <%arrayLength(hpcOmSchedule.threadTasks)%>
          static int finished; //set to 1 if the hpcom-threads should be destroyed

          <%threadDecl%>

          <%locks%>

          <%threadLocks%>
          <%threadLocks1%>

          void terminateHpcOmThreads()
          {
            finished = 1;

            //Start the threads one last time
            <%threadReleaseLocks%>
          }

          <%threadFuncs%>

          //using type: <%type%>
          static int initialized = 0;
          void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
          {
            if(!initialized)
            {
                <%initlocks%>
                <%threadLocksInit%>
                <%threadLocksInit1%>

                //set locks
                <%assignLocks%>

                <%threadAssignLocks%>
                <%threadAssignLocks1%>

                <%threadFuncCalls%>
                initialized = 1;
            }

            //Start the threads
            <%threadReleaseLocks%>

            //"join"
            <%threadAssignLocks1%>
          }
          >>
        case ("pthreads_spin") then
          let threadDecl = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_decl(i0); separator="\n"
          let threadFuncs = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_func(derivativEquations, name, n, hpcOmSchedule.threadTasks, type, i0, modelNamePrefixStr); separator="\n"
          let threadFuncCalls = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_call(name, n, i0); separator="\n"
          let threadStart = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => functionXXX_system0_HPCOM_PThread_start(i0); separator="\n"

          let threadLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_createLockByLockName(i0, "th_lock", "pthreads"); separator="\n"
          let threadLocks1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_createLockByLockName(i0, "th_lock1", "pthreads"); separator="\n"
          let threadLocksInit = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_initializeLockByLockName(i0, "th_lock", "pthreads"); separator="\n"
          let threadLocksInit1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_initializeLockByLockName(i0, "th_lock1", "pthreads"); separator="\n"
          let threadAssignLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_assignLockByLockName(i0, "th_lock", "pthreads"); separator="\n"
          let threadAssignLocks1 = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_assignLockByLockName(i0, "th_lock1", "pthreads"); separator="\n"
          let threadReleaseLocks = arrayList(hpcOmSchedule.threadTasks) |> tt hasindex i0 fromindex 0 => function_HPCOM_releaseLockByLockName(i0, "th_lock", "pthreads"); separator="\n"

          <<
          static int finished; //set to 1 if the hpcom-threads should be destroyed

          <%threadDecl%>

          <%locks%>

          <%threadLocks%>
          <%threadLocks1%>

          void terminateHpcOmThreads()
          {
            finished = 1;

            //Start the threads one last time
            <%threadReleaseLocks%>
          }

          <%threadFuncs%>

          //using type: <%type%>
          static int initialized = 0;
          void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
          {
            if(!initialized)
            {
                finished = 0;

                <%initlocks%>
                <%threadLocksInit%>
                <%threadLocksInit1%>

                //set locks
                <%assignLocks%>

                <%threadAssignLocks%>
                <%threadAssignLocks1%>

                <%threadFuncCalls%>
                initialized = 1;
            }

            //Start the threads
            <%threadReleaseLocks%>

            //"join"
            <%threadAssignLocks1%>
          }
          >>

end functionXXX_system_HPCOM;

template functionXXX_system0_HPCOM_Level(list<SimEqSystem> derivativEquations, String name, TaskList tasksOfLevel, String iType, String modelNamePrefixStr)
::=
  match(tasksOfLevel)
    case(PARALLELTASKLIST(__)) then
      let odeEqs = tasks |> task => functionXXX_system0_HPCOM_Level0Section(derivativEquations,name,task,iType,modelNamePrefixStr); separator="\n"
      <<
      #pragma omp sections
      {
         <%odeEqs%>
      }
      >>
    case(SERIALTASKLIST(__)) then
      let odeEqs = tasks |> task => functionXXX_system0_HPCOM_Level0(derivativEquations,name,task,iType,modelNamePrefixStr); separator="\n"
      <<
      #pragma omp master
      {
         <%odeEqs%>
      }
      #pragma omp barrier
      >>
    else
      <<

      >>
end functionXXX_system0_HPCOM_Level;

template functionXXX_system0_HPCOM_Level0(list<SimEqSystem> derivativEquations, String name, Task iTask, String iType, String modelNamePrefixStr)
::=
  <<
    <%function_HPCOM_Task(derivativEquations,name,iTask,iType,modelNamePrefixStr)%>
  >>
end functionXXX_system0_HPCOM_Level0;

template functionXXX_system0_HPCOM_Level0Section(list<SimEqSystem> derivativEquations, String name, Task iTask, String iType, String modelNamePrefixStr)
::=
  <<
  #pragma omp section
  {
    int fail=0;
    MMC_TRY_TOP()
    <%function_HPCOM_Task(derivativEquations,name,iTask,iType,modelNamePrefixStr)%>
    MMC_CATCH_TOP(fail=1)
    if (fail) {
      MMC_THROW_INTERNAL()
    }
  }
  >>
end functionXXX_system0_HPCOM_Level0Section;

template functionXXX_system0_HPCOM_TaskDep(list<tuple<Task,list<Integer>>> tasks, list<SimEqSystem> derivativEquations, String iType, String name, String modelNamePrefixStr)
::=
  let odeEqs = tasks |> t => functionXXX_system0_HPCOM_TaskDep0(t,derivativEquations, iType, name, modelNamePrefixStr); separator="\n"
  <<

  int t[0];
  #pragma omp parallel
  {
    #pragma omp master
    {
        <%odeEqs%>
    }
  }
  >>
end functionXXX_system0_HPCOM_TaskDep;

template functionXXX_system0_HPCOM_TaskDep0(tuple<Task,list<Integer>> taskIn, list<SimEqSystem> derivativEquations, String iType, String name, String modelNamePrefixStr)
::=
  match taskIn
    case ((task as CALCTASK(__),parents)) then
        let taskEqs = function_HPCOM_Task(derivativEquations, name, task, iType, modelNamePrefixStr); separator="\n"
        let parentDependencies = parents |> p => 't[<%p%>]'; separator = ","
        let depIn = if intGt(listLength(parents),0) then 'depend(in:<%parentDependencies%>)' else ""
        <<
        #pragma omp task <%depIn%> depend(out:t[<%task.index%>])
        {
            <%taskEqs%>
        }
        >>
end functionXXX_system0_HPCOM_TaskDep0;

template functionXXX_system0_HPCOM_Thread(list<SimEqSystem> derivativEquations, String name, array<list<Task>> threadTasks, String iType, String modelNamePrefixStr)
::=
  let odeEqs = arrayList(threadTasks) |> tt => functionXXX_system0_HPCOM_Thread0(derivativEquations,name,tt,iType,modelNamePrefixStr); separator="\n"
  match iType
    case ("openmp") then
      <<
      if (omp_get_dynamic())
        omp_set_dynamic(0);
      #pragma omp parallel sections num_threads(<%arrayLength(threadTasks)%>)
      {
         <%odeEqs%>
      }
      >>
    case ("pthreads") then
      <<
      //not implemented
      >>
    case ("pthreads_spin") then
      <<
      //not implemented
      >>

end functionXXX_system0_HPCOM_Thread;

template functionXXX_system0_HPCOM_Thread0(list<SimEqSystem> derivativEquations, String name, list<Task> threadTaskList, String iType, String modelNamePrefixStr)
::=
  let threadTasks = threadTaskList |> tt => function_HPCOM_Task(derivativEquations,name,tt,iType,modelNamePrefixStr); separator="\n"
  match iType
    case ("openmp") then
      <<
      #pragma omp section
      {
        int fail=0;
        MMC_TRY_TOP()
        <%threadTasks%>
        MMC_CATCH_TOP(fail=1)
        if (fail) {
          MMC_THROW_INTERNAL()
        }
      }
      >>
    case ("pthreads") then
      <<
      <%threadTasks%>
      >>
    case ("pthreads_spin") then
      <<
      <%threadTasks%>
      >>
end functionXXX_system0_HPCOM_Thread0;

template function_HPCOM_Task(list<SimEqSystem> derivativEquations, String name, Task iTask, String iType, String modelNamePrefixStr)
::=
  match iTask
    case (task as CALCTASK(__)) then
      let odeEqs = task.eqIdc |> eq => equationNamesHPCOM_Thread_(eq,derivativEquations,contextSimulationNonDiscrete,modelNamePrefixStr); separator="\n"
      <<
      // Task <%task.index%>
      <%odeEqs%>
      // End Task <%task.index%>
      >>
    case (task as CALCTASK_LEVEL(__)) then
      let odeEqs = task.eqIdc |> eq => equationNamesHPCOM_Thread_(eq,derivativEquations,contextSimulationNonDiscrete,modelNamePrefixStr); separator="\n"
      <<
      <%odeEqs%>
      >>
    case(task as DEPTASK(outgoing=false)) then
      let assLck = function_HPCOM_assignLockByDepTask(task, "lock", iType); separator="\n"
      <<
      <%assLck%>
      >>
    case(task as DEPTASK(outgoing=true)) then
      let relLck = function_HPCOM_releaseLockByDepTask(task, "lock", iType); separator="\n"
      <<
      <%relLck%>
      >>
end function_HPCOM_Task;

template function_HPCOM_getLockNameByDepTask(Task depTask)
::=
  match(depTask)
    case(DEPTASK(sourceTask=CALCTASK(index=sourceIdx), targetTask=CALCTASK(index=targetIdx))) then
      '<%sourceIdx%>_<%targetIdx%>'
    else
      'invalidLockTask'
  end match
end function_HPCOM_getLockNameByDepTask;

template function_HPCOM_initializeLockByDepTask(Task depTask, String lockPrefix, String iType)
::=
  let lockName = function_HPCOM_getLockNameByDepTask(depTask)
  <<
  <%function_HPCOM_initializeLockByLockName(lockName, lockPrefix, iType)%>
  >>
end function_HPCOM_initializeLockByDepTask;

template function_HPCOM_initializeLockByLockName(String lockName, String lockPrefix, String iType)
::=
  match iType
    case ("openmp") then
      <<
      omp_init_lock(&<%lockPrefix%>_<%lockName%>);
      >>
    case ("pthreads") then
      <<
      pthread_mutex_init(&<%lockPrefix%>_<%lockName%>, NULL);
      >>
    case ("pthreads_spin") then
      <<
      pthread_spin_init(&<%lockPrefix%>_<%lockName%>, 0);
      >>
end function_HPCOM_initializeLockByLockName;

template function_HPCOM_createLockByDepTask(Task depTask, String lockPrefix, String iType)
::=
  let lockName = function_HPCOM_getLockNameByDepTask(depTask)
  <<
  <%function_HPCOM_createLockByLockName(lockName, lockPrefix, iType)%>
  >>
end function_HPCOM_createLockByDepTask;

template function_HPCOM_createLockByLockName(String lockName, String lockPrefix, String iType)
::=
  match iType
    case ("openmp") then
      <<
      static omp_lock_t <%lockPrefix%>_<%lockName%>;
      >>
    case ("pthreads") then
      <<
      static pthread_mutex_t <%lockPrefix%>_<%lockName%>;
      >>
    case ("pthreads_spin") then
      <<
      static pthread_spinlock_t <%lockPrefix%>_<%lockName%>;
      >>
end function_HPCOM_createLockByLockName;

template function_HPCOM_assignLockByDepTask(Task depTask, String lockPrefix, String iType)
::=
  let lockName = function_HPCOM_getLockNameByDepTask(depTask)
  <<
  <%function_HPCOM_assignLockByLockName(lockName, lockPrefix, iType)%>
  >>
end function_HPCOM_assignLockByDepTask;

template function_HPCOM_assignLockByLockName(String lockName, String lockPrefix, String iType)
::=
  match iType
    case ("openmp") then
      <<
      omp_set_lock(&<%lockPrefix%>_<%lockName%>);
      >>
    case ("pthreads") then
      <<
      pthread_mutex_lock(&<%lockPrefix%>_<%lockName%>);
      >>
    case ("pthreads_spin") then
      <<
      pthread_spin_lock(&<%lockPrefix%>_<%lockName%>);
      >>
end function_HPCOM_assignLockByLockName;

template function_HPCOM_releaseLockByDepTask(Task depTask, String lockPrefix, String iType)
::=
  let lockName = function_HPCOM_getLockNameByDepTask(depTask)
  <<
  <%function_HPCOM_releaseLockByLockName(lockName, lockPrefix, iType)%>
  >>
end function_HPCOM_releaseLockByDepTask;

template function_HPCOM_releaseLockByLockName(String lockName, String lockPrefix, String iType)
::=
  match iType
    case ("openmp") then
      <<
      omp_unset_lock(&<%lockPrefix%>_<%lockName%>);
      >>
    case ("pthreads") then
      <<
      pthread_mutex_unlock(&<%lockPrefix%>_<%lockName%>);
      >>
    case ("pthreads_spin") then
      <<
      pthread_spin_unlock(&<%lockPrefix%>_<%lockName%>);
      >>
end function_HPCOM_releaseLockByLockName;

template functionXXX_system0_HPCOM_PThread_func(list<SimEqSystem> derivativEquations, String name, Integer n, array<list<Task>> threadTasks, String iType, Integer idx, String modelNamePrefixStr)
::=
  let taskEqs = functionXXX_system0_HPCOM_Thread0(derivativEquations, name, arrayGet(threadTasks,intAdd(idx,1)), iType, modelNamePrefixStr); separator="\n"
  let assLock = function_HPCOM_assignLockByLockName(idx, "th_lock", "pthreads"); separator="\n"
  let relLock = function_HPCOM_releaseLockByLockName(idx, "th_lock1", "pthreads"); separator="\n"
  <<
  void function<%name%>_system<%n%>_thread_<%idx%>(DATA *data, threadData_t *threadData)
  {
    MMC_TRY_TOP()
    while(1)
    {
      <%assLock%>

      if(finished)
         return;

      <%taskEqs%>
      <%relLock%>
    }
    MMC_CATCH_TOP(return;) /* No exit status?? */
  }
  >>
end functionXXX_system0_HPCOM_PThread_func;

template functionXXX_system0_HPCOM_PThread_call(String name, Integer n, Integer idx)
::=
  <<
  GC_pthread_create(&odeThread_<%idx%>, NULL, function<%name%>_system<%n%>_thread_<%idx%>, data);
  >>
end functionXXX_system0_HPCOM_PThread_call;

template functionXXX_system0_HPCOM_PThread_decl(Integer idx)
::=
  <<
  static pthread_t odeThread_<%idx%>;
  >>
end functionXXX_system0_HPCOM_PThread_decl;

template functionXXX_system0_HPCOM_PThread_start(Integer idx)
::=
  <<
  pthread_mutex_lock(&th_unlock_<%idx%>);
  >>
end functionXXX_system0_HPCOM_PThread_start;

template equationNamesHPCOM_Thread_(Integer idx, list<SimEqSystem> derivativEquations, Context context, String modelNamePrefixStr)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
match context
case SIMULATION_CONTEXT(genDiscrete=true) then
 match getSimCodeEqByIndex(derivativEquations, idx)
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = equationIndex(getSimCodeEqByIndex(derivativEquations, idx))
  <<
  <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(data, threadData);
  >>
else
 match getSimCodeEqByIndex(derivativEquations, idx)
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = equationIndex(getSimCodeEqByIndex(derivativEquations, idx))
  <<
  <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(data, threadData);
  >>
end equationNamesHPCOM_Thread_;

//----------------------------------
// End: Modified functions for HpcOm
//----------------------------------

template functionXXX_system(list<SimEqSystem> derivativEquations, String name, Integer n, String modelNamePrefixStr)
::=
  let odeEqs = derivativEquations |> eq => equationNames_(eq,contextSimulationNonDiscrete,modelNamePrefixStr); separator="\n"
  let forwardEqs = derivativEquations |> eq => equationForward_(eq,contextSimulationNonDiscrete,modelNamePrefixStr); separator="\n"
  <<

  /* forwarded equations */
  <%forwardEqs%>

  static void function<%name%>_system<%n%>(DATA *data, threadData_t *threadData)
  {
    <%odeEqs%>
  }
  >>
end functionXXX_system;

template functionXXX_systems(list<list<SimEqSystem>> eqs, String name, Text &loop, Text &varDecls, String modelNamePrefixStr)
::=
  let funcs = (eqs |> eq hasindex i0 fromindex 0 => functionXXX_system(eq,name,i0,modelNamePrefixStr) ; separator="\n")
  match listLength(eqs)
  case 0 then //empty case
    let &loop +=
        <<
        /* no <%name%> systems */
        >>
    ""
  case 1 then //1 function
    let &loop +=
        <<
        function<%name%>_system0(data, threadData);
        >>
    funcs //just the one function
  case nFuncs then //2 and more
    let funcNames = eqs |> e hasindex i0 fromindex 0 => 'function<%name%>_system<%i0%>' ; separator=",\n"
    let head = if Flags.isSet(Flags.PARMODAUTO) then '#pragma omp parallel for private(id) schedule(<%match noProc() case 0 then "dynamic" else "static"%>)'
    let &varDecls += 'int id;<%\n%>'

    let &loop +=
      /* Text for the loop body that calls the equations */
      <<
      <%head%>
      for(id=0; id<<%nFuncs%>; id++) {
        function<%name%>_systems[id](data, threadData);
      }
      >>
    /* Text before the function head */
    <<
    <%funcs%>
    static void (*function<%name%>_systems[<%nFuncs%>])(DATA *, threadData_t *threadData) = {
      <%funcNames%>
    };
    >>
end functionXXX_systems;


template equationNamesArrayFormat(SimEqSystem eq, Context context, String name, Integer arrayIndex, Text &arrayEqs, Text &forwardEqs, String modelNamePrefixStr)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
match context
case SIMULATION_CONTEXT(genDiscrete=true) then
 match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = equationIndex(eq)
  let &arrayEqs += '<%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>,<%\n%>'
  let &forwardEqs += 'extern void <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(DATA* data, threadData_t *threadData);<%\n%>'
  <<
  <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ix%>);' %>
  function<%name%>_systems[<%arrayIndex%>](data);
  <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ix%>);' %>
  >>
else
 match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = equationIndex(eq)
  let &arrayEqs += '<%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>,<%\n%>'
  let &forwardEqs += 'extern void <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(DATA* data, threadData_t *threadData);<%\n%>'
  <<
  <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ix%>);' %>
  // <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(data, threadData);
  function<%name%>_systems[<%arrayIndex%>](data);
  <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ix%>);' %>
  >>
end equationNamesArrayFormat;

template functionXXX_systems_arrayFormat(list<list<SimEqSystem>> eqlstlst, String name, Text &fncalls, Text &nrfuncs, Text &varDecls, String modelNamePrefixStr)
::=
match eqlstlst
  case ({})
  case ({{}}) then
  let &nrfuncs += "0"
  <<
  /* no <%name%> systems */
  static void (**function<%name%>_systems)(DATA *) = NULL;
  >>

  case eqlstlst as ({eqlst}) then
    let &nrfuncs += listLength(eqlst)
    let &arrayEqs = buffer ""
    let &forwardEqs = buffer ""
    let &fncalls += (eqlst |> eq hasindex i0 => equationNamesArrayFormat(eq,contextSimulationNonDiscrete,name,i0,arrayEqs,forwardEqs,modelNamePrefixStr); separator="\n")
    <<
    /* forwarded equations */
    <%forwardEqs%>

    static void (*function<%name%>_systems[<%nrfuncs%>])(DATA *,  threadData_t *) = {
      <%arrayEqs%>
    };

    >>

  case eqlstlst as ({eqlst::_}) then
  error(sourceInfo(), 'TODO more than ODE list in <%name%> systems')
end functionXXX_systems_arrayFormat;

template functionODE(list<list<SimEqSystem>> derivativEquations, Text method, Option<tuple<Schedule,Schedule,Schedule>> hpcOmSchedules, String modelNamePrefix)
 "Generates function in simulation file."
::=
  let () = System.tmpTickReset(0)
  let &nrfuncs = buffer ""
  let &varDecls2 = buffer ""
  let &varDecls = buffer ""
  let &fncalls = buffer ""
  let systems = if Flags.isSet(Flags.HPCOM) then
                    (functionXXX_systems_HPCOM(derivativEquations, "ODE", &fncalls, &varDecls, hpcOmSchedules, modelNamePrefix))
                else if Flags.isSet(Flags.PARMODAUTO) then
                    (functionXXX_systems_arrayFormat(derivativEquations, "ODE", &fncalls, &nrfuncs, &varDecls, modelNamePrefix))
                else
                    (functionXXX_systems(derivativEquations, "ODE", &fncalls, &varDecls, modelNamePrefix))
  /* let systems = functionXXX_systems(derivativEquations, "ODE", &fncalls, &varDecls) */
  let &tmp = buffer ""
  <<
  <%tmp%>
  <%systems%>

  int <%symbolName(modelNamePrefix,"functionODE")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <% if profileFunctions() then "rt_tick(SIM_TIMER_FUNCTION_ODE);" %>

    <%varDecls%>

    data->simulationInfo->callStatistics.functionODE++;

    <%if Flags.isSet(Flags.PARMODAUTO) then 'PM_functionODE(<%nrfuncs%>, data, threadData, functionODE_systems);'
    else '<%fncalls%>' %>

    <% if profileFunctions() then "rt_accumulate(SIM_TIMER_FUNCTION_ODE);" %>

    TRACE_POP
    return 0;
  }
  >>
end functionODE;

template functionAlgebraic(list<list<SimEqSystem>> algebraicEquations, String modelNamePrefix)
  "Generates function in simulation file."
::=
  let &varDecls = buffer ""
  let &nrfuncs = buffer ""
  let &fncalls = buffer ""
  let systems = if Flags.isSet(Flags.PARMODAUTO) then
                    (functionXXX_systems_arrayFormat(algebraicEquations, "Alg", &fncalls, &nrfuncs, &varDecls, modelNamePrefix))
                else
                    (functionXXX_systems(algebraicEquations, "Alg", &fncalls, &varDecls, modelNamePrefix))


  <<
  <%systems%>
  /* for continuous time variables */
  int <%symbolName(modelNamePrefix,"functionAlgebraics")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    <%if Flags.isSet(Flags.PARMODAUTO) then 'PM_functionAlg(<%nrfuncs%>, data, threadData, functionAlg_systems);'
    else '<%fncalls%>' %>

    <%symbolName(modelNamePrefix,"function_savePreSynchronous")%>(data, threadData);

    TRACE_POP
    return 0;
  }
  >>
end functionAlgebraic;

template functionDAE(list<SimEqSystem> allEquationsPlusWhen, String modelNamePrefix)
  "Generates function in simulation file.
  This is a helper of template simulationFile."
::=
  let &auxFunction = buffer ""
  let &varDecls = buffer ""
  let nrfuncs = listLength(allEquationsPlusWhen)
  let &eqfuncs = buffer ""
  let &eqArray = buffer ""
  let fncalls = if Flags.isSet(Flags.PARMODAUTO) then
                (allEquationsPlusWhen |> eq hasindex i0 =>
                    equation_arrayFormat(eq, "DAE", contextSimulationDiscrete, i0, &varDecls, &eqArray, &eqfuncs, modelNamePrefix)
                    ;separator="\n")
              else
                (allEquationsPlusWhen |> eq hasindex i0 =>
                    equation_(-1, eq, contextSimulationDiscrete, &varDecls, &eqfuncs, modelNamePrefix)
                    ;separator="\n")

  let eqArrayDecl = if Flags.isSet(Flags.PARMODAUTO) then
                <<
                static void (*functionDAE_systems[<%nrfuncs%>])(DATA *, threadData_t *) = {
                    <%eqArray%>
                };
                >>
              else
                ""


  <<
  <%auxFunction%>
  <%&eqfuncs%>

  <%eqArrayDecl%>

  int <%symbolName(modelNamePrefix,"functionDAE")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    int equationIndexes[1] = {0};<%/*reinits may use equation indexes, even though it has no equation...*/%>
    <%addRootsTempArray()%>
    <%varDecls%>

    data->simulationInfo->needToIterate = 0;
    data->simulationInfo->discreteCall = 1;
    <%if Flags.isSet(Flags.PARMODAUTO) then 'PM_functionDAE(<%nrfuncs%>, data, threadData, functionDAE_systems);'
    else '<%fncalls%>' %>
    data->simulationInfo->discreteCall = 0;

    TRACE_POP
    return 0;
  }
  >>
end functionDAE;

template functionZeroCrossing(list<ZeroCrossing> zeroCrossings, list<SimEqSystem> equationsForZeroCrossings, String modelNamePrefix)
"template functionZeroCrossing
  Generates function for ZeroCrossings in simulation file.
  This is a helper of template simulationFile."
::=
  let &varDecls = buffer ""
  let &tmp = buffer ""
  let &auxFunction = buffer ""
  let eqs = (equationsForZeroCrossings |> eq =>
       equation_(-1, eq, contextSimulationNonDiscrete, &varDecls, &tmp, modelNamePrefix)
      ;separator="\n")
  let forwardEqs = equationsForZeroCrossings |> eq => equationForward_(eq,contextSimulationNonDiscrete,modelNamePrefix); separator="\n"

  let &varDecls2 = buffer ""
  let zeroCrossingsCode = zeroCrossingsTpl(zeroCrossings, &varDecls2, &auxFunction)

  let resDesc = (zeroCrossings |> ZERO_CROSSING(__) => '"<%Util.escapeModelicaStringToCString(ExpressionDump.printExpStr(relation_))%>"'
    ;separator=",\n")

  let desc = match zeroCrossings
             case {} then
               <<
               const char *<%symbolName(modelNamePrefix,"zeroCrossingDescription")%>(int i, int **out_EquationIndexes)
               {
                 *out_EquationIndexes = NULL;
                 return "empty";
               }
               >>
             else
               <<
               const char *<%symbolName(modelNamePrefix,"zeroCrossingDescription")%>(int i, int **out_EquationIndexes)
               {
                 static const char *res[] = {<%resDesc%>};
                 <%zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
                   'static const int occurEqs<%i0%>[] = {<%listLength(occurEquLst)%><%occurEquLst |> i => ',<%i%>'%>};' ; separator = "\n"%>
                 static const int *occurEqs[] = {<%zeroCrossings |> ZERO_CROSSING(__) hasindex i0 => 'occurEqs<%i0%>' ; separator = ","%>};
                 *out_EquationIndexes = (int*) occurEqs[i];
                 return res[i];
               }
               >>

  <<
  <%desc%>

  /* forwarded equations */
  <%forwardEqs%>

  int <%symbolName(modelNamePrefix,"function_ZeroCrossingsEquations")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    data->simulationInfo->callStatistics.functionZeroCrossingsEquations++;

    <%eqs%>

    TRACE_POP
    return 0;
  }

  int <%symbolName(modelNamePrefix,"function_ZeroCrossings")%>(DATA *data, threadData_t *threadData, double *gout)
  {
    TRACE_PUSH
    <%varDecls2%>

    data->simulationInfo->callStatistics.functionZeroCrossings++;

    <%zeroCrossingsCode%>

    TRACE_POP
    return 0;
  }
  >>
end functionZeroCrossing;

template zeroCrossingsTpl(list<ZeroCrossing> zeroCrossings, Text &varDecls, Text &auxFunction)
 "Generates code for zero crossings."
::=
  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    zeroCrossingTpl(i0, relation_, &varDecls, &auxFunction)
  ;separator="\n";empty)
end zeroCrossingsTpl;


template zeroCrossingTpl(Integer index1, Exp relation, Text &varDecls, Text &auxFunction)
 "Generates code for a zero crossing."
::=
  match relation
  case exp as RELATION(__) then
    let &preExp = buffer ""
    let e1 = daeExp(exp, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (<%e1%>) ? 1 : -1;
    >>
  case (exp1 as LBINARY(__)) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (<%e1%>) ? 1 : -1;
    >>
  case (exp1 as LUNARY(__)) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (<%e1%>) ? 1 : -1;
    >>
  case CALL(path=IDENT(name="sample"), expLst={_, start, interval}) then
    << >>
  case CALL(path=IDENT(name="integer"), expLst={exp1, idx}) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    let indx = daeExp(idx, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (floor(<%e1%>) != floor(data->simulationInfo->mathEventsValuePre[<%indx%>])) ? 1 : -1;
    >>
  case CALL(path=IDENT(name="floor"), expLst={exp1, idx}) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    let indx = daeExp(idx, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (floor(<%e1%>) != floor(data->simulationInfo->mathEventsValuePre[<%indx%>])) ? 1 : -1;
    >>
  case CALL(path=IDENT(name="ceil"), expLst={exp1, idx}) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    let indx = daeExp(idx, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (ceil(<%e1%>) != ceil(data->simulationInfo->mathEventsValuePre[<%indx%>])) ? 1 : -1;
    >>
  case CALL(path=IDENT(name="div"), expLst={exp1, exp2, idx}) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextZeroCross, &preExp, &varDecls, &auxFunction)
    let e2 = daeExp(exp2, contextZeroCross, &preExp, &varDecls, &auxFunction)
    let indx = daeExp(idx, contextZeroCross, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    gout[<%index1%>] = (trunc(<%e1%>/<%e2%>) != trunc(data->simulationInfo->mathEventsValuePre[<%indx%>]/data->simulationInfo->mathEventsValuePre[<%indx%>+1])) ? 1 : -1;
    >>
  else
    error(sourceInfo(), ' UNKNOWN ZERO CROSSING for <%index1%>')
end zeroCrossingTpl;

template functionRelations(list<ZeroCrossing> relations, String modelNamePrefix) "template functionRelations
  Generates function in simulation file.
  This is a helper of template simulationFile."
::=
  let &auxFunction = buffer ""
  let &varDecls = buffer ""
  let relationsCode = relationsTpl(relations, contextZeroCross, &varDecls, &auxFunction)
  let relationsCodeElse = relationsTpl(relations, contextOther, &varDecls, &auxFunction)

  let resDesc = (relations |> ZERO_CROSSING(__) => '"<%ExpressionDump.printExpStr(relation_)%>"'
    ;separator=",\n")

  let desc = match relations
             case {} then
               <<
               const char *<%symbolName(modelNamePrefix,"relationDescription")%>(int i)
               {
                 return "empty";
               }
               >>
             else
               <<
               const char *<%symbolName(modelNamePrefix,"relationDescription")%>(int i)
               {
                 const char *res[] = {<%resDesc%>};
                 return res[i];
               }
               >>

  <<
  <%auxFunction%>
  <%desc%>

  int <%symbolName(modelNamePrefix,"function_updateRelations")%>(DATA *data, threadData_t *threadData, int evalforZeroCross)
  {
    TRACE_PUSH
    <%varDecls%>

    if(evalforZeroCross) {
      <%relationsCode%>
    } else {
      <%relationsCodeElse%>
    }

    TRACE_POP
    return 0;
  }
  >>
end functionRelations;

template relationsTpl(list<ZeroCrossing> relations, Context context, Text &varDecls, Text &auxFunction)
 "Generates code for zero crossings."
::=
  (relations |> ZERO_CROSSING(__) hasindex i0 =>
    relationTpl(i0, relation_, context, &varDecls, &auxFunction)
  ;separator="\n";empty)
end relationsTpl;


template relationTpl(Integer index1, Exp relation, Context context, Text &varDecls, Text &auxFunction)
 "Generates code for a zero crossing."
::=
  match relation
  case exp as RELATION(__) then
    let &preExp = buffer ""
    let res = daeExp(exp, context, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    data->simulationInfo->relations[<%index1%>] = <%res%>;
    >>
  else
    <<
    /* UNKNOWN Relation for <%index1%> */
    >>
end relationTpl;

template functionCheckForDiscreteChanges(list<ComponentRef> discreteModelVars, String modelNamePrefix) "template functionCheckForDiscreteChanges
  Generates function in simulation file.
  This is a helper of template simulationFile."
::=
  let changediscreteVars = (discreteModelVars |> var =>
    match var
    case CREF_QUAL(__)
    case CREF_IDENT(__) then
      <<
      if(<%cref(var)%> != $P$PRE<%cref(var)%>)
      {
        infoStreamPrint(LOG_EVENTS_V, 0, "discrete var changed: <%crefStr(var)%> from <%crefToPrintfArg(var)%> to <%crefToPrintfArg(var)%>", $P$PRE<%cref(var)%>, <%cref(var)%>);
        needToIterate = 1;
      }
      >>
      ;separator="\n")

  <<
  int <%symbolName(modelNamePrefix, "checkForDiscreteChanges")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    int needToIterate = 0;

    infoStreamPrint(LOG_EVENTS_V, 1, "check for discrete changes at time=%.12g", data->localData[0]->timeValue);
    <%changediscreteVars%>
    if (ACTIVE_STREAM(LOG_EVENTS_V)) messageClose(LOG_EVENTS_V);

    TRACE_POP
    return needToIterate;
  }
  >>
end functionCheckForDiscreteChanges;

template crefToPrintfArg(ComponentRef cr)
::=
  match crefType(cr)
  case "modelica_real" then "%g"
  case "modelica_integer" then "%ld"
  case "modelica_boolean" then "%d"
  case "modelica_string" then "%s"
  else error(sourceInfo(), 'Do not know what printf argument to give <%crefStr(cr)%>')
  end match
end crefToPrintfArg;

template crefType(ComponentRef cr) "template crefType
  Like cref but with cast if type is integer."
::=
  match cr
  case CREF_IDENT(__) then '<%expTypeModelica(identType)%>'
  case CREF_QUAL(__)  then '<%crefType(componentRef)%>'
  else "crefType:ERROR"
  end match
end crefType;

template functionAssertsforCheck(list<SimEqSystem> algAndEqAssertsEquations, String modelNamePrefix) "template functionAssertsforCheck
  Generates function in simulation file.
  This is a helper of template simulationFile."
::=
  let &varDecls = buffer ""
  let &tmp = buffer ""
  let algAndEqAssertsPart = (algAndEqAssertsEquations |> eq =>
    equation_(-1, eq, contextSimulationDiscrete, &varDecls, &tmp, modelNamePrefix)
    ;separator="\n")

  <<
  <%&tmp%>
  /* function to check assert after a step is done */
  int <%symbolName(modelNamePrefix,"checkForAsserts")%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%varDecls%>

    <%algAndEqAssertsPart%>

    TRACE_POP
    return 0;
  }
  >>
end functionAssertsforCheck;

template functionlinearmodel(ModelInfo modelInfo, String modelNamePrefix) "template functionlinearmodel
  Generates function in simulation file."
::=
  match modelInfo
  case MODELINFO(varInfo=VARINFO(__), vars=SIMVARS(__)) then
    let matrixA = genMatrix("A", varInfo.numStateVars, varInfo.numStateVars)
    let matrixB = genMatrix("B", varInfo.numStateVars, varInfo.numInVars)
    let matrixC = genMatrix("C", varInfo.numOutVars, varInfo.numStateVars)
    let matrixD = genMatrix("D", varInfo.numOutVars, varInfo.numInVars)
    let vectorX = genVector("x", varInfo.numStateVars, 0)
    let vectorU = genVector("u", varInfo.numInVars, 1)
    let vectorY = genVector("y", varInfo.numOutVars, 2)
    //string def_proctedpart("\n  Real x[<%varInfo.numStateVars%>](start=x0);\n  Real u[<%varInfo.numInVars%>](start=u0); \n  output Real y[<%varInfo.numOutVars%>]; \n");
    <<
    const char *<%symbolName(modelNamePrefix,"linear_model_frame")%>()
    {
      return "model linear_<%underscorePath(name)%>\n  parameter Integer n = <%varInfo.numStateVars%>; // states \n  parameter Integer k = <%varInfo.numInVars%>; // top-level inputs \n  parameter Integer l = <%varInfo.numOutVars%>; // top-level outputs \n"
      "  parameter Real x0[<%varInfo.numStateVars%>] = {%s};\n"
      "  parameter Real u0[<%varInfo.numInVars%>] = {%s};\n"
      <%matrixA%>
      <%matrixB%>
      <%matrixC%>
      <%matrixD%>
      <%vectorX%>
      <%vectorU%>
      <%vectorY%>
      "\n  <%getVarName(vars.stateVars, "x", varInfo.numStateVars )%>  <% getVarName(vars.inputVars, "u", varInfo.numInVars) %>  <%getVarName(vars.outputVars, "y", varInfo.numOutVars) %>\n"
      "equation\n  der(x) = A * x + B * u;\n  y = C * x + D * u;\nend linear_<%underscorePath(name)%>;\n";
    }
    >>
  end match
end functionlinearmodel;

template getVarName(list<SimVar> simVars, String arrayName, Integer arraySize) "template getVarName
  Generates name for a varables."
::=
  match simVars
  case {} then
    <<
    >>
  case (var :: restVars) then
    let rest = getVarName(restVars, arrayName, arraySize)
    let arrindex = decrementInt(arraySize,listLength(restVars))
    match var
    case SIMVAR(__) then
      <<Real <%arrayName%>_<%crefM(name)%> = <%arrayName%>[<%arrindex%>];\n  <%rest%>>>
    end match
  end match
end getVarName;

template genMatrix(String name, Integer row, Integer col) "template genMatrix
  Generates Matrix for linear model"
::=
  match row
  case 0 then
    <<"  parameter Real <%name%>[<%row%>,<%col%>] = zeros(<%row%>,<%col%>);%s\n">>
  case _ then
    match col
    case 0 then
      <<"  parameter Real <%name%>[<%row%>,<%col%>] = zeros(<%row%>,<%col%>);%s\n">>
    case _ then
      <<"  parameter Real <%name%>[<%row%>,<%col%>] = [%s];\n">>
    end match
  end match
end genMatrix;

template genVector(String name, Integer numIn, Integer flag) "template genVector
  Generates variables Vectors for linear model"
::=
  match flag
  case 0 then
    match numIn
    case 0 then
      <<"  Real <%name%>[<%numIn%>];\n">>
    case _ then
      <<"  Real <%name%>[<%numIn%>](start=<%name%>0);\n">>
    end match
  case 1 then
    match numIn
    case 0 then
      <<"  input Real <%name%>[<%numIn%>];\n">>
    case _ then
      <<"  input Real <%name%>[<%numIn%>](start= <%name%>0);\n">>
    end match
  case 2 then
    match numIn
    case 0 then
      <<"  output Real <%name%>[<%numIn%>];\n">>
    case _ then
      <<"  output Real <%name%>[<%numIn%>];\n">>
    end match
  end match
end genVector;

template functionAnalyticJacobians(list<JacobianMatrix> JacobianMatrixes,String modelNamePrefix) "template functionAnalyticJacobians
  This template generates source code for all given jacobians."
::=
  let initialjacMats = (JacobianMatrixes |> (mat, vars, name, (sparsepattern,_), colorList, maxColor, indexJacobian) =>
    initialAnalyticJacobians(mat, vars, name, sparsepattern, colorList, maxColor, modelNamePrefix); separator="\n")
  let jacMats = (JacobianMatrixes |> (mat, vars, name, sparsepattern, colorList, maxColor, indexJacobian) =>
    generateMatrix(mat, vars, name, modelNamePrefix) ;separator="\n")

  <<
  <%initialjacMats%>

  <%jacMats%>
  >>
end functionAnalyticJacobians;

template initialAnalyticJacobians(list<JacobianColumn> jacobianColumn, list<SimVar> seedVars, String matrixname, list<tuple<Integer,list<Integer>>> sparsepattern, list<list<Integer>> colorList, Integer maxColor, String modelNamePrefix)
"template initialAnalyticJacobians
  This template generates source code for functions that initialize the sparse-pattern for a single jacobian.
  This is a helper of template functionAnalyticJacobians"
::=
match seedVars
case {} then
<<
int <%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%matrixname%>(void* inData, threadData_t *threadData)
{
  TRACE_PUSH
  TRACE_POP
  return 1;
}
>>
case _ then
  match sparsepattern
  case _ then
      let &eachCrefParts = buffer ""
      let sp_size_index =  lengthListElements(unzipSecond(sparsepattern))
      let sizeleadindex = listLength(sparsepattern)
      let leadindex = (sparsepattern |> (i, indexes) =>
      <<
      <%listLength(indexes)%>
      >>
      ;separator=",")
      let indexElems = ( sparsepattern |> (i, indexes) hasindex index0 =>
        ( indexes |> indexrow =>
        <<
        <%indexrow%>
        >>
        ;separator=",")
      ;separator=",")
      let colorArray = (colorList |> (indexes) hasindex index0 =>
        let colorCol = ( indexes |> i_index =>
         <<data->simulationInfo->analyticJacobians[index].sparsePattern.colorCols[<%i_index%>] = <%intAdd(index0,1)%>;>>
        ;separator="\n")
      '<%colorCol%>'
      ;separator="\n")
      let indexColumn = (jacobianColumn |> (eqs,vars,indxColumn) => indxColumn;separator="\n")
      let tmpvarsSize = (jacobianColumn |> (_,vars,_) => listLength(vars);separator="\n")
      let index_ = listLength(seedVars)
      <<

      int <%symbolName(modelNamePrefix,"initialAnalyticJacobian")%><%matrixname%>(void* inData, threadData_t *threadData)
      {
        TRACE_PUSH
        DATA* data = ((DATA*)inData);
        int index = <%symbolName(modelNamePrefix,"INDEX_JAC_")%><%matrixname%>;

        int i;

        data->simulationInfo->analyticJacobians[index].sizeCols = <%index_%>;
        data->simulationInfo->analyticJacobians[index].sizeRows = <%indexColumn%>;
        data->simulationInfo->analyticJacobians[index].sizeTmpVars = <%tmpvarsSize%>;
        data->simulationInfo->analyticJacobians[index].seedVars = (modelica_real*) calloc(<%index_%>,sizeof(modelica_real));
        data->simulationInfo->analyticJacobians[index].resultVars = (modelica_real*) calloc(<%indexColumn%>,sizeof(modelica_real));
        data->simulationInfo->analyticJacobians[index].tmpVars = (modelica_real*) calloc(<%tmpvarsSize%>,sizeof(modelica_real));
        data->simulationInfo->analyticJacobians[index].sparsePattern.leadindex = (unsigned int*) malloc(<%sizeleadindex%>*sizeof(int));
        data->simulationInfo->analyticJacobians[index].sparsePattern.index = (unsigned int*) malloc(<%sp_size_index%>*sizeof(int));
        data->simulationInfo->analyticJacobians[index].sparsePattern.numberOfNoneZeros = <%sp_size_index%>;
        data->simulationInfo->analyticJacobians[index].sparsePattern.colorCols = (unsigned int*) malloc(<%index_%>*sizeof(int));
        data->simulationInfo->analyticJacobians[index].sparsePattern.maxColors = <%maxColor%>;
        data->simulationInfo->analyticJacobians[index].jacobian = NULL;

        /* write lead index of compressed sparse column*/
        const int tmp[<%sizeleadindex%>] = {<%leadindex%>};
        memcpy(data->simulationInfo->analyticJacobians[index].sparsePattern.leadindex, tmp, <%sizeleadindex%>*sizeof(int));

        for(i=1;i<<%sizeleadindex%>;++i)
            data->simulationInfo->analyticJacobians[index].sparsePattern.leadindex[i] += data->simulationInfo->analyticJacobians[index].sparsePattern.leadindex[i-1];

        /* call sparse index */
        const int tmpElem[<%sp_size_index%>] = {<%indexElems%>};
        memcpy(data->simulationInfo->analyticJacobians[index].sparsePattern.index, tmpElem, <%sp_size_index%>*sizeof(int));

        /* write color array */
        <%colorArray%>
        TRACE_POP
        return 0;
      }
      >>
   end match
end match
end initialAnalyticJacobians;

template generateMatrix(list<JacobianColumn> jacobianColumn, list<SimVar> seedVars, String matrixname, String modelNamePrefix)
  "This template generates source code for a single jacobian in dense format and sparse format.
  This is a helper of template functionAnalyticJacobians"
::=
  let indxColumn = (jacobianColumn |> (eqs,vars,indxColumn) => indxColumn)
  match indxColumn
  case "0" then
    <<
    int <%symbolName(modelNamePrefix,"functionJac")%><%matrixname%>_column(void* data, threadData_t *threadData)
    {
      TRACE_PUSH
      TRACE_POP
      return 0;
    }
    >>
  case _ then
    match seedVars
     case {} then
        <<
        int <%symbolName(modelNamePrefix,"functionJac")%><%matrixname%>_column(void* data, threadData_t *threadData)
        {
          TRACE_PUSH
          TRACE_POP
          return 0;
        }
        >>
      case _ then
        let jacMats = (jacobianColumn |> (eqs,vars,indxColumn) =>
          functionJac(eqs, vars, indxColumn, matrixname, modelNamePrefix)
          ;separator="\n")
        let indexColumn = (jacobianColumn |> (eqs,vars,indxColumn) =>
          indxColumn
          ;separator="\n")
        <<
        <%jacMats%>
        >>
     end match
  end match
end generateMatrix;

template functionJac(list<SimEqSystem> jacEquations, list<SimVar> tmpVars, String columnLength, String matrixName, String modelNamePrefix) "template functionJac
  This template generates functions for each column of a single jacobian.
  This is a helper of generateMatrix."
::=
  let &varDecls = buffer ""
  let &tmp = buffer ""
  let eqns_ = (jacEquations |> eq =>
    equation_(-1, eq, contextSimulationNonDiscrete, &varDecls, &tmp, modelNamePrefix); separator="\n")

  <<
  <%&tmp%>
  int <%symbolName(modelNamePrefix,"functionJac")%><%matrixName%>_column(void* inData, threadData_t *threadData)
  {
    TRACE_PUSH

    DATA* data = ((DATA*)inData);
    int index = <%symbolName(modelNamePrefix,"INDEX_JAC_")%><%matrixName%>;
    <%varDecls%>
    <%eqns_%>

    TRACE_POP
    return 0;
  }
  >>
end functionJac;

template equation_arrayFormat(SimEqSystem eq, String name, Context context, Integer arrayIndex, Text &varDecls, Text &eqArray, Text &eqfuncs, String modelNamePrefix)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  (
  let ix = equationIndex(eq) /*System.tmpTickIndex(10)*/
  let &tmp = buffer ""
  let &varD = buffer ""
  let &tempeqns = buffer ""
  let() = System.tmpTickResetIndex(0,1) /* Boxed array indices */
  let disc = match context
  case SIMULATION_CONTEXT(genDiscrete=true) then 1
  else 0
  let x = match eq
  case e as SES_SIMPLE_ASSIGN(__)
    then equationSimpleAssign(e, context, &varD, &tempeqns)
  case e as SES_ARRAY_CALL_ASSIGN(__)
    then equationArrayCallAssign(e, context, &varD, &tempeqns)
  case e as SES_IFEQUATION(__)
    then equationIfEquationAssign(e, context, &varD, &tempeqns, modelNamePrefix)
  case e as SES_ALGORITHM(__)
    then equationAlgorithm(e, context, &varD, &tempeqns)
  case e as SES_INVERSE_ALGORITHM(__)
    then equationAlgorithm(e, context, &varD, &tempeqns)
  case e as SES_LINEAR(__)
    then equationLinear(e, context, &varD)
  // no dynamic tearing
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
    let &tempeqns += (nls.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    equationNonlinear(e, context, &varD, modelNamePrefix)
  // dynamic tearing
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
    let &tempeqns += (nls.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    let &tempeqns += '<%\n%>'
    let &tempeqns += (at.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    equationNonlinear(e, context, &varD, modelNamePrefix)
  case e as SES_WHEN(__)
    then equationWhen(e, context, &varD, &tempeqns)
  case e as SES_RESIDUAL(__)
    then "NOT IMPLEMENTED EQUATION SES_RESIDUAL"
  case e as SES_MIXED(__)
    then equationMixed(e, context, &varD, &eqfuncs, modelNamePrefix)
  else
    "NOT IMPLEMENTED EQUATION equation_"

  let &eqArray += '<%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>, <%\n%>'
  let &varD += addRootsTempArray()
  let &eqfuncs +=
  <<

  <%tempeqns%>
  /*
   <%dumpEqs(fill(eq,1))%>
   */
  void <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    const int equationIndexes[2] = {1,<%ix%>};
    <%&varD%>
    <%x%>
    TRACE_POP
  }
  >>
  <<
  // <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>(data, threadData);
  function<%name%>_systems[<%arrayIndex%>](data, threadData);
  >>
  )
end equation_arrayFormat;

template equation_(Integer clockIndex, SimEqSystem eq, Context context, Text &varDecls, Text &eqs, String modelNamePrefix)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  (
  let ix = equationIndex(eq) /*System.tmpTickIndex(10)*/
  let ix2 = match eq
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
    equationIndexAlternativeTearing(eq)
  else
    ""
  let &tmp = buffer ""
  let &varD = buffer ""
  let &tempeqns = buffer ""
  let &tempeqns2 = buffer ""
  let() = System.tmpTickResetIndex(0,1) /* Boxed array indices */
  let disc = match context
  case SIMULATION_CONTEXT(genDiscrete=true) then 1
  else 0
  let x = match eq
  case e as SES_SIMPLE_ASSIGN(__)
    then equationSimpleAssign(e, context, &varD, &tempeqns)
  case e as SES_ARRAY_CALL_ASSIGN(__)
    then equationArrayCallAssign(e, context, &varD, &tempeqns)
  case e as SES_IFEQUATION(__)
    then equationIfEquationAssign(e, context, &varD, &tempeqns, modelNamePrefix)
  case e as SES_ALGORITHM(__)
    then equationAlgorithm(e, context, &varD, &tempeqns)
  case e as SES_INVERSE_ALGORITHM(__)
    then equationAlgorithm(e, context, &varD, &tempeqns)
  case e as SES_LINEAR(__)
    then equationLinear(e, context, &varD)
  // no dynamic tearing
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
    let &tempeqns += (nls.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    equationNonlinear(e, context, &varD, modelNamePrefix)
  // dynamic tearing
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
    let &tempeqns += (nls.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    equationNonlinear(e, context, &varD, modelNamePrefix)
  case e as SES_WHEN(__)
    then equationWhen(e, context, &varD, &tempeqns)
  case e as SES_RESIDUAL(__)
    then "NOT IMPLEMENTED EQUATION SES_RESIDUAL"
  case e as SES_MIXED(__)
    then equationMixed(e, context, &varD, &eqs, modelNamePrefix)
  case e as SES_FOR_LOOP(__)
    then equationForLoop(e, context, &varD, &tempeqns)
  else
    "NOT IMPLEMENTED EQUATION equation_"
  let x2 = match eq
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
    let &tempeqns2 += (at.eqs |> eq => 'void <%symbolName(modelNamePrefix,"eqFunction")%>_<%equationIndex(eq)%>(DATA*,threadData_t*);' ; separator = "\n")
    equationNonlinearAlternativeTearing(e, context, &varD, modelNamePrefix)
  else
    ""

  let &varD += addRootsTempArray()
  let clockIndex_ = if intLt(clockIndex, 0) then '' else 'const int clockIndex = <%clockIndex%>;'

  match eq
  case e as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
  let &eqs +=
  <<

  <%tempeqns%>
  /*
   <%dumpEqs(fill(eq,1))%>
   */
  int <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%clockIndex_%>
    const int equationIndexes[2] = {1,<%ix%>};
    <%&varD%>
    <%x%>
    TRACE_POP
  }

  <%tempeqns2%>
  /*
   <%dumpEqsAlternativeTearing(fill(eq,1))%>
   */
  void <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix2%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%clockIndex_%>
    const int equationIndexes[2] = {1,<%ix2%>};
    <%&varD%>
    <%x2%>
    TRACE_POP
  }
  >>

  <<
  <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ix2%>);' %>
  <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix2%>(data, threadData);
  <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ix2%>);' %>
  >>
  else
  let &eqs +=
  <<

  <%tempeqns%>
  /*
   <%dumpEqs(fill(eq,1))%>
   */
  void <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>(DATA *data, threadData_t *threadData)
  {
    TRACE_PUSH
    <%clockIndex_%>
    const int equationIndexes[2] = {1,<%ix%>};
    <%&varD%>
    <%x%>
    TRACE_POP
  }
  >>

  <<
  <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ix%>);' %>
  <%symbolName(modelNamePrefix,"eqFunction")%>_<%ix%>(data, threadData);
  <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ix%>);' %>
  >>
  )
end equation_;

template equationForward_(SimEqSystem eq, Context context, String modelNamePrefixStr)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
 match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = match eq
  case e as SES_NONLINEAR(alternativeTearing = SOME(NONLINEARSYSTEM)) then
    equationIndexAlternativeTearing(eq)
  else
    equationIndex(eq)
  end match
  <<
  extern void <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(DATA* data, threadData_t *threadData);
  >>
end equationForward_;

template equationNames_(SimEqSystem eq, Context context, String modelNamePrefixStr)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
 match eq
  case e as SES_ALGORITHM(statements={})
  then ""
  else
  let ix = match eq
  case e as SES_NONLINEAR(alternativeTearing = SOME(NONLINEARSYSTEM)) then
    equationIndexAlternativeTearing(eq)
  else
    equationIndex(eq)
  end match
  <<
  <% if profileAll() then 'SIM_PROF_TICK_EQ(<%ix%>);' %>
  <%symbolName(modelNamePrefixStr,"eqFunction")%>_<%ix%>(data, threadData);
  <% if profileAll() then 'SIM_PROF_ACC_EQ(<%ix%>);' %>
  >>
end equationNames_;

template equationSimpleAssign(SimEqSystem eq, Context context,
                              Text &varDecls, Text &auxFunction)
 "Generates an equation that is just a simple assignment."
::=
match eq
case SES_SIMPLE_ASSIGN(exp=CALL(path=IDENT(name="fail"))) then
  '<%generateThrow()%><%\n%>'
case SES_SIMPLE_ASSIGN(__) then
  let &preExp = buffer ""
  let expPart = daeExp(exp, context, &preExp, &varDecls, &auxFunction)
  <<
  <%modelicaLine(eqInfo(eq))%>
  <%preExp%>
  <%cref(cref)%> = <%expPart%>;
  <%endModelicaLine()%>
  >>
end equationSimpleAssign;

template equationForLoop(SimEqSystem eq, Context context, Text &varDecls, Text &auxFunction)
 "Generates an equation that is a for-loop."
::=
match eq
case SES_FOR_LOOP(__) then
  let &preExp = buffer ""
  let expPart = daeExp(exp, context, &preExp, &varDecls, &auxFunction)
  let crefPart = daeExp(crefExp(cref), context, &preExp, &varDecls, &auxFunction)
  //let bodyStr = daeExpIteratedCref(body)
  let start = printExpStr(startIt)
  let stop = printExpStr(endIt)
  let iterVar = daeExp(iter, context, &preExp, &varDecls, &auxFunction)
  <<
  <%modelicaLine(eqInfo(eq))%>
  modelica_integer  $P<%printExpStr(iter)%> = 0; // the iterator
  // the for-equation
  for($P<%printExpStr(iter)%> = <%start%>; $P<%printExpStr(iter)%> != <%stop%>+1; $P<%printExpStr(iter)%>++)
  {
    <%crefPart%> += <%expPart%>;
  }
  <%endModelicaLine()%>
  >>
end equationForLoop;


template equationArrayCallAssign(SimEqSystem eq, Context context,
                                 Text &varDecls, Text &auxFunction)
 "Generates equation on form 'cref_array = call(...)'."
::=
<<
<%modelicaLine(eqInfo(eq))%>
<%match eq

case eqn as SES_ARRAY_CALL_ASSIGN(lhs=lhs as CREF(__)) then
  let &preExp = buffer ""
  let expPart = daeExp(exp, context, &preExp, &varDecls, &auxFunction)
  let lhsstr = daeExpCrefLhs(lhs, context, &preExp, &varDecls, &auxFunction)
  match expTypeFromExpShort(eqn.exp)
  case "boolean" then
    <<
    <%preExp%>
    copy_boolean_array_data(<%expPart%>, &<%lhsstr%>);
    >>
  case "integer" then
    <<
    <%preExp%>
    copy_integer_array_data(<%expPart%>, &<%lhsstr%>);
    >>
  case "real" then
    <<
    <%preExp%>
    copy_real_array_data(<%expPart%>, &<%lhsstr%>);
    >>
  case "string" then
    <<
    <%preExp%>
    copy_string_array_data(<%expPart%>, &<%lhsstr%>);
    >>
  else error(sourceInfo(), 'No runtime support for this sort of array call: <%printExpStr(eqn.exp)%>')
%>
<%endModelicaLine()%>
>>
end equationArrayCallAssign;

template equationAlgorithm(SimEqSystem eq, Context context, Text &varDecls, Text &auxFunction)
 "Generates an equation that is an algorithm."
::=
match eq
case SES_ALGORITHM(__)
case SES_INVERSE_ALGORITHM(__) then
  (statements |> stmt =>
    algStatement(stmt, context, &varDecls, &auxFunction)
  ;separator="\n")
end equationAlgorithm;

template equationLinear(SimEqSystem eq, Context context, Text &varDecls)
 "Generates a linear equation system."
::=
match eq
// no dynamic tearing
case e as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
  <<
  /* Linear equation system */
  <% if profileSome() then 'SIM_PROF_TICK_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex);' %>
  <%ls.vars |> SIMVAR(__) hasindex i0 => 'data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].x[<%i0%>] = _<%cref(name)%>(1);' ;separator="\n"%>
  solve_linear_system(data, threadData, <%ls.indexLinearSystem%>);
  <%ls.vars |> SIMVAR(__) hasindex i0 => '<%cref(name)%> = data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].x[<%i0%>];' ;separator="\n"%>
  <% if profileSome() then 'SIM_PROF_ACC_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex);' %>
  >>

// dynamic tearing
case e as SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(LINEARSYSTEM(__))) then
  <<
  /* Linear equation system */
  <% if profileSome() then 'SIM_PROF_TICK_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex);' %>
  <%ls.vars |> SIMVAR(__) hasindex i0 => 'data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].x[<%i0%>] = _<%cref(name)%>(1);' ;separator="\n"%>
  solve_linear_system(data, threadData, <%ls.indexLinearSystem%>);
  <%ls.vars |> SIMVAR(__) hasindex i0 => '<%cref(name)%> = data->simulationInfo->linearSystemData[<%ls.indexLinearSystem%>].x[<%i0%>];' ;separator="\n"%>
  <% if profileSome() then 'SIM_PROF_ACC_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%ls.index%>).profileBlockIndex);' %>
  >>
end equationLinear;


template equationMixed(SimEqSystem eq, Context context, Text &varDecls, Text &tmp, String modelNamePrefixStr)
 "Generates a mixed equation system."
::=
match eq
case eqn as SES_MIXED(__) then
  let contEqs = equation_(-1, cont, context, &varDecls, &tmp, modelNamePrefixStr)
  let numDiscVarsStr = listLength(discVars)
  <<
  /* Continuous equation part in <%contEqs%> */
  <% if profileSome() then 'SIM_PROF_TICK_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%index%>).profileBlockIndex);' %>
  <%discVars |> SIMVAR(__) hasindex i0 => 'data->simulationInfo->mixedSystemData[<%eqn.indexMixedSystem%>].iterationVarsPtr[<%i0%>] = (modelica_boolean*)&<%cref(name)%>;' ;separator="\n"%>;
  <%discVars |> SIMVAR(__) hasindex i0 => 'data->simulationInfo->mixedSystemData[<%eqn.indexMixedSystem%>].iterationPreVarsPtr[<%i0%>] = (modelica_boolean*)&$P$PRE<%cref(name)%>;' ;separator="\n"%>;
  solve_mixed_system(data, <%indexMixedSystem%>);
  <% if profileSome() then 'SIM_PROF_ACC_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%index%>).profileBlockIndex);' %>
  >>
end equationMixed;


template equationNonlinear(SimEqSystem eq, Context context, Text &varDecls, String modelNamePrefix)
 "Generates a non linear equation system."
::=
  match eq
    case eq as SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = at) then
      let size = listLength(nls.crefs)
      let &tmp = buffer ""
      let innerBody = (nls.eqs |> eq2 =>
         functionExtraResidualsPreBody(eq2, &varDecls, &tmp, modelNamePrefix)
       ;separator="\n")
      let nonlinindx = nls.indexNonLinearSystem
      let returnval = match at case at as SOME(__) then 'return 1;' case at as NONE() then ''
      let returnval2 = match at case at as SOME(__) then 'return 0;' case at as NONE() then ''
      <<
      int retValue;
      if(ACTIVE_STREAM(LOG_DT))
      {
        infoStreamPrint(LOG_DT, 1, "Solving nonlinear system <%nls.index%> (STRICT TEARING SET if tearing enabled) at time = %18.10e", data->localData[0]->timeValue);
        messageClose(LOG_DT);
      }
      <% if profileSome() then
      <<
      SIM_PROF_TICK_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%nls.index%>).profileBlockIndex);
      SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%nls.index%>).profileBlockIndex,-1);
      >>
      %>
      /* extrapolate data */
      <%nls.crefs |> name hasindex i0 =>
        let namestr = cref(name)
        <<
        data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].nlsx[<%i0%>] = <%namestr%>;
        data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].nlsxOld[<%i0%>] = _<%namestr%>(1) /*old1*/;
        data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].nlsxExtrapolation[<%i0%>] = extraPolate(data, _<%namestr%>(1) /*old1*/, _<%namestr%>(2) /*old2*/, $P$ATTRIBUTE<%namestr%>.min, $P$ATTRIBUTE<%namestr%>.max);
        >>
      ;separator="\n"%>
      retValue = solve_nonlinear_system(data, threadData, <%nls.indexNonLinearSystem%>);
      /* check if solution process was successful */
      if (retValue > 0){
        const int indexes[2] = {1,<%nls.index%>};
        throwStreamPrintWithEquationIndexes(threadData, indexes, "Solving non-linear system <%nls.index%> failed at time=%.15g.\nFor more information please use -lv LOG_NLS.", data->localData[0]->timeValue);
        <%returnval2%>
      }
      /* write solution */
      <%nls.crefs |> name hasindex i0 => '<%cref(name)%> = data->simulationInfo->nonlinearSystemData[<%nls.indexNonLinearSystem%>].nlsx[<%i0%>];' ;separator="\n"%>
      <% if profileSome() then 'SIM_PROF_ACC_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%nls.index%>).profileBlockIndex);' %>
      <%returnval%>
      >>
end equationNonlinear;

template equationNonlinearAlternativeTearing(SimEqSystem eq, Context context, Text &varDecls, String modelNamePrefix)
 "Generates a non linear equation system."
::=
  match eq
    case eq as SES_NONLINEAR(nlSystem = nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
      let size = listLength(at.crefs)
      let &tmp = buffer ""
      let innerBody = (at.eqs |> eq2 =>
         functionExtraResidualsPreBody(eq2, &varDecls, &tmp, modelNamePrefix)
       ;separator="\n")
      let nonlinindx = at.indexNonLinearSystem
      <<
      int retValue;
      if(ACTIVE_STREAM(LOG_DT))
      {
        infoStreamPrint(LOG_DT, 1, "Solving nonlinear system <%at.index%> (CASUAL TEARING SET) at time = %18.10e", data->localData[0]->timeValue);
        messageClose(LOG_DT);
      }
      <% if profileSome() then
      <<
      SIM_PROF_TICK_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%at.index%>).profileBlockIndex);
      SIM_PROF_ADD_NCALL_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%at.index%>).profileBlockIndex,-1);
      >>
      %>
      /* extrapolate data */
      <%at.crefs |> name hasindex i0 =>
        let namestr = cref(name)
        <<
        data->simulationInfo->nonlinearSystemData[<%at.indexNonLinearSystem%>].nlsx[<%i0%>] = <%namestr%>;
        data->simulationInfo->nonlinearSystemData[<%at.indexNonLinearSystem%>].nlsxOld[<%i0%>] = _<%namestr%>(1) /*old1*/;
        data->simulationInfo->nonlinearSystemData[<%at.indexNonLinearSystem%>].nlsxExtrapolation[<%i0%>] = extraPolate(data, _<%namestr%>(1) /*old1*/, _<%namestr%>(2) /*old2*/,$P$ATTRIBUTE<%namestr%>.min, $P$ATTRIBUTE<%namestr%>.max);
        >>
      ;separator="\n"%>
      retValue = solve_nonlinear_system(data, threadData, <%at.indexNonLinearSystem%>);

      /* The casual tearing set found a solution */
      if (retValue == 0){
      /* write solution */
      <%at.crefs |> name hasindex i0 => '<%cref(name)%> = data->simulationInfo->nonlinearSystemData[<%at.indexNonLinearSystem%>].nlsx[<%i0%>];' ;separator="\n"%>
      <% if profileSome() then 'SIM_PROF_ACC_EQ(modelInfoGetEquation(&data->modelData->modelDataXml,<%at.index%>).profileBlockIndex);' %>
      }
      >>
end equationNonlinearAlternativeTearing;

template equationWhen(SimEqSystem eq, Context context, Text &varDecls, Text &auxFunction)
 "Generates a when equation."
::=
  match eq
    case SES_WHEN(whenStmtLst = whenStmtLst, conditions=conditions, elseWhen=NONE()) then
      let helpIf = if intGt(listLength(conditions), 0) then (conditions |> e => '(<%cref(e)%> && !$P$PRE<%cref(e)%> /* edge */)';separator=" || ") else '0'
      let assign = whenOperators(whenStmtLst, context, &varDecls, auxFunction)
      <<
      if(<%helpIf%>)
      {
        <%assign%>
      }
      >>
    case SES_WHEN(whenStmtLst = whenStmtLst, conditions=conditions, elseWhen=SOME(elseWhenEq)) then
      let helpIf = if intGt(listLength(conditions), 0) then (conditions |> e => '(<%cref(e)%> && !$P$PRE<%cref(e)%> /* edge */)';separator=" || ") else '0'
      let assign = whenOperators(whenStmtLst, context, &varDecls, auxFunction)
      let elseWhen = equationElseWhen(elseWhenEq,context,varDecls,&auxFunction)
      <<
      if(<%helpIf%>)
      {
        <%assign%>
      }
      <%elseWhen%>
      >>
end equationWhen;

template equationElseWhen(SimEqSystem eq, Context context, Text &varDecls, Text &auxFunction)
 "Generates a else when equation."
::=
match eq
case SES_WHEN(whenStmtLst = whenStmtLst, conditions=conditions, elseWhen=NONE()) then
  let helpIf = (conditions |> e => '(<%cref(e)%> && !$P$PRE<%cref(e)%> /* edge */)';separator=" || ")
  let assign = whenOperators(whenStmtLst, context, &varDecls, auxFunction)

  if intGt(listLength(conditions), 0) then
    <<
    else if(<%helpIf%>)
    {
      <%assign%>
    }
    >>
case SES_WHEN(whenStmtLst = whenStmtLst, conditions=conditions, elseWhen=SOME(elseWhenEq)) then
  let helpIf = (conditions |> e => '(<%cref(e)%> && !$P$PRE<%cref(e)%> /* edge */)';separator=" || ")
  let assign = whenOperators(whenStmtLst, context, &varDecls, auxFunction)
  let elseWhen = equationElseWhen(elseWhenEq, context, varDecls, auxFunction)
  let body = if intGt(listLength(conditions), 0) then
    <<
    else if(<%helpIf%>)
    {
      <%assign%>
    }
    >>

  <<
  <%body%>
  <%elseWhen%>
  >>
end equationElseWhen;

template whenOperators(list<WhenOperator> whenOps, Context context, Text &varDecls, Text &auxFunction)
  "Generates body statements for when equation."
::=
  let body = (whenOps |> whenOp =>
    match whenOp
    case ASSIGN(__) then whenAssign(left, typeof(right), right, context, &varDecls, &auxFunction)
    case REINIT(__) then
      let &preExp = buffer ""
      let val = daeExp(value, contextSimulationDiscrete, &preExp, &varDecls, &auxFunction)
      let lhs = match crefTypeConsiderSubs(stateVar)
         case DAE.T_ARRAY(__) then
           'copy_real_array_data_mem(<%val%>, &<%cref(stateVar)%>);'
         else
           '<%cref(stateVar)%> = <%val%>;'
      <<
      <%preExp%>
      <%lhs%>
      infoStreamPrint(LOG_EVENTS, 0, "reinit <%cref(stateVar)%> = <%crefToPrintfArg(stateVar)%>", <%cref(stateVar)%>);
      data->simulationInfo->needToIterate = 1;
      >>
    case TERMINATE(__) then
      let &preExp = buffer ""
      let msgVar = daeExp(message, contextSimulationDiscrete, &preExp, &varDecls, &auxFunction)
      <<
      <%preExp%>
      FILE_INFO info = {<%infoArgs(getElementSourceFileInfo(source))%>};
      omc_terminate(info, MMC_STRINGDATA(<%msgVar%>));
      >>
    case ASSERT(source=SOURCE(info=info)) then
      assertCommon(condition, List.fill(message,1), level, contextSimulationDiscrete, &varDecls, &auxFunction, info)
    case NORETCALL(__) then
      let &preExp = buffer ""
      let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, &varDecls, &auxFunction)
      <<
      <%preExp%>
      <% if isCIdentifier(expPart) then "" else '<%expPart%>;' %>
      >>
  ;separator="\n")
  <<
  <%body%>
  >>
end whenOperators;

template whenAssign(ComponentRef left, Type ty, Exp right, Context context, Text &varDecls, Text &auxFunction)
 "Generates assignment for when."
::=
match ty
  case T_ARRAY(__) then
    let &preExp = buffer ""
    let expPart = daeExp(right, context, &preExp, &varDecls, &auxFunction)
    match expTypeFromExpShort(right)
    case "boolean" then
      <<
      <%preExp%>
      copy_boolean_array_data_mem(<%expPart%>, &<%cref(left)%>);
      >>
    case "integer" then
      <<
      <%preExp%>
      copy_integer_array_data_mem(<%expPart%>, &<%cref(left)%>);
      >>
    case "real" then
      <<
      <%preExp%>
      copy_real_array_data_mem(<%expPart%>, &<%cref(left)%>);
      >>
    case "string" then
      <<
      <%preExp%>
      copy_string_array_data_mem(<%expPart%>, &<%cref(left)%>);
      >>
    else
      error(sourceInfo(), 'No runtime support for this sort of array call: <%cref(left)%> = <%printExpStr(right)%>')
    end match
  else
    let &preExp = buffer ""
    let exp = daeExp(right, context, &preExp, &varDecls, &auxFunction)
    <<
    <%preExp%>
    <%cref(left)%> = <%exp%>;
   >>
end whenAssign;

template equationIfEquationAssign(SimEqSystem eq, Context context, Text &varDecls, Text &eqnsDecls, String modelNamePrefixStr)
 "Generates a if equation."
::=
match eq
case SES_IFEQUATION(ifbranches=ifbranches, elsebranch=elsebranch) then
  let &preExp = buffer ""
  let IfEquation = (ifbranches |> (e, eqns) hasindex index0 =>
    let condition = daeExp(e, context, &preExp, &varDecls, &eqnsDecls)
    let ifequations = ( eqns |> eqn =>
       let eqnStr = equation_(-1, eqn, context, &varDecls, &eqnsDecls, modelNamePrefixStr)
       <<
       <%eqnStr%>
       >>

      ;separator="\n")
   let conditionline = if index0 then 'else if(<%condition%>)' else 'if(<%condition%>)'
    <<
    <%conditionline%>
    {
      <%ifequations%>
    }
    >>
    ;separator="\n")
  let elseequations = ( elsebranch |> eqn =>
     let eqnStr = equation_(-1, eqn, context, &varDecls, &eqnsDecls /*EQNBUF*/, modelNamePrefixStr)
       <<
       <%eqnStr%>
       >>
    ;separator="\n")
  <<
  <%preExp%>
  <%IfEquation%>else
  {
    <%elseequations%>
  }
  >>
end equationIfEquationAssign;

/* public */ template simulationLiteralsFile(String filePrefix, list<Exp> literals)
  "Generates the content of the C file for literals in the simulation case.
  used in Compiler/Template/CodegenFMU.tpl"
::=
  let &preLit = buffer ""
  let res = literals |> literal hasindex i0 fromindex 0 =>
    (if typeinfo() then '/* <%Util.escapeModelicaStringToCString(printExpStr(literal))%> */<%\n%>') +
    literalExpConst(literal,i0, &preLit)
    ; separator="\n";empty
  <<
  #ifdef __cplusplus
  extern "C" {
  #endif

  <%preLit%>
  <%res%>

  #ifdef __cplusplus
  }
  #endif<%\n%>
  >>
  /* adpro: leave a newline at the end of file to get rid of warnings! */
end simulationLiteralsFile;

/* public */ template simulationFunctionsFile(String filePrefix, list<Function> functions, Text &staticPrototypes)
 "Generates the content of the C file for functions in the simulation case.
  used in Compiler/Template/CodegenFMU.tpl"
::=
  <<
  #include "<%filePrefix%>_functions.h"
  #ifdef __cplusplus
  extern "C" {
  #endif

  #include "<%filePrefix%>_literals.h"
  #include "<%filePrefix%>_includes.h"

  <%if acceptParModelicaGrammar() then
  <<
  /* the OpenCL Kernels file name needed in libParModelicaExpl.a */
  const char* omc_ocl_kernels_source = "<%filePrefix%>_kernels.cl";
  /* the OpenCL program. Made global to avoid repeated builds */
  extern cl_program omc_ocl_program;
  /* The default OpenCL device. If not set (=0) show the selection option.*/
  unsigned int default_ocl_device = <%getDefaultOpenCLDevice()%>;
  >>
  %>

  <%if staticPrototypes then
  <<
  /* default, do not make protected functions static */
  #if !defined(PROTECTED_FUNCTION_STATIC)
  #define PROTECTED_FUNCTION_STATIC
  #endif
  <%staticPrototypes%>
  >>
  %>

  <%functionBodies(functions,true)%>

  #ifdef __cplusplus
  }
  #endif<%\n%>
  >>
  /* adpro: leave a newline at the end of file to get rid of warnings! */
end simulationFunctionsFile;

template simulationParModelicaKernelsFile(String filePrefix, list<Function> functions)
 "Generates the content of the C file for functions in the simulation case."
::=

  /* Reset the parfor loop id counter to 1*/
  let()= System.tmpTickResetIndex(0,20) /* parfor index */

  <<
  #include <ParModelica/explicit/openclrt/OCLRuntimeUtil.cl>

  // ParModelica Parallel Function headers.
  <%functionHeadersParModelica(filePrefix, functions)%>

  // Headers finish here.

  <%functionBodiesParModelica(functions)%>


  >>

end simulationParModelicaKernelsFile;

/* public */ template simulationFunctionsHeaderFile(String filePrefix, list<Function> functions, list<RecordDeclaration> recordDecls)
 "Generates the content of the C file for functions in the simulation case.
  used in Compiler/Template/CodegenFMU.tpl"
::=
  let &staticPrototypes = buffer ""
  <<
  #ifndef <%stringReplace(filePrefix,".","_")%>__H
  #define <%stringReplace(filePrefix,".","_")%>__H
  <%commonHeader(filePrefix)%>
  #include "simulation/simulation_runtime.h"
  #ifdef __cplusplus
  extern "C" {
  #endif
  <%\n%>
  <%recordDecls |> rd => recordDeclarationHeader(rd) ;separator="\n\n"%>
  <%\n%>
  <%functionHeaders(functions, true, staticPrototypes)%>
  #include "<%filePrefix%>_model.h"
  <%\n%>
  <%if staticPrototypes then
  <<
  /* default, do not make protected functions static */
  #if !defined(PROTECTED_FUNCTION_STATIC)
  #define PROTECTED_FUNCTION_STATIC
  #endif
  <%staticPrototypes%>
  >>
  %>
  <%\n%>
  #ifdef __cplusplus
  }
  #endif
  #endif<%\n%>
  <%\n%>
  >>
  /* adrpo: leave a newline at the end of file to get rid of the warning */
end simulationFunctionsHeaderFile;

template simulationMakefile(String target, SimCode simCode)
 "Generates the contents of the makefile for the simulation case."
::=
match getGeneralTarget(target)
case "msvc" then
match simCode
case SIMCODE(modelInfo=MODELINFO(__), makefileParams=MAKEFILE_PARAMS(__), simulationSettingsOpt = sopt) then
  let dirExtra = if modelInfo.directory then '/LIBPATH:"<%modelInfo.directory%>"' //else ""
  let libsStr = (makefileParams.libs |> lib => lib ;separator=" ")
  let libsPos1 = if not dirExtra then libsStr //else ""
  let libsPos2 = if dirExtra then libsStr // else ""
  let ParModelicaExpLibs = if acceptParModelicaGrammar() then 'ParModelicaExpl.lib OpenCL.lib' // else ""
  let extraCflags = match sopt case SOME(s as SIMULATION_SETTINGS(__)) then
    match s.method case "dassljac" then "-D_OMC_JACOBIAN "
  <<
  # Makefile generated by OpenModelica

  # Simulations use -O0 by default
  SIM_OR_DYNLOAD_OPT_LEVEL=
  MODELICAUSERCFLAGS=
  CC=cl
  CXX=cl
  EXEEXT=.exe
  DLLEXT=.dll

  # /Od - Optimization disabled
  # /EHa enable C++ EH (w/ SEH exceptions)
  # /fp:except - consider floating-point exceptions when generating code
  # /arch:SSE2 - enable use of instructions available with SSE2 enabled CPUs
  # /I - Include Directories
  # /DNOMINMAX - Define NOMINMAX (does what it says)
  # /TP - Use C++ Compiler
  CFLAGS=/Od /ZI /EHa /fp:except /I"<%makefileParams.omhome%>/include/omc/c" /I"<%makefileParams.omhome%>/include/omc/msvc/" /I. /DNOMINMAX /TP /DNO_INTERACTIVE_DEPENDENCY /DOPENMODELICA_XML_FROM_FILE_AT_RUNTIME <%if (Flags.isSet(Flags.HPCOM)) then '/openmp'%> <% if Flags.isSet(Flags.FMU_EXPERIMENTAL) then '/DFMU_EXPERIMENTAL' %>
  # /ZI enable Edit and Continue debug info
  CDFLAGS = /ZI

  # /MD - link with MSVCRT.LIB
  # /link - [linker options and libraries]
  # /LIBPATH: - Directories where libs can be found
  LDFLAGS=/MD /link /NODEFAULTLIB:libcmt /STACK:0x2000000 /pdb:"<%fileNamePrefix%>.pdb" /LIBPATH:"<%makefileParams.omhome%>/lib/<%getTriple()%>/omc/msvc/" /LIBPATH:"<%makefileParams.omhome%>/lib/<%getTriple()%>/omc/msvc/release/" <%dirExtra%> <%libsPos1%> <%libsPos2%> f2c.lib initialization.lib libexpat.lib math-support.lib meta.lib results.lib simulation.lib solver.lib sundials_kinsol.lib sundials_nvecserial.lib util.lib lapack_win32_MT.lib lis.lib  gc-lib.lib user32.lib pthreadVC2.lib wsock32.lib cminpack.lib umfpack.lib amd.lib

  # /MDd link with MSVCRTD.LIB debug lib
  # lib names should not be appended with a d just switch to lib/omc/msvc/debug


  FILEPREFIX=<%fileNamePrefix%>
  MAINFILE=$(FILEPREFIX).c
  MAINOBJ=$(FILEPREFIX).obj
  CFILES=<%fileNamePrefix%>_functions.c <%fileNamePrefix%>_records.c \
  <%fileNamePrefix%>_01exo.c <%fileNamePrefix%>_02nls.c <%fileNamePrefix%>_03lsy.c <%fileNamePrefix%>_04set.c <%fileNamePrefix%>_05evt.c <%fileNamePrefix%>_06inz.c <%fileNamePrefix%>_07dly.c \
  <%fileNamePrefix%>_08bnd.c <%fileNamePrefix%>_09alg.c <%fileNamePrefix%>_10asr.c <%fileNamePrefix%>_11mix.c <%fileNamePrefix%>_12jac.c <%fileNamePrefix%>_13opt.c <%fileNamePrefix%>_14lnz.c \
  <%fileNamePrefix%>_15syn.c
  OFILES=$(CFILES:.c=.obj)
  GENERATEDFILES=$(MAINFILE) $(FILEPREFIX)_functions.h $(FILEPREFIX).makefile $(CFILES)

  .PHONY: $(FILEPREFIX)$(EXEEXT)

  # This is to make sure that <%fileNamePrefix%>_*.c are always compiled.
  .PHONY: $(CFILES)

  $(FILEPREFIX)$(EXEEXT): $(MAINFILE) $(FILEPREFIX)_functions.h $(CFILES)
  <%\t%>$(CXX) /Fe$(FILEPREFIX)$(EXEEXT) $(MAINFILE) $(CFILES) $(CFLAGS) $(LDFLAGS)

  clean:
  <%\t%>rm -f *.obj *.lib *.exp *.c *.h *.xml *.libs *.log *.makefile *.pdb *.idb *.exe
  >>
end match
case "gcc" then
match simCode
case SIMCODE(modelInfo=MODELINFO(__), makefileParams=MAKEFILE_PARAMS(__), simulationSettingsOpt = sopt) then
  let dirExtra = if modelInfo.directory then '-L"<%modelInfo.directory%>"' //else ""
  let libsStr = (makefileParams.libs |> lib => lib ;separator=" ")
  let libsPos1 = if not dirExtra then libsStr //else ""
  let libsPos2 = if dirExtra then libsStr // else ""
  let ParModelicaExpLibs = if acceptParModelicaGrammar() then '-lParModelicaExpl -lOpenCL' // else ""
  let ParModelicaAutoLibs = if Flags.isSet(Flags.PARMODAUTO) then '-lParModelicaAuto -ltbb -lpugixml -lboost_system' // else ""
  let extraCflags = match sopt case SOME(s as SIMULATION_SETTINGS(__)) then
    match s.method case "dassljac" then "-D_OMC_JACOBIAN "

  <<
  # Makefile generated by OpenModelica

  # Simulations use -O3 by default
  CC=<%if boolOr(Flags.isSet(Flags.PARMODAUTO),acceptParModelicaGrammar()) then 'g++' else '<%makefileParams.ccompiler%>'%>
  CXX=<%makefileParams.cxxcompiler%>
  LINK=<%makefileParams.linker%>
  EXEEXT=<%makefileParams.exeext%>
  DLLEXT=<%makefileParams.dllext%>
  CFLAGS_BASED_ON_INIT_FILE=<%extraCflags%>
  DEBUG_FLAGS=<% if boolOr(acceptMetaModelicaGrammar(), Flags.isSet(Flags.GEN_DEBUG_SYMBOLS)) then "-O0 -g"%>
  CFLAGS=$(CFLAGS_BASED_ON_INIT_FILE) $(DEBUG_FLAGS) <%makefileParams.cflags%> <%match sopt case SOME(s as SIMULATION_SETTINGS(__)) then '<%s.cflags%> ' /* From the simulate() command */%> <% if Flags.isSet(Flags.FMU_EXPERIMENTAL) then '-DFMU_EXPERIMENTAL' %>
  <% if stringEq(Config.simCodeTarget(),"JavaScript") then 'OMC_EMCC_PRE_JS=<%makefileParams.omhome%>/lib/<%getTriple()%>/omc/emcc/pre.js<%\n%>'
  %>CPPFLAGS=<%makefileParams.includes ; separator=" "%> -I"<%makefileParams.omhome%>/include/omc/c" -I. -DOPENMODELICA_XML_FROM_FILE_AT_RUNTIME<% if stringEq(Config.simCodeTarget(),"JavaScript") then " -DOMC_EMCC"%>
  LDFLAGS=<%dirExtra%> <%
  if stringEq(Config.simCodeTarget(),"JavaScript") then <<-L'<%makefileParams.omhome%>/lib/<%getTriple()%>/omc/emcc' -lblas -llapack -lexpat -lSimulationRuntimeC -s TOTAL_MEMORY=805306368 -s OUTLINING_LIMIT=20000 --pre-js $(OMC_EMCC_PRE_JS)>>
  else <<-L"<%makefileParams.omhome%>/lib/<%getTriple()%>/omc" -L"<%makefileParams.omhome%>/lib" -Wl,<% if stringEq(makefileParams.platform, "win32") then "--stack,16777216,"%>-rpath,"<%makefileParams.omhome%>/lib/<%getTriple()%>/omc" -Wl,-rpath,"<%makefileParams.omhome%>/lib" <%ParModelicaExpLibs%> <%ParModelicaAutoLibs%> <%makefileParams.ldflags%> <%makefileParams.runtimelibs%> >>
  %>
  MAINFILE=<%fileNamePrefix%>.c
  MAINOBJ=<%fileNamePrefix%>.o
  CFILES=<%fileNamePrefix%>_functions.c <%fileNamePrefix%>_records.c \
  <%fileNamePrefix%>_01exo.c <%fileNamePrefix%>_02nls.c <%fileNamePrefix%>_03lsy.c <%fileNamePrefix%>_04set.c <%fileNamePrefix%>_05evt.c <%fileNamePrefix%>_06inz.c <%fileNamePrefix%>_07dly.c \
  <%fileNamePrefix%>_08bnd.c <%fileNamePrefix%>_09alg.c <%fileNamePrefix%>_10asr.c <%fileNamePrefix%>_11mix.c <%fileNamePrefix%>_12jac.c <%fileNamePrefix%>_13opt.c <%fileNamePrefix%>_14lnz.c \
  <%fileNamePrefix%>_15syn.c
  OFILES=$(CFILES:.c=.o)
  GENERATEDFILES=$(MAINFILE) <%fileNamePrefix%>.makefile <%fileNamePrefix%>_literals.h <%fileNamePrefix%>_functions.h $(CFILES)

  .PHONY: omc_main_target clean bundle

  # This is to make sure that <%fileNamePrefix%>_*.c are always compiled.
  .PHONY: $(CFILES)

  omc_main_target: $(MAINOBJ) <%fileNamePrefix%>_functions.h <%fileNamePrefix%>_literals.h $(OFILES)
  <%\t%>$(CC) -I. -o <%fileNamePrefix%>$(EXEEXT) $(MAINOBJ) $(OFILES) $(CPPFLAGS) <%dirExtra%> <%libsPos1%> <%libsPos2%> $(CFLAGS) $(LDFLAGS)
  <% if stringEq(Config.simCodeTarget(),"JavaScript") then '<%\t%>rm -f <%fileNamePrefix%>'%>
  <% if stringEq(Config.simCodeTarget(),"JavaScript") then '<%\t%>ln -s <%fileNamePrefix%>_node.js <%fileNamePrefix%>'%>
  <% if stringEq(Config.simCodeTarget(),"JavaScript") then '<%\t%>chmod +x <%fileNamePrefix%>_node.js'%>
  clean:
  <%\t%>@rm -f <%fileNamePrefix%>_records.o $(MAINOBJ)

  bundle:
  <%\t%>@tar -cvf <%fileNamePrefix%>_Files.tar $(GENERATEDFILES)
  >>
end match
else
  error(sourceInfo(), 'Target <%target%> is not handled!')
end simulationMakefile;

template xsdateTime(DateTime dt)
 "YYYY-MM-DDThh:mm:ssZ"
::=
  match dt
  case DATETIME(__) then '<%year%>-<%twodigit(mon)%>-<%twodigit(mday)%>T<%twodigit(hour)%>:<%twodigit(min)%>:<%twodigit(sec)%>Z'
end xsdateTime;

/* public */ template simulationInitFile(SimCode simCode, String guid)
 "Generates the contents of the makefile for the simulation case.
  used in Compiler/Template/CodegenFMU.tpl"
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(functions = functions, varInfo = vi as VARINFO(__), vars = vars as SIMVARS(__)),
             simulationSettingsOpt = SOME(s as SIMULATION_SETTINGS(__)), makefileParams = makefileParams as MAKEFILE_PARAMS(__))
  then
  <<
  <?xml version = "1.0" encoding="UTF-8"?>

  <!-- description of the model interface using an extention of the FMI standard -->
  <fmiModelDescription
    fmiVersion                          = "1.0"

    modelName                           = "<%Util.escapeModelicaStringToXmlString(dotPath(modelInfo.name))%>"
    modelIdentifier                     = "<%Util.escapeModelicaStringToXmlString(underscorePath(modelInfo.name))%>"

    OPENMODELICAHOME                    = "<%makefileParams.omhome%>"

    guid                                = "{<%guid%>}"

    description                         = "<%Util.escapeModelicaStringToXmlString(modelInfo.description)%>"
    generationTool                      = "OpenModelica Compiler <%getVersionNr()%>"
    generationDateAndTime               = "<%xsdateTime(getCurrentDateTime())%>"

    variableNamingConvention            = "structured"

    numberOfEventIndicators             = "<%vi.numZeroCrossings%>"  cmt_numberOfEventIndicators             = "NG:       number of zero crossings,                           FMI"
    numberOfTimeEvents                  = "<%vi.numTimeEvents%>"  cmt_numberOfTimeEvents                  = "NG_SAM:   number of zero crossings that are samples,          OMC"

    numberOfInputVariables              = "<%vi.numInVars%>"  cmt_numberOfInputVariables              = "NI:       number of inputvar on topmodel,                     OMC"
    numberOfOutputVariables             = "<%vi.numOutVars%>"  cmt_numberOfOutputVariables             = "NO:       number of outputvar on topmodel,                    OMC"

    numberOfExternalObjects             = "<%vi.numExternalObjects%>"  cmt_numberOfExternalObjects             = "NEXT:     number of external objects,                         OMC"
    numberOfFunctions                   = "<%listLength(functions)%>"  cmt_numberOfFunctions                   = "NFUNC:    number of functions used by the simulation,         OMC"

    numberOfContinuousStates            = "<%vi.numStateVars%>"  cmt_numberOfContinuousStates            = "NX:       number of states,                                   FMI"
    numberOfRealAlgebraicVariables      = "<%intAdd(vi.numAlgVars,intAdd(vi.numDiscreteReal,intAdd(vi.numOptimizeConstraints, vi.numOptimizeFinalConstraints)))%>"  cmt_numberOfRealAlgebraicVariables      = "NY:       number of real variables,                           OMC"
    numberOfRealAlgebraicAliasVariables = "<%vi.numAlgAliasVars%>"  cmt_numberOfRealAlgebraicAliasVariables = "NA:       number of alias variables,                          OMC"
    numberOfRealParameters              = "<%vi.numParams%>"  cmt_numberOfRealParameters              = "NP:       number of parameters,                               OMC"

    numberOfIntegerAlgebraicVariables   = "<%vi.numIntAlgVars%>"  cmt_numberOfIntegerAlgebraicVariables   = "NYINT:    number of alg. int variables,                       OMC"
    numberOfIntegerAliasVariables       = "<%vi.numIntAliasVars%>"  cmt_numberOfIntegerAliasVariables       = "NAINT:    number of alias int variables,                      OMC"
    numberOfIntegerParameters           = "<%vi.numIntParams%>"  cmt_numberOfIntegerParameters           = "NPINT:    number of int parameters,                           OMC"

    numberOfStringAlgebraicVariables    = "<%vi.numStringAlgVars%>"  cmt_numberOfStringAlgebraicVariables    = "NYSTR:    number of alg. string variables,                    OMC"
    numberOfStringAliasVariables        = "<%vi.numStringAliasVars%>"  cmt_numberOfStringAliasVariables        = "NASTR:    number of alias string variables,                   OMC"
    numberOfStringParameters            = "<%vi.numStringParamVars%>"  cmt_numberOfStringParameters            = "NPSTR:    number of string parameters,                        OMC"

    numberOfBooleanAlgebraicVariables   = "<%vi.numBoolAlgVars%>"  cmt_numberOfBooleanAlgebraicVariables   = "NYBOOL:   number of alg. bool variables,                      OMC"
    numberOfBooleanAliasVariables       = "<%vi.numBoolAliasVars%>"  cmt_numberOfBooleanAliasVariables       = "NABOOL:   number of alias bool variables,                     OMC"
    numberOfBooleanParameters           = "<%vi.numBoolParams%>"  cmt_numberOfBooleanParameters           = "NPBOOL:   number of bool parameters,                          OMC" >


    <!-- startTime, stopTime, tolerance are FMI specific, all others are OMC specific -->
    <DefaultExperiment
      startTime      = "<%s.startTime%>"
      stopTime       = "<%s.stopTime%>"
      stepSize       = "<%s.stepSize%>"
      tolerance      = "<%s.tolerance%>"
      solver         = "<%s.method%>"
      outputFormat   = "<%s.outputFormat%>"
      variableFilter = "<%s.variableFilter%>" />

    <!-- variables in the model -->
    <%ModelVariables(modelInfo)%>


  </fmiModelDescription>

  >>
end simulationInitFile;

template crefM(ComponentRef cr)
 "Generates Modelica equivalent name for component reference."
::=
  match cr
  case CREF_IDENT(ident = "xloc") then crefStr(cr)
  case CREF_IDENT(ident = "time") then "time"
  else "P" + crefToMStr(cr)
end crefM;

template equationInfo1(SimEqSystem eq, Text &preBuf, Text &eqnsDefines, Text &reverseProf)
::=
  match eq
    case SES_RESIDUAL(__) then
      '{<%index%>,"SES_RESIDUAL <%index%>",0,NULL}'
    case SES_SIMPLE_ASSIGN(__) then
      let var = '<%cref(cref)%>__varInfo'
      let &preBuf += 'const VAR_INFO** equationInfo_cref<%index%> = (const VAR_INFO**)calloc(1,sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += 'equationInfo_cref<%index%>[0] = &<%var%>;<%\n%>'
      '{<%index%>,"SES_SIMPLE_ASSIGN <%index%>",1,equationInfo_cref<%index%>}'
    case SES_ARRAY_CALL_ASSIGN(__) then
      '{<%index%>,"SES_ARRAY_CALL_ASSIGN <%index%>",0,NULL}'
    case SES_IFEQUATION(__) then
      let branches = ifbranches |> (_,eqs) => (eqs |> eq => equationInfo1(eq,preBuf,eqnsDefines,reverseProf) + ',<%\n%>')
      let elsebr = (elsebranch |> eq => equationInfo1(eq,preBuf,eqnsDefines,reverseProf) + ',<%\n%>')
      '<%branches%><%elsebr%>{<%index%>,"SES_IFEQUATION <%index%>",0,NULL}'
    case SES_ALGORITHM(__) then
      '{<%index%>,"SES_ALGORITHM <%index%>", 0, NULL}'
    case SES_WHEN(__) then
      '{<%index%>,"SES_WHEN <%index%>", 0, NULL}'
    // no dynamic tearing
    case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing=NONE()) then
      let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%ls.index%> = (const VAR_INFO**)malloc(<%listLength(ls.vars)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%ls.vars|>var hasindex i0 => 'equationInfo_crefs<%ls.index%>[<%i0%>] = &<%cref(varName(var))%>__varInfo;'; separator="\n"%>;'
      '{<%ls.index%>,"linear system <%ls.index%> (size <%listLength(ls.vars)%>)", <%listLength(ls.vars)%>, equationInfo_crefs<%ls.index%>}'
    case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing=NONE()) then
      let residuals = SimCodeUtil.sortEqSystems(nls.eqs) |> e => (equationInfo1(e,preBuf,eqnsDefines,reverseProf) + ',<%\n%>')
      let jac = match nls.jacobianMatrix case SOME(mat) then equationInfoMatrix(mat,preBuf,eqnsDefines,reverseProf)
      let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%nls.index%> = (const VAR_INFO**)malloc(<%listLength(nls.crefs)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%nls.crefs|>cr hasindex i0 => 'equationInfo_crefs<%nls.index%>[<%i0%>] = &<%cref(cr)%>__varInfo;'; separator="\n"%>;'
      '<%residuals%>{<%nls.index%>,"residualFunc<%nls.index%> (size <%listLength(nls.crefs)%>)", <%listLength(nls.crefs)%>, equationInfo_crefs<%nls.index%>}<%if jac then ',<%\n%><%jac%>'%>'
    // dynamic tearing
    case SES_LINEAR(lSystem=ls as LINEARSYSTEM(__), alternativeTearing = SOME(at as LINEARSYSTEM(__))) then
      // for strict tearing set
      let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%ls.index%> = (const VAR_INFO**)malloc(<%listLength(ls.vars)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%ls.vars|>var hasindex i0 => 'equationInfo_crefs<%ls.index%>[<%i0%>] = &<%cref(varName(var))%>__varInfo;'; separator="\n"%>;'
      // for casual tearing set
      // let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%at.index%> = (const VAR_INFO**)malloc(<%listLength(at.vars)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%at.vars|>var hasindex i0 => 'equationInfo_crefs<%at.index%>[<%i0%>] = &<%cref(varName(var))%>__varInfo;'; separator="\n"%>;'
      <<
        {<%ls.index%>,"linear system <%ls.index%> (size <%listLength(ls.vars)%>)", <%listLength(ls.vars)%>, equationInfo_crefs<%ls.index%>}
        {<%at.index%>,"linear system <%at.index%> (size <%listLength(at.vars)%>)", <%listLength(at.vars)%>, equationInfo_crefs<%at.index%>}
      >>
    case SES_NONLINEAR(nlSystem=nls as NONLINEARSYSTEM(__), alternativeTearing = SOME(at as NONLINEARSYSTEM(__))) then
      // for strict tearing set
      let residuals = SimCodeUtil.sortEqSystems(nls.eqs) |> e => (equationInfo1(e,preBuf,eqnsDefines,reverseProf) + ',<%\n%>')
      let jac = match nls.jacobianMatrix case SOME(mat) then equationInfoMatrix(mat,preBuf,eqnsDefines,reverseProf)
      let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%nls.index%> = (const VAR_INFO**)malloc(<%listLength(nls.crefs)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%nls.crefs|>cr hasindex i0 => 'equationInfo_crefs<%nls.index%>[<%i0%>] = &<%cref(cr)%>__varInfo;'; separator="\n"%>;'
      // for casual tearing set
      let residuals2 = SimCodeUtil.sortEqSystems(at.eqs) |> e => (equationInfo1(e,preBuf,eqnsDefines,reverseProf) + ',<%\n%>')
      let jac2 = match at.jacobianMatrix case SOME(mat) then equationInfoMatrix(mat,preBuf,eqnsDefines,reverseProf)
      // let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += 'const VAR_INFO** equationInfo_crefs<%at.index%> = (const VAR_INFO**)malloc(<%listLength(at.crefs)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%at.crefs|>cr hasindex i0 => 'equationInfo_crefs<%at.index%>[<%i0%>] = &<%cref(cr)%>__varInfo;'; separator="\n"%>;'
      <<
        <%residuals%>{<%nls.index%>,"residualFunc<%nls.index%> (size <%listLength(nls.crefs)%>)", <%listLength(nls.crefs)%>, equationInfo_crefs<%nls.index%>}<%if jac then ',<%\n%><%jac%>'%>
        <%residuals2%>{<%at.index%>,"residualFunc<%at.index%> (size <%listLength(at.crefs)%>)", <%listLength(at.crefs)%>, equationInfo_crefs<%at.index%>}<%if jac2 then ',<%\n%><%jac2%>'%>
      >>
    case SES_MIXED(__) then
      let conEqn = equationInfo1(cont,preBuf,eqnsDefines,reverseProf)
      let &eqnsDefines += functionSimProfDef(eq,System.tmpTick(),reverseProf)
      let &preBuf += '<%\n%>const VAR_INFO** equationInfo_crefs<%index%> = (const VAR_INFO**)malloc(<%listLength(discVars)%>*sizeof(VAR_INFO*));<%\n%>'
      let &preBuf += '<%discVars|>var hasindex i0 => 'equationInfo_crefs<%index%>[<%i0%>] = &<%cref(varName(var))%>__varInfo;'; separator="\n"%>;'
      '<%conEqn%>,<%\n%>{<%index%>,"MIXED<%index%>", <%listLength(discVars)%>, equationInfo_crefs<%index%>}'
    else '<%error(sourceInfo(), 'Unkown Equation Type in equationInfo1')%>'
end equationInfo1;

template equationInfoMatrix(JacobianMatrix jacobianMatrix, Text &preBuf, Text &eqnsDefines, Text &reverseProf)
::=
  match jacobianMatrix case (cols,_,_,_,_,_,_) then (cols |> (eqs,_,_) => (eqs |> eq => equationInfo1(eq,preBuf,eqnsDefines,reverseProf) ; separator = ',<%\n%>') ; separator = ',<%\n%>')
end equationInfoMatrix;

template equationInfo(list<SimEqSystem> eqs, list<StateSet> stateSets, Text &eqnsDefines, Text &reverseProf, Integer numEquations)
::=
  let() = System.tmpTickReset(0)
  match eqs
    case {} then "const struct EQUATION_INFO equation_info[1] = {{0, NULL}};"
    else
      let &preBuf = buffer ""
      let res =
        <<
        const struct EQUATION_INFO equationInfo[<%numEquations%>] = {
          {0, "Dummy Equation so we can index from 1", 0, NULL},
          <% listReverse(stateSets) |> st as SES_STATESET(__) =>
            '{<%index%>,"SES_STATESET <%index%>",0,NULL},<%\n%><%equationInfoMatrix(jacobianMatrix,preBuf,eqnsDefines,reverseProf)%>,<%\n%>'
          %>
          <% eqs |> eq => equationInfo1(eq,preBuf,eqnsDefines,reverseProf) ; separator=",\n"%>
        };
        /* Verify the data in the array to make sure certain assumptions hold */
        int i;
        for (i=0; i<<%numEquations%>; i++) {
          if (equationInfo[i].id != i) {
            fprintf(stderr, "equationInfo[i].id=%d, i=%d\n", equationInfo[i].id, i);
            assert(equationInfo[i].id == i);
          }
        }
        >>
      <<
      <%preBuf%>
      <%res%>
      >>
end equationInfo;

template ModelVariables(ModelInfo modelInfo)
 "Generates code for ModelVariables file for FMU target."
::=
  match modelInfo
    case MODELINFO(vars=SIMVARS(__),varInfo=VARINFO(numAlgVars= numAlgVars, numDiscreteReal = numDiscreteReal, numOptimizeConstraints = numOptimizeConstraints)) then
      <<
      <ModelVariables>
      <%System.tmpTickReset(1000)%>

      <%vars.stateVars       |> var hasindex i0 => ScalarVariable(var,i0,"rSta") ;separator="\n";empty%>
      <%vars.derivativeVars  |> var hasindex i0 => ScalarVariable(var,i0,"rDer") ;separator="\n";empty%>
      <%vars.algVars         |> var hasindex i0 => ScalarVariable(var,i0,"rAlg") ;separator="\n";empty%>
      <%vars.discreteAlgVars |> var hasindex i0 => ScalarVariable(var,intAdd(i0,numAlgVars),"rAlg") ;separator="\n";empty%>
      <%vars.realOptimizeConstraintsVars
                             |> var hasindex i0 => ScalarVariable(var,intAdd(i0,intAdd(numAlgVars,numDiscreteReal)),"rAlg") ;separator="\n";empty%>
      <%vars.realOptimizeFinalConstraintsVars
                             |> var hasindex i0 => ScalarVariable(var,intAdd(i0,intAdd(intAdd(numAlgVars,numOptimizeConstraints),numDiscreteReal)),"rAlg") ;separator="\n";empty%>
      <%vars.paramVars       |> var hasindex i0 => ScalarVariable(var,i0,"rPar") ;separator="\n";empty%>
      <%vars.aliasVars       |> var hasindex i0 => ScalarVariable(var,i0,"rAli") ;separator="\n";empty%>

      <%vars.intAlgVars      |> var hasindex i0 => ScalarVariable(var,i0,"iAlg") ;separator="\n";empty%>
      <%vars.intParamVars    |> var hasindex i0 => ScalarVariable(var,i0,"iPar") ;separator="\n";empty%>
      <%vars.intAliasVars    |> var hasindex i0 => ScalarVariable(var,i0,"iAli") ;separator="\n";empty%>

      <%vars.boolAlgVars     |> var hasindex i0 => ScalarVariable(var,i0,"bAlg") ;separator="\n";empty%>
      <%vars.boolParamVars   |> var hasindex i0 => ScalarVariable(var,i0,"bPar") ;separator="\n";empty%>
      <%vars.boolAliasVars   |> var hasindex i0 => ScalarVariable(var,i0,"bAli") ;separator="\n";empty%>

      <%vars.stringAlgVars   |> var hasindex i0 => ScalarVariable(var,i0,"sAlg") ;separator="\n";empty%>
      <%vars.stringParamVars |> var hasindex i0 => ScalarVariable(var,i0,"sPar") ;separator="\n";empty%>
      <%vars.stringAliasVars |> var hasindex i0 => ScalarVariable(var,i0,"sAli") ;separator="\n";empty%>
      </ModelVariables>
      >>
end ModelVariables;

template ScalarVariable(SimVar simVar, Integer classIndex, String classType)
 "Generates code for ScalarVariable file for FMU target."
::=
  match simVar
    case SIMVAR(__) then
      <<
      <ScalarVariable
      <%ScalarVariableAttribute(simVar, classIndex, classType)%>>
      <%ScalarVariableType(unit, displayUnit, minValue, maxValue, initialValue, nominalValue, isFixed, type_)%>
      </ScalarVariable>
      >>
end ScalarVariable;

template ScalarVariableAttribute(SimVar simVar, Integer classIndex, String classType)
 "Generates code for ScalarVariable Attribute file for FMU target."
::=
  match simVar
    case SIMVAR(source=SOURCE(info=info)) then
      let valueReference = '<%System.tmpTick()%>'
      let variability = getVariablity(varKind)
      let description = if comment then 'description = "<%Util.escapeModelicaStringToXmlString(comment)%>"'
      let alias = getAliasVar(aliasvar)
      let caus = getCausality(causality)
      let inputIndex = getInputIndexXml(simVar)
      <<
      name = "<%Util.escapeModelicaStringToXmlString(crefStrNoUnderscore(name))%>"
      valueReference = "<%valueReference%>"
      <%description%>
      variability = "<%variability%>" isDiscrete = "<%isDiscrete%>"
      causality = "<%caus%>"<%inputIndex%> isValueChangeable = "<%isValueChangeable%>"
      alias = <%alias%>
      classIndex = "<%classIndex%>" classType = "<%classType%>"
      isProtected = "<%isProtected%>" hideResult = "<%hideResult%>"
      <%getInfoArgs(info)%>
      >>
end ScalarVariableAttribute;

template getInputIndexXml(SimVar simVar)
::=
  match SimCodeUtil.getInputIndex(simVar)
    case -1 then ""
    case ix then ' inputIndex="<%ix%>"'
end getInputIndexXml;

template getInfoArgs(builtin.SourceInfo info)
::=
  match info
    case SOURCEINFO(__) then 'fileName = "<%Util.escapeModelicaStringToXmlString(fileName)%>" startLine = "<%lineNumberStart%>" startColumn = "<%columnNumberStart%>" endLine = "<%lineNumberEnd%>" endColumn = "<%columnNumberEnd%>" fileWritable = "<%if isReadOnly then false else true%>"'
end getInfoArgs;

template getCausality(Causality c)
 "Returns the Causality Attribute of ScalarVariable."
::=
  match c
    case NONECAUS(__) then "none"
    case INTERNAL(__) then "internal"
    case OUTPUT(__) then "output"
    case INPUT(__) then "input"
end getCausality;

/*****************************************************************************
 *         SECTION: GENERATE OPTIMIZATION IN SIMULATION FILE
 *****************************************************************************/

template optimizationComponents( list<DAE.ClassAttributes> classAttributes ,SimCode simCode, String modelNamePrefixStr)
  "Generates C for Objective Functions."
::=
    match classAttributes
    case{} then
        <<
        int <%symbolName(modelNamePrefixStr,"mayer")%>(DATA* data, modelica_real** res,short *i){return -1;}
        int <%symbolName(modelNamePrefixStr,"lagrange")%>(DATA* data, modelica_real** res, short * i1, short*i2){return -1;}
        int <%symbolName(modelNamePrefixStr,"pickUpBoundsForInputsInOptimization")%>(DATA* data, modelica_real* min, modelica_real* max, modelica_real*nominal, modelica_boolean *useNominal, char ** name, modelica_real * start, modelica_real * startTimeOpt){return -1;}
        int <%symbolName(modelNamePrefixStr,"setInputData")%>(DATA *data, const modelica_boolean file){return -1;}
        int <%symbolName(modelNamePrefixStr,"getTimeGrid")%>(DATA *data, modelica_integer * nsi, modelica_real**t){return -1;}
        >>
      else
        (classAttributes |> classAttribute => optimizationComponents1(classAttribute,simCode, modelNamePrefixStr); separator="\n")
end optimizationComponents;

template optimizationComponents1(ClassAttributes classAttribute, SimCode simCode, String modelNamePrefixStr)
"Generates C for class attributes of objective function."
::=
  match classAttribute
    case OPTIMIZATION_ATTRS(__) then
      let &varDecls = buffer ""
      let &preExp = buffer ""
      let &varDecls1 = buffer ""
      let &preExp1 = buffer ""
      let &auxFunction = buffer ""

      let objectiveFunction = match objetiveE
        case SOME(exp) then
          <<
          *res =  &$P$OMC$objectMayerTerm;
          #ifdef $P<%BackendDAE.optimizationMayerTermName%>$P$pDERC$PdummyVarC$indexdiffed
          *index_Dres = $P<%BackendDAE.optimizationMayerTermName%>$P$pDERC$PdummyVarC$indexdiffed;
          return 0;
          #endif
          >>

      let startTimeOpt = match startTimeE
        case SOME(exp) then
          let startTimeOptExp = daeExp(exp, contextOther, &preExp, &varDecls, &auxFunction)
          <<
          *startTimeOpt = <%startTimeOptExp%>;
          >>

      let objectiveIntegrand = match objectiveIntegrandE
        case SOME(exp) then
          <<
          *res =  &$P$OMC$objectLagrangeTerm;
          #ifdef $P<%BackendDAE.optimizationLagrangeTermName%>$P$pDERB$PdummyVarB$indexdiffed
          *index_DresB = $P<%BackendDAE.optimizationLagrangeTermName%>$P$pDERB$PdummyVarB$indexdiffed;
          *index_DresC = $P<%BackendDAE.optimizationLagrangeTermName%>$P$pDERC$PdummyVarC$indexdiffed;
          return 0;
          #endif
          >>

      let setInput = match simCode
        case simCode as SIMCODE(__) then
          match modelInfo
            case MODELINFO(vars=SIMVARS(__)) then
            <<
              if(file){
              <%vars.inputVars |> SIMVAR(varKind = OPT_LOOP_INPUT(replaceExp=cr)) hasindex i0 =>
              '<%cref(name)%> = <%cref(cr)%> ;'
              ;separator="\n"
              %>
              }
              <%vars.inputVars |> SIMVAR(__) hasindex i0 =>
              'data->simulationInfo->inputVars[<%i0%>] = <%cref(name)%>;'
              ;separator="\n"
              %>
            >>

      let getTG = match simCode
        case simCode as SIMCODE(__) then
          match modelInfo
            case MODELINFO(vars=SIMVARS(__)) then
            <<
              *nsi=(-1 <%vars.paramVars |> SIMVAR(varKind=OPT_TGRID(__)) hasindex i0 => '+1'
               ;separator=" "%>);
              *t = (modelica_real*) malloc((*nsi+1)*sizeof(modelica_real));
              <%vars.paramVars |> SIMVAR(varKind=OPT_TGRID(__)) hasindex i0 =>
              '(*t)[<%i0%>] = <%cref(name)%>;'
              ;separator="\n"
              %>
            >>

      let inputBounds = match simCode
        case simCode as SIMCODE(__) then
          match modelInfo
            case MODELINFO(vars=SIMVARS(__)) then
              <<
              <%vars.inputVars |> SIMVAR(__) hasindex i0 =>
              'min[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.min;<%\n%>max[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.max;<%\n%>nominal[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.nominal;<%\n%>useNominal[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.useNominal;<%\n%>name[<%i0%>] =(char *) <%cref(name)%>__varInfo.name;<%\n%>start[<%i0%>] = $P$ATTRIBUTE<%cref(name)%>.start;'
              ;separator="\n"%>
              >>
           <<
           <%auxFunction%>
           /* objectiveFunction */
           int <%symbolName(modelNamePrefixStr,"mayer")%>(DATA* data, modelica_real** res, short * index_Dres)
           {
             <%varDecls%>
             <%preExp%>
             <%objectiveFunction%>
             return  -1;
           }
           /* objectiveIntegrand */
           int <%symbolName(modelNamePrefixStr,"lagrange")%>(DATA* data, modelica_real** res, short * index_DresB, short *index_DresC)
           {
             <%varDecls1%>
             <%preExp1%>
             <%objectiveIntegrand%>
             return -1;
           }

           /* opt vars  */
           int <%symbolName(modelNamePrefixStr,"pickUpBoundsForInputsInOptimization")%>(DATA* data, modelica_real* min, modelica_real* max, modelica_real*nominal, modelica_boolean *useNominal, char ** name, modelica_real * start, modelica_real* startTimeOpt)
           {
             <%inputBounds%>
             *startTimeOpt = data->simulationInfo->startTime - 1.0;
             <%startTimeOpt%>
             return 0;
           }

           int <%symbolName(modelNamePrefixStr,"setInputData")%>(DATA *data, const modelica_boolean file)
           {
            TRACE_PUSH
            <%setInput%>
            TRACE_POP
            return 0;
           }
           int <%symbolName(modelNamePrefixStr,"getTimeGrid")%>(DATA *data, modelica_integer * nsi, modelica_real**t){
            <%getTG%>
            return 0;
           }
           >>
    else error(sourceInfo(), 'Unknown Constraint List')
end optimizationComponents1;

template functionXXX_systemPartial(list<SimEqSystem> derivativEquations, String name, Integer n, String modelNamePrefixStr, ModelInfo modelInfo)
::=
    let code =  match modelInfo
    case MODELINFO(vars=SIMVARS(derivativeVars=ders)) then
    (ders |> SIMVAR(__) hasindex i0 => equationNames_Partial(SimCodeUtil.computeDependencies(derivativEquations,name),modelNamePrefixStr,i0,crefStr(name)) ; separator="\n")
<<
static void <%modelNamePrefixStr%>_function<%name%><%n%>(DATA *data, threadData_t *threadData, int i)
{
  switch (i) {
  <%code%>
  }
}
>>
end functionXXX_systemPartial;


template functionXXX_systemsPartial(list<list<SimEqSystem>> eqs, String name, Text &loop, Text &varDecls, String modelNamePrefixStr, ModelInfo modelInfo)
::=
  let funcs = (eqs |> eq hasindex i0 fromindex 0 => functionXXX_systemPartial(eq,name,i0,modelNamePrefixStr,modelInfo) ; separator="\n")
  match listLength(eqs)
  case 0 then //empty case
    let &loop +=
        <<
        /* no <%name%> systems */
        >>
    ""
  case 1 then //1 function
    let &loop +=
        <<
        <%modelNamePrefixStr%>_function<%name%>0(data, threadData,i);
        >>
    funcs //just the one function
  case nFuncs then //2 and more
    let funcNames = eqs |> e hasindex i0 fromindex 0 => 'function<%name%>_system<%i0%>' ; separator=",\n"
    let head = if Flags.isSet(Flags.PARMODAUTO) then '#pragma omp parallel for private(id) schedule(<%match noProc() case 0 then "dynamic" else "static"%>)'
    let &varDecls += 'int id;<%\n%>'

    let &loop +=
      /* Text for the loop body that calls the equations */
      <<
      <%head%>
      for(id=0; id<<%nFuncs%>; id++) {
        function<%name%>_systems[id](data, threadData);
      }
      >>
    /* Text before the function head */
    <<
    <%funcs%>
    static void (*function<%name%>_systems[<%nFuncs%>])(DATA *, threadData_t *threadData) = {
      <%funcNames%>
    };
    >>
end functionXXX_systemsPartial;

template functionODEPartial(list<list<SimEqSystem>> derivativEquations, Text method, Option<tuple<Schedule,Schedule,Schedule>> hpcOmSchedules, String modelNamePrefix, ModelInfo modelInfo)
 "Generates function in simulation file."
::=
  let () = System.tmpTickReset(0)
  let &nrfuncs = buffer ""
  let &varDecls2 = buffer ""
  let &varDecls = buffer ""
  let &fncalls = buffer ""
  let systems = (functionXXX_systemsPartial(derivativEquations, "ODE_Partial", &fncalls, &varDecls, modelNamePrefix, modelInfo))
  let &tmp = buffer ""
  <<
  <%tmp%>
  <%systems%>

  void <%symbolName(modelNamePrefix,"functionODE_Partial")%>(DATA *data, threadData_t *threadData, int i)
  {
    TRACE_PUSH
    <% if profileFunctions() then "rt_tick(SIM_TIMER_FUNCTION_ODE);" %>

    <%varDecls%>

    //data->simulationInfo->callStatistics.functionODE++;

    <%fncalls%>

    TRACE_POP
  }
  >>
end functionODEPartial;

template equationNames_Partial(list<SimEqSystem> eqs, String modelNamePrefixStr, Integer i0, String cref_der)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  let odeEqs = eqs |> eq => equationNames_(eq,contextSimulationNonDiscrete,modelNamePrefixStr); separator="\n"
  <<
  case <%i0%>:
    // Assigning <%cref_der%>
    <%odeEqs%>
    break;
  >>
end equationNames_Partial;

annotation(__OpenModelica_Interface="backend");
end CodegenC;

// vim: filetype=susan sw=2 sts=2
