/*
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-2010, Linköpings University,
 * Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF THIS OSMC PUBLIC
 * LICENSE (OSMC-PL). ANY USE, REPRODUCTION OR DISTRIBUTION OF
 * THIS PROGRAM CONSTITUTES RECIPIENT'S ACCEPTANCE OF THE OSMC
 * PUBLIC LICENSE.
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from Linköpings University, either from the above address,
 * from the URL: http://www.ida.liu.se/projects/OpenModelica
 * and in the OpenModelica distribution.
 *
 * This program is distributed  WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS
 * OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 * For more information about the Qt-library visit TrollTech's webpage
 * regarding the Qt licence: http://www.trolltech.com/products/qt/licensing.html
 */

/*!
* \file VisualizeWindow.cpp
* \author Peter Arvidsson
* \date 2011
*/

#include "VisualizeWindow.h"

using namespace OMVisualize;

VisualizeWindow::VisualizeWindow(QStringList arguments, QWidget *parent)
    : QWidget(parent)
{
    mMaxZoomOut = -100;//Sets zoom range
    mMaxSpeed = 100;//Sets maximum speed
    mPausePressed = true;

    // reads the visualization file
    readVisualizationData(arguments);
    // reads the simulation file
    readSimulationData(arguments);
    mpOSGQGLGraphicsView = new osg::QGLGraphicsView;
    mpOSGQGLGraphicsScene = new osg::QOSGScene;
    mpTrackBall = new osgGA::TrackballManipulator;

    mpOSGQGLGraphicsView->setScene(mpOSGQGLGraphicsScene);
    mpOSGQGLGraphicsScene->setCameraManipulator(mpTrackBall);
    mpOSGQGLGraphicsScene->setSceneData(CreateScene());


    mpStartTimeLabel = new QLabel;
    mpStartTimeLabel->setNum(mStartTime);
    mpStopTimeLabel = new QLabel;
    mpStopTimeLabel->setNum(mStopTime);
    mpNowTime = new QLabel;
    QLabel *progressText = new QLabel;
    progressText->setText("Progress:");
    QLabel *startTimeText = new QLabel;
    startTimeText->setText("Start time:");
    QLabel *endTimeText = new QLabel;
    endTimeText->setText("End time:");
    QLabel *speedMultiplierText = new QLabel;
    speedMultiplierText->setText("Speed:");
    mpSpeed = new QLabel;
    QLabel *zoomText = new QLabel;
    zoomText->setText("Zoom:");
    mpZoom = new QLabel;

    mpPlayButton = new QPushButton;
    mpRestartButton = new QPushButton;
    mpNormalSpeedButton = new QPushButton;
    mpNormalZoomButton = new QPushButton;

    mpSliderTimer = new QTimer();

    mpProgressBar = new QProgressBar;//Create progressbar
    mpProgressBar->setRange(mStartTime, mStopTime*mNormalizeSlider);

    mpTimeSlider = new QSlider;//Create time slider
    mpTimeSlider->setOrientation(Qt::Horizontal);
    mpTimeSlider->setTickPosition(QSlider::NoTicks);
    mpTimeSlider->setRange(mStartTime, mStopTime*mNormalizeSlider);
    mpTimeSlider->setSliderPosition(0);

    mpSpeedSlider = new QSlider;//Create speed slider
    mpSpeedSlider->setOrientation(Qt::Vertical);
    mpSpeedSlider->setTickPosition(QSlider::NoTicks);
    mpSpeedSlider->setRange(0,mMaxSpeed);
    mpSpeedSlider->setSliderPosition(10);
    mpSpeed->setNum(mpSpeedSlider->value()/10);

    mpZoomSlider = new QSlider;//Create zoom slider
    mpZoomSlider->setOrientation(Qt::Vertical);
    mpZoomSlider->setTickPosition(QSlider::NoTicks);
    mpZoomSlider->setRange(mMaxZoomOut,0);
    mpZoomSlider->setSliderPosition(-(int)mpTrackBall->getDistance());
    mpZoom->setNum((int)mpTrackBall->getDistance());
    mHomePositionZoom = (double)mpTrackBall->getDistance();
    zoomButtonSlot();//To get the object in centre after get it to fit in window

    mpPlayButton->setText("PLAY");
    mpRestartButton->setText("RESTART");
    mpNormalSpeedButton->setText("SPEED 1");
    mpNormalZoomButton->setText("FIT TO WINDOW");
    //noHorizontal->setText("NO ROTATION");
    //noVertical->setText("NO ROTATION");
    mpTimeSlider->setValue(this->mStartTime);
    mpProgressBar->setValue(this->mStartTime);

    //Buttonlayout
    QGridLayout *gLayout = new QGridLayout; //Creation of button layout
    gLayout->addWidget(startTimeText, 1,4,1,1,Qt::AlignRight); //Adds the label that shows start time
    gLayout->addWidget(mpStartTimeLabel,1,5,1,1,Qt::AlignLeft); //Adds the label that shows start time for simulation (normaly 0)
    gLayout->addWidget(mpProgressBar,2,1,1,6,0); //Adds the progressbar
    gLayout->addWidget(endTimeText, 1,6,1,1,Qt::AlignRight); //Adds the label that shows end time
    gLayout->addWidget(mpStopTimeLabel,1,7,1,1,Qt::AlignLeft); //Adds the label that shows stop time for animation
    gLayout->addWidget(mpPlayButton,5,1,1,2,0); //Adds the playbutton
    gLayout->addWidget(mpRestartButton,5,3,1,2,0); //Adds the restart button
    gLayout->addWidget(mpTimeSlider,3,1,1,6,0); //Adds the slider
    gLayout->addWidget(progressText, 1,1,1,1,0); //Adds the label that shows progress
    gLayout->addWidget(mpNowTime, 1,2,1,1,0); //Adds the label that shows how far into the animation the widget are
    gLayout->addWidget(speedMultiplierText,1,8,1,1,Qt::AlignCenter);
    gLayout->addWidget(mpSpeed,1,8,1,1,Qt::AlignRight);
    gLayout->addWidget(mpSpeedSlider,2, 8, 3, 1, Qt::AlignCenter);
    gLayout->addWidget(mpNormalSpeedButton,5,8,1,1,Qt::AlignCenter); //Adds the speedbutton
    gLayout->addWidget(zoomText,1,10,1,1,Qt::AlignLeft);
    gLayout->addWidget(mpZoom,1,10,1,1,Qt::AlignCenter);
    gLayout->addWidget(mpZoomSlider,2, 10, 3, 1, Qt::AlignCenter);
    gLayout->addWidget(mpNormalZoomButton,5,10,1,1,Qt::AlignCenter); //Adds the zoombutton

    connect(mpPlayButton, SIGNAL(clicked()),this, SLOT(playSlot()));
    connect(mpRestartButton, SIGNAL(clicked()),this, SLOT(restartSlot()));
    connect(mpTimeSlider, SIGNAL(sliderMoved(int)), this, SLOT(slideChangeSlot(int)));
    connect(mpTimeSlider, SIGNAL(sliderReleased()), this, SLOT(slideReleaseSlot()));
    connect(mpSpeedSlider, SIGNAL(sliderReleased()), this, SLOT(speedSlideReleaseSlot()));
    connect(mpSpeedSlider, SIGNAL(sliderMoved(int)), this, SLOT(speedSlideChangeSlot(int)));
    connect(mpNormalSpeedButton, SIGNAL(clicked()),this, SLOT(speedButtonSlot()));
    connect(mpZoomSlider, SIGNAL(sliderMoved(int)), this, SLOT(zoomSlideChangeSlot(int)));
    connect(mpNormalZoomButton, SIGNAL(clicked()),this, SLOT(zoomButtonSlot()));
    connect(mpSliderTimer, SIGNAL(timeout()), this, SLOT(runningSlot()));

    QVBoxLayout *verticalLayout = new QVBoxLayout;
    verticalLayout->addWidget(mpOSGQGLGraphicsView);
    verticalLayout->addLayout(gLayout);
    setLayout(verticalLayout);
}

VisualizeWindow::~VisualizeWindow()
{

}

void VisualizeWindow::closeEvent(QCloseEvent *event)
{
    emit closingDown();
    event->accept();
}

/*! \brief Creates the scene. Reads the shapes, colors, sizes and positions from the created vectors and makes the animation from this data
 */
osg::Node* VisualizeWindow::CreateScene() {
    float angleX = 0, angleY = 0, angleZ = 0; //Rotation angles
    std::vector<path> vPath;
    vPath.resize(this->mObjVector.size());
    /*! \brief Vector that holds position at specific time in animation */
    osg::ref_ptr<osg::AnimationPath> RTemp;
    std::vector<animation> AnimationVector; /**< Holds all about the animation */

    //Create as many objects that is in the animation
    for (unsigned int i = 0; i < mObjVector.size(); i++)
    {
        AnimationVector.resize(AnimationVector.size()+1);
        //Check what shape the object shall have
        if (this->mObjVector[i].objectName == "ball")
        {
            AnimationVector[i].shape = new osg::Sphere(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX);
        }
        else if (this->mObjVector[i].shape == "box")
        {
            AnimationVector[i].shape = new osg::Box(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX, this->mObjVector[i].objSizeZ, this->mObjVector[i].objSizeY);
        }
        else if (this->mObjVector[i].shape == "cube")
        {
            AnimationVector[i].shape = new osg::Box(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX, this->mObjVector[i].objSizeZ, this->mObjVector[i].objSizeY);
        }
        else if (this->mObjVector[i].shape == "capsule")
        {
            AnimationVector[i].shape = new osg::Capsule(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX, this->mObjVector[i].objSizeY);
        }
        else if (this->mObjVector[i].shape == "cone")
        {
            AnimationVector[i].shape = new osg::Cone(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX, this->mObjVector[i].objSizeY);
        }
        else if (this->mObjVector[i].shape == "cylinder")
        {
            AnimationVector[i].shape = new osg::Cylinder(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX, this->mObjVector[i].objSizeY);
        }
        else if (this->mObjVector[i].shape == "sphere")
        {
            AnimationVector[i].shape = new osg::Sphere(osg::Vec3(this->mObjVector[i].offsetX, this->mObjVector[i].offsetZ, this->mObjVector[i].offsetY), this->mObjVector[i].objSizeX);
        }
        else //No accepted type.
        {
            //exit(1);
        }
        AnimationVector[i].shapeDrawable = new osg::ShapeDrawable(AnimationVector[i].shape);

        //Check what color the object shall have
        if (this->mObjVector[i].color == "blue")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(0.0f, 0.0f, 1.0f, 1.0f));
        }
        else if (this->mObjVector[i].color == "red")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(1.0f, 0.0f, 0.0f, 1.0f));
        }
        else if (this->mObjVector[i].color == "green")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(0.0f, 1.0f, 0.0f, 1.0f));
        }
        else if (this->mObjVector[i].color == "yellow")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(1.0f, 1.0f, 0.0f, 1.0f));
        }
        else if (this->mObjVector[i].color == "black")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(0.0f, 0.0f, 0.0f, 1.0f));
        }
        else if (this->mObjVector[i].color == "white")
        {
            AnimationVector[i].shapeDrawable->setColor(osg::Vec4(1.0f, 1.0f, 1.0f, 1.0f));
        }
        else //No accepted color
        {
            //exit(6);
        }

        AnimationVector[i].shapeGeode = new osg::Geode();


        //Animation PATH
        AnimationVector[i].mpTrans = new osg::Transform;
        AnimationVector[i].pMatTrans = new osg::PositionAttitudeTransform;
        AnimationVector[i].pMatTrans->addChild( AnimationVector[i].shapeGeode);

        RTemp = new osg::AnimationPath;
        RTemp->setLoopMode( osg::AnimationPath::NO_LOOPING); //LOOP,SWING,NO_LOOPING


        for (unsigned int j=0; j < this->mAnimVectors[0].vect.size(); j=j+4)
        {
            osg::AnimationPath::ControlPoint c0;

            if(this->mObjVector[i].sVisual == "PositionRotationSize")
            {
                osg::Vec3f A = osg::Vec3f(this->mAnimVectors[i].vect[j+1], this->mAnimVectors[i].vect[j+2], this->mAnimVectors[i].vect[j+3]);//Vector pointing to
                osg::Vec3f B = osg::Vec3f(this->mAnimVectorsB[i].vect[j+1], this->mAnimVectorsB[i].vect[j+2], this->mAnimVectorsB[i].vect[j+3]);//Vector pointing from

                osg::Vec3f C = A-B;
                C = C /C.length(); //Normalize the resulting vector
                //This can be wrong, if Z-axis is used it behave strange
                angleX = std::atan2(C.z(), C.y());
                angleY = std::atan2(C.x(),C.y());
                //angleZ // Can't find any use for this angle for now

                c0.setRotation(osg::Quat(angleX, osg::Vec3d(1.0, 0.0, 0.0),
                                         angleY, osg::Vec3d(0.0, 1.0, 0.0), angleZ, osg::Vec3d(0.0, 0.0, 1.0)));

                c0.setPosition(osg::Vec3(this->mAnimVectors[i].vect[j+1],this->mAnimVectors[i].vect[j+3],this->mAnimVectors[i].vect[j+2]));
            }
            else if(this->mObjVector[i].sVisual == "PositionRotationSizeOffset")
            {
                osg::Vec3f A = osg::Vec3f(this->mAnimVectors[i].vect[j+1], this->mAnimVectors[i].vect[j+2], this->mAnimVectors[i].vect[j+3]);//Vector pointing to
                osg::Vec3f B = osg::Vec3f(this->mAnimVectorsB[i].vect[j+1], this->mAnimVectorsB[i].vect[j+2], this->mAnimVectorsB[i].vect[j+3]);//Vector pointing from

                osg::Vec3f C = A-B;
                C = C /C.length(); //Normalize the resulting vector
                //This can be wrong, if Z-axis is used it behave strange
                angleX = std::atan2(C.z(), C.y());
                angleY = std::atan2(C.x(),C.y());
                //angleZ // Can't find any use for this angle for now

                c0.setRotation(osg::Quat(angleX, osg::Vec3d(1.0, 0.0, 0.0),
                                         angleY, osg::Vec3d(0.0, 1.0, 0.0),
                                         angleZ, osg::Vec3d(0.0, 0.0, 1.0)));

                c0.setPosition(osg::Vec3(this->mAnimVectors[i].vect[j+1],this->mAnimVectors[i].vect[j+3],this->mAnimVectors[i].vect[j+2]));
            }
            else if (this->mObjVector[i].sVisual == "PositionSize")
            {
                c0.setPosition(osg::Vec3(this->mAnimVectors[i].vect[j+1],this->mAnimVectors[i].vect[j+3],this->mAnimVectors[i].vect[j+2]));
            }
            else //Wrong simple visual type
            {
                throw VisualizeException("Error: Wrong simple visual type");
            }
            RTemp->insert( this->mAnimVectors[i].vect[j], c0 ); //Stupid but it seems to be needed
            vPath[i].vectorPath = RTemp;
            if(j==0)//To get all objects in right start position
            {
                AnimationVector[i].pMatTrans->setPosition(c0.getPosition());
                AnimationVector[i].pMatTrans->setAttitude(c0.getRotation().inverse());
            }
        }
    }
    this->mStartTime = vPath[0].vectorPath->getFirstTime();
    this->mStopTime = vPath[0].vectorPath->getLastTime();

    for(unsigned int m = 0; m < this->mObjVector.size(); m++) //Filling up with animation data
    {
        this->mCallBack.resize(this->mCallBack.size() + 1);
        this->mCallBack[m].rAniCallback = new osg::AnimationPathCallback( vPath[m].vectorPath.get() );
        this->mCallBack[m].rAniCallback->AnimationPathCallback::setPause(true);
        AnimationVector[m].pMatTrans->setUpdateCallback(this->mCallBack[m].rAniCallback.get());
        AnimationVector[m].shapeGeode->addDrawable(AnimationVector[m].shapeDrawable);
        AnimationVector[0].mpTrans->addChild(AnimationVector[m].pMatTrans);
    }
    return AnimationVector[0].mpTrans;
}

/*! brief Reads the .visualize file
  */
void VisualizeWindow::readVisualizationData(QStringList arguments)
{
    obj aObj; //Contains data about an object
    QString currentLine;
    QStringList values, tempList;
    if (arguments.size() < 2)
    {
        throw VisualizeException("Error: No visualization filename given");
    }
    QFile inFile(arguments[1]);
    if (!inFile.exists())
        throw VisualizeException("Error: Visualize file does not exist");
    inFile.open(QIODevice::ReadOnly);
    QTextStream *textStream = new QTextStream(&inFile);

    while (!textStream->atEnd())
    {
        currentLine = textStream->readLine();
        if (currentLine.isEmpty())
          continue;
        currentLine.remove('"');
        currentLine.remove("color=");
        currentLine.remove("shape=");
        currentLine.remove(":SimpleVisual.");
        tempList = currentLine.split(";");
        values = tempList[0].split(",");
        tempList.removeAt(0);
        values.append(tempList);
        aObj.objectName = values.at(0);
        aObj.color = values.at(1);
        aObj.shape = values.at(2);
        aObj.sVisual = values.at(3);
        this->mObjVector.push_back(aObj); //Putting everything together to the object
    }
}

/*! brief Check what kind of file that is input
  */
void VisualizeWindow::readSimulationData(QStringList arguments)
{
    if (arguments.size() < 3)
    {
        throw VisualizeException("Error: No simulation filename given");
    }

    QFileInfo simulationFileInfo(arguments.at(2));
    if (simulationFileInfo.suffix().compare("mat") == 0)
    {
        readMatFile(arguments.at(2));
    }
    else if (simulationFileInfo.suffix().compare("plt") == 0)
    {
        readPltFile(arguments.at(2));
    }
    else if (simulationFileInfo.suffix().compare("csv") == 0)
    {
        readCsvFile(arguments.at(2));
    }
}

/*! brief Reads the .mat file to be shown
  */
void VisualizeWindow::readMatFile(QString fileName)
{
    std::vector<double> numVector, numVectorB, timeVector, yVector, xVector, zVector, yVectorB, xVectorB, zVectorB;
    ModelicaMatVariable_t *MMV = new ModelicaMatVariable_t;
    const char* read;
    double *returned;
    QString qTemp;
    QStringList variablesList;
    ModelicaMatReader *matReader = new ModelicaMatReader;
    QFile inFile(fileName);
    if (!inFile.exists())
        throw VisualizeException("Error: File does not exist");

    //Filling up vector with the interesting variable names
    variablesList.append("time");
    variablesList.append(".frame_a[1]");
    variablesList.append(".frame_a[2]");
    variablesList.append(".frame_a[3]");
    variablesList.append(".frame_b[1]");
    variablesList.append(".frame_b[2]");
    variablesList.append(".frame_b[3]");
    variablesList.append(".size[1]");
    variablesList.append(".size[2]");
    variablesList.append(".size[3]");
    variablesList.append(".offset[1]");
    variablesList.append(".offset[2]");
    variablesList.append(".offset[3]");

    //Initialiseing MatReader

    read = omc_new_matlab4_reader(fileName.toStdString().c_str(), matReader);
    for(unsigned int i = 0; i< this->mObjVector.size(); i++)
    {
        //Reading TIME
        returned = omc_matlab4_read_vals(matReader, 1);//Time is always column 1
        for(unsigned int k = 0; k<matReader->nrows; k++)
        {
            timeVector.push_back(returned[k]);
        }

        //Reading the variables
        for(int variableNr = 1; variableNr <= 12; variableNr++)
        {
            qTemp.clear();
            qTemp = (this->mObjVector[i].objectName);
            qTemp.append(variablesList.at(variableNr));
            MMV = omc_matlab4_find_var(matReader, qTemp.toStdString().c_str());
            if(MMV==0)//The variable doesn't exist but can be correct. Some variables is not used all the time
            {
                if(variableNr == 4)
                {
                    xVectorB.push_back(0);//No frame_b
                }
                else if(variableNr == 5)
                {
                    yVectorB.push_back(0);//No frame_b
                }
                else if(variableNr == 6)
                {
                    zVectorB.push_back(0);//No frame_b
                }
                else if(variableNr == 10)
                {
                    this->mObjVector[i].offsetX = 0;//No offset
                }
                else if(variableNr == 11)
                {
                    this->mObjVector[i].offsetY = 0;//No offset
                }
                else if(variableNr == 12)
                {
                    this->mObjVector[i].offsetZ = 0;//No offset
                }
                else
                    throw VisualizeException("Error: Variable name fault");
            }
            else
            {
                returned = omc_matlab4_read_vals(matReader, MMV->index);//Something strange when more than two objects
                if(MMV->isParam == 1)//Could be strange results if not a single value
                {
                    double val;
                    if (omc_matlab4_val(&val,matReader,MMV,0.0))
                      throw VisualizeException("Parameter doesn't have a value : ");
                    returned[1] = val;
                }
                if(variableNr == 1)//frame_a[1]
                {
                    for(unsigned int k = 0; k<=matReader->nrows; k++)
                    {
                        xVector.push_back(returned[k]);
                    }
                }
                if(variableNr == 2)//frame_a[2]
                {
                    for(unsigned int k = 0; k<matReader->nrows; k++)
                    {
                        yVector.push_back(returned[k]);
                    }
                }
                if(variableNr == 3)//frame_a[3]
                {
                    for(unsigned int k = 0; k<matReader->nrows; k++)
                    {
                        zVector.push_back(returned[k]);
                    }
                }
                if(variableNr == 4)//frame_b[1]
                {
                    for(unsigned int k = 0; k<=matReader->nrows; k++)
                    {
                        xVectorB.push_back(returned[k]);
                    }
                }
                if(variableNr == 5)//frame_b[2]
                {
                    for(unsigned int k = 0; k<=matReader->nrows; k++)
                    {
                        yVectorB.push_back(returned[k]);
                    }
                }
                if(variableNr == 6)//frame_b[3]
                {
                    for(unsigned int k = 0; k<=matReader->nrows; k++)
                    {
                        zVectorB.push_back(returned[k]);
                    }
                }
                if(variableNr == 7)//size[1]
                    this->mObjVector[i].objSizeX = (float)returned[1];
                if(variableNr == 8)//size[2]
                    this->mObjVector[i].objSizeY = (float)returned[1];
                if(variableNr == 9)//size[3]
                    this->mObjVector[i].objSizeZ = (float)returned[1];
                if(variableNr == 10)//offset[1]
                    this->mObjVector[i].offsetX = (float)returned[1];
                if(variableNr == 11)//offset[2]
                    this->mObjVector[i].offsetY = (float)returned[1];
                if(variableNr == 12)//offset[3]
                    this->mObjVector[i].offsetZ = (float)returned[1];
            }
        }
        //Putting plotVector together
        numVector.clear();
        numVectorB.clear();
        for(unsigned int u=0;u<timeVector.size();u++)
        {
            numVector.push_back(timeVector[u]);
            numVector.push_back(xVector[u]);
            numVector.push_back(yVector[u]);
            numVector.push_back(zVector[u]);

            numVectorB.push_back(timeVector[u]);
            numVectorB.push_back(xVectorB[u]);
            numVectorB.push_back(yVectorB[u]);
            numVectorB.push_back(zVectorB[u]);
        }
        //Cleaning
        timeVector.clear();
        xVector.clear();
        yVector.clear();
        zVector.clear();
        xVectorB.clear();
        yVectorB.clear();
        zVectorB.clear();

        this->mAnimVectors.resize(this->mAnimVectors.size() + 1);
        this->mAnimVectorsB.resize(this->mAnimVectors.size());

        this->mAnimVectors[i].vect.swap(numVector);//Fill the class vectors for usage in other functions
        this->mAnimVectorsB[i].vect.swap(numVectorB);
        numVector.clear();
        numVectorB.clear();
    }
    omc_free_matlab4_reader(matReader);
}

/*! brief Reads the .plt file to be shown
  */
void VisualizeWindow::readPltFile(QString fileName)
{
    QString currentLine;
    QStringList values;
    QFile inFile(fileName);
    if (!inFile.exists())
        throw VisualizeException("Error: File does not exist");
    inFile.open(QIODevice::ReadOnly);
    QTextStream *textStream = new QTextStream(&inFile);
    std::vector<double> timeVector, yVector, xVector, zVector, yVectorB, xVectorB, zVectorB; //Vectors that will be filled with data from simulation file

    for (unsigned int nrObj = 0; nrObj < this->mObjVector.size(); nrObj++)//As many laps as there are objects
    {
        QStringList variablesPlotted;
        QStringList variablesList;
        this->mAnimVectors.resize(this->mAnimVectors.size() + 1);
        this->mAnimVectorsB.resize(this->mAnimVectors.size());
        QString qTemp = (this->mObjVector[nrObj].objectName);

        variablesList.append("time");
        variablesList.append(qTemp+".frame_a[1]");
        variablesList.append(qTemp+".frame_a[2]");
        variablesList.append(qTemp+".frame_a[3]");
        variablesList.append(qTemp+".frame_b[1]");
        variablesList.append(qTemp+".frame_b[2]");
        variablesList.append(qTemp+".frame_b[3]");
        variablesList.append(qTemp+".size[1]");
        variablesList.append(qTemp+".size[2]");
        variablesList.append(qTemp+".size[3]");
        variablesList.append(qTemp+".offset[1]");
        variablesList.append(qTemp+".offset[2]");
        variablesList.append(qTemp+".offset[3]");

        // read the interval size from the file
        int intervalSize = 0;
        textStream->seek(0);
        while (!textStream->atEnd())
        {
            currentLine = textStream->readLine();
            if (currentLine.startsWith("#IntervalSize"))
            {
                intervalSize = static_cast<QString>(currentLine.split("=").last()).toInt();
                break;
            }
        }
        // Read variable values and plot them
        while (!textStream->atEnd())
        {
            currentLine = textStream->readLine();
            QString currentVariable;
            if (currentLine.contains("DataSet:"))
            {
                currentVariable = currentLine.remove("DataSet: ");
                if (variablesList.contains(currentVariable))
                {
                    // read the variable values now
                    currentLine = textStream->readLine();
                    for(int j = 0; j < intervalSize; j++)
                    {
                        values = currentLine.split(",");

                        if(currentVariable == "time")
                            timeVector.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(1))
                            xVector.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(2))
                            yVector.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(3))
                            zVector.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(4))
                            xVectorB.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(5))
                            yVectorB.push_back(QString(values[1]).toDouble());
                        if(currentVariable == variablesList.at(6))
                            zVectorB.push_back(QString(values[1]).toDouble());
                        if((currentVariable == variablesList.at(7)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].objSizeX=QString(values[1]).toDouble();
                        if((currentVariable == variablesList.at(8)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].objSizeY=QString(values[1]).toDouble();
                        if((currentVariable == variablesList.at(9)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].objSizeZ=QString(values[1]).toDouble();
                        if((currentVariable == variablesList.at(10)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].offsetX=QString(values[1]).toDouble();
                        if((currentVariable == variablesList.at(11)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].offsetY=QString(values[1]).toDouble();
                        if((currentVariable == variablesList.at(12)) && (!variablesPlotted.contains(currentVariable)))//just read it once
                            this->mObjVector[nrObj].offsetZ=QString(values[1]).toDouble();

                        currentLine = textStream->readLine();
                    }
                variablesPlotted.append(currentVariable);
                }
            }
        }
        for(unsigned int i = 0; i< timeVector.size(); i++)
        {
            this->mAnimVectors[nrObj].vect.push_back(timeVector[i]);
            this->mAnimVectors[nrObj].vect.push_back(xVector[i]);
            this->mAnimVectors[nrObj].vect.push_back(yVector[i]);
            this->mAnimVectors[nrObj].vect.push_back(zVector[i]);
            this->mAnimVectorsB[nrObj].vect.push_back(1000);
            if(xVectorB.size()!=0)
            {
                this->mAnimVectorsB[nrObj].vect.push_back(xVectorB[i]);
                this->mAnimVectorsB[nrObj].vect.push_back(yVectorB[i]);
                this->mAnimVectorsB[nrObj].vect.push_back(zVectorB[i]);
            }
            else
            {
                this->mAnimVectorsB[nrObj].vect.push_back(0);
                this->mAnimVectorsB[nrObj].vect.push_back(0);
                this->mAnimVectorsB[nrObj].vect.push_back(0);
            }
        }
        timeVector.clear();
        xVector.clear();
        yVector.clear();
        zVector.clear();
        xVectorB.clear();
        yVectorB.clear();
        zVectorB.clear();
    }
    // close the file
    inFile.close();
    return;
}

/*! brief Reads the .csv file to be shown
  */
void VisualizeWindow::readCsvFile(QString fileName)
{
    QStringList variablesList;
    this->mAnimVectors.resize(this->mObjVector.size());
    this->mAnimVectorsB.resize(this->mObjVector.size());

    // open the file
    QFile inFile(fileName);
    if (!inFile.exists())
        throw VisualizeException("Error: File does not exist");
    inFile.open(QIODevice::ReadOnly);
    QTextStream *textStream = new QTextStream(&inFile);
    QString currentLine = textStream->readLine();
    currentLine.remove(QChar('"'));
    QStringList allVariablesInFile = currentLine.split(",");
    allVariablesInFile.removeLast();

    for (unsigned int nrObj = 0; nrObj < this->mObjVector.size(); nrObj++)
    {
        QString qTemp = (this->mObjVector[nrObj].objectName);
        variablesList.clear();
        variablesList.append("time");
        variablesList.append(qTemp+".frame_a[1]");
        variablesList.append(qTemp+".frame_a[2]");
        variablesList.append(qTemp+".frame_a[3]");
        variablesList.append(qTemp+".frame_b[1]");
        variablesList.append(qTemp+".frame_b[2]");
        variablesList.append(qTemp+".frame_b[3]");
        variablesList.append(qTemp+".size[1]");
        variablesList.append(qTemp+".size[2]");
        variablesList.append(qTemp+".size[3]");
        variablesList.append(qTemp+".offset[1]");
        variablesList.append(qTemp+".offset[2]");
        variablesList.append(qTemp+".offset[3]");
    }
    //Assign Values
    while(!textStream->atEnd())
    {
        currentLine = textStream->readLine();
        QStringList values = currentLine.split(",");
        values.removeLast();
        int r;
        for(unsigned int i = 0; i < this->mObjVector.size(); i++)
        {
            this->mAnimVectors[i].vect.push_back(values.at(allVariablesInFile.indexOf("time")).toDouble());
            this->mAnimVectors[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_a[1]")).toDouble());
            this->mAnimVectors[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_a[2]")).toDouble());
            this->mAnimVectors[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_a[3]")).toDouble());

            r = allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_b[1]");
            if (r != -1)//If no rotation is used
            {
                this->mAnimVectorsB[i].vect.push_back(values.at(allVariablesInFile.indexOf("time")).toDouble());
                this->mAnimVectorsB[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_b[1]")).toDouble());
                this->mAnimVectorsB[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_b[2]")).toDouble());
                this->mAnimVectorsB[i].vect.push_back(values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".frame_b[3]")).toDouble());
            }
            else//Rotation
            {
                this->mAnimVectorsB[i].vect.push_back(values.at(allVariablesInFile.indexOf("time")).toDouble());
                this->mAnimVectorsB[i].vect.push_back(0);
                this->mAnimVectorsB[i].vect.push_back(0);
                this->mAnimVectorsB[i].vect.push_back(0);

            }
            this->mObjVector[i].objSizeX = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".size[1]")).toFloat();
            this->mObjVector[i].objSizeY = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".size[2]")).toFloat();
            this->mObjVector[i].objSizeZ = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".size[3]")).toFloat();

            r = allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".offset[1]");
            if (r == -1)//If no offset is used
            {
                this->mObjVector[i].offsetX = 0;
                this->mObjVector[i].offsetY = 0;
                this->mObjVector[i].offsetZ = 0;
            }
            else
            {
                this->mObjVector[i].offsetX = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".offset[1]")).toDouble();
                this->mObjVector[i].offsetY = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".offset[2]")).toDouble();
                this->mObjVector[i].offsetZ = values.at(allVariablesInFile.indexOf((this->mObjVector[i].objectName)+".offset[3]")).toDouble();
            }
        }
    }
    // close the file
    inFile.close();
    return;
}


/*! \brief Handles how the play button behave
*/
void VisualizeWindow::playSlot()
{
    this->mPausePressedTime = this->mCallBack[0].rAniCallback->getAnimationTime();
    if (this->mPausePressed == TRUE)//Is paused
    {
        this->mPausePressed = FALSE;
        this->mpPlayButton->setText("PAUSE");
        for (unsigned int r = 0; r < this->mObjVector.size(); r++)
        {
            this->mCallBack[r].rAniCallback->AnimationPathCallback::setPause(FALSE);//Set animation run
        }
        this->mpSliderTimer->start();
    }
    else //Is running
    {
        this->mPausePressed = TRUE;
        this->mpPlayButton->setText("PLAY");

        for (unsigned int r = 0; r < this->mObjVector.size(); r++)
        {
            this->mCallBack[r].rAniCallback->AnimationPathCallback::setPause(TRUE);//Set animation pause
        }
        this->mpSliderTimer->stop();
    }
}

/*! \brief Handles restart button behaveour
  */
void VisualizeWindow::restartSlot()
{
    for (unsigned int r = 0; r < this->mObjVector.size(); r++)
    {
        this->mCallBack[r].rAniCallback->AnimationPathCallback::reset();//Reset animation
        this->mCallBack[r].rAniCallback->AnimationPathCallback::setPause(FALSE);//Start animation
        this->mpSliderTimer->start();//Slider set to start
        this->mProgressTime = 0;//No progress
        this->mpProgressBar->setValue(0);
        this->mpNowTime->setNum(0);//Where animation is, time = 0 on restart
        this->mpTimeSlider->setValue(0);
        this->mCallBack[r].rAniCallback->setTimeOffset(0);//Start animation from beginning
        this->mpPlayButton->setText("PAUSE");
        this->mPausePressed = FALSE;
    }
}

/*! \brief While running this updates widgets that change value during time
  */
void VisualizeWindow::runningSlot()
{
    mpZoomSlider->setSliderPosition(-(int)mpTrackBall->getDistance());
    mpZoom->setNum((int)mpTrackBall->getDistance());
    this->mProgressTime = this->mCallBack[0].rAniCallback->getAnimationTime();// + (this->mInterval/mNormalizeSlider);
    this->mpProgressBar->setValue(this->mProgressTime*mNormalizeSlider);
    this->mpTimeSlider->setValue(this->mProgressTime*mNormalizeSlider);
    QObject::connect(this->mpTimeSlider, SIGNAL(valueChanged(int)),this->mpProgressBar, SLOT(setValue(int)));
    if (this->mCallBack[0].rAniCallback->getAnimationTime() < 0)
    {
        this->mpNowTime->setNum(0);
    }
    else
    {
        this->mpNowTime->setNum(this->mCallBack[0].rAniCallback->getAnimationTime());
    }
    if (this->mCallBack[0].rAniCallback->getAnimationTime() > this->mStopTime)
    {
        this->mpNowTime->setNum(this->mStopTime);
    }
}

/*! \brief Handles time slideChangeSlot
  */
void VisualizeWindow::slideChangeSlot(int slideValue)
{
    this->slider = double(slideValue);//Set value
    for (unsigned int r = 0; r < this->mObjVector.size(); r++)
    {
        if (this->mPausePressed==FALSE)//If user change slider without have pressed pause before
        {
            for (unsigned int r = 0; r < this->mObjVector.size(); r++)
            {
                this->mCallBack[r].rAniCallback->AnimationPathCallback::setPause(TRUE);//Set animation pause
            }
            this->mPausePressed = TRUE;
            this->mpSliderTimer->stop();
        }
        this->mCallBack[r].rAniCallback->AnimationPathCallback::reset();//Reset the animation
        this->mCallBack[r].rAniCallback->setTimeOffset(-this->slider/mNormalizeSlider);//Set time offset. Is divided by 1000 to get normalization
        this->mpNowTime->setNum(this->slider/mNormalizeSlider);
    }
}
/*! \brief Handles time slideReleaseSlot
  */
void VisualizeWindow::slideReleaseSlot()
{

    for (unsigned int r = 0; r < this->mObjVector.size(); r++)
    {
        this->mpSliderTimer->start(this->slider/mNormalizeSlider);
        this->mCallBack[r].rAniCallback->AnimationPathCallback::setPause(TRUE);//Set animation pause
        this->mpSliderTimer->stop();
    }
}

/*! \brief Handles zoomSlideChangeSlot
  */
void VisualizeWindow::zoomSlideChangeSlot(int val)
{
    this->mpZoom->setNum(double(this->mpZoomSlider->value()));
    mpOSGQGLGraphicsScene->getCameraManipulator()->setHomePosition(osg::Vec3d(0.0, double(this->mpZoomSlider->value()), 0.0),
                                                                   osg::Vec3d(0.0, 0.0, 0.0),
                                                                   osg::Vec3d(0.0, 0.0, 1.0));
    mpOSGQGLGraphicsScene->home();
}

/*! \brief Handles zoomButtonSlot, sets zoomvalue to 0
  */
void VisualizeWindow::zoomButtonSlot()
{
    //this->mpZoom->setNum(0);
    this->mpZoomSlider->setSliderPosition(-mHomePositionZoom);
    this->mpZoom->setNum(-(int)mHomePositionZoom);
    mpOSGQGLGraphicsScene->getCameraManipulator()->setHomePosition(osg::Vec3d(0.0,  -mHomePositionZoom, 0.0),
                                                                   osg::Vec3d(0.0, 0.0, 0.0),
                                                                   osg::Vec3d(0.0, 0.0, 1.0));
    mpOSGQGLGraphicsScene->home();
}

/*! \brief Handles speedButtonSlot, sets speed to 1
  */
void VisualizeWindow::speedButtonSlot()
{
    for (unsigned int r = 0; r < this->mObjVector.size(); r++)
    {
        this->mCallBack[r].rAniCallback->setTimeMultiplier(1);
        this->mpSpeed->setNum(1);
        this->mpSpeedSlider->setSliderPosition(10);
    }
}

/*! \brief Handles speedSlideChangeSlot
  */
void VisualizeWindow::speedSlideChangeSlot(int val)
{
    this->mpSpeed->setNum(double(this->mpSpeedSlider->value())/10);
}

/*! \brief Handles speedSlideReleaseSlot, while release button it sets animationtime to 0 and sets the new time multiplier
  */
void VisualizeWindow::speedSlideReleaseSlot()
{
    for (unsigned int r = 0; r < this->mObjVector.size(); r++)
    {
        this->mCallBack[r].rAniCallback->AnimationPathCallback::reset();//Reset animation
        this->mCallBack[r].rAniCallback->setTimeOffset(0); //Start animation from beginning
        this->mCallBack[r].rAniCallback->setTimeMultiplier(double(this->mpSpeedSlider->value())/10);//Set speed
    }
}

void VisualizeWindow::receiveMessage(QStringList arguments)
{

}
