/*
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-2010, Linköpings University,
 * Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF THIS OSMC PUBLIC
 * LICENSE (OSMC-PL). ANY USE, REPRODUCTION OR DISTRIBUTION OF
 * THIS PROGRAM CONSTITUTES RECIPIENT'S ACCEPTANCE OF THE OSMC
 * PUBLIC LICENSE.
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from Linköpings University, either from the above address,
 * from the URL: http://www.ida.liu.se/projects/OpenModelica
 * and in the OpenModelica distribution.
 *
 * This program is distributed  WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS
 * OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 * For more information about the Qt-library visit TrollTech's webpage
 * regarding the Qt licence: http://www.trolltech.com/products/qt/licensing.html
 */

/*!
* \file VisualizeWindow.cpp
* \author Peter Arvidsson
* \date 2011
*/

#ifndef VISUALIZEWINDOW_H
#define VISUALIZEWINDOW_H

#include <QtGui>
#include <QtCore>
#include <QTextStream>

//STD c++ headers
#include <cmath>
#include <fstream>
#include <iostream>
#include <stdlib.h>
#include <vector>
#include <stdexcept>
#include "../../SimulationRuntime/c/util/read_matlab4.h"
#include "QOSGGraphics.h"

//OSG headers
#include <osgGA/TrackballManipulator>
#include <osg/MatrixTransform>
#include <osg/PositionAttitudeTransform>
#include <osg/ShapeDrawable>
#include <osgViewer/Viewer>
#include <osgViewer/ViewerEventHandlers>


namespace OMVisualize
{
class VisualizeWindow : public QWidget
{
  Q_OBJECT
private:
  /*! \brief Holds what distance the zoom have to have whole animation fit in window */
  double mHomePositionZoom;
  /*! \brief Pointer to the trackball that handle the camera position */
  osgGA::TrackballManipulator *mpTrackBall;
  /*! \brief Pointer to the mpOSGQGLGraphicsView */
  osg::QGLGraphicsView *mpOSGQGLGraphicsView;
  /*! \brief Pointer to the created QOSGScene */
  osg::QOSGScene *mpOSGQGLGraphicsScene;
  /*! \brief Int that holds how fast you can run your animation */
  int mMaxSpeed;
  /*! \brief Int that holds how far out from origo you can zoom out */
  int mMaxZoomOut;
  /*! \brief Int that holds 1000 to get smoother sliders without ticks */
  int mNormalizeSlider;
  /*! \brief Double value that holds the start time for the animation */
  double mStartTime;
  /*! \brief Is the label that shows the animation stop time */
  QLabel *mpStartTimeLabel;
  /*! \brief Double value that holds the stop time for the animation */
  double mStopTime;
  /*! \brief Is the label that shows the animation start time */
  QLabel *mpStopTimeLabel;
  /*! \brief Is the label that shows the animation current time */
  QLabel *mpNowTime;
  /*! \brief Is the time slider */
  QSlider *mpTimeSlider;
  /*! \brief Is the speed slider */
  QSlider *mpSpeedSlider;
  /*! \brief Is the speed slider */
  QSlider *mpZoomSlider;
  /*! \brief Is the label that shows animation speed */
  QLabel *mpSpeed;
  /*! \brief Is the label that shows the animation zoom */
  QLabel *mpZoom;
  /*! \brief Is used to send slider value between functions */
  double slider;
  /*! \brief Is used to remember time when pause pressed */
  double mPausePressedTime;
  /*! \brief Is used to remember if pause have been pressed or not */
  bool mPausePressed;
  /*! \brief Int value that holds the actual time in the animation */
  double mProgressTime;
  /*! \brief Pointer to the play button */
  QPushButton *mpPlayButton;
  /*! \brief Pointer to the restart button */
  QPushButton *mpRestartButton;
  /*! \brief Pointer to the speed button button */
  QPushButton *mpNormalSpeedButton;
  /*! \brief Pointer to the zoom button button */
  QPushButton *mpNormalZoomButton;
  /*! \brief Is used to show progress */
  QTimer *mpSliderTimer;
  /*! \brief Is the progress bar */
  QProgressBar *mpProgressBar;
  /*! \brief Description of one osg::AnimationPathCallback */
  struct animCallback{
      /*! \brief osg::ref_ptr<osg::AnimationPathCallback> */
      osg::ref_ptr<osg::AnimationPathCallback> rAniCallback;
  };
  /*! \brief Description of one osg::AnimationPath */
  struct path{
      /*! \brief osg::ref_ptr<osg::AnimationPath> */
      osg::ref_ptr<osg::AnimationPath> vectorPath;
  };
  /*! \brief Description of one animationobject */
  struct obj{
      /*! \brief Name of the object to animate */
      QString objectName;
      /*! \brief Shape of object. It is ok to use ball, box, capsule, cone, cylinder and sphere*/
      QString shape;
      /*! \brief Color of object. Colors that can be used is blue, red green, yellow, black and white */
      QString color;
      /*! \brief What kind of object is it.  */
      QString sVisual;
      /*! \brief Size in X coordinates */
      float objSizeX;
      /*! \brief Size in Y coordinates */
      float objSizeY;
      /*! \brief Size in Z coordinates */
      float objSizeZ;
      /*! \brief How the object is moved from zenit of animation in X-axis */
      float offsetX;
      /*! \brief How the object is moved from zenit of animation in Y-axis */
      float offsetY;
      /*! \brief How the object is moved from zenit of animation in Z-axis */
      float offsetZ;
  };
  /*! \brief Description of one std::vector for use in a vector matrix */
  struct v{
      std::vector<double> vect;
  };
  /*! \brief Description of one animation */
  struct animation{
      /*! \brief  osg::Shape* */
      osg::Shape *shape;
      /*! \brief osg::MatrixTransform* */
      //osg::MatrixTransform* pMatTrans;
      osg::PositionAttitudeTransform *pMatTrans;
      /*! \brief osg::Geode* */
      osg::Geode* shapeGeode;
      /*! \brief osg::ShapeDrawable* */
      osg::ShapeDrawable* shapeDrawable;
      /*! \brief osg::Transform* */
      osg::Transform* mpTrans;
  };
  /*! \brief Holds all objects and corresponding attributes */
  std::vector<obj> mObjVector;
  /*! \brief Vector that holds position Frame_a in animation */
  std::vector<v> mAnimVectors;
  /*! \brief Vector that holds position Frame_b in animation */
  std::vector<v> mAnimVectorsB;
  /*! \brief Vector that holds the data about the animated objects */
  std::vector<animCallback> mCallBack;
public:
  VisualizeWindow(QStringList arguments = QStringList(), QWidget *parent = 0);
  ~VisualizeWindow();
  void closeEvent(QCloseEvent *event);
  void receiveMessage(QStringList arguments);
  /*! \brief Function that creates the animation, filling it up with all data and makes it ready to run */
  osg::Node* CreateScene();
  void readVisualizationData(QStringList arguments);
  void readSimulationData(QStringList arguments);
  void readMatFile(QString fileName);
  void readPltFile(QString fileName);
  void readCsvFile(QString fileName);
public slots:
  /*! \brief Start animation */
  void playSlot();
  /*! \brief Resets the animation so it will start from beginning*/
  void restartSlot();
  /*! \brief Handles the zoom return button*/
  void zoomButtonSlot();
  /*! \brief Handles the reset speed button*/
  void speedButtonSlot();
  /*! \brief Handles the normal running of widget */
  void runningSlot();
  /*! \brief Handles the time slide release */
  void slideReleaseSlot();
  /*! \brief Handles the time slide change */
  void slideChangeSlot(int);
  /*! \brief Handles the speed slider release */
  void speedSlideReleaseSlot();
  /*! \brief Handles the speed slider release */
  void speedSlideChangeSlot(int);
  /*! \brief Handles the zoom slider change */
  void zoomSlideChangeSlot(int);
signals:
    void closingDown();
};

//Exception classes
class VisualizeException : public std::runtime_error
{
public:
  VisualizeException(const char *e) : std::runtime_error(e) {}
  VisualizeException(const QString str) : std::runtime_error(str.toStdString().c_str()) {}
};

class NoFileException : public VisualizeException
{
public:
  NoFileException(const char *fileName) : VisualizeException(fileName) {}
private:
  const char *temp;
};

class NoVariableException : public VisualizeException
{
public:
  NoVariableException(const char *varName) : VisualizeException(varName) {}
};

}
#endif // VISUALIZEWINDOW_H
