/*
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-2010, Linköpings University,
 * Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF THIS OSMC PUBLIC
 * LICENSE (OSMC-PL). ANY USE, REPRODUCTION OR DISTRIBUTION OF
 * THIS PROGRAM CONSTITUTES RECIPIENT'S ACCEPTANCE OF THE OSMC
 * PUBLIC LICENSE.
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from Linköpings University, either from the above address,
 * from the URL: http://www.ida.liu.se/projects/OpenModelica
 * and in the OpenModelica distribution.
 *
 * This program is distributed  WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS
 * OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 * For more information about the Qt-library visit TrollTech's webpage
 * regarding the Qt licence: http://www.trolltech.com/products/qt/licensing.html
 */

/*!
* \file main.cpp
* \author Peter Arvidsson
* \date 2011
*/

#include <QtGui/QApplication>
#include "VisualizeMainWindow.h"
#include "VisualizeApplication.h"

using namespace OMVisualize;

#define CONSUME_BOOL_ARG(i,n,var) { \
  if (0 == strcmp("true",argv[i]+n)) var = true; \
  else if (0 == strcmp("false",argv[i]+n)) var = false; \
  else {fprintf(stderr, "%s does not describe a boolean value\n", argv[i]);} \
  }

void printUsage()
{
  printf("Usage: OMVisualize [--visualizationfile=NAME] [--simulationfile=NAME]\n");
  printf("OPTIONS\n");
  printf("    --title=TITLE                    Sets the TITLE of the visualize window\n");
  printf("    --visualizationfile=NAME         Sets the NAME of the visualization file\n");
  printf("    --simulationfile=NAME            Sets the NAME of the simulation result file\n");
  printf("    --new-window=[true|false]        Create a MDI dialog in the visualize window\n");
}

/*! \brief Handles the main class
  */
int main(int argc, char *argv[])
{
  // read the arguments
  QStringList arguments;
  bool newApplication = false;
  QString title("Visualization by OpenModelica");
  QString visualizeFilename;
  QString simulationFilename;
  for(int i = 1; i < argc; i++)
  {
    if (strncmp(argv[i], "--visualizationfile=", 20) == 0) {
      visualizeFilename = argv[i]+20;
    } else if (strncmp(argv[i], "--simulationfile=", 17) == 0) {
      simulationFilename = argv[i]+17;
    } else if (strcmp(argv[i], "--help") == 0) {
      printUsage();
      return 1;
    } else if (strncmp(argv[i], "--title=", 7) == 0) {
      title = argv[i]+7;
    } else if (strncmp(argv[i], "--new-window=",13) == 0) {
      CONSUME_BOOL_ARG(i,13,newApplication);
    } else if (strncmp(argv[i], "--", 2) == 0) {
      fprintf(stderr, "Error: Unknown option: %s\n", argv[i]);
      return 1;
    }
  }
  if (visualizeFilename.length() == 0) {
    fprintf(stderr, "Error: No visualization filename given\n");
    printUsage();
    return 1;
  }
  if (simulationFilename.length() == 0) {
    fprintf(stderr, "Error: No simulation filename given\n");
    printUsage();
    return 1;
  }
  // Hack to get the expected format of VisualizeApplication. Yes, this is totally crazy :)
  arguments.append(argv[0]);
  arguments.append(visualizeFilename);
  arguments.append(simulationFilename);
  arguments.append(title);
  // create the visualize application object that is used to check that only one instance of application is running
  VisualizeApplication app(argc, argv, "OMVisualize");
  // create the visualize main window
  VisualizeMainWindow w;
  QObject::connect(&app, SIGNAL(messageAvailable(QStringList)), w.getVisualizeWindowContainer(), SLOT(updateCurrentWindow(QStringList)));
  QObject::connect(&app, SIGNAL(newApplicationLaunched(QStringList)), w.getVisualizeWindowContainer(), SLOT(addVisualizeWindow(QStringList)));
  try {
      if (!app.isRunning())
          w.addVisualizeWindow(arguments);
      // if there is no exception with visualize window then continue
      if (app.isRunning())
      {
          if (newApplication)
              app.launchNewApplication(arguments);
          else
              app.sendMessage(arguments);
          return 0;
      }
      w.show();
      return app.exec();
  } catch (VisualizeException &e)
  {
      QMessageBox *msgBox = new QMessageBox();
      msgBox->setWindowTitle(QString("OMVisualize - Error"));
      msgBox->setIcon(QMessageBox::Warning);
      msgBox->setText(QString(e.what()));
      msgBox->setStandardButtons(QMessageBox::Ok);
      msgBox->setDefaultButton(QMessageBox::Ok);
      msgBox->exec();
      return 1;
  }
}
