/*
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-2014, Open Source Modelica Consortium (OSMC),
 * c/o Linköpings universitet, Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF GPL VERSION 3 LICENSE OR
 * THIS OSMC PUBLIC LICENSE (OSMC-PL) VERSION 1.2.
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THE OSMC PUBLIC LICENSE OR THE GPL VERSION 3,
 * ACCORDING TO RECIPIENTS CHOICE.
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from OSMC, either from the above address,
 * from the URLs: http://www.ida.liu.se/projects/OpenModelica or
 * http://www.openmodelica.org, and in the OpenModelica distribution.
 * GNU version 3 is obtained from: http://www.gnu.org/copyleft/gpl.html.
 *
 * This program is distributed WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 */

encapsulated package CevalScript
" file:        CevalScript.mo
  package:     CevalScript
  description: Constant propagation of expressions


  This module handles scripting.

  Input:
    Env: Environment with bindings
    Exp: Expression to evaluate
    Bool flag determines whether the current instantiation is implicit
    InteractiveSymbolTable is optional, and used in interactive mode, e.g. from OMShell

  Output:
    Value: The evaluated value
    InteractiveSymbolTable: Modified symbol table
    Subscript list : Evaluates subscripts and generates constant expressions."

// public imports
import Absyn;
import Ceval;
import DAE;
import FCore;
import Error;
import GlobalScript;
import Interactive;
import Values;
import SimCode;

// protected imports
protected
import BaseHashSet;
import CevalScriptBackend;
import CevalFunction;
import ClassInf;
import ClassLoader;
import CodegenCFunctions;
import Config;
import Corba;
import DAEUtil;
import Debug;
import Dump;
import DynLoad;
import Expression;
import ExpressionDump;
import FBuiltin;
import Flags;
import FGraph;
import FNode;
import GC;
import GenerateAPIFunctionsTpl;
import Global;
import GlobalScriptUtil;
import Graph;
import HashSetString;
import Inst;
import InstFunction;
import List;
import Lookup;
import Mod;
import Prefix;
import Parser;
import Print;
import SCodeDump;
import SimCodeFunction;
import ExecStat.{execStat,execStatReset};
import StackOverflow;
import System;
import Static;
import SCode;
import SCodeUtil;
import Settings;
import SymbolTable;
import Tpl;
import Types;
import Unparsing;
import Util;
import ValuesUtil;
import ComponentReference;
import ErrorExt;

public

function ceval "
  This is a wrapper funtion to Ceval.ceval. The purpose of this
  function is to concetrate all the calls to Ceval.ceval made from
  the Script files. This will simplify the separation of the scripting
  environment from the FrontEnd"
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input DAE.Exp inExp;
  input Boolean inBoolean "impl";
  input Absyn.Msg inMsg;
  input Integer numIter;
  output FCore.Cache outCache;
  output Values.Value outValue;

  partial function ReductionOperator
    input Values.Value v1;
    input Values.Value v2;
    output Values.Value res;
  end ReductionOperator;
algorithm
  (outCache,outValue):=
  matchcontinue (inCache,inEnv,inExp,inBoolean,inMsg,numIter)
    local
      Boolean impl;
      FCore.Graph env;
      Absyn.Msg msg;
      list<Values.Value> vallst;
      list<DAE.Exp> expl;
      Values.Value newval,value;
      DAE.Exp e;
      Absyn.Path funcpath;
      FCore.Cache cache;

    // adrpo: TODO! this needs more work as if we don't have a symtab we run into unloading of dlls problem
    case (cache,env,(e as DAE.CALL(path = funcpath,expLst = expl)),impl,msg,_)
      equation
        // do not handle Connection.isRoot here!
        false = stringEq("Connection.isRoot", Absyn.pathString(funcpath));
        // do not roll back errors generated by evaluating the arguments
        (cache,vallst) = Ceval.cevalList(cache,env, expl, impl, msg, numIter);

        (cache,newval)= cevalCallFunction(cache, env, e, vallst, impl, msg, numIter+1);
      then
        (cache,newval);

    // Try Interactive functions last
    case (cache,env,(e as DAE.CALL()),(true),msg,_)
      equation
        (cache,value) = cevalInteractiveFunctions(cache, env, e, msg, numIter+1);
      then
        (cache,value);
    case (cache,env,e,impl,msg,_)
      equation
        (cache,value) = Ceval.ceval(cache,env,e,impl,msg,numIter+1);
      then
         (cache,value);
  end matchcontinue;
end ceval;

public function isCompleteFunction
"a function is complete if is:
 - not partial
 - not replaceable (without redeclare)
 - replaceable and called functions are not partial or not replaceable (without redeclare)"
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input Absyn.Path inFuncPath;
  output Boolean isComplete;
algorithm
 isComplete := matchcontinue(inCache, inEnv, inFuncPath)
   local
     FCore.Cache cache;
     FCore.Graph env;
     Absyn.Path fpath;

   // external functions are complete :)
   case (cache, env, fpath)
     equation
       (_, SCode.CLASS(classDef = SCode.PARTS(externalDecl = SOME(_))), _) = Lookup.lookupClass(cache, env, fpath);
     then
       true;

   // if is partial instantiation no function evaluation/generation
   case (_, _, _)
     equation
       true = System.getPartialInstantiation();
     then
       false;

   // partial functions are not complete!
   case (cache, env, fpath)
     equation
       (_, SCode.CLASS(partialPrefix = SCode.PARTIAL()), _) = Lookup.lookupClass(cache, env, fpath);
     then
       false;

   else true;

  end matchcontinue;
end isCompleteFunction;

public function compileModel "Compiles a model given a file-prefix, helper function to buildModel."
  input String fileprefix;
  input list<String> libs;
  input String workingDir = "";
  input list<String> makeVars = {};
protected
  String omhome = Settings.getInstallationDirectoryPath(),omhome_1 = System.stringReplace(omhome, "\"", "");
  String pd = System.pathDelimiter();
  String cdWorkingDir,setMakeVars,libsfilename,libs_str,s_call,filename,winCompileMode,workDir = (if stringEq(workingDir, "") then "" else workingDir + pd);
  String fileDLL = workDir + fileprefix + System.getDllExt(),
         fileEXE = workDir + fileprefix + System.getExeExt(),
         fileLOG = workDir + fileprefix + ".log";
  Integer numParallel,res;
  Boolean isWindows = System.os() == "Windows_NT";
  list<String> makeVarsNoBinding;
algorithm
  libsfilename := fileprefix + ".libs";
  libs_str := stringDelimitList(libs, " ");
  makeVarsNoBinding := makeVars; // OMC is stupid and wants to constant evaluate inputs with bindings for iterator variables...

  System.writeFile(libsfilename, libs_str);
  if isWindows then
    // We only need to set OPENMODELICAHOME on Windows, and set doesn't work in bash shells anyway
    // adrpo: 2010-10-05:
    //        whatever you do, DO NOT add a space before the && otherwise
    //        OPENMODELICAHOME that we set will contain a SPACE at the end!
    //        set OPENMODELICAHOME=DIR && actually adds the space between the DIR and &&
    //        to the environment variable! Don't ask me why, ask Microsoft.
    omhome := "set OPENMODELICAHOME=\"" + System.stringReplace(omhome_1, "/", "\\") + "\"&& ";
    setMakeVars := sum("set "+var+"&& " for var in makeVarsNoBinding);
    cdWorkingDir := if stringEmpty(workingDir) then "" else ("cd \"" + workingDir + "\"&& ");
    winCompileMode := if Config.getRunningTestsuite() then "serial" else "parallel";
    s_call := stringAppendList({omhome,cdWorkingDir,setMakeVars,"\"",omhome_1,pd,"share",pd,"omc",pd,"scripts",pd,"Compile","\""," ",fileprefix," ",Config.simulationCodeTarget()," ", System.openModelicaPlatform(), " ", winCompileMode});
  else
    numParallel := if Config.getRunningTestsuite() then 1 else Config.noProc();
    cdWorkingDir := if stringEmpty(workingDir) then "" else (" -C \"" + workingDir + "\"");
    setMakeVars := sum(" "+var for var in makeVarsNoBinding);
    s_call := stringAppendList({System.getMakeCommand()," -j",intString(numParallel),cdWorkingDir," -f ",fileprefix,".makefile",setMakeVars});
  end if;
  if Flags.isSet(Flags.DYN_LOAD) then
    Debug.traceln("compileModel: running " + s_call);
  end if;

  // remove .exe .dll .log!
  if System.regularFileExists(fileEXE) then
    0 := System.removeFile(fileEXE);
  end if;
  if System.regularFileExists(fileDLL) then
    0 := System.removeFile(fileDLL);
  end if;
  if System.regularFileExists(fileLOG) then
    0 := System.removeFile(fileLOG);
  end if;

  if Config.getRunningTestsuite() then
    System.appendFile(Config.getRunningTestsuiteFile(),
      fileEXE + "\n" + fileDLL + "\n" + fileLOG + "\n" + fileprefix + ".o\n" + fileprefix + ".libs\n" +
      fileprefix + "_records.o\n" + fileprefix + "_res.mat\n");
  end if;

  // call the system command to compile the model!
  if System.systemCall(s_call,if isWindows then "" else fileLOG) <> 0 then
    // We failed, print error
    if System.regularFileExists(fileLOG) then
      Error.addMessage(Error.SIMULATOR_BUILD_ERROR, {System.readFile(fileLOG)});
    elseif isWindows then
      // Check that it is a correct OPENMODELICAHOME, on Windows only
      s_call := stringAppendList({omhome_1,pd,"share",pd,"omc",pd,"scripts",pd,"Compile.bat"});
      if not System.regularFileExists(s_call) then
        Error.addMessage(Error.SIMULATOR_BUILD_ERROR, {stringAppendList({"command ",s_call," not found. Check $OPENMODELICAHOME"})});
      end if;
    end if;
    if Flags.isSet(Flags.DYN_LOAD) then
      Debug.trace("compileModel: failed!\n");
    end if;
    fail();
  end if;

  if Flags.isSet(Flags.DYN_LOAD) then
    Debug.trace("compileModel: successful!\n");
  end if;
end compileModel;

protected function loadFile "load the file or the directory structure if the file is named package.mo"
  input String name;
  input String encoding;
  input Absyn.Program p;
  input Boolean checkUses;
  output Absyn.Program outProgram;
protected
  String dir,filename,cname,prio,mp;
  list<String> rest;
algorithm
  true := System.regularFileExists(name);
  (dir,filename) := Util.getAbsoluteDirectoryAndFile(name);
  if filename == "package.mo" or filename == "package.moc" then
    cname::rest := System.strtok(List.last(System.strtok(dir,"/"))," ");
    prio := stringDelimitList(rest, " ");
    // send "" priority if that is it, don't send "default"
    // see https://trac.openmodelica.org/OpenModelica/ticket/2422
    // prio = if_(stringEq(prio,""), "default", prio);
    mp := System.realpath(dir + "/../") + System.groupDelimiter() + Settings.getModelicaPath(Config.getRunningTestsuite());
    (outProgram,true) := loadModel((Absyn.IDENT(cname),{prio},true)::{}, mp, p, true, true, checkUses, true, filename == "package.moc");
    return;
  end if;
  outProgram := Parser.parse(name,encoding);
  ClassLoader.checkOnLoadMessage(outProgram);
  outProgram := checkUsesAndUpdateProgram(outProgram, p, checkUses, Settings.getModelicaPath(Config.getRunningTestsuite()));
end loadFile;

protected function checkUsesAndUpdateProgram
  input Absyn.Program newp;
  input output Absyn.Program p;
  input Boolean checkUses;
  input String modelicaPath;
protected
  list<tuple<Absyn.Path,list<String>,Boolean>> modelsToLoad;
algorithm
  modelsToLoad := if checkUses then Interactive.getUsesAnnotationOrDefault(newp, requireExactVersion=true) else {};
  p := Interactive.updateProgram(newp, p);
  (p, _) := loadModel(modelsToLoad, modelicaPath, p, false, true, true, true, false);
end checkUsesAndUpdateProgram;

protected type LoadModelFoldArg =
  tuple<String /*modelicaPath*/, Boolean /*forceLoad*/, Boolean /*notifyLoad*/, Boolean /*checkUses*/, Boolean /*requireExactVersion*/, Boolean /*encrypted*/>;

public function loadModel
  input list<tuple<Absyn.Path,list<String>,Boolean /* Only use the first entry on the MODELICAPATH */>> imodelsToLoad;
  input String modelicaPath;
  input Absyn.Program ip;
  input Boolean forceLoad;
  input Boolean notifyLoad;
  input Boolean checkUses;
  input Boolean requireExactVersion;
  input Boolean encrypted = false;
  output Absyn.Program pnew;
  output Boolean success;
protected
  LoadModelFoldArg arg = (modelicaPath, forceLoad, notifyLoad, checkUses, requireExactVersion, encrypted);
algorithm
  (pnew, success) := List.fold1(imodelsToLoad, loadModel1, arg, (ip, true));
end loadModel;

protected function loadModel1
  input tuple<Absyn.Path,list<String>,Boolean> modelToLoad;
  input LoadModelFoldArg inArg;
  input tuple<Absyn.Program, Boolean> inTpl;
  output tuple<Absyn.Program, Boolean> outTpl;
protected
  list<tuple<Absyn.Path,list<String>,Boolean>> modelsToLoad;
  Boolean b, b1, success, forceLoad, notifyLoad, checkUses, requireExactVersion, onlyCheckFirstModelicaPath, encrypted;
  Absyn.Path path;
  list<String> versionsLst;
  String pathStr, versions, className, version, modelicaPath, thisModelicaPath;
  Absyn.Program p, pnew;
  Error.MessageTokens msgTokens;
algorithm
  (path, versionsLst, onlyCheckFirstModelicaPath) := modelToLoad;
  (modelicaPath, forceLoad, notifyLoad, checkUses, requireExactVersion, encrypted) := inArg;
  if onlyCheckFirstModelicaPath then
    /* Using loadFile() */
    thisModelicaPath::_ := System.strtok(modelicaPath, System.groupDelimiter());
  else
    thisModelicaPath := modelicaPath;
  end if;
  try
    (p, success) := inTpl;
    if checkModelLoaded(modelToLoad, p, forceLoad, NONE()) then
      pnew := Absyn.PROGRAM({}, Absyn.TOP());
      version := "";
    else
      pnew := ClassLoader.loadClass(path, versionsLst, thisModelicaPath, NONE(), requireExactVersion, encrypted);
      version := getPackageVersion(path, pnew);
      b := not notifyLoad or forceLoad;
      msgTokens := {Absyn.pathString(path), version};
      Error.assertionOrAddSourceMessage(b, Error.NOTIFY_NOT_LOADED, msgTokens, Absyn.dummyInfo);
    end if;
    p := Interactive.updateProgram(pnew, p);

    b := true;
    if checkUses then
      modelsToLoad := Interactive.getUsesAnnotationOrDefault(pnew, requireExactVersion);
      (p, b) := loadModel(modelsToLoad, modelicaPath, p, false, notifyLoad, checkUses, requireExactVersion, false);
    end if;
    outTpl := (p, success and b);
  else
    (p, _) := inTpl;
    pathStr := Absyn.pathString(path);
    versions := stringDelimitList(versionsLst, ",");
    msgTokens := {pathStr, versions, thisModelicaPath};
    if forceLoad then
      Error.addMessage(Error.LOAD_MODEL, msgTokens);
      outTpl := (p, false);
    else
      Error.addMessage(Error.NOTIFY_LOAD_MODEL_FAILED, msgTokens);
      outTpl := inTpl;
    end if;
  end try;
end loadModel1;

protected function checkModelLoaded
  input tuple<Absyn.Path,list<String>,Boolean> tpl;
  input Absyn.Program p;
  input Boolean forceLoad;
  input Option<String> failNonLoad;
  output Boolean loaded;
algorithm
  loaded := matchcontinue (tpl,p,forceLoad,failNonLoad)
    local
      Absyn.Class cdef;
      String str1,str2;
      Option<String> ostr2;
      Absyn.Path path;

    case (_,_,true,_) then false;
    case ((path,str1::_,_),_,false,_)
      equation
        cdef = Interactive.getPathedClassInProgram(path,p);
        ostr2 = Absyn.getNamedAnnotationInClass(cdef,Absyn.IDENT("version"),Interactive.getAnnotationStringValueOrFail);
        checkValidVersion(path,str1,ostr2);
      then true;
    case (_,_,_,NONE()) then false;
    case ((path,_,_),_,_,SOME(str2))
      equation
        str1 = Absyn.pathString(path);
        Error.addMessage(Error.INST_NON_LOADED, {str1,str2});
      then false;
  end matchcontinue;
end checkModelLoaded;

protected function checkValidVersion
  input Absyn.Path path;
  input String version;
  input Option<String> actualVersion;
algorithm
  _ := matchcontinue (path,version,actualVersion)
    local
      String pathStr,str1,str2;
    case (_,str1,SOME(str2))
      equation
        true = stringEq(str1,str2);
      then ();
    case (_,str1,SOME(str2))
      equation
        pathStr = Absyn.pathString(path);
        Error.addMessage(Error.LOAD_MODEL_DIFFERENT_VERSIONS,{pathStr,str1,str2});
      then ();
    case (_,str1,NONE())
      equation
        pathStr = Absyn.pathString(path);
        Error.addMessage(Error.LOAD_MODEL_DIFFERENT_VERSIONS,{pathStr,str1,"unknown"});
      then ();
  end matchcontinue;
end checkValidVersion;

public function cevalInteractiveFunctions
"defined in the interactive environment."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input DAE.Exp inExp "expression to evaluate";
  input Absyn.Msg msg;
  input Integer numIter;
  output FCore.Cache outCache;
  output Values.Value outValue;
algorithm
  (outCache,outValue) := matchcontinue (inCache,inEnv,inExp,msg,numIter)
    local
      FCore.Cache cache;
      FCore.Graph env;
      DAE.Exp exp;
      list<DAE.Exp> eLst;
      list<Values.Value> valLst;
      String name;
      Values.Value value;
      Real t1,t2,t;

      // This needs to be first because otherwise it takes 0 time to get the value :)
    case (cache,env,DAE.CALL(path = Absyn.IDENT(name = "timing"),expLst = {exp}),_,_)
      equation
        t1 = System.time();
        (cache,_) = Ceval.ceval(cache,env, exp, true, msg,numIter+1);
        t2 = System.time();
        t = t2 - t1;
      then
        (cache,Values.REAL(t));

    case (cache,env,DAE.CALL(path=Absyn.IDENT(name),attr=DAE.CALL_ATTR(builtin=true),expLst=eLst),_,_)
      equation
        (cache,valLst) = Ceval.cevalList(cache,env,eLst,true,msg,numIter);
        valLst = List.map1(valLst,evalCodeTypeName,env);
        (cache,value) = cevalInteractiveFunctions2(cache,env,name,valLst,msg);
      then
        (cache,value);

  end matchcontinue;
end cevalInteractiveFunctions;

public function cevalInteractiveFunctions2
"defined in the interactive environment."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input String inFunctionName;
  input list<Values.Value> inVals;
  input Absyn.Msg msg;
  output FCore.Cache outCache;
  output Values.Value outValue;
algorithm
  (outCache,outValue) := matchcontinue (inCache,inEnv,inFunctionName,inVals,msg)
    local
      String omdev,simflags,s1,s2,s3,str,str1,str2,str3,token,varid,cmd,executable,executable1,encoding,method_str,
             outputFormat_str,initfilename,pd,executableSuffixedExe,sim_call,result_file,filename_1,filename,filename1,filename2,
             call,str_1,mp,pathstr,name,cname,errMsg,errorStr,
             title,xLabel,yLabel,filename2,varNameStr,xml_filename,xml_contents,visvar_str,pwd,omhome,omlib,omcpath,os,
             platform,usercflags,senddata,res,workdir,gcc,confcmd,touch_file,uname,filenameprefix,compileDir,libDir,exeDir,configDir,from,to,
             gridStr, logXStr, logYStr, x1Str, x2Str, y1Str, y2Str, curveWidthStr, curveStyleStr, legendPosition, footer, autoScaleStr,scriptFile,logFile, simflags2, outputFile,
             systemPath, gccVersion, gd, strlinearizeTime, direction, suffix;
      list<DAE.Exp> simOptions;
      list<Values.Value> vals;
      Absyn.Path path,classpath,className,baseClassPath;
      SCode.Program scodeP,sp;
      Option<list<SCode.Element>> fp;
      FCore.Graph env;
      Absyn.Program p,ip,pnew,newp,ptot;
      list<Absyn.Program> newps;
      list<GlobalScript.Variable> iv;
      GlobalScript.SimulationOptions simOpt;
      Real startTime,stopTime,tolerance,reltol,reltolDiffMinMax,rangeDelta;
      DAE.Exp startTimeExp,stopTimeExp,toleranceExp,intervalExp;
      DAE.Type tp, ty;
      list<DAE.Type> tys;
      Absyn.Class absynClass;
      Absyn.ClassDef cdef;
      Absyn.Exp aexp;
      DAE.DAElist dae;
      array<list<Integer>> m,mt;
      Values.Value ret_val,simValue,value,v,cvar,cvar2,v1,v2,v3,gcStatRec;
      Absyn.ComponentRef cr,cr_1;
      Integer size,resI,i,i1,i2,i3,n,curveStyle,numberOfIntervals, status, access;
      list<Integer> is;
      list<String> vars_1,args,strings,strs,strs1,strs2,visvars,postOptModStrings,postOptModStringsOrg,mps,files,dirs;
      Real timeTotal,timeSimulation,timeStamp,val,x1,x2,y1,y2,r,r1,r2,linearizeTime,curveWidth,offset,offset1,offset2,scaleFactor,scaleFactor1,scaleFactor2;
      GlobalScript.Statements istmts;
      list<GlobalScript.Statements> istmtss;
      Boolean have_corba, bval, anyCode, b, b1, b2, externalWindow, logX, logY, autoScale, forceOMPlot, gcc_res, omcfound, rm_res, touch_res, uname_res,  ifcpp, ifmsvc,sort, builtin, showProtected, includeConstants, inputConnectors, outputConnectors, mergeAST;
      FCore.Cache cache;
      Absyn.ComponentRef  crefCName;
      list<tuple<String,Values.Value>> resultValues;
      list<Real> realVals;
      list<tuple<String,list<String>>> deps,depstransitive,depstransposed,depstransposedtransitive,depsmerged,depschanged;
      Absyn.CodeNode codeNode;
      list<Values.Value> cvars,vals2;
      list<Absyn.Path> paths;
      list<Absyn.NamedArg> nargs;
      list<Absyn.Class> classes;
      Absyn.Within within_;
      GlobalScript.SimulationOptions defaulSimOpt;
      SimCode.SimulationSettings simSettings;
      Boolean dumpExtractionSteps, requireExactVersion;
      list<tuple<Absyn.Path,list<String>>> uses;
      Config.LanguageStandard oldLanguageStd;
      SCode.Element cl;
      list<SCode.Element> cls, elts;
      list<String> names, namesPublic, namesProtected, namesChanged, fileNames;
      HashSetString.HashSet hashSetString;
      list<Boolean> blst;
      list<Error.TotalMessage> messages;
      Real stoptime,starttime,tol,stepsize,interval;
      String stoptime_str,stepsize_str,starttime_str,tol_str,num_intervalls_str,description,prefix;
      list<String> interfaceType;
      list<tuple<String,list<String>>> interfaceTypeAssoc;
      SCode.Encapsulated encflag;
      SCode.Restriction restr;
      list<list<Values.Value>> valsLst;
      Boolean new_inst;
      SymbolTable interactiveSymbolTable, interactiveSymbolTable2;
      GC.ProfStats gcStats;
      Absyn.Restriction restriction;

    case (cache,_,"parseString",{Values.STRING(str1),Values.STRING(str2)},_)
      equation
        Absyn.PROGRAM(classes=classes,within_=within_) = Parser.parsestring(str1,str2);
        paths = List.map(classes,Absyn.className);
        paths = List.map1r(paths,Absyn.joinWithinPath,within_);
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"parseString",_,_)
      then (cache,ValuesUtil.makeArray({}));

    case (cache,_,"parseFile",{Values.STRING(str1),Values.STRING(encoding)},_)
      equation
        // clear the errors before!
        Error.clearMessages() "Clear messages";
        Print.clearErrorBuf() "Clear error buffer";
        (paths) = Interactive.parseFile(str1, encoding);
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"loadFileInteractiveQualified",{Values.STRING(str1),Values.STRING(encoding)},_)
      equation
        // clear the errors before!
        Error.clearMessages() "Clear messages";
        Print.clearErrorBuf() "Clear error buffer";
        paths = Interactive.parseFile(str1, encoding, updateProgram=true);
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"loadFileInteractive",{Values.STRING(str1),Values.STRING(encoding)},_)
      equation
        pnew = loadFile(str1, encoding, SymbolTable.getAbsyn(), false) "System.regularFileExists(name) => 0 &    Parser.parse(name) => p1 &" ;
        vals = List.map(Interactive.getTopClassnames(pnew),ValuesUtil.makeCodeTypeName);
        SymbolTable.setAbsyn(pnew);
      then (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"getSourceFile",{Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        str = Interactive.getSourceFile(path, SymbolTable.getAbsyn());
      then
        (cache,Values.STRING(str));

    case (cache,_,"setSourceFile",{Values.CODE(Absyn.C_TYPENAME(path)),Values.STRING(str)},_)
      equation
        Values.ENUM_LITERAL(index=access) = Interactive.checkAccessAnnotationAndEncryption(path, SymbolTable.getAbsyn());
        if (access >= 9) then // i.e., The class is not encrypted.
          (b,p) = Interactive.setSourceFile(path, str, SymbolTable.getAbsyn());
          SymbolTable.setAbsyn(p);
        else
          Error.addMessage(Error.SAVE_ENCRYPTED_CLASS_ERROR, {});
          b = false;
        end if;
      then
        (cache,Values.BOOL(b));

    case (cache,_,"basename",{Values.STRING(str)},_)
      equation
        str = System.basename(str);
      then (cache,Values.STRING(str));

    case (cache,_,"dirname",{Values.STRING(str)},_)
      equation
        str = System.dirname(str);
      then (cache,Values.STRING(str));

    case (cache,_,"codeToString",{Values.CODE(codeNode)},_)
      equation
        str = Dump.printCodeStr(codeNode);
      then (cache,Values.STRING(str));

    case (cache,_,"typeOf",{Values.CODE(Absyn.C_VARIABLENAME(Absyn.CREF_IDENT(name = varid)))},_)
      equation
        tp = Interactive.getTypeOfVariable(varid, SymbolTable.getVars());
        str = Types.unparseType(tp);
      then
        (cache,Values.STRING(str));

    case (cache,_,"GC_gcollect_and_unmap",{},_)
      equation
        GC.gcollectAndUnmap();
      then (cache,Values.BOOL(true));

    case (cache,_,"GC_expand_hp",{Values.INTEGER(i)},_)
      equation
        b = GC.expandHeap(i);
      then (cache,Values.BOOL(b));

    case (cache,_,"GC_set_max_heap_size",{Values.INTEGER(i)},_)
      equation
        GC.setMaxHeapSize(i);
      then (cache,Values.BOOL(true));

    case (cache,_,"GC_get_prof_stats",{},_)
      equation
        gcStats = GC.getProfStats();
        gcStatRec = match gcStats
         case GC.PROFSTATS() then
           Values.RECORD(Absyn.IDENT("GC_PROFSTATS"),
            {
              Values.INTEGER(gcStats.heapsize_full),
              Values.INTEGER(gcStats.free_bytes_full),
              Values.INTEGER(gcStats.unmapped_bytes),
              Values.INTEGER(gcStats.bytes_allocd_since_gc),
              Values.INTEGER(gcStats.allocd_bytes_before_gc),
              Values.INTEGER(gcStats.bytes_allocd_since_gc+gcStats.allocd_bytes_before_gc),
              Values.INTEGER(gcStats.non_gc_bytes),
              Values.INTEGER(gcStats.gc_no),
              Values.INTEGER(gcStats.markers_m1),
              Values.INTEGER(gcStats.bytes_reclaimed_since_gc),
              Values.INTEGER(gcStats.reclaimed_bytes_before_gc)
            },
            {
              "heapsize_full",
              "free_bytes_full",
              "unmapped_bytes: ",
              "bytes_allocd_since_gc",
              "allocd_bytes_before_gc",
              "total_allocd_bytes",
              "non_gc_bytes",
              "gc_no",
              "markers_m1",
              "bytes_reclaimed_since_gc",
              "reclaimed_bytes_before_gc"
            },
            -1);
       end match;
      then (cache, gcStatRec);

    case (cache,_,"clear",{},_)
      algorithm
        SymbolTable.reset();
      then (cache,Values.BOOL(true));

    case (cache,_,"clearProgram",{},_)
      algorithm
        SymbolTable.clearProgram();
      then (cache,Values.BOOL(true));

    case (cache,_,"clearVariables",{},_)
      equation
        SymbolTable.setVars({});
      then (cache,Values.BOOL(true));

    // handle encryption
    case (cache,_,"list",_,_)
      equation
        // if AST contains encrypted class show nothing
        p = SymbolTable.getAbsyn();
        true = Interactive.astContainsEncryptedClass(p);
        Error.addMessage(Error.ACCESS_ENCRYPTED_PROTECTED_CONTENTS, {});
      then
        (cache,Values.STRING(""));

    case (cache,_,"list",{Values.CODE(Absyn.C_TYPENAME(Absyn.IDENT("AllLoadedClasses"))),Values.BOOL(false),Values.BOOL(false),Values.ENUM_LITERAL(name=path)},_)
      equation
        name = Absyn.pathLastIdent(path);
        str = match name
          case "Absyn" then Dump.unparseStr(SymbolTable.getAbsyn(), false);
          case "SCode" then SCodeDump.programStr(SymbolTable.getSCode());
          case "MetaModelicaInterface" then SCodeDump.programStr(SymbolTable.getSCode(), SCodeDump.OPTIONS(true,false,true,true,true,true,true,true,true));
          case "Internal" then System.anyStringCode(SymbolTable.getAbsyn());
          else "";
        end match;
      then
        (cache,Values.STRING(str));

    case (cache,_,"list",{Values.CODE(Absyn.C_TYPENAME(className)),Values.BOOL(b1),Values.BOOL(b2),Values.ENUM_LITERAL(name=path)},_)
      equation
        false = valueEq(Absyn.IDENT("AllLoadedClasses"),className);
        name = Absyn.pathLastIdent(path);
        p = SymbolTable.getAbsyn();
        scodeP = SymbolTable.getSCode();
        absynClass = Interactive.getPathedClassInProgram(className, p);
        absynClass = if b1 then Absyn.getFunctionInterface(absynClass) else absynClass;
        absynClass = if b2 then Absyn.getShortClass(absynClass) else absynClass;
        p = Absyn.PROGRAM({absynClass},Absyn.TOP());
        cl = SCodeUtil.getElementWithPathCheckBuiltin(scodeP, className);
        str = match name
          case "Absyn" then Dump.unparseStr(p, false);
          case "SCode" then SCodeDump.unparseElementStr(cl);
          case "MetaModelicaInterface" then SCodeDump.unparseElementStr(cl, SCodeDump.OPTIONS(true,false,true,true,true,true,true,true,true));
          case "Internal" then System.anyStringCode(p);
          else "";
        end match;
      then
        (cache,Values.STRING(str));

    case (cache,_,"list",_,_) then (cache,Values.STRING(""));

    case (cache,_,"listFile",{Values.CODE(Absyn.C_TYPENAME(className))},_)
      equation
        path = match className
          case Absyn.FULLYQUALIFIED() then className.path;
          else className;
        end match;
        // handle encryption
        Values.ENUM_LITERAL(index=access) = Interactive.checkAccessAnnotationAndEncryption(path, SymbolTable.getAbsyn());
        (absynClass as Absyn.CLASS(restriction=restriction, info=SOURCEINFO(fileName=str))) = Interactive.getPathedClassInProgram(className, SymbolTable.getAbsyn());
        /* If the class has Access.packageText annotation or higher
         * If the class has Access.nonPackageText annotation or higher and class is not a package
         */
        if ((access >= 7) or ((access >= 5) and not Absyn.isPackageRestriction(restriction))) then
          str = Dump.unparseStr(Absyn.PROGRAM({absynClass}, match path case Absyn.IDENT() then Absyn.TOP(); else Absyn.WITHIN(Absyn.stripLast(path)); end match), options=Dump.DUMPOPTIONS(str));
        else
          Error.addMessage(Error.ACCESS_ENCRYPTED_PROTECTED_CONTENTS, {});
          str = "";
        end if;
      then
        (cache,Values.STRING(str));

    case (cache,_,"listFile",_,_) then (cache,Values.STRING(""));

    case (cache,_,"sortStrings",{Values.ARRAY(valueLst=vals)},_)
      equation
        strs = List.map(vals, ValuesUtil.extractValueString);
        strs = List.sort(strs,Util.strcmpBool);
        v = ValuesUtil.makeArray(List.map(strs,ValuesUtil.makeString));
      then
        (cache,v);

    case (cache,_,"listVariables",{},_)
      equation
        v = ValuesUtil.makeArray(getVariableNames(SymbolTable.getVars(),{}));
      then
        (cache,v);

    case (cache,_,"setCompileCommand",{Values.STRING(cmd)},_)
      equation
        // cmd = Util.rawStringToInputString(cmd);
        Settings.setCompileCommand(cmd);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getCompileCommand",{},_)
      equation
        res = Settings.getCompileCommand();
      then
        (cache,Values.STRING(res));

    case (cache,_,"setTempDirectoryPath",{Values.STRING(cmd)},_)
      equation
        // cmd = Util.rawStringToInputString(cmd);
        Settings.setTempDirectoryPath(cmd);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getTempDirectoryPath",{},_)
      equation
        res = Settings.getTempDirectoryPath();
      then
        (cache,Values.STRING(res));

    case (cache,_,"setEnvironmentVar",{Values.STRING(varid),Values.STRING(str)},_)
      equation
        b = 0 == System.setEnv(varid,str,true);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"getEnvironmentVar",{Values.STRING(varid)},_)
      equation
        res = Util.makeValueOrDefault(System.readEnv, varid, "");
      then
        (cache,Values.STRING(res));

    case (cache,_,"setInstallationDirectoryPath",{Values.STRING(cmd)},_)
      equation
        // cmd = Util.rawStringToInputString(cmd);
        Settings.setInstallationDirectoryPath(cmd);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getInstallationDirectoryPath",{},_)
      equation
        res = Settings.getInstallationDirectoryPath();
      then
        (cache,Values.STRING(res));

    case (cache,_,"getModelicaPath",{},_)
      equation
        res = Settings.getModelicaPath(Config.getRunningTestsuite());
      then
        (cache,Values.STRING(res));

    case (cache,_,"setModelicaPath",{Values.STRING(cmd)},_)
      equation
        // cmd = Util.rawStringToInputString(cmd);
        Settings.setModelicaPath(cmd);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"setModelicaPath",_,_)
      then
        (cache,Values.BOOL(false));

    case (cache,_,"getLanguageStandard",{},_)
      equation
        res = Config.languageStandardString(Config.getLanguageStandard());
      then
        (cache,Values.STRING(res));

    case (cache,_,"reopenStandardStream",{Values.ENUM_LITERAL(index=i),Values.STRING(filename)},_)
      equation
        b = System.reopenStandardStream(i-1,filename);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"iconv",{Values.STRING(str),Values.STRING(from),Values.STRING(to)},_)
      equation
        str = System.iconv(str,from,to);
      then
        (cache,Values.STRING(str));

    case (cache,_,"getCompiler",{},_)
      equation
        str = System.getCCompiler();
      then
        (cache,Values.STRING(str));

    case (cache,_,"setCFlags",{Values.STRING(str)},_)
      equation
        System.setCFlags(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getCFlags",{},_)
      equation
        str = System.getCFlags();
      then
        (cache,Values.STRING(str));

    case (cache,_,"setCompiler",{Values.STRING(str)},_)
      equation
        System.setCCompiler(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getCXXCompiler",{},_)
      equation
        str = System.getCXXCompiler();
      then
        (cache,Values.STRING(str));

    case (cache,_,"setCXXCompiler",{Values.STRING(str)},_)
      equation
        System.setCXXCompiler(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"setCompilerFlags",{Values.STRING(str)},_)
      equation
        System.setCFlags(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getLinker",{},_)
      equation
        str = System.getLinker();
      then
        (cache,Values.STRING(str));

    case (cache,_,"setLinker",{Values.STRING(str)},_)
      equation
        System.setLinker(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getLinkerFlags",{},_)
      equation
        str = System.getLDFlags();
      then
        (cache,Values.STRING(str));

    case (cache,_,"setLinkerFlags",{Values.STRING(str)},_)
      equation
        System.setLDFlags(str);
      then
        (cache,Values.BOOL(true));

    case (_,_,"setCommandLineOptions",{Values.STRING(str)},_)
      equation
        new_inst = Flags.isSet(Flags.SCODE_INST);
        args = System.strtok(str, " ");
        {} = Flags.readArgs(args);

        // Invalidate the builtin cache if the newInst flag was toggled,
        // so we don't reuse the wrong builtin program.
        if new_inst <> Flags.isSet(Flags.SCODE_INST) then
          setGlobalRoot(Global.builtinIndex, {});
        end if;
      then
        (FCore.emptyCache(),Values.BOOL(true));

    case (cache,_,"setCommandLineOptions",_,_)
      then (cache,Values.BOOL(false));

    case (cache, _, "getCommandLineOptions", {}, _)
      then (cache, ValuesUtil.makeStringArray(Flags.unparseFlags()));

    case (cache, _, "getCommandLineOptions", _, _)
      then (cache, Values.META_FAIL());

    case (cache,_,"clearCommandLineOptions",{},_)
      equation
        Flags.resetDebugFlags();
        Flags.resetConfigFlags();
      then
        (cache,Values.BOOL(true));

    case (cache,_,"clearCommandLineOptions",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"clearDebugFlags",_,_)
      equation
        Flags.resetDebugFlags();
      then
        (cache,Values.BOOL(true));

    case (cache,_,"clearDebugFlags",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"getConfigFlagValidOptions",{Values.STRING(str)},_)
      equation
        (strs1,str,strs2) = Flags.getValidOptionsAndDescription(str);
        v1 = ValuesUtil.makeArray(List.map(strs1, ValuesUtil.makeString));
        v2 = Values.STRING(str);
        v3 = ValuesUtil.makeArray(List.map(strs2, ValuesUtil.makeString));
        v = Values.TUPLE({v1,v2,v3});
      then (cache,v);

    case (cache,_,"getConfigFlagValidOptions",{Values.STRING(_)},_)
      equation
        v1 = ValuesUtil.makeArray({});
        v2 = Values.STRING("");
        v3 = ValuesUtil.makeArray({});
        v = Values.TUPLE({v1,v2,v3});
      then (cache,v);

    case (cache,_,"cd",{Values.STRING("")},_)
      equation
        str_1 = System.pwd();
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"cd",{Values.STRING(str)},_)
      equation
        resI = System.cd(str);
        (resI == 0) = true;
        str_1 = System.pwd();
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"cd",{Values.STRING(str)},_)
      equation
        failure(true = System.directoryExists(str));
        res = stringAppendList({"Error, directory ",str," does not exist,"});
      then
        (cache,Values.STRING(res));

    case (cache,_,"mkdir",{Values.STRING(str)},_)
      equation
        true = System.directoryExists(str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"mkdir",{Values.STRING(str)},_)
      equation
        b = Util.createDirectoryTree(str);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"remove",{Values.STRING(str)},_)
      equation
        b = System.removeDirectory(str);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"getVersion",{Values.CODE(Absyn.C_TYPENAME(Absyn.IDENT("OpenModelica")))},_)
      equation
        str_1 = Settings.getVersionNr();
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"getVersion",{Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        str_1 = getPackageVersion(path,SymbolTable.getAbsyn());
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"getTempDirectoryPath",{},_)
      equation
        str_1 = Settings.getTempDirectoryPath();
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"system",{Values.STRING(str),Values.STRING(filename)},_)
      equation
        resI = System.systemCall(str,filename);
      then
        (cache,Values.INTEGER(resI));

    case (cache,_,"system_parallel",{Values.ARRAY(valueLst=vals),Values.INTEGER(i)},_)
      equation
        strs = List.map(vals, ValuesUtil.extractValueString);
        v = ValuesUtil.makeIntArray(System.systemCallParallel(strs,i));
      then
        (cache,v);

    case (cache,_,"timerClear",{Values.INTEGER(i)},_)
      equation
        System.realtimeClear(i);
      then
        (cache,Values.NORETCALL());

    case (cache,_,"timerTick",{Values.INTEGER(i)},_)
      equation
        System.realtimeTick(i);
      then
        (cache,Values.NORETCALL());

    case (cache,_,"timerTock",{Values.INTEGER(i)},_)
      equation
        true = System.realtimeNtick(i) > 0;
        r = System.realtimeTock(i);
      then
        (cache,Values.REAL(r));

    case (cache,_,"timerTock",_,_)
      then (cache,Values.REAL(-1.0));

    case (cache,_,"readFile",{Values.STRING(str)},_)
      equation
        str_1 = System.readFile(str);
      then (cache,Values.STRING(str_1));

    case (cache,_,"readFile",_,_)
      then (cache,Values.STRING(""));

    case (cache,_,"writeFile",{Values.STRING(str),Values.STRING(str1),Values.BOOL(false)},_)
      equation
        System.writeFile(str,str1);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"writeFile",{Values.STRING(str),Values.STRING(str1),Values.BOOL(true)},_)
      equation
        System.appendFile(str, str1);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"writeFile",_,_)
      then
        (cache,Values.BOOL(false));

    case (cache,_,"deleteFile",{Values.STRING(str)},_)
      equation
        b = if System.removeFile(str) == 0 then true else false;
      then
        (cache,Values.BOOL(b));

    case (cache,_,"compareFiles",{Values.STRING(str1),Values.STRING(str2)},_)
      equation
        b = System.fileContentsEqual(str1,str2);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"compareFilesAndMove",{Values.STRING(str1),Values.STRING(str2)},_)
      equation
        true = System.regularFileExists(str1);
        b = System.regularFileExists(str2) and System.fileContentsEqual(str1,str2);
        b = if not b then System.rename(str1,str2) else b;
      then
        (cache,Values.BOOL(b));

    case (cache,_,"compareFilesAndMove",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"readFileNoNumeric",{Values.STRING(str)},_)
      equation
        str_1 = System.readFileNoNumeric(str);
      then
        (cache,Values.STRING(str_1));

    case (cache,_,"getErrorString",{Values.BOOL(b)},_)
      equation
        str = Error.printMessagesStr(b);
      then
        (cache,Values.STRING(str));

    case (cache,_,"countMessages",_,_)
      equation
        i1 = Error.getNumMessages();
        i2 = Error.getNumErrorMessages();
        i3 = ErrorExt.getNumWarningMessages();
      then
        (cache,Values.TUPLE({Values.INTEGER(i1),Values.INTEGER(i2),Values.INTEGER(i3)}));

    case (cache,_,"clearMessages",{},_)
      equation
        Error.clearMessages();
      then
        (cache,Values.BOOL(true));

    case (cache,_,"getMessagesStringInternal",{Values.BOOL(true)},_)
      equation
        messages = List.unique(Error.getMessages());
        v = ValuesUtil.makeArray(List.map(messages, errorToValue));
      then
        (cache,v);

    case (cache,_,"getMessagesStringInternal",{Values.BOOL(false)},_)
      equation
        v = ValuesUtil.makeArray(List.map(Error.getMessages(), errorToValue));
      then
        (cache,v);

    case (cache,_,"stringTypeName",{Values.STRING(str)},_)
      equation
        path = Parser.stringPath(str);
      then (cache,Values.CODE(Absyn.C_TYPENAME(path)));

    case (cache,_,"stringVariableName",{Values.STRING(str)},_)
      equation
        cr = Parser.stringCref(str);
      then (cache,Values.CODE(Absyn.C_VARIABLENAME(cr)));

    case (cache,_,"typeNameString",{Values.CODE(A=Absyn.C_TYPENAME(path=path))},_)
      equation
        str = Absyn.pathString(path);
      then (cache,Values.STRING(str));

    case (cache,_,"typeNameStrings",{Values.CODE(A=Absyn.C_TYPENAME(path=path))},_)
      equation
        v = ValuesUtil.makeArray(List.map(Absyn.pathToStringList(path),ValuesUtil.makeString));
      then (cache,v);

    case (cache,_,"generateHeader",{Values.STRING(filename)},_)
      equation
        str = Tpl.tplString(Unparsing.programExternalHeader, SymbolTable.getSCode());
        System.writeFile(filename,str);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"generateHeader",_,_)
      then
        (cache,Values.BOOL(false));

    case (cache,env,"generateCode",{Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        (cache,Util.SUCCESS()) = Static.instantiateDaeFunction(cache, env, path, false, NONE(), true);
        (cache,_,_) = cevalGenerateFunction(cache,env,SymbolTable.getAbsyn(),path);
      then
        (cache,Values.BOOL(true));

    case (cache,_,"generateCode",_,_)
      then
        (cache,Values.BOOL(false));

    case (cache,env,"generateScriptingAPI",{Values.CODE(Absyn.C_TYPENAME(className)), Values.STRING(name)},_)
      algorithm
        scodeP := SymbolTable.getSCode();
        elts := match SCodeUtil.getElementWithPathCheckBuiltin(scodeP, className)
          case SCode.CLASS(classDef=SCode.PARTS(elementLst=elts)) then elts;
          case cl equation Error.addSourceMessage(Error.INTERNAL_ERROR, {Absyn.pathString(className) + " does not contain SCode.PARTS"}, SCode.elementInfo(cl)); then fail();
        end match;
        tys := {};
        for elt in elts loop
          _ := matchcontinue elt
            case SCode.CLASS(partialPrefix=SCode.NOT_PARTIAL(), restriction=SCode.R_FUNCTION(SCode.FR_EXTERNAL_FUNCTION()))
              algorithm
                (cache, ty, _) := Lookup.lookupType(cache, env, Absyn.suffixPath(className, elt.name), NONE() /*SOME(elt.info)*/);
                if isSimpleAPIFunction(ty) then
                  tys := ty::tys;
                end if;
              then ();
            else ();
          end matchcontinue;
        end for;
        s1 := Tpl.tplString(GenerateAPIFunctionsTpl.getCevalScriptInterface, tys);
        s2 := Tpl.tplString3(GenerateAPIFunctionsTpl.getQtInterface, tys, name + "::", name);
        s3 := Tpl.tplString2(GenerateAPIFunctionsTpl.getQtInterfaceHeaders, tys, name);
      then (cache,Values.TUPLE({Values.BOOL(true),Values.STRING(s1),Values.STRING(s2),Values.STRING(s3)}));

    case (cache,_,"generateScriptingAPI",_,_)
      then (cache,Values.TUPLE({Values.BOOL(false),Values.STRING(""),Values.STRING("")}));

    case (cache,_,"generateEntryPoint",{Values.STRING(filename),Values.CODE(Absyn.C_TYPENAME(path)),Values.STRING(str)},_)
      equation
        str = Tpl.tplString2(CodegenCFunctions.generateEntryPoint, path, str);
        System.writeFile(filename,str);
      then (cache,Values.BOOL(true));

    case (cache,_,"generateEntryPoint",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"checkInterfaceOfPackages",{Values.CODE(Absyn.C_TYPENAME(path)),Values.ARRAY(valueLst=vals)},_)
      equation
        sp = SymbolTable.getSCode();
        cl = SCode.getElementWithPath(sp,path);
        interfaceTypeAssoc = List.map1(vals, getInterfaceTypeAssocElt, SCode.elementInfo(cl));
        interfaceType = getInterfaceType(cl, interfaceTypeAssoc);
        List.map1_0(sp, verifyInterfaceType, interfaceType);
      then (cache,Values.BOOL(true));

    case (cache,_,"checkInterfaceOfPackages",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"generateSeparateCodeDependenciesMakefile",{Values.STRING(filename),Values.STRING(prefix),Values.STRING(suffix)},_)
      equation
        sp = SymbolTable.getSCode();
        names = List.filterMap(sp,SCode.getElementName);
        deps = Graph.buildGraph(names,buildDependencyGraphPublicImports,sp);
        strs = List.map3(sp,writeModuleDepends,prefix,suffix,deps);
        System.writeFile(filename,stringDelimitList(strs,"\n"));
      then (cache,Values.BOOL(true));

    case (cache,_,"generateSeparateCodeDependenciesMakefile",_,_)
      then (cache,Values.BOOL(false));

    case (cache,_,"generateSeparateCodeDependencies",{Values.STRING(suffix)},_)
      equation
        sp = SymbolTable.getSCode();
        names = List.filterMap(sp,SCode.getElementName);

        deps = Graph.buildGraph(names,buildDependencyGraph,sp);
        namesPublic = List.map(List.select(sp, containsPublicInterface), SCode.getElementName);
        namesChanged = List.filterMap1(sp,getChangedClass,suffix);
        hashSetString = HashSetString.emptyHashSet();
        hashSetString = List.fold(namesChanged,BaseHashSet.add,hashSetString);
        // print("namesChanged: " + stringDelimitList(namesChanged, ",") + "\n");

        depstransposed = Graph.transposeGraph(Graph.emptyGraph(names),deps,stringEq);
        depstransposedtransitive = Graph.buildGraph(namesPublic,buildTransitiveDependencyGraph,depstransposed);
        // depstransposedtransitive = List.sort(depstransposed, compareNumberOfDependencies);

        depstransitive = Graph.transposeGraph(Graph.emptyGraph(names),depstransposedtransitive,stringEq);
        depstransitive = List.sort(depstransitive, compareNumberOfDependencies);

        depsmerged = Graph.merge(deps,depstransitive,stringEq,compareDependencyNode);
        // depsmerged = List.sort(depsmerged, compareNumberOfDependencies);

        /*
         print("Total number of modules: " + intString(listLength(depsmerged)) + "\n");
         str = stringDelimitList(List.map(depsmerged, transitiveDependencyString), "\n");
         print(str + "\n");
        */

        depschanged = List.select1(depsmerged,isChanged,hashSetString);
        names = List.map(depschanged, Util.tuple21);
        // print("Files to recompile (" + intString(listLength(depschanged)) + "): " + stringDelimitList(names, ",") + "\n");
        fileNames = List.map1(names, stringAppend, suffix);
        _ = List.map(fileNames, System.removeFile);
        v = ValuesUtil.makeArray(List.map(names,ValuesUtil.makeString));
      then (cache,v);

    case (cache,_,"generateSeparateCodeDependencies",_,_)
      then (cache,Values.META_FAIL());

    case (cache,env,"generateSeparateCode",{v,Values.BOOL(b)},_)
      equation
        p = SymbolTable.getAbsyn();
        sp = SymbolTable.getSCode();
        name = getTypeNameIdent(v);
        setGlobalRoot(Global.instOnlyForcedFunctions,SOME(true));
        cl = List.getMemberOnTrue(name, sp, SCode.isClassNamed);
        (cache,env) = generateFunctions(cache,env,p,sp,{cl},b);
        setGlobalRoot(Global.instOnlyForcedFunctions,NONE());
      then (cache,Values.BOOL(true));

    case (_,_,"generateSeparateCode",{v,Values.BOOL(_)},_)
      equation
        sp = SymbolTable.getSCode();
        name = getTypeNameIdent(v);
        failure(_ = List.getMemberOnTrue(name, sp, SCode.isClassNamed));
        Error.addMessage(Error.LOOKUP_ERROR, {name,"<TOP>"});
      then fail();

    case (cache,_,"generateSeparateCode",_,_)
      equation
        setGlobalRoot(Global.instOnlyForcedFunctions,NONE());
      then (cache,Values.BOOL(false));

    case (_,_,"loadModel",{Values.CODE(Absyn.C_TYPENAME(path)),Values.ARRAY(valueLst=cvars),Values.BOOL(b),Values.STRING(str),Values.BOOL(requireExactVersion)},_)
      equation
        p = SymbolTable.getAbsyn();
        execStatReset();
        mp = Settings.getModelicaPath(Config.getRunningTestsuite());
        strings = List.map(cvars, ValuesUtil.extractValueString);
        /* If the user requests a custom version to parse as, set it up */
        oldLanguageStd = Config.getLanguageStandard();
        b1 = not stringEq(str,"");
        if b1 then
          Config.setLanguageStandard(Config.versionStringToStd(str));
        end if;
        (p,b) = loadModel({(path,strings,false)},mp,p,true,b,true,requireExactVersion,false);
        if b1 then
          Config.setLanguageStandard(oldLanguageStd);
        end if;
        Print.clearBuf();
        SymbolTable.setAbsyn(p);
        execStat("loadModel("+Absyn.pathString(path)+")");
      then (FCore.emptyCache(),Values.BOOL(b));

    case (cache,_,"loadModel",Values.CODE(Absyn.C_TYPENAME(path))::_,_)
      equation
        pathstr = Absyn.pathString(path);
        Error.addMessage(Error.LOAD_MODEL_ERROR, {pathstr});
      then
        (cache,Values.BOOL(false));

    case (_,_,"loadFile",Values.STRING(name)::Values.STRING(encoding)::Values.BOOL(b)::_,_)
      equation
        execStatReset();
        name = Util.testsuiteFriendlyPath(name);
        newp = loadFile(name, encoding, SymbolTable.getAbsyn(), b);
        execStat("loadFile("+name+")");
        SymbolTable.setAbsyn(newp);
      then
        (FCore.emptyCache(),Values.BOOL(true));

    case (cache,_,"loadFile",_,_)
      then (cache,Values.BOOL(false));

    case (_,_,"loadFiles",Values.ARRAY(valueLst=vals)::Values.STRING(encoding)::Values.INTEGER(i)::_,_)
      equation
        strs = List.mapMap(vals,ValuesUtil.extractValueString,Util.testsuiteFriendlyPath);
        newps = Parser.parallelParseFilesToProgramList(strs,encoding,numThreads=i);
        newp = List.fold(newps, function checkUsesAndUpdateProgram(checkUses=false, modelicaPath=Settings.getModelicaPath(Config.getRunningTestsuite())), SymbolTable.getAbsyn());
        SymbolTable.setAbsyn(newp);
      then
        (FCore.emptyCache(),Values.BOOL(true));

    case (cache,_,"loadFiles",_,_)
      equation
        // System.GC_enable();
      then (cache,Values.BOOL(false));

    case (_,_,"loadEncryptedPackage",Values.STRING(filename)::Values.STRING(workdir)::_,_)
      equation
        b = false;
        if (System.regularFileExists(filename)) then
          if (Util.endsWith(filename, ".mol")) then
            workdir = if System.directoryExists(workdir) then workdir else System.pwd();
            b = false;
            if (0 == System.systemCall("unzip -q -o -d \"" + workdir + "\" \"" +  filename + "\"")) then
              b = true;
              s1 = System.basename(filename);
              s2 = Util.removeLast4Char(s1);
              filename1 = workdir + "/" + s2 + "/" + s2 + ".moc";
              filename2 = workdir + "/" + s2 + "/package.moc";
              filename_1 = if System.regularFileExists(filename1) then filename1 else filename2;
              if (System.regularFileExists(filename_1)) then
                filename_1 = Util.testsuiteFriendlyPath(filename_1);
                p = SymbolTable.getAbsyn();
                newp = loadFile(filename_1, "UTF-8", p, true);
                execStat("loadFile("+filename_1+")");
                SymbolTable.setAbsyn(newp);
              else
                Error.addMessage(Error.ENCRYPTED_FILE_NOT_FOUND_ERROR, {filename1, filename2});
              end if;
            else
              Error.addMessage(Error.UNABLE_TO_UNZIP_FILE, {filename});
            end if;
          else
            Error.addMessage(Error.EXPECTED_ENCRYPTED_PACKAGE, {filename});
          end if;
        else
          Error.addMessage(Error.FILE_NOT_FOUND_ERROR, {filename});
        end if;
      then
        (FCore.emptyCache(),Values.BOOL(b));

    case (cache,_,"loadEncryptedPackage",_,_)
      then
        (cache,Values.BOOL(false));

    case (cache,_,"alarm",{Values.INTEGER(i)},_)
      equation
        i = System.alarm(i);
      then (cache,Values.INTEGER(i));

    case (cache,_,"getClassNames",{Values.CODE(Absyn.C_TYPENAME(Absyn.IDENT("AllLoadedClasses"))),Values.BOOL(false),_,Values.BOOL(sort),Values.BOOL(builtin),Values.BOOL(_),_},_)
      equation
        (ip,_) = FBuiltin.getInitialFunctions();
        p = SymbolTable.getAbsyn();
        p = if builtin then Interactive.updateProgram(p,ip) else p;
        paths = Interactive.getTopClassnames(p);
        paths = if sort then List.sort(paths, Absyn.pathGe) else paths;
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then
        (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"getClassNames",{Values.CODE(Absyn.C_TYPENAME(path)),Values.BOOL(false),Values.BOOL(b),Values.BOOL(sort),Values.BOOL(builtin),Values.BOOL(showProtected),Values.BOOL(includeConstants)},_)
      equation
        (ip,_) = FBuiltin.getInitialFunctions();
        p = SymbolTable.getAbsyn();
        p = if builtin then Interactive.updateProgram(p,ip) else p;
        paths = Interactive.getClassnamesInPath(path, p, showProtected, includeConstants);
        paths = if b then List.map1r(paths,Absyn.joinPaths,path) else paths;
        paths = if sort then List.sort(paths, Absyn.pathGe) else paths;
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then
        (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"getClassNames",{Values.CODE(Absyn.C_TYPENAME(Absyn.IDENT("AllLoadedClasses"))),Values.BOOL(true),_,Values.BOOL(sort),Values.BOOL(builtin),Values.BOOL(showProtected),Values.BOOL(includeConstants)},_)
      equation
        (ip,_) = FBuiltin.getInitialFunctions();
        p = SymbolTable.getAbsyn();
        p = if builtin then Interactive.updateProgram(p,ip) else p;
        (_,paths) = Interactive.getClassNamesRecursive(NONE(),p,showProtected,includeConstants,{});
        paths = listReverse(paths);
        paths = if sort then List.sort(paths, Absyn.pathGe) else paths;
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then
        (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"getClassNames",{Values.CODE(Absyn.C_TYPENAME(path)),Values.BOOL(true),_,Values.BOOL(sort),Values.BOOL(builtin),Values.BOOL(showProtected),Values.BOOL(includeConstants)},_)
      equation
        (ip,_) = FBuiltin.getInitialFunctions();
        p = SymbolTable.getAbsyn();
        p = if builtin then Interactive.updateProgram(p,ip) else p;
        (_,paths) = Interactive.getClassNamesRecursive(SOME(path),p,showProtected,includeConstants,{});
        paths = listReverse(paths);
        paths = if sort then List.sort(paths, Absyn.pathGe) else paths;
        vals = List.map(paths,ValuesUtil.makeCodeTypeName);
      then
        (cache,ValuesUtil.makeArray(vals));

    case (cache,_,"reloadClass",{Values.CODE(Absyn.C_TYPENAME(classpath)),Values.STRING(encoding)},_)
      equation
        Absyn.CLASS(info=SOURCEINFO(fileName=filename,lastModification=r2)) = Interactive.getPathedClassInProgram(classpath, SymbolTable.getAbsyn());
        (true,_,r1) = System.stat(filename);
        b = realEq(r1,r2);
        if not b then
          reloadClass(filename, encoding);
        end if;
      then (cache,Values.BOOL(true));

    case (cache,_,"reloadClass",{Values.CODE(Absyn.C_TYPENAME(classpath)),_},_)
      equation
        failure(_ = Interactive.getPathedClassInProgram(classpath, SymbolTable.getAbsyn()));
        str = Absyn.pathString(classpath);
        Error.addMessage(Error.LOAD_MODEL_ERROR, {str});
      then (cache,Values.BOOL(false));

    case (cache,_,"reloadClass",_,_)
      then (cache,Values.BOOL(false));

    case (_,_,"loadString",Values.STRING(str)::Values.STRING(name)::Values.STRING(encoding)::Values.BOOL(mergeAST)::_,_)
      equation
        str = if not (encoding == "UTF-8") then System.iconv(str, encoding, "UTF-8") else str;
        newp = Parser.parsestring(str,name);
        newp = Interactive.updateProgram(newp, SymbolTable.getAbsyn(), mergeAST);
        SymbolTable.setAbsyn(newp);
      then (FCore.emptyCache(), Values.BOOL(true));

    case (cache,_,"loadString",_,_)
    then (cache,Values.BOOL(false));

    case (cache,_,"help",{Values.STRING("")},_)
      equation
        str = Flags.printUsage();
      then
        (cache,Values.STRING(str));

    case (cache,_,"help",{Values.STRING(str)},_)
      equation
        str = Flags.printHelp({str});
      then
        (cache,Values.STRING(str));

    case (cache,_,"getTimeStamp",{Values.CODE(Absyn.C_TYPENAME(classpath))},_)
      equation
        Absyn.CLASS(info=SOURCEINFO(lastModification=r)) = Interactive.getPathedClassInProgram(classpath,SymbolTable.getAbsyn());
        str = System.ctime(r);
      then (cache,Values.TUPLE({Values.REAL(r),Values.STRING(str)}));

    case (cache,_,"getTimeStamp",_,_)
      then
        (cache,Values.TUPLE({Values.REAL(0.0),Values.STRING("")}));

    case (cache,_,"getClassRestriction",{Values.CODE(Absyn.C_TYPENAME(classpath))},_)
      equation
        str = Interactive.getClassRestriction(classpath, SymbolTable.getAbsyn());
      then
        (cache,Values.STRING(str));

    case (cache,_,"classAnnotationExists",{Values.CODE(Absyn.C_TYPENAME(classpath)),Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        b = Interactive.getNamedAnnotation(classpath, SymbolTable.getAbsyn(), path, SOME(false), isSome);
      then
        (cache,Values.BOOL(b));

    case (cache,_,"getBooleanClassAnnotation",{Values.CODE(Absyn.C_TYPENAME(classpath)),Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        Absyn.BOOL(b) = Interactive.getNamedAnnotation(classpath, SymbolTable.getAbsyn(), path, NONE(), Interactive.getAnnotationExp);
      then
        (cache,Values.BOOL(b));

    case (_,_,"getBooleanClassAnnotation",{Values.CODE(Absyn.C_TYPENAME(classpath)),Values.CODE(Absyn.C_TYPENAME(path))},_)
      equation
        str1 = Absyn.pathString(path);
        str2 = Absyn.pathString(classpath);
        Error.addMessage(Error.CLASS_ANNOTATION_DOES_NOT_EXIST, {str1,str2});
      then fail();

    case (cache,_,"strtok",{Values.STRING(str),Values.STRING(token)},_)
      equation
        vals = List.map(System.strtok(str,token), ValuesUtil.makeString);
        i = listLength(vals);
      then (cache,Values.ARRAY(vals,{i}));

    case (cache,_,"stringSplit",{Values.STRING(str),Values.STRING(token)},_)
      equation
        vals = List.map(Util.stringSplitAtChar(str,token), ValuesUtil.makeString);
        i = listLength(vals);
      then (cache,Values.ARRAY(vals,{i}));

    case (cache,_,"stringReplace",{Values.STRING(str1),Values.STRING(str2),Values.STRING(str3)},_)
      equation
        str = System.stringReplace(str1, str2, str3);
      then (cache,Values.STRING(str));

        /* Checks the installation of OpenModelica and tries to find common errors */
    case (cache,_,"checkSettings",{},_)
      equation
        vars_1 = {"OPENMODELICAHOME",
                  "OPENMODELICALIBRARY",
                  "OMC_PATH",
                  "SYSTEM_PATH",
                  "OMDEV_PATH",
                  "OMC_FOUND",
                  "MODELICAUSERCFLAGS",
                  "WORKING_DIRECTORY",
                  "CREATE_FILE_WORKS",
                  "REMOVE_FILE_WORKS",
                  "OS",
                  "SYSTEM_INFO",
                  "RTLIBS",
                  "C_COMPILER",
                  "C_COMPILER_VERSION",
                  "C_COMPILER_RESPONDING",
                  "HAVE_CORBA",
                  "CONFIGURE_CMDLINE"};
        omhome = Settings.getInstallationDirectoryPath();
        omlib = Settings.getModelicaPath(Config.getRunningTestsuite());
        omcpath = omhome + "/bin/omc" + System.getExeExt();
        systemPath = Util.makeValueOrDefault(System.readEnv,"PATH","");
        omdev = Util.makeValueOrDefault(System.readEnv,"OMDEV","");
        omcfound = System.regularFileExists(omcpath);
        os = System.os();
        touch_file = "omc.checksettings.create_file_test";
        usercflags = Util.makeValueOrDefault(System.readEnv,"MODELICAUSERCFLAGS","");
        workdir = System.pwd();
        touch_res = 0 == System.systemCall("touch " + touch_file, "");
        System.systemCall("uname -a", touch_file);
        uname = System.readFile(touch_file);
        rm_res = 0 == System.systemCall("rm " + touch_file, "");
        // _ = System.platform();
        senddata = System.getRTLibs();
        gcc = System.getCCompiler();
        have_corba = Corba.haveCorba();
        System.systemCall("rm -f " + touch_file, "");
        gcc_res = 0 == System.systemCall(gcc + " --version", touch_file);
        gccVersion = System.readFile(touch_file);
        System.systemCall("rm -f " + touch_file, "");
        confcmd = System.configureCommandLine();
        vals = {Values.STRING(omhome),
                Values.STRING(omlib),
                Values.STRING(omcpath),
                Values.STRING(systemPath),
                Values.STRING(omdev),
                Values.BOOL(omcfound),
                Values.STRING(usercflags),
                Values.STRING(workdir),
                Values.BOOL(touch_res),
                Values.BOOL(rm_res),
                Values.STRING(os),
                Values.STRING(uname),
                Values.STRING(senddata),
                Values.STRING(gcc),
                Values.STRING(gccVersion),
                Values.BOOL(gcc_res),
                Values.BOOL(have_corba),
                Values.STRING(confcmd)};
      then (cache,Values.RECORD(Absyn.IDENT("OpenModelica.Scripting.CheckSettingsResult"),vals,vars_1,-1));

    case (cache,_,"echo",{v as Values.BOOL(bval)},_)
      equation
        Settings.setEcho(if bval then 1 else 0);
      then (cache,v);

    case (cache,_,"numProcessors",{},_)
      equation
        i = Config.noProc();
      then (cache,Values.INTEGER(i));

    case (cache,_,"runScript",{Values.STRING(str)},_)
      equation
        str = Util.testsuiteFriendlyPath(str);
        istmts = Parser.parseexp(str);
        res = Interactive.evaluate(istmts, true);
      then
        (cache,Values.STRING(res));

    case (cache,_,"runScript",_,_)
      then (cache,Values.STRING("Failed"));

    case (_,_,"exit",{Values.INTEGER(i)},_)
      equation
        System.exit(i);
        /* Cannot reach here */
      then fail();

    case (cache,_,"getMemorySize",{},_)
      equation
        r = System.getMemorySize();
        v = Values.REAL(r);
      then (cache,v);

    else
      algorithm
        (cache,v) := CevalScriptBackend.cevalInteractiveFunctions3(inCache,inEnv,inFunctionName,inVals,msg);
      then (cache,v);

 end matchcontinue;
end cevalInteractiveFunctions2;

public function evalCodeTypeName
  input Values.Value val;
  input FCore.Graph env;
  output Values.Value res;
algorithm
  res := matchcontinue (val,env)
    local
      Absyn.Path path;
    case (Values.CODE(Absyn.C_TYPENAME(path as Absyn.IDENT(_) /* We only want to lookup idents in the symboltable; also speeds up e.g. simulate(Modelica.A.B.C) so we do not instantiate all classes */)),_)
      equation
        (_,_,_,DAE.VALBOUND(valBound=res as Values.CODE(A=Absyn.C_TYPENAME())),_,_,_,_,_) = Lookup.lookupVar(FCore.emptyCache(), env, ComponentReference.pathToCref(path));
      then res;
    else val;
  end matchcontinue;
end evalCodeTypeName;

protected function getVariableNames
  input list<GlobalScript.Variable> vars;
  input list<Values.Value> acc;
  output list<Values.Value> ovars;
algorithm
  ovars := match (vars,acc)
    local
      list<GlobalScript.Variable> vs;
      String p;
    case ({},_) then listReverse(acc);
    case (GlobalScript.IVAR(varIdent = "$echo") :: vs,_)
      then getVariableNames(vs,acc);
    case (GlobalScript.IVAR(varIdent = p) :: vs,_)
      then getVariableNames(vs,Values.CODE(Absyn.C_VARIABLENAME(Absyn.CREF_IDENT(p,{})))::acc);
  end match;
end getVariableNames;

public function getPackageVersion
  input Absyn.Path path;
  input Absyn.Program p;
  output String version = "";
protected
  Boolean evalParamAnn;
algorithm
  evalParamAnn := Config.getEvaluateParametersInAnnotations();
  Config.setEvaluateParametersInAnnotations(true);
  try
    Absyn.STRING(version) := Interactive.getNamedAnnotation(path, p, Absyn.IDENT("version"), SOME(Absyn.STRING("")), Interactive.getAnnotationExp);
  else
    version := "";
  end try;
  Config.setEvaluateParametersInAnnotations(evalParamAnn);
end getPackageVersion;

protected function errorToValue
  input Error.TotalMessage err;
  output Values.Value val;
algorithm
  val := match err
    local
      Absyn.Path msgpath;
      Values.Value tyVal,severityVal,infoVal;
      list<Values.Value> values;
      Util.TranslatableContent message;
      String msg_str;
      Integer id;
      Error.Severity severity;
      Error.MessageType ty;
      SourceInfo info;
    case Error.TOTALMESSAGE(Error.MESSAGE(id,ty,severity,message),info)
      equation
        msg_str = Util.translateContent(message);
        msgpath = Absyn.FULLYQUALIFIED(Absyn.QUALIFIED("OpenModelica",Absyn.QUALIFIED("Scripting",Absyn.IDENT("ErrorMessage"))));
        tyVal = errorTypeToValue(ty);
        severityVal = errorLevelToValue(severity);
        infoVal = infoToValue(info);
        values = {infoVal,Values.STRING(msg_str),tyVal,severityVal,Values.INTEGER(id)};
      then Values.RECORD(msgpath,values,{"info","message","kind","level","id"},-1);
  end match;
end errorToValue;

protected function infoToValue
  input SourceInfo info;
  output Values.Value val;
algorithm
  val := match info
    local
      list<Values.Value> values;
      Absyn.Path infopath;
      Integer ls,cs,le,ce;
      String filename;
      Boolean readonly;
    case SOURCEINFO(filename,readonly,ls,cs,le,ce,_)
      equation
        infopath = Absyn.FULLYQUALIFIED(Absyn.QUALIFIED("OpenModelica",Absyn.QUALIFIED("Scripting",Absyn.IDENT("SourceInfo"))));
        values = {Values.STRING(filename),Values.BOOL(readonly),Values.INTEGER(ls),Values.INTEGER(cs),Values.INTEGER(le),Values.INTEGER(ce)};
      then Values.RECORD(infopath,values,{"filename","readonly","lineStart","columnStart","lineEnd","columnEnd"},-1);
  end match;
end infoToValue;

protected function makeErrorEnumLiteral
  input String enumName;
  input String enumField;
  input Integer index;
  output Values.Value val;
  annotation(__OpenModelica_EarlyInline=true);
algorithm
  val := Values.ENUM_LITERAL(Absyn.FULLYQUALIFIED(Absyn.QUALIFIED("OpenModelica",Absyn.QUALIFIED("Scripting",Absyn.QUALIFIED(enumName,Absyn.IDENT(enumField))))),index);
end makeErrorEnumLiteral;

protected function errorTypeToValue
  input Error.MessageType ty;
  output Values.Value val;
algorithm
  val := match ty
    case Error.SYNTAX() then makeErrorEnumLiteral("ErrorKind","syntax",1);
    case Error.GRAMMAR() then makeErrorEnumLiteral("ErrorKind","grammar",2);
    case Error.TRANSLATION() then makeErrorEnumLiteral("ErrorKind","translation",3);
    case Error.SYMBOLIC() then makeErrorEnumLiteral("ErrorKind","symbolic",4);
    case Error.SIMULATION() then makeErrorEnumLiteral("ErrorKind","runtime",5);
    case Error.SCRIPTING() then makeErrorEnumLiteral("ErrorKind","scripting",6);
    else
      equation
        print("errorTypeToValue failed\n");
      then fail();
  end match;
end errorTypeToValue;

protected function errorLevelToValue
  input Error.Severity severity;
  output Values.Value val;
algorithm
  val := match severity
    case Error.ERROR() then makeErrorEnumLiteral("ErrorLevel","error",1);
    case Error.WARNING() then makeErrorEnumLiteral("ErrorLevel","warning",2);
    case Error.NOTIFICATION() then makeErrorEnumLiteral("ErrorLevel","notification",3);
    else
      equation
        print("errorLevelToValue failed\n");
      then fail();
  end match;
end errorLevelToValue;

protected function generateFunctionName
"@author adrpo:
 generate the function name from a path."
  input Absyn.Path functionPath;
  output String functionName;
algorithm
  functionName := Absyn.pathStringUnquoteReplaceDot(functionPath, "_");
end generateFunctionName;

protected function generateFunctionFileName
"@author adrpo:
 generate the function name from a path."
  input Absyn.Path functionPath;
  output String functionName;
algorithm
  functionName := matchcontinue(functionPath)
    local String name, n1, n2; Integer len;
    case (_)
      equation
        name = Absyn.pathStringUnquoteReplaceDot(functionPath, "_");
        len = stringLength(name);
        // not bigger than
        true = len > Global.maxFunctionFileLength;
        n1 = Absyn.pathFirstIdent(functionPath);
        n2 = Absyn.pathLastIdent(functionPath);
        name = System.unquoteIdentifier(n1 + "_" + n2);
        name = name + "_" + intString(tick());
      then
        name;
    else
      equation
        name = Absyn.pathStringUnquoteReplaceDot(functionPath, "_");
      then
        name;
  end matchcontinue;
end generateFunctionFileName;

public function getFunctionDependencies
"returns all function dependencies as paths, also the main function and the function tree"
  input FCore.Cache cache;
  input Absyn.Path functionName;
  output DAE.Function mainFunction "the main function";
  output list<Absyn.Path> dependencies "the dependencies as paths";
  output DAE.FunctionTree funcs "the function tree";
algorithm
  funcs := FCore.getFunctionTree(cache);
  // First check if the main function exists... If it does not it might be an interactive function...
  mainFunction := DAEUtil.getNamedFunction(functionName, funcs);
  dependencies := SimCodeFunction.getCalledFunctionsInFunction(functionName,funcs);
end getFunctionDependencies;

public function collectDependencies
"collects all function dependencies, also the main function, uniontypes, metarecords"
  input FCore.Cache inCache;
  input FCore.Graph env;
  input Absyn.Path functionName;
  output FCore.Cache outCache;
  output DAE.Function mainFunction;
  output list<DAE.Function> dependencies;
  output list<DAE.Type> metarecordTypes;
protected
  list<Absyn.Path> uniontypePaths,paths;
  DAE.FunctionTree funcs;
algorithm
  (mainFunction, paths, funcs) := getFunctionDependencies(inCache, functionName);
  // The list of functions is not ordered, so we need to filter out the main function...
  dependencies := List.map1(paths, DAEUtil.getNamedFunction, funcs);
  dependencies := List.setDifference(dependencies, {mainFunction});
  uniontypePaths := DAEUtil.getUniontypePaths(dependencies,{});
  (outCache,metarecordTypes) := Lookup.lookupMetarecordsRecursive(inCache, env, uniontypePaths);
end collectDependencies;

public function cevalGenerateFunction "Generates code for a given function name."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input Absyn.Program program;
  input Absyn.Path inPath;
  output FCore.Cache outCache;
  output String functionName;
  output String functionFileName;
algorithm
  (outCache,functionName,functionFileName) := matchcontinue (inCache,inEnv,program,inPath)
    local
      String pathstr, fileName;
      FCore.Graph env;
      Absyn.Path path;
      FCore.Cache cache;
      DAE.Function mainFunction;
      list<DAE.Function> d;
      list<DAE.Type> metarecordTypes;
      DAE.FunctionTree funcs;
    // template based translation
    case (cache, env, _, path)
      equation
        true = Flags.isSet(Flags.GEN);
        false = Flags.isSet(Flags.GENERATE_CODE_CHEAT);

        (cache, mainFunction, d, metarecordTypes) = collectDependencies(cache, env, path);

        pathstr  = generateFunctionName(path);
        fileName = generateFunctionFileName(path);
        SimCodeFunction.translateFunctions(program, fileName, SOME(mainFunction), d, metarecordTypes, {});
        compileModel(fileName, {});
      then
        (cache, pathstr, fileName);

    // Cheat if we want to generate code for Main.main
    // * Don't do dependency analysis of what functions to generate; just generate all of them
    // * Don't generate extra code for unreferenced MetaRecord types (for external functions)
    //   This could be an annotation instead anyway.
    // * Don't compile the generated files
    case (cache, _, _, path)
      equation
        true = Flags.isSet(Flags.GEN);
        true = Flags.isSet(Flags.GENERATE_CODE_CHEAT);
        funcs = FCore.getFunctionTree(cache);
        // First check if the main function exists... If it does not it might be an interactive function...
        pathstr = generateFunctionName(path);
        fileName = generateFunctionFileName(path);
        // The list of functions is not ordered, so we need to filter out the main function...
        d = DAEUtil.getFunctionList(funcs);
        SimCodeFunction.translateFunctions(program, fileName, NONE(), d, {}, {});
      then
        (cache, pathstr, fileName);

    case (cache, env, _, path)
      equation
        true = Flags.isSet(Flags.GEN);
        true = Flags.isSet(Flags.FAILTRACE);
        (cache,false) = Static.isExternalObjectFunction(cache,env,path);
        pathstr = generateFunctionName(path);
        fileName = generateFunctionFileName(path);
        Debug.trace("CevalScript.cevalGenerateFunction failed:\nfunction: " + pathstr + "\nfile: " + fileName + "\n");
      then
        fail();
  end matchcontinue;
end cevalGenerateFunction;

protected function matchQualifiedCalls
"Collects the packages used by the functions"
  input DAE.Exp inExp;
  input list<String> inAcc;
  output DAE.Exp outExp = inExp;
  output list<String> outAcc;
algorithm
  outAcc := match inExp
    local
      String name;

    case DAE.REDUCTION(reductionInfo = DAE.REDUCTIONINFO(path = Absyn.FULLYQUALIFIED(Absyn.QUALIFIED(name = name))))
      then List.consOnTrue(not listMember(name, inAcc), name, inAcc);

    case DAE.CALL(path = Absyn.FULLYQUALIFIED(Absyn.QUALIFIED(name = name)),
                  attr = DAE.CALL_ATTR(builtin = false))
      then List.consOnTrue(not listMember(name, inAcc), name, inAcc);

    case DAE.CREF(componentRef = DAE.CREF_QUAL(ident = name),
                  ty = DAE.T_FUNCTION_REFERENCE_FUNC(builtin = false))
      then List.consOnTrue(not listMember(name, inAcc), name, inAcc);

    case DAE.PARTEVALFUNCTION(path = Absyn.FULLYQUALIFIED(Absyn.QUALIFIED(name = name)))
      then List.consOnTrue(not listMember(name, inAcc), name, inAcc);

    else inAcc;
  end match;
end matchQualifiedCalls;

protected function instantiateDaeFunctions
  input FCore.Cache icache;
  input FCore.Graph ienv;
  input list<Absyn.Path> ipaths;
  output FCore.Cache outCache;
algorithm
  outCache := match (icache,ienv,ipaths)
    local
      Absyn.Path path;
      FCore.Cache cache; FCore.Graph env;
      list<Absyn.Path> paths;
    case (cache,_,{}) then cache;
    case (cache,env,path::paths)
      equation
        // print("force inst: " + Absyn.pathString(path));
        (cache,Util.SUCCESS()) = Static.instantiateDaeFunctionForceInst(cache,env,path,false,NONE(),true);
        // print(" ok\n");
        cache = instantiateDaeFunctions(cache,env,paths);
      then cache;
  end match;
end instantiateDaeFunctions;

function generateFunctions
  input FCore.Cache icache;
  input FCore.Graph ienv;
  input Absyn.Program p;
  input SCode.Program fullScodeProgram;
  input list<SCode.Element> isp;
  input Boolean cleanCache;
  output FCore.Cache cache;
  output FCore.Graph env;
algorithm
  (cache,env) := match (icache,ienv,p,isp,cleanCache)
    local
      String name;
      list<String> names,dependencies;
      list<Absyn.Path> paths;
      list<SCode.Element> elementLst;
      DAE.FunctionTree funcs;
      list<DAE.Function> d;
      list<tuple<String,list<String>>> acc;
      list<SCode.Element> sp;
      String file,nameHeader,str;
      Integer n;
      SourceInfo info;
      SCode.Element cl;
      SCode.Restriction restr;

    case (cache,env,_,{},_) then (cache,env);
    case (cache,env,_,(cl as SCode.CLASS(name=name,encapsulatedPrefix=SCode.ENCAPSULATED(),restriction=restr,info=info))::sp,_)
      algorithm
        _ := match restr
          case SCode.R_PACKAGE() then ();
          case SCode.R_UNIONTYPE() then ();
          else
            algorithm
              Error.addSourceMessage(Error.INTERNAL_ERROR, {"Only package and uniontype is supported as top-level classes in OpenModelica."}, info);
            then fail();
        end match;
        (cache,env) := generateFunctions2(cache,env,p,fullScodeProgram,cl,name,info,cleanCache);
        (cache,env) := generateFunctions(cache,env,p,fullScodeProgram,sp,cleanCache);
      then (cache,env);
    case (_,_,_,SCode.CLASS(encapsulatedPrefix=SCode.NOT_ENCAPSULATED(),name=name,info=info as SOURCEINFO(fileName=file))::_,_)
      equation
        (n,_) = System.regex(file, "ModelicaBuiltin.mo$", 1, false, false);
        Error.assertion(n > 0, "Not an encapsulated class (required for separate compilation): " + name, info);
      then fail();
  end match;
end generateFunctions;

function generateFunctions2
  input FCore.Cache icache;
  input FCore.Graph ienv;
  input Absyn.Program p;
  input SCode.Program sp;
  input SCode.Element cl;
  input String name;
  input SourceInfo info;
  input Boolean cleanCache;
  output FCore.Cache cache;
  output FCore.Graph env;
algorithm
  (cache,env) := matchcontinue (icache,ienv,p,cl,name,info,cleanCache)
    local
      list<String> names,dependencies,strs;
      list<Absyn.Path> paths, pathsMetarecord;
      DAE.FunctionTree funcs;
      list<DAE.Function> d;
      list<tuple<String,list<String>>> acc;
      String file,nameHeader,str;
      Integer n;
      FCore.Graph env2;
      FCore.Ref ref;
      FCore.Cache lookupCache;
      FCore.Children children;
      Absyn.Path path;
      list<SCode.Element> elements;
      list<DAE.Type> metarecords;
      DAE.Type t;

    case (cache,env,_,_,_,SOURCEINFO(fileName=file),_)
      equation
        (1,_) = System.regex(file, "ModelicaBuiltin.mo$", 1, false, false);
      then (cache,env);

    case (cache,env,_,_,_,_,_)
      algorithm
        cache := if cleanCache then FCore.emptyCache() else cache;

        if SCode.isPartial(cl) then
          paths := {};
          pathsMetarecord := {};
        else
          path := Absyn.FULLYQUALIFIED(Absyn.IDENT(name));
          elements := getNonPartialElementsForInstantiatedClass(sp, cl, path);
          (paths, pathsMetarecord) := List.fold22(elements, findFunctionsToCompile, path, sp, {}, {});
        end if;

        metarecords := {};
        for mr in pathsMetarecord loop
          (cache,t) := Lookup.lookupType(cache, env, mr, SOME(info));
          metarecords := t::metarecords;
        end for;

        cache := instantiateDaeFunctions(cache, env, paths);
        funcs := FCore.getFunctionTree(cache);
        d := List.map2(paths, DAEUtil.getNamedFunctionWithError, funcs, info);
        (_,(_,dependencies)) := DAEUtil.traverseDAEFunctions(d,Expression.traverseSubexpressionsHelper,(matchQualifiedCalls,{}));
        // print(name + " has dependencies: " + stringDelimitList(dependencies,",") + "\n");
        dependencies := List.sort(dependencies,Util.strcmpBool);
        dependencies := List.map1(dependencies,stringAppend,".h");
        nameHeader := name + ".h";
        strs := List.map1r(nameHeader::dependencies, stringAppend, "$(GEN_DIR)");
        System.writeFile(name + ".deps", "$(GEN_DIR)" + name + ".o: $(GEN_DIR)" + name + ".c" + " " + stringDelimitList(strs," "));
        dependencies := List.map1(dependencies,stringAppend,"\"");
        dependencies := List.map1r(dependencies,stringAppend,"#include \"");
        SimCodeFunction.translateFunctions(p, name, NONE(), d, {}, dependencies);
        str := Tpl.tplString(Unparsing.programExternalHeaderFromTypes, metarecords);
        System.writeFile(name + "_records.c","#include <meta/meta_modelica.h>\n" + str);
        cache := if cleanCache then icache else cache;
      then (cache,env);
    else
      equation
        Error.addSourceMessage(Error.SEPARATE_COMPILATION_PACKAGE_FAILED,{name},info);
      then fail();
  end matchcontinue;
end generateFunctions2;

function findFunctionsToCompile
  input SCode.Element elt;
  input Absyn.Path pathPrefix;
  input SCode.Program sp;
  input list<Absyn.Path> acc;
  input list<Absyn.Path> accMetarecord;
  output list<Absyn.Path> paths;
  output list<Absyn.Path> pathsMetarecord;
protected
  String name;
  Absyn.Path path;
  list<SCode.Element> elements;
algorithm
  SCode.CLASS(name=name) := elt;
  path := Absyn.joinPaths(pathPrefix, Absyn.IDENT(name));
  paths := if SCode.isFunction(elt) then path::acc else acc;
  pathsMetarecord := match elt
    case SCode.CLASS(restriction=SCode.R_METARECORD()) then path::accMetarecord;
    else accMetarecord;
  end match;
  elements := getNonPartialElementsForInstantiatedClass(sp, elt, path);
  (paths,pathsMetarecord) := List.fold22(elements, findFunctionsToCompile, path, sp, paths, pathsMetarecord);
end findFunctionsToCompile;

function getNonPartialElementsForInstantiatedClass "Gets the non-partial elements returned by instantiating the given path"
  input SCode.Program sp;
  input SCode.Element cl;
  input Absyn.Path p;
  output list<SCode.Element> elts;
protected
  FCore.Graph env;
  SCode.Element elt;
  Boolean skip;
  list<SCode.Element> eltsTmp;
algorithm
  skip := match cl
    case SCode.CLASS(classDef=SCode.CLASS_EXTENDS()) then false;
    case SCode.CLASS(classDef=SCode.PARTS(elementLst=eltsTmp)) then not List.exist(eltsTmp, SCode.isElementExtendsOrClassExtends);
    else true;
  end match;
  if not skip then
  try
    ErrorExt.setCheckpoint("getNonPartialElementsForInstantiatedClass");
    (, env) := Inst.instantiateClass(FCore.emptyCache(), InnerOuter.emptyInstHierarchy, sp, Absyn.makeNotFullyQualified(p), doSCodeDep=false);
    elts := FCore.RefTree.fold(FNode.children(FNode.fromRef(FGraph.lastScopeRef(env))),
      addNonPartialClassRef, {});
    ErrorExt.rollBack("getNonPartialElementsForInstantiatedClass");
    return;
  else
  end try;
  ErrorExt.rollBack("getNonPartialElementsForInstantiatedClass");
  end if;
  // Failed to instantiate the class; perhaps due to being a function
  // that cannot be instantiated using model restrictions.
  elts := match cl
    case SCode.CLASS(classDef=SCode.PARTS(elementLst=elts)) then list(e for e guard (not SCode.isPartial(e)) and SCode.isClass(e) in elts);
    else {};
  end match;
end getNonPartialElementsForInstantiatedClass;

protected function addNonPartialClassRef
  input FCore.Name name;
  input FCore.Ref ref;
  input list<SCode.Element> accum;
  output list<SCode.Element> classes;
protected
  SCode.Element e;
algorithm
  classes := match FNode.fromRef(ref)
    case FCore.N(data = FCore.CL(e = e as SCode.CLASS(partialPrefix = SCode.NOT_PARTIAL())))
      then e :: accum;

    else accum;
  end match;
end addNonPartialClassRef;

public function cevalCallFunction "This function evaluates CALL expressions, i.e. function calls.
  They are currently evaluated by generating code for the function and
  then dynamicly load the function and call it."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input DAE.Exp inExp;
  input list<Values.Value> inValuesValueLst;
  input Boolean impl;
  input Absyn.Msg inMsg;
  input Integer numIter;
  output FCore.Cache outCache;
  output Values.Value outValue;
algorithm
  (outCache,outValue) := matchcontinue (inCache,inEnv,inExp,inValuesValueLst,impl,inMsg,numIter)
    local
      Values.Value newval;
      FCore.Graph env;
      DAE.Exp e;
      Absyn.Path funcpath;
      list<DAE.Exp> expl;
      list<Values.Value> vallst, pubVallst, proVallst;
      Absyn.Msg msg;
      FCore.Cache cache;
      Absyn.Path complexName;
      list<DAE.Var> pubVarLst, proVarLst, varLst;
      list<String> pubVarNames, proVarNames, varNames;
      DAE.Type ty;
      SourceInfo info;
      String str;
      Boolean bIsCompleteFunction;

    // External functions that are "known" should be evaluated without compilation, e.g. all math functions
    case (cache,env,(DAE.CALL(path = funcpath)),vallst,_,msg,_)
      equation
        (cache,newval) = Ceval.cevalKnownExternalFuncs(cache,env, funcpath, vallst, msg);
      then
        (cache,newval);

    // This case prevents the constructor call of external objects of being evaluated
    case (cache,env,(DAE.CALL(path = funcpath)),_,_,msg,_)
      equation
        true = FGraph.isNotEmpty(env);
        cevalIsExternalObjectConstructor(cache,funcpath,env,msg);
      then
        fail();

    // Record constructors
    case(cache,env,(DAE.CALL(path = funcpath,attr = DAE.CALL_ATTR(ty = DAE.T_COMPLEX(complexClassType = ClassInf.RECORD(complexName), varLst=varLst)))),pubVallst,_,msg,_)
      equation
        if Flags.isSet(Flags.DYN_LOAD) then
          Debug.traceln("CALL: record constructor: func: " + Absyn.pathString(funcpath) + " type path: " + Absyn.pathString(complexName));
        end if;
        true = Absyn.pathEqual(funcpath,complexName);
        (pubVarLst,proVarLst) = List.splitOnTrue(varLst,Types.isPublicVar);
        expl = List.map1(proVarLst, Types.getBindingExp, funcpath);
        (cache,proVallst) = Ceval.cevalList(cache, env, expl, impl, msg, numIter);
        pubVarNames = List.map(pubVarLst,Expression.varName);
        proVarNames = List.map(proVarLst,Expression.varName);
        varNames = listAppend(pubVarNames, proVarNames);
        vallst = listAppend(pubVallst, proVallst);
        // fprintln(Flags.DYN_LOAD, "CALL: record constructor: [success] func: " + Absyn.pathString(funcpath));
      then
        (cache,Values.RECORD(funcpath,vallst,varNames,-1));

    // evaluate or generate non-partial and non-replaceable functions
    case (cache,env, DAE.CALL(path = funcpath, attr = DAE.CALL_ATTR(ty = ty, builtin = false)), _, _, msg, _)
      equation
        failure(cevalIsExternalObjectConstructor(cache, funcpath, env, msg));
        if Flags.isSet(Flags.DYN_LOAD) then
          Debug.traceln("CALL: try to evaluate or generate function: " + Absyn.pathString(funcpath));
        end if;

        bIsCompleteFunction = isCompleteFunction(cache, env, funcpath);
        false = Types.hasMetaArray(ty);

        if Flags.isSet(Flags.DYN_LOAD) then
          Debug.traceln("CALL: is complete function: " + Absyn.pathString(funcpath) + " " +  (if bIsCompleteFunction then "[true]" else "[false]"));
        end if;
        (cache, newval) = cevalCallFunctionEvaluateOrGenerate(inCache,inEnv,inExp,inValuesValueLst,impl,inMsg,bIsCompleteFunction);

        // Debug.fprintln(Flags.DYN_LOAD, "CALL: constant evaluation success: " + Absyn.pathString(funcpath));
      then
        (cache, newval);

    // partial and replaceable functions should not be evaluated!
    case (cache,env, DAE.CALL(path = funcpath, attr = DAE.CALL_ATTR( builtin = false)), _, _, msg, _)
      equation
        failure(cevalIsExternalObjectConstructor(cache, funcpath, env, msg));
        false = isCompleteFunction(cache, env, funcpath);

        if Flags.isSet(Flags.DYN_LOAD) then
          Debug.traceln("CALL: constant evaluation failed (not complete function): " + Absyn.pathString(funcpath));
        end if;
      then
        fail();

    case (cache,env, DAE.CALL(path = funcpath, attr = DAE.CALL_ATTR(ty = ty, builtin = false)), _, _, msg as Absyn.MSG(info), _)
      equation
        failure(cevalIsExternalObjectConstructor(cache, funcpath, env, msg));
        true = isCompleteFunction(cache, env, funcpath);
        true = Types.hasMetaArray(ty);
        str = ExpressionDump.printExpStr(inExp);
        Error.addSourceMessage(Error.FUNCTION_RETURNS_META_ARRAY, {str}, info);
      then fail();

  end matchcontinue;
end cevalCallFunction;

protected

function cevalCallFunctionEvaluateOrGenerate
"This function evaluates CALL expressions, i.e. function calls.
  They are currently evaluated by generating code for the function and
  then dynamicly load the function and call it."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input DAE.Exp inExp;
  input list<Values.Value> inValuesValueLst;
  input Boolean impl;
  input Absyn.Msg inMsg;
  input Boolean bIsCompleteFunction;
  output FCore.Cache outCache;
  output Values.Value outValue;
protected
  Integer numCheckpoints;
algorithm
  // Only add a stack overflow checkpoint for the top-most cevalCallFunctionEvaluateOrGenerate
  if isNone(getGlobalRoot(Global.stackoverFlowIndex)) then
    setGlobalRoot(Global.stackoverFlowIndex, SOME(1));
    numCheckpoints:=ErrorExt.getNumCheckpoints();
    try
      StackOverflow.clearStacktraceMessages();
      (outCache,outValue) := cevalCallFunctionEvaluateOrGenerate2(inCache,inEnv,inExp,inValuesValueLst,impl,inMsg,bIsCompleteFunction);
    else
      setGlobalRoot(Global.stackoverFlowIndex, NONE());
      ErrorExt.rollbackNumCheckpoints(ErrorExt.getNumCheckpoints()-numCheckpoints);
      Error.addInternalError("Stack overflow when evaluating function call: "+ExpressionDump.printExpStr(inExp)+"...\n"+stringDelimitList(StackOverflow.readableStacktraceMessages(), "\n"), match inMsg local SourceInfo info; case Absyn.MSG(info) then info; else sourceInfo(); end match);
      /* Do not fail or we can loop too much */
      StackOverflow.clearStacktraceMessages();
      outCache := inCache;
      outValue := Values.META_FAIL();
    end try annotation(__OpenModelica_stackOverflowCheckpoint=true);
    setGlobalRoot(Global.stackoverFlowIndex, NONE());
  else
    (outCache,outValue) := cevalCallFunctionEvaluateOrGenerate2(inCache,inEnv,inExp,inValuesValueLst,impl,inMsg,bIsCompleteFunction);
  end if;
end cevalCallFunctionEvaluateOrGenerate;

function cevalCallFunctionEvaluateOrGenerate2
"This function evaluates CALL expressions, i.e. function calls.
  They are currently evaluated by generating code for the function and
  then dynamicly load the function and call it."
  input FCore.Cache inCache;
  input FCore.Graph inEnv;
  input DAE.Exp inExp;
  input list<Values.Value> inValuesValueLst;
  input Boolean impl;
  input Absyn.Msg inMsg;
  input Boolean bIsCompleteFunction;
  output FCore.Cache outCache;
  output Values.Value outValue;
algorithm
  (outCache,outValue) := matchcontinue (inCache,inEnv,inExp,inValuesValueLst,impl,inMsg,bIsCompleteFunction)
    local
      Values.Value newval;
      FCore.Graph env;
      DAE.Exp e;
      Absyn.Path funcpath;
      list<DAE.Exp> expl;
      Boolean  print_debug;
      list<Values.Value> vallst;
      Absyn.Msg msg;
      FCore.Cache cache;
      Absyn.Program p;
      Integer libHandle, funcHandle;
      String fNew,fOld;
      Real buildTime, edit, build;
      Option<list<SCode.Element>> a;
      list<GlobalScript.Variable> c;
      String funcstr,f,fileName;
      String name;
      Boolean ppref, fpref, epref;
      Absyn.ClassDef    body;
      SourceInfo        info;
      Absyn.Within      w;
      list<Absyn.Path> functionDependencies;
      SCode.Element sc;
      SCode.ClassDef cdef;
      String error_Str;
      DAE.Function func;
      SCode.Restriction res;
      Absyn.FunctionRestriction funcRest;
      DAE.Type ty;

    // try function interpretation
    case (cache,env, DAE.CALL(path = funcpath, attr = DAE.CALL_ATTR(builtin = false)), vallst, _, msg, _)
      equation
        true = Flags.isSet(Flags.EVAL_FUNC);
        failure(cevalIsExternalObjectConstructor(cache, funcpath, env, msg));
        // bcall1(Flags.isSet(Flags.DYN_LOAD), print,"[dynload]: try constant evaluation: " + Absyn.pathString(funcpath) + "\n");
        (cache,
         sc as SCode.CLASS(partialPrefix = SCode.NOT_PARTIAL()),
         env) = Lookup.lookupClass(cache, env, funcpath);
        isCevaluableFunction(sc);
        (cache, env, _) = InstFunction.implicitFunctionInstantiation(
          cache,
          env,
          InnerOuter.emptyInstHierarchy,
          DAE.NOMOD(),
          Prefix.NOPRE(),
          sc,
          {});
        func = FCore.getCachedInstFunc(cache, funcpath);
        (cache, newval) = CevalFunction.evaluate(cache, env, func, vallst);
        // bcall1(Flags.isSet(Flags.DYN_LOAD), print, "[dynload]: constant evaluation SUCCESS: " + Absyn.pathString(funcpath) + "\n");
      then
        (cache, newval);

    // not in CF list, we have a symbol table, generate function and update symtab
    case (cache,env,(DAE.CALL(path = funcpath,attr = DAE.CALL_ATTR(ty=ty, builtin = false))),vallst,_, msg, _)
      guard (bIsCompleteFunction and Flags.isSet(Flags.GEN)) // yeha! we have a symboltable!
      algorithm
        failure(cevalIsExternalObjectConstructor(cache,funcpath,env,msg));

        if Flags.isSet(Flags.DYN_LOAD) then
          print("[dynload]: [SOME SYMTAB] not in in CF list: " + Absyn.pathString(funcpath) + "\n");
        end if;

        p := SymbolTable.getAbsyn();
        // now is safe to generate code
        (cache, funcstr, fileName) := cevalGenerateFunction(cache, env, p, funcpath);
        print_debug := Flags.isSet(Flags.DYN_LOAD);
        libHandle := System.loadLibrary(fileName, print_debug);
        funcHandle := System.lookupFunction(libHandle, stringAppend("in_", funcstr));
        execStatReset();
        newval := DynLoad.executeFunction(funcHandle, vallst, print_debug);
        execStat("executeFunction("+Absyn.pathString(funcpath)+")");

        System.freeLibrary(libHandle, print_debug);
        // update the build time in the class!
        Absyn.CLASS(_,_,_,_,Absyn.R_FUNCTION(_),_,info) := Interactive.getPathedClassInProgram(funcpath, p);

        w := Interactive.buildWithin(funcpath);

        if Flags.isSet(Flags.DYN_LOAD) then
          print("[dynload]: Updating build time for function path: " + Absyn.pathString(funcpath) + " within: " + Dump.unparseWithin(w) + "\n");
        end if;

        // p = Interactive.updateProgram(Absyn.PROGRAM({Absyn.CLASS(name,ppref,fpref,epref,Absyn.R_FUNCTION(funcRest),body,info)},w,ts), p);
        f := Absyn.getFileNameFromInfo(info);

        if Flags.isSet(Flags.DYN_LOAD) then
          print("[dynload]: [SOME SYMTAB] not in in CF list [finished]: " +
          Absyn.pathString(funcpath) + "\n");
        end if;
      then
        (cache,newval);

    case (_,_,(DAE.CALL(path = funcpath)),_,_,_,_)
      equation
        if Flags.isSet(Flags.DYN_LOAD) then
          print("[dynload]: FAILED to constant evaluate function: " + Absyn.pathString(funcpath) + "\n");
        end if;
        //TODO: readd this when testsuite is okay.
        //Error.addMessage(Error.FAILED_TO_EVALUATE_FUNCTION, {error_Str});
        false = Flags.isSet(Flags.GEN);
        true = Flags.isSet(Flags.FAILTRACE);
        Debug.trace("- codegeneration is turned off. switch \"nogen\" flag off\n");
      then
        fail();

  end matchcontinue;
end cevalCallFunctionEvaluateOrGenerate2;

function cevalIsExternalObjectConstructor
  input FCore.Cache cache;
  input Absyn.Path funcpath;
  input FCore.Graph env;
  input Absyn.Msg msg;
protected
  Absyn.Path funcpath2;
  DAE.Type tp;
  Option<SourceInfo> info;
algorithm
  _ := match(cache, funcpath, env, msg)
    case (_, _, FCore.EG(_), Absyn.NO_MSG()) then fail();
    case (_, _, _, Absyn.NO_MSG())
      equation
        (funcpath2, Absyn.IDENT("constructor")) = Absyn.splitQualAndIdentPath(funcpath);
        info = if valueEq(msg, Absyn.NO_MSG()) then NONE() else SOME(Absyn.dummyInfo);
        (_, tp, _) = Lookup.lookupType(cache, env, funcpath2, info);
        Types.externalObjectConstructorType(tp);
      then
        ();
  end match;
end cevalIsExternalObjectConstructor;

protected function checkLibraryUsage
  input String inLibrary;
  input Absyn.Exp inExp;
  output Boolean isUsed;
algorithm
  isUsed := match(inLibrary, inExp)
    local
      String s;
      list<Absyn.Exp> exps;

    case (_, Absyn.STRING(s)) then stringEq(s, inLibrary);
    case (_, Absyn.ARRAY(exps))
      then List.isMemberOnTrue(inLibrary, exps, checkLibraryUsage);
  end match;
end checkLibraryUsage;

function isCevaluableFunction
  "Checks if an element is a function or external function that can be evaluated
  by CevalFunction."
  input SCode.Element inElement;
algorithm
  _ := match(inElement)
    local
      String fid;
      SCode.Mod mod;
      Absyn.Exp lib;

    //only some external functions.
    case (SCode.CLASS(restriction = SCode.R_FUNCTION(SCode.FR_EXTERNAL_FUNCTION(_)),
      classDef = SCode.PARTS(externalDecl = SOME(SCode.EXTERNALDECL(
        funcName = SOME(fid),
        annotation_ = SOME(SCode.ANNOTATION(mod)))))))
      equation
        SCode.MOD(binding = SOME(lib)) = Mod.getUnelabedSubMod(mod, "Library");
        true = checkLibraryUsage("Lapack", lib) or checkLibraryUsage("lapack", lib);
        isCevaluableFunction2(fid);
      then
        ();

    // All other functions can be evaluated.
    case (SCode.CLASS(restriction = SCode.R_FUNCTION(_))) then ();

  end match;
end isCevaluableFunction;

function isCevaluableFunction2
  "Checks if a function name belongs to a known external function that we can
  constant evaluate."
  input String inFuncName;
algorithm
  _ := match(inFuncName)
    local
      // Lapack functions.
      case "dgbsv" then ();
      case "dgeev" then ();
      case "dgegv" then ();
      case "dgels" then ();
      case "dgelsx" then ();
      case "dgeqpf" then ();
      case "dgesv" then ();
      case "dgesvd" then ();
      case "dgetrf" then ();
      case "dgetri" then ();
      case "dgetrs" then ();
      case "dgglse" then ();
      case "dgtsv" then ();
      case "dorgqr" then ();
  end match;
end isCevaluableFunction2;

function isSimpleAPIFunction
  input DAE.Type ty;
  output Boolean b;
algorithm
  b := match ty
    case DAE.T_FUNCTION(functionAttributes=DAE.FUNCTION_ATTRIBUTES(isBuiltin=DAE.FUNCTION_BUILTIN())) then
      isSimpleAPIFunctionArg(ty.funcResultType) and
      min(match fa case DAE.FUNCARG() then isSimpleAPIFunctionArg(fa.ty); end match for fa in ty.funcArg);
    else false;
  end match;
end isSimpleAPIFunction;

function isSimpleAPIFunctionArg
  input DAE.Type ty;
  output Boolean b;
algorithm
  b := match ty
    case DAE.T_INTEGER() then true;
    case DAE.T_REAL() then true;
    case DAE.T_BOOL() then true;
    case DAE.T_STRING() then true;
    case DAE.T_NORETCALL() then true;
    case DAE.T_ARRAY() then isSimpleAPIFunctionArg(ty.ty);
    case DAE.T_CODE(ty=DAE.C_TYPENAME()) then true;
    case DAE.T_TUPLE() then min(isSimpleAPIFunctionArg(t) for t in ty.types);
    else false;
  end match;
end isSimpleAPIFunctionArg;

function verifyInterfaceType
  input SCode.Element elt;
  input list<String> expected;
algorithm
  _ := matchcontinue (elt,expected)
    local
      String str,name;
      SCode.Annotation ann;
      SourceInfo info;
    case (SCode.CLASS(restriction=SCode.R_METARECORD(moved=true)),_) then ();
    case (SCode.CLASS(cmt=SCode.COMMENT(annotation_=SOME(ann))),name::_)
      equation
        (Absyn.STRING(str),info) = SCode.getNamedAnnotation(ann,"__OpenModelica_Interface");
        Error.assertionOrAddSourceMessage(listMember(str, expected), Error.MISMATCHING_INTERFACE_TYPE, {str,name}, info);
      then ();
    else
      equation
        print(SCodeDump.unparseElementStr(elt)+"\n");
        Error.addSourceMessage(Error.MISSING_INTERFACE_TYPE,{},SCode.elementInfo(elt));
      then fail();
  end matchcontinue;
end verifyInterfaceType;

function getInterfaceType
  input SCode.Element elt;
  input list<tuple<String,list<String>>> assoc;
  output list<String> it;
algorithm
  it := matchcontinue (elt,assoc)
    local
      String name;
      SCode.Annotation ann;
      String str;
      SourceInfo info;
    case (SCode.CLASS(cmt=SCode.COMMENT(annotation_=SOME(ann))),_)
      equation
        (Absyn.STRING(str),_) = SCode.getNamedAnnotation(ann,"__OpenModelica_Interface");
        it = Util.assoc(str,assoc);
      then it;
    else
      equation
        Error.addSourceMessage(Error.MISSING_INTERFACE_TYPE,{},SCode.elementInfo(elt));
      then fail();
  end matchcontinue;
end getInterfaceType;

function getInterfaceTypeAssocElt
  input Values.Value val;
  input SourceInfo info;
  output tuple<String,list<String>> assoc;
algorithm
  assoc := match (val,info)
    local
      String str;
      list<String> strs;
      list<Values.Value> vals;
    case (Values.ARRAY(valueLst=Values.STRING("")::_),_)
      equation
        Error.addSourceMessage(Error.MISSING_INTERFACE_TYPE,{},info);
      then fail();
    case (Values.ARRAY(valueLst=Values.STRING(str)::vals),_)
      equation
        strs = List.select(List.map(vals,ValuesUtil.extractValueString), Util.isNotEmptyString);
      then ((str,str::strs));
  end match;
end getInterfaceTypeAssocElt;

protected function buildDependencyGraph
  input String name;
  input SCode.Program sp;
  output list<String> edges;
algorithm
  edges := match (name,sp)
    local
      list<SCode.Element> elts;
    case (_,_)
      equation
        SCode.CLASS(classDef=SCode.PARTS(elementLst=elts)) = List.getMemberOnTrue(name, sp, SCode.isClassNamed);
        (_,_,_,elts,_) = SCode.partitionElements(elts);
      then List.map(elts, importDepenency);
  end match;
end buildDependencyGraph;

protected function buildDependencyGraphPublicImports
  input String name;
  input SCode.Program sp;
  output list<String> edges;
algorithm
  edges := match (name,sp)
    local
      list<SCode.Element> elts;
    case (_,_)
      equation
        SCode.CLASS(classDef=SCode.PARTS(elementLst=elts)) = List.getMemberOnTrue(name, sp, SCode.isClassNamed);
        elts = List.select(elts,SCode.elementIsPublicImport);
      then List.map(elts, importDepenency);
  end match;
end buildDependencyGraphPublicImports;

protected function buildTransitiveDependencyGraph
  input String name;
  input list<tuple<String,list<String>>> oldgraph;
  output list<String> edges;
algorithm
  edges := matchcontinue (name,oldgraph)
    local
      String str;
    case (_,_) then List.setDifference(Graph.allReachableNodes(({name},{}),oldgraph,stringEq),{name});
    else
      equation
        str = "CevalScript.buildTransitiveDependencyGraph failed: " + name;
        Error.addMessage(Error.INTERNAL_ERROR, {str});
      then fail();
  end matchcontinue;
end buildTransitiveDependencyGraph;

protected function importDepenency
  input SCode.Element simp;
  output String name;
algorithm
  name := match simp
    local
      Absyn.Import imp;
      SourceInfo info;
      String str;
      Absyn.Path path;
    case SCode.IMPORT(imp=Absyn.NAMED_IMPORT(path=path)) then Absyn.pathFirstIdent(path);
    case SCode.IMPORT(imp=Absyn.NAMED_IMPORT(path=path)) then Absyn.pathFirstIdent(path);
    case SCode.IMPORT(imp=Absyn.QUAL_IMPORT(path=path)) then Absyn.pathFirstIdent(path);
    case SCode.IMPORT(imp=Absyn.UNQUAL_IMPORT(path=path)) then Absyn.pathFirstIdent(path);
    case SCode.IMPORT(imp=Absyn.GROUP_IMPORT(prefix=path)) then Absyn.pathFirstIdent(path);
    case SCode.IMPORT(imp=imp,info=info)
      equation
        str = "CevalScript.importDepenency could not handle:" + Dump.unparseImportStr(imp);
        Error.addSourceMessage(Error.INTERNAL_ERROR,{str},info);
      then fail();
  end match;
end importDepenency;

protected function compareNumberOfDependencies
  input tuple<String,list<String>> node1;
  input tuple<String,list<String>> node2;
  output Boolean cmp;
protected
  list<String> deps1,deps2;
algorithm
  (_,deps1) := node1;
  (_,deps2) := node2;
  cmp := listLength(deps1) >= listLength(deps2);
end compareNumberOfDependencies;

protected function compareDependencyNode
  input tuple<String,list<String>> node1;
  input tuple<String,list<String>> node2;
  output Boolean cmp;
protected
  String s1,s2;
algorithm
  (s1,_) := node1;
  (s2,_) := node2;
  cmp := Util.strcmpBool(s1,s2);
end compareDependencyNode;

protected function dependencyString
  input tuple<String,list<String>> deps;
  output String str;
protected
  list<String> strs;
algorithm
  (str,strs) := deps;
  str := str + " (" + intString(listLength(strs)) + "): " + stringDelimitList(strs, ",");
end dependencyString;

protected function transitiveDependencyString
  input tuple<String,list<String>> deps;
  output String str;
protected
  list<String> strs;
algorithm
  (str,strs) := deps;
  str := intString(listLength(strs)) + ": ("+str+") " + stringDelimitList(strs, ",");
end transitiveDependencyString;

protected function containsPublicInterface
  input SCode.Element elt;
  output Boolean b;
algorithm
  b := match elt
    local
      list<SCode.Element> elts;
      String name;
    case SCode.CLASS(restriction=SCode.R_PACKAGE(), encapsulatedPrefix=SCode.ENCAPSULATED(), classDef=SCode.PARTS(elementLst=elts))
      then List.exist(elts, containsPublicInterface2);
    else
      equation
        name = SCode.elementName(elt);
        name = "CevalScript.containsPublicInterface failed: " + name;
        Error.addMessage(Error.INTERNAL_ERROR, {name});
      then fail();
  end match;
end containsPublicInterface;

protected function containsPublicInterface2
  "If the package contains a public type or constant, we depend on this package also through other modules"
  input SCode.Element elt;
  output Boolean b;
algorithm
  b := match elt
    local
      String name;
    case SCode.IMPORT() then false;
    case SCode.EXTENDS() then false;
    case SCode.CLASS(restriction=SCode.R_FUNCTION(_)) then false;
    case SCode.COMPONENT(prefixes=SCode.PREFIXES(visibility=SCode.PUBLIC()))
      equation
        // print("public component " + name + ": ");
      then true;
    case SCode.CLASS(prefixes=SCode.PREFIXES(visibility=SCode.PUBLIC()))
      equation
        // print("public class " + name + ": ");
      then true;
    else false;
  end match;
end containsPublicInterface2;

protected function containsImport
  input SCode.Element elt;
  input SCode.Visibility visibility;
  output Boolean b;
algorithm
  b := match (elt,visibility)
    local
      list<SCode.Element> elts;
      String name;
    case (SCode.CLASS(restriction=SCode.R_PACKAGE(), encapsulatedPrefix=SCode.ENCAPSULATED(), classDef=SCode.PARTS(elementLst=elts)),_)
      then List.exist1(elts, containsImport2, visibility);
    else
      equation
        name = SCode.elementName(elt);
        name = "CevalScript.containsPublicInterface failed: " + name;
        Error.addMessage(Error.INTERNAL_ERROR, {name});
      then fail();
  end match;
end containsImport;

protected function containsImport2
  "If the package contains a public type or constant, we depend on this package also through other modules"
  input SCode.Element elt;
  input SCode.Visibility visibility;
  output Boolean b;
algorithm
  b := match (elt,visibility)
    local
      String name;
    case (SCode.IMPORT(visibility=SCode.PUBLIC()),SCode.PUBLIC()) then true;
    case (SCode.IMPORT(visibility=SCode.PROTECTED()),SCode.PROTECTED()) then true;
    else false;
  end match;
end containsImport2;

protected function printInterfaceString
  input SCode.Element elt;
protected
  String str;
algorithm
  SCode.CLASS(name=str) := elt;
  print(str + ": " + boolString(containsPublicInterface(elt)) + "\n");
end printInterfaceString;

protected function writeModuleDepends
  input SCode.Element cl;
  input String prefix;
  input String suffix;
  input list<tuple<String,list<String>>> deps;
  output String str;
algorithm
  str := matchcontinue (cl,prefix,suffix,deps)
    local
      String name,fileName,tmp1;
      list<String> allDepends,protectedDepends,tmp2;
      list<SCode.Element> elts;
      SourceInfo info;
    case (SCode.CLASS(name=name, classDef=SCode.PARTS(elementLst=elts), info = SOURCEINFO()),_,_,_)
      equation
        protectedDepends = List.map(List.select(elts,SCode.elementIsProtectedImport),importDepenency);
        protectedDepends = List.select(protectedDepends, isNotBuiltinImport);
        _::allDepends = Graph.allReachableNodes((name::protectedDepends,{}),deps,stringEq);
        allDepends = List.map1r(allDepends, stringAppend, prefix);
        allDepends = List.map1(allDepends, stringAppend, ".interface.mo");
        str = prefix + name + suffix + ": $(RELPATH_" + name + ") " + stringDelimitList(allDepends," ");
      then str;
    case (SCode.CLASS(name=name, classDef=SCode.PARTS(elementLst=elts), info=info),_,_,_)
      algorithm
        protectedDepends := List.map(List.select(elts,SCode.elementIsProtectedImport),importDepenency);
        protectedDepends := List.select(protectedDepends, isNotBuiltinImport);
        allDepends := list(Util.tuple21(e) for e in deps);
        for d in protectedDepends loop
          if not listMember(d, allDepends) then
            Error.addSourceMessage(Error.GENERATE_SEPARATE_CODE_DEPENDENCIES_FAILED_UNKNOWN_PACKAGE, {name,name,d}, info);
            fail();
          end if;
        end for;
        for dep in deps loop
          (tmp1,tmp2) := dep;
          for d in tmp2 loop
            if not listMember(d, allDepends) then
              Error.addSourceMessage(Error.GENERATE_SEPARATE_CODE_DEPENDENCIES_FAILED_UNKNOWN_PACKAGE, {name,tmp1,d}, info);
              fail();
            end if;
          end for;
        end for;
      then fail();
    case (SCode.CLASS(name=name,info=info),_,_,_)
      equation
        Error.addSourceMessage(Error.GENERATE_SEPARATE_CODE_DEPENDENCIES_FAILED, {name}, info);
      then fail();
  end matchcontinue;
end writeModuleDepends;

protected function isNotBuiltinImport
  input String module;
  output Boolean b = module <> "MetaModelica";
end isNotBuiltinImport;

protected function getTypeNameIdent
  input Values.Value val;
  output String str;
algorithm
  Values.CODE(Absyn.C_TYPENAME(Absyn.IDENT(str))) := val;
end getTypeNameIdent;

protected function getChangedClass
  input SCode.Element elt;
  input String suffix;
  output String name;
algorithm
  name := matchcontinue (elt,suffix)
    local
      String fileName;
    case (SCode.CLASS(name=name,info=SOURCEINFO()),_)
      equation
        false = System.regularFileExists(name + suffix);
      then name;
    case (SCode.CLASS(name=name,info=SOURCEINFO(fileName=fileName)),_)
      equation
        true = System.fileIsNewerThan(fileName, name + suffix);
      then name;
  end matchcontinue;
end getChangedClass;

protected function isChanged
  input tuple<String,list<String>> node;
  input HashSetString.HashSet hs;
  output Boolean b;
protected
  String str;
  list<String> strs;
algorithm
  (str,strs) := node;
  b := List.exist1(str::strs,BaseHashSet.has,hs);
  // print(str + ": " +  boolString(b) + "\n");
end isChanged;

protected function reloadClass
  input String filename;
  input String encoding;
protected
  Absyn.Program p,newp;
algorithm
  newp := Parser.parse(filename,encoding); /* Don't use the classloader since that can pull in entire directory structures. We only want to reload one single file. */
  newp := Interactive.updateProgram(newp, SymbolTable.getAbsyn());
  SymbolTable.setAbsyn(newp);
end reloadClass;

public function getFullPathFromUri
  input Absyn.Program program;
  input String uri;
  input Boolean printError;
  output String path;
protected
  String str1,str2,str3;
algorithm
  (str1,str2,str3) := System.uriToClassAndPath(uri);
  path := getBasePathFromUri(str1,str2,program,Settings.getModelicaPath(Config.getRunningTestsuite()),printError) + str3;
end getFullPathFromUri;

protected function getBasePathFromUri "Handle modelica:// URIs"
  input String scheme;
  input String iname;
  input Absyn.Program program;
  input String modelicaPath;
  input Boolean printError;
  output String basePath;
algorithm
  basePath := matchcontinue (scheme,iname,program,modelicaPath,printError)
    local
      Boolean isDir;
      list<String> mps,names;
      String gd,mp,bp,str,name,version,fileName;
    case ("modelica://",name,_,_,_)
      equation
        (name::names) = System.strtok(name,".");
        Absyn.CLASS(info=SOURCEINFO(fileName=fileName)) = Interactive.getPathedClassInProgram(Absyn.IDENT(name),program);
        mp = System.dirname(fileName);
        bp = findModelicaPath2(mp,names,"",true);
      then bp;
    case ("modelica://",name,_,mp,_)
      equation
        (name::names) = System.strtok(name,".");
        failure(_ = Interactive.getPathedClassInProgram(Absyn.IDENT(name),program));
        gd = System.groupDelimiter();
        mps = System.strtok(mp, gd);
        (mp,name,isDir) = System.getLoadModelPath(name, {"default"}, mps);
        mp = if isDir then mp + name else mp;
        bp = findModelicaPath2(mp,names,"",true);
      then bp;
    case ("file://",_,_,_,_) then "";
    case ("modelica://",name,_,mp,true)
      equation
        name::_ = System.strtok(name,".");
        str = "Could not resolve modelica://" + name + " with MODELICAPATH: " + mp;
        Error.addMessage(Error.COMPILER_ERROR,{str});
      then fail();
  end matchcontinue;
end getBasePathFromUri;

protected function findModelicaPath "Handle modelica:// URIs"
  input list<String> imps;
  input list<String> names;
  input String version;
  output String basePath;
algorithm
  basePath := matchcontinue (imps,names,version)
    local
      String mp;
      list<String> mps;

    case (mp::_,_,_)
      then findModelicaPath2(mp,names,version,false);
    case (_::mps,_,_)
      then findModelicaPath(mps,names,version);
  end matchcontinue;
end findModelicaPath;

protected function findModelicaPath2 "Handle modelica:// URIs"
  input String mp;
  input list<String> inames;
  input String version;
  input Boolean b;
  output String basePath;
algorithm
  basePath := matchcontinue (mp,inames,version,b)
    local
      list<String> names;
      String name,file;

    case (_,name::names,_,_)
      equation
        false = stringEq(version,"");
        file = mp + "/" + name + " " + version;
        true = System.directoryExists(file);
        // print("Found file 1: " + file + "\n");
      then findModelicaPath2(file,names,"",true);
    case (_,name::_,_,_)
      equation
        false = stringEq(version,"");
        file = mp + "/" + name + " " + version + ".mo";
        true = System.regularFileExists(file);
        // print("Found file 2: " + file + "\n");
      then mp;

    case (_,name::names,_,_)
      equation
        file = mp + "/" + name;
        true = System.directoryExists(file);
        // print("Found file 3: " + file + "\n");
      then findModelicaPath2(file,names,"",true);
    case (_,name::_,_,_)
      equation
        file = mp + "/" + name + ".mo";
        true = System.regularFileExists(file);
        // print("Found file 4: " + file + "\n");
      then mp;

      // This class is part of the current package.mo, or whatever...
    case (_,_,_,true)
      equation
        // print("Did not find file 5: " + mp + " - " + name + "\n");
      then mp;
  end matchcontinue;
end findModelicaPath2;

annotation(__OpenModelica_Interface="backend");
end CevalScript;
