/*
* This file is part of OpenModelica.
*
* Copyright (c) 1998-2020, Open Source Modelica Consortium (OSMC),
* c/o Linköpings universitet, Department of Computer and Information Science,
* SE-58183 Linköping, Sweden.
*
* All rights reserved.
*
* THIS PROGRAM IS PROVIDED UNDER THE TERMS OF GPL VERSION 3 LICENSE OR
* THIS OSMC PUBLIC LICENSE (OSMC-PL) VERSION 1.2.
* ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS PROGRAM CONSTITUTES
* RECIPIENT'S ACCEPTANCE OF THE OSMC PUBLIC LICENSE OR THE GPL VERSION 3,
* ACCORDING TO RECIPIENTS CHOICE.
*
* The OpenModelica software and the Open Source Modelica
* Consortium (OSMC) Public License (OSMC-PL) are obtained
* from OSMC, either from the above address,
* from the URLs: http://www.ida.liu.se/projects/OpenModelica or
* http://www.openmodelica.org, and in the OpenModelica distribution.
* GNU version 3 is obtained from: http://www.gnu.org/copyleft/gpl.html.
*
* This program is distributed WITHOUT ANY WARRANTY; without
* even the implied warranty of  MERCHANTABILITY or FITNESS
* FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
* IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS OF OSMC-PL.
*
* See the full OSMC Public License conditions for more details.
*
*/
encapsulated package NBCausalize
"file:        NBCausalize.mo
 package:     NBCausalize
 description: This file contains the functions which perform the causalization process;
"

public
  import Module = NBModule;

protected
  // NF imports
  import ComponentRef = NFComponentRef;
  import Dimension = NFDimension;
  import Expression = NFExpression;
  import NFFlatten.{FunctionTree, FunctionTreeImpl};
  import InstNode = NFInstNode.InstNode;
  import SBGraphUtil = NFSBGraphUtil;
  import Subscript = NFSubscript;
  import Type = NFType;
  import Variable = NFVariable;
  import NFArrayConnections.NameVertexTable;

  // Backend imports
  import Adjacency = NBAdjacency;
  import BackendDAE = NBackendDAE;
  import BEquation = NBEquation;
  import Differentiate = NBDifferentiate;
  import NBEquation.{Equation, EquationPointers, EqData, EquationAttributes};
  import Matching = NBMatching;
  import Sorting = NBSorting;
  import StrongComponent = NBStrongComponent;
  import System = NBSystem;
  import BVariable = NBVariable;
  import NBVariable.{VariablePointers, VarData};

  // util imports
  import BackendUtil = NBBackendUtil;
  import Error;
  import List;
  import StringUtil;
  import UnorderedSet;

  // SetBased Graph imports
  import SBGraph.BipartiteIncidenceList;
  import SBGraph.VertexDescriptor;
  import SBGraph.SetType;
  import NBAdjacency.BipartiteGraph;
  import SBInterval;
  import SBMultiInterval;
  import SBPWLinearMap;
  import SBSet;
  import NBGraphUtil.{SetVertex, SetEdge};

  // ############################################################
  //                      Main Functions
  // ############################################################

public
  function main extends Module.wrapper;
    input System.SystemType systemType;
  protected
    Module.causalizeInterface func = getModule();
  algorithm
    bdae := match (systemType, bdae)
      local
        list<System.System> systems;
        VarData varData;
        EqData eqData;
        FunctionTree funcTree;

      case (System.SystemType.ODE, BackendDAE.MAIN(ode = systems, varData = varData, eqData = eqData, funcTree = funcTree))
        algorithm
          (systems, varData, eqData, funcTree) := applyModule(systems, varData, eqData, funcTree, func);
          bdae.ode := systems;
          bdae.varData := varData;
          bdae.eqData := eqData;
          bdae.funcTree := funcTree;
      then bdae;

      case (System.SystemType.INI, BackendDAE.MAIN(init = systems, varData = varData, eqData = eqData, funcTree = funcTree))
        algorithm
          if Flags.isSet(Flags.INITIALIZATION) then
            print(StringUtil.headline_1("Balance Initialization") + "\n");
          end if;
          (systems, varData, eqData, funcTree) := applyModule(systems, varData, eqData, funcTree, func);
          bdae.init := systems;
          if Util.isSome(bdae.init_0) then
            (systems, varData, eqData, funcTree) := applyModule(Util.getOption(bdae.init_0), varData, eqData, funcTree, func);
            bdae.init_0 := SOME(systems);
          end if;
          bdae.varData := varData;
          bdae.eqData := eqData;
          bdae.funcTree := funcTree;
      then bdae;

      case (System.SystemType.DAE, BackendDAE.MAIN(dae = SOME(systems), varData = varData, eqData = eqData, funcTree = funcTree))
        algorithm
          (systems, varData, eqData, funcTree) := applyModule(systems, varData, eqData, funcTree, causalizeDAEMode);
          bdae.dae := SOME(systems);
          bdae.varData := varData;
          bdae.eqData := eqData;
          bdae.funcTree := funcTree;
      then bdae;

      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed with system type " + System.System.systemTypeString(systemType) + "!"});
      then fail();
    end match;
  end main;

  function applyModule
    input list<System.System> systems;
    output list<System.System> new_systems = {};
    input output VarData varData;
    input output EqData eqData;
    input output FunctionTree funcTree;
    input Module.causalizeInterface func;
  protected
    System.System new_system;
  algorithm
    for system in systems loop
      (new_system, varData, eqData, funcTree) := func(system, varData, eqData, funcTree);
      new_systems := new_system :: new_systems;
    end for;
    new_systems := listReverse(new_systems);
  end applyModule;

  function simple
    input VariablePointers vars;
    input EquationPointers eqs;
    input Adjacency.MatrixType matrixType = NBAdjacency.MatrixType.PSEUDO;
    output list<StrongComponent> comps;
  protected
    Adjacency.Matrix adj;
    Matching matching;
  algorithm
     // create scalar adjacency matrix for now
    adj := Adjacency.Matrix.create(vars, eqs, matrixType);
    matching := Matching.regular(NBMatching.EMPTY_MATCHING, adj);
    comps := Sorting.tarjan(adj, matching, vars, eqs);
  end simple;

  function getModule
    "Returns the module function that was chosen by the user."
    output Module.causalizeInterface func;
  protected
    String flag = Flags.getConfigString(Flags.MATCHING_ALGORITHM);
  algorithm
    (func) := match flag
      case "PFPlusExt"  then causalizePseudoArray;
      case "SBGraph"    then causalizeArray;
      case "pseudo"     then causalizePseudoArray;
      /* ... New causalize modules have to be added here */
      else algorithm
        Error.addMessage(Error.INTERNAL_ERROR,{getInstanceName() + " failed for unknown option: " + flag});
      then fail();
    end match;
  end getModule;

  // ############################################################
  //                Protected Functions and Types
  // ############################################################

protected
  function causalizePseudoArray extends Module.causalizeInterface;
  protected
    VariablePointers variables;
    EquationPointers equations;
    Adjacency.Matrix adj;
    Matching matching;
    list<StrongComponent> comps;
  algorithm
    (variables, equations, adj, matching, comps) := match system.systemType
      local
        list<Pointer<Variable>> fixable, unfixable;
        list<Pointer<Equation>> initials, simulation;

      case NBSystem.SystemType.INI algorithm
        (fixable, unfixable)    := List.splitOnTrue(VariablePointers.toList(system.unknowns), BVariable.isFixable);
        (initials, simulation)  := List.splitOnTrue(EquationPointers.toList(system.equations), Equation.isInitial);
        matching                := NBMatching.EMPTY_MATCHING;

        // #################################################
        // Phase I: match initial equations <-> unfixable vars
        // #################################################
        variables := VariablePointers.fromList(unfixable);
        equations := EquationPointers.fromList(initials);
        adj := Adjacency.Matrix.create(variables, equations, NBAdjacency.MatrixType.PSEUDO, NBAdjacency.MatrixStrictness.SOLVABLE);
        // do not resolve potential singular systems in Phase I or II! -> regular matching
        matching := Matching.regular(matching, adj, true, true);

        // #################################################
        // Phase II: match all equations <-> unfixables
        // #################################################
        (adj, variables, equations) := Adjacency.Matrix.expand(adj, variables, equations, {}, simulation);
        // do not resolve potential singular systems in Phase I or II! -> regular matching
        matching := Matching.regular(matching, adj, true, true);

        // #################################################
        // Phase III: match all equations <-> all vars
        // #################################################
        (adj, variables, equations) := Adjacency.Matrix.expand(adj, variables, equations, fixable, {});
        (matching, adj, variables, equations, funcTree, varData, eqData) := Matching.singular(matching, adj, variables, equations, funcTree, varData, eqData, system.systemType, false, true, false);

        adj := Adjacency.Matrix.create(variables, equations, NBAdjacency.MatrixType.PSEUDO, NBAdjacency.MatrixStrictness.FULL);
        comps := Sorting.tarjan(adj, matching, variables, equations);
      then (variables, equations, adj, matching, comps);

      else algorithm
        // compress the arrays to remove gaps
        variables := VariablePointers.compress(system.unknowns);
        equations := EquationPointers.compress(system.equations);

        // create solvable adjacency matrix for matching and full for sorting
        adj := Adjacency.Matrix.create(variables, equations, NBAdjacency.MatrixType.PSEUDO, NBAdjacency.MatrixStrictness.SOLVABLE);
        (matching, adj, variables, equations, funcTree, varData, eqData) := Matching.singular(NBMatching.EMPTY_MATCHING, adj, variables, equations, funcTree, varData, eqData, system.systemType, false, true);
        adj := Adjacency.Matrix.create(variables, equations, NBAdjacency.MatrixType.PSEUDO, NBAdjacency.MatrixStrictness.FULL);
        comps := Sorting.tarjan(adj, matching, variables, equations);
      then (variables, equations, adj, matching, comps);
    end match;

    system.unknowns := variables;
    system.equations := equations;
    system.adjacencyMatrix := SOME(adj);
    system.matching := SOME(matching);
    system.strongComponents := SOME(listArray(comps));
  end causalizePseudoArray;

  function causalizeArray extends Module.causalizeInterface;
  protected
    VariablePointers variables;
    EquationPointers equations;
    Adjacency.Matrix adj;
    Matching matching;
    list<StrongComponent> comps;
  algorithm
    // compress the arrays to remove gaps
    variables := VariablePointers.compress(system.unknowns);
    equations := EquationPointers.compress(system.equations);

    // create scalar adjacency matrix for now
    adj := Adjacency.Matrix.create(variables, equations, NBAdjacency.MatrixType.ARRAY, NBAdjacency.MatrixStrictness.SOLVABLE);
    matching := Matching.regular(NBMatching.EMPTY_MATCHING, adj);
  end causalizeArray;

  function causalizeDAEMode extends Module.causalizeInterface;
  protected
    Pointer<list<StrongComponent>> acc = Pointer.create({});
  algorithm
    // create all components as residuals for now
    // ToDo: use tearing to get inner/tmp equations
    EquationPointers.mapPtr(system.equations, function StrongComponent.makeDAEModeResidualTraverse(acc = acc));
    system.strongComponents := SOME(List.listArrayReverse(Pointer.access(acc)));
  end causalizeDAEMode;

  annotation(__OpenModelica_Interface="backend");
end NBCausalize;
