
// -*- mode: c++; c-indent-level: 4; c++-member-init-indent: 8; comment-column: 35; -*-

//-----------------------------------------------------------------------------
// eoLinearTopology.h
// (c) OPAC 2007
/*
    This library...

    Contact: paradiseo-help@lists.gforge.inria.fr, http://paradiseo.gforge.inria.fr
 */
//-----------------------------------------------------------------------------

#ifndef EOLINEARTOPOLOGY_H_
#define EOLINEARTOPOLOGY_H_

//-----------------------------------------------------------------------------
#include <eoPop.h>
#include <eoTopology.h>
#include <eoSocialNeighborhood.h>
//-----------------------------------------------------------------------------


/**
 *  One of the local best strategies. Each particle has a fixed number of neighbours, ans
 *  the neighborhood is social.
 *  The topology is never modified during the flight.
 */
template < class POT > class eoLinearTopology:public eoTopology <
            POT >
{

public:

    /**
     * Build the topology made of _neighborhoodSize neighborhoods.
     * @param _neighborhoods - The size of each neighborhood. 
     */
    eoLinearTopology (unsigned _neighborhoodSize):neighborhoodSize (_neighborhoodSize),isSetup(false){}


    /**
       * Build the neighborhoods contained in the topology.
       * @param _pop - The population used to build the neighborhoods.
       * If it remains several particle (because _pop.size()%neighborhoodSize !=0), there are inserted
       * in the last neighborhood. So it may be possible to have a bigger neighborhood.
       */
    void setup(const eoPop<POT> & _pop)
    {
        if (!isSetup)
        {
            // consitency check
            if (neighborhoodSize >= _pop.size()){
                std::string s;
                s.append (" Invalid neighborhood size in eoLinearTopology ");
                throw std::runtime_error (s);
            }

            unsigned howManyNeighborhood=_pop.size()/ neighborhoodSize;

            // build all the neighborhoods
            for (unsigned i=0;i< howManyNeighborhood;i++)
            {
                eoSocialNeighborhood<POT> currentNghd;

                currentNghd.best(_pop[i*neighborhoodSize]);
                for (unsigned k=i*neighborhoodSize;k < neighborhoodSize*(i+1);k++)
                {
                    currentNghd.put(k);
                    if (_pop[k].fitness() > currentNghd.best().fitness())
                        currentNghd.best(_pop[k]);
                }
                neighborhoods.push_back(currentNghd);
            }

            // assign the last neighborhood to the remaining particles
            if (_pop.size()%neighborhoodSize !=0)
            {
                for (unsigned z=_pop.size()-1;z >= (_pop.size()-_pop.size()%neighborhoodSize);z--){
                    neighborhoods.back().put(z);

                    if (_pop[z].fitness() > neighborhoods.back().best().fitness())
                        neighborhoods.back().best(_pop[z]);
                }
            }

            isSetup=true;
        }
        else
        {
            // Should activate this part ?
            /*
               std::string s;
               s.append (" Linear topology already setup in eoLinearTopology");
               throw std::runtime_error (s);
               */
        }

    }


    /**
     * Retrieve the neighboorhood of a particle.
     * @return _indice - The particle indice (in the population)
     */
    unsigned retrieveNeighborhoodByIndice(unsigned _indice)
    {
    	unsigned i=0;
        for (i=0;i< neighborhoods.size();i++)
        {
            if (neighborhoods[i].contains(_indice))
            {
                return i;
            }
        }
        return i;
    }


    /**
    * Update the neighborhood: update the particle's best fitness and the best particle 
    * of the corresponding neighborhood.
    */
    void updateNeighborhood(POT & _po,unsigned _indice)
    {
        // update the best fitness of the particle
        if (_po.fitness() > _po.best())
        {
            _po.best(_po.fitness());
        }

        // update the best in its neighborhood
        unsigned theGoodNhbd= retrieveNeighborhoodByIndice(_indice);
        if (_po.fitness() > neighborhoods[theGoodNhbd].best().fitness())
        {
            neighborhoods[theGoodNhbd].best(_po);
        }
    }


    /**
    * Return the best informative of a particle. Could be itself.
    * @param _indice - The indice of a particle in the population
    * @return POT & - The best particle in the neighborhood of the particle whose indice is _indice
    */
    POT & best (unsigned  _indice)
    {
        unsigned theGoodNhbd= retrieveNeighborhoodByIndice(_indice);

        return (neighborhoods[theGoodNhbd].best());
    }


    /**
     * Print the structure of the topology on the standrad output.
     */
    void printOn()
    {
        for (unsigned i=0;i< neighborhoods.size();i++)
        {
            std::cout << "{ " ;
            for (unsigned j=0;j< neighborhoods[i].size();j++)
            {
                std::cout << neighborhoods[i].get(j) << " ";
            }
            std::cout << "}" << std::endl;
        }
    }


protected:
	std::vector<eoSocialNeighborhood<POT> >  neighborhoods;
    unsigned neighborhoodSize; // the size of each neighborhood
    
    bool isSetup;

};

#endif /*EOLINEARTOPOLOGY_H_ */








