#include "blaswrap.h"
#include "f2c.h"

/* This file contains all routines used by dgesv for solving linear systems
   of equations. It includes both lapack routines and blas routines.
   Peter Aronsson, MathCore Engineering 2005-06-16.
*/


/* Subroutine */ int dgesv_(integer *n, integer *nrhs, doublereal *a, integer
  *lda, integer *ipiv, doublereal *b, integer *ldb, integer *info)
{
/*  -- LAPACK driver routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       March 31, 1993


    Purpose
    =======

    DGESV computes the solution to a real system of linear equations
       A * X = B,
    where A is an N-by-N matrix and X and B are N-by-NRHS matrices.

    The LU decomposition with partial pivoting and row interchanges is
    used to factor A as
       A = P * L * U,
    where P is a permutation matrix, L is unit lower triangular, and U is
    upper triangular.  The factored form of A is then used to solve the
    system of equations A * X = B.

    Arguments
    =========

    N       (input) INTEGER
            The number of linear equations, i.e., the order of the
            matrix A.  N >= 0.

    NRHS    (input) INTEGER
            The number of right hand sides, i.e., the number of columns
            of the matrix B.  NRHS >= 0.

    A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
            On entry, the N-by-N coefficient matrix A.
            On exit, the factors L and U from the factorization
            A = P*L*U; the unit diagonal elements of L are not stored.

    LDA     (input) INTEGER
            The leading dimension of the array A.  LDA >= max(1,N).

    IPIV    (output) INTEGER array, dimension (N)
            The pivot indices that define the permutation matrix P;
            row i of the matrix was interchanged with row IPIV(i).

    B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
            On entry, the N-by-NRHS matrix of right hand side matrix B.
            On exit, if INFO = 0, the N-by-NRHS solution matrix X.

    LDB     (input) INTEGER
            The leading dimension of the array B.  LDB >= max(1,N).

    INFO    (output) INTEGER
            = 0:  successful exit
            < 0:  if INFO = -i, the i-th argument had an illegal value
            > 0:  if INFO = i, U(i,i) is exactly zero.  The factorization
                  has been completed, but the factor U is exactly
                  singular, so the solution could not be computed.

    =====================================================================


       Test the input parameters.

       Parameter adjustments */
    /* System generated locals */
    integer a_dim1, a_offset, b_dim1, b_offset, i__1;
    /* Local variables */
    extern /* Subroutine */ int dgetrf_(integer *, integer *, doublereal *,
      integer *, integer *, integer *), xerbla_(char *, integer *), dgetrs_(char *, integer *, integer *, doublereal *,
      integer *, integer *, doublereal *, integer *, integer *);

    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    --ipiv;
    b_dim1 = *ldb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;

    /* Function Body */
    *info = 0;
    if (*n < 0) {
  *info = -1;
    } else if (*nrhs < 0) {
  *info = -2;
    } else if (*lda < max(1,*n)) {
  *info = -4;
    } else if (*ldb < max(1,*n)) {
  *info = -7;
    }
    if (*info != 0) {
  i__1 = -(*info);
  xerbla_("DGESV ", &i__1);
  return 0;
    }

/*     Compute the LU factorization of A. */

    dgetrf_(n, n, &a[a_offset], lda, &ipiv[1], info);
    if (*info == 0) {

/*        Solve the system A*X = B, overwriting B with X. */

  dgetrs_("No transpose", n, nrhs, &a[a_offset], lda, &ipiv[1], &b[
    b_offset], ldb, info);
    }
    return 0;

/*     End of DGESV */

} /* dgesv_ */


/* Subroutine */ int dgetrs_(char *trans, integer *n, integer *nrhs,
  doublereal *a, integer *lda, integer *ipiv, doublereal *b, integer *
  ldb, integer *info)
{
/*  -- LAPACK routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       March 31, 1993


    Purpose
    =======

    DGETRS solves a system of linear equations
       A * X = B  or  A' * X = B
    with a general N-by-N matrix A using the LU factorization computed
    by DGETRF.

    Arguments
    =========

    TRANS   (input) CHARACTER*1
            Specifies the form of the system of equations:
            = 'N':  A * X = B  (No transpose)
            = 'T':  A'* X = B  (Transpose)
            = 'C':  A'* X = B  (Conjugate transpose = Transpose)

    N       (input) INTEGER
            The order of the matrix A.  N >= 0.

    NRHS    (input) INTEGER
            The number of right hand sides, i.e., the number of columns
            of the matrix B.  NRHS >= 0.

    A       (input) DOUBLE PRECISION array, dimension (LDA,N)
            The factors L and U from the factorization A = P*L*U
            as computed by DGETRF.

    LDA     (input) INTEGER
            The leading dimension of the array A.  LDA >= max(1,N).

    IPIV    (input) INTEGER array, dimension (N)
            The pivot indices from DGETRF; for 1<=i<=N, row i of the
            matrix was interchanged with row IPIV(i).

    B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
            On entry, the right hand side matrix B.
            On exit, the solution matrix X.

    LDB     (input) INTEGER
            The leading dimension of the array B.  LDB >= max(1,N).

    INFO    (output) INTEGER
            = 0:  successful exit
            < 0:  if INFO = -i, the i-th argument had an illegal value

    =====================================================================


       Test the input parameters.

       Parameter adjustments */
    /* Table of constant values */
    static integer c__1 = 1;
    static doublereal c_b12 = 1.;
    static integer c_n1 = -1;

    /* System generated locals */
    integer a_dim1, a_offset, b_dim1, b_offset, i__1;
    /* Local variables */
    extern logical lsame_(char *, char *);
    extern /* Subroutine */ int dtrsm_(char *, char *, char *, char *,
      integer *, integer *, doublereal *, doublereal *, integer *,
      doublereal *, integer *), xerbla_(
      char *, integer *), dlaswp_(integer *, doublereal *,
      integer *, integer *, integer *, integer *, integer *);
    static logical notran;


    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    --ipiv;
    b_dim1 = *ldb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;

    /* Function Body */
    *info = 0;
    notran = lsame_(trans, "N");
    if (! notran && ! lsame_(trans, "T") && ! lsame_(
      trans, "C")) {
  *info = -1;
    } else if (*n < 0) {
  *info = -2;
    } else if (*nrhs < 0) {
  *info = -3;
    } else if (*lda < max(1,*n)) {
  *info = -5;
    } else if (*ldb < max(1,*n)) {
  *info = -8;
    }
    if (*info != 0) {
  i__1 = -(*info);
  xerbla_("DGETRS", &i__1);
  return 0;
    }

/*     Quick return if possible */

    if (*n == 0 || *nrhs == 0) {
  return 0;
    }

    if (notran) {

/*        Solve A * X = B.

          Apply row interchanges to the right hand sides. */

  dlaswp_(nrhs, &b[b_offset], ldb, &c__1, n, &ipiv[1], &c__1);

/*        Solve L*X = B, overwriting B with X. */

  dtrsm_("Left", "Lower", "No transpose", "Unit", n, nrhs, &c_b12, &a[
    a_offset], lda, &b[b_offset], ldb);

/*        Solve U*X = B, overwriting B with X. */

  dtrsm_("Left", "Upper", "No transpose", "Non-unit", n, nrhs, &c_b12, &
    a[a_offset], lda, &b[b_offset], ldb);
    } else {

/*        Solve A' * X = B.

          Solve U'*X = B, overwriting B with X. */

  dtrsm_("Left", "Upper", "Transpose", "Non-unit", n, nrhs, &c_b12, &a[
    a_offset], lda, &b[b_offset], ldb);

/*        Solve L'*X = B, overwriting B with X. */

  dtrsm_("Left", "Lower", "Transpose", "Unit", n, nrhs, &c_b12, &a[
    a_offset], lda, &b[b_offset], ldb);

/*        Apply row interchanges to the solution vectors. */

  dlaswp_(nrhs, &b[b_offset], ldb, &c__1, n, &ipiv[1], &c_n1);
    }

    return 0;

/*     End of DGETRS */

} /* dgetrs_ */

/* Subroutine */ int dtrsm_(char *side, char *uplo, char *transa, char *diag,
  integer *m, integer *n, doublereal *alpha, doublereal *a, integer *
  lda, doublereal *b, integer *ldb)
{
    /* System generated locals */
    integer a_dim1, a_offset, b_dim1, b_offset, i__1, i__2, i__3;
    /* Local variables */
    static integer info;
    static doublereal temp;
    static integer i__, j, k;
    static logical lside;
    extern logical lsame_(char *, char *);
    static integer nrowa;
    static logical upper;
    extern /* Subroutine */ int xerbla_(char *, integer *);
    static logical nounit;
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]
#define b_ref(a_1,a_2) b[(a_2)*b_dim1 + a_1]
/*  Purpose
    =======
    DTRSM  solves one of the matrix equations
       op( A )*X = alpha*B,   or   X*op( A ) = alpha*B,
    where alpha is a scalar, X and B are m by n matrices, A is a unit, or
    non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
       op( A ) = A   or   op( A ) = A'.
    The matrix X is overwritten on B.
    Parameters
    ==========
    SIDE   - CHARACTER*1.
             On entry, SIDE specifies whether op( A ) appears on the left
             or right of X as follows:
                SIDE = 'L' or 'l'   op( A )*X = alpha*B.
                SIDE = 'R' or 'r'   X*op( A ) = alpha*B.
             Unchanged on exit.
    UPLO   - CHARACTER*1.
             On entry, UPLO specifies whether the matrix A is an upper or
             lower triangular matrix as follows:
                UPLO = 'U' or 'u'   A is an upper triangular matrix.
                UPLO = 'L' or 'l'   A is a lower triangular matrix.
             Unchanged on exit.
    TRANSA - CHARACTER*1.
             On entry, TRANSA specifies the form of op( A ) to be used in
             the matrix multiplication as follows:
                TRANSA = 'N' or 'n'   op( A ) = A.
                TRANSA = 'T' or 't'   op( A ) = A'.
                TRANSA = 'C' or 'c'   op( A ) = A'.
             Unchanged on exit.
    DIAG   - CHARACTER*1.
             On entry, DIAG specifies whether or not A is unit triangular
             as follows:
                DIAG = 'U' or 'u'   A is assumed to be unit triangular.
                DIAG = 'N' or 'n'   A is not assumed to be unit
                                    triangular.
             Unchanged on exit.
    M      - INTEGER.
             On entry, M specifies the number of rows of B. M must be at
             least zero.
             Unchanged on exit.
    N      - INTEGER.
             On entry, N specifies the number of columns of B.  N must be
             at least zero.
             Unchanged on exit.
    ALPHA  - DOUBLE PRECISION.
             On entry,  ALPHA specifies the scalar  alpha. When  alpha is
             zero then  A is not referenced and  B need not be set before
             entry.
             Unchanged on exit.
    A      - DOUBLE PRECISION array of DIMENSION ( LDA, k ), where k is m
             when  SIDE = 'L' or 'l'  and is  n  when  SIDE = 'R' or 'r'.
             Before entry  with  UPLO = 'U' or 'u',  the  leading  k by k
             upper triangular part of the array  A must contain the upper
             triangular matrix  and the strictly lower triangular part of
             A is not referenced.
             Before entry  with  UPLO = 'L' or 'l',  the  leading  k by k
             lower triangular part of the array  A must contain the lower
             triangular matrix  and the strictly upper triangular part of
             A is not referenced.
             Note that when  DIAG = 'U' or 'u',  the diagonal elements of
             A  are not referenced either,  but are assumed to be  unity.
             Unchanged on exit.
    LDA    - INTEGER.
             On entry, LDA specifies the first dimension of A as declared
             in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
             LDA  must be at least  max( 1, m ),  when  SIDE = 'R' or 'r'
             then LDA must be at least max( 1, n ).
             Unchanged on exit.
    B      - DOUBLE PRECISION array of DIMENSION ( LDB, n ).
             Before entry,  the leading  m by n part of the array  B must
             contain  the  right-hand  side  matrix  B,  and  on exit  is
             overwritten by the solution matrix  X.
    LDB    - INTEGER.
             On entry, LDB specifies the first dimension of B as declared
             in  the  calling  (sub)  program.   LDB  must  be  at  least
             max( 1, m ).
             Unchanged on exit.
    Level 3 Blas routine.
    -- Written on 8-February-1989.
       Jack Dongarra, Argonne National Laboratory.
       Iain Duff, AERE Harwell.
       Jeremy Du Croz, Numerical Algorithms Group Ltd.
       Sven Hammarling, Numerical Algorithms Group Ltd.
       Test the input parameters.
       Parameter adjustments */
    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    b_dim1 = *ldb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    /* Function Body */
    lside = lsame_(side, "L");
    if (lside) {
  nrowa = *m;
    } else {
  nrowa = *n;
    }
    nounit = lsame_(diag, "N");
    upper = lsame_(uplo, "U");
    info = 0;
    if (! lside && ! lsame_(side, "R")) {
  info = 1;
    } else if (! upper && ! lsame_(uplo, "L")) {
  info = 2;
    } else if (! lsame_(transa, "N") && ! lsame_(transa,
       "T") && ! lsame_(transa, "C")) {
  info = 3;
    } else if (! lsame_(diag, "U") && ! lsame_(diag,
      "N")) {
  info = 4;
    } else if (*m < 0) {
  info = 5;
    } else if (*n < 0) {
  info = 6;
    } else if (*lda < max(1,nrowa)) {
  info = 9;
    } else if (*ldb < max(1,*m)) {
  info = 11;
    }
    if (info != 0) {
  xerbla_("DTRSM ", &info);
  return 0;
    }
/*     Quick return if possible. */
    if (*n == 0) {
  return 0;
    }
/*     And when  alpha.eq.zero. */
    if (*alpha == 0.) {
  i__1 = *n;
  for (j = 1; j <= i__1; ++j) {
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
    b_ref(i__, j) = 0.;
/* L10: */
      }
/* L20: */
  }
  return 0;
    }
/*     Start the operations. */
    if (lside) {
  if (lsame_(transa, "N")) {
/*           Form  B := alpha*inv( A )*B. */
      if (upper) {
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
        if (*alpha != 1.) {
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, j) = *alpha * b_ref(i__, j);
/* L30: */
      }
        }
        for (k = *m; k >= 1; --k) {
      if (b_ref(k, j) != 0.) {
          if (nounit) {
        b_ref(k, j) = b_ref(k, j) / a_ref(k, k);
          }
          i__2 = k - 1;
          for (i__ = 1; i__ <= i__2; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - b_ref(k, j) *
          a_ref(i__, k);
/* L40: */
          }
      }
/* L50: */
        }
/* L60: */
    }
      } else {
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
        if (*alpha != 1.) {
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, j) = *alpha * b_ref(i__, j);
/* L70: */
      }
        }
        i__2 = *m;
        for (k = 1; k <= i__2; ++k) {
      if (b_ref(k, j) != 0.) {
          if (nounit) {
        b_ref(k, j) = b_ref(k, j) / a_ref(k, k);
          }
          i__3 = *m;
          for (i__ = k + 1; i__ <= i__3; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - b_ref(k, j) *
          a_ref(i__, k);
/* L80: */
          }
      }
/* L90: */
        }
/* L100: */
    }
      }
  } else {
/*           Form  B := alpha*inv( A' )*B. */
      if (upper) {
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
        i__2 = *m;
        for (i__ = 1; i__ <= i__2; ++i__) {
      temp = *alpha * b_ref(i__, j);
      i__3 = i__ - 1;
      for (k = 1; k <= i__3; ++k) {
          temp -= a_ref(k, i__) * b_ref(k, j);
/* L110: */
      }
      if (nounit) {
          temp /= a_ref(i__, i__);
      }
      b_ref(i__, j) = temp;
/* L120: */
        }
/* L130: */
    }
      } else {
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
        for (i__ = *m; i__ >= 1; --i__) {
      temp = *alpha * b_ref(i__, j);
      i__2 = *m;
      for (k = i__ + 1; k <= i__2; ++k) {
          temp -= a_ref(k, i__) * b_ref(k, j);
/* L140: */
      }
      if (nounit) {
          temp /= a_ref(i__, i__);
      }
      b_ref(i__, j) = temp;
/* L150: */
        }
/* L160: */
    }
      }
  }
    } else {
  if (lsame_(transa, "N")) {
/*           Form  B := alpha*B*inv( A ). */
      if (upper) {
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
        if (*alpha != 1.) {
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, j) = *alpha * b_ref(i__, j);
/* L170: */
      }
        }
        i__2 = j - 1;
        for (k = 1; k <= i__2; ++k) {
      if (a_ref(k, j) != 0.) {
          i__3 = *m;
          for (i__ = 1; i__ <= i__3; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - a_ref(k, j) *
          b_ref(i__, k);
/* L180: */
          }
      }
/* L190: */
        }
        if (nounit) {
      temp = 1. / a_ref(j, j);
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, j) = temp * b_ref(i__, j);
/* L200: */
      }
        }
/* L210: */
    }
      } else {
    for (j = *n; j >= 1; --j) {
        if (*alpha != 1.) {
      i__1 = *m;
      for (i__ = 1; i__ <= i__1; ++i__) {
          b_ref(i__, j) = *alpha * b_ref(i__, j);
/* L220: */
      }
        }
        i__1 = *n;
        for (k = j + 1; k <= i__1; ++k) {
      if (a_ref(k, j) != 0.) {
          i__2 = *m;
          for (i__ = 1; i__ <= i__2; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - a_ref(k, j) *
          b_ref(i__, k);
/* L230: */
          }
      }
/* L240: */
        }
        if (nounit) {
      temp = 1. / a_ref(j, j);
      i__1 = *m;
      for (i__ = 1; i__ <= i__1; ++i__) {
          b_ref(i__, j) = temp * b_ref(i__, j);
/* L250: */
      }
        }
/* L260: */
    }
      }
  } else {
/*           Form  B := alpha*B*inv( A' ). */
      if (upper) {
    for (k = *n; k >= 1; --k) {
        if (nounit) {
      temp = 1. / a_ref(k, k);
      i__1 = *m;
      for (i__ = 1; i__ <= i__1; ++i__) {
          b_ref(i__, k) = temp * b_ref(i__, k);
/* L270: */
      }
        }
        i__1 = k - 1;
        for (j = 1; j <= i__1; ++j) {
      if (a_ref(j, k) != 0.) {
          temp = a_ref(j, k);
          i__2 = *m;
          for (i__ = 1; i__ <= i__2; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - temp * b_ref(
          i__, k);
/* L280: */
          }
      }
/* L290: */
        }
        if (*alpha != 1.) {
      i__1 = *m;
      for (i__ = 1; i__ <= i__1; ++i__) {
          b_ref(i__, k) = *alpha * b_ref(i__, k);
/* L300: */
      }
        }
/* L310: */
    }
      } else {
    i__1 = *n;
    for (k = 1; k <= i__1; ++k) {
        if (nounit) {
      temp = 1. / a_ref(k, k);
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, k) = temp * b_ref(i__, k);
/* L320: */
      }
        }
        i__2 = *n;
        for (j = k + 1; j <= i__2; ++j) {
      if (a_ref(j, k) != 0.) {
          temp = a_ref(j, k);
          i__3 = *m;
          for (i__ = 1; i__ <= i__3; ++i__) {
        b_ref(i__, j) = b_ref(i__, j) - temp * b_ref(
          i__, k);
/* L330: */
          }
      }
/* L340: */
        }
        if (*alpha != 1.) {
      i__2 = *m;
      for (i__ = 1; i__ <= i__2; ++i__) {
          b_ref(i__, k) = *alpha * b_ref(i__, k);
/* L350: */
      }
        }
/* L360: */
    }
      }
  }
    }
    return 0;
/*     End of DTRSM . */
} /* dtrsm_ */

#undef a_ref
#undef b_ref


/* Subroutine */ int dgetrf_(integer *m, integer *n, doublereal *a, integer *
  lda, integer *ipiv, integer *info)
{
/*  -- LAPACK routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       March 31, 1993


    Purpose
    =======

    DGETRF computes an LU factorization of a general M-by-N matrix A
    using partial pivoting with row interchanges.

    The factorization has the form
       A = P * L * U
    where P is a permutation matrix, L is lower triangular with unit
    diagonal elements (lower trapezoidal if m > n), and U is upper
    triangular (upper trapezoidal if m < n).

    This is the right-looking Level 3 BLAS version of the algorithm.

    Arguments
    =========

    M       (input) INTEGER
            The number of rows of the matrix A.  M >= 0.

    N       (input) INTEGER
            The number of columns of the matrix A.  N >= 0.

    A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
            On entry, the M-by-N matrix to be factored.
            On exit, the factors L and U from the factorization
            A = P*L*U; the unit diagonal elements of L are not stored.

    LDA     (input) INTEGER
            The leading dimension of the array A.  LDA >= max(1,M).

    IPIV    (output) INTEGER array, dimension (min(M,N))
            The pivot indices; for 1 <= i <= min(M,N), row i of the
            matrix was interchanged with row IPIV(i).

    INFO    (output) INTEGER
            = 0:  successful exit
            < 0:  if INFO = -i, the i-th argument had an illegal value
            > 0:  if INFO = i, U(i,i) is exactly zero. The factorization
                  has been completed, but the factor U is exactly
                  singular, and division by zero will occur if it is used
                  to solve a system of equations.

    =====================================================================


       Test the input parameters.

       Parameter adjustments */
    /* Table of constant values */
    static integer c__1 = 1;
    static integer c_n1 = -1;
    static doublereal c_b16 = 1.;
    static doublereal c_b19 = -1.;

    /* System generated locals */
    integer a_dim1, a_offset, i__1, i__2, i__3, i__4, i__5;
    /* Local variables */
    static integer i__, j;
    extern /* Subroutine */ int dgemm_(char *, char *, integer *, integer *,
      integer *, doublereal *, doublereal *, integer *, doublereal *,
      integer *, doublereal *, doublereal *, integer *);
    static integer iinfo;
    extern /* Subroutine */ int dtrsm_(char *, char *, char *, char *,
      integer *, integer *, doublereal *, doublereal *, integer *,
      doublereal *, integer *), dgetf2_(
      integer *, integer *, doublereal *, integer *, integer *, integer
      *);
    static integer jb, nb;
    extern /* Subroutine */ int xerbla_(char *, integer *);
    extern integer ilaenv_(integer *, char *, char *, integer *, integer *,
      integer *, integer *, ftnlen, ftnlen);
    extern /* Subroutine */ int dlaswp_(integer *, doublereal *, integer *,
      integer *, integer *, integer *, integer *);
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]


    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    --ipiv;

    /* Function Body */
    *info = 0;
    if (*m < 0) {
  *info = -1;
    } else if (*n < 0) {
  *info = -2;
    } else if (*lda < max(1,*m)) {
  *info = -4;
    }
    if (*info != 0) {
  i__1 = -(*info);
  xerbla_("DGETRF", &i__1);
  return 0;
    }

/*     Quick return if possible */

    if (*m == 0 || *n == 0) {
  return 0;
    }

/*     Determine the block size for this environment. */

    nb = ilaenv_(&c__1, "DGETRF", " ", m, n, &c_n1, &c_n1, (ftnlen)6, (ftnlen)
      1);
    if (nb <= 1 || nb >= min(*m,*n)) {

/*        Use unblocked code. */

  dgetf2_(m, n, &a[a_offset], lda, &ipiv[1], info);
    } else {

/*        Use blocked code. */

  i__1 = min(*m,*n);
  i__2 = nb;
  for (j = 1; i__2 < 0 ? j >= i__1 : j <= i__1; j += i__2) {
/* Computing MIN */
      i__3 = min(*m,*n) - j + 1;
      jb = min(i__3,nb);

/*           Factor diagonal and subdiagonal blocks and test for exact
             singularity. */

      i__3 = *m - j + 1;
      dgetf2_(&i__3, &jb, &a_ref(j, j), lda, &ipiv[j], &iinfo);

/*           Adjust INFO and the pivot indices. */

      if (*info == 0 && iinfo > 0) {
    *info = iinfo + j - 1;
      }
/* Computing MIN */
      i__4 = *m, i__5 = j + jb - 1;
      i__3 = min(i__4,i__5);
      for (i__ = j; i__ <= i__3; ++i__) {
    ipiv[i__] = j - 1 + ipiv[i__];
/* L10: */
      }

/*           Apply interchanges to columns 1:J-1. */

      i__3 = j - 1;
      i__4 = j + jb - 1;
      dlaswp_(&i__3, &a[a_offset], lda, &j, &i__4, &ipiv[1], &c__1);

      if (j + jb <= *n) {

/*              Apply interchanges to columns J+JB:N. */

    i__3 = *n - j - jb + 1;
    i__4 = j + jb - 1;
    dlaswp_(&i__3, &a_ref(1, j + jb), lda, &j, &i__4, &ipiv[1], &
      c__1);

/*              Compute block row of U. */

    i__3 = *n - j - jb + 1;
    dtrsm_("Left", "Lower", "No transpose", "Unit", &jb, &i__3, &
      c_b16, &a_ref(j, j), lda, &a_ref(j, j + jb), lda);
    if (j + jb <= *m) {

/*                 Update trailing submatrix. */

        i__3 = *m - j - jb + 1;
        i__4 = *n - j - jb + 1;
        dgemm_("No transpose", "No transpose", &i__3, &i__4, &jb,
          &c_b19, &a_ref(j + jb, j), lda, &a_ref(j, j + jb),
           lda, &c_b16, &a_ref(j + jb, j + jb), lda);
    }
      }
/* L20: */
  }
    }
    return 0;

/*     End of DGETRF */

} /* dgetrf_ */

#undef a_ref

/* Subroutine */ int xerbla_(char *srname, integer *info)
{
/*  -- LAPACK auxiliary routine (preliminary version) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       February 29, 1992


    Purpose
    =======

    XERBLA  is an error handler for the LAPACK routines.
    It is called by an LAPACK routine if an input parameter has an
    invalid value.  A message is printed and execution stops.

    Installers may consider modifying the STOP statement in order to
    call system-specific exception-handling facilities.

    Arguments
    =========

    SRNAME  (input) CHARACTER*6
            The name of the routine which called XERBLA.

    INFO    (input) INTEGER
            The position of the invalid parameter in the parameter list
            of the calling routine. */
    /* Table of constant values */
    static integer c__1 = 1;

    /* Format strings */
    static char fmt_9999[] = "(\002 ** On entry to \002,a6,\002 parameter nu"
      "mber \002,i2,\002 had \002,\002an illegal value\002)";
    /* Builtin functions */
    integer s_wsfe(cilist *), do_fio(integer *, char *, ftnlen), e_wsfe(void);
    /* Subroutine */ int s_stop(char *, ftnlen);
    /* Fortran I/O blocks */
    static cilist io___1 = { 0, 6, 0, fmt_9999, 0 };




    s_wsfe(&io___1);
    do_fio(&c__1, srname, (ftnlen)6);
    do_fio(&c__1, (char *)&(*info), (ftnlen)sizeof(integer));
    e_wsfe();

    s_stop("", (ftnlen)0);


/*     End of XERBLA */

    return 0;
} /* xerbla_ */

/* Subroutine */ int dgemm_(char *transa, char *transb, integer *m, integer *
  n, integer *k, doublereal *alpha, doublereal *a, integer *lda,
  doublereal *b, integer *ldb, doublereal *beta, doublereal *c__,
  integer *ldc)
{
    /* System generated locals */
    integer a_dim1, a_offset, b_dim1, b_offset, c_dim1, c_offset, i__1, i__2,
      i__3;
    /* Local variables */
    static integer info;
    static logical nota, notb;
    static doublereal temp;
    static integer i__, j, l, ncola;
    extern logical lsame_(char *, char *);
    static integer nrowa, nrowb;
    extern /* Subroutine */ int xerbla_(char *, integer *);
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]
#define b_ref(a_1,a_2) b[(a_2)*b_dim1 + a_1]
#define c___ref(a_1,a_2) c__[(a_2)*c_dim1 + a_1]
/*  Purpose
    =======
    DGEMM  performs one of the matrix-matrix operations
       C := alpha*op( A )*op( B ) + beta*C,
    where  op( X ) is one of
       op( X ) = X   or   op( X ) = X',
    alpha and beta are scalars, and A, B and C are matrices, with op( A )
    an m by k matrix,  op( B )  a  k by n matrix and  C an m by n matrix.
    Parameters
    ==========
    TRANSA - CHARACTER*1.
             On entry, TRANSA specifies the form of op( A ) to be used in
             the matrix multiplication as follows:
                TRANSA = 'N' or 'n',  op( A ) = A.
                TRANSA = 'T' or 't',  op( A ) = A'.
                TRANSA = 'C' or 'c',  op( A ) = A'.
             Unchanged on exit.
    TRANSB - CHARACTER*1.
             On entry, TRANSB specifies the form of op( B ) to be used in
             the matrix multiplication as follows:
                TRANSB = 'N' or 'n',  op( B ) = B.
                TRANSB = 'T' or 't',  op( B ) = B'.
                TRANSB = 'C' or 'c',  op( B ) = B'.
             Unchanged on exit.
    M      - INTEGER.
             On entry,  M  specifies  the number  of rows  of the  matrix
             op( A )  and of the  matrix  C.  M  must  be at least  zero.
             Unchanged on exit.
    N      - INTEGER.
             On entry,  N  specifies the number  of columns of the matrix
             op( B ) and the number of columns of the matrix C. N must be
             at least zero.
             Unchanged on exit.
    K      - INTEGER.
             On entry,  K  specifies  the number of columns of the matrix
             op( A ) and the number of rows of the matrix op( B ). K must
             be at least  zero.
             Unchanged on exit.
    ALPHA  - DOUBLE PRECISION.
             On entry, ALPHA specifies the scalar alpha.
             Unchanged on exit.
    A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
             k  when  TRANSA = 'N' or 'n',  and is  m  otherwise.
             Before entry with  TRANSA = 'N' or 'n',  the leading  m by k
             part of the array  A  must contain the matrix  A,  otherwise
             the leading  k by m  part of the array  A  must contain  the
             matrix A.
             Unchanged on exit.
    LDA    - INTEGER.
             On entry, LDA specifies the first dimension of A as declared
             in the calling (sub) program. When  TRANSA = 'N' or 'n' then
             LDA must be at least  max( 1, m ), otherwise  LDA must be at
             least  max( 1, k ).
             Unchanged on exit.
    B      - DOUBLE PRECISION array of DIMENSION ( LDB, kb ), where kb is
             n  when  TRANSB = 'N' or 'n',  and is  k  otherwise.
             Before entry with  TRANSB = 'N' or 'n',  the leading  k by n
             part of the array  B  must contain the matrix  B,  otherwise
             the leading  n by k  part of the array  B  must contain  the
             matrix B.
             Unchanged on exit.
    LDB    - INTEGER.
             On entry, LDB specifies the first dimension of B as declared
             in the calling (sub) program. When  TRANSB = 'N' or 'n' then
             LDB must be at least  max( 1, k ), otherwise  LDB must be at
             least  max( 1, n ).
             Unchanged on exit.
    BETA   - DOUBLE PRECISION.
             On entry,  BETA  specifies the scalar  beta.  When  BETA  is
             supplied as zero then C need not be set on input.
             Unchanged on exit.
    C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
             Before entry, the leading  m by n  part of the array  C must
             contain the matrix  C,  except when  beta  is zero, in which
             case C need not be set on entry.
             On exit, the array  C  is overwritten by the  m by n  matrix
             ( alpha*op( A )*op( B ) + beta*C ).
    LDC    - INTEGER.
             On entry, LDC specifies the first dimension of C as declared
             in  the  calling  (sub)  program.   LDC  must  be  at  least
             max( 1, m ).
             Unchanged on exit.
    Level 3 Blas routine.
    -- Written on 8-February-1989.
       Jack Dongarra, Argonne National Laboratory.
       Iain Duff, AERE Harwell.
       Jeremy Du Croz, Numerical Algorithms Group Ltd.
       Sven Hammarling, Numerical Algorithms Group Ltd.
       Set  NOTA  and  NOTB  as  true if  A  and  B  respectively are not
       transposed and set  NROWA, NCOLA and  NROWB  as the number of rows
       and  columns of  A  and the  number of  rows  of  B  respectively.
       Parameter adjustments */
    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    b_dim1 = *ldb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    c_dim1 = *ldc;
    c_offset = 1 + c_dim1 * 1;
    c__ -= c_offset;
    /* Function Body */
    nota = lsame_(transa, "N");
    notb = lsame_(transb, "N");
    if (nota) {
  nrowa = *m;
  ncola = *k;
    } else {
  nrowa = *k;
  ncola = *m;
    }
    if (notb) {
  nrowb = *k;
    } else {
  nrowb = *n;
    }
/*     Test the input parameters. */
    info = 0;
    if (! nota && ! lsame_(transa, "C") && ! lsame_(
      transa, "T")) {
  info = 1;
    } else if (! notb && ! lsame_(transb, "C") && !
      lsame_(transb, "T")) {
  info = 2;
    } else if (*m < 0) {
  info = 3;
    } else if (*n < 0) {
  info = 4;
    } else if (*k < 0) {
  info = 5;
    } else if (*lda < max(1,nrowa)) {
  info = 8;
    } else if (*ldb < max(1,nrowb)) {
  info = 10;
    } else if (*ldc < max(1,*m)) {
  info = 13;
    }
    if (info != 0) {
  xerbla_("DGEMM ", &info);
  return 0;
    }
/*     Quick return if possible. */
    if (*m == 0 || *n == 0 || ((*alpha == 0. || *k == 0) && *beta == 1.)) {
  return 0;
    }
/*     And if  alpha.eq.zero. */
    if (*alpha == 0.) {
  if (*beta == 0.) {
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        c___ref(i__, j) = 0.;
/* L10: */
    }
/* L20: */
      }
  } else {
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        c___ref(i__, j) = *beta * c___ref(i__, j);
/* L30: */
    }
/* L40: */
      }
  }
  return 0;
    }
/*     Start the operations. */
    if (notb) {
  if (nota) {
/*           Form  C := alpha*A*B + beta*C. */
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    if (*beta == 0.) {
        i__2 = *m;
        for (i__ = 1; i__ <= i__2; ++i__) {
      c___ref(i__, j) = 0.;
/* L50: */
        }
    } else if (*beta != 1.) {
        i__2 = *m;
        for (i__ = 1; i__ <= i__2; ++i__) {
      c___ref(i__, j) = *beta * c___ref(i__, j);
/* L60: */
        }
    }
    i__2 = *k;
    for (l = 1; l <= i__2; ++l) {
        if (b_ref(l, j) != 0.) {
      temp = *alpha * b_ref(l, j);
      i__3 = *m;
      for (i__ = 1; i__ <= i__3; ++i__) {
          c___ref(i__, j) = c___ref(i__, j) + temp * a_ref(
            i__, l);
/* L70: */
      }
        }
/* L80: */
    }
/* L90: */
      }
  } else {
/*           Form  C := alpha*A'*B + beta*C */
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        temp = 0.;
        i__3 = *k;
        for (l = 1; l <= i__3; ++l) {
      temp += a_ref(l, i__) * b_ref(l, j);
/* L100: */
        }
        if (*beta == 0.) {
      c___ref(i__, j) = *alpha * temp;
        } else {
      c___ref(i__, j) = *alpha * temp + *beta * c___ref(i__,
         j);
        }
/* L110: */
    }
/* L120: */
      }
  }
    } else {
  if (nota) {
/*           Form  C := alpha*A*B' + beta*C */
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    if (*beta == 0.) {
        i__2 = *m;
        for (i__ = 1; i__ <= i__2; ++i__) {
      c___ref(i__, j) = 0.;
/* L130: */
        }
    } else if (*beta != 1.) {
        i__2 = *m;
        for (i__ = 1; i__ <= i__2; ++i__) {
      c___ref(i__, j) = *beta * c___ref(i__, j);
/* L140: */
        }
    }
    i__2 = *k;
    for (l = 1; l <= i__2; ++l) {
        if (b_ref(j, l) != 0.) {
      temp = *alpha * b_ref(j, l);
      i__3 = *m;
      for (i__ = 1; i__ <= i__3; ++i__) {
          c___ref(i__, j) = c___ref(i__, j) + temp * a_ref(
            i__, l);
/* L150: */
      }
        }
/* L160: */
    }
/* L170: */
      }
  } else {
/*           Form  C := alpha*A'*B' + beta*C */
      i__1 = *n;
      for (j = 1; j <= i__1; ++j) {
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        temp = 0.;
        i__3 = *k;
        for (l = 1; l <= i__3; ++l) {
      temp += a_ref(l, i__) * b_ref(j, l);
/* L180: */
        }
        if (*beta == 0.) {
      c___ref(i__, j) = *alpha * temp;
        } else {
      c___ref(i__, j) = *alpha * temp + *beta * c___ref(i__,
         j);
        }
/* L190: */
    }
/* L200: */
      }
  }
    }
    return 0;
/*     End of DGEMM . */
} /* dgemm_ */
#undef c___ref
#undef b_ref
#undef a_ref

/* Subroutine */ int dlaswp_(integer *n, doublereal *a, integer *lda, integer
  *k1, integer *k2, integer *ipiv, integer *incx)
{
/*  -- LAPACK auxiliary routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       June 30, 1999


    Purpose
    =======

    DLASWP performs a series of row interchanges on the matrix A.
    One row interchange is initiated for each of rows K1 through K2 of A.

    Arguments
    =========

    N       (input) INTEGER
            The number of columns of the matrix A.

    A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
            On entry, the matrix of column dimension N to which the row
            interchanges will be applied.
            On exit, the permuted matrix.

    LDA     (input) INTEGER
            The leading dimension of the array A.

    K1      (input) INTEGER
            The first element of IPIV for which a row interchange will
            be done.

    K2      (input) INTEGER
            The last element of IPIV for which a row interchange will
            be done.

    IPIV    (input) INTEGER array, dimension (M*abs(INCX))
            The vector of pivot indices.  Only the elements in positions
            K1 through K2 of IPIV are accessed.
            IPIV(K) = L implies rows K and L are to be interchanged.

    INCX    (input) INTEGER
            The increment between successive values of IPIV.  If IPIV
            is negative, the pivots are applied in reverse order.

    Further Details
    ===============

    Modified by
     R. C. Whaley, Computer Science Dept., Univ. of Tenn., Knoxville, USA

   =====================================================================


       Interchange row I with row IPIV(I) for each of rows K1 through K2.

       Parameter adjustments */
    /* System generated locals */
    integer a_dim1, a_offset, i__1, i__2, i__3, i__4;
    /* Local variables */
    static doublereal temp;
    static integer i__, j, k, i1, i2, n32, ip, ix, ix0, inc;
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]

    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    --ipiv;

    /* Function Body */
    if (*incx > 0) {
  ix0 = *k1;
  i1 = *k1;
  i2 = *k2;
  inc = 1;
    } else if (*incx < 0) {
  ix0 = (1 - *k2) * *incx + 1;
  i1 = *k2;
  i2 = *k1;
  inc = -1;
    } else {
  return 0;
    }

    n32 = *n / 32 << 5;
    if (n32 != 0) {
  i__1 = n32;
  for (j = 1; j <= i__1; j += 32) {
      ix = ix0;
      i__2 = i2;
      i__3 = inc;
      for (i__ = i1; i__3 < 0 ? i__ >= i__2 : i__ <= i__2; i__ += i__3)
        {
    ip = ipiv[ix];
    if (ip != i__) {
        i__4 = j + 31;
        for (k = j; k <= i__4; ++k) {
      temp = a_ref(i__, k);
      a_ref(i__, k) = a_ref(ip, k);
      a_ref(ip, k) = temp;
/* L10: */
        }
    }
    ix += *incx;
/* L20: */
      }
/* L30: */
  }
    }
    if (n32 != *n) {
  ++n32;
  ix = ix0;
  i__1 = i2;
  i__3 = inc;
  for (i__ = i1; i__3 < 0 ? i__ >= i__1 : i__ <= i__1; i__ += i__3) {
      ip = ipiv[ix];
      if (ip != i__) {
    i__2 = *n;
    for (k = n32; k <= i__2; ++k) {
        temp = a_ref(i__, k);
        a_ref(i__, k) = a_ref(ip, k);
        a_ref(ip, k) = temp;
/* L40: */
    }
      }
      ix += *incx;
/* L50: */
  }
    }

    return 0;

/*     End of DLASWP */

} /* dlaswp_ */

#undef a_ref


logical lsame_(char *ca, char *cb)
{
/*  -- LAPACK auxiliary routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       September 30, 1994


    Purpose
    =======

    LSAME returns .TRUE. if CA is the same letter as CB regardless of
    case.

    Arguments
    =========

    CA      (input) CHARACTER*1
    CB      (input) CHARACTER*1
            CA and CB specify the single characters to be compared.

   =====================================================================



       Test if the characters are equal */
    /* System generated locals */
    logical ret_val;
    /* Local variables */
    static integer inta, intb, zcode;


    ret_val = *(unsigned char *)ca == *(unsigned char *)cb;
    if (ret_val) {
  return ret_val;
    }

/*     Now test for equivalence if both characters are alphabetic. */

    zcode = 'Z';

/*     Use 'Z' rather than 'A' so that ASCII can be detected on Prime
       machines, on which ICHAR returns a value with bit 8 set.
       ICHAR('A') on Prime machines returns 193 which is the same as
       ICHAR('A') on an EBCDIC machine. */

    inta = *(unsigned char *)ca;
    intb = *(unsigned char *)cb;

    if (zcode == 90 || zcode == 122) {

/*        ASCII is assumed - ZCODE is the ASCII code of either lower o
r
          upper case 'Z'. */

  if (inta >= 97 && inta <= 122) {
      inta += -32;
  }
  if (intb >= 97 && intb <= 122) {
      intb += -32;
  }

    } else if (zcode == 233 || zcode == 169) {

/*        EBCDIC is assumed - ZCODE is the EBCDIC code of either lower
 or
          upper case 'Z'. */

  if ((inta >= 129 && inta <= 137) || (inta >= 145 && inta <= 153) || (inta
    >= 162 && inta <= 169)) {
      inta += 64;
  }
  if ((intb >= 129 && intb <= 137) || (intb >= 145 && intb <= 153) || (intb
    >= 162 && intb <= 169)) {
      intb += 64;
  }

    } else if (zcode == 218 || zcode == 250) {

/*        ASCII is assumed, on Prime machines - ZCODE is the ASCII cod
e
          plus 128 of either lower or upper case 'Z'. */

  if (inta >= 225 && inta <= 250) {
      inta += -32;
  }
  if (intb >= 225 && intb <= 250) {
      intb += -32;
  }
    }
    ret_val = inta == intb;

/*     RETURN

       End of LSAME */

    return ret_val;
} /* lsame_ */

integer ilaenv_(integer *ispec, char *name__, char *opts, integer *n1,
  integer *n2, integer *n3, integer *n4, ftnlen name_len, ftnlen
  opts_len)
{
/*  -- LAPACK auxiliary routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       June 30, 1999


    Purpose
    =======

    ILAENV is called from the LAPACK routines to choose problem-dependent
    parameters for the local environment.  See ISPEC for a description of
    the parameters.

    This version provides a set of parameters which should give good,
    but not optimal, performance on many of the currently available
    computers.  Users are encouraged to modify this subroutine to set
    the tuning parameters for their particular machine using the option
    and problem size information in the arguments.

    This routine will not function correctly if it is converted to all
    lower case.  Converting it to all upper case is allowed.

    Arguments
    =========

    ISPEC   (input) INTEGER
            Specifies the parameter to be returned as the value of
            ILAENV.
            = 1: the optimal blocksize; if this value is 1, an unblocked
                 algorithm will give the best performance.
            = 2: the minimum block size for which the block routine
                 should be used; if the usable block size is less than
                 this value, an unblocked routine should be used.
            = 3: the crossover point (in a block routine, for N less
                 than this value, an unblocked routine should be used)
            = 4: the number of shifts, used in the nonsymmetric
                 eigenvalue routines
            = 5: the minimum column dimension for blocking to be used;
                 rectangular blocks must have dimension at least k by m,
                 where k is given by ILAENV(2,...) and m by ILAENV(5,...)
            = 6: the crossover point for the SVD (when reducing an m by n
                 matrix to bidiagonal form, if max(m,n)/min(m,n) exceeds
                 this value, a QR factorization is used first to reduce
                 the matrix to a triangular form.)
            = 7: the number of processors
            = 8: the crossover point for the multishift QR and QZ methods
                 for nonsymmetric eigenvalue problems.
            = 9: maximum size of the subproblems at the bottom of the
                 computation tree in the divide-and-conquer algorithm
                 (used by xGELSD and xGESDD)
            =10: ieee NaN arithmetic can be trusted not to trap
            =11: infinity arithmetic can be trusted not to trap

    NAME    (input) CHARACTER*(*)
            The name of the calling subroutine, in either upper case or
            lower case.

    OPTS    (input) CHARACTER*(*)
            The character options to the subroutine NAME, concatenated
            into a single character string.  For example, UPLO = 'U',
            TRANS = 'T', and DIAG = 'N' for a triangular routine would
            be specified as OPTS = 'UTN'.

    N1      (input) INTEGER
    N2      (input) INTEGER
    N3      (input) INTEGER
    N4      (input) INTEGER
            Problem dimensions for the subroutine NAME; these may not all
            be required.

   (ILAENV) (output) INTEGER
            >= 0: the value of the parameter specified by ISPEC
            < 0:  if ILAENV = -k, the k-th argument had an illegal value.

    Further Details
    ===============

    The following conventions have been used when calling ILAENV from the
    LAPACK routines:
    1)  OPTS is a concatenation of all of the character options to
        subroutine NAME, in the same order that they appear in the
        argument list for NAME, even if they are not used in determining
        the value of the parameter specified by ISPEC.
    2)  The problem dimensions N1, N2, N3, N4 are specified in the order
        that they appear in the argument list for NAME.  N1 is used
        first, N2 second, and so on, and unused problem dimensions are
        passed a value of -1.
    3)  The parameter value returned by ILAENV is checked for validity in
        the calling subroutine.  For example, ILAENV is used to retrieve
        the optimal blocksize for STRTRI as follows:

        NB = ILAENV( 1, 'STRTRI', UPLO // DIAG, N, -1, -1, -1 )
        IF( NB.LE.1 ) NB = MAX( 1, N )

    ===================================================================== */
    /* Table of constant values */
    static integer c__0 = 0;
    static real c_b162 = 0.f;
    static real c_b163 = 1.f;
    static integer c__1 = 1;

    /* System generated locals */
    integer ret_val;
    /* Builtin functions
       Subroutine */ int s_copy(char *, char *, ftnlen, ftnlen);
    integer s_cmp(char *, char *, ftnlen, ftnlen);
    /* Local variables */
    static integer i__;
    static logical cname, sname;
    static integer nbmin;
    static char c1[1], c2[2], c3[3], c4[2];
    static integer ic, nb;
    extern integer ieeeck_(integer *, real *, real *);
    static integer iz, nx;
    static char subnam[6];




    switch (*ispec) {
  case 1:  goto L100;
  case 2:  goto L100;
  case 3:  goto L100;
  case 4:  goto L400;
  case 5:  goto L500;
  case 6:  goto L600;
  case 7:  goto L700;
  case 8:  goto L800;
  case 9:  goto L900;
  case 10:  goto L1000;
  case 11:  goto L1100;
    }

/*     Invalid value for ISPEC */

    ret_val = -1;
    return ret_val;

L100:

/*     Convert NAME to upper case if the first character is lower case. */

    ret_val = 1;
    s_copy(subnam, name__, (ftnlen)6, name_len);
    ic = *(unsigned char *)subnam;
    iz = 'Z';
    if (iz == 90 || iz == 122) {

/*        ASCII character set */

  if (ic >= 97 && ic <= 122) {
      *(unsigned char *)subnam = (char) (ic - 32);
      for (i__ = 2; i__ <= 6; ++i__) {
    ic = *(unsigned char *)&subnam[i__ - 1];
    if (ic >= 97 && ic <= 122) {
        *(unsigned char *)&subnam[i__ - 1] = (char) (ic - 32);
    }
/* L10: */
      }
  }

    } else if (iz == 233 || iz == 169) {

/*        EBCDIC character set */

  if ((ic >= 129 && ic <= 137) || (ic >= 145 && ic <= 153) || (ic >= 162 &&
    ic <= 169)) {
      *(unsigned char *)subnam = (char) (ic + 64);
      for (i__ = 2; i__ <= 6; ++i__) {
    ic = *(unsigned char *)&subnam[i__ - 1];
    if ((ic >= 129 && ic <= 137) || (ic >= 145 && ic <= 153) || (ic >=
      162 && ic <= 169)) {
        *(unsigned char *)&subnam[i__ - 1] = (char) (ic + 64);
    }
/* L20: */
      }
  }

    } else if (iz == 218 || iz == 250) {

/*        Prime machines:  ASCII+128 */

  if (ic >= 225 && ic <= 250) {
      *(unsigned char *)subnam = (char) (ic - 32);
      for (i__ = 2; i__ <= 6; ++i__) {
    ic = *(unsigned char *)&subnam[i__ - 1];
    if (ic >= 225 && ic <= 250) {
        *(unsigned char *)&subnam[i__ - 1] = (char) (ic - 32);
    }
/* L30: */
      }
  }
    }

    *(unsigned char *)c1 = *(unsigned char *)subnam;
    sname = *(unsigned char *)c1 == 'S' || *(unsigned char *)c1 == 'D';
    cname = *(unsigned char *)c1 == 'C' || *(unsigned char *)c1 == 'Z';
    if (! (cname || sname)) {
  return ret_val;
    }
    s_copy(c2, subnam + 1, (ftnlen)2, (ftnlen)2);
    s_copy(c3, subnam + 3, (ftnlen)3, (ftnlen)3);
    s_copy(c4, c3 + 1, (ftnlen)2, (ftnlen)2);

    switch (*ispec) {
  case 1:  goto L110;
  case 2:  goto L200;
  case 3:  goto L300;
    }

L110:

/*     ISPEC = 1:  block size

       In these examples, separate code is provided for setting NB for
       real and complex.  We assume that NB will take the same value in
       single or double precision. */

    nb = 1;

    if (s_cmp(c2, "GE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  } else if (s_cmp(c3, "QRF", (ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3,
    "RQF", (ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3, "LQF", (ftnlen)
    3, (ftnlen)3) == 0 || s_cmp(c3, "QLF", (ftnlen)3, (ftnlen)3)
    == 0) {
      if (sname) {
    nb = 32;
      } else {
    nb = 32;
      }
  } else if (s_cmp(c3, "HRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 32;
      } else {
    nb = 32;
      }
  } else if (s_cmp(c3, "BRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 32;
      } else {
    nb = 32;
      }
  } else if (s_cmp(c3, "TRI", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  }
    } else if (s_cmp(c2, "PO", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  }
    } else if (s_cmp(c2, "SY", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  } else if (sname && s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 32;
  } else if (sname && s_cmp(c3, "GST", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 64;
  }
    } else if (cname && s_cmp(c2, "HE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 64;
  } else if (s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 32;
  } else if (s_cmp(c3, "GST", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 64;
  }
    } else if (sname && s_cmp(c2, "OR", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nb = 32;
      }
  } else if (*(unsigned char *)c3 == 'M') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nb = 32;
      }
  }
    } else if (cname && s_cmp(c2, "UN", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nb = 32;
      }
  } else if (*(unsigned char *)c3 == 'M') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nb = 32;
      }
  }
    } else if (s_cmp(c2, "GB", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    if (*n4 <= 64) {
        nb = 1;
    } else {
        nb = 32;
    }
      } else {
    if (*n4 <= 64) {
        nb = 1;
    } else {
        nb = 32;
    }
      }
  }
    } else if (s_cmp(c2, "PB", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    if (*n2 <= 64) {
        nb = 1;
    } else {
        nb = 32;
    }
      } else {
    if (*n2 <= 64) {
        nb = 1;
    } else {
        nb = 32;
    }
      }
  }
    } else if (s_cmp(c2, "TR", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRI", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  }
    } else if (s_cmp(c2, "LA", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "UUM", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nb = 64;
      } else {
    nb = 64;
      }
  }
    } else if (sname && s_cmp(c2, "ST", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "EBZ", (ftnlen)3, (ftnlen)3) == 0) {
      nb = 1;
  }
    }
    ret_val = nb;
    return ret_val;

L200:

/*     ISPEC = 2:  minimum block size */

    nbmin = 2;
    if (s_cmp(c2, "GE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "QRF", (ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3, "RQF", (
    ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3, "LQF", (ftnlen)3, (
    ftnlen)3) == 0 || s_cmp(c3, "QLF", (ftnlen)3, (ftnlen)3) == 0)
     {
      if (sname) {
    nbmin = 2;
      } else {
    nbmin = 2;
      }
  } else if (s_cmp(c3, "HRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nbmin = 2;
      } else {
    nbmin = 2;
      }
  } else if (s_cmp(c3, "BRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nbmin = 2;
      } else {
    nbmin = 2;
      }
  } else if (s_cmp(c3, "TRI", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nbmin = 2;
      } else {
    nbmin = 2;
      }
  }
    } else if (s_cmp(c2, "SY", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRF", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nbmin = 8;
      } else {
    nbmin = 8;
      }
  } else if (sname && s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nbmin = 2;
  }
    } else if (cname && s_cmp(c2, "HE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nbmin = 2;
  }
    } else if (sname && s_cmp(c2, "OR", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nbmin = 2;
      }
  } else if (*(unsigned char *)c3 == 'M') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nbmin = 2;
      }
  }
    } else if (cname && s_cmp(c2, "UN", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nbmin = 2;
      }
  } else if (*(unsigned char *)c3 == 'M') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nbmin = 2;
      }
  }
    }
    ret_val = nbmin;
    return ret_val;

L300:

/*     ISPEC = 3:  crossover point */

    nx = 0;
    if (s_cmp(c2, "GE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "QRF", (ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3, "RQF", (
    ftnlen)3, (ftnlen)3) == 0 || s_cmp(c3, "LQF", (ftnlen)3, (
    ftnlen)3) == 0 || s_cmp(c3, "QLF", (ftnlen)3, (ftnlen)3) == 0)
     {
      if (sname) {
    nx = 128;
      } else {
    nx = 128;
      }
  } else if (s_cmp(c3, "HRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nx = 128;
      } else {
    nx = 128;
      }
  } else if (s_cmp(c3, "BRD", (ftnlen)3, (ftnlen)3) == 0) {
      if (sname) {
    nx = 128;
      } else {
    nx = 128;
      }
  }
    } else if (s_cmp(c2, "SY", (ftnlen)2, (ftnlen)2) == 0) {
  if (sname && s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nx = 32;
  }
    } else if (cname && s_cmp(c2, "HE", (ftnlen)2, (ftnlen)2) == 0) {
  if (s_cmp(c3, "TRD", (ftnlen)3, (ftnlen)3) == 0) {
      nx = 32;
  }
    } else if (sname && s_cmp(c2, "OR", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nx = 128;
      }
  }
    } else if (cname && s_cmp(c2, "UN", (ftnlen)2, (ftnlen)2) == 0) {
  if (*(unsigned char *)c3 == 'G') {
      if (s_cmp(c4, "QR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "RQ",
        (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "LQ", (ftnlen)2, (
        ftnlen)2) == 0 || s_cmp(c4, "QL", (ftnlen)2, (ftnlen)2) ==
         0 || s_cmp(c4, "HR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(
        c4, "TR", (ftnlen)2, (ftnlen)2) == 0 || s_cmp(c4, "BR", (
        ftnlen)2, (ftnlen)2) == 0) {
    nx = 128;
      }
  }
    }
    ret_val = nx;
    return ret_val;

L400:

/*     ISPEC = 4:  number of shifts (used by xHSEQR) */

    ret_val = 6;
    return ret_val;

L500:

/*     ISPEC = 5:  minimum column dimension (not used) */

    ret_val = 2;
    return ret_val;

L600:

/*     ISPEC = 6:  crossover point for SVD (used by xGELSS and xGESVD) */

    ret_val = (integer) ((real) min(*n1,*n2) * 1.6f);
    return ret_val;

L700:

/*     ISPEC = 7:  number of processors (not used) */

    ret_val = 1;
    return ret_val;

L800:

/*     ISPEC = 8:  crossover point for multishift (used by xHSEQR) */

    ret_val = 50;
    return ret_val;

L900:

/*     ISPEC = 9:  maximum size of the subproblems at the bottom of the
                   computation tree in the divide-and-conquer algorithm
                   (used by xGELSD and xGESDD) */

    ret_val = 25;
    return ret_val;

L1000:

/*     ISPEC = 10: ieee NaN arithmetic can be trusted not to trap

       ILAENV = 0 */
    ret_val = 1;
    if (ret_val == 1) {
  ret_val = ieeeck_(&c__0, &c_b162, &c_b163);
    }
    return ret_val;

L1100:

/*     ISPEC = 11: infinity arithmetic can be trusted not to trap

       ILAENV = 0 */
    ret_val = 1;
    if (ret_val == 1) {
  ret_val = ieeeck_(&c__1, &c_b162, &c_b163);
    }
    return ret_val;

/*     End of ILAENV */

} /* ilaenv_ */

integer ieeeck_(integer *ispec, real *zero, real *one)
{
/*  -- LAPACK auxiliary routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       June 30, 1998


    Purpose
    =======

    IEEECK is called from the ILAENV to verify that Infinity and
    possibly NaN arithmetic is safe (i.e. will not trap).

    Arguments
    =========

    ISPEC   (input) INTEGER
            Specifies whether to test just for inifinity arithmetic
            or whether to test for infinity and NaN arithmetic.
            = 0: Verify infinity arithmetic only.
            = 1: Verify infinity and NaN arithmetic.

    ZERO    (input) REAL
            Must contain the value 0.0
            This is passed to prevent the compiler from optimizing
            away this code.

    ONE     (input) REAL
            Must contain the value 1.0
            This is passed to prevent the compiler from optimizing
            away this code.

    RETURN VALUE:  INTEGER
            = 0:  Arithmetic failed to produce the correct answers
            = 1:  Arithmetic produced the correct answers */
    /* System generated locals */
    integer ret_val;
    /* Local variables */
    static real neginf, posinf, negzro, newzro, nan1, nan2, nan3, nan4, nan5,
      nan6;


    ret_val = 1;

    posinf = *one / *zero;
    if (posinf <= *one) {
  ret_val = 0;
  return ret_val;
    }

    neginf = -(*one) / *zero;
    if (neginf >= *zero) {
  ret_val = 0;
  return ret_val;
    }

    negzro = *one / (neginf + *one);
    if (negzro != *zero) {
  ret_val = 0;
  return ret_val;
    }

    neginf = *one / negzro;
    if (neginf >= *zero) {
  ret_val = 0;
  return ret_val;
    }

    newzro = negzro + *zero;
    if (newzro != *zero) {
  ret_val = 0;
  return ret_val;
    }

    posinf = *one / newzro;
    if (posinf <= *one) {
  ret_val = 0;
  return ret_val;
    }

    neginf *= posinf;
    if (neginf >= *zero) {
  ret_val = 0;
  return ret_val;
    }

    posinf *= posinf;
    if (posinf <= *one) {
  ret_val = 0;
  return ret_val;
    }




/*     Return if we were only asked to check infinity arithmetic */

    if (*ispec == 0) {
  return ret_val;
    }

    nan1 = posinf + neginf;

    nan2 = posinf / neginf;

    nan3 = posinf / posinf;

    nan4 = posinf * *zero;

    nan5 = neginf * negzro;

    nan6 = nan5 * 0.f;

    if (nan1 == nan1) {
  ret_val = 0;
  return ret_val;
    }

    if (nan2 == nan2) {
  ret_val = 0;
  return ret_val;
    }

    if (nan3 == nan3) {
  ret_val = 0;
  return ret_val;
    }

    if (nan4 == nan4) {
  ret_val = 0;
  return ret_val;
    }

    if (nan5 == nan5) {
  ret_val = 0;
  return ret_val;
    }

    if (nan6 == nan6) {
  ret_val = 0;
  return ret_val;
    }

    return ret_val;
} /* ieeeck_ */

/* Subroutine */ int dgetf2_(integer *m, integer *n, doublereal *a, integer *
  lda, integer *ipiv, integer *info)
{
/*  -- LAPACK routine (version 3.0) --
       Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
       Courant Institute, Argonne National Lab, and Rice University
       June 30, 1992


    Purpose
    =======

    DGETF2 computes an LU factorization of a general m-by-n matrix A
    using partial pivoting with row interchanges.

    The factorization has the form
       A = P * L * U
    where P is a permutation matrix, L is lower triangular with unit
    diagonal elements (lower trapezoidal if m > n), and U is upper
    triangular (upper trapezoidal if m < n).

    This is the right-looking Level 2 BLAS version of the algorithm.

    Arguments
    =========

    M       (input) INTEGER
            The number of rows of the matrix A.  M >= 0.

    N       (input) INTEGER
            The number of columns of the matrix A.  N >= 0.

    A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
            On entry, the m by n matrix to be factored.
            On exit, the factors L and U from the factorization
            A = P*L*U; the unit diagonal elements of L are not stored.

    LDA     (input) INTEGER
            The leading dimension of the array A.  LDA >= max(1,M).

    IPIV    (output) INTEGER array, dimension (min(M,N))
            The pivot indices; for 1 <= i <= min(M,N), row i of the
            matrix was interchanged with row IPIV(i).

    INFO    (output) INTEGER
            = 0: successful exit
            < 0: if INFO = -k, the k-th argument had an illegal value
            > 0: if INFO = k, U(k,k) is exactly zero. The factorization
                 has been completed, but the factor U is exactly
                 singular, and division by zero will occur if it is used
                 to solve a system of equations.

    =====================================================================


       Test the input parameters.

       Parameter adjustments */
    /* Table of constant values */
    static integer c__1 = 1;
    static doublereal c_b6 = -1.;

    /* System generated locals */
    integer a_dim1, a_offset, i__1, i__2, i__3;
    doublereal d__1;
    /* Local variables */
    extern /* Subroutine */ int dger_(integer *, integer *, doublereal *,
      doublereal *, integer *, doublereal *, integer *, doublereal *,
      integer *);
    static integer j;
    extern /* Subroutine */ int dscal_(integer *, doublereal *, doublereal *,
      integer *), dswap_(integer *, doublereal *, integer *, doublereal
      *, integer *);
    static integer jp;
    extern integer idamax_(integer *, doublereal *, integer *);
    extern /* Subroutine */ int xerbla_(char *, integer *);
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]


    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    --ipiv;

    /* Function Body */
    *info = 0;
    if (*m < 0) {
  *info = -1;
    } else if (*n < 0) {
  *info = -2;
    } else if (*lda < max(1,*m)) {
  *info = -4;
    }
    if (*info != 0) {
  i__1 = -(*info);
  xerbla_("DGETF2", &i__1);
  return 0;
    }

/*     Quick return if possible */

    if (*m == 0 || *n == 0) {
  return 0;
    }

    i__1 = min(*m,*n);
    for (j = 1; j <= i__1; ++j) {

/*        Find pivot and test for singularity. */

  i__2 = *m - j + 1;
  jp = j - 1 + idamax_(&i__2, &a_ref(j, j), &c__1);
  ipiv[j] = jp;
  if (a_ref(jp, j) != 0.) {

/*           Apply the interchange to columns 1:N. */

      if (jp != j) {
    dswap_(n, &a_ref(j, 1), lda, &a_ref(jp, 1), lda);
      }

/*           Compute elements J+1:M of J-th column. */

      if (j < *m) {
    i__2 = *m - j;
    d__1 = 1. / a_ref(j, j);
    dscal_(&i__2, &d__1, &a_ref(j + 1, j), &c__1);
      }

  } else if (*info == 0) {

      *info = j;
  }

  if (j < min(*m,*n)) {

/*           Update trailing submatrix. */

      i__2 = *m - j;
      i__3 = *n - j;
      dger_(&i__2, &i__3, &c_b6, &a_ref(j + 1, j), &c__1, &a_ref(j, j +
        1), lda, &a_ref(j + 1, j + 1), lda);
  }
/* L10: */
    }
    return 0;

/*     End of DGETF2 */

} /* dgetf2_ */

#undef a_ref


/* Subroutine */ int dger_(integer *m, integer *n, doublereal *alpha,
  doublereal *x, integer *incx, doublereal *y, integer *incy,
  doublereal *a, integer *lda)
{
    /* System generated locals */
    integer a_dim1, a_offset, i__1, i__2;
    /* Local variables */
    static integer info;
    static doublereal temp;
    static integer i__, j, ix, jy, kx;
    extern /* Subroutine */ int xerbla_(char *, integer *);
#define a_ref(a_1,a_2) a[(a_2)*a_dim1 + a_1]
/*  Purpose
    =======
    DGER   performs the rank 1 operation
       A := alpha*x*y' + A,
    where alpha is a scalar, x is an m element vector, y is an n element
    vector and A is an m by n matrix.
    Parameters
    ==========
    M      - INTEGER.
             On entry, M specifies the number of rows of the matrix A.
             M must be at least zero.
             Unchanged on exit.
    N      - INTEGER.
             On entry, N specifies the number of columns of the matrix A.
             N must be at least zero.
             Unchanged on exit.
    ALPHA  - DOUBLE PRECISION.
             On entry, ALPHA specifies the scalar alpha.
             Unchanged on exit.
    X      - DOUBLE PRECISION array of dimension at least
             ( 1 + ( m - 1 )*abs( INCX ) ).
             Before entry, the incremented array X must contain the m
             element vector x.
             Unchanged on exit.
    INCX   - INTEGER.
             On entry, INCX specifies the increment for the elements of
             X. INCX must not be zero.
             Unchanged on exit.
    Y      - DOUBLE PRECISION array of dimension at least
             ( 1 + ( n - 1 )*abs( INCY ) ).
             Before entry, the incremented array Y must contain the n
             element vector y.
             Unchanged on exit.
    INCY   - INTEGER.
             On entry, INCY specifies the increment for the elements of
             Y. INCY must not be zero.
             Unchanged on exit.
    A      - DOUBLE PRECISION array of DIMENSION ( LDA, n ).
             Before entry, the leading m by n part of the array A must
             contain the matrix of coefficients. On exit, A is
             overwritten by the updated matrix.
    LDA    - INTEGER.
             On entry, LDA specifies the first dimension of A as declared
             in the calling (sub) program. LDA must be at least
             max( 1, m ).
             Unchanged on exit.
    Level 2 Blas routine.
    -- Written on 22-October-1986.
       Jack Dongarra, Argonne National Lab.
       Jeremy Du Croz, Nag Central Office.
       Sven Hammarling, Nag Central Office.
       Richard Hanson, Sandia National Labs.
       Test the input parameters.
       Parameter adjustments */
    --x;
    --y;
    a_dim1 = *lda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    /* Function Body */
    info = 0;
    if (*m < 0) {
  info = 1;
    } else if (*n < 0) {
  info = 2;
    } else if (*incx == 0) {
  info = 5;
    } else if (*incy == 0) {
  info = 7;
    } else if (*lda < max(1,*m)) {
  info = 9;
    }
    if (info != 0) {
  xerbla_("DGER  ", &info);
  return 0;
    }
/*     Quick return if possible. */
    if (*m == 0 || *n == 0 || *alpha == 0.) {
  return 0;
    }
/*     Start the operations. In this version the elements of A are
       accessed sequentially with one pass through A. */
    if (*incy > 0) {
  jy = 1;
    } else {
  jy = 1 - (*n - 1) * *incy;
    }
    if (*incx == 1) {
  i__1 = *n;
  for (j = 1; j <= i__1; ++j) {
      if (y[jy] != 0.) {
    temp = *alpha * y[jy];
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        a_ref(i__, j) = a_ref(i__, j) + x[i__] * temp;
/* L10: */
    }
      }
      jy += *incy;
/* L20: */
  }
    } else {
  if (*incx > 0) {
      kx = 1;
  } else {
      kx = 1 - (*m - 1) * *incx;
  }
  i__1 = *n;
  for (j = 1; j <= i__1; ++j) {
      if (y[jy] != 0.) {
    temp = *alpha * y[jy];
    ix = kx;
    i__2 = *m;
    for (i__ = 1; i__ <= i__2; ++i__) {
        a_ref(i__, j) = a_ref(i__, j) + x[ix] * temp;
        ix += *incx;
/* L30: */
    }
      }
      jy += *incy;
/* L40: */
  }
    }
    return 0;
/*     End of DGER  . */
} /* dger_ */
#undef a_ref

/* Subroutine */ int dscal_(integer *n, doublereal *da, doublereal *dx,
  integer *incx)
{
    /* System generated locals */
    integer i__1, i__2;
    /* Local variables */
    static integer i__, m, nincx, mp1;
/*     scales a vector by a constant.
       uses unrolled loops for increment equal to one.
       jack dongarra, linpack, 3/11/78.
       modified 3/93 to return if incx .le. 0.
       modified 12/3/93, array(1) declarations changed to array(*)
       Parameter adjustments */
    --dx;
    /* Function Body */
    if (*n <= 0 || *incx <= 0) {
  return 0;
    }
    if (*incx == 1) {
  goto L20;
    }
/*        code for increment not equal to 1 */
    nincx = *n * *incx;
    i__1 = nincx;
    i__2 = *incx;
    for (i__ = 1; i__2 < 0 ? i__ >= i__1 : i__ <= i__1; i__ += i__2) {
  dx[i__] = *da * dx[i__];
/* L10: */
    }
    return 0;
/*        code for increment equal to 1
          clean-up loop */
L20:
    m = *n % 5;
    if (m == 0) {
  goto L40;
    }
    i__2 = m;
    for (i__ = 1; i__ <= i__2; ++i__) {
  dx[i__] = *da * dx[i__];
/* L30: */
    }
    if (*n < 5) {
  return 0;
    }
L40:
    mp1 = m + 1;
    i__2 = *n;
    for (i__ = mp1; i__ <= i__2; i__ += 5) {
  dx[i__] = *da * dx[i__];
  dx[i__ + 1] = *da * dx[i__ + 1];
  dx[i__ + 2] = *da * dx[i__ + 2];
  dx[i__ + 3] = *da * dx[i__ + 3];
  dx[i__ + 4] = *da * dx[i__ + 4];
/* L50: */
    }
    return 0;
} /* dscal_ */


/* Subroutine */ int dswap_(integer *n, doublereal *dx, integer *incx,
  doublereal *dy, integer *incy)
{
    /* System generated locals */
    integer i__1;
    /* Local variables */
    static integer i__, m;
    static doublereal dtemp;
    static integer ix, iy, mp1;
/*     interchanges two vectors.
       uses unrolled loops for increments equal one.
       jack dongarra, linpack, 3/11/78.
       modified 12/3/93, array(1) declarations changed to array(*)
       Parameter adjustments */
    --dy;
    --dx;
    /* Function Body */
    if (*n <= 0) {
  return 0;
    }
    if (*incx == 1 && *incy == 1) {
  goto L20;
    }
/*       code for unequal increments or equal increments not equal
           to 1 */
    ix = 1;
    iy = 1;
    if (*incx < 0) {
  ix = (-(*n) + 1) * *incx + 1;
    }
    if (*incy < 0) {
  iy = (-(*n) + 1) * *incy + 1;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
  dtemp = dx[ix];
  dx[ix] = dy[iy];
  dy[iy] = dtemp;
  ix += *incx;
  iy += *incy;
/* L10: */
    }
    return 0;
/*       code for both increments equal to 1
         clean-up loop */
L20:
    m = *n % 3;
    if (m == 0) {
  goto L40;
    }
    i__1 = m;
    for (i__ = 1; i__ <= i__1; ++i__) {
  dtemp = dx[i__];
  dx[i__] = dy[i__];
  dy[i__] = dtemp;
/* L30: */
    }
    if (*n < 3) {
  return 0;
    }
L40:
    mp1 = m + 1;
    i__1 = *n;
    for (i__ = mp1; i__ <= i__1; i__ += 3) {
  dtemp = dx[i__];
  dx[i__] = dy[i__];
  dy[i__] = dtemp;
  dtemp = dx[i__ + 1];
  dx[i__ + 1] = dy[i__ + 1];
  dy[i__ + 1] = dtemp;
  dtemp = dx[i__ + 2];
  dx[i__ + 2] = dy[i__ + 2];
  dy[i__ + 2] = dtemp;
/* L50: */
    }
    return 0;
} /* dswap_ */


integer idamax_(integer *n, doublereal *dx, integer *incx)
{
    /* System generated locals */
    integer ret_val, i__1;
    doublereal d__1;
    /* Local variables */
    static doublereal dmax__;
    static integer i__, ix;
/*     finds the index of element having max. absolute value.
       jack dongarra, linpack, 3/11/78.
       modified 3/93 to return if incx .le. 0.
       modified 12/3/93, array(1) declarations changed to array(*)
       Parameter adjustments */
    --dx;
    /* Function Body */
    ret_val = 0;
    if (*n < 1 || *incx <= 0) {
  return ret_val;
    }
    ret_val = 1;
    if (*n == 1) {
  return ret_val;
    }
    if (*incx == 1) {
  goto L20;
    }
/*        code for increment not equal to 1 */
    ix = 1;
    dmax__ = abs(dx[1]);
    ix += *incx;
    i__1 = *n;
    for (i__ = 2; i__ <= i__1; ++i__) {
  if ((d__1 = dx[ix], abs(d__1)) <= dmax__) {
      goto L5;
  }
  ret_val = i__;
  dmax__ = (d__1 = dx[ix], abs(d__1));
L5:
  ix += *incx;
/* L10: */
    }
    return ret_val;
/*        code for increment equal to 1 */
L20:
    dmax__ = abs(dx[1]);
    i__1 = *n;
    for (i__ = 2; i__ <= i__1; ++i__) {
  if ((d__1 = dx[i__], abs(d__1)) <= dmax__) {
      goto L30;
  }
  ret_val = i__;
  dmax__ = (d__1 = dx[i__], abs(d__1));
L30:
  ;
    }
    return ret_val;
} /* idamax_ */

