// This file defines templates for transforming Modelica code to C# code.

package SimCodeCSharp

import interface SimCodeTV;

// SECTION: SIMULATION TARGET, ROOT TEMPLATE

template translateModel(SimCode simCode) ::=
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
    let()= textFile(simulationFile(simCode), '<%dotPath(modelInfo.name)%>.cs')
    //let()= textFile(simulationFunctionsFile(simCode), '<%modelInfo.name%>_functions.cs')
    "" // empty result of the top-level template .., only side effects
end translateModel;


// SECTION: SIMULATION TARGET, C# FILE SPECIFIC TEMPLATES


template simulationFile(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<<
// Simulation code for <%dotPath(modelInfo.name)%> generated by the OpenModelica Compiler.

using System;
using Bodylight.Solvers;
namespace Bodylight.Models<%modelNameSpace(modelInfo.name)%>
{
  public partial class <%lastIdentOfPath(modelInfo.name)%> : DAESystem
  {
         
    <%modelDataMembers(modelInfo, simCode)%>

    <%let fbody = simulationFunctionsBody(simCode)
      if fbody then 
        <<
        #region Functions
        <%fbody%>
        #endregion
        >>
      else "//** No functions **"
    %>
    
    <%functionDaeOutput(nonStateContEquations, removedEquations, simCode)%>

    <%functionDaeOutput2(nonStateDiscEquations, removedEquations, simCode)%>

    <%functionInput(modelInfo, simCode)%>

    <%functionOutput(modelInfo, simCode)%>

    <%functionZeroCrossing(zeroCrossings, simCode)%>

    <%functionHandleZeroCrossing(simCode)%>

    <%functionUpdateDependents(allEquations, helpVarInfo, simCode)%>
    
    <%functionUpdateDepend(allEquationsPlusWhen, simCode)%>

    <%functionOnlyZeroCrossing(zeroCrossings, simCode)%>

    <%functionStoreDelayed(simCode)%>
    
    <%functionWhen(simCode)%>

    <%functionOde(stateContEquations, simCode)%>

    <%functionInitial(initialEquations, simCode)%>

    <%functionInitialResidual(residualEquations, simCode)%>
    
    <%functionExtraResiduals(allEquations, simCode)%>
    
    <%functionBoundParameters(parameterEquations, simCode)%>

    <%functionCheckForDiscreteVarChanges(simCode)%>
    
  }
}
>>
end simulationFile;

template modelNameSpace(Path modelName) ::=
  match modelName
  case QUALIFIED(__) then '.<%name%><%modelNameSpace(path)%>'
  case FULLYQUALIFIED(__) then modelNameSpace(path)
end modelNameSpace;

template lastIdentOfPath(Path modelName) ::=
  match modelName
  case QUALIFIED(__) then lastIdentOfPath(path)
  case IDENT(__)     then name
  case FULLYQUALIFIED(__) then lastIdentOfPath(path)
end lastIdentOfPath;

template simulationFunctionsBody(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
    (functions |> fn => match fn  
          case FUNCTION(__)           then 
            <<
            <%functionBodyRegularFunction(fn, simCode)%>
            >>
          case EXTERNAL_FUNCTION(__)  then 'EXTERNAL_FUN_NOT_IMPLEMETED(name=<%dotPath(name)%>)'
          case RECORD_CONSTRUCTOR(__) then 'RECORD_CONSTRUCTOR_NOT_IMPLEMENTED(name=<%dotPath(name)%>)'
          else "UNKNOWN_FUNCTION"
    ;separator="\n")
end simulationFunctionsBody;

/*
//not used now
template simulationFunctionsFile(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<<
// Simulation functions code for <%modelInfo.name%> generated by the OpenModelica Compiler.

using System;
using Bodylight.Solvers;
namespace Bodylight.Models
{
  public partial class <%dotPath(modelInfo.name)%> : DAESystem
  {
    
    <%simulationFunctionsBody(simCode)%>
          
  }
}
>>
end simulationFunctionsFile;
*/

template recordDeclaration(RecordDeclaration recDecl, SimCode simCode)
 "Generates structs for a record declaration."
::=
  match recDecl
  case RECORD_DECL_FULL(__) then
    <<
    struct <%name%> {
      <%variables |> var as VARIABLE(__) => '<%varType(var)%> <%crefStr(var.name,simCode)%>;' ;separator="\n"%>
    };    
    >> 
  case RECORD_DECL_DEF(__) then "RECORD_DECL_DEF_NOT_SUPPORTED"
  else "UNKNOWN_RECORD_DECL_"
end recordDeclaration;



template functionBodyRegularFunction(Function fn, SimCode simCode)
 "Generates the body for a Modelica/MetaModelica function."
::=
match fn
case FUNCTION(__) then
  let()= System.tmpTickReset(1)
  let fname = underscorePath(name)
  let retType = match outVars 
                case fv :: _ then varType(fv) 
                else "void"
  let retVar = match outVars 
               case (fv as VARIABLE(__)) :: _ then crefStr(fv.name,simCode) 
  let varInits = variableDeclarations |> var => varInit(var, simCode) 
                 ;separator="\n" //TODO:special end of line,see varInit    
  let bodyPart = (body |> stmt  => funStatement(stmt, simCode) ;separator="\n")
  <<
  <%retType%> _<%fname%>(<%functionArguments |> var => funArgDefinition(var,simCode) ;separator=", "%>)
  {
    <%varInits%>

    <%bodyPart%>
    
    _return:
    <%if outVars then 'return <%retVar%>;'%>
  }
  
  >>
end functionBodyRegularFunction;

template varInit(Variable var, SimCode simCode)
 "Generates code to initialize variables."
::=
match var
case var as VARIABLE(__) then
  if instDims then
    let varName = crefStr(var.name,simCode)
    let &preDimsExp = buffer ""
    let instDimsInit = (var.instDims |> exp =>
          daeExp(exp, contextFunction, &preDimsExp, simCode)
        ;separator=", ")
    <<
    <%preDimsExp%>
    var <%varName%> = new <%expTypeArray(var.ty,listLength(instDims))%>(<%instDimsInit%>);<% 
      //special \n solution ... the new line is dependent on this line, but the varInit as a whole does not put \n
      match var.value  case SOME(CREF(componentRef = cr)) then 
      '<%\n%><%varName%>.CopyFrom(<%crefStr(cr,simCode)%>);'
    %>  
    >>
  else
    <<
    <%varType(var)%> <%crefStr(var.name,simCode)%>;
    >>
else "UNSUPPORTED_VARIABLE_vatInit"

end varInit;

template funArgDefinition(Variable var, SimCode simCode)
::=
  match var
  case VARIABLE(__) then '<%varType(var)%> <%crefStr(name,simCode)%>'
  case FUNCTION_PTR(__) then 'funArgDefinition_UNSUPPORTED_fnptr <%name%>'
  else "UNSUPPORTED_VARIABLE_funArgDefinition"
end funArgDefinition;

template funStatement(Statement stmt, SimCode simCode)
 "Generates function statements."
::=
  match stmt
  case ALGORITHM(__) then
    (statementLst |> stmt =>
      algStatement(stmt, contextFunction, simCode)
    ;separator="\n") 
  else
    "NOT IMPLEMENTED FUN STATEMENT"
end funStatement;

constant String localRepresentationArrayDefines =
"var X = states; var Xd = statesDerivatives; var Y = algebraics; var P = parameters; var H = helpVars;
var preX = savedStates; var preXd = savedStatesDerivatives; var preY = savedAlgebraics; var preH = savedHelpVars;
var preYI = savedAlgebraicsInt; var preYB = savedAlgebraicsBool; var YI = algebraicsInt; var YB = algebraicsBool; var PI = parametersInt; var PB = parametersBool; ";
       

template modelDataMembers(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
const int 
  NHELP = <%varInfo.numHelpVars%>, NG = <%varInfo.numZeroCrossings%>,
  NX = <%varInfo.numStateVars%>, NY = <%varInfo.numAlgVars%>, NP = <%varInfo.numParams%>,
  NYI = <%varInfo.numIntAlgVars%>, NYB = <%varInfo.numBoolAlgVars%>,
  NPI = <%varInfo.numIntParams%>, NPB = <%varInfo.numBoolParams%>,
  NO = <%varInfo.numOutVars%>, NI = <%varInfo.numInVars%>, NR = <%varInfo.numResiduals%>,
  NEXT = <%varInfo.numExternalObjects%>, NYSTR = <%varInfo.numStringAlgVars%>, NPSTR = <%varInfo.numStringParamVars%>;

public override string ModelName        { get { return "<%dotPath(name)%>"; }}
public override int HelpVarsCount       { get { return NHELP; } }
public override int ZeroCrossingsCount  { get { return NG; } }
public override int StatesCount         { get { return NX; } }
public override int AlgebraicsCount     { get { return NY; } }
public override int AlgebraicsIntCount  { get { return NYI; } }
public override int AlgebraicsBoolCount { get { return NYB; } }
public override int ParametersCount     { get { return NP; } }
public override int ParametersIntCount  { get { return NPI; } }
public override int ParametersBoolCount { get { return NPB; } }
        
public override int OutputsCount   { get { return NO; } }
public override int InputsCount    { get { return NI; } }
public override int ResidualsCount { get { return NR; } }
//public int ExternalObjectsCount { get { return NEXT; } }        
public override int MaximumOrder { get { return 5; } }
public override int StringVarsCount { get { return NYSTR; } }
public override int StringParametersCount { get { return NPSTR; } }

#region VariableInfos
public static readonly SimVarInfo[] VariableInfosStatic = new[] {
	<%{  
		varInfos("State", vars.stateVars, false, simCode),
		varInfos("StateDer", vars.derivativeVars, false, simCode),
		varInfos("Algebraic", vars.algVars, false, simCode),
		varInfos("AlgebraicInt", vars.intAlgVars, false, simCode),
		varInfos("AlgebraicBool", vars.boolAlgVars, false, simCode),
		varInfos("Parameter", vars.paramVars, true, simCode),
		varInfos("ParameterInt", vars.intParamVars, true, simCode),
		varInfos("ParameterBool", vars.boolParamVars, true, simCode)
	} ;separator=",\n\n"%>
};
public override SimVarInfo[] VariableInfos { get { return VariableInfosStatic; } }
#endregion

#region InitialFixed
private static readonly bool[] InitialFixedStatic = new bool[NX + NX + NY + NYI +NYB + NP + NPI + NPB] {
    <%{ (if vars.stateVars      then "//states\n" + initFixed(vars.stateVars, simCode)),
        (if vars.derivativeVars then "//derivatives\n" + initFixed(vars.derivativeVars, simCode)),
        (if vars.algVars        then "//algebraics\n" + initFixed(vars.algVars, simCode)), 
        (if vars.intAlgVars     then "//algebraicsInt\n" + initFixed(vars.intAlgVars, simCode)), 
        (if vars.boolAlgVars    then "//algebraicsBool\n" + initFixed(vars.boolAlgVars, simCode)), 
        (if vars.paramVars      then "//parameters\n" + initFixed(vars.paramVars, simCode)),
        (if vars.intParamVars   then "//parametersInt\n" + initFixed(vars.intParamVars, simCode)),
        (if vars.boolParamVars  then "//parametersBool\n" + initFixed(vars.boolParamVars, simCode))
      } ;separator=",\n"%>
};
public override bool[] InitialFixed { get { return InitialFixedStatic; } }
#endregion

#region Constructor - Init Data
public <%lastIdentOfPath(name)%>() {
    CreateData();
    
    //**** states *****
	<%initVals("states", vars.stateVars, simCode)%>
	  	
	//**** state derivatives *****
	//all are default values ... 0.0
	<%/* TODO: is it correct to have derivatives without initial values at all ?? 
	  initVals("statesDerivatives", vars.derivativeVars, simCode)
	*/%>
	
	//**** algebraics *****  
	<%initVals("algebraics", vars.algVars, simCode)%>
	
	//**** algebraics Int *****  
	<%initVals("algebraicsInt", vars.intAlgVars, simCode)%>
	
	//**** algebraics Bool *****  
	<%initVals("algebraicsBool", vars.boolAlgVars, simCode)%>
	
	//**** parameters *****  
	<%initVals("parameters", vars.paramVars, simCode)%>
	
	//**** parameters Int *****  
	<%initVals("parametersInt", vars.intParamVars, simCode)%>
	
	//**** parameters Bool *****  
	<%initVals("parametersBool", vars.boolParamVars, simCode)%>
}
#endregion
>>
end modelDataMembers;

template varInfos(String typeName, list<SimVar> varsLst, Boolean isMInd, SimCode simCode) ::=
  if varsLst then 
    <<
    #region <%typeName%> variable infos
    <% varsLst |> SIMVAR(__) => 
       <<
	   new SimVarInfo( "<%crefStr(name, simCode)%>", "<%comment%>", SimVarType.<%typeName%>, <%index%>, <%isMInd%>)
	   >> ;separator=",\n"
	%>
    #endregion<%\n%>    
    >>
end varInfos;

template initVals(String arrName, list<SimVar> varsLst, SimCode simCode) ::=
  varsLst |> sv as SIMVAR(__) =>
	match initialValue 
  	case SOME(v) then 
      let &preExp = buffer "" //dummy ... the value is always a constant
      match daeExp(v, contextOther, &preExp, simCode) 
      case vStr as "0"
      case vStr as "0.0"
      case vStr as "(0)" then
        '//<%arrName%>[<%sv.index%>] = <%vStr%>; //<%crefStr(sv.name, simCode)%> --> zero val' 
      case vStr then
       '<%arrName%>[<%sv.index%>] = <%vStr%>; //<%crefStr(sv.name, simCode)%>'
  	  end match
  	else '//<%arrName%>[<%sv.index%>] = 0.0; //<%crefStr(sv.name, simCode)%> --> default val'	
  ;separator="\n"
end initVals;

template initFixed(list<SimVar> simVarLst, SimCode simCode) ::=
  (simVarLst |> SIMVAR(__) => '<%isFixed%> /* <%crefStr(name, simCode)%> */' ;separator=",\n")
end initFixed;

template functionDaeOutput(list<SimEqSystem> nonStateContEquations,
                  list<SimEqSystem> removedEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
/* for continuous time variables */
public override void FunDAEOutput()
{
  <% localRepresentationArrayDefines %>
  <% nonStateContEquations |> it => equation_(it,contextSimulationNonDiscrete, simCode) ;separator="\n"%>
  <% removedEquations      |> it => equation_(it,contextSimulationNonDiscrete, simCode) ;separator="\n"%>
}
>>
end functionDaeOutput;

template functionDaeOutput2(list<SimEqSystem> nonStateDiscEquations,
                   list<SimEqSystem> removedEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
/* for discrete time variables */
public override void FunDAEOutput2()
{
  <% localRepresentationArrayDefines %>
  <% nonStateDiscEquations |> it => equation_(it,contextSimulationDiscrete, simCode) ;separator="\n"%>
  <% removedEquations      |> it => equation_(it,contextSimulationDiscrete, simCode) ;separator="\n"%>
}
>>
end functionDaeOutput2;

template functionInput(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
public override void InputFun()
{
  <% localRepresentationArrayDefines %>
  <%vars.inputVars |> SIMVAR(__) hasindex i0 => 
  <<
  <%cref(name, simCode)%> = inputVars[<%i0%>];
  >> ;separator="\n"%>
}
>>
end functionInput;

template functionOutput(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
public override void OutputFun()
{
  <% localRepresentationArrayDefines %>
  // * not yet
  <%vars.outputVars |> SIMVAR(__) hasindex i0 =>
  <<
  //outputVars[<%i0%>] = <%cref(name, simCode)%>;
  >> ;separator="\n"%>  
}
>>
end functionOutput;

template functionZeroCrossing(list<ZeroCrossing> zeroCrossingLst, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunZeroCrossing(double time, double[] x, double[] xd, double[] gout)
{
  <% localRepresentationArrayDefines %>
  var timeBackup = this.time;
  this.time = time;

  FunODE();
  FunDAEOutput();
  
  <%zeroCrossingLst |> ZERO_CROSSING(__) hasindex i0 => zeroCrossing(relation_, i0, simCode) ;separator="\n"%>  

  this.time = timeBackup;
}
>>
end functionZeroCrossing;


// This function should only save in cases. The rest is done in
// function_updateDependents.
template functionHandleZeroCrossing(SimCode simCode) ::=
match simCode case SIMCODE(__) then
<<
public override void FunHandleZeroCrossing(int index)
{  
  <% localRepresentationArrayDefines %>
  switch(index) {
    <% zeroCrossingsNeedSave |> toSaveLst hasindex i0 =>
    <<
    case <%i0%>:
      <% toSaveLst |> SIMVAR(__) =>
         '<%preCref(name, simCode)%> = <%cref(name, simCode)%>; //save()'
      ;separator="\n" %>
      break;
    >> ;separator="\n"%>
    default:
       break;
  }
}
>>
end functionHandleZeroCrossing;

template functionUpdateDependents(list<SimEqSystem> allEquations, list<HelpVarInfo> helpVarInfoLst, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunUpdateDependents()
{
  <% localRepresentationArrayDefines %>
  //inUpdate=initial()?0:1;
  isInUpdate = ! isInit;

  <%allEquations |> it => equation_(it, contextSimulationDiscrete, simCode) ;separator="\n"%>
  <%helpVarInfoLst |> (in1, exp, _)  =>
      let &preExp = buffer ""
      let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, simCode)
      <<
      <%preExp%>
      helpVars[<%in1%>] = <%expPart%> ? 1.0 : 0.0;<%/*???TODO: ? 1.0 : 0.0;*/%>
      >>
  ;separator="\n"%>

  isInUpdate = false;
}
>>
end functionUpdateDependents;

// All when equations should go in here too according to Willi
// And something about if-eqs being sorted ans not just added to end
template functionUpdateDepend(list<SimEqSystem> allEquationsPlusWhen, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunUpdateDepend()
{
  <% localRepresentationArrayDefines %>
  isInUpdate = ! isInit;
  
  <%allEquationsPlusWhen |> it => equation_(it, contextSimulationDiscrete, simCode) ;separator="\n"%>  

  isInUpdate = false;
}
>>
end functionUpdateDepend;

template functionOnlyZeroCrossing(list<ZeroCrossing> zeroCrossingLst, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunOnlyZeroCrossings(double time, double[] gout) //TODO:??time in original is *t only ... how is it called?
{
  <% localRepresentationArrayDefines %>
  <%zeroCrossingLst |> ZERO_CROSSING(__) hasindex i0 => zeroCrossing(relation_, i0, simCode) ;separator="\n"%>  
}
>>
end functionOnlyZeroCrossing;

template zeroCrossing(Exp it, Integer index, SimCode simCode) ::=
  match it
  case RELATION(__) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextOther, &preExp, simCode)
    let e2 = daeExp(exp2, contextOther, &preExp, simCode)
    <<
    {<%preExp%>var _zen = zeroCrossingEnabled[<%index%>]; //ZEROCROSSING(<%index%>, <%zeroCrossingOpFunc(operator)%>(<%e1%>, <%e2%>));
    gout[<%index%>] = (_zen != 0) ? _zen * (<%match operator
                                           case LESS(__)
                                           case LESSEQ(__)    then '<%e1%>-<%e2%>'
                                           case GREATER(__)
                                           case GREATEREQ(__) then '<%e2%>-<%e1%>'
                                           case EQUAL(__) then '<%e2%> == <%e1%>'
                                           case NEQUAL(__) then '<%e2%> != <%e1%>'    			
                                           else "!!!unsupported ZC operator!!!"
                                          %>) : 1.0; }
    >>    
  case CALL(path=IDENT(name="sample"), expLst={start, interval}) then
    let &preExp = buffer ""
    let eStart = daeExp(start, contextOther, &preExp, simCode)
    let eInterval = daeExp(interval, contextOther, &preExp, simCode)
    <<
    {<%preExp%>var _zen = zeroCrossingEnabled[<%index%>]; //ZEROCROSSING(<%index%>, Sample(*t, <%eStart%>, <%eInterval%>));
    gout[<%index%>] = (_zen != 0) ? _zen * Sample(time, <%eStart%>, <%eInterval%>) : 1.0; }
    >> 
  case _ then
    <<
    ZERO_CROSSING_ERROR
    >>
end zeroCrossing;

template zeroCrossingOpFunc(Operator it) ::=
  match it
  case LESS(__)      then "Less"
  case GREATER(__)   then "Greater"
  case LESSEQ(__)    then "LessEq"
  case GREATEREQ(__) then "GreaterEq"
  case EQUAL(__)     then "!!!Equal"
  case NEQUAL(__)    then "!!!NEqual"
  else "!!!unsupported ZC operator!!!"
  
end zeroCrossingOpFunc;

// New runtime function. What should it do?
template functionStoreDelayed(SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunStoreDelayed()
{ 
  <% localRepresentationArrayDefines %> 
}
>>
end functionStoreDelayed;

template functionWhen(SimCode simCode) ::=
let()= System.tmpTickReset(1)
match simCode case SIMCODE(__) then
<<
public override void FunWhen(int i)
{
  <% localRepresentationArrayDefines %>
  switch(i) {
    <%whenClauses |> SIM_WHEN_CLAUSE(__) hasindex i0 =>
    <<
    case <%i0%>:
      <%functionWhen_caseEquation(whenEq, simCode)%>
      <%reinits |> REINIT(__)  =>
        let &preExp = buffer ""
        let valueExp = daeExp(value, contextSimulationDiscrete, &preExp, simCode)
      <<
      <%preExp%>
      <%cref(stateVar, simCode)%> = <%valueExp%>;
      >> ;separator="\n"%>
      break;
    >> ;separator="\n"%>
    default:
      break;
  }
}
>>
end functionWhen;

template functionWhen_caseEquation(Option<WhenEquation> it, SimCode simCode) ::=
match it
case SOME(weq as WHEN_EQ(__)) then
let &preExp = buffer ""
let expPart = daeExp(weq.right, contextSimulationDiscrete, &preExp, simCode)
<<
<%preCref(weq.left, simCode)%> = <%cref(weq.left, simCode)%>; //save()
<%preExp%>
<%cref(weq.left, simCode)%> = <%expPart%>;
>>
end functionWhen_caseEquation;

template functionOde(list<SimEqSystem> stateContEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunODE()
{
  <% localRepresentationArrayDefines %>
  <%stateContEquations |> it => equation_(it, contextOther, simCode) ;separator="\n"%>
}
>>
end functionOde;

//TODO:??there is st more in the trunk
template functionInitial(list<SimEqSystem> initialEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void InitialFun()
{
  <% localRepresentationArrayDefines %>
  <%initialEquations |> saeq as SES_SIMPLE_ASSIGN(__) => equation_(saeq, contextOther, simCode) ;separator="\n"%>

  //if (sim_verbose) {
    <%initialEquations |> SES_SIMPLE_ASSIGN(__) =>
    <<
    //Debug.WriteLine("Setting variable start value: {0}(start={1})", "<%cref(cref, simCode)%>", <%cref(cref, simCode)%>);
    >> ;separator="\n"%>
  //}
}
>>
end functionInitial;

template functionInitialResidual(list<SimEqSystem> residualEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void InitialResidual()
{
  <% localRepresentationArrayDefines %>
  int _i = 0;

  <%residualEquations |> SES_RESIDUAL(__)  =>
    match exp case DAE.SCONST(__) then
      'initialResiduals[_i++] = 0;'
    else
      let &preExp = buffer ""
      let expPart = daeExp(exp, contextSimulationNonDiscrete, &preExp, simCode) // ??contextOther
      <<
      <%preExp%>
      initialResiduals[_i++] = <%expPart%>;
      >>
  ;separator="\n"%>
}
>>
end functionInitialResidual;

template functionExtraResiduals(list<SimEqSystem> allEquations, SimCode simCode) ::=
  let()= System.tmpTickReset(1)
  (allEquations |> SES_NONLINEAR(__) =>
	<<
	int ResidualFun<%index%>(int n, double[] xloc, double[] res, int iflag)
	{
	   <% localRepresentationArrayDefines %>
	   <%eqs |> saeq as SES_SIMPLE_ASSIGN(__) =>
         equation_(saeq, contextOther, simCode)
         ;separator="\n"
       %>
	   <%eqs |> SES_RESIDUAL(__) hasindex i0 =>
	     let &preExp = buffer ""
	     let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, simCode)
	     <<
	     <%preExp%>
	     res[<%i0%>] = <%expPart%>;
	     >> ;separator="\n"
	   %>
	   return 0;
	}
	>> ;separator="\n")
end functionExtraResiduals;

template functionBoundParameters(list<SimEqSystem> parameterEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void BoundParameters()
{
  <% localRepresentationArrayDefines %>
  <% parameterEquations
     |> saeq as SES_SIMPLE_ASSIGN(__)  => equation_(saeq, contextOther, simCode) ;separator="\n"%>
}
>>
end functionBoundParameters;

// TODO: Is the -1 thing really correct? It seems to work.
template functionCheckForDiscreteVarChanges(SimCode simCode) ::=
match simCode case SIMCODE(__) then
<<
public override bool CheckForDiscreteVarChanges()
{
  <% localRepresentationArrayDefines %>
  //var needToIterate = false;

  //edge(H[i])
  <% helpVarInfo |> (id1, exp, id2) => match id2 case -1 then ""
     else
       'if (H[<%id1%>]!=0.0 && preH[<%id1%>]==0.0) EventQueue.Add(<%id2%> + NG);'
    ;separator="\n" %>

  //TODO: changeDiscreteVar(i) and to get the i from ComponentRef
  //if change()
  <%discreteModelVars |> it =>
  <<
  if (<%preCref(it, simCode)%> != <%cref(it, simCode)%>) return true; /*needToIterate = true; */
  >> ;separator="\n"%>
  
  for (int i = 0; i < H.Length; i++) {
    //change(H[i]) ?? TODO: not sure if it can be only 1.0 or 0.0
    if (H[i] != preH[i])
      return true; //needToIterate=true;
  }

  return false; //needToIterate;
}
>>
end functionCheckForDiscreteVarChanges;

template daeExpToReal(Exp exp, Context context, Text &preExp, SimCode simCode) ::=
	daeExp(exp, context, &preExp, simCode)
	+ (match expTypeFromExp(exp) case "bool" then " ?1.0:0.0") //TODO: a HACK ?
	  
end daeExpToReal;

// Residual equations are not handled here
template equation_(SimEqSystem eq, Context context, SimCode simCode) ::=
match eq
case SES_SIMPLE_ASSIGN(__) then
  let &preExp = buffer ""
  let expPart = daeExpToReal(exp, context, &preExp, simCode)
  <<
  <%preExp%>
  <%cref(cref, simCode)%> = <%expPart%>;
  >>
case SES_ARRAY_CALL_ASSIGN(__) then "SES_ARRAY_CALL_ASSIGN"
case SES_ALGORITHM(__) then "SES_ALGORITHM"

case SES_LINEAR(__) then
  let uid = System.tmpTick()
  let size = listLength(vars)
  let aname = 'A<%uid%>'
  let bname = 'b<%uid%>'
  <<
  var <%aname%> = new double[<%size%>*<%size%>]; //declare_matrix(<%aname%>, <%size%>, <%size%>);
  var <%bname%> = new double[<%size%>]; //declare_vector(<%bname%>, <%size%>);
  <%simJac |> (row, col, eq as SES_RESIDUAL(__)) =>
     let &preExp = buffer ""
     let expPart = daeExpToReal(eq.exp, context, &preExp, simCode)
     '<%preExp%><%aname%>[<%row%>+<%col%>*<%size%>] = <%expPart%>; //set_matrix_elt(<%aname%>, <%row%>, <%col%>, <%size%>, <%expPart%>);'
  ;separator="\n"%>
  <%beqs|> it hasindex i0 =>
     let &preExp = buffer ""
     let expPart = daeExpToReal(it, context, &preExp, simCode)
     '<%preExp%><%bname%>[<%i0%>] = <%expPart%>; //set_vector_elt(<%bname%>, <%i0%>, <%expPart%>);'
  ;separator="\n"%>
  <%if partOfMixed then 
      'if(SolveLinearSystemMixed(<%aname%>, <%bname%>, <%size%>, <%uid%>) != 0) found_solution = -1;'
    else 
      'SolveLinearSystem(<%aname%>, <%bname%>, <%size%>, <%uid%>);'
  %>
  <%vars |> SIMVAR(__) hasindex i0 => '<%cref(name, simCode)%> = <%bname%>[<%i0%>]; //get_vector_elt(<%bname%>, <%i0%>);' ;separator="\n"%>
  >>
case SES_MIXED(__) then
  let uid = System.tmpTick()
  let contEqs = equation_(cont, context, simCode)
  let numDiscVarsStr = listLength(discVars) 
  let valuesLenStr = listLength(values)
  let &preDisc = buffer "" /*BUFD*/
  let discLoc2 = 
    discEqs |> discEq as SES_SIMPLE_ASSIGN(__) hasindex i0 =>
        <<
        double discrete_loc2_<%i0%> = <%daeExpToReal(discEq.exp, context, &preDisc, simCode)%>;
        <%cref(discEq.cref, simCode)%> = discrete_loc2_<%i0%>;
        >>
        ;separator="\n"
  	/*
  	match discEqs
    case { discEq as SES_SIMPLE_ASSIGN(__) } then
      <<
      <%cref(discEq.cref, simCode)%> = <%daeExpToReal(discEq.exp, context, &preDisc, simCode)%>;
      double discrete_loc2_0 = <%cref(discEq.cref, simCode)%>;
      >>
  	case discEqs then
      <<
      var discrete_loc2 = new double[<%numDiscVarsStr%>];
      <%discEqs |> discEq as SES_SIMPLE_ASSIGN(__) hasindex i0 =>
        <<
        <%cref(discEq.cref, simCode)%> = <%daeExpToReal(exp, context, &preDisc, simCode)%>;
        discrete_loc2[<%i0%>] = <%cref(discEq.cref, simCode)%>;
        >>
        ;separator="\n"%>
      >>
      */
  <<
  // *** mixed_equation_system(<%numDiscVarsStr%>) ***
  { int found_solution = 0;
    int cur_value_indx = -1;
    var values = new double[]{<%values ;separator=", "%>};
    do {
	  <%
	    discVars |> SIMVAR(__) hasindex i0 => 
	      'double discrete_loc_<%i0%> = <%cref(name, simCode)%>;'
	    ;separator=",\n"
	   
	   /*match discVars
	    case { var as SIMVAR(__) } then
	      <<
	      double discrete_loc_0 = <%cref(var.name, simCode)%>;
          >>
	    case discVars then
	      <<
	      var discrete_loc = new double[<%numDiscVarsStr%>] {
	        <%discVars |> SIMVAR(__) => cref(name, simCode) ;separator=",\n"%>
	      };
	      >> */
	  %>
	  {
	    <%contEqs%>
	  }
	  <%preDisc%>
	  <%discLoc2%>
	  {
	    // check_discrete_values(<%numDiscVarsStr%>, <%valuesLenStr%>);
	    if (found_solution == -1) { /*system of equations failed*/
            found_solution = 0;
        } else {
            found_solution = 1;
            <% 
              discVars |> SIMVAR(__) hasindex i0 =>
                <<
                if ( Math.Abs(discrete_loc_<%i0%> - discrete_loc2_<%i0%>) > 1e-12) found_solution = 0;                
                >> ;separator="\nelse "
              /*match discVars
	    	  case { SIMVAR(__) } then
	    	    <<
	    	    if ( Math.Abs(discrete_loc_0 - discrete_loc2_0) > 1e-12) {
                    found_solution = 0;
                }
                >>
              case discVars then
                <<
                for (int i=0; i < <%numDiscVarsStr%>; i++) {
	                if ( Math.Abs(discrete_loc[i] - discrete_loc2[i]) > 1e-12) {
	                    found_solution = 0;
	                }
	            }
	            >>*/
	        %>
        }
        if (found_solution == 0) { //!found_solution
            cur_value_indx++;
            if (cur_value_indx >= <%valuesLenStr%>/<%numDiscVarsStr%>) {
                found_solution = -1; //?? failed ??
                System.Diagnostics.Debug.WriteLine("Mixed system id=" + <%uid%> + "failed.");
            } else {
              var curValOffset = cur_value_indx*<%numDiscVarsStr%>;
              <% discVars |> SIMVAR(__) hasindex i0 =>
                  '<%cref(name, simCode)%> = values[curValOffset+<%i0%>];'
                 ;separator="\n"
              %>
            }
        }             
      }
    } while (found_solution == 0);
  } // *** mixed_equation_system_end(<%numDiscVarsStr%>) ***
  >>

case SES_NONLINEAR(__) then 
  let size = listLength(crefs)
  <<
  //start_nonlinear_system(<%size%>);
  { var oldX = oldStates; var oldXd = oldStatesDerivatives;  var old2X = oldStates2; var old2Xd = oldStatesDerivatives2;
    var oldY = oldAlgebraics; var old2Y = oldAlgebraics2; var oldTimeDelta = oldTime - oldTime2; 
    var nls_x = new double[<%size%>]; var nls_xold = new double[<%size%>];
    <%crefs |> name hasindex i0 =>
      <<
      nls_x[<%i0%>] = /*extraPolate(<%crefStr(name, simCode)%>)*/ oldTimeDelta == 0.0 ? <%cref(name, simCode)%> : (time * (<%oldCref(name, simCode)%> - <%old2Cref(name, simCode)%>) + (oldTime * <%old2Cref(name, simCode)%> - oldTime2 * <%oldCref(name, simCode)%>)) / oldTimeDelta;          
      nls_xold[<%i0%>] = <%oldCref(name, simCode)%>;
      >> ;separator="\n"
    %>
    SolveNonlinearSystem(ResidualFun<%index%>, nls_x, nls_xold, <%index%>);
    <%crefs |> name hasindex i0 => 
    '<%cref(name, simCode)%> = nls_x[<%i0%>];' ;separator="\n"
    %>
  } //end_nonlinear_system();
  >>
  
case SES_WHEN(__) then
  let &preExp = buffer ""
  let helpIf = (conditions |> (e, hidx) =>
      let hInit = daeExp(e, context, &preExp, simCode)
      let &preExp += 'H[<%hidx%>] = <%hInit%> ? 1.0 : 0.0;' //TODO: ??? type
      '/*edge(H[<%hidx%>])*/(H[<%hidx%>]!=0.0 && preH[<%hidx%>]==0.0)'
    ;separator=" || ")
  let &preExp2 = buffer ""
  let rightExp = daeExp(right, context, &preExp2, simCode)
  <<
  <%preExp%>
  if (<%helpIf%>) {
    <%preExp2%>
    <%cref(left, simCode)%> = <%rightExp%>;
  } else {
    <%cref(left, simCode)%> = <%preCref(left, simCode)%>;
  }
  >>

else  
  "UNKNOWN_equation"
  
end equation_;

// SECTION: SIMULATION TARGET, FUNCTIONS FILE SPECIFIC TEMPLATES
// not yet implemented

// SECTION: GENERAL TEMPLATES, COMPONENT REFERENCES

template cref(ComponentRef cr, SimCode simCode) ::=
  match cr
  case CREF_IDENT(ident = "xloc") then crefStr(cr, simCode) //TODO: ??xloc
  case CREF_IDENT(ident = "time") then "time"
  else '/*<% crefStr(cr, simCode) %>*/<% representationCref(cr, simCode) %>'
  // not needed ... //the space at the start is important to avoid ambiguity with "/" operator to become "//" line comment
/*
  match cr
  //TODO: ?? case CREF_IDENT(ident = "xloc") then crefStr(cr)
  case CREF_IDENT(__) then replaceDollarWorkaround(ident)
  case _ then "CREF_NOT_IDENT"
*/
end cref;

template representationCref(ComponentRef inCref, SimCode simCode) ::=
  cref2simvar(inCref, simCode) |> SIMVAR(__) =>
	'<%representationArrayName(varKind, type_)%>[<% index %>]'
end representationCref;

template representationArrayName(VarKind varKind, ExpType type_) ::=
  match varKind 
  case VARIABLE(__)    then "Y" + representationArrayNameTypePostfix(type_)
  case STATE(__)       then "X"
  case STATE_DER(__)   then "Xd"
  case DUMMY_DER(__)   then "Y" // => algebraics
  case DUMMY_STATE(__) then "Y" // => algebraics
  case DISCRETE(__)    then 'Y<%representationArrayNameTypePostfix(type_)%>/*d*/'
  case PARAM(__)       then "P" + representationArrayNameTypePostfix(type_)
  case CONST then "CONST_VAR_KIND"
  else "BAD_VARKIND"
end representationArrayName;

template representationArrayNameTypePostfix(ExpType type_) ::=
  match type_ 
  case ET_INT(__)  then "I"
  case ET_BOOL(__) then "B"
  case ET_REAL(__) then ""
  else "BAD_ARRAY_NAME_POSTFIX"
end representationArrayNameTypePostfix;

template preCref(ComponentRef cr, SimCode simCode) ::=
'/*pre(<%crefStr(cr, simCode)%>)*/pre<%representationCref(cr, simCode)%>'
end preCref;

template derCref(ComponentRef cr, SimCode simCode) ::=
'/*derCall!!(<% crefStr(cr, simCode) %>)*/<%representationCref(derComponentRef(cr), simCode)%>'
end derCref;

template oldCref(ComponentRef cr, SimCode simCode) ::=
'/*old(<%crefStr(cr, simCode)%>)*/old<%representationCref(cr, simCode)%>'
end oldCref;

template old2Cref(ComponentRef cr, SimCode simCode) ::=
'/*old2(<%crefStr(cr, simCode)%>)*/old2<%representationCref(cr, simCode)%>'
end old2Cref;


template contextCref(ComponentRef cr, Context context, SimCode simCode)
  "Generates code for a component reference depending on which context we're in."
::=
  match context
  case FUNCTION_CONTEXT(__) then crefStr(cr, simCode)
  else cref(cr, simCode)
end contextCref;

template crefStr(ComponentRef cr, SimCode simCode)
 "Generates the name of a variable for variable name array."
::=
  match cr
  case CREF_IDENT(__) then ident + subscriptsStr(subscriptLst, simCode)
  case CREF_QUAL(ident = "$DER") then 'der(<%crefStr(componentRef, simCode)%>)'
  case CREF_QUAL(__) then '<%ident%><%subscriptsStr(subscriptLst, simCode)%>.<%crefStr(componentRef, simCode)%>'
  else "CREF_NOT_IDENT_OR_QUAL"
end crefStr;

template subscriptsStr(list<Subscript> subscripts, SimCode simCode)
 "Generares subscript part of the name."
::=
  if subscripts then
    '[<% subscripts |> s => match s
           case INDEX(__) 
           case SLICE(__) then
              let &preExp = buffer "" //dummy ??
              daeExp(exp, contextFunction, &preExp, simCode)
           case WHOLEDIM(__) then "WHOLEDIM"
           else "UNKNOWN_SUBSCRIPT"
      ;separator="," %>]'
end subscriptsStr;


/*
template crefSubscript(ComponentRef it, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then replaceDollarWorkaround(ident) + subscripts(subscriptLst)
  case _ then "CREF_NOT_IDENT"
end crefSubscript;

template subscripts(list<Subscript> subscriptsLst) ::=
  if subscriptsLst then '[<%subscriptsLst |> it => subscript(it) ;separator=","%>]'
end subscripts;

template subscript(Subscript it) ::=
  match it
  case INDEX(exp = ICONST(__)) then exp.integer
  case _ then "SUBSCRIPT_NOT_CONSTANT"
end subscript;

*/


// SECTION: GENERAL TEMPLATES, PATHS

template dotPath(Path it) ::=
  match it
  case QUALIFIED(__)      then '<%name%>.<%dotPath(path)%>'
  case IDENT(__)          then name
  case FULLYQUALIFIED(__) then dotPath(path)
end dotPath;

//TODO: not 100% back-convertible e.g. Awkward_.Model vs. Awkward._Model, both -> Awkward___Model 
template underscorePath(Path it) ::=
  match it
  case QUALIFIED(__)      then '<%System.stringReplace(name, "_", "__")%>_<%underscorePath(path)%>'
  case IDENT(__)          then System.stringReplace(name, "_", "__")
  case FULLYQUALIFIED(__) then underscorePath(path)
end underscorePath;


// SECTION: GENERAL TEMPLATES, FUNCTION GENERATION
// not yet implemented



// Codegen.generateStatement
template algStatement(DAE.Statement it, Context context, SimCode simCode) ::=
  match it
  case STMT_ASSIGN(exp1 = CREF(componentRef = WILD(__)), exp = e) then
    let &preExp = buffer "" 
    let expPart = daeExp(e, context, &preExp, simCode)
    <<
    <%preExp%>
    <%expPart%>
    >>    
  case STMT_ASSIGN(exp1 = CREF(__)) then
    let &preExp = buffer ""
    let expPart = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    <%scalarLhsCref(exp1, context, &preExp, simCode)%> = <%expPart%>;
    >>
  case STMT_ASSIGN(__) then
    let &preExp = buffer ""
    let expPart1 = daeExp(exp1, context, &preExp, simCode)
    let expPart2 = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    <%expPart1%> = <%expPart2%>;
    >>
  //works only for array name as IDENT with subscripts 
  case STMT_ASSIGN_ARR(componentRef = CREF_IDENT(subscriptLst=subs as (_ :: _))) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp, simCode)
     let spec = daeExpCrefRhsIndexSpec(subs, context, &preExp, simCode)
     <<
     <%preExp%>
     <%componentRef.ident%>.AssignSpec(<%spec%>, <%expPart%>.A);
     >>
  case STMT_ASSIGN_ARR(__) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp, simCode)
     <<
     <%preExp%>
     ArrayCopy(<%expPart%>.A, <%contextCref(componentRef, context, simCode)%>.A);
     >>     
  case STMT_IF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    <%elseExpr(else_, context, simCode)%>
    >>
  case STMT_FOR(range=rng as RANGE(__)) then
    let identType = expType(type_, iterIsArray) //TODO: ?? what is this for ... no array typed iterator is possible ???
    let identTypeShort = expTypeShort(type_)
    let stmtStr = (statementLst |> stmt => algStatement(stmt, context, simCode)
                   ;separator="\n")
    algStmtForRange_impl(rng, iter, identType, identTypeShort, stmtStr, context, simCode)
  
  case s as STMT_FOR(__) then "algStmtForGeneric_NOT_IMPLEMENTED"
    //  algStmtForGeneric(s, context, &varDecls /*BUFC*/)
  
  case STMT_WHILE(__)  then
    let &preExp = buffer ""
    let var = daeExp(exp, context, &preExp, simCode)
    <<
    for(;;) {
      <%preExp%>
      if (!<%var%>) break;
      <%statementLst |> stmt => algStatement(stmt, context, simCode) ;separator="\n"%>
    }
    >>
  
  case _ then "NOT_IMPLEMENTED_ALG_STATEMENT"
 /*
  case STMT_FOR(range = rng as RANGE(__)) then
    <<
    {
      <%expTypeA(type_, boolean)%> 
        _r1 = <%daeExp(rng.exp,isSimulationCode)%>,
        _r2 = <%match rng.expOption case SOME(eo) then daeExp(eo,isSimulationCode) else "(1)"%>,
        _r3 = <%daeExp(rng.range,isSimulationCode)%>,
        <%iter%>;

      for (<%iter%> = _r1; in_range_<%expTypeShort(type_)%>(<%iter%>, _r1, _r3); <%iter%> += _r2) {
        <%statementLst |> it => algStatement(it) ;separator="\n" /* ??CONTEXT(codeContext,expContext,IN_FOR_LOOP(loopContext)*/ %>        
      }
    } /*end for*/
    >>
  */
end algStatement;

//TODO: in_range_integer is not correct against Modelica specification when e.g. 10:1
//TODO: optimize representation for range, so that template can know that the range is ascendent and/or bounded by a constatnt(not necessary)
template algStmtForRange_impl(Exp range, Ident iterator, String type, String shortType, Text body, Context context, SimCode simCode)
 "The implementation of algStmtForRange, which is also used by daeExpReduction."
::=
match range
case RANGE(__) then
  let iterName = iterator
  let &stopVar = buffer ""
  let &preExp = buffer ""
  let startValue = daeExp(exp, context, &preExp, simCode)
  let stopValue = daeExp(range, context, &preExp, simCode)
  match expOption 
  case SOME(eo) then
    let &stepVar = buffer "" 
    let stepValue = daeExp(eo, context, &preExp, simCode)
    <<
    <%preExp%>
    <%tempDecl(type, &stepVar)%> = <%stepValue%>; <%tempDecl(type, &stopVar)%> = <%stopValue%>;
    for(<%iterName%> = <%startValue%>;(<%stepVar%> > 0? <%iterName%><=<%stopVar%> : <%stopVar%><=<%iterName%>); <%iterName%> += <%stepVar%>) { 
      <%body%>
    }
    >>
  else //optimization for 1 step
    <<
    <%preExp%>
    <%tempDecl(type, &stopVar)%> = <%stopValue%>;
    for(<%iterName%> = <%startValue%>; <%iterName%><=<%stopVar%>; <%iterName%> += 1) { 
      <%body%>
    }
    >>
end algStmtForRange_impl;

template elseExpr(DAE.Else it, Context context, SimCode simCode) ::= 
  match it
  case NOELSE(__) then ""
  case ELSEIF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp, simCode)
    <<
    else {
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    <%elseExpr(else_, context, simCode)%>
    }
    >>
  case ELSE(__) then
    <<
    else {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    >>
end elseExpr;
    

template scalarLhsCref(Exp ecr, Context context, Text &preExp, SimCode simCode) ::=
match ecr
case ecr as CREF(componentRef=CREF_IDENT(subscriptLst=subs)) then
  if crefNoSub(ecr.componentRef) then
    contextCref(ecr.componentRef, context, simCode)
  else
    daeExpCrefRhs(ecr, context, &preExp, simCode)
case ecr as CREF(componentRef=CREF_QUAL(__)) then
    contextCref(ecr.componentRef, context, simCode)
else
    "ONLY_IDENT_OR_QUAL_CREF_SUPPORTED_SLHS"
end scalarLhsCref;

/*
template scalarLhsCref(ComponentRef it, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then replaceDollarWorkaround(ident)
  case CREF_QUAL(__)  then '<%ident%>.<%scalarLhsCref(componentRef)%>'
end scalarLhsCref;
*/

//TODO: this wrong for qualified integers !
template rhsCref(ComponentRef it, ExpType ty, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then '<%rhsCrefType(ty)%><%replaceDollarWorkaround(ident)%>'
  case CREF_QUAL(__)  then '<%rhsCrefType(ty)%><%ident%>.<%rhsCref(componentRef,ty, simCode)%>'
  case _          then "rhsCref:ERROR"
end rhsCref;

template rhsCrefType(ExpType it) ::=
  match it
  case ET_INT(__) then "(int)"
  case _      then ""
end rhsCrefType;


template replaceDollarWorkaround(String ident) ::=
  stringReplace(
    stringReplace(ident,"$DER","Der_"), 
    "$", "")
end replaceDollarWorkaround;

// SECTION: GENERAL TEMPLATES, EXPRESSIONS
  
template daeExp(Exp inExp, Context context, Text &preExp, SimCode simCode) ::=
  match inExp
  case ICONST(__)     then integer
  case RCONST(__)     then real
  case SCONST(__)     then '"<%Util.escapeModelicaStringToCString(string)%>"'
  case BCONST(__)     then if bool then "(1)" else "(0)"
  case ENUM_LITERAL(__) then '<%index%>/*ENUM:<%dotPath(name)%>*/'
  case CREF(ty = ET_FUNCTION_REFERENCE_FUNC(__)) then 'FUNC_REF_NOT_SUPPORTED(cr=<%crefStr(componentRef,simCode)%>)'
  case CREF(ty = ET_COMPLEX(complexClassType = RECORD(__))) then
    match context case FUNCTION_CONTEXT(__) then
      daeExpCrefRhs(inExp, context, &preExp, simCode)
    else
      "daeExpRecordCrefRhs_NOT_YET" //daeExpRecordCrefRhs(t, cr, context, preExp, varDecls)
  case CREF(__)       then daeExpCrefRhs(inExp, context, &preExp, simCode)
  case LBINARY(__)
  case BINARY(__)     then daeExpBinary(operator, exp1, exp2, context, &preExp, simCode)
  case LUNARY(__)
  case UNARY(__)      then daeExpUnary(operator, exp, context, &preExp, simCode)
  case RELATION(__)   then daeExpRelation(operator, exp1, exp2, context, &preExp, simCode)
  case IFEXP(__)      then daeExpIf(expCond, expThen, expElse, context, &preExp, simCode)
  case CALL(__)       then daeExpCall(inExp, context, &preExp, simCode)
  // PARTEVALFUNCTION
  case ARRAY(__)      then daeExpArray(inExp, context, &preExp, simCode)
  case MATRIX(__)     then daeExpMatrix(inExp, context, &preExp, simCode)
  case RANGE(__)      then "RANGE_NOT_IMPLEMENTED"
  case TUPLE(__)      then "TUPLE_NOT_IMPLEMENTED"
  case CAST(__)       then daeExpCast(inExp, context, &preExp, simCode)
  case ASUB(__)       then daeExpAsub(inExp, context, &preExp, simCode)
  case SIZE(__)       then daeExpSize(inExp, context, &preExp, simCode)
  case CODE(__)       then "CODE_NOT_IMPLEMENTED"
  case REDUCTION(__)  then "REDUCTION_NOT_IMPLEMENTED"
  case END(__)        then "END_NOT_IMPLEMENTED"
  //case VALUEBLOCK(__) then "VALUEBLOCK_NOT_IMPLEMENTED"
  case LIST(__)       then "LIST_NOT_IMPLEMENTED"
  case CONS(__)       then "CONS_NOT_IMPLEMENTED"
  // META_TUPLE
  // META_OPTION
  // METARECORDCALL
  case _          then "UNKNOWN_EXP"
end daeExp;

template daeExpSize(Exp esize, Context context, Text &preExp, SimCode simCode)
 "Generates code for a size expression."
::=
  match esize
  case SIZE(exp=CREF(__), sz=SOME(dim)) then
    let expPart = daeExp(exp, context, &preExp, simCode)
    match dim
    case ICONST(__) then 
      '<%expPart%>.size<%integer%>'
    else
      '<%expPart%>.size(<%daeExp(dim, context, &preExp, simCode)%>)'    
  else "size_X_NOT_IMPLEMENTED"
end daeExpSize;

  
template daeExpCrefRhs(Exp ecr, Context context, Text &preExp, SimCode simCode) ::=
  match ecr
  case ecr as CREF(componentRef = cr) then
    let box = daeExpCrefRhsArrayBox(ecr, context, &preExp, simCode)
    if box then
      box
    else if crefIsScalar(cr, context) then
      match ecr.ty 
      case ET_INT(__)  then '(int)<%contextCref(cr, context, simCode)%>'
      case ET_BOOL(__) then '(<%contextCref(cr, context, simCode)%> !=0.0)'
      else contextCref(cr, context, simCode)
    else if crefSubIsScalar(cr) then
      // The array subscript results in a scalar
      //let ety = match ecr.ty case ET_ARRAY(__) then "array" else "noarray"
      let arrName = contextCref(crefStripLastSubs(cr), context, simCode)
      //let arrayType = expTypeArray(ecr.ty,listLength(crefSubs(cr)))
      //let dimsLenStr = listLength(crefSubs(cr))
      //let dimsValuesStr = (crefSubs(cr) |> INDEX(__) => daeExp(exp, context, &preExp, simCode) ;separator=", ")
      <<
      <%arrName%>[<%crefSubs(cr) |> INDEX(__) => daeExp(exp, context, &preExp, simCode) 
                     ;separator=", "
                   %>]      
      >>
    else
      // The array subscript denotes a slice
      let arrName = contextCref(crefStripLastSubs(cr), context, simCode)
      //let crsubs = crefSubs(cr)
      //let arrayType = expTypeArray(ecr.ty, listLength(crsubs))
      let spec = daeExpCrefRhsIndexSpec(crefSubs(cr), context, &preExp, simCode)
      '<%arrName%>.Array1Spec(<%spec%>)'
  else
    "UNKNOWN_RHS_CREF"  
end daeExpCrefRhs;

template daeExpCrefRhsArrayBox(Exp exp, Context context, Text &preExp, SimCode simCode) ::=
match exp
case ecr as CREF(ty=ET_ARRAY(ty=aty,arrayDimensions=dims)) then
  match context 
  case FUNCTION_CONTEXT(__) then ""
  else
  //case SIMULATION(__) //TODO: ?? is it the same as "else than FUNCTION_CONTEXT" ? 
  //case OTHER(__)      then
    // For context simulation and other array variables must be boxed into a real_array
    // object since they are represented only in a double array.
    let &tmpArr = buffer ""
    let arrType = expTypeArray(aty, listLength(dims))
    let dimsValuesStr = (dims |> dim => dimension(dim) ;separator=", ")
    let &preExp += match cref2simvar(ecr.componentRef, simCode) case SIMVAR(__) then  
         '<%tempDecl("var", &tmpArr)%> = new <%arrType%>(<%dimsValuesStr%>, <%index%>-1, /*<%crefStr(ecr.componentRef, simCode)%>*/<%representationArrayName(varKind,type_)%>);<%\n%>'
    tmpArr
end daeExpCrefRhsArrayBox;


template daeExpCrefRhsIndexSpec(list<Subscript> subs, Context context, Text &preExp, SimCode simCode) ::=
  subs |> sub => match sub
       case INDEX(__) then
         'new int[]{<%daeExp(exp, context, &preExp, simCode)%>-1}'
       case WHOLEDIM(__) then
         "null"
       case SLICE(__) then
         match exp
         case ARRAY(scalar=true, ty=ET_INT(__)) then
           <<
           new int[]{<% array |> e => '(<%expTypeFromExp(e)%>)<%daeExp(e, context, &preExp, simCode)%>-1' 
                        ;separator=","
                      %>
           >>
         else "UKNOWN_SLICE_EXP"
  ;separator=", "
end daeExpCrefRhsIndexSpec;


template daeExpAsub(Exp aexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for an asub expression."
::=
  match aexp
  case ASUB(exp=RANGE(ty=t), sub={idx}) then
    'ASUB_EASY_CASE'
  case ASUB(exp=ASUB(
              exp=ASUB(
                exp=ASUB(exp=e, sub={ICONST(integer=i)}),
                sub={ICONST(integer=j)}),
              sub={ICONST(integer=k)}),
            sub={ICONST(integer=l)}) then
    'ASUB_4D'
  case ASUB(exp=ASUB(
              exp=ASUB(exp=e, sub={ICONST(integer=i)}),
              sub={ICONST(integer=j)}),
            sub={ICONST(integer=k)}) then
    'ASUB_3D'
  case ASUB(exp=ASUB(exp=e, sub={ICONST(integer=i)}),
            sub={ICONST(integer=j)}) then
    '<%daeExp(e, context, &preExp, simCode)%>[<%i%>,<%j%>]'
  case ASUB(exp=e, sub={ICONST(integer=i)}) then
    'ASUB_ARRAY'
  case ASUB(exp=ecr as CREF(componentRef = cr), sub = subs) then
    match context case FUNCTION_CONTEXT(__) then
      daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context, &preExp, simCode)
    else //SIMULATION or OTHER contexts
      match ecr.ty
      case ET_ARRAY(ty = ET_REAL(__), arrayDimensions = dims) then //daeExpCrefRhsArrayBox
        <<
        /*<% crefStr(cr, simCode) 
           %>[]*/<% cref2simvar(cr, simCode) |> SIMVAR(__) =>
	                 let &constSum = buffer index
	                 let baseSub = asubSubsripts(dims, subs, &constSum, context, &preExp, simCode)
	                 <<
	                 <%representationArrayName(varKind,type_)%>[<%constSum%><%baseSub%>]
	                 >>	                      
	              %>
	    >> 
      else "ASUB_SIMULATION_OTHER_ERROR"
        
      
    //cref(ecr.componentRef, simCode)
    /*
    let arrName = daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context, &preExp, simCode)
    match context case SIMULATION(__) then
      arrayScalarRhs(ecr.ty, subs, arrName, context, &preExp, simCode)
    else
      arrName
     */      
  else
    'OTHER_ASUB__ERROR'   
end daeExpAsub;

template asubSubsripts(list<Dimension> dims, list<Exp> subs, Text &constSum,
                       Context context, Text &preExp, SimCode simCode)
 "Helper to daeExpAsub."
::=
  match subs case s :: subsRest then
  	match dims  case _ :: dimsRest then
  		let subStr = daeExp(s, context, &preExp, simCode)
  		if dimsRest then //not last
  		   let ds = dimsRest |> dim => dimension(dim) ;separator="*"
  		   //if ds then //TODO: assuming every dimension is SOME, is it true ??
  		   let &constSum += '-(<%ds%>)' //-1 * ds
  		   '+<%subStr%>*(<%ds%>)<% asubSubsripts(dimsRest, subsRest, &constSum, context, &preExp, simCode) %>'
  		else //the last sub, add it to constSum (better optimization on compilation)
  		   let &constSum += '-1 + <% subStr %>'
  		   ""  		
  	else "ERROR_asubSubsripts_not_enough_dims" 
end asubSubsripts;

// TODO: Optimize as in Codegen
// TODO: Use this function in other places where almost the same thing is hard
//       coded
// not used yet 
template arrayScalarRhs(ExpType ty, list<Exp> subs, Text arrName, Context context,
                        Text &preExp, SimCode simCode)
 "Helper to daeExpAsub."
::=
  let arrayType = expTypeArray(ty, listLength(subs))
  let dimsValuesStr = (subs |> exp => daeExp(exp, context, &preExp, simCode)
                       ;separator=", ")
  <<
  (*ASR<%arrayType%>_element_addr(&<%arrName%>, <%/*dimsLen*/%>, <%dimsValuesStr%>))
  >>
end arrayScalarRhs;








template daeExpBinary(Operator it, Exp exp1, Exp exp2, Context context, Text &preExp, SimCode simCode) ::=
  let e1 = daeExp(exp1, context, &preExp, simCode)
  let e2 = daeExp(exp2, context, &preExp, simCode)
  match it
  case ADD(__) then '(<%e1%> + <%e2%>)'
  case SUB(__) then '(<%e1%> - <%e2%>)'
  case MUL(__) then '(<%e1%> * <%e2%>)'
  case DIV(__) then '(<%e1%> / <%e2%>)'
  case POW(__) then 'Math.Pow(<%e1%>, <%e2%>)'
  case AND(__) then '(<%e1%> && <%e2%>)'
  case OR(__)  then '(<%e1%> || <%e2%>)'
  case _   then "daeExpBinary:ERR"
end daeExpBinary;


template daeExpUnary(Operator it, Exp exp, Context context, Text &preExp, SimCode simCode) ::=
  let e = daeExp(exp, context, &preExp, simCode)
  match it
  case UMINUS(__)     then '(-<%e%>)'
  case UPLUS(__)      then '(<%e%>)'
  case NOT(__)        then '(!<%e%>)'
  case UMINUS_ARR(__) then "UMINUS_ARR_NOT_IMPLEMENTED"
  case UPLUS_ARR(__)  then "UPLUS_ARR_NOT_IMPLEMENTED"
  case _          then "daeExpUnary:ERR"

end daeExpUnary;


template daeExpRelation(Operator op, Exp exp1, Exp exp2, Context context, Text &preExp, SimCode simCode) ::=
  let e1 = daeExp(exp1, context, &preExp, simCode)
  let e2 = daeExp(exp2, context, &preExp, simCode)
  let simrel = daeExpSimRelation(context, op, e1, e2, &preExp) 
  if  simrel then simrel
  else //non-SIMULATION context or precise equality 
    match op
    case LESS(ty = ET_BOOL(__))        then '(!<%e1%> && <%e2%>)'
    case LESS(ty = ET_STRING(__))      then "# string comparison not supported\n"
    case LESS(ty = ET_INT(__))
    case LESS(ty = ET_REAL(__))        then '(<%e1%> < <%e2%>)'
    case GREATER(ty = ET_BOOL(__))     then '(<%e1%> && !<%e2%>)'
    case GREATER(ty = ET_STRING(__))   then "# string comparison not supported\n"
    case GREATER(ty = ET_INT(__))
    case GREATER(ty = ET_REAL(__))     then '(<%e1%> > <%e2%>)'
    case LESSEQ(ty = ET_BOOL(__))      then '(!<%e1%> || <%e2%>)'
    case LESSEQ(ty = ET_STRING(__))    then "# string comparison not supported\n"
    case LESSEQ(ty = ET_INT(__))
    case LESSEQ(ty = ET_REAL(__))      then '(<%e1%> <= <%e2%>)'
    case GREATEREQ(ty = ET_BOOL(__))   then '(<%e1%> || !<%e2%>)'
    case GREATEREQ(ty = ET_STRING(__)) then "# string comparison not supported\n"
    case GREATEREQ(ty = ET_INT(__))
    case GREATEREQ(ty = ET_REAL(__))   then '(<%e1%> >= <%e2%>)'
    case EQUAL(ty = ET_BOOL(__))       then '((!<%e1%> && !<%e2%>) || (<%e1%> && <%e2%>))'
    case EQUAL(ty = ET_STRING(__))
    case EQUAL(ty = ET_INT(__))
    case EQUAL(ty = ET_REAL(__))       then '(<%e1%> == <%e2%>)'
    case NEQUAL(ty = ET_BOOL(__))      then '((!<%e1%> && <%e2%>) || (<%e1%> && !<%e2%>))'
    case NEQUAL(ty = ET_STRING(__))
    case NEQUAL(ty = ET_INT(__))
    case NEQUAL(ty = ET_REAL(__))      then '(<%e1%> != <%e2%>)'
    case _                         then "daeExpRelation:ERR"
end daeExpRelation;


template daeExpSimRelation(Context it, Operator op, Text e1, Text e2, Text &preExp) ::=
	match it
	case SIMULATION(__) then 
	   match op
	   case LESS(__)      then SimRelationSimple(e1, e2, " <", &preExp)
	   case LESSEQ(__)    then SimRelationEqual(e1, e2, " <", &preExp)
	   case GREATER(__)   then SimRelationSimple(e1, e2, " >", &preExp)
	   case GREATEREQ(__) then SimRelationEqual(e1, e2, " >", &preExp)
	   end match
end daeExpSimRelation;


template SimRelationSimple(Text e1, Text e2, String op, Text &preExp) ::=
  let &res = buffer ""    
  let &preExp += 
    <<
    // RELATION( <%e1%><%op%> <%e2%> ) macro expansion
    <%tempDecl("bool", res)%> = <%e1%><%op%> <%e2%>; if (!<%res%> && isInUpdate && (<%e1%><%op%>= <%e2%>)) { SwapOldVars(); double res1 = <%e1%> - <%e2%>;  SwapOldVars12(); <%res%> = res1<%op%>= (<%e1%> - <%e2%>); SwapOldVars2(); }<%\n%>
    >>
  res 
end SimRelationSimple;


template SimRelationEqual(Text e1, Text e2, String op, Text &preExp) ::=
  let &res = buffer ""    
  let &preExp += 
    <<
    // RELATION( <%e1%><%op%>= <%e2%> ) macro expansion
    <%tempDecl("bool", res)%>;  if (isInUpdate) { <%res%> = <%e1%><%op%> <%e2%>;  if(!<%res%> && (<%e1%><%op%>= <%e2%>)) {  SwapOldVars(); double res1 = <%e1%> - <%e2%>;  SwapOldVars12(); <%res%> = res1<%op%>= (<%e1%> - <%e2%>); SwapOldVars2(); }  } else <%res%> = <%e1%><%op%>= <%e2%>;<%\n%>
    >>
  res
end SimRelationEqual;


template daeExpIf(Exp cond, Exp then_, Exp else_, Context context, Text &preExp, SimCode simCode) ::=
  let condExp = daeExp(cond, context, &preExp, simCode)
  let &resVar = buffer ""
  let &preExpThen = buffer ""
  let eThen = daeExp(then_, context, &preExpThen, simCode)
  let &preExpElse = buffer ""
  let eElse = daeExp(else_, context, &preExpElse, simCode)
  let &preExp +=  
  <<
  <%tempDecl(expTypeFromExpArrayIf(then_), resVar)%>;
  if (<%condExp%><%/*if expTypeFromExp(cond) is "bool" then " != 0.0"*/%>) { //cond type is <%expTypeFromExp(cond)%>
    <%preExpThen%>
    <%resVar%> = <%eThen%>;
  } else {
    <%preExpElse%>
    <%resVar%> = <%eElse%>;
  }<%\n%>
  >>
  resVar

/*<<
(<%daeExp(cond, context, &preExp, simCode)
  %> ? <%daeExp(then_, context, &preExp, simCode)%> : <%daeExp(else_, context, &preExp, simCode)%>)
>>*/
end daeExpIf;


template daeExpCall(Exp it, Context context, Text &preExp, SimCode simCode) ::=
  match it
  // special builtins
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="DIVISION"),
            expLst={e1, e2, DAE.SCONST(string=string)}) then
    let var1 = daeExp(e1, context, &preExp, simCode)
    let msg = Util.escapeModelicaStringToCString(string)
    match e2 
    case RCONST(__) then 
     	//match rr case 0.0 then 'DivBy0(<%var1%>,0.0,"<%msg%>")'
     	//else 
     	'<%var1%> / <%daeExp(e2, context, &preExp, simCode)%>'
    case _ then
     	let var2 = daeExp(e2, context, &preExp, simCode)
    	'(<%var2%>!=0.0 ? <%var1%> / <%var2%> : DivBy0(<%var1%>,<%var2%>,"<%msg%>"))' 
    end match 
    
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="der"), expLst={arg as CREF(__)}) then
    derCref(arg.componentRef, simCode)
    
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="pre"), expLst={arg as CREF(__)}) then
    //let retType = expType(arg.ty)
    <<
    <%match arg.ty case ET_INT(__) then "(int)"
    %><% preCref(arg.componentRef, simCode) %>
    >>
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="min"), expLst={e1,e2}) then
    'Math.Min(<%daeExp(e1, context, &preExp, simCode)%>,<%daeExp(e2, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="max"), expLst={e1,e2}) then
    'Math.Max(<%daeExp(e1, context, &preExp, simCode)%>,<%daeExp(e2, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="log"), expLst={s1}) then
    'Math.Log(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="exp"), expLst={s1}) then
    'Math.Exp(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="sin"), expLst={s1}) then
    'Math.Sin(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="sqrt"), expLst={s1}) then
    'Math.Sqrt(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(tuple_=false, builtin=true,
            path=IDENT(name="noEvent"), expLst={s1}) then
    '(/*noEvent*/<%daeExp(s1, context, &preExp, simCode)%>)'
  
    
  // TODO: add more special builtins (Codegen.generateBuiltinFunction)
  // no return calls
  case CALL(tuple_=false, ty=ET_NORETCALL(__)) then
    let argStr = (expLst |> it => daeExp(it, context, &preExp, simCode) ;separator=", ")
    let &preExp += '<%underscorePrefix(builtin)%><%underscorePath(path)%>(<%argStr%>);<%\n%>'
    <<
    /* NORETCALL */
    >>
  // non tuple calls (single return value)
  case CALL(tuple_=false) then
    let argStr = (expLst |> it => daeExp(it, context, &preExp, simCode) ;separator=", ")
    let funName = underscorePath(path)
    <<
    <%underscorePrefix(builtin)
    %><%funName%>(<%argStr%>)<%if not builtin then '/* !!!TODO:.<%funName%>_rettype_1 */'%>
    >>
  case _ then "daeExpCall:NOT_YET_IMPLEMENTED"
end daeExpCall;

template daeExpArray(Exp aexp, Context context, Text &preExp, SimCode simCode) ::=
  match hackArrayReverseToCref(aexp, context)
  case cr as CREF(__) then daeExpCrefRhs(cr, context, &preExp, simCode)
  case a as ARRAY(__) then
    if scalar then 
  	  let &arrayVar = buffer ""
	  let params = a.array |> e => '(<%expTypeFromExp(e)%>)<%daeExp(e, context, &preExp, simCode)%>' 
	               ;separator=", "
	  let &preExp += '<%tempDecl("var",&arrayVar)%> = new <%expTypeArray(a.ty,1)%>(<%listLength(a.array)%>,-1,new[]{<%params%>});<%\n%>'
	  arrayVar
    else
      "NON_SCALAR_ARRAY_notYetImplemeted"
end daeExpArray;

template daeExpMatrix(Exp mexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for a cast expression."
::=
  match hackMatrixReverseToCref(mexp, context)  
  case cr as CREF(__) then daeExpCrefRhs(cr, context, &preExp, simCode)
  case MATRIX(scalar={{}}) // special case for empty matrix: create dimensional array Real[0,1]
  case MATRIX(scalar={})   // special case for empty array: create dimensional array Real[0,1] 
    then
    let &tmp = buffer ""
    let &preExp += '<%tempDecl("var",&tmp)%> = new <%expTypeArray(ty,2)%>(0,1);<%\n%>'
    tmp
  //only scalar orthogonal matrix for now  
  case m as MATRIX(scalar=(row1::_)) then
    let &tmp = buffer ""
    let matArr = m.scalar |> row =>
                       (row |> (elem,isScalar) =>
                           if isScalar then daeExp(elem, context, &preExp, simCode)
                           else "MATRIX_NON_SCALAR_NYI"
                        ;separator=", ")
                 ;separator=",\n" 
    let &preExp += 
       <<
       <%tempDecl("var",&tmp)%> = new <%expTypeArray(m.ty,2)%>(<%listLength(m.scalar)%>,<%listLength(row1)%>,-1, new[]{
         <%matArr ;anchor%>
       });<%\n%>
       >>
    tmp     
end daeExpMatrix;


template daeExpCast(Exp cexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for a cast expression."
::=
match cexp
case CAST(__) then
  let expVar = daeExp(exp, context, &preExp, simCode)
  match ty
  case ET_INT(__)   then '((int)<%expVar%>)'
  case ET_REAL(__)  then '((double)<%expVar%>)'
  else "NOT_IMPLEMENTED_CAST"    
end daeExpCast;



template underscorePrefix(Boolean builtin) ::=
  match builtin
  case true then ""
  case false then "_"
end underscorePrefix;

// SECTION: GENERAL TEMPLATES, TEMPORARY VARIABLES

//newVar parameter is assumed to be empty or it can hold an identifier prefix
template tempDecl(String ty, Text &newVar) ::=
  let &newVar += '_tmp<%System.tmpTick()%>'
  <<
  <%ty%> <%newVar%>
  >>
end tempDecl;
  


// SECTION: GENERAL TEMPLATES, TYPES

template varType(Variable var)
 "Generates type for a variable."
::=
match var
case VARIABLE(__) then
  if instDims then
    expTypeArray(ty, listLength(instDims))
  else
    expTypeArrayIf(ty)
end varType;

// TODO: Check with Codegen
template expTypeShort(DAE.ExpType it) ::=
  match it
  case ET_INT(__)    then "int"
  case ET_REAL(__)   then "double"
  case ET_STRING(__) then "string"
  case ET_BOOL(__)   then "bool"
  case ET_OTHER(__)  then "OTHER_TYPE_NOT_SUPPORTED"
  case ET_ARRAY(__)  then expTypeShort(ty)   
  case ET_COMPLEX(complexClassType=EXTERNAL_OBJ(__)) then "COMPLEX_EXTERNAL_TYPE_NOT_SUPPORTED"
  case ET_COMPLEX(__) then '/*struct*/<%underscorePath(name)%>'
  case ET_METATYPE(__) case ET_BOXED(__) then "META_TYPE_NOT_SUPPORTED"
  case ET_FUNCTION_REFERENCE_VAR(__) then "FN_PTR_NOT_SUPPORTED"
  else "expTypeShort_ERROR"
end expTypeShort;


template expType(DAE.ExpType ty, Boolean isArray)
 "Generate type helper."
::=
  if isArray 
  then 'expType_<%expTypeArray(ty,0)%>_NOT_YET'
  else expTypeShort(ty)
end expType;

template expTypeArray(DAE.ExpType ty, Integer dims) ::=
<<
SimArray<%dims%><<%expTypeShort(ty)%>>
>>
end expTypeArray;

template expTypeArrayIf(DAE.ExpType ty)
 "Generate type helper."
::=
  match ty
  case ET_ARRAY(__) then expTypeArray(ty, listLength(arrayDimensions))
  else expTypeShort(ty)
end expTypeArrayIf;


template expTypeFromExpArrayIf(Exp exp) ::=
  expTypeFromExp(exp)
end expTypeFromExpArrayIf;


template expTypeFromExp(Exp it) ::=
  match it
  case ICONST(__)    then "int"
  case RCONST(__)    then "double"
  case SCONST(__)    then "string"
  case BCONST(__)    then "bool"
  case BINARY(__)
  case UNARY(__)
  case LBINARY(__)
  case LUNARY(__)     then expTypeFromOp(operator)
  case RELATION(__)   then "bool" //TODO: a HACK, it was expTypeFromOp(operator)
  case IFEXP(__)      then expTypeFromExp(expThen)
  case CALL(__)       then expTypeShort(ty)
  case ARRAY(__)
  case MATRIX(__)
  case RANGE(__)
  case CAST(__)
  case CREF(__)
  case CODE(__)       then expTypeShort(ty)
  case ASUB(__)       then expTypeFromExp(exp)
  case REDUCTION(__)  then expTypeFromExp(expr)
  case _          then "expTypeFromExp:ERROR"
end expTypeFromExp;


template expTypeFromOp(Operator it) ::=
  match it
  case ADD(__)
  case SUB(__)
  case MUL(__)
  case DIV(__)
  case POW(__)
  case UMINUS(__)
  case UPLUS(__)
  case UMINUS_ARR(__)
  case UPLUS_ARR(__)
  case ADD_ARR(__)
  case SUB_ARR(__)
  case MUL_ARR(__)
  case DIV_ARR(__)
  case MUL_SCALAR_ARRAY(__)
  case MUL_ARRAY_SCALAR(__)
  case ADD_SCALAR_ARRAY(__)
  case ADD_ARRAY_SCALAR(__)
  case SUB_SCALAR_ARRAY(__)
  case SUB_ARRAY_SCALAR(__)
  case MUL_SCALAR_PRODUCT(__)
  case MUL_MATRIX_PRODUCT(__)
  case DIV_ARRAY_SCALAR(__)
  case DIV_SCALAR_ARRAY(__)
  case POW_ARRAY_SCALAR(__)
  case POW_SCALAR_ARRAY(__)
  case POW_ARR(__)
  case POW_ARR2(__)
  case LESS(__)
  case LESSEQ(__)
  case GREATER(__)
  case GREATEREQ(__)
  case EQUAL(__)
  case NEQUAL(__)       then  expTypeShort(ty)
  case AND(__)
  case OR(__)
  case NOT(__) then "bool"
  case _ then "expTypeFromOp:ERROR"
end expTypeFromOp;

template dimension(Dimension d)
::=
  match d
  case DAE.DIM_INTEGER(__) then integer
  case DAE.DIM_UNKNOWN(__) then ":"
  else "INVALID_DIMENSION"
end dimension;


end SimCodeCSharp;
// vim: filetype=susan sw=2 sts=2
