// $Id: EIItem.cpp 10144 2011-10-18 11:42:23Z hubert.thieriot $
/**
 * This file is part of OpenModelica.
 *
 * Copyright (c) 1998-CurrentYear, Open Source Modelica Consortium (OSMC),
 * c/o Linköpings universitet, Department of Computer and Information Science,
 * SE-58183 Linköping, Sweden.
 *
 * All rights reserved.
 *
 * THIS PROGRAM IS PROVIDED UNDER THE TERMS OF GPL VERSION 3 LICENSE OR 
 * THIS OSMC PUBLIC LICENSE (OSMC-PL). 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS PROGRAM CONSTITUTES RECIPIENT'S ACCEPTANCE
 * OF THE OSMC PUBLIC LICENSE OR THE GPL VERSION 3, ACCORDING TO RECIPIENTS CHOICE. 
 *
 * The OpenModelica software and the Open Source Modelica
 * Consortium (OSMC) Public License (OSMC-PL) are obtained
 * from OSMC, either from the above address,
 * from the URLs: http://www.ida.liu.se/projects/OpenModelica or  
 * http://www.openmodelica.org, and in the OpenModelica distribution. 
 * GNU version 3 is obtained from: http://www.gnu.org/copyleft/gpl.html.
 *
 * This program is distributed WITHOUT ANY WARRANTY; without
 * even the implied warranty of  MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE, EXCEPT AS EXPRESSLY SET FORTH
 * IN THE BY RECIPIENT SELECTED SUBSIDIARY LICENSE CONDITIONS OF OSMC-PL.
 *
 * See the full OSMC Public License conditions for more details.
 *
 * Main contributor 2010, Hubert Thierot, CEP - ARMINES (France)
 * Main contributor 2010, Hubert Thierot, CEP - ARMINES (France)

 	@file EIItem.cpp
 	@brief Comments for file documentation.
 	@author Hubert Thieriot, hubert.thieriot@mines-paristech.fr
 	Company : CEP - ARMINES (France)
 	http://www-cep.ensmp.fr/english/
        @version

  */
#include "EIItem.h"

namespace EI
{
EIItem::EIItem()
{
        _parent = NULL;
}


EIItem::EIItem(EIItem* parent,QString name)
{
        _parent = parent;
        setName(name);
        _checked = true;
}

EIItem::EIItem(const EIItem & item):MOItem(item)
{
        _parent = item._parent;
        _checked = item._checked;
	
        for(int i=0;i<item.childCount();i++)
                addChild(item.child(i)->clone());
}

EIItem::EIItem(QDomElement & domEl)
{
        QDomNamedNodeMap attributes = domEl.attributes();
        QString fieldName;
        QString fieldValue;

        for(int i=0;i<attributes.count();i++)
        {
                fieldName = attributes.item(i).toAttr().name();
                fieldName.replace(XMLTools::space()," ");
                fieldValue = attributes.item(i).toAttr().value();
                fieldValue.replace(XMLTools::space()," ");
                MOItem::setFieldValue(fieldName,QVariant(fieldValue));
        }

        _editableFields << EIItem::NAME;
}


EIItem::~EIItem(void)
{
    QString msg = "deleted EIItem"+name();
    qDebug(msg.toLatin1().data());
    emit deleted();
	clear();
}

EIItem* EIItem::parent()const
{
    return _parent;
}

EIItem* EIItem::clone() const
{
	EIItem* newItem = new EIItem(*this);
	return newItem;
}

//*****************************
//Data managment
//*****************************

QVariant EIItem::getFieldValue(int iField, int role) const
{
	switch(iField)
	{
	case NAME:
		return _name;
		break;
	case CHECKED:
                return _checked;
		break;
	default :
		return QVariant();
		break;
	}
}

bool EIItem::setFieldValue(int iField, QVariant value)
{
	switch(iField)
	{
	case NAME:
		_name = value.toString();
		break;
	case CHECKED:
                _checked = value.toBool();
		break;
	default :
		return false;
	}
	emit modified();
	return true;
}

QString EIItem::sFieldName(int iField, int role)
{
	switch(iField)
	{
	case NAME:
		return "Name";
	case CHECKED:
                return "Checked";
	default :
		return "-";
	}
}


//*****************************
//Children
//*****************************
int EIItem::childCount() const
{
        return _children.size();
}

int EIItem::streamChildCount() const
{
	int nbComp=0;
        for(int i=0;i<_children.size();i++)
                if(_children.at(i)->getEIType()==EI::STREAM)
			nbComp++;

	return nbComp;
}

int EIItem::groupChildCount() const
{
	int nbModel=0;
        for(int i=0;i<_children.size();i++)
                if(_children.at(i)->getEIType()==EI::GROUP)
			nbModel++;

	return nbModel;
}

EIItem* EIItem::child(int nRow) const
{
        if((nRow>-1)&&(nRow<_children.count()))
                return _children.at(nRow);
	else
		return NULL;
}

EIItem* EIItem::streamChild(int nRow) const
{
	int iCurStream=-1;
	int curIndex=0;
        while((curIndex<_children.size())&&(iCurStream<nRow))
	{
                if(_children.at(curIndex)->getEIType()==EI::STREAM)
			iCurStream++;
	
		curIndex++;
	}

	if(iCurStream==nRow)
                return _children.at(curIndex-1);
	else
		return NULL;
}



EIItem* EIItem::groupChild(int nRow) const
{
	int iCurGroup=-1;
	int curIndex=0;
        while((curIndex<_children.size())&&(iCurGroup<nRow))
	{
                if(_children.at(curIndex)->getEIType()==EI::GROUP)
			iCurGroup++;
	
		curIndex++;
	}

	if(iCurGroup==nRow)
                return _children.at(curIndex-1);
	else
		return NULL;
}

int EIItem::indexInParent()
{
    if(parent()==NULL)
        return -1;


    //looking for row number of child in parent
    int nbBrothers = parent()->childCount();
    bool found = false;
    int iC=0;

    while(!found && iC<nbBrothers)
    {
        found = (parent()->child(iC)==this);
        if(!found)
            iC++;
    }
    if(!found)
        return -1;
    else
        return iC;
}



bool EIItem::removeChild(int i)
{
	if((i>-1)&&(i<childCount()))
	{
		delete child(i);
                _children.removeAt(i);
                return true;
	}
        else
            return false;
}
bool EIItem::removeChild(EIItem * item)
{
        int i=_children.indexOf(item);

        if(i>-1)
        {
	removeChild(i);
            return true;
}
        else
            return false;
}

bool EIItem::removeDescendant(EIItem* item)
{
    bool ok = removeChild(item);
    int iChild=0;

    while(!ok && (iChild<childCount()))
    {
        ok = child(iChild)->removeChild(item);
        iChild++;
    }
    return ok;
}

void EIItem::removeUncheckedDescendants()
{
    int i=0;
    while(i<childCount())
    {
        if(!child(i)->isChecked())
            removeChild(i);
        else
        {
            child(i)->removeUncheckedDescendants();
            i++;
    }
}
}

int EIItem::findChild(QVariant itemFieldValue, int iField)
{
        bool found = false;
        int i=0;
        int nbChildren=childCount();
        QVariant curFieldValue;

        while((!found)&&(i<nbChildren))
        {
                curFieldValue=child(i)->getFieldValue(iField);
                found=(itemFieldValue == curFieldValue);
                i++;
        }
        if(!found)
        {
                return -1;
        }
        else
        {
                return i-1;
        }
}


void EIItem::emitModified()
{
	emit modified();
}

int EIItem::depth() const
{
	QString fullName=_name;
        EIItem *curParent = parent();

	if(curParent==NULL)
		return  0;
	else
		return curParent->depth()+1;
}


void EIItem::clear()
{
	clearDescendants();
	emit modified();
        _parent = NULL;
	_name = QString();
	emit cleared();
}

void EIItem::clearDescendants()
{
        while(_children.size()>0)
	{
                delete _children.at(0);
                _children.removeAt(0);
	}
	 emit modified();
}

bool EIItem::addChild(EIItem *child, int i)
{

	bool ok=false;
	if(child && (this->getEIType()!=EI::STREAM))
	{
		child->setParent(this);
		// verify its name is unique compared to brother
		QStringList brothersNames;
                for(int i=0;i<this->_children.size();i++)
                        brothersNames.push_back(this->_children.at(i)->name());

		child->checkUniqueItemName(brothersNames);
                int index;
                if(i>=0)
                    index=i;
                else
                    index = _children.size()+i+1;
		
                _children.insert(i,child);
		
		ok = true;

		emit addedChild(child);
		connect(child,SIGNAL(modified()),this,SIGNAL(modified()));
	}
	if(ok)
		emit modified();

	return ok;
}




void EIItem::setParent(EIItem *parent)
{
        if(_parent != parent)
	{
                _parent = parent;
		emit modified();
	}
}

QString EIItem::getStrToolTip()
{
	QString toolTip;
	toolTip += ("EI item : " + _name);
	return toolTip;
}

bool EIItem::isChecked()
{
        return _checked;
}

void EIItem::setChecked(bool checked)
{
        if(checked!=_checked)
	{
                setFieldValue(CHECKED,checked);

		// if false, uncheck children
                if(!checked)
		{
		for(int i=0;i<childCount();i++)
			child(i)->setChecked(false);
		}
		
		// if true, check parent
                if(_parent && checked)
                        _parent->setChecked(true),
		
		emit modified();
	}
}

///**
//  * \brief returns model it is involved in. If is empty, looks for first parent which got a non-empty modelname
//  *
//  */
//QString  EIItem::model()
//{
//    if(getEIType()==EI::MODELCONTAINER)
//        return this->getFieldValue(EIModelContainer::MODEL);
//    else
//    {
//        if(parent()==NULL)
//            return _model;
//        else
//            return parent()->model();
//    }
//}

QDomElement EIItem::toXmlData(QDomDocument & doc)
{
	QDomElement result = MOItem::toXmlData(doc);
        //replace full name with short name
        result.setAttribute(getFieldName(EIItem::NAME),name(EI::SHORT));

	for(int i=0;i<childCount();i++)
		result.appendChild(child(i)->toXmlData(doc));

	return result;
}

QString EIItem::name(EI::NameFormat type)const
{
	if(type == EI::SHORT)
		return _name;
	else
	{
		QString fullName=_name;
                EIItem *curParent = parent();

                while((curParent!=NULL)&&(curParent->parent()!=NULL))
		{
			fullName.insert(0,curParent->name(EI::SHORT)+".");
                        curParent = curParent->parent();
		}
		return fullName;
	}
}

bool EIItem::isValid(MOOptVector* variables, QString &errMsg) const
{
    bool ok = true;

//    if(_name.isEmpty())
//    {
//        ok = false;
//        errMsg+= "empty name \n";
//    }

    bool hasAccent = (_name.indexOf(QRegExp("é|è| |ü|ö|ô|â)"))>-1);
    if(hasAccent)
    {
        ok = false;
        errMsg+="accents in "+_name;
    }

    return ok;
}


QString EIItem::model() const
{
    if(parent()==NULL)
        return QString();
    else
        return parent()->model();
}


/**
* This function is used to get references involved this EIItem.
* For generic EIItem, there is no reference.
* This is virtual since references can come from specific fields (e.g. from EIStream)
*/
QStringList EIItem::references() const
{
    return QStringList();
}

/**
* This function is used to replace reference by its numerical value extracted from variables vector.
* Be careful : this functions introduces a lost of information since reference name is replaced by its numerical value.
* This is a virtual function since all inheriting class have different fields.
*/
bool EIItem::numerize(MOOptVector* variables)
{
    return true;
}

}
