/*
 * The HumModOMCTotal.mo model is GPL v2 (http://www.gnu.org/licenses/gpl-2.0.html).
 * contact Pavol at: Pavol.Privitzer@lf1.cuni.cz if you have questions about the model
 */


within ;
package Modelica "Modelica Standard Library (Version 3.1)"
extends Modelica.Icons.Library;

  package Blocks
    "Library of basic input/output control blocks (continuous, discrete, logical, table blocks)"
  import SI = Modelica.SIunits;
  extends Modelica.Icons.Library2;

    package Continuous
      "Library of continuous control blocks with internal states"
      import Modelica.Blocks.Interfaces;
      import Modelica.SIunits;
      extends Modelica.Icons.Library;

      block Integrator "Output the integral of the input signal"
        import Modelica.Blocks.Types.Init;
        parameter Real k=1 "Integrator gain";

        /* InitialState is the default, because it was the default in Modelica 2.2
     and therefore this setting is backward compatible
  */
        parameter Modelica.Blocks.Types.Init initType=Modelica.Blocks.Types.Init.InitialState
          "Type of initialization (1: no init, 2: steady state, 3,4: initial output)"
                                                                                          annotation(Evaluate=true,
            Dialog(group="Initialization"));
        parameter Real y_start=0 "Initial or guess value of output (= state)" 
          annotation (Dialog(group="Initialization"));
        extends Interfaces.SISO(y(start=y_start));


      initial equation
        if initType == Init.SteadyState then
           der(y) = 0;
        elseif initType == Init.InitialState or 
               initType == Init.InitialOutput then
          y = y_start;
        end if;
      equation
        der(y) = k*u;
        annotation (
          Documentation(info="<html>
<p>
This blocks computes output <b>y</b> (element-wise) as
<i>integral</i> of the input <b>u</b> multiplied with
the gain <i>k</i>:
</p>
<pre>
         k
     y = - u
         s
</pre>

<p>
It might be difficult to initialize the integrator in steady state.
This is discussed in the description of package
<a href=\"Modelica://Modelica.Blocks.Continuous#info\">Continuous</a>.
</p>

</html>
"),       Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-80,78},{-80,-90}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,-80},{82,-80}}, color={192,192,192}),
              Polygon(
                points={{90,-80},{68,-72},{68,-88},{90,-80}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{0,-10},{60,-70}},
                lineColor={192,192,192},
                textString="I"),
              Text(
                extent={{-150,-150},{150,-110}},
                lineColor={0,0,0},
                textString="k=%k"),
              Line(points={{-80,-80},{80,80}}, color={0,0,127})}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(extent={{-60,60},{60,-60}}, lineColor={0,0,255}),
              Line(points={{-100,0},{-60,0}}, color={0,0,255}),
              Line(points={{60,0},{100,0}}, color={0,0,255}),
              Text(
                extent={{-36,60},{32,2}},
                lineColor={0,0,0},
                textString="k"),
              Text(
                extent={{-32,0},{36,-58}},
                lineColor={0,0,0},
                textString="s"),
              Line(points={{-46,0},{46,0}}, color={0,0,0})}));
      end Integrator;
      annotation (
        Documentation(info="<html>
<p>
This package contains basic <b>continuous</b> input/output blocks
described by differential equations.
</p>

<p>
All blocks of this package can be initialized in different
ways controlled by parameter <b>initType</b>. The possible
values of initType are defined in
<a href=\"Modelica://Modelica.Blocks.Types.Init\">Modelica.Blocks.Types.Init</a>:
</p>

<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>Name</b></td>
      <td valign=\"top\"><b>Description</b></td></tr>

  <tr><td valign=\"top\"><b>Init.NoInit</b></td>
      <td valign=\"top\">no initialization (start values are used as guess values with fixed=false)</td></tr>

  <tr><td valign=\"top\"><b>Init.SteadyState</b></td>
      <td valign=\"top\">steady state initialization (derivatives of states are zero)</td></tr>

  <tr><td valign=\"top\"><b>Init.InitialState</b></td>
      <td valign=\"top\">Initialization with initial states</td></tr>

  <tr><td valign=\"top\"><b>Init.InitialOutput</b></td>
      <td valign=\"top\">Initialization with initial outputs (and steady state of the states if possibles)</td></tr>
</table>

<p>
For backward compatibility reasons the default of all blocks is
<b>Init.NoInit</b>, with the exception of Integrator and LimIntegrator
where the default is <b>Init.InitialState</b> (this was the initialization
defined in version 2.2 of the Modelica standard library).
</p>

<p>
In many cases, the most useful initial condition is
<b>Init.SteadyState</b> because initial transients are then no longer
present. The drawback is that in combination with a non-linear
plant, non-linear algebraic equations occur that might be
difficult to solve if appropriate guess values for the
iteration variables are not provided (i.e. start values with fixed=false).
However, it is often already useful to just initialize
the linear blocks from the Continuous blocks library in SteadyState.
This is uncritical, because only linear algebraic equations occur.
If Init.NoInit is set, then the start values for the states are
interpreted as <b>guess</b> values and are propagated to the
states with fixed=<b>false</b>.
</p>

<p>
Note, initialization with Init.SteadyState is usually difficult
for a block that contains an integrator
(Integrator, LimIntegrator, PI, PID, LimPID).
This is due to the basic equation of an integrator:
</p>

<pre>
  <b>initial equation</b>
     <b>der</b>(y) = 0;   // Init.SteadyState
  <b>equation</b>
     <b>der</b>(y) = k*u;
</pre>

<p>
The steady state equation leads to the condition that the input to the
integrator is zero. If the input u is already (directly or indirectly) defined
by another initial condition, then the initialization problem is <b>singular</b>
(has none or infinitely many solutions). This situation occurs often
for mechanical systems, where, e.g., u = desiredSpeed - measuredSpeed and
since speed is both a state and a derivative, it is always defined by
Init.InitialState or Init.SteadyState initializtion.
</p>

<p>
In such a case, <b>Init.NoInit</b> has to be selected for the integrator
and an additional initial equation has to be added to the system
to which the integrator is connected. E.g., useful initial conditions
for a 1-dim. rotational inertia controlled by a PI controller are that
<b>angle</b>, <b>speed</b>, and <b>acceleration</b> of the inertia are zero.
</p>

</html>
"));
    end Continuous;

    package Interfaces
      "Library of connectors and partial models for input/output blocks"
      import Modelica.SIunits;
        extends Modelica.Icons.Library;

    connector RealInput = input Real "'input Real' as connector" 
      annotation (defaultComponentName="u",
      Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid)},
           coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
      Diagram(coordinateSystem(
            preserveAspectRatio=true, initialScale=0.2,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,0,127},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

    connector RealOutput = output Real "'output Real' as connector" 
      annotation (defaultComponentName="y",
      Icon(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)}),
      Diagram(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{-100,50},{0,0},{-100,-50},{-100,50}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{30,110},{30,60}},
              lineColor={0,0,127},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one output signal of type Real.
</p>
</html>"));

    connector BooleanInput = input Boolean "'input Boolean' as connector" 
      annotation (defaultComponentName="u",
           Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={255,0,255},
              fillColor={255,0,255},
              fillPattern=FillPattern.Solid)},
                coordinateSystem(extent={{-100,-100},{100,100}},
            preserveAspectRatio=true, initialScale=0.2)),    Diagram(coordinateSystem(
            preserveAspectRatio=true, initialScale=0.2,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={255,0,255},
              fillColor={255,0,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={255,0,255},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Boolean.
</p>
</html>"));

    connector BooleanOutput = output Boolean "'output Boolean' as connector" 
                                      annotation (defaultComponentName="y",
      Icon(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={255,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)}),
      Diagram(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{-100,50},{0,0},{-100,-50},{-100,50}},
              lineColor={255,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{30,110},{30,60}},
              lineColor={255,0,255},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one output signal of type Boolean.
</p>
</html>"));

        partial block BlockIcon "Basic graphical layout of input/output block"

          annotation (
            Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255})}),
          Documentation(info="<html>
<p>
Block that has only the basic icon for an input/output
block (no declarations, no equations). Most blocks
of package Modelica.Blocks inherit directly or indirectly
from this block.
</p>
</html>"));

        end BlockIcon;

        partial block SO "Single Output continuous control block"
          extends BlockIcon;

          RealOutput y "Connector of Real output signal" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));
          annotation (
            Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics),
          Documentation(info="<html>
<p>
Block has one continuous Real output signal.
</p>
</html>"));

        end SO;

        partial block SISO
        "Single Input Single Output continuous control block"
          extends BlockIcon;

          RealInput u "Connector of Real input signal" 
            annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
          RealOutput y "Connector of Real output signal" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));
          annotation (
          Documentation(info="<html>
<p>
Block has one continuous Real input and one continuous Real output signal.
</p>
</html>"));
        end SISO;

        partial block SI2SO
        "2 Single Input / 1 Single Output continuous control block"
          extends BlockIcon;

          RealInput u1 "Connector of Real input signal 1" 
            annotation (Placement(transformation(extent={{-140,40},{-100,80}},
                rotation=0)));
          RealInput u2 "Connector of Real input signal 2" 
            annotation (Placement(transformation(extent={{-140,-80},{-100,-40}},
                rotation=0)));
          RealOutput y "Connector of Real output signal" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

          annotation (
            Documentation(info="<html>
<p>
Block has two continuous Real input signals u1 and u2 and one
continuous Real output signal y.
</p>
</html>"),  Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics));

        end SI2SO;

        partial block MISO
        "Multiple Input Single Output continuous control block"

          extends BlockIcon;
          parameter Integer nin=1 "Number of inputs";
          RealInput u[nin] "Connector of Real input signals" 
            annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
          RealOutput y "Connector of Real output signal" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));
          annotation (Documentation(info="<HTML>
<p>
Block has a vector of continuous Real input signals and
one continuous Real output signal.
</p>
</HTML>
"));
        end MISO;

      partial block partialBooleanBlockIcon
        "Basic graphical layout of logical block"

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={210,210,210},
                fillPattern=FillPattern.Solid,
                borderPattern=BorderPattern.Raised), Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255})}),                        Documentation(info="<html>
<p>
Block that has only the basic icon for an input/output,
Boolean block (no declarations, no equations) used especially
in the Blocks.Logical library.
</p>
</html>"));
      end partialBooleanBlockIcon;

      partial block partialBooleanSource
        "Partial source block (has 1 output Boolean signal and an appropriate default icon)"
        extends partialBooleanBlockIcon;

        Blocks.Interfaces.BooleanOutput y "Connector of Boolean output signal" 
          annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Polygon(
                points={{-80,88},{-88,66},{-72,66},{-80,88}},
                lineColor={255,0,255},
                fillColor={255,0,255},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,66},{-80,-82}}, color={255,0,255}),
              Line(points={{-90,-70},{72,-70}}, color={255,0,255}),
              Polygon(
                points={{90,-70},{68,-62},{68,-78},{90,-70}},
                lineColor={255,0,255},
                fillColor={255,0,255},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{71,7},{85,-7}},
                lineColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillPattern=FillPattern.Solid)}),
            Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}), graphics={
              Polygon(
                points={{-70,92},{-76,70},{-64,70},{-70,92}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Line(points={{-70,70},{-70,-88}}, color={95,95,95}),
              Line(points={{-90,-70},{68,-70}}, color={95,95,95}),
              Polygon(
                points={{90,-70},{68,-64},{68,-76},{90,-70}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{54,-80},{106,-92}},
                lineColor={0,0,0},
                textString="time"),
              Text(
                extent={{-64,92},{-46,74}},
                lineColor={0,0,0},
                textString="y")}),
          Documentation(info="<html>
<p>
Basic block for Boolean sources of package Blocks.Sources.
This component has one continuous Boolean output signal y
and a 3D icon (e.g. used in Blocks.Logical library).
</p>
</html>"));

      end partialBooleanSource;

      partial block partialBooleanThresholdComparison
        "Partial block to compare the Real input u with a threshold and provide the result as 1 Boolean output signal"

        parameter Real threshold=0 "Comparison with respect to threshold";

        Blocks.Interfaces.RealInput u "Connector of Boolean input signal" 
                                      annotation (Placement(transformation(extent={
                  {-140,-20},{-100,20}}, rotation=0)));
        Blocks.Interfaces.BooleanOutput y "Connector of Boolean output signal" 
                                          annotation (Placement(transformation(
                extent={{100,-10},{120,10}}, rotation=0)));

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={210,210,210},
                fillPattern=FillPattern.Solid,
                borderPattern=BorderPattern.Raised),
              Text(
                extent={{-150,-140},{150,-110}},
                lineColor={0,0,0},
                textString="%threshold"),
              Ellipse(
                extent={{71,7},{85,-7}},
                lineColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillPattern=FillPattern.Solid)}),
            Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),
                    graphics),
          Documentation(info="<html>
<p>
Block has one continuous Real input and one continuous Boolean output signal
as well as a 3D icon (e.g. used in Blocks.Logical library).
</p>
</html>"));

      end partialBooleanThresholdComparison;

      partial block partialBooleanComparison
        "Partial block with 2 Real input and 1 Boolean output signal (the result of a comparison of the two Real inputs"

        Blocks.Interfaces.RealInput u1
          "Connector of first Boolean input signal" 
                                       annotation (Placement(transformation(extent=
                  {{-140,-20},{-100,20}}, rotation=0)));
        Blocks.Interfaces.RealInput u2
          "Connector of second Boolean input signal" 
                                       annotation (Placement(transformation(extent=
                  {{-140,-100},{-100,-60}}, rotation=0)));
        Blocks.Interfaces.BooleanOutput y "Connector of Boolean output signal" 
          annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={210,210,210},
                fillPattern=FillPattern.Solid,
                borderPattern=BorderPattern.Raised),
              Ellipse(
                extent={{73,7},{87,-7}},
                lineColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillColor=DynamicSelect({235,235,235}, if y > 0.5 then {0,255,0} else 
                          {235,235,235}),
                fillPattern=FillPattern.Solid),
              Ellipse(extent={{32,10},{52,-10}}, lineColor={0,0,255}),
              Line(points={{-100,-80},{42,-80},{42,0}}, color={0,0,255})}),
            Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),
                    graphics),
          Documentation(info="<html>
<p>
Block has two continuous Real input and one continuous Boolean output signal
as a result of the comparision of the two input signals. The block
has a 3D icon (e.g. used in Blocks.Logical library).
</p>
</html>"));

      end partialBooleanComparison;
        annotation (
          Documentation(info="<HTML>
<p>
This package contains interface definitions for
<b>continuous</b> input/output blocks with Real,
Integer and Boolean signals. Furthermore, it contains
partial models for continuous and discrete blocks.
</p>

</HTML>
",     revisions="<html>
<ul>
<li><i>Oct. 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Added several new interfaces. <a href=\"../Documentation/ChangeNotes1.5.html\">Detailed description</a> available.
<li><i>Oct. 24, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       RealInputSignal renamed to RealInput. RealOutputSignal renamed to
       output RealOutput. GraphBlock renamed to BlockIcon. SISOreal renamed to
       SISO. SOreal renamed to SO. I2SOreal renamed to M2SO.
       SignalGenerator renamed to SignalSource. Introduced the following
       new models: MIMO, MIMOs, SVcontrol, MVcontrol, DiscreteBlockIcon,
       DiscreteBlock, DiscreteSISO, DiscreteMIMO, DiscreteMIMOs,
       BooleanBlockIcon, BooleanSISO, BooleanSignalSource, MI2BooleanMOs.</li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>
"));
    end Interfaces;

    package Logical
      "Library of components with Boolean input and output signals"
        extends Modelica.Icons.Library;

      block GreaterThreshold
        "Output y is true, if input u is greater than threshold"
        extends Blocks.Interfaces.partialBooleanThresholdComparison;
      equation
        y = u > threshold;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,-40},{60,40}},
                lineColor={0,0,0},
                textString=">")}),
                            Diagram(coordinateSystem(preserveAspectRatio=true,
                extent={{-100,-100},{100,100}}),
                                    graphics),
          Documentation(info="<html>
<p>
The output is <b>true</b> if the Real input is greater than
parameter <b>threshold</b>, otherwise
the output is <b>false</b>.
</p>
</html>"));
      end GreaterThreshold;

      block GreaterEqualThreshold
        "Output y is true, if input u is greater or equal than threshold"

        extends Blocks.Interfaces.partialBooleanThresholdComparison;
      equation
        y = u >= threshold;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,-40},{60,40}},
                lineColor={0,0,0},
                textString=">=")}),
                                Diagram(coordinateSystem(preserveAspectRatio=true,
                extent={{-100,-100},{100,100}}),
                                        graphics),
          Documentation(info="<html>
<p>
The output is <b>true</b> if the Real input is greater than or equal to
parameter <b>threshold</b>, otherwise
the output is <b>false</b>.
</p>
</html>"));
      end GreaterEqualThreshold;

      block LessEqualThreshold
        "Output y is true, if input u is less or equal than threshold"
        extends Blocks.Interfaces.partialBooleanThresholdComparison;
      equation
        y = u <= threshold;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,-40},{60,40}},
                lineColor={0,0,0},
                textString="<=")}),
                                Diagram(coordinateSystem(preserveAspectRatio=true,
                extent={{-100,-100},{100,100}}),
                                        graphics),
          Documentation(info="<html>
<p>
The output is <b>true</b> if the Real input is less than or equal to
parameter <b>threshold</b>, otherwise
the output is <b>false</b>.
</p>
</html>"));
      end LessEqualThreshold;

      block LessEqual
        "Output y is true, if input u1 is less or equal as input u2"
        extends Blocks.Interfaces.partialBooleanComparison;


      equation
        y = u1 <= u2;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Ellipse(extent={{32,10},{52,-10}}, lineColor={0,0,255}),
              Line(points={{-100,-80},{42,-80},{42,0}}, color={0,0,255}),
              Text(
                extent={{-80,-60},{20,60}},
                lineColor={0,0,0},
                textString="<=")}),
            Documentation(info="<html>
<p>
The output is <b>true</b> if Real input u1 is less than or equal to
Real input u2, otherwise the output is <b>false</b>.
</p>
</html>"));
      end LessEqual;

      block Switch "Switch between two Real signals"
        extends Blocks.Interfaces.partialBooleanBlockIcon;
        Blocks.Interfaces.RealInput u1 "Connector of first Real input signal" 
                                       annotation (Placement(transformation(extent=
                  {{-140,60},{-100,100}}, rotation=0)));
        Blocks.Interfaces.BooleanInput u2 "Connector of Boolean input signal" 
                                          annotation (Placement(transformation(
                extent={{-140,-20},{-100,20}}, rotation=0)));
        Blocks.Interfaces.RealInput u3 "Connector of second Real input signal" 
                                       annotation (Placement(transformation(extent=
                  {{-140,-100},{-100,-60}}, rotation=0)));
        Blocks.Interfaces.RealOutput y "Connector of Real output signal" 
                                       annotation (Placement(transformation(extent=
                  {{100,-10},{120,10}}, rotation=0)));

      equation
        y = if u2 then u1 else u3;
        annotation (defaultComponentName="switch1",
          Documentation(info="<html>
<p>The Logical.Switch switches, depending on the
logical connector u2 (the middle connector)
between the two possible input signals
u1 (upper connector) and u3 (lower connector).</p>
<p>If u2 is <b>true</b>, the output signal y is set equal to
u1, else it is set equal to u3.</p>
</html>
"),       Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(
                points={{12,0},{100,0}},
                pattern=LinePattern.Solid,
                thickness=0.25,
                arrow={Arrow.None,Arrow.None},
                color={0,0,255}),
              Line(
                points={{-100,0},{-40,0}},
                color={255,0,127},
                pattern=LinePattern.Solid,
                thickness=0.25,
                arrow={Arrow.None,Arrow.None}),
              Line(
                points={{-100,-80},{-40,-80},{-40,-80}},
                pattern=LinePattern.Solid,
                thickness=0.25,
                arrow={Arrow.None,Arrow.None},
                color={0,0,255}),
              Line(points={{-40,12},{-40,-12}}, color={255,0,127}),
              Line(points={{-100,80},{-38,80}}, color={0,0,255}),
              Line(
                points={{-38,80},{6,2}},
                thickness=1,
                color={0,0,255}),
              Ellipse(
                extent={{2,8},{18,-6}},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid,
                lineColor={0,0,255})}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics));
      end Switch;
      annotation(Documentation(info="<html>
<p>
This package provides blocks with Boolean input and output signals
to describe logical networks. A typical example for a logical
network built with package Logical is shown in the next figure:
</p>
<p align=\"center\">
<img src=\"../Images/Blocks/LogicalNetwork1.png\">
</p>

<p>
The actual value of Boolean input and/or output signals is displayed
in the respective block icon as \"circle\", where \"white\" color means
value <b>false</b> and \"green\" color means value <b>true</b>. These
values are visualized in a diagram animation.
</p>
</html>"));
    end Logical;

    package Math "Library of mathematical functions as input/output blocks"
      import Modelica.SIunits;
      import Modelica.Blocks.Interfaces;
      extends Modelica.Icons.Library;

      block InverseBlockConstraints
        "Construct inverse model by requiring that two inputs and two outputs are identical (replaces the previously, unbalanced, TwoInputs and TwoOutputs blocks)"

        Modelica.Blocks.Interfaces.RealInput u1 "Input signal 1 (u1 = u2)" 
                                annotation (Placement(transformation(extent={{-240,
                  -20},{-200,20}}, rotation=0), iconTransformation(extent={{-240,-20},
                  {-200,20}})));
        Modelica.Blocks.Interfaces.RealInput u2 "Input signal 2 (u1 = u2)" 
                                annotation (Placement(transformation(extent={{-140,
                  -20},{-180,20}}, rotation=0), iconTransformation(extent={{-140,-20},
                  {-180,20}})));
        Modelica.Blocks.Interfaces.RealOutput y1 "Output signal 1 (y1 = y2)" 
                                 annotation (Placement(transformation(extent={{200,-10},
                  {220,10}},      rotation=0), iconTransformation(extent={{200,-10},{
                  220,10}})));
        Modelica.Blocks.Interfaces.RealOutput y2 "Output signal 2 (y2 = y2)" 
                                 annotation (Placement(transformation(extent={{10,-10},
                  {-10,10}},       rotation=0,
              origin={170,0}),  iconTransformation(extent={{180,-10},{160,10}})));

      equation
        u1 = u2;
        y1 = y2;
        annotation(__Dymola_structurallyIncomplete=true,
          Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-200,-120},{
                  200,120}}),
                  graphics),
          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-200,-120},{200,
                  120}}), graphics={
              Line(
                points={{180,0},{200,0}},
                color={0,0,127},
                smooth=Smooth.None),
              Line(
                points={{-200,0},{-180,0}},
                color={0,0,127},
                smooth=Smooth.None),
              Rectangle(extent={{-190,120},{190,-120}}, lineColor={135,135,135})}),
          Documentation(info="<html>
<p>
Exchange input and ouput signals of a block, i.e., the previous
block inputs become block outputs and the previous block outputs become
block inputs. This block is used to construct inverse models.
Its usage is demonstrated in example:
<a href=\"Modelica://Modelica.Blocks.Examples.InverseModel\">Modelica.Blocks.Examples.InverseModel</a>.
</p>

<p>
Note, if a block shall be inverted that has several input and output blocks,
then this can be easily achieved by using a vector of InverseBlockConstraints
instances:
</p>

<pre>
   InverseBlockConstraint invert[3];  // Block to be inverted has 3 input signals
</pre>
</html>"));
      end InverseBlockConstraints;

          block Gain "Output the product of a gain value with the input signal"

            parameter Real k(start=1) "Gain value multiplied with input signal";
      public
            Interfaces.RealInput u "Input signal connector" 
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
            Interfaces.RealOutput y "Output signal connector" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

          equation
            y = k*u;
            annotation (
              Documentation(info="
<HTML>
<p>
This block computes output <i>y</i> as
<i>product</i> of gain <i>k</i> with the
input <i>u</i>:
</p>
<pre>
    y = k * u;
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-140},{150,-100}},
                lineColor={0,0,0},
                textString="k=%k"),
              Text(
                extent={{-150,140},{150,100}},
                textString="%name",
                lineColor={0,0,255})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-76,38},{0,-34}},
                textString="k",
                lineColor={0,0,255})}));
          end Gain;

          block Sum "Output the sum of the elements of the input vector"
            extends Interfaces.MISO;
            parameter Real k[nin]=ones(nin) "Optional: sum coefficients";
          equation
            y = k*u;
            annotation (defaultComponentName="sum1",
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as
<i>sum</i> of the elements of the input signal vector
<b>u</b>:
</p>
<pre>
    <b>y</b> = <b>u</b>[1] + <b>u</b>[2] + ...;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   nin = 3;

  results in the following equations:

     y = u[1] + u[2] + u[3];
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Line(
                points={{26,42},{-34,42},{6,2},{-34,-38},{26,-38}},
                color={0,0,0},
                thickness=0.25), Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Line(
                points={{26,42},{-34,42},{6,2},{-34,-38},{26,-38}},
                color={0,0,0},
                thickness=0.25)}));
          end Sum;

          block Feedback
        "Output difference between commanded and feedback input"

            input Interfaces.RealInput u1 annotation (Placement(transformation(
                extent={{-100,-20},{-60,20}}, rotation=0)));
            input Interfaces.RealInput u2 
              annotation (Placement(transformation(
              origin={0,-80},
              extent={{-20,-20},{20,20}},
              rotation=90)));
            output Interfaces.RealOutput y annotation (Placement(transformation(
                extent={{80,-10},{100,10}}, rotation=0)));

          equation
            y = u1 - u2;
            annotation (
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as <i>difference</i> of the
commanded input <b>u1</b> and the feedback
input <b>u2</b>:
</p>
<pre>
    <b>y</b> = <b>u1</b> - <b>u2</b>;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   n = 2

  results in the following equations:

     y = u1 - u2
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Ellipse(
                extent={{-20,20},{20,-20}},
                lineColor={0,0,127},
                fillColor={235,235,235},
                fillPattern=FillPattern.Solid),
              Line(points={{-60,0},{-20,0}}, color={0,0,127}),
              Line(points={{20,0},{80,0}}, color={0,0,127}),
              Line(points={{0,-20},{0,-60}}, color={0,0,127}),
              Text(
                extent={{-14,0},{82,-94}},
                lineColor={0,0,0},
                textString="-"),
              Text(
                extent={{-150,94},{150,44}},
                textString="%name",
                lineColor={0,0,255})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Ellipse(
                extent={{-20,20},{20,-20}},
                pattern=LinePattern.Solid,
                lineThickness=0.25,
                fillColor={235,235,235},
                fillPattern=FillPattern.Solid,
                lineColor={0,0,255}),
              Line(points={{-60,0},{-20,0}}, color={0,0,255}),
              Line(points={{20,0},{80,0}}, color={0,0,255}),
              Line(points={{0,-20},{0,-60}}, color={0,0,255}),
              Text(
                extent={{-12,10},{84,-84}},
                lineColor={0,0,0},
                textString="-")}));
          end Feedback;

          block Add "Output the sum of the two inputs"
            extends Interfaces.SI2SO;
            parameter Real k1=+1 "Gain of upper input";
            parameter Real k2=+1 "Gain of lower input";

          equation
            y = k1*u1 + k2*u2;
            annotation (
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as <i>sum</i> of the
two input signals <b>u1</b> and <b>u2</b>:
</p>
<pre>
    <b>y</b> = k1*<b>u1</b> + k2*<b>u2</b>;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   k1= +2, k2= -3

  results in the following equations:

     y = 2 * u1 - 3 * u2
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Text(
                extent={{-98,-52},{7,-92}},
                lineColor={0,0,0},
                textString="%k2"),
              Text(
                extent={{-100,90},{5,50}},
                lineColor={0,0,0},
                textString="%k1"),
              Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255}),
              Line(points={{-100,60},{-40,60},{-30,40}}, color={0,0,255}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,255}),
              Line(points={{-100,-60},{-40,-60},{-30,-40}}, color={0,0,255}),
              Line(points={{-15,-25.99},{15,25.99}}, color={0,0,0}),
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{50,0},{100,0}}, color={0,0,255}),
              Line(points={{-100,60},{-74,24},{-44,24}}, color={0,0,127}),
              Line(points={{-100,-60},{-74,-28},{-42,-28}}, color={0,0,127}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,127}),
              Line(points={{50,0},{100,0}}, color={0,0,127}),
              Text(
                extent={{-38,34},{38,-34}},
                lineColor={0,0,0},
                textString="+"),
              Text(
                extent={{-100,52},{5,92}},
                lineColor={0,0,0},
                textString="%k1"),
              Text(
                extent={{-100,-52},{5,-92}},
                lineColor={0,0,0},
                textString="%k2")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-98,-52},{7,-92}},
                lineColor={0,0,0},
                textString="%k2"),
              Text(
                extent={{-100,90},{5,50}},
                lineColor={0,0,0},
                textString="%k1"),
              Line(points={{-100,60},{-40,60},{-30,40}}, color={0,0,255}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,255}),
              Line(points={{-100,-60},{-40,-60},{-30,-40}}, color={0,0,255}),
              Line(points={{-15,-25.99},{15,25.99}}, color={0,0,0}),
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{50,0},{100,0}}, color={0,0,255}),
              Line(points={{-100,60},{-74,24},{-44,24}}, color={0,0,127}),
              Line(points={{-100,-60},{-74,-28},{-42,-28}}, color={0,0,127}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,127}),
              Line(points={{50,0},{100,0}}, color={0,0,127}),
              Text(
                extent={{-38,34},{38,-34}},
                lineColor={0,0,0},
                textString="+"),
              Text(
                extent={{-100,52},{5,92}},
                lineColor={0,0,0},
                textString="k1"),
              Text(
                extent={{-100,-52},{5,-92}},
                lineColor={0,0,0},
                textString="k2")}));
          end Add;

          block Add3 "Output the sum of the three inputs"
            extends Interfaces.BlockIcon;

            parameter Real k1=+1 "Gain of upper input";
            parameter Real k2=+1 "Gain of middle input";
            parameter Real k3=+1 "Gain of lower input";
            input Interfaces.RealInput u1 "Connector 1 of Real input signals" 
              annotation (Placement(transformation(extent={{-140,60},{-100,100}},
                rotation=0)));
            input Interfaces.RealInput u2 "Connector 2 of Real input signals" 
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
            input Interfaces.RealInput u3 "Connector 3 of Real input signals" 
              annotation (Placement(transformation(extent={{-140,-100},{-100,-60}},
                rotation=0)));
            output Interfaces.RealOutput y "Connector of Real output signals" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

          equation
            y = k1*u1 + k2*u2 + k3*u3;
            annotation (
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as <i>sum</i> of the
three input signals <b>u1</b>, <b>u2</b> and <b>u3</b>:
</p>
<pre>
    <b>y</b> = k1*<b>u1</b> + k2*<b>u2</b> + k3*<b>u3</b>;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   k1= +2, k2= -3, k3=1;

  results in the following equations:

     y = 2 * u1 - 3 * u2 + u3;
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Text(
                extent={{-100,50},{5,90}},
                lineColor={0,0,0},
                textString="%k1"),
              Text(
                extent={{-100,-20},{5,20}},
                lineColor={0,0,0},
                textString="%k2"),
              Text(
                extent={{-100,-50},{5,-90}},
                lineColor={0,0,0},
                textString="%k3"),
              Text(
                extent={{2,36},{100,-44}},
                lineColor={0,0,0},
                textString="+")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-100,50},{5,90}},
                lineColor={0,0,0},
                textString="%k1"),
              Text(
                extent={{-100,-20},{5,20}},
                lineColor={0,0,0},
                textString="%k2"),
              Text(
                extent={{-100,-50},{5,-90}},
                lineColor={0,0,0},
                textString="%k3"),
              Text(
                extent={{2,36},{100,-44}},
                lineColor={0,0,0},
                textString="+"),
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-100,50},{5,90}},
                lineColor={0,0,0},
                textString="k1"),
              Text(
                extent={{-100,-20},{5,20}},
                lineColor={0,0,0},
                textString="k2"),
              Text(
                extent={{-100,-50},{5,-90}},
                lineColor={0,0,0},
                textString="k3"),
              Text(
                extent={{2,36},{100,-44}},
                lineColor={0,0,0},
                textString="+")}));
          end Add3;

          block Product "Output product of the two inputs"
            extends Interfaces.SI2SO;

          equation
            y = u1*u2;
            annotation (
              Documentation(info="
<HTML>
<p>
This blocks computes the output <b>y</b> (element-wise)
as <i>product</i> of the corresponding elements of
the two inputs <b>u1</b> and <b>u2</b>:
</p>
<pre>
    y = u1 * u2;
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-100,60},{-40,60},{-30,40}}, color={0,0,127}),
              Line(points={{-100,-60},{-40,-60},{-30,-40}}, color={0,0,127}),
              Line(points={{50,0},{100,0}}, color={0,0,127}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Line(points={{-15,25.99},{15,-25.99}}, color={0,0,0}),
              Line(points={{-15,-25.99},{15,25.99}}, color={0,0,0}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,127})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{-100,60},{-40,60},{-30,40}}, color={0,0,255}),
              Line(points={{-100,-60},{-40,-60},{-30,-40}}, color={0,0,255}),
              Line(points={{50,0},{100,0}}, color={0,0,255}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Line(points={{-15,25.99},{15,-25.99}}, color={0,0,0}),
              Line(points={{-15,-25.99},{15,25.99}}, color={0,0,0}),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,255})}));
          end Product;

          block Division "Output first input divided by second input"
            extends Interfaces.SI2SO;

          equation
            y = u1/u2;
            annotation (
              Documentation(info="
<HTML>
<p>
This block computes the output <b>y</b> (element-wise)
by <i>dividing</i> the corresponding elements of
the two inputs <b>u1</b> and <b>u2</b>:
</p>
<pre>
    y = u1 / u2;
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{50,0},{100,0}}, color={0,0,127}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Ellipse(
                extent={{-5,20},{5,30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-5,-20},{5,-30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,127}),
              Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255}),
              Line(points={{-100,60},{-66,60},{-40,30}}, color={0,0,127}),
              Line(points={{-100,-60},{0,-60},{0,-50}}, color={0,0,127})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{50,0},{100,0}}, color={0,0,255}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Ellipse(
                extent={{-5,20},{5,30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-5,-20},{5,-30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,255}),
              Line(points={{-100,60},{-66,60},{-40,30}}, color={0,0,255}),
              Line(points={{-100,-60},{0,-60},{0,-50}}, color={0,0,255})}));
          end Division;

          block Log10
        "Output the base 10 logarithm of the input (input > 0 required)"

            extends Interfaces.SISO;
          equation
            y = Modelica.Math.log10(u);
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Polygon(
                points={{90,0},{68,8},{68,-8},{90,0}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,0},{68,0}}, color={192,192,192}),
              Line(points={{-79.8,-80},{-79.2,-50.6},{-78.4,-37},{-77.6,-28},{-76.8,
                    -21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,
                    28},{-47,38.1},{-31.8,48.1},{-10.1,58},{22.1,68},{68.7,78.1},{
                    80,80}}, color={0,0,0}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,-80},{-80,68}}, color={192,192,192}),
              Text(
                extent={{-30,-22},{60,-70}},
                lineColor={192,192,192},
                textString="log10")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-80,80},{-88,80}}, color={192,192,192}),
              Line(points={{-80,-80},{-88,-80}}, color={192,192,192}),
              Line(points={{-80,-90},{-80,84}}, color={192,192,192}),
              Text(
                extent={{-65,96},{-38,78}},
                lineColor={160,160,164},
                textString="y"),
              Polygon(
                points={{-80,100},{-86,84},{-74,84},{-80,100}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-100,0},{84,0}}, color={192,192,192}),
              Polygon(
                points={{100,0},{84,6},{84,-6},{100,0}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-79.8,-80},{-79.2,-50.6},{-78.4,-37},{-77.6,-28},{-76.8,
                    -21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,
                    28},{-47,38.1},{-31.8,48.1},{-10.1,58},{22.1,68},{68.7,78.1},{
                    80,80}}, color={0,0,0}),
              Text(
                extent={{70,-3},{90,-23}},
                textString="20",
                lineColor={0,0,255}),
              Text(
                extent={{-78,-1},{-58,-21}},
                textString="1",
                lineColor={0,0,255}),
              Text(
                extent={{-109,72},{-89,88}},
                textString=" 1.3",
                lineColor={0,0,255}),
              Text(
                extent={{-109,-88},{-89,-72}},
                textString="-1.3",
                lineColor={0,0,255}),
              Text(
                extent={{62,30},{90,10}},
                lineColor={160,160,164},
                textString="u")}),
              Documentation(info="<HTML>
<p>
This blocks computes the output <b>y</b> as the
<i>base 10 logarithm</i> of the input <b>u</b>:
</p>
<pre>
    y = <b>log10</b>( u );
</pre>
<p>
An error occurs if the elements of the input <b>u</b> are
zero or negative.
</p>

<p>
<img src=\"../Images/Math/log10.png\">
</p>

</HTML>
"));
          end Log10;

      block Max "Pass through the largest signal"
        extends Interfaces.SI2SO;
      equation
        y = max(u1, u2);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,36},{90,-36}},
                lineColor={160,160,164},
                textString="max()")}),
                                Documentation(info="<html>
<p>
This block computes the output <b>y</b> as <i>maximum</i>
of the two Real inputs <b>u1</b> and <b>u2</b>:
</p>
<pre>    y = <b>max</b> ( u1 , u2 );
</pre>
</html>
"));
      end Max;

      block Min "Pass through the smallest signal"
        extends Interfaces.SI2SO;
      equation
         y = min(u1, u2);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,36},{90,-36}},
                lineColor={160,160,164},
                textString="min()")}),
                                Documentation(info="<html>
<p>
This block computes the output <b>y</b> as <i>minimum</i> of
the two Real inputs <b>u1</b> and <b>u2</b>:
</p>
<pre>    y = <b>min</b> ( u1 , u2 );
</pre>
</html>
"));
      end Min;
      annotation (
        Documentation(info="
<HTML>
<p>
This package contains basic <b>mathematical operations</b>,
such as summation and multiplication, and basic <b>mathematical
functions</b>, such as <b>sqrt</b> and <b>sin</b>, as
input/output blocks. All blocks of this library can be either
connected with continuous blocks or with sampled-data blocks.
</p>
</HTML>
",     revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       New blocks added: RealToInteger, IntegerToReal, Max, Min, Edge, BooleanChange, IntegerChange.</li>
<li><i>August 7, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized (partly based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist).
</li>
</ul>
</html>"));
    end Math;

    package Sources
      "Library of signal source blocks generating Real and Boolean signals"
      import Modelica.Blocks.Interfaces;
      import Modelica.SIunits;
          extends Modelica.Icons.Library;

          block Constant "Generate constant signal of type Real"
            parameter Real k(start=1) "Constant output value";
            extends Interfaces.SO;

          equation
            y = k;
            annotation (defaultComponentName="const",
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-80,68},{-80,-80}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,-70},{82,-70}}, color={192,192,192}),
              Polygon(
                points={{90,-70},{68,-62},{68,-78},{90,-70}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,0},{80,0}}, color={0,0,0}),
              Text(
                extent={{-150,-150},{150,-110}},
                lineColor={0,0,0},
                textString="k=%k")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Polygon(
                points={{-80,90},{-86,68},{-74,68},{-80,90}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,68},{-80,-80}}, color={95,95,95}),
              Line(
                points={{-80,0},{80,0}},
                color={0,0,255},
                thickness=0.5),
              Line(points={{-90,-70},{82,-70}}, color={95,95,95}),
              Polygon(
                points={{90,-70},{68,-64},{68,-76},{90,-70}},
                lineColor={95,95,95},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-83,92},{-30,74}},
                lineColor={0,0,0},
                textString="y"),
              Text(
                extent={{70,-80},{94,-100}},
                lineColor={0,0,0},
                textString="time"),
              Text(
                extent={{-101,8},{-81,-12}},
                lineColor={0,0,0},
                textString="k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>

<p>
<img src=\"../Images/Blocks/Sources/Constant.png\">
</p>
</html>"));
          end Constant;
          annotation (
            Documentation(info="<HTML>
<p>
This package contains <b>source</b> components, i.e., blocks which
have only output signals. These blocks are used as signal generators
for Real, Integer and Boolean signals.
</p>

<p>
All Real source signals (with the exception of the Constant source)
have at least the following two parameters:
</p>

<table border=1 cellspacing=0 cellpadding=2>
  <tr><td valign=\"top\"><b>offset</b></td>
      <td valign=\"top\">Value which is added to the signal</td>
  </tr>
  <tr><td valign=\"top\"><b>startTime</b></td>
      <td valign=\"top\">Start time of signal. For time &lt; startTime,
                the output y is set to offset.</td>
  </tr>
</table>

<p>
The <b>offset</b> parameter is especially useful in order to shift
the corresponding source, such that at initial time the system
is stationary. To determine the corresponding value of offset,
usually requires a trimming calculation.
</p>
</HTML>
",     revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Integer sources added. Step, TimeTable and BooleanStep slightly changed.</li>
<li><i>Nov. 8, 1999</i>
       by <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
       <A HREF=\"mailto:schneider@eas.iis.fhg.de\">schneider@eas.iis.fhg.de</A>,
       <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       New sources: Exponentials, TimeTable. Trapezoid slightly enhanced
       (nperiod=-1 is an infinite number of periods).</li>
<li><i>Oct. 31, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       <a href=\"mailto:clauss@eas.iis.fhg.de\">Christoph Clau&szlig;</a>,
       <A HREF=\"mailto:schneider@eas.iis.fhg.de\">schneider@eas.iis.fhg.de</A>,
       All sources vectorized. New sources: ExpSine, Trapezoid,
       BooleanConstant, BooleanStep, BooleanPulse, SampleTrigger.
       Improved documentation, especially detailed description of
       signals in diagram layer.</li>
<li><i>June 29, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>"));
    end Sources;

    package Types
      "Library of constants and types with choices, especially to build menus"
      extends Modelica.Icons.Library;

      type Init = enumeration(
          NoInit
            "No initialization (start values are used as guess values with fixed=false)", 

          SteadyState
            "Steady state initialization (derivatives of states are zero)",
          InitialState "Initialization with initial states",
          InitialOutput
            "Initialization with initial outputs (and steady state of the states if possibles)")
        "Enumeration defining initialization of a block" 
          annotation (Evaluate=true, Documentation(info="<html>

</html>"));

      annotation ( Documentation(info="<HTML>
<p>
In this package <b>types</b> and <b>constants</b> are defined that are used
in library Modelica.Blocks. The types have additional annotation choices
definitions that define the menus to be built up in the graphical
user interface when the type is used as parameter in a declaration.
</p>
</HTML>"));
    end Types;
  annotation (
    Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
        graphics={
        Rectangle(extent={{-32,-6},{16,-35}}, lineColor={0,0,0}),
        Rectangle(extent={{-32,-56},{16,-85}}, lineColor={0,0,0}),
        Line(points={{16,-20},{49,-20},{49,-71},{16,-71}}, color={0,0,0}),
        Line(points={{-32,-72},{-64,-72},{-64,-21},{-32,-21}}, color={0,0,0}),
        Polygon(
          points={{16,-71},{29,-67},{29,-74},{16,-71}},
          lineColor={0,0,0},
          fillColor={0,0,0},
          fillPattern=FillPattern.Solid),
        Polygon(
          points={{-32,-21},{-46,-17},{-46,-25},{-32,-21}},
          lineColor={0,0,0},
          fillColor={0,0,0},
          fillPattern=FillPattern.Solid)}),
                            Documentation(info="<html>
<p>
This library contains input/output blocks to build up block diagrams.
</p>

<dl>
<dt><b>Main Author:</b>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
    Oberpfaffenhofen<br>
    Postfach 1116<br>
    D-82230 Wessling<br>
    email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
</dl>
<p>
Copyright &copy; 1998-2009, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p>
</HTML>
",   revisions="<html>
<ul>
<li><i>June 23, 2004</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Introduced new block connectors and adapated all blocks to the new connectors.
       Included subpackages Continuous, Discrete, Logical, Nonlinear from
       package ModelicaAdditions.Blocks.
       Included subpackage ModelicaAdditions.Table in Modelica.Blocks.Sources
       and in the new package Modelica.Blocks.Tables.
       Added new blocks to Blocks.Sources and Blocks.Logical.
       </li>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       New subpackage Examples, additional components.
       </li>
<li><i>June 20, 2000</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a> and
       Michael Tiller:<br>
       Introduced a replaceable signal type into
       Blocks.Interfaces.RealInput/RealOutput:
<pre>
   replaceable type SignalType = Real
</pre>
       in order that the type of the signal of an input/output block
       can be changed to a physical type, for example:
<pre>
   Sine sin1(outPort(redeclare type SignalType=Modelica.SIunits.Torque))
</pre>
      </li>
<li><i>Sept. 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Renamed to Blocks. New subpackages Math, Nonlinear.
       Additional components in subpackages Interfaces, Continuous
       and Sources. </li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version, based on an existing Dymola library
       of Dieter Moormann and Hilding Elmqvist.</li>
</ul>
</html>"));
  end Blocks;

  package StateGraph
    "Library of hierarchical state machine components to model discrete event and reactive systems"
  extends Modelica.Icons.Library2;

  package Interfaces "Connectors and partial models"
    extends Modelica.Icons.Library;

    connector Step_in "Input port of a step"
      output Boolean occupied "true, if step is active" annotation (HideResult=true);
      input Boolean set "true, if transition fires and step is activated" 
        annotation (HideResult=true);
      annotation (
     Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
                points={{-100,100},{100,0},{-100,-100},{-100,100}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid)}),
                                              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
                points={{0,50},{100,0},{0,-50},{0,50}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid), Text(
                extent={{-141,100},{100,60}},
                lineColor={0,0,0},
                textString="%name")}),
          Documentation(info="<html>

</html>"));
    end Step_in;

    connector Step_out "Output port of a step"
      output Boolean available "true, if step is active" annotation (HideResult=true);

      input Boolean reset "true, if transition fires and step is deactivated" 
        annotation (HideResult=true);

      annotation (Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
                                Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Rectangle(
                extent={{-100,50},{0,-50}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-100,100},{186,58}},
                lineColor={0,0,0},
                textString="%name")}),
          Documentation(info="<html>

</html>"));
    end Step_out;

    connector Transition_in "Input port of a transition"
      input Boolean available
          "true, if step connected to the transition input is active" 
        annotation (HideResult=true);
      output Boolean reset
          "true, if transition fires and the step connected to the transition input is deactivated"
        annotation (HideResult=true);

      annotation (Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
                points={{-100,100},{100,0},{-100,-100},{-100,100}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid)}),
                                              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
                points={{0,50},{100,0},{0,-50},{0,50}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid), Text(
                extent={{-141,100},{100,60}},
                lineColor={0,0,0},
                textString="%name")}),
          Documentation(info="<html>

</html>"));
    end Transition_in;

    connector Transition_out "Output port of a transition"
      input Boolean occupied
          "true, if step connected to the transition output is active" 
        annotation (HideResult=true);
      output Boolean set
          "true, if transition fires and step connected to the transition output becomes active"
        annotation (HideResult=true);

      annotation (Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
                                Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Text(
                extent={{-100,100},{146,60}},
                lineColor={0,0,0},
                textString="%name"), Rectangle(
                extent={{-100,50},{0,-50}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
          Documentation(info="<html>

</html>"));
    end Transition_out;

    connector CompositeStepStatePort_in
        "Communication port between a CompositeStep and the ordinary steps within the CompositeStep (suspend/resume are inputs)"

      input Boolean suspend
          "= true, if suspend transition of CompositeStep fires";
      input Boolean resume
          "= true, if resume transition of CompositeStep fires";
      Real activeStepsDummy
          "Dummy variable in order that connector fulfills restriction of connector"
                                                                                  annotation(HideResult=true);
      flow Real activeSteps "Number of active steps in the CompositeStep";
      annotation(Documentation(info="<html>

</html>"));
    end CompositeStepStatePort_in;

    connector CompositeStepStatePort_out
        "Communication port between a CompositeStep and the ordinary steps within the CompositeStep (suspend/resume are outputs)"

      output Boolean suspend
          "= true, if suspend transition of CompositeStep fires";
      output Boolean resume
          "= true, if resume transition of CompositeStep fires";
      Real activeStepsDummy
          "Dummy variable in order that connector fulfills restriction of connector"
                                                                                   annotation(HideResult=true);
      flow Real activeSteps "Number of active steps in the CompositeStep";
      annotation (Documentation(info="<html>

</html>"));
    end CompositeStepStatePort_out;

    partial block PartialStep
        "Partial step with one input and one output transition port"

      parameter Integer nIn(min=0) = 1 "Number of input connections";
      parameter Integer nOut(min=0) = 1 "Number of output connections";

      /* localActive is introduced since component 'Step' has Boolean variable 'active'
     and component 'StepWithSignal' has connector instance 'active' defined
     and both components inherit from PartialStep
  */
      output Boolean localActive
          "= true if step is active, otherwise the step is not active" 
        annotation (HideResult=true);
      Interfaces.Step_in inPort[nIn] "Vector of step input connectors" 
        annotation (Placement(transformation(extent={{-120,10},{-100,-10}},
                rotation=0)));
      Interfaces.Step_out outPort[nOut] "Vector of step output connectors" 
        annotation (Placement(transformation(extent={{100,5},{110,-5}}, rotation=
                  0)));
      protected
      outer Interfaces.CompositeStepState stateGraphRoot;
      model OuterStatePort
        CompositeStepStatePort_in subgraphStatePort;
      end OuterStatePort;
      OuterStatePort outerStatePort;

      Boolean newActive "Value of active in the next iteration" 
        annotation (HideResult=true);
      Boolean oldActive "Value of active when CompositeStep was aborted";
    initial equation
      pre(newActive) = pre(localActive);
      pre(oldActive) = pre(localActive);
    equation
      connect(outerStatePort.subgraphStatePort, stateGraphRoot.subgraphStatePort);

      // Check that connections to the connector are correct
      for i in 1:nIn loop

      assert(cardinality(inPort[i]) <= 1,
             "Connector is connected to more than one transition (this is not allowed)");
      end for;

      for i in 1:nOut loop

      assert(cardinality(outPort[i]) <= 1,
             "Connector is connected to more than one transition (this is not allowed)");
      end for;

      // set active state
      localActive = pre(newActive);
      newActive = if outerStatePort.subgraphStatePort.resume then 
                       oldActive else 
                       ( StateGraph.Temporary.anyTrue(inPort.set) or 
                            localActive
                         and not StateGraph.Temporary.anyTrue(outPort.reset))
                       and not outerStatePort.subgraphStatePort.suspend;

      // Remember state for suspend action
      when outerStatePort.subgraphStatePort.suspend then
        oldActive = localActive;
      end when;

      // Report state to CompositeStep
      outerStatePort.subgraphStatePort.activeSteps = if localActive then 1.0 else 0.0;

      // Report state to input and output transitions
      for i in 1:nIn loop
        inPort[i].occupied = if i == 1 then localActive else 
                                            inPort[i-1].occupied or 
                                            inPort[i-1].set;
      end for;

      for i in 1:nOut loop
         outPort[i].available = if i == 1 then localActive else 
                                               outPort[i-1].available and not 
                                               outPort[i-1].reset;
      end for;

      // Default setting, if an inPort or an outPort is not connected
      for i in 1:nIn loop
        if cardinality(inPort[i]) == 0 then
          inPort[i].set = false;
        end if;
      end for;

      for i in 1:nOut loop
        if cardinality(outPort[i]) == 0 then
          outPort[i].reset = false;
        end if;
      end for;
      annotation(Documentation(info="<html>

</html>"),       Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                         graphics),
        Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                  100}}),
             graphics),
        Coordsys(grid=[1,1], component=[20,20]));
    end PartialStep;

    partial block PartialTransition
        "Partial transition with input and output connections"
      input Boolean localCondition "= true, if transition may fire" 
        annotation (HideResult=true);
      parameter Boolean enableTimer=false "= true, if timer is enabled" 
        annotation (Evaluate=true, Dialog(group="Timer"));
      parameter Modelica.SIunits.Time waitTime(min=0) = 0
          "Wait time before transition fires" 
        annotation (Dialog(group="Timer", enable=enableTimer));
      output Modelica.SIunits.Time t
          "= actual waiting time (transition will fire when t > waitTime)";
      output Boolean enableFire "= true, if all firing conditions are true";
      output Boolean fire "= true, if transition fires" annotation (HideResult=true);

      StateGraph.Interfaces.Transition_in inPort
          "Vector of transition input connectors" 
        annotation (Placement(transformation(extent={{-50,-10},{-30,10}},
                rotation=0)));
      StateGraph.Interfaces.Transition_out outPort
          "Vector of transition output connectors" 
        annotation (Placement(transformation(extent={{10,-5},{20,5}}, rotation=0)));
      protected
      Modelica.SIunits.Time t_start
          "Time instant at which the transition would fire, if waitTime would be zero";
      Real t_dummy;
    initial equation

      pre(enableFire) = false;
    equation
      assert(cardinality(inPort) == 1,
        "Connector inPort is not connected to exactly one other connector");
      assert(cardinality(outPort) == 1,
        "Connector outPort is not connected to exactly one other connector");

      // Handling of Timer
      if enableTimer then
        when enableFire then
          t_start = time;
        end when;
        t_dummy = time - t_start;
        t = if enableFire then t_dummy else 0;
        fire = enableFire and time >= t_start + waitTime;
      else
        t_start = 0;
        t_dummy = 0;
        t = 0;
        fire = enableFire;
      end if;

      // Determine fire setting and report it to the connected step
      enableFire = localCondition and inPort.available and not outPort.occupied;
      inPort.reset = fire;
      outPort.set = fire;
      annotation(Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics),
        Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics),
        Documentation(info="<html>

</html>"));
    end PartialTransition;

    model CompositeStepState
        "Communication channel between CompositeSteps and steps in the CompositeStep"

      output Boolean suspend = false;
      output Boolean resume =  false;
      CompositeStepStatePort_out subgraphStatePort;

    /*
    missingInnerMessage="No \"stateGraphRoot\" component is defined on highest level
of the StateGraph. A stateGraphRoot component is automatically introduced.
In order to get rid of this message, drag StateGraph.StateGraphRoot into the
top level your model.");
*/
    equation
      suspend = subgraphStatePort.suspend;
      resume  = subgraphStatePort.resume;
      subgraphStatePort.activeStepsDummy = 0;
      annotation (
        defaultComponentName="stateGraphRoot",
        defaultComponentPrefixes="inner",
        missingInnerMessage="A \"stateGraphRoot\" component was automatically introduced.",
          Documentation(info="<html>

</html>"));
    end CompositeStepState;
      annotation (Documentation(info="<html>

</html>"));
  end Interfaces;

  block InitialStepWithSignal
      "Initial step (= step that is active when simulation starts). Connector 'active' is true when the step is active"

    extends Interfaces.PartialStep;


    Modelica.Blocks.Interfaces.BooleanOutput active 
      annotation (Placement(transformation(
            origin={0,-110},
            extent={{-10,-10},{10,10}},
            rotation=270)));
  initial equation
    active = true;
  equation
    active = localActive;
    annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}), graphics={Rectangle(extent={{-100,100},{100,-100}},
                lineColor={0,0,0}), Rectangle(extent={{-80,80},{80,-80}},
                lineColor={0,0,0})}),
         Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                100}}), graphics={
            Text(
              extent={{-200,110},{200,150}},
              textString="%name",
              lineColor={0,0,255}),
            Rectangle(
              extent={{-100,100},{100,-100}},
              fillColor=DynamicSelect({255,255,255}, if active > 0.5 then {0,255,
                  0} else {255,255,255}),
              lineColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-92,-50},{94,-68}},
              lineColor={0,0,0},
              textString="active"),
            Rectangle(extent={{-80,80},{80,-80}}, lineColor={0,0,0})}),
        Documentation(info="<html>

</html>"));
  end InitialStepWithSignal;

  block StepWithSignal
      "Ordinary step (= step that is not active when simulation starts). Connector 'active' is true when the step is active"

    extends Interfaces.PartialStep;


    Modelica.Blocks.Interfaces.BooleanOutput active 
      annotation (Placement(transformation(
            origin={0,-110},
            extent={{-10,-10},{10,10}},
            rotation=270)));
  initial equation
    active = false;
  equation
    active = localActive;
    annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}), graphics={Rectangle(extent={{-100,100},{100,-100}},
                lineColor={0,0,0})}),
         Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                100}}), graphics={
            Text(
              extent={{-200,110},{200,150}},
              textString="%name",
              lineColor={0,0,255}),
            Rectangle(
              extent={{-100,100},{100,-100}},
              fillColor=DynamicSelect({255,255,255}, if active > 0.5 then {0,255,
                  0} else {255,255,255}),
              lineColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-92,-74},{94,-92}},
              lineColor={0,0,0},
              textString="active")}),
        Documentation(info="<html>

</html>"));
  end StepWithSignal;

  block TransitionWithSignal
      "Transition where the fire condition is set by a Boolean input signal"

    Modelica.Blocks.Interfaces.BooleanInput condition 
      annotation (Placement(transformation(
            origin={0,-120},
            extent={{-20,-20},{20,20}},
            rotation=90)));

    extends Interfaces.PartialTransition(final localCondition=condition);

    annotation (
      Icon(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={
            Text(
              extent={{20,20},{200,45}},
              lineColor=DynamicSelect({0,0,0}, if enableTimer < 0.5 then {255,255,
                  255} else {0,0,0}),
              textString="%waitTime"),
            Rectangle(
              extent={{-10,100},{10,-100}},
              fillColor=DynamicSelect({0,0,0}, if enableFire > 0.5 then {0,255,0} else 
                        {0,0,0}),
              lineColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Line(points={{-30,0},{-10,0}}, color={0,0,0}),
            Text(
              extent={{200,110},{-200,150}},
              textString="%name",
              lineColor={0,0,255}),
            Ellipse(
              extent={{7,-81},{-7,-95}},
              lineColor=DynamicSelect({0,0,0}, if condition > 0.5 then {0,255,0} else 
                        {0,0,0}),
              fillColor=DynamicSelect({0,0,0}, if condition > 0.5 then {0,255,0} else 
                        {0,0,0}),
              fillPattern=FillPattern.Solid)}),
      Diagram(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Line(points={{-31,0},{-11,0}}, color={0,0,0}),
              Rectangle(
              extent={{-10,100},{10,-100}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid)}),
        Documentation(info="<html>

</html>"));

  end TransitionWithSignal;

  package Temporary
      "Components that will be provided by other libraries in the future"
    extends Modelica.Icons.Library;

    function anyTrue
        "Returns true, if at least on element of the Boolean input vector is true"

      extends Modelica.Icons.Function;
      input Boolean b[:];
      output Boolean result;
    algorithm
      result := false;
      for i in 1:size(b,1) loop
         result := result or b[i];
      end for;
        annotation (Documentation(info="<html>

</html>"));
    end anyTrue;
    annotation (Documentation(info="<html>
<p>
This library is just temporarily present. The components of
this library will be present in the future in the Modelica
standard library (with the new block connectors) and in the
UserInteraction library that is currently under development.
</p>
</html>"));
  end Temporary;
  annotation (
    Documentation(info="<html>
<p>
Library <b>StateGraph</b> is a <b>free</b> Modelica package providing
components to model <b>discrete event</b> and <b>reactive</b>
systems in a convenient
way. It is based on the JGraphChart method and
takes advantage of Modelica features for
the \"action\" language. JGraphChart is a further development of
Grafcet to include elements of StateCharts that are not present
in Grafcet/Sequential Function Charts. Therefore, the StateGraph
library has a similar modeling power as StateCharts but avoids
some deficiences of StateCharts.
</p>
<p>
For an introduction, have especially a look at:
</p>
<ul>
<li> <a href=\"Modelica://Modelica.StateGraph.UsersGuide\">StateGraph.UsersGuide</a>
     discusses the most important aspects how to use this library.</li>
<li> <a href=\"Modelica://Modelica.StateGraph.Examples\">StateGraph.Examples</a>
     contains examples that demonstrate the usage of this library.</li>
</ul>
<p>
A typical model generated with this library is shown
in the next figure where on the left hand side a two-tank
system with a tank controller and on the right hand side the
top-level part of the tank controller as a StateGraph is shown:
</p>
<p>
<img src=\"../Images/StateGraph/Examples/ControlledTanks1_small.png\">
<img src=\"../Images/StateGraph/Examples/ControlledTanks2_small.png\">
</p>
<p>
The unique feature of the StateGraph library with respect to JGraphCharts,
Grafcet, Sequential Function Charts, and StateCharts, is Modelica's
\"single assignment rule\" that requires that every variable is defined
by exactly one equation. This leads to a different \"action\" definition
as in these formalisms. The advantage is that the translator can either
determine a useful evaluation sequence by equation sorting or
reports an error if this is not possible, e.g., because a model
would lead to a non-determinism or to a dead-lock. As a side effect,
this leads also to simpler and more easier to understand models and
global variables are no longer needed (whereas in JGraphCharts,
Grafcet, Sequential Function Charts and StateCharts global variables
are nearly always needed).
</p>
<p>
The StateGraph library is currently available in a beta release.
The available components will most likely not be changed for the
release version. It is planned to improve the convenience of
building models with the StateGraph library for the release version
(this may require to introduce some additional annotations).
It is planned to include the StateGraph library in the
Modelica standard library.
It is most useful to combine this libray with the Modelica libraries
</p>
<ul>
<li><b>Modelica.Blocks.Logical</b> that provides
    components available in PLCs (programmable logic controllers). </li>
<li><b>UserInteraction</b> that provides components to
    interactively communicate with models in a running simulation.</li>
</ul>

<p>
Copyright &copy; 1998-2009, Modelica Association and DLR
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p><br>

</HTML>
"),   Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Rectangle(extent={{-88,-20},{-50,-54}}, lineColor={0,0,0}),
          Line(points={{-50,-38},{-24,-38}}, color={0,0,0}),
          Polygon(
            points={{-24,-32},{-12,-38},{-24,-44},{-24,-32}},
            lineColor={0,0,0},
            fillColor={0,0,0},
            fillPattern=FillPattern.Solid),
          Line(points={{-12,-6},{-12,-76}}, color={0,0,0}),
          Line(points={{-12,-38},{14,-38}}, color={0,0,0}),
          Polygon(
            points={{14,-32},{26,-38},{14,-44},{14,-32}},
            lineColor={0,0,0},
            fillColor={0,0,0},
            fillPattern=FillPattern.Solid),
          Rectangle(extent={{26,-22},{64,-56}}, lineColor={0,0,0})}));
  end StateGraph;

  package Math
    "Library of mathematical functions (e.g., sin, cos) and of functions operating on vectors and matrices"
  import SI = Modelica.SIunits;
  extends Modelica.Icons.Library2;

  function asin "Inverse sine (-1 <= u <= 1)"
    extends baseIcon2;
    input Real u;
    output SI.Angle y;


  external "C" y=  asin(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}},
          grid={2,2}), graphics={
          Line(points={{-90,0},{68,0}}, color={192,192,192}),
          Polygon(
            points={{90,0},{68,8},{68,-8},{90,0}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,-80},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,
                -49.8},{-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,
                52.7},{75.2,62.2},{77.6,67.5},{80,80}}, color={0,0,0}),
          Text(
            extent={{-88,78},{-16,30}},
            lineColor={192,192,192},
            textString="asin")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}},
          grid={2,2}), graphics={
          Text(
            extent={{-40,-72},{-15,-88}},
            textString="-pi/2",
            lineColor={0,0,255}),
          Text(
            extent={{-38,88},{-13,72}},
            textString=" pi/2",
            lineColor={0,0,255}),
          Text(
            extent={{68,-9},{88,-29}},
            textString="+1",
            lineColor={0,0,255}),
          Text(
            extent={{-90,21},{-70,1}},
            textString="-1",
            lineColor={0,0,255}),
          Line(points={{-100,0},{84,0}}, color={95,95,95}),
          Polygon(
            points={{98,0},{82,6},{82,-6},{98,0}},
            lineColor={95,95,95},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Line(
            points={{-80,-80},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,-49.8},
                {-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,52.7},{
                75.2,62.2},{77.6,67.5},{80,80}},
            color={0,0,255},
            thickness=0.5),
          Text(
            extent={{82,24},{102,4}},
            lineColor={95,95,95},
            textString="u"),
          Line(
            points={{0,80},{86,80}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{80,86},{80,-10}},
            color={175,175,175},
            smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = asin(u), with -1 &le; u &le; +1:
</p>

<p>
<img src=\"../Images/Math/asin.png\">
</p>
</html>"),   Library="ModelicaExternalC");
  end asin;

  function log10 "Base 10 logarithm (u shall be > 0)"
    extends baseIcon1;
    input Real u;
    output Real y;


  external "C" y=  log10(u);
    annotation (
      Icon(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}},
          grid={2,2}), graphics={
          Line(points={{-90,0},{68,0}}, color={192,192,192}),
          Polygon(
            points={{90,0},{68,8},{68,-8},{90,0}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Line(points={{-79.8,-80},{-79.2,-50.6},{-78.4,-37},{-77.6,-28},{-76.8,-21.3},
                {-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28},
                {-47,38.1},{-31.8,48.1},{-10.1,58},{22.1,68},{68.7,78.1},{80,80}},
              color={0,0,0}),
          Text(
            extent={{-30,-22},{60,-70}},
            lineColor={192,192,192},
            textString="log10")}),
      Diagram(coordinateSystem(
          preserveAspectRatio=true,
          extent={{-100,-100},{100,100}},
          grid={2,2}), graphics={
          Line(points={{-100,0},{84,0}}, color={95,95,95}),
          Polygon(
            points={{98,0},{82,6},{82,-6},{98,0}},
            lineColor={95,95,95},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid),
          Line(
            points={{-77.8,-80},{-77.2,-50.6},{-76.4,-37},{-75.6,-28},{-74.8,-21.3},
                {-73.2,-11.4},{-70.8,-1.31},{-67.5,8.08},{-62.7,17.9},{-55.5,28},
                {-45,38.1},{-29.8,48.1},{-8.1,58},{24.1,68},{70.7,78.1},{82,80}},
            color={0,0,255},
            thickness=0.5),
          Text(
            extent={{66,-13},{86,-33}},
            textString="20",
            lineColor={0,0,255}),
          Text(
            extent={{-78,-1},{-58,-21}},
            textString="1",
            lineColor={0,0,255}),
          Text(
            extent={{-83,62},{-63,78}},
            textString=" 1.3",
            lineColor={0,0,255}),
          Text(
            extent={{80,24},{100,4}},
            lineColor={95,95,95},
            textString="u"),
          Text(
            extent={{-100,9},{-80,-11}},
            textString="0",
            lineColor={0,0,255}),
          Line(
            points={{-80,80},{86,80}},
            color={175,175,175},
            smooth=Smooth.None),
          Line(
            points={{80,92},{80,-12}},
            color={175,175,175},
            smooth=Smooth.None)}),
      Documentation(info="<html>
<p>
This function returns y = log10(u),
with u &gt; 0:
</p>

<p>
<img src=\"../Images/Math/log10.png\">
</p>
</html>"),   Library="ModelicaExternalC");
  end log10;

  partial function baseIcon1
      "Basic icon for mathematical function with y-axis on left side"

    annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
              -100},{100,100}}), graphics={
          Rectangle(
            extent={{-100,100},{100,-100}},
            lineColor={0,0,0},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Line(points={{-80,-80},{-80,68}}, color={192,192,192}),
          Polygon(
            points={{-80,90},{-88,68},{-72,68},{-80,90}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Text(
            extent={{-150,150},{150,110}},
            textString="%name",
            lineColor={0,0,255})}),                          Diagram(
          coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Line(points={{-80,80},{-88,80}}, color={95,95,95}),
          Line(points={{-80,-80},{-88,-80}}, color={95,95,95}),
          Line(points={{-80,-90},{-80,84}}, color={95,95,95}),
          Text(
            extent={{-75,104},{-55,84}},
            lineColor={95,95,95},
            textString="y"),
          Polygon(
            points={{-80,98},{-86,82},{-74,82},{-80,98}},
            lineColor={95,95,95},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid)}),
      Documentation(info="<html>
<p>
Icon for a mathematical function, consisting of an y-axis on the left side.
It is expected, that an x-axis is added and a plot of the function.
</p>
</html>"));
  end baseIcon1;

  partial function baseIcon2
      "Basic icon for mathematical function with y-axis in middle"

    annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
              -100},{100,100}}), graphics={
          Rectangle(
            extent={{-100,100},{100,-100}},
            lineColor={0,0,0},
            fillColor={255,255,255},
            fillPattern=FillPattern.Solid),
          Line(points={{0,-80},{0,68}}, color={192,192,192}),
          Polygon(
            points={{0,90},{-8,68},{8,68},{0,90}},
            lineColor={192,192,192},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Text(
            extent={{-150,150},{150,110}},
            textString="%name",
            lineColor={0,0,255})}),                          Diagram(graphics={
          Line(points={{0,80},{-8,80}}, color={95,95,95}),
          Line(points={{0,-80},{-8,-80}}, color={95,95,95}),
          Line(points={{0,-90},{0,84}}, color={95,95,95}),
          Text(
            extent={{5,104},{25,84}},
            lineColor={95,95,95},
            textString="y"),
          Polygon(
            points={{0,98},{-6,82},{6,82},{0,98}},
            lineColor={95,95,95},
            fillColor={95,95,95},
            fillPattern=FillPattern.Solid)}),
      Documentation(revisions="<html>
<p>
Icon for a mathematical function, consisting of an y-axis in the middle.
It is expected, that an x-axis is added and a plot of the function.
</p>
</html>"));
  end baseIcon2;
  annotation (
    Invisible=true,
    Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
        graphics={Text(
          extent={{-59,-9},{42,-56}},
          lineColor={0,0,0},
          textString="f(x)")}),
    Documentation(info="<HTML>
<p>
This package contains <b>basic mathematical functions</b> (such as sin(..)),
as well as functions operating on <b>vectors</b> and <b>matrices</b>.
</p>

<dl>
<dt><b>Main Author:</b>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e.V. (DLR)<br>
    Institut f&uuml;r Robotik und Mechatronik<br>
    Postfach 1116<br>
    D-82230 Wessling<br>
    Germany<br>
    email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
</dl>

<p>
Copyright &copy; 1998-2009, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p><br>
</HTML>
",   revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Function tempInterpol2 added.</li>
<li><i>Oct. 24, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Icons for icon and diagram level introduced.</li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized.</li>
</ul>

</html>"));
  end Math;

  package Utilities
    "Library of utility functions dedicated to scripting (operating on files, streams, strings, system)"
    extends Modelica.Icons.Library;

    /*######
    package Streams "Read from files and write to files"
      extends Modelica.Icons.Library;
      annotation (
        Documentation(info="<HTML>
<h4>Library content</h4>
<p>
Package <b>Streams</b> contains functions to input and output strings
to a message window or on files. Note that a string is interpreted
and displayed as html text (e.g., with print(..) or error(..))
if it is enclosed with the Modelica html quotation, e.g.,
</p>
<center>
string = \"&lt;html&gt; first line &lt;br&gt; second line &lt;/html&gt;\".
</center>
<p>
It is a quality of implementation, whether (a) all tags of html are supported
or only a subset, (b) how html tags are interpreted if the output device
does not allow to display formatted text.
</p>
<p>
In the table below an example call to every function is given:
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><th><b><i>Function/type</i></b></th><th><b><i>Description</i></b></th></tr>
  <tr><td valign=\"top\"><a href=\"Modelica://Modelica.Utilities.Streams.print\">print</a>(string)<br>
          <a href=\"Modelica://Modelica.Utilities.Streams.print\">print</a>(string,fileName)</td>
      <td valign=\"top\"> Print string \"string\" or vector of strings to message window or on
           file \"fileName\".</td>
  </tr>
  <tr><td valign=\"top\">stringVector =
         <a href=\"Modelica://Modelica.Utilities.Streams.readFile\">readFile</a>(fileName)</td>
      <td valign=\"top\"> Read complete text file and return it as a vector of strings.</td>
  </tr>
  <tr><td valign=\"top\">(string, endOfFile) =
         <a href=\"Modelica://Modelica.Utilities.Streams.readLine\">readLine</a>(fileName, lineNumber)</td>
      <td valign=\"top\">Returns from the file the content of line lineNumber.</td>
  </tr>
  <tr><td valign=\"top\">lines =
         <a href=\"Modelica://Modelica.Utilities.Streams.countLines\">countLines</a>(fileName)</td>
      <td valign=\"top\">Returns the number of lines in a file.</td>
  </tr>
  <tr><td valign=\"top\"><a href=\"Modelica://Modelica.Utilities.Streams.error\">error</a>(string)</td>
      <td valign=\"top\"> Print error message \"string\" to message window
           and cancel all actions</td>
  </tr>
  <tr><td valign=\"top\"><a href=\"Modelica://Modelica.Utilities.Streams.close\">close</a>(fileName)</td>
      <td valign=\"top\"> Close file if it is still open. Ignore call if
           file is already closed or does not exist. </td>
  </tr>
</table>
<p>
Use functions <b>scanXXX</b> from package
<a href=\"Modelica://Modelica.Utilities.Strings\">Strings</a>
to parse a string.
</p>
<p>
If Real, Integer or Boolean values shall be printed
or used in an error message, they have to be first converted
to strings with the builtin operator
<a href=\"Modelica://ModelicaReference.Operators.string\">String</a>(...).
Example:
</p>
<pre>
  <b>if</b> x &lt; 0 <b>or</b> x &gt; 1 <b>then</b>
     Streams.error(\"x (= \" + String(x) + \") has to be in the range 0 .. 1\");
  <b>end if</b>;
</pre>
</HTML>
"));
*/

    /* ########
      function readLine
      "Reads a line of text from a file and returns it in a string"
        extends Modelica.Icons.Function;
        input String fileName "Name of the file that shall be read" 
                            annotation(Dialog(__Dymola_loadSelector(filter="Text files (*.txt)",
                            caption="Open text file for reading")));
        input Integer lineNumber(min=1) "Number of line to read";
        output String string "Line of text";
        output Boolean endOfFile
        "If true, end-of-file was reached when trying to read line";
       external "C" string = ModelicaInternal_readLine(fileName,lineNumber,endOfFile);
        annotation (Library="ModelicaExternalC",Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
(string, endOfFile) = Streams.<b>readLine</b>(fileName, lineNumber)
</pre></blockquote>
<h4>Description</h4>
<p>
Function <b>readLine</b>(..) opens the given file, reads enough of the
content to get the requested line, and returns the line as a string.
Lines are separated by LF or CR-LF; the returned string does not
contain the line separator. The file might remain open after
the call.
</p>
<p>
If lineNumber > countLines(fileName), an empty string is returned
and endOfFile=true. Otherwise endOfFile=false.
</p>
</html>"));
      end readLine;
*/
    /* ###
      function error "Print error message and cancel all actions"
        extends Modelica.Icons.Function;
        input String string "String to be printed to error message window";
        external "C" ModelicaError(string);
        annotation (Library="ModelicaExternalC",
      Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
Streams.<b>error</b>(string);
</pre></blockquote>
<h4>Description</h4>
<p>
Print the string \"string\" as error message and
cancel all actions. Line breaks are characterized
by \"\\n\" in the string.
</p>
<h4>Example</h4>
<blockquote><pre>
  Streams.error(\"x (= \" + String(x) + \")\\nhas to be in the range 0 .. 1\");
</pre></blockquote>
<h4>See also</h4>
<p>
<a href=\"Modelica://Modelica.Utilities.Streams\">Streams</a>,
<a href=\"Modelica://Modelica.Utilities.Streams.print\">Streams.print</a>,
<a href=\"Modelica://ModelicaReference.Operators.string\">String</a>
</p>
</html>"));
*/

    /* ###
      end error;
    end Streams;
*/
    package Strings "Operations on strings"

      function length "Returns length of string"
        extends Modelica.Icons.Function;
        input String string;
        output Integer result "Number of characters of string";
      external "C" result = ModelicaStrings_length(string);
        annotation (Library="ModelicaExternalC", Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
Strings.<b>length</b>(string);
</pre></blockquote>
<h4>Description</h4>
<p>
Returns the number of characters of \"string\".
</p>
</html>"));
      end length;

      function substring "Returns a substring defined by start and end index"

        extends Modelica.Icons.Function;
        input String string "String from which a substring is inquired";
        input Integer startIndex(min=1)
          "Character position of substring begin (index=1 is first character in string)";
        input Integer endIndex(min=1) "Character position of substring end";
        output String result
          "String containing substring string[startIndex:endIndex]";
      external "C" result = 
                          ModelicaStrings_substring(string,startIndex,endIndex);
        annotation (Library="ModelicaExternalC",
      Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
string2 = Strings.<b>substring</b>(string, startIndex, endIndex);
</pre></blockquote>
<h4>Description</h4>
<p>
This function returns
the substring from position startIndex
up to and including position endIndex of \"string\" .
</p>
<p>
If index, startIndex, or endIndex are not correct, e.g.,
if endIndex &gt; length(string), an assert is triggered.
</p>
<h4>Example</h4>
<blockquote><pre>
  string1 := \"This is line 111\";
  string2 := Strings.substring(string1,9,12); // string2 = \"line\"
</pre></blockquote>
</html>"));
      end substring;

      function repeat "Repeat a string n times"
        extends Modelica.Icons.Function;
        input Integer n(min=0) = 1 "Number of occurences";
        input String string=" " "String that is repeated";
        output String repeatedString "String containing n concatenated strings";
      algorithm
        repeatedString :="";
        for i in 1:n loop
           repeatedString := repeatedString + string;
        end for;
        annotation (
      Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
string2 = Strings.<b>repeat</b>(n);
string2 = Strings.<b>repeat</b>(n, string=\" \");
</pre></blockquote>
<h4>Description</h4>
<p>
The first form returns a string consisting of n blanks.
</p>
<p>
The second form returns a string consisting of n substrings
defined by the optional argument \"string\".
</p>
</html>"));
      end repeat;
      extends Modelica.Icons.Library;

      function scanReal
        "Scan for the next Real number and trigger an assert if not present"
        extends Modelica.Icons.Function;
        input String string "String to be scanned";
        input Integer startIndex(min=1)=1
          "Start scanning of string at character startIndex";
        input Boolean unsigned=false
          "= true, if Real token shall not start with a sign";
        input String message=""
          "Message used in error message if scan is not successful";
        output Real number "Value of real number";
        output Integer nextIndex "index of character after the found number";
      algorithm
        (nextIndex, number) :=Advanced.scanReal(string, startIndex, unsigned);
        if nextIndex == startIndex then
           nextIndex :=Advanced.skipWhiteSpace(string, startIndex);
           if unsigned then
              syntaxError(string, nextIndex, "Expected a Real number without a sign " + message);
           else
              syntaxError(string, nextIndex, "Expected a Real number " + message);
           end if;
        end if;
        annotation (Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
             number = Strings.<b>scanReal</b>(string);
(number, nextIndex) = Strings.<b>scanReal</b>(string, startIndex=1,
                                            unsigned=false, message=\"\");
</pre></blockquote>
<h4>Description</h4>
<p>
The first form, \"scanReal(string)\", scans \"string\" for a
Real number with leading white space and returns the value.
</p>
<p>
The second form, \"scanReal(string,startIndex,unsigned)\",
scans the string starting at index
\"startIndex\", checks whether the next token is a Real literal
and returns its value as a Real number, as well as the
index directly after the Real number.
If the optional argument \"unsigned\" is <b>true</b>,
the real number shall not have a leading \"+\" or \"-\" sign.
</p>
<p>
If the required Real number with leading white space
is not present in \"string\",  an assert is triggered.
</p>
</html>"));
      end scanReal;

      function syntaxError
        "Print an error message, a string and the index at which scanning detected an error"
        extends Modelica.Icons.Function;
        input String string "String that has an error at position index";
        input Integer index
          "Index of string at which scanning detected an error";
        input String message="" "String printed at end of error message";

      protected
        Integer maxIndex = 40;
        Integer maxLenString = 60;
        Integer lenString = length(string);
        String errString;
        Integer index2 = if index < 1 then 1 else if index > lenString then lenString else index;
      algorithm
      // if "string" is too long, skip part of the string when printing it
         if index2 <= maxIndex then
           errString := string;
         else
           errString := "... " + substring(string, index2-maxIndex, lenString);
           index2 := maxIndex + 5; // To mark right position
         end if;

         if length(errString) > maxLenString then
            errString := substring(errString, 1, maxLenString) + " ...";
         end if;

      // Print error message
       /*####
         Streams.error("Syntax error at character " + String(index) + " of\n" +
                       errString + "\n" +
                       repeat(index2-1, " ") + "*" + "\n" +
                       message);
       */
        annotation (Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
Strings.<b>syntaxError</b>(string, index, message);
</pre></blockquote>
<h4>Description</h4>
<p>
Function <b>syntaxError</b> prints an error message in the
following form:
</p>
<pre>
   Syntax error at column &lt;index&gt; of
   &lt;string&gt;
       ^       // shows character that is wrong
   &lt;message&gt;
</pre>
<p>
where the strings withing &lt;..&gt; are the actual values of the
input arguments of the function.
</p>
<p>
If the given string is too long, only a relevant
part of the string is printed.
</p>
</html>"));
      end syntaxError;

      package Advanced "Advanced scanning functions"

        function scanReal "Scans a signed real number"
          extends Modelica.Icons.Function;
          input String string;
          input Integer startIndex(min=1)=1 "Index where scanning starts";
          input Boolean unsigned=false
            "= true, if number shall not start with '+' or '-'";
          output Integer nextIndex
            "Index after the found token (success=true) or index at which scanning failed (success=false)";
          output Real number "Value of Real number";
          external "C" ModelicaStrings_scanReal(string, startIndex, unsigned, nextIndex, number);
          annotation (Library="ModelicaExternalC",Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
(nextIndex, realNumber) = <b>scanReal</b>(string, startIndex=1, unsigned=false);
</pre></blockquote>
<h4>Description</h4>
<p>
Starts scanning of \"string\" at position \"startIndex\".
First skips white space and scans afterwards a number
of type Real with an optional sign according to the Modelica grammar:
</p>
<pre>
    real     ::= [sign] unsigned [fraction] [exponent]
    sign     ::= '+' | '-'
    unsigned ::= digit [unsigned]
    fraction ::= '.' [unsigned]
    exponent ::= ('e' | 'E') [sign] unsigned
    digit    ::= '0'|'1'|'2'|'3'|'4'|'5'|'6'|'7'|'8'|'9'
</pre>
<p>
If successful, the function returns nextIndex = index of character
directly after the found real number, as well as the value
in the second output argument.
</p>
<p>
If not successful, on return nextIndex = startIndex and
the second output argument is zero.
</p>
<p>
If the optional argument \"unsigned\" is <b>true</b>, the number
shall not start with '+' or '-'. The default of \"unsigned\" is <b>false</b>.
</p>
<h4>See also</h4>
<a href=\"Modelica://Modelica.Utilities.Strings.Advanced\">Strings.Advanced</a>.
</html>
"));
        end scanReal;

        function skipWhiteSpace "Scans white space"
          extends Modelica.Icons.Function;
          input String string;
          input Integer startIndex(min=1)=1;
          output Integer nextIndex;
          external "C" nextIndex = ModelicaStrings_skipWhiteSpace(string, startIndex);
          annotation (Library="ModelicaExternalC",Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
nextIndex = <b>skipWhiteSpace</b>(string, startIndex);
</pre></blockquote>
<h4>Description</h4>
<p>
Starts scanning of \"string\" at position \"startIndex\" and
skips white space. The function returns nextIndex = index of character
of the first non white space character.
</p>
<h4>See also</h4>
<a href=\"Modelica://Modelica.Utilities.Strings.Advanced\">Strings.Advanced</a>.
</html>
"));
        end skipWhiteSpace;
        annotation (Documentation(info="<html>
<h4>Library content</h4>
<p>
Package <b>Strings.Advanced</b> contains basic scanning
functions. These functions should be <b>not called</b> directly, because
it is much simpler to utilize the higher level functions \"Strings.scanXXX\".
The functions of the \"Strings.Advanced\" library provide
the basic interface in order to implement the higher level
functions in package \"Strings\".
</p>
<p>
Library \"Advanced\" provides the following functions:
</p>
<pre>
  (nextIndex, realNumber)    = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanReal\">scanReal</a>        (string, startIndex, unsigned=false);
  (nextIndex, integerNumber) = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanInteger\">scanInteger</a>     (string, startIndex, unsigned=false);
  (nextIndex, string2)       = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanString\">scanString</a>      (string, startIndex);
  (nextIndex, identifier)    = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanIdentifier\">scanIdentifier</a>  (string, startIndex);
   nextIndex                 = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.skipWhiteSpace\">skipWhiteSpace</a>  (string, startIndex);
   nextIndex                 = <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.skipLineComments\">skipLineComments</a>(string, startIndex);
</pre>
<p>
All functions perform the following actions:
</p>
<ol>
<li> Scanning starts at character position \"startIndex\" of
     \"string\" (startIndex has a default of 1).
<li> First, white space is skipped, such as blanks (\" \"), tabs (\"\\t\"), or newline (\"\\n\")</li>
<li> Afterwards, the required token is scanned.</li>
<li> If successful, on return nextIndex = index of character
     directly after the found token and the token value is returned
     as second output argument.<br>
     If not successful, on return nextIndex = startIndex.
     </li>
</ol>
<p>
The following additional rules apply for the scanning:
</p>
<ul>
<li> Function <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanReal\">scanReal</a>:<br>
     Scans a full number including one optional leading \"+\" or \"-\" (if unsigned=false)
     according to the Modelica grammar. For example, \"+1.23e-5\", \"0.123\" are
     Real numbers, but \".1\" is not.
     Note, an Integer number, such as \"123\" is also treated as a Real number.<br>&nbsp;</li>
<li> Function <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanInteger\">scanInteger</a>:<br>
     Scans an Integer number including one optional leading \"+\"
     or \"-\" (if unsigned=false) according to the Modelica (and C/C++) grammar.
     For example, \"+123\", \"20\" are Integer numbers.
     Note, a Real number, such as \"123.4\" is not an Integer and
     scanInteger returns nextIndex = startIndex.<br>&nbsp;</li>
<li> Function <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanString\">scanString</a>:<br>
     Scans a String according to the Modelica (and C/C++) grammar, e.g.,
     \"This is a \"string\"\" is a valid string token.<br>&nbsp;</li>
<li> Function <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.scanIdentifier\">scanIdentifier</a>:<br>
     Scans a Modelica identifier, i.e., the identifier starts either
     with a letter, followed by letters, digits or \"_\".
     For example, \"w_rel\", \"T12\".<br>&nbsp;</li>
<li> Function <a href=\"Modelica://Modelica.Utilities.Strings.Advanced.skipLineComments\">skipLineComments</a><br>
     Skips white space and Modelica (C/C++) line comments iteratively.
     A line comment starts with \"//\" and ends either with an
     end-of-line (\"\\n\") or the end of the \"string\". </li>
</ul>
</html>
"));
      end Advanced;
      annotation (
        Documentation(info="<HTML>
<h4>Library content</h4>
<p>
Package <b>Strings</b> contains functions to manipulate strings.
</p>
<p>
In the table below an example
call to every function is given using the <b>default</b> options.
</p>
<table border=1 cellspacing=0 cellpadding=2>
  <tr><th><b><i>Function</i></b></th><th><b><i>Description</i></b></th></tr>
  <tr><td valign=\"top\">len = <a href=\"Modelica://Modelica.Utilities.Strings.length\">length</a>(string)</td>
      <td valign=\"top\">Returns length of string</td></tr>
  <tr><td valign=\"top\">string2 = <a href=\"Modelica://Modelica.Utilities.Strings.substring\">substring</a>(string1,startIndex,endIndex)
       </td>
      <td valign=\"top\">Returns a substring defined by start and end index</td></tr>
  <tr><td valign=\"top\">result = <a href=\"Modelica://Modelica.Utilities.Strings.repeat\">repeat</a>(n)<br>
 result = <a href=\"Modelica://Modelica.Utilities.Strings.repeat\">repeat</a>(n,string)</td>
      <td valign=\"top\">Repeat a blank or a string n times.</td></tr>
  <tr><td valign=\"top\">result = <a href=\"Modelica://Modelica.Utilities.Strings.compare\">compare</a>(string1, string2)</td>
      <td valign=\"top\">Compares two substrings with regards to alphabetical order</td></tr>
  <tr><td valign=\"top\">identical =
<a href=\"Modelica://Modelica.Utilities.Strings.isEqual\">isEqual</a>(string1,string2)</td>
      <td valign=\"top\">Determine whether two strings are identical</td></tr>
  <tr><td valign=\"top\">result = <a href=\"Modelica://Modelica.Utilities.Strings.count\">count</a>(string,searchString)</td>
      <td valign=\"top\">Count the number of occurrences of a string</td></tr>
  <tr>
<td valign=\"top\">index = <a href=\"Modelica://Modelica.Utilities.Strings.find\">find</a>(string,searchString)</td>
      <td valign=\"top\">Find first occurrence of a string in another string</td></tr>
<tr>
<td valign=\"top\">index = <a href=\"Modelica://Modelica.Utilities.Strings.findLast\">findLast</a>(string,searchString)</td>
      <td valign=\"top\">Find last occurrence of a string in another string</td></tr>
  <tr><td valign=\"top\">string2 = <a href=\"Modelica://Modelica.Utilities.Strings.replace\">replace</a>(string,searchString,replaceString)</td>
      <td valign=\"top\">Replace one or all occurrences of a string</td></tr>
  <tr><td valign=\"top\">stringVector2 = <a href=\"Modelica://Modelica.Utilities.Strings.sort\">sort</a>(stringVector1)</td>
      <td valign=\"top\">Sort vector of strings in alphabetic order</td></tr>
  <tr><td valign=\"top\">(token, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanToken\">scanToken</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for a token (Real/Integer/Boolean/String/Identifier/Delimiter/NoToken)</td></tr>
  <tr><td valign=\"top\">(number, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanReal\">scanReal</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for a Real constant</td></tr>
  <tr><td valign=\"top\">(number, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanInteger\">scanInteger</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for an Integer constant</td></tr>
  <tr><td valign=\"top\">(boolean, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanBoolean\">scanBoolean</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for a Boolean constant</td></tr>
  <tr><td valign=\"top\">(string2, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanString\">scanString</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for a String constant</td></tr>
  <tr><td valign=\"top\">(identifier, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanIdentifier\">scanIdentifier</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for an identifier</td></tr>
  <tr><td valign=\"top\">(delimiter, index) = <a href=\"Modelica://Modelica.Utilities.Strings.scanDelimiter\">scanDelimiter</a>(string,startIndex)</td>
      <td valign=\"top\">Scan for delimiters</td></tr>
  <tr><td valign=\"top\"><a href=\"Modelica://Modelica.Utilities.Strings.scanNoToken\">scanNoToken</a>(string,startIndex)</td>
      <td valign=\"top\">Check that remaining part of string consists solely of <br>
          white space or line comments (\"// ...\\n\").</td></tr>
  <tr><td valign=\"top\"><a href=\"Modelica://Modelica.Utilities.Strings.syntaxError\">syntaxError</a>(string,index,message)</td>
      <td valign=\"top\"> Print a \"syntax error message\" as well as a string and the <br>
           index at which scanning detected an error</td></tr>
</table>
<p>
The functions \"compare\", \"isEqual\", \"count\", \"find\", \"findLast\", \"replace\", \"sort\"
have the optional
input argument <b>caseSensitive</b> with default <b>true</b>.
If <b>false</b>, the operation is carried out without taking
into account whether a character is upper or lower case.
</p>
</HTML>"));
    end Strings;
      annotation (
  Documentation(info="<html>
<p>
This package contains Modelica <b>functions</b> that are
especially suited for <b>scripting</b>. The functions might
be used to work with strings, read data from file, write data
to file or copy, move and remove files.
</p>
<p>
For an introduction, have especially a look at:
</p>
<ul>
<li> <a href=\"Modelica://Modelica.Utilities.UsersGuide\">Modelica.Utilities.User's Guide</a>
     discusses the most important aspects of this library.</li>
<li> <a href=\"Modelica://Modelica.Utilities.Examples\">Modelica.Utilities.Examples</a>
     contains examples that demonstrate the usage of this library.</li>
</ul>
<p>
The following main sublibraries are available:
</p>
<ul>
<li> <a href=\"Modelica://Modelica.Utilities.Files\">Files</a>
     provides functions to operate on files and directories, e.g.,
     to copy, move, remove files.</li>
<li> <a href=\"Modelica://Modelica.Utilities.Streams\">Streams</a>
     provides functions to read from files and write to files.</li>
<li> <a href=\"Modelica://Modelica.Utilities.Strings\">Strings</a>
     provides functions to operate on strings. E.g.
     substring, find, replace, sort, scanToken.</li>
<li> <a href=\"Modelica://Modelica.Utilities.System\">System</a>
     provides functions to interact with the environment.
     E.g., get or set the working directory or environment
     variables and to send a command to the default shell.</li>
</ul>

<p>
Copyright &copy; 1998-2009, Modelica Association, DLR and Dynasim.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p><br>
</html>
"));
  end Utilities;

  package Constants
    "Library of mathematical constants and constants of nature (e.g., pi, eps, R, sigma)"
    import SI = Modelica.SIunits;
    import NonSI = Modelica.SIunits.Conversions.NonSIunits;
    extends Modelica.Icons.Library2;

    final constant Real pi=2*Modelica.Math.asin(1.0);
    annotation (
      Documentation(info="<html>
<p>
This package provides often needed constants from mathematics, machine
dependent constants and constants from nature. The latter constants
(name, value, description) are from the following source:
</p>

<dl>
<dt>Peter J. Mohr and Barry N. Taylor (1999):</dt>
<dd><b>CODATA Recommended Values of the Fundamental Physical Constants: 1998</b>.
    Journal of Physical and Chemical Reference Data, Vol. 28, No. 6, 1999 and
    Reviews of Modern Physics, Vol. 72, No. 2, 2000. See also <a href=
\"http://physics.nist.gov/cuu/Constants/\">http://physics.nist.gov/cuu/Constants/</a></dd>
</dl>

<p>CODATA is the Committee on Data for Science and Technology.</p>

<dl>
<dt><b>Main Author:</b></dt>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum f&uuml;r Luft und Raumfahrt e. V. (DLR)<br>
    Oberpfaffenhofen<br>
    Postfach 11 16<br>
    D-82230 We&szlig;ling<br>
    email: <a href=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</a></dd>
</dl>


<p>
Copyright &copy; 1998-2009, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p><br>
</html>
",   revisions="<html>
<ul>
<li><i>Nov 8, 2004</i>
       by <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Constants updated according to 2002 CODATA values.</li>
<li><i>Dec 9, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Constants updated according to 1998 CODATA values. Using names, values
       and description text from this source. Included magnetic and
       electric constant.</li>
<li><i>Sep 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Constants eps, inf, small introduced.</li>
<li><i>Nov 15, 1997</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized.</li>
</ul>
</html>"),
      Invisible=true,
      Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Line(
            points={{-34,-38},{12,-38}},
            color={0,0,0},
            thickness=0.5),
          Line(
            points={{-20,-38},{-24,-48},{-28,-56},{-34,-64}},
            color={0,0,0},
            thickness=0.5),
          Line(
            points={{-2,-38},{2,-46},{8,-56},{14,-64}},
            color={0,0,0},
            thickness=0.5)}),
      Diagram(graphics={
          Rectangle(
            extent={{200,162},{380,312}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{200,312},{220,332},{400,332},{380,312},{200,312}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{400,332},{400,182},{380,162},{380,312},{400,332}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Text(
            extent={{210,302},{370,272}},
            lineColor={160,160,164},
            textString="Library"),
          Line(
            points={{266,224},{312,224}},
            color={0,0,0},
            thickness=1),
          Line(
            points={{280,224},{276,214},{272,206},{266,198}},
            color={0,0,0},
            thickness=1),
          Line(
            points={{298,224},{302,216},{308,206},{314,198}},
            color={0,0,0},
            thickness=1),
          Text(
            extent={{152,412},{458,334}},
            lineColor={255,0,0},
            textString="Modelica.Constants")}));
  end Constants;

  package Icons "Library of icons"

    partial package Library "Icon for library"

      annotation (             Icon(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={
          Rectangle(
            extent={{-100,-100},{80,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{-100,50},{-80,70},{100,70},{80,50},{-100,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{100,70},{100,-80},{80,-100},{80,50},{100,70}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Text(
            extent={{-85,35},{65,-85}},
            lineColor={0,0,255},
            textString="Library"),
          Text(
            extent={{-120,122},{120,73}},
            lineColor={255,0,0},
            textString="%name")}),
        Documentation(info="<html>
<p>
This icon is designed for a <b>library</b>.
</p>
</html>"));
    end Library;

    partial package Library2
      "Icon for library where additional icon elements shall be added"

      annotation (             Icon(coordinateSystem(
            preserveAspectRatio=true,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={
            Rectangle(
              extent={{-100,-100},{80,50}},
              fillColor={235,235,235},
              fillPattern=FillPattern.Solid,
              lineColor={0,0,255}),
            Polygon(
              points={{-100,50},{-80,70},{100,70},{80,50},{-100,50}},
              fillColor={235,235,235},
              fillPattern=FillPattern.Solid,
              lineColor={0,0,255}),
            Polygon(
              points={{100,70},{100,-80},{80,-100},{80,50},{100,70}},
              fillColor={235,235,235},
              fillPattern=FillPattern.Solid,
              lineColor={0,0,255}),
            Text(
              extent={{-120,125},{120,70}},
              lineColor={255,0,0},
              textString="%name"),
            Text(
              extent={{-90,40},{70,10}},
              lineColor={160,160,164},
              textString="Library")}),
        Documentation(info="<html>
<p>
This icon is designed for a <b>package</b> where a package
specific graphic is additionally included in the icon.
</p>
</html>"));
    end Library2;

    partial function Function "Icon for a function"

      annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}), graphics={
            Text(
              extent={{-140,162},{136,102}},
              textString="%name",
              lineColor={0,0,255}),
            Ellipse(
              extent={{-100,100},{100,-100}},
              lineColor={255,127,0},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-100,100},{100,-100}},
              lineColor={255,127,0},
              textString="f")}),
                          Documentation(info="<html>
<p>
This icon is designed for a <b>function</b>
</p>
</html>"));
    end Function;
    annotation (
      Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Rectangle(
            extent={{-100,-100},{80,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{-100,50},{-80,70},{100,70},{80,50},{-100,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{100,70},{100,-80},{80,-100},{80,50},{100,70}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Text(
            extent={{-120,135},{120,70}},
            lineColor={255,0,0},
            textString="%name"),
          Text(
            extent={{-90,40},{70,10}},
            lineColor={160,160,164},
            textString="Library"),
          Rectangle(
            extent={{-100,-100},{80,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{-100,50},{-80,70},{100,70},{80,50},{-100,50}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{100,70},{100,-80},{80,-100},{80,50},{100,70}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Text(
            extent={{-90,40},{70,10}},
            lineColor={160,160,164},
            textString="Library"),
          Polygon(
            points={{-64,-20},{-50,-4},{50,-4},{36,-20},{-64,-20},{-64,-20}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Rectangle(
            extent={{-64,-20},{36,-84}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid),
          Text(
            extent={{-60,-24},{32,-38}},
            lineColor={128,128,128},
            textString="Library"),
          Polygon(
            points={{50,-4},{50,-70},{36,-84},{36,-20},{50,-4}},
            lineColor={0,0,0},
            fillColor={192,192,192},
            fillPattern=FillPattern.Solid)}),
                              Documentation(info="<html>
<p>
This package contains definitions for the graphical layout of
components which may be used in different libraries.
The icons can be utilized by inheriting them in the desired class
using \"extends\" or by directly copying the \"icon\" layer.
</p>

<dl>
<dt><b>Main Author:</b>
<dd><a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a><br>
    Deutsches Zentrum fuer Luft und Raumfahrt e.V. (DLR)<br>
    Oberpfaffenhofen<br>
    Postfach 1116<br>
    D-82230 Wessling<br>
    email: <A HREF=\"mailto:Martin.Otter@dlr.de\">Martin.Otter@dlr.de</A><br>
</dl>

<p>
Copyright &copy; 1998-2009, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p><br>
</HTML>
",   revisions="<html>
<ul>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Added new icons <b>Function</b>, <b>Enumerations</b> and <b>Record</b>.</li>
<li><i>June 6, 2000</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Replaced <b>model</b> keyword by <b>package</b> if the main
       usage is for inheriting from a package.<br>
       New icons <b>GearIcon</b> and <b>MotorIcon</b>.</li>
<li><i>Sept. 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Renaming package Icon to Icons.
       Model Advanced removed (icon not accepted on the Modelica meeting).
       New model Library2, which is the Library icon with enough place
       to add library specific elements in the icon. Icon also used in diagram
       level for models Info, TranslationalSensor, RotationalSensor.</li>
<li><i>July 15, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Model Caution renamed to Advanced, model Sensor renamed to
       TranslationalSensor, new model RotationalSensor.</li>
<li><i>June 30, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Realized a first version.</li>
</ul>
<br>
</html>"));
  end Icons;

  package SIunits
    "Library of type and unit definitions based on SI units according to ISO 31-1992"
    extends Modelica.Icons.Library2;

    package Conversions
      "Conversion functions to/from non SI units and type definitions of non SI units"
      extends Modelica.Icons.Library2;

      package NonSIunits "Type definitions of non SI units"
        extends Modelica.Icons.Library2;

        type Angle_deg = Real (final quantity="Angle", final unit="deg")
          "Angle in degree";
        annotation (Documentation(info="<HTML>
<p>
This package provides predefined types, such as <b>Angle_deg</b> (angle in
degree), <b>AngularVelocity_rpm</b> (angular velocity in revolutions per
minute) or <b>Temperature_degF</b> (temperature in degree Fahrenheit),
which are in common use but are not part of the international standard on
units according to ISO 31-1992 \"General principles concerning quantities,
units and symbols\" and ISO 1000-1992 \"SI units and recommendations for
the use of their multiples and of certain other units\".</p>
<p>If possible, the types in this package should not be used. Use instead
types of package Modelica.SIunits. For more information on units, see also
the book of Francois Cardarelli <b>Scientific Unit Conversion - A
Practical Guide to Metrication</b> (Springer 1997).</p>
<p>Some units, such as <b>Temperature_degC/Temp_C</b> are both defined in
Modelica.SIunits and in Modelica.Conversions.NonSIunits. The reason is that these
definitions have been placed erroneously in Modelica.SIunits although they
are not SIunits. For backward compatibility, these type definitions are
still kept in Modelica.SIunits.</p>
</HTML>
"),   Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
                  100}}), graphics={Text(
                extent={{-66,-13},{52,-67}},
                lineColor={0,0,0},
                textString="[km/h]")}));
      end NonSIunits;

      function from_deg "Convert from degree to radian"
        extends ConversionIcon;
        input NonSIunits.Angle_deg degree "degree value";
        output Angle radian "radian value";
      algorithm
        radian := (Modelica.Constants.pi/180.0)*degree;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-20,100},{-100,20}},
                lineColor={0,0,0},
                textString="deg"), Text(
                extent={{100,-20},{20,-100}},
                lineColor={0,0,0},
                textString="rad")}));
      end from_deg;

      partial function ConversionIcon "Base icon for conversion functions"
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={191,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,0},{30,0}}, color={191,0,0}),
              Polygon(
                points={{90,0},{30,20},{30,-20},{90,0}},
                lineColor={191,0,0},
                fillColor={191,0,0},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-115,155},{115,105}},
                textString="%name",
                lineColor={0,0,255})}));
      end ConversionIcon;
      annotation (Icon(coordinateSystem(preserveAspectRatio=true,
                       extent={{-100,-100},{100,100}}), graphics={
            Text(
              extent={{-33,-7},{-92,-67}},
              lineColor={0,0,0},
              lineThickness=1,
              textString="C"),
            Text(
              extent={{82,-7},{22,-67}},
              lineColor={0,0,0},
              textString="K"),
            Line(points={{-26,-36},{6,-36}}, color={0,0,0}),
            Polygon(
              points={{6,-28},{6,-45},{26,-37},{6,-28}},
              pattern=LinePattern.None,
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid,
              lineColor={0,0,255})}),
                                Documentation(info="<HTML>
<p>This package provides conversion functions from the non SI Units
defined in package Modelica.SIunits.Conversions.NonSIunits to the
corresponding SI Units defined in package Modelica.SIunits and vice
versa. It is recommended to use these functions in the following
way (note, that all functions have one Real input and one Real output
argument):</p>
<pre>
  <b>import</b> SI = Modelica.SIunits;
  <b>import</b> Modelica.SIunits.Conversions.*;
     ...
  <b>parameter</b> SI.Temperature     T   = from_degC(25);   // convert 25 degree Celsius to Kelvin
  <b>parameter</b> SI.Angle           phi = from_deg(180);   // convert 180 degree to radian
  <b>parameter</b> SI.AngularVelocity w   = from_rpm(3600);  // convert 3600 revolutions per minutes
                                                      // to radian per seconds
</pre>

</HTML>
"));
    end Conversions;

    type Angle = Real (
        final quantity="Angle",
        final unit="rad",
        displayUnit="deg");

    type Time = Real (final quantity="Time", final unit="s");

    type Pressure = Real (
        final quantity="Pressure",
        final unit="Pa",
        displayUnit="bar");
    annotation (
      Invisible=true,
      Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={Text(
            extent={{-63,-13},{45,-67}},
            lineColor={0,0,0},
            textString="[kg.m2]")}),
      Documentation(info="<html>
<p>This package provides predefined types, such as <i>Mass</i>,
<i>Angle</i>, <i>Time</i>, based on the international standard
on units, e.g.,
</p>

<pre>   <b>type</b> Angle = Real(<b>final</b> quantity = \"Angle\",
                     <b>final</b> unit     = \"rad\",
                     displayUnit    = \"deg\");
</pre>

<p>
as well as conversion functions from non SI-units to SI-units
and vice versa in subpackage
<a href=\"Modelica://Modelica.SIunits.Conversions\">Conversions</a>.
</p>

<p>
For an introduction how units are used in the Modelica standard library
with package SIunits, have a look at:
<a href=\"Modelica://Modelica.SIunits.UsersGuide.HowToUseSIunits\">How to use SIunits</a>.
</p>

<p>
Copyright &copy; 1998-2009, Modelica Association and DLR.
</p>
<p>
<i>This Modelica package is <b>free</b> software; it can be redistributed and/or modified
under the terms of the <b>Modelica license</b>, see the license conditions
and the accompanying <b>disclaimer</b>
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense\">here</a>.</i>
</p>

</html>",   revisions="<html>
<ul>
<li><i>Dec. 14, 2005</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Add User's Guide and removed \"min\" values for Resistance and Conductance.</li>
<li><i>October 21, 2002</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
       Added new package <b>Conversions</b>. Corrected typo <i>Wavelenght</i>.</li>
<li><i>June 6, 2000</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Introduced the following new types<br>
       type Temperature = ThermodynamicTemperature;<br>
       types DerDensityByEnthalpy, DerDensityByPressure,
       DerDensityByTemperature, DerEnthalpyByPressure,
       DerEnergyByDensity, DerEnergyByPressure<br>
       Attribute \"final\" removed from min and max values
       in order that these values can still be changed to narrow
       the allowed range of values.<br>
       Quantity=\"Stress\" removed from type \"Stress\", in order
       that a type \"Stress\" can be connected to a type \"Pressure\".</li>
<li><i>Oct. 27, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       New types due to electrical library: Transconductance, InversePotential,
       Damping.</li>
<li><i>Sept. 18, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Renamed from SIunit to SIunits. Subpackages expanded, i.e., the
       SIunits package, does no longer contain subpackages.</li>
<li><i>Aug 12, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Type \"Pressure\" renamed to \"AbsolutePressure\" and introduced a new
       type \"Pressure\" which does not contain a minimum of zero in order
       to allow convenient handling of relative pressure. Redefined
       BulkModulus as an alias to AbsolutePressure instead of Stress, since
       needed in hydraulics.</li>
<li><i>June 29, 1999</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
       Bug-fix: Double definition of \"Compressibility\" removed
       and appropriate \"extends Heat\" clause introduced in
       package SolidStatePhysics to incorporate ThermodynamicTemperature.</li>
<li><i>April 8, 1998</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and Astrid Jaschinski:<br>
       Complete ISO 31 chapters realized.</li>
<li><i>Nov. 15, 1997</i>
       by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
       and <a href=\"http://www.control.lth.se/~hubertus/\">Hubertus Tummescheit</a>:<br>
       Some chapters realized.</li>
</ul>
</html>"),
      Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
              100}}), graphics={
          Rectangle(
            extent={{169,86},{349,236}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{169,236},{189,256},{369,256},{349,236},{169,236}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Polygon(
            points={{369,256},{369,106},{349,86},{349,236},{369,256}},
            fillColor={235,235,235},
            fillPattern=FillPattern.Solid,
            lineColor={0,0,255}),
          Text(
            extent={{179,226},{339,196}},
            lineColor={160,160,164},
            textString="Library"),
          Text(
            extent={{206,173},{314,119}},
            lineColor={0,0,0},
            textString="[kg.m2]"),
          Text(
            extent={{163,320},{406,264}},
            lineColor={255,0,0},
            textString="Modelica.SIunits")}));
  end SIunits;
annotation (
preferredView="info",
version="3.1",
versionBuild=4,
versionDate="2009-08-14",
dateModified = "2009-08-28 08:30:00Z",
revisionId="$Id: package.mo,v 1.1.1.2 2009/09/07 15:17:14 Dag Exp $",
conversion(
 noneFromVersion="3.0.1",
 noneFromVersion="3.0",
 from(version="2.1", script="Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2", script="Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2.1", script="Scripts/ConvertModelica_from_2.2.2_to_3.0.mos"),
 from(version="2.2.2", script="Scripts/ConvertModelica_from_2.2.2_to_3.0.mos")),
__Dymola_classOrder={"UsersGuide","Blocks","StateGraph","Electrical","Magnetic","Mechanics","Fluid","Media","Thermal",
      "Math","Utilities","Constants", "Icons", "SIunits"},
Settings(NewStateSelection=true),
Documentation(info="<HTML>
<p>
Package <b>Modelica</b> is a <b>standardized</b> and <b>free</b> package
that is developed together with the Modelica language from the
Modelica Association, see
<a href=\"http://www.Modelica.org\">http://www.Modelica.org</a>.
It is also called <b>Modelica Standard Library</b>.
It provides model components in many domains that are based on
standardized interface definitions. Some typical examples are shown
in the next figure:
</p>

<p>
<img src=\"../Images/UsersGuide/ModelicaLibraries.png\">
</p>

<p>
For an introduction, have especially a look at:
</p>
<ul>
<li> <a href=\"Modelica://Modelica.UsersGuide.Overview\">Overview</a>
  provides an overview of the Modelica Standard Library
  inside the <a href=\"Modelica://Modelica.UsersGuide\">User's Guide</a>.</li>
<li><a href=\"Modelica://Modelica.UsersGuide.ReleaseNotes\">Release Notes</a>
 summarizes the changes of new versions of this package.</li>
<li> <a href=\"Modelica://Modelica.UsersGuide.Contact\">Contact</a>
  lists the contributors of the Modelica Standard Library.</li>
<li> <a href=\"../help/Documentation/ModelicaStandardLibrary.pdf\">ModelicaStandardLibrary.pdf</a>
  is the complete documentation of the library in pdf format.
<li> The <b>Examples</b> packages in the various libraries, demonstrate
  how to use the components of the corresponding sublibrary.</li>
</ul>

<p>
This version of the Modelica Standard Library consists of
</p>
<ul>
<li> <b>922</b> models and blocks, and</li>
<li> <b>615</b> functions
</ul>
<p>
that are directly usable (= number of public, non-partial classes).
</p>


<p>
<b>Licensed by the Modelica Association under the Modelica License 2</b><br>
Copyright &copy; 1998-2009, ABB, arsenal research, T.&nbsp;Bdrich, DLR, Dynasim, Fraunhofer, Modelon,
TU Hamburg-Harburg, Politecnico di Milano.
</p>

<p>
<i>This Modelica package is <u>free</u> software and
the use is completely at <u>your own risk</u>;
it can be redistributed and/or modified under the terms of the
Modelica license 2, see the license conditions (including the
disclaimer of warranty)
<a href=\"Modelica://Modelica.UsersGuide.ModelicaLicense2\">here</a></u>
or at
<a href=\"http://www.Modelica.org/licenses/ModelicaLicense2\">
http://www.Modelica.org/licenses/ModelicaLicense2</a>.
</p>

</HTML>
"));
end Modelica;


package HumMod

  model HumMod_GolemEdition 
   "Main model.
   The HumModOMCTotal.mo model is GPL v2 (http://www.gnu.org/licenses/gpl-2.0.html).
   contact Pavol at: Pavol.Privitzer@lf1.cuni.cz if you have questions about the model"
  
  import HumMod;

  HumMod.Blood.CVS_ CVS 
      annotation (Placement(transformation(extent={{40,78},{60,100}})));
    HumMod.Metabolism.NutrientsAndMetabolism nutrientsAndMetabolism 
      annotation (Placement(transformation(extent={{-56,70},{-36,90}})));
    Electrolytes.Electrolytes2 electrolytes 
      annotation (Placement(transformation(extent={{74,-28},{94,-8}})));
    Hormones.Hormones hormones 
      annotation (Placement(transformation(extent={{40,12},{60,32}})));
    Nerves.Nerves nerves 
      annotation (Placement(transformation(extent={{74,44},{94,64}})));
    HumMod.Setup.SetupFromFile setup 
      annotation (Placement(transformation(extent={{-16,-100},{4,-80}})));
    Water.Water water 
      annotation (Placement(transformation(extent={{-90,30},{-70,50}})));
    Proteins.Proteins proteins 
      annotation (Placement(transformation(extent={{-54,-8},{-34,12}})));
    HumMod.Status.TissuesFitness status 
      annotation (Placement(transformation(extent={{42,-74},{62,-54}})));
    Gases.Gases gases 
      annotation (Placement(transformation(extent={{-76,-56},{-56,-36}})));
    HumMod.setupVars.WriteVariablesToFile writeVariablesToFile 
      annotation (Placement(transformation(extent={{-40,-82},{-20,-62}})));
  equation

    connect(setup.busConnector, hormones.busConnector) annotation (Line(
        points={{2,-84},{2,27.9},{43.9,27.9}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, proteins.busConnector) annotation (Line(
        points={{2,-84},{2,10.3},{-37.7,10.3}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, CVS.busConnector) annotation (Line(
        points={{2,-84},{2,97.8},{42,97.8}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, nutrientsAndMetabolism.busConnector) 
      annotation (Line(
        points={{2,-84},{2,88},{-38,88}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, water.busConnector) annotation (Line(
        points={{2,-84},{2,47},{-73,47}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, nerves.busConnector) annotation (Line(
        points={{2,-84},{2,62},{76,62}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(status.busConnector, setup.busConnector)         annotation (Line(
        points={{43.2,-54.8},{2,-54.8},{2,-84}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(electrolytes.busConnector, setup.busConnector) annotation (Line(
        points={{74,-14},{2,-14},{2,-84}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(gases.busConnector, setup.busConnector) annotation (Line(
        points={{-58,-38},{2,-38},{2,-84}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    connect(setup.busConnector, writeVariablesToFile.busConnector) annotation (
        Line(
        points={{2,-84},{2,-64},{-30,-64}},
        color={0,0,255},
        thickness=0.5,
        smooth=Smooth.None));
    annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
            -100},{100,100}}),        graphics), Documentation(info="<html>
<p><h4><font color=\"#008000\">QHP Golem Edition</font></h4></p>
<p>Signal bus connect all submodels with their signal inputs/outputs variables.</p>
</html>", revisions="<html>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0\"><tr>
<td><p>Author:</p></td>
<td><p>Marek Matejak</p></td>
</tr>
<tr>
<td><p>Copyright:</p></td>
<td><p>In public domains</p></td>
</tr>
<tr>
<td><p>By:</p></td>
<td><p>Charles University, Prague</p></td>
</tr>
<tr>
<td><p>Date of:</p></td>
<td><p>2009</p></td>
</tr>
<tr>
<td><p>References:</p></td>
<td><p>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</p></td>
</tr>
<tr>
<td></td>
<td><p><br/><br/><br/>Siggaard Andersen: OSA (2005), University of Copenhagen</p></td>
</tr>
<tr>
<td></td>
<td><p><br/><br/><br/>Noriaki Ikeda: A model of overall regulation of body fluids (1979), Kitasato University</p></td>
</tr>
</table>
</html>"),
      Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{100,
            100}}), graphics={Bitmap(extent={{-100,100},{102,-100}}, fileName=
              "icons/golem.png")}),
      experiment(NumberOfIntervals=200, Tolerance=0.01),
      experimentSetupOutput,
      Commands(file="view.mos" "view"));
  end HumMod_GolemEdition;

  package Library "Physical domains library"

    package Interfaces "Abstract Interfaces"

      partial model BaseModel

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,127,0},
                fillColor={215,215,215},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-100,-74},{100,-52}},
                lineColor={0,0,177},
                fillPattern=FillPattern.VerticalCylinder,
                fillColor={215,215,215},
                textString="%name")}), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end BaseModel;

      partial model BaseFactorIcon

        RealInput_ yBase annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,80}),
              iconTransformation(extent={{-10,10},{10,30}}, rotation=-90)));
        RealOutput_ y annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-60}),
              iconTransformation(extent={{-10,-30},{10,-10}}, rotation=-90)));

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={95,95,95},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-90,-10},{92,10}},
                textString="%name",
                lineColor={0,0,0})}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end BaseFactorIcon;

      partial model BaseFactorIcon2

        RealInput_ yBase annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,100}),
              iconTransformation(extent={{-10,-10},{10,10}},rotation=270,
              origin={0,50})));
        RealOutput_ y annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-80}),
              iconTransformation(extent={{-10,-10},{10,10}},  rotation=270,
              origin={0,-52})));

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,50},{100,-50}},
                lineColor={0,127,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-86,-36},{100,40}},
                textString="%name",
                lineColor={0,0,0},
                fillPattern=FillPattern.Sphere)}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end BaseFactorIcon2;

      partial model BaseFactorIcon3

        RealInput_ yBase annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,80}),
              iconTransformation(extent={{-10,10},{10,30}}, rotation=-90)));
        RealOutput_ y annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-60}),
              iconTransformation(extent={{-10,-30},{10,-10}}, rotation=-90)));

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={0,127,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-90,-10},{92,10}},
                textString="%name",
                lineColor={0,0,0})}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end BaseFactorIcon3;

      partial model BaseFactorIcon4

        RealInput_ yBase annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,80}),
              iconTransformation(extent={{-10,10},{10,30}}, rotation=-90)));
        RealOutput_ y annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-60}),
              iconTransformation(extent={{-10,-30},{10,-10}}, rotation=-90)));

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={127,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-90,-10},{92,10}},
                textString="%name",
                lineColor={0,0,0})}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end BaseFactorIcon4;

      partial model BaseFactorIcon5

        RealInput_ yBase annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,80}),
              iconTransformation(extent={{-10,10},{10,30}}, rotation=-90)));
        RealOutput_ y annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-60}),
              iconTransformation(extent={{-10,-30},{10,-10}}, rotation=-90)));

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-90,-10},{92,10}},
                textString="%name",
                lineColor={0,0,0})}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end BaseFactorIcon5;

      partial model BaseResistorModel

       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-70,30},{70,-30}},
                lineColor={0,0,0},
                fillColor={241,241,241},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,0},{-70,0}}, color={0,0,0}),
              Line(points={{70,0},{90,0}}, color={0,0,0})}));
      end BaseResistorModel;

      partial connector SignalBusBlue "Icon for signal bus"

        annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.2), graphics={
            Rectangle(
              extent={{-20,2},{20,-2}},
              lineColor={255,204,51},
              lineThickness=0.5),
            Polygon(
              points={{-80,50},{80,50},{100,30},{80,-40},{60,-50},{-60,-50},{-80,
                  -40},{-100,30},{-80,50}},
              lineColor={0,0,0},
              fillColor={0,0,255},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-65,25},{-55,15}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-5,25},{5,15}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{55,25},{65,15}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-35,-15},{-25,-25}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{25,-15},{35,-25}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid)}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.2), graphics={
            Polygon(
              points={{-40,25},{40,25},{50,15},{40,-20},{30,-25},{-30,-25},{-40,
                  -20},{-50,15},{-40,25}},
              lineColor={0,0,0},
              fillColor={0,0,255},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-32.5,7.5},{-27.5,12.5}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-2.5,12.5},{2.5,7.5}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{27.5,12.5},{32.5,7.5}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{-17.5,-7.5},{-12.5,-12.5}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Ellipse(
              extent={{12.5,-7.5},{17.5,-12.5}},
              lineColor={0,0,0},
              fillColor={0,0,0},
              fillPattern=FillPattern.Solid),
            Text(
              extent={{-150,70},{150,40}},
              lineColor={0,0,0},
              textString="%name")}),
          Documentation(info="<html>
<p>
This icon is designed for a <b>signal bus</b> connector.
</p>
</html>"));

      end SignalBusBlue;

      connector RealInput = input Real "'input Real' as connector" 
        annotation (defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{98,-50},{724,58}},
              lineColor={0,0,127},
              textString="%name")},
             coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
        Diagram(coordinateSystem(
              preserveAspectRatio=true, initialScale=0.2,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,0,127},
              textString="%name")}),
          Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

      connector RealInput_ =input Real "'input Real' as connector" 
        annotation (defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid)},
             coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
        Diagram(coordinateSystem(
              preserveAspectRatio=true, initialScale=0.2,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,0,127},
              fillColor={0,0,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,0,127},
              textString="%name")}),
          Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

      connector RealOutput = output Real "'output Real' as connector" 
        annotation (defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-690,-58},{-100,60}},
              lineColor={0,0,127},
              textString="%name")},
             coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
        Diagram(coordinateSystem(
              preserveAspectRatio=true, initialScale=0.2,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,0,127},
              textString="%name")}),
          Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

      connector RealOutput_ =output Real "'output Real' as connector" 
        annotation (defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)},
             coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
        Diagram(coordinateSystem(
              preserveAspectRatio=true, initialScale=0.2,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,0,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-100,-50},{358,-92}},
              lineColor={0,0,255},
              textString="%name")}),
          Documentation(info="<html>
<p>
Connector with one input signal of type Real.
</p>
</html>"));

    connector BooleanInput = input Boolean "'input Boolean' as connector" 
      annotation (defaultComponentName="u",
           Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={255,0,255},
              fillColor={255,0,255},
              fillPattern=FillPattern.Solid)},
                coordinateSystem(extent={{-100,-100},{100,100}},
            preserveAspectRatio=true, initialScale=0.2)),    Diagram(coordinateSystem(
            preserveAspectRatio=true, initialScale=0.2,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={255,0,255},
              fillColor={255,0,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={255,0,255},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Boolean.
</p>
</html>"));

    connector BooleanOutput =output Boolean "'output Boolean' as connector" 
      annotation (defaultComponentName="u",
           Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={255,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid)},
                coordinateSystem(extent={{-100,-100},{100,100}},
            preserveAspectRatio=true, initialScale=0.2)),    Diagram(coordinateSystem(
            preserveAspectRatio=true, initialScale=0.2,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={255,0,255},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={255,0,255},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Boolean.
</p>
</html>"));

        expandable connector BusConnector
        "Empty control bus that is adapted to the signals connected to it"
          extends HumMod.Library.Interfaces.SignalBusBlue;

          annotation (
            Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                {100,100}}), graphics={Rectangle(
              extent={{-20,2},{22,-2}},
              lineColor={0,0,255},
              lineThickness=0.5)}),
            Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}),
                    graphics),
            Documentation(info="<html>
<p>
This connector defines the \"expandable connector\" ControlBus that
is used as bus in the
<a href=\"Modelica://Modelica.Blocks.Examples.BusUsage\">BusUsage</a> example.
Note, this connector is \"empty\". When using it, the actual content is
constructed by the signals connected to this bus.
</p>
</html>"));

        end BusConnector;

    partial block SIMO "Single Input Multiple Output continuous control block"
      parameter Integer nout=1 "Number of outputs";
          Modelica.Blocks.Interfaces.RealInput u
          "Connector of Real input signal" 
            annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
          Modelica.Blocks.Interfaces.RealOutput y[nout]
          "Connector of Real output signals" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

      annotation (Documentation(info="<HTML>
<p> Block has one continuous Real input signal and a
    vector of continuous Real output signals.</p>
 
</HTML>
"),   Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},{100,
                  100}}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
    end SIMO;

        partial block SISO
        "Single Input Single Output continuous control block"

          Modelica.Blocks.Interfaces.RealInput u
          "Connector of Real input signals" 
            annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
          Modelica.Blocks.Interfaces.RealOutput y
          "Connector of Real output signal" 
            annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));
          annotation (Documentation(info="<HTML>
<p>
Block has a vector of continuous Real input signals and
one continuous Real output signal.
</p>
</HTML>
"),         Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
        end SISO;

      partial class ConversionIcon "Base icon for conversion functions"

        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={191,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{-74,0},{30,0}}, color={191,0,0}),
              Polygon(
                points={{80,0},{20,20},{20,-20},{80,0}},
                lineColor={191,0,0},
                fillColor={191,0,0},
                fillPattern=FillPattern.Solid)}));

      end ConversionIcon;

      package Tissues

        model SkeletalMuscle

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-70,92},{40,-80}},
                    fileName="icons/sval.png")}));
        end SkeletalMuscle;

        model Bone

          annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                    -100},{100,100}}), graphics={Rectangle(
                  extent={{-100,100},{100,-100}},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  pattern=LinePattern.None), Bitmap(extent={{-88,96},{52,-84}},
                    fileName="icons/bone.jpg")}));
        end Bone;

        model OtherTissue

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-74,60},{48,-94}},
                    fileName="icons/pojivovaTkan.png")}));
        end OtherTissue;

        model RespiratoryMuscle

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-66,86},{44,-86}},
                    fileName="icons/respiracniSvaly.png")}));
        end RespiratoryMuscle;

        model Fat

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-84,86},{38,-68}},
                    fileName="icons/fat.jpg")}));
        end Fat;

        model Skin

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-74,56},{48,-84}},
                    fileName="icons/skin.jpg")}));
        end Skin;

        model Brain

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-78,62},{44,-92}},
                    fileName="icons/brain.jpg")}));
        end Brain;

        model GITract

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-80,74},{42,-80}},
                    fileName="icons/traviciTrakt.png")}));
        end GITract;

        model LeftHeart

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-80,100},{44,-84}},
                    fileName="icons/srdceLeva.png")}));
        end LeftHeart;

        model RightHeart

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-78,70},{44,-84}},
                    fileName="icons/srdceprava.png")}));
        end RightHeart;

        model Liver

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-84,34},{86,-100}},
                    fileName="icons/jatra.png")}));
        end Liver;

        model Kidney

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-92,86},{40,-88}},
                    fileName="icons/kidney.jpg")}));
        end Kidney;

        model Bladder

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Rectangle(
                  extent={{-100,100},{100,-100}},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  pattern=LinePattern.None), Bitmap(extent={{-92,86},{40,-88}},
                    fileName="icons/mocovyMechyr.png")}));
        end Bladder;

        model GILumen_

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,
                      -100}}, fileName="icons/traviciTrakt2.png")}));
        end GILumen_;

        model ThyroidGland

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-92,86},{40,-88}},
                    fileName="icons/stitnaZlaza.png")}));
        end ThyroidGland;

        model Pancreas

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,
                      -100}}, fileName="icons/pankreas.png")}));
        end Pancreas;

        model AdrenalGland

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,
                      -100}}, fileName="icons/nadledviny.png")}));
        end AdrenalGland;

        model Lungs

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-70,92},{40,-80}},
                    fileName="icons/plice.png")}));
        end Lungs;

        model Tissues

          annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                    -100},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,100},{100,-100}},
                  lineColor={0,0,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid),
                Bitmap(extent={{-46,98},{32,26}}, fileName="icons/bone.jpg"),
                Bitmap(extent={{-62,-10},{50,-92}}, fileName="icons/skin.jpg"),
                Bitmap(extent={{-38,90},{64,-20}}, fileName="icons/sval.png"),
                Bitmap(extent={{-10,36},{102,-56}}, fileName=
                      "icons/traviciTrakt.png"),
                Text(
                  extent={{64,-66},{94,-96}},
                  lineColor={0,0,0},
                  textString="..."),
                Bitmap(extent={{-98,36},{-24,-30}}, fileName="icons/brain.jpg")}));

        end Tissues;

        model Peritoneum

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-70,92},{40,-80}},
                    fileName="icons/peritoneum.png")}));
        end Peritoneum;

        model Plasma

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-70,92},{40,-80}},
                    fileName="icons/plazma.png")}));
        end Plasma;

        model Hypophysis

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,
                      -100}}, fileName="icons/hypofyza.png")}));
        end Hypophysis;

        model Heart

          annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,
                      -100}}, fileName="icons/heart.png")}));
        end Heart;
      end Tissues;
    end Interfaces;

    package NonSIunits "Non SI-units Support"

     constant Real PaTOmmHg(final unit="mmHg/Pa") = 760/101325;

     type Pressure_mmHg = Real (final quantity="Pressure", final unit="mmHg")
        "Pressure in mmHg";

      function to_mmHg "Convert from Pascal to mmHg"
        extends Modelica.SIunits.Conversions.ConversionIcon;
        input Modelica.SIunits.Pressure Pa "Pascal value";
        output Pressure_mmHg mmHg "mmHg value";
        //1 mmHg = 1 torr; 1/760 atm = 1 torr; 1 atm = 101325 Pa   =>   1 mmHg = 101325/760 Pa
        constant Real PaTOmmHg(final unit="mmHg/Pa") = 760/101325;
      algorithm
        mmHg := Pa * PaTOmmHg;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-20,100},{-100,20}},
                lineColor={0,0,0},
                textString="Pa"), Text(
                extent={{100,-20},{20,-100}},
                lineColor={0,0,0},
                textString="mmHg")}));
      end to_mmHg;
    end NonSIunits;

    package Curves "Empirical Dependence of Two Variables"
      /*    
      model Curve
      "2D natural cubic interpolation spline defined with (x,y,slope) points"
      // extends Library.BaseModel;
       parameter Real x[:](fixed=true);
       parameter Real y[:](fixed=true);
       parameter Real slope[:](fixed=true);

       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-100,-60},{-60,-20}}),
              iconTransformation(extent={{-120,-20},{-80,20}})));
       HumMod.Library.Interfaces.RealOutput_ val 
                       annotation (Placement(transformation(extent={{60,-20},{100,20}}),
              iconTransformation(extent={{82,-20},{122,20}})));

       parameter Integer iconNP(fixed=true) = 20;
       parameter Real iconU[iconNP](fixed=false);
       parameter Real iconPoint[iconNP,2](final unit="mm",fixed=false); // = {{-70,-42},{-22,38},{32,36},{82,-74},{72,-36}};
       parameter Integer iconFrom[20](fixed=false);
       Real iconActualPoint[2,2](final unit="mm");

       Integer iFrom(start=1);

       parameter Real a[size(x,1)-1,4](fixed=false);
      //   parameter Real a[:,:] = QHP.Library.Curves.pavol_version.SplineCoeficients( x,y,slope);

       parameter Real c1(fixed=false);
       parameter Real c2(fixed=false);

      initial equation
        // only icon drawing
       iconPoint[1,1] = -80;
       iconU[1]=min(x[j] for j in 1:size(x,1));
       iconPoint[1,2] = (-50 +
                        (100/(max(y[j] for j in 1:size(y,1))-min(y[j] for j in 1:size(y,1)))) *
                          (a[iconFrom[1],1]*(min(x[j] for j in 1:size(x,1)))^3 +
                           a[iconFrom[1],2]*(min(x[j] for j in 1:size(x,1)))^2 +
                           a[iconFrom[1],3]*(min(x[j] for j in 1:size(x,1))) +
                           a[iconFrom[1],4]
                           - min(y[j] for j in 1:size(y,1))));
       iconFrom[1] = if (size(x,1)==4) then (if (min(x[j] for j in 1:size(x,1)))<=x[2] then 1 else 
                          if (min(x[j] for j in 1:size(x,1)))<=x[3] then 2 else 3) else 
                     if (size(x,1)==3) then (if (min(x[j] for j in 1:size(x,1)))<=x[2] then 1 else 2) else 
                     1;

       for i in 1:(iconNP-1) loop
          iconPoint[i+1,1] = i*(160/(iconNP-1)) - 80;
      //x.. min(x)+ (i*(max(x)-min(x)))/(iconNP-1);
          iconU[i+1]=(min(x[j] for j in 1:size(x,1))+(i*(max(x[j] for j in 1:size(x,1))-min(x[j] for j in 1:size(x,1))))/(iconNP-1));
          iconPoint[i+1,2] = (-50 +
                        (100/(max(y[j] for j in 1:size(y,1))-min(y[j] for j in 1:size(y,1)))) *
                          (a[iconFrom[i],1]*iconU[i+1]^3 +
                           a[iconFrom[i],2]*iconU[i+1]^2 +
                           a[iconFrom[i],3]*iconU[i+1] +
                           a[iconFrom[i],4]
                           - min(y[j] for j in 1:size(y,1))));
          iconFrom[i+1] = if (size(x,1)==8) then (if iconU[i+1]<=x[2] then 1 else if iconU[i+1]<=x[3] then 2 else if iconU[i+1]<x[4] then 3 else if iconU[i+1]<x[5] then 4 else if iconU[i+1]<x[6] then 5 else if iconU[i+1]<x[7] then 6 else 7) else 
                          if (size(x,1)==7) then (if iconU[i+1]<=x[2] then 1 else if iconU[i+1]<=x[3] then 2 else if iconU[i+1]<x[4] then 3 else if iconU[i+1]<x[5] then 4 else if iconU[i+1]<x[6] then 5 else 6) else 
                          if (size(x,1)==6) then (if iconU[i+1]<=x[2] then 1 else if iconU[i+1]<=x[3] then 2 else if iconU[i+1]<x[4] then 3 else if iconU[i+1]<x[4] then 4 else 5) else 
                          if (size(x,1)==5) then (if iconU[i+1]<=x[2] then 1 else if iconU[i+1]<=x[3] then 2 else if iconU[i+1]<x[4] then 3 else 4) else 
                          if (size(x,1)==4) then (if iconU[i+1]<=x[2] then 1 else if iconU[i+1]<=x[3] then 2 else 3) else 
                          if (size(x,1)==3) then (if iconU[i+1]<=x[2] then 1 else 2) else 
                             1;
       end for;

        //main curve
        for i in 1:size(x,1)-1 loop
         a[i,1]*x[i]^3 + a[i,2]*x[i]^2 + a[i,3]*x[i] + a[i,4] = y[i];
         3*a[i,1]*x[i]^2 + 2*a[i,2]*x[i] + a[i,3] = slope[i];
         a[i,1]*x[i+1]^3 + a[i,2]*x[i+1]^2 + a[i,3]*x[i+1] + a[i,4] = y[i+1];
         3*a[i,1]*x[i+1]^2 + 2*a[i,2]*x[i+1] + a[i,3] = slope[i+1];
        end for;

        slope[1]*x[1]+c1 = y[1];
        slope[size(x,1)]*x[size(x,1)]+c2 = y[size(x,1)];

      equation
       // only icon drawing
       iconActualPoint[1,1]=(u-min(x[j] for j in 1:size(x,1)))*160/(max(x[j] for j in 1:size(x,1))-min(x[j] for j in 1:size(x,1))) - 80 -2;
       iconActualPoint[1,2]=(val-min(y[j] for j in 1:size(y,1)))*100/(max(y[j] for j in 1:size(y,1))-min(y[j] for j in 1:size(y,1))) - 50 -2;
       iconActualPoint[2,1]=iconActualPoint[1,1]+4;
       iconActualPoint[2,2]=iconActualPoint[1,2]+4;

      //main equations
       val = if (u<x[1]) then slope[1]*u + c1 else 
              if (u>x[size(x,1)]) then slope[size(x,1)]*u + c2 else 
              a[iFrom,1]*u^3 + a[iFrom,2]*u^2 + a[iFrom,3]*u + a[iFrom,4];

       iFrom =   if (size(x,1)==8) then (if u<=x[2] then 1 else if u<=x[3] then 2 else if u<x[4] then 3 else if u<x[5] then 4 else if u<x[6] then 5 else if u<x[7] then 6 else 7) else 
                 if (size(x,1)==7) then (if u<=x[2] then 1 else if u<=x[3] then 2 else if u<x[4] then 3 else if u<x[5] then 4 else if u<x[6] then 5 else 6) else 
                 if (size(x,1)==6) then (if u<=x[2] then 1 else if u<=x[3] then 2 else if u<x[4] then 3 else if u<x[4] then 4 else 5) else 
                 if (size(x,1)==5) then (if u<=x[2] then 1 else if u<=x[3] then 2 else if u<x[4] then 3 else 4) else 
                 if (size(x,1)==4) then (if u<=x[2] then 1 else if u<=x[3] then 2 else 3) else 
                 if (size(x,1)==3) then (if u<=x[2] then 1 else 2) else 
                     1;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-94,-44},{100,-100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%name"),
              Text(
                extent={{-100,100},{100,50}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="curve"),
              Line(
                points=DynamicSelect({{-100,0},{100,0}}, iconPoint),
                color={0,0,0},
                smooth=Smooth.None),
              Ellipse(
                extent=DynamicSelect({{0,0},{0,0}}, iconActualPoint),
                lineColor={0,0,0},
                fillColor={255,0,0},
                fillPattern=FillPattern.Solid)}),
       Diagram(coordinateSystem(preserveAspectRatio=false,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));

      end Curve;
      */

      model Curve
        "2D natural cubic interpolation spline defined with (x,y,slope) points"

               parameter Real x[:];
               parameter Real y[:];
               parameter Real slope[:];

               HumMod.Library.Interfaces.RealInput_ u 
                            annotation (Placement(transformation(extent={{-100,-60},{-60,-20}}),
                      iconTransformation(extent={{-120,-20},{-80,20}})));
               HumMod.Library.Interfaces.RealOutput_ val 
                               annotation (Placement(transformation(extent={{60,-20},{100,20}}),
                      iconTransformation(extent={{82,-20},{122,20}})));

      protected
              parameter Real a[:,:] = SplineCoeficients(x,y,slope);

      equation
             val = Spline(x,a, u);

             annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                -100},{100,100}}),        graphics));
      end Curve;

     function SplineCoeficients

          input Real[:] x;
          input Real[:] y;
          input Real[:] slope;

          output Real[size(x,1)+1,4] a;

      protected
        Integer n;
        Integer i;
        Real x1;
        Real x2;

        Real y1;
        Real y2;
        Real slope1;
        Real slope2;

     algorithm
        n := size(x,1);
        for i in 2:n loop
              x1:=x[i-1];
              x2:=x[i];
              y1:=y[i-1];
              y2:=y[i];
              slope1:=slope[i-1];
              slope2:=slope[i];

              a[i,1]:=-(-x2*slope2 - x2*slope1 + x1*slope2 + x1*slope1 + 2*y2 - 2*y1)/(x2 - x1)^3;
              a[i,2]:=(-x2^2*slope2-2*x2^2*slope1-3*x2*y1+x2*slope1*x1+3*x2*y2-x2*slope2*x1-3*y1*x1+slope1*x1^2+3*y2*x1+2*slope2*x1^2)/(x2-x1)^3;
              a[i,3]:=-(-slope1*x2^3-2*x2^2*slope2*x1-x2^2*slope1*x1+x2*slope2*x1^2+2*x2*slope1*x1^2+6*x2*x1*y2-6*x2*x1*y1+slope2*x1^3)/(x2-x1)^3;
              a[i,4]:=(-slope1*x2^3*x1+y1*x2^3-slope2*x1^2*x2^2+slope1*x1^2*x2^2-3*y1*x2^2*x1+3*y2*x1^2*x2+slope2*x1^3*x2-y2*x1^3)/(x2-x1)^3;
        end for;
        a[1,  :] := { 0, 0, slope[1], y[1] - x[1]*slope[1]};
        a[n+1,:] := { 0, 0, slope[n], y[n] - x[n]*slope[n]};

     end SplineCoeficients;

     function Spline

          input Real[:] x;
          input Real[:,4] a;
          input Real xVal;

          output Real yVal;
     //     output Real outExtra;
      protected
        Integer index;
        Integer n;

     algorithm
            // Najdi interval, ve kterem se nachazi xVal

            if (xVal <= x[1]) then

              yVal := xVal * a[1,3] + a[1,4];

            else
              n := size(x,1);
              if (xVal>=x[n]) then

                 yVal := xVal * a[n+1,3] + a[n+1,4];

              else
                 index := 2;
                 while  xVal > x[index] and index < n loop
                   index:=index+1;
                 end while;
                 yVal := ((a[index,1]*xVal + a[index,2])*xVal + a[index,3])*xVal + a[index,4];
              end if;
           end if;
       //    outExtra := xVal + yVal;
     end Spline;

    end Curves;

    package Factors "Multiplication Effect Types"

      model SimpleMultiply "multiplication"
       extends HumMod.Library.Interfaces.BaseFactorIcon;
       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-102,-24},{-62,16}}),
              iconTransformation(extent={{-108,-10},{-88,10}})));

       Modelica.Blocks.Math.Product product  annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-32})));
        Real effect;
      equation
        effect = u;
        connect(yBase, product.u1) annotation (Line(
            points={{6,80},{6,30},{6,-20},{6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, y) annotation (Line(
            points={{-2.02067e-015,-43},{-2.02067e-015,-55.5},{0,-55.5},{0,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(u, product.u2) annotation (Line(
            points={{-82,-4},{-6,-4},{-6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>", info="<html>
<p><h4>y = yBase * u</h4></p>
</html>"));
      end SimpleMultiply;

      model SplineValue "calculate multiplication factor from spline value"
       extends HumMod.Library.Interfaces.BaseFactorIcon4;
       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-102,-24},{-62,16}}),
              iconTransformation(extent={{-108,-10},{-88,10}})));

       parameter Real[:,3] data;
        Curves.Curve curve(
          x=data[:, 1],
          y=data[:, 2],
          slope=data[:, 3]) 
          annotation (Placement(transformation(extent={{-46,-10},{-26,10}})));
        Modelica.Blocks.Math.Product product annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-32})));

        Real effect;
      equation
        effect = curve.val;
        connect(curve.u, u) annotation (Line(
            points={{-46,0},{-64,0},{-64,-4},{-82,-4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(yBase, product.u1) annotation (Line(
            points={{6,80},{6,30},{6,-20},{6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(curve.val, product.u2) annotation (Line(
            points={{-25.8,0},{-6,0},{-6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, y) annotation (Line(
            points={{-2.02067e-015,-43},{-2.02067e-015,-55.5},{0,-55.5},{0,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end SplineValue;

      model DelayedToSpline
        "adapt the signal, from which is by curve multiplication coeficient calculated"
       extends HumMod.Library.Interfaces.BaseFactorIcon5;
       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-118,44},{-78,
                  84}}),
              iconTransformation(extent={{-108,-10},{-88,10}})));
       parameter Real Tau = 40;
       parameter Real initialValue = 1; //40;
       parameter Real[:,3] data;
       parameter String adaptationSignalName;
        Curves.Curve curve(
          x=data[:, 1],
          y=data[:, 2],
          slope=data[:, 3]) 
          annotation (Placement(transformation(extent={{-38,-16},{-18,4}})));
        Modelica.Blocks.Math.Product product annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-32})));
        Blocks.Integrator integrator(                    k=(1/Tau)/Library.SecPerMin,
            y_start=initialValue,
          stateName=adaptationSignalName) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-60,14})));
        Modelica.Blocks.Math.Feedback feedback annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-60,46})));
        Real effect;
      equation
        effect = curve.val;
        connect(yBase, product.u1) annotation (Line(
            points={{6,80},{6,30},{6,-20},{6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, y) annotation (Line(
            points={{-2.02067e-015,-43},{-2.02067e-015,-55.5},{0,-55.5},{0,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(feedback.y, integrator.u) annotation (Line(
            points={{-60,37},{-60,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, feedback.u2) annotation (Line(
            points={{-60,3},{-60,-6},{-84,-6},{-84,46},{-68,46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(feedback.u1, u) annotation (Line(
            points={{-60,54},{-60,64},{-98,64}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, curve.u) annotation (Line(
            points={{-60,3},{-60,-6},{-38,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(curve.val, product.u2) annotation (Line(
            points={{-17.8,-6},{-6,-6},{-6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end DelayedToSpline;

      model SplineDelayByDay
        "adapt the value of multiplication coeficient calculated from curve"
       extends HumMod.Library.Interfaces.BaseFactorIcon3;
       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-118,44},{-78,
                  84}}),
              iconTransformation(extent={{-108,-10},{-88,10}})));
       parameter Real Tau;
       parameter Real[:,3] data;
        Curves.Curve curve(
          x=data[:, 1],
          y=data[:, 2],
          slope=data[:, 3]) 
          annotation (Placement(transformation(extent={{-68,58},{-48,78}})));
        Modelica.Blocks.Math.Product product annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-32})));
        Modelica.Blocks.Continuous.Integrator integrator(
            y_start=1, k=(1/(Tau*1440))/Library.SecPerMin,
          initType=Modelica.Blocks.Types.Init.SteadyState) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-26,12})));
        Modelica.Blocks.Math.Feedback feedback annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-26,44})));
        Real effect;
      equation
        effect = integrator.y;
        connect(curve.u, u) annotation (Line(
            points={{-68,68},{-83,68},{-83,64},{-98,64}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(yBase, product.u1) annotation (Line(
            points={{6,80},{6,30},{6,-20},{6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, y) annotation (Line(
            points={{-2.02067e-015,-43},{-2.02067e-015,-55.5},{0,-55.5},{0,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(curve.val, feedback.u1) annotation (Line(
            points={{-47.8,68},{-26,68},{-26,52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(feedback.y, integrator.u) annotation (Line(
            points={{-26,35},{-26,29.5},{-26,24},{-26,24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, feedback.u2) annotation (Line(
            points={{-26,1},{-26,-8},{-50,-8},{-50,44},{-34,44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, product.u2) annotation (Line(
            points={{-26,1},{-26,-8},{-6,-8},{-6,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end SplineDelayByDay;

      model SplineDelayFactorByDayWithFailture
        "combination of SplineDelayByDay and ZeroIfFalse"
       extends HumMod.Library.Interfaces.BaseFactorIcon2;
       HumMod.Library.Interfaces.RealInput_ u 
                    annotation (Placement(transformation(extent={{-118,6},{-78,46}}),
              iconTransformation(extent={{-108,-30},{-88,-10}})));
       parameter Real Tau;
       parameter Real[:,3] data;
        Curves.Curve curve(
          x=data[:, 1],
          y=data[:, 2],
          slope=data[:, 3]) 
          annotation (Placement(transformation(extent={{-76,20},{-56,40}})));
        Modelica.Blocks.Math.Product product annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-50})));
        Modelica.Blocks.Continuous.Integrator integrator(                y_start=
             1, k=(1/(Tau*1440))/Library.SecPerMin,
          initType=Modelica.Blocks.Types.Init.SteadyState) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-14,-6})));
        Modelica.Blocks.Math.Feedback feedback annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-14,26})));
        Modelica.Blocks.Logical.Switch switch1 
          annotation (Placement(transformation(extent={{-36,40},{-16,60}})));
        HumMod.Library.Blocks.Constant Constant1(k=0) 
          annotation (Placement(transformation(extent={{-82,62},{-62,82}})));
        HumMod.Library.Interfaces.BooleanInput Failed 
                            annotation (Placement(transformation(extent={{-118,30},{
                  -78,70}}), iconTransformation(extent={{-108,30},{-88,50}})));
        Real effect;
      equation
        effect = integrator.y;
        connect(curve.u, u) annotation (Line(
            points={{-76,30},{-87,30},{-87,26},{-98,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(yBase, product.u1) annotation (Line(
            points={{6,100},{6,31},{6,-38},{6,-38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, y) annotation (Line(
            points={{-2.02067e-015,-61},{-2.02067e-015,-55.5},{0,-55.5},{0,-80}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(feedback.y, integrator.u) annotation (Line(
            points={{-14,17},{-14,14.25},{-14,14.25},{-14,11.5},{-14,6},{-14,6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, feedback.u2) annotation (Line(
            points={{-14,-17},{-14,-26},{-38,-26},{-38,26},{-22,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(integrator.y, product.u2) annotation (Line(
            points={{-14,-17},{-14,-26},{-6,-26},{-6,-38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(switch1.y, feedback.u1) annotation (Line(
            points={{-15,50},{-14,50},{-14,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(curve.val, switch1.u3) annotation (Line(
            points={{-55.8,30},{-54,30},{-54,42},{-38,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Constant1.y, switch1.u1) annotation (Line(
            points={{-61,72},{-58,72},{-58,58},{-38,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(switch1.u2, Failed) annotation (Line(
            points={{-38,50},{-98,50}},
            color={255,0,255},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true,  extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end SplineDelayFactorByDayWithFailture;
    end Factors;

    package Blocks "Base Signal Blocks Library"

        block BooleanConstant "Generate constant signal of type Boolean"
          parameter Boolean k=true "Constant output value";
          extends Modelica.Blocks.Interfaces.partialBooleanSource;

        equation
          y = k;
          annotation (
            Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Line(points={{-80,0},{80,0}}, color={0,0,0}),
                Text(
                extent={{-150,-140},{150,-110}},
                lineColor={0,0,0},
                textString="%k")}),
            Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(
                points={{-70,0},{80,0}},
                color={0,0,255},
                thickness=0.5),
              Text(
                extent={{-69,20},{-49,0}},
                lineColor={0,0,0},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid,
                textString="k"),
              Text(
                extent={{-96,6},{-76,-4}},
                lineColor={0,0,0},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid,
                textString="true"),
              Text(
                extent={{-98,-58},{-72,-68}},
                lineColor={0,0,0},
                fillColor={95,95,95},
                fillPattern=FillPattern.Solid,
                textString="false")}),
            Documentation(info="<html>
<p>
The Boolean output y is a constant signal:
</p>
 
<p>
<img src=\"../Images/Blocks/Sources/BooleanConstant.png\">
</p>
</html>"));
        end BooleanConstant;

          block Parts "Divide the input value by weights"
            extends HumMod.Library.Interfaces.SIMO;
            parameter Real w[nout]=ones(nout) "Optional: weight coeficients";
      protected
           Real coef;
           Real weight[nout];
          equation
            ones(nout)*weight = 1;
            for i in 1:nout loop
                weight[i] = w[i] * coef;
                y[i] = u * weight[i];
            end for;
            annotation (defaultComponentName="parts1",
              Documentation(info="
<HTML>
<p>
This blocks divide input value <b>y</b> to
<i>outputs</i> by weights. The sum of output values is equal to input value.
<b>u</b>:
</p>
<pre>
    <b>y</b> = (<b>u</b>[1] + <b>u</b>[2] + ... + <b>u</b>[n]) / n;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   nin = 3;
 
  results in the following equations:
 
     y = (u[1] + u[2] + u[3]) / 3;
</pre>
 
</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="Parts")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                textString="AVG")}));
          end Parts;

          block Add "Output the addition of a value with the input signal"

            parameter Real k(start=1) "value added to input signal";
      public
            Modelica.Blocks.Interfaces.RealInput u "Input signal connector" 
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0)));
            Modelica.Blocks.Interfaces.RealOutput y "Output signal connector" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                rotation=0)));

          equation
            y = k+u;
            annotation (
              Documentation(info="
<HTML>
<p>
This block computes output <i>y</i> as
<i>product</i> of gain <i>k</i> with the
input <i>u</i>:
</p>
<pre>
    y = k * u;
</pre>
 
</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-98,-52},{30,34}},
                lineColor={0,0,0},
                textString="%k+u"),
              Text(
                extent={{-150,140},{150,100}},
                textString="%name",
                lineColor={0,0,255})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Polygon(
                points={{-100,-100},{-100,100},{100,0},{-100,-100}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-76,38},{0,-34}},
                lineColor={0,0,255},
                textString="k")}));
          end Add;

          block Pow "the power of parameter"
            extends HumMod.Library.Interfaces.SISO;
            parameter Real power_base=10 "base";
          equation
            y = power_base^u;
             annotation (defaultComponentName="avg1",
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as
<i>average value</i> of the elements of the input signal vector
<b>u</b>:
</p>
<pre>
    <b>y</b> = (<b>u</b>[1] + <b>u</b>[2] + ... + <b>u</b>[n]) / n;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   nin = 3;
 
  results in the following equations:
 
     y = (u[1] + u[2] + u[3]) / 3;
</pre>
 
</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
                  initialScale=0.04), graphics={Text(
                extent={{-100,-40},{100,40}},
                lineColor={0,0,0},
                textString="%power_base^u")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
                  initialScale=0.04), graphics={Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-100,-46},{100,52}},
                lineColor={0,0,255},
                textString="Pow")}));
          end Pow;

          block Inv "Output the inverse value of the input"
            extends Modelica.Blocks.Interfaces.SISO;
          equation
            y = 1/u;
            annotation (defaultComponentName="inv1",
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                textString="1/u")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics),
              Documentation(info="<HTML>
<p>
This blocks computes the output <b>y</b>
as <i>inverse value</i> of the input <b>u</b>:
</p>
<pre>
    y = 1 / u ;
</pre>
 
</HTML>
"));
          end Inv;

          block MultiProduct
        "Output the product of the elements of the input vector"
            extends Modelica.Blocks.Interfaces.MISO;
          equation
            y = product(u);
            annotation (defaultComponentName="product1",
              Documentation(info="
<HTML>
<p>
This blocks computes output <b>y</b> as
<i>product</i> of the elements of the input signal vector
<b>u</b>:
</p>
<pre>
    <b>y</b> = <b>u</b>[1] * <b>u</b>[2] * ...;
</pre>
<p>
Example:
</p>
<pre>
     parameter:   nin = 3;

  results in the following equations:

     y = u[1] * u[2] * u[3];
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255}),
              Line(
                points={{56,38},{-48,38},{-10,38},{-10,-42},{-10,-42}},
                color={0,0,0},
                thickness=0.25),
              Line(
                points={{16,38},{16,-44}},
                color={0,0,0},
                smooth=Smooth.None)}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(
                points={{26,42},{-34,42},{-10,42},{-10,-38},{-10,-38}},
                color={0,0,0},
                thickness=0.25),
              Line(
                points={{0,42},{0,-38}},
                color={0,0,0},
                smooth=Smooth.None)}));
          end MultiProduct;

          block Log10AsEffect
        "Output the base 10 logarithm of the input > 1, or 0 otherwise"

            extends Modelica.Blocks.Interfaces.SISO;
          equation
            y = if u>1 then Modelica.Math.log10(u) else 0;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Polygon(
                points={{90,0},{68,8},{68,-8},{90,0}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,0},{68,0}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-80,-80},{-80,68}}, color={192,192,192}),
              Text(
                extent={{-44,-56},{94,-80}},
                lineColor={192,192,192},
                textString="max(log10,0)"),
              Line(points={{-100,0},{-80,0},{-78,0},{-74,0},{-76,0},{-74,0},{-72,
                    0},{-69.5,6.08},{-64.7,15.9},{-57.5,26},{-47,36.1},{-31.8,
                    46.1},{-10.1,56},{22.1,66},{68.7,76.1},{80,78}}, color={0,0,
                    0})}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-80,80},{-88,80}}, color={192,192,192}),
              Line(points={{-80,-80},{-88,-80}}, color={192,192,192}),
              Line(points={{-80,-90},{-80,84}}, color={192,192,192}),
              Text(
                extent={{-65,96},{-38,78}},
                lineColor={160,160,164},
                textString="y"),
              Polygon(
                points={{-80,100},{-86,84},{-74,84},{-80,100}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-100,0},{84,0}}, color={192,192,192}),
              Polygon(
                points={{100,0},{84,6},{84,-6},{100,0}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-100,0},{-80,0},{-78,0},{-74,0},{-76,0},{-74,0},{-72,0},
                    {-69.5,6.08},{-64.7,15.9},{-57.5,26},{-47,36.1},{-31.8,46.1},{
                    -10.1,56},{22.1,66},{68.7,76.1},{80,78}}, color={0,0,0}),
              Text(
                extent={{70,-3},{90,-23}},
                textString="20",
                lineColor={0,0,255}),
              Text(
                extent={{-78,-1},{-58,-21}},
                textString="1",
                lineColor={0,0,255}),
              Text(
                extent={{-109,72},{-89,88}},
                textString=" 1.3",
                lineColor={0,0,255}),
              Text(
                extent={{-109,-88},{-89,-72}},
                textString="-1.3",
                lineColor={0,0,255}),
              Text(
                extent={{62,30},{90,10}},
                lineColor={160,160,164},
                textString="u")}),
              Documentation(info="<HTML>
<p>
This blocks computes the output <b>y</b> as the
<i>base 10 logarithm</i> of the input <b>u</b>:
</p>
<pre>
    y = <b>log10</b>( u );
</pre>
<p>
An error occurs if the elements of the input <b>u</b> are
zero or negative.
</p>

<p>
<img src=\"../Images/Math/log10.png\">
</p>

</HTML>
"));
          end Log10AsEffect;

          block Constant "Generate constant signal of type Real"
            parameter Real k(start=1) "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-40},{100,46}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end Constant;

          block FractConstant "Generate constant signal in part from 1"
            parameter Real k(start=1, final unit="%") "Part in percent";

            HumMod.Library.Interfaces.RealOutput_ y(final unit="1")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k/100;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-94,-50},{94,56}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid,
                textString="%k%%")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end FractConstant;

          block Fract2Constant
        "Generate constant signal y as part on interval <0,1> and signal 1-y"
            parameter Real k(start=1, final unit="%") "Part in percent";

            HumMod.Library.Interfaces.RealOutput_ y(final unit="1")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0), iconTransformation(extent={{100,48},{120,68}})));

            HumMod.Library.Interfaces.RealOutput_ y2(final unit="1")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0), iconTransformation(extent={{100,-62},{120,-42}})));
          equation
            y = k/100;
            y2 = 1-y;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,2},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k %%"),
              Text(
                extent={{-100,-100},{100,-2}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="100-%k %%")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end Fract2Constant;

          block PressureConstant
        "Generate constant signal of type Pressure_mmHg"
            parameter Real k(start=1, final quantity="Pressure", final unit="mmHg")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Pressure",
            final unit="mmHg") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
        connect(y, y) annotation (Line(
            points={{110,0},{110,0}},
            color={0,0,127},
            smooth=Smooth.None));
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,-6},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-98,-82},{98,-20}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mmHg")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end PressureConstant;

          block VolumeConstant "Generate constant signal of type Volume_ml"
            parameter Real k(start=1, final quantity="Volume", final unit="ml")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Volume",
            final unit="ml") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));


          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k ml")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end VolumeConstant;

          block OsmolarityConstant "Generate constant signal of type mOsm"
            parameter Real k(start=1, final quantity="Osmolarity", final unit="mOsm")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Osmolarity",
            final unit="mOsm") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            connect(y, y) annotation (Line(
                points={{110,0},{106,0},{106,0},{110,0}},
                color={0,0,127},
                smooth=Smooth.None));
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-98,6},{100,86}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-80},{100,-18}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mOsm")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end OsmolarityConstant;

          block TemperatureConstant
        "Generate constant signal of type temperature in Celsius degrees"
            parameter Real k(start=1, final quantity="Temperature", final unit="degC")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity=
              "Temperature", final unit="degC")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k C")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end TemperatureConstant;

          block FlowConstant "Generate constant signal in units ml/min"
            parameter Real k(start=1, final quantity="Flow", final unit="ml/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="ml/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="ml")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end FlowConstant;

          block CaloriesConstant
        "Generate constant signal of type Energy in calories"
            parameter Real k(start=1, final quantity="Energy", final unit="Cal")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Energy",
            final unit="Cal") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k Cal")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end CaloriesConstant;

          block ElectrolytesMassConstant
        "Generate constant signal of type Mass_mEq"
            parameter Real k(start=1, final quantity="Mass", final unit="mEq")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Mass",
            final unit="mEq") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k mEq")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ElectrolytesMassConstant;

          block ComplianceConstant "Generate constant signal in units ml/mmHg"
            parameter Real k(start=1, final quantity="Compliance", final unit="ml/mmHg")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Compliance",
            final unit="ml/mmHg") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mmHg"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="ml")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ComplianceConstant;

          block CondConstant "Generate constant signal in units (ml/min)/mmHg"
            parameter Real k(start=1, final quantity="Conductance", final unit="ml/(min.mmHg)")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity=
              "Conductance", final unit="ml/(min.mmHg)")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-86},{98,-50}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min.mmHg"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-98,-40},{94,0}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="ml")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end CondConstant;

          block ElectrolytesConcentrationConstant_per_l
        "Generate constant signal of type mEq/L"
            parameter Real k(start=1, final quantity="Concentration", final unit
            =                                                                    "mEq/l")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity=
              "Concentration", final unit="mEq/l")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mEq"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="l")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ElectrolytesConcentrationConstant_per_l;

          block ConcentrationConstant_pg_per_ml
        "Generate constant signal in units pg/ml"
            parameter Real k(start=1, final quantity="Concentration", final unit
            =                                                                    "pg/ml")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity=
              "Concentration", final unit="pg/ml")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="ml"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="pG")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ConcentrationConstant_pg_per_ml;

          block ConcentrationConstant_uU_per_ml
        "Generate constant signal in units uU/ml"
            parameter Real k(start=1, final quantity="Concentration", final unit
            =                                                                    "uU/ml")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity=
              "Concentration", final unit="uU/ml")
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="ml"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="uU")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ConcentrationConstant_uU_per_ml;

          block MassFlowConstant "Generate constant signal in units mg/min"
            parameter Real k(start=1, final quantity="Flow", final unit="mg/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="mg/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mg")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end MassFlowConstant;

          block ElectrolytesFlowConstant
        "Generate constant signal of type Mass_mEq_per_min"
            parameter Real k(start=1, final quantity="Flow", final unit="mEq/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="mEq/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,-6},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="mEq")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ElectrolytesFlowConstant;

          block HormoneFlowConstant_nG
        "Generate constant signal in units nG/min"
            parameter Real k(start=1, final quantity="Flow", final unit="ng/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="ng/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="nG")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end HormoneFlowConstant_nG;

          block HormoneFlowConstant_pmol
        "Generate constant signal in units pmol/min"
            parameter Real k(start=1, final quantity="Flow", final unit="pmol/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="pmol/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="pmol")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end HormoneFlowConstant_pmol;

          block HormoneFlowConstant_uG
        "Generate constant signal in units uG/min"
            parameter Real k(start=1, final quantity="Flow", final unit="ug/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="ug/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="uG")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end HormoneFlowConstant_uG;

          block HormoneFlowConstant_GU
        "Generate constant signal in units U/min"
            parameter Real k(start=1, final quantity="Flow", final unit="GU/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="GU/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="GU")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end HormoneFlowConstant_GU;

          block CaloriesFlowConstant
        "Generate constant signal of type Energy Flow in calories per minute"
            parameter Real k(start=1, final quantity="Flow", final unit="Cal/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="Cal/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,2},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-52},{100,24}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-44},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="Cal")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end CaloriesFlowConstant;

          block MassFlowConstant_kg "Generate constant signal in units kg/min"
            parameter Real k(start=1, final quantity="Flow", final unit="kg/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="kg/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,20},{100,92}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="kg")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end MassFlowConstant_kg;

          block SafeDivision
        "Output first input divided by second input, if the second input is zero return parameter"
            extends Modelica.Blocks.Interfaces.SI2SO;
            parameter Real ZeroDivisorResult=0;
            parameter Real Zero=1e-20;
          equation
            y = if u2<=Zero and u2>=-Zero then ZeroDivisorResult else u1/u2;
            annotation (
              Documentation(info="
<HTML>
<p>
This block computes the output <b>y</b> (element-wise)
by <i>dividing</i> the corresponding elements of
the two inputs <b>u1</b> and <b>u2</b>:
</p>
<pre>
    y = u1 / u2;
</pre>

</HTML>
"),           Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{50,0},{100,0}}, color={0,0,127}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Ellipse(
                extent={{-5,20},{5,30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-5,-20},{5,-30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,127}),
              Text(
                extent={{-150,150},{150,110}},
                textString="%name",
                lineColor={0,0,255}),
              Line(points={{-100,60},{-66,60},{-40,30}}, color={0,0,127}),
              Line(points={{-100,-60},{0,-60},{0,-50}}, color={0,0,127}),
              Text(
                extent={{-96,-62},{98,-98}},
                lineColor={0,0,0},
                textString="u/0 = %ZeroDivisorResult")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(points={{50,0},{100,0}}, color={0,0,255}),
              Line(points={{-30,0},{30,0}}, color={0,0,0}),
              Ellipse(
                extent={{-5,20},{5,30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(
                extent={{-5,-20},{5,-30}},
                lineColor={0,0,0},
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid),
              Ellipse(extent={{-50,50},{50,-50}}, lineColor={0,0,255}),
              Line(points={{-100,60},{-66,60},{-40,30}}, color={0,0,255}),
              Line(points={{-100,-60},{0,-60},{0,-50}}, color={0,0,255})}),
              uses(Modelica(version="3.1")));
          end SafeDivision;

      block Min "Pass through the smallest signal"
        extends Modelica.Blocks.Interfaces.MISO;
      equation
         y = min(u);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-90,36},{90,-36}},
                lineColor={160,160,164},
                textString="min()")}),
                                Documentation(info="<html>
<p>
This block computes the output <b>y</b> as <i>minimum</i> of
the Real inputs <b>u[1]</b>,<b>u[2]</b> .. <b>u[nin]</b>:
</p>
<pre>    y = <b>min</b> ( u );
</pre>
</html>
"));
      end Min;

          block HormoneFlowConstant_U "Generate constant signal in units U/min"
            parameter Real k(start=1, final quantity="Flow", final unit="U/min")
          "Constant output value";

            HumMod.Library.Interfaces.RealOutput_ y(final quantity="Flow",
            final unit="U/min") "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}),
              Text(
                extent={{-100,0},{100,100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%k"),
              Text(
                extent={{-100,-100},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="min"),
              Text(
                extent={{-100,-46},{100,30}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="____"),
              Text(
                extent={{-100,-38},{100,16}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="U")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Text(
                extent={{-100,-100},{100,100}},
                lineColor={0,0,0},
                textString="%k")}),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end HormoneFlowConstant_U;

      model Integrator
        extends Utilities.DynamicState;

        parameter Real k=1 "Integrator gain";

        parameter Real y_start=0 "Initial or guess value of output (= state)" 
          annotation (Dialog(group="Initialization"));
        extends Interfaces.SISO(y(start=y_start));


      equation
        stateValue = y;
        changePerMin = u*Library.SecPerMin;

        annotation (
          Documentation(info="<html>
<p>
This blocks computes output <b>y</b> (element-wise) as
<i>integral</i> of the input <b>u</b> multiplied with
the gain <i>k</i>:
</p>
<pre>
         k
     y = - u
         s
</pre>

<p>
It might be difficult to initialize the integrator in steady state.
This is discussed in the description of package
<a href=\"Modelica://Modelica.Blocks.Continuous#info\">Continuous</a>.
</p>

</html>
"),       Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Line(points={{-80,78},{-80,-90}}, color={192,192,192}),
              Polygon(
                points={{-80,90},{-88,68},{-72,68},{-80,90}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Line(points={{-90,-80},{82,-80}}, color={192,192,192}),
              Polygon(
                points={{90,-80},{68,-72},{68,-88},{90,-80}},
                lineColor={192,192,192},
                fillColor={192,192,192},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{0,-10},{60,-70}},
                lineColor={192,192,192},
                textString="I"),
              Text(
                extent={{-150,-150},{150,-110}},
                lineColor={0,0,0},
                textString="k=%k"),
              Line(points={{-80,-80},{80,80}}, color={0,0,127}),
              Text(
                extent={{-150,106},{150,146}},
                lineColor={0,0,0},
                textString="%stateName")}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={
              Rectangle(extent={{-60,60},{60,-60}}, lineColor={0,0,255}),
              Line(points={{-100,0},{-60,0}}, color={0,0,255}),
              Line(points={{60,0},{100,0}}, color={0,0,255}),
              Text(
                extent={{-36,60},{32,2}},
                lineColor={0,0,0},
                textString="k"),
              Text(
                extent={{-32,0},{36,-58}},
                lineColor={0,0,0},
                textString="s"),
              Line(points={{-46,0},{46,0}}, color={0,0,0})}));
      end Integrator;
    end Blocks;

    package PressureFlow "Hydraulic Physical Domain"

      connector PressureFlow "Pressure[mmHg] and Flow[ml/min]"
        Real pressure( final quantity="Pressure", final unit="mmHg");
        flow Real q( final quantity="Flow", final unit="ml/min") "flow";
        annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague, Czech Republic</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
</table>
</html>", info="<html>
<p><font style=\"font-size: 9pt; \">This connector connects hydraulic domains elements. The elements contains one equation for each his pressure-flow connector. The equation defines relation between variables in the connector. Variables are hydraulic pressure and volume flow of hydraulic medium. The pressure is the same in each connector that are connected together. The sum of flow in connectors connected together is zero (</font><b><font style=\"font-size: 9pt; \">Kirchhoff&apos;s circuit laws</font></b><font style=\"font-size: 9pt; \">).</font> </p>
</html>"));
      end PressureFlow;

      connector PositivePressureFlow "Pressure[mmHg] and Inflow[ml/min]"
        extends PressureFlow;

      annotation (
          defaultComponentName="q_in",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={0,0,0},
                lineThickness=1), Polygon(
                points={{0,100},{100,0},{0,-100},{-100,0},{0,100}},
                lineColor={0,0,0},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={0,0,0})}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))));
      end PositivePressureFlow;

      connector NegativePressureFlow
        "Pressure[mmHg] and negative Outflow[ml/min]"
        extends PressureFlow;

      annotation (
          defaultComponentName="q_out",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={0,0,0},
                lineThickness=1), Polygon(
                points={{0,100},{100,0},{0,-100},{-100,0},{0,100}},
                lineColor={0,0,0},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={200,200,200})}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))));
      end NegativePressureFlow;

      model FlowMeasure
        "Convert connector volume flow value to signal flow value"

        NegativePressureFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{18,-10},{38,10}}), iconTransformation(
                extent={{50,-10},{70,10}})));


        PositivePressureFlow q_in annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-70,-10},
                  {-50,10}})));
        HumMod.Library.Interfaces.RealOutput_ actualFlow(final quantity="Flow",
            final unit="ml/min") 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=90,
              origin={0,54})));
      equation
        q_in.q + q_out.q = 0;
        q_out.pressure = q_in.pressure;

        actualFlow = q_in.q;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Bitmap(extent={{-60,60},{60,-60}},
                  fileName="icons/flowMeassure.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FlowMeasure;

      model PressureMeasure
        "Convert connector hydraulic pressure value to signal flow value"


        HumMod.Library.PressureFlow.PositivePressureFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-30,-50},
                  {-10,-30}})));
        HumMod.Library.Interfaces.RealOutput_ actualPressure 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=0,
              origin={52,-20})));
      equation

        actualPressure = q_in.pressure;
        q_in.q = 0;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Bitmap(extent={{-56,60},{64,-60}},
                  fileName="icons/pressureMeassure.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end PressureMeasure;

      partial model ResistorBase "Hydraulic Volume Flow Resistance"
       extends HumMod.Library.Interfaces.BaseResistorModel;
        PositivePressureFlow q_in 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-10},{-90,10}})));
        NegativePressureFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
      Real ActualConductance;
      equation
        ActualConductance = if 
                              (q_in.pressure == q_out.pressure) then 0 else q_in.q/(q_in.pressure - q_out.pressure);
        annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague, Czech Republic</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
</table>
</html>",       info="<html>
<p>This hydraulic conductance (resistance) element contains two connector sides. No hydraulic medium volume is changing in this element during simulation. That means that sum of flow in both connector sides is zero. The flow through element is determined by <b>Ohm&apos;s law</b>. It is used conductance (=1/resistance) because it could be numerical zero better then infinity in resistance. </p>
</html>"));
      end ResistorBase;

      partial model ResistorBase2 "Hydraulic Volume Flow Resistance"
       extends HumMod.Library.Interfaces.BaseResistorModel;
        PositivePressureFlow q_in 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(
                extent={{90,-10},{110,10}})));
        NegativePressureFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-10},{-90,10}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
      Real ActualConductance;
      equation
        ActualConductance = if 
                              (q_in.pressure == q_out.pressure) then 0 else q_in.q/(q_in.pressure - q_out.pressure);
        annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague, Czech Republic</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
</table>
</html>",       info="<html>
<p>This hydraulic conductance (resistance) element contains two connector sides. No hydraulic medium volume is changing in this element during simulation. That means that sum of flow in both connector sides is zero. The flow through element is determined by <b>Ohm&apos;s law</b>. It is used conductance (=1/resistance) because it could be numerical zero better then infinity in resistance. </p>
</html>"), Icon(graphics));
      end ResistorBase2;

      partial model Resistor
       extends ResistorBase;
      equation
        q_in.q + q_out.q = 0;
        annotation (Icon(graphics));
      end Resistor;

      model ResistorWithCond
       extends Resistor;
        HumMod.Library.Interfaces.RealInput_ cond(final quantity="Conductance",
            final unit="ml/(min.mmHg)") 
                                 annotation (extent=[-5,30; 5,50], rotation=-90);
      equation
        q_in.q = cond * (q_in.pressure - q_out.pressure);
        annotation (Icon(graphics={Text(
                extent={{-70,-30},{70,30}},
                textString="%name",
                lineColor={0,0,255})}));
      end ResistorWithCond;

      model ResistorWithCondParam
       extends Resistor;
       parameter Real cond(  final quantity="Conductance", final unit="ml/(min.mmHg)");
      equation
        q_in.q = cond * (q_in.pressure - q_out.pressure);

        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}),
                            graphics), Icon(graphics={
              Text(
                extent={{-70,0},{70,30}},
                lineColor={0,0,0},
                textString="%cond"),
              Text(
                extent={{-134,-90},{154,-30}},
                textString="%name",
                lineColor={0,0,255}),
              Text(
                extent={{-70,-20},{70,0}},
                lineColor={0,0,0},
                textString="ml/min/mmHg")}));
      end ResistorWithCondParam;

      model OneWayResistorWithCondParam
       extends Resistor;
       parameter Real cond(  final quantity="Conductance", final unit="ml/(min.mmHg)");
       parameter Real zero = 1.E-5;
        HumMod.Library.Interfaces.RealOutput_ actualFlow(final quantity="Flow",
            final unit="ml/min") 
                               annotation (Placement(transformation(extent={{72,-48},
                  {112,-8}}), iconTransformation(extent={{70,-32},{94,-8}})));
      equation
        q_in.q = max(cond * (q_in.pressure - q_out.pressure), zero);
        actualFlow = q_in.q;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}}),
                            graphics), Icon(graphics={
              Text(
                extent={{-70,-30},{28,30}},
                lineColor={0,0,0},
                textString="%cond"),
              Text(
                extent={{-134,-90},{154,-30}},
                textString="%name",
                lineColor={0,0,255}),
              Line(
                points={{30,30},{30,-30},{70,0},{30,30}},
                color={0,0,255},
                smooth=Smooth.None)}));
      end OneWayResistorWithCondParam;

      model PumpBase "Defined flow to/from/in system by real signal"


        Interfaces.RealInput_ desiredFlow( quantity="Flow", unit = "ml/min")
          "desired volume flow value"                                                                    annotation (Placement(transformation(
                extent={{-66,50},{-26,90}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,60})));
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-40},{100,60}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,35},{80,10},{-80,-15},{-80,35}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-90},{150,-50}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague, Czech Republic</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
</table>
</html>",       info="<html>
<p><font style=\"font-size: 9pt; \">This element needs to be connected only to next hydraulic elements, which contain calculation of hydraulic pressure in connector. It is because equation contains only </font><b><font style=\"font-size: 9pt; \">hydraulic volume flow</font></b><font style=\"font-size: 9pt; \"> variable, which is set to value of input signal variable. </font></p>
</html>"));
      end PumpBase;

      model InputPump
        extends PumpBase;
        NegativePressureFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,0},{110,20}}),   iconTransformation(
                extent={{90,0},{110,20}})));

      equation
        q_out.q = - desiredFlow;

        annotation (Icon(graphics));
      end InputPump;

      model OutputPump
        extends PumpBase;
        PositivePressureFlow q_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent=
                  {{-110,-10},{-90,10}})));

      equation
        q_in.q = desiredFlow;
      end OutputPump;

      model Pump
        extends PumpBase;
        NegativePressureFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
        PositivePressureFlow q_in annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));
      equation
        q_in.q + q_out.q = 0;
        q_in.q = desiredFlow;
      end Pump;

      model PressurePumpBase
        "Defined pressure to/from/in system by real signal"

        Interfaces.RealInput_ desiredPressure(quantity="Pressure", unit="mmHg")
          "desired pressure flow value"                                                                  annotation (Placement(transformation(
                extent={{-66,50},{-26,90}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,60})));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-80,60},{80,-60}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={255,170,170},
                fillColor={255,170,170},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-94},{150,-54}},
                textString="%name",
                lineColor={0,0,0},
                fillPattern=FillPattern.Solid,
                fillColor={255,170,170}),
              Text(
                extent={{-36,-72},{-152,98}},
                lineColor={255,170,170},
                fillColor={255,170,170},
                fillPattern=FillPattern.Solid,
                textString="")}));
      end PressurePumpBase;

      model InputPressurePump
        extends PressurePumpBase;
        NegativePressureFlow p_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,0},{110,20}}),   iconTransformation(
                extent={{70,-10},{90,10}})));

      equation
        p_out.pressure = desiredPressure;

        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics));
      end InputPressurePump;

      model OutputPressurePump
        extends PressurePumpBase;
        PositivePressureFlow p_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,0},{110,20}}),   iconTransformation(
                extent={{-90,-10},{-70,10}})));

      equation
        p_in.pressure = desiredPressure;

        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics), Icon(graphics));
      end OutputPressurePump;

      model ReabsorbtionWithMinimalOutflow
        "Divide inflow to outflow and reabsorbtion if it is under defined treshold"

        Library.PressureFlow.PositivePressureFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-100,40},{-60,80}}),  iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.PressureFlow.NegativePressureFlow Outflow 
          annotation (Placement(transformation(extent={{58,40},{98,80}}),
              iconTransformation(extent={{90,-10},{110,10}})));
        Library.PressureFlow.NegativePressureFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-12,-98},{28,-58}}), iconTransformation(
                extent={{-10,-110},{10,-90}})));
        HumMod.Library.Interfaces.RealInput_ FractReab 
                                     annotation (Placement(transformation(extent={{-60,-26},
                  {-48,-14}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=180,
              origin={94,-56})));
        HumMod.Library.Interfaces.RealInput_ OutflowMin(final quantity="Flow",
            final unit="ml/min")     annotation (Placement(transformation(extent={{-44,84},
                  {-32,96}}),  iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=180,
              origin={34,50})));
        FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{-38,22},{-18,2}})));
        Factors.SimpleMultiply simpleMultiply annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-28,-20})));
        Pump reabsorbtion annotation (Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=270,
              origin={10,-34})));
        Pump MinimalFlow annotation (Placement(transformation(
              extent={{10,10},{-10,-10}},
              rotation=180,
              origin={-2,60})));
        FlowMeasure flowMeasure1 
          annotation (Placement(transformation(extent={{-66,50},{-46,70}})));
        Modelica.Blocks.Math.Min min 
          annotation (Placement(transformation(extent={{-24,82},{-14,92}})));
      equation
        connect(flowMeasure.actualFlow, simpleMultiply.yBase) annotation (Line(
            points={{-28,6.6},{-28,-18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(simpleMultiply.u, FractReab) annotation (Line(
            points={{-37.8,-20},{-54,-20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(reabsorbtion.q_out, Reabsorbtion) annotation (Line(
            points={{10,-44},{10,-78},{8,-78}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(simpleMultiply.y, reabsorbtion.desiredFlow) annotation (Line(
            points={{-28,-22},{-28,-34},{4,-34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.q_out, Outflow) annotation (Line(
            points={{-22,12},{43,12},{43,60},{78,60}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, reabsorbtion.q_in) annotation (Line(
            points={{-22,12},{10,12},{10,-24}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(Inflow, flowMeasure1.q_in) annotation (Line(
            points={{-80,60},{-62,60}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure1.q_out, flowMeasure.q_in) annotation (Line(
            points={{-50,60},{-42,60},{-42,12},{-34,12}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure1.q_out, MinimalFlow.q_in) annotation (Line(
            points={{-50,60},{-12,60}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(MinimalFlow.q_out, Outflow) annotation (Line(
            points={{8,60},{78,60}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(OutflowMin, min.u1) annotation (Line(
            points={{-38,90},{-25,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure1.actualFlow, min.u2) annotation (Line(
            points={{-56,65.4},{-56,84},{-25,84}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(min.y, MinimalFlow.desiredFlow) annotation (Line(
            points={{-13.5,87},{-2,87},{-2,66}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-100,134},{100,106}},
                lineColor={0,0,255},
                textString="%name"), Bitmap(extent={{-100,100},{100,-100}},
                  fileName="icons/reabsorbtion2.jpg")}),
                                       Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ReabsorbtionWithMinimalOutflow;

      model Reabsorbtion "Divide inflow to outflow and reabsorbtion"

        Library.PressureFlow.PositivePressureFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-106,-18},{-66,22}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.PressureFlow.NegativePressureFlow Outflow 
          annotation (Placement(transformation(extent={{50,-18},{90,22}}),
              iconTransformation(extent={{90,-10},{110,10}})));
        Library.PressureFlow.NegativePressureFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-20,-100},{20,-60}}),iconTransformation(
                extent={{-10,-110},{10,-90}})));
        HumMod.Library.Interfaces.RealInput_ FractReab 
                                     annotation (Placement(transformation(extent={{-74,-38},
                  {-58,-22}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=180,
              origin={104,-46})));
        FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{-48,12},{-28,-8}})));
        Factors.SimpleMultiply simpleMultiply annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-38,-30})));
        Pump pump annotation (Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=270,
              origin={0,-44})));
      equation
        connect(Inflow, flowMeasure.q_in) annotation (Line(
            points={{-86,2},{-44,2}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, simpleMultiply.yBase) annotation (Line(
            points={{-38,-3.4},{-38,-28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(simpleMultiply.u, FractReab) annotation (Line(
            points={{-47.8,-30},{-66,-30}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pump.q_out, Reabsorbtion) annotation (Line(
            points={{-1.83697e-015,-54},{0,-54},{0,-80},{0,-80}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(simpleMultiply.y, pump.desiredFlow) annotation (Line(
            points={{-38,-32},{-38,-44},{-6,-44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.q_out, Outflow) annotation (Line(
            points={{-32,2},{70,2}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, pump.q_in) annotation (Line(
            points={{-32,2},{1.83697e-015,2},{1.83697e-015,-34}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{-100,130},{100,108}},
                lineColor={0,0,255},
                textString="%name"), Bitmap(extent={{-100,100},{100,-100}},
                  fileName="icons/reabsorbtion.png")}),
                                       Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end Reabsorbtion;

      model GravityHydrostaticDifferenceWithPumpEffect
        "Create hydrostatic pressure between connectors in different altitude with specific pressure pump effect"

        Library.PressureFlow.PositivePressureFlow q_up 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{62,38},
                  {82,58}})));


        Library.PressureFlow.PositivePressureFlow q_down 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{60,-64},
                  {80,-44}})));
        Library.Interfaces.RealInput_ height(final unit="m") 
                                                     annotation (Placement(transformation(extent={{22,
                  -18},{62,22}}), iconTransformation(extent={{15,-15},{-15,15}},
              rotation=180,
              origin={-61,-3})));

       // constant Real g(final unit="m/s2")=9.81; //gravity acceleration constant
        parameter Real ro(final unit="kg/m3")=1060; //liquid density

       //TorsoArtyGradient  =  TorsoCM * Gravity_Gz * ArtyFractGz;  //Hydrostatic pressure: P=ro.g.h
         //cm *  .01 m/cm * 9.81 m/s * 1060 kg/m3 * 1/133.322 mmHg/Pa
         //= height[cm] * .77996129671022036873134216408395 [mmHg]
         //Blood density = 1060 kg/m3: Cutnell, John & Johnson, Kenneth. Physics, Fourth Edition. Wiley, 1998: 308.

         //bloodDensity = 1060 kg/m3
         //gravityAcceleration = 9.81 m/s2
         //Pa2mmHg = 1/133.322 mmHg/Pa
         //cm2m = .01 m/cm
         //averageHydrostaticHeight = TorsoCM * postureCoef[Status_Posture]
         //hydrostaticPressure = averageHydrostaticHeight * cm2m * bloodDensity * gravityAcceleration * Pa2mmHg
         //                    = .77996 * averageHydrostaticHeight

        Library.Interfaces.RealInput_ pumpEffect     annotation (Placement(transformation(extent={{22,
                  -18},{62,22}}), iconTransformation(extent={{15,-15},{-15,15}},
              rotation=90,
              origin={-15,95})));

        Library.Interfaces.RealInput_ G(final unit="m/(s.s)") 
                                                     annotation (Placement(transformation(extent={{22,
                  -18},{62,22}}), iconTransformation(extent={{15,-15},{-15,15}},
              rotation=270,
              origin={15,-85})));
      equation
        q_down.pressure = q_up.pressure + Library.NonSIunits.to_mmHg(G*ro*height)
          *pumpEffect;
        q_up.q + q_down.q = 0;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-30,-58},{30,72}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-62,-21},{62,21}},
                textString="%name",
                lineColor={0,0,255},
                origin={0,5},
                rotation=90),
              Line(
                points={{-28,44},{0,72},{28,44}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Bitmap(extent={{-100,102},{100,-98}}, fileName=
                    "icons/hydrostaticGradient.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end GravityHydrostaticDifferenceWithPumpEffect;

      model GravityHydrostaticDifference
        "Create hydrostatic pressure between connectors in different altitude"

        Library.PressureFlow.PositivePressureFlow q_up 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{62,38},
                  {82,58}})));


        Library.PressureFlow.PositivePressureFlow q_down 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{60,-66},
                  {80,-46}})));
        Library.Interfaces.RealInput_ height(final unit="m") 
                                                     annotation (Placement(transformation(extent={{22,
                  -18},{62,22}}), iconTransformation(extent={{15,-15},{-15,15}},
              rotation=180,
              origin={-61,-5})));

      //  constant Real g(final unit="m/s2")=9.81; //gravity acceleration constant
        parameter Real ro(final unit="kg/m3")=1060; //liquid density

       //TorsoArtyGradient  =  TorsoCM * Gravity_Gz * ArtyFractGz;  //Hydrostatic pressure: P=ro.g.h
         //cm *  .01 m/cm * 9.81 m/s * 1060 kg/m3 * 1/133.322 mmHg/Pa
         //= height[cm] * .77996129671022036873134216408395 [mmHg]
         //Blood density = 1060 kg/m3: Cutnell, John & Johnson, Kenneth. Physics, Fourth Edition. Wiley, 1998: 308.

         //bloodDensity = 1060 kg/m3
         //gravityAcceleration = 9.81 m/s2
         //Pa2mmHg = 1/133.322 mmHg/Pa
         //cm2m = .01 m/cm
         //averageHydrostaticHeight = TorsoCM * postureCoef[Status_Posture]
         //hydrostaticPressure = averageHydrostaticHeight * cm2m * bloodDensity * gravityAcceleration * Pa2mmHg
         //                    = .77996 * averageHydrostaticHeight

        Library.Interfaces.RealInput_ G(final unit="m/(s.s)") 
                                                     annotation (Placement(transformation(extent={{22,
                  -18},{62,22}}), iconTransformation(extent={{15,-15},{-15,15}},
              rotation=270,
              origin={15,-85})));
      equation
        q_down.pressure = q_up.pressure + Library.NonSIunits.to_mmHg(G*ro*height);
        q_up.q + q_down.q = 0;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-30,-58},{30,72}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-62,-21},{62,21}},
                textString="%name",
                lineColor={0,0,255},
                origin={0,5},
                rotation=90),
              Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/hydrostaticGradient.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end GravityHydrostaticDifference;

        model VolumeCompartement
        "Generate constant pressure independ on inflow or outflow"
         //extends Library.Interfaces.BaseModel;
          extends HumMod.Library.Utilities.DynamicState;

          parameter Real pressure=0;

          PositivePressureFlow con 
            annotation (Placement(transformation(extent={{100,-20},{140,20}},
                rotation=0), iconTransformation(extent={{-120,-20},{-80,20}})));

          parameter Real initialVolume;

          Interfaces.RealOutput_ Volume(start=initialVolume) annotation (Placement(transformation(extent={{
                    -8,-120},{32,-80}}), iconTransformation(
                extent={{-20,-20},{20,20}},
                rotation=270,
                origin={12,-100})));
        //initial equation
        //  Volume = initialVolume;
        equation
          /*if STEADY then
    der(Volume) = 0;
  else
    der(Volume) = con.q/Library.SecPerMin;
  end if;
*/
          /*if Volume<=0 and con.q<=0 then
    der(Volume) = 0;
  else
    con.pressure = pressure;
  end if;
 */
          if Volume>0 or con.q>0 then
                    con.pressure = pressure;
                  else
                    con.q = 0;
                  end if;

          stateValue = Volume;
          changePerMin = con.q;
          annotation (
            Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),
            graphics,
                  lineColor={0,0,0}), Rectangle(
                extent={{-100,100},{100,-100}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid));
        end VolumeCompartement;

        model PressureControledCompartment
        "Multiple PressureFlow connector with pressures from multiple inputs"
         extends Library.Interfaces.BaseModel;
         extends HumMod.Library.Utilities.DynamicState;

          parameter Integer ncon=1 "Number of inputs";
          Library.Interfaces.RealInput_ pressure[ncon](final quantity="Pressure",
              final unit="mmHg") "Connector of Pressure values input signals" 
            annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                rotation=0), iconTransformation(extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,100})));
          PositivePressureFlow y[ncon]
          "Connector of PressureFlow output connectors" 
            annotation (Placement(transformation(extent={{100,-20},{140,20}},
                rotation=0), iconTransformation(extent={{-120,-20},{-80,20}})));

          parameter Real initialVolume;
          Interfaces.RealOutput_ Volume(start=initialVolume, final quantity="Volume", final unit =                    "ml") 
            annotation (Placement(transformation(extent={{-20,-120},{20,-80}}),
              iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,-100})));

        //initial equation
        //  Volume = initialVolume;
         // der(Volume) = 0;
        equation
        /*  if STEADY then
      der(Volume) = 0;
  else
      der(Volume) = (ones(ncon)*y.q)/Library.SecPerMin;
  end if;
*/
          for i in 1:ncon loop
              y[i].pressure = pressure[i];
          end for;

          stateValue = Volume;
          changePerMin = ones(ncon)*y.q;
          annotation (Documentation(info="<html>
<p>Model has a vector of continuous Real input signals as pressures for vector of pressure-flow connectors. </p>
<p>Usage in tests: Set defaul volume&GT;0 and try to set STEADY in instances to &QUOT;false&QUOT;!</p>
</html>",  revisions=
             "<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),  Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),
                            graphics));
        end PressureControledCompartment;

      model FromMLtoMMOL "conversion of flow units from ml to mmol"
        extends Modelica.SIunits.Conversions.ConversionIcon;
        PositivePressureFlow q_ML "flow in ml" annotation (Placement(transformation(
                extent={{-110,-12},{-90,8}}), iconTransformation(extent={{-110,-10},{-90,
                  10}})));
        NegativePressureFlow q_MMOL "flow in mmol" annotation (Placement(
              transformation(extent={{-110,-12},{-90,8}}), iconTransformation(extent={
                  {90,-10},{110,10}})));

        parameter Real mmolPerML(final unit="mmol/ml")
          "this density depends on substance";

      equation
        q_ML.q * mmolPerML + q_MMOL.q = 0;
        q_ML.pressure = q_MMOL.pressure;
        annotation (Icon(graphics={
              Text(
                extent={{-56,60},{-94,8}},
                lineColor={0,0,0},
                textString="ml"),
              Text(
                extent={{96,-20},{-28,-70}},
                lineColor={0,0,0},
                textString="mmol"),
              Text(
                extent={{100,-108},{-102,-130}},
                lineColor={0,0,0},
                textString="%mmolPerML mmol/ml")}), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FromMLtoMMOL;
      annotation (Documentation(revisions="<html>
</html>"));
    end PressureFlow;

    package VolumeFlow "Volume and Volume Flow domains"

      connector VolumeFlow
        Real volume(  final quantity="Volume", final unit="ml");
        flow Real q(  final quantity="Flow", final unit="ml/min");
      end VolumeFlow;

      connector PositiveVolumeFlow
        extends VolumeFlow;

      annotation (
          defaultComponentName="q_in",
          Coordsys(
            extent=[-100, -100; 100, 100],
            grid=[1, 1],
            component=[20, 20],
            scale=0.2),
          Icon(coordinateSystem(extent=[-100, -100; 100, 100]), Polygon(points=[-50,-5;
                  0,40; 50,-5; 0,-51; -50,-5],            style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0}))),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>"));
      end PositiveVolumeFlow;

      connector NegativeVolumeFlow
         extends VolumeFlow;

      annotation (
          defaultComponentName="q_out",
          Coordsys(
            extent=[-100, -100; 100, 100],
            grid=[1, 1],
            component=[20, 20],
            scale=0.2),
          Icon(coordinateSystem(extent=[-100, -100; 100, 100]), Polygon(points=[-50,-5;
                  0,40; 50,-5; 0,-51; -50,-5],            style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={255,255,255}))),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={255,255,255})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>"));
      end NegativeVolumeFlow;

      model VolumeMeasure


        PositiveVolumeFlow q_in   annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-20,-60},
                  {20,-20}})));
        HumMod.Library.Interfaces.RealOutput_ actualVolume 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=90,
              origin={0,40})));
      equation
        q_in.q = 0;

        actualVolume = q_in.volume;

        annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-40,-40},{40,40}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-28,18},{30,-40}},
                lineColor={0,0,0},
                fillPattern=FillPattern.VerticalCylinder,
                fillColor={215,215,215},
                textString="Vol")}),    Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics));
      end VolumeMeasure;

      model InputPump

        NegativeVolumeFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
        HumMod.Library.Interfaces.RealInput desiredFlow(final quantity="Flow",
            final unit="ml/min")      annotation ( extent = [-10,50;10,70], rotation = -90);


      equation
        q_out.q = - desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
                    Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end InputPump;

      model OutputPump

        PositiveVolumeFlow q_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        HumMod.Library.Interfaces.RealInput desiredFlow(final quantity="Flow",
            final unit="ml/min")      annotation ( extent = [-10,50;10,70], rotation = -90);


      equation
        q_in.q = desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
                    Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end OutputPump;
    end VolumeFlow;

    package ConcentrationFlow "Concentration Physical Domain"

      replaceable type Concentration = Real (final quantity="Concentration");

      replaceable type SoluteFlow = Real (final quantity="Flow");

      replaceable type SoluteMass_ = Real (final quantity="Mass");

      connector ConcentrationFlow "Concentration and Solute flow"
        Concentration conc;
        flow SoluteFlow q;
        annotation (Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ConcentrationFlow;

      connector PositiveConcentrationFlow "Concentration and Solute inflow"
        extends HumMod.Library.ConcentrationFlow.ConcentrationFlow;

      annotation (
          defaultComponentName="q_in",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={200,0,0},
                lineThickness=1), Polygon(
                points={{0,100},{100,0},{0,-100},{-100,0},{0,100}},
                lineColor={200,0,0},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={200,0,0})}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>", revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));

      end PositiveConcentrationFlow;

      connector NegativeConcentrationFlow
        "Concentration and negative Solute outflow"
        extends HumMod.Library.ConcentrationFlow.ConcentrationFlow;

      annotation (
          defaultComponentName="q_out",
          Icon(coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=false),
              graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={200,0,0},
                lineThickness=1), Polygon(
                points={{-100,0},{0,100},{100,0},{0,-100},{-100,0}},
                lineColor={200,0,0},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={255,240,240})}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={255,255,255})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>", revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));

      end NegativeConcentrationFlow;

      model FlowMeasure

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{18,-10},{38,10}}), iconTransformation(
                extent={{30,-10},{50,10}})));


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-50,-10},
                  {-30,10}})));
        HumMod.Library.Interfaces.RealOutput_ actualFlow 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=90,
              origin={0,50})));
      equation
        q_in.q + q_out.q = 0;
        q_out.conc = q_in.conc;

        actualFlow = q_in.q;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-40,30},{40,-30}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FlowMeasure;

      model ConcentrationMeasure
        parameter String unitsString="";
        parameter Real toAnotherUnitCoef=1;


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-10,-10},
                  {10,10}})));
        HumMod.Library.Interfaces.RealOutput_ actualConc 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=90,
              origin={0,40})));
      equation

        actualConc =  toAnotherUnitCoef * q_in.conc;
        q_in.q = 0;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Text(
                extent={{-48,-24},{48,-40}},
                lineColor={0,0,0},
                textString="%unitsString"), Rectangle(
                extent={{-20,20},{20,-20}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ConcentrationMeasure;

      partial model ResistorBase
       extends HumMod.Library.Interfaces.BaseResistorModel;
        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-96,-10},{-76,10}}), iconTransformation(
                extent={{-96,-10},{-76,10}})));
        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(
                extent={{76,-10},{96,10}})));
        annotation (Icon(graphics={Bitmap(extent={{-100,100},{100,-100}},
                  fileName="icons/diffusion.png")}), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorBase;

      partial model Resistor
       extends ResistorBase;
      equation
        q_in.q + q_out.q = 0;
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end Resistor;

      model ResistorWithCondParam
       extends Resistor;
       parameter Real cond
          "speed of solute in dependence on concentration gradient";
      equation
        q_in.q = cond * (q_in.conc - q_out.conc);
        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}),
                            graphics), Icon(graphics={Text(
                extent={{-70,38},{70,80}},
                lineColor={0,0,0},
                textString="%cond"), Text(
                extent={{-142,-90},{142,-40}},
                textString="%name",
                lineColor={0,0,255})}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorWithCondParam;

      model SolventFlowPump

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out
          "second side connector with value of q (solute mass flow) and conc (concentration)"
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));

        Interfaces.RealInput_ solventFlow
          "solvent flow (solution volume flow = solventFlow + solute volume flow)"
                                      annotation ( extent = [-10,50;10,70], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in
          "first side connector with value of q (solute mass flow) and conc (concentration)"
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

      equation
        q_in.q + q_out.q = 0;
        q_in.q = if initial() or (solventFlow>=0) then 
           solventFlow*q_in.conc else 
           solventFlow*q_out.conc;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
</table>
</html>",       info="<html>
<p><h4><font color=\"#008000\">Bidirectional mass flow by concentration</font></h4></p>
<p>Possible field values: </p>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0.1\"><tr>
<td></td>
<td><p align=\"center\">forward flow</p></td>
<td><p align=\"center\">backward flow</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>solventFlow</h4></p></td>
<td><p align=\"center\">&GT;=0</p></td>
<td><p align=\"center\">&LT;0</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>q_in.q</h4></p></td>
<td><p align=\"center\">=solventFlow*q_in.conc</p></td>
<td><p align=\"center\">=solventFlow*q_out.conc</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>q_out.q</h4></p></td>
<td><p align=\"center\">=-q_in.q</p></td>
<td><p align=\"center\">=-q_in.q</p></td>
</tr>
</table>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SolventFlowPump;

      model SolventOutflowPump

        HumMod.Library.Interfaces.RealInput solventFlow "solvent outflow" 
                                      annotation ( extent = [-10,50;10,70], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in
          "solute outflow"        annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

        parameter Real K=1 "part of real mass flow in solution outflow";
      equation
        q_in.q = K*solventFlow*q_in.conc;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255}),
              Text(
                extent={{-100,-30},{100,-50}},
                lineColor={0,0,0},
                textString="K=%K")}),   Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
</table>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SolventOutflowPump;

      model Synthesis
        parameter SoluteFlow SynthesisBasic =   0.01;
        parameter Real[:,3] data =  {{ 20.0,  3.0,  0.0}, { 28.0,  1.0,  -0.2}, { 40.0,  0.0,  0.0}}
          "COPEffect";
        Library.Curves.Curve c(x=data[:,1],y=data[:,2],slope=data[:,3]);

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));

        Real COP;
        SoluteMass_ synthetized( start=0);
      protected
        parameter Real C1 =    320.0;
        parameter Real C2 =    1160.0;
      equation
        COP = if  q_out.conc > 0.0 then   ( C1 * q_out.conc)  + ( C2 * q_out.conc * q_out.conc) else   0.0;
        c.u=COP;
        q_out.q = - SynthesisBasic*c.val;

        der(synthetized) = -q_out.q / Library.SecPerMin;
       annotation (
          defaultComponentName="synthesis",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-100,-50},{90,50}},
                lineColor={0,0,255},
                textString="%name")}),  Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end Synthesis;

      model Degradation
        parameter SoluteFlow DegradationBasic =   0.01;
        parameter Real[:,3] data =  {{ 0.00,  0.0,  0.0}, { 0.07,  1.0,  40.0}, { 0.09,  6.0,  0.0}}
          "ProteinEffect";
        Library.Curves.Curve c(x=data[:,1],y=data[:,2],slope=data[:,3]);
        PositiveConcentrationFlow q_in 
          annotation (Placement(transformation(extent={{-100,0},{-60,40}}),
              iconTransformation(extent={{-110,-10},{-90,10}})));


      SoluteMass_ degraded( start=0);
      equation
        c.u=q_in.conc;
        q_in.q = DegradationBasic*c.val;

        der(degraded) = q_in.q / Library.SecPerMin;
       annotation (
          defaultComponentName="degradation",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-88,-50},{100,50}},
                lineColor={0,0,255},
                textString="%name")}),  Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end Degradation;

      model InputPump

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={
                  {50,-10},{70,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow "speed of solute flow"
                                                                      annotation ( extent = [-10,30;10,50], rotation = -90);


      equation
        q_out.q = - desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-30},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,20},{50,0},{-48,-21},{-48,20}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-92,-54},{80,-30}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end InputPump;

      model OutputPump

        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{-70,-10},
                  {-50,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow 
                                       annotation ( extent = [-10,30;10,50], rotation = -90);


      equation
        q_in.q = desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-32},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,18},{50,-2},{-48,-26},{-48,18}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-78,-54},{72,-32}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end OutputPump;

      model SoluteFlowPump

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
        HumMod.Library.Interfaces.RealInput soluteFlow 
                                      annotation ( extent = [-10,50;10,70], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));
      equation
        q_in.q + q_out.q = 0;
        q_in.q = soluteFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SoluteFlowPump;

      model PartialPressure
        "partial gas concentration in ml/ml multiplied by ambient pressure"

        Library.PressureFlow.NegativePressureFlow outflow annotation (Placement(
              transformation(extent={{-20,-120},{20,-80}}), iconTransformation(extent=
                 {{-10,-110},{10,-90}})));
        Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
          annotation (Placement(transformation(extent={{-20,80},{20,120}}),
              iconTransformation(extent={{-10,90},{10,110}})));
        Interfaces.RealInput_ ambientPressure(final unit="mmHg") annotation (Placement(transformation(extent={{
                  -60,-20},{-20,20}}), iconTransformation(extent={{-60,-20},{-20,20}})));
      equation
        q_in.q+outflow.q=0;
        outflow.pressure=q_in.conc*ambientPressure;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-40,100},{40,-100}},
                lineColor={0,0,255},
                fillColor={215,215,215},
                fillPattern=FillPattern.Solid)}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end PartialPressure;

      model Dilution
      // parameter Real DilutionCoef=1-47/760;


        NegativeConcentrationFlow q_diluted 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,10},{110,30}}), iconTransformation(extent={{90,50},
                  {110,70}})));
        PositiveConcentrationFlow q_concentrated 
                                  annotation (Placement(
              transformation(extent={{-130,-28},{-90,12}}), iconTransformation(extent={{-110,50},
                  {-90,70}})));
        Interfaces.RealInput_ addedSolventPart
          "part of added solvent volume(not containing solute) normalized to whole summed volume"
          annotation (Placement(transformation(extent={{-16,8},{24,48}}),
              iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=0,
              origin={-92,-68})));
      equation
        q_diluted.q+q_concentrated.q=0;
        q_diluted.conc = (1-addedSolventPart) * q_concentrated.conc;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Bitmap(extent={{-100,100},{100,-102}},
                  fileName="icons/dilution.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end Dilution;

      model Reabsorbtion

        Library.ConcentrationFlow.PositiveConcentrationFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-120,-18},{-80,22}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Outflow 
          annotation (Placement(transformation(extent={{0,-100},{40,-60}}),
              iconTransformation(extent={{90,-10},{110,10}})));

        Library.ConcentrationFlow.NegativeConcentrationFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-20,-100},{20,-60}}),iconTransformation(
                extent={{-10,-50},{10,-30}})));
        HumMod.Library.Interfaces.RealInput_ ReabsorbedFract(final unit="1") 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                      rotation=-90,
              origin={0,40})));

      equation
        Outflow.q + Inflow.q + Reabsorbtion.q = 0;
        Outflow.conc = Inflow.conc;
        Reabsorbtion.q = - ReabsorbedFract * Inflow.q;
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(
                points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,-28},
                    {6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,8},{-58,
                    12},{-70,14}},
                color={0,0,255},
                smooth=Smooth.None),
              Text(
                extent={{12,-54},{166,-84}},
                lineColor={0,0,255},
                textString="%name")}), Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end Reabsorbtion;

      model FractReabsorbtion

        Library.ConcentrationFlow.PositiveConcentrationFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-120,-18},{-80,22}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Outflow 
          annotation (Placement(transformation(extent={{0,-100},{40,-60}}),
              iconTransformation(extent={{90,-10},{110,10}})));

        Library.ConcentrationFlow.NegativeConcentrationFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-20,-100},{20,-60}}),iconTransformation(
                extent={{-10,-50},{10,-30}})));
        HumMod.Library.Interfaces.RealInput_ Normal(final unit="1") 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                      rotation=-90,
              origin={-60,40})));
        HumMod.Library.Interfaces.RealInput_ Effects(final unit="1") 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{60,20},{100,60}},rotation=-90)));

      parameter SoluteFlow MaxReab=14 "maximum reabsorbtion solute flow";
        Interfaces.RealOutput_ ReabFract(final unit="1") annotation (Placement(transformation(extent={{80,-60},
                  {120,-20}}), iconTransformation(extent={{80,-60},{120,-20}})));
      equation
        Outflow.q + Inflow.q + Reabsorbtion.q = 0;
        Outflow.conc = Inflow.conc;
        Reabsorbtion.q = -min(ReabFract * Inflow.q, MaxReab);
        ReabFract = if (Normal<=0) or (Effects<=0) then 0 else if Normal>1 then 1 else Normal^(1/Effects);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(
                points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,-28},
                    {6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,8},{-58,
                    12},{-70,14}},
                color={0,0,255},
                smooth=Smooth.None),
              Text(
                extent={{12,-54},{166,-84}},
                lineColor={0,0,255},
                textString="%name"),
              Text(
                extent={{-100,-40},{-4,-62}},
                lineColor={0,0,255},
                textString="%MaxReab = MaxReab")}),
                                       Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FractReabsorbtion;

      model FractReabsorbtion2

        Library.ConcentrationFlow.PositiveConcentrationFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-120,-18},{-80,22}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Outflow 
          annotation (Placement(transformation(extent={{0,-100},{40,-60}}),
              iconTransformation(extent={{90,-10},{110,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-20,-100},{20,-60}}),iconTransformation(
                extent={{-10,-50},{10,-30}})));
        HumMod.Library.Interfaces.RealInput_ Normal(final unit="1") 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                      rotation=-90,
              origin={-60,40})));
        HumMod.Library.Interfaces.RealInput_ Effects(final unit="1") 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                      rotation=-90,
              origin={80,40})));
        Real ReabFract(final unit="1");
        HumMod.Library.Interfaces.RealInput_ MaxReab 
                                     annotation (Placement(transformation(extent={{-20,20},{20,
                  60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                      rotation=90,
              origin={-60,-42})));
      equation
        Outflow.q + Inflow.q + Reabsorbtion.q = 0;
        Outflow.conc = Inflow.conc;
        Reabsorbtion.q = -min(ReabFract * Inflow.q, MaxReab);
        ReabFract = if (Normal<=0) or (Effects<=0) then 0 else if Normal>1 then 1 else Normal^(1/Effects);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(
                points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,-28},
                    {6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,8},{-58,
                    12},{-70,14}},
                color={0,0,255},
                smooth=Smooth.None),
              Text(
                extent={{12,-42},{166,-72}},
                lineColor={0,0,255},
                textString="%name")}), Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FractReabsorbtion2;

      model ConstLimitedReabsorbtion

        Library.ConcentrationFlow.PositiveConcentrationFlow Inflow 
                                                       annotation (Placement(
              transformation(extent={{-120,-18},{-80,22}}), iconTransformation(
                extent={{-110,-10},{-90,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Outflow 
          annotation (Placement(transformation(extent={{0,-100},{40,-60}}),
              iconTransformation(extent={{90,-10},{110,10}})));
        Library.ConcentrationFlow.NegativeConcentrationFlow Reabsorbtion 
                                                         annotation (Placement(
              transformation(extent={{-20,-100},{20,-60}}),iconTransformation(
                extent={{-10,-50},{10,-30}})));

      parameter SoluteFlow MaxReab=250 "maximum reabsorbtion solute flow";
      equation
        Outflow.q + Inflow.q + Reabsorbtion.q = 0;
        Outflow.conc = Inflow.conc;
        Reabsorbtion.q = -min( Inflow.q, MaxReab);
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Line(
                points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,-28},
                    {6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,8},{-58,
                    12},{-70,14}},
                color={0,0,255},
                smooth=Smooth.None),
              Text(
                extent={{12,-42},{166,-72}},
                lineColor={0,0,255},
                textString="%name"),
              Text(
                extent={{-100,-40},{-4,-62}},
                lineColor={0,0,255},
                textString="%MaxReab = MaxReab")}),
                                       Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ConstLimitedReabsorbtion;

      model FlowConcentrationMeasure

        PositiveConcentrationFlow q_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent=
                  {{-110,-10},{-90,10}})));
        HumMod.Library.Interfaces.RealInput_ SolventFlow(final quantity="Flow",
            final unit="ml/min")       annotation ( extent = [-10,50;10,70], rotation = -90);


        Interfaces.RealInput_ AdditionalSoluteFlow annotation (Placement(
              transformation(extent={{-60,60},{-20,100}}), iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={0,-60})));
        Interfaces.RealOutput_ Conc annotation (Placement(transformation(extent=
                 {{82,-20},{122,20}}), iconTransformation(extent={{82,-20},{122,
                  20}})));
      equation
        Conc = q_in.conc + AdditionalSoluteFlow/SolventFlow;
        q_in.q = 0;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid), Text(
                extent={{-88,-50},{80,50}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end FlowConcentrationMeasure;

      model SimpleReaction

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
        HumMod.Library.Interfaces.RealInput_ coef
          "who much units of q_out produce one unit of q_in" 
                                      annotation ( extent = [-10,30;10,50], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));
      equation
        q_out.q + coef*q_in.q = 0;
        q_out.conc = coef*q_in.conc;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-28},{100,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-128,-60},{142,-34}},
                textString="%name",
                lineColor={0,0,255}),
              Polygon(
                points={{-60,4},{-60,2},{54,2},{54,2},{18,12},{18,4},{-60,4}},
                lineColor={0,0,0},
                smooth=Smooth.None,
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid)}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SimpleReaction;

      model SimpleReaction2

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,20},{110,40}}), iconTransformation(extent={{90,
                  20},{110,40}})));
        HumMod.Library.Interfaces.RealInput_ coef
          "who much units of q_out produce one unit of q_in" 
                                      annotation ( extent = [-10,30;10,50], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));
        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out2 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={{
                  90,-40},{110,-20}})));
        Interfaces.RealInput_ coef2
          "who much units of q_out2 produce one unit of q_in"                           annotation (Placement(transformation(extent={{-54,
                  20},{-14,60}}), iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={60,40})));
      equation
        q_out.q + coef*q_in.q = 0;
        q_out2.q + coef2*q_in.q = 0;
        q_out.conc = coef*q_in.conc;
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-30},{100,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-128,-66},{142,-40}},
                textString="%name",
                lineColor={0,0,255}),
              Polygon(
                points={{-60,4},{-60,2},{54,2},{54,2},{18,12},{18,4},{-60,4}},
                lineColor={0,0,0},
                smooth=Smooth.None,
                fillColor={0,0,0},
                fillPattern=FillPattern.Solid)}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SimpleReaction2;

      model UnlimitedStorage
        //extends HumMod.Library.Interfaces.BaseModel;

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                                   annotation (Placement(
              transformation(extent={{62,-32},{102,8}}),  iconTransformation(extent={{-8,-10},
                  {12,10}})));
        parameter Real concentration(final unit="%");

      equation
        q_out.conc = 0.01*concentration;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Text(
                extent={{-100,52},{100,74}},
                lineColor={0,0,0},
                fillPattern=FillPattern.VerticalCylinder,
                fillColor={215,215,215},
                textString="(%concentration%%)")}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end UnlimitedStorage;

      model ConcentrationCompartment
        extends HumMod.Library.Utilities.DynamicState;
        //extends QHP.Library.Interfaces.BaseModel;

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                                   annotation (Placement(
              transformation(extent={{62,-32},{102,8}}),  iconTransformation(extent={{-10,-10},
                  {10,10}})));
        parameter Real initialSoluteMass;

        HumMod.Library.Interfaces.RealInput_ SolventVolume(final quantity=
              "Volume", final unit="ml") 
                                 annotation (Placement(transformation(extent={{-120,68},{-80,108}}),
              iconTransformation(extent={{-100,40},{-60,80}})));
        HumMod.Library.Interfaces.RealOutput_ SoluteMass(start=initialSoluteMass) 
          annotation (Placement(transformation(extent={{-20,-120},{20,-80}}, rotation=
                 -90,
              origin={102,-102}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,-78})));

      //initial equation
      //  SoluteMass = initialSoluteMass;
      equation
          q_out.conc = if (SolventVolume>0) then SoluteMass / SolventVolume else 0;
      //  q_out.conc = if initial() or (SolventVolume>0) then SoluteMass / SolventVolume else 0;
      //  der(SoluteMass) = q_out.q / Library.SecPerMin;
        stateValue = SoluteMass;

        changePerMin = q_out.q;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/concentrationCompartement.png"), Text(
                extent={{-22,-102},{220,-136}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ConcentrationCompartment;

      model MassStorageCompartment
        //extends QHP.Library.Interfaces.BaseModel;

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                                   annotation (Placement(
              transformation(extent={{62,-32},{102,8}}),  iconTransformation(extent={{-10,-10},
                  {10,10}})));
        parameter SoluteMass_ initialSoluteMass;

        HumMod.Library.Interfaces.RealOutput_ SoluteMass 
          annotation (Placement(transformation(extent={{-14,-120},{26,-80}},rotation=-90,
              origin={0,2}),
              iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,-74})));

      initial equation
        SoluteMass = initialSoluteMass;
      equation
        q_out.conc = SoluteMass;
        der(SoluteMass) = q_out.q / Library.SecPerMin;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/massStorage.png"), Text(
                extent={{-94,134},{98,106}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end MassStorageCompartment;

      model SolventFlowPump_InitialPatch

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out
          "second side connector with value of q (solute mass flow) and conc (concentration)"
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));

        Interfaces.RealInput_ solventFlow
          "solvent flow (solution volume flow = solventFlow + solute volume flow)"
                                      annotation ( extent = [-10,50;10,70], rotation = -90);


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in
          "first side connector with value of q (solute mass flow) and conc (concentration)"
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

      equation
        q_in.q + q_out.q = 0;
        q_in.q = solventFlow*q_in.conc;

       // assert(solventFlow < 0, "BAD FLOW DIRECTION !!!");
       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
</table>
</html>",       info="<html>
<p><h4><font color=\"#008000\">Bidirectional mass flow by concentration</font></h4></p>
<p>Possible field values: </p>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0.1\"><tr>
<td></td>
<td><p align=\"center\">forward flow</p></td>
<td><p align=\"center\">backward flow</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>solventFlow</h4></p></td>
<td><p align=\"center\">&GT;=0</p></td>
<td><p align=\"center\">&LT;0</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>q_in.q</h4></p></td>
<td><p align=\"center\">=solventFlow*q_in.conc</p></td>
<td><p align=\"center\">=solventFlow*q_out.conc</p></td>
</tr>
<tr>
<td><p align=\"center\"><h4>q_out.q</h4></p></td>
<td><p align=\"center\">=-q_in.q</p></td>
<td><p align=\"center\">=-q_in.q</p></td>
</tr>
</table>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end SolventFlowPump_InitialPatch;
    end ConcentrationFlow;

    package Semipermeable "Osmotic Physical Domain"

       connector OsmoticFlow
        "H2O flow throught semipermeable membrane by osmotic pressure gradient"
        Real o(quantity="Osmolarity", unit="mOsm");
        flow Real q(quantity="Flow", unit="ml/min");
        annotation (Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
       end OsmoticFlow;

      connector PositiveOsmoticFlow "H2O inflow"
        extends OsmoticFlow;

      annotation (
          defaultComponentName="q_in",
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={127,127,0},
                lineThickness=1), Polygon(
                points={{0,100},{100,0},{0,-100},{-100,0},{0,100}},
                lineColor={0,0,0},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={127,127,0})}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={0,0,0})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>", revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));

      end PositiveOsmoticFlow;

      connector NegativeOsmoticFlow "H2O outflow"
        extends OsmoticFlow;

      annotation (
          defaultComponentName="q_out",
          Icon(coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=false),
              graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={127,127,0},
                lineThickness=1), Polygon(
                points={{-100,0},{0,100},{100,0},{0,-100},{-100,0}},
                smooth=Smooth.None,
                fillPattern=FillPattern.Solid,
                fillColor={225,235,129},
                pattern=LinePattern.None)}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={255,255,255})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>", revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));

      end NegativeOsmoticFlow;

      model FlowMeasure

        NegativeOsmoticFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{18,-10},{38,10}}), iconTransformation(
                extent={{78,-42},{98,-22}})));


        PositiveOsmoticFlow q_in  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent={{-100,
                  -44},{-80,-24}})));
        HumMod.Library.Interfaces.RealOutput_ actualFlow(final quantity="Flow",
            final unit="ml/min") 
                               annotation (Placement(transformation(extent={{-20,30},{20,70}}),
              iconTransformation(extent={{-20,-20},{20,20}},
                                                           rotation=90,
              origin={-2,82})));
      equation
        q_in.q + q_out.q = 0;
        q_out.o = q_in.o;

        actualFlow = q_in.q;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={Bitmap(extent={{-100,102},{100,-102}},
                  fileName="icons/flowMeassure.png")}),
                                        Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end FlowMeasure;

      partial model ResistorBase "semipermeable membrane"
       extends HumMod.Library.Interfaces.BaseResistorModel;
        PositiveOsmoticFlow q_in 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-10},{-90,10}})));
        NegativeOsmoticFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorBase;

      partial model Resistor
       extends ResistorBase;
      equation
        q_in.q + q_out.q = 0;
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end Resistor;

      model ResistorWithCondParam
       extends Resistor;
       parameter Real cond "H2O membrane permeability";
      equation
        q_in.q = cond * (q_out.o - q_in.o);
        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}),
                            graphics), Icon(graphics={Text(
                extent={{-70,-30},{70,30}},
                lineColor={0,0,0},
                textString="%cond"), Text(
                extent={{-134,-90},{154,-30}},
                textString="%name",
                lineColor={0,0,255})}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorWithCondParam;

      model InputPump

        NegativeOsmoticFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={
                  {50,-10},{70,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow "pure H2O inflow" 
                                                                      annotation ( extent = [-10,30;10,50], rotation = -90);


      equation
        q_out.q = - desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-30},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,20},{50,0},{-48,-21},{-48,20}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-92,-54},{80,-30}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end InputPump;

      model OutputPump

        PositiveOsmoticFlow q_in 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-8},{-90,12}}), iconTransformation(extent={{-70,-10},
                  {-50,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow 
                                       annotation ( extent = [-10,30;10,50], rotation = -90);


      equation
        q_in.q = desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-32},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,18},{50,-2},{-48,-26},{-48,18}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-78,-54},{72,-32}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end OutputPump;

      model ColloidOsmolarity "set osmolarity from protein mass flow"
        extends Interfaces.ConversionIcon;
        Interfaces.RealInput_ proteinMassFlow(unit="g/min") 
                                annotation (Placement(transformation(extent={{-20,80},
                  {20,120}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={48,116})));
        PressureFlow.PositivePressureFlow q_in "hydraulic pressure" 
                                  annotation (Placement(transformation(extent={{-110,-110},
                  {-90,-90}}),iconTransformation(extent={{-110,-110},{-90,-90}})));
        NegativeOsmoticFlow q_out(o(final unit="g/ml"))
          "colloid osmotic pressure" 
          annotation (Placement(transformation(extent={{90,-10},{110,10}}),
              iconTransformation(extent={{90,-10},{110,10}})));

      parameter Real C1 =   320.0;
      parameter Real C2 =   1160.0;

        Interfaces.RealOutput_ P annotation (Placement(transformation(extent={{42,86},
                  {82,126}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={-60,-120})));
      equation
        q_in.q + q_out.q = 0;
        q_out.o = abs(proteinMassFlow/q_in.q);
        P = q_in.pressure;

        annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}},
              initialScale=0.04), graphics={Text(
                extent={{22,92},{84,18}},
                lineColor={0,0,0},
                textString="O"), Text(
                extent={{-94,-12},{-10,-90}},
                lineColor={0,0,0},
                textString="P")}), Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              initialScale=0.04), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end ColloidOsmolarity;

      model ColloidHydraulicPressure
        "set pressure as sum of osmotic pressure(from osmoles) and hydrostatic/hydrodynamic pressure(from signal)"
        extends Interfaces.ConversionIcon;
        Interfaces.RealInput_ hydraulicPressure(unit="mmHg") 
                                annotation (Placement(transformation(extent={{-20,80},
                  {20,120}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={40,120})));
        PressureFlow.NegativePressureFlow q_out
          "pressure on semipermeable membrane wall = osmotic + hydrostatic" 
                                  annotation (Placement(transformation(extent={{90,-10},
                  {110,10}}), iconTransformation(extent={{90,-10},{110,10}})));
        PositiveOsmoticFlow q_in(o(unit="g")) "osmoles" 
          annotation (Placement(transformation(extent={{-110,-10},{-90,10}})));

      parameter Real C1 =   320.0;
      parameter Real C2 =   1160.0;

      equation
        q_in.q + q_out.q = 0;
        q_out.pressure = hydraulicPressure - ( (C1 * q_in.o)  + ( C2 * (q_in.o^2)));

        annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}},
              initialScale=0.04), graphics={Text(
                extent={{-94,-10},{-32,-84}},
                lineColor={0,0,0},
                textString="O"), Text(
                extent={{8,92},{92,14}},
                lineColor={0,0,0},
                textString="P")}), Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              initialScale=0.04), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end ColloidHydraulicPressure;

      model ColloidHydraulicPressure2
       extends Interfaces.ConversionIcon;
        Interfaces.RealInput_ hydraulicPressure(unit="mmHg") 
                                annotation (Placement(transformation(extent={{-20,80},
                  {20,120}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={40,120})));
        PressureFlow.NegativePressureFlow q_out
          "pressure on semipermeable membrane wall = osmotic + hydrostatic" 
                                  annotation (Placement(transformation(extent={{90,-10},
                  {110,10}}), iconTransformation(extent={{90,-10},{110,10}})));
        PositiveOsmoticFlow q_in(o(unit="g")) "osmoles" 
          annotation (Placement(transformation(extent={{-110,-10},{-90,10}})));

      parameter Real C1 =   320.0;
      parameter Real C2 =   1160.0;

      PressureFlow.NegativePressureFlow withoutCOP
          "only hydrostatic pressure without colloid osmotic pressure" 
                                  annotation (Placement(transformation(extent={{90,90},
                  {110,110}}),iconTransformation(extent={{90,90},{110,110}})));

      equation
        q_in.q + q_out.q + withoutCOP.q = 0;
        q_out.pressure = hydraulicPressure - ( (C1 * q_in.o)  + ( C2 * (q_in.o^2)));

        withoutCOP.pressure = hydraulicPressure;

        annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}},
              initialScale=0.04), graphics={Text(
                extent={{-94,-10},{-32,-84}},
                lineColor={0,0,0},
                textString="O"), Text(
                extent={{8,92},{92,14}},
                lineColor={0,0,0},
                textString="P")}), Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              initialScale=0.04), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end ColloidHydraulicPressure2;

      model ColloidHydraulicPressure0
       extends Interfaces.ConversionIcon;
        Interfaces.RealInput_ hydraulicPressure(unit="mmHg") 
                                annotation (Placement(transformation(extent={{-20,80},
                  {20,120}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={40,120})));
        PositiveOsmoticFlow q_in(o(unit="g")) "osmoles" 
          annotation (Placement(transformation(extent={{-110,-10},{-90,10}})));

      parameter Real C1 =   320.0;
      parameter Real C2 =   1160.0;

      PressureFlow.NegativePressureFlow withoutCOP
          "only hydrostatic pressure without colloid osmotic pressure" 
                                  annotation (Placement(transformation(extent={{90,90},
                  {110,110}}),iconTransformation(extent={{90,90},{110,110}})));

      equation
        q_in.q + withoutCOP.q = 0;
        withoutCOP.pressure = hydraulicPressure;

        annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}},
              initialScale=0.04), graphics={Text(
                extent={{-94,-10},{-32,-84}},
                lineColor={0,0,0},
                textString="O"), Text(
                extent={{8,92},{92,14}},
                lineColor={0,0,0},
                textString="P")}), Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              initialScale=0.04), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid)}));
      end ColloidHydraulicPressure0;

      model OsmoticPump
        extends OsmoticPumpBase;
        PositiveOsmoticFlow q_in 
          annotation (Placement(transformation(extent={{-90,-10},{-70,10}})));
        NegativeOsmoticFlow q_out annotation (Placement(transformation(extent={{70,-10},
                  {90,10}}), iconTransformation(extent={{70,-10},{90,10}})));
      equation
        q_in.q + q_out.q = 0;
        q_in.o = desiredOsmoles;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
                                                     graphics));
      end OsmoticPump;

      model OsmoticPumpBase "Defined osmoles to/from/in system by real signal"

        Interfaces.RealInput_ desiredOsmoles(quantity="Osmolarity", unit="mOsm")
          "desired pressure flow value"                                                                  annotation (Placement(transformation(
                extent={{-66,50},{-26,90}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,60})));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Rectangle(
                extent={{-80,60},{80,-60}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={255,170,170},
                fillColor={255,170,170},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-94},{150,-54}},
                textString="%name",
                lineColor={0,0,0},
                fillPattern=FillPattern.Solid,
                fillColor={255,170,170}),
              Text(
                extent={{-36,-72},{-152,98}},
                lineColor={255,170,170},
                fillColor={255,170,170},
                fillPattern=FillPattern.Solid,
                textString="")}), Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics));
      end OsmoticPumpBase;
    end Semipermeable;

    package HeatFlow "Temperature Physical Domain"

      connector HeatFlowConnector "Heat flow connector"
        Real T(final unit="K") "temperature";
        flow Real q(final unit="kCal/min") "heat flow";
        annotation (Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end HeatFlowConnector;

      connector PositiveHeatFlow "Heat inflow"
        extends HumMod.Library.HeatFlow.HeatFlowConnector;

        annotation (Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-18,10},{22,-10}},
                lineColor={255,128,0},
                lineThickness=1), Ellipse(
                extent={{-100,100},{100,-100}},
                lineColor={176,88,0},
                fillColor={255,128,0},
                fillPattern=FillPattern.Solid)}), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end PositiveHeatFlow;

      connector NegativeHeatFlow "Heat outflow"
        extends HumMod.Library.HeatFlow.HeatFlowConnector;

      annotation (
          defaultComponentName="q_out",
          Coordsys(
            extent=[-100, -100; 100, 100],
            grid=[1, 1],
            component=[20, 20],
            scale=0.2),
          Icon(coordinateSystem(extent={{-100,-100},{100,100}},
                preserveAspectRatio=true),                      Polygon(points=[-100,0;
                  0,100; 100,0; 0,-100; -100,0],            style(
                color=74,
                rgbcolor={200,200,0},
                fillColor=0,
                rgbfillColor={255,240,240})),
            graphics={Rectangle(
                extent={{-20,10},{20,-10}},
                lineColor={255,128,0},
                lineThickness=1), Ellipse(
                extent={{-100,100},{100,-100}},
                lineColor={176,88,0},
                fillColor={255,170,85},
                fillPattern=FillPattern.Solid)}),
          Diagram(Polygon(points=[-21,-3; 5,23; 31,-3; 5,-29; -21,-3],   style(
                color=74,
                rgbcolor={0,0,0},
                fillColor=0,
                rgbfillColor={255,255,255})), Text(
              extent=[-105,-38; 115,-83],
              string="%name",
              style(color=0, rgbcolor={0,0,0}))),
          Documentation(info="<html>
<p>
Connector with one flow signal of type Real.
</p>
</html>", revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end NegativeHeatFlow;

      partial model ResistorBase
       extends HumMod.Library.Interfaces.BaseResistorModel;
        PositiveHeatFlow q_in annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-10},{-90,10}})));
        NegativeHeatFlow q_out annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));
      Real ActualConductance;
      equation
        ActualConductance = if 
                              (q_in.T == q_out.T) then 0 else q_in.q/(q_in.T - q_out.T);
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorBase;

      partial model Resistor
       extends ResistorBase;
      equation
        q_in.q + q_out.q = 0;
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end Resistor;

      model ResistorWithCond
       extends Resistor;
        HumMod.Library.Interfaces.RealInput_ cond(final quantity="Conductance",
            final unit="kCal/(min.K)") 
                                 annotation (extent=[-5,30; 5,50], rotation=-90);
      equation
        q_in.q = cond * (q_in.T - q_out.T);
        annotation (Icon(graphics={Text(
                extent={{-70,-30},{70,30}},
                textString="%name",
                lineColor={0,0,255})}), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end ResistorWithCond;

      model ResistorWithCondParam
       extends Resistor;
       parameter Real cond;
      equation
        q_in.q = cond * (q_in.T - q_out.T);
        annotation (Icon(graphics={Text(
                extent={{-70,-8},{70,10}},
                lineColor={0,0,0},
                textString="%cond (kcal/min)/K")}),
                                        Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}),
                          graphics));
      end ResistorWithCondParam;

      model HeatFlux "flow circuit through mass with different temperature"

        NegativeHeatFlow q_out "surrounding mass" annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-10,-60},{10,-40}}), iconTransformation(extent={
                  {-10,-60},{10,-40}})));
        Interfaces.RealInput_ substanceFlow(final unit="kg/min")
          "flowing speed in circuit"  annotation ( extent = [-10,50;10,70], rotation = -90);


        PositiveHeatFlow q_in "flow circuit"     annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

      parameter Real specificHeat(unit="kCal/kg/K") "of flow circuit medium";

      equation
        q_in.q + q_out.q = 0;
        q_in.q = substanceFlow*(q_in.T-q_out.T)*specificHeat;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{20,-84},{320,-44}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end HeatFlux;

      model HeatVaporOutflow
        "heat outflow through vaporization and outflowing heatted steam"

        Interfaces.RealInput_ liquidOutflow(final unit="kg/min")
          "speed of vaporization"  annotation ( extent = [-10,50;10,70], rotation = -90);


        PositiveHeatFlow q_in "flow circuit"     annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));
      parameter Real TempToolsVapor(final unit="kCal/kg")
          "needed heat to vaporization";
      parameter Real specificHeat(final unit="kCal/kg/K") "of liquid";

      equation
        q_in.q = liquidOutflow*(q_in.T*specificHeat + TempToolsVapor);

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-80,25},{80,0},{-80,-25},{-80,25}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{20,-84},{320,-44}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end HeatVaporOutflow;

      model InputPump

        NegativeHeatFlow q_out annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={
                  {50,-10},{70,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow "speed of heat flow" 
                                                                      annotation ( extent = [-10,30;10,50], rotation = -90);


      equation
        q_out.q = - desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-30},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,20},{50,0},{-48,-21},{-48,20}},
                lineColor={0,0,127},
                fillColor={0,0,127},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-92,-54},{80,-30}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end InputPump;

      model HeatAccumulation
        "accumulating heat to substance mass with specific heat constant"
        extends HumMod.Library.Utilities.DynamicState;
        extends Library.Interfaces.BaseModel;
        PositiveHeatFlow q_in(T(start=initialTemperature))
          "heat inflow/outflow connector" 
          annotation (Placement(transformation(extent={{-20,-20},{20,20}})));

        parameter Real initialTemperature(final unit="K")=310.15
          "to calculate initial heat";                                                        //==37 degC
        parameter Real specificHeat(final unit="kCal/kg/K")=1
          "of the mass, where the heat are accumulated";
        parameter Real weightPar(final unit="kg") = 1.0;

        Real heatMass(start=weightPar*specificHeat*initialTemperature, final unit
            =                                                                     "kCal")
          "accumulated heat";
        Real T_degC(final unit="degC") "temperature in celsius degrees";
        Interfaces.RealInput_ weight(final unit="kg")
          "weight of mass, where the heat are accumulated"                            annotation (Placement(transformation(extent={{-122,60},
                  {-82,100}}), iconTransformation(extent={{-120,60},{-80,100}})));
        Interfaces.RealOutput_ T(unit="degC") annotation (Placement(transformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,-100})));
      //initial equation
      //  heatMass=weight*specificHeat*initialTemperature;
      equation
      //  der(heatMass)=q_in.q/Library.SecPerMin;
        q_in.T=heatMass/(weight*specificHeat);
        T_degC = q_in.T - 273.15;
        T=T_degC;
        stateValue = heatMass;

        changePerMin = q_in.q;
        annotation (Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"), Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics));
      end HeatAccumulation;

      model AmbientTemperature "constant temperature, undefinned heat flow"
        extends Library.Interfaces.BaseModel;
        PositiveHeatFlow q_in "heat inflow/outflow connector" 
          annotation (Placement(transformation(extent={{-20,-20},{20,20}})));

        parameter Real Temperature(final unit="K")=295.37
          "to calculate initial heat";
                                                                //==22 degC
        Real T_degC(final unit="degC") "temperature in celsius degrees";
      equation
        q_in.T=Temperature;
        T_degC = q_in.T - 273.15;
        annotation (Icon(graphics), Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end AmbientTemperature;

      model Pump

        NegativeHeatFlow q_out annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={
                  {50,-10},{70,10}})));
        HumMod.Library.Interfaces.RealInput_ desiredFlow(unit="ml/min")
          "speed of liquid flow"                                      annotation ( extent = [-10,30;10,50], rotation = -90);


        PositiveHeatFlow q_in 
          annotation (Placement(transformation(extent={{-70,-10},{-50,10}})));

        parameter Real specificHeat(unit="kCal/ml/K") "of flow circuit medium";
      equation
        q_in.q + q_out.q = 0;
        q_in.q = specificHeat * q_in.T * desiredFlow;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-60,-30},{60,30}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Polygon(
                points={{-48,20},{50,0},{-48,-21},{-48,20}},
                lineColor={255,0,0},
                fillColor={255,0,0},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-92,-54},{80,-30}},
                textString="%name",
                lineColor={0,0,255})}), Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"),          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end Pump;
    end HeatFlow;

    constant Real SecPerMin(unit="s/min") = 60
      "Conversion coeficient from minutes to seconds";

    package Utilities

      function readRealParameter "Read the value of a Real parameter from file"
        import Modelica.Utilities.*;
        extends Modelica.Icons.Function;

        //parameter String PARAMETERS_FILE = "parameters/default.txt";
        input String fileName "Name of file"       annotation(Dialog(
                               __Dymola_loadSelector(filter="Text files (*.txt)",
                               caption="Open file in which Real parameters are present")));

        input String name "Name of parameter";
        output Real result "Actual value of parameter on file";


      protected
        String line;
        Integer nextIndex;
        Integer iline=1;
        Boolean found = false;
        Boolean endOfFile=false;
      algorithm
       //#(line, endOfFile) := Streams.readLine(fileName, iline);
       /* ##########  
        while not found and not endOfFile loop
             if line == name then
                 // name found, get value of "name = value;"
                 (line, endOfFile) :=Streams.readLine(fileName, iline+1);
                 result := Strings.scanReal(line, 1);
                 found := true;
              else
                 // wrong name, skip line
                 iline := iline + 2;
                 // read next line
                 (line, endOfFile) :=Streams.readLine(fileName, iline);
              end if;
        end while;

        if not found then
           Streams.error("Parameter \"" + name + "\" not found in file \"" + fileName + "\"");
        end if;
			*/
        result := 1.0;

        annotation (Documentation(info="<html>
<h4>Syntax</h4>
<blockquote><pre>
result = <b>readRealParameter</b>(fileName, name);
</pre></blockquote>
<h4>Description</h4>
<p>
This function demonstrates how a function can be implemented
that reads the value of a parameter from file. The function
performs the following actions:
</p>
<ol>
<li> It opens file \"fileName\" and reads the lines of the file.</li>
<li> In every line, Modelica line comments (\"// ... end-of-line\")
     are skipped </li>
<li> If a line consists of \"name = expression\" and the \"name\"
     in this line is identical to the second argument \"name\"
     of the function call, the expression calculator Examples.expression
     is used to evaluate the expression after the \"=\" character.
     The expression can optionally be terminated with a \";\".</li>
<li> The result of the expression evaluation is returned as
     the value of the parameter \"name\". </li>
</ol>
<h4>Example</h4>
<p>
On file \"test.txt\" the following lines might be present:
</p>
<blockquote><pre>
// Motor data
J        = 2.3     // inertia
w_rel0   = 1.5*2;  // relative angular velocity
phi_rel0 = pi/3
</pre></blockquote>
<p>
The function returns the value \"3.0\" when called as:
</p>
<blockquote><pre>
readRealParameter(\"test.txt\", \"w_rel0\")
</pre></blockquote>
</html>"));
      end readRealParameter;

      type Init = enumeration(
          NoInit
            "No abstract initialization (start values are used as guess values with fixed=false or from start=value in instances)", 

          CalculateInitialSteadyState
            "Steady state initialization (derivatives of states are zero)",
          FromFile "Initialization from file");

      partial model DynamicState
        extends DynamicStateSteady;
      //  extends DynamicStateDynamic;
      //  extends DynamicStateDynamicNoInit;

        import Modelica.Utilities.*;

        parameter String stateName
          "state name must be unique for each instance";
        parameter String PARAMETERS_FILE = "setup/default.txt"; //"parameters/default.txt"; //"setup/old_saveInit.txt"; //
        parameter String PARAMETERS2_FILE = "setup/initialState.txt";

        Real stateValue "state must be connected in inherited class definition";
        Real changePerMin "dynamic change of state value per minute";
      initial equation
        if initType == Init.CalculateInitialSteadyState then
          der(stateValue)=0;
        elseif initType == Init.FromFile then
          stateValue=Library.Utilities.readRealParameter( PARAMETERS_FILE, stateName);
        end if;

      equation
        if SAVE_VALUES and initial() then // and terminal() then
           //Streams.print( stateName, PARAMETERS2_FILE);
           //Streams.print( String(stateValue), PARAMETERS2_FILE);
        end if;

        if STEADY then
          changePerMin = 0;
      //    der(stateValue) = 0;
        else
          der(stateValue) =  changePerMin / Library.SecPerMin;  //derivation time in Modelica is in seconds
        end if;

      end DynamicState;

      partial model DynamicStateSteady

        parameter Boolean STEADY = true;
        parameter Boolean SAVE_VALUES = true;
        parameter Init initType = Init.NoInit;

      end DynamicStateSteady;

          block ConstantFromFile
        "Generate constant signal of type Real with value from file"
            parameter Real k(fixed=false) "Constant output value";
            parameter String SETUP_FILE = "setup/default.txt";
            parameter String varName;

            HumMod.Library.Interfaces.RealOutput_ y
          "Connector of Real output signal" 
              annotation (Placement(transformation(extent={{100,-10},{120,10}},
                  rotation=0)));

          initial equation
            k=Library.Utilities.readRealParameter( SETUP_FILE, varName);
          equation
            y = k;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.1), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-10},{100,10}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%varName")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.1), graphics),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end ConstantFromFile;

          block VariableToFile "Save variable to file"
            import Modelica.Utilities.*;

            parameter String SAVE_FILE = "setup/saveInit.txt";
            parameter String varName;

            Interfaces.RealInput_ y "Connector of Real input signal" 
              annotation (Placement(transformation(extent={{-100,-10},{-80,10}},
                  rotation=0), iconTransformation(extent={{-120,-10},{-100,10}})));


          equation
            //if terminal() then
            if initial() then
               //Streams.print( varName, SAVE_FILE);
               //Streams.print( String(y), SAVE_FILE);
            end if;
            annotation (
              Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics={Rectangle(
                extent={{-100,20},{100,-20}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-100,-10},{100,10}},
                lineColor={0,0,0},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%varName")}),
              Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2},
              initialScale=0.04), graphics),
          Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
          end VariableToFile;
    end Utilities;
    annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2008-2009</td>
</tr>
</table>
</html>"));
  end Library;

  package Blood "Blood and Cardio-Vascular System"

    package VascularCompartments "Blood Cardio-Vascular Distribution"

      package heart "Heart Ventricle Components"

        model Diastole
          Real iconPoint[20,2](final unit="mm");
          Real iconActualPoint[2,2](final unit="mm");

          Library.PressureFlow.PositivePressureFlow inflow annotation (Placement(
                transformation(extent={{-122,-24},{-82,16}}), iconTransformation(extent={{-120,20},
                    {-80,60}})));

          HumMod.Library.Interfaces.RealInput_ externalPressure(final quantity=
                "Pressure", final unit="mmHg")
            "pericardium pressure around ventricle" 
                                         annotation ( Placement(transformation(extent={{94,-58},
                    {114,-38}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=180,
                origin={100,-60})));

        parameter Real stiffnes "parametrization of end diastolic volume curve";
        parameter Real n_Diastole
            "parametrization of end systolic volume curve";
        parameter Real Abasic_Diastole
            "parametrization of end diastolic volume curve";

          HumMod.Library.Interfaces.RealOutput EDV(final quantity="Volume",
              final unit="ml")   annotation (Placement(transformation(extent={{-74,-74},
                    {-34,-34}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={100,80})));

          HumMod.Library.Interfaces.RealOutput_ P(final quantity="Pressure",
              final unit="mmHg") annotation (Placement(transformation(extent={{-74,-74},
                    {-34,-34}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={60,-60})));
          HumMod.Library.Interfaces.RealOutput_ Stiffness 
                                 annotation (Placement(transformation(extent={{-74,-74},
                    {-34,-34}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-68,100})));
        equation
          inflow.q = 0;
          P = inflow.pressure;
          EDV = ((inflow.pressure-externalPressure)/(stiffnes*Abasic_Diastole))^(1/n_Diastole);
          Stiffness = stiffnes;

        // drawing icon
          for i in 1:size(iconPoint,1) loop
            iconPoint[i,1]=((i-1)/(size(iconPoint,1)))*180 - 90;
            (iconPoint[i,1]+90)*200/180 = ( ( ( (iconPoint[i,2]+40)*200/130)-externalPressure)/(stiffnes*Abasic_Diastole))^(1/n_Diastole);
          end for;
          iconActualPoint[1,1]=EDV*180/200-90 -2;
          iconActualPoint[1,2]=inflow.pressure*130/200-40 -2;
          iconActualPoint[2,1]=iconActualPoint[1,1]+4;
          iconActualPoint[2,2]=iconActualPoint[1,2]+4;
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,100},{100,-60}},
                  lineColor={255,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={255,255,255}),
                Text(
                  extent={{-100,-60},{100,-100}},
                  lineColor={0,0,255},
                  textString="%name"),
                Line(
                  points=DynamicSelect({{-100,0},{100,0}}, iconPoint),
                  color={0,0,255},
                  smooth=Smooth.None),
                Ellipse(
                  extent=DynamicSelect({{0,0},{0,0}}, iconActualPoint),
                  lineColor={0,0,0},
                  fillColor={255,0,0},
                  fillPattern=FillPattern.Solid)}),
                                              Diagram(coordinateSystem(
                  preserveAspectRatio=true,  extent={{-100,-100},{100,100}}), graphics));
        end Diastole;

        model Systole

          Real iconPoint[20,2](final unit="mm");
          Real iconActualPoint[2,2](final unit="mm");

          Library.PressureFlow.NegativePressureFlow outflow 
            annotation (Placement(transformation(extent={{80,20},{120,60}}),
                iconTransformation(extent={{80,20},{120,60}})));

          HumMod.Library.Interfaces.RealInput_ contractility
            "heart muscle contractility [xNormal]" 
                                         annotation ( extent = [70,90;90,110], rotation = -90);

        parameter Real n_Systole
            "parametrization of end diastolic volume curve";
        parameter Real Abasic_Systole
            "parametrization of end systolic volume curve";
        parameter Real additionalPressure_Systolic
            "parametrization of end systolic volume curve";

          HumMod.Library.Interfaces.RealOutput ESV(final quantity="Volume",
              final unit="ml")   annotation (Placement(transformation(extent={{-74,-74},
                    {-34,-34}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=180,
                origin={-100,80})));

          Library.Interfaces.RealInput_ externalPressure(      final quantity="Pressure", final unit
              =                                                                                     "mmHg")
            "pressure around ventricle"                                           annotation (
        Placement(transformation(extent={{-110,-70},{-90,-50}}), iconTransformation(
                  extent={{-110,-70},{-90,-50}})));
          HumMod.Library.Interfaces.RealOutput_ P(final quantity="Pressure",
              final unit="mmHg") annotation (Placement(transformation(extent={{-74,-74},
                    {-34,-34}}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-60,-60})));
        equation
          outflow.q = 0;
          P=outflow.pressure;
          ESV = ((outflow.pressure+additionalPressure_Systolic-externalPressure)/(contractility*Abasic_Systole))^(1/n_Systole);

        // drawing icon
          for i in 1:size(iconPoint,1) loop
            iconPoint[i,1]=((i-1)/(size(iconPoint,1)))*180 - 90;
            (iconPoint[i,1]+90)*200/180= ((((iconPoint[i,2]+40)*200/130)+additionalPressure_Systolic-externalPressure)/(contractility*Abasic_Systole))^(1/n_Systole);
          end for;
          iconActualPoint[1,1]=ESV*180/200-90 -2;
          iconActualPoint[1,2]=outflow.pressure*130/200-40 -2;
          iconActualPoint[2,1]=iconActualPoint[1,1]+4;
          iconActualPoint[2,2]=iconActualPoint[1,2]+4;
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,100},{100,-60}},
                  lineColor={255,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={255,255,255}),
                Text(
                  extent={{-100,-60},{100,-100}},
                  lineColor={0,0,255},
                  textString="%name"),
                Line(
                  points=DynamicSelect({{-100,0},{100,0}}, iconPoint),
                  color={0,0,255},
                  smooth=Smooth.None),
                Ellipse(
                  extent=DynamicSelect({{0,0},{0,0}}, iconActualPoint),
                  lineColor={0,0,0},
                  fillColor={255,0,0},
                  fillPattern=FillPattern.Solid)}),      Diagram(coordinateSystem(
                  preserveAspectRatio=true,  extent={{-100,-100},{100,100}}), graphics));
        end Systole;

        model Ventricle
         extends HumMod.Blood.VascularCompartments.Interfaces.IVentricle;


          HumMod.Library.Blocks.Constant basicContractility(k=
                contractilityBasic) 
            annotation (Placement(transformation(extent={{60,2},{68,10}})));

          HumMod.Nerves.BetaReceptorsActivityFactor betaReceptorsActivityFactor
            annotation (Placement(transformation(extent={{64,-20},{84,0}})));

          Modelica.Blocks.Math.Add Vol_SteadyState(k1=.5, k2=.5) 
                                                     annotation (Placement(
                transformation(
                extent={{10,-10},{-10,10}},
                rotation=270,
                origin={-34,18})));

          Diastole diastole(
            stiffnes=stiffnes,
            n_Diastole=n_Diastole,
            Abasic_Diastole=Abasic_Diastole) 
            annotation (Placement(transformation(extent={{-76,-44},{-56,-24}})));

          Systole systole(
            n_Systole=n_Systole,
            Abasic_Systole=Abasic_Systole,
            additionalPressure_Systolic=additionalPressure_Systolic) 
            annotation (Placement(transformation(extent={{56,-44},{76,-24}})));

          Modelica.Blocks.Math.Feedback StrokeVolume 
            annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-8,-6})));

          VentricleVolumeAndPumping2 ventricle(
            initialVolume=initialVol,
            K=K,
            BasicCompliance=BasicCompliance) 
                   annotation (Placement(transformation(extent={{-8,66},{12,86}})));

          Modelica.Blocks.Math.Product BloodFlow     annotation (Placement(
                transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={16,40})));

        equation
          connect(basicContractility.y, betaReceptorsActivityFactor.yBase) 
                                                                 annotation (Line(
              points={{68.4,6},{74,6},{74,-3}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(bloodFlowSignals.BetaPool_Effect, betaReceptorsActivityFactor.BetaPool_Effect) 
            annotation (Line(
              points={{22,2},{22,-4},{64.2,-4}},
              color={0,0,255},
              thickness=0.5,
              smooth=Smooth.None), Text(
              string="%first",
              index=-1,
              extent={{-6,3},{-6,3}}));
          connect(bloodFlowSignals.BetaBlocade_Effect, betaReceptorsActivityFactor.BetaBlockade_Effect) 
            annotation (Line(
              points={{22,2},{22,-10},{64.2,-10}},
              color={0,0,255},
              thickness=0.5,
              smooth=Smooth.None), Text(
              string="%first",
              index=-1,
              extent={{-6,3},{-6,3}}));
          connect(bloodFlowSignals.GangliaGeneral_NA, betaReceptorsActivityFactor.GangliaGeneral_NA) 
            annotation (Line(
              points={{22,2},{22,-16},{64.2,-16}},
              color={0,0,255},
              thickness=0.5,
              smooth=Smooth.None), Text(
              string="%first",
              index=-1,
              extent={{-6,3},{-6,3}}));
          connect(diastole.externalPressure, bloodFlowSignals.Pericardium_Pressure) 
            annotation (Line(
              points={{-56,-40},{22,-40},{22,2}},
              color={0,0,127},
              smooth=Smooth.None), Text(
              string="%second",
              index=1,
              extent={{6,3},{6,3}}));
          connect(bloodFlowSignals.Pericardium_Pressure, systole.externalPressure) 
            annotation (Line(
              points={{22,2},{22,-40},{56,-40}},
              color={0,0,255},
              thickness=0.5,
              smooth=Smooth.None), Text(
              string="%first",
              index=-1,
              extent={{-6,3},{-6,3}}));
          connect(betaReceptorsActivityFactor.y, systole.contractility) 
            annotation (Line(
              points={{74,-17},{74,-24}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(diastole.EDV,StrokeVolume. u1) annotation (Line(
              points={{-56,-26},{-40,-26},{-40,-6},{-16,-6}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Vol_SteadyState.u2, diastole.EDV) annotation (Line(
              points={{-40,6},{-40,-26},{-56,-26}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(systole.outflow, q_out) annotation (Line(
              points={{76,-30},{86,-30},{86,100},{-20,100}},
              color={0,0,0},
              smooth=Smooth.None));
          connect(q_in, diastole.inflow) annotation (Line(
              points={{20,80},{-82,80},{-82,-30},{-76,-30}},
              color={0,0,0},
              smooth=Smooth.None));
          connect(systole.ESV, Vol_SteadyState.u1) annotation (Line(
              points={{56,-26},{-28,-26},{-28,6}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(systole.ESV,StrokeVolume. u2) annotation (Line(
              points={{56,-26},{-8,-26},{-8,-14}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Vol_SteadyState.y, ventricle.VentricleSteadyStateVolume) 
            annotation (Line(
              points={{-34,29},{-34,58},{-6,58},{-6,66}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(q_in, ventricle.q_in)                 annotation (Line(
              points={{20,80},{-54,80},{-54,76},{-8,76}},
              color={0,0,0},
              smooth=Smooth.None));
          connect(ventricle.q_out, q_out)                 annotation (Line(
              points={{12,76},{56,76},{56,100},{-20,100}},
              color={0,0,0},
              smooth=Smooth.None));
          connect(StrokeVolume.y, BloodFlow.u1)     annotation (Line(
              points={{1,-6},{10,-6},{10,28}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(BloodFlow.y, ventricle.BloodFlow)                 annotation (Line(
              points={{16,51},{16,58},{10,58},{10,66}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(ventricle.Volume, Vol)                 annotation (Line(
              points={{12.4,71},{14.7,71},{14.7,93},{17,93}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(BloodFlow.u2, bloodFlowSignals.HeartVentricleRate) 
            annotation (Line(
              points={{22,28},{22,2}},
              color={0,0,127},
              smooth=Smooth.None), Text(
              string="%second",
              index=1,
              extent={{6,3},{6,3}}));
          connect(BloodFlow.y, CO) annotation (Line(
              points={{16,51},{16,59},{77,59}},
              color={0,0,127},
              smooth=Smooth.None));
         annotation (
            Documentation(info="<HTML>
<PRE>
Heart ventricle.
 
like
QHP 2008 / VascularCompartments/LeftVentricle/*
 
Created : 16-May-05
Last Modified : 05-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
 
The pressure-volume characteristics of both ventricles in diastole
and systole will be described using a power function.
 
  P = A * (V ^ n)
 
P = Pressure
V = Volume
A = Scale Factor
n = curvature
 
The function is invertable.
 
  V = (P/A) ^ (1/n)
 
Some typical values
 
                                 Right                 Left
                           ---------------------- ------------------------
Nominal Vol                     87.5                  87.5
Nominal TMP                      3.0                   6.0
Compliance                      29.2                  14.6
Pericardial P                   -3.0                  -3.0
 
DIASTOLE
  EDP                            1.0                   5.0
  EDV                          125.0                 125.0
  TMP                            4.0                   8.0
  n                              2.0                   2.0
  V^n                        15625.0               15625.0
  A                              0.00026               0.00051
 
SYSTOLE
  ESP                           22.0                 120.0
  ESV                           50.0                  50.0
  TMP                           25.0                 123.0
  n                              0.5                   0.5
  V^n                            7.07                  7.07
  A                              3.53                 17.39
</PRE>
</HTML>
",       revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"),  Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                    {100,100}}), graphics={Text(
                  extent={{-82,6},{76,-8}},
                  lineColor={0,0,255},
                  textString="(initial %initialVol ml)")}),
                                          Diagram(coordinateSystem(preserveAspectRatio=true,
                           extent={{-100,-100},{100,100}}), graphics={Text(
                  extent={{-32,62},{-8,60}},
                  lineColor={0,0,0},
                  textString="Volume_SteadyState"), Text(
                  extent={{8,62},{32,60}},
                  lineColor={0,0,0},
                  textString="Blood_Inflow")}),
            DymolaStoredErrors);
        end Ventricle;

          model VentricleVolumeAndPumping2
          "Multiple PressureFlow connector with pressures from multiple inputs"
           extends Library.Interfaces.BaseModel;
           extends HumMod.Library.Utilities.DynamicState;

            Library.Interfaces.RealInput_ BloodFlow(final quantity="Flow", final unit
              =  "ml/min") "heart cardiac output" 
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                  rotation=0), iconTransformation(extent={{-20,-20},{20,20}},
                rotation=90,
                origin={80,-100})));
            Library.PressureFlow.PositivePressureFlow q_in 
               annotation (Placement(transformation(extent={{100,-20},{140,20}},
                  rotation=0), iconTransformation(extent={{-110,-10},{-90,10}})));

            parameter Real initialVolume(final quantity="Volume", final unit="ml");

            Library.PressureFlow.NegativePressureFlow q_out 
                                   annotation (extent=[-10, -110; 10, -90], Placement(
                  transformation(extent={{90,-10},{110,10}}), iconTransformation(extent={{90,-10},
                      {110,10}})));

            Real delta(final quantity="Flow", final unit="ml/min");

            Library.Interfaces.RealInput_ VentricleSteadyStateVolume(final
              quantity=
                  "Volume", final unit="ml")
            "heart ventricle steady state volume" 
              annotation (Placement(transformation(extent={{-140,-20},{-100,20}},
                  rotation=0), iconTransformation(extent={{-20,-20},{20,20}},
                rotation=90,
                origin={-80,-100})));

            parameter Real K(final unit="1/min") = 1;
            parameter Real V0(final quantity="Volume", final unit="ml") = 0;
            parameter Real BasicCompliance(final quantity="Compliance", final unit
              =                                                                    "ml/mmHg");
            Library.Interfaces.RealOutput_ Volume(start=initialVolume, final
              quantity =                                                              "Volume", final unit
              =                                                                                            "ml") annotation (Placement(transformation(
                    extent={{84,-70},{124,-30}}), iconTransformation(extent={{84,-70},{
                      124,-30}})));

          //  Real Compliance(final quantity="Compliance", final unit="ml/mmHg");
          //  Real StressedVol(final quantity="Volume", final unit="ml");
          //  Real Pressure(final quantity="Pressure", final unit="mmHg");
          //initial equation
          //  Volume = initialVolume;
            //der(Volume) = 0;
          equation

            delta = (VentricleSteadyStateVolume - Volume)*K;
          /*  if STEADY then
    der(Volume) = 0;
  else
    der(Volume) = delta / Library.SecPerMin;
  end if;
*/
            q_in.q + q_out.q = delta;
            q_in.q = if (delta<0) then BloodFlow else BloodFlow+delta;

            //increase (delta>0) average ventricle volume means flow delta from q_in (veins)
            //decrease (delta<0) average ventricle volume means flow delta to q_to (atrys)

           // StressedVol  =  max( Volume - V0,  0.0);
           // Compliance  =  BasicCompliance / VentricleDiastole_Stiffness;
           // Pressure  =  ( StressedVol / Compliance)  + ExternalPressure;

            stateValue = Volume;
            changePerMin = delta;
            annotation (Documentation(info="<HTML>
<p>
Model has a vector of continuous Real input signals as pressures for
vector of pressure-flow connectors.
</p>
</HTML>
"),           Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                    -100},{100,100}}), graphics={Rectangle(
                  extent={{-40,-24},{42,22}},
                  lineColor={0,0,127},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid), Polygon(
                  points={{-30,12},{34,-2},{-30,-18},{-30,12}},
                  lineColor={0,0,127},
                  fillColor={0,0,127},
                  fillPattern=FillPattern.Solid)}),
              DymolaStoredErrors);
          end VentricleVolumeAndPumping2;
      end heart;

      model SystemicVeinsElacticBloodCompartment
      // extends QHP.Library.Interfaces.BaseModel;


        Library.PressureFlow.PositivePressureFlow referencePoint 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-110,-10},{-90,10}}), iconTransformation(extent={{-10,-10},
                  {10,10}},
              rotation=0,
              origin={0,0})));
        Library.Interfaces.RealInput_ ExternalPressure(     final quantity="Pressure", final unit
            =                                                                                     "mmHg")
          "external pressure around the compartment"                            annotation (extent = [-10,90;10,110], rotation = -90);
        Library.Interfaces.RealInput_ Compliance(     final quantity="Compliance", final unit
            =                                                                                 "ml/mmHg")
          "elasticity of the stressed walls"                             annotation (extent = [50,90;70,110], rotation = -90);

        Modelica.Blocks.Interfaces.RealOutput Pressure(  final quantity="Pressure", final unit
            =                                                                                  "mmHg")
          "blod pressure in compartment" 
          annotation (Placement(transformation(extent={{30,-110},{50,-90}}, rotation=-90)));
        Modelica.Blocks.Interfaces.RealOutput Vol(final quantity="Volume", final unit =      "ml")
          "blood volume in compartment" 
          annotation (Placement(transformation(extent={{-50,-110},{-30,-90}}, rotation=-90),
              iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-40,-100})));
        Library.Interfaces.BusConnector organBloodFlowSignals annotation (Placement(
              transformation(extent={{-88,72},{-68,92}}), iconTransformation(
                extent={{88,51},{108,71}})));
        HumMod.Blood.VascularCompartments.VascularElasticBloodCompartment
          systemicVeins(initialVol=initialSystemisVeinsVol,
          stateName="SystemicVeins.Vol") "systemic veins" 
          annotation ( Placement(transformation(extent={{-14,-10},{6,10}})));
        Library.Factors.SplineValue V0_A2_Effect( data={{0.0,1.05,0.0},{1.3,1.00,-0.10},
              {3.0,0.85,0.0}}) 
          annotation (Placement(transformation(extent={{-30,7},{-10,27}})));
        Library.Blocks.VolumeConstant const4(
                                k=1700) 
          annotation (Placement(transformation(extent={{-32,48},{-23,39}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors(
          NEURALK=.333,
          HUMORALK=.5,
          data={{0.0,1.2,0.0},{1.0,1.0,-0.30},{3.0,0.6,0.0}}) 
          annotation (Placement(transformation(extent={{-30,20},{-10,40}})));
                                                            //,
          /* ##SympsOnConductance(iFrom(start=1))*/
        Modelica.Blocks.Interfaces.RealOutput V0(final quantity="Volume", final unit
            =                                                                        "ml")
          "maximal zero pressure blood volume in compartment" 
          annotation (Placement(transformation(extent={{-4.5,-4.5},{4.5,4.5}},rotation=180,
              origin={-27.5,5.5}),
              iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={-68,100})));

        parameter Real initialSystemisVeinsVol(final quantity="Volume", final unit =   "ml")= 2329.57; // = 2200;// = 2980;

        Modelica.Blocks.Interfaces.RealOutput NormalizedVolume(final quantity=
              "NormalizedVolume", final unit="1")
          "maximal zero pressure blood volume in compartment" 
          annotation (Placement(transformation(extent={{-4.5,-4.5},{4.5,4.5}},rotation=0,
              origin={22.5,9.5}),
              iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-100,56})));
      equation
        connect(V0_A2_Effect.y,systemicVeins. V0) annotation (Line(
            points={{-20,15},{-20,12},{-10,12},{-10,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors.y,V0_A2_Effect. yBase) annotation (Line(
            points={{-20,23},{-20,19}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors.yBase,const4. y) annotation (Line(
            points={{-20,37},{-20,43.5},{-22.55,43.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeins.referencePoint, referencePoint) annotation (Line(
            points={{-4,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(systemicVeins.Vol, Vol) annotation (Line(
            points={{-8,-10},{-8,-67},{-40,-67},{-40,-100}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeins.Pressure, Pressure) annotation (Line(
            points={{0,-10},{0,-67},{40,-67},{40,-100}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeins.ExternalPressure, ExternalPressure) annotation (Line(
            points={{-4,10},{-4,67},{0,67},{0,100}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeins.Compliance, Compliance) annotation (Line(
            points={{2,10},{2,50},{60,50},{60,100}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, V0_A2_Effect.u) annotation (
            Line(
            points={{-78,82},{-78,17},{-29.8,17}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors.GangliaGeneral_NA) 
          annotation (Line(
            points={{-78,82},{-78,24},{-29.8,24}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors.AlphaPool_Effect) 
          annotation (Line(
            points={{-78,82},{-78,36},{-29.8,36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-78,82},{-78,30},{-29.8,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(V0_A2_Effect.y, V0) annotation (Line(
            points={{-20,15},{-20,5.5},{-27.5,5.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeins.NormalizedVolume, NormalizedVolume) annotation (
            Line(
            points={{6,6.6},{12,6.6},{12,9.5},{22.5,9.5}},
            color={0,0,127},
            smooth=Smooth.None));
       annotation (
          Documentation(info="<HTML>
<PRE>
  Elastic blood flow compartment. 
 
like 
QHP 2008 / VascularCompartments/SystemicArtys.DES
 
Created : 21-Oct-04
Last Modified : 13-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"), Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Bitmap(extent={{-100,100},{100,-100}},
                  fileName="icons/elastickyKompartment.png")}),
          Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics));
      end SystemicVeinsElacticBloodCompartment;

      model SequesteredBlood
       extends Interfaces.ISequesteredBlood;


      parameter Real initialVol(  final quantity="Volume", final unit="ml")
          "initial compartment blood volume";
       parameter Real[:,3] data;

        Library.PressureFlow.PressureControledCompartment
          pressureControledCompartment(initialVolume=initialVol) 
          annotation (Placement(transformation(extent={{-16,-10},{4,10}})));
        Library.Curves.Curve curve(
          x=data[:, 1],
          y=data[:, 2],
          slope=data[:, 3]) 
          annotation (Placement(transformation(extent={{-60,50},{-40,70}})));

        Modelica.Blocks.Math.Add add annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-6,40})));
      equation
        connect(pressureControledCompartment.Volume, curve.u) annotation (Line(
            points={{-6,-10},{-6,-18},{-70,-18},{-70,60},{-60,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pressureControledCompartment.Volume, Vol) annotation (Line(
            points={{-6,-10},{-6,-60},{-40,-60},{-40,-100}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(referencePoint, pressureControledCompartment.y[1]) annotation (Line(
            points={{-100,0},{-16,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(add.y, pressureControledCompartment.pressure[1]) annotation (Line(
            points={{-6,29},{-6,19.5},{-6,19.5},{-6,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(curve.val, add.u2) annotation (Line(
            points={{-39.8,60},{-12,60},{-12,52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ExternalPressure, add.u1) annotation (Line(
            points={{-90,100},{-90,52},{1.77636e-015,52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add.y, Pressure) annotation (Line(
            points={{-6,29},{-6,20},{40,20},{40,-100}},
            color={0,0,127},
            smooth=Smooth.None));
       annotation (
          Documentation(info="<HTML>
<PRE>
  Elastic blood flow compartment. 
 
like 
QHP 2008 / VascularCompartments/SystemicArtys.DES
 
Created : 21-Oct-04
Last Modified : 13-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"), Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics={Text(
                extent={{-100,-16},{100,-32}},
                lineColor={0,0,255},
                textString="(initial %initialVol ml)"), Bitmap(extent={{-100,
                    100},{100,-100}}, fileName="icons/sequerestedVolume.png")}),
          Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),
                          graphics));
      end SequesteredBlood;

      package Interfaces "Abstract Interfaces"

        partial model IVascularElasticBloodCompartment
         //extends QHP.Library.Interfaces.BaseModel;


          Library.PressureFlow.PositivePressureFlow referencePoint 
                                annotation (extent=[-10, -110; 10, -90], Placement(
                transformation(extent={{-110,-10},{-90,10}}), iconTransformation(extent=
                   {{-10,-10},{10,10}})));
          Library.Interfaces.RealInput_ V0(    final quantity="Volume", final unit
              =                                                                    "ml")
            "maximal nonstressed volume"                                                                              annotation (extent = [-70,90;-50,110], rotation = -90);
          Library.Interfaces.RealInput_ ExternalPressure(   final quantity="Pressure", final unit
              =                                                                                   "mmHg")
            "external pressure around the compartment"                            annotation (extent = [-10,90;10,110], rotation = -90);
          Library.Interfaces.RealInput_ Compliance(     final quantity="Compliance", final unit
              =                                                                                 "ml/mmHg")
            "elasticity of the stressed walls"                             annotation (extent = [50,90;70,110], rotation = -90);

          Modelica.Blocks.Interfaces.RealOutput Pressure(  final quantity="Pressure", final unit
              =                                                                                  "mmHg")
            "blod pressure in compartment" 
            annotation (Placement(transformation(extent={{30,-110},{50,-90}}, rotation=-90)));
         // Real Vol "blood volume in compartment";

          parameter Real initialVol = 0;
          Modelica.Blocks.Interfaces.RealOutput Vol(start=initialVol, final
              quantity =                                                             "Volume", final unit
              ="ml") "blood volume in compartment" 
            annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-40,-100})));

         // parameter Real initialVol = 0;
         annotation (
            preferredView = text,
            Documentation(info="<HTML>
<PRE>
  Elastic blood flow compartment. 
 
like 
QHP 2008 / VascularCompartments/SystemicArtys.DES
 
Created : 21-Oct-04
Last Modified : 13-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",       revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),   Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                    -100},{100,100}}), graphics),
            Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}),
                                graphics));
        end IVascularElasticBloodCompartment;

        model ISequesteredBlood
        // extends QHP.Library.Interfaces.BaseModel;


          Library.PressureFlow.PositivePressureFlow referencePoint 
                                annotation (extent=[-10, -110; 10, -90], Placement(
                transformation(extent={{-110,-10},{-90,10}}), iconTransformation(extent=
                   {{-10,-10},{10,10}})));
          Library.Interfaces.RealInput_ ExternalPressure(   final quantity="Pressure", final unit
              =                                                                                   "mmHg")
            "external pressure around the compartment"                            annotation (extent = [-100,90;-80,110], rotation = -90);

          Modelica.Blocks.Interfaces.RealOutput Pressure(  final quantity="Pressure", final unit
              =                                                                                  "mmHg")
            "blod pressure in compartment" 
            annotation (Placement(transformation(extent={{30,-110},{50,-90}}, rotation=-90)));
          Modelica.Blocks.Interfaces.RealOutput Vol(  final quantity="Volume", final unit
              =                                                                           "ml")
            "blood volume in compartment" 
            annotation (Placement(transformation(extent={{-50,-110},{-30,-90}}, rotation=-90)));

         annotation (
            Documentation(info="<HTML>
<PRE>
  Elastic blood flow compartment. 
 
like 
QHP 2008 / VascularCompartments/SystemicArtys.DES
 
Created : 21-Oct-04
Last Modified : 13-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",       revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),   Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}),
                  graphics),
            Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}),
                            graphics));
        end ISequesteredBlood;

        partial model IVentricle

          parameter Real initialVol( final quantity="Volume", final unit="ml") = 90;
          parameter Real initialESV( final quantity="Volume", final unit="ml") = 50;

        parameter Real stiffnes = 1
            "parametrization of end diastolic volume curve";
        parameter Real n_Diastole = 2
            "parametrization of end systolic volume curve";
        parameter Real n_Systole=.5
            "parametrization of end diastolic volume curve";
        parameter Real Abasic_Diastole = 0.00051
            "parametrization of end diastolic volume curve";
        parameter Real Abasic_Systole = 17.39
            "parametrization of end systolic volume curve";
        parameter Real additionalPressure_Systolic = 24
            "parametrization of end systolic volume curve";
        parameter Real contractilityBasic = 1
            "parametrization of end systolic volume curve";
        parameter Real K(  final quantity="TimeCoefficient", final unit="1/min") = 1
            "time adaptation coeficient of average ventricle blood volume";
        parameter Real BasicCompliance(final quantity="Compliance", final unit="ml/mmHg") = 1;

        parameter Real MaxContractionCompliance(final quantity="Compliance", final unit
              =                                                                         "ml/mmHg") = 1;
        parameter Real Cond1=1;
        parameter Real Cond2=1;

          Library.PressureFlow.PositivePressureFlow q_in 
                                annotation (extent=[-10, -110; 10, -90], Placement(
                transformation(extent={{10,70},{30,90}}),
                iconTransformation(extent={{10,70},{30,90}})));
          Library.PressureFlow.NegativePressureFlow q_out 
                                 annotation (extent=[-10, -110; 10, -90], Placement(
                transformation(extent={{-30,90},{-10,110}}),iconTransformation(
                  extent={{-30,90},{-10,110}})));


          HumMod.Library.Interfaces.BusConnector bloodFlowSignals 
                                                annotation (Placement(transformation(
                  extent={{12,-8},{32,12}}),   iconTransformation(extent={{-10,-64},{10,
                    -44}})));

          Library.Interfaces.RealOutput_ Vol(final quantity="Volume", final unit
              =                                                                  "ml") annotation (Placement(transformation(
                  extent={{14,90},{20,96}}),    iconTransformation(
                extent={{-20,-20},{20,20}},
                rotation=270,
                origin={40,-40})));
          Library.Interfaces.RealOutput_ CO(final quantity="Flow", final unit=
                "ml/min")                                                              annotation (Placement(transformation(
                  extent={{74,56},{80,62}}),    iconTransformation(
                extent={{-20,-20},{20,20}},
                rotation=180,
                origin={-68,-20})));

         annotation (
            Documentation(info="<HTML>
<PRE>
Heart ventricle.
 
like
QHP 2008 / VascularCompartments/LeftVentricle/*
 
Created : 16-May-05
Last Modified : 05-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
 
The pressure-volume characteristics of both ventricles in diastole
and systole will be described using a power function.
 
  P = A * (V ^ n)
 
P = Pressure
V = Volume
A = Scale Factor
n = curvature
 
The function is invertable.
 
  V = (P/A) ^ (1/n)
 
Some typical values
 
                                 Right                 Left
                           ---------------------- ------------------------
Nominal Vol                     87.5                  87.5
Nominal TMP                      3.0                   6.0
Compliance                      29.2                  14.6
Pericardial P                   -3.0                  -3.0
 
DIASTOLE
  EDP                            1.0                   5.0
  EDV                          125.0                 125.0
  TMP                            4.0                   8.0
  n                              2.0                   2.0
  V^n                        15625.0               15625.0
  A                              0.00026               0.00051
 
SYSTOLE
  ESP                           22.0                 120.0
  ESV                           50.0                  50.0
  TMP                           25.0                 123.0
  n                              0.5                   0.5
  V^n                            7.07                  7.07
  A                              3.53                 17.39
</PRE>
</HTML>
",       revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),  Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                    {100,100}}), graphics={Text(
                  extent={{-100,-126},{100,-92}},
                  textString="%name",
                  lineColor={0,0,0}), Bitmap(extent={{-100,100},{100,-100}},
                    fileName="icons/komoraSrdce.png")}),
                                          Diagram(coordinateSystem(preserveAspectRatio=true,
                           extent={{-100,-100},{100,100}}), graphics),
            DymolaStoredErrors);
        end IVentricle;
      end Interfaces;

      model VascularElasticBloodCompartment
        extends HumMod.Library.Utilities.DynamicState;
        extends
          HumMod.Blood.VascularCompartments.Interfaces.IVascularElasticBloodCompartment;


        Real StressedVolume(   final quantity="Volume", final unit="ml");

        Modelica.Blocks.Interfaces.RealOutput NormalizedVolume(final quantity="NormalizedVolume",
            final unit="1")
          "actual volume divided by standard compartement volume" 
          annotation (Placement(transformation(extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-40,-100}), iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={100,66})));
      //initial equation
      //  Vol = initialVol;
       // der(Vol) = 0;
      equation

        StressedVolume = max(Vol-V0,0);
        Pressure = (StressedVolume/Compliance) + ExternalPressure;
        referencePoint.pressure = Pressure;

        NormalizedVolume = if initialVol==0 then 0 else Vol/initialVol;

        stateValue = Vol;
        changePerMin = referencePoint.q;

       annotation (
          preferredView = text,
          Documentation(info="<HTML>
<PRE>
  Elastic blood flow compartment. 
 
like 
QHP 2008 / VascularCompartments/SystemicArtys.DES
 
Created : 21-Oct-04
Last Modified : 13-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"), Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}), graphics={Text(
                extent={{-100,-16},{100,-32}},
                lineColor={0,0,255},
                textString="(initial %initialVol ml)"), Bitmap(extent={{-100,
                    100},{100,-100}}, fileName="icons/elastickyKompartment.png")}));
      end VascularElasticBloodCompartment;

      model PulmonaryCirculation
        extends Library.PressureFlow.ResistorBase;

        VascularElasticBloodCompartment pulmCapys(initialVol=200.141,
          stateName="PulmCapys.Vol") "pulmonary capilaries" 
                                          annotation (Placement(transformation(extent={{-22,-10},
                  {-2,10}})));
        VascularElasticBloodCompartment pulmArty(initialVol=200.488,
          stateName="PulmArty.Vol") 
                   annotation (Placement(transformation(extent={{-86,-10},{-66,
                  10}})));
        VascularElasticBloodCompartment pulmVeins(initialVol=210.463,
          stateName="PulmVeins.Vol") "pulmonary veins" 
                                     annotation (Placement(transformation(extent={{38,-10},
                  {58,10}})));
        Library.PressureFlow.ResistorWithCondParam pulmArtyConductance(cond=1350) 
          annotation (Placement(transformation(extent={{-61,-11},{-39,11}})));
        Library.Blocks.VolumeConstant const13(k=110) 
          annotation (Placement(transformation(extent={{-94,16},{-86,24}})));
        Library.Blocks.ComplianceConstant const14(k=5.3) 
          annotation (Placement(transformation(extent={{-54,16},{-62,24}})));
        Library.Blocks.VolumeConstant const15(k=140) 
          annotation (Placement(transformation(extent={{-28,16},{-20,24}})));
        Library.Blocks.ComplianceConstant const16(k=4.6) 
          annotation (Placement(transformation(extent={{4,16},{-4,24}})));
        Library.PressureFlow.ResistorWithCondParam pulmCapysConductance(cond=
              1800) 
          annotation (Placement(transformation(extent={{2,-11},{24,11}})));
        Library.Blocks.VolumeConstant const17(k=150) 
          annotation (Placement(transformation(extent={{30,16},{38,24}})));
        Library.Blocks.ComplianceConstant const18(k=6) 
          annotation (Placement(transformation(extent={{66,14},{58,22}})));
        Library.PressureFlow.ResistorWithCondParam pulmVeinsConductance(cond=
              5400) 
          annotation (Placement(transformation(extent={{62,-11},{84,11}})));
        Library.Interfaces.BusConnector busConnector
          "signals of organ bood flow resistence"                                          annotation (Placement(
              transformation(extent={{18,66},{38,86}}),    iconTransformation(
                extent={{-108,42},{-92,58}})));
        LungBloodFlow lungBloodFlow 
          annotation (Placement(transformation(extent={{-26,74},{-6,94}})));
        Library.PressureFlow.FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{-42,-10},{-22,10}})));
        Modelica.Blocks.Math.Sum sum1(nin=3) 
          annotation (Placement(transformation(extent={{-28,28},{-20,36}})));
        Modelica.Blocks.Math.Sum sum2(nin=2) 
          annotation (Placement(transformation(extent={{82,-38},{74,-30}})));
      equation
        connect(pulmArty.referencePoint,pulmArtyConductance. q_in) annotation (Line(
            points={{-76,0},{-61,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(pulmArty.ExternalPressure,busConnector. Thorax_AvePressure) annotation (Line(
            points={{-76,10},{-76,14},{28,14},{28,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmCapys.ExternalPressure,busConnector. Thorax_AvePressure) annotation (Line(
            points={{-12,10},{-12,14},{28,14},{28,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmVeins.ExternalPressure,busConnector. Thorax_AvePressure) annotation (Line(
            points={{48,10},{48,14},{28,14},{28,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmArty.V0,const13. y) annotation (Line(
            points={{-82,10},{-82,20},{-85.6,20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmArty.Compliance,const14. y) annotation (Line(
            points={{-70,10},{-70,20},{-62.4,20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const15.y,pulmCapys. V0) annotation (Line(
            points={{-19.6,20},{-18,20},{-18,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const16.y,pulmCapys. Compliance) annotation (Line(
            points={{-4.4,20},{-6,20},{-6,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmCapys.referencePoint,pulmCapysConductance. q_in) annotation (Line(
            points={{-12,0},{2,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(pulmCapysConductance.q_out,pulmVeins. referencePoint) annotation (Line(
            points={{24,0},{48,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(pulmVeins.V0,const17. y) annotation (Line(
            points={{42,10},{42,20},{38.4,20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const18.y,pulmVeins. Compliance) annotation (Line(
            points={{57.6,18},{54,18},{54,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmVeins.referencePoint,pulmVeinsConductance. q_in) annotation (Line(
            points={{48,0},{62,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(lungBloodFlow.AlveolarVentilated, busConnector.AlveolarVentilated_BloodFlow) 
          annotation (Line(
            points={{-6,90},{28,90},{28,76}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));

        connect(pulmArtyConductance.q_out, flowMeasure.q_in) annotation (Line(
            points={{-39,0},{-38,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, pulmCapys.referencePoint) annotation (Line(
            points={{-26,0},{-12,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, lungBloodFlow.CardiacOutput) 
          annotation (Line(
            points={{-32,5.4},{-32,90},{-26,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const13.y, sum1.u[1]) annotation (Line(
            points={{-85.6,20},{-82,20},{-82,32},{-28,32},{-28,31.4667},{-28.8,
                31.4667}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(const15.y, sum1.u[2]) annotation (Line(
            points={{-19.6,20},{-18,20},{-18,26},{-80,26},{-80,32},{-28.8,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const17.y, sum1.u[3]) annotation (Line(
            points={{38.4,20},{42,20},{42,26},{-78,26},{-78,32.5333},{-28.8,
                32.5333}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmCapys.Vol, sum2.u[1]) annotation (Line(
            points={{-16,-10},{-16,-42},{88,-42},{88,-34.4},{82.8,-34.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmVeins.Vol, sum2.u[2]) annotation (Line(
            points={{44,-10},{44,-20},{88,-20},{88,-32},{82.8,-33.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pulmCapys.Pressure, busConnector.PulmCapys_Pressure) annotation (Line(
            points={{-8,-10},{-8,-20},{28,-20},{28,76}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-6},{6,-6}}));
        connect(sum1.y, busConnector.PulmonaryCirculation_V0) annotation (Line(
            points={{-19.6,32},{28,32},{28,76}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(pulmArty.Vol, busConnector.PulmonaryCirculation_DeoxygenatedBloodVolume) 
          annotation (Line(
            points={{-80,-10},{-80,-28},{28,-28},{28,76}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-16},{6,-16}}));
        connect(busConnector.PulmonaryCirculation_OxygenatedBloodVolume, sum2.y) 
          annotation (Line(
            points={{28,76},{28,-34},{73.6,-34}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(q_in, pulmArty.referencePoint) annotation (Line(
            points={{-100,0},{-76,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(pulmVeinsConductance.q_out, q_out) annotation (Line(
            points={{84,0},{100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Bitmap(extent={{-100,104},{100,-100}}, fileName=
                    "icons/perfusion2.png"),
              Text(
                extent={{-108,-48},{106,-70}},
                lineColor={0,0,255},
                textString="%name"),
              Bitmap(extent={{-34,46},{34,-48}}, fileName="icons/plice.png")}),
        Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>", info="<html>
<p>Blood volume (and pressures) distribution between </p>
<p><ul>
<li>pulmonary arteries, </li>
<li>pulmonary veins and </li>
<li>pulmonary capilaries.</li>
</ul></p>
<p>Blood flow through ventilated alveoli.</p>
</html>"));
      end PulmonaryCirculation;

      model SystemicCirculation
        extends Library.PressureFlow.ResistorBase2;

        /*
  Library.PressureFlow.NegativePressureFlow fromSystemicVeins annotation (
      Placement(transformation(extent={{-110,-10},{-90,10}}),iconTransformation(
          extent={{-112,44},{-92,64}})));
  Library.PressureFlow.PositivePressureFlow systemicArteries 
    annotation (Placement(transformation(extent={{88,-10},{108,10}}),
        iconTransformation(extent={{90,46},{110,66}})));
*/


        VascularElasticBloodCompartment systemicArtys(initialVol=1000.36,
          stateName="SystemicArtys.Vol") 
          annotation (Placement(transformation(extent={{65,-10},{86,11}})));
        Library.Blocks.VolumeConstant V0_artys(k=850) 
          annotation (Placement(transformation(extent={{98,18},{88,28}})));
        Library.Blocks.ComplianceConstant const7(k=88.6) 
          annotation (Placement(transformation(extent={{-27,10},{-35,18}})));
        OrganFlow.PeripheralFlow peripheral 
          annotation (Placement(transformation(extent={{-2,-14},{26,14}})));
        Library.PressureFlow.InputPump volumeCorrections 
          annotation (Placement(transformation(extent={{-28,24},{-42,38}})));
        OrganFlow.SystemicVeins veins(BaseConductance=856)
          "scaled to coronary vessels reorganisation" 
          annotation (Placement(transformation(extent={{-72,-8},{-56,8}})));
        SystemicVeinsElacticBloodCompartment
          systemicVeinsElacticBloodCompartment(systemicVeins(STEADY=false)) 
          annotation (Placement(transformation(extent={{-54,-10},{-34,10}})));
        Library.Blocks.PressureConstant const4(k=0) 
          annotation (Placement(transformation(extent={{86,14},{79,21}})));
        Library.Blocks.PressureConstant const6(k=0) 
          annotation (Placement(transformation(extent={{-36,16},{-43,23}})));
        Library.Interfaces.BusConnector busConnector
          "signals of organ bood flow resistence"                                          annotation (Placement(
              transformation(extent={{36,50},{56,70}}),    iconTransformation(
                extent={{-110,32},{-94,48}})));
        Library.Blocks.ComplianceConstant const5(k=1.55) 
          annotation (Placement(transformation(extent={{96,8},{89,15}})));
        OrganFlow.CollapsingVeins collapsedVeins 
                                                annotation (Placement(transformation(
              extent={{-10,10.5},{10,-10.5}},
              rotation=180,
              origin={-84.5,0})));
        Library.PressureFlow.GravityHydrostaticDifference
          gravityHydrostaticDifference annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={55,35})));
        Library.PressureFlow.PressureMeasure pressureMeasure annotation (
            Placement(transformation(
              extent={{8.5,-8.5},{-8.5,8.5}},
              rotation=0,
              origin={78.5,45.5})));
        Library.PressureFlow.ResistorWithCondParam legsArtys(cond=40) 
          annotation (Placement(transformation(extent={{80,-76},{64,-60}})));
        Library.Blocks.PressureConstant const8(k=0) 
          annotation (Placement(transformation(extent={{93,-58},{86,-51}})));
        Library.PressureFlow.GravityHydrostaticDifference hydrostaticDifference
          annotation (Placement(transformation(extent={{-10,-10},{10,10}},
              rotation=0,
              origin={54,-62})));
        SequesteredBlood sequesteredBlood(data={{0,0,10^(-10)},{50,97,1.0},{200,150,0.5}},
            initialVol=50.0044,
          pressureControledCompartment(stateName="BVSeqArtys.Vol")) 
          annotation (Placement(transformation(extent={{80,-78},{100,-58}})));
        SequesteredBlood sequesteredBlood1(                data={{0,-100,2.00},{150,11,
              0.11},{600,50,0.15}}, initialVol=120.691,
          pressureControledCompartment(stateName="BVSeqVeins.Vol")) 
          annotation (Placement(transformation(extent={{-62,-72},{-44,-54}})));
        Library.PressureFlow.ResistorWithCondParam legsVeins(cond=100) 
          annotation (Placement(transformation(extent={{-42,-70},{-28,-56}})));
        Library.PressureFlow.GravityHydrostaticDifferenceWithPumpEffect
          hydrostaticDifference1 
          annotation (Placement(transformation(extent={{-5,-68},{-25,-48}})));
        Library.Blocks.PressureConstant const12(k=0) 
          annotation (Placement(transformation(extent={{-52,-50},{-59,-43}})));
        Modelica.Blocks.Math.Gain G(k=9.81) annotation (Placement(transformation(
              extent={{4,-4},{-4,4}},
              rotation=90,
              origin={34,26})));
      equation
        connect(systemicArtys.Compliance, const5.y) annotation (Line(
            points={{81.8,11},{81.8,11.5},{88.65,11.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicArtys.V0, V0_artys.y) 
                                            annotation (Line(
            points={{69.2,11},{69.2,23},{87.5,23}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector, peripheral.organBloodFlowSignals)          annotation (
            Line(
            points={{46,60},{46,2.66},{21.66,2.66}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(veins.organBloodFlowSignals,busConnector)           annotation (
           Line(
            points={{-58.48,1.52},{-60,1.52},{-60,60},{46,60}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(const7.y,systemicVeinsElacticBloodCompartment. Compliance) 
          annotation (Line(
            points={{-35.4,14},{-38.2,14},{-38.2,10},{-38,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(veins.q_out,systemicVeinsElacticBloodCompartment. referencePoint) 
          annotation (Line(
            points={{-56,0},{-44,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(busConnector,systemicVeinsElacticBloodCompartment. organBloodFlowSignals) 
          annotation (Line(
            points={{46,60},{-26,60},{-26,6.1},{-34.2,6.1}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(systemicVeinsElacticBloodCompartment.referencePoint,volumeCorrections. q_out) 
          annotation (Line(
            points={{-44,0},{-46,0},{-46,31.7},{-42,31.7}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(systemicArtys.ExternalPressure,const4. y) annotation (Line(
            points={{75.5,11},{75.5,17.5},{78.65,17.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeinsElacticBloodCompartment.ExternalPressure,const6. y) 
          annotation (Line(
            points={{-44,10},{-44,19.5},{-43.35,19.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.BloodVolume_change,volumeCorrections. desiredFlow) 
          annotation (Line(
            points={{46,60},{-35,60},{-35,35.2}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(systemicArtys.Compliance,const5. y) annotation (Line(
            points={{81.8,11},{81.8,11.5},{88.65,11.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(collapsedVeins.ExternalPressure, busConnector.Thorax_AvePressure) 
          annotation (Line(
            points={{-90.5,2.1},{-90,2.1},{-90,60},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(veins.q_in, collapsedVeins.q_in) annotation (Line(
            points={{-72,0},{-74.5,0},{-74.5,-1.22465e-015}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(pressureMeasure.actualPressure,busConnector. CarotidSinus_Pressure) 
          annotation (Line(
            points={{74.08,43.8},{46,43.8},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gravityHydrostaticDifference.height, busConnector.CarotidSinusHeight) 
          annotation (Line(
            points={{48.9,34.5},{46,34},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gravityHydrostaticDifference.q_down, systemicArtys.referencePoint) 
          annotation (Line(
            points={{62,29.4},{62,0.5},{75.5,0.5}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(legsArtys.q_in, sequesteredBlood.referencePoint) annotation (Line(
            points={{80,-68},{90,-68}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(sequesteredBlood.ExternalPressure, const8.y) annotation (Line(
            points={{81,-58},{81,-54.5},{85.65,-54.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(legsArtys.q_out, hydrostaticDifference.q_down) annotation (Line(
            points={{64,-68},{63.5,-68},{63.5,-67.6},{61,-67.6}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(hydrostaticDifference.q_up, systemicArtys.referencePoint) annotation (
           Line(
            points={{61.2,-57.2},{61.2,0},{75.5,0},{75.5,0.5}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(sequesteredBlood1.ExternalPressure, const12.y) annotation (Line(
            points={{-61.1,-54},{-61.1,-46.5},{-59.35,-46.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sequesteredBlood1.referencePoint, legsVeins.q_in) annotation (
            Line(
            points={{-53,-63},{-42,-63}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(legsVeins.q_out, hydrostaticDifference1.q_down) annotation (
            Line(
            points={{-28,-63},{-28,-63.4},{-22,-63.4}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(hydrostaticDifference1.q_up,
          systemicVeinsElacticBloodCompartment.referencePoint) annotation (Line(
            points={{-22.2,-53.2},{-22.2,0},{-44,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(hydrostaticDifference1.height, busConnector.LowerTorsoVeinHeight) 
          annotation (Line(
            points={{-8.9,-58.3},{46,-58.3},{46,60}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(hydrostaticDifference1.pumpEffect, busConnector.Exercise_MusclePump_Effect) 
          annotation (Line(
            points={{-13.5,-48.5},{46,-48.5},{46,60}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(systemicArtys.Pressure, busConnector.SystemicArtys_Pressure) 
          annotation (Line(
            points={{79.7,-10},{80,-10},{80,-28},{46,-28},{46,60}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{3,-3},{3,-3}}));

        connect(gravityHydrostaticDifference.q_up, pressureMeasure.q_in) 
          annotation (Line(
            points={{62.2,39.8},{80,39.8},{80,42.1},{80.2,42.1}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(hydrostaticDifference.height, busConnector.LowerTorsoArtyHeight) 
          annotation (Line(
            points={{47.9,-62.5},{46,-62.5},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pressureMeasure.actualPressure, busConnector.CarotidSinusArteryPressure) 
          annotation (Line(
            points={{74.08,43.8},{46,43.8},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeinsElacticBloodCompartment.Pressure, busConnector.SystemicVeins_Pressure) 
          annotation (Line(
            points={{-40,-10},{-40,-28},{46,-28},{46,60}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(peripheral.q_in, systemicVeinsElacticBloodCompartment.referencePoint) 
          annotation (Line(
            points={{-2,1.77636e-015},{-44,1.77636e-015},{-44,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(peripheral.q_out, systemicArtys.referencePoint) annotation (
            Line(
            points={{26,1.77636e-015},{26,0.5},{75.5,0.5}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(systemicVeinsElacticBloodCompartment.V0, busConnector.SystemicVeins_V0) 
          annotation (Line(
            points={{-50.8,10},{-50.8,60},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(V0_artys.y, busConnector.SystemicArtys_V0) annotation (Line(
            points={{87.5,23},{70,23},{70,22},{46,22},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeinsElacticBloodCompartment.Vol, busConnector.SystemicVeins_DeoxygenatedBloodVolume) 
          annotation (Line(
            points={{-48,-10},{-48,-28},{46,-28},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sequesteredBlood1.Vol, busConnector.LegVeins_DeoxygenatedBloodVolume) 
          annotation (Line(
            points={{-56.6,-72},{-58,-72},{-58,-80},{46,-80},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicArtys.Vol, busConnector.SystemicArtys_OxygenatedBloodVolume) 
          annotation (Line(
            points={{71.3,-10},{72,-10},{72,-28},{46,-28},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sequesteredBlood.Vol, busConnector.LegArtys_OxygenatedBloodVolume) 
          annotation (Line(
            points={{86,-78},{86,-80},{46,-80},{46,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_out, q_out) annotation (Line(
            points={{-100,0},{-100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(q_in, systemicArtys.referencePoint) annotation (Line(
            points={{100,0},{99.75,0},{99.75,0.5},{75.5,0.5}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(collapsedVeins.q_out, q_out) annotation (Line(
            points={{-94.5,1.22465e-015},{-96.25,1.22465e-015},{-96.25,0},{-100,
                0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector.Gravity_Gz, G.u) annotation (Line(
            points={{46,60},{46,38},{34,38},{34,30.8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(G.y, gravityHydrostaticDifference.G) annotation (Line(
            points={{34,21.6},{34,16},{56.5,16},{56.5,26.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(G.y, hydrostaticDifference1.G) annotation (Line(
            points={{34,21.6},{34,-80},{-16.5,-80},{-16.5,-66.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(G.y, hydrostaticDifference.G) annotation (Line(
            points={{34,21.6},{34,-80},{55.5,-80},{55.5,-70.5}},
            color={0,0,127},
            smooth=Smooth.None));
       annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Text(
                extent={{42,46},{74,46}},
                lineColor={0,0,255},
                textString="Carotid Artery Pressure")}),
                                       Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-100,-10},{100,-34}},
                lineColor={0,0,255},
                textString="%name"),
              Bitmap(extent={{-100,78},{102,-86}}, fileName=
                    "icons/systemicCirculation.jpg"),
              Text(
                extent={{-104,-54},{110,-76}},
                lineColor={0,0,255},
                textString="%name")}),
        Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>", info="<html>
<p>Blood volume (and pressure) distributions between </p>
<p><ul>
<li>systemic arteries, </li>
<li>systemic veins and</li>
<li>portal (splanchnic) vein.</li>
</ul></p>
<p>Blood flow throught: </p>
<p><ul>
<li>GIT, hepatic artery, liver</li>
<li>Kidney</li>
<li>CNS</li>
<li>Skeletal muscle</li>
<li>Respiratory muscle</li>
<li>Skin</li>
<li>Bones</li>
<li>Adipose tissue</li>
<li>Other tissue</li>
</ul></p>
</html>"));
      end SystemicCirculation;

      model Heart

        VascularElasticBloodCompartment RightAtrium(initialVol=51.6454,
          stateName="RightAtrium.Vol") "right atrium" 
          annotation (Placement(transformation(extent={{-75,-1},{-55,19}})));
        Library.Blocks.ComplianceConstant const(k=12.5) 
          annotation (Placement(transformation(extent={{-48,18},{-55,25}})));
        Library.Blocks.VolumeConstant const1(k=0) 
          annotation (Placement(transformation(extent={{-78,21},{-72,27}})));
        heart.Ventricle rightVentricle(
          stiffnes=1.0,
          n_Diastole=2.0,
          n_Systole=0.5,
          Abasic_Diastole=0.00026,
          Abasic_Systole=3.53,
          contractilityBasic=1,
          K=1,
          additionalPressure_Systolic=9,
          BasicCompliance=29.2,
          Cond1=60000000,
          Cond2=60000000,
          MaxContractionCompliance=2,
          initialVol=87.5,
          ventricle(stateName="RightVentricle.Vol")) 
          annotation (Placement(transformation(extent={{-6,-46},{-50,-2}})));
        VascularElasticBloodCompartment LeftAtrium(initialVol=50.5035,
          stateName="LeftAtrium.Vol") "left atrium" 
                               annotation (Placement(transformation(extent={{46,-8},
                  {66,12}})));
        Library.Blocks.VolumeConstant const2(k=0) 
          annotation (Placement(transformation(extent={{59,19},{53,25}})));
        Library.Blocks.ComplianceConstant const3(k=6.25) 
          annotation (Placement(transformation(extent={{75,11},{68,18}})));
        heart.Ventricle leftVentricle(
          stiffnes=1,
          n_Diastole=2,
          n_Systole=0.5,
          Abasic_Diastole=0.00051,
          Abasic_Systole=17.39,
          additionalPressure_Systolic=24,
          contractilityBasic=1,
          K=1,
          BasicCompliance=14.6,
          MaxContractionCompliance=0.4,
          Cond1=60000000,
          Cond2=60000000,
          initialVol=87.5,
          ventricle(stateName="LeftVentricle.Vol")) 
               annotation (Placement(transformation(extent={{10,-46},{54,-2}})));
        Library.Interfaces.BusConnector busConnector
          "signals of organ bood flow resistence"                                          annotation (Placement(
              transformation(extent={{-4,32},{16,52}}),    iconTransformation(
                extent={{-108,-38},{-92,-22}})));
        Library.PressureFlow.PositivePressureFlow rightAtrium
          "blood inflow to right atrium" 
                                       annotation (
            Placement(transformation(extent={{-102,-24},{-82,-4}}),
              iconTransformation(extent={{-66,30},{-46,50}})));
        Library.PressureFlow.NegativePressureFlow fromRightVentricle
          "blood outflow to pulmonary circulation" 
          annotation (Placement(transformation(extent={{-20,-12},{0,8}}),
              iconTransformation(extent={{-48,-38},{-28,-18}})));
        Library.PressureFlow.NegativePressureFlow fromLeftVentricle
          "blood outflow to aorta" 
                                 annotation (
           Placement(transformation(extent={{20,30},{40,50}}),
              iconTransformation(extent={{14,-38},{34,-18}})));
        Library.PressureFlow.PositivePressureFlow leftAtrium
          "blood inflow to left atrium" 
                                      annotation (
            Placement(transformation(extent={{78,-24},{98,-4}}),
              iconTransformation(extent={{34,32},{54,52}})));
        Modelica.Blocks.Math.Sum sum1(nin=2) 
          annotation (Placement(transformation(extent={{-32,26},{-24,34}})));
        Modelica.Blocks.Math.Sum sum3(nin=2) 
          annotation (Placement(transformation(extent={{-56,-94},{-48,-86}})));
        Modelica.Blocks.Math.Sum sum2(nin=2) 
          annotation (Placement(transformation(extent={{-4,-4},{4,4}},
              rotation=180,
              origin={70,-90})));
        OrganFlow.LeftHeart leftCororaryCirculation(BasicLargeVeselsConductance=
             50*0.93677109469950528615290800207779, BasicSmallVeselsConductance=
             2.2*0.93677109469950528615290800207779)
          "scaled to normal pressure gradient 94 mmHg" 
          annotation (Placement(transformation(extent={{-34,72},{-8,98}})));
        OrganFlow.RightHeart rightCororaryCirculation(
            BasicLargeVeselsConductance=10*0.93677109469950528615290800207779,
            BasicSmallVeselsConductance=0.4*0.93677109469950528615290800207779)
          "scaled to normal pressure gradient 94 mmHg" 
          annotation (Placement(transformation(extent={{-34,48},{-8,74}})));
        Modelica.Blocks.Math.Feedback rightAtrium_TMP 
          annotation (Placement(transformation(extent={{-70,34},{-54,50}})));
        Modelica.Blocks.Math.Feedback leftAtrium_TMP 
          annotation (Placement(transformation(extent={{64,34},{48,50}})));
      equation
        connect(RightAtrium.V0, const1.y) annotation (Line(
            points={{-71,19},{-71,24},{-71.7,24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RightAtrium.Compliance, const.y) annotation (Line(
            points={{-59,19},{-59,21.5},{-55.35,21.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RightAtrium.referencePoint, rightVentricle.q_in) annotation (Line(
            points={{-65,9},{-66,8},{-32,8},{-32,-6.4},{-32.4,-6.4}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(busConnector.Pericardium_Pressure, RightAtrium.ExternalPressure) 
          annotation (Line(
            points={{6,42},{6,14},{-65,14},{-65,19}},
            color={0,0,255},
            thickness=.1,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{110,10},{110,10}}));
        connect(busConnector,rightVentricle. bloodFlowSignals)          annotation (
            Line(
            points={{6,42},{6,-37},{-28,-37},{-28,-35.88}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(LeftAtrium.V0,const2. y) annotation (Line(
            points={{50,12},{50,22},{52.7,22}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeftAtrium.Compliance,const3. y) annotation (Line(
            points={{62,12},{62,14.5},{67.65,14.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeftAtrium.referencePoint,leftVentricle. q_in) annotation (Line(
            points={{56,2},{44,2},{44,-6.4},{36.4,-6.4}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(busConnector,leftVentricle. bloodFlowSignals)          annotation (
            Line(
            points={{6,42},{6,-37},{32,-37},{32,-35.88}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.Pericardium_Pressure,LeftAtrium. ExternalPressure) 
          annotation (Line(
            points={{6,42},{6,14},{56,14},{56,12}},
            color={0,0,255},
            thickness=0.1,
            smooth=Smooth.None));
        connect(rightAtrium, RightAtrium.referencePoint) annotation (Line(
            points={{-92,-14},{-92,9},{-65,9}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(rightVentricle.q_out, fromRightVentricle) annotation (Line(
            points={{-23.6,-2},{-10,-2}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(leftVentricle.q_out, fromLeftVentricle) annotation (Line(
            points={{27.6,-2},{26,-2},{26,40},{30,40}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(LeftAtrium.referencePoint, leftAtrium) annotation (Line(
            points={{56,2},{88,2},{88,-14}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector.CardiacOutput, leftVentricle.CO) annotation (Line(
            points={{6,42},{6,-28},{17.04,-28},{17.04,-28.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-10},{-6,-10}}));
        connect(RightAtrium.Pressure, busConnector.RightAtrium_Pressure) annotation (
            Line(
            points={{-61,-1},{-61,-68},{6,-68},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-5},{6,-5}}));
        connect(const1.y, sum1.u[1]) annotation (Line(
            points={{-71.7,24},{-70,24},{-70,30},{-32.8,30},{-32.8,29.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const2.y, sum1.u[2]) annotation (Line(
            points={{52.7,22},{-38,22},{-38,30.4},{-32.8,30.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RightAtrium.Vol, sum3.u[1]) annotation (Line(
            points={{-69,-1},{-69,-90.4},{-56.8,-90.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(rightVentricle.Vol, sum3.u[2]) annotation (Line(
            points={{-36.8,-32.8},{-36,-32.8},{-36,-34},{-70,-34},{-70,-89.6},{
                -56.8,-89.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeftAtrium.Vol, sum2.u[1]) annotation (Line(
            points={{52,-8},{52,-42},{78,-42},{78,-89.6},{74.8,-89.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(leftVentricle.Vol, sum2.u[2]) annotation (Line(
            points={{40.8,-32.8},{40.8,-32},{78,-32},{78,-90.4},{74.8,-90.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeftAtrium.Pressure, busConnector.LeftAtrium_Pressure) annotation (
            Line(
            points={{60,-8},{60,-68},{6,-68},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(sum3.y, busConnector.Heart_DeoxygenatedBloodVolume) annotation (
           Line(
            points={{-47.6,-90},{6,-90},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(busConnector.Heart_OxygenatedBloodVolume, sum2.y) annotation (
            Line(
            points={{6,42},{6,-90},{65.6,-90}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(sum1.y, busConnector.Heart_V0) annotation (Line(
            points={{-23.6,30},{6,30},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(rightCororaryCirculation.organBloodFlowSignals, busConnector) 
                                                                annotation (
            Line(
            points={{-12.03,63.47},{6,63.47},{6,42}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(leftCororaryCirculation.organBloodFlowSignals, busConnector) 
                                                               annotation (Line(
            points={{-12.03,87.47},{6,87.47},{6,42}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(leftCororaryCirculation.BloodFlow, busConnector.leftHeart_BloodFlow) 
          annotation (Line(
            points={{-11.9,82.4},{6,82.4},{6,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(rightCororaryCirculation.BloodFlow, busConnector.rightHeart_BloodFlow) 
          annotation (Line(
            points={{-11.9,58.4},{6,58.4},{6,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(leftCororaryCirculation.BloodFlow, busConnector.LeftHeart_BloodFlow) 
          annotation (Line(
            points={{-11.9,82.4},{6,82.4},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(rightCororaryCirculation.BloodFlow, busConnector.RightHeart_BloodFlow) 
          annotation (Line(
            points={{-11.9,58.4},{6,58.4},{6,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(leftVentricle.q_out, rightCororaryCirculation.q_out) 
          annotation (Line(
            points={{27.6,-2},{26,-2},{26,60},{-8,60},{-8,61}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(leftVentricle.q_out, leftCororaryCirculation.q_out) annotation (
           Line(
            points={{27.6,-2},{26,-2},{26,84},{-8,84},{-8,85}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(rightCororaryCirculation.q_in, RightAtrium.referencePoint) 
          annotation (Line(
            points={{-34,61},{-82,61},{-82,9},{-65,9}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(leftCororaryCirculation.q_in, RightAtrium.referencePoint) 
          annotation (Line(
            points={{-34,85},{-82,85},{-82,9},{-65,9}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(RightAtrium.Pressure, rightAtrium_TMP.u1) annotation (Line(
            points={{-61,-1},{-61,-12},{-78,-12},{-78,42},{-68.4,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(rightAtrium_TMP.u2, busConnector.Pericardium_Pressure) annotation (
            Line(
            points={{-62,35.6},{-62,14},{6,14},{6,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeftAtrium.Pressure, leftAtrium_TMP.u1)  annotation (Line(
            points={{60,-8},{60,-12},{74,-12},{74,42},{62.4,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Pericardium_Pressure, leftAtrium_TMP.u2)  annotation (
            Line(
            points={{6,42},{6,14},{56,14},{56,35.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(rightAtrium_TMP.y, busConnector.rightAtrium_TMP) annotation (
            Line(
            points={{-54.8,42},{-24,42},{-24,42},{6,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(leftAtrium_TMP.y, busConnector.leftAtrium_TMP) annotation (Line(
            points={{48.8,42},{6,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/heart.png"), Text(
                extent={{-164,-60},{156,-88}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(info="<html>
<p>Calculates hearth pumping (left and right atria and ventricles, as wel as coronary circulation). By pressure-flow connectors communicate with pulmonary and systemic circulation.</p>
<p>Blood volume (and pressure) distributions between </p>
<p><ul>
<li>right/left atrium and</li>
<li>right/left ventricle.</li>
</ul></p>
<p>Generate average blood flow throught ventricles.</p>
<p>On the contrary of Tom Coleman models: coronary circulation outflows to right atrium, average blood volume in the hearth are calculated more simple (with the same results in steady state).</p>
</html>", revisions="<html>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0\"><tr>
<td><p>Author:</p></td>
<td><p>Marek Matejak, Jiri Kofranek</p></td>
</tr>
<tr>
<td><p>Design:</p></td>
<td><p>Zuzana Rubaninska</p></td>
</tr>
<tr>
<td><p>Copyright:</p></td>
<td><p>In public domains</p></td>
</tr>
<tr>
<td><p>By:</p></td>
<td><p>Charles University, Prague</p></td>
</tr>
<tr>
<td><p>Date of:</p></td>
<td><p>january 2009, August 2010</p></td>
</tr>
<tr>
<td><p>References:</p></td>
<td><p>Tom Coleman: QHP 2008 beta 3,, HumMod 2010, University of Mississippi Medical Center</p></td>
</tr>
</table>
</html>"));
      end Heart;

      model LungBloodFlow

        Library.Interfaces.RealInput_ CardiacOutput 
          annotation (Placement(transformation(extent={{-120,40},{-80,80}})));

        parameter Real BasicRLShunt(final unit="ml/min") =   220.0;
        parameter Real RightHemithorax_Pressure(final unit="mmHg") =   -4;
        parameter Real LeftHemithorax_Pressure(final unit="mmHg") =   -4;

        Real PressureGradientRightLeft(final unit="mmHg");
        Real Thorax_RightLungFlowFract(final unit="1");
        Real Thorax_LeftLungFlowFract(final unit="1");
        Real RightHemithorax_LungInflation(final unit="1");
        Real LeftHemithorax_LungInflation(final unit="1");
        Real Total(final unit="ml/min");
        Real RightLeftShunt(final unit="ml/min");
        Real Alveolar(final unit="ml/min");
        Real RightLungTotal(final unit="ml/min");
        Real RightLungVentilated(final unit="ml/min");
        Real RightLungShunt(final unit="ml/min");
        Real LeftLungTotal(final unit="ml/min");
        Real LeftLungVentilated(final unit="ml/min");
        Real LeftLungShunt(final unit="ml/min");
        Real AlveolarShunt(final unit="ml/min");
        Real TotalShunt(final unit="ml/min");

        parameter Real[:,3] PressureOnInflation =   {{ -4.0,  1.0,  0}, {  4.0,  0.0,  0}};
        parameter Real[:,3] PressureGradientOnFlowDist =   {{ -25,  0.0,  0}, {   0,  0.5,  0.03}, {  25,  1.0,  0}};

        Library.Interfaces.RealOutput_ AlveolarVentilated(final unit="ml/min") annotation (Placement(transformation(extent={{80,40},{120,80}}),
              iconTransformation(extent={{80,40},{120,80}})));
        Library.Curves.Curve Thorax_PressureGradientOnFlowDist(
          x=PressureGradientOnFlowDist[:, 1],
          y=PressureGradientOnFlowDist[:, 2],
          slope=PressureGradientOnFlowDist[:, 3]) 
          annotation (Placement(transformation(extent={{-40,52},{-20,72}})));
        Library.Curves.Curve Thorax_PressureOnInflationR(
          x=PressureOnInflation[:, 1],
          y=PressureOnInflation[:, 2],
          slope=PressureOnInflation[:, 3]) 
          annotation (Placement(transformation(extent={{52,46},{72,66}})));
        Library.Curves.Curve Thorax_PressureOnInflationL(
          x=PressureOnInflation[:, 1],
          y=PressureOnInflation[:, 2],
          slope=PressureOnInflation[:, 3]) 
          annotation (Placement(transformation(extent={{-2,-24},{18,-4}})));
      equation
             PressureGradientRightLeft  =  RightHemithorax_Pressure - LeftHemithorax_Pressure;

             Thorax_PressureGradientOnFlowDist.u = PressureGradientRightLeft;
             Thorax_LeftLungFlowFract  =  Thorax_PressureGradientOnFlowDist.val;
             Thorax_RightLungFlowFract  =  1.0 - Thorax_LeftLungFlowFract;

             Thorax_PressureOnInflationR.u = RightHemithorax_Pressure;
             RightHemithorax_LungInflation = Thorax_PressureOnInflationR.val;
             Thorax_PressureOnInflationL.u = LeftHemithorax_Pressure;
             LeftHemithorax_LungInflation = Thorax_PressureOnInflationL.val;

             Total  =  CardiacOutput;
             RightLeftShunt  =  min(BasicRLShunt,Total);
             Alveolar  =  Total - RightLeftShunt;
             RightLungTotal  =  Alveolar * Thorax_RightLungFlowFract;
             RightLungVentilated  =  RightLungTotal * RightHemithorax_LungInflation;
             RightLungShunt  =  RightLungTotal - RightLungVentilated;
             LeftLungTotal  =  Alveolar * Thorax_LeftLungFlowFract;
             LeftLungVentilated  =  LeftLungTotal * LeftHemithorax_LungInflation;
             LeftLungShunt  =  LeftLungTotal - LeftLungVentilated;
             AlveolarVentilated  =  RightLungVentilated + LeftLungVentilated;
             AlveolarShunt  =  RightLungShunt + LeftLungShunt;
             TotalShunt  =  RightLeftShunt + AlveolarShunt;

        annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                  -100},{100,100}}), graphics), Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>", info="<html>
<p>Calculation of blood flow fraction through ventilated alveoli.</p>
</html>"),Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,-100},
                  {100,100}}), graphics={Bitmap(extent={{-100,100},{100,-100}},
                  fileName="icons/plicniZkrat.png"), Text(
                extent={{-102,-92},{112,-114}},
                lineColor={0,0,255},
                textString="%name")}));
      end LungBloodFlow;
    annotation (Documentation(revisions="<html>
</html>"));
    end VascularCompartments;

    package OrganFlow "Tissue vessels hydraulic conductance (=1/resistance)"

      package ConductanceFactors
        "Multiplication factors on conductance (=1/resistance)"

        model HeartInfraction
         extends HumMod.Library.Interfaces.BaseFactorIcon;

         parameter Real DamagedArea_percent = 0;

          Library.Blocks.Constant Constant0(k=DamagedArea_percent) 
            annotation (Placement(transformation(extent={{-94,-12},{-74,8}})));
          Library.Blocks.Constant Constant1(k=1) 
            annotation (Placement(transformation(extent={{-60,6},{-40,26}})));
          Library.Blocks.Constant Constant2(k=100) annotation (Placement(
                transformation(extent={{-94,-38},{-74,-18}})));
          Modelica.Blocks.Math.Division division 
            annotation (Placement(transformation(extent={{-60,-24},{-40,-4}})));
          Modelica.Blocks.Math.Feedback feedback 
            annotation (Placement(transformation(extent={{-32,6},{-12,26}})));
          Modelica.Blocks.Math.Product product annotation (Placement(
                transformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={0,-30})));
        equation

          connect(product.y, y) annotation (Line(
              points={{-2.02067e-015,-41},{-2.02067e-015,-52.5},{0,-52.5},{0,
                  -60}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product.u1, yBase) annotation (Line(
              points={{6,-18},{6,-18},{6,80}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(feedback.y, product.u2) annotation (Line(
              points={{-13,16},{-6,16},{-6,-18}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant0.y, division.u1) annotation (Line(
              points={{-73,-2},{-68,-2},{-68,-8},{-62,-8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant2.y, division.u2) annotation (Line(
              points={{-73,-28},{-68,-28},{-68,-20},{-62,-20}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant1.y, feedback.u1) annotation (Line(
              points={{-39,16},{-30,16}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(division.y, feedback.u2) annotation (Line(
              points={{-39,-14},{-22,-14},{-22,8}},
              color={0,0,127},
              smooth=Smooth.None));
        end HeartInfraction;

        model MetabolicVasolidation2
          extends HumMod.Library.Interfaces.BaseFactorIcon5;
          HumMod.Library.Interfaces.RealInput_ O2Need 
                       annotation (Placement(transformation(extent={{-110,60},{-82,88}}),
                iconTransformation(extent={{-108,-10},{-88,10}})));

         parameter Real[:,3] data = {{   50,  1.0,  0}, { 1000,  3.5,  0.003}, { 3000,  5.5,  0}};
         parameter Real OnTau = 0.2;
         parameter Real OffTau = 1;
         parameter Real initialEffectValue = 1;

          Library.Curves.Curve SteadyState(
            x=data[:, 1],
            y=data[:, 2],
            slope=data[:, 3]) 
            annotation (Placement(transformation(extent={{-72,64},{-52,84}})));
          Modelica.Blocks.Continuous.Integrator effect(     y_start=initialEffectValue, k=1/
                Library.SecPerMin) 
            annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-20,-14})));
          Modelica.Blocks.Math.Feedback feedback annotation (Placement(
                transformation(
                extent={{-8,-12},{12,-32}},
                rotation=270,
                origin={8,64})));
          Modelica.Blocks.Math.Product product annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={0,-46})));
          Modelica.Blocks.Math.Product product1 annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-20,16})));
          Modelica.Blocks.Logical.Switch switch1 
            annotation (Placement(transformation(extent={{-54,24},{-34,44}})));
          Modelica.Blocks.Logical.GreaterThreshold greaterThreshold 
            annotation (Placement(transformation(extent={{-72,30},{-64,38}})));
          Library.Blocks.Inv OnK  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-86,8})));
          Library.Blocks.Inv OffK  annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,8})));
          Library.Blocks.Constant Constant0(k=OnTau) 
                                                 annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-86,-22})));
          Library.Blocks.Constant Constant1(k=OffTau) 
                                                 annotation (Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=90,
                origin={-60,-22})));

        equation
          connect(SteadyState.val, feedback.u1) 
                                          annotation (Line(
              points={{-51.8,74},{-14,74},{-14,70}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(effect.y, feedback.u2)     annotation (Line(
              points={{-20,-25},{-20,-28},{-2,-28},{-2,62},{-6,62}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(SteadyState.u, O2Need)  annotation (Line(
              points={{-72,74},{-96,74}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product.u2, effect.y)     annotation (Line(
              points={{-6,-34},{-6,-28},{-20,-28},{-20,-25}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product.u1, yBase) annotation (Line(
              points={{6,-34},{6,80},{6,80}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product.y, y) annotation (Line(
              points={{-2.02067e-015,-57},{-2.02067e-015,-61.5},{0,-61.5},{0,
                  -60}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product1.y, effect.u)     annotation (Line(
              points={{-20,5},{-20,3.25},{-20,3.25},{-20,1.5},{-20,-2},{-20,-2}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(product1.u1, feedback.y) annotation (Line(
              points={{-14,28},{-14,34.25},{-14,34.25},{-14,40.5},{-14,53},{-14,
                  53}},
              color={0,0,127},
              smooth=Smooth.None));

          connect(switch1.y, product1.u2) annotation (Line(
              points={{-33,34},{-26,34},{-26,28}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(greaterThreshold.u, feedback.y) annotation (Line(
              points={{-72.8,34},{-76,34},{-76,50},{-14,50},{-14,53}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(greaterThreshold.y, switch1.u2) annotation (Line(
              points={{-63.6,34},{-56,34}},
              color={255,0,255},
              smooth=Smooth.None));
          connect(OnK.y, switch1.u1)  annotation (Line(
              points={{-86,19},{-86,42},{-56,42}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(OffK.y, switch1.u3)  annotation (Line(
              points={{-60,19},{-60,26},{-56,26}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant1.y, OffK.u)  annotation (Line(
              points={{-60,-11},{-60,-4}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant0.y, OnK.u)  annotation (Line(
              points={{-86,-11},{-86,-4}},
              color={0,0,127},
              smooth=Smooth.None));
          annotation (Icon(graphics), Diagram(graphics));
        end MetabolicVasolidation2;
      end ConductanceFactors;

      model CollapsingVeins
        extends Library.PressureFlow.Resistor;
        Library.Interfaces.RealInput_ ExternalPressure annotation (Placement(
              transformation(extent={{40,0},{80,40}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={60,20})));

        parameter Real PR1LL( final quantity="Pressure", final unit="mmHg") = 0
          "start-collapsing sucking pressure, when external pressure is zero";
      equation
         q_in.pressure = if (q_out.pressure > (PR1LL+ExternalPressure)) then q_out.pressure else (PR1LL+ExternalPressure);
        annotation (Icon(graphics={
              Rectangle(
                extent={{-100,38},{100,-40}},
                lineColor={255,255,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-64,10},{56,-16}},
                lineColor={0,0,255},
                textString="%name"),
              Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/kolaps.png")}));
      end CollapsingVeins;

      partial model Base
        extends Library.PressureFlow.ResistorBase;
         HumMod.Library.Interfaces.BusConnector organBloodFlowSignals
          "signals of organ bood flow resistence" 
          annotation (Placement(transformation(extent={{-100,80},{-80,100}}),
              iconTransformation(extent={{60,10},{78,28}})));

        Library.Interfaces.RealOutput_ BloodFlow annotation (Placement(transformation(
                extent={{-92,8},{-82,18}}),  iconTransformation(extent={{60,-30},{80,-10}})));
        Library.PressureFlow.FlowMeasure flowMeasure annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-88,0})));
      equation

        connect(q_in, flowMeasure.q_in) annotation (Line(
            points={{-100,0},{-94,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, BloodFlow) annotation (Line(
            points={{-88,5.4},{-88,14},{-87,14},{-87,13}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Documentation(info="<HTML>
<PRE>
Blood flow variable resistor abstract model.
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
</table>
</html>"),      Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
              graphics={
              Polygon(
                points={{14,9},{-14,1},{14,-9},{14,-1},{14,9}},
                lineColor={135,135,135},
                smooth=Smooth.None,
                fillColor={175,175,175},
                fillPattern=FillPattern.Solid,
                origin={34,-61},
                rotation=180),
              Rectangle(
                extent={{-34,-60},{20,-62}},
                lineColor={135,135,135},
                fillPattern=FillPattern.Solid,
                fillColor={175,175,175}),
              Bitmap(extent={{-178,110},{158,-104}}, fileName=
                    "icons/perfusion.png")}));
      end Base;

      partial model BaseBadDirection
        extends Library.PressureFlow.ResistorBase;
         HumMod.Library.Interfaces.BusConnector organBloodFlowSignals
          "signals of organ bood flow resistence" 
          annotation (Placement(transformation(extent={{-100,80},{-80,100}}),
              iconTransformation(extent={{60,10},{78,28}})));

        Library.Interfaces.RealOutput_ BloodFlow annotation (Placement(transformation(
                extent={{84,6},{96,18}}),    iconTransformation(extent={{60,-30},{80,-10}})));
        Library.PressureFlow.FlowMeasure flowMeasure annotation (Placement(
              transformation(
              extent={{10,-10},{-10,10}},
              rotation=0,
              origin={88,0})));
      equation

        connect(q_out, flowMeasure.q_in) annotation (Line(
            points={{100,0},{94,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, BloodFlow) annotation (Line(
            points={{88,5.4},{88,12},{90,12}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Documentation(info="<HTML>
<PRE>
Blood flow variable resistor abstract model.
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
</table>
</html>"),      Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(graphics={
              Text(
                extent={{-70,-20},{70,20}},
                textString="%name",
                lineColor={0,0,255}),
              Polygon(
                points={{-18,-52},{-46,-60},{-18,-70},{-18,-62},{-18,-52}},
                lineColor={135,135,135},
                smooth=Smooth.None,
                fillColor={175,175,175},
                fillPattern=FillPattern.Solid),
              Rectangle(
                extent={{-18,-60},{36,-62}},
                lineColor={135,135,135},
                fillPattern=FillPattern.Solid,
                fillColor={175,175,175}),
              Bitmap(extent={{-184,112},{152,-102}}, fileName=
                    "icons/perfusion2.png")}));
      end BaseBadDirection;

      model SystemicVeins
        //extends QHP.Blood.OrganFlow.BaseBadDirection;
        extends Library.PressureFlow.ResistorBase;
        Library.PressureFlow.ResistorWithCond systemicVeinsConductance 
          annotation (Placement(transformation(extent={{-20,-10},{0,10}})));
        Library.Factors.SimpleMultiply ViscosityEffect 
          annotation (Placement(transformation(extent={{-20,0},{0,20}})));
        Library.Factors.SimpleMultiply ExerciseEffect 
          annotation (Placement(transformation(extent={{-20,7},{0,27}})));
        Library.Factors.SimpleMultiply CollapseEffect 
          annotation (Placement(transformation(extent={{-20,14},{0,34}})));
        Library.Blocks.CondConstant const12(k=BaseConductance) 
          annotation (Placement(transformation(extent={{-27,45},{-15,57}})));
        parameter Real BaseConductance( final quantity="Conductance", final unit
            =                                                                    "ml/(min.mmHg)") = 692;

        Library.PressureFlow.FlowMeasure flowMeasure annotation (Placement(
              transformation(
              extent={{10,-10},{-10,10}},
              rotation=0,
              origin={80,0})));
        Library.Interfaces.RealOutput_ BloodFlow annotation (Placement(transformation(
                extent={{76,6},{88,18}}),    iconTransformation(extent={{60,-30},{80,-10}})));
         Library.Interfaces.BusConnector organBloodFlowSignals
          "signals of organ bood flow resistence" 
          annotation (Placement(transformation(extent={{-96,77},{-76,97}}),
              iconTransformation(extent={{60,10},{78,28}})));
      equation
        connect(systemicVeinsConductance.cond,ViscosityEffect. y) annotation (Line(
            points={{-10,4},{-10,8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ViscosityEffect.yBase,ExerciseEffect. y) annotation (Line(
            points={{-10,12},{-10,15}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ExerciseEffect.yBase,CollapseEffect. y) annotation (Line(
            points={{-10,19},{-10,19.5},{-10.2,19.5},{-10.2,20},{-10,20},{-10,
                22}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, systemicVeinsConductance.q_in) annotation (Line(
            points={{-100,0},{-20,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect,
          ViscosityEffect.u) annotation (Line(
            points={{-86,87},{-86,10},{-19.8,10}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Exercise_MusclePump_Effect,
          ExerciseEffect.u) annotation (Line(
            points={{-86,87},{-86,17},{-19.8,17}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.BloodVol_CollapsedEffect, CollapseEffect.u) 
          annotation (Line(
            points={{-86,87},{-86,24},{-19.8,24}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(const12.y, CollapseEffect.yBase) annotation (Line(
            points={{-14.4,51},{-10,51},{-10,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(systemicVeinsConductance.q_out, flowMeasure.q_out) annotation (
            Line(
            points={{0,0},{74,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(q_out,flowMeasure. q_in) annotation (Line(
            points={{100,0},{86,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow,BloodFlow)  annotation (Line(
            points={{80,5.4},{80,12},{82,12}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics), Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Text(
                extent={{-70,-30},{70,30}},
                lineColor={0,0,0},
                textString="%BaseConductance ml/min/mmHg"), Text(
                extent={{-101,-58},{99,-30}},
                lineColor={0,0,255},
                textString="%name")}));
      end SystemicVeins;

      model GITract
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
      extends Library.Interfaces.Tissues.GITract;

        HumMod.Library.Factors.SplineValue A2Effect(data={{0.0,1.05,0},{1.3,
              1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{0,22},{20,42}})));
        HumMod.Library.Factors.SplineValue ADHEffect(data={{0.8,1.0,0},{3.0,0.1,
              0}}) 
          annotation (Placement(transformation(extent={{0,-2},{20,18}})));
        HumMod.Library.Blocks.CondConstant BasicConductance(k=11.2) 
          annotation (Placement(transformation(extent={{-6,74},{6,86}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia 
          annotation (Placement(transformation(extent={{0,50},{20,70}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity 
          annotation (Placement(transformation(extent={{0,58},{20,78}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture Vasculature(
            Tau=30, data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}) 
          annotation (Placement(transformation(extent={{0,40},{20,60}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors(data={{0.0,
              1.3,0},{1.0,1.0,-0.3},{5.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{0,10},{20,30}})));
        Library.PressureFlow.ResistorWithCond GITract 
          annotation (Placement(transformation(extent={{20,-10},{0,10}})));
        HumMod.Library.Factors.SplineValue PO2OnConductance(data={{10,2.0,0},{
              30,1.0,0}}) 
          annotation (Placement(transformation(extent={{0,30},{20,50}})));

      equation
        connect(Vasculature.y, PO2OnConductance.yBase) annotation (Line(
            points={{10,44.8},{10,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.y, Anesthesia.yBase) annotation (Line(
            points={{10,66},{10,62}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature.yBase, Anesthesia.y) annotation (Line(
            points={{10,55},{10,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect.y, GITract.cond) annotation (Line(
            points={{10,6},{10,4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PO2OnConductance.y, A2Effect.yBase) annotation (Line(
            points={{10,38},{10,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(BasicConductance.y, Viscosity.yBase)         annotation (Line(
            points={{6.6,80},{10,80},{10,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-52,90},{-52,20},{0.2,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-52,90},{-52,26},{0.2,26}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{-52,90},{-52,14},{0.2,14}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect.u) annotation (Line(
            points={{-90,90},{-52,90},{-52,8},{0.2,8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect.u) annotation (Line(
            points={{-90,90},{-52,90},{-52,32},{0.2,32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GITract_PO2, PO2OnConductance.u) annotation (Line(
            points={{-90,90},{-52,90},{-52,40},{0.2,40}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GITract_Function_Failed, Vasculature.Failed) annotation (
           Line(
            points={{-90,90},{-52,90},{-52,54},{0.2,54}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GITract_PO2, Vasculature.u) annotation (Line(
            points={{-90,90},{-52,90},{-52,40},{-4,40},{-4,48},{0.2,48}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia.u) 
          annotation (Line(
            points={{-90,90},{-52,90},{-52,60},{0.2,60}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity.u) annotation (
            Line(
            points={{-90,90},{-52,90},{-52,68},{0.2,68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(AplhaReceptors.yBase, A2Effect.y) annotation (Line(
            points={{10,27},{10,30}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors.y, ADHEffect.yBase) annotation (Line(
            points={{10,13},{10,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, GITract.q_out) annotation (Line(
            points={{-100,0},{0,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(GITract.q_in, flowMeasure.q_out) annotation (Line(
            points={{20,0},{52,0},{52,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in gastro interstitial tract.
 
QHP 2008 / GITract-Flow
 
Created : 23-May-05
Last Modified : 31-Mar-08
Author : Tom Coleman
Copyright : 2005-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(coordinateSystem(
                preserveAspectRatio=true,  extent={{-100,-100},{100,100}}),
              graphics={Text(
                extent={{-120,-106},{80,-86}},
                lineColor={0,0,255},
                textString="%name")}));
      end GITract;

      model Bone
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.Bone;

        HumMod.Library.Factors.SplineValue A2Effect_Bone(data={{0.0,1.05,0},{
              1.3,1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{-12,28},{8,48}})));
        HumMod.Library.Factors.SplineValue ADHEffect_Bone(data={{0.8,1.0,0},{
              3.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{-12,20},{8,40}})));
        HumMod.Library.Blocks.CondConstant Bone_BasicConductance(k=3.62859)
          "scaled to new coronary circulation" 
          annotation (Placement(transformation(extent={{-16,72},{-4,84}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors_Bone(data={{
              0.0,1.3,0},{1.0,1.0,-0.3},{5.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{-12,6},{8,26}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity_Bone 
          annotation (Placement(transformation(extent={{-12,58},{8,78}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia_Bone 
          annotation (Placement(transformation(extent={{-12,50},{8,70}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_Bone(data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}, Tau=30) 
          annotation (Placement(transformation(extent={{-12,39},{8,59}})));
        Library.PressureFlow.ResistorWithCond bone 
          annotation (Placement(transformation(extent={{-12,-10},{8,10}})));
      equation
        connect(Anesthesia_Bone.yBase,Viscosity_Bone. y) annotation (Line(
            points={{-2,62},{-2,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Anesthesia_Bone.y,Vasculature_Bone. yBase) annotation (Line(
            points={{-2,58},{-2,54}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_Bone.y,A2Effect_Bone. yBase) annotation (Line(
            points={{-2,43.8},{-2,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Bone_BasicConductance.y,Viscosity_Bone. yBase) annotation (Line(
            points={{-3.4,78},{-2,78},{-2,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, bone.q_in) annotation (Line(
            points={{-100,0},{-12,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity_Bone.u) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,68},{-11.8,68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia_Bone.u) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,60},{-11.8,60}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Bone_Function_Failed, Vasculature_Bone.Failed) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,53},{-11.8,53}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Bone_PO2, Vasculature_Bone.u) annotation (Line(
            points={{-90,90},{-75,90},{-75,47},{-11.8,47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect_Bone.u) annotation (Line(
            points={{-90,90},{-75,90},{-75,38},{-11.8,38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors_Bone.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,10},{-11.8,10}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_Bone.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,15.7},{-12,16},{-11.8,16}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_Bone.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-75,90},{-75,22},{-11.8,22}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect_Bone.u) annotation (Line(
            points={{-90,90},{-75,90},{-75,30},{-11.8,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(A2Effect_Bone.y, ADHEffect_Bone.yBase) annotation (Line(
            points={{-2,36},{-2,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect_Bone.y, AplhaReceptors_Bone.yBase) annotation (Line(
            points={{-2,28},{-2,23}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors_Bone.y, bone.cond) annotation (Line(
            points={{-2,9},{-2,4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(bone.q_out, flowMeasure.q_out) annotation (Line(
            points={{8,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in bones.
 
QHP 2008 / Bone-Flow
 
Created : 23-May-05
Last Modified : 19-Dec-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}},
              initialScale=0.1,
              grid={1,1}),                 graphics), Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              initialScale=0.1,
              grid={1,1}), graphics={Text(
                extent={{-100,-90},{100,-70}},
                lineColor={0,0,255},
                textString="%name")}));
      end Bone;

      model Brain
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.Brain;

        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_Brain(Tau=30, data={{27,1.2,0},{37,1.0,-0.03},{47,0.8,0}}) 
          annotation (Placement(transformation(extent={{-52,16},{-72,36}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity_Brain 
          annotation (Placement(transformation(extent={{-52,28},{-72,48}})));
        HumMod.Library.Factors.SplineValue PO2OnTension(data={{22,0.0,0},{36,
              1.0,0.02},{60,1.2,0}}) 
          annotation (Placement(transformation(extent={{22,24},{2,44}})));
                                   /* ##, curve(iFrom(start=2))*/
        HumMod.Library.Factors.SplineValue PCO2OnTension(data={{20,1.8,0},{45,
              1.0,-0.05},{75,0.0,0}}) 
          annotation (Placement(transformation(extent={{22,16},{2,36}})));
                                    /* ##, curve(iFrom(start=2))*/
        HumMod.Library.Factors.SplineValue TensionEffect(data={{0.0,2.2,0},{1.0,
              1.0,-0.5},{2.0,0.6,0}}) 
          annotation (Placement(transformation(extent={{-52,4},{-72,24}})));
                                    /* ##, curve(iFrom(start=1))*/
        Library.PressureFlow.ResistorWithCond brain 
          annotation (Placement(transformation(extent={{-72,-10},{-52,10}})));
        HumMod.Library.Blocks.CondConstant Brain_BasicConductance(k=9.1) 
          annotation (Placement(transformation(extent={{-76,42},{-64,54}})));
      equation
        connect(Viscosity_Brain.y,Vasculature_Brain. yBase) annotation (Line(
            points={{-62,36},{-62,31}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Brain_BasicConductance.y,Viscosity_Brain. yBase) annotation (
            Line(
            points={{-63.4,48},{-62,48},{-62,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PO2OnTension.y,PCO2OnTension. yBase) annotation (Line(
            points={{12,32},{12,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_Brain.y,TensionEffect. yBase) annotation (Line(
            points={{-62,20.8},{-62,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(TensionEffect.y,brain. cond) annotation (Line(
            points={{-62,12},{-62,4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(TensionEffect.u,PCO2OnTension. y) annotation (Line(
            points={{-52.2,14},{12,14},{12,24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, brain.q_in) annotation (Line(
            points={{-100,0},{-72,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(PO2OnTension.u, organBloodFlowSignals.Brain_PO2) annotation (Line(
            points={{21.8,34},{72,34},{72,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(PCO2OnTension.u, organBloodFlowSignals.Brain_PCO2) annotation (Line(
            points={{21.8,26},{72,26},{72,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Vasculature_Brain.Failed, organBloodFlowSignals.Brain_Function_Failed) 
          annotation (Line(
            points={{-52.2,30},{-8,30},{-8,90},{-90,90}},
            color={255,0,255},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(PO2OnTension.yBase, organBloodFlowSignals.Anesthesia_VascularConductance) 
          annotation (Line(
            points={{12,36},{12,46},{72,46},{72,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,13},{6,13}}));
        connect(Vasculature_Brain.u, organBloodFlowSignals.Brain_PO2) 
          annotation (Line(
            points={{-52.2,24},{-8,24},{-8,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Viscosity_Brain.u, organBloodFlowSignals.Viscosity_ConductanceEffect) 
          annotation (Line(
            points={{-52.2,38},{-8,38},{-8,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(brain.q_out, flowMeasure.q_out) annotation (Line(
            points={{-52,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in brain.
 
QHP 2008 / Brain-Flow
 
Created : 23-May-05
Last Modified : 20-Dec-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}),
                    Diagram(graphics));
      end Brain;

      model Fat
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.Fat;

        HumMod.Library.Factors.SplineValue A2Effect_Fat(data={{0.0,1.05,0},{1.3,
              1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{0,28},{20,48}})));
        HumMod.Library.Factors.SplineValue ADHEffect_Fat(data={{0.8,1.0,0},{3.0,
              0.1,0}}) 
          annotation (Placement(transformation(extent={{0,2},{20,22}})));
        Library.PressureFlow.ResistorWithCond fat 
          annotation (Placement(transformation(extent={{0,-10},{20,10}})));
        HumMod.Library.Blocks.CondConstant Fat_BasicConductance(k=2.7) 
          annotation (Placement(transformation(extent={{-4,70},{8,82}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors_Fat(data={{
              0.0,1.3,0},{1.0,1.0,-0.3},{5.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{0,16},{20,36}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity_Fat 
          annotation (Placement(transformation(extent={{0,56},{20,76}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia_Fat 
          annotation (Placement(transformation(extent={{0,48},{20,68}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_Fat(data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}, Tau=30) 
          annotation (Placement(transformation(extent={{0,38},{20,58}})));
      equation
        connect(Anesthesia_Fat.yBase,Viscosity_Fat. y)   annotation (Line(
            points={{10,60},{10,64}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Anesthesia_Fat.y,Vasculature_Fat. yBase)   annotation (Line(
            points={{10,56},{10,53}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_Fat.y,A2Effect_Fat. yBase) annotation (Line(
            points={{10,42.8},{10,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity_Fat.yBase,Fat_BasicConductance. y) annotation (Line(
            points={{10,68},{10,76},{8.6,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fat.cond,ADHEffect_Fat. y) annotation (Line(
            points={{10,4},{10,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity_Fat.u) 
          annotation (Line(
            points={{-90,90},{-56,90},{-56,66},{0.2,66}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia_Fat.u) 
          annotation (Line(
            points={{-90,90},{-56,90},{-56,58},{0.2,58}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Fat_Function_Failed, Vasculature_Fat.Failed) annotation (
           Line(
            points={{-90,90},{-56,90},{-56,52},{0.2,52}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Fat_PO2, Vasculature_Fat.u) annotation (Line(
            points={{-90,90},{-56,90},{-56,46},{0.2,46}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect_Fat.u) annotation (Line(
            points={{-90,90},{-56,90},{-56,38},{0.2,38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors_Fat.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{-56,90},{-56,20},{0.2,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_Fat.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-56,90},{-56,32},{0.2,32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_Fat.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-56,90},{-56,26},{0.2,26}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect_Fat.u) annotation (Line(
            points={{-90,90},{-56,90},{-56,12},{0.2,12}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(q_in, fat.q_in) annotation (Line(
            points={{-100,0},{0,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(AplhaReceptors_Fat.yBase, A2Effect_Fat.y) annotation (Line(
            points={{10,33},{10,36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors_Fat.y, ADHEffect_Fat.yBase) annotation (Line(
            points={{10,19},{10,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fat.q_out, flowMeasure.q_out) annotation (Line(
            points={{20,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in adipose tissue.
 
QHP 2008 / Fat-Flow
 
Created : 23-May-05
Last Modified : 19-Dec-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}));
      end Fat;

      model Kidney
        extends HumMod.Blood.OrganFlow.Base;
        extends Library.Interfaces.Tissues.Kidney;

        HumMod.Library.Factors.SimpleMultiply NephronCountEffect_AfferentArtery
          annotation (Placement(transformation(extent={{-10,34},{10,54}})));
        HumMod.Library.Blocks.CondConstant Afferent_BasicConductance(k=34) 
          annotation (Placement(transformation(extent={{-14,48},{-2,60}})));
        VariableResistorWithMyogenicResponse AfferentArtery(
            PressureChangeOnMyogenicCondEffect={{-20.0,1.2,0.0},{0.0,1.0,-0.02},
              {20.0,0.8,0.0}}, pressureChange(start=0, fixed=true)) 
          annotation (Placement(transformation(extent={{-12,-12},{12,12}})));
        HumMod.Library.Factors.SplineValue A2Effect2(data={{0.0,1.2,0.0},{1.3,
              1.0,-0.4},{3.0,0.6,0.0}}) 
          annotation (Placement(transformation(extent={{62,18},{82,38}})));
        HumMod.Library.Blocks.CondConstant KidneyEfferent_BasicConductance(k=23) 
          annotation (Placement(transformation(extent={{58,48},{70,60}})));
        HumMod.Library.Factors.SimpleMultiply
          NephronCountEffect_KidneyEfferentArtery 
          annotation (Placement(transformation(extent={{62,34},{82,54}})));
        Library.PressureFlow.ResistorWithCond EfferentArtery 
          annotation (Placement(transformation(extent={{60,-12},{84,12}})));
        Library.PressureFlow.ResistorWithCondParam AcurateArtery(cond=600) 
          annotation (Placement(transformation(extent={{-54,-8},{-38,8}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor
          AplhaReceptors_KidneyAfferentArtery(data={{1.5,1.0,0.0},{7.0,0.9,0.0}}) 
          annotation (Placement(transformation(extent={{-10,14},{10,34}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor
          AplhaReceptors_KidneyEfferentArtery(data={{1.5,1.0,0.0},{7.0,0.9,0.0}}) 
          annotation (Placement(transformation(extent={{62,6},{82,26}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia_KidneyEfferentArtery 
          annotation (Placement(transformation(extent={{62,26},{82,46}})));
        Library.PressureFlow.PressureMeasure pressureMeasure annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={16,8})));
        Modelica.Blocks.Math.Product KidneyPlasmaFlow annotation (Placement(
              transformation(
              extent={{-5,-5},{5,5}},
              rotation=90,
              origin={-67,71})));
        Library.PressureFlow.PressureMeasure pressureMeasure1 
                                                             annotation (Placement(
              transformation(
              extent={{10,-10},{-10,10}},
              rotation=0,
              origin={-24,8})));
        Modelica.Blocks.Math.Gain IFP(k=0.042) 
          annotation (Placement(transformation(extent={{-50,-86},{-58,-78}})));
        Library.Factors.DelayedToSpline TGFEffect( data={{0.0,1.2,0.0},{1.3,1.0,
              -0.4},{3.0,0.6,0.0}}, Tau=1,
          initialValue=1.01309,
          adaptationSignalName="TGF-Vascular.Signal")
          "Macula Densa TGF vascular signal delay and effect to afferent arteriole"
          annotation (Placement(transformation(extent={{-10,26},{10,46}})));
        HumMod.Library.Factors.SplineValue FurosemideEffect(data={{0.0,1.0,0.0},
              {1.3,0.2,0.0}})
          "furosemide outflow on Macula Densa Na sensibility" 
          annotation (Placement(transformation(extent={{22,-52},{42,-32}})));
        HumMod.Library.Factors.SplineValue NaEffect_MaculaDensa(data={{0,0.0,
              0.0},{48,1.0,0.03},{100,3.0,0.0}}) 
          annotation (Placement(transformation(extent={{22,-48},{42,-28}})));
        HumMod.Library.Factors.SplineValue ANP_Effect(data={{0.0,1.2,0.0},{1.3,
              1.0,-0.3},{2.7,0.8,0.0}}) 
          annotation (Placement(transformation(extent={{22,-56},{42,-36}})));
       HumMod.Library.Factors.SplineValue A2Effect3(data={{0.0,0.0,0.0},{0.2,
              0.6,0.05},{1.3,1.0,0.1},{3.0,8.0,0.0}}) 
          annotation (Placement(transformation(extent={{22,-60},{42,-40}})));
        HumMod.Library.Blocks.Constant MedulaDensa_BaseTGFSignal(k=1) 
          annotation (Placement(transformation(extent={{22,-34},{30,-26}})));
        VasaRecta vasaRecta 
          annotation (Placement(transformation(extent={{-22,-92},{-2,-72}})));
      equation
        connect(Afferent_BasicConductance.y, NephronCountEffect_AfferentArtery.yBase) 
                                                                   annotation (Line(
            points={{-1.4,54},{0,54},{0,46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Kidney_NephronCount_Total_xNormal,
          NephronCountEffect_AfferentArtery.u) annotation (Line(
            points={{-90,90},{-90,44},{-9.8,44}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_KidneyAfferentArtery.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-90,30},{-9.8,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_KidneyAfferentArtery.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-90,24},{-9.8,24}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA,
          AplhaReceptors_KidneyAfferentArtery.GangliaGeneral_NA) annotation (
            Line(
            points={{-90,90},{-90,18},{-9.8,18}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect2.u) annotation (Line(
            points={{-90,90},{22,90},{22,28},{62.2,28}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_KidneyEfferentArtery.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{22,90},{22,22},{62.2,22}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA,
          AplhaReceptors_KidneyEfferentArtery.GangliaGeneral_NA) annotation (
            Line(
            points={{-90,90},{22,90},{22,10},{62.2,10}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_KidneyEfferentArtery.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{22,90},{22,16},{62.2,16}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Kidney_NephronCount_Total_xNormal,
          NephronCountEffect_KidneyEfferentArtery.u) annotation (Line(
            points={{-90,90},{22,90},{22,44},{62.2,44}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance,
          Anesthesia_KidneyEfferentArtery.u) annotation (Line(
            points={{-90,90},{22,90},{22,36},{62.2,36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(KidneyEfferent_BasicConductance.y,
          NephronCountEffect_KidneyEfferentArtery.yBase) annotation (Line(
            points={{70.6,54},{72,54},{72,46}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(A2Effect2.y, AplhaReceptors_KidneyEfferentArtery.yBase) 
          annotation (Line(
            points={{72,26},{72,23}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(NephronCountEffect_KidneyEfferentArtery.y,
          Anesthesia_KidneyEfferentArtery.yBase) annotation (Line(
            points={{72,42},{72,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Anesthesia_KidneyEfferentArtery.y, A2Effect2.yBase) annotation (
           Line(
            points={{72,34},{72,30}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors_KidneyEfferentArtery.y, EfferentArtery.cond) 
          annotation (Line(
            points={{72,9},{72,4.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors_KidneyAfferentArtery.y, AfferentArtery.cond) 
          annotation (Line(
            points={{0,17},{0,4.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(pressureMeasure.q_in, AfferentArtery.q_out) annotation (Line(
            points={{14,4},{14,0},{12,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(pressureMeasure.actualPressure, organBloodFlowSignals.GlomerulusBloodPressure) 
          annotation (Line(
            points={{21.2,6},{21.2,-16},{-90,-16},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-3},{6,-3}}));
        connect(organBloodFlowSignals.BloodVol_PVCrit, KidneyPlasmaFlow.u2) 
          annotation (Line(
            points={{-90,90},{-54,90},{-54,60},{-64,60},{-64,65}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{6,-3},{6,-3}}));
        connect(KidneyPlasmaFlow.y, organBloodFlowSignals.KidneyPlasmaFlow) 
          annotation (Line(
            points={{-67,76.5},{-67,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{3,3},{3,3}}));
        connect(AcurateArtery.q_out, AfferentArtery.q_in) annotation (Line(
            points={{-38,0},{-12,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(EfferentArtery.q_in, AfferentArtery.q_out) annotation (Line(
            points={{60,0},{12,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(EfferentArtery.q_out, q_out) annotation (Line(
            points={{84,0},{100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(AfferentArtery.q_in, pressureMeasure1.q_in) annotation (Line(
            points={{-12,0},{-22,0},{-22,4}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(IFP.u, pressureMeasure1.actualPressure) annotation (Line(
            points={{-49.2,-82},{-32,-82},{-32,6},{-29.2,6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ANP_Effect.y,A2Effect3. yBase) annotation (Line(
            points={{32,-48},{32,-48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(FurosemideEffect.y,ANP_Effect. yBase) annotation (Line(
            points={{32,-44},{32,-44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(NaEffect_MaculaDensa.y,FurosemideEffect. yBase) annotation (
            Line(
            points={{32,-40},{32,-40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MedulaDensa_BaseTGFSignal.y,NaEffect_MaculaDensa. yBase) 
          annotation (Line(
            points={{30.4,-30},{32,-30},{32,-36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(A2Effect3.y,TGFEffect. u) annotation (Line(
            points={{32,-52},{32,-54},{-18,-54},{-18,36},{-9.8,36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(NephronCountEffect_AfferentArtery.y, TGFEffect.yBase) annotation (
            Line(
            points={{0,42},{0,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(TGFEffect.y, AplhaReceptors_KidneyAfferentArtery.yBase) annotation (
            Line(
            points={{0,34},{0,31}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.FurosemidePool_Loss, FurosemideEffect.u) 
          annotation (Line(
            points={{-90,90},{-90,-42},{22.2,-42}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ANPPool_Log10Conc, ANP_Effect.u) annotation (
            Line(
            points={{-90,90},{-90,-46},{22.2,-46}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect3.u) annotation (Line(
            points={{-90,90},{-90,-50},{22.2,-50}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(flowMeasure.actualFlow, KidneyPlasmaFlow.u1) annotation (Line(
            points={{-88,5.4},{-88,50},{-70,50},{-70,65}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.q_out, AcurateArtery.q_in) annotation (Line(
            points={{-82,0},{-54,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, vasaRecta.busConnector) annotation (Line(
            points={{-90,90},{-90,-74},{-20,-74}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(pressureMeasure1.actualPressure, vasaRecta.ArcuateArtery_Pressure) 
          annotation (Line(
            points={{-29.2,6},{-32,6},{-32,-82},{-20,-82}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.NephronIFP_Pressure, IFP.y) annotation (
            Line(
            points={{-90,90},{-90,-82},{-58.4,-82}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.MD_Na, NaEffect_MaculaDensa.u) 
          annotation (Line(
            points={{-90,90},{-90,-38},{22.2,-38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in kidneys.
 
QHP 2008 / Kidney-Flow
 
Created : 09-Apr-05
Last Modified : 09-Apr-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}), graphics={Rectangle(extent={{14,-24},{50,-62}},
                  lineColor={127,0,0}), Text(
                extent={{14,-62},{50,-56}},
                lineColor={0,0,255},
                textString="Macula Densa")}),         Icon(coordinateSystem(
                preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
              graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}),
                    Diagram(graphics));
      end Kidney;

    model VariableResistorWithMyogenicResponse
      extends HumMod.Library.PressureFlow.Resistor; //(q_in(pressure(start=95.5)),q_out(pressure(start=61.5)));
      Library.Interfaces.RealInput_ cond(    final quantity="Conductance",       final unit
            =                                                                               "ml/(min.mmHg)") 
          annotation (Placement(transformation(extent={{-100,92},             {-80,112}}), iconTransformation(
                   extent={{-10,-10},{10,10}},         rotation=270,         origin={0,40})));
      Real myogenicEffect;
      Real pressureChange(start=0.161874);
      Real pressureChangeSteadyState;
      Real adaptedPressure(start=77.9546);
      Real interlobarPressure;
      parameter Real K_PressureChange(final unit="1/min") =  2.0;
      parameter Real Tau_PressureAdoption(final unit="hod") =  4.0;
      parameter Real[:,3] PressureChangeOnMyogenicCondEffect;
      Library.Curves.Curve PressureChangeOnCondEffect(     x=PressureChangeOnMyogenicCondEffect[:, 1],
      y=PressureChangeOnMyogenicCondEffect[:, 2],     slope=PressureChangeOnMyogenicCondEffect[:, 3]) 
      annotation (Placement(transformation(extent={{14,26},{34,46}})));

    initial equation
      der(adaptedPressure) = 0;
     // der(pressureChange) = 0;
    equation
      q_in.q = myogenicEffect * cond * (q_in.pressure - q_out.pressure);
      interlobarPressure = (q_in.pressure + q_out.pressure)/2;  //stredny tlak v arteriole
      der(adaptedPressure) = (1/(60*Tau_PressureAdoption)) * (interlobarPressure-adaptedPressure); //receptory sa adaptuju na tlak (polcas adaptacie nastane za 2hod a 45min)
      pressureChangeSteadyState = interlobarPressure-adaptedPressure;  //zmena tlaku sa taktiez adaptuje - neviem preco? => potom to vizera, ze stimulacia receptorov je postupny proces (polcas nabudenia stimulacie je 20 sekund)
      der(pressureChange) = K_PressureChange * (pressureChangeSteadyState - pressureChange);
      PressureChangeOnCondEffect.u = pressureChange; //zmena tlaku aktivuje receptory
      myogenicEffect = PressureChangeOnCondEffect.val; //stimulacia receptorov ma vplyv na vodivost (odpor) krvi aferentnej arterie
      annotation (Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}}),
      graphics), Icon(graphics={
              Text(
                extent={{-110,-68},{106,-32}},
                textString="%name",
                lineColor={0,0,255}),
              Text(
                extent={{-68,-20},{66,12}},
                lineColor={0,0,0},
                textString="blood flow"),
              Rectangle(
                extent={{-70,30},{70,-30}},
                lineColor={0,0,255},
                fillColor={215,215,215},
                fillPattern=FillPattern.Sphere),
              Text(
                extent={{-54,20},{52,-18}},
                lineColor={0,0,0},
                fillPattern=FillPattern.Sphere,
                fillColor={215,215,215},
                textString="Myogenic")}),
            Documentation(info="<html> <pre>Receptory sa adaptuju na priemerny tlak v arteriole - polcas adaptacie nastane za 2hod a 45min. 
        Zmena tlaku sa taktiez adaptuje - neviem preco? =&GT; potom to vizera, ze stimulacia receptorov je postupny proces (polcas nabudenia stimulacie je 20 sekund). 
        Zmena tlaku aktivuje receptory, ktorych stimulacia ma vplyv na vasokonstrikciu danej arterioly.</pre> </html>"));
    end VariableResistorWithMyogenicResponse;

      model Skin
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.Skin;

        HumMod.Library.Factors.SplineValue LocalTempVsNA(data={{1.2,1.0,0},{1.5,
              0.0,0}}) 
          annotation (Placement(transformation(extent={{-36,52},{-16,72}})));
        Library.PressureFlow.ResistorWithCond skin 
          annotation (Placement(transformation(extent={{56,-12},{80,12}})));
        HumMod.Library.Factors.SplineValue A2Effect_Skin(data={{0.0,1.05,0},{
              1.3,1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{58,30},{78,50}})));
        HumMod.Library.Factors.SplineValue ADHEffect_Skin(data={{0.8,1.0,0},{
              3.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{58,6},{78,26}})));
        HumMod.Library.Blocks.CondConstant Skin_BasicConductance_(k=1.6) 
          annotation (Placement(transformation(extent={{54,72},{66,84}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors_Skin(data={{
              0.0,0.3,0},{1.0,0.0,-0.3},{5.0,-0.9,0}}) 
          annotation (Placement(transformation(extent={{-50,26},{-30,46}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity_Skin 
          annotation (Placement(transformation(extent={{58,58},{78,78}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia_Skin 
          annotation (Placement(transformation(extent={{58,50},{78,70}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_Skin(Tau=30, data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}) 
          annotation (Placement(transformation(extent={{58,40},{78,60}})));
        HumMod.Library.Factors.SplineValue LocalTempEffect(data={{10.8,-0.8,0},
              {29.0,0.0,0.1},{45.0,4.0,0}}) 
          annotation (Placement(transformation(extent={{-34,38},{-14,58}})));
        HumMod.Library.Blocks.Constant Skin_BasicConductance_1(k=1) 
          annotation (Placement(transformation(extent={{-20,70},{-24,66}})));
        HumMod.Library.Factors.SimpleMultiply LocalTempEffect_Skin 
          annotation (Placement(transformation(extent={{58,22},{78,42}})));
        HumMod.Library.Factors.SimpleMultiply SympsEffect_Skin 
          annotation (Placement(transformation(extent={{58,14},{78,34}})));
        HumMod.Library.Factors.SplineValue SympsDilateEffect(data={{0.0,0.3,0},
              {1.0,1.0,2.2},{4.0,8.0,0}}) 
          annotation (Placement(transformation(extent={{58,-2},{78,18}})));
        HumMod.Library.Blocks.Add add(k=1) 
          annotation (Placement(transformation(extent={{24,28},{32,36}})));
        HumMod.Library.Blocks.Add add1(k=1) 
          annotation (Placement(transformation(extent={{14,20},{22,28}})));
      equation
        connect(Anesthesia_Skin.yBase,Viscosity_Skin. y) annotation (Line(
            points={{68,62},{68,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Anesthesia_Skin.y,Vasculature_Skin. yBase) annotation (Line(
            points={{68,58},{68,55}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_Skin.y,A2Effect_Skin. yBase) annotation (Line(
            points={{68,44.8},{68,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Skin_BasicConductance_.y,Viscosity_Skin. yBase) 
                                                               annotation (Line(
            points={{66.6,78},{68,78},{68,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LocalTempVsNA.yBase,Skin_BasicConductance_1. y) annotation (Line(
            points={{-26,64},{-26,68},{-24.2,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LocalTempEffect.yBase,LocalTempVsNA. y) annotation (Line(
            points={{-24,50},{-24,56},{-26,56},{-26,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LocalTempEffect_Skin.yBase,A2Effect_Skin. y) annotation (Line(
            points={{68,34},{68,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LocalTempEffect_Skin.y,SympsEffect_Skin. yBase) annotation (Line(
            points={{68,30},{68,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SympsEffect_Skin.y,ADHEffect_Skin. yBase) annotation (Line(
            points={{68,22},{68,18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect_Skin.y,SympsDilateEffect. yBase) annotation (Line(
            points={{68,14},{68,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SympsDilateEffect.y,skin. cond) annotation (Line(
            points={{68,6},{68,4.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity_Skin.u) 
          annotation (Line(
            points={{-90,90},{-12,90},{-12,68},{58.2,68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia_Skin.u) 
          annotation (Line(
            points={{-90,90},{-12,90},{-12,60},{58.2,60}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Skin_Function_Failed, Vasculature_Skin.Failed) 
          annotation (Line(
            points={{-90,90},{-12,90},{-12,54},{58.2,54}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Skin_PO2, Vasculature_Skin.u) annotation (Line(
            points={{-90,90},{-12,90},{-12,48},{58.2,48}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect_Skin.u) annotation (Line(
            points={{-90,90},{-12,90},{-12,40},{58.2,40}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect_Skin.u) annotation (Line(
            points={{-90,90},{-12,90},{-12,16},{58.2,16}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.HypothalamusSkinFlow_NervesActivity, SympsDilateEffect.u) 
          annotation (Line(
            points={{-90,90},{-88,90},{-88,8},{58.2,8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.HypothalamusSkinFlow_NervesActivity, LocalTempVsNA.u) 
          annotation (Line(
            points={{-90,90},{-88,90},{-88,62},{-35.8,62}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.skin_T, LocalTempEffect.u) annotation (Line(
            points={{-90,90},{-90,48},{-33.8,48}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_Skin.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-90,42},{-49.8,42}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_Skin.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-90,36},{-49.8,36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors_Skin.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{-90,30},{-49.8,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(q_in, skin.q_in) annotation (Line(
            points={{-100,0},{56,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(AplhaReceptors_Skin.yBase, LocalTempVsNA.y) annotation (Line(
            points={{-40,43},{-40,56},{-26,56},{-26,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LocalTempEffect.y, add.u) annotation (Line(
            points={{-24,46},{-24,32},{23.2,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add.y, LocalTempEffect_Skin.u) annotation (Line(
            points={{32.4,32},{58.2,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SympsEffect_Skin.u, add1.y) annotation (Line(
            points={{58.2,24},{22.4,24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add1.u, AplhaReceptors_Skin.y) annotation (Line(
            points={{13.2,24},{-40,24},{-40,29}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(skin.q_out, flowMeasure.q_out) annotation (Line(
            points={{80,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
         annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in skin.
 
QHP 2008 / Skin-Flow
 
Created : 23-May-05
Last Modified : 01-Jul-06
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}));
      end Skin;

      model OtherTissue
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.OtherTissue;

        HumMod.Library.Blocks.CondConstant OtherTissue_BasicConductance(k=4.2) 
          annotation (Placement(transformation(extent={{20,80},{32,92}})));
        Library.PressureFlow.ResistorWithCond OtherTissue 
          annotation (Placement(transformation(extent={{22,-12},{46,12}})));
        HumMod.Library.Factors.SplineValue A2Effect_OtherTissue(data={{0.0,1.05,
              0},{1.3,1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{24,28},{44,48}})));
        HumMod.Library.Factors.SplineValue ADHEffect_OtherTissue(data={{0.8,1.0,
              0},{3.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{24,2},{44,22}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia_OtherTissue 
          annotation (Placement(transformation(extent={{24,56},{44,76}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity_OtherTissue 
          annotation (Placement(transformation(extent={{24,64},{44,84}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_OtherTissue(Tau=30, data={{41,1.2,0},{51,1.0,-0.03},{61,
              0.8,0}}) 
          annotation (Placement(transformation(extent={{24,46},{44,66}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor AplhaReceptors_OtherTissue(
            data={{0.0,1.3,0},{1.0,1.0,-0.3},{5.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{24,16},{44,36}})));
        HumMod.Library.Factors.SplineValue PO2OnConductance_OtherTissue(data={{
              10,2.0,0},{30,1.0,0}}) 
          annotation (Placement(transformation(extent={{24,36},{44,56}})));
      equation
        connect(Vasculature_OtherTissue.y,PO2OnConductance_OtherTissue. yBase) 
                                                       annotation (Line(
            points={{34,50.8},{34,48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity_OtherTissue.y,Anesthesia_OtherTissue. yBase) 
                                               annotation (Line(
            points={{34,72},{34,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_OtherTissue.yBase,Anesthesia_OtherTissue. y) 
                                                 annotation (Line(
            points={{34,61},{34,64}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PO2OnConductance_OtherTissue.y,A2Effect_OtherTissue. yBase) 
                                                    annotation (Line(
            points={{34,44},{34,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(OtherTissue_BasicConductance.y,Viscosity_OtherTissue. yBase) 
          annotation (Line(
            points={{32.6,86},{34,86},{34,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect_OtherTissue.y, OtherTissue.cond) annotation (Line(
            points={{34,10},{34,4.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, OtherTissue.q_in) annotation (Line(
            points={{-100,0},{22,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity_OtherTissue.u) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,74},{24.2,74}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia_OtherTissue.u) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,66},{24.2,66}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.OtherTissue_Function_Failed, Vasculature_OtherTissue.Failed) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,60},{24.2,60}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.OtherTissue_PO2, Vasculature_OtherTissue.u) annotation (
            Line(
            points={{-90,90},{-40,90},{-40,46},{22,46},{22,54},{24.2,54}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals.OtherTissue_PO2, PO2OnConductance_OtherTissue.u) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,46},{24.2,46}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.A2Pool_Log10Conc, A2Effect_OtherTissue.u) annotation (
            Line(
            points={{-90,90},{-40,90},{-40,38},{24.2,38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, AplhaReceptors_OtherTissue.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,32},{24.2,32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, AplhaReceptors_OtherTissue.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,20},{24.2,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, AplhaReceptors_OtherTissue.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{-40,90},{-40,26},{24.2,26}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect_OtherTissue.u) annotation (
            Line(
            points={{-90,90},{-40,90},{-40,12},{24.2,12}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(AplhaReceptors_OtherTissue.yBase, A2Effect_OtherTissue.y) annotation (
           Line(
            points={{34,33},{34,36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AplhaReceptors_OtherTissue.y, ADHEffect_OtherTissue.yBase) 
          annotation (Line(
            points={{34,19},{34,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(OtherTissue.q_out, flowMeasure.q_out) annotation (Line(
            points={{46,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in other tissue.
 
 
QHP 2008 / OtherTissue-Flow
 
Created : 31-Mar-05
Last Modified : 12-Dec-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}));
      end OtherTissue;

      model SkeletalMuscle
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.SkeletalMuscle;

        HumMod.Library.Factors.SplineValue A2Effect(data={{0.0,1.05,0},{1.3,
              1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{-42,-36},{-62,-16}})));
        HumMod.Library.Blocks.CondConstant SkeletalMuscle_BasicConductance(k=
              7.2) 
          annotation (Placement(transformation(extent={{-68,30},{-56,42}})));
        HumMod.Library.Factors.SplineValue ADHEffect(data={{0.8,1.0,0},{3.0,0.1,
              0}}) 
          annotation (Placement(transformation(extent={{-42,-60},{-62,-40}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity 
          annotation (Placement(transformation(extent={{-42,16},{-62,36}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia 
          annotation (Placement(transformation(extent={{-42,8},{-62,28}})));
        Library.PressureFlow.ResistorWithCond SkeletalMuscle 
          annotation (Placement(transformation(extent={{-66,-74},{-38,-46}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor aplhaReceptorsActivityFactor(
            data={{0.0,1.3,0},{1.0,1.0,-0.2},{4.0,0.5,0}}) 
          annotation (Placement(transformation(extent={{-42,-48},{-62,-28}})));
            /* ##, SympsOnConductance(
              iFrom(start=1))*/
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture
          Vasculature_skeletalMuscle(data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}},
            Tau=30) 
          annotation (Placement(transformation(extent={{-42,-2},{-62,18}})));
        HumMod.Library.Factors.SplineValue PO2Effect(data={{0,4.0,0},{25,2.5,-0.2},
              {35,1.0,0}}) 
          annotation (Placement(transformation(extent={{-42,-12},{-62,8}})));
                         /* ##, curve(iFrom(start=2))*/
        HumMod.Library.Factors.SplineValue IntensityEffect(data={{0,0.0,0.007},
              {300,1.0,0.0}}) 
          annotation (Placement(transformation(extent={{24,44},{4,64}})));
        HumMod.Library.Factors.SplineValue RateEffect(data={{0,0.0,0.04},{60,
              1.0,0.0}}) 
          annotation (Placement(transformation(extent={{24,36},{4,56}})));
        HumMod.Library.Factors.SimpleMultiply MusclePumping_SkeletalMuscle 
          annotation (Placement(transformation(extent={{-42,-28},{-62,-8}})));
        HumMod.Library.Blocks.Constant const(k=1) 
          annotation (Placement(transformation(extent={{8,60},{12,64}})));
        ConductanceFactors.MetabolicVasolidation2 metabolicVasolidation(
          data={{50,1.0,0},{1000,3.5,0.003},{3000,5.5,0}},
          OnTau=0.2,
          OffTau=1) 
          annotation (Placement(transformation(extent={{-42,-20},{-62,0}})));
        HumMod.Library.Blocks.Add add(k=1) 
          annotation (Placement(transformation(extent={{-4,-22},{-12,-14}})));
      equation
        connect(SkeletalMuscle.q_in, q_in) annotation (Line(
            points={{-66,-60},{-92,-60},{-92,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(Viscosity.y,Anesthesia. yBase) annotation (Line(
            points={{-52,24},{-52,20}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_skeletalMuscle.yBase,Anesthesia. y) annotation (
            Line(
            points={{-52,13},{-52,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature_skeletalMuscle.y,PO2Effect. yBase) annotation (Line(
            points={{-52,2.8},{-52,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MusclePumping_SkeletalMuscle.y,A2Effect. yBase) annotation (
            Line(
            points={{-52,-20},{-52,-24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.yBase,SkeletalMuscle_BasicConductance. y) annotation (
           Line(
            points={{-52,28},{-52,36},{-55.4,36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SkeletalMuscle.cond, ADHEffect.y) annotation (Line(
            points={{-52,-54.4},{-52,-52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const.y, IntensityEffect.yBase) annotation (Line(
            points={{12.2,62},{14,62},{14,56}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(IntensityEffect.y, RateEffect.yBase) annotation (Line(
            points={{14,52},{14,48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.u, organBloodFlowSignals.Viscosity_ConductanceEffect) annotation (
            Line(
            points={{-42.2,26},{0,26},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Anesthesia.u, organBloodFlowSignals.Anesthesia_VascularConductance) 
          annotation (Line(
            points={{-42.2,18},{0,18},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Vasculature_skeletalMuscle.Failed, organBloodFlowSignals.SkeletalMuscle_Function_Failed) 
          annotation (Line(
            points={{-42.2,12},{0,12},{0,90},{-90,90}},
            color={255,0,255},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Vasculature_skeletalMuscle.u, organBloodFlowSignals.SkeletalMuscle_PO2) 
          annotation (Line(
            points={{-42.2,6},{0,6},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(PO2Effect.u, organBloodFlowSignals.SkeletalMuscle_PO2) annotation (Line(
            points={{-42.2,-2},{0,-2},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(metabolicVasolidation.O2Need, organBloodFlowSignals.SkeletalMuscle_O2Need) 
          annotation (Line(
            points={{-42.2,-10},{0,-10},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(A2Effect.u, organBloodFlowSignals.A2Pool_Log10Conc) annotation (Line(
            points={{-42.2,-26},{0,-26},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.AlphaPool_Effect, organBloodFlowSignals.AlphaPool_Effect) 
          annotation (Line(
            points={{-42.2,-32},{0,-32},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.AlphaBlockade_Effect, organBloodFlowSignals.AlphaBlocade_Effect) 
          annotation (Line(
            points={{-42.2,-38},{0,-38},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.GangliaGeneral_NA, organBloodFlowSignals.GangliaGeneral_NA) 
          annotation (Line(
            points={{-42.2,-44},{0,-44},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(ADHEffect.u, organBloodFlowSignals.ADHPool_Log10Conc) annotation (Line(
            points={{-42.2,-50},{0,-50},{0,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(IntensityEffect.u, organBloodFlowSignals.Exercise_Metabolism_MotionWatts) 
          annotation (Line(
            points={{23.8,54},{82,54},{82,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(RateEffect.u, organBloodFlowSignals.Exercise_Metabolism_ContractionRate) 
          annotation (Line(
            points={{23.8,46},{82,46},{82,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(metabolicVasolidation.yBase, PO2Effect.y) annotation (Line(
            points={{-52,-8},{-52,-4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(metabolicVasolidation.y, MusclePumping_SkeletalMuscle.yBase) 
          annotation (Line(
            points={{-52,-12},{-52,-16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(A2Effect.y, aplhaReceptorsActivityFactor.yBase) annotation (Line(
            points={{-52,-28},{-52,-31}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect.yBase, aplhaReceptorsActivityFactor.y) annotation (Line(
            points={{-52,-48},{-52,-45}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MusclePumping_SkeletalMuscle.u, add.y) annotation (Line(
            points={{-42.2,-18},{-12.4,-18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(add.u, RateEffect.y) annotation (Line(
            points={{-3.2,-18},{14,-18},{14,44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SkeletalMuscle.q_out, flowMeasure.q_out) annotation (Line(
            points={{-38,-60},{38,-60},{38,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in skeletal muscle.
 
QHP 2008 / SkeletalMuscle-Flow
 
Created : 23-May-05
Last Modified : 04-Aug-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}));
      end SkeletalMuscle;

      model RespiratoryMuscle
        extends HumMod.Blood.OrganFlow.BaseBadDirection;
        extends Library.Interfaces.Tissues.RespiratoryMuscle;

        HumMod.Library.Factors.SplineValue A2Effect(data={{0.0,1.05,0},{1.3,
              1.00,-0.08},{3.5,0.50,0}}) 
          annotation (Placement(transformation(extent={{-6,10},{-26,30}})));
        HumMod.Library.Blocks.CondConstant BasicConductance(k=1.1) 
          annotation (Placement(transformation(extent={{-32,70},{-20,82}})));
        HumMod.Library.Factors.SplineValue ADHEffect(data={{0.8,1.0,0},{3.0,0.1,
              0}}) 
          annotation (Placement(transformation(extent={{-6,-14},{-26,6}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity 
          annotation (Placement(transformation(extent={{-6,54},{-26,74}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia 
          annotation (Placement(transformation(extent={{-6,46},{-26,66}})));
        Library.PressureFlow.ResistorWithCond respiratorylMuscle 
          annotation (Placement(transformation(extent={{-30,-30},{-2,-2}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor aplhaReceptorsActivityFactor(
            data={{0.0,1.3,0},{1.0,1.0,-0.3},{5.0,0.1,0}}) 
          annotation (Placement(transformation(extent={{-6,-2},{-26,18}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture Vasculature(
            Tau=30, data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}) 
          annotation (Placement(transformation(extent={{-6,36},{-26,56}})));
        HumMod.Library.Factors.SplineValue PO2Effect(data={{10,2.0,0},{30,1.0,0}}) 
          annotation (Placement(transformation(extent={{-6,26},{-26,46}})));
        HumMod.Library.Factors.SplineValue MetabolismEffect(data={{6,1.0,0},{12,
              1.3,0.08},{400,24.0,0}}) 
          annotation (Placement(transformation(extent={{-6,18},{-26,38}})));
      equation
        connect(respiratorylMuscle.q_in, q_in) 
                                           annotation (Line(
            points={{-30,-16},{-90,-16},{-90,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(Viscosity.y,Anesthesia. yBase) annotation (Line(
            points={{-16,62},{-16,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature.yBase, Anesthesia.y)                annotation (
            Line(
            points={{-16,51},{-16,54}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature.y, PO2Effect.yBase)                annotation (Line(
            points={{-16,40.8},{-16,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.yBase, BasicConductance.y)                annotation (
           Line(
            points={{-16,66},{-16,76},{-19.4,76}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(respiratorylMuscle.cond, ADHEffect.y) annotation (Line(
            points={{-16,-10.4},{-16,-9.2},{-16,-6},{-16,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.u, organBloodFlowSignals.Viscosity_ConductanceEffect) annotation (
            Line(
            points={{-6.2,64},{62,64},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Anesthesia.u, organBloodFlowSignals.Anesthesia_VascularConductance) 
          annotation (Line(
            points={{-6.2,56},{62,56},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Vasculature.Failed, organBloodFlowSignals.RespiratoryMuscle_Function_Failed) 
          annotation (Line(
            points={{-6.2,50},{62,50},{62,90},{-90,90}},
            color={255,0,255},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Vasculature.u, organBloodFlowSignals.RespiratoryMuscle_PO2) annotation (Line(
            points={{-6.2,44},{62,44},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(PO2Effect.u, organBloodFlowSignals.RespiratoryMuscle_PO2) annotation (Line(
            points={{-6.2,36},{62,36},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(A2Effect.u, organBloodFlowSignals.A2Pool_Log10Conc) annotation (Line(
            points={{-6.2,20},{62,20},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.AlphaPool_Effect, organBloodFlowSignals.AlphaPool_Effect) 
          annotation (Line(
            points={{-6.2,14},{62,14},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.AlphaBlockade_Effect, organBloodFlowSignals.AlphaBlocade_Effect) 
          annotation (Line(
            points={{-6.2,8},{62,8},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(aplhaReceptorsActivityFactor.GangliaGeneral_NA, organBloodFlowSignals.GangliaGeneral_NA) 
          annotation (Line(
            points={{-6.2,2},{62,2},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(ADHEffect.u, organBloodFlowSignals.ADHPool_Log10Conc) annotation (Line(
            points={{-6.2,-4},{62,-4},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(PO2Effect.y, MetabolismEffect.yBase) annotation (Line(
            points={{-16,34},{-16,30}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MetabolismEffect.u, organBloodFlowSignals.RespiratoryMuscle_O2Need) 
          annotation (Line(
            points={{-6.2,28},{62,28},{62,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(MetabolismEffect.y, A2Effect.yBase) annotation (Line(
            points={{-16,26},{-16,22}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(aplhaReceptorsActivityFactor.yBase, A2Effect.y) annotation (Line(
            points={{-16,15},{-16,18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(aplhaReceptorsActivityFactor.y, ADHEffect.yBase) annotation (Line(
            points={{-16,1},{-16,-2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(respiratorylMuscle.q_out, flowMeasure.q_out) annotation (Line(
            points={{-2,-16},{70,-16},{70,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in respiratory musclet.
 
QHP 2008 / RespiratoryMuscle-Flow
 
Created : 23-May-05
Last Modified : 19-Dec-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-102,-100},{98,-80}},
                lineColor={0,0,255},
                textString="%name")}));
      end RespiratoryMuscle;

      model Heart
        extends HumMod.Blood.OrganFlow.BaseBadDirection;

      parameter Real BasicLargeVeselsConductance( final quantity="Conductance", final unit
            =                                                                              "ml/(min.mmHg)");
      parameter Real BasicSmallVeselsConductance( final quantity="Conductance", final unit
            =                                                                              "ml/(min.mmHg)");

        HumMod.Library.Factors.SplineValue PO2Effect(data={{12,2.0,0},{17,1.0,-0.04},
              {30,0.8,0}}) 
          annotation (Placement(transformation(extent={{60,20},{80,40}})));
        HumMod.Library.Factors.SplineValue ADHEffect2(data={{0.8,1.0,0},{3.0,
              0.1,0}}) 
          annotation (Placement(transformation(extent={{60,58},{80,78}})));
        HumMod.Library.Factors.SplineValue MetabolismEffect(data={{30,1.0,0},{
              100,3.0,0}}) 
          annotation (Placement(transformation(extent={{60,10},{80,30}})));
        HumMod.Library.Factors.SimpleMultiply Anesthesia 
          annotation (Placement(transformation(extent={{60,66},{80,86}})));
        Library.PressureFlow.ResistorWithCond SmallVessel 
          annotation (Placement(transformation(extent={{56,-14},{84,14}})));
        HumMod.Library.Blocks.CondConstant LargeVesselBasicConductance(k=
              BasicLargeVeselsConductance) 
          annotation (Placement(transformation(extent={{-6,18},{-18,30}})));
        Library.PressureFlow.ResistorWithCond LargeVessel 
          annotation (Placement(transformation(extent={{-34,-14},{-6,14}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity1 
          annotation (Placement(transformation(extent={{-30,4},{-10,24}})));
        HumMod.Library.Blocks.CondConstant SmallVesselBasicConductance(k=
              BasicSmallVeselsConductance) 
          annotation (Placement(transformation(extent={{90,86},{78,98}})));
        HumMod.Nerves.AplhaReceptorsActivityFactor aplhaReceptorsActivityFactor(
            data={{0.0,1.3,0},{1.0,1.0,-0.16},{4.0,0.8,0}}) 
          annotation (Placement(transformation(extent={{60,46},{80,66}})));
        ConductanceFactors.HeartInfraction Infraction 
          annotation (Placement(transformation(extent={{60,2},{80,22}})));
        HumMod.Library.Factors.SplineDelayFactorByDayWithFailture Vasculature(
            data={{41,1.2,0},{51,1.0,-0.03},{61,0.8,0}}, Tau=30) 
          annotation (Placement(transformation(extent={{60,30},{80,50}})));
        HumMod.Library.Factors.SimpleMultiply Viscosity 
          annotation (Placement(transformation(extent={{60,74},{80,94}})));
      equation
        connect(LargeVesselBasicConductance.y, Viscosity1.yBase) annotation (Line(
            points={{-18.6,24},{-20,24},{-20,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity1.y, LargeVessel.cond) annotation (Line(
            points={{-20,12},{-20,5.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Viscosity.y, Anesthesia.yBase) annotation (Line(
            points={{70,82},{70,78}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Infraction.y, SmallVessel.cond) annotation (Line(
            points={{70,10},{70,5.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(organBloodFlowSignals.Anesthesia_VascularConductance, Anesthesia.u) 
          annotation (Line(
            points={{-90,90},{6,90},{6,76},{60.2,76}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity.u) annotation (
            Line(
            points={{-90,90},{6,90},{6,84},{60.2,84}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.ADHPool_Log10Conc, ADHEffect2.u) annotation (Line(
            points={{-90,90},{6,90},{6,68},{60.2,68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaPool_Effect, aplhaReceptorsActivityFactor.AlphaPool_Effect) 
          annotation (Line(
            points={{-90,90},{6,90},{6,62},{60.2,62}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.AlphaBlocade_Effect, aplhaReceptorsActivityFactor.AlphaBlockade_Effect) 
          annotation (Line(
            points={{-90,90},{6,90},{6,56},{60.2,56}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.GangliaGeneral_NA, aplhaReceptorsActivityFactor.GangliaGeneral_NA) 
          annotation (Line(
            points={{-90,90},{6,90},{6,50},{60.2,50}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.Viscosity_ConductanceEffect, Viscosity1.u) annotation (
            Line(
            points={{-90,90},{-74,90},{-74,14},{-29.8,14}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(q_in, LargeVessel.q_in) annotation (Line(
            points={{-100,0},{-67,0},{-67,1.77636e-015},{-34,1.77636e-015}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(LargeVessel.q_out, SmallVessel.q_in) annotation (Line(
            points={{-6,1.77636e-015},{14,1.77636e-015},{16,0},{56,1.77636e-015}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(Viscosity.yBase, SmallVesselBasicConductance.y) annotation (
            Line(
            points={{70,86},{70,92},{77.4,92}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Anesthesia.y, ADHEffect2.yBase) annotation (Line(
            points={{70,74},{70,70}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ADHEffect2.y, aplhaReceptorsActivityFactor.yBase) annotation (
            Line(
            points={{70,66},{70,63}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(aplhaReceptorsActivityFactor.y, Vasculature.yBase) annotation (
            Line(
            points={{70,49},{70,45}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Vasculature.y, PO2Effect.yBase) annotation (Line(
            points={{70,34.8},{70,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PO2Effect.y, MetabolismEffect.yBase) annotation (Line(
            points={{70,28},{70,22}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MetabolismEffect.y, Infraction.yBase) annotation (Line(
            points={{70,18},{70,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.q_out, SmallVessel.q_out) annotation (Line(
            points={{82,0},{82,1.77636e-015},{84,1.77636e-015}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in left heart.
 
QHP 2008 / LeftHeart-Flow
 
Created : 23-May-05
Last Modified : 23-May-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-100,-100},{100,-80}},
                lineColor={0,0,255},
                textString="%name")}),
                    Diagram(graphics));
      end Heart;

      model PeripheralFlow
        import QHP = HumMod;
        extends QHP.Blood.OrganFlow.BaseBadDirection;
        //extends Library.PressureFlow.ResistorBase;


        QHP.Blood.OrganFlow.Bone bone 
          annotation (Placement(transformation(extent={{-38,64},{-12,90}})));
        QHP.Blood.OrganFlow.Brain brain 
          annotation (Placement(transformation(extent={{18,46},{44,72}})));
        QHP.Blood.OrganFlow.Fat fat 
          annotation (Placement(transformation(extent={{-38,26},{-12,52}})));
        QHP.Blood.OrganFlow.Kidney kidney 
          annotation (Placement(transformation(extent={{44,8},{18,34}})));
        QHP.Blood.OrganFlow.Skin skin 
          annotation (Placement(transformation(extent={{22,-34},{46,-10}})));
        QHP.Blood.OrganFlow.OtherTissue otherTissue 
          annotation (Placement(transformation(extent={{-36,-98},{-10,-72}})));
        QHP.Blood.OrganFlow.SkeletalMuscle skeletalMuscle 
          annotation (Placement(transformation(extent={{-34,-54},{-8,-28}})));
        QHP.Blood.OrganFlow.RespiratoryMuscle respiratoryMuscle 
          annotation (Placement(transformation(extent={{20,-78},{46,-52}})));
         QHP.Library.Interfaces.BusConnector organBloodFlowSignals
          "signals of organ bood flow resistence" 
          annotation (Placement(transformation(extent={{-100,80},{-80,100}}),
              iconTransformation(extent={{60,10},{78,28}})));
         QHP.Blood.OrganFlow.SplanchnicCirculation splanchnicCirculation 
          annotation (Placement(transformation(extent={{-46,-18},{-8,20}})));
      equation

        connect(bone.q_in, q_in) annotation (Line(
            points={{-38,77},{-76,77},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(brain.q_in, q_in) annotation (Line(
            points={{18,59},{-76,59},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(fat.q_in, q_in) annotation (Line(
            points={{-38,39},{-76,39},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(skin.q_in, q_in) annotation (Line(
            points={{22,-22},{-76,-22},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(otherTissue.q_in, q_in)  annotation (Line(
            points={{-36,-85},{-76,-85},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(skeletalMuscle.q_in, q_in)  annotation (Line(
            points={{-34,-41},{-76,-41},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(respiratoryMuscle.q_in, q_in) annotation (Line(
            points={{20,-65},{-76,-65},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(respiratoryMuscle.q_out,flowMeasure.q_out) annotation (Line(
            points={{46,-65},{76,-65},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(skeletalMuscle.q_out,flowMeasure.q_out)  annotation (Line(
            points={{-8,-41},{76,-41},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(otherTissue.q_out,flowMeasure.q_out)  annotation (Line(
            points={{-10,-85},{76,-85},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(skin.q_out,flowMeasure.q_out) annotation (Line(
            points={{46,-22},{76,-22},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(fat.q_out,flowMeasure.q_out) annotation (Line(
            points={{-12,39},{76,39},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(brain.q_out,flowMeasure.q_out) annotation (Line(
            points={{44,59},{76,59},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(bone.q_out,flowMeasure.q_out) annotation (Line(
            points={{-12,77},{76,77},{76,0},{82,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(organBloodFlowSignals, otherTissue.organBloodFlowSignals)  annotation (Line(
            points={{-90,90},{52,90},{52,-82.53},{-14.03,-82.53}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, skin.organBloodFlowSignals) annotation (Line(
            points={{-90,90},{52,90},{52,-19.72},{42.28,-19.72}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, kidney.organBloodFlowSignals) annotation (Line(
            points={{-90,90},{14,90},{14,23.47},{22.03,23.47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, fat.organBloodFlowSignals) annotation (Line(
            points={{-90,90},{52,90},{52,41.47},{-16.03,41.47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, brain.organBloodFlowSignals) annotation (Line(
            points={{-90,90},{52,90},{52,61.47},{39.97,61.47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, bone.organBloodFlowSignals) annotation (Line(
            points={{-90,90},{52,90},{52,79.47},{-16.03,79.47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, skeletalMuscle.organBloodFlowSignals)  annotation (Line(
            points={{-90,90},{52,90},{52,-38.53},{-12.03,-38.53}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(organBloodFlowSignals, respiratoryMuscle.organBloodFlowSignals) annotation (
            Line(
            points={{-90,90},{52,90},{52,-62.53},{41.97,-62.53}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(bone.BloodFlow, organBloodFlowSignals.bone_BloodFlow) 
          annotation (Line(
            points={{-15.9,74.4},{52,74.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(brain.BloodFlow, organBloodFlowSignals.brain_BloodFlow) 
          annotation (Line(
            points={{40.1,56.4},{52,56.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fat.BloodFlow, organBloodFlowSignals.fat_BloodFlow) annotation (
           Line(
            points={{-15.9,36.4},{52,36.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(kidney.BloodFlow, organBloodFlowSignals.kidney_BloodFlow) 
          annotation (Line(
            points={{21.9,18.4},{14,18.4},{14,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(skin.BloodFlow, organBloodFlowSignals.skin_BloodFlow) 
          annotation (Line(
            points={{42.4,-24.4},{52,-24.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(skeletalMuscle.BloodFlow, organBloodFlowSignals.skeletalMuscle_BloodFlow) 
          annotation (Line(
            points={{-11.9,-43.6},{52,-43.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(respiratoryMuscle.BloodFlow, organBloodFlowSignals.respiratoryMuscle_BloodFlow) 
          annotation (Line(
            points={{42.1,-67.6},{52,-67.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(otherTissue.BloodFlow, organBloodFlowSignals.otherTissue_BloodFlow) 
          annotation (Line(
            points={{-13.9,-87.6},{52,-87.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));

          connect(bone.BloodFlow, organBloodFlowSignals.Bone_BloodFlow) 
          annotation (Line(
            points={{-15.9,74.4},{52,74.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(brain.BloodFlow, organBloodFlowSignals.Brain_BloodFlow) 
          annotation (Line(
            points={{40.1,56.4},{52,56.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(fat.BloodFlow, organBloodFlowSignals.Fat_BloodFlow) annotation (
           Line(
            points={{-15.9,36.4},{52,36.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(kidney.BloodFlow, organBloodFlowSignals.Kidney_BloodFlow) 
          annotation (Line(
            points={{21.9,18.4},{14,18.4},{14,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(skin.BloodFlow, organBloodFlowSignals.Skin_BloodFlow) 
          annotation (Line(
            points={{42.4,-24.4},{52,-24.4},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(skeletalMuscle.BloodFlow, organBloodFlowSignals.SkeletalMuscle_BloodFlow) 
          annotation (Line(
            points={{-11.9,-43.6},{52,-43.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(respiratoryMuscle.BloodFlow, organBloodFlowSignals.RespiratoryMuscle_BloodFlow) 
          annotation (Line(
            points={{42.1,-67.6},{52,-67.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(otherTissue.BloodFlow, organBloodFlowSignals.OtherTissue_BloodFlow) 
          annotation (Line(
            points={{-13.9,-87.6},{52,-87.6},{52,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(kidney.q_in,flowMeasure.q_out) annotation (Line(
            points={{44,21},{76,21},{76,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(kidney.q_out, q_in) annotation (Line(
            points={{18,21},{-76,21},{-76,0},{-100,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));

        connect(splanchnicCirculation.q_in, flowMeasure.q_out) annotation (Line(
            points={{-8,1},{76,1},{76,0},{82,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(splanchnicCirculation.q_out, q_in) annotation (Line(
            points={{-46,1},{-76,1},{-76,0},{-100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(splanchnicCirculation.busConnector, organBloodFlowSignals) 
          annotation (Line(
            points={{-46,10.5},{-90,10.5},{-90,90}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        annotation (
          Documentation(info="<HTML>
<PRE>
Blood resistance in peripheral organs except hepatic artery, gastro interstitional tract, vena porte, liver and liver veins.
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),             graphics), Icon(graphics={Text(
                extent={{-120,-108},{80,-80}},
                lineColor={0,0,255},
                textString="%name"), Bitmap(extent={{-66,98},{38,-80}},
                  fileName="icons/krevniReciste.png")}),
                    Diagram(graphics));
      end PeripheralFlow;

      model VasaRecta

        Library.PressureFlow.ResistorWithCond resistorWithCond 
          annotation (Placement(transformation(extent={{2,0},{22,20}})));
        Library.Factors.SplineValue OsmOnConductance(data={{600,1.4,0},{1100,1.0,-0.0006},
              {2000,0.8,0}}) 
          annotation (Placement(transformation(extent={{2,38},{22,58}})));
        Library.Factors.SplineValue A2OnConductance(data={{0.0,1.3,0},{1.3,1.0,-0.6},{
              2.0,0.5,0}}) 
          annotation (Placement(transformation(extent={{2,26},{22,46}})));
        Library.Factors.SplineValue SympsOnConductance(data={{0.0,1.1,0},{1.0,1.0,-0.13},
              {1.4,0.6,0}}) 
          annotation (Placement(transformation(extent={{2,18},{22,38}})));
        Library.Blocks.CondConstant condConstant(k=0.27) 
          annotation (Placement(transformation(extent={{-4,62},{8,74}})));
        Library.Interfaces.BusConnector busConnector
          "signals of organ bood flow resistence"                                          annotation (Placement(
              transformation(extent={{-96,64},{-74,86}}),  iconTransformation(
                extent={{-100,60},{-60,100}})));
        Library.PressureFlow.FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{32,20},{52,0}})));
        Library.PressureFlow.InputPump inputPump 
          annotation (Placement(transformation(extent={{-54,-48},{-34,-28}})));
        Modelica.Blocks.Math.Add Osm 
          annotation (Placement(transformation(extent={{-14,44},{-6,52}})));
        Library.Interfaces.RealInput_ ArcuateArtery_Pressure annotation (
            Placement(transformation(extent={{-80,20},{-68,32}}),
              iconTransformation(extent={{-100,-20},{-60,20}})));
        Library.Factors.SplineValue NephroneADHOnConductance(
          data={{0.0,1.4,0},{0.3,1.0,-0.4},{1.0,0.9,0}}) 
          annotation (Placement(transformation(extent={{2,46},{22,66}})));
        Library.PressureFlow.InputPressurePump arcuateArtery 
          annotation (Placement(transformation(extent={{-74,0},{-54,20}})));
        Library.PressureFlow.OutputPressurePump veins 
          annotation (Placement(transformation(extent={{54,0},{74,20}})));
      equation
        connect(SympsOnConductance.y, resistorWithCond.cond) annotation (Line(
            points={{12,26},{12,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(A2OnConductance.y, SympsOnConductance.yBase) annotation (Line(
            points={{12,34},{12,30}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(OsmOnConductance.y, A2OnConductance.yBase) annotation (Line(
            points={{12,46},{12,38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.A2Pool_Log10Conc, A2OnConductance.u) annotation (Line(
            points={{-85,75},{-60,75},{-60,36},{2.2,36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.Kidney_Alpha_NA, SympsOnConductance.u) annotation (Line(
            points={{-85,75},{-60,75},{-60,28},{2.2,28}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(resistorWithCond.q_out, flowMeasure.q_in) annotation (Line(
            points={{22,10},{36,10}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, busConnector.VasaRecta_Outflow) annotation (
            Line(
            points={{42,4.6},{42,-10},{84,-10},{84,75},{-85,75}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-12},{6,-12}}));
        connect(inputPump.q_out, flowMeasure.q_in) annotation (Line(
            points={{-34,-37},{26,-37},{26,10},{36,10}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(Osm.y, OsmOnConductance.u) annotation (Line(
            points={{-5.6,48},{2.2,48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.MedullaNa_Osmolarity, Osm.u1) annotation (Line(
            points={{-85,75},{-60,75},{-60,50.4},{-14.8,50.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.MedullaUrea_Osmolarity, Osm.u2) annotation (Line(
            points={{-85,75},{-60,75},{-60,45.6},{-14.8,45.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.CD_H2O_Reab, inputPump.desiredFlow) annotation (
            Line(
            points={{-85,75},{-76,75},{-76,-22},{-44,-22},{-44,-32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(NephroneADHOnConductance.y, OsmOnConductance.yBase) annotation (
           Line(
            points={{12,54},{12,50}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(condConstant.y, NephroneADHOnConductance.yBase) annotation (
            Line(
            points={{8.6,68},{12,68},{12,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.NephronADH, NephroneADHOnConductance.u) 
          annotation (Line(
            points={{-85,75},{-60,75},{-60,56},{2.2,56}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(arcuateArtery.p_out, resistorWithCond.q_in) annotation (Line(
            points={{-56,10},{2,10}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(ArcuateArtery_Pressure, arcuateArtery.desiredPressure) 
          annotation (Line(
            points={{-74,26},{-64,26},{-64,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(veins.desiredPressure, busConnector.SystemicVeins_Pressure) 
          annotation (Line(
            points={{64,16},{64,75},{-85,75}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(flowMeasure.q_out, veins.p_in) annotation (Line(
            points={{48,10},{56,10}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                               graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2010</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
      end VasaRecta;

      model LeftHeart
        extends Heart;//(PO2Effect(curve(iFrom(start=2))));
        extends Library.Interfaces.Tissues.LeftHeart;
      equation
        connect(organBloodFlowSignals.LeftHeart_Function_Failed, Vasculature.Failed) 
          annotation (Line(
            points={{-90,90},{6,90},{6,44},{60.2,44}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.LeftHeart_PO2, PO2Effect.u) annotation (
            Line(
            points={{-90,90},{6,90},{6,30},{60.2,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.LeftHeart_O2Need, MetabolismEffect.u) 
          annotation (Line(
            points={{-90,90},{6,90},{6,20},{60.2,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.LeftHeart_PO2, Vasculature.u) annotation (
           Line(
            points={{-90,90},{6,90},{6,38},{60.2,38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        annotation (Diagram(graphics));
      end LeftHeart;

      model RightHeart
        extends Heart(SmallVessel(q_in(pressure(start=44))));
        extends Library.Interfaces.Tissues.RightHeart;
      equation
        connect(organBloodFlowSignals.RightHeart_PO2, Vasculature.u) 
          annotation (Line(
            points={{-90,90},{6,90},{6,38},{60.2,38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.RightHeart_PO2, PO2Effect.u) annotation (
            Line(
            points={{-90,90},{6,90},{6,30},{60.2,30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.RightHeart_O2Need, MetabolismEffect.u) 
          annotation (Line(
            points={{-90,90},{6,90},{6,20},{60.2,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(organBloodFlowSignals.RightHeart_Function_Failed, Vasculature.Failed) 
          annotation (Line(
            points={{-90,90},{6,90},{6,44},{60.2,44}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        annotation (Diagram(graphics));
      end RightHeart;

      model SplanchnicCirculation
        extends Library.PressureFlow.ResistorBase2;

        Library.Interfaces.BusConnector busConnector
          "signals of organ bood flow resistence"                                          annotation (Placement(
              transformation(extent={{56,-56},{76,-36}}),  iconTransformation(
                extent={{-108,42},{-92,58}})));

        VascularCompartments.VascularElasticBloodCompartment portalVein(
            initialVol=1009.99,
          stateName="SplanchnicVeins.Vol") 
          annotation (Placement(transformation(extent={{-16,-10},{5,11}})));
        OrganFlow.GITract GITract 
          annotation (Placement(transformation(extent={{34,-12},{58,12}})));
        Library.Blocks.PressureConstant const9(k=0) 
          annotation (Placement(transformation(extent={{3,15},{-4,22}})));
        Library.Blocks.ComplianceConstant const10(k=62.5) 
          annotation (Placement(transformation(extent={{12,10},{4,18}})));
        Library.Blocks.VolumeConstant const11(k=500) 
          annotation (Placement(transformation(extent={{-27,11},{-19,19}})));
        Liver liver 
          annotation (Placement(transformation(extent={{-56,34},{-32,58}})));
      equation

        connect(portalVein.referencePoint,GITract. q_in)     annotation (Line(
            points={{-5.5,0.5},{20,0},{34,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GITract.organBloodFlowSignals,busConnector)           annotation (Line(
            points={{54.28,2.28},{66,2.28},{66,-46}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(portalVein.ExternalPressure,const9. y)     annotation (Line(
            points={{-5.5,11},{-5.5,18.5},{-4.35,18.5}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(portalVein.Compliance,const10. y)     annotation (Line(
            points={{0.8,11},{0.8,14},{3.6,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(portalVein.V0,const11. y)     annotation (Line(
            points={{-11.8,11},{-11.8,15},{-18.6,15}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(GITract.BloodFlow, busConnector.GITract_BloodFlow) annotation (
            Line(
            points={{54.4,-2.4},{66,-2.4},{66,-46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(portalVein.Pressure, busConnector.SplanchnicVeins_Pressure) 
          annotation (Line(
            points={{-1.3,-10},{-1.3,-46},{66,-46}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{3,-3},{3,-3}}));
        connect(const11.y, busConnector.PortalVein_V0) annotation (Line(
            points={{-18.6,15},{-14,15},{-14,-46},{66,-46}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(portalVein.Vol, busConnector.SplanchnicCirculation_DeoxygenatedBloodVolume) 
          annotation (Line(
            points={{-9.7,-10},{-9.7,-46},{66,-46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(q_in, GITract.q_out) annotation (Line(
            points={{100,0},{58,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(q_in, liver.q_out) annotation (Line(
            points={{100,0},{84,0},{84,46},{-32,46}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(portalVein.referencePoint, liver.portalVein) annotation (Line(
            points={{-5.5,0.5},{-41.36,0.5},{-41.36,40.24}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(liver.q_in, q_out) annotation (Line(
            points={{-56,46},{-78,46},{-78,0},{-100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector, liver.organBloodFlowSignals) annotation (Line(
            points={{66,-46},{66,48.28},{-35.72,48.28}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics),       Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/perfusion2.png"),
              Text(
                extent={{-108,-48},{106,-70}},
                lineColor={0,0,255},
                textString="%name"),
              Bitmap(extent={{-34,46},{34,-48}}, fileName=
                    "icons/traviciTrakt03.png")}),
        Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>january 2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>", info="<html>
<p>Blood volume (and pressures) distribution between </p>
<p><ul>
<li>pulmonary arteries, </li>
<li>pulmonary veins and </li>
<li>pulmonary capilaries.</li>
</ul></p>
<p>Blood flow through ventilated alveoli.</p>
</html>"));
      end SplanchnicCirculation;

      model Liver
        extends BaseBadDirection;
        extends Library.Interfaces.Tissues.Liver;
        Library.PressureFlow.PositivePressureFlow portalVein 
                              annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{-14,-10},{6,10}}),    iconTransformation(extent={{12,-58},
                  {32,-38}})));
        Library.PressureFlow.ResistorWithCondParam splachnicVeinsConductance(
            cond=178*5.4826929484432128619694687329025) 
          annotation (Placement(transformation(extent={{-70,-8},{-54,8}})));
        Library.PressureFlow.FlowMeasure flowMeasure1 annotation (Placement(
              transformation(
              extent={{-10,10},{10,-10}},
              rotation=180,
              origin={-82,0})));
        Library.PressureFlow.FlowMeasure flowMeasure2 annotation (Placement(
              transformation(
              extent={{-10,10},{10,-10}},
              rotation=180,
              origin={-26,0})));
        Library.PressureFlow.ResistorWithCondParam HepaticArtyConductance(cond=
              2.8) 
          annotation (Placement(transformation(extent={{-4,34},{16,54}})));
        Library.PressureFlow.FlowMeasure flowMeasure3 
                                                     annotation (Placement(
              transformation(
              extent={{-10,10},{10,-10}},
              rotation=180,
              origin={42,44})));
      equation
        connect(HepaticArtyConductance.q_in,splachnicVeinsConductance. q_out) 
          annotation (Line(
            points={{-4,44},{-51,44},{-51,0},{-54,0}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure1.q_in,splachnicVeinsConductance. q_in) annotation (
            Line(
            points={{-76,-7.34788e-016},{-74,-7.34788e-016},{-74,0},{-70,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure1.actualFlow, organBloodFlowSignals.Liver_BloodFlow) 
          annotation (Line(
            points={{-82,5.4},{-82,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure2.actualFlow, organBloodFlowSignals.PortalVein_BloodFlow) 
          annotation (Line(
            points={{-26,5.4},{-26,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(splachnicVeinsConductance.q_out,flowMeasure2. q_out) annotation (Line(
            points={{-54,0},{-44,0},{-44,7.34788e-016},{-32,7.34788e-016}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure3.q_out, HepaticArtyConductance.q_out) 
                                                                 annotation (
            Line(
            points={{36,44},{16,44}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure3.actualFlow, organBloodFlowSignals.HepaticArty_BloodFlow) 
          annotation (Line(
            points={{42,49.4},{42,90},{-90,90}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(flowMeasure1.q_out, q_in) annotation (Line(
            points={{-88,7.34788e-016},{-94,7.34788e-016},{-94,0},{-100,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(portalVein, flowMeasure2.q_in) annotation (Line(
            points={{-4,0},{-12,0},{-12,-7.34788e-016},{-20,-7.34788e-016}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, flowMeasure3.q_in) annotation (Line(
            points={{82,0},{72,0},{72,44},{48,44}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}}),
                               graphics), Diagram(graphics));
      end Liver;
    end OrganFlow;

    package BlooodVolume "Red Cells and Blood Properties"

      model RedCells2

        Library.PressureFlow.OutputPump RBCClearance 
          annotation (Placement(transformation(extent={{48,-10},{68,10}})));
        Library.PressureFlow.VolumeCompartement RBCVolume(initialVolume=2373.2,
            stateName="RBCVol.Vol") 
          annotation (Placement(transformation(extent={{0,-10},{20,10}})));
        Library.PressureFlow.InputPump transfusion 
          annotation (Placement(transformation(extent={{-48,26},{-28,46}})));
        Library.PressureFlow.OutputPump hemorrhage 
          annotation (Placement(transformation(extent={{48,26},{68,46}})));
        Library.Factors.SplineDelayByDay EPOEffect(
          Tau=3,
          data={{0.0,0.0,0},{1.3,1.0,1.0},{4.0,4.0,0}}) 
          annotation (Placement(transformation(extent={{-76,4},{-56,24}})));
        Library.Blocks.FlowConstant RBCBaseSecretionRate(    k=0.013889) 
          annotation (Placement(transformation(extent={{-80,20},{-70,30}})));
        Library.PressureFlow.InputPump RBCProduction 
          annotation (Placement(transformation(extent={{-76,-10},{-56,10}})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-98,54},{-78,74}})));
        Modelica.Blocks.Math.Gain gain(k=1/120*(1/1440)) 
          annotation (Placement(transformation(extent={{34,-14},{42,-6}})));
        Modelica.Blocks.Math.Gain H2O(k=1 - 0.34) 
          annotation (Placement(transformation(extent={{36,-44},{44,-36}})));
      equation
        connect(busConnector.Transfusion_RBCRate,transfusion. desiredFlow) 
                                                            annotation (Line(
            points={{-88,64},{-86,64},{-86,58},{-38,58},{-38,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Hemorrhage_RBCRate, hemorrhage.desiredFlow) 
                                                            annotation (Line(
            points={{-88,64},{-86,64},{-86,58},{58,58},{58,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RBCBaseSecretionRate.y,EPOEffect. yBase)              annotation (
            Line(
            points={{-69.5,25},{-66,25},{-66,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(EPOEffect.u,busConnector. EPOPool_Log10Conc)              annotation (Line(
            points={{-75.8,14},{-86.9,14},{-86.9,64},{-88,64}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RBCProduction.desiredFlow, EPOEffect.y) 
                                                 annotation (Line(
            points={{-66,6},{-66,12}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RBCProduction.q_out, RBCVolume.con) 
                                                   annotation (Line(
            points={{-56,1},{-30.6,1},{-30.6,0},{0,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(transfusion.q_out, RBCVolume.con) annotation (Line(
            points={{-28,37},{-14,37},{-14,0},{0,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(RBCVolume.con, hemorrhage.q_in)  annotation (Line(
            points={{0,0},{26,0},{26,36},{48,36}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(RBCVolume.con, RBCClearance.q_in) annotation (Line(
            points={{0,0},{48,0}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(RBCVolume.Volume, gain.u) annotation (Line(
            points={{11.2,-10},{33.2,-10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.y, RBCClearance.desiredFlow) annotation (Line(
            points={{42.4,-10},{44,-10},{44,10},{58,10},{58,6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RBCVolume.Volume,busConnector. RBCVol_Vol) annotation (Line(
            points={{11.2,-10},{11.2,-24},{100,-24},{100,58},{-86,58},{-86,64},
                {-88,64}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-5},{6,-5}}));
        connect(RBCVolume.Volume, H2O.u) annotation (Line(
            points={{11.2,-10},{10,-10},{10,-40},{35.2,-40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(H2O.y, busConnector.RBCH2O_Vol) annotation (Line(
            points={{44.4,-40},{100,-40},{100,58},{-88,58},{-88,64}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));

        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-70,80},{70,-40}}, fileName=
                    "icons/krvinka.png"), Text(
                extent={{-136,-46},{148,-68}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
      end RedCells2;

      model BloodProperties

        Modelica.Blocks.Math.Add BloodVolume 
          annotation (Placement(transformation(extent={{14,-70},{22,-62}})));
        Modelica.Blocks.Math.Min CollapsedEffect 
          annotation (Placement(transformation(extent={{50,-84},{58,-76}})));
        Modelica.Blocks.Math.Division division 
          annotation (Placement(transformation(extent={{38,-78},{46,-70}})));
        Library.Blocks.Constant const(
                               k=1) 
          annotation (Placement(transformation(extent={{38,-94},{46,-86}})));
        Modelica.Blocks.Math.Division HtcFract 
          annotation (Placement(transformation(extent={{32,-46},{40,-38}})));
        Library.Curves.Curve HtcOnVisc(
          x={0,.44,.8},
          y={.5,1,5},
          slope={.8,3,30}) 
          annotation (Placement(transformation(extent={{46,-52},{66,-32}})));
        Modelica.Blocks.Math.Division division1 
          annotation (Placement(transformation(extent={{78,-40},{86,-32}})));
        Library.Blocks.Constant const1(
                               k=1) 
          annotation (Placement(transformation(extent={{64,-30},{72,-22}})));
        Library.Blocks.Constant const2(
                               k=1) 
          annotation (Placement(transformation(extent={{28,-14},{36,-6}})));
        Modelica.Blocks.Math.Feedback PVCrit 
          annotation (Placement(transformation(extent={{38,-14},{46,-6}})));
        Library.Blocks.Constant Constant4(k=8.4) 
          annotation (Placement(transformation(extent={{-6,8},{-14,16}})));
        Library.Blocks.Constant Constant1(k=0.44) 
          annotation (Placement(transformation(extent={{-66,-8},{-58,0}})));
        Library.Factors.SimpleMultiply hematocritEffect 
          annotation (Placement(transformation(extent={{-30,-10},{-10,10}})));
        Modelica.Blocks.Math.Division division2 
          annotation (Placement(transformation(extent={{-50,-4},{-42,4}})));
        Library.Blocks.Constant Constant5(k=5.4) 
          annotation (Placement(transformation(extent={{26,90},{18,98}})));
        Library.Blocks.Constant Constant6(k=0.005) 
          annotation (Placement(transformation(extent={{26,82},{18,90}})));
        Library.Blocks.Constant Constant7(k=0.005) 
          annotation (Placement(transformation(extent={{26,74},{18,82}})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-104,58},{-84,78}})));
        Modelica.Blocks.Math.Division division3(y(unit="mmol/l")) 
          annotation (Placement(transformation(extent={{-4,-4},{4,4}},
              rotation=0,
              origin={10,2})));
        Modelica.Blocks.Math.Feedback feedback 
          annotation (Placement(transformation(extent={{-20,32},{-40,52}})));
        Modelica.Blocks.Math.Add BloodVolume1 
          annotation (Placement(transformation(extent={{-46,24},{-38,32}})));
        Modelica.Blocks.Math.Product product 
          annotation (Placement(transformation(extent={{66,2},{74,10}})));
        Modelica.Blocks.Math.Sum V0(nin=5) 
          annotation (Placement(transformation(extent={{-32,-90},{-24,-82}})));
        Modelica.Blocks.Math.Sum ArtysVol(nin=4) "volume of oxygenated blood" 
          annotation (Placement(transformation(extent={{-28,-34},{-20,-26}})));
        Modelica.Blocks.Math.Sum VeinsVol(nin=5) "volume of deoxygenated blood"
          annotation (Placement(transformation(extent={{-28,-54},{-20,-46}})));
      equation
        connect(busConnector.PlasmaVol_Vol, BloodVolume.u2) 
                                                    annotation (Line(
            points={{-94,68},{-94,-68.4},{13.2,-68.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.RBCVol_Vol, BloodVolume.u1) 
                                                  annotation (Line(
            points={{-94,68},{-94,-63.6},{13.2,-63.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(BloodVolume.y, busConnector.BloobVolume) 
                                                 annotation (Line(
            points={{22.4,-66},{90,-66},{90,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(division.y, CollapsedEffect.u1) 
                                    annotation (Line(
            points={{46.4,-74},{47.8,-74},{47.8,-77.6},{49.2,-77.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Vesseles_V0,division. u2) annotation (Line(
            points={{-94,68},{-94,-74},{37.2,-74},{37.2,-76.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(BloodVolume.y, division.u1) 
                                    annotation (Line(
            points={{22.4,-66},{28,-66},{28,-71.6},{37.2,-71.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const.y, CollapsedEffect.u2) 
                                 annotation (Line(
            points={{46.4,-90},{48,-90},{48,-82.4},{49.2,-82.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(CollapsedEffect.y, busConnector.BloodVol_CollapsedEffect) 
                                                              annotation (Line(
            points={{58.4,-80},{90,-80},{90,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.u2, BloodVolume.y) 
                                    annotation (Line(
            points={{31.2,-44.4},{28,-44.4},{28,-66},{22.4,-66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.u1,busConnector. RBCVol_Vol) annotation (Line(
            points={{31.2,-39.6},{-94,-39.6},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.y,HtcOnVisc. u) annotation (Line(
            points={{40.4,-42},{46,-42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const1.y,division1. u1) annotation (Line(
            points={{72.4,-26},{74,-26},{74,-33.6},{77.2,-33.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcOnVisc.val,division1. u2) annotation (Line(
            points={{66.2,-42},{75.5,-42},{75.5,-38.4},{77.2,-38.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(division1.y,busConnector. Viscosity_ConductanceEffect) annotation (Line(
            points={{86.4,-36},{90,-36},{90,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.y,busConnector. BloodVol_Hct) annotation (Line(
            points={{40.4,-42},{42,-42},{42,-20},{-94,-20},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const2.y,PVCrit. u1) annotation (Line(
            points={{36.4,-10},{38.8,-10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.y,PVCrit. u2) annotation (Line(
            points={{40.4,-42},{42,-42},{42,-13.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PVCrit.y,busConnector. BloodVol_PVCrit) annotation (Line(
            points={{45.6,-10},{90,-10},{90,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Constant4.y,hematocritEffect. yBase) annotation (Line(
            points={{-14.4,12},{-20,12},{-20,2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(hematocritEffect.y,busConnector. ctHb) annotation (Line(
            points={{-20,-2},{-20,-10},{-94,-10},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(division2.y,hematocritEffect. u) annotation (Line(
            points={{-41.6,0},{-29.8,0}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Constant1.y,division2. u2) annotation (Line(
            points={{-57.6,-4},{-55.2,-4},{-55.2,-2.4},{-50.8,-2.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.y,division2. u1) annotation (Line(
            points={{40.4,-42},{42,-42},{42,-20},{-72,-20},{-72,2.4},{-50.8,2.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.cDPG,Constant5. y) annotation (Line(
            points={{-94,68},{-4,68},{-4,94},{17.6,94}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.FMetHb,Constant6. y) annotation (Line(
            points={{-94,68},{-4,68},{-4,86},{17.6,86}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.FHbF,Constant7. y) annotation (Line(
            points={{-94,68},{-4,68},{-4,78},{17.6,78}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Constant4.y, division3.u1) annotation (Line(
            points={{-14.4,12},{-20,12},{-20,4},{5.2,4},{5.2,4.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(HtcFract.y, division3.u2) annotation (Line(
            points={{40.4,-42},{42,-42},{42,-20},{-2,-20},{-2,-0.4},{5.2,-0.4}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(division3.y, busConnector.ctHb_ery) annotation (Line(
            points={{14.4,2},{20,2},{20,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(BloodVolume.y, feedback.u1) annotation (Line(
            points={{22.4,-66},{28,-66},{28,42},{-22,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.ArtysVol, BloodVolume1.u1) annotation (Line(
            points={{-94,68},{-94,30},{-46.8,30},{-46.8,30.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.VeinsVol, BloodVolume1.u2) annotation (Line(
            points={{-94,68},{-94,26},{-46.8,26},{-46.8,25.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(BloodVolume1.y, feedback.u2) annotation (Line(
            points={{-37.6,28},{-30,28},{-30,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.BloodVolume_change, feedback.y) annotation (Line(
            points={{-94,68},{-94,42},{-39,42}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(PVCrit.y, product.u2) annotation (Line(
            points={{45.6,-10},{54,-10},{54,3.6},{65.2,3.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.PortalVein_BloodFlow, product.u1) annotation (Line(
            points={{-94,68},{54,68},{54,8.4},{65.2,8.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(product.y, busConnector.PortalVein_PlasmaFlow) annotation (Line(
            points={{74.4,6},{90,6},{90,68},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(V0.y, busConnector.Vesseles_V0) annotation (Line(
            points={{-23.6,-86},{0,-86},{0,-78},{-94,-78},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(VeinsVol.y, busConnector.VeinsVol) annotation (Line(
            points={{-19.6,-50},{0,-50},{0,-42},{-94,-42},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(ArtysVol.y, busConnector.ArtysVol) annotation (Line(
            points={{-19.6,-30},{0,-30},{0,-22},{-94,-22},{-94,68}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(busConnector.PulmonaryCirculation_OxygenatedBloodVolume, ArtysVol.u[1]) 
          annotation (Line(
            points={{-94,68},{-94,-26},{-30,-26},{-30,-30.6},{-28.8,-30.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,6},{-6,6}}));
        connect(busConnector.Heart_OxygenatedBloodVolume, ArtysVol.u[2]) annotation (
            Line(
            points={{-94,68},{-94,-30.2},{-28.8,-30.2}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,2},{-6,2}}));
        connect(busConnector.SystemicArtys_OxygenatedBloodVolume, ArtysVol.u[3]) 
          annotation (Line(
            points={{-94,68},{-94,-34},{-30,-34},{-30,-29.8},{-28.8,-29.8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-2},{-6,-2}}));
        connect(busConnector.LegArtys_OxygenatedBloodVolume, ArtysVol.u[4]) 
          annotation (Line(
            points={{-94,68},{-94,-38},{-30,-38},{-30,-29.4},{-28.8,-29.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-6},{-6,-6}}));

        connect(busConnector.PulmonaryCirculation_DeoxygenatedBloodVolume, VeinsVol.u[1]) 
          annotation (Line(
            points={{-94,68},{-94,-46},{-30,-46},{-30,-50.64},{-28.8,-50.64}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,6},{-6,6}}));
        connect(busConnector.Heart_DeoxygenatedBloodVolume, VeinsVol.u[2]) annotation (
            Line(
            points={{-94,68},{-94,-50.32},{-28.8,-50.32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,2},{-6,2}}));
        connect(busConnector.LegVeins_DeoxygenatedBloodVolume, VeinsVol.u[3]) 
          annotation (Line(
            points={{-94,68},{-94,-54},{-30,-54},{-30,-50},{-28.8,-50}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-2},{-6,-2}}));
        connect(busConnector.SplanchnicCirculation_DeoxygenatedBloodVolume, VeinsVol.u[4]) 
          annotation (Line(
            points={{-94,68},{-94,-58},{-30,-58},{-30,-49.68},{-28.8,-49.68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-6},{-6,-6}}));
        connect(busConnector.SystemicVeins_DeoxygenatedBloodVolume, VeinsVol.u[5]) 
          annotation (Line(
            points={{-94,68},{-94,-62},{-30,-62},{-30,-49.36},{-28.8,-49.36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-10},{-6,-10}}));

        connect(busConnector.PulmonaryCirculation_V0, V0.u[1]) 
          annotation (Line(
            points={{-94,68},{-94,-82},{-34,-82},{-34,-86.64},{-32.8,-86.64}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,6},{-6,6}}));
        connect(busConnector.Heart_V0, V0.u[2]) annotation (
            Line(
            points={{-94,68},{-94,-86.32},{-32.8,-86.32}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,2},{-6,2}}));
        connect(busConnector.SystemicArtys_V0, V0.u[3]) 
          annotation (Line(
            points={{-94,68},{-94,-90},{-34,-90},{-34,-86},{-32.8,-86}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-2},{-6,-2}}));
        connect(busConnector.PortalVein_V0, V0.u[4]) annotation (Line(
            points={{-94,68},{-94,-94},{-34,-94},{-34,-86},{-32.8,-85.68}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-6},{-6,-6}}));
        connect(busConnector.SystemicVeins_V0, V0.u[5]) annotation (Line(
            points={{-94,68},{-94,-98},{-34,-98},{-34,-86},{-32.8,-85.36}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-10},{-6,-10}}));

        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                               graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-102,100},{102,-100}}, fileName=
                    "icons/bloodProperties.jpg"), Text(
                extent={{-142,-108},{142,-130}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(info="<html>
<p>BloodVolume .. whole blood volume</p>
<p>BloodVol_Hct .. haematocrit</p>
<p>BloodVol_PVCrit .. 1-BloodVol_Hct, plamacrit</p>
<p>ctHb_ery .. concentration of heme group in erythrocytes = 4 * MCHC (mean corpuscular hemoglobin concentration)</p>
<p>ctHb .. concentration of heme group in whole blood</p>
<p>FHbF .. featal Hb fraction from all Hb</p>
<p>FMetHb .. metha-Hb fraction from all Hb</p>
<p>cDPG .. concentration of 2,3-diphosphoglycerate in erythrocytes</p>
<p>Viscosity_ConductanceEffect .. multiply coeficient of blood resistance through vesseles</p>
<p>BloodVol_CollapsedEffect .. effect of small blood volume on systemic veins collasping</p>
<p>Vesseles_V0 .. volume of no-pressure generating blood in all vessels.</p>
<p>ArtysVol .. volume of all oxigenated blood</p>
<p>VeinsVol .. volume of all deoxigenated blood</p>
</html>", revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
      end BloodProperties;
    end BlooodVolume;

    model CVS_
      extends HumMod.Library.Utilities.DynamicState(stateName="BloodVol.Vol", initType=Library.Utilities.Init.NoInit);

      VascularCompartments.Heart heart 
        annotation (Placement(transformation(extent={{0,-6},{26,20}})));
      VascularCompartments.PulmonaryCirculation pulmonaryCirculation 
        annotation (Placement(transformation(extent={{0,46},{28,74}})));
      VascularCompartments.SystemicCirculation systemicCirculation 
        annotation (Placement(transformation(extent={{-2,-68},{32,-34}})));
      Library.Interfaces.BusConnector busConnector
        "signals of organ bood flow resistence"                                          annotation (Placement(
            transformation(extent={{-38,-8},{-16,14}}),  iconTransformation(
              extent={{-100,60},{-60,100}})));
      BlooodVolume.RedCells2 redCells 
        annotation (Placement(transformation(extent={{-34,26},{-54,46}})));
      BlooodVolume.BloodProperties bloodProperties 
        annotation (Placement(transformation(extent={{-34,-34},{-54,-14}})));

    initial equation
      if STEADY then
         bloodProperties.feedback.y = 0;
      end if;

    equation
      connect(busConnector, pulmonaryCirculation.busConnector) annotation (
          Line(
          points={{-27,3},{-26,3},{-26,67},{0,67}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(busConnector, systemicCirculation.busConnector) annotation (
          Line(
          points={{-27,3},{-26,3},{-26,-44.2},{-2.34,-44.2}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(busConnector, redCells.busConnector)    annotation (Line(
          points={{-27,3},{-26,3},{-26,42.4},{-35.2,42.4}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(busConnector, bloodProperties.busConnector) annotation (Line(
          points={{-27,3},{-26,3},{-26,-17.2},{-34.6,-17.2}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(busConnector, heart.busConnector) annotation (Line(
          points={{-27,3},{-25.5,3},{-25.5,3.1},{0,3.1}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(heart.fromLeftVentricle, systemicCirculation.q_in) annotation (
          Line(
          points={{16.12,3.36},{16.12,18},{32,18},{32,-51}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(systemicCirculation.q_out, heart.rightAtrium) annotation (Line(
          points={{-2,-51},{-4,-51},{-4,12.2},{5.72,12.2}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(pulmonaryCirculation.q_in, heart.fromRightVentricle) annotation (
          Line(
          points={{0,60},{-4,60},{-4,18},{8.06,18},{8.06,3.36}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(pulmonaryCirculation.q_out, heart.leftAtrium) annotation (Line(
          points={{28,60},{36,60},{36,12.46},{18.72,12.46}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));

      stateValue = bloodProperties.BloodVolume1.y;

      annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}),       graphics), Documentation(info="<html>
<p><h4><font color=\"#008000\">Cardiovascular system</font></h4></p>
<p><font style=\"color: #008000; \">Connect heart with pulmonary and systemic part. This connection contains pressures and blood flows before and after heart.</font></p>
<p><font style=\"color: #008000; \">Output signal variables about blood volume between lungs capilaries and systemic tissue capilaries are used in gass transport submodels like O2 or CO2.</font></p>
<p><font style=\"color: #008000; \">Output signal Vesseles_V0 is used for systemic veins collapsing in massive blood lost.</font></p>
<p><font style=\"color: #008000; \">The blood properties convert known variables values to another meanings (haematocrit, viscosity, ..).</font></p>
<p><font style=\"color: #008000; \">Red cells contains volume, loss and gains all red blood cells.</font></p>
</html>", revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Design:</td>
<td>Zuzana Rubaninska</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"),
        Icon(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},{
              100,100}}), graphics={Bitmap(extent={{-100,100},{100,-100}},
              fileName="icons/csv.png"), Text(
            extent={{-116,-96},{126,-130}},
            lineColor={0,0,255},
            textString="%name")}));
    end CVS_;
  annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
  end Blood;

  package Water "Body Water"

    package Hydrostatics
      "Hydrostatic pressure difference in upper, midle and lower torso"

      type Posture = enumeration(
          Lying,
          Sitting,
          Standing,
          Tilting);

      model TorsoHydrostatics
       extends HumMod.Library.Interfaces.BaseModel;


         parameter Real Alpha =   0.1667
          "torso: capilary pressure coeficient between artery and vein pressure";
         parameter Real Gravity_Gz = 1 "gravity constant / 10";

         parameter Real TorsoCM(final quantity="Height", final unit="cm")
          "torso: center of gravity - height";

         parameter Real artyFractGz[Posture]
          "torso: arty hydrostatic effects for posturing";
         parameter Real veinFractGz[Posture]
          "torso: vein hydrostatic effects for posturing";

         Real ArtyFractGz;
         Real VeinFractGz;
         Real TorsoArtyGradient(final quantity="Pressure", final unit="mmHg");
         Real TorsoVeinGradient(final quantity="Pressure", final unit="mmHg");
         Real Arty(final quantity="Pressure", final unit="mmHg");
         Real Vein(final quantity="Pressure", final unit="mmHg");

        HumMod.Library.Interfaces.RealInput Pump_Effect(final unit="1")
          "xNormal"                              annotation (Placement(transformation(extent={{
                  -120,-100},{-80,-60}}), iconTransformation(extent={{-120,-100},{-80,
                  -60}})));
        HumMod.Library.Interfaces.RealInput fromPressure(final quantity=
              "Pressure", final unit="mmHg")
          "torso: systemic arteries pressure"                             annotation (Placement(transformation(extent={{
                  -120,0},{-80,40}}), iconTransformation(extent={{-120,0},{-80,40}})));
        HumMod.Library.Interfaces.RealInput toPressure(final quantity=
              "Pressure", final unit="mmHg") "torso: systemic veins pressure" 
                                                                             annotation (Placement(transformation(extent={{-120,
                  -40},{-80,0}}),       iconTransformation(extent={{-120,-40},{-80,0}})));
        PostureInput Status_Posture "Lying, Sitting, Standing or Tilting"           annotation (Placement(transformation(
                extent={{-120,60},{-80,100}}), iconTransformation(extent={{-120,60},{
                  -80,100}})));
        HumMod.Library.Interfaces.RealOutput_ Capy(final quantity="Pressure",
            final unit="mmHg") "torso: average capilaries pressure"         annotation (Placement(transformation(extent={{80,-20},
                  {120,20}}), iconTransformation(extent={{80,-20},{120,20}})));
      equation
         ArtyFractGz = artyFractGz[Status_Posture];
         VeinFractGz = veinFractGz[Status_Posture]; //in legs veins are valves

         TorsoArtyGradient  =  TorsoCM * Gravity_Gz * ArtyFractGz;  //Hydrostatic pressure: P=ro.g.h
         //cm *  .01 m/cm * 9.81 m/s * 1060 kg/m3 * 1/133.322 mmHg/Pa
         //= height[cm] * .77996129671022036873134216408395 [mmHg]
         //Blood density = 1060 kg/m3: Cutnell, John & Johnson, Kenneth. Physics, Fourth Edition. Wiley, 1998: 308.

         //bloodDensity = 1060 kg/m3
         //gravityAcceleration = 9.81 m/s2
         //Pa2mmHg = 1/133.322 mmHg/Pa
         //cm2m = .01 m/cm
         //averageHydrostaticHeight = TorsoCM * postureCoef[Status_Posture]
         //hydrostaticPressure = averageHydrostaticHeight * cm2m * bloodDensity * gravityAcceleration * Pa2mmHg
         //                    = .77996 * averageHydrostaticHeight

         TorsoVeinGradient  =  TorsoCM * Gravity_Gz * VeinFractGz;

         Arty  = max( (fromPressure + TorsoArtyGradient), 0);
         Vein  = max( toPressure + ( TorsoVeinGradient * Pump_Effect));  //only lower torso has LegMusclePump_Effect
         Capy  = max( ( Alpha * Arty)   + ( 1.0 - Alpha)  * Vein, 0);
       annotation (
      Documentation(info="<HTML>
<PRE>
QHP 2008 / Hydrostatics
 
Created : 18-Jun-05
Last Modified : 18-Jun-05
Author : Tom Coleman
Copyright Status : In Public Domain
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}), graphics));
      end TorsoHydrostatics;

      model Hydrostatics
        //extends QHP.Library.Interfaces.BaseModel;

         parameter Real TiltTable_Degrees( final quantity="Angle", final unit="Deg")= 0;

        TorsoHydrostatics UpperTorsoHydrostatics(                         veinFractGz=
             {0,1,1,sin(Modelica.SIunits.Conversions.from_deg(TiltTable_Degrees))},
          TorsoCM=-10,
          artyFractGz={0,1,1,sin(Modelica.SIunits.Conversions.from_deg(
              TiltTable_Degrees))}) 
          annotation (Placement(transformation(extent={{-12,34},{16,62}})));
        TorsoHydrostatics LowerTorsoHydrostatics(
          TorsoCM=50.0,
          artyFractGz={0,.7,1,sin(Modelica.SIunits.Conversions.from_deg(
              TiltTable_Degrees))},
          veinFractGz={.2,.7,1,.2 + sin(Modelica.SIunits.Conversions.from_deg(
              TiltTable_Degrees))}) 
          annotation (Placement(transformation(extent={{-12,-42},{16,-14}})));
        TorsoHydrostatics MiddleTorsoHydrostatics(
                                            artyFractGz={0,1,1,sin(
              Modelica.SIunits.Conversions.from_deg(TiltTable_Degrees))}, veinFractGz=
             {0,1,1,sin(Modelica.SIunits.Conversions.from_deg(TiltTable_Degrees))},
          TorsoCM=4) 
          annotation (Placement(transformation(extent={{-12,-4},{16,24}})));
        HumMod.Library.Blocks.Constant PumpEffect(k=1) 
          annotation (Placement(transformation(extent={{-40,32},{-36,36}})));
        HumMod.Library.Interfaces.RealInput SystemicArtys_Pressure(final
            quantity="Pressure", final unit="mmHg") 
          annotation (Placement(transformation(extent={{-44,56},{-40,60}}),
              iconTransformation(extent={{-100,-30},{-80,-10}})));
        HumMod.Library.Interfaces.RealInput RightAtrium_Pressure(final quantity
            ="Pressure", final unit="mmHg") 
          annotation (Placement(transformation(extent={{-34,66},{-30,70}}),
              iconTransformation(extent={{-100,10},{-80,30}})));
        HumMod.Library.Interfaces.RealInput LegMusclePump_Effect(final quantity
            ="Flow", final unit="ml/min") 
          annotation (Placement(transformation(extent={{-46,-42},{-38,-34}}),
              iconTransformation(extent={{-100,-90},{-80,-70}})));
        Library.Interfaces.RealInput_ Status_Posture
          "Lying, Sitting, Standing or Tilting"                                   annotation (Placement(transformation(
                extent={{-90,22},{-86,26}}), iconTransformation(extent={{-100,70},{-80,
                  90}})));
        HumMod.Library.Interfaces.RealOutput RegionalPressure_UpperCapy(final
            quantity="Pressure", final unit="mmHg") 
                                                annotation (Placement(transformation(
                extent={{42,44},{50,52}}),     iconTransformation(extent={{90,20},{
                  110,40}})));
        HumMod.Library.Interfaces.RealOutput RegionalPressure_MiddleCapy(final
            quantity="Pressure", final unit="mmHg") 
                                                annotation (Placement(transformation(
                extent={{42,6},{50,14}}),      iconTransformation(extent={{90,-20},{
                  110,0}})));
        HumMod.Library.Interfaces.RealOutput RegionalPressure_LowerCapy(final
            quantity="Pressure", final unit="mmHg") 
                                                annotation (Placement(transformation(
                extent={{42,-32},{50,-24}}),   iconTransformation(extent={{90,-60},{
                  110,-40}})));
        Real2Posture real2Posture 
          annotation (Placement(transformation(extent={{-72,20},{-64,28}})));
      equation
        connect(RightAtrium_Pressure,LowerTorsoHydrostatics. toPressure) annotation (
            Line(
            points={{-32,68},{-28,68},{-28,-30.8},{-12,-30.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RightAtrium_Pressure,MiddleTorsoHydrostatics. toPressure) annotation (
           Line(
            points={{-32,68},{-28,68},{-28,7.2},{-12,7.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(RightAtrium_Pressure,UpperTorsoHydrostatics. toPressure) annotation (
            Line(
            points={{-32,68},{-28,68},{-28,45.2},{-12,45.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PumpEffect.y,UpperTorsoHydrostatics. Pump_Effect) annotation (Line(
            points={{-35.8,34},{-26,34},{-26,36.8},{-12,36.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MiddleTorsoHydrostatics.Pump_Effect,PumpEffect. y) annotation (Line(
            points={{-12,-1.2},{-26,-1.2},{-26,34},{-35.8,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SystemicArtys_Pressure,LowerTorsoHydrostatics. fromPressure) 
          annotation (Line(
            points={{-42,58},{-30,58},{-30,-25.2},{-12,-25.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SystemicArtys_Pressure,MiddleTorsoHydrostatics. fromPressure) 
          annotation (Line(
            points={{-42,58},{-30,58},{-30,12.8},{-12,12.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SystemicArtys_Pressure,UpperTorsoHydrostatics. fromPressure) 
          annotation (Line(
            points={{-42,58},{-30,58},{-30,50.8},{-12,50.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LegMusclePump_Effect,LowerTorsoHydrostatics. Pump_Effect) annotation (
           Line(
            points={{-42,-38},{-32,-38},{-32,-39.2},{-12,-39.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(UpperTorsoHydrostatics.Capy, RegionalPressure_UpperCapy) annotation (
            Line(
            points={{16,48},{46,48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(MiddleTorsoHydrostatics.Capy, RegionalPressure_MiddleCapy) 
          annotation (Line(
            points={{16,10},{31,10},{31,10},{46,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LowerTorsoHydrostatics.Capy, RegionalPressure_LowerCapy) annotation (
            Line(
            points={{16,-28},{46,-28}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(UpperTorsoHydrostatics.Status_Posture, real2Posture.y) annotation (
            Line(
            points={{-12,59.2},{-32,59.2},{-32,24},{-63.2,24}},
            color={0,127,127},
            smooth=Smooth.None));

        connect(real2Posture.y, MiddleTorsoHydrostatics.Status_Posture) annotation (
            Line(
            points={{-63.2,24},{-32,24},{-32,21.2},{-12,21.2}},
            color={0,127,127},
            smooth=Smooth.None));
        connect(LowerTorsoHydrostatics.Status_Posture, real2Posture.y) annotation (
            Line(
            points={{-12,-16.8},{-32,-16.8},{-32,24},{-63.2,24}},
            color={0,127,127},
            smooth=Smooth.None));
        connect(real2Posture.u, Status_Posture) annotation (Line(
            points={{-72.8,24},{-88,24}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/hydrostatika.png"),
              Text(
                extent={{-100,122},{100,102}},
                lineColor={0,0,255},
                textString="%name"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0})}));
      end Hydrostatics;

      connector PostureInput = 
                            input Posture "'input Posture' as connector" 
        annotation (defaultComponentName="u",
        Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,127,127},
              fillColor={0,127,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{100,-50},{700,50}},
              lineColor={0,127,127},
              textString="%name")},
             coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
        Diagram(coordinateSystem(
              preserveAspectRatio=true, initialScale=0.2,
              extent={{-100,-100},{100,100}},
              grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,127,127},
              fillColor={0,127,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,127,127},
              textString="%name")}),
          Documentation(info="<html>
<p>
Connector with one input signal of type Posture.
</p>
</html>"));

      connector PostureOutput = 
                          output Posture "'input Posture' as connector" 
      annotation (defaultComponentName="u",
      Icon(graphics={Polygon(
              points={{-100,100},{100,0},{-100,-100},{-100,100}},
              lineColor={0,127,127},
              fillColor={255,255,255},
              fillPattern=FillPattern.Solid), Text(
              extent={{100,-50},{700,50}},
              lineColor={0,127,127},
              textString="%name")},
           coordinateSystem(extent={{-100,-100},{100,100}}, preserveAspectRatio=true, initialScale=0.2)),
      Diagram(coordinateSystem(
            preserveAspectRatio=true, initialScale=0.2,
            extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Polygon(
              points={{0,50},{100,0},{0,-50},{0,50}},
              lineColor={0,127,127},
              fillColor={0,127,127},
              fillPattern=FillPattern.Solid), Text(
              extent={{-10,85},{-10,60}},
              lineColor={0,127,127},
              textString="%name")}),
        Documentation(info="<html>
<p>
Connector with one input signal of type Posture.
</p>
</html>"));

    model Real2Posture "Convert Real to type Posture"
      extends Library.Interfaces.ConversionIcon;

      PostureOutput y "Connector of Real output signal" 
        annotation (Placement(transformation(extent={{60,-20},{80,0}},
            rotation=0), iconTransformation(extent={{100,-20},{140,20}})));

      Library.Interfaces.RealInput_ u annotation (Placement(transformation(extent={{-130,
                -20},{-90,20}}),      iconTransformation(extent={{-140,-20},{-100,20}})));
    equation
      y = integer(u);
      annotation (
        Icon(coordinateSystem(
        preserveAspectRatio=true,
        extent={{-100,-100},{100,100}},
        grid={2,2},
        initialScale=0.04), graphics),
        Diagram(coordinateSystem(
        preserveAspectRatio=true,
        extent={{-100,-100},{100,100}},
        grid={2,2},
        initialScale=0.04), graphics),
    Documentation(info="<html>
<p>
The Real output y is a constant signal:
</p>
</html>"));
    end Real2Posture;
    end Hydrostatics;

    package Osmoles "Intracellular vs. Extracellular Water"

      model OsmBody
        //extends QHP.Library.Interfaces.BaseModel;

        HumMod.Library.Interfaces.RealInput OsmECFV_Electrolytes(final unit=
              "mOsm")                                annotation (Placement(transformation(
                extent={{-120,50},{-80,90}}), iconTransformation(extent={{-100,88},{-76,
                  112}})));
        HumMod.Library.Interfaces.RealInput OsmCell_Electrolytes(final unit=
              "mOsm")                                annotation (Placement(transformation(
                extent={{-120,-10},{-80,30}}),
                                             iconTransformation(extent={{-100,48},{-76,
                  72}})));
        HumMod.Library.Interfaces.RealInput UreaECF(final unit="mOsm") 
                                                        annotation (Placement(
              transformation(extent={{-120,20},{-80,60}}), iconTransformation(extent={{-100,8},
                  {-76,32}})));
        HumMod.Library.Interfaces.RealInput UreaICF(final unit="mOsm") 
                                                        annotation (Placement(
              transformation(extent={{-120,-40},{-80,0}}),iconTransformation(extent={{-100,
                  -32},{-76,-8}})));
        HumMod.Library.Interfaces.RealInput BodyH2O_Vol(final unit="ml")
          "all water in body"                                  annotation (Placement(transformation(extent={{-120,
                  -110},{-80,-70}}),   iconTransformation(extent={{-100,-112},{-76,-88}})));
        HumMod.Library.Interfaces.RealOutput ECFV(final unit="ml")
          "extracellular water"                            annotation (Placement(transformation(extent={{60,-80},
                  {100,-40}}),iconTransformation(extent={{92,8},{116,32}})));
        HumMod.Library.Interfaces.RealOutput ICFV(final unit="ml")
          "intracellular water"                            annotation (Placement(transformation(extent={{60,60},
                  {100,100}}),iconTransformation(extent={{92,68},{116,92}})));

        parameter Real Dissociation = 0.890;

        Real OsmECFV_NonElectrolytes;
        Real OsmCell_NonElectrolytes;
        Real Electrolytes;
        Real NonElectrolytes;
        Real Total;
        Real ECFVActiveElectrolytes;
        Real ICFVActiveElectrolytes;
        Real ActiveElectrolytes;
        Real ECFVActiveOsmoles;
        Real ICFVActiveOsmoles;
        Real ActiveOsmoles;
      //  Real Osm_conc_CellWall;
        //Real Osm_conc_Osmoreceptors;
        HumMod.Library.Interfaces.RealOutput OsmBody_Osm_conc_CellWalls(final unit
            =    "mOsm/ml")                                       annotation (
            Placement(transformation(extent={{60,-20},{100,20}}), iconTransformation(
                extent={{92,-52},{116,-28}})));
        HumMod.Library.Interfaces.RealOutput Osmoreceptors(final unit="mOsm/ml") 
                                                                  annotation (
            Placement(transformation(extent={{60,-54},{100,-14}}),iconTransformation(
                extent={{92,-92},{116,-68}})));
        HumMod.Library.Interfaces.RealInput GlucoseECF(final unit="mOsm") 
                                                        annotation (Placement(
              transformation(extent={{-120,-40},{-80,0}}),iconTransformation(extent={{-100,
                  -72},{-76,-48}})));
      equation
            OsmECFV_NonElectrolytes = UreaECF + GlucoseECF + 340.0;
            OsmCell_NonElectrolytes = UreaICF + 354.0;

             Electrolytes  =  OsmECFV_Electrolytes + OsmCell_Electrolytes;
             NonElectrolytes  =  OsmECFV_NonElectrolytes + OsmCell_NonElectrolytes;
             Total  =  Electrolytes + NonElectrolytes;
             ECFVActiveElectrolytes  =  Dissociation * OsmECFV_Electrolytes;
             ICFVActiveElectrolytes  =  Dissociation * OsmCell_Electrolytes;
             ActiveElectrolytes  =  ECFVActiveElectrolytes + ICFVActiveElectrolytes;
             ECFVActiveOsmoles  =  ECFVActiveElectrolytes + OsmECFV_NonElectrolytes;
             ICFVActiveOsmoles  =  ICFVActiveElectrolytes + OsmCell_NonElectrolytes;
             ActiveOsmoles  =  ECFVActiveOsmoles + ICFVActiveOsmoles;
             OsmBody_Osm_conc_CellWalls  =  ActiveOsmoles / BodyH2O_Vol;
             Osmoreceptors  =  ActiveElectrolytes / BodyH2O_Vol;
             ICFV  =  ( ICFVActiveOsmoles / ActiveOsmoles)  * BodyH2O_Vol;
             ECFV  =  BodyH2O_Vol - ICFV;
      annotation (
      Documentation(info="<html>
<pre>QHP 2008 / OsmBody
 
Created : 31-May-05
Last Modified : 24-Mar-08
Author : Tom Coleman
Copyright : 2005-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0 </pre>
<p><br/><br/>Intracellular water calculation from cells osmotic pressure.</p>
</html>",
       revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),       Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                      graphics),
          Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}}),
                          graphics),
                    Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},
                  {100,100}}),       graphics));
      end OsmBody;
    end Osmoles;

    package WaterCompartments "Body Water Distribution"

      model Outtake

        Library.VolumeFlow.PositiveVolumeFlow H2OLoss annotation (
            Placement(transformation(extent={{-120,-20},{-80,20}}),
              iconTransformation(extent={{-110,-10},{-90,10}})));

      parameter Real H2OTarget(final quantity="Flow",final unit="ml/min") = 0;

      parameter Real H2OMassEffect[ :,3] = {{  0.0,  0.0,  0.0}, { 50.0,  1.0,  0.0}};

        Library.VolumeFlow.OutputPump outputPump 
          annotation (Placement(transformation(extent={{20,-20},{60,20}})));
        Library.Blocks.FlowConstant Constant0(k=H2OTarget) 
          annotation (Placement(transformation(extent={{-60,56},{-40,76}})));
        Library.Curves.Curve curve(x=H2OMassEffect[:,1],y=H2OMassEffect[:,2],slope=H2OMassEffect[:,3]) 
          annotation (Placement(transformation(extent={{-60,24},{-40,44}})));
        Modelica.Blocks.Math.Product product 
          annotation (Placement(transformation(extent={{-8,30},{12,50}})));
        Library.VolumeFlow.VolumeMeasure volumeMeasure 
          annotation (Placement(transformation(extent={{-86,-6},{-66,14}})));
      equation

        connect(curve.val, product.u2) annotation (Line(
            points={{-39.8,34},{-10,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.y, outputPump.desiredFlow) annotation (Line(
            points={{13,40},{40,40},{40,12}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(product.u1, Constant0.y) annotation (Line(
            points={{-10,46},{-22,46},{-22,66},{-39,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(H2OLoss, volumeMeasure.q_in) annotation (Line(
            points={{-100,0},{-76,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(volumeMeasure.q_in, outputPump.q_in) annotation (Line(
            points={{-76,0},{20,0}},
            color={0,0,0},
            smooth=Smooth.None));
        connect(volumeMeasure.actualVolume, curve.u) annotation (Line(
            points={{-76,8},{-76,34},{-60,34}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={Rectangle(
                extent={{-100,40},{100,-40}},
                lineColor={0,0,255},
                fillPattern=FillPattern.Solid,
                fillColor={255,255,255}), Text(
                extent={{-90,-40},{100,40}},
                lineColor={0,0,255},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid,
                textString="%name")}), Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics));
      end Outtake;

      package Kidney "Kidney Water Excretion"

        model CD_H2OChannels
          extends HumMod.Library.Utilities.DynamicState(stateName="CD_H2OChannels.Inactive");
          extends HumMod.Library.Interfaces.BaseModel;

        parameter Real initialActive(final unit="1") = 1;
        parameter Real InactivateK(final unit="1/m1") = 0.000125;
        parameter Real ReactivateK(final unit="1/m1") = 0.0004;

          Library.PressureFlow.PositivePressureFlow CD_H2O_Reab annotation (Placement(
                transformation(extent={{-20,80},{20,120}}), iconTransformation(extent={
                    {-20,80},{20,120}})));

          Real Inactive( start = 2-initialActive, final unit="1");
          HumMod.Library.Interfaces.RealOutput_ Active(final unit="1") 
                                     annotation (Placement(transformation(extent={{96,46},
                    {136,86}}), iconTransformation(extent={{96,46},{136,86}})));
          Library.PressureFlow.NegativePressureFlow q_out annotation (Placement(
                transformation(extent={{-20,-120},{20,-80}}), iconTransformation(extent={{-20,
                    -120},{20,-80}})));
        equation
          q_out.q + CD_H2O_Reab.q = 0;
          q_out.pressure = CD_H2O_Reab.pressure;

        //  der(Inactive) = (InactivateK * CD_H2O_Reab.q - ReactivateK * Inactive) / Library.SecPerMin;
          Active = 2 - Inactive;

          stateValue = Inactive;
          changePerMin = InactivateK * CD_H2O_Reab.q - ReactivateK * Inactive;
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}), graphics={Text(
                  extent={{-100,-22},{100,-38}},
                  lineColor={0,0,255},
                  textString="(initial %initialActive)")}));
        end CD_H2OChannels;

        model DistalTubule

          Library.PressureFlow.PositivePressureFlow Inflow 
                                                         annotation (Placement(
                transformation(extent={{-120,-18},{-80,22}}), iconTransformation(
                  extent={{-120,-20},{-80,20}})));
          Library.PressureFlow.NegativePressureFlow Outflow 
            annotation (Placement(transformation(extent={{80,-20},{120,20}})));
          Library.PressureFlow.NegativePressureFlow Reabsorbtion 
                                                           annotation (Placement(
                transformation(extent={{-20,-58},{20,-18}}), iconTransformation(
                  extent={{-20,-58},{20,-18}})));
          HumMod.Library.Interfaces.RealInput_ DesiredFlow(final quantity=
                "Flow", final unit="ml/min") 
                                       annotation (Placement(transformation(extent={{-20,20},{20,
                    60}}), iconTransformation(extent={{-20,-20},{20,20}},
                                                                        rotation=-90,
                origin={60,40})));
        equation
          Outflow.q + Inflow.q + Reabsorbtion.q = 0;
          Inflow.pressure = Outflow.pressure;
          Outflow.q = - DesiredFlow;
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,40},{100,-40}},
                  lineColor={127,127,0},
                  fillColor={255,255,170},
                  fillPattern=FillPattern.HorizontalCylinder),
                Line(
                  points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,
                      -28},{6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,
                      8},{-58,12},{-70,14}},
                  color={0,0,255},
                  smooth=Smooth.None),
                Text(
                  extent={{12,-42},{166,-72}},
                  lineColor={0,0,255},
                  textString="%name"),
                Polygon(
                  points={{20,14},{20,-14},{82,-2},{20,14}},
                  lineColor={0,0,255},
                  smooth=Smooth.None,
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid)}),
                                         Diagram(coordinateSystem(preserveAspectRatio=true,
                          extent={{-100,-100},{100,100}}), graphics));
        end DistalTubule;

        model Glomerulus4


          Library.Semipermeable.ColloidHydraulicPressure0
            colloidHydraulicPressure 
            annotation (Placement(transformation(extent={{-30,18},{-38,26}})));
          Library.Semipermeable.ColloidOsmolarity colloidOsmolarity(q_in(q(
                  start=-569.708))) 
            annotation (Placement(transformation(extent={{42,18},{34,26}})));
          Library.PressureFlow.ResistorWithCond Kf 
            annotation (Placement(transformation(extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-4,-54})));
          Library.Factors.SimpleMultiply NephronCountEffect 
            annotation (Placement(transformation(extent={{24,-54},{4,-34}})));
          Library.Blocks.Constant const(
                                 k=20) 
            annotation (Placement(transformation(extent={{24,-40},{18,-34}})));
          Library.Semipermeable.ColloidHydraulicPressure
            colloidHydraulicPressure1(q_in(q(start=132.696))) 
                                      annotation (Placement(transformation(
                extent={{-4,-4},{4,4}},
                rotation=270,
                origin={-4,-2})));
          Library.PressureFlow.PositivePressureFlow fromAffHydraulic 
                                                         annotation (Placement(
                transformation(extent={{-76,16},{-56,36}}),   iconTransformation(
                  extent={{-120,-20},{-80,20}})));
          Library.PressureFlow.NegativePressureFlow Filtration 
                                                           annotation (Placement(
                transformation(extent={{-14,-86},{6,-66}}),  iconTransformation(
                  extent={{-20,-58},{20,-18}})));
          Library.PressureFlow.NegativePressureFlow toEffHydraulic 
            annotation (Placement(transformation(extent={{64,8},{84,28}}),
                iconTransformation(extent={{80,-20},{120,20}})));
          Library.Interfaces.RealInput_ Protein_massFlow(    final quantity="Flow",
              final unit="g/min")                annotation (Placement(transformation(extent={{20,30},
                    {30,40}}),
                           iconTransformation(extent={{-30,-20},{10,20}},rotation=-90,
                origin={-60,40})));
          Library.Interfaces.RealInput_ NephronCount_xNormal(   final unit="1") 
            annotation (Placement(transformation(extent={{42,-46},{38,-42}}),
                iconTransformation(extent={{20,-20},{-20,20}},
                rotation=90,
                origin={60,50})));
        equation

          connect(colloidOsmolarity.q_out, colloidHydraulicPressure.q_in) 
            annotation (Line(
              points={{34,22},{-30,22}},
              color={127,127,0},
              thickness=1,
              smooth=Smooth.None));
          connect(colloidOsmolarity.P, colloidHydraulicPressure.hydraulicPressure) 
            annotation (Line(
              points={{40.4,17.2},{40.4,16},{8,16},{8,40},{-35.6,40},{-35.6,26.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(NephronCountEffect.y,Kf. cond) annotation (Line(
              points={{14,-46},{14,-54},{0,-54}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(NephronCountEffect.yBase,const. y) annotation (Line(
              points={{14,-42},{14,-38},{17.7,-38},{17.7,-37}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(NephronCountEffect.u,NephronCount_xNormal)  annotation (Line(
              points={{23.8,-44},{40,-44}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Kf.q_out, Filtration) annotation (Line(
              points={{-4,-64},{-4,-70},{-4,-70},{-4,-76}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(Kf.q_in, colloidHydraulicPressure1.q_out) annotation (Line(
              points={{-4,-44},{-4,-34.5},{-4,-34.5},{-4,-25},{-4,-6},{-4,-6}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(colloidHydraulicPressure1.q_in, colloidHydraulicPressure.q_in) 
            annotation (Line(
              points={{-4,2},{-4,22},{-30,22}},
              color={127,127,0},
              thickness=1,
              smooth=Smooth.None));
          connect(colloidOsmolarity.P, colloidHydraulicPressure1.hydraulicPressure) 
            annotation (Line(
              points={{40.4,17.2},{40.4,16},{8,16},{8,-3.6},{0.8,-3.6}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(fromAffHydraulic, colloidHydraulicPressure.withoutCOP) 
            annotation (Line(
              points={{-66,26},{-38,26}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(colloidOsmolarity.q_in, toEffHydraulic) annotation (Line(
              points={{42,18},{74,18}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(colloidOsmolarity.proteinMassFlow, Protein_massFlow) annotation (Line(
              points={{36.08,26.64},{36.08,35},{25,35}},
              color={0,0,127},
              smooth=Smooth.None));
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -100},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,40},{100,-40}},
                  lineColor={127,0,0},
                  fillPattern=FillPattern.HorizontalCylinder,
                  fillColor={255,255,255}),
                Line(
                  points={{-70,14},{-70,-18},{-52,-12},{-36,-14},{-18,-20},{-2,
                      -28},{6,-36},{8,-40},{6,-22},{0,-12},{-8,-6},{-22,2},{-40,
                      8},{-58,12},{-70,14}},
                  color={0,0,255},
                  smooth=Smooth.None),
                Text(
                  extent={{12,-42},{166,-72}},
                  lineColor={0,0,255},
                  textString="%name")}), Diagram(coordinateSystem(preserveAspectRatio=true,
                          extent={{-100,-100},{100,100}}), graphics={
                Text(
                  extent={{-78,40},{-52,28}},
                  lineColor={0,127,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  textString="(plasma flow)"),
                Text(
                  extent={{58,32},{94,22}},
                  lineColor={0,127,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  textString="(hydraulic pressure)"),
                Text(
                  extent={{14,48},{48,42}},
                  lineColor={0,127,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  textString="(COP_Eff)"),
                Text(
                  extent={{2,-58},{30,-56}},
                  lineColor={0,0,255},
                  textString="ultrafiltration coefficient")}));
        end Glomerulus4;

        model Nephrons2
          import QHP = HumMod;
         //extends Library.BaseModel;
          Library.PressureFlow.NegativePressureFlow
            EfferentArtery_Water_Hydraulic "outgoing plasma from kidneys" 
                                                         annotation (Placement(
                transformation(extent={{-100,-22},{-80,-2}}), iconTransformation(extent={{22,90},
                    {42,110}})));
          Library.PressureFlow.NegativePressureFlow urine "H2O excretion" 
                                                          annotation (Placement(
                transformation(extent={{82,-8},{104,12}}),  iconTransformation(extent={{36,-80},
                    {56,-60}})));
          Library.PressureFlow.ResistorWithCondParam ProximalTubule_Conductance(
              cond=7, q_in(pressure(start=18.9565))) 
            annotation (Placement(transformation(extent={{-8,-8},{12,12}}, rotation=-90,
                origin={-64,18})));
          QHP.Library.PressureFlow.Reabsorbtion LoopOfHenle 
            annotation (Placement(transformation(extent={{-46,-58},{-28,-40}})));
          QHP.Library.PressureFlow.Reabsorbtion ProximalTubule 
            annotation (Placement(transformation(extent={{-11,-11},{9,9}},  rotation=-90,
                origin={-61,-9})));
          QHP.Library.Interfaces.RealInput PT_Na_FractReab(final unit="1") 
            annotation (Placement(transformation(extent={{2,-2},{-2,2}},
                rotation=180,
                origin={-74,-26}),
                iconTransformation(extent={{100,20},{90,30}})));
          Modelica.Blocks.Math.Gain gain(k=0.37) 
            annotation (Placement(transformation(extent={{-3,-3},{3,3}},      rotation=180,
                origin={-17,-53})));
          QHP.Library.Interfaces.RealInput LH_Na_FractReab(final unit="1") 
            annotation (Placement(transformation(extent={{0,-56},{-6,-50}}),
                iconTransformation(extent={{100,10},{90,20}})));
          QHP.Library.Interfaces.RealInput DT_Na_Outflow(final quantity="Flow", final unit
              =                                                                            "mEq/min") 
            annotation (Placement(transformation(extent={{0,30},{4,34}}),
                iconTransformation(extent={{100,0},{90,10}})));
          QHP.Library.Factors.SimpleMultiply ADHEffect 
            annotation (Placement(transformation(extent={{-4,6},{16,26}})));
          QHP.Library.Interfaces.RealInput KidneyProtein_massFlow(final
              quantity =                                                         "Flow", final unit
              =                                                                                     "g/min") 
            annotation (Placement(transformation(extent={{-76,82},{-70,88}}),
                iconTransformation(extent={{100,50},{90,60}})));
          QHP.Library.PressureFlow.ReabsorbtionWithMinimalOutflow
            CollectingDuct 
            annotation (Placement(transformation(extent={{40,-8},{60,12}})));
          QHP.Library.Interfaces.RealInput CD_Na_Outflow(final quantity="Flow", final unit
              =                                                                            "mEq/min") 
            annotation (Placement(transformation(extent={{82,50},{78,54}}),
                iconTransformation(extent={{100,-10},{90,0}})));
          QHP.Library.Interfaces.RealInput CD_K_Outflow(final quantity="Flow", final unit
              =                                                                           "mEq/min") 
            annotation (Placement(transformation(extent={{82,46},{78,50}}),
                iconTransformation(extent={{100,-20},{90,-10}})));
          QHP.Library.Interfaces.RealInput CD_NH4_Outflow(final quantity="Flow", final unit
              =                                                                             "mEq/min") 
            annotation (Placement(transformation(extent={{82,42},{78,46}}),
                iconTransformation(extent={{100,-30},{90,-20}})));
          QHP.Library.Interfaces.RealInput CD_Glucose_Outflow(final quantity="Flow", final unit
              =                                                                                 "mg/min") 
            annotation (Placement(transformation(extent={{88,36},{84,40}}),
                iconTransformation(extent={{100,-40},{90,-30}})));
          QHP.Library.Interfaces.RealInput MedullaNa_conc(final quantity="Concentration", final unit
              =                                                                                      "mEq/ml") 
            annotation (Placement(transformation(extent={{5.3333,-5.3333},{
                    -2.66666,2.66666}},
                rotation=0,
                origin={96.6667,19.3333}),
                iconTransformation(extent={{100,-50},{90,-40}})));
          Modelica.Blocks.Math.Gain gain1(k=.5) 
            annotation (Placement(transformation(extent={{72,36},{68,40}})));
          Modelica.Blocks.Math.Sum sum1(nin=4) 
            annotation (Placement(transformation(extent={{3,-3},{-5,5}},
                rotation=90,
                origin={67,29})));
          QHP.Library.Factors.SimpleMultiply MedullaNaEffect 
            annotation (Placement(transformation(extent={{76,8},{56,28}})));
          QHP.Water.WaterCompartments.Kidney.Glomerulus4 glomerulus 
            annotation (Placement(transformation(extent={{-72,66},{-52,86}})));
          Library.PressureFlow.PositivePressureFlow
            AfferentArtery_Water_Hydraulic "ingoing plasma to kidney" 
                                                                  annotation (Placement(
                transformation(extent={{-88,66},{-68,86}}), iconTransformation(extent={{-20,92},
                    {0,112}})));
          CD_H2OChannels H2OChannels(initialActive=0.969492) 
            annotation (Placement(transformation(extent={{40,-58},{60,-38}})));
          QHP.Water.WaterCompartments.Kidney.DistalTubule distalTubule 
            annotation (Placement(transformation(extent={{-10,-8},{10,12}})));
          QHP.Library.Blocks.Inv inv1 annotation (Placement(transformation(
                extent={{-2,-2},{2,2}},
                rotation=0,
                origin={-8,16})));
          QHP.Library.Blocks.Inv inv2 annotation (Placement(transformation(
                extent={{-5,-5},{3,3}},
                rotation=180,
                origin={83,17})));
          QHP.Library.PressureFlow.FlowMeasure flowMeasure annotation (Placement(
                transformation(
                extent={{10,-10},{-10,10}},
                rotation=90,
                origin={-62,42})));
          QHP.Library.Interfaces.RealOutput_ GFR "glomerulus filtrate" 
            annotation (Placement(transformation(
                extent={{-4,-4},{4,4}},
                rotation=180,
                origin={-76,42}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=180,
                origin={-110,50})));
          QHP.Library.Interfaces.RealInput NephronCount_xNormal(final unit="1") 
            annotation (Placement(transformation(extent={{-40,84},{-44,88}}),
                iconTransformation(extent={{100,40},{90,50}})));
          QHP.Library.PressureFlow.FlowMeasure flowMeasure1 
            annotation (Placement(transformation(extent={{74,12},{94,-8}})));
          QHP.Library.Interfaces.RealOutput_ CD_H2O_Outflow "H2O to urine" 
            annotation (Placement(transformation(
                extent={{-4,-4},{4,4}},
                rotation=0,
                origin={92,-10}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=180,
                origin={-110,-60})));
          QHP.Library.Factors.SplineValue NephronADHOnPerm(
            data={{0.0,0.3,0},{2.0,1.0,0.5},{10.0,3.0,0}}) annotation (
              Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=270,
                origin={72,-24})));
          QHP.Library.Factors.SplineValue PermOnOutflow(data={{0.3,0.00,0},{1.0,
                0.93,0.1},{3.0,1.00,0}}) annotation (Placement(transformation(
                extent={{10,-10},{-10,10}},
                rotation=270,
                origin={66,-10})));
          QHP.Library.Blocks.Constant Constant0(k=1) 
            annotation (Placement(transformation(extent={{76,-12},{72,-8}})));
          QHP.Library.Interfaces.RealInput ADH(final quantity="Concentration",
              final unit="pg/ml") 
            annotation (Placement(transformation(extent={{-40,20},{-32,28}}),
                iconTransformation(extent={{100,30},{90,40}})));
          QHP.Library.Blocks.Constant Constant1(k=1) 
            annotation (Placement(transformation(extent={{-20,28},{-14,34}})));
          QHP.Library.Factors.SplineValue NephronADHEffect(
            data={{0.0,0.060,0},{2.0,0.110,0.02},{10.0,0.160,0}}) annotation (
              Placement(transformation(
                extent={{-10,-10},{10,10}},
                rotation=0,
                origin={-12,24})));
          QHP.Library.PressureFlow.FlowMeasure flowMeasure2 
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=180,
                origin={16,-62})));
          QHP.Library.Interfaces.RealOutput_ CD_H2O_Reab "H2O to urine" 
            annotation (Placement(transformation(
                extent={{-4,-4},{4,4}},
                rotation=0,
                origin={22,-50}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=180,
                origin={-110,-80})));
          QHP.Library.PressureFlow.FlowMeasure flowMeasure3 
            annotation (Placement(transformation(extent={{-10,10},{10,-10}},
                rotation=90,
                origin={-22,-20})));
          QHP.Library.Interfaces.RealOutput_ LH_H2O_Outflow
            "H2O from loop of Henle" 
            annotation (Placement(transformation(
                extent={{-4,-4},{4,4}},
                rotation=0,
                origin={-8,-20}), iconTransformation(
                extent={{-10,-10},{10,10}},
                rotation=270,
                origin={-72,-70})));
        equation

          connect(ProximalTubule.Reabsorbtion, EfferentArtery_Water_Hydraulic) 
                                                       annotation (Line(
              points={{-72,-8},{-90,-8},{-90,-12}},
              color={0,0,0},
              smooth=Smooth.None,
              thickness=1));
          connect(ProximalTubule.FractReab, PT_Na_FractReab) annotation (Line(
              points={{-66.6,-18.4},{-66,-18.4},{-66,-26},{-74,-26}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(LoopOfHenle.Reabsorbtion, EfferentArtery_Water_Hydraulic) 
                                                    annotation (Line(
              points={{-37,-58},{-37,-62},{-90,-62},{-90,-12}},
              color={0,0,0},
              smooth=Smooth.None,
              thickness=1));
          connect(gain.u, LH_Na_FractReab) annotation (Line(
              points={{-13.4,-53},{-3,-53}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(DT_Na_Outflow, ADHEffect.yBase)       annotation (Line(
              points={{2,32},{6,32},{6,18}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(CD_Glucose_Outflow, gain1.u) annotation (Line(
              points={{86,38},{72.4,38}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(gain1.y, sum1.u[1]) annotation (Line(
              points={{67.8,38},{66.6,38},{66.6,32.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(CD_NH4_Outflow, sum1.u[2]) annotation (Line(
              points={{80,44},{66,44},{66,32.8},{66.2,32.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(CD_K_Outflow, sum1.u[3]) annotation (Line(
              points={{80,48},{65.3,48},{65.3,32.8},{65.8,32.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(CD_Na_Outflow, sum1.u[4]) annotation (Line(
              points={{80,52},{66,52},{66,32.8},{65.4,32.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(MedullaNaEffect.yBase, sum1.y)   annotation (Line(
              points={{66,20},{66,23.6}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(distalTubule.Reabsorbtion, EfferentArtery_Water_Hydraulic) 
                                                         annotation (Line(
              points={{0,-1.8},{0,-62},{-90,-62},{-90,-12}},
              color={0,0,0},
              smooth=Smooth.None,
              thickness=1));
          connect(CollectingDuct.Inflow, distalTubule.Outflow) annotation (Line(
              points={{40,2},{10,2}},
              color={0,0,0},
              smooth=Smooth.None,
              thickness=1));
          connect(distalTubule.DesiredFlow, ADHEffect.y)       annotation (Line(
              points={{6,6},{6,14}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(inv2.u, MedullaNa_conc) annotation (Line(
              points={{88.8,18},{93.4,18},{93.4,18},{98,18}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(inv1.y, ADHEffect.u)  annotation (Line(
              points={{-5.8,16},{-3.8,16}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(inv2.y, MedullaNaEffect.u) annotation (Line(
              points={{79.6,18},{75.8,18}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(ProximalTubule_Conductance.q_out, ProximalTubule.Inflow) 
            annotation (Line(
              points={{-62,6},{-62,2}},
              color={0,0,0},
              smooth=Smooth.None,
              thickness=1));
          connect(flowMeasure.q_out, ProximalTubule_Conductance.q_in) 
            annotation (Line(
              points={{-62,36},{-62,26}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure.actualFlow, GFR) annotation (Line(
              points={{-67.4,42},{-76,42}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(ProximalTubule.Outflow, LoopOfHenle.Inflow) annotation (Line(
              points={{-62,-18},{-62,-48},{-46,-48},{-46,-49}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(KidneyProtein_massFlow, glomerulus.Protein_massFlow) 
            annotation (Line(
              points={{-73,85},{-68,85},{-68,81}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(glomerulus.NephronCount_xNormal, NephronCount_xNormal) 
            annotation (Line(
              points={{-56,81},{-56,86},{-42,86}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(AfferentArtery_Water_Hydraulic, glomerulus.fromAffHydraulic) 
            annotation (Line(
              points={{-78,76},{-72,76}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(glomerulus.Filtration, flowMeasure.q_in) annotation (Line(
              points={{-62,72.2},{-62,48}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(glomerulus.toEffHydraulic, EfferentArtery_Water_Hydraulic) 
            annotation (Line(
              points={{-52,76},{-32,76},{-32,94},{-90,94},{-90,-12}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(CollectingDuct.Outflow, flowMeasure1.q_in) annotation (Line(
              points={{60,2},{78,2}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure1.q_out, urine) annotation (Line(
              points={{90,2},{93,2}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure1.actualFlow, CD_H2O_Outflow) annotation (Line(
              points={{84,-3.4},{84,-10},{92,-10}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(H2OChannels.Active, NephronADHOnPerm.yBase) annotation (Line(
              points={{61.6,-41.4},{78,-41.4},{78,-24},{74,-24}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(NephronADHOnPerm.y, PermOnOutflow.u) annotation (Line(
              points={{70,-24},{66,-24},{66,-19.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(PermOnOutflow.y, CollectingDuct.FractReab) annotation (Line(
              points={{64,-10},{62,-10},{62,-4},{60,-4},{60,-3.6},{59.4,-3.6}},
              color={0,0,127},
              smooth=Smooth.None));

          connect(Constant0.y, PermOnOutflow.yBase) annotation (Line(
              points={{71.8,-10},{68,-10}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(CollectingDuct.Reabsorbtion, H2OChannels.CD_H2O_Reab) 
            annotation (Line(
              points={{50,-8},{50,-38}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(ADH, NephronADHOnPerm.u) annotation (Line(
              points={{-36,24},{-32,24},{-32,-68},{72,-68},{72,-33.8}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(Constant1.y, NephronADHEffect.yBase) annotation (Line(
              points={{-13.7,31},{-12,31},{-12,26}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(NephronADHEffect.y, inv1.u) annotation (Line(
              points={{-12,22},{-12,16},{-10.4,16}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(ADH, NephronADHEffect.u) annotation (Line(
              points={{-36,24},{-21.8,24}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(LoopOfHenle.FractReab, gain.y) annotation (Line(
              points={{-27.64,-53.14},{-23.82,-53.14},{-23.82,-53},{-20.3,-53}},
              color={0,0,127},
              smooth=Smooth.None));

          connect(CollectingDuct.OutflowMin, MedullaNaEffect.y) annotation (
              Line(
              points={{53.4,7},{66,7},{66,16}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(H2OChannels.q_out, flowMeasure2.q_in) annotation (Line(
              points={{50,-58},{50,-62},{22,-62}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure2.q_out, EfferentArtery_Water_Hydraulic) 
            annotation (Line(
              points={{10,-62},{-90,-62},{-90,-12}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure2.actualFlow, CD_H2O_Reab) annotation (Line(
              points={{16,-56.6},{16,-50},{22,-50}},
              color={0,0,127},
              smooth=Smooth.None));
          connect(LoopOfHenle.Outflow, flowMeasure3.q_in) annotation (Line(
              points={{-28,-49},{-22,-49},{-22,-26}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure3.q_out, distalTubule.Inflow) annotation (Line(
              points={{-22,-14},{-22,2},{-10,2}},
              color={0,0,0},
              thickness=1,
              smooth=Smooth.None));
          connect(flowMeasure3.actualFlow, LH_H2O_Outflow) annotation (Line(
              points={{-16.6,-20},{-8,-20}},
              color={0,0,127},
              smooth=Smooth.None));
          annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                    -70},{100,100}}), graphics={
                Rectangle(
                  extent={{-100,50},{100,-70}},
                  lineColor={99,0,99},
                  fillColor={215,215,215},
                  fillPattern=FillPattern.VerticalCylinder),
                Text(
                  extent={{-72,-102},{76,-76}},
                  lineColor={0,0,0},
                  fillColor={255,255,255},
                  fillPattern=FillPattern.Solid,
                  textString="%name"),
                Bitmap(extent={{-96,96},{106,-74}}, fileName="icons/nefron.png")}),
                                                  Diagram(coordinateSystem(
                  preserveAspectRatio=true, extent={{-100,-70},{100,100}}),
                graphics));
        end Nephrons2;
      end Kidney;

      model Torso2
      extends HumMod.Library.Interfaces.Tissues.OtherTissue;


        HumMod.Library.Interfaces.RealInput_ InterstitialProtein_Mass(final
            quantity="Mass", final unit="g") "interstitial protein mass"                  annotation (Placement(transformation(extent={{60,34},
                  {52,42}}),
                         iconTransformation(extent={{110,-110},{90,-90}})));
        HumMod.Library.Interfaces.RealInput_ CellH2O_Vol_part(final quantity=
              "Volume", final unit="ml") "water in cells"         annotation (Placement(transformation(
                extent={{72,26},{64,34}}),  iconTransformation(extent={{110,-70},{90,-50}})));

        parameter Real initialVolume(final quantity="Volume",final unit="ml") = 9800
          "initial water volume in torso";
        parameter Real PressureVolume[ :,3] = {{   600.0,  -30.0,   0.01}, {  2400.0,   -4.8,   0.0004}, {  6000.0,    0.0,   0.0004}, { 12000.0,   50.0,   0.01}}
          "curve of water pressure, that depends on interstitial water volume";
        parameter Real CapilaryWallPermeability(final unit="ml/(min.mmHg)") = 1
          "Capilary wall permeability for water";
        parameter Real InterstitialPressureOnLymphFlow[:,3]={{-14.0,0.0,0.0},{-4.0,1.0,0.1},{2.0,8.0,4.0},{6.0,25.0,0.0}}
          "dependence between interstitial water hydraulic pressure and lymph flow";
        parameter Real NormalLymphFlowRate(final unit="ml/min") = 0.4;

        HumMod.Library.Interfaces.RealOutput_ Vol(final quantity="Volume",
            final unit="ml") "volume of all water in torso"         annotation (Placement(transformation(extent={{32,-58},
                  {42,-48}}),  iconTransformation(extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-60,-100})));
        HumMod.Library.Interfaces.RealOutput_ InterstitialVol(final quantity=
              "Volume", final unit="ml")
          "volume of interstitial water in torso"                   annotation (Placement(
              transformation(extent={{48,8},{58,18}}),    iconTransformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={0,-100})));
        Library.Curves.Curve PressureVolumeCurve(
          x=PressureVolume[:,1],y=PressureVolume[:,2],slope=PressureVolume[:,3])
          "curve of water hydraulic pressure, that depends on interstitial water volume"
          annotation (Placement(transformation(extent={{56,-16},{76,4}})));
        TorsoExtravascularWater H2O(NotpermeableSolutes(unit="g"),
            initialWaterVolume=initialVolume) 
          annotation (Placement(transformation(extent={{40,20},{20,40}})));
        Library.Semipermeable.PositiveOsmoticFlow q_in(o(final unit="g/ml"))
          "plasma capillary wall osmotic connector"    annotation (Placement(
              transformation(extent={{-110,10},{-90,30}}), iconTransformation(extent={{-110,30},
                  {-90,50}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure toPressure 
          annotation (Placement(transformation(extent={{-86,16},{-78,24}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure2 toPressure1 
          annotation (Placement(transformation(extent={{6,34},{-2,26}})));
        Library.PressureFlow.ResistorWithCondParam CapillaryWall(cond=
              CapilaryWallPermeability) 
          annotation (Placement(transformation(extent={{-65,32},{-43,54}})));
        Library.Interfaces.RealOutput_ LymphFlow(
                                                final quantity="Flow", final unit
            =                                                                     "ml/min") 
                                        annotation (Placement(transformation(extent={{-33,4},
                  {-27,10}}),       iconTransformation(extent={{-10,-10},{10,10}},
              rotation=270,
              origin={60,-100})));
        HumMod.Library.Interfaces.RealInput_ capilaryPressure(final quantity=
              "Pressure", final unit="mmHg")
          "average capilary hydraulic pressure in torso"                                annotation (Placement(transformation(extent={{5,-5},{
                  -5,5}},
              rotation=180,
              origin={-92,40}),
                         iconTransformation(extent={{110,-30},{90,-10}})));
        Library.PressureFlow.Pump pump annotation (Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=180,
              origin={-46,0})));
        Library.Factors.SplineValue InterstitialPressureEffect(data=
              InterstitialPressureOnLymphFlow) 
          annotation (Placement(transformation(extent={{-36,2},{-56,22}})));
        Library.Blocks.FlowConstant flowConstant(k=NormalLymphFlowRate) 
          annotation (Placement(transformation(extent={{-56,16},{-48,24}})));
        HumMod.Library.Interfaces.RealInput_ HeatInsensible_H2O(final quantity=
              "Flow", final unit="ml/min") "vaporization"                                 annotation (Placement(transformation(extent={{32,90},
                  {24,98}}),
                         iconTransformation(extent={{110,10},{90,30}})));
        HumMod.Library.Interfaces.RealInput_ HeatSweating_H2O(final quantity=
              "Flow", final unit="ml/min") "sweating"             annotation (Placement(transformation(
                extent={{-16,80},{-8,88}}), iconTransformation(extent={{110,50},{90,70}})));
        HumMod.Library.Interfaces.RealInput_ MetabolicH2O_Rate(final quantity=
              "Flow", final unit="ml/min") "water from cellular metabolism"             annotation (Placement(transformation(extent={{5,-5},{
                  -5,5}},
              rotation=180,
              origin={-30,74}),
                         iconTransformation(extent={{110,90},{90,110}})));
        Library.Semipermeable.InputPump metabolic 
          annotation (Placement(transformation(extent={{-30,54},{-10,74}})));
        Library.Semipermeable.OutputPump sweat 
          annotation (Placement(transformation(extent={{10,64},{-10,84}})));
        Library.Semipermeable.OutputPump vapor 
          annotation (Placement(transformation(extent={{26,74},{6,94}})));
      equation
        connect(toPressure.q_in, q_in) annotation (Line(
            points={{-86,20},{-100,20}},
            color={127,127,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(toPressure1.q_in, H2O.q_out)                     annotation (Line(
            points={{6,30},{30,30}},
            color={127,127,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(toPressure.q_out, CapillaryWall.q_in) 
                                                     annotation (Line(
            points={{-78,20},{-72,20},{-72,43},{-65,43}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(CapillaryWall.q_out, toPressure1.q_out) 
                                                       annotation (Line(
            points={{-43,43},{-12,43},{-12,30},{-2,30}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(InterstitialProtein_Mass, H2O.NotpermeableSolutes) 
          annotation (Line(
            points={{56,38},{40,38}},
            color={0,0,127},
            pattern=LinePattern.None,
            smooth=Smooth.None));
        connect(InterstitialPressureEffect.y, pump.desiredFlow) annotation (
            Line(
            points={{-46,10},{-46,6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowConstant.y, InterstitialPressureEffect.yBase) annotation (
            Line(
            points={{-47.6,20},{-46,20},{-46,14}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(InterstitialPressureEffect.y, LymphFlow) annotation (Line(
            points={{-46,10},{-46,7},{-30,7}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(PressureVolumeCurve.val, InterstitialPressureEffect.u) 
          annotation (Line(
            points={{76.2,-6},{84,-6},{84,-64},{0,-64},{0,12},{-36.2,12}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toPressure1.withoutCOP, pump.q_in) annotation (Line(
            points={{-2,26},{-12,26},{-12,0},{-36,0},{-36,-1.22465e-015}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(pump.q_out, toPressure.q_out) annotation (Line(
            points={{-56,1.22465e-015},{-72,1.22465e-015},{-72,20},{-78,20}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(MetabolicH2O_Rate, metabolic.desiredFlow) annotation (Line(
            points={{-30,74},{-20,74},{-20,68}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(metabolic.q_out, H2O.q_out)                     annotation (Line(
            points={{-14,64},{30,64},{30,30}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(vapor.desiredFlow, HeatInsensible_H2O) annotation (Line(
            points={{16,88},{16,94},{28,94}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(sweat.desiredFlow, HeatSweating_H2O) annotation (Line(
            points={{0,78},{0,84},{-12,84}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(vapor.q_in, H2O.q_out)                     annotation (Line(
            points={{22,84},{30,84},{30,30}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(sweat.q_in, H2O.q_out)                     annotation (Line(
            points={{6,74},{30,74},{30,30}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(toPressure1.hydraulicPressure, PressureVolumeCurve.val) annotation (
           Line(
            points={{0.4,25.2},{0.4,-64},{84,-64},{84,-6},{76.2,-6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(capilaryPressure, toPressure.hydraulicPressure) annotation (
            Line(
            points={{-92,40},{-80.4,40},{-80.4,24.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(CellH2O_Vol_part, H2O.ICFV) annotation (Line(
            points={{68,30},{54,30},{54,34},{40,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(H2O.ECFV, InterstitialVol) annotation (Line(
            points={{26,20},{26,12},{53,13}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(H2O.WaterVolume, Vol) annotation (Line(
            points={{30,20},{32,20},{32,-53},{37,-53}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(H2O.ECFV, PressureVolumeCurve.u) annotation (Line(
            points={{26,20},{26,-6},{56,-6}},
            color={0,0,127},
            smooth=Smooth.None));
          annotation (
      Documentation(info="<HTML>
<PRE>
Torso water compartment.
 
like
QHP 2008 / TissueH2O/UT_H2O.DES
and
QHP 2008 / TissueH2O/InterstitialWater/UT_InterstitialWater.DES
 
Created : 24-Mar-08
Last Modified : 24-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
        Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-100,64},{100,48}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)"),
              Text(
                extent={{-100,94},{100,74}},
                lineColor={0,0,255},
                textString="%name"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0})}));
      end Torso2;

      model Lungs
      extends HumMod.Library.Interfaces.Tissues.Lungs;

       parameter Real initialVolume(unit="ml") = 0;

        HumMod.Library.Interfaces.RealOutput_ Vol(final quantity="Volume",
            final unit="ml") "volume of all water in torso"         annotation (Placement(transformation(extent={{32,8},{
                  42,18}}),    iconTransformation(extent={{-10,-110},{10,-90}},rotation=-90)));
        Library.Semipermeable.PositiveOsmoticFlow q_in(o(final unit="g/ml")) 
                                                       annotation (Placement(
              transformation(extent={{-110,10},{-90,30}}), iconTransformation(extent={
                  {-10,30},{10,50}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure toPressure 
          annotation (Placement(transformation(extent={{-86,16},{-78,24}})));
        HumMod.Library.Interfaces.RealInput_ capilaryPressure(final quantity=
              "Pressure", final unit="mmHg")
          "average capilary hydraulic pressure in torso"                                annotation (Placement(transformation(extent={{5,-5},{
                  -5,5}},
              rotation=180,
              origin={-95,41}),
                         iconTransformation(extent={{108,40},{88,60}})));
        Library.PressureFlow.VolumeCompartement excessLungWater(initialVolume=
              initialVolume,
          stateName="ExcessLungWater.Volume",
          STEADY=false) 
          annotation (Placement(transformation(extent={{11,19},{31,39}})));
        Library.PressureFlow.ResistorWithCondParam PulmCapys(cond=3) 
          annotation (Placement(transformation(extent={{-38,24},{-16,46}})));
        Library.PressureFlow.Pump PulmLymph     annotation (Placement(
              transformation(
              extent={{-6,-6},{6,6}},
              rotation=180,
              origin={-28,20})));
        Library.Curves.Curve ExcessLungWaterOnLymph(
          x={10,1000},
          y={0,1},
          slope={0,0}) 
          annotation (Placement(transformation(extent={{-8,-20},{12,0}})));
      equation
        connect(toPressure.q_in, q_in) annotation (Line(
            points={{-86,20},{-100,20}},
            color={127,127,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(excessLungWater.Volume,ExcessLungWaterOnLymph. u) annotation (
            Line(
            points={{22.2,19},{22,10},{-14,10},{-14,-10},{-8,-10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ExcessLungWaterOnLymph.val,PulmLymph. desiredFlow) annotation (
            Line(
            points={{12.2,-10},{20,-10},{20,-28},{-28,-28},{-28,16.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(toPressure.q_out, PulmCapys.q_in) annotation (Line(
            points={{-78,20},{-58,20},{-58,35},{-38,35}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(toPressure.q_out, PulmLymph.q_out) annotation (Line(
            points={{-78,20},{-34,20}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(PulmCapys.q_out, excessLungWater.con) annotation (Line(
            points={{-16,35},{-2,35},{-2,29},{11,29}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(excessLungWater.con, PulmLymph.q_in) annotation (Line(
            points={{11,29},{-2,29},{-2,20},{-22,20}},
            color={0,0,0},
            pattern=LinePattern.None,
            thickness=1,
            smooth=Smooth.None));
        connect(excessLungWater.Volume, Vol) annotation (Line(
            points={{22.2,19},{22.2,13},{37,13}},
            color={0,0,127},
            pattern=LinePattern.None,
            smooth=Smooth.None));
        connect(capilaryPressure, toPressure.hydraulicPressure) annotation (Line(
            points={{-95,41},{-80.4,41},{-80.4,24.8}},
            color={0,0,127},
            smooth=Smooth.None));
          annotation (
      Documentation(info="<HTML>
<PRE>
Torso water compartment.
 
like
QHP 2008 / TissueH2O/UT_H2O.DES
and
QHP 2008 / TissueH2O/InterstitialWater/UT_InterstitialWater.DES
 
Created : 24-Mar-08
Last Modified : 24-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
        Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-100,-50},{100,-66}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0}),
              Text(
                extent={{-100,88},{100,72}},
                lineColor={0,0,255},
                textString="%name")}));
      end Lungs;

      model Plasma2
        extends HumMod.Library.Utilities.DynamicState;
        extends HumMod.Library.Interfaces.Tissues.Plasma;

        parameter Real initialVolume(final quantity="Volume",final unit="ml") = 3000
          "initial plasma water volume";

        HumMod.Library.Interfaces.RealOutput_ Volume(start=initialVolume, final
            quantity =                                                                   "Volume",
            final unit="ml") "volume of plasma water"       annotation (Placement(transformation(extent={{78,66},
                  {118,106}}),iconTransformation(extent={{88,76},{112,100}})));

        Library.Semipermeable.PositiveOsmoticFlow q_in(o(unit="g/ml")) 
                                                       annotation (Placement(
              transformation(extent={{-50,10},{-30,30}}),iconTransformation(extent={{-50,10},
                  {-30,30}})));
        HumMod.Library.Interfaces.RealInput_ PlasmaProtein_Mass(final quantity=
              "Mass", final unit="g")                annotation (Placement(
              transformation(extent={{-120,-50},{-80,-10}}), iconTransformation(
                extent={{-100,80},{-80,100}})));
      //initial equation
      //  Volume = initialVolume;
      equation
        //der(Volume) = q_in.q  /Library.SecPerMin;

        q_in.o = PlasmaProtein_Mass/Volume;

        stateValue = Volume;
        changePerMin = q_in.q;

        annotation (
      Documentation(info="<HTML>
<PRE>
QHP 2008 / BloodVolume/PlasmaVol.DES
 
Created : 21-Oct-04
Last Modified : 20-Mar-08
Author : Tom Coleman
Copyright : 2005-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
        Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-18,72},{124,58}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0}),
              Text(
                extent={{-100,-80},{100,-100}},
                lineColor={0,0,255},
                textString="%name")}));
      end Plasma2;

      model Peritoneum2
      extends HumMod.Library.Interfaces.Tissues.Peritoneum;
        Library.Semipermeable.PositiveOsmoticFlow flux
          "plasma proteins concentration"               annotation (Placement(
              transformation(extent={{56,30},{76,50}}),      iconTransformation(
                extent={{-10,30},{10,50}})));

        parameter Real initialVolume(final quantity="Volume",final unit="ml") = 0
          "initial water in peritoneum";

        HumMod.Library.Interfaces.RealOutput_ Volume(final quantity="Volume",
            final unit="ml") "water volume in peritoneum" 
          annotation (Placement(transformation(extent={{-82,24},{-74,32}}),   iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={0,-100})));
        Library.PressureFlow.OneWayResistorWithCondParam peritoneumLymph(cond=1.5, zero=0) 
          annotation (Placement(transformation(extent={{-46,16},{-30,32}})));
        Library.PressureFlow.OneWayResistorWithCondParam HepaticEffusion(cond=0.9, zero=0) 
          annotation (Placement(transformation(extent={{-34,54},{-50,70}})));
        Library.Interfaces.RealOutput_ PeritoneumSpace_Gain(   final quantity="Flow",final unit
            =                                                                                   "ml/min") 
                                                annotation (Placement(transformation(
                extent={{-46,64},{-40,70}}),   iconTransformation(extent={{92,-80},
                  {112,-60}})));
        Library.Interfaces.RealOutput_ PeritoneumSpace_Loss(   final quantity="Flow",final unit
            =                                                                                   "ml/min") 
                                                annotation (Placement(transformation(
                extent={{-16,8},{-10,14}}),    iconTransformation(extent={{92,-80},
                  {112,-100}})));
        Library.Interfaces.RealInput_ SplanchnicVeins_Pressure(     final
            quantity =                                                             "Pressure",final unit
            =                                                                                            "mmHg")
          "pressure in splanchnic veins"                                          annotation (Placement(
              transformation(extent={{-18,68},{-10,76}}),     iconTransformation(
                extent={{-100,50},{-80,70}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure0
          colloidhydraulicPressure0_1 
          annotation (Placement(transformation(extent={{14,54},{6,62}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure0
          colloidhydraulicPressure0_2 
          annotation (Placement(transformation(extent={{14,16},{6,24}})));
        Library.Blocks.PressureConstant ExternalPressure(k=9) 
          annotation (Placement(transformation(extent={{-90,80},{-82,88}})));
        Blood.VascularCompartments.VascularElasticBloodCompartment
          vascularElasticBloodCompartment(initialVol=initialVolume, stateName=
              "PeritoneumSpace.Volume",
          STEADY=false) 
          annotation (Placement(transformation(extent={{-90,34},{-70,54}})));
        Library.Blocks.VolumeConstant volumeConstant(k=0) 
          annotation (Placement(transformation(extent={{-96,56},{-88,64}})));
        Library.Blocks.ComplianceConstant complianceConstant(k=760) 
          annotation (Placement(transformation(extent={{-64,56},{-72,64}})));

      equation
        connect(HepaticEffusion.actualFlow, PeritoneumSpace_Gain)    annotation (Line(
            points={{-48.56,60.4},{-52,60.4},{-52,67},{-43,67}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(peritoneumLymph.actualFlow, PeritoneumSpace_Loss)    annotation (Line(
            points={{-31.44,22.4},{-26,22.4},{-26,11},{-13,11}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flux, colloidhydraulicPressure0_1.q_in) annotation (Line(
            points={{66,40},{22,40},{22,58},{14,58}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(colloidhydraulicPressure0_1.withoutCOP, HepaticEffusion.q_in) 
          annotation (Line(
            points={{6,62},{-34,62}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(colloidhydraulicPressure0_2.q_in, flux) annotation (Line(
            points={{14,20},{22,20},{22,40},{66,40}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(peritoneumLymph.q_out, colloidhydraulicPressure0_2.withoutCOP) 
          annotation (Line(
            points={{-30,24},{6,24}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(vascularElasticBloodCompartment.Vol, Volume) annotation (Line(
            points={{-84,34},{-84,28},{-78,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ExternalPressure.y, vascularElasticBloodCompartment.ExternalPressure) 
          annotation (Line(
            points={{-81.6,84},{-80,84},{-80,54}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(volumeConstant.y, vascularElasticBloodCompartment.V0) annotation (
            Line(
            points={{-87.6,60},{-86,60},{-86,54}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(complianceConstant.y, vascularElasticBloodCompartment.Compliance) 
          annotation (Line(
            points={{-72.4,60},{-74,60},{-74,54}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(vascularElasticBloodCompartment.referencePoint, HepaticEffusion.q_out) 
          annotation (Line(
            points={{-80,44},{-60,44},{-60,62},{-50,62}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(vascularElasticBloodCompartment.referencePoint, peritoneumLymph.q_in) 
          annotation (Line(
            points={{-80,44},{-60,44},{-60,24},{-46,24}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(colloidhydraulicPressure0_1.hydraulicPressure,
          SplanchnicVeins_Pressure) annotation (Line(
            points={{8.4,62.8},{8.4,72},{-14,72}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(colloidhydraulicPressure0_2.hydraulicPressure, ExternalPressure.y) 
          annotation (Line(
            points={{8.4,24.8},{8.4,32},{-28,32},{-28,84},{-81.6,84}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (
      Documentation(info="<HTML>
<PRE>
QHP 2008 / Peritoneum
 
Created : 19-Mar-08
Last Modified : 19-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : QHP 2008
Schema : 2008.0
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),       Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics={
              Text(
                extent={{-100,120},{100,104}},
                lineColor={0,0,255},
                textString="%name"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0}),
              Text(
                extent={{-100,-6},{100,-22}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)")}),
          Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,
                  -100},{100,100}}), graphics));
      end Peritoneum2;

      model GILumen2
        import QHP = HumMod;
      extends QHP.Library.Interfaces.Tissues.GILumen_;


        QHP.Library.Interfaces.RealInput_ GILumenSodium_Mass(final quantity="Mass",final unit
            =                                                                                 "mEq")
          "sodium in gastro intestinal lumen"                                             annotation (Placement(transformation(extent={{-78,58},
                  {-68,68}}),
                         iconTransformation(extent={{-100,90},{-80,110}})));
        QHP.Library.Interfaces.RealInput_ GILumenPotassium_Mass(final quantity="Mass",final unit
            =                                                                                    "mEq")
          "potasium in gastro intestinal lumen"                   annotation (Placement(transformation(
                extent={{-78,50},{-68,60}}),iconTransformation(extent={{-100,30},{-80,
                  50}})));

        Library.Semipermeable.NegativeOsmoticFlow toPlasma(o(unit="g/l")) 
                                                                       annotation (
            Placement(transformation(extent={{80,44},{100,64}}),  iconTransformation(
                extent={{90,-10},{110,10}})));
      parameter Real Fiber_mass(final quantity="Mass", final unit="mOsm") = 0.043;
      parameter Real Na_EqToAllConnectedOsm(final unit="mOsm/mEq") = 2;
      parameter Real K_EqToAllConnectedOsm(final unit="mOsm/mEq") = 2;
      //parameter Real initialVolume(final quantity="Volume", final unit="ml") =  1000;

        Library.Interfaces.RealOutput_ Vol annotation (Placement(transformation(
                extent={{-12,58},{-4,66}}),  iconTransformation(extent={{82,60},{122,100}})));
        QHP.Water.WaterCompartments.GILumen_ GILumen(
          Fiber_mass=Fiber_mass,
          Na_EqToAllConnectedOsm=Na_EqToAllConnectedOsm,
          K_EqToAllConnectedOsm=K_EqToAllConnectedOsm,
          stateName="GILumenVolume.Mass",
          STEADY=false) 
          annotation (Placement(transformation(extent={{-35,44},{-15,64}})));
        Outtake vomitus 
          annotation (Placement(transformation(extent={{-21,32},{-7,46}})));
        Outtake diarrhea 
          annotation (Placement(transformation(extent={{-29,32},{-43,46}})));
        Library.Semipermeable.ResistorWithCondParam absorbtion1(        cond=150) 
          annotation (Placement(transformation(extent={{-8,-8},{8,8}},
              rotation=0,
              origin={12,54})));
        Library.VolumeFlow.InputPump Diet 
          annotation (Placement(transformation(extent={{-65,8},{-55,18}})));
        QHP.Library.Interfaces.RealInput_ OsmBody_Osm_conc_CellWalls(                                final unit
            =                                                                                                  "mOsm/ml") 
                                                     annotation (Placement(
              transformation(extent={{38,68},{48,78}}),      iconTransformation(
                extent={{-100,-30},{-80,-10}})));
        QHP.Library.Semipermeable.FlowMeasure flowMeasure annotation (Placement(
              transformation(
              extent={{-6,6},{6,-6}},
              rotation=0,
              origin={38,52})));
        QHP.Library.Interfaces.RealInput_ OsmBody_Osmoreceptors(final quantity=
              "Osmolarity", final unit="mOsm/L")
          "potasium in gastro intestinal lumen"                   annotation (Placement(transformation(
                extent={{-104,20},{-94,30}}),
                                            iconTransformation(extent={{-100,-90},{-80,
                  -70}})));
        QHP.Library.Factors.SplineValue DietThirst(data={{233,0.0,0},{253,2.0,
              0.2},{313,30.0,0}}) 
          annotation (Placement(transformation(extent={{-70,16},{-50,36}})));
        QHP.Library.Blocks.Constant Constant0(k=1/1.440) 
          annotation (Placement(transformation(extent={{-72,32},{-64,40}})));
        Modelica.Blocks.Math.Gain ML_TO_L(k=1000) 
          annotation (Placement(transformation(extent={{-82,22},{-76,28}})));
        Library.Interfaces.RealOutput_ Absorbtion(unit="ml/min") 
                                           annotation (Placement(transformation(
                extent={{38,38},{46,46}}),   iconTransformation(extent={{82,20},
                  {122,60}})));
        QHP.Library.Semipermeable.OsmoticPump osmoticPump 
          annotation (Placement(transformation(extent={{52,44},{74,64}})));
        Modelica.Blocks.Math.Gain per_ml(k=1) 
          annotation (Placement(transformation(extent={{56,64},{62,70}})));
        Library.Interfaces.RealOutput_ Intake(unit="ml/min") 
                                           annotation (Placement(transformation(
                extent={{-42,16},{-34,24}}), iconTransformation(extent={{80,-100},
                  {120,-60}})));
      equation

        connect(GILumen.outtake,diarrhea. H2OLoss) annotation (Line(
            points={{-25,44},{-25,39},{-29,39}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GILumen.outtake,vomitus. H2OLoss) annotation (Line(
            points={{-25,44},{-25,39},{-21,39}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GILumen.absorbtion, absorbtion1.q_in) 
                                                     annotation (Line(
            points={{-15,54},{4,54}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GILumen.intake,Diet. q_out) annotation (Line(
            points={{-35,50},{-49,50},{-49,13},{-55,13}},
            color={0,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GILumen.Vol, Vol) annotation (Line(
            points={{-14.8,62},{-8,62}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(GILumenSodium_Mass, GILumen.GILumenSodium_Mass) annotation (Line(
            points={{-73,63},{-59.5,63},{-59.5,62},{-34,62}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(GILumenPotassium_Mass, GILumen.GILumenPotassium_Mass) annotation (
            Line(
            points={{-73,55},{-59.5,55},{-59.5,56},{-34,56}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(absorbtion1.q_out, flowMeasure.q_in) annotation (Line(
            points={{20,54},{27.5,54},{27.5,54.04},{32.6,54.04}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(DietThirst.y, Diet.desiredFlow) annotation (Line(
            points={{-60,24},{-60,16}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Constant0.y, DietThirst.yBase) annotation (Line(
            points={{-63.6,36},{-60,36},{-60,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ML_TO_L.y, DietThirst.u) annotation (Line(
            points={{-75.7,25},{-72.85,25},{-72.85,26},{-69.8,26}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(OsmBody_Osmoreceptors, ML_TO_L.u) annotation (Line(
            points={{-99,25},{-82.6,25}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, Absorbtion) 
                                                annotation (Line(
            points={{37.88,47.08},{37.88,42},{42,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.q_out, osmoticPump.q_in) annotation (Line(
            points={{43.28,53.92},{53.4,53.92},{53.4,54},{54.2,54}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(osmoticPump.q_out, toPlasma) annotation (Line(
            points={{71.8,54},{90,54}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(OsmBody_Osm_conc_CellWalls, per_ml.u) annotation (Line(
            points={{43,73},{49.5,73},{49.5,67},{55.4,67}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(per_ml.y, osmoticPump.desiredOsmoles) annotation (Line(
            points={{62.3,67},{63,67},{63,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(DietThirst.y, Intake) annotation (Line(
            points={{-60,24},{-60,20},{-38,20}},
            color={0,0,127},
            smooth=Smooth.None));

          annotation (
      Documentation(info="<HTML>
<PRE>
QHP 2008 / GILumenH2O.REF
 
Created : 13-Mar-08
Last Modified : 13-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : QHP 2008
Schema : 2008.0
 
Volume ========================================
 
Typical lumen volume is 1000 mL on an intake of 1.4 mL/Min.
This number is based on lumen volume being 1.4% body weight
from
 
   JCI 36:289-296, 1957.
 
Data suggests a 1L oral load is nearly all absorbed in
1 Hr.  Average rate would be 17 mL/Hr with a peak rate
of about 30.
 
   Baldes & Smirk. J.Physiol. 82:62,1934.
 
Osmolar gradient is assumed to be 0.200 mOsm/mL.
 
   Rate = Perm * Grad
   30 = Perm * 0.200
   Perm = 150
 
Temperature ===================================
 
Typical lumen heat content is 155 kCal.
 
Specific heat (SH) of H2O is 0.001 (kCal/G)/DegK.
 
To get temperature from heat
 
   DegK = 1000.0 * kCal/G (or mL)
   DegC = DegK - 273.15
   DegF = ( 9 / 5 ) * DegC + 32
 
To get heat from temperature
 
   kCal = 0.001 * DegK * mL
   kCal/Min = 0.001 * DegK * mL/Min
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
        Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-100,-42},{100,-58}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0}),
              Text(
                extent={{-102,124},{98,108}},
                lineColor={0,0,255},
                textString="%name")}));
      end GILumen2;

      model Kidney2
       extends HumMod.Library.Interfaces.Tissues.Kidney;
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-62,72},{-42,92}}),
              iconTransformation(extent={{-106,-10},{-86,10}})));
        Kidney.Nephrons2 Nephron 
          annotation (Placement(transformation(extent={{36,-68},{-9,-30}})));
        Modelica.Blocks.Math.Product product 
          annotation (Placement(transformation(extent={{-20,-16},{-12,-8}})));
        Library.Semipermeable.PositiveOsmoticFlow q_in(o(unit="g/ml")) 
                                                       annotation (Placement(
              transformation(extent={{2,34},{22,54}}),   iconTransformation(extent={{2,34},{
                  22,54}})));
        Library.PressureFlow.NegativePressureFlow urine 
          annotation (Placement(transformation(extent={{-8,-110},{12,-90}}),
              iconTransformation(extent={{-8,-110},{12,-90}})));
        HumMod.Library.Semipermeable.ColloidHydraulicPressure0
          colloidhydraulicPressure0_1 
          annotation (Placement(transformation(extent={{20,40},{28,48}})));
        Library.PressureFlow.Pump kidneyFlow annotation (Placement(transformation(
              extent={{-10,10},{10,-10}},
              rotation=270,
              origin={38,22})));
        Modelica.Blocks.Math.Division conc 
          annotation (Placement(transformation(extent={{-48,-26},{-40,-18}})));
      equation

        connect(busConnector.Kidney_NephronCount_Total_xNormal, Nephron.NephronCount_xNormal) 
          annotation (Line(
            points={{-52,82},{-52,-42.2941},{-7.875,-42.2941}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.PT_Na_FractReab, Nephron.PT_Na_FractReab) 
          annotation (Line(
            points={{-52,82},{-52,-46.7647},{-7.875,-46.7647}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.LH_Na_FractReab, Nephron.LH_Na_FractReab) 
          annotation (Line(
            points={{-52,82},{-52,-49},{-7.875,-49}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.DT_Na_Outflow, Nephron.DT_Na_Outflow) 
          annotation (Line(
            points={{-52,82},{-52,-51.2353},{-7.875,-51.2353}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.CD_Na_Outflow, Nephron.CD_Na_Outflow) 
          annotation (Line(
            points={{-52,82},{-52,-53.4706},{-7.875,-53.4706}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.CD_K_Outflow, Nephron.CD_K_Outflow) 
          annotation (Line(
            points={{-52,82},{-52,-55.7059},{-7.875,-55.7059}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.CD_NH4_Outflow, Nephron.CD_NH4_Outflow) 
          annotation (Line(
            points={{-52,82},{-52,-57.9412},{-7.875,-57.9412}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.CD_Glucose_Outflow, Nephron.CD_Glucose_Outflow) 
          annotation (Line(
            points={{-52,82},{-52,-60.1765},{-7.875,-60.1765}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(busConnector.MedullaNa_conc, Nephron.MedullaNa_conc) 
          annotation (Line(
            points={{-52,82},{-52,-62.4118},{-7.875,-62.4118}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None));
        connect(Nephron.KidneyProtein_massFlow, product.y)          annotation (
           Line(
            points={{-7.875,-40.0588},{-10,-40},{-14,-40},{-14,-30},{-8,-30},{
                -8,-12},{-11.6,-12}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.KidneyPlasmaFlow, product.u1)         annotation (
            Line(
            points={{-52,82},{-52,-9.6},{-20.8,-9.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(conc.y, product.u2)                        annotation (
            Line(
            points={{-39.6,-22},{-36,-22},{-36,-14},{-20.8,-14},{-20.8,-14.4}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="PlasmaProtein_conc",
            index=-1,
            extent={{-6,2},{-6,2}}));
        connect(busConnector.PlasmaProtein_Mass, conc.u1) 
                                            annotation (Line(
            points={{-52,82},{-52,-19.6},{-48.8,-19.6}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Nephron.GFR, busConnector.GlomerulusFiltrate_GFR) 
          annotation (Line(
            points={{38.25,-41.1765},{68,-41.1765},{68,82},{-52,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Nephron.urine, urine) annotation (Line(
            points={{3.15,-68},{3.15,-68.8235},{2,-68.8235},{2,-100}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector.GlomerulusBloodPressure,
          colloidhydraulicPressure0_1.hydraulicPressure) annotation (Line(
            points={{-52,82},{-52,50},{25.6,50},{25.6,48.8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(q_in, colloidhydraulicPressure0_1.q_in) annotation (Line(
            points={{12,44},{20,44}},
            color={127,127,0},
            thickness=1,
            smooth=Smooth.None));
        connect(kidneyFlow.q_in, colloidhydraulicPressure0_1.withoutCOP) 
          annotation (Line(
            points={{38,32},{38,48},{28,48}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector.PlasmaVol, conc.u2) 
          annotation (Line(
            points={{-52,82},{-52,-24.4},{-48.8,-24.4}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.KidneyPlasmaFlow, kidneyFlow.desiredFlow) 
          annotation (Line(
            points={{-52,82},{-52,22},{32,22}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(kidneyFlow.q_out, Nephron.AfferentArtery_Water_Hydraulic) 
          annotation (Line(
            points={{38,12},{38,-29.5529},{15.75,-29.5529}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(colloidhydraulicPressure0_1.withoutCOP, Nephron.EfferentArtery_Water_Hydraulic) 
          annotation (Line(
            points={{28,48},{28,66},{-2,66},{-2,-30},{6.3,-30}},
            color={0,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(Nephron.CD_H2O_Outflow, busConnector.CD_H2O_Outflow) 
          annotation (Line(
            points={{38.25,-65.7647},{68,-65.7647},{68,82},{-52,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Nephron.GFR, busConnector.Glomerulus_GFR) annotation (Line(
            points={{38.25,-41.1765},{68,-41.1765},{68,82},{-52,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Nephron.CD_H2O_Reab, busConnector.CD_H2O_Reab) annotation (Line(
            points={{38.25,-70.2353},{68,-70.2353},{68,82},{-52,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(Nephron.LH_H2O_Outflow, busConnector.LH_H2O_Outflow) annotation (Line(
            points={{29.7,-68},{30,-68},{30,-80},{80,-80},{80,82},{-52,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,-9},{6,-9}}));
        connect(busConnector.NephronADH, Nephron.ADH) annotation (Line(
            points={{-52,82},{-52,-44.5294},{-7.875,-44.5294}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                               graphics), Icon(coordinateSystem(
                preserveAspectRatio=false, extent={{-100,-100},{100,100}}),
              graphics={Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,
                    0,0}), Text(
                extent={{-100,120},{100,104}},
                lineColor={0,0,255},
                textString="%name")}));
      end Kidney2;

      model TorsoExtravascularWater
        extends HumMod.Library.Utilities.DynamicState;
        extends HumMod.Library.Interfaces.Tissues.OtherTissue;
        Library.Semipermeable.NegativeOsmoticFlow q_out(o(final unit="g")) 
                                   annotation (Placement(
              transformation(extent={{62,-32},{102,8}}),  iconTransformation(extent={{-10,-10},
                  {10,10}})));
        parameter Real initialWaterVolume(final quantity="Volume", unit="ml");

        HumMod.Library.Interfaces.RealInput_ NotpermeableSolutes(quantity=
              "Mass", unit="g") 
                        annotation (Placement(transformation(extent={{-120,60},
                  {-80,100}}),
              iconTransformation(extent={{-120,60},{-80,100}})));
        HumMod.Library.Interfaces.RealOutput_ WaterVolume(start=initialWaterVolume, final
            quantity=
              "Volume", unit="ml") 
          annotation (Placement(transformation(extent={{-20,-120},{20,-80}}, rotation=
                 -90)));

        HumMod.Library.Interfaces.RealInput_ ICFV(quantity="Volume", unit="ml")
          "intracellular water volume" 
                        annotation (Placement(transformation(extent={{-120,20},
                  {-80,60}}),
              iconTransformation(extent={{-120,20},{-80,60}})));
        Library.Interfaces.RealOutput_ ECFV "extracellular water" annotation (
            Placement(transformation(extent={{34,-108},{74,-68}}),
              iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={40,-100})));
      /*initial equation 
  if STEADY then
     WaterVolume = initialWaterVolume;
  end if;
*/
      equation
        ECFV = WaterVolume - ICFV;
        q_out.o = if (ECFV>0) then NotpermeableSolutes / ECFV else 0;
      //  der(WaterVolume) = q_out.q / Library.SecPerMin;
        stateValue = WaterVolume;

        changePerMin = q_out.q;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,
                    0,0}), Text(
                extent={{-100,88},{100,68}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<p><i>2009-2010</i></p>
<p>Marek Matejak, Charles University, Prague, Czech Republic </p>
</html>"));
      end TorsoExtravascularWater;

      model GILumen_
        extends HumMod.Library.Utilities.DynamicState;
        extends HumMod.Library.Interfaces.Tissues.GILumen_;


        HumMod.Library.Interfaces.RealInput_ GILumenSodium_Mass(final quantity=
              "Mass", final unit="mEq") "sodium in gastro intestinal lumen"               annotation (Placement(transformation(extent={{120,20},
                  {80,60}}),
                         iconTransformation(extent={{-100,70},{-80,90}})));
        HumMod.Library.Interfaces.RealInput_ GILumenPotassium_Mass(final
            quantity="Mass", final unit="mEq")
          "potasium in gastro intestinal lumen"                   annotation (Placement(transformation(
                extent={{120,60},{80,100}}),iconTransformation(extent={{-100,10},{-80,
                  30}})));

        Library.Semipermeable.NegativeOsmoticFlow absorbtion           annotation (
            Placement(transformation(extent={{60,-20},{100,20}}), iconTransformation(
                extent={{90,-10},{110,10}})));
        Library.VolumeFlow.NegativeVolumeFlow outtake annotation (Placement(
              transformation(extent={{-20,-120},{20,-80}}), iconTransformation(extent=
                 {{-20,-120},{20,-80}})));
        Library.VolumeFlow.PositiveVolumeFlow intake 
          annotation (Placement(transformation(extent={{-120,-60},{-80,-20}}),
              iconTransformation(extent={{-120,-60},{-80,-20}})));
      parameter Real Fiber_mass(final quantity="Mass", final unit="mOsm") = 0.043;
      parameter Real Na_EqToAllConnectedOsm(final unit="mOsm/mEq") = 2;
      parameter Real K_EqToAllConnectedOsm(final unit="mOsm/mEq") = 2;
      parameter Real initialVolume(final quantity="Volume", final unit="ml") = 949.201;
      Real mass(final quantity="Volume",final unit="ml",start = initialVolume)
          "water volume in gastro intestinal lumen";
      Real OsmNa;
      Real OsmK;
      Real Fiber;

        Library.Interfaces.RealOutput_ Vol annotation (Placement(transformation(
                extent={{82,60},{122,100}}), iconTransformation(extent={{82,60},{122,100}})));
      equation
      //   der(mass) = (intake.q + absorbtion.q + outtake.q)/Library.SecPerMin;

         intake.volume = mass;
         OsmNa=Na_EqToAllConnectedOsm*GILumenSodium_Mass/mass;
         OsmK = K_EqToAllConnectedOsm*GILumenPotassium_Mass/mass;
         Fiber =Fiber_mass; ///mass;
         absorbtion.o = (Fiber + OsmNa + OsmK);
         outtake.volume = mass;

         Vol = mass;

         stateValue = mass;
         changePerMin = intake.q + absorbtion.q + outtake.q;
          annotation (
      Documentation(info="<HTML>
<PRE>
QHP 2008 / GILumenH2O.REF
 
Created : 13-Mar-08
Last Modified : 13-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : QHP 2008
Schema : 2008.0
 
Volume ========================================
 
Typical lumen volume is 1000 mL on an intake of 1.4 mL/Min.
This number is based on lumen volume being 1.4% body weight
from
 
   JCI 36:289-296, 1957.
 
Data suggests a 1L oral load is nearly all absorbed in
1 Hr.  Average rate would be 17 mL/Hr with a peak rate
of about 30.
 
   Baldes & Smirk. J.Physiol. 82:62,1934.
 
Osmolar gradient is assumed to be 0.200 mOsm/mL.
 
   Rate = Perm * Grad
   30 = Perm * 0.200
   Perm = 150
 
Temperature ===================================
 
Typical lumen heat content is 155 kCal.
 
Specific heat (SH) of H2O is 0.001 (kCal/G)/DegK.
 
To get temperature from heat
 
   DegK = 1000.0 * kCal/G (or mL)
   DegC = DegK - 273.15
   DegF = ( 9 / 5 ) * DegC + 32
 
To get heat from temperature
 
   kCal = 0.001 * DegK * mL
   kCal/Min = 0.001 * DegK * mL/Min
</PRE>
</HTML>
",     revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
        Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},
                  {100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={
              Text(
                extent={{-100,-10},{100,-26}},
                lineColor={0,0,255},
                textString="(initial %initialVolume ml)"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0}),
              Text(
                extent={{-100,120},{100,104}},
                lineColor={0,0,255},
                textString="%name")}));
      end GILumen_;

      model Bladder
        extends Library.PressureFlow.VolumeCompartement;
        extends HumMod.Library.Interfaces.Tissues.Bladder;
        annotation (Icon(graphics={Text(
                extent={{-100,96},{100,80}},
                lineColor={0,0,255},
                textString="%name")}));
      end Bladder;
    end WaterCompartments;

    package TissuesVolume
      "Division of intracellular and interstitial water into tissues"

      model Tissue
        "compute tissue size from global interstitial and cell H20 volume"
        //extends Library.Interfaces.BaseModel;

        parameter Real FractIFV;
        parameter Real FractOrganH2O;

        Library.Interfaces.RealInput_ InterstitialWater_Vol(final unit="ml") 
          annotation (Placement(transformation(extent={{-120,60},{-80,100}})));
        Library.Interfaces.RealInput_ CellH2O_Vol(final unit="ml") 
          annotation (Placement(transformation(extent={{-120,20},{-80,60}})));
        Library.Interfaces.RealOutput_ LiquidVol(final unit="ml")
          "all tissue water volume" 
          annotation (Placement(transformation(extent={{80,60},{120,100}}),
              iconTransformation(extent={{80,60},{120,100}})));
        Library.Interfaces.RealOutput_ OrganH2O(final unit="ml")
          "tissue cells water volume" 
          annotation (Placement(transformation(extent={{80,20},{120,60}}),
              iconTransformation(extent={{80,20},{120,60}})));

        Library.Interfaces.RealOutput_ InterstitialWater(final unit="ml")
          "tissue interstitial water volume" 
          annotation (Placement(transformation(extent={{80,20},{120,60}}),
              iconTransformation(extent={{80,-20},{120,20}})));
      equation
        OrganH2O = FractOrganH2O * CellH2O_Vol;
        LiquidVol = FractIFV*InterstitialWater_Vol + OrganH2O;
        InterstitialWater = LiquidVol - OrganH2O;
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),
                               graphics), Icon(coordinateSystem(preserveAspectRatio=false,
                        extent={{-100,-100},{100,100}}), graphics={
              Text(
                extent={{-72,64},{64,96}},
                lineColor={0,0,0},
                fillPattern=FillPattern.VerticalCylinder,
                fillColor={215,215,215},
                textString="IST .. %FractIFV"),
              Text(
                extent={{-72,26},{64,58}},
                lineColor={0,0,0},
                fillPattern=FillPattern.VerticalCylinder,
                fillColor={215,215,215},
                textString="ICF .. %FractOrganH2O"),
              Text(
                extent={{-98,-104},{102,-120}},
                lineColor={0,0,255},
                textString="%name"),
              Rectangle(extent={{-100,100},{100,-100}}, lineColor={0,0,0})}));
      end Tissue;

      model Tissues
        import QHP = HumMod;
       extends Library.Interfaces.Tissues.Tissues;
        SkeletalMuscle skeletalMuscle(FractIFV=0.597041124206978, FractOrganH2O=0.597041124206978) 
          annotation (Placement(transformation(extent={{-54,62},{-34,82}})));
        Bone bone(FractIFV=7.5535107692334e-002, FractOrganH2O=7.5535107692334e-002) 
          annotation (Placement(transformation(extent={{-54,36},{-34,56}})));
        Fat fat(FractIFV=6.86793125415955e-002, FractOrganH2O=6.86793125415955e-002) 
          annotation (Placement(transformation(extent={{-54,-54},{-34,-34}})));
        Brain brain(FractIFV=2.13850278381526e-002, FractOrganH2O=2.13850278381526e-002) 
          annotation (Placement(transformation(extent={{52,28},{72,48}})));
        QHP.Water.TissuesVolume.RightHeart rightHeart(FractIFV=7.11479063056408e-004,
            FractOrganH2O=7.11479063056408e-004) 
          annotation (Placement(transformation(extent={{52,-92},{72,-72}})));
        RespiratoryMuscle respiratoryMuscle(FractIFV=6.71126519181567e-002,
            FractOrganH2O=6.71126519181567e-002) 
          annotation (Placement(transformation(extent={{-54,-24},{-34,-4}})));
        OtherTissue otherTissue(FractIFV=6.70823116596042e-002, FractOrganH2O=6.70823116596042e-002) 
          annotation (Placement(transformation(extent={{-54,8},{-34,28}})));
        Liver liver(FractIFV=2.84998184687167e-002, FractOrganH2O=2.84998184687167e-002) 
          annotation (Placement(transformation(extent={{52,58},{72,78}})));
        QHP.Water.TissuesVolume.LeftHeart leftHeart(FractIFV=4.26887437833845e-003,
            FractOrganH2O=4.26887437833845e-003) 
          annotation (Placement(transformation(extent={{52,-62},{72,-42}})));
        QHP.Water.TissuesVolume.Kidney kidney(FractIFV=4.71608978940247e-003,
            FractOrganH2O=4.71608978940247e-003) 
          annotation (Placement(transformation(extent={{52,-34},{72,-14}})));
        QHP.Water.TissuesVolume.GITract GITract(FractIFV=2.34991370540916e-002,
            FractOrganH2O=2.34991370540916e-002) 
          annotation (Placement(transformation(extent={{52,-2},{72,18}})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-108,72},{-68,112}})));
        Skin skin(FractIFV=4.14690653895735e-002, FractOrganH2O=4.14690653895735e-002) 
          annotation (Placement(transformation(extent={{-54,-84},{-34,-64}})));

      //Real ifv;
      //Real organ;
      equation

        connect(busConnector.InterstitialWater_Vol, bone.InterstitialWater_Vol)      annotation (
           Line(
            points={{-88,92},{-88,54},{-54,54}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));

        connect(busConnector.InterstitialWater_Vol, brain.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,46},{52,46}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, fat.InterstitialWater_Vol)      annotation (
            Line(
            points={{-88,92},{-88,-36},{-54,-36}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, GITract.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,16},{52,16}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, kidney.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-16},{52,-16}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, leftHeart.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-44},{52,-44}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, liver.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,76},{52,76}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, otherTissue.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{-88,26},{-54,26}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, respiratoryMuscle.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{-88,-6},{-54,-6}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, rightHeart.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-74},{52,-74}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, skin.InterstitialWater_Vol)      annotation (
           Line(
            points={{-88,92},{-88,-66},{-54,-66}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.InterstitialWater_Vol, skeletalMuscle.InterstitialWater_Vol) 
          annotation (Line(
            points={{-88,92},{-87.5,92},{-87.5,80},{-54,80}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));

        connect(busConnector.CellH2O_Vol, bone.CellH2O_Vol)         annotation (
           Line(
            points={{-88,92},{-88,50},{-54,50}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, brain.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,42},{52,42}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, fat.CellH2O_Vol)         annotation (
            Line(
            points={{-88,92},{-88,-40},{-54,-40}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, GITract.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,12},{52,12}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, kidney.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-20},{52,-20}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, leftHeart.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-48},{52,-48}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, liver.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,72},{52,72}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, otherTissue.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{-88,22},{-54,22}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, respiratoryMuscle.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{-88,-10},{-54,-10}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, rightHeart.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{6,92},{6,-78},{52,-78}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, skin.CellH2O_Vol)         annotation (
           Line(
            points={{-88,92},{-88,-70},{-54,-70}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));
        connect(busConnector.CellH2O_Vol, skeletalMuscle.CellH2O_Vol) 
          annotation (Line(
            points={{-88,92},{-87.5,92},{-87.5,76},{-54,76}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-5,2},{-5,2}}));

        connect(bone.LiquidVol, busConnector.bone_LiquidVol)       annotation (
           Line(
            points={{-34,54},{6,54},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(brain.LiquidVol, busConnector.brain_LiquidVol) 
          annotation (Line(
            points={{72,46},{98,46},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(fat.LiquidVol, busConnector.fat_LiquidVol)     annotation (
            Line(
            points={{-34,-36},{6,-36},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(GITract.LiquidVol, busConnector.GITract_LiquidVol) 
          annotation (Line(
            points={{72,16},{98,16},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(kidney.LiquidVol, busConnector.kidney_LiquidVol) 
          annotation (Line(
            points={{72,-16},{98,-16},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(leftHeart.LiquidVol, busConnector.leftHeart_LiquidVol) 
          annotation (Line(
            points={{72,-44},{98,-44},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(liver.LiquidVol, busConnector.liver_LiquidVol) 
          annotation (Line(
            points={{72,76},{98,76},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(otherTissue.LiquidVol, busConnector.otherTissue_LiquidVol) 
          annotation (Line(
            points={{-34,26},{6,26},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(respiratoryMuscle.LiquidVol, busConnector.respiratoryMuscle_LiquidVol) 
          annotation (Line(
            points={{-34,-6},{6,-6},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(rightHeart.LiquidVol, busConnector.rightHeart_LiquidVol) 
          annotation (Line(
            points={{72,-74},{98,-74},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skin.LiquidVol, busConnector.skin_LiquidVol)     annotation (
           Line(
            points={{-34,-66},{6,-66},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skeletalMuscle.LiquidVol, busConnector.skeletalMuscle_LiquidVol) 
          annotation (Line(
            points={{-34,80},{5.5,80},{5.5,92},{-88,92}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));

       connect(bone.LiquidVol, busConnector.Bone_LiquidVol)       annotation (
           Line(
            points={{-34,54},{6,54},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(brain.LiquidVol, busConnector.Brain_LiquidVol) 
          annotation (Line(
            points={{72,46},{98,46},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(fat.LiquidVol, busConnector.Fat_LiquidVol)     annotation (
            Line(
            points={{-34,-36},{6,-36},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(kidney.LiquidVol, busConnector.Kidney_LiquidVol) 
          annotation (Line(
            points={{72,-16},{98,-16},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(leftHeart.LiquidVol, busConnector.LeftHeart_LiquidVol) 
          annotation (Line(
            points={{72,-44},{98,-44},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(liver.LiquidVol, busConnector.Liver_LiquidVol) 
          annotation (Line(
            points={{72,76},{98,76},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(otherTissue.LiquidVol, busConnector.OtherTissue_LiquidVol) 
          annotation (Line(
            points={{-34,26},{6,26},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(respiratoryMuscle.LiquidVol, busConnector.RespiratoryMuscle_LiquidVol) 
          annotation (Line(
            points={{-34,-6},{6,-6},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(rightHeart.LiquidVol, busConnector.RightHeart_LiquidVol) 
          annotation (Line(
            points={{72,-74},{98,-74},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skin.LiquidVol, busConnector.Skin_LiquidVol)     annotation (
           Line(
            points={{-34,-66},{6,-66},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skeletalMuscle.LiquidVol, busConnector.SkeletalMuscle_LiquidVol) 
          annotation (Line(
            points={{-34,80},{5.5,80},{5.5,92},{-88,92}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));

        connect(bone.OrganH2O, busConnector.bone_CellH2OVol)       annotation (
           Line(
            points={{-34,50},{6,50},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(brain.OrganH2O, busConnector.brain_CellH2OVol) 
          annotation (Line(
            points={{72,42},{98,42},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(fat.OrganH2O, busConnector.fat_CellH2OVol)     annotation (
            Line(
            points={{-34,-40},{6,-40},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(GITract.OrganH2O, busConnector.GITract_CellH2OVol) 
          annotation (Line(
            points={{72,12},{98,12},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(kidney.OrganH2O, busConnector.kidney_CellH2OVol) 
          annotation (Line(
            points={{72,-20},{98,-20},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(leftHeart.OrganH2O, busConnector.leftHeart_CellH2OVol) 
          annotation (Line(
            points={{72,-48},{98,-48},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(liver.OrganH2O, busConnector.liver_CellH2OVol) 
          annotation (Line(
            points={{72,72},{98,72},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(otherTissue.OrganH2O, busConnector.otherTissue_CellH2OVol) 
          annotation (Line(
            points={{-34,22},{6,22},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(respiratoryMuscle.OrganH2O, busConnector.respiratoryMuscle_CellH2OVol) 
          annotation (Line(
            points={{-34,-10},{6,-10},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(rightHeart.OrganH2O, busConnector.rightHeart_CellH2OVol) 
          annotation (Line(
            points={{72,-78},{98,-78},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skin.OrganH2O, busConnector.skin_CellH2OVol)     annotation (
           Line(
            points={{-34,-70},{6,-70},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skeletalMuscle.OrganH2O, busConnector.skeletalMuscle_CellH2OVol) 
          annotation (Line(
            points={{-34,76},{5.5,76},{5.5,92},{-88,92}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));

       connect(bone.InterstitialWater, busConnector.bone_InterstitialWater)       annotation (
           Line(
            points={{-34,46},{6,46},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(brain.InterstitialWater, busConnector.brain_InterstitialWater) 
          annotation (Line(
            points={{72,38},{98,38},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(fat.InterstitialWater, busConnector.fat_InterstitialWater)     annotation (
            Line(
            points={{-34,-44},{6,-44},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(GITract.InterstitialWater, busConnector.GITract_InterstitialWater) 
          annotation (Line(
            points={{72,8},{98,8},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(kidney.InterstitialWater, busConnector.kidney_InterstitialWater) 
          annotation (Line(
            points={{72,-24},{98,-24},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(leftHeart.InterstitialWater, busConnector.leftHeart_InterstitialWater) 
          annotation (Line(
            points={{72,-52},{98,-52},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(liver.InterstitialWater, busConnector.liver_InterstitialWater) 
          annotation (Line(
            points={{72,68},{98,68},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(otherTissue.InterstitialWater, busConnector.otherTissue_InterstitialWater) 
          annotation (Line(
            points={{-34,18},{6,18},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(respiratoryMuscle.InterstitialWater, busConnector.respiratoryMuscle_InterstitialWater) 
          annotation (Line(
            points={{-34,-14},{6,-14},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(rightHeart.InterstitialWater, busConnector.rightHeart_InterstitialWater) 
          annotation (Line(
            points={{72,-82},{98,-82},{98,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skin.InterstitialWater, busConnector.skin_InterstitialWater)     annotation (
           Line(
            points={{-34,-74},{6,-74},{6,92},{-88,92}},
            color={0,127,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
        connect(skeletalMuscle.InterstitialWater, busConnector.skeletalMuscle_InterstitialWater) 
          annotation (Line(
            points={{-34,72},{5.5,72},{5.5,92},{-88,92}},
            color={127,0,0},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{5,2},{5,2}}));
      //ifv=bone.FractIFV +brain.FractIFV +fat.FractIFV +GITract.FractIFV +kidney.FractIFV +leftHeart.FractIFV +liver.FractIFV +otherTissue.FractIFV +respiratoryMuscle.FractIFV +rightHeart.FractIFV +skin.FractIFV +skeletalMuscle.FractIFV;
      //organ=bone.FractOrganH2O +brain.FractOrganH2O +fat.FractOrganH2O +GITract.FractOrganH2O +kidney.FractOrganH2O +leftHeart.FractOrganH2O +liver.FractOrganH2O +otherTissue.FractOrganH2O +respiratoryMuscle.FractOrganH2O +rightHeart.FractOrganH2O +skin.FractOrganH2O +skeletalMuscle.FractOrganH2O;
      /*
 assert(bone.FractIFV +
brain.FractIFV +
fat.FractIFV +
GITract.FractIFV +
kidney.FractIFV +
leftHeart.FractIFV +
liver.FractIFV +
otherTissue.FractIFV +
respiratoryMuscle.FractIFV +
rightHeart.FractIFV +
skin.FractIFV +
skeletalMuscle.FractIFV <> 1, "Water.TissuesVolume.Tissues: Sum of FractIFV is not 1!");
 assert(bone.FractOrganH2O +
brain.FractOrganH2O +
fat.FractOrganH2O +
GITract.FractOrganH2O +
kidney.FractOrganH2O +
leftHeart.FractOrganH2O +
liver.FractOrganH2O +
otherTissue.FractOrganH2O +
respiratoryMuscle.FractOrganH2O +
rightHeart.FractOrganH2O +
skin.FractOrganH2O +
skeletalMuscle.FractOrganH2O <> 1, "Water.TissuesVolume.Tissues: Sum of FractOrganH20 is not 1!");
*/
        annotation (Icon(graphics={Text(
                extent={{-100,-104},{100,-120}},
                lineColor={0,0,255},
                textString="%name")}),
                                 Diagram(coordinateSystem(preserveAspectRatio=true,
                        extent={{-100,-100},{100,100}}), graphics));
      end Tissues;

      model SkeletalMuscle
        extends HumMod.Library.Interfaces.Tissues.SkeletalMuscle;
        extends Tissue;
      end SkeletalMuscle;

      model Bone
        extends HumMod.Library.Interfaces.Tissues.Bone;
        extends Tissue;
      end Bone;

      model OtherTissue
        extends HumMod.Library.Interfaces.Tissues.OtherTissue;
        extends Tissue;
      end OtherTissue;

      model RespiratoryMuscle
        extends HumMod.Library.Interfaces.Tissues.RespiratoryMuscle;
        extends Tissue;
      end RespiratoryMuscle;

      model Fat
        extends HumMod.Library.Interfaces.Tissues.Fat;
        extends Tissue;
      end Fat;

      model Skin
        extends HumMod.Library.Interfaces.Tissues.Skin;
        extends Tissue;
      end Skin;

      model Liver
        extends HumMod.Library.Interfaces.Tissues.Liver;
        extends Tissue;
      end Liver;

      model Brain
        extends HumMod.Library.Interfaces.Tissues.Brain;
        extends Tissue;
      end Brain;

      model GITract
        extends HumMod.Library.Interfaces.Tissues.GITract;
        extends Tissue;
      end GITract;

      model Kidney
        extends HumMod.Library.Interfaces.Tissues.Kidney;
        extends Tissue;
      end Kidney;

      model LeftHeart
        extends HumMod.Library.Interfaces.Tissues.LeftHeart;
        extends Tissue;
      end LeftHeart;

      model RightHeart
        extends HumMod.Library.Interfaces.Tissues.RightHeart;
        extends Tissue;
      end RightHeart;
    end TissuesVolume;

    model WaterProperties
    // extends Library.Interfaces.BaseModel;
      Osmoles.OsmBody osmBody "intra/extracellular water" 
        annotation (Placement(transformation(extent={{-84,-22},{-64,-2}})));
      Hydrostatics.Hydrostatics hydrostatics 
        annotation (Placement(transformation(extent={{-62,62},{-38,86}})));
      Modelica.Blocks.Math.Feedback sub 
        annotation (Placement(transformation(extent={{-4,4},{4,-4}},
            rotation=90,
            origin={-34,4})));
      Library.Blocks.Parts CellH2O(         nout=3, w={.2,.5,.3}) 
        annotation (Placement(transformation(extent={{-28,22},{-16,34}})));
      Modelica.Blocks.Math.Sum BodyH2O(nin=5) 
        annotation (Placement(transformation(extent={{32,-30},{24,-22}})));
      Library.Interfaces.BusConnector busConnector 
        annotation (Placement(transformation(extent={{-12,34},{0,46}}),
            iconTransformation(extent={{-118,62},{-80,100}})));
      TissuesVolume.Tissues tissues 
        annotation (Placement(transformation(extent={{38,-92},{58,-72}})));
      Modelica.Blocks.Math.Sum InterstitialWater(nin=3) 
        annotation (Placement(transformation(extent={{42,-48},{34,-40}})));
    equation
      connect(hydrostatics.RegionalPressure_UpperCapy, busConnector.RegionalPressure_UpperCapy) 
        annotation (Line(
          points={{-38,77.6},{-6,77.6},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(hydrostatics.RegionalPressure_MiddleCapy, busConnector.RegionalPressure_MiddleCapy) 
        annotation (Line(
          points={{-38,72.8},{-6,72.8},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(hydrostatics.RegionalPressure_LowerCapy, busConnector.RegionalPressure_LowerCapy) 
        annotation (Line(
          points={{-38,68},{-6,68},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(osmBody.ICFV, sub.u1)      annotation (Line(
          points={{-63.6,-4},{-34,-4},{-34,0.8}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(sub.y, CellH2O.u)      annotation (Line(
          points={{-34,7.6},{-34,28},{-29.2,28}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(CellH2O.y[1], busConnector.UT_Cell_H2O)  annotation (Line(
          points={{-15.4,27.6},{-12,24},{-6,24},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(CellH2O.y[2], busConnector.MT_Cell_H2O)  annotation (Line(
          points={{-15.4,28},{-6,28},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(CellH2O.y[3], busConnector.LT_Cell_H2O)  annotation (Line(
          points={{-15.4,28.4},{-12,32},{-6,32},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(BodyH2O.y,osmBody. BodyH2O_Vol) annotation (Line(
          points={{23.6,-26},{-96,-26},{-96,-22},{-82.8,-22}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(osmBody.OsmBody_Osm_conc_CellWalls, busConnector.OsmBody_Osm_conc_CellWalls) 
        annotation (Line(
          points={{-63.6,-16},{-6,-16},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(busConnector.Status_Posture, hydrostatics.Status_Posture) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,83.6},{-60.8,83.6}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.SystemicArtys_Pressure, hydrostatics.SystemicArtys_Pressure) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,71.6},{-60.8,71.6}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.RightAtrium_Pressure, hydrostatics.RightAtrium_Pressure) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,76.4},{-60.8,76.4}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.LegMusclePump_Effect, hydrostatics.LegMusclePump_Effect) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,64.4},{-60.8,64.4}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.OsmECFV_Electrolytes, osmBody.OsmECFV_Electrolytes) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,-2},{-82.8,-2}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.OsmCell_Electrolytes, osmBody.OsmCell_Electrolytes) 
        annotation (Line(
          points={{-6,40},{-86,40},{-86,-6},{-82.8,-6}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.RBCH2O_Vol, sub.u2)              annotation (Line(
          points={{-6,40},{-86,40},{-86,4},{-37.2,4}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));

      connect(busConnector, tissues.busConnector)         annotation (Line(
          points={{-6,40},{-6,-74},{39.2,-74},{39.2,-72.8}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(sub.y, busConnector.CellH2O_Vol)              annotation (Line(
          points={{-34,7.6},{-34,40},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(InterstitialWater.y, busConnector.InterstitialWater_Vol) 
        annotation (Line(
          points={{33.6,-44},{-6,-44},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(osmBody.ECFV, busConnector.ECFV_Vol)         annotation (Line(
          points={{-63.6,-10},{-6,-10},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));

      connect(BodyH2O.y, busConnector.BodyH2O_Vol) annotation (Line(
          points={{23.6,-26},{-6,-26},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(osmBody.ICFV, busConnector.ICFV_Vol) annotation (Line(
          points={{-63.6,-4},{-6,-4},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(osmBody.Osmoreceptors, busConnector.Osmreceptors) annotation (
          Line(
          points={{-63.6,-20},{-6,-20},{-6,40}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(busConnector.UreaECF_Osmoles, osmBody.UreaECF) annotation (Line(
          points={{-6,40},{-86,40},{-86,-10},{-82.8,-10}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.UreaICF_Osmoles, osmBody.UreaICF) annotation (Line(
          points={{-6,40},{-86,40},{-86,-14},{-82.8,-14}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.GlucoseECF_Osmoles, osmBody.GlucoseECF) annotation (Line(
          points={{-6,40},{-86,40},{-86,-18},{-82.8,-18}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));

      connect(busConnector.PlasmaVol,BodyH2O. u[1])   annotation (Line(
          points={{-6,40},{100,40},{100,-26},{32.8,-26},{32.8,-26.64}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(busConnector.LT_H2O_Vol,BodyH2O. u[2])  annotation (Line(
          points={{-6,40},{100,40},{100,-26.32},{32.8,-26.32}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(busConnector.MT_H2O_Vol,BodyH2O. u[3])  annotation (Line(
          points={{-6,40},{100,40},{100,-26},{32.8,-26}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(busConnector.UT_H2O_Vol,BodyH2O. u[4])  annotation (Line(
          points={{-6,40},{100,40},{100,-25.68},{32.8,-25.68}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(busConnector.RBCH2O_Vol, BodyH2O.u[5]) annotation (Line(
          points={{-6,40},{88,40},{88,-25.36},{32.8,-25.36}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
    /*  connect(busConnector.ExcessLungWater_Volume, BodyH2O.u[6]) annotation (Line(
      points={{-6,40},{88,40},{88,-25.5429},{32.8,-25.5429}},
      color={0,0,255},
      thickness=0.5,
      smooth=Smooth.None));
  connect(busConnector.PeritoneumSpace_Vol, BodyH2O.u[7]) annotation (Line(
      points={{-6,40},{88,40},{88,-25.3143},{32.8,-25.3143}},
      color={0,0,255},
      thickness=0.5,
      smooth=Smooth.None));
*/
      connect(busConnector.UT_InterstitialWater_Vol, InterstitialWater.u[1]) 
        annotation (Line(
          points={{-6,40},{88,40},{88,-44.5333},{42.8,-44.5333}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.MT_InterstitialWater_Vol, InterstitialWater.u[2]) 
        annotation (Line(
          points={{-6,40},{88,40},{88,-44},{42.8,-44}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(busConnector.LT_InterstitialWater_Vol, InterstitialWater.u[3]) 
        annotation (Line(
          points={{-6,40},{88,40},{88,-43.4667},{42.8,-43.4667}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));

      annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                -100},{100,100}}),      graphics), Icon(coordinateSystem(
              preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
            graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                  "icons/water.png"), Text(
              extent={{-112,-82},{130,-116}},
              lineColor={0,0,255},
              textString="%name")}),
        Documentation(revisions="<html>
<table cellspacing=\"2\" cellpadding=\"0\" border=\"0\"><tr>
<td><p>Author:</p></td>
<td><p>Marek Matejak</p></td>
</tr>
<tr>
<td><p>Design:</p></td>
<td><p>Zuzana Rubaninska</p></td>
</tr>
<tr>
<td><p>Copyright:</p></td>
<td><p>In public domains</p></td>
</tr>
<tr>
<td><p>By:</p></td>
<td><p>Charles University, Prague</p></td>
</tr>
<tr>
<td><p>Date of:</p></td>
<td><p>2009</p></td>
</tr>
<tr>
<td><p>References:</p></td>
<td><p>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</p></td>
</tr>
</table>
</html>",     info="<html>
<p>Distribution of H2O in whole body:</p>
<p><ul>
<li>blood plasma</li>
<li>erythrocytes</li>
<li>upper torso interstitium</li>
<li>midle torso interstitium</li>
<li>lower torso interstitium</li>
<li>intracellular water (constant fraction coeficient in different tissues(torsos))</li>
</ul></p>
<p><br/>Water redistribution flows:</p>
<p><ul>
<li>semipermeable capilary walls (plasma - GIT/interstitium/glomerulus filtrate)</li>
<li>lymph from upper/lower/midle torso</li>
<li>kidney nephron filtration/reabsorbtion</li>
</ul></p>
<p><br/>driving by hydraulic, hydrostatic and coloid osmotic pressures.</p>
</html>"));
    end WaterProperties;

    model Water
      import QHP = HumMod;
      //extends QHP.Library.Interfaces.BaseModel;

      QHP.Water.WaterCompartments.Plasma2 plasma(initialVolume=3020.4,
          stateName="PlasmaVol.Vol") 
        annotation (Placement(transformation(extent={{-31,31},{-11,51}})));
      QHP.Library.Semipermeable.OutputPump Hemorrhage 
        annotation (Placement(transformation(extent={{32,-82},{47,-67}})));
      QHP.Library.Semipermeable.InputPump Transfusion 
        annotation (Placement(transformation(extent={{66,-88},{51,-73}})));
      QHP.Library.Semipermeable.InputPump IVDrip 
        annotation (Placement(transformation(extent={{82,-95},{67,-80}})));
      QHP.Library.Interfaces.BusConnector busConnector 
        annotation (Placement(transformation(extent={{-106,81},{-86,101}}),
            iconTransformation(extent={{60,60},{80,80}})));
      QHP.Water.WaterCompartments.Lungs Lungs 
        annotation (Placement(transformation(extent={{-10,-43},{-30,-23}})));
      QHP.Water.WaterCompartments.Torso2 UpperTorso(
        CapilaryWallPermeability=1,
        NormalLymphFlowRate=0.4,
        PressureVolume={{600.0,-30.0,0.01},{2000.0,-4.8,0.0004},{5000.0,0.0,
            0.0004},{12000.0,50.0,0.01}},
        InterstitialPressureOnLymphFlow={{-14.0,0.0,0.0},{-4.0,1.0,0.1},{2.0,
            8.0,4.0},{6.0,25.0,0.0}},
        H2O(stateName="UT_H2O.Vol"),
        initialVolume=8473) 
        annotation (Placement(transformation(extent={{33,35},{53,55}})));
      QHP.Water.WaterCompartments.Torso2 MiddleTorso(
        CapilaryWallPermeability=0.44,
        NormalLymphFlowRate=0.8,
        PressureVolume={{1200.0,-30.0,0.01},{4800.0,-4.8,0.0004},{12000.0,0.0,
            0.0004},{24000.0,50.0,0.01}},
        InterstitialPressureOnLymphFlow={{-14.0,0.0,0.0},{-4.0,1.0,0.1},{2.0,
            8.0,4.0},{6.0,25.0,0.0}},
        H2O(stateName="MT_H2O.Vol"),
        initialVolume=18613) 
        annotation (Placement(transformation(extent={{32,-13},{52,7}})));
      QHP.Water.WaterCompartments.Torso2 LowerTorso(
        CapilaryWallPermeability=0.27,
        NormalLymphFlowRate=1.3,
        PressureVolume={{600.0,-30.0,0.02},{3000.0,-4.8,0.0004},{4000.0,-4.0,
            0.0004},{6000.0,50.0,0.03}},
        InterstitialPressureOnLymphFlow={{-14.0,0.0,0.0},{-4.0,1.0,0.1},{2.0,
            8.0,4.0},{6.0,25.0,0.0}},
        initialVolume=10876.6,
        H2O(stateName="LT_H2O.Vol")) 
        annotation (Placement(transformation(extent={{32,-50},{52,-30}})));
      QHP.Water.WaterCompartments.Peritoneum2 Peritoneum 
        annotation (Placement(transformation(extent={{-69,-68},{-49,-48}})));
      QHP.Water.WaterCompartments.GILumen2 GILumen 
        annotation (Placement(transformation(extent={{-67,61},{-47,81}})));
      QHP.Water.WaterCompartments.Kidney2 Kidney 
        annotation (Placement(transformation(extent={{-67,10},{-47,30}})));
      QHP.Water.WaterCompartments.Bladder Bladder(
        stateName="BladderVolume.Mass",
        STEADY=false,
        initialVolume=200) 
        annotation (Placement(transformation(extent={{-47,-14},{-27,6}})));
      QHP.Library.PressureFlow.OutputPump bladderVoidFlow 
        annotation (Placement(transformation(extent={{-66,-17},{-81,-2}})));
      QHP.Library.Blocks.Parts MetabolicH2O(nout=3, w={25,50,25}) 
        annotation (Placement(transformation(extent={{70,80},{79,89}})));
      QHP.Library.Blocks.Parts HeatSweating(nout=3) 
        annotation (Placement(transformation(extent={{56,74},{65,83}})));
      QHP.Library.Blocks.Parts HeatInsensibleSkin(nout=3) 
        annotation (Placement(transformation(extent={{42,67},{51,76}})));
      Modelica.Blocks.Math.Add insensible 
        annotation (Placement(transformation(extent={{64,13},{57,20}})));
      QHP.Water.WaterProperties waterProperties 
        annotation (Placement(transformation(extent={{-83,-94},{-63,-74}})));
    equation

      connect(IVDrip.desiredFlow, busConnector.IVDrip_H2ORate)         annotation (
          Line(
          points={{74.5,-84.5},{97,-84.5},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(Transfusion.desiredFlow, busConnector.Transfusion_PlasmaRate) 
        annotation (Line(
          points={{58.5,-77.5},{97,-77.5},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(Hemorrhage.desiredFlow, busConnector.Hemorrhage_PlasmaRate) 
        annotation (Line(
          points={{39.5,-71.5},{97,-71.5},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(plasma.Volume, busConnector.PlasmaVol_Vol)         annotation (
          Line(
          points={{-11,49.8},{-11,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(busConnector.PlasmaProtein_Mass, plasma.PlasmaProtein_Mass) 
        annotation (Line(
          points={{-96,91},{-96,50},{-30,50}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(plasma.q_in, Lungs.q_in) annotation (Line(
          points={{-25,43},{1,43},{1,-29},{-20,-29}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_in, UpperTorso.q_in) annotation (Line(
          points={{-25,43},{1,43},{1,49},{33,49}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_in, MiddleTorso.q_in) annotation (Line(
          points={{-25,43},{1,43},{1,1},{32,1}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_in, LowerTorso.q_in) annotation (Line(
          points={{-25,43},{1,43},{1,-36},{32,-36}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(Peritoneum.flux, plasma.q_in) annotation (Line(
          points={{-59,-54},{-45,-54},{-45,-59},{1,-59},{1,43},{-25,43}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(busConnector.SplanchnicVeins_Pressure, Peritoneum.SplanchnicVeins_Pressure) 
        annotation (Line(
          points={{-96,91},{-96,-52},{-68,-52}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(GILumen.toPlasma, plasma.q_in) annotation (Line(
          points={{-47,71},{-25,71},{-25,43}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(busConnector.GILumenSodium_Mass, GILumen.GILumenSodium_Mass) 
        annotation (Line(
          points={{-96,91},{-96,81},{-66,81}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.GILumenPotassium_Mass, GILumen.GILumenPotassium_Mass) 
        annotation (Line(
          points={{-96,91},{-96,75},{-66,75}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.PulmCapys_Pressure, Lungs.capilaryPressure) 
        annotation (Line(
          points={{-96,91},{-96,-28},{-29.8,-28}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.OsmBody_Osm_conc_CellWalls, GILumen.OsmBody_Osm_conc_CellWalls) 
        annotation (Line(
          points={{-96,91},{-96,69},{-66,69}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(plasma.q_in, Hemorrhage.q_in) annotation (Line(
          points={{-25,43},{1,43},{1,-74.5},{35,-74.5}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_in, Transfusion.q_out) annotation (Line(
          points={{-25,43},{1,43},{1,-79},{54,-79},{54,-80.5}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(IVDrip.q_out, plasma.q_in) annotation (Line(
          points={{70,-87.5},{1,-87.5},{1,43},{-25,43}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(Kidney.q_in, plasma.q_in) annotation (Line(
          points={{-55.8,24.4},{-25,43}},
          color={127,127,0},
          thickness=1,
          smooth=Smooth.None));
      connect(bladderVoidFlow.q_in,Bladder. con) annotation (Line(
          points={{-66,-9.5},{-47,-9.5},{-47,-4}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(busConnector.BladderVoidFlow, bladderVoidFlow.desiredFlow) 
        annotation (Line(
          points={{-96,91},{-96,-4},{-73.5,-4},{-73.5,-5}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(Bladder.Volume, busConnector.BladderVolume_Mass) 
        annotation (Line(
          points={{-35.8,-14},{-35.8,-17},{-96,-17},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(Kidney.urine, Bladder.con) annotation (Line(
          points={{-56.8,10},{-56.8,-4},{-47,-4}},
          color={0,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(busConnector.MetabolicH2O_Rate, MetabolicH2O.u)             annotation (Line(
          points={{-96,91},{24,91},{24,84.5},{69.1,84.5}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.HeatInsensibleSkin_H2O, HeatInsensibleSkin.u)       annotation (
          Line(
          points={{-96,91},{24,91},{24,71.5},{41.1,71.5}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(HeatInsensibleSkin.y[1], UpperTorso.HeatInsensible_H2O) annotation (
          Line(
          points={{51.45,71.2},{92,71.2},{92,47},{53,47}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="UT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(HeatInsensibleSkin.y[2], insensible.u2) annotation (
          Line(
          points={{51.45,71.5},{92,71.5},{92,14.4},{64.7,14.4}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="MT_HeatInsensible_H2O",
          index=-1,
          extent={{28,2},{28,2}}));
      connect(HeatInsensibleSkin.y[3], LowerTorso.HeatInsensible_H2O) annotation (
          Line(
          points={{51.45,71.8},{92,71.8},{92,-38},{52,-38}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="LT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(MetabolicH2O.y[1], UpperTorso.MetabolicH2O_Rate) annotation (Line(
          points={{79.45,84.2},{94,84.2},{94,55},{53,55}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="UT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(MetabolicH2O.y[2], MiddleTorso.MetabolicH2O_Rate) annotation (Line(
          points={{79.45,84.5},{94,84.5},{94,7},{52,7}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="MT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(MetabolicH2O.y[3], LowerTorso.MetabolicH2O_Rate) annotation (Line(
          points={{79.45,84.8},{94,84.8},{94,-30},{52,-30}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="LT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(HeatSweating.y[1], UpperTorso.HeatSweating_H2O) annotation (Line(
          points={{65.45,78.2},{93,78.2},{93,51},{53,51}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="UT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(HeatSweating.y[2], MiddleTorso.HeatSweating_H2O) annotation (Line(
          points={{65.45,78.5},{93,78.5},{93,3},{52,3}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="MT_%second",
          index=-1,
          extent={{30,2},{30,2}}));
      connect(HeatSweating.y[3], LowerTorso.HeatSweating_H2O) annotation (Line(
          points={{65.45,78.8},{93,78.8},{93,-34},{52,-34}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="LT_%second",
          index=-1,
          extent={{30,2},{30,2}}));

      connect(insensible.u1, busConnector.HeatInsensibleLung_H2O) 
        annotation (Line(
          points={{64.7,18.6},{97,18.6},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{5,2},{5,2}}));
      connect(insensible.y, MiddleTorso.HeatInsensible_H2O) annotation (Line(
          points={{56.65,16.5},{55,16.5},{55,-1},{52,-1}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(UpperTorso.CellH2O_Vol_part, busConnector.UT_Cell_H2O) 
        annotation (Line(
          points={{53,39},{97,39},{97,91},{-96,91}},
          color={0,0,0},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(UpperTorso.capilaryPressure, busConnector.RegionalPressure_UpperCapy) 
        annotation (Line(
          points={{53,43},{97,43},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(UpperTorso.InterstitialProtein_Mass, busConnector.UT_InterstitialProtein_Mass) 
        annotation (Line(
          points={{53,35},{97,35},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(MiddleTorso.CellH2O_Vol_part, busConnector.MT_Cell_H2O) 
        annotation (Line(
          points={{52,-9},{97,-9},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(MiddleTorso.capilaryPressure, busConnector.RegionalPressure_MiddleCapy) 
        annotation (Line(
          points={{52,-5},{97,-5},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(MiddleTorso.InterstitialProtein_Mass, busConnector.MT_InterstitialProtein_Mass) 
        annotation (Line(
          points={{52,-13},{97,-13},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(LowerTorso.CellH2O_Vol_part, busConnector.LT_Cell_H2O) 
        annotation (Line(
          points={{52,-46},{97,-46},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));
      connect(LowerTorso.InterstitialProtein_Mass, busConnector.LT_InterstitialProtein_Mass) 
        annotation (Line(
          points={{52,-50},{97,-50},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,2},{6,2}}));

      connect(busConnector, Kidney.busConnector) annotation (Line(
          points={{-96,91},{-96,20},{-66.6,20}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(plasma.Volume, busConnector.PlasmaVol) annotation (Line(
          points={{-11,49.8},{-11,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(GILumen.Vol, busConnector.GILumenVolume_Mass) annotation (Line(
          points={{-46.8,79},{-25,79},{-25,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(plasma.Volume, busConnector.PlasmaVol)  annotation (Line(
          points={{-11,49.8},{-11,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,1},{3,1}}));
      connect(UpperTorso.InterstitialVol, busConnector.UT_InterstitialWater_Vol)  annotation (
          Line(
          points={{43,35},{43,28},{97,28},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-6},{3,-6}}));
      connect(UpperTorso.Vol, busConnector.UT_H2O_Vol)  annotation (Line(
          points={{37,35},{37,24},{97,24},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-9},{3,-9}}));
      connect(UpperTorso.LymphFlow, busConnector. UT_LymphFlow) annotation (Line(
          points={{49,35},{49,31},{97,31},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-3},{3,-3}}));
      connect(MiddleTorso.InterstitialVol, busConnector.MT_InterstitialWater_Vol)  annotation (
          Line(
          points={{42,-13},{42,-20},{97,-20},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-6},{3,-6}}));
      connect(MiddleTorso.Vol, busConnector.MT_H2O_Vol)  annotation (Line(
          points={{36,-13},{36,-24},{97,-24},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-9},{3,-9}}));
      connect(MiddleTorso.LymphFlow, busConnector. MT_LymphFlow) annotation (Line(
          points={{48,-13},{48,-17},{97,-17},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-3},{3,-3}}));
      connect(LowerTorso.InterstitialVol, busConnector.LT_InterstitialWater_Vol)  annotation (
          Line(
          points={{42,-50},{42,-57},{97,-57},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-6},{3,-6}}));
      connect(LowerTorso.Vol, busConnector.LT_H2O_Vol)  annotation (Line(
          points={{36,-50},{36,-61},{97,-61},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-9},{3,-9}}));
      connect(LowerTorso.LymphFlow, busConnector. LT_LymphFlow) annotation (Line(
          points={{48,-50},{48,-54},{97,-54},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,-3},{3,-3}}));
      connect(busConnector.PeritoneumSpace_Vol, Peritoneum.Volume) annotation (Line(
          points={{-96,91},{-96,-68},{-59,-68}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(Peritoneum.PeritoneumSpace_Loss, busConnector.PeritoneumSpace_Loss) 
        annotation (Line(
          points={{-48.8,-67},{11,-67},{11,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,1},{3,1}}));
      connect(Peritoneum.PeritoneumSpace_Gain, busConnector.PeritoneumSpace_Gain) 
        annotation (Line(
          points={{-48.8,-65},{11,-65},{11,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{3,1},{3,1}}));

      connect(busConnector, waterProperties.busConnector) annotation (Line(
          points={{-96,91},{-96,-75},{-82.9,-75},{-82.9,-75.9}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None));
      connect(LowerTorso.capilaryPressure, busConnector.RegionalPressure_UpperCapy) 
        annotation (Line(
          points={{52,-42},{97,-42},{97,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(busConnector.Osmreceptors, GILumen.OsmBody_Osmoreceptors) 
        annotation (Line(
          points={{-96,91},{-96,63},{-66,63}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(GILumen.Intake, busConnector.GILumenVolume_Intake) annotation (
          Line(
          points={{-47,63},{-25,63},{-25,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(GILumen.Absorbtion, busConnector.GILumenVolume_Absorption) 
        annotation (Line(
          points={{-46.8,75},{-25,75},{-25,91},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(busConnector.SweatDuct_H2OOutflow, HeatSweating.u) annotation (
          Line(
          points={{-96,91},{24,91},{24,78.5},{55.1,78.5}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(Lungs.Vol, busConnector.ExcessLungWater_Volume) annotation (Line(
          points={{-20,-43},{-21,-43},{-21,-44},{-96,-44},{-96,91}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (
    Documentation(info="<HTML>
<PRE>
QHP 2008 / H2O Reference
 
Created : 26-Jun-06
Last Modified : 24-Mar-08
Author : Tom Coleman
Copyright : 2008-2008
By : University of Mississippi Medical Center
Solver : DES 2005
Schema : 2005.0
 
There are several different ways to describe the body's
water compartments while implementing mass balance.
 
Water compartments can initially be divided into two
basic types: intracellular (ICFV) and extracellular (ECFV).
 
Intracellular compartments are red blood cells and
(all other) cells.
 
Extracellular compartments are plasma and interstitium.
 
Red Cells     =  1.6
Cells         = 26.4
                -----
ICFV          = 28.0
 
Interstitium  = 12.0
Plasma        =  3.0
                -----
ECFV          = 15.0
 
Total         = 43.0
 
The problem with this scheme is that water cannot move
freely between plasma and interstitium but it can move
freely between interstitium and cell H2O.
 
So, we'll divide the body's water compartments into
vascular and extravascular. The vascular compartment
is futher subdivided into RBC water and plasma.
 
Red Cells     =  1.6
Plasma        =  3.0
                 ---
Vascular      =  4.6
 
Cells         = 26.4 (ICFV less RBC's H2O)
Interstitium  = 12.0
                ----
Extravascular = 38.4
 
Total         = 43.0
 
To relax the nomenclature a bit, extravascular water
is referred to as tissue H2O.
 
This scheme is implemented using three integrals: plasma,
RBC and extravascular.
 
Exchanges are internal and external.
 
Internal exchanges are capillary
filtration and lymph flow.
 
External exchanges are many: absorption
from gut, urine, IV drip, transfusion,
hemorrhage ...
 
External exchanges influence the derivative
of either plasma or extravasular H2O.
 
Traditional compartments are also
represented here.
 
Red Cell H2O  =  1584
Organ H2O     = 26428
                -----
Cell H2O      = 28012
 
IFV           = 11988
Plasma        =  3000
                -----
Extracellular = 14988
 
Total         = 43000
</PRE>
</HTML>
",   revisions="<html>
<ul>
<li><i>  </i>
       </li>
<li><i> january 2009 </i><br> 
       by Marek Matejak in Modelica initially implemented (consulted with Jiri Kofranek), Charles University, Prague<br>
       orginaly described by Tom Coleman in QHP 2008 beta 3, University of Mississippi Medical Center
       </li>
</ul>
</html>"),
       Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                100,100}},
            grid={1,1}),                graphics), Icon(coordinateSystem(
              preserveAspectRatio=true, extent={{-100,-100},{100,100}},
            grid={1,1}), graphics={Bitmap(extent={{-100,100},{100,-100}},
              fileName="icons/water.png"), Text(
            extent={{-139,-99},{142,-124}},
            lineColor={0,0,255},
            textString="%name")}));
    end Water;
    annotation (Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
              -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,-100}},
              fileName="icons/golem.png")}), Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
  end Water;

  package Proteins "Body Proteins"

    model Proteins

      Library.Interfaces.BusConnector busConnector 
        annotation (Placement(transformation(extent={{82,4},{102,24}}),
            iconTransformation(extent={{46,66},{80,100}})));
      Library.ConcentrationFlow.ResistorWithCondParam UT_Capillary(cond=.22) 
        annotation (Placement(transformation(extent={{-2,78},{10,90}})));
      Library.ConcentrationFlow.ResistorWithCondParam MT_Capillary(cond=0.44) 
        annotation (Placement(transformation(extent={{-2,32},{10,44}})));
      Library.ConcentrationFlow.ResistorWithCondParam LT_Capillary(cond=0.71) 
        annotation (Placement(transformation(extent={{0,-8},{12,4}})));
      Library.ConcentrationFlow.OutputPump Hemorrhage 
        annotation (Placement(transformation(extent={{-14,-102},{6,-82}})));
      Library.ConcentrationFlow.InputPump Transfusion 
        annotation (Placement(transformation(extent={{-36,-94},{-16,-74}})));
      Library.ConcentrationFlow.InputPump IVDrip 
        annotation (Placement(transformation(extent={{-44,-76},{-24,-56}})));
      Library.ConcentrationFlow.SolventFlowPump UT_Lymph 
        annotation (Placement(transformation(extent={{10,74},{0,64}})));
      Library.ConcentrationFlow.SolventFlowPump MT_Lymph 
        annotation (Placement(transformation(extent={{10,28},{0,18}})));
      Library.ConcentrationFlow.SolventFlowPump LT_Lymph 
        annotation (Placement(transformation(extent={{10,-12},{0,-22}})));
      Library.ConcentrationFlow.SolventFlowPump Change 
        annotation (Placement(transformation(extent={{-44,-36},{-56,-24}})));
      Library.ConcentrationFlow.ConcentrationCompartment plasma(
          initialSoluteMass=211.361, stateName="PlasmaProtein.Mass") 
        annotation (Placement(transformation(extent={{-76,32},{-56,52}})));
      Library.ConcentrationFlow.ConcentrationCompartment UpperTorso(
          initialSoluteMass=74.3639, stateName="UT_InterstitialProtein.Mass") 
        annotation (Placement(transformation(extent={{42,64},{22,84}})));
      Library.ConcentrationFlow.ConcentrationCompartment MiddleTorso(
          initialSoluteMass=148.686, stateName="MT_InterstitialProtein.Mass") 
        annotation (Placement(transformation(extent={{42,18},{22,38}})));
      Library.ConcentrationFlow.ConcentrationCompartment LowerTorso(
          initialSoluteMass=76.115, stateName="LT_InterstitialProtein.Mass") 
        annotation (Placement(transformation(extent={{42,-22},{22,-2}})));
      Library.ConcentrationFlow.ConcentrationCompartment peritoneum(
          initialSoluteMass=0,
        stateName="PeritoneumProtein.Mass",
        STEADY=false) 
        annotation (Placement(transformation(extent={{-80,-40},{-60,-20}})));
      Library.ConcentrationFlow.Synthesis synthesis(SynthesisBasic=0.01) 
        annotation (Placement(transformation(extent={{-68,80},{-48,100}})));
      Library.ConcentrationFlow.Degradation degradation(DegradationBasic=0.01) 
        annotation (Placement(transformation(extent={{28,-56},{48,-36}})));
      Library.ConcentrationFlow.ResistorWithCondParam GlomerulusProtein_Perm(cond=0) 
        annotation (Placement(transformation(extent={{8,-78},{28,-58}})));
      Library.ConcentrationFlow.ConcentrationCompartment Bladder(
        stateName="BladderProtein.Mass",
        STEADY=false,
        initialSoluteMass=0) 
        annotation (Placement(transformation(extent={{74,-78},{54,-58}})));
      Library.Blocks.Parts alb_glb(nout=2, w={60,40})
        "distribution to albumins(60%) and globulins(40%) " 
        annotation (Placement(transformation(extent={{-62,20},{-56,26}})));
      Modelica.Blocks.Math.Gain alb_molar_mass(
        y(unit="mmol/l"),
        k(unit="mmol/g") = 1/66.438,
        u(unit="g/l")) 
        annotation (Placement(transformation(extent={{-48,26},{-44,30}})));
      Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure(
          unitsString="g/l", toAnotherUnitCoef=1000) 
        annotation (Placement(transformation(extent={{-80,14},{-60,34}})));
      Modelica.Blocks.Math.Feedback PeritoneumChange annotation (Placement(
            transformation(
            extent={{-6,-6},{6,6}},
            rotation=270,
            origin={-50,-14})));
    equation
     connect(Transfusion.desiredFlow, busConnector.Transfusion_ProteinRate) annotation (Line(
          points={{-26,-80},{-26,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));
      connect(IVDrip.desiredFlow, busConnector.IVDripProteinRate) 
                                                  annotation (Line(
          points={{-34,-62},{-34,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));
      connect(busConnector.Hemorrhage_ProteinRate,Hemorrhage. desiredFlow) 
                                                             annotation (Line(
          points={{92,14},{-4,14},{-4,-88}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.PlasmaVol_Vol,plasma. SolventVolume) annotation (Line(
          points={{92,14},{-84,14},{-84,48},{-74,48}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(UpperTorso.SolventVolume, busConnector.UT_InterstitialWater_Vol) 
                                                    annotation (Line(
          points={{40,80},{50,80},{50,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));
      connect(MiddleTorso.SolventVolume, busConnector.MT_InterstitialWater_Vol) 
                                                     annotation (Line(
          points={{40,34},{50,34},{50,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));
      connect(LowerTorso.SolventVolume, busConnector.LT_InterstitialWater_Vol) 
                                                    annotation (Line(
          points={{40,-6},{50,-6},{50,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));
      connect(busConnector.PeritoneumSpace_Vol,peritoneum. SolventVolume) annotation (Line(
          points={{92,14},{-84,14},{-84,-24},{-78,-24}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.LT_LymphFlow,LT_Lymph. solventFlow) annotation (Line(
          points={{92,14},{50,14},{50,-26},{6,-26},{6,-24},{5,-20}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.MT_LymphFlow,MT_Lymph. solventFlow) annotation (Line(
          points={{92,14},{6,14},{6,20},{5,20}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.UT_LymphFlow,UT_Lymph. solventFlow) annotation (Line(
          points={{92,14},{50,14},{50,60},{6,60},{6,64},{5,66}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.LT_InterstitialProtein_Mass,LowerTorso. SoluteMass) annotation (
          Line(
          points={{92,14},{50,14},{50,-26},{32,-26},{32,-19.8}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.MT_InterstitialProtein_Mass,MiddleTorso. SoluteMass) annotation (
          Line(
          points={{92,14},{32,14},{32,20.2}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(busConnector.UT_InterstitialProtein_Mass,UpperTorso. SoluteMass) annotation (
          Line(
          points={{92,14},{50,14},{50,60},{32,60},{32,66.2}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%first",
     index=-1,
     extent={{-6,3},{-6,3}}));
      connect(plasma.SoluteMass, busConnector.PlasmaProtein_Mass)  annotation (Line(
          points={{-66,34.2},{-66,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
     string="%second",
     index=1,
     extent={{6,3},{6,3}}));

      connect(synthesis.q_out, plasma.q_out) annotation (Line(
          points={{-48,90},{-14,90},{-14,42},{-66,42}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(UT_Capillary.q_out, UpperTorso.q_out) annotation (Line(
          points={{9.16,84},{16,84},{16,74},{32,74}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(UT_Lymph.q_in, UpperTorso.q_out) annotation (Line(
          points={{10,69},{16,69},{16,74},{32,74}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, UT_Capillary.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,84},{-1.16,84}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, UT_Lymph.q_out) annotation (Line(
          points={{-66,42},{-14,42},{-14,69},{0,69}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, MT_Capillary.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,38},{-1.16,38}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(MT_Capillary.q_out, MiddleTorso.q_out) annotation (Line(
          points={{9.16,38},{20,38},{20,28},{32,28}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(MT_Lymph.q_in, MiddleTorso.q_out) annotation (Line(
          points={{10,23},{20,23},{20,28},{32,28}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, MT_Lymph.q_out) annotation (Line(
          points={{-66,42},{-14,42},{-14,23},{0,23}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, LT_Capillary.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,-2},{0.84,-2}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, LT_Lymph.q_out) annotation (Line(
          points={{-66,42},{-14,42},{-14,-16},{0,-16},{0,-17}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, GlomerulusProtein_Perm.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,-68},{9.4,-68}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, Hemorrhage.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,-92},{-10,-92}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, Transfusion.q_out) annotation (Line(
          points={{-66,42},{-14,42},{-14,-84},{-20,-84}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, IVDrip.q_out) annotation (Line(
          points={{-66,42},{-14,42},{-14,-66},{-28,-66}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, degradation.q_in) annotation (Line(
          points={{-66,42},{-14,42},{-14,-46},{28,-46}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(plasma.q_out, Change.q_in) 
                                       annotation (Line(
          points={{-66,42},{-14,42},{-14,-30},{-44,-30}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(LT_Capillary.q_out, LowerTorso.q_out) annotation (Line(
          points={{11.16,-2},{20,-2},{20,-12},{32,-12}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(LT_Lymph.q_in, LowerTorso.q_out) annotation (Line(
          points={{10,-17},{20,-17},{20,-12},{32,-12}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(Change.q_out, peritoneum.q_out) 
                                            annotation (Line(
          points={{-56,-30},{-70,-30}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(GlomerulusProtein_Perm.q_out, Bladder.q_out) annotation (Line(
          points={{26.6,-68},{64,-68}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(alb_molar_mass.u, alb_glb.y[1]) annotation (Line(
          points={{-48.4,28},{-54,28},{-54,22.85},{-55.7,22.85}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(alb_molar_mass.y, busConnector.ctAlb) annotation (Line(
          points={{-43.8,28},{-28,28},{-28,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      connect(alb_glb.y[2], busConnector.ctGlb) annotation (Line(
          points={{-55.7,23.15},{-54,23.15},{-54,18},{-30,18},{-30,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,-3},{6,-3}}));
      connect(concentrationMeasure.actualConc, alb_glb.u) annotation (Line(
          points={{-70,28},{-70,23},{-62.6,23}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(concentrationMeasure.q_in, plasma.q_out) annotation (Line(
          points={{-70,24},{-74,24},{-74,42},{-66,42}},
          color={200,0,0},
          thickness=1,
          smooth=Smooth.None));
      connect(busConnector.PeritoneumSpace_Gain, PeritoneumChange.u1) 
        annotation (Line(
          points={{92,14},{-50,14},{-50,-9.2}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(busConnector.PeritoneumSpace_Loss, PeritoneumChange.u2) 
        annotation (Line(
          points={{92,14},{-84,14},{-84,-14},{-54.8,-14}},
          color={0,0,255},
          thickness=0.5,
          smooth=Smooth.None), Text(
          string="%first",
          index=-1,
          extent={{-6,3},{-6,3}}));
      connect(PeritoneumChange.y, Change.solventFlow) annotation (Line(
          points={{-50,-19.4},{-50,-26.4}},
          color={0,0,127},
          smooth=Smooth.None));
      connect(Bladder.SolventVolume, busConnector.BladderVolume_Mass) 
        annotation (Line(
          points={{72,-62},{96,-62},{96,14},{92,14}},
          color={0,0,127},
          smooth=Smooth.None), Text(
          string="%second",
          index=1,
          extent={{6,3},{6,3}}));
      annotation (Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,
              -100},{100,100}}), graphics={Bitmap(extent={{-100,100},{100,-100}},
              fileName="icons/protein.png"), Text(
            extent={{-114,-86},{128,-120}},
            lineColor={0,0,255},
            textString="%name")}),   Diagram(coordinateSystem(preserveAspectRatio=true,
              extent={{-100,-100},{100,100}}), graphics));
    end Proteins;
    annotation (Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
  end Proteins;

  package Electrolytes "Body Electrolytes"

    package Sodium "Body Sodium Distribution"

      model GlomerulusCationFiltration

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

        Library.Interfaces.RealInput_ otherCations( final quantity="Concentration", final unit
            =                                                                                  "mEq/l") annotation (Placement(
              transformation(extent={{-78,30},{-38,70}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={-60,50})));
        Library.Interfaces.RealInput_ ProteinAnions(final quantity="Concentration",
            final unit="mEq/l")                                                                               annotation (Placement(
              transformation(extent={{-40,30},{0,70}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={-20,50})));
        Real KAdjustment;
        Real Cations( final quantity="Concentration", final unit = "mEq/l");
        Real Anions( final quantity="Concentration", final unit = "mEq/l");
      equation
        q_in.q + q_out.q = 0;
        Cations = q_in.conc*1000+otherCations;
        Anions = Cations;
        KAdjustment = (Cations-(Anions-ProteinAnions))/(Cations+(Anions-ProteinAnions));
        q_out.conc = (1-KAdjustment)*q_in.conc;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255}),
              Line(
                points={{0,42},{0,26}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,20},{0,4}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,-4},{0,-20}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,-26},{0,-42}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5)}),       Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
                    Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end GlomerulusCationFiltration;

      model Sodium2
        //extends Library.Interfaces.BaseModel;
        Library.ConcentrationFlow.ConcentrationCompartment NaPool(
            initialSoluteMass=2058.45, stateName="NaPool.Mass") 
          annotation (Placement(transformation(extent={{-86,14},{-66,34}})));
        Library.ConcentrationFlow.ConcentrationCompartment GILumen(
            initialSoluteMass=80, stateName="GILumenSodium.Mass") 
          annotation (Placement(transformation(extent={{-60,-46},{-40,-26}})));
        Library.ConcentrationFlow.SoluteFlowPump Absorbtion 
          annotation (Placement(transformation(extent={{-44,-10},{-56,-22}})));
        Modelica.Blocks.Math.Gain Perm(k=.0015) 
          annotation (Placement(transformation(extent={{-40,-56},{-34,-50}})));
        Library.ConcentrationFlow.OutputPump Hemorrhage 
          annotation (Placement(transformation(extent={{-58,64},{-38,84}})));
        Library.ConcentrationFlow.OutputPump DialyzerActivity 
          annotation (Placement(transformation(extent={{-58,50},{-38,70}})));
        Library.ConcentrationFlow.OutputPump SweatDuct 
          annotation (Placement(transformation(extent={{-58,76},{-38,96}})));
        Library.ConcentrationFlow.InputPump IVDrip 
          annotation (Placement(transformation(extent={{-78,70},{-58,90}})));
        Library.ConcentrationFlow.InputPump Transfusion 
          annotation (Placement(transformation(extent={{-78,56},{-58,76}})));
        Library.ConcentrationFlow.SolventFlowPump_InitialPatch
          glomerulusSudiumRate 
          annotation (Placement(transformation(extent={{-16,14},{4,34}})));
        HumMod.Electrolytes.Sodium.GlomerulusCationFiltration glomerulus 
          annotation (Placement(transformation(extent={{-40,14},{-20,34}})));
        Library.ConcentrationFlow.FractReabsorbtion PT 
          annotation (Placement(transformation(extent={{8,14},{28,34}})));
        Library.Blocks.FractConstant const1(k=58) 
          annotation (Placement(transformation(extent={{2,34},{8,40}})));
        HumMod.Library.Factors.SplineValue IFPEffect(data={{1.0,1.4,0},{4.0,1.0,
              -0.2},{7.0,0.3,0}}) 
          annotation (Placement(transformation(extent={{14,32},{34,52}})));
        HumMod.Library.Factors.SplineValue ANPEffect(data={{0.0,1.2,0},{1.3,1.0,
              -0.2},{2.7,0.6,0}}) 
          annotation (Placement(transformation(extent={{14,40},{34,60}})));
        HumMod.Library.Factors.SplineValue SympsEffect(data={{0.6,0.6,0},{1.0,
              1.0,0.5},{4.0,1.5,0}}) 
          annotation (Placement(transformation(extent={{14,48},{34,68}})));
        HumMod.Library.Factors.SplineValue A2Effect(data={{0.7,0.8,0},{1.3,1.0,
              0.8},{1.6,1.2,0}}) 
          annotation (Placement(transformation(extent={{14,56},{34,76}})));
        Library.ConcentrationFlow.FractReabsorbtion LH(MaxReab=7) 
          annotation (Placement(transformation(extent={{68,14},{88,34}})));
        Library.Blocks.FractConstant const2(k=75) 
          annotation (Placement(transformation(extent={{64,30},{70,36}})));
        Library.ConcentrationFlow.FractReabsorbtion2 DT 
          annotation (Placement(transformation(extent={{80,-74},{60,-54}})));
        Library.ConcentrationFlow.FractReabsorbtion CD(MaxReab=.7) 
          annotation (Placement(transformation(extent={{30,-74},{10,-54}})));
        Library.Blocks.FractConstant const3(k=75) 
          annotation (Placement(transformation(extent={{18,-58},{24,-52}})));
        Library.Blocks.FractConstant const4(k=75) 
          annotation (Placement(transformation(extent={{68,-58},{74,-52}})));
        Library.ConcentrationFlow.ConcentrationCompartment Bladder(
            initialSoluteMass=0,
          stateName="BladderSodium.Mass",
          STEADY=false) 
          annotation (Placement(transformation(extent={{-28,-26},{-8,-6}})));
        HumMod.Library.Factors.SplineValue Furosemide(data={{0.0,1.0,-1},{0.1,
              0.0,0}}) 
          annotation (Placement(transformation(extent={{64,62},{84,82}})));
        Library.Factors.SplineValue AldoEffect(data={{0.0,0.7,0},{10.0,1.0,0}}) 
          annotation (Placement(transformation(extent={{76,40},{96,60}})));
        HumMod.Library.Factors.SplineValue LoadEffect(data={{0.0,3.0,0},{7.2,
              1.0,-0.2},{20.0,0.5,0}}) 
          annotation (Placement(transformation(extent={{76,32},{96,52}})));
        HumMod.Library.Factors.SimpleMultiply FurosemideEffect 
          annotation (Placement(transformation(extent={{76,48},{96,68}})));
        HumMod.Library.Factors.SimpleMultiply Filtering_xNormal 
          annotation (Placement(transformation(extent={{64,54},{84,74}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{50,14},{70,34}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure1 
          annotation (Placement(transformation(extent={{98,-74},{78,-54}})));
        HumMod.Library.Factors.SplineValue LoadEffect1(data={{0.0,2.0,0},{1.6,
              1.0,0}}) 
          annotation (Placement(transformation(extent={{72,-56},{52,-36}})));
         HumMod.Library.Factors.SplineValue ThiazideEffect(data={{0.0,1.0,-2.0},
              {0.6,0.2,0.0}}) 
          annotation (Placement(transformation(extent={{72,-48},{52,-28}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure2 
          annotation (Placement(transformation(extent={{58,-74},{38,-54}})));
        HumMod.Library.Factors.SplineValue LoadEffect2(data={{0.0,2.0,0},{0.4,
              1.0,0}}) 
          annotation (Placement(transformation(extent={{22,-58},{2,-38}})));
        HumMod.Library.Factors.SplineValue ANPEffect2(data={{0.0,1.2,0},{1.3,
              1.0,-0.4},{2.7,0.2,0}}) 
          annotation (Placement(transformation(extent={{22,-50},{2,-30}})));
        Library.Factors.SimpleMultiply AldoEffect2 
          annotation (Placement(transformation(extent={{10,-10},{-10,10}},
              rotation=270,
              origin={84,-78})));
        HumMod.Library.Blocks.Constant const5(k=2) 
          annotation (Placement(transformation(extent={{94,-80},{90,-76}})));
        Library.ConcentrationFlow.InputPump Diet 
          annotation (Placement(transformation(extent={{-90,-60},{-70,-40}})));
        Library.ConcentrationFlow.OutputPump Diarrhea 
          annotation (Placement(transformation(extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-80,-64})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure3 
          annotation (Placement(transformation(extent={{10,-54},{-10,-74}})));
        Library.ConcentrationFlow.ConcentrationCompartment Medulla(
                                       stateName="MedullaNa.Mass",
            initialSoluteMass=13) 
          annotation (Placement(transformation(extent={{28,-102},{48,-82}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure(
            unitsString="mEq/l", toAnotherUnitCoef=1000) 
          annotation (Placement(transformation(
              extent={{-9,-9},{9,9}},
              rotation=270,
              origin={57,-97})));
        Library.ConcentrationFlow.SolventFlowPump VasaRectaOutflow 
          annotation (Placement(transformation(extent={{-4,-100},{-20,-84}})));
        Modelica.Blocks.Math.Gain gain(k=.03) 
          annotation (Placement(transformation(extent={{-28,-88},{-20,-80}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure1(
            unitsString="mEq/l", toAnotherUnitCoef=1000) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-92,36})));
        Library.ConcentrationFlow.SolventOutflowPump bladderVoid 
          annotation (Placement(transformation(extent={{12,-22},{24,-10}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure4 
          annotation (Placement(transformation(extent={{64,-68},{44,-88}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure5 
          annotation (Placement(transformation(extent={{80,16},{60,-4}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure6 
          annotation (Placement(transformation(extent={{10,10},{-10,-10}},
              rotation=90,
              origin={18,12})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-98,86},{-86,98}}),
              iconTransformation(extent={{60,60},{100,100}})));
        Modelica.Blocks.Math.Gain Osm(k=2) 
          annotation (Placement(transformation(extent={{68,-100},{74,-94}})));
        Library.Factors.SplineValue AldoEffect1(
          data={{0.0,0.5,0},{12.0,1.0,0.08},{50.0,3.0,0}}) 
          annotation (Placement(transformation(extent={{52,-40},{72,-20}})));
        Modelica.Blocks.Math.Division division 
          annotation (Placement(transformation(extent={{80,-16},{86,-10}})));
        Modelica.Blocks.Math.Gain ml2l(k=1000) 
          annotation (Placement(transformation(extent={{90,-16},{96,-10}})));
        Library.Factors.SplineValue LeptinEffect2(data={{0,3.0,0},{8,1.0,-0.04},{
              50,0.0,0}}) 
          annotation (Placement(transformation(extent={{-90,-50},{-70,-30}})));
        Library.Blocks.Constant Constant7(k=180/1440) 
          annotation (Placement(transformation(extent={{-90,-36},{-82,-28}})));
      Real ECF_conc;
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure2(
            unitsString="mEq/ml", toAnotherUnitCoef=1) 
          annotation (Placement(transformation(
              extent={{-9,-9},{9,9}},
              rotation=270,
              origin={57,-93})));
      equation
        ECF_conc = concentrationMeasure1.actualConc;

      connect(NaPool.SolventVolume, busConnector. ECFV_Vol) annotation (Line(
            points={{-84,30},{-97,30},{-97,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(GILumen.SolventVolume, busConnector. GILumenVolume_Mass) annotation (Line(
            points={{-58,-30},{-97,-30},{-97,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(GILumen.q_out, Absorbtion.q_in) annotation (Line(
            points={{-50,-36},{-38,-36},{-38,-16},{-44,-16}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Absorbtion.q_out, NaPool.q_out) annotation (Line(
            points={{-56,-16},{-100,-16},{-100,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Hemorrhage.q_in, NaPool.q_out) annotation (Line(
            points={{-54,74},{-58,74},{-58,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(DialyzerActivity.q_in, NaPool.q_out) annotation (Line(
            points={{-54,60},{-58,60},{-58,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(NaPool.q_out, SweatDuct.q_in) annotation (Line(
            points={{-76,24},{-58,24},{-58,86},{-54,86}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Transfusion.q_out, NaPool.q_out) annotation (Line(
            points={{-62,66},{-58,66},{-58,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(IVDrip.q_out, NaPool.q_out) annotation (Line(
            points={{-62,80},{-58,80},{-58,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(glomerulusSudiumRate.solventFlow, busConnector. GlomerulusFiltrate_GFR) 
          annotation (Line(
            points={{-6,30},{-6,38},{-32,38},{-32,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(glomerulus.q_in, NaPool.q_out) annotation (Line(
            points={{-40,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(glomerulus.q_out, glomerulusSudiumRate.q_in) annotation (Line(
            points={{-20,24},{-16,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(glomerulusSudiumRate.q_out, PT.Inflow) annotation (Line(
            points={{4,24},{8,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(const1.y, PT.Normal) annotation (Line(
            points={{8.3,37},{12,37},{12,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(A2Effect.yBase, busConnector. KidneyFunctionEffect) annotation (Line(
            points={{24,68},{24,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(A2Effect.y, SympsEffect.yBase) annotation (Line(
            points={{24,64},{24,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(SympsEffect.y, ANPEffect.yBase) annotation (Line(
            points={{24,56},{24,52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ANPEffect.y, IFPEffect.yBase) annotation (Line(
            points={{24,48},{24,44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(IFPEffect.y, PT.Effects) annotation (Line(
            points={{24,40},{24,28},{26,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.A2Pool_Log10Conc, A2Effect.u) annotation (Line(
            points={{-92,92},{-18.9,92},{-18.9,66},{14.2,66}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(SympsEffect.u, busConnector. KidneyAlpha_PT_NA) annotation (Line(
            points={{14.2,58},{-18.9,58},{-18.9,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(busConnector.NephronANP_Log10Conc, ANPEffect.u) annotation (Line(
            points={{-92,92},{-18.9,92},{-18.9,50},{14.2,50}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(busConnector.NephronIFP_Pressure, IFPEffect.u) annotation (Line(
            points={{-92,92},{-18.9,92},{-18.9,42},{14.2,42}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(const2.y, LH.Normal) annotation (Line(
            points={{70.3,33},{72,33},{72,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const3.y, CD.Normal) annotation (Line(
            points={{24.3,-55},{26,-55},{26,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(const4.y, DT.Normal) annotation (Line(
            points={{74.3,-55},{76,-55},{76,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Bladder.SolventVolume, busConnector. BladderVolume_Mass) annotation (Line(
            points={{-26,-10},{-98,-10},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(Furosemide.u, busConnector. FurosemidePool_Furosemide_conc) 
                                                  annotation (Line(
            points={{64.2,72},{48,72},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(AldoEffect.y, LoadEffect.yBase) annotation (Line(
            points={{86,48},{86,44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LoadEffect.y, LH.Effects) annotation (Line(
            points={{86,40},{86,28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(FurosemideEffect.y, AldoEffect.yBase) annotation (Line(
            points={{86,56},{86,52}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Filtering_xNormal.u, busConnector. Kidney_NephronCount_Filtering_xNormal) 
          annotation (Line(
            points={{64.2,64},{48,64},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(Filtering_xNormal.y, FurosemideEffect.u) annotation (Line(
            points={{74,62},{74,58},{76.2,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Furosemide.y, Filtering_xNormal.yBase) annotation (Line(
            points={{74,70},{74,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.KidneyFunctionEffect, Furosemide.yBase) annotation (Line(
            points={{-92,92},{74,92},{74,74}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(FurosemideEffect.yBase, busConnector. KidneyFunctionEffect) annotation (Line(
            points={{86,60},{86,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(PT.Outflow, flowMeasure.q_in) annotation (Line(
            points={{28,24},{56,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure.q_out, LH.Inflow) annotation (Line(
            points={{64,24},{68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure.actualFlow, LoadEffect.u) annotation (Line(
            points={{60,29},{60,42},{76.2,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(DT.Inflow, flowMeasure1.q_out) annotation (Line(
            points={{80,-64},{84,-64}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure1.q_in, LH.Outflow) annotation (Line(
            points={{92,-64},{98,-64},{98,24},{88,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(LoadEffect1.y, DT.Effects) annotation (Line(
            points={{62,-48},{62,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LoadEffect1.u, flowMeasure1.actualFlow) annotation (Line(
            points={{71.8,-46},{88,-46},{88,-59}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ThiazideEffect.u, busConnector. ThiazidePool_Thiazide_conc) annotation (Line(
            points={{71.8,-38},{99.9,-38},{99.9,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(CD.Inflow, flowMeasure2.q_out) annotation (Line(
            points={{30,-64},{44,-64}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure2.q_in, DT.Outflow) annotation (Line(
            points={{52,-64},{60,-64}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(LoadEffect2.y, CD.Effects) annotation (Line(
            points={{12,-50},{12,-60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LoadEffect2.u, flowMeasure2.actualFlow) annotation (Line(
            points={{21.8,-48},{48,-48},{48,-59}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ANPEffect2.y, LoadEffect2.yBase) annotation (Line(
            points={{12,-42},{12,-46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ANPEffect2.yBase, busConnector. KidneyFunctionEffect) annotation (Line(
            points={{12,-38},{12,-36},{48,-36},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(ANPEffect2.u, busConnector. NephronANP_Log10Conc) annotation (Line(
            points={{21.8,-40},{48,-40},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(IVDrip.desiredFlow, busConnector. IVDrip_NaRate) annotation (Line(
            points={{-68,84},{-68,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(Transfusion.desiredFlow, busConnector. Transfusion_NaRate) annotation (Line(
            points={{-68,70},{-60,70},{-60,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(SweatDuct.desiredFlow, busConnector. SweatDuct_NaRate) annotation (Line(
            points={{-48,90},{-48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(const5.y, AldoEffect2.yBase) annotation (Line(
            points={{89.8,-78},{86,-78}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AldoEffect2.y, DT.MaxReab) annotation (Line(
            points={{82,-78},{76,-78},{76,-68.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Hemorrhage_NaRate, Hemorrhage.desiredFlow) annotation (Line(
            points={{-92,92},{-47.5,92},{-47.5,78},{-48,78}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(DialyzerActivity.desiredFlow, busConnector. DialyzerActivity_Na_Flux) 
          annotation (Line(
            points={{-48,64},{-48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(glomerulus.otherCations, busConnector. KPool) annotation (Line(
            points={{-36,29},{-36,38},{-32,38},{-32,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(glomerulus.ProteinAnions, busConnector. BloodIons_ProteinAnions) annotation (
            Line(
            points={{-32,29},{-32,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(Diet.q_out, GILumen.q_out) annotation (Line(
            points={{-74,-50},{-58,-50},{-58,-36},{-50,-36}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Diarrhea.desiredFlow, busConnector. GILumenDiarrhea_NaLoss) 
                                                         annotation (Line(
            points={{-80,-68},{-80,-70},{-98,-70},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(GILumen.q_out, Diarrhea.q_in) annotation (Line(
            points={{-50,-36},{-58,-36},{-58,-64},{-74,-64}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(GILumen.SoluteMass, Perm.u) annotation (Line(
            points={{-50,-43.8},{-50,-53},{-40.6,-53}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Perm.y, Absorbtion.soluteFlow) annotation (Line(
            points={{-33.7,-53},{-32,-54},{-32,-24},{-50,-24},{-50,-19.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(CD.Outflow, flowMeasure3.q_in) annotation (Line(
            points={{10,-64},{4,-64}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure3.q_out, Bladder.q_out) annotation (Line(
            points={{-4,-64},{-6,-64},{-6,-16},{-18,-16}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(CD.Reabsorbtion, Medulla.q_out) annotation (Line(
            points={{20,-68},{20,-92},{38,-92}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Medulla.q_out, concentrationMeasure.q_in) annotation (Line(
            points={{38,-92},{42,-92},{42,-97},{57,-97}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Medulla.q_out, VasaRectaOutflow.q_in) annotation (Line(
            points={{38,-92},{-4,-92}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(VasaRectaOutflow.q_out, NaPool.q_out) annotation (Line(
            points={{-20,-92},{-100,-92},{-100,24},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(gain.y, VasaRectaOutflow.solventFlow) annotation (Line(
            points={{-19.6,-84},{-12,-84},{-12,-87.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.VasaRecta_Outflow, gain.u) annotation (Line(
            points={{-92,92},{-98.4,92},{-98.4,-84},{-28.8,-84}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-6,3},{-6,3}}));
        connect(Medulla.SolventVolume, busConnector. Medulla_Volume) annotation (Line(
            points={{30,-86},{26,-86},{26,-78},{-98,-78},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(NaPool.SoluteMass, busConnector. NaPool_mass) annotation (Line(
            points={{-76,16.2},{-76,14},{-98,14},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(concentrationMeasure1.q_in, NaPool.q_out) annotation (Line(
            points={{-92,36},{-76,36},{-76,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Bladder.q_out, bladderVoid.q_in) annotation (Line(
            points={{-18,-16},{12,-16}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(bladderVoid.solventFlow, busConnector. BladderVoidFlow) annotation (Line(
            points={{18,-12.4},{18,-6},{-98,-6},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(GILumen.SoluteMass, busConnector. GILumenSodium_Mass) annotation (Line(
            points={{-50,-43.8},{-50,-60},{-76,-60},{-76,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(AldoEffect1.y, busConnector. DT_AldosteroneEffect) annotation (Line(
            points={{62,-32},{62,-34},{100,-34},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(DT.Reabsorbtion, flowMeasure4.q_in) annotation (Line(
            points={{70,-68},{70,-78},{58,-78}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure4.q_out, NaPool.q_out) annotation (Line(
            points={{50,-78},{-100,-78},{-100,24},{-76,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure5.actualFlow, busConnector. LH_Na_Reab) annotation (Line(
            points={{70,1},{100,1},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(LH.Reabsorbtion, flowMeasure5.q_in) annotation (Line(
            points={{78,20},{78,6},{74,6}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure5.q_out, NaPool.q_out) annotation (Line(
            points={{66,6},{-100,6},{-100,24},{-76,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure6.actualFlow, busConnector. PT_Na_Reab) annotation (Line(
            points={{23,12},{47.5,12},{47.5,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(PT.Reabsorbtion, flowMeasure6.q_in) annotation (Line(
            points={{18,20},{18,16}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure6.q_out, NaPool.q_out) annotation (Line(
            points={{18,8},{18,6},{-100,6},{-100,24},{-76,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));

        connect(concentrationMeasure1.actualConc, busConnector.NaPool_conc_per_liter) 
          annotation (Line(
            points={{-92,40},{-92,46},{-98,46},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(concentrationMeasure.actualConc, Osm.u) annotation (Line(
            points={{60.6,-97},{67.4,-97}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Osm.y, busConnector.MedullaNa_Osmolarity) annotation (Line(
            points={{74.3,-97},{100,-97},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(busConnector.KidneyFunctionEffect, AldoEffect1.yBase) annotation (
            Line(
            points={{-92,92},{48,92},{48,-28},{62,-28}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(AldoEffect1.y, ThiazideEffect.yBase) annotation (Line(
            points={{62,-32},{62,-36}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ThiazideEffect.y, LoadEffect1.yBase) annotation (Line(
            points={{62,-40},{62,-44}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AldoEffect1.y, AldoEffect2.u) annotation (Line(
            points={{62,-32},{62,-34},{96,-34},{96,-90},{84,-90},{84,-87.8}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LH.ReabFract, busConnector. LH_Na_FractReab) annotation (Line(
            points={{88,20},{100,20},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(PT.ReabFract, busConnector. PT_Na_FractReab) annotation (Line(
            points={{28,20},{48,20},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(flowMeasure2.actualFlow, busConnector. DT_Na_Outflow) annotation (Line(
            points={{48,-59},{48,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(flowMeasure3.actualFlow, busConnector. CD_Na_Outflow) annotation (Line(
            points={{0,-69},{0,-74},{-98,-74},{-98,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(flowMeasure4.actualFlow, busConnector. DT_Na_Reab) annotation (Line(
            points={{54,-83},{54,-90},{100,-90},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{6,3},{6,3}}));
        connect(flowMeasure1.actualFlow, division.u1) annotation (Line(
            points={{88,-59},{88,-24},{74,-24},{74,-11.2},{79.4,-11.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.LH_H2O_Outflow, division.u2) annotation (Line(
            points={{-92,92},{-98,92},{-98,-6},{56,-6},{56,-14.8},{79.4,-14.8}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));

        connect(ml2l.u, division.y) annotation (Line(
            points={{89.4,-13},{89.4,-12.5},{86.3,-12.5},{86.3,-13}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ml2l.y, busConnector.MD_Na) annotation (Line(
            points={{96.3,-13},{100,-14},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(busConnector.NephronAldo_conc_in_nG_per_dl, AldoEffect.u) 
          annotation (Line(
            points={{-92,92},{48,92},{48,50},{76.2,50}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.NephronAldo_conc_in_nG_per_dl, AldoEffect1.u) 
          annotation (Line(
            points={{-92,92},{48,92},{48,-30},{52.2,-30}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Constant7.y,LeptinEffect2. yBase) annotation (Line(
            points={{-81.6,-32},{-80,-32},{-80,-38}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeptinEffect2.y, Diet.desiredFlow) annotation (Line(
            points={{-80,-42},{-80,-46}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Leptin, LeptinEffect2.u) annotation (Line(
            points={{-92,92},{-98,92},{-98,-40},{-89.8,-40}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Medulla.q_out, concentrationMeasure2.q_in) annotation (Line(
            points={{38,-92},{52,-92},{52,-93},{57,-93}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(concentrationMeasure2.actualConc, busConnector.MedullaNa_conc) 
          annotation (Line(
            points={{60.6,-93},{100,-93},{100,92},{-92,92}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/Na.jpg"), Text(
                extent={{-110,-104},{110,-130}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
      end Sodium2;
    end Sodium;

    package Potassium "Body Potassium Distribution"

      model IkedaPotasiumIntoICFFactor

        Library.Interfaces.RealInput_ yBase 
                         annotation (Placement(transformation(extent={{-20,-20},{
                  20,20}},
              rotation=270,
              origin={6,100}),
              iconTransformation(extent={{-10,-10},{10,10}},rotation=270,
              origin={0,50})));
        Library.Interfaces.RealOutput_ y 
                      annotation (Placement(transformation(extent={{-20,-20},{20,
                  20}},
              rotation=270,
              origin={0,-80}),
              iconTransformation(extent={{-10,-10},{10,10}},  rotation=270,
              origin={0,-52})));

        Library.Interfaces.RealInput_ Artys_pH annotation (Placement(transformation(
                extent={{-118,10},{-78,50}}), iconTransformation(extent={{-120,10},{-80,
                  50}})));
        Library.Interfaces.RealInput_ PotasiumECF_conc annotation (Placement(
              transformation(extent={{-118,-28},{-78,12}}),iconTransformation(extent={
                  {-120,-50},{-80,-10}})));

        Real effect;
      equation
        effect = 1+0.5*log(PotasiumECF_conc/(56.744-7.06*Artys_pH));
        y = yBase*effect;
       annotation (
          Icon(coordinateSystem(
              preserveAspectRatio=true,
              extent={{-100,-100},{100,100}},
              grid={2,2}), graphics={Rectangle(
                extent={{-100,50},{100,-50}},
                lineColor={127,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Sphere), Text(
                extent={{-86,-36},{100,40}},
                textString="%name",
                lineColor={0,0,0},
                fillPattern=FillPattern.Sphere)}), Diagram(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics));
      end IkedaPotasiumIntoICFFactor;

      model Potassium2
        //extends Library.Interfaces.BaseModel;
        Library.ConcentrationFlow.ConcentrationCompartment KPool(
            initialSoluteMass=62.897, stateName="KPool.Mass") 
          annotation (Placement(transformation(extent={{-34,14},{-14,34}})));
        Library.ConcentrationFlow.ConcentrationCompartment GILumen(
            initialSoluteMass=25, stateName="GILumenPotassium.Mass") 
          annotation (Placement(transformation(extent={{-18,-36},{2,-16}})));
        Library.ConcentrationFlow.SoluteFlowPump Absorbtion 
          annotation (Placement(transformation(extent={{6,-20},{18,-32}})));
        Modelica.Blocks.Math.Gain Perm(k=.002) 
          annotation (Placement(transformation(extent={{0,-48},{6,-42}})));
        Library.ConcentrationFlow.OutputPump Hemorrhage 
          annotation (Placement(transformation(extent={{20,-86},{40,-66}})));
        Library.ConcentrationFlow.OutputPump DialyzerActivity 
          annotation (Placement(transformation(extent={{20,-100},{40,-80}})));
        Library.ConcentrationFlow.OutputPump SweatDuct 
          annotation (Placement(transformation(extent={{20,-72},{40,-52}})));
        Library.ConcentrationFlow.InputPump IVDrip 
          annotation (Placement(transformation(extent={{0,-80},{20,-60}})));
        Library.ConcentrationFlow.InputPump Transfusion 
          annotation (Placement(transformation(extent={{0,-94},{20,-74}})));
        Library.ConcentrationFlow.ConcentrationCompartment Bladder(
            initialSoluteMass=0,
          STEADY=false,
          stateName="BladderPotassium.Mass") 
          annotation (Placement(transformation(extent={{94,12},{74,32}})));
        HumMod.Library.Factors.SplineValue NaEffect(data={{0.0,0.3,0},{0.4,1.0,
              1.5},{4.0,3.0,0}}) 
          annotation (Placement(transformation(extent={{44,30},{24,50}})));
        Library.Factors.SplineValue AldoEffect(data={{0.0,0.3,0},{12.0,1.0,0.06},{50.0,
              3.0,0}}) 
          annotation (Placement(transformation(extent={{44,38},{24,58}})));
        HumMod.Library.Factors.SplineValue ThiazideEffect(data={{0.0,1.0,2.0},{
              0.6,2.0,0}}) 
          annotation (Placement(transformation(extent={{44,46},{24,66}})));
        Library.ConcentrationFlow.InputPump Diet 
          annotation (Placement(transformation(extent={{-66,-42},{-46,-22}})));
        Library.ConcentrationFlow.OutputPump Diarrhea 
          annotation (Placement(transformation(extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-56,-44})));
        Library.ConcentrationFlow.ConcentrationCompartment KCell(
            initialSoluteMass=3769.74, stateName="KCell.Mass") 
          annotation (Placement(transformation(extent={{-30,70},{-10,90}})));
        Library.ConcentrationFlow.SoluteFlowPump KFluxToCell 
          annotation (Placement(transformation(extent={{-6,-6},{6,6}},
              rotation=90,
              origin={-30,58})));
        Modelica.Blocks.Math.Gain Perm1(k=.002) 
          annotation (Placement(transformation(extent={{-3,-3},{3,3}},
              rotation=90,
              origin={-55,53})));
        Library.ConcentrationFlow.SoluteFlowPump KFluxToPool 
          annotation (Placement(transformation(extent={{-6,-6},{6,6}},
              rotation=270,
              origin={-12,48})));
        Modelica.Blocks.Math.Feedback feedback 
          annotation (Placement(transformation(extent={{-10,62},{-2,54}})));
        HumMod.Library.Blocks.ElectrolytesMassConstant KCell_CaptiveMass(k=2180) 
          annotation (Placement(transformation(extent={{8,76},{-2,66}})));
        Modelica.Blocks.Math.Gain Perm2(k=7.4e-5) 
          annotation (Placement(transformation(extent={{-3,-3},{3,3}},
              rotation=270,
              origin={3,53})));
        Library.Blocks.ElectrolytesFlowConstant electrolytesFlowConstant(k=.05) 
          annotation (Placement(transformation(extent={{18,76},{30,88}})));
        Library.ConcentrationFlow.SoluteFlowPump DT_K 
          annotation (Placement(transformation(extent={{-6,-6},{6,6}},
              rotation=0,
              origin={34,22})));
        HumMod.Library.Factors.SplineValue KEffect(data={{0.0,0.0,0},{4.4,1.0,
              0.5},{5.5,3.0,0}}) 
          annotation (Placement(transformation(extent={{24,22},{44,42}})));
        Modelica.Blocks.Math.Gain mEq_per_L(k=1000) 
          annotation (Placement(transformation(extent={{-3,-3},{3,3}},
              rotation=0,
              origin={15,31})));
        Modelica.Blocks.Math.Division division 
          annotation (Placement(transformation(extent={{-18,6},{-14,10}})));
        Library.Factors.SimpleMultiply KidneyFunction 
          annotation (Placement(transformation(extent={{44,54},{24,74}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={30,6})));
        Modelica.Blocks.Math.Gain gain(k=1000) 
          annotation (Placement(transformation(extent={{40,4},{44,8}})));
        Library.Factors.SplineDelayByDay splineDelayByDay(              data={{
              0,0.9,0.0},{300,1.0,0.00025},{1500,1.1,0.0}}, Tau=120*1440) 
                                                            annotation (
            Placement(transformation(
              extent={{10,-10},{-10,10}},
              rotation=90,
              origin={-48,58})));
        Library.ConcentrationFlow.SolventOutflowPump bladderVoid 
          annotation (Placement(transformation(extent={{82,-20},{94,-8}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure1(
            unitsString="mEq/l", toAnotherUnitCoef=1000) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-8,80})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure annotation (Placement(
              transformation(
              extent={{-8,8},{8,-8}},
              rotation=90,
              origin={-24,34})));
        Library.ConcentrationFlow.SoluteFlowPump KFluxToCellWithGlucose 
          annotation (Placement(transformation(extent={{-6,-6},{6,6}},
              rotation=90,
              origin={-68,60})));
        Modelica.Blocks.Math.Gain CGL3(k=.03)
          "glucose flow into cells to potassium flow into cells" 
          annotation (Placement(transformation(extent={{-2,-2},{2,2}},
              rotation=0,
              origin={-78,60})));
        IkedaPotasiumIntoICFFactor IkedaIntoICF annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=90,
              origin={-40,58})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure2(
            unitsString="mEq/l", toAnotherUnitCoef=1000) 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-48,24})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-94,88},{-82,100}}),
              iconTransformation(extent={{60,60},{100,100}})));
        Modelica.Blocks.Math.Add3 YGLS "Ikeda glucose to cells flow" 
          annotation (Placement(transformation(extent={{-84,66},{-74,76}})));
        Library.Blocks.ElectrolytesFlowConstant electrolytesFlowConstant1(k=0) 
          annotation (Placement(transformation(extent={{-90,46},{-80,56}})));
        Library.Blocks.Constant Constant7(k=70/1440) 
          annotation (Placement(transformation(extent={{-66,-16},{-58,-8}})));
        Library.Factors.SplineValue LeptinEffect2(data={{0,3.0,0},{8,1.0,-0.04},{
              50,0.0,0}}) 
          annotation (Placement(transformation(extent={{-66,-30},{-46,-10}})));
        Library.Blocks.VolumeConstant volumeConstant(k=14294.8) 
          annotation (Placement(transformation(extent={{-86,16},{-78,24}})));

      Real ECF_conc;
      equation
        ECF_conc=concentrationMeasure2.actualConc;
        connect(GILumen.SolventVolume, busConnector. GILumenVolume_Mass) annotation (Line(
            points={{-16,-20},{-24,-20},{-24,-4},{-96,-4},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.q_out, Absorbtion.q_in) annotation (Line(
            points={{-8,-26},{6,-26}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Absorbtion.q_out, KPool.q_out)  annotation (Line(
            points={{18,-26},{22,-26},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Hemorrhage.q_in, KPool.q_out)  annotation (Line(
            points={{24,-76},{22,-76},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(DialyzerActivity.q_in, KPool.q_out)  annotation (Line(
            points={{24,-90},{22,-90},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(KPool.q_out, SweatDuct.q_in)  annotation (Line(
            points={{-24,24},{22,24},{22,-62},{24,-62}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Transfusion.q_out, KPool.q_out)  annotation (Line(
            points={{16,-84},{22,-84},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(IVDrip.q_out, KPool.q_out)  annotation (Line(
            points={{16,-70},{22,-70},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Bladder.SolventVolume, busConnector. BladderVolume_Mass) annotation (Line(
            points={{92,28},{98,28},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(AldoEffect.y, NaEffect.yBase)     annotation (Line(
            points={{34,46},{34,42}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ThiazideEffect.y, AldoEffect.yBase)  annotation (Line(
            points={{34,54},{34,50}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ThiazideEffect.u, busConnector. ThiazidePool_Thiazide_conc) annotation (Line(
            points={{43.8,56},{98,56},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(IVDrip.desiredFlow, busConnector. IVDrip_KRate)  annotation (Line(
            points={{10,-66},{10,-64},{-96,-64},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Transfusion.desiredFlow, busConnector. Transfusion_KRate)  annotation (Line(
            points={{10,-80},{-96,-80},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(SweatDuct.desiredFlow, busConnector. SweatDuct_KRate)  annotation (Line(
            points={{30,-58},{30,-54},{98,-54},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(busConnector.Hemorrhage_KRate, Hemorrhage.desiredFlow)  annotation (Line(
            points={{-88,94},{98,94},{98,-72},{30,-72}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(DialyzerActivity.desiredFlow, busConnector. DialyzerActivity_K_Flux) 
          annotation (Line(
            points={{30,-86},{98,-86},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Diet.q_out, GILumen.q_out) annotation (Line(
            points={{-50,-32},{-16,-32},{-16,-26},{-8,-26}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Diarrhea.desiredFlow, busConnector. GILumenDiarrhea_KLoss) 
                                                         annotation (Line(
            points={{-56,-48},{-96,-48},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.q_out, Diarrhea.q_in) annotation (Line(
            points={{-8,-26},{-16,-26},{-16,-44},{-50,-44}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(KCell.SolventVolume, busConnector. CellH2O_Vol) annotation (Line(
            points={{-28,86},{-95,86},{-95,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Perm.y, Absorbtion.soluteFlow) annotation (Line(
            points={{6.3,-45},{12,-45},{12,-29.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Perm.u, GILumen.SoluteMass) annotation (Line(
            points={{-0.6,-45},{-8,-45},{-8,-33.8}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(KFluxToCell.q_out, KCell.q_out) annotation (Line(
            points={{-30,64},{-30,80},{-20,80}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(KCell.q_out, KFluxToPool.q_in) annotation (Line(
            points={{-20,80},{-12,80},{-12,54}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(KPool.SoluteMass, Perm1.u) annotation (Line(
            points={{-24,16.2},{-24,10},{-55,10},{-55,49.4}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KCell.SoluteMass, feedback.u1) annotation (Line(
            points={{-20,72.2},{-20,58},{-9.2,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KCell_CaptiveMass.y, feedback.u2) annotation (Line(
            points={{-2.5,71},{-6,71},{-6,61.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(feedback.y, Perm2.u) annotation (Line(
            points={{-2.4,58},{3.3,58},{3.3,56.6},{3,56.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KFluxToPool.soluteFlow, Perm2.y) annotation (Line(
            points={{-8.4,48},{3,48},{3,49.7}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(NaEffect.u, busConnector. DT_Na_Outflow) annotation (Line(
            points={{43.8,40},{98,40},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(KPool.q_out, DT_K.q_in) annotation (Line(
            points={{-24,24},{22,24},{22,22},{28,22}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(NaEffect.y, KEffect.yBase) annotation (Line(
            points={{34,38},{34,34}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KEffect.y, DT_K.soluteFlow) annotation (Line(
            points={{34,30},{34,25.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(mEq_per_L.y, KEffect.u) annotation (Line(
            points={{18.3,31},{21.15,31},{21.15,32},{24.2,32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(division.u1, KPool.SoluteMass) annotation (Line(
            points={{-18.4,9.2},{-24,9.2},{-24,16.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.ECFV_Vol, division.u2) annotation (Line(
            points={{-88,94},{-96,94},{-96,6.8},{-18.4,6.8}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(division.y, mEq_per_L.u) annotation (Line(
            points={{-13.8,8},{-8,8},{-8,31},{11.4,31}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(electrolytesFlowConstant.y, KidneyFunction.yBase) annotation (
            Line(
            points={{30.6,82},{34,82},{34,66}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KidneyFunction.y, ThiazideEffect.yBase) annotation (Line(
            points={{34,62},{34,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(KidneyFunction.u, busConnector. KidneyFunctionEffect) annotation (Line(
            points={{43.8,64},{98,64},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(DT_K.q_out, Bladder.q_out) annotation (Line(
            points={{40,22},{84,22}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(busConnector.KCell_Mass, KCell.SoluteMass) annotation (Line(
            points={{-88,94},{-4,94},{-4,72},{-20,72},{-20,72.2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(KPool.SoluteMass, busConnector. KPool_mass) annotation (Line(
            points={{-24,16.2},{-24,10},{-96,10},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure.q_in, KPool.q_out) annotation (Line(
            points={{30,6},{22,6},{22,24},{-24,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(concentrationMeasure.actualConc, gain.u) annotation (Line(
            points={{34,6},{36.8,6},{36.8,6},{39.6,6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.y, busConnector. KPool_conc_per_liter) annotation (Line(
            points={{44.2,6},{98,6},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Perm1.y, splineDelayByDay.yBase) annotation (Line(
            points={{-55,56.3},{-55,58},{-50,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(splineDelayByDay.u, busConnector. AldoPool_Aldo) annotation (Line(
            points={{-48,67.8},{-48,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Bladder.q_out, bladderVoid.q_in) annotation (Line(
            points={{84,22},{66,22},{66,-14},{82,-14}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(bladderVoid.solventFlow, busConnector. BladderVoidFlow) annotation (Line(
            points={{88,-10.4},{88,-6},{98,-6},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.SoluteMass, busConnector. GILumenPotasium_Mass) annotation (Line(
            points={{-8,-33.8},{-8,-54},{-96,-54},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(KCell.q_out, concentrationMeasure1.q_in) annotation (Line(
            points={{-20,80},{-8,80}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(concentrationMeasure1.actualConc, busConnector. KCell_conc) annotation (Line(
            points={{-8,84},{-8,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(KPool.q_out, flowMeasure.q_in) annotation (Line(
            points={{-24,24},{-24,30.8}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, KFluxToPool.q_out) annotation (Line(
            points={{-24,37.2},{-24,38},{-12,38},{-12,42}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.q_out, KFluxToCell.q_in) annotation (Line(
            points={{-24,37.2},{-24,38},{-30,38},{-30,52}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, busConnector. PotassiumToCells) annotation (Line(
            points={{-20,34},{-95,34},{-95,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(flowMeasure.q_out, KFluxToCellWithGlucose.q_in) annotation (
            Line(
            points={{-24,37.2},{-24,38},{-68,38},{-68,54}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(KFluxToCellWithGlucose.q_out, KCell.q_out) annotation (Line(
            points={{-68,66},{-68,80},{-20,80}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(IkedaIntoICF.y, KFluxToCell.soluteFlow) annotation (Line(
            points={{-34.8,58},{-33.6,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(splineDelayByDay.y, IkedaIntoICF.yBase) annotation (Line(
            points={{-46,58},{-45,58}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(IkedaIntoICF.Artys_pH, busConnector. Artys_pH) annotation (Line(
            points={{-43,48},{-44,46},{-44,42},{-96,42},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure2.q_in, KPool.q_out) annotation (Line(
            points={{-48,24},{-24,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(concentrationMeasure2.actualConc, IkedaIntoICF.PotasiumECF_conc) 
          annotation (Line(
            points={{-48,28},{-48,40},{-37,40},{-37,48}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(concentrationMeasure2.actualConc, busConnector. KPool_per_liter) annotation (
            Line(
            points={{-48,28},{-48,30},{-96,30},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));

        connect(busConnector.skeletalMuscle_GlucoseToCellStorageFlow,YGLS. u2) 
                   annotation (Line(
            points={{-88,94},{-96,94},{-96,71},{-85,71}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.liver_GlucoseToCellStorageFlow,YGLS. u1) 
          annotation (Line(
            points={{-88,94},{-96,94},{-96,75},{-85,75}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.respiratoryMuscle_GlucoseToCellStorageFlow,YGLS. u3) 
          annotation (Line(
            points={{-88,94},{-96,94},{-96,67},{-85,67}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(YGLS.y, CGL3.u) annotation (Line(
            points={{-73.5,71},{-72,71},{-72,64},{-82,64},{-82,60},{-80.4,60}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(KEffect.y, busConnector. CD_K_Outflow) annotation (Line(
            points={{34,30},{98,30},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure2.actualConc, busConnector.KPool) 
          annotation (Line(
            points={{-48,28},{-96,28},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(GILumen.SoluteMass, busConnector.GILumenPotassium_Mass) 
          annotation (Line(
            points={{-8,-33.8},{-8,-54},{-96,-54},{-96,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(electrolytesFlowConstant1.y, KFluxToCellWithGlucose.soluteFlow) 
          annotation (Line(
            points={{-79.5,51},{-74,51},{-74,60},{-71.6,60}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Leptin, LeptinEffect2.u) annotation (Line(
            points={{-88,94},{-96,94},{-96,-20},{-65.8,-20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Constant7.y,LeptinEffect2. yBase) annotation (Line(
            points={{-57.6,-12},{-56,-12},{-56,-18}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeptinEffect2.y, Diet.desiredFlow) annotation (Line(
            points={{-56,-22},{-56,-28}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(AldoEffect.u, busConnector.NephronAldo_conc_in_nG_per_dl) annotation (
           Line(
            points={{43.8,48},{98,48},{98,94},{-88,94}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(volumeConstant.y, KPool.SolventVolume) annotation (Line(
            points={{-77.6,20},{-40,20},{-40,30},{-32,30}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/K.jpg"), Text(
                extent={{-112,-102},{108,-128}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr><tr>
<td></td>
<td>Noriaki Ikeda: A model of overall regulation of body fluids (1979), Kitasato University</td>
</tr>
</table>
</html>", info="<html>
<p>TODO: Ikeda - Potassium glucose flow from ECF to ICF with insulin affects potassium flow to cells.</p>
</html>"));
      end Potassium2;
    end Potassium;

    package Chloride "Body Chloride Distribution"

      model Chloride2
       // extends Library.Interfaces.BaseModel;
        Library.ConcentrationFlow.ConcentrationCompartment ClPool(
            initialSoluteMass=1515.25,
          stateName="ClPool.Mass",
          STEADY=false) 
          annotation (Placement(transformation(extent={{-78,14},{-58,34}})));
        Library.ConcentrationFlow.ConcentrationCompartment GILumen(
            initialSoluteMass=90, stateName="GILumenChloride.Mass") 
          annotation (Placement(transformation(extent={{-44,-36},{-24,-16}})));
        Library.ConcentrationFlow.SoluteFlowPump Absorbtion 
          annotation (Placement(transformation(extent={{-20,-20},{-8,-32}})));
        Modelica.Blocks.Math.Gain Perm(k=.0015) 
          annotation (Placement(transformation(extent={{-26,-48},{-20,-42}})));
        Library.ConcentrationFlow.OutputPump Hemorrhage 
          annotation (Placement(transformation(extent={{-6,-86},{14,-66}})));
        Library.ConcentrationFlow.OutputPump DialyzerActivity 
          annotation (Placement(transformation(extent={{-6,-100},{14,-80}})));
        Library.ConcentrationFlow.OutputPump SweatDuct 
          annotation (Placement(transformation(extent={{-6,-72},{14,-52}})));
        Library.ConcentrationFlow.InputPump IVDrip 
          annotation (Placement(transformation(extent={{-26,-80},{-6,-60}})));
        Library.ConcentrationFlow.InputPump Transfusion 
          annotation (Placement(transformation(extent={{-26,-94},{-6,-74}})));
        Library.ConcentrationFlow.ConcentrationCompartment Bladder(
            initialSoluteMass=0,
          stateName="BladderChloride.Mass",
          STEADY=false) 
          annotation (Placement(transformation(extent={{50,14},{70,34}})));
        Library.ConcentrationFlow.InputPump Diet 
          annotation (Placement(transformation(extent={{-82,-46},{-62,-26}})));
        Library.ConcentrationFlow.OutputPump Diarrhea 
          annotation (Placement(transformation(extent={{-10,-10},{10,10}},
              rotation=180,
              origin={-52,-44})));
        HumMod.Library.Factors.SplineValue PhEffect(data={{7.00,1.00,0},{7.45,
              0.93,-0.5},{7.80,0.00,0}}) 
          annotation (Placement(transformation(extent={{16,42},{36,62}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-56,40})));
        Modelica.Blocks.Math.Gain gain(k=1000) 
          annotation (Placement(transformation(extent={{-46,38},{-42,42}})));
        Library.ConcentrationFlow.SolventOutflowPump bladderVoid 
          annotation (Placement(transformation(extent={{58,-14},{70,-2}})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-92,-4},{-80,8}}),
              iconTransformation(extent={{62,62},{100,100}})));
        Library.Blocks.Constant Constant7(k=200/1440) 
          annotation (Placement(transformation(extent={{-82,-22},{-74,-14}})));
        Library.Factors.SplineValue LeptinEffect2(data={{0,3.0,0},{8,1.0,-0.04},{
              50,0.0,0}}) 
          annotation (Placement(transformation(extent={{-82,-36},{-62,-16}})));
        Library.ConcentrationFlow.SoluteFlowPump CD_Outflow 
          annotation (Placement(transformation(extent={{10,30},{22,18}})));
        Modelica.Blocks.Math.Sum CD_Cations_Outflow(nin=3) 
          annotation (Placement(transformation(extent={{42,-38},{52,-28}})));
        Modelica.Blocks.Math.Sum CD_AnionsLessCl_Outflow(nin=3) 
          annotation (Placement(transformation(extent={{42,-52},{52,-42}})));
        Modelica.Blocks.Math.Feedback CD_Cl_Outflow 
          annotation (Placement(transformation(extent={{60,-42},{78,-24}})));

      Real ECF_conc;
      equation
         ECF_conc = gain.y;
       connect(ClPool.SolventVolume, busConnector. ECFV_Vol) annotation (Line(
            points={{-76,30},{-81,30},{-81,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.SolventVolume, busConnector. GILumenVolume_Mass) annotation (Line(
            points={{-42,-20},{-49,-20},{-49,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.q_out, Absorbtion.q_in) annotation (Line(
            points={{-34,-26},{-20,-26}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Absorbtion.q_out, ClPool.q_out) annotation (Line(
            points={{-8,-26},{-6,-26},{-6,24},{-68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Hemorrhage.q_in, ClPool.q_out) annotation (Line(
            points={{-2,-76},{-6,-76},{-6,24},{-68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(DialyzerActivity.q_in, ClPool.q_out) annotation (Line(
            points={{-2,-90},{-6,-90},{-6,24},{-68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(ClPool.q_out, SweatDuct.q_in) annotation (Line(
            points={{-68,24},{-6,24},{-6,-62},{-2,-62}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Transfusion.q_out, ClPool.q_out) annotation (Line(
            points={{-10,-84},{-6,-84},{-6,24},{-68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(IVDrip.q_out, ClPool.q_out) annotation (Line(
            points={{-10,-70},{-6,-70},{-6,24},{-68,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Bladder.SolventVolume, busConnector. BladderVolume_Mass) annotation (Line(
            points={{52,30},{46,30},{46,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(IVDrip.desiredFlow, busConnector. IVDrip_ClRate) annotation (Line(
            points={{-16,-66},{-16,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Transfusion.desiredFlow, busConnector. Transfusion_ClRate) annotation (Line(
            points={{-16,-80},{-38,-80},{-38,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(SweatDuct.desiredFlow, busConnector. SweatDuct_ClRate) annotation (Line(
            points={{4,-58},{4,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(busConnector.Hemorrhage_ClRate, Hemorrhage.desiredFlow) annotation (Line(
            points={{-86,2},{4.5,2},{4.5,-72},{4,-72}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(DialyzerActivity.desiredFlow, busConnector. DialyzerActivity_Cl_Flux) 
          annotation (Line(
            points={{4,-86},{4,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Diet.q_out, GILumen.q_out) annotation (Line(
            points={{-66,-36},{-42,-36},{-42,-26},{-34,-26}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Diarrhea.desiredFlow, busConnector.GILumenVomitus_ClLoss) 
                                                         annotation (Line(
            points={{-52,-48},{-52,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(GILumen.q_out, Diarrhea.q_in) annotation (Line(
            points={{-34,-26},{-42,-26},{-42,-44},{-46,-44}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Perm.y, Absorbtion.soluteFlow) annotation (Line(
            points={{-19.7,-45},{-14,-45},{-14,-29.6}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(Perm.u, GILumen.SoluteMass) annotation (Line(
            points={{-26.6,-45},{-34,-45},{-34,-33.8}},
            color={0,0,127},
            smooth=Smooth.None));

        connect(busConnector.CollectingDuct_NetSumCats,PhEffect. yBase) annotation (Line(
            points={{-86,2},{-90,2},{-90,64},{26,64},{26,54}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(ClPool.SoluteMass, busConnector. ClPool_mass) annotation (Line(
            points={{-68,16.2},{-68,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure.actualConc, gain.u) annotation (Line(
            points={{-52,40},{-46.4,40}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.y, busConnector. ClPool_conc_per_liter) annotation (Line(
            points={{-41.8,40},{-69.9,40},{-69.9,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(ClPool.q_out, concentrationMeasure.q_in) annotation (Line(
            points={{-68,24},{-62,24},{-62,40},{-56,40}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(Bladder.q_out, bladderVoid.q_in) annotation (Line(
            points={{60,24},{50,24},{50,-8},{58,-8}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(bladderVoid.solventFlow, busConnector. BladderVoidFlow) annotation (Line(
            points={{64,-4.4},{64,2},{-86,2}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));

        connect(busConnector.Artys_pH,PhEffect. u) annotation (Line(
            points={{-86,2},{-90,2},{-90,52},{16.2,52}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.Leptin, LeptinEffect2.u) annotation (Line(
            points={{-86,2},{-90,2},{-90,-26},{-81.8,-26}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Constant7.y,LeptinEffect2. yBase) annotation (Line(
            points={{-73.6,-18},{-72,-18},{-72,-24}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeptinEffect2.y, Diet.desiredFlow) annotation (Line(
            points={{-72,-28},{-72,-32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(ClPool.q_out, CD_Outflow.q_in) annotation (Line(
            points={{-68,24},{10,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(CD_Outflow.q_out, Bladder.q_out) annotation (Line(
            points={{22,24},{60,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(busConnector.CD_Na_Outflow, CD_Cations_Outflow.u[1]) annotation (Line(
            points={{-86,2},{10,2},{10,-32},{40,-32},{40,-33.6667},{41,-33.6667}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.CD_NH4_Outflow, CD_Cations_Outflow.u[2]) annotation (
            Line(
            points={{-86,2},{10,2},{10,-38},{41,-38},{41,-33}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,0},{-6,0}}));
        connect(busConnector.CD_K_Outflow, CD_Cations_Outflow.u[3]) annotation (Line(
            points={{-86,2},{10,2},{10,-34},{40,-34},{40,-32.3333},{41,-32.3333}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-3},{-6,-3}}));
        connect(busConnector.CD_PO4_mEq_Outflow, CD_AnionsLessCl_Outflow.u[1]) 
          annotation (Line(
            points={{-86,2},{10,2},{10,-46},{40,-46},{40,-47.6667},{41,-47.6667}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(busConnector.CD_SO4_Outflow, CD_AnionsLessCl_Outflow.u[2]) 
          annotation (Line(
            points={{-86,2},{10,2},{10,-48},{40,-48},{41,-47}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,0},{-6,0}}));
        connect(busConnector.CD_KA_Outflow, CD_AnionsLessCl_Outflow.u[3]) annotation (
           Line(
            points={{-86,2},{10,2},{10,-52},{40,-52},{40,-46.3333},{41,-46.3333}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,-3},{-6,-3}}));
        connect(CD_Cations_Outflow.y, CD_Cl_Outflow.u1) annotation (Line(
            points={{52.5,-33},{52,-34},{52,-33},{61.8,-33}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(CD_AnionsLessCl_Outflow.y, CD_Cl_Outflow.u2) annotation (Line(
            points={{52.5,-47},{69,-40.2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(CD_Cl_Outflow.y, CD_Outflow.soluteFlow) annotation (Line(
            points={{77.1,-33},{86,-33},{86,-18},{16,-18},{16,20.4}},
            color={0,0,127},
            smooth=Smooth.None));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/Cl.jpg"), Text(
                extent={{-112,-102},{108,-128}},
                lineColor={0,0,255},
                textString="%name")}),
          Documentation(revisions="<html>
<table>
<tr>
<td>Author:</td>
<td>Marek Matejak</td>
</tr>
<tr>
<td>Copyright:</td>
<td>In public domains</td>
</tr>
<tr>
<td>By:</td>
<td>Charles University, Prague</td>
</tr>
<tr>
<td>Date of:</td>
<td>2009</td>
</tr>
<tr>
<td>References:</td>
<td>Tom Coleman: QHP 2008 beta 3, University of Mississippi Medical Center</td>
</tr>
</table>
</html>"));
      end Chloride2;
    end Chloride;

    package Phosphate "Body Phosphate Distribution"

      model GlomerulusStrongAnionFiltration

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

        Library.Interfaces.RealInput_ Cations( final quantity="Concentration", final unit =    "mEq/l") annotation (Placement(
              transformation(extent={{-80,30},{-40,70}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={-60,50})));
        Library.Interfaces.RealInput_ ProteinAnions(final quantity=
              "Concentration", final unit="mEq/l")                                                            annotation (Placement(
              transformation(extent={{-40,30},{0,70}}), iconTransformation(
              extent={{-20,-20},{20,20}},
              rotation=270,
              origin={-20,50})));
        Real KAdjustment;
        //Real Cations( final quantity="Concentration", final unit = "mEq/l");
        Real Anions( final quantity="Concentration", final unit = "mEq/l");
      //  Real ProteinAnions( final quantity="Concentration", final unit = "mEq/l");
      /*  Library.Interfaces.RealInput_ HCO3(final quantity="Concentration",
      final unit="mEq/l")                                                                               annotation (Placement(
        transformation(extent={{0,30},{40,70}}),  iconTransformation(
        extent={{-20,-20},{20,20}},
        rotation=270,
        origin={20,50})));
*/
      equation
        q_in.q + q_out.q = 0;
        //Cations = q_in.conc*1000+otherCations;
        Anions = Cations;
      //  ProteinAnions = Anions - otherStrongAnions - q_in.conc*1000 - HCO3;
        KAdjustment = (Cations-(Anions-ProteinAnions))/(Cations+(Anions-ProteinAnions));
        q_out.conc = (1+KAdjustment)*q_in.conc;

       annotation (
          Icon(coordinateSystem(preserveAspectRatio=true, extent={{-100,-100},{
                  100,100}}), graphics={
              Rectangle(
                extent={{-100,-50},{100,50}},
                lineColor={0,0,127},
                fillColor={255,255,255},
                fillPattern=FillPattern.Solid),
              Text(
                extent={{-150,-100},{150,-60}},
                textString="%name",
                lineColor={0,0,255}),
              Line(
                points={{0,42},{0,26}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,20},{0,4}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,-4},{0,-20}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5),
              Line(
                points={{0,-26},{0,-42}},
                color={0,0,255},
                smooth=Smooth.None,
                thickness=0.5)}),       Diagram(coordinateSystem(preserveAspectRatio=true,
                         extent={{-100,-100},{100,100}}), graphics),
                    Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}), graphics));
      end GlomerulusStrongAnionFiltration;

      model Phosphate2
       // extends Library.Interfaces.BaseModel;
        Library.ConcentrationFlow.ConcentrationCompartment PO4Pool(
            initialSoluteMass=2.43011, stateName="PO4Pool.Mass") 
          annotation (Placement(transformation(extent={{-64,14},{-44,34}})));
        Library.ConcentrationFlow.ConcentrationCompartment Bladder(
            initialSoluteMass=0,
          stateName="BladderPhosphate.Mass",
          STEADY=false) 
          annotation (Placement(transformation(extent={{50,14},{70,34}})));
        Library.ConcentrationFlow.InputPump Diet 
          annotation (Placement(transformation(extent={{-84,-4},{-64,16}})));
        Library.ConcentrationFlow.SolventFlowPump glomerulusPhosphateRate 
          annotation (Placement(transformation(extent={{0,14},{20,34}})));
        GlomerulusStrongAnionFiltration glomerulus 
          annotation (Placement(transformation(extent={{-28,14},{-8,34}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure 
          annotation (Placement(transformation(
              extent={{-10,-10},{10,10}},
              rotation=270,
              origin={-28,2})));
        Modelica.Blocks.Math.Gain gain(k=1000) 
          annotation (Placement(transformation(extent={{-18,0},{-14,4}})));
        Library.ConcentrationFlow.FlowMeasure flowMeasure 
          annotation (Placement(transformation(extent={{20,34},{40,14}})));
        Modelica.Blocks.Math.Gain gain1(k=.5) 
          annotation (Placement(transformation(extent={{-50,-14},{-46,-10}})));
        Library.ConcentrationFlow.SolventOutflowPump bladderVoid 
          annotation (Placement(transformation(extent={{66,-20},{78,-8}})));
        Library.ConcentrationFlow.ConcentrationMeasure concentrationMeasure1(
            unitsString="mmol/l", toAnotherUnitCoef=1000) annotation (Placement(
              transformation(
              extent={{-10,-10},{10,10}},
              rotation=0,
              origin={-60,60})));
        Library.Interfaces.BusConnector busConnector 
          annotation (Placement(transformation(extent={{-90,76},{-78,88}}),
              iconTransformation(extent={{60,60},{100,100}})));
        Library.Blocks.Constant Constant7(k=30/1440) 
          annotation (Placement(transformation(extent={{-84,24},{-76,32}})));
        Library.Factors.SplineValue LeptinEffect2(data={{0,3.0,0},{8,1.0,-0.04},{
              50,0.0,0}}) 
          annotation (Placement(transformation(extent={{-84,10},{-64,30}})));
        AcidDissociation acidDissociation 
          annotation (Placement(transformation(extent={{30,-50},{50,-30}})));
      Real ECF_conc;
      equation
        ECF_conc = concentrationMeasure1.actualConc;
       connect(PO4Pool.SolventVolume, busConnector. ECFV_Vol) 
                                                annotation (Line(
            points={{-62,30},{-67,30},{-67,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Bladder.SolventVolume, busConnector. BladderVolume_Mass) annotation (Line(
            points={{52,30},{46,30},{46,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));

        connect(Diet.q_out, PO4Pool.q_out) annotation (Line(
            points={{-68,6},{-60,6},{-60,24},{-54,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(glomerulusPhosphateRate.solventFlow, busConnector. GlomerulusFiltrate_GFR) 
          annotation (Line(
            points={{10,30},{10,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(glomerulus.q_out, glomerulusPhosphateRate.q_in) 
                                                             annotation (Line(
            points={{-8,24},{0,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(PO4Pool.q_out, glomerulus.q_in) annotation (Line(
            points={{-54,24},{-28,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(concentrationMeasure.actualConc, gain.u) annotation (Line(
            points={{-24,2},{-21.2,2},{-21.2,2},{-18.4,2}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain.y, busConnector. PO4Pool_conc_per_liter) annotation (Line(
            points={{-13.8,2},{-48.9,2},{-48.9,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure.q_in, PO4Pool.q_out) annotation (Line(
            points={{-28,2},{-34,2},{-34,24},{-54,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(flowMeasure.actualFlow, busConnector. CD_PO4_Outflow) annotation (Line(
            points={{30,19},{30,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(flowMeasure.q_out, Bladder.q_out) annotation (Line(
            points={{34,24},{60,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(glomerulusPhosphateRate.q_out, flowMeasure.q_in) annotation (
            Line(
            points={{20,24},{26,24}},
            color={200,0,0},
            smooth=Smooth.None,
            thickness=1));
        connect(busConnector.BloodIons_Cations, glomerulus.Cations) annotation (Line(
            points={{-84,82},{-24,82},{-24,29}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%first",
       index=-1,
       extent={{-3,1},{-3,1}}));
        connect(PO4Pool.SoluteMass, gain1.u) annotation (Line(
            points={{-54,16.2},{-54,-12},{-50.4,-12}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(gain1.y, busConnector. PO4Pool_Osmoles) annotation (Line(
            points={{-45.8,-12},{-64.9,-12},{-64.9,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(Bladder.q_out, bladderVoid.q_in) annotation (Line(
            points={{60,24},{50,24},{50,-14},{66,-14}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));
        connect(bladderVoid.solventFlow, busConnector. BladderVoidFlow) annotation (Line(
            points={{72,-10.4},{72,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure1.actualConc, busConnector. ctPO4) annotation (Line(
            points={{-60,64},{-60,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
       string="%second",
       index=1,
       extent={{3,1},{3,1}}));
        connect(concentrationMeasure1.q_in, PO4Pool.q_out) annotation (Line(
            points={{-60,60},{-54,60},{-54,24}},
            color={200,0,0},
            thickness=1,
            smooth=Smooth.None));

        connect(busConnector.Leptin, LeptinEffect2.u) annotation (Line(
            points={{-84,82},{-92,82},{-92,20},{-83.8,20}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(Constant7.y,LeptinEffect2. yBase) annotation (Line(
            points={{-75.6,28},{-74,28},{-74,22}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(LeptinEffect2.y, Diet.desiredFlow) annotation (Line(
            points={{-74,18},{-74,10}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(flowMeasure.actualFlow, acidDissociation.CD_tPO4) annotation (Line(
            points={{30,19},{32,19},{32,-20},{20,-20},{20,-32},{30,-32}},
            color={0,0,127},
            smooth=Smooth.None));
        connect(busConnector.Urine_pH, acidDissociation.PHU) annotation (Line(
            points={{-84,82},{-92,82},{-92,-38},{30,-38},{30,-38}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        connect(acidDissociation.CD_PO4_mEq, busConnector.CD_PO4_mEq_Outflow) 
          annotation (Line(
            points={{50,-46},{98,-46},{98,82},{-84,82}},
            color={0,0,127},
            smooth=Smooth.None), Text(
            string="%second",
            index=1,
            extent={{6,3},{6,3}}));
        connect(busConnector.BloodIons_ProteinAnions, glomerulus.ProteinAnions) 
          annotation (Line(
            points={{-84,82},{-20,82},{-20,29}},
            color={0,0,255},
            thickness=0.5,
            smooth=Smooth.None), Text(
            string="%first",
            index=-1,
            extent={{-6,3},{-6,3}}));
        annotation (Diagram(coordinateSystem(preserveAspectRatio=true, extent={{-100,
                  -100},{100,100}}),       graphics), Icon(coordinateSystem(
                preserveAspectRatio=true, extent={{-100,-100},{100,100}}),
              graphics={Bitmap(extent={{-100,100},{100,-100}}, fileName=
                    "icons/fosfat.png"), Text(
                extent={{-108,-98},{112,-124}},
                lineColor={0,0,255},
                textString="%name")}));
      end Phosphate2;

      model AcidDissociation

        Library.Interfaces.RealInput CD_tPO4(unit="mmol/min")
          "total PO4 in mmol/min" 
          annotation (Placement(transformation(extent={{-120,60},{-80,100}}),
              iconTransformation(extent={{-120,60},{-80,100}})));
        Library.Interfaces.RealInput PHU "pH of urine in collecting duct" annotation (
           Placement(transformation(extent={{-120,60},{-80,100}}), iconTransformation(
                extent={{-120,0},{-80,40}})));
        Library.Interfaces.RealOutput CD_PO4_mEq(unit="mEq/min") annotation (
            Placement(transformation(extent={{80,-80},{120,-40}}), iconTransformation(
                extent={{80,-80},{120,-40}})));

        Real H2PO4;
        Real HPO4;
      equation
        HPO4/H2PO4 = 10^(PHU-6.8);
        CD_tPO4 = HPO4 + H2PO4;
        CD_PO4_mEq = 2*HPO4 + H2PO4;
      end AcidDissociation;
    end Phosphate;

    package Sulphate "Body Sulphate Distribution"

      model GlomerulusStrongAnionFiltration

        HumMod.Library.ConcentrationFlow.NegativeConcentrationFlow q_out 
                               annotation (extent=[-10, -110; 10, -90], Placement(
              transformation(extent={{90,-10},{110,10}})));


        HumMod.Library.ConcentrationFlow.PositiveConcentrationFlow q_in 
                                  annotation (Placement(
              transformation(extent={{-120,-20},{-80,20}}), iconTransformation(extent=
                 {{-110,-10},{-90,10}})));

        Library.Interfaces.RealInput_ Cations( final quantity="Concentration", final unit =    "mEq/l") annotation (Placement(
              transformation(extent={{-80,30},{-40,70}}), ic