// This file defines templates for transforming Modelica code to C# code.

package CodegenCSharp

import interface SimCodeTV;

// SECTION: SIMULATION TARGET, ROOT TEMPLATE

template translateModel(SimCode simCode) ::=
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
    let()= textFile(simulationFile(simCode), '<%dotPath(modelInfo.name)%>.cs')
    //let()= textFile(simulationFunctionsFile(simCode), '<%modelInfo.name%>_functions.cs')
    "" // empty result of the top-level template .., only side effects
end translateModel;


// SECTION: SIMULATION TARGET, C# FILE SPECIFIC TEMPLATES


template simulationFile(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<<
// Simulation code for <%dotPath(modelInfo.name)%> generated by the OpenModelica Compiler.

using System;
using Bodylight.Solvers;
namespace Bodylight.Models<%modelNameSpace(modelInfo.name)%>
{
  public partial class <%lastIdentOfPath(modelInfo.name)%> : DAESystem
  {
    <%modelDataMembers(modelInfo, simCode)%>
    
    <%let fbody = simulationFunctionsBody(simCode)
      if fbody then 
        <<
        #region Functions
        <%fbody%>
        #endregion
        >>
      else "//** No functions **"
    %>
    
    <%functionCallExternalObjectConstructors(extObjInfo, simCode)%>
    
    <%functionExtraResiduals(allEquations, simCode)%>
      
    <%functionInput(modelInfo, simCode)%>

    <%functionOutput(modelInfo, simCode)%>
    
    <%functionInitSample(sampleConditions, simCode)%>    
    
    <%functionSampleEquations(sampleEquations, simCode)%>
    
    <%functionStoreDelayed(simCode)%>

    <%functionInitial(startValueEquations, simCode)%>

    <%functionInitialResidual(residualEquations, simCode)%>

    <%functionBoundParameters(parameterEquations, simCode)%>

    <%functionODE(odeEquations, simCode)%>

    <%functionAlgebraic(algebraicEquations, simCode)%>

    <%functionAliasEquation(removedEquations, simCode)%>

    <%functionDAE(allEquations, whenClauses, helpVarInfo, simCode)%>

    <%functionOnlyZeroCrossing(zeroCrossings, simCode)%>
    
    <%functionCheckForDiscreteChanges(simCode)%>

    <%/*functionAssertsforCheck(algorithmAndEquationAsserts)*/%>
    
  }
}
>>
end simulationFile;

template modelNameSpace(Path modelName) ::=
  match modelName
  case QUALIFIED(__) then '.<%name%><%modelNameSpace(path)%>'
  case FULLYQUALIFIED(__) then modelNameSpace(path)
end modelNameSpace;

template lastIdentOfPath(Path modelName) ::=
  match modelName
  case QUALIFIED(__) then lastIdentOfPath(path)
  case IDENT(__)     then name
  case FULLYQUALIFIED(__) then lastIdentOfPath(path)
end lastIdentOfPath;

template simulationFunctionsBody(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = modelInfo as MODELINFO(__)) then
    (modelInfo.functions |> fn => match fn  
          case FUNCTION(__)           then functionBodyRegularFunction(fn, simCode)
          case EXTERNAL_FUNCTION(__)  then functionBodyExternalFunction(fn, simCode) //'EXTERNAL_FUN_NOT_IMPLEMETED(name=<%dotPath(name)%>)'
          case RECORD_CONSTRUCTOR(__) then 'RECORD_CONSTRUCTOR_NOT_IMPLEMENTED(name=<%dotPath(name)%>)'
          else "UNKNOWN_FUNCTION"
    ;separator="\n")
end simulationFunctionsBody;

/*
//not used now
template simulationFunctionsFile(SimCode simCode) ::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<<
// Simulation functions code for <%modelInfo.name%> generated by the OpenModelica Compiler.

using System;
using Bodylight.Solvers;
namespace Bodylight.Models
{
  public partial class <%dotPath(modelInfo.name)%> : DAESystem
  {
    
    <%simulationFunctionsBody(simCode)%>
          
  }
}
>>
end simulationFunctionsFile;
*/

template recordDeclaration(RecordDeclaration recDecl, SimCode simCode)
 "Generates structs for a record declaration."
::=
  match recDecl
  case RECORD_DECL_FULL(__) then
    <<
    struct <%name%> {
      <%variables |> var as VARIABLE(__) => '<%varType(var)%> <%crefStr(var.name,simCode)%>;' ;separator="\n"%>
    };    
    >> 
  case RECORD_DECL_DEF(__) then "RECORD_DECL_DEF_NOT_SUPPORTED"
  else "UNKNOWN_RECORD_DECL_"
end recordDeclaration;



template functionBodyRegularFunction(Function fn, SimCode simCode)
 "Generates the body for a Modelica/MetaModelica function."
::=
match fn
case FUNCTION(__) then
  let()= System.tmpTickReset(1)
  let fname = underscorePath(name)
  let retType = match outVars //a hack, only one output var for now
                case fv :: _ then varType(fv) 
                else "void"
  let retVar = match outVars 
               case (fv as VARIABLE(__)) :: _ then crefStr(fv.name,simCode) 
  let varInits = variableDeclarations |> var => varInit(var, simCode) 
                 ;separator="\n" //TODO:special end of line,see varInit    
  let bodyPart = (body |> stmt  => funStatement(stmt, simCode) ;separator="\n")
  <<
  <%retType%> _<%fname%>(<%functionArguments |> var => funArgDefinition(var,simCode) ;separator=", "%>)
  {
    <%varInits%>

    <%bodyPart%>
    
    _return:
    <%if outVars then 'return <%retVar%>;'%>
  }
  
  >>
end functionBodyRegularFunction;


template functionBodyExternalFunction(Function fn, SimCode simCode)
 "Generates the body for an external function (just a wrapper)."
::=
match fn
case efn as EXTERNAL_FUNCTION(__) then
  let()= System.tmpTickReset(1)
  let fname = underscorePath(name)
  let retType = match outVars 
                case fv :: _ then varType(fv)
                else "void"
  let &preExp = buffer "" /*BUFD*/
  let callPart = //extFunCall(fn, &preExp /*BUFC*/, &varDecls /*BUFD*/)
      let args = (extArgs |> arg => extArg(arg, &preExp, simCode) ;separator=", ")
      let lib = hackGetFirstExternalFunctionLib(libs) // match libs case fnLib::_ then fnLib else "NO_EXT_LIB" 
      'Bodylight.Solvers.ExternalLibraries.<%lib%>.<%extName%>(<%args%>);'
  <<
  static <%retType%> _<%fname%>(<%funArgs |> var => funArgDefinition(var,simCode) ;separator=", "%>)
  {
    <%preExp%>
    <%if outVars then "return "
    %><%callPart%>    
  }
  
  >>  
end functionBodyExternalFunction;


template extArg(SimExtArg extArg, Text &preExp, SimCode simCode)
 "Helper to extFunCall."
::=
  match extArg
  case SIMEXTARG(cref=c, outputIndex=oi, isArray=true, type_=t) then
    let name = if oi then 'NOT_SUPPORTED_out.targ<%oi%>' else crefStr(c,simCode)
    name
    //let shortTypeStr = expTypeShort(t)
    //'(<%extType(t,isInput,true)%>) data_of_<%shortTypeStr%>_array(&(<%name%>))'
  case SIMEXTARG(cref=c, isInput=ii, outputIndex=0, type_=t) then
    let cr = crefStr(c,simCode)
    cr    
  case SIMEXTARG(cref=c, isInput=ii, outputIndex=oi, type_=t) then
    'NOT_SUPPORTED_ext<%oi%>'
  case SIMEXTARGEXP(__) then
    daeExp(exp, contextFunction, &preExp, simCode)
  case SIMEXTARGSIZE(cref=c) then
    let name = if outputIndex then 'NOT_SUPPORTED_SIZE_out.targ<%outputIndex%>' else crefStr(c,simCode)
    match exp
    case ICONST(__) then 
      '<%name%>.size<%integer%>'
    else
      '<%name%>.size(<%daeExp(exp, contextFunction, &preExp, simCode)%>)'
end extArg;


template varInit(Variable var, SimCode simCode)
 "Generates code to initialize variables."
::=
match var
case var as VARIABLE(__) then
  if instDims then
    let varName = crefStr(var.name,simCode)
    let &preDimsExp = buffer ""
    let instDimsInit = (var.instDims |> exp =>
          daeExp(exp, contextFunction, &preDimsExp, simCode)
        ;separator=", ")
    <<
    <%preDimsExp%>
    var <%varName%> = new <%expTypeArray(var.ty,listLength(instDims))%>(<%instDimsInit%>);<% 
      //special \n solution ... the new line is dependent on this line, but the varInit as a whole does not put \n
      match var.value  case SOME(CREF(componentRef = cr)) then 
      '<%\n%><%varName%>.CopyFrom(<%crefStr(cr,simCode)%>);'
    %>  
    >>
  else
    <<
    <%varType(var)%> <%crefStr(var.name,simCode)%>;
    >>
else "UNSUPPORTED_VARIABLE_varInit"

end varInit;

template funArgDefinition(Variable var, SimCode simCode)
::=
  match var
  case VARIABLE(__) then '<%varType(var)%> <%crefStr(name,simCode)%>'
  case FUNCTION_PTR(__) then 'funArgDefinition_UNSUPPORTED_fnptr <%name%>'
  else "UNSUPPORTED_VARIABLE_funArgDefinition"
end funArgDefinition;

template funStatement(Statement stmt, SimCode simCode)
 "Generates function statements."
::=
  match stmt
  case ALGORITHM(__) then
    (statementLst |> stmt =>
      algStatement(stmt, contextFunction, simCode)
    ;separator="\n") 
  else
    "NOT IMPLEMENTED FUN STATEMENT"
end funStatement;

constant String localRepresentationArrayDefines =
"var X = states; var Xd = statesDerivatives; var Y = algebraics; var P = parameters; var H = helpVars; var EO = externalObjects;
var preX = savedStates; var preXd = savedStatesDerivatives; var preY = savedAlgebraics; var preH = savedHelpVars;
var preYI = savedAlgebraicsInt; var preYB = savedAlgebraicsBool; var YI = algebraicsInt; var YB = algebraicsBool; var PI = parametersInt; var PB = parametersBool; ";
       

template modelDataMembers(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
#region Model description
const int 
  NHELP = <%varInfo.numHelpVars%>, 
  NG = <%varInfo.numZeroCrossings%>,
  NG_SAM = <%varInfo.numTimeEvents%>,
  NX = <%varInfo.numStateVars%>, 
  NY = <%varInfo.numAlgVars%>, 
  NA = <%varInfo.numAlgAliasVars%>, // number of alias variables
  NP = <%varInfo.numParams%>,
  NO = <%varInfo.numOutVars%>, 
  NI = <%varInfo.numInVars%>, 
  NR = <%varInfo.numResiduals%>,
  NEXT = <%varInfo.numExternalObjects%>, 
  //NFUNC = <%listLength(functions)%>, // number of functions used by the simulation
  NYSTR = <%varInfo.numStringAlgVars%>, 
  NASTR = <%varInfo.numStringAliasVars%>, // number of alias string variables
  NYI = <%varInfo.numIntAlgVars%>,
  NAI = <%varInfo.numIntAliasVars%>, // number of alias int variables
  NYB = <%varInfo.numBoolAlgVars%>,
  NAB = <%varInfo.numBoolAliasVars%>, // number of alias bool variables
  NPI = <%varInfo.numIntParams%>, 
  NPB = <%varInfo.numBoolParams%>,
  NPSTR = <%varInfo.numStringParamVars%>;

public override string ModelName        { get { return "<%dotPath(name)%>"; }}
public override int HelpVarsCount       { get { return NHELP; } }
public override int ZeroCrossingsCount  { get { return NG; } }
public override int SampleTypesCount    { get { return NG_SAM; } }
public override int StatesCount         { get { return NX; } }
public override int AlgebraicsCount     { get { return NY; } }
public override int AlgebraicsIntCount  { get { return NYI; } }
public override int AlgebraicsBoolCount { get { return NYB; } }
public override int ParametersCount     { get { return NP; } }
public override int ParametersIntCount  { get { return NPI; } }
public override int ParametersBoolCount { get { return NPB; } }
        
public override int OutputsCount   { get { return NO; } }
public override int InputsCount    { get { return NI; } }
public override int ResidualsCount { get { return NR; } }
public override int ExternalObjectsCount { get { return NEXT; } }        
public override int MaximumOrder { get { return 5; } }
public override int StringVarsCount { get { return NYSTR; } }
public override int StringParametersCount { get { return NPSTR; } }

#endregion

#region VariableInfos
public static readonly SimVarInfo[] VariableInfosStatic = new[] {
    <%{  
        varInfos("State", vars.stateVars, false, simCode),
        varInfos("StateDer", vars.derivativeVars, false, simCode),
        varInfos("Algebraic", vars.algVars, false, simCode),
        varInfos("alias Algebraic", vars.aliasVars, false, simCode),
        varInfos("AlgebraicInt", vars.intAlgVars, false, simCode),
        varInfos("alias AlgebraicInt", vars.intAliasVars, false, simCode),
        varInfos("AlgebraicBool", vars.boolAlgVars, false, simCode),
        varInfos("alias AlgebraicBool", vars.boolAliasVars, false, simCode),
        varInfos("Parameter", vars.paramVars, true, simCode),
        varInfos("ParameterInt", vars.intParamVars, true, simCode),
        varInfos("ParameterBool", vars.boolParamVars, true, simCode)
    } ;separator=",\n\n"%>
};
public override SimVarInfo[] VariableInfos { get { return VariableInfosStatic; } }
#endregion

#region InitialFixed
private static readonly bool[] InitialFixedStatic = new bool[NX + NX + NY + NYI +NYB + NP + NPI + NPB] {
    <%{ (if vars.stateVars      then "//states\n" + initFixed(vars.stateVars, simCode)),
        (if vars.derivativeVars then "//derivatives\n" + initFixed(vars.derivativeVars, simCode)),
        (if vars.algVars        then "//algebraics\n" + initFixed(vars.algVars, simCode)), 
        (if vars.intAlgVars     then "//algebraicsInt\n" + initFixed(vars.intAlgVars, simCode)), 
        (if vars.boolAlgVars    then "//algebraicsBool\n" + initFixed(vars.boolAlgVars, simCode)), 
        (if vars.paramVars      then "//parameters\n" + initFixed(vars.paramVars, simCode)),
        (if vars.intParamVars   then "//parametersInt\n" + initFixed(vars.intParamVars, simCode)),
        (if vars.boolParamVars  then "//parametersBool\n" + initFixed(vars.boolParamVars, simCode))
      } ;separator=",\n"%>
};
public override bool[] InitialFixed { get { return InitialFixedStatic; } }
#endregion

#region Constructor - Init Data
public <%lastIdentOfPath(name)%>() {
    CreateData();
    
    //**** states *****
    <%initVals("states", vars.stateVars, simCode)%>
          
    //**** state derivatives *****
    //all are default values ... 0.0
    <%/* TODO: is it correct to have derivatives without initial values at all ?? 
      initVals("statesDerivatives", vars.derivativeVars, simCode)
    */%>
    
    //**** algebraics *****  
    <%initVals("algebraics", vars.algVars, simCode)%>
    
    //**** algebraics Int *****  
    <%initVals("algebraicsInt", vars.intAlgVars, simCode)%>
    
    //**** algebraics Bool *****  
    <%initVals("algebraicsBool", vars.boolAlgVars, simCode)%>
    
    //**** parameters *****  
    <%initVals("parameters", vars.paramVars, simCode)%>
    
    //**** parameters Int *****  
    <%initVals("parametersInt", vars.intParamVars, simCode)%>
    
    //**** parameters Bool *****  
    <%initVals("parametersBool", vars.boolParamVars, simCode)%>
}
#endregion
>>
end modelDataMembers;

template simVarTypeName(VarKind varKind, Type type_) ::=
  match varKind 
  case VARIABLE(__)    then "Algebraic" + simVarTypeNamePostfix(type_)
  case STATE(__)       then "State"
  case STATE_DER(__)   then "StateDer"
  case DUMMY_DER(__)   //then "Algebraic" 
  case DUMMY_STATE(__) then "Algebraic" 
  case DISCRETE(__)    then 'Algebraic<%simVarTypeNamePostfix(type_)%>/*d*/'
  case PARAM(__)       then "Parameter" + simVarTypeNamePostfix(type_)
  else error(sourceInfo(), "Unexpected simVarTypeName varKind")
  //case EXTOBJ(__)      then "EO"
  //case CONST(__)       then "CONST_VAR_KIND"
end simVarTypeName;

template simVarTypeNamePostfix(Type type_) ::=
  match type_ 
  case T_INTEGER(__)  then "Int"
  case T_REAL(__) then ""
  case T_BOOL(__) then "Bool"
  case T_STRING(__) then "String"
  case T_ENUMERATION(__) then "/*Enum*/"
  //case T_COMPLEX(__) then "Complex"
  //case T_ARRAY(__) then "Array"
  //case T_OTHER(__) then "Other"
  else error(sourceInfo(), "Unknown simVarTypeNamePostfix")
end simVarTypeNamePostfix;

template varInfos(String regionName, list<SimVar> varsLst, Boolean isMInd, SimCode simCode) ::=
  if varsLst then 
    <<
    #region <%regionName%> variable infos
    <% varsLst |> sv as SIMVAR(__) => 
       let typeIdxAndAliasMode = 
          match aliasvar
          case NOALIAS(__)      then '<%simVarTypeName(sv.varKind, sv.type_)%>,<%sv.index%>, 0'
          case ALIAS(__)        then '<%cref2simvar(varName, simCode) |> SIMVAR(__) => '<%simVarTypeName(varKind, type_)%>, <%index%>'%>, 1/*alias to <%crefStr(varName, simCode)%>*/'
          case NEGATEDALIAS(__) then '<%cref2simvar(varName, simCode) |> SIMVAR(__) => '<%simVarTypeName(varKind, type_)%>, <%index%>'%>, -1/*alias to <%crefStr(varName, simCode)%>*/'
          else error(sourceInfo(), "Unknown alias var type")           
       <<
       new SimVarInfo( "<%crefStr(name, simCode)%>", "<%Util.escapeModelicaStringToCString(comment)%>", SimVarType.<%typeIdxAndAliasMode%>)
       >> ;separator=",\n"
    %>
    #endregion<%\n%>    
    >>
end varInfos;

template initVals(String arrName, list<SimVar> varsLst, SimCode simCode) ::=
  varsLst |> sv as SIMVAR(__) =>
    match initialValue 
      case SOME(v) then 
      let &preExp = buffer "" //dummy ... the value is always a constant
      match daeExp(v, contextOther, &preExp, simCode) 
      case vStr as "0"
      case vStr as "0.0"
      case vStr as "false"
      case vStr as "(0)" then
        '//<%arrName%>[<%sv.index%>] = <%vStr%>; //<%crefStr(sv.name, simCode)%> --> zero val' 
      //case "true" then //a hack for now
      // '<%arrName%>[<%sv.index%>] = 1.0; //true //<%crefStr(sv.name, simCode)%>'
      case vStr then
       '<%arrName%>[<%sv.index%>] = <%vStr%>; //<%crefStr(sv.name, simCode)%>'
        end match
      else '//<%arrName%>[<%sv.index%>] = 0.0; //<%crefStr(sv.name, simCode)%> --> default val'    
  ;separator="\n"
end initVals;

template initFixed(list<SimVar> simVarLst, SimCode simCode) ::=
  (simVarLst |> SIMVAR(__) => '<%isFixed%> /* <%crefStr(name, simCode)%> */' ;separator=",\n")
end initFixed;


template functionCallExternalObjectConstructors(ExtObjInfo extObjInfo, SimCode simCode)
 "Generates external objects construction calls."
::=
match extObjInfo
case EXTOBJINFO(__) then
  <<
  public override void FunCallExternalObjectConstructors() {
    <% localRepresentationArrayDefines %>
    <% vars |> var as SIMVAR(initialValue=SOME(exp)) => 
      let &preExp = buffer "" /*BUFD*/
      let arg = daeExp(exp, contextOther, &preExp, simCode)
      <<
      <%preExp%>
      <%cref(var.name, simCode)%> = <%arg%>;
      >>
    %>
    <%aliases |> (var1, var2) => '<%cref(var1,simCode)%> = <%cref(var2,simCode)%>;' ;separator="\n"%>
  }

  >>
end functionCallExternalObjectConstructors;



template functionAlgebraic(list<SimEqSystem> algebraicEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunAlgebraics()
{
  <% localRepresentationArrayDefines %>
  <% algebraicEquations |> it => equation_(it,contextSimulationNonDiscrete, simCode) ;separator="\n"%>
}
>>
end functionAlgebraic;


template functionAliasEquation(list<SimEqSystem> removedEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunAliasEquations()
{
  <% localRepresentationArrayDefines %>
  <% removedEquations  |> eq => equation_(eq, contextSimulationNonDiscrete, simCode) ;separator="\n"%>
}
>>
end functionAliasEquation;

template functionInput(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
public override void InputFun()
{
  <% localRepresentationArrayDefines %>
  <%vars.inputVars |> SIMVAR(__) hasindex i0 => 
  <<
  <%cref(name, simCode)%> = inputVars[<%i0%>];
  >> ;separator="\n"%>
}
>>
end functionInput;

template functionOutput(ModelInfo modelInfo, SimCode simCode) ::=
match modelInfo
case MODELINFO(varInfo = VARINFO(__), vars = SIMVARS(__)) then
<<
public override void OutputFun()
{
  <% localRepresentationArrayDefines %>
  // * not yet
  <%vars.outputVars |> SIMVAR(__) hasindex i0 =>
  <<
  //outputVars[<%i0%>] = <%cref(name, simCode)%>;
  >> ;separator="\n"%>  
}
>>
end functionOutput;


template functionInitSample(list<SampleCondition> sampleConditions, SimCode simCode)
  "Generates function initSample() in simulation file."
::=
  /* Initializes the raw time events of the simulation using the now
     calcualted parameters. */  
  <<
  public override void FunSampleInit()
  {
    <% localRepresentationArrayDefines %>
    var SA = Samples.SamplesArr;
    <%if sampleConditions then "int i = 0; // Current index"%>
    <%sampleConditions |> (relation, zcIndex)  =>
        match relation
        case RELATION(__) then
          <<
          /* <%zcIndex%> Not a time event */
          >>
        case CALL(path=IDENT(name="sample"), expLst={start, interval,_}) then
          let &preExp = buffer "" /*BUFD*/
          let startE = daeExp(start, contextOther, &preExp, simCode)
          let intervalE = daeExp(interval, contextOther, &preExp, simCode)
          <<
          <%preExp%>
          SA[i++] = new OneSample(<%intervalE%>, <%startE%>, <%zcIndex%>, false);
          >>
        else
         <<
         /* UNKNOWN ZERO CROSSING for <%zcIndex%> */
         >> 
    %>
  }
  >>
end functionInitSample;

template functionSampleEquations(list<SimEqSystem> sampleEqns, SimCode simCode)
 "Generates function for sample equations."
::=
  <<
  public override void FunUpdateSample()
  {
    <% localRepresentationArrayDefines %>
    
    <%sampleEqns |> eq =>
      equation_(eq, contextSimulationDiscrete, simCode)
     ;separator="\n"
    %>
  }
>>
end functionSampleEquations;


template functionDAE(list<SimEqSystem> allEquationsPlusWhen,
                     list<SimWhenClause> whenClauses,  list<HelpVarInfo> helpVarInfo,
                     SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override bool FunDAE()
{
  <% localRepresentationArrayDefines %>
  var needToIterate = false;
  isInUpdate = ! isInit;
  
  <%allEquationsPlusWhen |> eq => equation_(eq, contextSimulationDiscrete, simCode) ;separator="\n"%>  

  <%whenClauses |> when hasindex i0 =>  genreinits(when, i0, simCode)
     ;separator="\n"
  %>
  
  isInUpdate = false;
  return needToIterate;
}
>>
end functionDAE;


template genreinits(SimWhenClause whenClauses, Integer widx, SimCode simCode)
" Generates reinit statemeant"
::=

match whenClauses
case SIM_WHEN_CLAUSE(__) then
if reinits then  
  let &preExp = buffer "" /*BUFD*/
  //let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |> (e, hidx) =>
      let helpInit = daeExp(e, contextSimulationDiscrete, &preExp, simCode)
      let &preExp += 'H[<%hidx%>] = <%helpInit%> ?1.0:0.0;'
      edgeHelpVar(hidx)
    ;separator=" || ")  
  <<

  //For whenclause index: <%widx%>
  <%preExp%>
  if (<%helpIf%>) { 
    <%functionWhenReinitStatementThen(reinits, simCode)%>
  }
  >>
end genreinits;


template functionWhenReinitStatementThen(list<WhenOperator> reinits, SimCode simCode)
 "Generates re-init statement for when equation."
::=
  reinits |> reinit =>
    match reinit
    case REINIT(__) then 
      let &preExp = buffer ""
      let val = daeExp(value, contextSimulationDiscrete, &preExp, simCode)
      <<
      <%preExp%>
      <%cref(stateVar, simCode)%> = <%val%>;
      needToIterate = true;
      >>
    case TERMINATE(__) then 
      let &preExp = buffer ""
      let msgVar = daeExp(message, contextSimulationDiscrete, &preExp, simCode)
      <<  
      <%preExp%> 
      MODELICA_TERMINATE(<%msgVar%>);
      >>
    case ASSERT(source=SOURCE(info=info)) then 
      assertCommon(condition, message, contextSimulationDiscrete, info, simCode)
   ;separator="\n"

end functionWhenReinitStatementThen;

template infoArgs(Info info)
::=
  match info
  case INFO(__) then '"<%fileName%>",<%lineNumberStart%>,<%columnNumberStart%>,<%lineNumberEnd%>,<%columnNumberEnd%>,<%if isReadOnly then 1 else 0%>'
end infoArgs;


template assertCommon(Exp condition, Exp message, Context context, Info info, SimCode simCode)
::=
  let &preExpCond = buffer ""
  let condVar = daeExp(condition, context, &preExpCond, simCode)
  let &preExpMsg = buffer ""
  let msgVar = daeExp(message, context, &preExpMsg, simCode)
  <<
  <%preExpCond%>
  if (!<%condVar%>) {
    <%preExpMsg%>
    omc_fileInfo info = {<%infoArgs(info)%>};
    MODELICA_ASSERT(info, <%msgVar%>);
  }
  >>
end assertCommon;



template functionOnlyZeroCrossing(list<ZeroCrossing> zeroCrossingLst, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunOnlyZeroCrossings(double time, double[] gout) //TODO:??time in original is *t only ... how is it called?
{
  <% localRepresentationArrayDefines %>
  <%zeroCrossingLst |> ZERO_CROSSING(__) hasindex i0 => zeroCrossing(relation_, i0, simCode) ;separator="\n"%>  
}
>>
end functionOnlyZeroCrossing;

template zeroCrossing(Exp it, Integer index, SimCode simCode) ::=
  match it
  case RELATION(__) then
    let &preExp = buffer ""
    let e1 = daeExp(exp1, contextOther, &preExp, simCode)
    let e2 = daeExp(exp2, contextOther, &preExp, simCode)
    <<
    <%preExp%>
    gout[<%index%>] = <%
                    match operator
                    case GREATER(__)
                    case LESSEQ(__)    then '<%e1%> - <%e2%>' //space is mandatory here ... (X--1) must be (X - -1)
                    case LESS(__)
                    case GREATEREQ(__) then '<%e2%> - <%e1%>'
                    case EQUAL(__)
                    case NEQUAL(__) then '/*BE AWARE*/Math.Abs(<%e1%> - <%e2%>)' // maybe a HACK, but mathematically correct against the domain change test: (gout > 0) <> (gout_old > 0)               
                    else "!!!unsupported ZC operator!!!"
                   %>;      
    >>
    /* to be deleted
     {var _zen = zeroCrossingEnabled[<%index%>]; //ZEROCROSSING(<%index%>, <%zeroCrossingOpFunc(operator)%>(<%e1%>, <%e2%>));

     if (eulerInUse || _zen!=0.0){
          <%preExp%>
          var _exp = (<%
                    match operator
                    case LESS(__)
                    case LESSEQ(__)    then '<%e1%> - <%e2%>' //space is mandatory here ... (X--1) must be (X - -1)
                    case GREATER(__)
                    case GREATEREQ(__) then '<%e2%> - <%e1%>'
                    case EQUAL(__) then '<%e2%> == <%e1%>'
                    case NEQUAL(__) then '<%e2%> != <%e1%>'             
                    else "!!!unsupported ZC operator!!!"
                   %>);         
          gout[<%index%>] = eulerInUse ? exp : _zen*exp;
        }
     else
        gout[<%index%>] = 1.0;
    }*/
    
    /*  to be deleted  
    <<
    {<%preExp%>var _zen = zeroCrossingEnabled[<%index%>]; //ZEROCROSSING(<%index%>, <%zeroCrossingOpFunc(operator)%>(<%e1%>, <%e2%>));
    gout[<%index%>] = (_zen != 0) ? _zen * (<%match operator
                                           case LESS(__)
                                           case LESSEQ(__)    then '<%e1%> - <%e2%>' //space is mandatory here ... (X--1) must be (X - -1)
                                           case GREATER(__)
                                           case GREATEREQ(__) then '<%e2%> - <%e1%>'
                                           case EQUAL(__) then '<%e2%> == <%e1%>'
                                           case NEQUAL(__) then '<%e2%> != <%e1%>'                
                                           else "!!!unsupported ZC operator!!!"
                                          %>) : 1.0; }
    >>
    */
  case CALL(path=IDENT(name="sample"), expLst={start, interval}) then
    let &preExp = buffer "" //is ignored
    let eStart = daeExp(start, contextOther, &preExp, simCode)
    let eInterval = daeExp(interval, contextOther, &preExp, simCode)
    <<
    //ZEROCROSSING(<%index%>, Sample(*t, <%eStart%>, <%eInterval%>));
    >>
    //the old sample, to be deleted in the next iteration
    //<<
    //{<%preExp%>var _zen = zeroCrossingEnabled[<%index%>]; //ZEROCROSSING(<%index%>, Sample(*t, <%eStart%>, <%eInterval%>));
    //gout[<%index%>] = (_zen != 0) ? _zen * Sample(time, <%eStart%>, <%eInterval%>) : 1.0; }
    //>> 
  case _ then
    <<
    ZERO_CROSSING_ERROR
    >>
end zeroCrossing;

template zeroCrossingOpFunc(Operator it) ::=
  match it
  case LESS(__)      then "Less"
  case GREATER(__)   then "Greater"
  case LESSEQ(__)    then "LessEq"
  case GREATEREQ(__) then "GreaterEq"
  case EQUAL(__)     then "!!!Equal"
  case NEQUAL(__)    then "!!!NEqual"
  else "!!!unsupported ZC operator!!!"
  
end zeroCrossingOpFunc;

// New runtime function. What should it do?
template functionStoreDelayed(SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunStoreDelayed()
{ 
  <% localRepresentationArrayDefines %> 
}
>>
end functionStoreDelayed;
    
template functionODE(list<list<SimEqSystem>> stateContEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void FunODE()
{
  <% localRepresentationArrayDefines %>
  <%stateContEquations |> eqs => 
     (eqs |> it => equation_(it, contextOther, simCode) ;separator="\n") 
     ;separator="\n"%>
}
>>
end functionODE;

//TODO:??there is st more in the trunk
template functionInitial(list<SimEqSystem> startValueEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void InitialFun()
{
  <% localRepresentationArrayDefines %>
  <%startValueEquations |> saeq as SES_SIMPLE_ASSIGN(__) => equation_(saeq, contextOther, simCode) ;separator="\n"%>

  //if (sim_verbose) {
    <%startValueEquations |> SES_SIMPLE_ASSIGN(__) =>
    <<
    //Debug.WriteLine("Setting variable start value: {0}(start={1})", "<%cref(cref, simCode)%>", <%cref(cref, simCode)%>);
    >> ;separator="\n"%>
  //}
}
>>
end functionInitial;

template functionInitialResidual(list<SimEqSystem> residualEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void InitialResidual()
{
  <% localRepresentationArrayDefines %>
  var startX = startStates; var startYB = startAlgebraicsBool; var startYI = startAlgebraicsInt;
  int _i = 0;
  const double _lambda = 1.0;
  
  <%residualEquations |> SES_RESIDUAL(__)  =>
    match exp case DAE.SCONST(__) then
      'initialResiduals[_i++] = 0;'
    else
      let &preExp = buffer ""
      let expPart = daeExp(exp, contextSimulationNonDiscrete, &preExp, simCode) // ??contextOther
      <<
      <%preExp%>
      initialResiduals[_i++] = <%expPart%>;
      >>
  ;separator="\n"%>
}
>>
end functionInitialResidual;

template functionExtraResiduals(list<SimEqSystem> allEquations, SimCode simCode) ::=
  let()= System.tmpTickReset(1)
  (allEquations |> eq => match eq
    case SES_MIXED(__)     then functionExtraResidual(cont, simCode)
    case SES_NONLINEAR(__) then functionExtraResidual(eq, simCode)
  ;separator="\n")
end functionExtraResiduals;

template functionExtraResidual(SimEqSystem equ, SimCode simCode) ::=
  match equ 
  case SES_NONLINEAR(__) then
    <<
    int ResidualFun<%index%>(int n, double[] xloc, double[] res, int iflag)
    {
       <% localRepresentationArrayDefines %>
       <%eqs |> saeq as SES_SIMPLE_ASSIGN(__) =>
         equation_(saeq, contextOther, simCode)
         ;separator="\n"
       %>
       <%eqs |> SES_RESIDUAL(__) hasindex i0 =>
         let &preExp = buffer ""
         let expPart = daeExp(exp, contextSimulationDiscrete, &preExp, simCode)
         <<
         <%preExp%>
         res[<%i0%>] = <%expPart%>;
         >> ;separator="\n"
       %>
       return 0;
    }
    >> 
end functionExtraResidual;

template functionBoundParameters(list<SimEqSystem> parameterEquations, SimCode simCode) ::=
let()= System.tmpTickReset(1)
<<
public override void BoundParameters()
{
  <% localRepresentationArrayDefines %>
  <% parameterEquations
     |> saeq as SES_SIMPLE_ASSIGN(__)  => equation_(saeq, contextOther, simCode) ;separator="\n"%>
}
>>
end functionBoundParameters;

// TODO: Is the -1 thing really correct? It seems to work.
template functionCheckForDiscreteChanges(SimCode simCode) ::=
match simCode case SIMCODE(__) then
<<
public override bool CheckForDiscreteChanges()
{
  <% localRepresentationArrayDefines %>
  //var needToIterate = false;
<%/* old stuff
  //edge(H[i])
  <% helpVarInfo |> (id1, exp, id2) => match id2 case -1 then ""
     else
       'if (<%edgeHelpVar(id1)%>) EventQueue.Add(<%id2%> + NG);'
    ;separator="\n" %>
*/%>
  //if change()
  <%discreteModelVars |> var => 
     match var 
     case CREF_QUAL(__) 
     case CREF_IDENT(__) then
     <<
     if (<%preCref(var, simCode)%> != <%cref(var, simCode)%>) return true;
     >> ;separator="\n"
  %>
<%/* old stuff
  for (int i = 0; i < H.Length; i++) {
    //change(H[i]) ?? TODO: not sure if it can be only 1.0 or 0.0
    if (H[i] != preH[i])
      return true; //needToIterate=true;
  }
*/%>
  return false; //needToIterate;
}
>>
end functionCheckForDiscreteChanges;

template edgeHelpVar(String idx) ::=
   '(/*edge(h[<%idx%>])*/H[<%idx%>]!=0.0 && preH[<%idx%>]==0.0)'
end edgeHelpVar;

//TODO: eliminate this entirely  ??
template daeExpToReal(Exp exp, Context context, Text &preExp, SimCode simCode) ::=
    daeExp(exp, context, &preExp, simCode)
    + (match expTypeFromExp(exp) case "bool" then " ?1.0:0.0") //TODO: a HACK ?
      
end daeExpToReal;

// Residual equations are not handled here
template equation_(SimEqSystem eq, Context context, SimCode simCode) ::=
match eq
case SES_SIMPLE_ASSIGN(__) then
  let &preExp = buffer ""
  let expPart = daeExp(exp, context, &preExp, simCode) //was daeExpToReal
  <<
  <%preExp%>
  <%cref(cref, simCode)%> = <%expPart%>;
  >>
case SES_ARRAY_CALL_ASSIGN(__) then "SES_ARRAY_CALL_ASSIGN"
case SES_ALGORITHM(__) then
  (statements |> stmt =>
    algStatement(stmt, context, simCode)
  ;separator="\n")

case SES_LINEAR(__) then
  let uid = System.tmpTick()
  let size = listLength(vars)
  let aname = 'A<%uid%>'
  let bname = 'b<%uid%>'
  <<
  var <%aname%> = new double[<%size%>*<%size%>]; //declare_matrix(<%aname%>, <%size%>, <%size%>);
  var <%bname%> = new double[<%size%>]; //declare_vector(<%bname%>, <%size%>);
  <%simJac |> (row, col, eq as SES_RESIDUAL(__)) =>
     let &preExp = buffer ""
     let expPart = daeExpToReal(eq.exp, context, &preExp, simCode) 
     '<%preExp%><%aname%>[<%row%>+<%col%>*<%size%>] = <%expPart%>; //set_matrix_elt(<%aname%>, <%row%>, <%col%>, <%size%>, <%expPart%>);'
  ;separator="\n"%>
  <%beqs|> it hasindex i0 =>
     let &preExp = buffer ""
     let expPart = daeExpToReal(it, context, &preExp, simCode)
     '<%preExp%><%bname%>[<%i0%>] = <%expPart%>; //set_vector_elt(<%bname%>, <%i0%>, <%expPart%>);'
  ;separator="\n"%>
  <%if partOfMixed then 
      'if(SolveLinearSystemMixed(<%aname%>, <%bname%>, <%size%>, <%uid%>) != 0) found_solution = -1;'
    else 
      'SolveLinearSystem(<%aname%>, <%bname%>, <%size%>, <%uid%>);'
  %>
  <%vars |> SIMVAR(__) hasindex i0 => '<%cref(name, simCode)%> = <%bname%>[<%i0%>]; //get_vector_elt(<%bname%>, <%i0%>);' ;separator="\n"%>
  >>
case SES_MIXED(__) then
  let uid = System.tmpTick()
  let contEqs = equation_(cont, context, simCode)
  let numDiscVarsStr = listLength(discVars) 
  let valuesLenStr = listLength(values)
  let &preDisc = buffer "" /*BUFD*/
  let discLoc2 = 
    discEqs |> discEq as SES_SIMPLE_ASSIGN(__) hasindex i0 =>
        <<
        <%cref(discEq.cref, simCode)%> = <%daeExp(discEq.exp, context, &preDisc, simCode)%>;
        double discrete_loc2_<%i0%> = <%crefToReal(discEq.cref, simCode)%>;
        >>
        //double discrete_loc2_<%i0%> = <%daeExpToReal(discEq.exp, context, &preDisc, simCode)%>;
        //<%cref(discEq.cref, simCode)%> = discrete_loc2_<%i0%>;
        
        ;separator="\n"
      /*
      match discEqs
    case { discEq as SES_SIMPLE_ASSIGN(__) } then
      <<
      <%cref(discEq.cref, simCode)%> = <%daeExpToReal(discEq.exp, context, &preDisc, simCode)%>;
      double discrete_loc2_0 = <%crefToReal(discEq.cref, simCode)%>;
      >>
      case discEqs then
      <<
      var discrete_loc2 = new double[<%numDiscVarsStr%>];
      <%discEqs |> discEq as SES_SIMPLE_ASSIGN(__) hasindex i0 =>
        <<
        <%cref(discEq.cref, simCode)%> = <%daeExpToReal(exp, context, &preDisc, simCode)%>;
        discrete_loc2[<%i0%>] = <%crefToReal(discEq.cref, simCode)%>;
        >>
        ;separator="\n"%>
      >>
      */
  <<
  // *** mixed_equation_system(<%numDiscVarsStr%>) ***
  { int found_solution = 0;
    int cur_value_indx = -1;
    var values = new double[]{<%values ;separator=", "%>};
    do {
      <%
        discVars |> SIMVAR(__) hasindex i0 => 
          'double discrete_loc_<%i0%> = <%crefToReal(name, simCode)%>;'
        ;separator=",\n"
       
       /*match discVars
        case { var as SIMVAR(__) } then
          <<
          double discrete_loc_0 = <%crefToReal(var.name, simCode)%>;
          >>
        case discVars then
          <<
          var discrete_loc = new double[<%numDiscVarsStr%>] {
            <%discVars |> SIMVAR(__) => cref(name, simCode) ;separator=",\n"%>
          };
          >> */
      %>
      {
        <%contEqs%>
      }
      <%preDisc%>
      <%discLoc2%>
      {
        // check_discrete_values(<%numDiscVarsStr%>, <%valuesLenStr%>);
        if (found_solution == -1) { /*system of equations failed*/
            found_solution = 0;
        } else {
            found_solution = 1;
            <% 
              discVars |> SIMVAR(__) hasindex i0 =>
                <<
                if ( Math.Abs(discrete_loc_<%i0%> - discrete_loc2_<%i0%>) > 1e-12) found_solution = 0;                
                >> ;separator="\nelse "
              /*match discVars
              case { SIMVAR(__) } then
                <<
                if ( Math.Abs(discrete_loc_0 - discrete_loc2_0) > 1e-12) {
                    found_solution = 0;
                }
                >>
              case discVars then
                <<
                for (int i=0; i < <%numDiscVarsStr%>; i++) {
                    if ( Math.Abs(discrete_loc[i] - discrete_loc2[i]) > 1e-12) {
                        found_solution = 0;
                    }
                }
                >>*/
            %>
        }
        if (found_solution == 0) { //!found_solution
            cur_value_indx++;
            if (cur_value_indx >= <%valuesLenStr%>/<%numDiscVarsStr%>) {
                found_solution = -1; //?? failed ??
                System.Diagnostics.Debug.WriteLine("Mixed system id=" + <%uid%> + "failed.");
            } else {
              var curValOffset = cur_value_indx*<%numDiscVarsStr%>;
              <% discVars |> SIMVAR(__) hasindex i0 =>
                  '<%cref(name, simCode)%> = <%convertRealExpForCref('values[curValOffset+<%i0%>]', name, simCode)%>;'
                 ;separator="\n"
              %>
            }
        }             
      }
    } while (found_solution == 0);
  } // *** mixed_equation_system_end(<%numDiscVarsStr%>) ***
  >>

case SES_NONLINEAR(__) then 
  let size = listLength(crefs)
  <<
  //start_nonlinear_system(<%size%>);
  { var oldX = statesOld; var oldXd = statesDerivativesOld;  var old2X = statesOld2; var old2Xd = statesDerivativesOld2;
    var oldY = algebraicsOld; var old2Y = algebraicsOld2; var oldTimeDelta = timeOld - timeOld2; 
    var nls_x = new double[<%size%>]; var nls_xold = new double[<%size%>];
    <%crefs |> name hasindex i0 =>
      <<
      nls_x[<%i0%>] = /*extraPolate(<%crefStr(name, simCode)%>)*/ oldTimeDelta == 0.0 ? <%cref(name, simCode)%> : (time * (<%oldCref(name, simCode)%> - <%old2Cref(name, simCode)%>) + (timeOld * <%old2Cref(name, simCode)%> - timeOld2 * <%oldCref(name, simCode)%>)) / oldTimeDelta;          
      nls_xold[<%i0%>] = <%oldCref(name, simCode)%>;
      >> ;separator="\n"
    %>
    SolveNonlinearSystem(ResidualFun<%index%>, nls_x, nls_xold, <%index%>);
    <%crefs |> name hasindex i0 => 
    '<%cref(name, simCode)%> = <%convertRealExpForCref('nls_x[<%i0%>]', name, simCode)%>;' 
      ;separator="\n"
    %>
  } //end_nonlinear_system();
  >>
  
case SES_WHEN(__) then
  let &preExp = buffer ""
  let helpIf = (conditions |> (e, hidx) =>
      let hInit = daeExp(e, context, &preExp, simCode)
      let &preExp += 'H[<%hidx%>] = <%hInit%> ? 1.0 : 0.0;' //TODO: ??? type
      edgeHelpVar(hidx)
      //'/*edge(H[<%hidx%>])*/(H[<%hidx%>]!=0.0 && preH[<%hidx%>]==0.0)'
    ;separator=" || ")
  let &preExp2 = buffer ""
  let rightExp = daeExp(right, context, &preExp2, simCode)
  <<
  <%preExp%>
  if (<%helpIf%>) {
    <%preExp2%>
    <%cref(left, simCode)%> = <%rightExp%>;
  } else {
    <%cref(left, simCode)%> = <%preCref(left, simCode)%>;
  }
  >>

else  
  "UNKNOWN_equation"
  
end equation_;

// SECTION: SIMULATION TARGET, FUNCTIONS FILE SPECIFIC TEMPLATES
// not yet implemented

// SECTION: GENERAL TEMPLATES, COMPONENT REFERENCES

template cref(ComponentRef cr, SimCode simCode) ::=
  match cr
  case CREF_IDENT(ident = "xloc") then crefStr(cr, simCode) //TODO: ??xloc
  case CREF_IDENT(ident = "time") then "time"
  case CREF_IDENT(ident = "$_lambda") then "_lambda"
  else '/*<% crefStr(cr, simCode) %>*/<% representationCref(cr, simCode) %>'
  // not needed ... //the space at the start is important to avoid ambiguity with "/" operator to become "//" line comment
/*
  match cr
  //TODO: ?? case CREF_IDENT(ident = "xloc") then crefStr(cr)
  case CREF_IDENT(__) then replaceDollarWorkaround(ident)
  case _ then "CREF_NOT_IDENT"
*/
end cref;

template representationCref(ComponentRef inCref, SimCode simCode) ::=
  cref2simvar(inCref, simCode) |> SIMVAR(__) =>
    '<%representationArrayName(varKind, type_)%>[<% index %>]'
end representationCref;

template representationArrayName(VarKind varKind, Type type_) ::=
  match varKind 
  case VARIABLE(__)    then "Y" + representationArrayNameTypePostfix(type_)
  case STATE(__)       then "X"
  case STATE_DER(__)   then "Xd"
  case DUMMY_DER(__)   then "Y" // => algebraics
  case DUMMY_STATE(__) then "Y" // => algebraics
  case DISCRETE(__)    then 'Y<%representationArrayNameTypePostfix(type_)%>/*d*/'
  case PARAM(__)       then "P" + representationArrayNameTypePostfix(type_)
  case EXTOBJ(__)      then "EO"
  case CONST(__)       then "CONST_VAR_KIND"
  else "BAD_VARKIND"
end representationArrayName;

template representationArrayNameTypePostfix(Type type_) ::=
  match type_ 
  case T_INTEGER(__)  then "I"
  case T_BOOL(__) then "B"
  case T_REAL(__) then ""
  else "BAD_ARRAY_NAME_POSTFIX"
end representationArrayNameTypePostfix;

//TODO: a HACK ?
template daeExpRealConversionPostfix(Exp exp, SimCode simCode) ::=
  match exp
  case CREF(__) then
     match cref2simvar(componentRef, simCode) 
     case SIMVAR(type_ = T_BOOL(__)) then "?1.0:0.0"
  //TODO: make conversion also for other expressions
end daeExpRealConversionPostfix;

//TODO: a HACK ?
template crefToReal(ComponentRef cr, SimCode simCode) ::=
<</*(double)<% crefStr(cr, simCode) 
            %>*/<% cref2simvar(cr, simCode) |> SIMVAR(__) => //representationCref(cr, simCode)
                   '<%representationArrayName(varKind, type_)%>[<% index %>]'
                   + (match type_ case T_BOOL(__) 
                      then "?1.0:0.0")
                %>
>>
end crefToReal;

//TODO: a HACK ?
template convertRealExpForCref(Text realExp, ComponentRef cr, SimCode simCode) ::=
  cref2simvar(cr, simCode) |> SIMVAR(__) =>
     match type_ 
     case T_BOOL(__) then '(<%realExp%>) != 0.0'
     case T_INTEGER(__) then '(int)(<%realExp%>)'
     else realExp                

end convertRealExpForCref;

template preCref(ComponentRef cr, SimCode simCode) ::=
'/*pre(<%crefStr(cr, simCode)%>)*/pre<%representationCref(cr, simCode)%>'
end preCref;

template derCref(ComponentRef cr, SimCode simCode) ::=
'/*derCall!!(<% crefStr(cr, simCode) %>)*/<%representationCref(derComponentRef(cr), simCode)%>'
end derCref;

template oldCref(ComponentRef cr, SimCode simCode) ::=
'/*old(<%crefStr(cr, simCode)%>)*/old<%representationCref(cr, simCode)%>'
end oldCref;

template old2Cref(ComponentRef cr, SimCode simCode) ::=
'/*old2(<%crefStr(cr, simCode)%>)*/old2<%representationCref(cr, simCode)%>'
end old2Cref;

template startCref(ComponentRef cr, SimCode simCode) ::=
'/*start(<%crefStr(cr, simCode)%>)*/start<%representationCref(cr, simCode)%>'
end startCref;

template contextCref(ComponentRef cr, Context context, SimCode simCode)
  "Generates code for a component reference depending on which context we're in."
::=
  match context
  case FUNCTION_CONTEXT(__) then crefStr(cr, simCode)
  else cref(cr, simCode)
end contextCref;

template crefStr(ComponentRef cr, SimCode simCode)
 "Generates the name of a variable for variable name array."
::=
  match cr
  case CREF_IDENT(__) then csharpIdent(ident) + subscriptsStr(subscriptLst, simCode)
  case CREF_QUAL(ident = "$DER") then 'der(<%crefStr(componentRef, simCode)%>)'
  case CREF_QUAL(__) then '<%ident%><%subscriptsStr(subscriptLst, simCode)%>.<%crefStr(componentRef, simCode)%>'
  else "CREF_NOT_IDENT_OR_QUAL"
end crefStr;

template csharpIdent(String ident)
 "Generates the name of a variable for variable name array."
::=
  match ident
  case "string" then "@string"
  case "int"    then "@int"
  else ident
end csharpIdent;

template subscriptsStr(list<Subscript> subscripts, SimCode simCode)
 "Generares subscript part of the name."
::=
  if subscripts then
    '[<% subscripts |> s => match s
           case INDEX(__) 
           case SLICE(__) then
              let &preExp = buffer "" //dummy ??
              daeExp(exp, contextFunction, &preExp, simCode)
           case WHOLEDIM(__) then "WHOLEDIM"
           else "UNKNOWN_SUBSCRIPT"
      ;separator="," %>]'
end subscriptsStr;


/*
template crefSubscript(ComponentRef it, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then replaceDollarWorkaround(ident) + subscripts(subscriptLst)
  case _ then "CREF_NOT_IDENT"
end crefSubscript;

template subscripts(list<Subscript> subscriptsLst) ::=
  if subscriptsLst then '[<%subscriptsLst |> it => subscript(it) ;separator=","%>]'
end subscripts;

template subscript(Subscript it) ::=
  match it
  case INDEX(exp = ICONST(__)) then exp.integer
  case _ then "SUBSCRIPT_NOT_CONSTANT"
end subscript;

*/


// SECTION: GENERAL TEMPLATES, PATHS

template dotPath(Path it) ::=
  match it
  case QUALIFIED(__)      then '<%name%>.<%dotPath(path)%>'
  case IDENT(__)          then name
  case FULLYQUALIFIED(__) then dotPath(path)
end dotPath;

//TODO: not 100% back-convertible e.g. Awkward_.Model vs. Awkward._Model, both -> Awkward___Model 
template underscorePath(Path it) ::=
  match it
  case QUALIFIED(__)      then '<%System.stringReplace(name, "_", "__")%>_<%underscorePath(path)%>'
  case IDENT(__)          then System.stringReplace(name, "_", "__")
  case FULLYQUALIFIED(__) then underscorePath(path)
end underscorePath;


// SECTION: GENERAL TEMPLATES, FUNCTION GENERATION
// not yet implemented



// Codegen.generateStatement
template algStatement(DAE.Statement stmt, Context context, SimCode simCode) ::=
  match stmt
  case STMT_ASSIGN(exp1 = CREF(componentRef = WILD(__)), exp = e) then
    let &preExp = buffer "" 
    let expPart = daeExp(e, context, &preExp, simCode)
    <<
    <%preExp%>
    <%expPart%>
    >>    
  case STMT_ASSIGN(exp1 = CREF(__)) then
    let &preExp = buffer ""
    let expPart = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    <%scalarLhsCref(exp1, context, &preExp, simCode)%> = <%expPart%>;
    >>
  case STMT_ASSIGN(__) then
    let &preExp = buffer ""
    let expPart1 = daeExp(exp1, context, &preExp, simCode)
    let expPart2 = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    <%expPart1%> = <%expPart2%>;
    >>
  //works only for array name as IDENT with subscripts 
  case STMT_ASSIGN_ARR(componentRef = CREF_IDENT(subscriptLst=subs as (_ :: _))) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp, simCode)
     let spec = daeExpCrefRhsIndexSpec(subs, context, &preExp, simCode)
     <<
     <%preExp%>
     <%componentRef.ident%>.AssignSpec(<%spec%>, <%expPart%>.A);
     >>
  case STMT_ASSIGN_ARR(__) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp, simCode)
     <<
     <%preExp%>
     ArrayCopy(<%expPart%>.A, <%contextCref(componentRef, context, simCode)%>.A);
     >>     
  case STMT_IF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp, simCode)
    <<
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    <%elseExpr(else_, context, simCode)%>
    >>
  case STMT_FOR(range=rng as RANGE(__)) then
    let identType = expType(type_, iterIsArray) //TODO: ?? what is this for ... no array typed iterator is possible ???
    let identTypeShort = expTypeShort(type_)
    let stmtStr = (statementLst |> stmt => algStatement(stmt, context, simCode)
                   ;separator="\n")
    algStmtForRange_impl(rng, iter, identType, identTypeShort, stmtStr, context, simCode)
  
  case s as STMT_FOR(__) then "algStmtForGeneric_NOT_IMPLEMENTED"
    //  algStmtForGeneric(s, context, &varDecls /*BUFC*/)
  
  case STMT_WHILE(__)  then
    let &preExp = buffer ""
    let var = daeExp(exp, context, &preExp, simCode)
    <<
    for(;;) {
      <%preExp%>
      if (!<%var%>) break;
      <%statementLst |> stmt => algStatement(stmt, context, simCode) ;separator="\n"%>
    }
    >>
  
  case STMT_TUPLE_ASSIGN(__)   then "STMT_TUPLE_ASSIGN_NI"
  case STMT_ASSERT(__)         then "STMT_ASSERT_NI"
  case STMT_TERMINATE(__)      then "STMT_TERMINATE_NI"
  case STMT_WHEN(__)           then algStmtWhen(stmt, context, simCode)
  case STMT_BREAK(__)          then 'break; //break stmt<%\n%>'
  case STMT_FAILURE(__)        then "STMT_FAILURE_NI"
  case STMT_TRY(__)            then "STMT_TRY_NI"
  case STMT_CATCH(__)          then "STMT_CATCH_NI"
  case STMT_THROW(__)          then "STMT_THROW_NI"
  case STMT_RETURN(__)         then "STMT_RETURN_NI"
  case STMT_NORETCALL(__)      then "STMT_NORETCALL_NI"
  case STMT_REINIT(__)         then "STMT_REINIT_NI"
  
  case _ then "NOT_IMPLEMENTED_ALG_STATEMENT"
 /*
  case STMT_FOR(range = rng as RANGE(__)) then
    <<
    {
      <%expTypeA(type_, boolean)%> 
        _r1 = <%daeExp(rng.exp,isSimulationCode)%>,
        _r2 = <%match rng.expOption case SOME(eo) then daeExp(eo,isSimulationCode) else "(1)"%>,
        _r3 = <%daeExp(rng.range,isSimulationCode)%>,
        <%iter%>;

      for (<%iter%> = _r1; in_range_<%expTypeShort(type_)%>(<%iter%>, _r1, _r3); <%iter%> += _r2) {
        <%statementLst |> it => algStatement(it) ;separator="\n" /* ??CONTEXT(codeContext,expContext,IN_FOR_LOOP(loopContext)*/ %>        
      }
    } /*end for*/
    >>
  */
end algStatement;

//TODO: in_range_integer is not correct against Modelica specification when e.g. 10:1
//TODO: optimize representation for range, so that template can know that the range is ascendent and/or bounded by a constatnt(not necessary)
template algStmtForRange_impl(Exp range, Ident iterator, String type, String shortType, Text body, Context context, SimCode simCode)
 "The implementation of algStmtForRange, which is also used by daeExpReduction."
::=
match range
case RANGE(__) then
  let iterName = iterator
  let &stopVar = buffer ""
  let &preExp = buffer ""
  let startValue = daeExp(exp, context, &preExp, simCode)
  let stopValue = daeExp(range, context, &preExp, simCode)
  match expOption 
  case SOME(eo) then
    let &stepVar = buffer "" 
    let stepValue = daeExp(eo, context, &preExp, simCode)
    <<
    <%preExp%>
    <%tempDecl(type, &stepVar)%> = <%stepValue%>; <%tempDecl(type, &stopVar)%> = <%stopValue%>;
    for(<%iterName%> = <%startValue%>;(<%stepVar%> > 0? <%iterName%><=<%stopVar%> : <%stopVar%><=<%iterName%>); <%iterName%> += <%stepVar%>) { 
      <%body%>
    }
    >>
  else //optimization for 1 step
    <<
    <%preExp%>
    <%tempDecl(type, &stopVar)%> = <%stopValue%>;
    for(<%iterName%> = <%startValue%>; <%iterName%><=<%stopVar%>; <%iterName%> += 1) { 
      <%body%>
    }
    >>
end algStmtForRange_impl;

template elseExpr(DAE.Else it, Context context, SimCode simCode) ::= 
  match it
  case NOELSE(__) then ""
  case ELSEIF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp, simCode)
    <<
    else {
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    <%elseExpr(else_, context, simCode)%>
    }
    >>
  case ELSE(__) then
    <<
    else {
      <%statementLst |> it => algStatement(it, context, simCode) ;separator="\n"%>
    }
    >>
end elseExpr;


template algStmtWhen(DAE.Statement when, Context context, SimCode simCode)
 "Generates a when algorithm statement."
::=
match context
case SIMULATION(genDiscrete=true) then
  match when
  case STMT_WHEN(__) then
    <<
    <%algStatementWhenPre(when, simCode)%>
    if (<%helpVarIndices |> idx => edgeHelpVar(idx) ;separator=" || "%>) {
      <% statementLst |> stmt =>  algStatement(stmt, context, simCode)
         ;separator="\n" %>
    }
    <%algStatementWhenElse(elseWhen, simCode)%>
    >>
  end match
end algStmtWhen;


template algStatementWhenPre(DAE.Statement stmt, SimCode simCode)
 "Helper to algStmtWhen."
::=
  match stmt
  case STMT_WHEN(exp=ARRAY(array=el)) then
    let &preExp = buffer "" /*BUFD*/
    let assignments = algStatementWhenPreAssigns(el, helpVarIndices, &preExp, simCode)
    <<
    <%preExp%>
    <%assignments%>
    <%match elseWhen case SOME(ew) then  algStatementWhenPre(ew, simCode)%>
    >>
  case when as STMT_WHEN(__) then
    match helpVarIndices
    case {i} then
      let &preExp = buffer "" /*BUFD*/
      let res = daeExp(when.exp, contextSimulationDiscrete, &preExp, simCode) //TODO: ??? type conversion
      <<
      <%preExp%>
      H[<%i%>] = <%res%> ? 1.0 : 0.0;
      <%match when.elseWhen case SOME(ew) then  algStatementWhenPre(ew, simCode)%>
      >>
end algStatementWhenPre;


template algStatementWhenElse(Option<DAE.Statement> stmt, SimCode simCode)
 "Helper to algStmtWhen."
::=
match stmt
case SOME(when as STMT_WHEN(__)) then
  let elseCondStr = (when.helpVarIndices |> idx => edgeHelpVar(idx)
                      ;separator=" || ")
  <<
  else if (<%elseCondStr%>) {
    <% when.statementLst |> stmt =>  algStatement(stmt, contextSimulationDiscrete, simCode)
       ;separator="\n"%>
  }
  <%algStatementWhenElse(when.elseWhen, simCode)%>
  >>
end algStatementWhenElse;


template algStatementWhenPreAssigns(list<Exp> exps, list<Integer> ints, Text &preExp, SimCode simCode)
 "Helper to algStatementWhenPre.
  The lists exps and ints should be of the same length. Iterating over two
  lists like this is not so well supported in Susan, so it looks a bit ugly."
::=
  match exps
  case (firstExp :: restExps) then
    match ints
    case (firstInt :: restInts) then
      <<
      H[<%firstInt%>] = <%daeExp(firstExp, contextSimulationDiscrete, &preExp, simCode)%>;
      <%algStatementWhenPreAssigns(restExps, restInts, &preExp, simCode)%>
      >>
end algStatementWhenPreAssigns;
    

template scalarLhsCref(Exp ecr, Context context, Text &preExp, SimCode simCode) ::=
match ecr
case ecr as CREF(componentRef=CREF_IDENT(subscriptLst=subs)) then
  if crefNoSub(ecr.componentRef) then
    contextCref(ecr.componentRef, context, simCode)
  else
    daeExpCrefRhs(ecr, context, &preExp, simCode)
case ecr as CREF(componentRef=CREF_QUAL(__)) then
    contextCref(ecr.componentRef, context, simCode)
else
    "ONLY_IDENT_OR_QUAL_CREF_SUPPORTED_SLHS"
end scalarLhsCref;

/*
template scalarLhsCref(ComponentRef it, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then replaceDollarWorkaround(ident)
  case CREF_QUAL(__)  then '<%ident%>.<%scalarLhsCref(componentRef)%>'
end scalarLhsCref;
*/

//TODO: this wrong for qualified integers !
template rhsCref(ComponentRef it, Type ty, SimCode simCode) ::=
  match it
  case CREF_IDENT(__) then '<%rhsCrefType(ty)%><%replaceDollarWorkaround(ident)%>'
  case CREF_QUAL(__)  then '<%rhsCrefType(ty)%><%ident%>.<%rhsCref(componentRef,ty, simCode)%>'
  case _          then "rhsCref:ERROR"
end rhsCref;

template rhsCrefType(Type it) ::=
  match it
  case T_INTEGER(__) then "(int)"
  case _      then ""
end rhsCrefType;


template replaceDollarWorkaround(String ident) ::=
  stringReplace(
    stringReplace(ident,"$DER","Der_"), 
    "$", "")
end replaceDollarWorkaround;

// SECTION: GENERAL TEMPLATES, EXPRESSIONS
  
template daeExp(Exp inExp, Context context, Text &preExp, SimCode simCode) ::=
  match inExp
  case ICONST(__)     then integer
  case RCONST(__)     then real
  case SCONST(__)     then '"<%Util.escapeModelicaStringToCString(string)%>"'
  case BCONST(__)     then if bool then "true" else "false" //"(1)" else "(0)"
  case ENUM_LITERAL(__) then '<%index%>/*ENUM:<%dotPath(name)%>*/'
  case CREF(ty = T_FUNCTION_REFERENCE_FUNC(__)) then 'FUNC_REF_NOT_SUPPORTED(cr=<%crefStr(componentRef,simCode)%>)'
  case CREF(ty = T_COMPLEX(complexClassType = RECORD(__))) then
    match context case FUNCTION_CONTEXT(__) then
      daeExpCrefRhs(inExp, context, &preExp, simCode)
    else
      "daeExpRecordCrefRhs_NOT_YET" //daeExpRecordCrefRhs(t, cr, context, preExp, varDecls)
  case CREF(__)       then daeExpCrefRhs(inExp, context, &preExp, simCode)
  case LBINARY(__)
  case BINARY(__)     then daeExpBinary(operator, exp1, exp2, context, &preExp, simCode)
  case LUNARY(__)
  case UNARY(__)      then daeExpUnary(operator, exp, context, &preExp, simCode)
  case RELATION(__)   then daeExpRelation(inExp, context, &preExp, simCode)
  case IFEXP(__)      then daeExpIf(expCond, expThen, expElse, context, &preExp, simCode)
  case CALL(__)       then daeExpCall(inExp, context, &preExp, simCode)
  // PARTEVALFUNCTION
  case ARRAY(__)      then daeExpArray(inExp, context, &preExp, simCode)
  case MATRIX(__)     then daeExpMatrix(inExp, context, &preExp, simCode)
  case RANGE(__)      then "RANGE_NOT_IMPLEMENTED"
  case TUPLE(__)      then "TUPLE_NOT_IMPLEMENTED"
  case CAST(__)       then daeExpCast(inExp, context, &preExp, simCode)
  case ASUB(__)       then daeExpAsub(inExp, context, &preExp, simCode)
  case SIZE(__)       then daeExpSize(inExp, context, &preExp, simCode)
  case CODE(__)       then "CODE_NOT_IMPLEMENTED"
  case REDUCTION(__)  then "REDUCTION_NOT_IMPLEMENTED"
  //case VALUEBLOCK(__) then "VALUEBLOCK_NOT_IMPLEMENTED"
  case LIST(__)       then "LIST_NOT_IMPLEMENTED"
  case CONS(__)       then "CONS_NOT_IMPLEMENTED"
  // META_TUPLE
  // META_OPTION
  // METARECORDCALL
  case _          then "UNKNOWN_EXP"
end daeExp;

template daeExpSize(Exp esize, Context context, Text &preExp, SimCode simCode)
 "Generates code for a size expression."
::=
  match esize
  case SIZE(exp=CREF(__), sz=SOME(dim)) then
    let expPart = daeExp(exp, context, &preExp, simCode)
    match dim
    case ICONST(__) then 
      '<%expPart%>.size<%integer%>'
    else
      '<%expPart%>.size(<%daeExp(dim, context, &preExp, simCode)%>)'    
  else "size_X_NOT_IMPLEMENTED"
end daeExpSize;

  
template daeExpCrefRhs(Exp ecr, Context context, Text &preExp, SimCode simCode) ::=
  match ecr
  case ecr as CREF(componentRef = cr) then
    let box = daeExpCrefRhsArrayBox(ecr, context, &preExp, simCode)
    if box then
      box
    else if crefIsScalar(cr, context) then
      //match ecr.ty 
      //case T_INTEGER(__)  then '(int)<%contextCref(cr, context, simCode)%>'
      //case T_BOOL(__) then 
      //    match context
      //    case FUNCTION_CONTEXT(__) then contextCref(cr, context, simCode) //TODO: a hack!
      //    else '(<%contextCref(cr, context, simCode)%> !=0.0)'
      //else 
      contextCref(cr, context, simCode)
    else if crefSubIsScalar(cr) then
      // The array subscript results in a scalar
      //let ety = match ecr.ty case T_ARRAY(__) then "array" else "noarray"
      let arrName = contextCref(crefStripLastSubs(cr), context, simCode)
      //let arrayType = expTypeArray(ecr.ty,listLength(crefSubs(cr)))
      //let dimsLenStr = listLength(crefSubs(cr))
      //let dimsValuesStr = (crefSubs(cr) |> INDEX(__) => daeExp(exp, context, &preExp, simCode) ;separator=", ")
      <<
      <%arrName%>[<%crefSubs(cr) |> INDEX(__) => daeExp(exp, context, &preExp, simCode) 
                     ;separator=", "
                   %>]      
      >>
    else
      // The array subscript denotes a slice
      let arrName = contextCref(crefStripLastSubs(cr), context, simCode)
      //let crsubs = crefSubs(cr)
      //let arrayType = expTypeArray(ecr.ty, listLength(crsubs))
      let spec = daeExpCrefRhsIndexSpec(crefSubs(cr), context, &preExp, simCode)
      '<%arrName%>.Array1Spec(<%spec%>)'
  else
    "UNKNOWN_RHS_CREF"  
end daeExpCrefRhs;

template daeExpCrefRhsArrayBox(Exp exp, Context context, Text &preExp, SimCode simCode) ::=
match exp
case ecr as CREF(ty=T_ARRAY(ty=aty,dims=dims)) then
  match context 
  case FUNCTION_CONTEXT(__) then ""
  else
  //case SIMULATION(__) //TODO: ?? is it the same as "else than FUNCTION_CONTEXT" ? 
  //case OTHER(__)      then
    // For context simulation and other array variables must be boxed into a real_array
    // object since they are represented only in a double array.
    let &tmpArr = buffer ""
    let arrType = expTypeArray(aty, listLength(dims))
    let dimsValuesStr = (dims |> dim => dimension(dim) ;separator=", ")
    let &preExp += match cref2simvar(ecr.componentRef, simCode) case SIMVAR(__) then  
         '<%tempDecl("var", &tmpArr)%> = new <%arrType%>(<%dimsValuesStr%>, <%index%>-1, /*<%crefStr(ecr.componentRef, simCode)%>*/<%representationArrayName(varKind,type_)%>);<%\n%>'
    tmpArr
end daeExpCrefRhsArrayBox;


template daeExpCrefRhsIndexSpec(list<Subscript> subs, Context context, Text &preExp, SimCode simCode) ::=
  subs |> sub => match sub
       case INDEX(__) then
         'new int[]{<%daeExp(exp, context, &preExp, simCode)%>-1}'
       case WHOLEDIM(__) then
         "null"
       case SLICE(__) then
         match exp
         case ARRAY(scalar=true, ty=T_INTEGER(__)) then
           <<
           new int[]{<% array |> e => '(<%expTypeFromExp(e)%>)<%daeExp(e, context, &preExp, simCode)%>-1' 
                        ;separator=","
                      %>
           >>
         else "UKNOWN_SLICE_EXP"
  ;separator=", "
end daeExpCrefRhsIndexSpec;


template daeExpAsub(Exp aexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for an asub expression."
::=
  match aexp
  case ASUB(exp=RANGE(ty=t), sub={idx}) then
    'ASUB_EASY_CASE'
  case ASUB(exp=ASUB(
              exp=ASUB(
                exp=ASUB(exp=e, sub={ICONST(integer=i)}),
                sub={ICONST(integer=j)}),
              sub={ICONST(integer=k)}),
            sub={ICONST(integer=l)}) then
    'ASUB_4D'
  case ASUB(exp=ASUB(
              exp=ASUB(exp=e, sub={ICONST(integer=i)}),
              sub={ICONST(integer=j)}),
            sub={ICONST(integer=k)}) then
    'ASUB_3D'
  case ASUB(exp=ASUB(exp=e, sub={ICONST(integer=i)}),
            sub={ICONST(integer=j)}) then
    '<%daeExp(e, context, &preExp, simCode)%>[<%i%>,<%j%>]'
  case ASUB(exp=e, sub={ICONST(integer=i)}) then
    'ASUB_ARRAY'
  case ASUB(exp=ecr as CREF(componentRef = cr), sub = subs) then
    match context case FUNCTION_CONTEXT(__) then
      daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context, &preExp, simCode)
    else //SIMULATION or OTHER contexts
      match ecr.ty
      case T_ARRAY(ty = T_REAL(__), dims = dims) then //daeExpCrefRhsArrayBox
        <<
        /*<% crefStr(cr, simCode) 
           %>[]*/<% cref2simvar(cr, simCode) |> SIMVAR(__) =>
                     let &constSum = buffer index
                     let baseSub = asubSubsripts(dims, subs, &constSum, context, &preExp, simCode)
                     <<
                     <%representationArrayName(varKind,type_)%>[<%constSum%><%baseSub%>]
                     >>                          
                  %>
        >> 
      else "ASUB_SIMULATION_OTHER_ERROR"
        
      
    //cref(ecr.componentRef, simCode)
    /*
    let arrName = daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context, &preExp, simCode)
    match context case SIMULATION(__) then
      arrayScalarRhs(ecr.ty, subs, arrName, context, &preExp, simCode)
    else
      arrName
     */      
  case ASUB(exp=exp as ARRAY(scalar=true), sub={idx}) then
      "ASUB_FAST_ONE"
  case ASUB(exp=e, sub=indexes) then
    <<
    <%daeExp(e, context, &preExp, simCode)
    %>[<%indexes |> index => '<%daeExp(index, context, &preExp, simCode)%>' ;separator=", "%>]
    >>
  else
    'OTHER_ASUB__ERROR'   
end daeExpAsub;

template asubSubsripts(list<Dimension> dims, list<Exp> subs, Text &constSum,
                       Context context, Text &preExp, SimCode simCode)
 "Helper to daeExpAsub."
::=
  match subs case s :: subsRest then
      match dims  case _ :: dimsRest then
          let subStr = daeExp(s, context, &preExp, simCode)
          if dimsRest then //not last
             let ds = dimsRest |> dim => dimension(dim) ;separator="*"
             //if ds then //TODO: assuming every dimension is SOME, is it true ??
             let &constSum += '-(<%ds%>)' //-1 * ds
             '+<%subStr%>*(<%ds%>)<% asubSubsripts(dimsRest, subsRest, &constSum, context, &preExp, simCode) %>'
          else //the last sub, add it to constSum (better optimization on compilation)
             let &constSum += '-1 + <% subStr %>'
             ""          
      else "ERROR_asubSubsripts_not_enough_dims" 
end asubSubsripts;

// TODO: Optimize as in Codegen
// TODO: Use this function in other places where almost the same thing is hard
//       coded
// not used yet 
template arrayScalarRhs(Type ty, list<Exp> subs, Text arrName, Context context,
                        Text &preExp, SimCode simCode)
 "Helper to daeExpAsub."
::=
  let arrayType = expTypeArray(ty, listLength(subs))
  let dimsValuesStr = (subs |> exp => daeExp(exp, context, &preExp, simCode)
                       ;separator=", ")
  <<
  (*ASR<%arrayType%>_element_addr(&<%arrName%>, <%/*dimsLen*/%>, <%dimsValuesStr%>))
  >>
end arrayScalarRhs;








template daeExpBinary(Operator it, Exp exp1, Exp exp2, Context context, Text &preExp, SimCode simCode) ::=
  let e1 = daeExp(exp1, context, &preExp, simCode)
  let e2 = daeExp(exp2, context, &preExp, simCode)
  match it
  case ADD(__) then '(<%e1%> + <%e2%>)'
  //TODO FIX: bad type of SUB operator when on bools; the following does not work
  //case SUB(ty = T_BOOL(__)) then '((<%e1%>?1.0:0.0) - (<%e2%>?1.0:0.0))' //in InitialResidual() ...
  case SUB(__) then 
      //match expTypeFromExp(exp1) //TODO FIX: bool typed CREF returns "double"; the following is not working, too 
      //case "bool" then '((<%e1%>?1.0:0.0) - (<%e2%>?1.0:0.0))' //in InitialResidual() ...
      //HACK!!
      let boolConv = daeExpRealConversionPostfix(exp1, simCode)           
      '((<%e1%><%boolConv%>) - (<%e2%><%boolConv%>))'
  case MUL(__) then '(<%e1%> * <%e2%>)'
  case DIV(__) then '(<%e1%> / <%e2%>)'
  case POW(__) then 'Math.Pow(<%e1%>, <%e2%>)'
  case AND(__) then '(<%e1%> && <%e2%>)'
  case OR(__)  then '(<%e1%> || <%e2%>)'
  case _   then "daeExpBinary:ERR"
end daeExpBinary;


template daeExpUnary(Operator it, Exp exp, Context context, Text &preExp, SimCode simCode) ::=
  let e = daeExp(exp, context, &preExp, simCode)
  match it
  case UMINUS(__)     then '(-<%e%>)'
  case NOT(__)        then '(!<%e%>)'
  case UMINUS_ARR(__) then "UMINUS_ARR_NOT_IMPLEMENTED"
  case _          then "daeExpUnary:ERR"

end daeExpUnary;


template daeExpRelation(Exp inExp, Context context, Text &preExp, SimCode simCode) ::=
  match inExp
  case RELATION(__) then
    let e1 = daeExp(exp1, context, &preExp, simCode)
    let e2 = daeExp(exp2, context, &preExp, simCode)
    let simrel = daeExpSimRelation(inExp, context, e1, e2, &preExp) 
    if  simrel then simrel
    else //non-SIMULATION context or precise equality 
      match operator
      case LESS(ty = T_BOOL(__))        then '(!<%e1%> && <%e2%>)'
      case LESS(ty = T_STRING(__))      then "# string comparison not supported\n"
      case LESS(ty = T_INTEGER(__))
      case LESS(ty = T_REAL(__))        then '(<%e1%> < <%e2%>)'
      case GREATER(ty = T_BOOL(__))     then '(<%e1%> && !<%e2%>)'
      case GREATER(ty = T_STRING(__))   then "# string comparison not supported\n"
      case GREATER(ty = T_INTEGER(__))
      case GREATER(ty = T_REAL(__))     then '(<%e1%> > <%e2%>)'
      case LESSEQ(ty = T_BOOL(__))      then '(!<%e1%> || <%e2%>)'
      case LESSEQ(ty = T_STRING(__))    then "# string comparison not supported\n"
      case LESSEQ(ty = T_INTEGER(__))
      case LESSEQ(ty = T_REAL(__))      then '(<%e1%> <= <%e2%>)'
      case GREATEREQ(ty = T_BOOL(__))   then '(<%e1%> || !<%e2%>)'
      case GREATEREQ(ty = T_STRING(__)) then "# string comparison not supported\n"
      case GREATEREQ(ty = T_INTEGER(__))
      case GREATEREQ(ty = T_REAL(__))   then '(<%e1%> >= <%e2%>)'
      case EQUAL(ty = T_BOOL(__))       then '((!<%e1%> && !<%e2%>) || (<%e1%> && <%e2%>))'
      case EQUAL(ty = T_STRING(__))
      case EQUAL(ty = T_INTEGER(__))
      case EQUAL(ty = T_REAL(__))       then '(<%e1%> == <%e2%>)'
      case NEQUAL(ty = T_BOOL(__))      then '((!<%e1%> && <%e2%>) || (<%e1%> && !<%e2%>))'
      case NEQUAL(ty = T_STRING(__))
      case NEQUAL(ty = T_INTEGER(__))
      case NEQUAL(ty = T_REAL(__))      then '(<%e1%> != <%e2%>)'
      case _                         then "daeExpRelation:ERR"
end daeExpRelation;


template daeExpSimRelation(Exp inExp, Context it, Text e1, Text e2, Text &preExp) ::=
match it
case sim as SIMULATION(__) then
  match inExp
  case RELATION(__) then
     let op = 
         (match operator 
          case LESS(__)      then " < "
         case LESSEQ(__)    then " <= "
         case GREATER(__)   then " > "
         case GREATEREQ(__) then " >= "
         case EQUAL(__)     then " == "
         case NEQUAL(__)    then " != "
         case _             then " daeExpSimRelation:ERR ")
     let &res = buffer ""
     let &preExp +=
       match index
       case -1 then
         '<%tempDecl("bool", res)%> = <%e1%><%op%><%e2%>;<%\n%>'
       else if sim.genDiscrete then
               <<
               <%tempDecl("bool", res)%> = <%e1%><%op%><%e2%>;
               backuprelations[<%index%>] = <%res%>;<%\n%>
               >>
            else
               '<%tempDecl("bool", res)%> = backuprelations[<%index%>];<%\n%>'
     res
       /*
       match operator
       case LESS(__)      then SimRelationSimple(inExp, e1, e2, " < ", &preExp)
       case LESSEQ(__)    then SimRelationEqual(e1, e2, " <= ", &preExp)
       case GREATER(__)   then SimRelationSimple(e1, e2, " > ", &preExp)
       case GREATEREQ(__) then SimRelationEqual(e1, e2, " >= ", &preExp)
       end match
       */
end daeExpSimRelation;


template SimRelationSimple(Text e1, Text e2, String op, Text &preExp) ::=
  let &res = buffer ""    
  let &preExp += 
    <<
    // RELATION( <%e1%><%op%> <%e2%> ) macro expansion
    
    >>
    //<%tempDecl("bool", res)%> = <%e1%><%op%> <%e2%>; if (!<%res%> && isInUpdate && (<%e1%><%op%>= <%e2%>)) { SwapOldVars(); double res1 = <%e1%> - <%e2%>;  SwapOldVars12(); <%res%> = res1<%op%>= (<%e1%> - <%e2%>); SwapOldVars2(); }<%\n%>
  res 
end SimRelationSimple;


template SimRelationEqual(Text e1, Text e2, String op, Text &preExp) ::=
  let &res = buffer ""    
  let &preExp += 
    <<
    // RELATION( <%e1%><%op%>= <%e2%> ) macro expansion
    <%tempDecl("bool", res)%>;  if (isInUpdate) { <%res%> = <%e1%><%op%> <%e2%>;  if(!<%res%> && (<%e1%><%op%>= <%e2%>)) {  SwapOldVars(); double res1 = <%e1%> - <%e2%>;  SwapOldVars12(); <%res%> = res1<%op%>= (<%e1%> - <%e2%>); SwapOldVars2(); }  } else <%res%> = <%e1%><%op%>= <%e2%>;<%\n%>
    >>
  res
end SimRelationEqual;


template daeExpIf(Exp cond, Exp then_, Exp else_, Context context, Text &preExp, SimCode simCode) ::=
  let condExp = daeExp(cond, context, &preExp, simCode)
  let &resVar = buffer ""
  let &preExpThen = buffer ""
  let eThen = daeExp(then_, context, &preExpThen, simCode)
  let &preExpElse = buffer ""
  let eElse = daeExp(else_, context, &preExpElse, simCode)
  let &preExp +=  
  <<
  <%tempDecl(expTypeFromExpArrayIf(then_), resVar)%>;
  if (<%condExp%><%/*if expTypeFromExp(cond) is "bool" then " != 0.0"*/%>) { //cond type is <%expTypeFromExp(cond)%>
    <%preExpThen%>
    <%resVar%> = <%eThen%>;
  } else {
    <%preExpElse%>
    <%resVar%> = <%eElse%>;
  }<%\n%>
  >>
  resVar

/*<<
(<%daeExp(cond, context, &preExp, simCode)
  %> ? <%daeExp(then_, context, &preExp, simCode)%> : <%daeExp(else_, context, &preExp, simCode)%>)
>>*/
end daeExpIf;


template daeExpCall(Exp it, Context context, Text &preExp, SimCode simCode) ::=
  match it
  // special builtins
  case CALL(path=IDENT(name="DIVISION"),
            expLst={e1, e2, e3 as SHARED_LITERAL(__)}) then
    let var1 = daeExp(e1, context, &preExp, simCode)
    let string = //TODO: a local hack here to retrieve the shared litral ... make more like it was designed to
       match simCode
       case SIMCODE(__) then
              match listNth(literals, e3.index)
              case DAE.SCONST(__) then string
              else "TemplErr:division msg string not recognized"        
    let msg = Util.escapeModelicaStringToCString(string)
    match e2 
    case RCONST(__) then 
         //match rr case 0.0 then 'DivBy0(<%var1%>,0.0,"<%msg%>")'
         //else 
         '<%var1%> / <%daeExp(e2, context, &preExp, simCode)%>'
    case _ then
         let var2 = daeExp(e2, context, &preExp, simCode)
        '(<%var2%>!=0.0 ? <%var1%> / <%var2%> : DivBy0(<%var1%>,<%var2%>,"<%msg%>"))' 
    end match 
    
  
  case CALL(path=IDENT(name="der"), expLst={arg as CREF(__)}) then
    derCref(arg.componentRef, simCode)
    
  case CALL(path=IDENT(name="pre"), expLst={arg as CREF(__)}) then
    //let retType = expType(arg.ty)
    <<
    <%match arg.ty case T_INTEGER(__) then "(int)"
    %><% preCref(arg.componentRef, simCode) %>
    >>
  
  case CALL(path=IDENT(name="integer"), expLst={toBeCasted}) 
  case CALL(path=IDENT(name="Integer"), expLst={toBeCasted}) then
    let castedVar = daeExp(toBeCasted, context, &preExp, simCode)
    '((int)<%castedVar%>)'
    
  case CALL(path=IDENT(name="$_start"), expLst={arg as CREF(__)}) then
    startCref(arg.componentRef, simCode)
    
  //'(/*edge(h[<%idx%>])*/H[<%idx%>]!=0.0 && preH[<%idx%>]==0.0)'
  case CALL(path=IDENT(name="edge"), expLst={arg as CREF(__)}) then
    <<
    (/*edge*/<% cref(arg.componentRef, simCode) %> && !(<% preCref(arg.componentRef, simCode) %>))
    >>
  
  case CALL(path=IDENT(name="max"), expLst={e1,e2}) then
    'Math.Max(<%daeExp(e1, context, &preExp, simCode)%>,<%daeExp(e2, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="min"), expLst={e1,e2}) then
    'Math.Min(<%daeExp(e1, context, &preExp, simCode)%>,<%daeExp(e2, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="mod"), expLst={e1,e2}, attr=attr as CALL_ATTR(__)) then
    let var1 = daeExp(e1, context, &preExp, simCode)
    let var2 = daeExp(e2, context, &preExp, simCode)
    'Mod_<%expTypeShort(attr.ty)%>(<%var1%>,<%var2%>)'
  
  case CALL(path=IDENT(name="abs"), expLst={e1}, attr=CALL_ATTR(ty = T_INTEGER(__))) then
    let var1 = daeExp(e1, context, &preExp, simCode)
    'Abs_int(<%var1%>)'
    
  case CALL(path=IDENT(name="abs"), expLst={s1}) then
    'Math.Abs(<%daeExp(s1, context, &preExp, simCode)%>)'
    
  case CALL(path=IDENT(name="log"), expLst={s1}) then
    'Math.Log(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="log10"), expLst={s1}) then
    'Math.Log10(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="exp"), expLst={s1}) then
    'Math.Exp(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="sin"), expLst={s1}) then
    'Math.Sin(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="sqrt"), expLst={s1}) then
    'Math.Sqrt(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="tanh"), expLst={s1}) then
    'Math.Tanh(<%daeExp(s1, context, &preExp, simCode)%>)'
  
  case CALL(path=IDENT(name="noEvent"), expLst={s1}) then
    '(/*noEvent*/<%daeExp(s1, context, &preExp, simCode)%>)'
  
    
  // TODO: add more special builtins (Codegen.generateBuiltinFunction)
  // no return calls
  case CALL(attr=CALL_ATTR(tuple_=false,ty=T_NORETCALL(__))) then
    let argStr = (expLst |> it => daeExp(it, context, &preExp, simCode) ;separator=", ")
    let &preExp += '<%underscorePrefix(attr.builtin)%><%underscorePath(path)%>(<%argStr%>);<%\n%>'
    <<
    /* NORETCALL */
    >>
  // non tuple calls (single return value)
  case CALL(attr=attr as CALL_ATTR(tuple_=false)) then
    let argStr = (expLst |> it => daeExp(it, context, &preExp, simCode) ;separator=", ")
    let funName = underscorePath(path)
    <<
    <%underscorePrefix(attr.builtin)
    %><%funName%>(<%argStr%>)<%if not attr.builtin then '/* !!!TODO:.<%funName%>_rettype_1 */'%>
    >>
  case _ then "daeExpCall:NOT_YT_IMPLEMENTED"
end daeExpCall;

template daeExpArray(Exp aexp, Context context, Text &preExp, SimCode simCode) ::=
  match hackArrayReverseToCref(aexp, context)
  case cr as CREF(__) then daeExpCrefRhs(cr, context, &preExp, simCode)
  case a as ARRAY(__) then
    if scalar then 
        let &arrayVar = buffer ""
      let params = a.array |> e => '(<%expTypeFromExp(e)%>)<%daeExp(e, context, &preExp, simCode)%>' 
                   ;separator=", "
      let &preExp += '<%tempDecl("var",&arrayVar)%> = new <%expTypeArray(a.ty,1)%>(<%listLength(a.array)%>,-1,new[]{<%params%>});<%\n%>'
      arrayVar
    else
      "NON_SCALAR_ARRAY_notYetImplemeted"
end daeExpArray;

template daeExpMatrix(Exp mexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for a cast expression."
::=
  match hackMatrixReverseToCref(mexp, context)  
  case cr as CREF(__) then daeExpCrefRhs(cr, context, &preExp, simCode)
  case MATRIX(matrix={{}}) // special case for empty matrix: create dimensional array Real[0,1]
  case MATRIX(matrix={})   // special case for empty array: create dimensional array Real[0,1] 
    then
    let &tmp = buffer ""
    let &preExp += '<%tempDecl("var",&tmp)%> = new <%expTypeArray(ty,2)%>(0,1);<%\n%>'
    tmp
  //only scalar orthogonal matrix for now  
  case m as MATRIX(matrix=(row1::_)) then
    let &tmp = buffer ""
    let matArr = m.matrix |> row =>
                       (row |> elem =>
                           daeExp(elem, context, &preExp, simCode)
                        ;separator=", ")
                 ;separator=",\n" 
    let &preExp += 
       <<
       <%tempDecl("var",&tmp)%> = new <%expTypeArray(m.ty,2)%>(<%listLength(m.matrix)%>,<%listLength(row1)%>,-1, new[]{
         <%matArr ;anchor%>
       });<%\n%>
       >>
    tmp     
end daeExpMatrix;


template daeExpCast(Exp cexp, Context context, Text &preExp, SimCode simCode)
 "Generates code for a cast expression."
::=
match cexp
case CAST(__) then
  let expVar = daeExp(exp, context, &preExp, simCode)
  match ty
  case T_INTEGER(__)   then '((int)<%expVar%>)'
  case T_REAL(__)  then '((double)<%expVar%>)'
  else "NOT_IMPLEMENTED_CAST"    
end daeExpCast;



template underscorePrefix(Boolean builtin) ::=
  match builtin
  case true then ""
  case false then "_"
end underscorePrefix;

// SECTION: GENERAL TEMPLATES, TEMPORARY VARIABLES

//newVar parameter is assumed to be empty or it can hold an identifier prefix
template tempDecl(String ty, Text &newVar) ::=
  let &newVar += '_tmp<%System.tmpTick()%>'
  <<
  <%ty%> <%newVar%>
  >>
end tempDecl;
  


// SECTION: GENERAL TEMPLATES, TYPES

template varType(Variable var)
 "Generates type for a variable."
::=
match var
case VARIABLE(__) then
  if instDims then
    expTypeArray(ty, listLength(instDims))
  else
    expTypeArrayIf(ty)
end varType;

// TODO: Check with Codegen
template expTypeShort(DAE.Type it) ::=
  match it
  case T_INTEGER(__)    then "int"
  case T_REAL(__)   then "double"
  case T_STRING(__) then "string"
  case T_BOOL(__)   then "bool"
  case T_UNKNOWN(__)  then "UNKNOWN_TYPE_NOT_SUPPORTED"
  case T_ANYTYPE(__)  then "ANYTYPE_TYPE_NOT_SUPPORTED"
  case T_ARRAY(__)  then expTypeShort(ty)   
  case T_COMPLEX(complexClassType=EXTERNAL_OBJ(__)) then "object"
  case T_COMPLEX(__) then '/*struct*/<%underscorePath(ClassInf.getStateName(complexClassType))%>'
  case T_METATYPE(__) case T_METABOXED(__) then "META_TYPE_NOT_SUPPORTED"
  case T_FUNCTION_REFERENCE_VAR(__) then "FN_PTR_NOT_SUPPORTED"
  else "expTypeShort_ERROR"
end expTypeShort;


template expType(DAE.Type ty, Boolean isArray)
 "Generate type helper."
::=
  if isArray 
  then 'expType_<%expTypeArray(ty,0)%>_NOT_YET'
  else expTypeShort(ty)
end expType;

template expTypeArray(DAE.Type ty, Integer dims) ::=
<<
SimArray<%dims%><<%expTypeShort(ty)%>>
>>
end expTypeArray;

template expTypeArrayIf(DAE.Type ty)
 "Generate type helper."
::=
  match ty
  case T_ARRAY(__) then expTypeArray(ty, listLength(dims))
  else expTypeShort(ty)
end expTypeArrayIf;


template expTypeFromExpArrayIf(Exp exp) ::=
  expTypeFromExp(exp)
end expTypeFromExpArrayIf;


template expTypeFromExp(Exp it) ::=
  match it
  case ICONST(__)    then "int"
  case RCONST(__)    then "double"
  case SCONST(__)    then "string"
  case BCONST(__)    then "bool"
  case BINARY(__)
  case UNARY(__)
  case LBINARY(__)
  case LUNARY(__)     then expTypeFromOp(operator)
  case RELATION(__)   then "bool" //TODO: a HACK, it was expTypeFromOp(operator)
  case IFEXP(__)      then expTypeFromExp(expThen)
  case CALL(attr=attr as CALL_ATTR(__)) then expTypeShort(attr.ty)
  case ARRAY(__)
  case MATRIX(__)
  case RANGE(__)
  case CAST(__)
  case CREF(__)
  case CODE(__)       then expTypeShort(ty)
  case ASUB(__)       then expTypeFromExp(exp)
  case REDUCTION(__)  then expTypeFromExp(expr)
  case _          then "expTypeFromExp:ERROR"
end expTypeFromExp;


template expTypeFromOp(Operator it) ::=
  match it
  case ADD(__)
  case SUB(__)
  case MUL(__)
  case DIV(__)
  case POW(__)
  case UMINUS(__)
  case UMINUS_ARR(__)
  case ADD_ARR(__)
  case SUB_ARR(__)
  case MUL_ARR(__)
  case DIV_ARR(__)
  case MUL_ARRAY_SCALAR(__)
  case ADD_ARRAY_SCALAR(__)
  case SUB_SCALAR_ARRAY(__)
  case MUL_SCALAR_PRODUCT(__)
  case MUL_MATRIX_PRODUCT(__)
  case DIV_ARRAY_SCALAR(__)
  case DIV_SCALAR_ARRAY(__)
  case POW_ARRAY_SCALAR(__)
  case POW_SCALAR_ARRAY(__)
  case POW_ARR(__)
  case POW_ARR2(__)
  case LESS(__)
  case LESSEQ(__)
  case GREATER(__)
  case GREATEREQ(__)
  case EQUAL(__)
  case NEQUAL(__)       then  expTypeShort(ty)
  case AND(__)
  case OR(__)
  case NOT(__) then "bool"
  case _ then "expTypeFromOp:ERROR"
end expTypeFromOp;

template dimension(Dimension d)
::=
  match d
  case DAE.DIM_INTEGER(__) then integer
  case DAE.DIM_UNKNOWN(__) then ":"
  else "INVALID_DIMENSION"
end dimension;


template error(Absyn.Info srcInfo, String errMessage)
"Example source template error reporting template to be used together with the sourceInfo() magic function.
Usage: error(sourceInfo(), <<message>>) "
::=
let() = Tpl.addSourceTemplateError(errMessage, srcInfo)
<<

#error <% Error.infoStr(srcInfo) %>  <% errMessage %><%\n%>
>>
end error;

//for completeness; although the error() template above is preferable
template errorMsg(String errMessage)
"Example template error reporting template
 that is reporting only the error message without the usage of source information."
::=
let() = Tpl.addTemplateError(errMessage)
<<

#error <% errMessage %><%\n%>
>>
end errorMsg;


end CodegenCSharp;
// vim: filetype=susan sw=2 sts=2
