package CodegenCpp

import interface SimCodeTV;

// SECTION: SIMULATION TARGET, ROOT TEMPLATE

template translateModel(SimCode simCode) ::=
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
  let()= textFile(simulationHeaderFile(simCode), '<%lastIdentOfPath(modelInfo.name)%>.h')
  let()= textFile(simulationCppFile(simCode), '<%lastIdentOfPath(modelInfo.name)%>.cpp')
  let()= textFile(simulationFunctionsHeaderFile(simCode,modelInfo.functions), 'Functions.h')
  let()= textFile(simulationFunctionsFile(simCode, modelInfo.functions), 'Functions.cpp')
  let()= textFile(simulationMakefile(simCode), '<%fileNamePrefix%>.makefile')
//  algloopfiles(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)  
  algloopfiles(allEquations,simCode) 
   
    // empty result of the top-level template .., only side effects
end translateModel;

template translateFunctions(FunctionCode functionCode)
 "Generates C code and Makefile for compiling and calling Modelica and
  MetaModelica functions." 
::=
  match functionCode
  case FUNCTIONCODE(__) then
  
  "" // Return empty result since result written to files directly
end translateFunctions;

template simulationHeaderFile(SimCode simCode)
 "Generates code for header file for simulation target."
::=
match simCode
case SIMCODE(__) then
  <<
   <%generateHeaderInlcudeString(simCode)%>
   <%generateClassDeclarationCode(simCode)%>
   
   
   >>
end simulationHeaderFile;



template algloopHeaderFile(SimCode simCode,SimEqSystem eq)
 "Generates code for header file for simulation target."
::=
match simCode
case SIMCODE(__) then
  <<
   <%generateAlgloopHeaderInlcudeString(simCode)%>
   <%generateAlgloopClassDeclarationCode(simCode,eq)%>
   
   >>
end algloopHeaderFile;

template simulationFunctionsFile(SimCode simCode, list<Function> functions)
 "Generates the content of the Cpp file for functions in the simulation case."
::=
match simCode
case SIMCODE(modelInfo=MODELINFO(__)) then
  <<
  #include "Modelica.h"
  #include "Functions.h"
 
   Functions::Functions() 
   { 
   } 

   Functions::~Functions() 
   { 
   } 
    void Functions::Assert(bool cond,string msg)
    {
        if(!cond)
            throw std::runtime_error(msg);
    }
    <%functionBodies(functions,simCode)%>
  >>
  
end simulationFunctionsFile;

template simulationFunctionsHeaderFile(SimCode simCode, list<Function> functions)
::=
match simCode
case SIMCODE(modelInfo=MODELINFO(__)) then
  <<
  #pragma once
  #include "Math/Implementation/ArrayOperations.h"
  #include "Math/Implementation/Functions.h"
   #include "LibrariesConfig.h"
  /*****************************************************************************
  *    
  * Simulation code for FunctionCall functions generated by the OpenModelica Compiler.
  *
  *****************************************************************************/
  //external c functions
  extern "C" {
      <%externfunctionHeaderDefinition(functions)%>
  }
  <%functionHeaderBodies1(functions,simCode)%>
  
  class Functions
     { 
      public: 
        Functions(); 
       ~Functions(); 
       <%functionHeaderBodies2(functions,simCode)%>
       void Assert(bool cond,string msg);
      private:
       <%functionHeaderBodies3(functions,simCode)%>
       
     };
  >>
  
end simulationFunctionsHeaderFile;

template simulationMakefile(SimCode simCode)
 "Generates the contents of the makefile for the simulation case."
::=
match simCode
case SIMCODE(modelInfo=MODELINFO(__), makefileParams=MAKEFILE_PARAMS(__), simulationSettingsOpt = sopt) then
let extraCflags = match sopt case SOME(s as SIMULATION_SETTINGS(__)) then
    '<%if s.measureTime then "-D_OMC_MEASURE_TIME "%> <%match s.method
       case "inline-euler" then "-D_OMC_INLINE_EULER"
       case "inline-rungekutta" then "-D_OMC_INLINE_RK"%>'
<<
# Makefile generated by OpenModelica
include ModelicaConfic.inc
# Simulations use -O3 by default
SIM_OR_DYNLOAD_OPT_LEVEL=-O3
CC=<%makefileParams.ccompiler%>
CXX=<%makefileParams.cxxcompiler%>
LINK=<%makefileParams.linker%>
EXEEXT=<%makefileParams.exeext%>
DLLEXT=<%makefileParams.dllext%>
CFLAGS_BASED_ON_INIT_FILE=<%extraCflags%>
CFLAGS=$(CFLAGS_BASED_ON_INIT_FILE) -I"<%makefileParams.omhome%>/include/omc/cpp" -I"$(BOOST_INCLUDE)" <%makefileParams.includes ; separator=" "%> <%makefileParams.cflags%> <%match sopt case SOME(s as SIMULATION_SETTINGS(__)) then s.cflags %>
LDFLAGS=-L"<%makefileParams.omhome%>/bin" -L$(BOOST_LIBS)

MAINFILE=<%lastIdentOfPath(modelInfo.name)%><% if acceptMetaModelicaGrammar() then ".conv"%>.cpp
FUNCTIONFILE=Functions.cpp

.PHONY: <%lastIdentOfPath(modelInfo.name)%>
<%lastIdentOfPath(modelInfo.name)%>: $(MAINFILE) 
<%\t%>$(CXX) -shared -I. -o $(MODELICA_SYSTEM_LIB) $(MAINFILE) $(FUNCTIONFILE)  <%algloopcppfilenames(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>   -L"..//Test//Base/"    $(CFLAGS)  $(LDFLAGS) -lSystem -lMath -lModelicaExternalC -Wl,-Bstatic  -Wl,-Bdynamic
     
>>
end simulationMakefile;

template simulationCppFile(SimCode simCode)
 "Generates code for main cpp file for simulation target."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   #include "Modelica.h"
   #include "<%lastIdentOfPath(modelInfo.name)%>.h"


   using boost::extensions::factory;
   BOOST_EXTENSION_TYPE_MAP_FUNCTION {
    types.get<std::map<std::string, factory<IDAESystem,IGlobalSettings&> > >()
    ["ModelicaSystem"].set<<%lastIdentOfPath(modelInfo.name)%>>();
    }
    
    <%lastIdentOfPath(modelInfo.name)%>::<%lastIdentOfPath(modelInfo.name)%>(IGlobalSettings& globalSettings) 
   :SystemDefaultImplementation()
    <%simulationInitFile(simCode)%>
    { 
    //Number of equations
    <%dimension1(simCode)%>
    _dimZeroFunc= <%zerocrosslength(simCode)%>;
    //Number of residues
    <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then 
      <<     
      _dimResidues=<%numResidues(allEquations)%>;
      >>
    else
      <<
      _dimResidues=0;
      >>
    %>
    //DAE's are not supported yet, Index reduction is enabled
    _dimAE = 0; // algebraic equations
    //Initialize the state vector
    SystemDefaultImplementation::init();
    //Instantiate auxiliary object for event handling functionality
    _event_handling.resetHelpVar =  boost::bind(&<%lastIdentOfPath(modelInfo.name)%>::resetHelpVar, this, _1);
    _historyImpl = new HistoryImplType(globalSettings);
    <%arrayReindex(modelInfo)%>
    //Initialize array elements
    <%arrayInit(simCode)%>
    //Load Algloopsover library
    type_map types;

    std::string algsover_name(SYSTEM_LIB);
    if(!load_single_library(types, algsover_name))
        throw std::invalid_argument("Algsover library could not be loaded");
    std::map<std::string, factory<IAlgLoopSolverFactory> >::iterator iter;
    std::map<std::string, factory<IAlgLoopSolverFactory> >& factories(types.get());
    iter = factories.find("AlgLoopSolverFactory");
    if (iter ==factories.end()) 
    {
        throw std::invalid_argument("No AlgLoopSolverFactory  found");
    }
    <%generateAlgloopsolvers(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>
    }
    <%lastIdentOfPath(modelInfo.name)%>::~<%lastIdentOfPath(modelInfo.name)%>()
    { 
    delete _historyImpl;
    }
   <%init(simCode)%>
   void <%lastIdentOfPath(modelInfo.name)%>::resetTimeEvents()
   {
    <%resetTimeEvents(sampleConditions,whenClauses,simCode)%>
   }
   <%Update(simCode)%>  
   <%writeoutput(simCode)%>
   <%DefaultImplementationCode(simCode)%>
   <%checkForDiscreteEvents(discreteModelVars,simCode)%>
   <%giveZeroFunc1(zeroCrossings,simCode)%>
   <%giveConditions(simCode)%>
   <%setConditions(simCode)%>
   <%saveConditions(simCode)%>
   <%isODE(simCode)%>
   <%DimZeroFunc(simCode)%>
   <%DimInitEquations(simCode)%>
   <%DimUnfixedStates(simCode)%>
   <%DimUnfixedParameters(simCode)%>
   <%DimIntialResiduals(simCode)%>
   <%GetIntialStatus(simCode)%>
   <%SetIntialStatus(simCode)%>
   <%checkConditions(zeroCrossings,whenClauses,simCode)%>
   <%handleSystemEvents(zeroCrossings,whenClauses,simCode)%>
   <%saveall(modelInfo,simCode)%>
   <%resethelpvar(whenClauses,simCode)%>
   <%LabeledDAE(modelInfo.labels,simCode)%>
   >>
end simulationCppFile;


template algloopCppFile(SimCode simCode,SimEqSystem eq)
 "Generates code for main cpp file for algloop system ."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
   let &varDecls = buffer "" 
   let &arrayInit = buffer ""
   let constructorParams = ConstructorParamAlgloop(modelInfo)
   let iniAlgloopParamas = InitAlgloopParams(modelInfo,arrayInit)
   
match eq
    case SES_LINEAR(__)
    case SES_NONLINEAR(__) then
    
  <<
   #include "Modelica.h"
   #include "<%modelname%>Algloop<%index%>.h"
   <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then '#include "Math/Implementation/ArrayOperations.h"'%>

   
    
    <%modelname%>Algloop<%index%>::<%modelname%>Algloop<%index%>(
                                                                <%constructorParams%>,double* z,double* zDot,EventHandling& event_handling
                                                                 ) 
   :AlgLoopDefaultImplementation()
   ,_residuals(NULL)
   ,_z(z)
   ,_zDot(zDot)
   ,_event_handling(event_handling)
   <%alocateLinearSystem(eq)%>
   <%iniAlgloopParamas%>
    { 
      
      <%initAlgloopDimension(eq,varDecls)%>
      
    }
    
   <%modelname%>Algloop<%index%>::~<%modelname%>Algloop<%index%>() 
    { 
               if(_residuals) delete [] _residuals;
    }
   <%algloopRHSCode(simCode,eq)%>
   <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then algloopResiduals(simCode,eq)%>
   <%initAlgloop(simCode,eq)%>
   <%upateAlgloopNonLinear(simCode,eq)%>
   <%upateAlgloopLinear(simCode,eq)%>    
   <%AlgloopDefaultImplementationCode(simCode,eq)%>
   <%getAMatrixCode(simCode,eq)%>
   <%isLinearCode(simCode,eq)%>
    >>
end algloopCppFile;


template upateAlgloopNonLinear( SimCode simCode,SimEqSystem eqn)
  "Generates functions in simulation file."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let () = System.tmpTickReset(0)
  let modelname = lastIdentOfPath(modelInfo.name)
  match eqn
     //case eq as SES_MIXED(__) then functionExtraResiduals(fill(eq.cont,1),simCode)
     case eq as SES_NONLINEAR(__) then
     let &varDecls = buffer "" /*BUFD*/
     let algs = (eq.eqs |> eq2 as SES_ALGORITHM(__) =>
         equation_(eq2, contextOther, &varDecls /*BUFD*/,simCode)
       ;separator="\n")      
     let prebody = (eq.eqs |> eq2 as SES_SIMPLE_ASSIGN(__) =>
         equation_(eq2, contextOther, &varDecls /*BUFD*/,simCode)
       ;separator="\n")   
     let body = (eq.eqs |> eq2 as SES_RESIDUAL(__) hasindex i0 =>
         let &preExp = buffer "" /*BUFD*/
         let expPart = daeExp(eq2.exp, contextSimulationDiscrete,
                            &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
         '<%preExp%>_residuals[<%i0%>] = <%expPart%>;'
         
       ;separator="\n")
  
  <<
  void <%modelname%>Algloop<%index%>::update(const IContinous::UPDATE command)
  {
        <%varDecls%>
       if(command & IContinous::CONTINOUS)
      {
          
           <%algs%>
           <%prebody%>
           <%body%>
      }
      
  }
  >>
end upateAlgloopNonLinear;
  



template upateAlgloopLinear( SimCode simCode,SimEqSystem eqn)
 "Generates functions in simulation file."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let () = System.tmpTickReset(0)
  let modelname = lastIdentOfPath(modelInfo.name)
 match eqn
 case SES_LINEAR(__) then
  let uid = System.tmpTick()
  let size = listLength(vars)
  let aname = 'A<%uid%>'
  let bname = 'b<%uid%>'
    let &varDecls = buffer "" /*BUFD*/
     let &preExp = buffer "" /*BUFD*/


 
 let Amatrix= 
    (simJac |> (row, col, eq as SES_RESIDUAL(__)) =>
      let expPart = daeExp(eq.exp, contextSimulationNonDiscrete, &preExp /*BUFC*/,  &varDecls /*BUFD*/,simCode)
      '<%preExp%>_A[<%row%>][<%col%>]=<%expPart%>;'
  ;separator="\n")
  
 
 
 let bvector =  (beqs |> exp hasindex i0 =>

     let expPart = daeExp(exp, contextSimulationNonDiscrete, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
     '<%preExp%>_b[<%i0%>]=<%expPart%>;'
  ;separator="\n")
 
  
 <<
   void <%modelname%>Algloop<%index%>::update(const IContinous::UPDATE command)
  {
      <%varDecls%>
      <%Amatrix%>
      <%bvector%>      
     
  }
     
     >>  
  
end upateAlgloopLinear;

template inlineVars(Context context, list<SimVar> simvars)
::= match context case INLINE_CONTEXT(__) then match simvars
case {} then ''
else <<

<%simvars |> var => match var case SIMVAR(name = cr as CREF_QUAL(ident = "$DER")) then 'inline_integrate(<%cref(cr)%>);' ;separator="\n"%>
>>
end inlineVars;

template functionBodies(list<Function> functions,SimCode simCode)
 "Generates the body for a set of functions."
::=
  (functions |> fn => functionBody(fn, false,simCode) ;separator="\n")
end functionBodies;

template functionBody(Function fn, Boolean inFunc,SimCode simCode)
 "Generates the body for a function."
::=
  match fn
  case fn as FUNCTION(__)           then functionBodyRegularFunction(fn, inFunc,simCode)
  case fn as EXTERNAL_FUNCTION(__)  then functionBodyExternalFunction(fn, inFunc,simCode)
  case fn as RECORD_CONSTRUCTOR(__) then ""
end functionBody;

template externfunctionHeaderDefinition(list<Function> functions)
 "Generates the body for a set of functions."
::=
  (functions |> fn => extFunDef(fn) ;separator="\n")
end externfunctionHeaderDefinition;

template functionHeaderBodies1(list<Function> functions,SimCode simCode)
 "Generates the body for a set of functions."
::=
  (functions |> fn => functionHeaderBody1(fn,simCode) ;separator="\n")
end functionHeaderBodies1;

template functionHeaderBody1(Function fn,SimCode simCode)
 "Generates the body for a function."
::=
  match fn
  case fn as FUNCTION(__)           then functionHeaderRegularFunction1(fn,simCode)
  case fn as EXTERNAL_FUNCTION(__)  then functionHeaderExternFunction(fn,simCode)
  case fn as RECORD_CONSTRUCTOR(__) then ""
end functionHeaderBody1;

template functionHeaderBodies2(list<Function> functions,SimCode simCode)
 "Generates the body for a set of functions."
::=
  (functions |> fn => functionHeaderBody2(fn,simCode) ;separator="\n")
end functionHeaderBodies2;

template functionHeaderBody2(Function fn,SimCode simCode)
 "Generates the body for a function."
::=
  match fn
  case fn as FUNCTION(__)           then functionHeaderRegularFunction2(fn,simCode)
  case fn as EXTERNAL_FUNCTION(__)  then functionHeaderRegularFunction2(fn,simCode)
  case fn as RECORD_CONSTRUCTOR(__) then ""
end functionHeaderBody2;

template functionHeaderBodies3(list<Function> functions,SimCode simCode)
 "Generates the body for a set of functions."
::=
  (functions |> fn => functionHeaderBody3(fn,simCode) ;separator="\n")
end functionHeaderBodies3;

template functionHeaderBody3(Function fn,SimCode simCode)
 "Generates the body for a function."
::=
  match fn
  case fn as FUNCTION(__)           then functionHeaderRegularFunction3(fn,simCode)
  case fn as EXTERNAL_FUNCTION(__)  then functionHeaderRegularFunction3(fn,simCode)
  case fn as RECORD_CONSTRUCTOR(__) then ""
end functionHeaderBody3;


template extFunDef(Function fn)
 "Generates function header for an external function."
::=
match fn
case func as EXTERNAL_FUNCTION(__) then
 
  //let fn_name = extFunctionName(extName, language)
  let fargsStr = extFunDefArgs(extArgs, language)
  'extern <%extReturnType(extReturn)%> <%extName%>(<%fargsStr%>);'
end extFunDef;

template extFunctionName(String name, String language)
::=
  match language
  case "C" then '<%name%>'
  else error(sourceInfo(), 'Unsupport external language: <%language%>')
end extFunctionName;

template extFunDefArgs(list<SimExtArg> args, String language)
::=
  match language
  case "C" then (args |> arg => extFunDefArg(arg) ;separator=", ")
  else error(sourceInfo(), 'Unsupport external language: <%language%>')
end extFunDefArgs;

template extFunDefArg(SimExtArg extArg)
 "Generates the definition of an external function argument.
  Assume that language is C for now."
::=
  match extArg
  case SIMEXTARG(cref=c, isInput=ii, isArray=ia, type_=t) then
    let name = contextCref2(c,contextFunction)
    let typeStr = extType2(t,ii,ia)
    <<
    <%typeStr%> /*<%name%>*/
    >>
  case SIMEXTARGEXP(__) then
    let typeStr = extType2(type_,true,false)
    <<
    <%typeStr%>
    >>
  case SIMEXTARGSIZE(cref=c) then
    <<
    size_t
    >>
end extFunDefArg;



template extType2(Type type, Boolean isInput, Boolean isArray)
 "Generates type for external function argument or return value."
::=
  let s = match type
  case T_INTEGER(__)         then "int"
  case T_REAL(__)        then "double"
  case T_STRING(__)      then "const char*"
  case T_BOOL(__)        then "int"
  case T_ENUMERATION(__) then "int"
  case T_ARRAY(__)       then extType2(ty,isInput,true)
  case T_COMPLEX(complexClassType=EXTERNAL_OBJ(__))
                      then "void *"
  case T_COMPLEX(complexClassType=RECORD(path=rname))
                      then 'struct <%underscorePath(rname)%>'
  case T_METATYPE(__) case T_METABOXED(__)    then "modelica_metatype"
  else error(sourceInfo(), 'Unknown external C type <%typeString(type)%>')
  match type case T_ARRAY(__) then s else if isInput then (if isArray then '<%match s case "const char*" then "" else "const "%><%s%>*' else s) else '<%s%>*'
end extType2;

template extReturnType(SimExtArg extArg)
 "Generates return type for external function."
::=
  match extArg
  case ex as SIMEXTARG(__)    then extType2(type_,true /*Treat this as an input (pass by value)*/,false)
  case SIMNOEXTARG(__)  then "void"
  case SIMEXTARGEXP(__) then error(sourceInfo(), 'Expression types are unsupported as return arguments <%printExpStr(exp)%>')
  else error(sourceInfo(), "Unsupported return argument")
end extReturnType;


template functionHeaderRegularFunction1(Function fn,SimCode simCode)
::=
match fn
case FUNCTION(__) then 
  let fname = underscorePath(name)
  << 
    typedef tuple< <%outVars |> var => funReturnDefinition1(var,simCode) ;separator=", "%> >  <%fname%>RetType;
    typedef tuple< <%outVars |> var => funReturnDefinition2(var,simCode) ;separator=", "%> > <%fname%>RefRetType;
  >>
end functionHeaderRegularFunction1;



template functionHeaderExternFunction(Function fn,SimCode simCode)
::=
match fn
case EXTERNAL_FUNCTION(__) then 
  let fname = underscorePath(name)
  << 
    typedef tuple< <%outVars |> var => funReturnDefinition1(var,simCode) ;separator=", "%> >  <%fname%>RetType;
  >>
end functionHeaderExternFunction;

template functionHeaderRegularFunction2(Function fn,SimCode simCode)
::=
match fn
case FUNCTION(__) then 
  let fname = underscorePath(name)
  << 
        <%fname%>RetType <%fname%>(<%functionArguments |> var => funArgDefinition(var,simCode) ;separator=", "%>);
  >>
case EXTERNAL_FUNCTION(__) then 
let fname = underscorePath(name)
   <<                               
        <%fname%>RetType <%fname%>(<%funArgs |> var => funArgDefinition(var,simCode) ;separator=", "%>);
   >>
end functionHeaderRegularFunction2;

template functionHeaderRegularFunction3(Function fn,SimCode simCode)
::=
match fn
case FUNCTION(__) then 
  let fname = underscorePath(name)
  << 
        <%fname%>RetType _<%fname%>;
  >>
 case EXTERNAL_FUNCTION(__) then 
 let fname = underscorePath(name)
 << 
        <%fname%>RetType _<%fname%>;
  >>
end functionHeaderRegularFunction3;

template functionBodyRegularFunction(Function fn, Boolean inFunc,SimCode simCode)
 "Generates the body for a Modelica/MetaModelica function."
::=
match fn
case FUNCTION(__) then
  let()= System.tmpTickReset(1)
  let fname = underscorePath(name)
  let retType = if outVars then '<%fname%>RetType' else "void"
  let &varDecls = buffer "" /*BUFD*/
  let &varInits = buffer "" /*BUFD*/
  //let retVar = if outVars then tempDecl(retType, &varDecls /*BUFD*/)
  //let stateVar = if not acceptMetaModelicaGrammar() then tempDecl("state", &varDecls /*BUFD*/)
  let _ = (variableDeclarations |> var hasindex i1 fromindex 1 =>
      varInit(var, "", i1, &varDecls /*BUFD*/, &varInits /*BUFC*/,simCode) ; empty /* increase the counter! */
    )
  //let addRootsInputs = (functionArguments |> var => addRoots(var) ;separator="\n")
  //let addRootsOutputs = (outVars |> var => addRoots(var) ;separator="\n")
  //let funArgs = (functionArguments |> var => functionArg(var, &varInits) ;separator="\n")
  let bodyPart = (body |> stmt  => funStatement(stmt, &varDecls /*BUFD*/,simCode) ;separator="\n")
  let &outVarInits = buffer ""
  let &outVarCopy = buffer ""
  let &outVarAssign = buffer ""
 
     let _ = (outVars |> var hasindex i1 fromindex 0 =>
     varOutput(fn, var,i1, &varDecls, &outVarInits, &outVarCopy, &outVarAssign, simCode)
      ;separator="\n"; empty /* increase the counter! */
     )
   
  
  
  
 
  
  //let boxedFn = if acceptMetaModelicaGrammar() then functionBodyBoxed(fn)
  <<
  <%retType%> Functions::<%fname%>(<%functionArguments |> var => funArgDefinition(var,simCode) ;separator=", "%>)
  {  
    <%varDecls%>
    <%outVarInits%>
    <%varInits%>
    <%bodyPart%>
    <%outVarAssign%>
    return <%if outVars then '_<%fname%>' %>;
  }
  
  <% if inFunc then
  <<
  int in_<%fname%>(type_description * inArgs, type_description * outVar)
  {
    <%functionArguments |> var => '<%funArgDefinition(var,simCode)%>;' ;separator="\n"%>
    <%if outVars then '<%retType%> out;'%>
  
    //MMC_TRY_TOP()


    
    return 0; 
  }
  >>
  %>
  
 
  >>
end functionBodyRegularFunction;


template functionBodyExternalFunction(Function fn, Boolean inFunc,SimCode simCode)
 "Generates the body for an external function (just a wrapper)."
::=
match fn
case efn as EXTERNAL_FUNCTION(__) then
  let()= System.tmpTickReset(1)
  let fname = underscorePath(name)
  let retType = if outVars then '<%fname%>RetType' else "void"
  let &preExp = buffer "" /*BUFD*/
  let &varDecls = buffer "" /*BUFD*/
  // make sure the variable is named "out", doh!
   let retVar = if outVars then '_<%fname%>'  
  let &outVarInits = buffer ""
  let callPart = extFunCall(fn, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  let _ = ( outVars |> var hasindex i1 fromindex 1 => 
            varInit(var, retVar, i1, &varDecls /*BUFD*/, &outVarInits /*BUFC*/,simCode)
            ; empty /* increase the counter! */ 
          )   
  
  let fnBody = <<
  <%retType%> Functions::<%fname%>(<%funArgs |> var => funArgDefinition(var,simCode) ;separator=", "%>)
  {
    /* functionBodyExternalFunction: varDecls */
    <%varDecls%>
    /* functionBodyExternalFunction: preExp */
    <%preExp%>
    /* functionBodyExternalFunction: outputAlloc */
    <%outVarInits%>
    /* functionBodyExternalFunction: callPart */
    <%callPart%>
     /* functionBodyExternalFunction: return */
     return <%retVar%>;
  }
  >>
  <<
  <% if dynamicLoad then
  <<  
  ptrT_<%extFunctionName(extName, language)%> ptr_<%extFunctionName(extName, language)%>=NULL;
  >> %>  
  <%fnBody%>

  <% if inFunc then
  <<
  int in_<%fname%>(type_description * inArgs, type_description * outVar)
  {
    <%funArgs |> VARIABLE(__) => '<%expTypeArrayIf(ty)%> <%contextCref(name,contextFunction,simCode)%>;' ;separator="\n"%>
    <%retType%> out;
    <%funArgs |> arg as VARIABLE(__) => readInVar(arg,simCode) ;separator="\n"%>
    MMC_TRY_TOP()
    out = _<%fname%>(<%funArgs |> VARIABLE(__) => contextCref(name,contextFunction,simCode) ;separator=", "%>);
    MMC_CATCH_TOP(return 1)
    <%outVars |> var as VARIABLE(__) hasindex i1 fromindex 1 => writeOutVar(var, i1) ;separator="\n"%>
    return 0;
  }
  >> %>
  
  
  >>
end functionBodyExternalFunction;

template writeOutVar(Variable var, Integer index)
 "Generates code for writing a variable to outVar."

::=
  match var
  case VARIABLE(ty=T_COMPLEX(complexClassType=RECORD(__))) then
    <<
    write_modelica_record(outVar, <%writeOutVarRecordMembers(ty, index, "")%>);
    >>
  case VARIABLE(__) then

    <<
    write_<%varType(var)%>(outVar, &out.targTest8<%index%>);
    >>
end writeOutVar;


template writeOutVarRecordMembers(Type type, Integer index, String prefix)
 "Helper to writeOutVar."
::=
match type
case T_COMPLEX(varLst=vl, complexClassType = n) then
  let basename = underscorePath(ClassInf.getStateName(n))
  let args = (vl |> subvar as TYPES_VAR(__) =>
      match ty case T_COMPLEX(__) then
        let newPrefix = '<%prefix%>.<%subvar.name%>'
        '<%expTypeRW(ty)%>, <%writeOutVarRecordMembers(ty, index, newPrefix)%>'
      else
        '<%expTypeRW(ty)%>, &(out.targTest7<%index%><%prefix%>.<%subvar.name%>)'
    ;separator=", ")
  <<
  &<%basename%>__desc<%if args then ', <%args%>'%>, TYPE_DESC_NONE
  >>
end writeOutVarRecordMembers;
template expTypeRW(DAE.Type type)
 "Helper to writeOutVarRecordMembers."
::=
  match type
  case T_INTEGER(__)         then "TYPE_DESC_INT"
  case T_REAL(__)        then "TYPE_DESC_REAL"
  case T_STRING(__)      then "TYPE_DESC_STRING"
  case T_BOOL(__)        then "TYPE_DESC_BOOL"
  case T_ENUMERATION(__) then "TYPE_DESC_INT"  
  case T_ARRAY(__)       then '<%expTypeRW(ty)%>_ARRAY'
  case T_COMPLEX(complexClassType=RECORD(__))
                      then "TYPE_DESC_RECORD"
  case T_METATYPE(__) case T_METABOXED(__)    then "TYPE_DESC_MMC"
end expTypeRW;

template readInVar(Variable var,SimCode simCode)
 "Generates code for reading a variable from inArgs."
::=
  match var
  case VARIABLE(name=cr, ty=T_COMPLEX(complexClassType=RECORD(__))) then
    <<
    if (read_modelica_record(&inArgs, <%readInVarRecordMembers(ty, contextCref(cr,contextFunction,simCode))%>)) return 1;
    >>
  case VARIABLE(name=cr, ty=T_STRING(__)) then
    <<
    if (read_<%expTypeArrayIf(ty)%>(&inArgs, <%if not acceptMetaModelicaGrammar() then "(char**)"%> &<%contextCref(name,contextFunction,simCode)%>)) return 1;
    >>
  case VARIABLE(__) then
    <<
    if (read_<%expTypeArrayIf(ty)%>(&inArgs, &<%contextCref(name,contextFunction,simCode)%>)) return 1;
    >>
end readInVar;


template readInVarRecordMembers(Type type, String prefix)
 "Helper to readInVar."
::=
match type
case T_COMPLEX(varLst=vl) then
  (vl |> subvar as TYPES_VAR(__) =>
    match ty case T_COMPLEX(__) then
      let newPrefix = '<%prefix%>.<%subvar.name%>'
      readInVarRecordMembers(ty, newPrefix)
    else
      '&(<%prefix%>.<%subvar.name%>)'
  ;separator=", ")
end readInVarRecordMembers;

template outDecl(String ty, Text &varDecls /*BUFP*/)
 "Declares a temporary variable in varDecls and returns the name."
::=
  let newVar = 'out'
  let &varDecls += '<%ty%> <%newVar%>;<%\n%>'
  newVar
end outDecl;

template extFunCall(Function fun, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates the call to an external function."
::=
match fun
case EXTERNAL_FUNCTION(__) then
  match language
  case "C" then extFunCallC(fun, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
 end extFunCall;



template extFunCallC(Function fun, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates the call to an external C function."
::=
match fun
case EXTERNAL_FUNCTION(__) then
  /* adpro: 2011-06-24 do vardecls -> extArgs as there might be some sets in there! */
  let varDecs = (List.union(extArgs, extArgs) |> arg => extFunCallVardecl(arg, &varDecls /*BUFD*/) ;separator="\n")
  //let fname = if dynamicLoad then 'ptr_<%extFunctionName(extName, language)%>' else '<%extName%>'
  let fname = underscorePath(name)
  let dynamicCheck = if dynamicLoad then 
  <<
  if (<%fname%>==NULL) {
    MODELICA_TERMINATE("dynamic external function <%extFunctionName(extName, language)%> not set!")
  } else
  >>
    else ''
  let args = (extArgs |> arg =>
      extArg(arg, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    ;separator=", ")
  let returnAssign = match extReturn case SIMEXTARG(cref=c) then
      '<%extVarName2(c)%> = '
    else
      ""
  <<
  <%varDecs%>
  <%match extReturn case SIMEXTARG(__) then extFunCallVardecl(extReturn, &varDecls /*BUFD*/)%>
  <%dynamicCheck%>
  <%returnAssign%><%extName%>(<%args%>);
  <%extArgs |> arg => extFunCallVarcopy(arg,fname) ;separator="\n"%>
  <%match extReturn case SIMEXTARG(__) then extFunCallVarcopy(extReturn,fname)%>
  >>
end extFunCallC;



template extFunCallVarcopy(SimExtArg arg, String fnName)
 "Helper to extFunCall."
::=
match arg
case SIMEXTARG(outputIndex=oi, isArray=false, type_=ty, cref=c) then
  match oi case 0 then
    ""
  else
    let cr = '<%extVarName2(c)%>'
    let assginBegin = 'get<<%intAdd(-1,oi)%>>(' 
      let assginEnd = ')'
    
    <<
     <%assginBegin%>_<%fnName%><%assginEnd%> = <%cr%>;
    >>
end extFunCallVarcopy;

template expTypeModelica(DAE.Type ty)
 "Generate type helper."
::=
  expTypeFlag(ty, 2)
end expTypeModelica;


template extArg(SimExtArg extArg, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Helper to extFunCall."
::=
  match extArg
  case SIMEXTARG(cref=c, outputIndex=oi, isArray=true, type_=t) then
    let name = if oi then 'out.targTest5<%oi%>' else contextCref2(c,contextFunction)
    let shortTypeStr = expTypeShort(t)
    ' <%name%>.data()'
  case SIMEXTARG(cref=c, isInput=ii, outputIndex=0, type_=t) then
    let cr = '<%contextCref2(c,contextFunction)%>'
    if acceptMetaModelicaGrammar() then
      (match t case T_STRING(__) then 'MMC_STRINGDATA(<%cr%>)' else '<%cr%>_ext')
    else
      '<%cr%><%match t case T_STRING(__) then ".c_str()" else "_ext"%>'
  case SIMEXTARG(cref=c, isInput=ii, outputIndex=oi, type_=t) then
    '&<%extVarName2(c)%>'
  case SIMEXTARGEXP(__) then
    daeExternalCExp(exp, contextFunction, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case SIMEXTARGSIZE(cref=c) then
    let typeStr = expTypeShort(type_)
    let name = if outputIndex then 'out.targTest4<%outputIndex%>' else contextCref2(c,contextFunction)
    let dim = daeExp(exp, contextFunction, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    '<%name%>.shape()[<%dim%> -1]'
    
end extArg;


template daeExternalCExp(Exp exp, Context context, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
  "Like daeExp, but also converts the type to external C"
::=
  match typeof(exp)
    case T_ARRAY(__) then  // Array-expressions
      let shortTypeStr = expTypeShort(typeof(exp))
      '<%daeExp(exp, context, &preExp, &varDecls,simCode)%>).data()'
    else daeExp(exp, context, &preExp, &varDecls,simCode)
    
    
end daeExternalCExp;

template extFunCallVardecl(SimExtArg arg, Text &varDecls /*BUFP*/)
 "Helper to extFunCall."
::=
  match arg
  case SIMEXTARG(isInput=true, isArray=false, type_=ty, cref=c) then
    match ty case T_STRING(__) then
      ""
    else
      let &varDecls += '<%extType2(ty,true,false)%> <%extVarName2(c)%>;<%\n%>'
      <<
      <%extVarName2(c)%> = (<%extType2(ty,true,false)%>)<%contextCref2(c,contextFunction)%>;
      >>
  case SIMEXTARG(outputIndex=oi, isArray=false, type_=ty, cref=c) then
    match oi case 0 then
      ""
    else
      let &varDecls += '<%extType2(ty,true,false)%> <%extVarName2(c)%>;<%\n%>'
      ""
end extFunCallVardecl;


template varOutput(Function fn, Variable var, Integer ix, Text &varDecls,
          Text &varInits, Text &varCopy, Text &varAssign, SimCode simCode)
 "Generates code to copy result value from a function to dest."
::=
match fn
case FUNCTION(__) then
 let fname = underscorePath(name)
match var
/* The storage size of arrays is known at call time, so they can be allocated
 * before set_memory_state. Strings are not known, so we copy them, etc...
 */
case var as VARIABLE(ty = T_STRING(__)) then 
    if not acceptMetaModelicaGrammar() then
      // We need to strdup() all strings, then allocate them on the memory pool again, then free the temporary string
      let strVar = tempDecl("modelica_string_t", &varDecls)
      let &varCopy += '<%strVar%> = strdup(<%contextCref(var.name,contextFunction,simCode)%>);<%\n%>'
      let &varAssign +=
        <<
        _<%fname%> = init_modelica_string(<%strVar%>);
        free(<%strVar%>);<%\n%>
        >>
      ""
    else
      let &varAssign += '_<%fname%>= <%contextCref(var.name,contextFunction,simCode)%>;<%\n%>'
      ""
case var as VARIABLE(__) then
  let marker = '<%contextCref(var.name,contextFunction,simCode)%>'
  let &varInits += '/* varOutput varInits(<%marker%>) */ <%\n%>'
  let &varAssign += '/* varOutput varAssign(<%marker%>) */ <%\n%>'
  let instDimsInit = (instDims |> exp =>
      daeExp(exp, contextFunction, &varInits /*BUFC*/, &varDecls /*BUFD*/,simCode)
    ;separator="][")
  let assginBegin = 'get<<%ix%>>(' 
  let assginEnd = ')'
  if instDims then
    let &varInits += '<%assginBegin%>_<%fname%><%assginEnd%>.resize((boost::extents[<%instDimsInit%>]));
    <%assginBegin%>_<%fname%><%assginEnd%>.reindex(1);<%\n%>'
    let &varAssign += '<%assginBegin%>_<%fname%><%assginEnd%>=<%contextCref(var.name,contextFunction,simCode)%>;<%\n%>'
    ""
  else
   // let &varInits += initRecordMembers(var)
    let &varAssign += ' <%assginBegin%>_<%fname%><%assginEnd%> = <%contextCref(var.name,contextFunction,simCode)%>;<%\n%>'
    ""
case var as FUNCTION_PTR(__) then
    let &varAssign += '_<%fname%> = (modelica_fnptr) _<%var.name%>;<%\n%>'
    ""
end varOutput;

template varInit(Variable var, String outStruct, Integer i, Text &varDecls /*BUFP*/, Text &varInits /*BUFP*/, SimCode simCode)
 "Generates code to initialize variables.
  Does not return anything: just appends declarations to buffers."
::=
match var
case var as VARIABLE(__) then
  let varName = '<%contextCref(var.name,contextFunction,simCode)%>'
  let typ = '<%varType(var)%>'
  let initVar = match typ case "modelica_metatype" then ' = NULL' else ''
  let addRoot = match typ case "modelica_metatype" then ' mmc_GC_add_root(&<%varName%>, mmc_GC_local_state, "<%varName%>");' else ''  
  let &varDecls += if instDims then 'multi_array<<%expTypeShort(var.ty)%>,<%listLength(instDims)%>> <%varName%><%initVar%>;<%addRoot%><%\n%>' else '<%typ%> <%varName%><%initVar%>;<%addRoot%><%\n%>'
  let varName = if outStruct then '<%outStruct%>.targTest3<%i%>' else '<%contextCref(var.name,contextFunction,simCode)%>'
  let instDimsInit = (instDims |> exp =>
      daeExp(exp, contextFunction, &varInits /*BUFC*/, &varDecls /*BUFD*/,simCode)
    ;separator="][") 
  if instDims then
    (match var.value
    case SOME(exp) then
      let &varInits += '<%varName%>.resize((boost::extents[<%instDimsInit%>]));
      <%varName%>.reindex(1);<%\n%>'
      let defaultValue = varDefaultValue(var, outStruct, i, varName, &varDecls, &varInits,simCode)
      let &varInits += defaultValue
      let var_name = if outStruct then 
        '<%extVarName(var.name,simCode)%>' else 
        '<%contextCref(var.name, contextFunction,simCode)%>' 
      let defaultValue1 = '<%var_name%> = <%daeExp(exp, contextFunction, &varInits  /*BUFC*/, &varDecls /*BUFD*/,simCode)%>;<%\n%>'
      let &varInits += defaultValue1
      " "
    else
      let &varInits += '<%varName%>.resize((boost::extents[<%instDimsInit%>]));
      <%varName%>.reindex(1);<%\n%>'
      let defaultValue = varDefaultValue(var, outStruct, i, varName, &varDecls, &varInits,simCode)
     let &varInits += defaultValue
      "")
  else
    (match var.value
    case SOME(exp) then
      let defaultValue = '<%contextCref(var.name,contextFunction,simCode)%> = <%daeExp(exp, contextFunction, &varInits  /*BUFC*/, &varDecls /*BUFD*/,simCode)%>;<%\n%>'
      let &varInits += defaultValue

      " "
    else
      "")
case var as FUNCTION_PTR(__) then
  let &ignore = buffer ""
  let &varDecls += functionArg(var,&ignore)
  ""

end varInit;

template functionArg(Variable var, Text &varInit)
"Shared code for function arguments that are part of the function variables and valueblocks.
Valueblocks need to declare a reference to the function while input variables
need to initialize."
::=
match var
case var as FUNCTION_PTR(__) then
  let typelist = (args |> arg => mmcVarType(arg) ;separator=", ")
  let rettype = '<%name%>RetType'
  match tys
    case {} then
      let &varInit += '_<%name%> = (void(*)(<%typelist%>)) <%name%><%\n%>;'
      'void(*_<%name%>)(<%typelist%>);<%\n%>'
    else

      let &varInit += '_<%name%> = (<%rettype%>(*)(<%typelist%>)) <%name%>;<%\n%>'
      <<
      <% tys |> arg hasindex i1 fromindex 1 => '#define <%rettype%>_<%i1%> targTest2<%i1%>' ; separator="\n" %>
      typedef struct <%rettype%>_s
      {
        <% tys |> ty hasindex i1 fromindex 1 => 'modelica_<%mmcTypeShort(ty)%> targTest1<%i1%>;' ; separator="\n" %> 
      } <%rettype%>;
      <%rettype%>(*_<%name%>)(<%typelist%>);<%\n%>
      >>
  end match
end functionArg;

template mmcVarType(Variable var)
::=
  match var
  case VARIABLE(__) then 'modelica_<%mmcTypeShort(ty)%>'
  case FUNCTION_PTR(__) then 'modelica_fnptr'
end mmcVarType;

template mmcTypeShort(DAE.Type type)
::=
  match type
  case T_INTEGER(__)                     then "integer"
  case T_REAL(__)                    then "real"
  case T_STRING(__)                  then "string"
  case T_BOOL(__)                    then "integer"
  case T_ENUMERATION(__)             then "integer"
  case T_ARRAY(__)                   then "array"
  case T_METATYPE(__) case T_METABOXED(__)                then "metatype"
  case T_FUNCTION_REFERENCE_VAR(__)  then "fnptr"
  else "mmcTypeShort:ERROR"
end mmcTypeShort;

template extVarName(ComponentRef cr, SimCode simCode)
::= '<%contextCref(cr,contextFunction,simCode)%>_ext'
end extVarName;

template extVarName2(ComponentRef cr)
::= '<%contextCref2(cr,contextFunction)%>_ext'
end extVarName2;

template varDefaultValue(Variable var, String outStruct, Integer i, String lhsVarName,  Text &varDecls /*BUFP*/, Text &varInits /*BUFP*/,SimCode simCode)
::=
match var
case var as VARIABLE(__) then
  match value
  case SOME(CREF(componentRef = cr)) then
    'copy_<%expTypeShort(var.ty)%>_array_data(&<%contextCref(cr,contextFunction,simCode)%>, &<%outStruct%>.targTest9<%i%>);<%\n%>'
  case SOME(arr as ARRAY(__)) then
    let arrayExp = '<%daeExp(arr, contextFunction, &varInits /*BUFC*/, &varDecls /*BUFD*/,simCode)%>'
    <<
    copy_<%expTypeShort(var.ty)%>_array_data(&<%arrayExp%>, &<%lhsVarName%>);<%\n%>
    >>
end varDefaultValue;


template funArgDefinition(Variable var,SimCode simCode)
::=
  match var
  case VARIABLE(__) then '<%varType1(var)%> <%contextCref(name,contextFunction,simCode)%>'
  case FUNCTION_PTR(__) then 'modelica_fnptr <%name%>'
end funArgDefinition;

template funExtArgDefinition(SimExtArg extArg,SimCode simCode)
::=
  match extArg
  case SIMEXTARG(cref=c, isInput=ii, isArray=ia, type_=t) then
    let name = contextCref(c,contextFunction,simCode)
    let typeStr = expTypeFlag(t,5)
    <<
    <%typeStr%> <%name%>
    >>
  case SIMEXTARGEXP(__) then
    let typeStr = expTypeFlag(type_,5)
    <<
    <%typeStr%>
    >>
  case SIMEXTARGSIZE(cref=c) then
    <<
    size_t
    >>
end funExtArgDefinition;

template funReturnDefinition1(Variable var,SimCode simCode)
::=
  match var
  case VARIABLE(__) then '<%varType1(var)%>'
  case FUNCTION_PTR(__) then 'modelica_fnptr <%name%>'
end funReturnDefinition1;

template funReturnDefinition2(Variable var,SimCode simCode)
::=
  match var
  case VARIABLE(__) then '<%varType2(var)%>'
  case FUNCTION_PTR(__) then 'modelica_fnptr <%name%>'
end funReturnDefinition2;

template varType(Variable var)
 "Generates type for a variable."
::=
match var
case var as VARIABLE(__) then
  if instDims then
    expTypeArray(var.ty)
  else
    expTypeArrayIf(var.ty)
end varType;


template varType1(Variable var)
::=
match var
case var as VARIABLE(__) then
     if instDims then 'multi_array<<%expTypeShort(var.ty)%>,<%listLength(instDims)%>> ' else expTypeFlag(var.ty, 5)
end varType1;

template varType2(Variable var)
::=
match var
case var as VARIABLE(__) then
     if instDims then 'multi_array_ref<<%expTypeShort(var.ty)%>,<%listLength(instDims)%>> ' else expTypeFlag(var.ty, 5)
end varType2;




template funStatement(Statement stmt, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates function statements."
::=
  match stmt
  case ALGORITHM(__) then
    (statementLst |> stmt =>
      algStatement(stmt, contextFunction, &varDecls /*BUFD*/,simCode)
    ;separator="\n") 
  else
    "NOT IMPLEMENTED FUN STATEMENT"
end funStatement;

template init(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__))  then
   let () = System.tmpTickReset(0)
   let &varDecls = buffer "" /*BUFD*/
   let initVariables = initvar(modelInfo,simCode)
   let initFunctions = functionInitial(startValueEquations,varDecls,simCode)
   let initZeroCrossings = functionOnlyZeroCrossing(zeroCrossings,varDecls,simCode)
   let initTimeEventFunctions = timeEventCondition(sampleConditions,varDecls,simCode)
   let initEventHandling = eventHandlingInit(simCode)
   let initOutputIndices = outputIndices(modelInfo)
   let initBoundParameters = boundParameters(parameterEquations,varDecls,simCode)
   let initALgloopSolvers = initAlgloopsolvers(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)
   
  <<
   void <%lastIdentOfPath(modelInfo.name)%>::init(double ts,double te)
   {
    <%varDecls%>  
    <%initVariables%>
   <%initFunctions%>
   checkConditions(0,true);
    <%initTimeEventFunctions%>
    <%initEventHandling%>
    _event_handling.init(this,<%helpvarlength(simCode)%>);
      vector<unsigned int> var_ouputs_idx;
    <%initOutputIndices%>
   _historyImpl->setOutputs(var_ouputs_idx);
   _historyImpl->clear();
    <%initALgloopSolvers%>
    //initialize equations
   <%initBoundParameters%>
  
    }
   >>
end init;

template initAlgloop(SimCode simCode,SimEqSystem eq)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
  let initalgvars = initAlgloopvars(modelInfo,simCode)
  let &varDecls = buffer "" 
  match eq
  case SES_NONLINEAR(__) then
  <<
  void <%modelname%>Algloop<%index%>::init()
  {  
         AlgLoopDefaultImplementation::init();
         <%initAlgloopEquation(eq,varDecls,simCode)%>
         if(_residuals) 
        {
            delete [] _residuals;
            _residuals = NULL;
        }
        _residuals = new double[_dim[0]];

    // Update the equations once before start of simulation
    update(IContinous::ALL);
   }
  >>
 case SES_LINEAR(__) then
   <<
     void <%modelname%>Algloop<%index%>::init()
     {  
         AlgLoopDefaultImplementation::init();
         <%initAlgloopEquation(eq,varDecls,simCode)%>
         
        // Update the equations once before start of simulation
        update(IContinous::ALL);
     }
   >>
 
end initAlgloop;


template getAMatrixCode(SimCode simCode,SimEqSystem eq)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
  let initalgvars = initAlgloopvars(modelInfo,simCode)
  let &varDecls = buffer "" 
  match eq
  case SES_NONLINEAR(__) then
  <<
  void <%modelname%>Algloop<%index%>::giveAMatrix(double* A_matrix)
  {  
         
   }
  >>
 case SES_LINEAR(__) then
   <<
     void <%modelname%>Algloop<%index%>::giveAMatrix(double* A_matrix)
     {  
          memcpy(A_matrix,_A.data(),_dim[0]*_dim[0]*sizeof(double));
     }
   >>
 
end getAMatrixCode;


template algloopRHSCode(SimCode simCode,SimEqSystem eq)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
  let initalgvars = initAlgloopvars(modelInfo,simCode)
  let &varDecls = buffer "" 
  match eq
  case SES_NONLINEAR(__) then
  <<
  void <%modelname%>Algloop<%index%>::giveRHS(double* doubleResiduals, int* intResiduals, bool* boolResiduals)
    {
    if(doubleResiduals)
        memcpy(doubleResiduals,_residuals,_dim[0]*sizeof(double));
    }
  >>
 case SES_LINEAR(__) then
   <<
    void <%modelname%>Algloop<%index%>::giveRHS(double* doubleResiduals, int* intResiduals, bool* boolResiduals)
    {
        if(doubleResiduals)
        memcpy(doubleResiduals,_b.data(),_dim[0]*sizeof(double));
    }
   >>
 
end algloopRHSCode;


template algloopResiduals(SimCode simCode,SimEqSystem eq)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
match eq
 case SES_LINEAR(__) then
   <<
    int <%modelname%>Algloop<%index%>::giveDimResiduals(int index)
    {
      if(index == 0) return _dim[0];
      else if(index == 1) return _dim[1];
      else if(index == 2) return _dim[2];
    }
   
    void <%modelname%>Algloop<%index%>::giveResiduals(double* doubleResiduals, int* intResiduals, bool* boolResiduals)
    {        
        ublas::matrix<double> A=toMatrix(_dim[0],_dim[0],_A.data());
        double* doubleUnknowns = new double[_dim[0]];
        int* intUnknowns = new int[_dim[1]];
        bool* boolUnknowns = new bool[_dim[2]];
        giveVars(doubleUnknowns,intUnknowns,boolUnknowns);
        ublas::vector<double> x=toVector(_dim[0],doubleUnknowns);
        ublas::vector<double> b=toVector(_dim[0],_b.data());
        b=ublas::prod(ublas::trans(A),x)-b;         
        if(doubleResiduals) std::copy(b.data().begin(), b.data().end(), doubleResiduals);
    }
   >>  
 case SES_NONLINEAR(__) then
    <<
    int <%modelname%>Algloop<%index%>::giveDimResiduals(int index)
    {
      if(index == 0) return _dim[0];
      else if(index == 1) return _dim[1];
      else if(index == 2) return _dim[2];
    }
   
    void <%modelname%>Algloop<%index%>::giveResiduals(double* doubleResiduals, int* intResiduals, bool* boolResiduals)
    {        
          if(doubleResiduals)
        memcpy(doubleResiduals,_residuals,_dim[0]*sizeof(double));
    }
   >>  
 case SES_MIXED(__) then algloopResiduals(simCode,cont)
end algloopResiduals;

template isLinearCode(SimCode simCode,SimEqSystem eq)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
  let initalgvars = initAlgloopvars(modelInfo,simCode)
  let &varDecls = buffer "" 
  match eq
  case SES_NONLINEAR(__) then
  <<
  bool <%modelname%>Algloop<%index%>::isLinear()
  {  
         return false;
   }
  >>
 case SES_LINEAR(__) then
   <<
     bool <%modelname%>Algloop<%index%>::isLinear()
     {  
          return true;
     }
   >>
 
end isLinearCode;


template initAlgloopEquation(SimEqSystem eq, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates a non linear equation system."
::=
match eq
case SES_NONLINEAR(__) then
  let size = listLength(crefs)
  <<
   
   <%crefs |> name hasindex i0 =>
    let namestr = cref1(name,simCode)
    <<
    _xd[<%i0%>] = <%namestr%>;
     >>
  ;separator="\n"%>
   >>
 case SES_LINEAR(__)then
  let &varDecls = buffer "" /*BUFD*/
     let &preExp = buffer "" /*BUFD*/
 let Amatrix= 
    (simJac |> (row, col, eq as SES_RESIDUAL(__)) =>
      let expPart = daeExp(eq.exp, contextSimulationNonDiscrete, &preExp /*BUFC*/,  &varDecls /*BUFD*/,simCode)
      '<%preExp%>_A[<%row%>][<%col%>]=<%expPart%>;'
  ;separator="\n")
  
 
 
 let bvector =  (beqs |> exp hasindex i0 =>

     let expPart = daeExp(exp, contextSimulationNonDiscrete, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
     '<%preExp%>_b[<%i0%>]=<%expPart%>;'
  ;separator="\n")
 
 <<
      <%varDecls%>
      <%Amatrix%>
      <%bvector%>
  >> 
   
end initAlgloopEquation;






template giveAlgloopvars(SimEqSystem eq,SimCode simCode)
 "Generates a non linear equation system."
::=
match eq
case SES_NONLINEAR(__) then
  let size = listLength(crefs)
  <<
   
   <%crefs |> name hasindex i0 =>
    let namestr = cref1(name,simCode)
    <<
      doubleUnknowns[<%i0%>] = <%namestr%>;
     >>
  ;separator="\n"%>
   >>
 case SES_LINEAR(__) then
   <<
   
      <%vars |> SIMVAR(__) hasindex i0 => 'doubleUnknowns[<%i0%>] =<%cref1(name,simCode)%>;' ;separator="\n"%><%inlineVars(contextSimulationNonDiscrete,vars)%>
   >>  
 
end giveAlgloopvars;




template writeAlgloopvars(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      writeAlgloopvars2(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator=" ")
  
  <<
  <%algloopsolver%>
  >>
end writeAlgloopvars;


template writeAlgloopvars2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case e as SES_NONLINEAR(__) then
    let size = listLength(crefs)
  <<
   double algloopvars<%index%>[<%size%>];
   _algLoop<%index%>->giveVars(algloopvars<%index%>,NULL,NULL);
   <%crefs |> name hasindex i0 =>
    let namestr = cref(name)
    <<
     <%namestr%> = algloopvars<%index%>[<%i0%>];
     >>
  ;separator="\n"%>
  
   >>
  case e as SES_LINEAR(__) then
    let size = listLength(vars)
    let algloopid = index
  <<
   double algloopvars<%algloopid%>[<%size%>];
   _algLoop<%index%>->giveVars(algloopvars<%algloopid%>,NULL,NULL);
   
    <%vars |> SIMVAR(__) hasindex i0 => '<%cref1(name,simCode)%> = algloopvars<%algloopid%>[<%i0%>];' ;separator="\n"%>
 
  
   >>
 end writeAlgloopvars2;





template setAlgloopvars(SimEqSystem eq,SimCode simCode)
 "Generates a non linear equation system."
::=
match eq
case SES_NONLINEAR(__) then
  let size = listLength(crefs)
  <<
   
   <%crefs |> name hasindex i0 =>
    let namestr = cref1(name,simCode)
    <<
    <%namestr%>  = doubleUnknowns[<%i0%>];
     >>
  ;separator="\n"%>
   >>
  case SES_LINEAR(__) then
  <<
   
   <%vars |> SIMVAR(__) hasindex i0 => '<%cref1(name,simCode)%>=doubleUnknowns[<%i0%>];' ;separator="\n"%><%inlineVars(contextSimulationNonDiscrete,vars)%>
  
   >>
end setAlgloopvars;

template initAlgloopDimension(SimEqSystem eq, Text &varDecls /*BUFP*/)
 "Generates a non linear equation system."
::=
match eq
case SES_NONLINEAR(__) then
  let size = listLength(crefs)
  <<
    // Number of unknowns/equations according to type (0: double, 1: int, 2: bool)
    _dim[0] = <%size%>;
    _dim[1] = 0;
    _dim[2] = 0;
  >>
  case SES_LINEAR(__) then
  let size = listLength(vars)
  <<
    // Number of unknowns/equations according to type (0: double, 1: int, 2: bool)
    _dim[0] = <%size%>;
    _dim[1] = 0;
    _dim[2] = 0;
    fill_array(_A,0.0);
    fill_array(_b,0.0);
  >>
 
end initAlgloopDimension;

template alocateLinearSystem(SimEqSystem eq)
 "Generates a non linear equation system."
::=
match eq
case SES_LINEAR(__) then
   let size = listLength(vars)
  <<
    ,_A(boost::extents[<%size%>][<%size%>],boost::fortran_storage_order())
   ,_b(boost::extents[<%size%>])
  >>
 
end alocateLinearSystem;

template Update(SimCode simCode)
::=
match simCode
case SIMCODE(__) then
  <<
    <%update(allEquations,whenClauses,simCode)%>
  >>
end Update;
/*<%update(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>*/

  
template writeoutput(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<  
   void <%lastIdentOfPath(modelInfo.name)%>::writeOutput(const OUTPUT command)
   {
      
      //Write head line
    if (command & HEAD_LINE)
    {
    vector<string> head;
    head+= <%writeoutput1(modelInfo)%>
    _historyImpl->write(head);
    }
    //Write the current values
    else
    {
    
    HistoryImplType::value_type_v v(<%numAlgvars(modelInfo)%>+<%numStatevars(modelInfo)%>);
    HistoryImplType::value_type_dv v2(<%numDerivativevars(modelInfo)%>);
    <%writeoutput2(modelInfo)%>
    <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then
      <<
      HistoryImplType::value_type_r v3(<%numResidues(allEquations)%>);      
      <%(allEquations |> eqs => (eqs |> eq => writeoutputAlgloopsolvers(eq,simCode));separator="\n")%>
      double residues [] = {<%(allEquations |> eqn => writeoutput3(eqn, simCode));separator=","%>};
      for(int i=0;i<<%numResidues(allEquations)%>;i++) v3(i) = residues[i];
      _historyImpl->write(v,v2,v3,time);
      >>
    else
      <<
      _historyImpl->write(v,v2,time);
      >>
    %>
    }
   }
  >>
  //<%writeAlgloopvars(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>
end writeoutput;

template writeoutputAlgloopsolvers(SimEqSystem eq, SimCode simCode)
::=
  match eq
  case SES_LINEAR(__)
  case SES_NONLINEAR(__)
  case SES_MIXED(__)
    then 
    let num = index
    match simCode
    case SIMCODE(modelInfo = MODELINFO(__)) then    
    <<
    double* doubleResiduals<%num%> = new double[_algLoop<%num%>->giveDimResiduals(0)];
    int* intResiduals<%num%> = new int[_algLoop<%num%>->giveDimResiduals(1)];
    bool* boolResiduals<%num%> = new bool[_algLoop<%num%>->giveDimResiduals(2)];
    _algLoop<%num%>->giveResiduals(doubleResiduals<%num%>,intResiduals<%num%>,boolResiduals<%num%>);
  
    >>
  else
    " "
 end writeoutputAlgloopsolvers;

template writeoutput3(SimEqSystem eqn, SimCode simCode)
::=
  match eqn
  case SES_RESIDUAL(__) then
  <<
  >>
  case  SES_SIMPLE_ASSIGN(__) then
  <<
  <%cref1(cref,simCode)%>
  >>
  case SES_ARRAY_CALL_ASSIGN(__) then
  <<
  >>
  case SES_ALGORITHM(__) then
  <<
  >>
  case e as SES_LINEAR(__) then
  <<
  <%(vars |> var hasindex myindex2 => writeoutput4(e.index,myindex2));separator=","%>
  >>
  case e as SES_NONLINEAR(__) then
  <<
  <%(eqs |> eq hasindex myindex2 => writeoutput4(e.index,myindex2));separator=","%>
  >>
  case SES_MIXED(__) then writeoutput3(cont,simCode)
  case SES_WHEN(__) then
  <<
  >>
  else
  <<
  >>
end writeoutput3;

template writeoutput4(Integer index, Integer myindex2)
::=
 <<
 *(doubleResiduals<%index%>+<%myindex2%>)
 >>
end writeoutput4;

template generateHeaderInlcudeString(SimCode simCode)
 "Generates header part of simulation file."
::=
match simCode
case SIMCODE(modelInfo=MODELINFO(__), extObjInfo=EXTOBJINFO(__)) then
  <<
  #pragma once
  #define BOOST_EXTENSION_SYSTEM_DECL BOOST_EXTENSION_IMPORT_DECL
  #define BOOST_EXTENSION_EVENTHANDLING_DECL BOOST_EXTENSION_IMPORT_DECL
  #include <boost\shared_ptr.hpp>
  #include "System/Implementation/SystemDefaultImplementation.h"
  #include "Math/Implementation/ArrayOperations.h"
  #include "System/Implementation/EventHandling.h"
  #include "Settingsfactory/Interfaces/IGlobalSettings.h"
  #include "System/Interfaces/IAlgLoopSolverFactory.h"
  #include "System/Interfaces/IAlgLoopSolver.h"
  #include "Functions.h"
  <%algloopfilesInclude(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>
  #include "DataExchange/Interfaces/IHistory.h"
  #include "HistoryImpl.h"
  <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then
  <<
  #include "ReduceDAE/Interfaces/IReduceDAE.h"
  #include "policies/BufferReaderWriter.h"
  typedef HistoryImpl<BufferReaderWriter,<%numAlgvars(modelInfo)%>+<%numStatevars(modelInfo)%>,<%numDerivativevars(modelInfo)%>,<%numResidues(allEquations)%>> HistoryImplType;  

  >>
  else
  <<
  #include "policies/TextFileWriter.h"
  typedef HistoryImpl<TextFileWriter,<%numAlgvars(modelInfo)%>+<%numStatevars(modelInfo)%>,<%numDerivativevars(modelInfo)%>,0> HistoryImplType;
  
  >>%>
  /*****************************************************************************
  * 
  * Simulation code for <%lastIdentOfPath(modelInfo.name)%> generated by the OpenModelica Compiler.
  * System class <%lastIdentOfPath(modelInfo.name)%> implements the Interface IDAESystem
  *
  *****************************************************************************/
   >>
end generateHeaderInlcudeString;



template generateAlgloopHeaderInlcudeString(SimCode simCode)
 "Generates header part of simulation file."
::=
match simCode
case SIMCODE(modelInfo=MODELINFO(__), extObjInfo=EXTOBJINFO(__)) then
  <<
  #pragma once
  #define BOOST_EXTENSION_ALGLOOPDEFAULTIMPL_DECL BOOST_EXTENSION_IMPORT_DECL
  #define BOOST_EXTENSION_EVENTHANDLING_DECL BOOST_EXTENSION_IMPORT_DECL 
  #include "System/Interfaces/IDAESystem.h"
  #include "System/Implementation/AlgLoopDefaultImplementation.h"
  #include "System/Implementation/EventHandling.h"
  #include "Functions.h"
  >>
end generateAlgloopHeaderInlcudeString;

template generateClassDeclarationCode(SimCode simCode)
 "Generates class declarations."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
  class <%lastIdentOfPath(modelInfo.name)%>: public IDAESystem ,public IContinous ,public IEvent ,public ISystemProperties, public  ISystemInitialization <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then ',public IReduceDAE '%>,public SystemDefaultImplementation
  { 
  public: 
      <%lastIdentOfPath(modelInfo.name)%>(IGlobalSettings& globalSettings); 
      ~<%lastIdentOfPath(modelInfo.name)%>();
      
       <%generateMethodDeclarationCode(simCode)%>
  
  private:    
    //Methods:
    

     void resetHelpVar(const int index);
     
     //Variables:
     EventHandling _event_handling;
     
     <%MemberVariable(modelInfo)%>
     <%conditionvariable(zeroCrossings,sampleConditions,simCode)%>
     Functions _functions;
     HistoryImplType* _historyImpl;
     boost::shared_ptr<IAlgLoopSolverFactory>
        _algLoopSolverFactory;    ///< Factory that provides an appropriate solver
     <%generateAlgloopsolverVariables(odeEquations,algebraicEquations,whenClauses,parameterEquations,simCode)%>
   };
  >>
end generateClassDeclarationCode;

template generateAlgloopClassDeclarationCode(SimCode simCode,SimEqSystem eq)
 "Generates class declarations."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  let modelname = lastIdentOfPath(modelInfo.name)
  let algvars = MemberVariableAlgloop(modelInfo)
  let constructorParams = ConstructorParamAlgloop(modelInfo)
  match eq
      case SES_LINEAR(__)
    case SES_NONLINEAR(__) then
  <<
  class <%modelname%>Algloop<%index%>: public IAlgLoop, public AlgLoopDefaultImplementation
  { 
  public: 
      <%modelname%>Algloop<%index%>(
                                       <%constructorParams%>
                                       ,double* z,double* zDot
                                       ,EventHandling& event_handling
                                      ); 
      ~<%modelname%>Algloop<%index%>();
      
       <%generateAlgloopMethodDeclarationCode(simCode)%>
  
  private:    
    Functions _functions;
    double         *_residuals;        ///< Auxillary variables
    //states
    double* _z;
    //state derivatives
    double* _zDot;
    // A matrix
    boost::multi_array<double,2> _A;
    //b vector
    boost::multi_array<double,1> _b;
     <%algvars%>
    EventHandling& _event_handling; 
   };
  >>
end generateAlgloopClassDeclarationCode;


template DefaultImplementationCode(SimCode simCode)

::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<< 
  // Release instance
void <%lastIdentOfPath(modelInfo.name)%>::destroy()
{
    delete this;
}

// Set current integration time 
void <%lastIdentOfPath(modelInfo.name)%>::setTime(const double& t)
{
    SystemDefaultImplementation::setTime(t);
}

// Returns the vector with all time events 
event_times_type <%lastIdentOfPath(modelInfo.name)%>::getTimeEvents()
{
    return _event_handling.getTimeEvents();
}

// Provide number (dimension) of variables according to the index
int <%lastIdentOfPath(modelInfo.name)%>::getDimVars(const INDEX index) const
{
    return(SystemDefaultImplementation::getDimVars(index));
}

// Provide number (dimension) of right hand sides (equations and/or residuals) according to the index
int <%lastIdentOfPath(modelInfo.name)%>::getDimRHS(const INDEX index ) const
{
     return(SystemDefaultImplementation::getDimRHS(index));
}

// Provide variables with given index to the system
void <%lastIdentOfPath(modelInfo.name)%>::giveVars(double* z, const INDEX index)
{    
    SystemDefaultImplementation::giveVars(z,index);
}

// Set variables with given index to the system
void <%lastIdentOfPath(modelInfo.name)%>::setVars(const double* z, const INDEX index)
{
    SystemDefaultImplementation::setVars(z,index);
}
 
// Provide the right hand side (according to the index) 
void <%lastIdentOfPath(modelInfo.name)%>::giveRHS(double* f, const INDEX index)
{
<%if Flags.isSet(Flags.WRITE_TO_BUFFER) then
<<
if(index == IContinous::ALL_RESIDUES)
{
    <%(allEquations |> eqs => (eqs |> eq => writeoutputAlgloopsolvers(eq,simCode));separator="\n")%>
    double residues [] = {<%(allEquations |> eqn => writeoutput3(eqn, simCode));separator=","%>};
    for(int i=0;i<<%numResidues(allEquations)%>;i++) *(f+i) = residues[i];
}
else SystemDefaultImplementation::giveRHS(f,index);
>>
else
<<
    SystemDefaultImplementation::giveRHS(f,index);
>>%>
}

void <%lastIdentOfPath(modelInfo.name)%>::giveJacobianSparsityPattern(SparcityPattern pattern)
{
  throw std::runtime_error("giveJacobianSparsityPattern is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::giveJacobian(SparseMatrix matrix)
{
  throw std::runtime_error("giveJacobian is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::giveMassSparsityPattern(SparcityPattern pattern)
{
  throw std::runtime_error("giveMassSparsityPattern is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::giveMassMatrix(SparseMatrix matrix)
{
  throw std::runtime_error("giveMassMatrix is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::giveConstraintSparsityPattern(SparcityPattern pattern)
{
  throw std::runtime_error("giveConstraintSparsityPattern is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::giveConstraint(SparseMatrix matrix)
{
  throw std::runtime_error("giveConstraint is not yet implemented");    
}

bool <%lastIdentOfPath(modelInfo.name)%>::isAutonomous()
{
  throw std::runtime_error("isAutonomous is not yet implemented");    
}

bool <%lastIdentOfPath(modelInfo.name)%>::isTimeInvariant()
{
  throw std::runtime_error("isTimeInvariant is not yet implemented");    
}

bool <%lastIdentOfPath(modelInfo.name)%>::isAlgebraic()
{
  return false; // Indexreduction is enabled
}

bool <%lastIdentOfPath(modelInfo.name)%>::isExplicit()
{
  return true; // At the moment only explicit form is supported
}

bool <%lastIdentOfPath(modelInfo.name)%>::hasConstantMass()
{
  throw std::runtime_error("hasConstantMass is not yet implemented");    
}

bool <%lastIdentOfPath(modelInfo.name)%>::hasStateDependentMass()
{
  throw std::runtime_error("hasStateDependentMass is not yet implemented");    
}

bool <%lastIdentOfPath(modelInfo.name)%>::provideSymbolicJacobian()
{
  throw std::runtime_error("provideSymbolicJacobian is not yet implemented");    
}

void <%lastIdentOfPath(modelInfo.name)%>::handleEvent(unsigned long index)
{
 <%handleEvent(sampleConditions,simCode)%>
}
>>
end DefaultImplementationCode;


template AlgloopDefaultImplementationCode(SimCode simCode,SimEqSystem eq)

::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
let modelname = lastIdentOfPath(modelInfo.name)
match eq
case SES_LINEAR(__)
case SES_NONLINEAR(__) then
<< 

/// Provide number (dimension) of variables according to data type
int  <%modelname%>Algloop<%index%>::getDimVars(const IAlgLoop::DATATYPE type) const    
{
    return(AlgLoopDefaultImplementation::getDimVars(type));
};

/// Provide number (dimension) of residuals according to data type
int  <%modelname%>Algloop<%index%>::getDimRHS(const IAlgLoop::DATATYPE type ) const
{
    return(AlgLoopDefaultImplementation::getDimRHS(type));
};

/// Provide number (dimension) of inputs according to data type
int  <%modelname%>Algloop<%index%>::getDimInputs(const IAlgLoop::DATATYPE type )     
{
    return(AlgLoopDefaultImplementation::getDimInputs(type));    
};

/// Provide number (dimension) of outputs according to data type
int  <%modelname%>Algloop<%index%>::getDimOutputs(const IAlgLoop::DATATYPE type )     
{
    return(AlgLoopDefaultImplementation::getDimOutputs(type));    
};

/// Add contraint value to algebraic loop
void  <%modelname%>Algloop<%index%>::addInputs(const double* doubleInputs, const int* intInputs, const bool* boolInputs)
{
    AlgLoopDefaultImplementation::addInputs(doubleInputs, intInputs, boolInputs);
};

/// Add outputs of algebraic loop
void  <%modelname%>Algloop<%index%>::addOutputs(double* doubleOutputs, int* intOutputs, bool* boolOutputs)
{
    AlgLoopDefaultImplementation::addOutputs(doubleOutputs, intOutputs, boolOutputs);;
};



/// Provide variables with given index to the system
void  <%modelname%>Algloop<%index%>::giveVars(double* doubleUnknowns, int* intUnknowns, bool* boolUnknowns)
{    
    AlgLoopDefaultImplementation::giveVars(doubleUnknowns, intUnknowns, boolUnknowns);
    //workaroud until names of algloop vars are replaced in simcode
    <%giveAlgloopvars(eq,simCode)%>
};

/// Set variables with given index to the system
void  <%modelname%>Algloop<%index%>::setVars(const double* doubleUnknowns, const int* intUnknowns, const bool* boolUnknowns)
{
    //workaroud until names of algloop vars are replaced in simcode  
    <%setAlgloopvars(eq,simCode)%>
    AlgLoopDefaultImplementation::setVars(doubleUnknowns, intUnknowns, boolUnknowns);
};




/// Set stream for output
void  <%modelname%>Algloop<%index%>::setOutput(ostream* outputStream) 
{
    AlgLoopDefaultImplementation::setOutput(outputStream);
};
>>
end AlgloopDefaultImplementationCode;


template generateMethodDeclarationCode(SimCode simCode)

::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<< 
     //Releases the Modelica System 
    virtual void destroy();
    //provide number (dimension) of variables according to the index
    virtual int getDimVars(const INDEX index = ALL_VARS) const;
    //Provide number (dimension) of right hand sides (equations and/or residuals) according to the index 
    virtual int getDimRHS(const INDEX index = ALL_VARS)const;
      //Resets all time events
    virtual void resetTimeEvents();
    //Set current integration time 
    virtual void setTime(const double& t);
    //Provide variables with given index to the system 
    virtual void giveVars(double* z, const INDEX index = ALL_VARS);
    //Set variables with given index to the system
    virtual void setVars(const double* z, const INDEX index = ALL_VARS);
    //Update transfer behavior of the system of equations according to command given by solver
    virtual void update(const UPDATE command =IContinous::UNDEF_UPDATE);
    //Provide the right hand side (according to the index)
    virtual void giveRHS(double* f, const INDEX index = ALL_VARS);
    //Output routine (to be called by the solver after every successful integration step)
    virtual void writeOutput(const OUTPUT command = UNDEF_OUTPUT);
    //Provide pattern for Jacobian
    virtual void giveJacobianSparsityPattern(SparcityPattern pattern);
    //Provide Jacobian
    virtual void giveJacobian(SparseMatrix matrix);
    //Provide pattern for mass matrix
    virtual void giveMassSparsityPattern(SparcityPattern pattern); 
    //Provide mass matrix 
    virtual void giveMassMatrix(SparseMatrix matrix);
    //Provide pattern for global constraint jacobian 
    virtual void giveConstraintSparsityPattern(SparcityPattern pattern);
    //Provide global constraint jacobian 
    virtual void giveConstraint(SparseMatrix matrix);
    //Provide number (dimension) of zero functions 
    virtual int getDimZeroFunc();
    //Provides current values of root/zero functions 
     virtual void giveZeroFunc(double* f);
    virtual void giveConditions(bool* c);
    virtual void setConditions(bool* c);
    //Called to check conditions for event-handling
    virtual void checkConditions(unsigned int, bool all);
    virtual void saveConditions();
    //Called to handle all  events occured at same time  
    virtual void handleSystemEvents(const bool* events);
    //Called to handle an event  
    virtual void handleEvent(unsigned long index);
    //Checks if a discrete variable has changed and triggers an event 
    virtual bool checkForDiscreteEvents();
     //Saves all variables before an event is handled, is needed for the pre, edge and change operator
    virtual void saveAll();
    //Returns the vector with all time events 
    virtual event_times_type getTimeEvents();
    // No input
    virtual bool isAutonomous();
    // Time is not present
    virtual bool isTimeInvariant();
    // M is regular 
    virtual bool isODE();
    // M is singular 
    virtual bool isAlgebraic();
    //M = identity 
    virtual bool isExplicit(); 
    // M does not depend on t, z 
    virtual bool hasConstantMass(); 
    // M depends on z 
    virtual bool hasStateDependentMass(); 
    // System is able to provide the Jacobian symbolically 
    virtual bool provideSymbolicJacobian();
    //Systeminitialization methods
    virtual unsigned int getDimInitEquations() ; 
    virtual unsigned int getDimUnfixedStates() ; 
    virtual unsigned int getDimUnfixedParameters(); 
    virtual unsigned int getDimIntialResiduals() ;
    virtual bool initial();
    virtual void setInitial(bool);
    /// (Re-) initialize the system of equations and bounded parameters
    virtual void init(double ts,double te);
    
    <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then 
    <<
    // Returns the history object to query simulation results 
    virtual IHistory* getHistory();
    // Returns labels for a labeled DAE
    virtual label_list_type getLabels();
    //Sets all algebraic and state varibales for current time
    virtual void setVariables(const ublas::vector<double>& variables, const ublas::vector<double>& variables2);

    >>%>
>>
end generateMethodDeclarationCode;

template generateAlgloopMethodDeclarationCode(SimCode simCode)

::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
<< 
    /// Provide number (dimension) of variables according to data type
    virtual int getDimVars(const IAlgLoop::DATATYPE type = IAlgLoop::ALL) const    ;
    /// Provide number (dimension) of residuals according to data type
    virtual int getDimRHS(const IAlgLoop::DATATYPE type = IAlgLoop::ALL) const    ;
    /// Provide number (dimension) of inputs according to data type
    virtual int getDimInputs(const IAlgLoop::DATATYPE type = IAlgLoop::ALL)     ;
    /// Provide number (dimension) of outputs according to data type
    virtual int getDimOutputs(const IAlgLoop::DATATYPE type = IAlgLoop::ALL)    ;
    /// Add contraint value to algebraic loop
    virtual void addInputs(const double* doubleInputs, const int* intInputs, const bool* boolInputs);
    /// Add outputs of algebraic loop
    virtual void addOutputs(double* doubleOutputs, int* intOutputs, bool* boolOutputs)    ;
    /// (Re-) initialize the system of equations
    virtual void init();
    /// Provide variables with given index to the system
    virtual void giveVars(double* doubleUnknowns, int* intUnknowns, bool* boolUnknowns)    ;
    /// Set variables with given index to the system
    virtual void setVars(const double* doubleUnknowns, const int* intUnknowns, const bool* boolUnknowns)    ;
    /// Update transfer behavior of the system of equations according to command given by solver
    virtual void update(const  IContinous::UPDATE command =IContinous::UNDEF_UPDATE);
    /// Provide the right hand side (according to the index)
    virtual void giveRHS(double* doubleResiduals, int* intResiduals, bool* boolResiduals);
    <%if Flags.isSet(Flags.WRITE_TO_BUFFER) then
    <<
    /// Provide dimensions of residuals for linear equation systems
    virtual int giveDimResiduals(int index);    
    /// Provide the residuals for linear equation systems
    virtual void giveResiduals(double* doubleResiduals, int* intResiduals, bool* boolResiduals);
    >>%>
    /// Output routine (to be called by the solver after every successful integration step)
    virtual void giveAMatrix(double* A_matrix);
    virtual bool isLinear();
    /// Set stream for output
    virtual void setOutput(ostream* outputStream)     ;
    
>>
//void writeOutput(HistoryImplType::value_type_v& v ,vector<string>& head ,const IDAESystem::OUTPUT command  = IDAESystem::UNDEF_OUTPUT);
end generateAlgloopMethodDeclarationCode;

template MemberVariable(ModelInfo modelInfo)
 "Define membervariable in simulation file."
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  <<
  <%vars.algVars |> var =>
    MemberVariableDefine2(var, "algebraics")
  ;separator="\n"%>
  <%vars.paramVars |> var =>
    MemberVariableDefine2(var, "parameters")
  ;separator="\n"%>
   <%vars.aliasVars |> var =>
    MemberVariableDefine2(var, "aliasVars")
  ;separator="\n"%>
  <%vars.intAlgVars |> var =>
    MemberVariableDefine("int", var, "intVariables.algebraics")
  ;separator="\n"%>
  <%vars.intParamVars |> var =>
    MemberVariableDefine("int", var, "intVariables.parameters")
  ;separator="\n"%>
   <%vars.intAliasVars |> var =>
    MemberVariableDefine("int", var, "intVariables.AliasVars")
  ;separator="\n"%>
  <%vars.boolAlgVars |> var =>
    MemberVariableDefine("bool",var, "boolVariables.algebraics")
  ;separator="\n"%>
  <%vars.boolParamVars |> var =>
    MemberVariableDefine("bool",var, "boolVariables.parameters")
  ;separator="\n"%>
   <%vars.boolAliasVars |> var =>
    MemberVariableDefine("bool ",var, "boolVariables.AliasVars")
  ;separator="\n"%>   
  <%vars.stringAlgVars |> var =>
    MemberVariableDefine("string",var, "stringVariables.algebraics")
  ;separator="\n"%>
  <%vars.stringParamVars |> var =>
    MemberVariableDefine("string",var, "stringVariables.parameters")
  ;separator="\n"%>
  <%vars.stringAliasVars |> var =>
    MemberVariableDefine("string",var, "stringVariables.AliasVars")
  ;separator="\n"%>
   <%vars.constVars |> var =>
    MemberVariableDefine2(var, "constvariables")
  ;separator="\n"%>
   <%vars.intConstVars |> var =>
    MemberVariableDefine("const int", var, "intConstvariables")
  ;separator="\n"%>
   <%vars.boolConstVars |> var =>
    MemberVariableDefine("const bool", var, "boolConstvariables")
  ;separator="\n"%>
   <%vars.stringConstVars |> var =>
    MemberVariableDefine("const string",var, "stringConstvariables")
  ;separator="\n"%>
  >>
end MemberVariable;

template MemberVariableAlgloop(ModelInfo modelInfo)
 "Define membervariable in simulation file."
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  
  <<
  <%vars.algVars |> var =>
    MemberVariableDefineReference2(var, "algebraics","")
  ;separator=";\n"%>
  <%if vars.algVars then ";" else " "%>  
  <%vars.paramVars |> var =>
    MemberVariableDefineReference2(var, "parameters","")
  ;separator=";\n"%>
  <%if vars.paramVars then ";" else " "%>
 
   <%vars.aliasVars |> var =>
    MemberVariableDefineReference2(var, "aliasVars","")
  ;separator=";\n"%>
  <%if vars.aliasVars then ";" else " "%>

  <%vars.intAlgVars |> var =>
    MemberVariableDefineReference("int", var, "intVariables.algebraics","")
  ;separator=";\n"%>
  <%if vars.intAlgVars then ";" else " "%>
  
  <%vars.intParamVars |> var =>
    MemberVariableDefineReference("int", var, "intVariables.parameters","")
  ;separator=";\n"%>
  <%if vars.intParamVars then ";" else " "%>
   
   
   <%vars.intAliasVars |> var =>
   MemberVariableDefineReference("int", var, "intVariables.AliasVars","")
  ;separator=";\n"%>
  <%if vars.intAliasVars then ";" else " "%>
  
  <%vars.boolAlgVars |> var =>
    MemberVariableDefineReference("bool",var, "boolVariables.algebraics","")
  ;separator=";\n"%>
  <%if vars.boolAlgVars then ";" else " "%>
  
  <%vars.boolParamVars |> var =>
    MemberVariableDefineReference("bool",var, "boolVariables.parameters","")
  ;separator=";\n"%>
  <%if vars.boolParamVars then ";" else " "%>
   
   <%vars.boolAliasVars |> var =>
     MemberVariableDefineReference("bool ",var, "boolVariables.AliasVars","")
  ;separator=";\n"%>
  <%if vars.boolAliasVars then ";" else " "%>
  
  
     
  <%vars.stringAlgVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.algebraics","")
  ;separator=";\n"%>
  <%if vars.stringAlgVars then ";" else " "%>
  
  <%vars.stringParamVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.parameters","")
  ;separator=";\n"%>
  <%if vars.stringParamVars then ";" else " "%>
  
  <%vars.stringAliasVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.AliasVars","")
  ;separator=";\n"%>
  <%if vars.stringAliasVars then ";" else " "%>
  
  
   <%vars.constVars |> var =>
    MemberVariableDefineReference2(var, "constvariables","")
  ;separator=";\n"%>
  <%if vars.constVars then ";" else " "%>
   
   <%vars.intConstVars |> var =>
    MemberVariableDefineReference("const int", var, "intConstvariables","")
  ;separator=";\n"%>
  <%if vars.intConstVars then ";" else " "%>
   
   
   <%vars.boolConstVars |> var =>
    MemberVariableDefineReference("const bool", var, "boolConstvariables","")
  ;separator=";\n"%>
  <%if vars.boolConstVars then ";" else " "%>
   
   <%vars.stringConstVars |> var =>
    MemberVariableDefineReference("const string",var, "stringConstvariables","")
  ;separator=";\n"%>
  <%if vars.stringConstVars then ";" else " "%>
  
  >>
end MemberVariableAlgloop;



template ConstructorParamAlgloop(ModelInfo modelInfo)
 "Define membervariable in simulation file."
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  
    
  <<

  <%vars.algVars |> var =>
    MemberVariableDefineReference2(var, "algebraics","_")
  ;separator=", "%>
 
   <%if vars.paramVars then "," else ""%> 
  <%vars.paramVars |> var =>
    MemberVariableDefineReference2(var, "parameters","_")
  ;separator=", "%>
  
  <%if vars.aliasVars then "," else ""%> 
   <%vars.aliasVars |> var =>
    MemberVariableDefineReference2(var, "aliasVars","_")
  ;separator=", "%>
 
  <%if vars.intAlgVars then "," else ""%> 
  <%vars.intAlgVars |> var =>
    MemberVariableDefineReference("int", var, "intVariables.algebraics","_")
  ;separator=", "%>
  
  <%if vars.intParamVars then "," else ""%>
  <%vars.intParamVars |> var =>
    MemberVariableDefineReference("int", var, "intVariables.parameters","_")
  ;separator=", "%>
  
  <%if vars.intAliasVars then "," else ""%>
   <%vars.intAliasVars |> var =>
    MemberVariableDefineReference("int", var, "intVariables.AliasVars","_")
  ;separator=", "%>
  
  <%if vars.boolAlgVars then "," else ""%>
  <%vars.boolAlgVars |> var =>
    MemberVariableDefineReference("bool",var, "boolVariables.algebraics","_")
  ;separator=", "%>
  
  <%if vars.boolParamVars then "," else ""%>
  <%vars.boolParamVars |> var =>
    MemberVariableDefineReference("bool",var, "boolVariables.parameters","_")
  ;separator=", "%>
  
  <%if vars.boolAliasVars then "," else ""%>
   <%vars.boolAliasVars |> var =>
    MemberVariableDefineReference("bool ",var, "boolVariables.AliasVars","_")
  ;separator=", "%>
  
  <%if vars.stringAlgVars then "," else "" %>   
  <%vars.stringAlgVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.algebraics","_")
  ;separator=", "%>
  
  
  <%if vars.stringParamVars then "," else ""%>  
  <%vars.stringParamVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.parameters","_")
  ;separator=", "%>
  
   <%if vars.stringAliasVars then "," else ""%>
  <%vars.stringAliasVars |> var =>
    MemberVariableDefineReference("string",var, "stringVariables.AliasVars","_")
  ;separator=", "%>
 
  <%if vars.constVars then "," else ""%>
   <%vars.constVars |> var =>
    MemberVariableDefineReference2(var, "constvariables","_")
  ;separator=", "%>
  
  <%if vars.intConstVars then "," else ""%>
   <%vars.intConstVars |> var =>
    MemberVariableDefineReference("const int", var, "intConstvariables","_")
  ;separator=", "%>
  
  <%if vars.boolConstVars then "," else "" %>
   <%vars.boolConstVars |> var =>
    MemberVariableDefineReference("const bool", var, "boolConstvariables","_")
  ;separator=", "%>
  
  <%if vars.stringConstVars then "," else ""%>
   <%vars.stringConstVars |> var =>
    MemberVariableDefineReference("const string",var, "stringConstvariables","_")
  ;separator=", "%>
  >>
end ConstructorParamAlgloop;

template CallAlgloopParams(ModelInfo modelInfo)
 "Define membervariable in simulation file."
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  
    
  <<

  <%vars.algVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
 
   <%if vars.paramVars then "," else ""%> 
  <%vars.paramVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.aliasVars then "," else ""%> 
   <%vars.aliasVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
 
  <%if vars.intAlgVars then "," else ""%> 
  <%vars.intAlgVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.intParamVars then "," else ""%>
  <%vars.intParamVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.intAliasVars then "," else ""%>
   <%vars.intAliasVars |> var =>
    CallAlgloopParam( var)
  ;separator=", "%>
  
  <%if vars.boolAlgVars then "," else ""%>
  <%vars.boolAlgVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.boolParamVars then "," else ""%>
  <%vars.boolParamVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.boolAliasVars then "," else ""%>
   <%vars.boolAliasVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.stringAlgVars then "," else "" %>   
  <%vars.stringAlgVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  
  <%if vars.stringParamVars then "," else ""%>  
  <%vars.stringParamVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
   <%if vars.stringAliasVars then "," else ""%>
  <%vars.stringAliasVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
 
  <%if vars.constVars then "," else ""%>
   <%vars.constVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.intConstVars then "," else ""%>
   <%vars.intConstVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.boolConstVars then "," else "" %>
   <%vars.boolConstVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  
  <%if vars.stringConstVars then "," else ""%>
   <%vars.stringConstVars |> var =>
    CallAlgloopParam(var)
  ;separator=", "%>
  >>
end CallAlgloopParams;



template InitAlgloopParams(ModelInfo modelInfo,Text& arrayInit)
 "Define membervariable in simulation file."
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  
    
  <<
  <%if vars.algVars then "," else ""%>
  <%vars.algVars |> var =>
    InitAlgloopParam(var, "algebraics",arrayInit)
  ;separator="\n,"%>

  <%if vars.paramVars then "," else ""%>
  <%vars.paramVars |> var =>
    InitAlgloopParam(var, "parameters",arrayInit)
  ;separator="\n,"%>

  <%if vars.aliasVars then "," else ""%>
   <%vars.aliasVars |> var =>
    InitAlgloopParam(var, "aliasVars",arrayInit)
  ;separator="\n,"%>
  
  <%if vars.intAlgVars then "," else ""%>
  <%vars.intAlgVars |> var =>
    InitAlgloopParam( var, "intVariables.algebraics",arrayInit)
  ;separator="\n,"%>

  <%if vars.intParamVars then "," else ""%>
  <%vars.intParamVars |> var =>
    InitAlgloopParam( var, "intVariables.parameters",arrayInit)
  ;separator="\n,"%>

  <%if vars.intAliasVars then "," else ""%>    
   <%vars.intAliasVars |> var =>
    InitAlgloopParam( var, "intVariables.AliasVars",arrayInit)
  ;separator="\n,"%>

  <%if vars.boolAlgVars then "," else ""%>
  <%vars.boolAlgVars |> var =>
    InitAlgloopParam(var, "boolVariables.algebraics",arrayInit)
  ;separator="\n,"%>

  <%if vars.boolParamVars then "," else ""%>
  <%vars.boolParamVars |> var =>
    InitAlgloopParam(var, "boolVariables.parameters",arrayInit)
  ;separator="\n,"%>

   <%if vars.boolAliasVars then "," else ""%>
   <%vars.boolAliasVars |> var =>
    InitAlgloopParam(var, "boolVariables.AliasVars",arrayInit)
  ;separator="\n,"%>
  
  <%if vars.stringAlgVars then "," else ""%>
  <%vars.stringAlgVars |> var =>
    InitAlgloopParam(var, "stringVariables.algebraics",arrayInit)
  ;separator="\n,"%>

  <%if vars.stringParamVars then "," else "" %>
  <%vars.stringParamVars |> var =>
    InitAlgloopParam(var, "stringVariables.parameters",arrayInit)
  ;separator="\n,"%>
 
  <%if vars.stringAliasVars then "," else "" %>
  <%vars.stringAliasVars |> var =>
    InitAlgloopParam(var, "stringVariables.AliasVars",arrayInit)
  ;separator="\n,"%>

  <%if vars.constVars then "," else ""%>
   <%vars.constVars |> var =>
    InitAlgloopParam(var, "constvariables",arrayInit)
  ;separator="\n,"%>

  <%if vars.intConstVars then "," else ""%>
   <%vars.intConstVars |> var =>
    InitAlgloopParam( var, "intConstvariables",arrayInit)
  ;separator="\n,"%>

   <%if vars.boolConstVars  then "," else ""%>
   <%vars.boolConstVars |> var =>
    InitAlgloopParam( var, "boolConstvariables",arrayInit)
  ;separator="\n,"%>

  <%if vars.stringConstVars then "," else ""%>
   <%vars.stringConstVars |> var =>
    InitAlgloopParam(var, "stringConstvariables",arrayInit)
  ;separator="\n,"%>
  >>
end InitAlgloopParams;

template MemberVariableDefine(String type,SimVar simVar, String arrayName)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%type%> <%cref(name)%>;
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(numArrayElement)%>> <%arraycref(name)%>;
      >>
    case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(numArrayElement)%>> <%arraycref(name)%>;
      >>   
end MemberVariableDefine;

template MemberVariableDefineReference(String type,SimVar simVar, String arrayName,String pre)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%type%>& <%pre%><%cref(name)%>
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>>& <%pre%><%arraycref(name)%>
      >>
     case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>>& <%pre%><%arraycref(name)%>
      >>
end MemberVariableDefineReference;


template MemberVariableDefine2(SimVar simVar, String arrayName)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%variableType(type_)%> <%cref(name)%>;
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>> <%arraycref(name)%>;
      >>
    case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>> <%arraycref(name)%>;
      >>
end MemberVariableDefine2;


template InitAlgloopParam(SimVar simVar, String arrayName,Text& arrayInit)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%cref(name)%>(_<%cref(name)%>)
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      //let &arrayInit+= ',<%arraycref(name)%>=_<%arraycref(name)%>'
      '<%arraycref(name)%>(_<%arraycref(name)%>)'
    case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      //let &arrayInit+= ' ,<%arraycref(name)%>= _<%arraycref(name)%>'
      '<%arraycref(name)%>( _<%arraycref(name)%>)'  
end InitAlgloopParam;

template CallAlgloopParam(SimVar simVar)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%cref(name)%>
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      //let &arrayInit+= ',<%arraycref(name)%>=_<%arraycref(name)%>'
      '<%arraycref(name)%>'
    case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      //let &arrayInit+= ' ,<%arraycref(name)%>= _<%arraycref(name)%>'
      '<%arraycref(name)%>'   
end CallAlgloopParam;

template MemberVariableDefineReference2(SimVar simVar, String arrayName,String pre)
::=
match simVar
      case SIMVAR(numArrayElement={}) then
      <<
      <%variableType(type_)%>& <%pre%><%cref(name)%>
      >>
    case v as SIMVAR(name=CREF_IDENT(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>>& <%pre%><%arraycref(name)%>
      >>
    case v as SIMVAR(name=CREF_QUAL(__),arrayCref=SOME(_),numArrayElement=num) then
      <<
      multi_array<<%variableType(type_)%>,<%listLength(num)%>>& <%pre%><%arraycref(name)%>
      >>  
end MemberVariableDefineReference2;


template arrayConstruct(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(vars = vars as SIMVARS(__)) 
  then
  <<
  <%arrayConstruct1(vars.algVars)%>
  <%arrayConstruct1(vars.intAlgVars)%>
  <%arrayConstruct1(vars.boolAlgVars)%>
  <%arrayConstruct1(vars.stringAlgVars)%>
  <%arrayConstruct1(vars.paramVars)%>
  <%arrayConstruct1(vars.intParamVars)%>
  <%arrayConstruct1(vars.boolParamVars)%>
  <%arrayConstruct1(vars.stringParamVars)%>
  <%arrayConstruct1(vars.aliasVars)%>
  <%arrayConstruct1(vars.intAliasVars)%>
  <%arrayConstruct1(vars.boolAliasVars)%>
  <%arrayConstruct1(vars.stringAliasVars)%>
  <%arrayConstruct1(vars.constVars)%>
  <%arrayConstruct1(vars.intConstVars)%>
  <%arrayConstruct1(vars.boolConstVars)%>
  <%arrayConstruct1(vars.stringConstVars)%>
  >>
end arrayConstruct;

template arrayConstruct1(list<SimVar> varsLst) ::=
  varsLst |> v as SIMVAR(arrayCref=SOME(_)) =>
  <<
  ,<%arraycref(name)%>(boost::extents[<%v.numArrayElement;separator="]["%>])
  >> 
  ;separator="\n"
end arrayConstruct1;

template variableType(DAE.Type type)
 "Generates integer for use in arrays in global data section."
::=
  match type
  case T_REAL(__)        then "double"
  case T_STRING(__)      then "string"
  case T_INTEGER(__)         then "int"
  case T_BOOL(__)        then "bool"
end variableType;

template lastIdentOfPath(Path modelName) ::=
  match modelName
  case QUALIFIED(__) then lastIdentOfPath(path)
  case IDENT(__)     then name
  case FULLYQUALIFIED(__) then lastIdentOfPath(path)
end lastIdentOfPath;

template cref(ComponentRef cr)
 "Generates C equivalent name for component reference."
::=
  match cr
  case CREF_IDENT(ident = "time") then "time"
  case WILD(__) then ''
  else "_"+crefToCStr(cr)
end cref;

template cref2(ComponentRef cr)
 "Generates C equivalent name for component reference."
::=
  match cr
  case CREF_IDENT(ident = "xloc") then '<%crefStr(cr)%>'
  case CREF_IDENT(ident = "time") then "time"
  case WILD(__) then ''
  else "_"+crefToCStr(cr)
end cref2;

template crefToCStr(ComponentRef cr)
 "Helper function to cref."
::=
  match cr
  case CREF_IDENT(__) then '<%ident%><%subscriptsToCStr(subscriptLst)%>'
  case CREF_QUAL(__) then '<%ident%><%subscriptsToCStr(subscriptLst)%>_P_<%crefToCStr(componentRef)%>'
  case WILD(__) then ''
  else "CREF_NOT_IDENT_OR_QUAL"
end crefToCStr;

template subscriptsToCStr(list<Subscript> subscripts)
::=
  if subscripts then
    '[<%subscripts |> s => subscriptToCStr(s) ;separator="]["%>]'
end subscriptsToCStr;

template subscriptToCStr(Subscript subscript)
::=
  match subscript
  case INDEX(exp=ICONST(integer=i)) then i
  case SLICE(exp=ICONST(integer=i)) then i
  case WHOLEDIM(__) then "WHOLEDIM"
  else "UNKNOWN_SUBSCRIPT"
end subscriptToCStr;

template arraycref(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(ident = "xloc") then crefStr(cr)
  case CREF_IDENT(ident = "time") then "time"
  case WILD(__) then ''
  else "_"+crefToCStr1(cr)
end arraycref;

template crefToCStr1(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(__) then '<%ident%>'
  case CREF_QUAL(__) then               '<%ident%><%subscriptsToCStr(subscriptLst)%>_P_<%crefToCStr1(componentRef)%>'
  case WILD(__) then ''
  else "CREF_NOT_IDENT_OR_QUAL"
end crefToCStr1;

template crefStr(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(ident = "xloc") then '_xd<%subscriptsStr(subscriptLst)%>'
  case CREF_IDENT(__) then '<%ident%><%subscriptsStr(subscriptLst)%>'
  // Are these even needed? Function context should only have CREF_IDENT :)
  case CREF_QUAL(ident = "$DER") then 'der(<%crefStr(componentRef)%>)'
  case CREF_QUAL(__) then '<%ident%><%subscriptsStr(subscriptLst)%>.<%crefStr(componentRef)%>'
  else "CREF_NOT_IDENT_OR_QUAL"
end crefStr;

template subscriptsStr(list<Subscript> subscripts)
 "Generares subscript part of the name."
::=
  if subscripts then
    '[<%subscripts |> s => subscriptStr(s) ;separator=","%>]'
end subscriptsStr;

template subscriptStr(Subscript subscript)
 "Generates a single subscript.
  Only works for constant integer indicies."
::=
  match subscript
  case INDEX(exp=ICONST(integer=i)) then i
  case SLICE(exp=ICONST(integer=i)) then i
  case WHOLEDIM(__) then "WHOLEDIM"
  else "UNKNOWN_SUBSCRIPT"
end subscriptStr;

template simulationInitFile(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(varInfo = vi as VARINFO(__), vars = vars as SIMVARS(__))) 
  then
  <<
  <%arrayConstruct(modelInfo)%>
  <%initVals(vars.constVars,simCode)%>
  <%initVals(vars.intConstVars,simCode)%>
  <%initVals(vars.boolConstVars,simCode)%>
  <%initVals(vars.stringConstVars,simCode)%>
  <%initVals(vars.paramVars,simCode)%>
  <%initVals(vars.intParamVars,simCode)%>
  <%initVals(vars.boolParamVars,simCode)%>
  <%initVals(vars.stringParamVars,simCode)%>
  >>
end simulationInitFile;

template initVals(list<SimVar> varsLst,SimCode simCode) ::=
  varsLst |> SIMVAR(numArrayElement={}) =>
  <<
  ,<%cref(name)%>(<%match initialValue 
  case SOME(v) then initVal(v)
  else "0"
  %>) 
  >>  
  ;separator="\n"
end initVals;

template arrayInit(SimCode simCode)
 "Generates the contents of the makefile for the simulation case."
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(varInfo = vi as VARINFO(__), vars = vars as SIMVARS(__))) 
  then
  <<
  <%initVals1(vars.paramVars,simCode)%>
  <%initVals1(vars.intParamVars,simCode)%>
  <%initVals1(vars.boolParamVars,simCode)%>
  <%initVals1(vars.stringParamVars,simCode)%>
  <%initVals1(vars.constVars,simCode)%>
  <%initVals1(vars.intConstVars,simCode)%>
  <%initVals1(vars.boolConstVars,simCode)%>
  <%initVals1(vars.stringConstVars,simCode)%>
  >>
end arrayInit;

template initVals1(list<SimVar> varsLst, SimCode simCode) ::=
  varsLst |> (var as SIMVAR(__)) => 
  initVals2(var,simCode)
  ;separator="\n"
end initVals1;

template initVals2(SimVar var, SimCode simCode) ::=
  match var
  case SIMVAR(numArrayElement = {}) then ''
  case SIMVAR(__) then '<%cref(name)%>=<%match initialValue 
    case SOME(v) then initVal(v)
      else "0"
    %>;'
end initVals2;


template arrayReindex(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(vars = vars as SIMVARS(__)) 
  then
  <<
  <%arrayReindex1(vars.algVars)%>
  <%arrayReindex1(vars.intAlgVars)%>
  <%arrayReindex1(vars.boolAlgVars)%>
  <%arrayReindex1(vars.stringAlgVars)%>
  <%arrayReindex1(vars.paramVars)%>
  <%arrayReindex1(vars.intParamVars)%>
  <%arrayReindex1(vars.boolParamVars)%>
  <%arrayReindex1(vars.stringParamVars)%>
  <%arrayReindex1(vars.aliasVars)%>
  <%arrayReindex1(vars.intAliasVars)%>
  <%arrayReindex1(vars.boolAliasVars)%>
  <%arrayReindex1(vars.stringAliasVars)%>
  <%arrayReindex1(vars.constVars)%>
  <%arrayReindex1(vars.intConstVars)%>
  <%arrayReindex1(vars.boolConstVars)%>
  <%arrayReindex1(vars.stringConstVars)%>
  >>
end arrayReindex;

template arrayReindex1(list<SimVar> varsLst) ::=
  varsLst |> SIMVAR(arrayCref=SOME(_)) =>
  <<
  <%arraycref(name)%>.reindex(1);
  >> 
  ;separator="\n"
end arrayReindex1;


template initVal(Exp initialValue) 
::=
  match initialValue 
  case ICONST(__) then integer
  case RCONST(__) then real
  case SCONST(__) then '"<%Util.escapeModelicaStringToCString(string)%>"'
  case BCONST(__) then if bool then "true" else "false"
  case ENUM_LITERAL(__) then '<%index%>/*ENUM:<%dotPath(name)%>*/'
  else "*ERROR* initial value of unknown type"
end initVal;

template dotPath(Path path)
 "Generates paths with components separated by dots."
::=
  match path
  case QUALIFIED(__)      then '<%name%>.<%dotPath(path)%>'

  case IDENT(__)          then name
  case FULLYQUALIFIED(__) then dotPath(path)
end dotPath;

template writeoutput1(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
  << 
      <%{(vars.algVars |> SIMVAR(__) =>
        ' "<%crefStr(name)%>" '
      ;separator=","),
      (vars.intAlgVars |> SIMVAR(__) =>
        ' "<%crefStr(name)%>" '
      ;separator=","),
      (vars.boolAlgVars |> SIMVAR(__) =>
        ' "<%crefStr(name)%>" '
      ;separator=","),
      (vars.stateVars |> SIMVAR(__) =>
        ' "<%crefStr(name)%>" '
      ;separator=","),
      (vars.derivativeVars |> SIMVAR(__) =>
        ' "<%crefStr(name)%>" '
      ;separator=",")}
    ;separator=","%>;
  >>  
end writeoutput1;

template numResidues(list<SimEqSystem> allEquations)
::=
(allEquations |> eqn => numResidues2(eqn);separator="+")
end numResidues;

template numResidues2(SimEqSystem eqn)
::=
match eqn
case SES_RESIDUAL(__) then
<<
>>
case  SES_SIMPLE_ASSIGN(__) then
<<
1
>>
case SES_ARRAY_CALL_ASSIGN(__) then
<<
>>
case SES_ALGORITHM(__) then
<<
>>
case lin as SES_LINEAR(__) then
<<
<%(vars |> var => '1');separator="+"%>
>>
case SES_NONLINEAR(__) then
<<
<%(eqs |> eq => '1');separator="+"%>
>>
case SES_MIXED(__) then numResidues2(cont)
case SES_WHEN(__) then
<<
>>
else
<<
>>
end numResidues2;

template numStatevars(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.numStateVars%>
>>
end numStatevars;

template numAlgvars(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.numAlgVars%>+<%varInfo.numIntAlgVars%>+<%varInfo.numBoolAlgVars%>
>>
end numAlgvars;

template numAlgvar(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.numAlgVars%>
>>
end numAlgvar;

template numIntAlgvar(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.numIntAlgVars%>
>>
end numIntAlgvar;

template numBoolAlgvar(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.numBoolAlgVars%>
>>
end numBoolAlgvar;

template numDerivativevars(ModelInfo modelInfo)
::=
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
<<
<%varInfo.dimODE1stOrder%>+<%varInfo.dimODE2ndOrder%>
>>
end numDerivativevars;


template writeoutput2(ModelInfo modelInfo)

::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then


 <<  
    
     <%vars.algVars |> SIMVAR(__) hasindex i0 =>'v(<%i0%>)=<%cref(name)%>;'%>
     <%vars.intAlgVars |> SIMVAR(__) hasindex i1 =>'v(<%i1%>+<%numAlgvar(modelInfo)%>)=<%cref(name)%>;'%>
     <%vars.boolAlgVars |> SIMVAR(__)hasindex i2  =>'v(<%i2%>+ <%numAlgvar(modelInfo)%> +<%numIntAlgvar(modelInfo)%>)=<%cref(name)%>;'%>
     <%(vars.stateVars  |> SIMVAR(__) hasindex i3 =>' v(<%i3%> +<%numAlgvar(modelInfo)%> +<%numIntAlgvar(modelInfo)%> +<%numBoolAlgvar(modelInfo)%>)=_z[<%index%>]; ')%>
     <%(vars.derivativeVars  |> SIMVAR(__) hasindex i4 =>' v2(<%i4%>)=_zDot[<%index%>]; ')%>
   
 >>
end writeoutput2;


template saveall(ModelInfo modelInfo, SimCode simCode)
 
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(vars = vars as SIMVARS(__))) 
  then
  <<  
    void <%lastIdentOfPath(modelInfo.name)%>::saveAll()
    { 
      <%{(vars.algVars |> SIMVAR(__) =>
        '_event_handling.save(<%cref(name)%>,"<%cref(name)%>");'
      ;separator="\n"),
      (vars.intAlgVars |> SIMVAR(__) =>
       '_event_handling.save(<%cref(name)%>,"<%cref(name)%>");'
      ;separator="\n"),
      (vars.boolAlgVars |> SIMVAR(__) =>
        '_event_handling.save(<%cref(name)%>,"<%cref(name)%>");'
      ;separator="\n"),
      (vars.stateVars |> SIMVAR(__) =>
        '_event_handling.save(_z[<%index%>],"<%cref(name)%>");'
      ;separator="\n")}
     ;separator="\n"%>
     
     _event_handling.saveH();
    
    }
  >>  
  /*
  //save all zero crossing condtions
   <%saveconditionvar(zeroCrossings,simCode)%>
   */  
end saveall;

template initvar(ModelInfo modelInfo,SimCode simCode)
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
 <<  
  <%initValst(vars.stateVars, simCode)%> 
  <%initValst(vars.derivativeVars, simCode)%>
  <%initValst(vars.algVars, simCode)%>
  <%initValst(vars.intAlgVars, simCode)%>
  <%initValst(vars.boolAlgVars, simCode)%>
  <%initValst(vars.aliasVars, simCode)%>
  <%initValst(vars.intAliasVars, simCode)%>
  <%initValst(vars.boolAliasVars, simCode)%>
 >>
end initvar;

template initAlgloopvars(ModelInfo modelInfo,SimCode simCode)
::=
match modelInfo
case MODELINFO(vars=SIMVARS(__)) then
 <<  
  <%initValst(vars.algVars, simCode)%>
  <%initValst(vars.intAlgVars, simCode)%>
  <%initValst(vars.boolAlgVars, simCode)%>
  >>
end initAlgloopvars;

template boundParameters(list<SimEqSystem> parameterEquations,Text &varDecls,SimCode simCode)
 "Generates function in simulation file."
::=
  
  let &tmp = buffer ""
  let body = (parameterEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(eq, contextOther, &varDecls /*BUFD*/, simCode)
    ;separator="\n")
  let divbody = (parameterEquations |> eq as SES_ALGORITHM(__) =>
      equation_(eq, contextOther, &varDecls /*BUFD*/, simCode)
    ;separator="\n")    
  <<
    
    <%body%>
    <%divbody%>
   >>
end boundParameters;

template outputIndices(ModelInfo modelInfo)
::= match modelInfo
case MODELINFO(varInfo=VARINFO(__),vars=SIMVARS(__)) then
    if varInfo.numOutVars then
    <<
    var_ouputs_idx+=<%
    {(vars.algVars |> SIMVAR(__) => if isOutput(causality) then '<%index%>';separator=","),
    (vars.intAlgVars |> SIMVAR(__) => if isOutput(causality) then '<%numAlgvar(modelInfo)%>+<%index%>';separator=","),
    (vars.boolAlgVars |> SIMVAR(__) => if isOutput(causality) then '<%numAlgvar(modelInfo)%>+<%numIntAlgvar(modelInfo)%>+<%index%>';separator=","),
    (vars.stateVars  |> SIMVAR(__) => if isOutput(causality) then '<%numAlgvars(modelInfo)%>+<%index%>';separator=","),
    (vars.derivativeVars  |> SIMVAR(__) => if isOutput(causality) then '<%numAlgvars(modelInfo)%>+<%numStatevars(modelInfo)%>+<%index%>';separator=",")};separator=","%>;
    >>
end outputIndices;

template isOutput(Causality c)
 "Returns the Causality Attribute of a Variable."
::=
match c
  case OUTPUT(__) then "output"
end isOutput;

template initValst(list<SimVar> varsLst, SimCode simCode) ::=
  varsLst |> sv as SIMVAR(__) =>
    match initialValue 
      case SOME(v) then 
      let &preExp = buffer "" //dummy ... the value is always a constant
      let &varDecls = buffer ""
      match daeExp(v, contextOther, &preExp, &varDecls,simCode)
      case vStr as "0"
      case vStr as "0.0"
      case vStr as "(0)" then
       '<%cref1(sv.name,simCode)%>=<%vStr%>;//<%cref(sv.name)%>'  
      case vStr as "" then
       '<%cref1(sv.name,simCode)%>=0;//<%cref(sv.name)%>' 
      case vStr then
       '<%cref1(sv.name,simCode)%>=<%vStr%>;//<%cref(sv.name)%>' 
        end match
      else '<%cref1(sv.name,simCode)%>=0;'
  ;separator="\n"
end initValst;

//template initValst1(list<SimVar> varsLst, SimCode simCode) ::=
  //varsLst |> sv as SIMVAR(__) =>
    // match aliasvar 
      //case ALIAS(__) then 
       //'<%cref1(sv.name,simCode)%>=$<%crefStr(varName)%>;'
       //else
      //'<%initValst(varsLst,simCode)%>'
//end initValst1;

template eventHandlingInit(SimCode simCode)

::=
match simCode
case SIMCODE(__) then
  << 
      for(int i=0;i<=<%helpvarlength(simCode)%>;++i) { handleEvent(i); }
  >>
end eventHandlingInit;
  

template dimension1(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(varInfo = vi as VARINFO(__)))
then
    <<
    <%modelInfo.varInfo |> VARINFO(__) =>'<%match dimODE1stOrder 
    case SOME(v) then 
               <<
               _dimODE1stOrder = <%vi.dimODE1stOrder%>;
               >>
      else ""%> ';separator=";\n"%>
    <%modelInfo.varInfo |> VARINFO(__) =>'<%match dimODE2ndOrder 
    case SOME(v) then 
                <<
                _dimODE2ndOrder = <%vi.dimODE2ndOrder%>;
                >> 
      else ""%> ';separator=";\n"%>  
    >>
end dimension1;

template isODE(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(varInfo = vi as VARINFO(__)))
then
<< 
   bool <%lastIdentOfPath(modelInfo.name)%>::isODE()
 {
  return <%vi.numStateVars%>>0 ;
 }
>>
end isODE;


template contextArrayCref(ComponentRef cr, Context context)
 "Generates code for an array component reference depending on the context."
::=
  match context
  case FUNCTION_CONTEXT(__) then "_" + arrayCrefStr(cr)
  else arrayCrefCStr(cr)
end contextArrayCref;

template arrayCrefStr(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(__) then '<%ident%>'
  case CREF_QUAL(__) then '<%ident%>.<%arrayCrefStr(componentRef)%>'
  else "CREF_NOT_IDENT_OR_QUAL"
end arrayCrefStr;

template expTypeFlag(DAE.Type ty, Integer flag)

::=
  match flag
  case 1 then
    // we want the short typesmuwww.
    expTypeShort(ty)
  case 2 then
    // we want the "modelica type"
    match ty case T_COMPLEX(complexClassType=EXTERNAL_OBJ(__)) then
      '<%expTypeShort(ty)%>'
    else match ty case T_COMPLEX(__) then
      'struct <%underscorePath(ClassInf.getStateName(complexClassType))%>'
    else
      '<%expTypeShort(ty)%>'
  case 3 then
    // we want the "array type"
    '<%expTypeShort(ty)%>'  
  case 4 then
    match ty
    case T_ARRAY(__) then '<%expTypeShort(ty)%>'
    else expTypeFlag(ty, 2)
    end match
  case 5 then
    match ty
    case T_ARRAY(dims=dims) then 'multi_array_ref<<%expTypeShort(ty)%>,<%listLength(dims)%>>'
    else expTypeFlag(ty, 2)
    end match
  case 6 then
    match ty
    case T_ARRAY(dims=dims) then 'multi_array<<%expTypeShort(ty)%>,<%listLength(dims)%>>'
    else expTypeFlag(ty, 2)
    end match
  case 7 then
     match ty
    case T_ARRAY(dims=dims)
    then
     'multi_array<<%expTypeShort(ty)%>,<%listLength(dims)%>>'
    end match
 
end expTypeFlag;

template expTypeArray(DAE.Type ty)

::=
  expTypeFlag(ty, 3)
end expTypeArray;

template expTypeArrayforDim(DAE.Type ty)

::=
  expTypeFlag(ty, 6)
end expTypeArrayforDim;

template expTypeShort(DAE.Type type)

::=
  match type
  case T_INTEGER(__)         then "int"  
  case T_REAL(__)        then "double"
  case T_STRING(__)      then if acceptMetaModelicaGrammar() then "metatype" else "string"
  case T_BOOL(__)        then "bool"
  case T_ENUMERATION(__) then "int"  
  case T_UNKNOWN(__)     then "complex"
  case T_ANYTYPE(__)     then "complex"
  case T_ARRAY(__)       then expTypeShort(ty)   
  case T_COMPLEX(complexClassType=EXTERNAL_OBJ(__))
                      then "complex"
  case T_COMPLEX(__)     then 'struct <%underscorePath(ClassInf.getStateName(complexClassType))%>'  
  case T_METATYPE(__) case T_METABOXED(__)    then "metatype"
  case T_FUNCTION_REFERENCE_VAR(__) then "fnptr"
  else "expTypeShort:ERROR"
end expTypeShort;

template dimension(Dimension d)
::=
  match d
  case DAE.DIM_INTEGER(__) then integer
  case DAE.DIM_ENUM(__) then size
  case DAE.DIM_UNKNOWN(__) then ":"
  else "INVALID_DIMENSION"
end dimension;

template arrayCrefCStr(ComponentRef cr)
::= '_<%arrayCrefCStr2(cr)%>'
end arrayCrefCStr;

template arrayCrefCStr2(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(__) then '<%unquoteIdentifier(ident)%>'
  case CREF_QUAL(__) then '<%unquoteIdentifier(ident)%>_P_<%arrayCrefCStr2(componentRef)%>'
  else "CREF_NOT_IDENT_OR_QUAL"
end arrayCrefCStr2;

template underscorePath(Path path)
 "Generate paths with components separated by underscores.
  Replaces also the . in identifiers with _. 
  The dot might happen for world.gravityAccleration"
::=
  match path
  case QUALIFIED(__) then
    '<%replaceDotAndUnderscore(name)%>_<%underscorePath(path)%>'
  case IDENT(__) then
    replaceDotAndUnderscore(name)
  case FULLYQUALIFIED(__) then
    underscorePath(path)
end underscorePath;

template replaceDotAndUnderscore(String str)
 "Replace _ with __ and dot in identifiers with _"
::=
  match str
  case name then
    let str_dots = System.stringReplace(name,".", "_")  
    let str_underscores = System.stringReplace(str_dots, "_", "__")
    '<%str_underscores%>'
end replaceDotAndUnderscore;

template tempDecl(String ty, Text &varDecls /*BUFP*/)
 "Declares a temporary variable in varDecls and returns the name."
::=
  let newVar = 'tmp<%System.tmpTick()%>'
  let &varDecls += '<%ty%> <%newVar%>;<%\n%>'
  newVar
end tempDecl;


template contextCref(ComponentRef cr, Context context,SimCode simCode)
  "Generates code for a component reference depending on which context we're in."
::=
  match context
  case FUNCTION_CONTEXT(__) then crefStr(cr)
  else cref1(cr,simCode)
end contextCref;

template contextCref2(ComponentRef cr, Context context)
  "Generates code for a component reference depending on which context we're in."
::=
  match context
  case FUNCTION_CONTEXT(__) then crefStr(cr)
  else ""
end contextCref2;

template crefFunctionName(ComponentRef cr)
::=
  match cr
  case CREF_IDENT(__) then 
    System.stringReplace(unquoteIdentifier(ident), "_", "__")
  case CREF_QUAL(__) then 
    '<%System.stringReplace(unquoteIdentifier(ident), "_", "__")%>_<%crefFunctionName(componentRef)%>'
end crefFunctionName;

template functionInitial(list<SimEqSystem> startValueEquations,Text &varDecls,SimCode simCode)

::=
  

  let eqPart = (startValueEquations |> eq as SES_SIMPLE_ASSIGN(__) =>
      equation_(eq, contextOther, &varDecls,simCode)
    ;separator="\n")
  <<
    
    <%eqPart%>
  >>
end functionInitial;


template equation_(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case e as SES_SIMPLE_ASSIGN(__)
    then equationSimpleAssign(e, context,&varDecls,simCode)
  case e as SES_ALGORITHM(__)
    then equationAlgorithm(e, context, &varDecls /*BUFD*/,simCode)
  case e as SES_WHEN(__)
    then equationWhen(e, context, &varDecls /*BUFD*/,simCode)
  case e as SES_ARRAY_CALL_ASSIGN(__)
    then equationArrayCallAssign(e, context, &varDecls /*BUFD*/,simCode)
  case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
    <<

    
     if(!(command & IContinous::RANKING))
    {
       if(_algLoopSolver<%index%>) _algLoopSolver<%index%>->solve(command);       
    }
       else _algLoop<%index%>->init();
       //if(_algLoopSolver<%index%>) _algLoopSolver<%index%>->solve(command);

      >>
  case e as SES_MIXED(__)
    then
    <<
     <%equationMixed(e, context, &varDecls, simCode)%>
     >>
  else
    "NOT IMPLEMENTED EQUATION" 
end equation_;


template equationMixed(SimEqSystem eq, Context context, Text &varDecls /*BUFP*/, SimCode simCode)
 "Generates a mixed equation system."
::=
match eq
case SES_MIXED(__) then
  let contEqs = equation_(cont, context, &varDecls /*BUFD*/, simCode)
  let numDiscVarsStr = listLength(discVars) 
  let valuesLenStr = listLength(values)
  let &preDisc = buffer "" /*BUFD*/
  let num = index
  let discvars2 = (discEqs |> SES_SIMPLE_ASSIGN(__) hasindex i0 =>
      let expPart = daeExp(exp, context, &preDisc /*BUFC*/, &varDecls /*BUFD*/,simCode)
      <<
      <%cref(cref)%> = <%expPart%>;
      new_disc_vars<%num%>[<%i0%>] = <%cref(cref)%>;
      >>
    ;separator="\n")
  <<
      bool values<%num%>[<%valuesLenStr%>] = {<%values ;separator=", "%>};
      bool pre_disc_vars<%num%>[<%numDiscVarsStr%>];
      bool new_disc_vars<%num%>[<%numDiscVarsStr%>];
      bool restart<%num%> = true;
      int iter<%num%>=0;
      int max_iter<%num%> = (<%valuesLenStr%> / <%numDiscVarsStr%>)+1;
       while(restart<%num%> && !(iter<%num%> > max_iter<%num%>))
       {
         <%discVars |> SIMVAR(__) hasindex i0 => 'pre_disc_vars<%num%>[<%i0%>] = <%cref(name)%>;' ;separator="\n"%> 
          <%contEqs%>
           checkConditions(0,true);
          <%preDisc%>
         <%discvars2%>
         bool* cur_disc_vars<%num%>[<%numDiscVarsStr%>]= {<%discVars |> SIMVAR(__) => '&<%cref(name)%>' ;separator=", "%>};
       restart<%num%>=!(_event_handling.CheckDiscreteValues(values<%num%>,pre_disc_vars<%num%>,new_disc_vars<%num%>,cur_disc_vars<%num%>,<%numDiscVarsStr%>,iter<%num%>,<%valuesLenStr%>));
       iter<%num%>++;
    }
    if(iter<%num%>>max_iter<%num%> && (restart<%num%> == true) )
    {
        //throw std::runtime_error("Number of iteration steps exceeded for discrete varibales check . ");
        cout << "Number of iteration steps exceeded for discrete varibales check at time " << time << std::endl;
    }
  
  >>
end equationMixed;

template generateAlgloopsolvers(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      generateAlgloopsolvers2(eq, contextOther, &varDecls /*BUFC*/,simCode) ;separator="\n")
    ;separator="\n")
  
  <<
   _algLoopSolverFactory = boost::shared_ptr<IAlgLoopSolverFactory>(iter->second.create());
  <%algloopsolver%>
  >>
end generateAlgloopsolvers;


template generateAlgloopsolvers2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
  let num = index
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
  <<

  _algLoop<%num%> =  boost::shared_ptr<<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>>(new <%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>(
                                                                                                                                                  <%CallAlgloopParams(modelInfo)%>,_z,_zDot,_event_handling
                                                                                                                                                  )
                                                                                                                                  );
  _algLoopSolver<%num%> = boost::shared_ptr<IAlgLoopSolver>(_algLoopSolverFactory->createAlgLoopSolver(_algLoop<%num%>.get()));
   >>
   end match
  case e as SES_MIXED(cont = eq_sys)
  then
   <<
   <%generateAlgloopsolvers2(eq_sys,context,varDecls,simCode)%>
   >> 
  else
    ""
 end generateAlgloopsolvers2;




template generateAlgloopsolverVariables(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      generateAlgloopsolverVariables2(eq, contextOther, &varDecls /*BUFC*/,simCode);separator="\n")
    ;separator="\n")
  
  <<
  <%algloopsolver%>
  >>
end generateAlgloopsolverVariables;


template generateAlgloopsolverVariables2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
   case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
  let num = index
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
  <<
      boost::shared_ptr<<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>>  //Algloop  which holds equation system
        _algLoop<%num%>;    
    boost::shared_ptr<IAlgLoopSolver>
        _algLoopSolver<%num%>;        ///< Solver for algebraic loop */
   >>
   end match
   case e as SES_MIXED(cont = eq_sys)
  then
   <<
   <%generateAlgloopsolverVariables2(eq_sys,context,varDecls,simCode)%>
   >>
  else
    ""
 end generateAlgloopsolverVariables2;


template initAlgloopsolvers(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      initAlgloopsolvers2(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator="\n")
  
  <<
  <%algloopsolver%>
  >>
end initAlgloopsolvers;


template initAlgloopsolvers2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
   case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
  let num = index
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
  <<
      // Initialize the solver
    if(_algLoopSolver<%num%>)
        _algLoopSolver<%num%>->init();
  >>
   end match
   case e as SES_MIXED(cont = eq_sys)
  then
   <<
   <%initAlgloopsolvers2(eq_sys,context,varDecls,simCode)%>
   >>
  else
    " "
 end initAlgloopsolvers2;

template algloopfilesInclude(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  <<
  <% continousEquations |> eqs => (eqs |> eq =>
      algloopfilesInclude2(eq, contextOther, &varDecls /*BUFC*/,simCode) ;separator="\n" )
    ;separator="\n" %>
  >>
end algloopfilesInclude;


template algloopfilesInclude2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
   case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
      let num = index
      match simCode
          case SIMCODE(modelInfo = MODELINFO(__)) then    
         <<#include "<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.h ">>
   end match
  case e as SES_MIXED(cont = eq_sys)
  then
   <<
   <%algloopfilesInclude2(eq_sys,context,varDecls,simCode)%>
   >>
  else
       "" 
 end algloopfilesInclude2;
 
/* 
template algloopfiles(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      algloopfiles2(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator="\n")
  
  <<
  <%algloopsolver%>
  >>
end algloopfiles;
*/

// use allEquations instead of odeEquations, because only allEquations are labeled for reduction algorithms
template algloopfiles(list<SimEqSystem> allEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (allEquations |> eqs =>
      algloopfiles2(eqs, contextOther, &varDecls /*BUFC*/,simCode)
    ;separator="\n")
  
  <<
  <%algloopsolver%>
  >>
end algloopfiles;

template algloopfiles2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
  case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
  let num = index
      match simCode
          case SIMCODE(modelInfo = MODELINFO(__)) then    
              let()= textFile(algloopHeaderFile(simCode,eq), '<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.h')
              let()= textFile(algloopCppFile(simCode,eq), '<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.cpp')
            " "
        end match
  case e as SES_MIXED(cont = eq_sys)
    then 
      let num = index
      match simCode
          case SIMCODE(modelInfo = MODELINFO(__)) then    
              let()= textFile(algloopHeaderFile(simCode, eq_sys), '<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.h')
              let()= textFile(algloopCppFile(simCode, eq_sys), '<%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.cpp')
            " "
        end match
  else
    " "
 end algloopfiles2;



template algloopcppfilenames(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let algloopsolver = (continousEquations |> eqs => (eqs |> eq =>
      algloopcppfilenames2(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator=" ")
  
  <<
  <%algloopsolver%>
  >>
end algloopcppfilenames;


template algloopcppfilenames2(SimEqSystem eq, Context context, Text &varDecls, SimCode simCode)
 "Generates an equation.
  This template should not be used for a SES_RESIDUAL.
  Residual equations are handled differently."
::=
  match eq
   case SES_LINEAR(__)
  case e as SES_NONLINEAR(__)
    then 
  let num = index
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
   <<
   <%lastIdentOfPath(modelInfo.name)%>Algloop<%num%>.cpp
   >>
   end match
   case e as SES_MIXED(cont = eq_sys)
  then
   <<
   <%algloopcppfilenames2(eq_sys,context,varDecls,simCode)%>
   >>
 else
    " "
 end algloopcppfilenames2;





template equationArrayCallAssign(SimEqSystem eq, Context context,
                                 Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates equation on form 'cref_array = call(...)'."
::=
match eq

case eqn as SES_ARRAY_CALL_ASSIGN(__) then
  let &preExp = buffer "" /*BUFD*/
  let expPart = daeExp(exp, context, &preExp /*BUF  let &preExp = buffer "" /*BUFD*/
  let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |> (e, hidx) =>
      let helpInit = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let &helpInits += 'localData->helpVars[<%hidx%>] = <%helpInit%>;'
      'localData->helpVars[<%hidx%>] && !localData->helpVars_saved[<%hidx%>] /* edge */'
    ;separator=" || ")C*/, &varDecls /*BUFD*/,simCode)
  match expTypeFromExpShort(eqn.exp)
  case "boolean" then
    let tvar = tempDecl("boolean_array", &varDecls /*BUFD*/)
    //let &preExp += 'cast_integer_array_to_real(&<%expPart%>, &<%tvar%>);<%\n%>'
    <<
    <%preExp%>
    <%cref(eqn.componentRef)%>=<%expPart%>;
    >>
  case "int" then
    let tvar = tempDecl("integer_array", &varDecls /*BUFD*/)
    //let &preExp += 'cast_integer_array_to_real(&<%expPart%>, &<%tvar%>);<%\n%>'
    <<
    <%preExp%>
    <%cref(eqn.componentRef)%>=<%expPart%>;
    >>
  case "double" then
    <<
    <%preExp%>
    <%cref(eqn.componentRef)%>=<%expPart%>;
    >>
 
end equationArrayCallAssign;

template inlineArray(Context context, String arr, ComponentRef c)
::= match context case INLINE_CONTEXT(__) then match c
case CREF_QUAL(ident = "$DER") then <<

inline_integrate_array(size_of_dimension_real_array(<%arr%>,1),<%cref(c)%>);
>>
end inlineArray;
 
template equationWhen(SimEqSystem eq, Context context, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates a when equation."
::=
match eq
 case SES_WHEN(__) then
  let &preExp = buffer ""
  let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |>(e, hidx) =>
    let helpInit = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let &preExp +=""
      '_event_handling.edge(_event_handling[<%hidx%>],"h<%hidx%>")'
   ;separator="||")
  let &preExp2 = buffer ""
  let rightExp = daeExp(right, context, &preExp2,&varDecls,simCode)
  <<
  <%preExp%>
  <%helpInits%>
  if (<%helpIf%>) {
    <%preExp2%>
    <%cref1(left, simCode)%> = <%rightExp%>;
  } 
  >>
 else  
  "UNKNOWN_equation"
end equationWhen;

template helpvarvector(list<SimWhenClause> whenClauses,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let reinit = (whenClauses |> when hasindex i0 =>
      helpvarvector1(when, contextOther,&varDecls,i0,simCode)
    ;separator="")
  <<
    <%reinit%>
  >>
end helpvarvector;

template helpvarvector1(SimWhenClause whenClauses,Context context, Text &varDecls,Integer int,SimCode simCode)
::=
match whenClauses
case SIM_WHEN_CLAUSE(__) then
  let &preExp = buffer "" /*BUFD*/
  let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |> (e, hidx) =>
      let helpInit = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let &preExp +='h[<%hidx%>]=<%helpInit%>;<%\n%>'
      ""
   ;separator="")      
<<
 <%preExp%>
  <%helpIf%>
>>
end helpvarvector1;

template resethelpvar(list<SimWhenClause> whenClauses,SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   void <%lastIdentOfPath(modelInfo.name)%>::resetHelpVar(const int index)
   {
    <%resethelpvar2(whenClauses,simCode)%>
   }
  >>
end resethelpvar;

template resethelpvar2(list<SimWhenClause> whenClauses,SimCode simCode)

::=
  let &varDecls = buffer "" /*BUFD*/
  let reinit = (whenClauses |> when hasindex i0 =>
      resethelpvar1(when, contextOther,&varDecls,i0,simCode)
    ;separator="\n")
  <<
    <%reinit%>
  >>
end resethelpvar2;

template resethelpvar1(SimWhenClause whenClauses,Context context, Text &varDecls,Integer int,SimCode simCode)
::=
  match whenClauses
  case SIM_WHEN_CLAUSE(__) then
  let &preExp = buffer "" /*BUFD*/
  let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |> (e, hidx) =>
      let helpInit = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
   let &preExp += ""
    ' if(index==<%hidx%>)
     _event_handling.setHelpVar(<%hidx%>,<%helpInit%>);
    '
   )
   
 <<
  <%preExp%>
  <%helpIf%>
 >>

end resethelpvar1;

template preCref(ComponentRef cr, SimCode simCode) ::=
'pre<%representationCref(cr, simCode)%>'
end preCref;

template equationSimpleAssign(SimEqSystem eq, Context context,Text &varDecls,
                              SimCode simCode)
 "Generates an equation that is just a simple assignment."
::=
match eq
case SES_SIMPLE_ASSIGN(__) then
  let &preExp = buffer "" /*BUFD*/
  let expPart = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  <<
  <%preExp%>
  <%cref1(cref, simCode)%>=<%expPart%>;
  >>

end equationSimpleAssign;





template daeExp(Exp exp, Context context, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for an expression."
::=
  match exp
  case e as ICONST(__)          then '<%integer%>' /* Yes, we need to cast int to long on 64-bit arch... */
  case e as RCONST(__)          then real
  case e as BCONST(__)          then if bool then "(1)" else "(0)" 
  case e as ENUM_LITERAL(__)    then index
  case e as CREF(__)            then daeExpCrefRhs(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as CAST(__)            then daeExpCast(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as CONS(__)            then "Cons not supported yet"
  case e as SCONST(__)          then daeExpSconst(string, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as UNARY(__)           then daeExpUnary(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as LBINARY(__)         then daeExpLbinary(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as LUNARY(__)          then daeExpLunary(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as BINARY(__)          then daeExpBinary(operator, exp1, exp2, context, &preExp, &varDecls,simCode)
  case e as IFEXP(__)           then daeExpIf(expCond, expThen, expElse, context, &preExp /*BUFC*/, &varDecls /*BUFD*/, simCode)
  case e as RELATION(__)        then daeExpRelation(operator, index,exp1, exp2, context, &preExp, &varDecls,simCode)
  case e as CALL(__)            then daeExpCall(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as ASUB(__)            then daeExpAsub(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as MATRIX(__)          then daeExpMatrix(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as RANGE(__)           then "Range not supported yet"
  case e as ASUB(__)            then "Asub not supported yet"
  case e as TSUB(__)            then "Tsub not supported yet"
  case e as REDUCTION(__)       then "Reduction not supported yet"
  case e as ARRAY(__)           then daeExpArray(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as SIZE(__)            then daeExpSize(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case e as SHARED_LITERAL(__)  then ""
  else "ErrorExp"
end daeExp;

template daeExpSharedLiteral(Exp exp, Context context, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a match expression."
::=
match exp case exp as SHARED_LITERAL(__) then '"_OMC_LIT<%exp.index%>"'
end daeExpSharedLiteral;

template daeExpSize(Exp exp, Context context, Text &preExp /*BUFP*/,
                    Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a size expression."
::=
  match exp
  case SIZE(exp=CREF(__), sz=SOME(dim)) then
    let expPart = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let dimPart = daeExp(dim, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let resVar = tempDecl("size_t", &varDecls /*BUFD*/)
    let typeStr = '<%expTypeArray(exp.ty)%>'
    let &preExp += '<%resVar%> = <%expPart%>.shape()[<%dimPart%>-1];<%\n%>'
    resVar
  else "size(X) not implemented"
end daeExpSize;


template daeExpMatrix(Exp exp, Context context, Text &preExp /*BUFP*/,
                      Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a matrix expression."
::=
  match exp
  case MATRIX(matrix={{}})  // special case for empty matrix: create dimensional array Real[0,1]
  case MATRIX(matrix={})    // special case for empty array: create dimensional array Real[0,1] 
    then    
    let typestr = expTypeArray(ty)
    let arrayTypeStr = 'boost::multi_array<<%typestr%>,2>'
    let tmp = tempDecl(arrayTypeStr, &varDecls /*BUFD*/)
   // let &preExp += 'alloc_<%arrayTypeStr%>(&<%tmp%>, test2, 0, 1);<%\n%>'
    tmp
   case m as MATRIX(matrix=(row1::_)) then
     let arrayTypeStr = expTypeArray(ty)
       let arrayDim = expTypeArrayforDim(ty)
       let &tmp = buffer "" /*BUFD*/
     let arrayVar = tempDecl(arrayTypeStr, &tmp /*BUFD*/)
     let &vals = buffer "" /*BUFD*/
       let dim_cols = listLength(row1)
   
    let params = (m.matrix |> row =>
        let vars = daeExpMatrixRow(row, context, &varDecls,&preExp,simCode)
        '<%vars%>'
      ;separator=",")
     
     let &preExp += '
     <%arrayDim%><%arrayVar%>(boost::extents[<%listLength(m.matrix)%>][<%dim_cols%>]);
     <%arrayVar%>.reindex(1);
     <%arrayTypeStr%> <%arrayVar%>_data[]={<%params%>};
    <%arrayVar%>.assign(<%arrayVar%>_data,<%arrayVar%>_data+ (<%listLength(m.matrix)%> * <%dim_cols%>));<%\n%>'
    
     arrayVar
end daeExpMatrix;


template daeExpMatrixRow(list<Exp> row,
                         Context context, 
                         Text &varDecls /*BUFP*/,Text &preExp /*BUFP*/,SimCode simCode)
 "Helper to daeExpMatrix."
::=

   let varLstStr = (row |> e =>
      let expVar = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      '<%expVar%>'
    ;separator=",")
  varLstStr
end daeExpMatrixRow;


template daeExpArray(Exp exp, Context context, Text &preExp /*BUFP*/,
                     Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for an array expression."
::=
match exp
case ARRAY(__) then
  let arrayTypeStr = expTypeArray(ty)

  let arrayDim = expTypeArrayforDim(ty)
  let arrayVar = tempDecl(arrayTypeStr, &varDecls /*BUFD*/)
  // let scalarPrefix = if scalar then "scalar_" else ""
  //let scalarRef = if scalar then "&" else ""
  let &tmpVar = buffer ""
  let params = (array |> e =>
    '<%daeExp(e, context, &preExp /*BUFC*/, &tmpVar /*BUFD*/,simCode)%>'
   ;separator=", ")
   let &preExp += '
   <%arrayDim%><%arrayVar%>(boost::extents[<%listLength(array)%>]);
   <%arrayVar%>.reindex(1);
   <%arrayTypeStr%> <%arrayVar%>_data[]={<%params%>};
   <%arrayVar%>.assign(<%arrayVar%>_data,<%arrayVar%>_data+<%listLength(array)%>);<%\n%>'
  arrayVar
end daeExpArray;


//template daeExpAsub(Exp exp, Context context, Text &preExp /*BUFP*/,
//                    Text &varDecls /*BUFP*/,SimCode simCode)
// "Generates code for an asub expression."
//::=
// match exp
//   case ASUB(exp=ecr as CREF(__), sub=subs) then
//    let arrName = daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context,
//                              &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
//    match context case FUNCTION_CONTEXT(__)  then
//      arrName
//    else
//      arrayScalarRhs(exp, subs, arrName, context, &preExp, &varDecls,simCode)

//end daeExpAsub;


template daeExpAsub(Exp inExp, Context context, Text &preExp /*BUFP*/,
                    Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for an asub expression."
::=
  match expTypeFromExpShort(inExp)
  case "metatype" then
  // MetaModelica Array
    (match inExp case ASUB(exp=e, sub={idx}) then
      let e1 = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let idx1 = daeExp(idx, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      'arrayGet(<%e1%>,<%idx1%>) /* DAE.ASUB */')
  // Modelica Array
  else
  match inExp
  
  case ASUB(exp=ASUB(__)) then
    error(sourceInfo(),'Nested array subscripting *should* have been handled by the routine creating the asub, but for some reason it was not: <%printExpStr(exp)%>')

  // Faster asub: Do not construct a whole new array just to access one subscript
  case ASUB(exp=exp as ARRAY(scalar=true), sub={idx}) then
    let res = tempDecl(expTypeFromExpModelica(exp),&varDecls)
    let idx1 = daeExp(idx, context, &preExp, &varDecls,simCode)
    let expl = (exp.array |> e hasindex i1 fromindex 1 =>
      let &caseVarDecls = buffer ""
      let &casePreExp = buffer ""
      let v = daeExp(e, context, &casePreExp, &caseVarDecls,simCode)
      <<
      case <%i1%>: {
        <%&caseVarDecls%>
        <%&casePreExp%>
        <%res%> = <%v%>;
        break;
      }
      >> ; separator = "\n")
    let &preExp +=
    <<
    switch (<%idx1%>) { /* ASUB */
    <%expl%>
    default:
      assert(NULL == "index out of bounds");
    }
    >>
    res
  
  case ASUB(exp=RANGE(ty=t), sub={idx}) then
    error(sourceInfo(),'ASUB_EASY_CASE <%printExpStr(exp)%>')
  
 case ASUB(exp=ecr as CREF(__), sub=subs) then
    let arrName = daeExpCrefRhs(buildCrefExpFromAsub(ecr, subs), context,
                              &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    match context case FUNCTION_CONTEXT(__)  then
      arrName
    else
      arrayScalarRhs(ecr, subs, arrName, context, &preExp, &varDecls,simCode)

  
  case ASUB(exp=e, sub=indexes) then
  let exp = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
 // let typeShort = expTypeFromExpShort(e)
  let expIndexes = (indexes |> index => '<%daeExpASubIndex(index, context, &preExp, &varDecls,simCode)%>' ;separator=", ")
   //'<%typeShort%>_get<%match listLength(indexes) case 1 then "" case i then '_<%i%>D'%>(&<%exp%>, <%expIndexes%>)'
  '(<%exp%>)[<%expIndexes%>+1]'
  case exp then
    error(sourceInfo(),'OTHER_ASUB <%printExpStr(exp)%>')
end daeExpAsub;

template daeExpASubIndex(Exp exp, Context context, Text &preExp, Text &varDecls,SimCode simCode)
::=
match exp
  case ICONST(__) then incrementInt(integer,-1)
  case ENUM_LITERAL(__) then incrementInt(index,-1)
  else daeExp(exp,context,&preExp,&varDecls,simCode)
end daeExpASubIndex;


template arrayScalarRhs(Exp exp, list<Exp> subs, String arrName, Context context,
               Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Helper to daeExpAsub."
::=
   match exp
   case ASUB(exp=ecr as CREF(__)) then
  let arrayType = expTypeArray(exp.ty)
  let dimsLenStr = listLength(subs)
  let dimsValuesStr = (subs |> exp =>
      daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)

    ;separator="][")
  match arrayType
    case "metatype_array" then
      'arrayGet(<%arrName%>,<%dimsValuesStr%>) /*arrayScalarRhs*/'
    else
      <<
      <%arrayCrefCStr(ecr.componentRef)%>[<%dimsValuesStr%>]
      >>
end arrayScalarRhs;

template daeExpCast(Exp exp, Context context, Text &preExp /*BUFP*/,
                    Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a cast expression."
::=
match exp
case CAST(__) then
  let expVar = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match ty
  case T_INTEGER(__)   then '((int)<%expVar%>)'  
  case T_REAL(__)  then '((double)<%expVar%>)'
  case T_ENUMERATION(__)   then '((modelica_integer)<%expVar%>)'
  case T_BOOL(__)   then '((bool)<%expVar%>)'  
  case T_ARRAY(__) then
    let arrayTypeStr = expTypeArray(ty)
    let tvar = tempDecl(arrayTypeStr, &varDecls /*BUFD*/)
    let to = expTypeShort(ty)
    let from = expTypeFromExpShort(exp)
    let &preExp += 'cast_<%from%>_array_to_<%to%>(&<%expVar%>, &<%tvar%>);<%\n%>'
    '<%tvar%>'
  else 
    '(<%expVar%>) /* could not cast, using the variable as it is */'
end daeExpCast;

template daeExpCall(Exp call, Context context, Text &preExp /*BUFP*/,
                    Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a function call."
::=
  match call
  // special builtins
  
  case CALL(path=IDENT(name="edge"),
            expLst={e1}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    '_event_handling.edge(<%var1%>,"<%var1%>")'
    
  case CALL(path=IDENT(name="pre"),
            expLst={arg as CREF(__)}) then
    let var1 = daeExp(arg, context, &preExp, &varDecls,simCode)
    '_event_handling.pre(<%var1%>,"<%cref(arg.componentRef)%>")'
    
  case CALL(path=IDENT(name="sample"), expLst={start, interval,index}) then
    let &preExp = buffer "" /*BUFD*/
    let eStart = daeExp(start, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
    let eInterval = daeExp(interval, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
    let eIndex = daeExp(index, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
     '_conditions1[<%eIndex%>]'
  case CALL(path=IDENT(name="initial") ) then
      'initial()'
   
   case CALL(path=IDENT(name="DIVISION"),
            expLst={e1, e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
     
    'division(<%var1%>,<%var2%>,"<%var1%>/<%var2%> because <%var2%>  == 0")'      
  
  case CALL(path=IDENT(name="DIVISION"),
            expLst={e1, e2, DAE.SCONST(string=string)}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
     let var3 = Util.escapeModelicaStringToCString(string)
    'division(<%var1%>,<%var2%>,"<%var3%>")'
   
   case CALL(path=IDENT(name="sign"),
            expLst={e1}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
     'sgn(<%var1%>)'
   case CALL(path=IDENT(name="DIVISION"))
            then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
   // let typeStr = expTypeShort(attr.ty )
   let typeStr ="double"
    let retVar = tempDecl(typeStr, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = division(<%argStr%>,"division by zero");<%\n%>'
    '<%retVar%>'              
  
  case CALL(path=IDENT(name="DIVISION_ARRAY_SCALAR"),
            expLst={e1, e2, DAE.SCONST(string=string)},attr=attr as CALL_ATTR(__)) then
    let type = match attr.ty case T_ARRAY(ty=T_INTEGER(__)) then "integer_array" 
                        case T_ARRAY(ty=T_ENUMERATION(__)) then "integer_array"
                        else "real_array"
    let var = tempDecl(type, &varDecls)
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    let var3 = Util.escapeModelicaStringToCString(string)
    let &preExp += 'division_alloc_<%type%>_scalar(&<%var1%>, <%var2%>, &<%var%>,"<%var3%>");<%\n%>'
    '<%var%>'
    
  case CALL(path=IDENT(name="der"), expLst={arg as CREF(__)}) then
    representationCref2(arg.componentRef,simCode)
  case CALL(path=IDENT(name="pre"), expLst={arg as CREF(__)}) then
    let retType = '<%expTypeArrayIf(arg.ty)%>'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let cast = match arg.ty case T_INTEGER(__) then "(int)" 
                            case T_ENUMERATION(__) then "(int)" //else ""
    let &preExp += '<%retVar%> = <%cast%>pre(<%cref(arg.componentRef)%>);<%\n%>'
    '<%retVar%>'
  
  case CALL(path=IDENT(name="print"), expLst={e1}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    if acceptMetaModelicaGrammar() then 'print(<%var1%>)' else 'puts(<%var1%>)'
  
  case CALL(path=IDENT(name="max"), attr=CALL_ATTR(ty = T_REAL(__)), expLst={e1,e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'fmax(<%var1%>,<%var2%>)'
  
  case CALL(path=IDENT(name="max"), expLst={e1,e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'int_max((int)<%var1%>,(int)<%var2%>)'
  
  case CALL(attr=CALL_ATTR(ty = T_REAL(__)),
            path=IDENT(name="min"), expLst={e1,e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'fmin(<%var1%>,<%var2%>)'
  
  case CALL(path=IDENT(name="min"), expLst={e1,e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'int_min((int)<%var1%>,(int)<%var2%>)'
  
  case CALL(path=IDENT(name="abs"), expLst={e1}, attr=CALL_ATTR(ty = T_INTEGER(__))) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    'labs(<%var1%>)'
  
  case CALL(path=IDENT(name="abs"), expLst={e1}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    'fabs(<%var1%>)'
  
  case CALL(path=IDENT(name="sqrt"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let typeStr = expTypeShort(attr.ty )
    let retVar = tempDecl(typeStr, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = sqrt(<%argStr%>);<%\n%>'
    '<%retVar%>' 
  
  case CALL(path=IDENT(name="sin"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>_rettype'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = <%daeExpCallBuiltinPrefix(attr.builtin)%><%funName%>(<%argStr%>);<%\n%>'
    if attr.builtin then '<%retVar%>' else '<%retVar%>.<%retType%>_1'
    
   case CALL(path=IDENT(name="cos"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>_rettype'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = <%daeExpCallBuiltinPrefix(attr.builtin)%><%funName%>(<%argStr%>);<%\n%>'
    if attr.builtin then '<%retVar%>' else '<%retVar%>.<%retType%>_1'

   case CALL(path=IDENT(name="tan"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>_rettype'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = <%daeExpCallBuiltinPrefix(attr.builtin)%><%funName%>(<%argStr%>);<%\n%>'
    if attr.builtin then '<%retVar%>' else '<%retVar%>.<%retType%>_1'
    
   case CALL(path=IDENT(name="atan"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>_rettype'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = <%daeExpCallBuiltinPrefix(attr.builtin)%><%funName%>(<%argStr%>);<%\n%>'
    if attr.builtin then '<%retVar%>' else '<%retVar%>.<%retType%>_1'
    
   case CALL(path=IDENT(name="exp"),
            expLst={e1},attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>_rettype'
    let retVar = tempDecl(retType, &varDecls /*BUFD*/)
    let &preExp += '<%retVar%> = <%daeExpCallBuiltinPrefix(attr.builtin)%><%funName%>(<%argStr%>);<%\n%>'
    if attr.builtin then '<%retVar%>' else '<%retVar%>.<%retType%>_1'
  
  case CALL(path=IDENT(name="div"), expLst={e1,e2}, attr=CALL_ATTR(ty = T_INTEGER(__))) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'ldiv(<%var1%>,<%var2%>).quot'
  
  case CALL(path=IDENT(name="div"), expLst={e1,e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'trunc(<%var1%>/<%var2%>)'
  
  case CALL(path=IDENT(name="mod"), expLst={e1,e2}, attr=attr as CALL_ATTR(__)) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    'modelica_mod_<%expTypeShort(attr.ty)%>(<%var1%>,<%var2%>)'
  
  case CALL(path=IDENT(name="max"), expLst={array}) then
    let &tmpVar = buffer "" /*BUFD*/
    let expVar = daeExp(array, context, &preExp /*BUFC*/, &tmpVar /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(array)%>'
    let tvar = tempDecl(expTypeFromExpModelica(array), &varDecls /*BUFD*/)
    let &preExp += '<%tvar%> = min_max<<%arr_tp_str%>,1>(<%expVar%>).second;<%\n%>'
    '<%tvar%>'
    
  case CALL(path=IDENT(name="min"), expLst={array}) then
    let &tmpVar = buffer "" /*BUFD*/
    let expVar = daeExp(array, context, &preExp /*BUFC*/, &tmpVar /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(array)%>'
    let tvar = tempDecl(expTypeFromExpModelica(array), &varDecls /*BUFD*/)
    let &preExp += '<%tvar%> = min_max<<%arr_tp_str%>,1>(<%expVar%>).first;<%\n%>'
    '<%tvar%>'
  
  case CALL(path=IDENT(name="fill"), expLst=val::dims, attr=attr as CALL_ATTR(__)) then
    let valExp = daeExp(val, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let dimsExp = (dims |> dim =>
      daeExp(dim, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode) ;separator="][")
    let ty_str = '<%expTypeArray(attr.ty)%>'
    let tmp_type_str =  'multi_array<<%ty_str%>,<%listLength(dims)%>>'
    
    let tvar = tempDecl(tmp_type_str, &varDecls /*BUFD*/)
    
    let &varDecls += '<%tvar%>.resize((boost::extents[<%dimsExp%>]));
    <%tvar%>.reindex(1);<%\n%>'
    
    let &preExp += 'fill_array<<%ty_str%>,<%listLength(dims)%>>(<%tvar%>, <%valExp%>);<%\n%>'
    '<%tvar%>'
  
    
  case CALL(path=IDENT(name="cat"), expLst=dim::arrays, attr=attr as CALL_ATTR(__)) then
    let dim_exp = daeExp(dim, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let arrays_exp = (arrays |> array =>
      daeExp(array, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode) ;separator=", &")
    let ty_str = '<%expTypeArray(attr.ty)%>'
    let tvar = tempDecl(ty_str, &varDecls /*BUFD*/)
    let &preExp += 'cat_alloc_<%ty_str%>(<%dim_exp%>, &<%tvar%>, <%listLength(arrays)%>, &<%arrays_exp%>);<%\n%>'
    '<%tvar%>'
    
  case CALL(path=IDENT(name="promote"), expLst={A, n}) then
    let var1 = daeExp(A, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let var2 = daeExp(n, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(A)%>'
    let tvar = tempDecl(arr_tp_str, &varDecls /*BUFD*/)
    let &preExp += 'promote_alloc_<%arr_tp_str%>(&<%var1%>, <%var2%>, &<%tvar%>);<%\n%>'
    '<%tvar%>'
  
  case CALL(path=IDENT(name="transpose"), expLst={A}) then
    let var1 = daeExp(A, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(A)%>'
    let tvar = tempDecl(arr_tp_str, &varDecls /*BUFD*/)
    let &preExp += 'transpose_alloc_<%arr_tp_str%>(&<%var1%>, &<%tvar%>);<%\n%>'
    '<%tvar%>'
   
   case CALL(path=IDENT(name="cross"), expLst={v1, v2}) then
    let var1 = daeExp(v1, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let var2 = daeExp(v2, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(v1)%>'
    let tvar = tempDecl(arr_tp_str, &varDecls /*BUFD*/)
    let &preExp += 'cross_alloc_<%arr_tp_str%>(&<%var1%>, &<%var2%>, &<%tvar%>);<%\n%>'
    '<%tvar%>'
  
  case CALL(path=IDENT(name="identity"), expLst={A}) then
    let var1 = daeExp(A, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let arr_tp_str = '<%expTypeFromExpArray(A)%>'
    let tvar = tempDecl(arr_tp_str, &varDecls /*BUFD*/)
    let &preExp += 'identity_alloc_<%arr_tp_str%>(<%var1%>, &<%tvar%>);<%\n%>'
    '<%tvar%>'
   
   case CALL(path=IDENT(name="rem"),
             expLst={e1, e2}) then
    let var1 = daeExp(e1, context, &preExp, &varDecls,simCode)
    let var2 = daeExp(e2, context, &preExp, &varDecls,simCode)
    let typeStr = expTypeFromExpShort(e1)
    'modelica_rem_<%typeStr%>(<%var1%>,<%var2%>)'
    
   case CALL(path=IDENT(name="String"),
             expLst={s, format}) then
    let tvar = tempDecl("modelica_string", &varDecls /*BUFD*/)
    let sExp = daeExp(s, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let formatExp = daeExp(format, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let typeStr = expTypeFromExpModelica(s)
    let &preExp += '<%tvar%> = <%typeStr%>_to_modelica_string_format(<%sExp%>, <%formatExp%>);<%\n%>'
    '<%tvar%>'
    
   case CALL(path=IDENT(name="String"),
             expLst={s, minlen, leftjust}) then
    let tvar = tempDecl("modelica_string", &varDecls /*BUFD*/)
    let sExp = daeExp(s, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let minlenExp = daeExp(minlen, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let leftjustExp = daeExp(leftjust, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let typeStr = expTypeFromExpModelica(s)
    let &preExp += '<%tvar%> = <%typeStr%>_to_modelica_string(<%sExp%>, <%minlenExp%>, <%leftjustExp%>);<%\n%>'
    '<%tvar%>'
  
  case CALL(path=IDENT(name="String"),
            expLst={s, minlen, leftjust, signdig}) then
    let tvar = tempDecl("modelica_string", &varDecls /*BUFD*/)
    let sExp = daeExp(s, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let minlenExp = daeExp(minlen, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let leftjustExp = daeExp(leftjust, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let signdigExp = daeExp(signdig, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let &preExp += '<%tvar%> = double_to_modelica_string(<%sExp%>, <%minlenExp%>, <%leftjustExp%>, <%signdigExp%>);<%\n%>'
    '<%tvar%>'
    
  case CALL(path=IDENT(name="delay"),
            expLst={ICONST(integer=index), e, d, delayMax}) then
    let tvar = tempDecl("double", &varDecls /*BUFD*/)
    let var1 = daeExp(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let var2 = daeExp(d, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let var3 = daeExp(delayMax, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let &preExp += '<%tvar%> = delayImpl(<%index%>, <%var1%>, time, <%var2%>, <%var3%>);<%\n%>'
    '<%tvar%>'
  
  case CALL(path=IDENT(name="integer"),
            expLst={toBeCasted}) then
    let castedVar = daeExp(toBeCasted, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    '((int)<%castedVar%>)'
    
   case CALL(path=IDENT(name="Integer"),
             expLst={toBeCasted}) then
    let castedVar = daeExp(toBeCasted, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    '((int)<%castedVar%>)'
  
  case CALL(path=IDENT(name="clock"), expLst={}) then
    'mmc_clock()'

  case CALL(path=IDENT(name="noEvent"),
            expLst={e1}) then
    daeExp(e1, context, &preExp, &varDecls,simCode)
  
  case CALL(path=IDENT(name="anyString"),
            expLst={e1}) then
    'mmc_anyString(<%daeExp(e1, context, &preExp, &varDecls,simCode)%>)'
    
  case CALL(path=IDENT(name="mmc_get_field"),
            expLst={s1, ICONST(integer=i)}) then
    let tvar = tempDecl("modelica_metatype", &varDecls /*BUFD*/)
    let expPart = daeExp(s1, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let &preExp += '<%tvar%> = MMC_FETCH(MMC_OFFSET(MMC_UNTAGPTR(<%expPart%>), <%i%>));<%\n%>'
    '<%tvar%>'
  
  case exp as CALL(attr=attr as CALL_ATTR(__)) then
    let argStr = (expLst |> exp => '<%daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)%>' ;separator=", ")
    let funName = '<%underscorePath(path)%>'
    let retType = '<%funName%>RetType'
    let retVar = match exp
      case CALL(attr=CALL_ATTR(ty=T_NORETCALL(__))) then ""
      else tempDecl(retType, &varDecls)
    let &preExp += match context case FUNCTION_CONTEXT(__) then'<%if retVar then '<%retVar%> = '%><%funName%>(<%argStr%>);<%\n%>'
    /*let &preExp += match context case OTHER(__) then*/ else '<%if retVar then '<%retVar%> = '%>_functions.<%funName%>(<%argStr%>);<%\n%>'
    
    match exp
      // no return calls
      case CALL(attr=CALL_ATTR(ty=T_NORETCALL(__))) then '/* NORETCALL */'
      // non tuple calls (single return value)
      case CALL(attr=CALL_ATTR(tuple_=false)) then
       if attr.builtin then '<%retVar%>' else 'get<0>(<%retVar%>)'
      // tuple calls (multiple return values)
      case CALL(attr=CALL_ATTR(tuple_=true)) then
        '<%retVar%>'
  
    
end daeExpCall;

template expTypeFromExpShort(Exp exp)

::=
  expTypeFromExpFlag(exp, 1)
end expTypeFromExpShort;

template expTypeFromExpModelica(Exp exp)

::=
  expTypeFromExpFlag(exp, 2)
end expTypeFromExpModelica;

template expTypeFromExpArray(Exp exp)

::=
  expTypeFromExpFlag(exp, 3)
end expTypeFromExpArray;

template assertCommon(Exp condition, Exp message, Context context, Text &varDecls, Info info,SimCode simCode)
::=
  let &preExpCond = buffer ""
  let &preExpMsg = buffer ""
  let condVar = daeExp(condition, context, &preExpCond, &varDecls,simCode)
  let msgVar = daeExp(message, context, &preExpMsg, &varDecls,simCode)
  <<
  <%preExpCond%>
  if (!<%condVar%>) {
    <%preExpMsg%>
    omc_fileInfo info = {<%infoArgs(info)%>};
    MODELICA_ASSERT(info, <%if acceptMetaModelicaGrammar() then 'MMC_STRINGDATA(<%msgVar%>)' else msgVar%>);
  }
  >>
end assertCommon;

template infoArgs(Info info)
::=
  match info
  case INFO(__) then '"<%fileName%>",<%lineNumberStart%>,<%columnNumberStart%>,<%lineNumberEnd%>,<%columnNumberEnd%>,<%isReadOnly%>'
end infoArgs;


template underscorePrefix(Boolean builtin) ::=
  match builtin
  case true then ""
  case false then "_"
end underscorePrefix;

template daeExpCallBuiltinPrefix(Boolean builtin)
 "Helper to daeExpCall."
::=
  match builtin
  case true  then ""
  case false then "_"
end daeExpCallBuiltinPrefix;


template daeExpLunary(Exp exp, Context context, Text &preExp /*BUFP*/,
                      Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a logical unary expression."
::=
match exp
case LUNARY(__) then
  let e = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match operator
  case NOT(__) then '(!<%e%>)'
end daeExpLunary;

template daeExpLbinary(Exp exp, Context context, Text &preExp /*BUFP*/,
                       Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a logical binary expression."
::=
match exp
case LBINARY(__) then
  let e1 = daeExp(exp1, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  let e2 = daeExp(exp2, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match operator
  case AND(__) then '(<%e1%> && <%e2%>)'
  case OR(__)  then '(<%e1%> || <%e2%>)'
  else "daeExpLbinary:ERR"
end daeExpLbinary;

template daeExpBinary(Operator it, Exp exp1, Exp exp2, Context context, Text &preExp,Text &varDecls /*BUFP*/, SimCode simCode) ::=
  let e1 = daeExp(exp1, context, &preExp, &varDecls, simCode)
  let e2 = daeExp(exp2, context, &preExp, &varDecls, simCode)
  match it
  case ADD(__) then '(<%e1%> + <%e2%>)'
  case SUB(__) then '(<%e1%> - <%e2%>)'
  case MUL(__) then '(<%e1%> * <%e2%>)'
  case DIV(__) then '(<%e1%> / <%e2%>)'
  case POW(__) then 'pow(<%e1%>, <%e2%>)'
  case AND(__) then '(<%e1%> && <%e2%>)'
  case OR(__)  then '(<%e1%> || <%e2%>)'
  case MUL_ARRAY_SCALAR(ty=T_ARRAY(dims=dims)) then
    let type = match ty case T_ARRAY(ty=T_INTEGER(__)) then 'multi_array<int,<%listLength(dims)%>>'
                        case T_ARRAY(ty=T_ENUMERATION(__)) then 'multi_array<int,<%listLength(dims)%>>'
                        else 'multi_array<double,<%listLength(dims)%>>'
    let type1 = match ty case T_ARRAY(ty=T_INTEGER(__)) then "int"
                        case T_ARRAY(ty=T_ENUMERATION(__)) then "int"
                        else "double"
    //let var = tempDecl(type,&varDecls /*BUFD*/)
    let var1 = tempDecl1(type,e1,&varDecls /*BUFD*/)
    let &preExp += '<%var1%>=multiply_array<<%type1%>,<%listLength(dims)%>>(<%e1%>, <%e2%>);<%\n%>'
    '<%var1%>'  
  case MUL_MATRIX_PRODUCT(ty=T_ARRAY(dims=dims)) then
    let type = match ty case T_ARRAY(ty=T_INTEGER(__)) then 'multi_array<int,<%listLength(dims)%>>'
                        case T_ARRAY(ty=T_ENUMERATION(__)) then 'multi_array<int,<%listLength(dims)%>>'
                        else 'multi_array<double,<%listLength(dims)%>>'
    let type1 = match ty case T_ARRAY(ty=T_INTEGER(__)) then "int"
                        case T_ARRAY(ty=T_ENUMERATION(__)) then "int"
                        else "double"
    //let var = tempDecl(type,&varDecls /*BUFD*/)
    let var1 = tempDecl1(type,e1,&varDecls /*BUFD*/)
    let &preExp += '<%var1%>=multiply_array<<%type1%>,<%listLength(dims)%>>(<%e1%>, <%e2%>);<%\n%>'
    '<%var1%>'  
  case DIV_ARRAY_SCALAR(ty=T_ARRAY(dims=dims)) then
  let type = match ty case T_ARRAY(ty=T_INTEGER(__)) then "integer_array" 
                        case T_ARRAY(ty=T_ENUMERATION(__)) then "integer_array"
                        else "real_array"
    let var = tempDecl(type, &varDecls /*BUFD*/)
    let &preExp += 'div_alloc_<%type%>_scalar(&<%e1%>, <%e2%>, &<%var%>);<%\n%>'
    '<%var%>'
  case _   then "daeExpBinary:ERR"
end daeExpBinary;

template tempDecl1(String ty, String exp, Text &varDecls /*BUFP*/)
 "Declares a temporary variable in varDecls and returns the name."
::=
  let newVar = 'tmp<%System.tmpTick()%>'
  let newVar1 = '<%newVar%>(<%exp%>)'
  let &varDecls += '<%ty%> <%newVar1%>;<%\n%>'
  newVar
end tempDecl1;


template daeExpSconst(String string, Context context, Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a string constant."
::=
  '"<%Util.escapeModelicaStringToCString(string)%>"'
end daeExpSconst;

template daeExpUnary(Exp exp, Context context, Text &preExp /*BUFP*/,
                     Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a unary expression."
::=
match exp
case UNARY(__) then
  let e = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match operator
  case UMINUS(__)     then '(-<%e%>)'
  case UMINUS_ARR(ty=T_ARRAY(ty=T_REAL(__))) then
    let &preExp += 'usub_real_array(&<%e%>);<%\n%>'
    '<%e%>'
  case UMINUS_ARR(__) then 'unary minus for non-real arrays not implemented'
  else "daeExpUnary:ERR"
end daeExpUnary;


template daeExpCrefRhs(Exp exp, Context context, Text &preExp /*BUFP*/,
                       Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a component reference on the right hand side of an
 expression."
::=
  match exp
  // A record cref without subscripts (i.e. a record instance) is handled
  // by daeExpRecordCrefRhs only in a simulation context, not in a function.
  case CREF(componentRef = cr, ty = t as T_COMPLEX(complexClassType = RECORD(path = _))) then
    match context case FUNCTION_CONTEXT(__) then
      daeExpCrefRhs2(exp, context, &preExp, &varDecls,simCode)
    else
      daeExpRecordCrefRhs(t, cr, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  case CREF(componentRef = cr, ty = T_FUNCTION_REFERENCE_FUNC(__)) then
    '((modelica_fnptr)boxptr_<%crefFunctionName(cr)%>)'
  case CREF(componentRef = cr, ty = T_FUNCTION_REFERENCE_VAR(__)) then
    '((modelica_fnptr) _<%crefStr(cr)%>)'
  else daeExpCrefRhs2(exp, context, &preExp, &varDecls,simCode)
end daeExpCrefRhs;

template daeExpRecordCrefRhs(DAE.Type ty, ComponentRef cr, Context context, Text &preExp /*BUFP*/,
                       Text &varDecls /*BUFP*/,SimCode simCode)
::=
match ty
case T_COMPLEX(complexClassType = record_state, varLst = var_lst) then
  let vars = var_lst |> v => daeExp(makeCrefRecordExp(cr,v), context, &preExp, &varDecls,simCode) 
             ;separator=", "
  let record_type_name = underscorePath(ClassInf.getStateName(record_state))
  let ret_type = '<%record_type_name%>RetType'
  let ret_var = tempDecl(ret_type, &varDecls)
  let &preExp += '<%ret_var%> = _functions.<%record_type_name%>(<%vars%>);<%\n%>'
  '<%ret_var%>'
end daeExpRecordCrefRhs;

template daeExpCrefRhs2(Exp ecr, Context context, Text &preExp /*BUFP*/,
                       Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a component reference."
::=
  match ecr
  case ecr as CREF(componentRef=cr, ty=ty) then
      let box = daeExpCrefRhsArrayBox(ecr, context, &preExp, &varDecls,simCode)
    if box then
      box
    else if crefIsScalar(cr, context) then
      let cast = match ty case T_INTEGER(__) then ""
                          case T_ENUMERATION(__) then "" //else ""
      '<%cast%><%contextCref(cr,context,simCode)%>'
    else 
     if crefSubIsScalar(cr) then
      // The array subscript results in a scalar
      let arrName = contextCref(crefStripLastSubs(cr), context,simCode)
      let arrayType = expTypeArray(ty)
      //let dimsLenStr = listLength(crefSubs(cr))
      let dimsValuesStr = (crefSubs(cr) |> INDEX(__) =>
          daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
        ;separator="][")
      match arrayType
        case "metatype_array" then
          'arrayGet(<%arrName%>,<%dimsValuesStr%>) /* DAE.CREF */'
        else
         <<
          <%arrName%>[<%dimsValuesStr%>]
          >>
    else
      // The array subscript denotes a slice
      let arrName = contextArrayCref(cr, context)
      let arrayType = expTypeArray(ty)
      let tmp = tempDecl(arrayType, &varDecls /*BUFD*/)
      let spec1 = daeExpCrefRhsIndexSpec(crefSubs(cr), context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let &preExp += 'index_alloc_<%arrayType%>(&<%arrName%>, &<%spec1%>, &<%tmp%>);<%\n%>'
      tmp
end daeExpCrefRhs2;

template daeExpCrefRhsIndexSpec(list<Subscript> subs, Context context,
                                Text &preExp /*BUFP*/, Text &varDecls /*BUFP*/,SimCode simCode)
 "Helper to daeExpCrefRhs."
::=
  let nridx_str = listLength(subs)
  let idx_str = (subs |> sub =>
      match sub
      case INDEX(__) then
        let expPart = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
        <<
        (0), make_index_array(1, (int) <%expPart%>), 'S'

        >>
      case WHOLEDIM(__) then
        <<
        (1), (int*)0, 'W'
        >>
      case SLICE(__) then
        let expPart = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
        let tmp = tempDecl("int", &varDecls /*BUFD*/)
        let &preExp += '<%tmp%> = size_of_dimension_integer_array(<%expPart%>, 1);<%\n%>'
        <<
        (int) <%tmp%>, integer_array_make_index_array(&<%expPart%>), 'A'
        >>
    ;separator=", ")
  let tmp = tempDecl("index_spec_t", &varDecls /*BUFD*/)
  let &preExp += 'create_index_spec(&<%tmp%>, <%nridx_str%>, <%idx_str%>);<%\n%>'
  tmp
end daeExpCrefRhsIndexSpec;

template daeExpCrefRhsArrayBox(Exp ecr, Context context, Text &preExp /*BUFP*/,
                               Text &varDecls /*BUFP*/,SimCode simCode)
 "Helper to daeExpCrefRhs."
::=
match ecr
case ecr as CREF(ty=T_ARRAY(ty=aty,dims=dims)) then
  match context
  case FUNCTION_CONTEXT(__) then ''
  else
    // For context simulation and other array variables must be boxed into a real_array
    // object since they are represented only in a double array.
    //let tmpArr = tempDecl(expTypeArray(aty), &varDecls /*BUFD*/)
    //let &preExp += '<%tmpArr%> = <%arrayCrefCStr(ecr.componentRef)%>;<%\n%>'
    let tmpArr = '<%arrayCrefCStr(ecr.componentRef)%>'
    //let dimsLenStr = listLength(dims)
    //let type = expTypeShort(aty)
    //let &preExp += '<%type%>_array_create(&<%tmpArr%>, ((modelica_<%type%>*)&(<%arrayCrefCStr(ecr.componentRef)%>)), <%dimsLenStr%>);<%\n%>'
    tmpArr
end daeExpCrefRhsArrayBox;

template cref1(ComponentRef cr, SimCode simCode) ::=
  match cr
  case CREF_IDENT(ident = "xloc") then '<%representationCref(cr, simCode) %>'
  case CREF_IDENT(ident = "time") then "time"
  else '<%representationCref(cr, simCode) %>'
end cref1;

template representationCref(ComponentRef inCref, SimCode simCode) ::=
  cref2simvar(inCref, simCode) |> SIMVAR(__) =>
     match varKind 
  case STATE(__)        then
    << <%representationCref1(inCref,simCode)%> >>
  case STATE_DER(__)   then 
    << <%representationCref2(inCref,simCode)%> >>
  else << <%cref(inCref)%>>>
end representationCref;


template representationCref1(ComponentRef inCref, SimCode simCode) ::=
  cref2simvar(inCref, simCode) |> SIMVAR(__) =>
    match index
   case -1 then 
   << <%cref2(inCref)%> >>
   case _  then
   << _z[<%index%>] >>
end representationCref1;

template representationCref2(ComponentRef inCref, SimCode simCode) ::=
  cref2simvar(inCref, simCode) |> SIMVAR(__) =>'_zDot[<%index%>]'
end representationCref2;

template helpvarlength(SimCode simCode) 
::=
match simCode
case SIMCODE(__) then
  let size = listLength(helpVarInfo)
  <<
   <%size%>
  >>
end helpvarlength;

template zerocrosslength(SimCode simCode) 
::=
match simCode
case SIMCODE(__) then
  let size = listLength(zeroCrossings)
  <<
     <%size%>
  >>
end zerocrosslength;


template DimZeroFunc(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   int <%lastIdentOfPath(modelInfo.name)%>::getDimZeroFunc()
    {
      return _dimZeroFunc;
    }
  >>
end DimZeroFunc;




template DimInitEquations(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
  <<
   unsigned int <%lastIdentOfPath(modelInfo.name)%>::getDimInitEquations()
    {
      return <%varInfo.numInitEquations%>;
    }
  >>
  end match
end DimInitEquations;

template DimUnfixedStates(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   unsigned int <%lastIdentOfPath(modelInfo.name)%>::getDimUnfixedStates()
    {
      return 0;
    }
  >>
end DimUnfixedStates;


template DimUnfixedParameters(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
match modelInfo
case MODELINFO(varInfo=VARINFO(__)) then
  <<
   unsigned int <%lastIdentOfPath(modelInfo.name)%>::getDimUnfixedParameters()
    {
      return <%varInfo.numResiduals%>;
    }
  >>
  end match
end DimUnfixedParameters;


template DimIntialResiduals(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   unsigned int <%lastIdentOfPath(modelInfo.name)%>::getDimIntialResiduals()
    {
      return 0;
    }
  >>
end DimIntialResiduals;

template SetIntialStatus(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   void <%lastIdentOfPath(modelInfo.name)%>::setInitial(bool status)
    {
      _initial = status;
    }
  >>
end SetIntialStatus;

template GetIntialStatus(SimCode simCode)
::=
match simCode
case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
   bool <%lastIdentOfPath(modelInfo.name)%>::initial()
    {
      return _initial;
    }
  >>
end GetIntialStatus;


template daeExpRelation(Operator op, Integer index,Exp exp1, Exp exp2, Context context, Text &preExp,Text &varDecls,SimCode simCode) ::=
  let e1 = daeExp(exp1, context, &preExp, &varDecls,simCode)
  let e2 = daeExp(exp2, context, &preExp, &varDecls,simCode)
  match index 
  case -1 then 
     match op
    case LESS(ty = T_BOOL(__))        then '(!<%e1%> && <%e2%>)'
    case LESS(ty = T_STRING(__))      then "# string comparison not supported\n"
    case LESS(ty = T_INTEGER(__))
    case LESS(ty = T_REAL(__))        then '(<%e1%> < <%e2%>)'
    case LESS(ty = T_ENUMERATION(__))      then '(<%e1%> < <%e2%>)'
     
    case GREATER(ty = T_BOOL(__))     then '(<%e1%> && !<%e2%>)'
    case GREATER(ty = T_STRING(__))   then "# string comparison not supported\n"
    case GREATER(ty = T_INTEGER(__))
    case GREATER(ty = T_REAL(__))     then '(<%e1%> > <%e2%>)'
     case GREATER(ty = T_ENUMERATION(__))   then '(<%e1%> > <%e2%>)'
     
    case LESSEQ(ty = T_BOOL(__))      then '(!<%e1%> || <%e2%>)'
    case LESSEQ(ty = T_STRING(__))    then "# string comparison not supported\n"
    case LESSEQ(ty = T_INTEGER(__))
    case LESSEQ(ty = T_REAL(__))       then '(<%e1%> <= <%e2%>)'
    case LESSEQ(ty = T_ENUMERATION(__))    then '(<%e1%> <= <%e2%>)'
    
    case GREATEREQ(ty = T_BOOL(__))   then '(<%e1%> || !<%e2%>)'
    case GREATEREQ(ty = T_STRING(__)) then "# string comparison not supported\n"
    case GREATEREQ(ty = T_INTEGER(__))
    case GREATEREQ(ty = T_REAL(__))   then '(<%e1%> >= <%e2%>)'
     case GREATEREQ(ty = T_ENUMERATION(__)) then '(<%e1%> >= <%e2%>)'
     
    case EQUAL(ty = T_BOOL(__))       then '((!<%e1%> && !<%e2%>) || (<%e1%> && <%e2%>))'
    case EQUAL(ty = T_STRING(__))
    case EQUAL(ty = T_INTEGER(__))
    case EQUAL(ty = T_REAL(__))       then '(<%e1%> == <%e2%>)'
    case EQUAL(ty = T_ENUMERATION(__))     then '(<%e1%> == <%e2%>)'   
     
    case NEQUAL(ty = T_BOOL(__))      then '((!<%e1%> && <%e2%>) || (<%e1%> && !<%e2%>))'
    case NEQUAL(ty = T_STRING(__))
    case NEQUAL(ty = T_INTEGER(__))
    case NEQUAL(ty = T_REAL(__))      then '(<%e1%> != <%e2%>)'
    case NEQUAL(ty = T_ENUMERATION(__))    then '(<%e1%> != <%e2%>)'
    
    case _                            then "daeExpRelation:ERR"
      end match
  case _ then 
     match op
    case LESS(ty = T_BOOL(__))        then '_conditions1[<%index%>]'
    case LESS(ty = T_STRING(__))      then "# string comparison not supported\n"
    case LESS(ty = T_INTEGER(__))
    case LESS(ty = T_REAL(__))        then '_conditions1[<%index%>]'
    
    case GREATER(ty = T_BOOL(__))     then '_conditions1[<%index%>]'
    case GREATER(ty = T_STRING(__))   then "# string comparison not supported\n"
    case GREATER(ty = T_INTEGER(__))
    case GREATER(ty = T_REAL(__))     then '_conditions1[<%index%>]'
    
    case LESSEQ(ty = T_BOOL(__))      then '_conditions1[<%index%>]'
    case LESSEQ(ty = T_STRING(__))    then "# string comparison not supported\n"
    case LESSEQ(ty = T_INTEGER(__))
    case LESSEQ(ty = T_REAL(__))       then '_conditions1[<%index%>]'
    
    case GREATEREQ(ty = T_BOOL(__))   then '_conditions1[<%index%>]'
    case GREATEREQ(ty = T_STRING(__)) then "# string comparison not supported\n"
    case GREATEREQ(ty = T_INTEGER(__))
    case GREATEREQ(ty = T_REAL(__))   then '_conditions1[<%index%>]'
    
    case EQUAL(ty = T_BOOL(__))       then '_conditions1[<%index%>]'
    case EQUAL(ty = T_STRING(__))
    case EQUAL(ty = T_INTEGER(__))
    case EQUAL(ty = T_REAL(__))       then '_conditions1[<%index%>]'
    
    case NEQUAL(ty = T_BOOL(__))      then '_conditions1[<%index%>]'
    case NEQUAL(ty = T_STRING(__))
    case NEQUAL(ty = T_INTEGER(__))
    case NEQUAL(ty = T_REAL(__))      then '_conditions1[<%index%>]'
    case _                         then "daeExpRelationCondition:ERR"
      end match
end daeExpRelation;


template daeExpIf(Exp cond, Exp then_, Exp else_, Context context, Text &preExp, Text &varDecls,SimCode simCode) ::=
  let condExp = daeExp(cond, context, &preExp, &varDecls,simCode)
  let &preExpThen = buffer ""
  let eThen = daeExp(then_, context, &preExpThen, &varDecls,simCode)
  let &preExpElse = buffer ""
  let eElse = daeExp(else_, context, &preExpElse /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let condVar = tempDecl("bool", &varDecls /*BUFD*/)
      let resVarType = expTypeFromExpArrayIf(then_)
      let resVar = tempDecl(resVarType, &varDecls /*BUFD*/)
      let &preExp +=  
      <<
      <%condVar%> = <%condExp%>;
      if (<%condVar%>) {
        <%preExpThen%>
        <%resVar%> = <%eThen%>;
      } else {
        <%preExpElse%>
        <%resVar%> = <%eElse%>;
      }<%\n%>
      >>
      resVar
end daeExpIf;

template expTypeFromExpArrayIf(Exp exp) ::=
  expTypeFromExp(exp)
end expTypeFromExpArrayIf;

template expTypeFromExp(Exp it) ::=
  match it
  case ICONST(__)    then "int"
  case RCONST(__)    then "double"
  case SCONST(__)    then "string"
  case BCONST(__)    then "bool"
  case BINARY(__)
  case UNARY(__)
  case LBINARY(__)
  case LUNARY(__)     then expTypeFromOp(operator)
  case RELATION(__)   then "bool" //TODO: a HACK, it was expTypeFromOp(operator)
  case IFEXP(__)      then expTypeFromExp(expThen)
  case CALL(attr=CALL_ATTR(__))       then expTypeShort(attr.ty)
  case ARRAY(__)
  case MATRIX(__)
  case RANGE(__)
  case CAST(__)
  case CREF(__)
  case CODE(__)       then expTypeShort(ty)
  case ASUB(__)       then expTypeFromExp(exp)
  case REDUCTION(__)  then expTypeFromExp(expr)
  case _          then "expTypeFromExp:ERROR"
end expTypeFromExp;


template expTypeFromOp(Operator it) ::=
  match it
  case ADD(__)
  case SUB(__)
  case MUL(__)
  case DIV(__)
  case POW(__)
  case UMINUS(__)
  case UMINUS_ARR(__)
  case ADD_ARR(__)
  case SUB_ARR(__)
  case MUL_ARR(__)
  case DIV_ARR(__)
  case MUL_ARRAY_SCALAR(__)
  case ADD_ARRAY_SCALAR(__)
  case SUB_SCALAR_ARRAY(__)
  case MUL_SCALAR_PRODUCT(__)
  case MUL_MATRIX_PRODUCT(__)
  case DIV_ARRAY_SCALAR(__)
  case DIV_SCALAR_ARRAY(__)
  case POW_ARRAY_SCALAR(__)
  case POW_SCALAR_ARRAY(__)
  case POW_ARR(__)
  case POW_ARR2(__)
  case LESS(__)
  case LESSEQ(__)
  case GREATER(__)
  case GREATEREQ(__)
  case EQUAL(__)
  case NEQUAL(__)       then  expTypeShort(ty)
  case AND(__)
  case OR(__)
  case NOT(__) then "bool"
  case _ then "expTypeFromOp:ERROR"
end expTypeFromOp;

template equationAlgorithm(SimEqSystem eq, Context context,Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates an equation that is an algorithm."
::=
match eq
case SES_ALGORITHM(__) then
  (statements |> stmt =>
    algStatement(stmt, context, &varDecls /*BUFD*/,simCode)
  ;separator="\n") 
end equationAlgorithm;

template algStatement(DAE.Statement it, Context context, Text &varDecls,SimCode simCode) ::=
  match it
  case STMT_ASSIGN(exp1 = CREF(componentRef = WILD(__)), exp = e) then
    let &preExp = buffer "" 
    let expPart = daeExp(e, context, &preExp, &varDecls, simCode)
    <<
    <%preExp%>
    <%expPart%>
    >>
  case STMT_ASSIGN(exp1 = CREF(__)) then
    let &preExp = buffer ""
    let expPart = daeExp(exp, context, &preExp,&varDecls, simCode)
    <<
    <%preExp%>
    <%scalarLhsCref(exp1, context, &preExp, &varDecls, simCode)%> = <%expPart%>;
    >>  
  case STMT_ASSIGN(__) then
    let &preExp = buffer ""
    let expPart1 = daeExp(exp1, context, &preExp, &varDecls,simCode)
    let expPart2 = daeExp(exp, context, &preExp, &varDecls,simCode)
    <<
    <%preExp%>
    <%expPart1%> = <%expPart2%>;
    >>  
  case STMT_ASSIGN_ARR(componentRef = CREF_IDENT(subscriptLst=subs as (_ :: _))) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp,&varDecls, simCode)
     let spec = daeExpCrefRhsIndexSpec(subs, context, &preExp, &varDecls, simCode)
     <<
     <%preExp%>
     <%componentRef.ident%>.AssignSpec(<%spec%>, <%expPart%>.A);
     >>
  case STMT_ASSIGN_ARR(__) then
     let &preExp = buffer ""
     let expPart = daeExp(exp, context, &preExp, &varDecls,simCode)
     <<
     <%preExp%>
     <%contextCref(componentRef, context,simCode)%>=<%expPart%>;
     >>  
  case STMT_IF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp,&varDecls, simCode)
    <<
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context,&varDecls,simCode) ;separator="\n"%>
    }
    <%elseExpr(else_, context,&preExp, &varDecls, simCode)%>
    >>

  case STMT_FOR(range=rng as RANGE(__)) then
    let identType = expType(type_, iterIsArray) //TODO: ?? what is this for ... no array typed iterator is possible ???
    let identTypeShort = expTypeShort(type_)
    let stmtStr = (statementLst |> stmt => algStatement(stmt, context, &varDecls,simCode)
                   ;separator="\n")
    algStmtForRange_impl(rng, iter, identType, identTypeShort, stmtStr, context, &varDecls,simCode)
  
  case STMT_WHILE(__)  then
    let &preExp = buffer ""
    let var = daeExp(exp, context, &preExp,&varDecls, simCode)
    <<
    while (1) {
    <%preExp%>
    if (!<%var%>) break;
      <%statementLst |> stmt => algStatement(stmt, context,&varDecls,simCode) ;separator="\n"%>
    }
    >>
     
  case s as STMT_TUPLE_ASSIGN(__)   then algStmtTupleAssign(s, context, &varDecls /*BUFD*/,simCode)
  case STMT_ASSERT(__)         then 
   let &preExp = buffer "" 
   let condExp = daeExp(cond, context, &preExp,&varDecls, simCode)
   let msgExp = daeExp(msg, context, &preExp,&varDecls, simCode)
  'Assert(<%condExp%>,<%msgExp%>);'
  
  case STMT_TERMINATE(__)      then "STMT_TERMINATE_NI"
  case STMT_WHEN(__)           then algStmtWhen(it, context,&varDecls, simCode)
  case STMT_FOR(__)            
     then 
   algStmtForGeneric(it, context,&varDecls, simCode)
  case STMT_BREAK(__)          then 'break; //break stmt<%\n%>'
  case STMT_FAILURE(__)        then "STMT_FAILURE_NI"
  case STMT_TRY(__)            then "STMT_TRY_NI"
  case STMT_CATCH(__)          then "STMT_CATCH_NI"
  case STMT_THROW(__)          then "STMT_THROW_NI"
  case STMT_RETURN(__)         then "STMT_RETURN_NI"
  case STMT_NORETCALL(__)      then "STMT_NORETCALL_NI"
  case STMT_REINIT(__)         then "STMT_REINIT_NI"
  case _ then "NOT_IMPLEMENTED_ALG_STATEMENT"
      
end algStatement;



template algStmtTupleAssign(DAE.Statement stmt, Context context,
                   Text &varDecls /*BUFP*/, SimCode simCode)
 "Generates a tuple assigment algorithm statement."
::=
match stmt
case STMT_TUPLE_ASSIGN(exp=CALL(__)) then
  let &preExp = buffer "" /*BUFD*/
  let &afterExp = buffer "" /*BUFD*/
  let crefs = (expExpLst |> e => ExpressionDump.printExpStr(e) ;separator=", ")
  let marker = '(<%crefs%>) = <%ExpressionDump.printExpStr(exp)%>'
  let &preExp += '/* algStmtTupleAssign: preExp buffer created for <%marker%> */<%\n%>' 
  let &afterExp += '/* algStmtTupleAssign: afterExp buffer created for <%marker%> */<%\n%>'
  let retStruct = daeExp(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/, simCode)
  let lhsCrefs = (expExpLst |> cr hasindex i1 fromindex 0 =>
                    let rhsStr = 'get<<%i1%>>(<%retStruct%>)'
                    writeLhsCref(cr, rhsStr, context, &afterExp /*BUFC*/, &varDecls /*BUFD*/ , simCode)
                  ;separator="\n")
  <<
  /* algStmtTupleAssign: preExp printout <%marker%>*/
  <%preExp%>
  /* algStmtTupleAssign: writeLhsCref <%marker%> */
  <%lhsCrefs%>
  /* algStmtTupleAssign: afterExp printout <%marker%> */
  <%afterExp%>
  >>

else error(sourceInfo(), 'algStmtTupleAssign failed')
end algStmtTupleAssign;



template error(Absyn.Info srcInfo, String errMessage)
"Example source template error reporting template to be used together with the sourceInfo() magic function.
Usage: error(sourceInfo(), <<message>>) "
::=
let() = Tpl.addSourceTemplateError(errMessage, srcInfo)
<<

#error "<% Error.infoStr(srcInfo) %> <% errMessage %>"<%\n%>
>>
end error;

//for completeness; although the error() template above is preferable
template errorMsg(String errMessage)
"Example template error reporting template
 that is reporting only the error message without the usage of source infotmation."
::=
let() = Tpl.addTemplateError(errMessage)
<<

#error "<% errMessage %>"<%\n%>
>>
end errorMsg;

template algStmtForGeneric(DAE.Statement stmt, Context context, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates a for algorithm statement where range is not RANGE."
::=
match stmt
case STMT_FOR(__) then
  let iterType = expType(type_, iterIsArray)
  let arrayType = expTypeArray(type_)


  let stmtStr = (statementLst |> stmt => 
    algStatement(stmt, context, &varDecls,simCode) ;separator="\n")
  algStmtForGeneric_impl(range, iter, iterType, arrayType, iterIsArray, stmtStr, 
    context, &varDecls,simCode)
end algStmtForGeneric;

template algStmtForGeneric_impl(Exp exp, Ident iterator, String type, 
  String arrayType, Boolean iterIsArray, Text &body, Context context, Text &varDecls,SimCode simCode)
 "The implementation of algStmtForGeneric, which is also used by daeExpReduction."
::=
  let iterName = contextIteratorName(iterator, context)
  //let stateVar = if not acceptMetaModelicaGrammar() then tempDecl("state", &varDecls)
  //let tvar = tempDecl("int", &varDecls)
  //let ivar = tempDecl(type, &varDecls)
  let &preExp = buffer ""
  let &tmpVar = buffer ""
  let evar = daeExp(exp, context, &preExp, &tmpVar,simCode)
  //let stmtStuff = if iterIsArray then
  //    'simple_index_alloc_<%type%>1(&<%evar%>, <%tvar%>, &<%ivar%>);'
  //  else
  //    '<%iterName%> = *(<%arrayType%>_element_addr1(&<%evar%>, 1, <%tvar%>));'
  <<
  <%preExp%>
    <%type%> <%iterName%>;
   BOOST_FOREACH( short <%iterName%>,  <%evar%> ){ 
      <%body%> 
    }  
  >>

end algStmtForGeneric_impl;

template contextIteratorName(Ident name, Context context)
  "Generates code for an iterator variable."
::=
  match context
  case FUNCTION_CONTEXT(__) then "_" + name
  else name
end contextIteratorName;


template algStmtWhen(DAE.Statement when, Context context, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates a when algorithm statement."
::=
match context
case SIMULATION(genDiscrete=true) then
  match when
  case STMT_WHEN(__) then
    <<
    
    if (<%helpVarIndices |> idx => edgeHelpVar(idx) ;separator=" || "%>) {
      <% statementLst |> stmt =>  algStatement(stmt, context, &varDecls,simCode)
         ;separator="\n" %>
    }
    <%algStatementWhenElse(elseWhen, &varDecls,simCode)%>
    >>
  end match
end algStmtWhen;

template algStatementWhenElse(Option<DAE.Statement> stmt, Text &varDecls /*BUFP*/,SimCode simCode)
 "Helper to algStmtWhen."
::=
match stmt
case SOME(when as STMT_WHEN(__)) then
  let elseCondStr = (when.helpVarIndices |> idx => edgeHelpVar(idx)
                      ;separator=" || ")
  <<
  else if (<%elseCondStr%>) {
    <% when.statementLst |> stmt =>  algStatement(stmt, contextSimulationDiscrete,&varDecls,simCode)
       ;separator="\n"%>
  }
  <%algStatementWhenElse(when.elseWhen, &varDecls,simCode)%>
  >>
end algStatementWhenElse;

template edgeHelpVar(String idx) ::=
   '_event_handling.edge(_event_handling[<%idx%>],"h<%idx%>")'
end edgeHelpVar;

template algStmtForRange_impl(Exp range, Ident iterator, String type, String shortType, Text body, Context context, Text &varDecls,SimCode simCode)
 "The implementation of algStmtForRange, which is also used by daeExpReduction."
::=
match range
case RANGE(__) then
  let iterName = iterator
  let &stopVar = buffer ""
  let &preExp = buffer ""
  let startValue = daeExp(exp, context, &preExp, &varDecls,simCode)
  let stopValue = daeExp(range, context, &preExp, &varDecls,simCode)
  let res = tempDecl(type, &varDecls /*BUFC*/)
  match expOption 
  case SOME(eo) then
    let &stepVar = buffer "" 
    let stepValue = daeExp(eo, context, &preExp,&varDecls,simCode)
    <<
    <%preExp%>
    for(<%type%> <%iterName%> = <%startValue%>;(<%stepValue%> > 0? <%iterName%><=<%stopValue%> : <%stopValue%><=<%iterName%>); <%iterName%> += <%stepValue%>) { 
      <%body%>
    }
    >>
  else //optimization for 1 step
    <<
    <%preExp%>
    for(<%type%> <%iterName%> = <%startValue%>; <%iterName%><=<%stopValue%>; <%iterName%> += 1) { 
      <%body%>
    }
    >>
end algStmtForRange_impl;


template writeLhsCref(Exp exp, String rhsStr, Context context, Text &preExp /*BUFP*/,
              Text &varDecls /*BUFP*/, SimCode simCode)
 "Generates code for writing a returnStructur to var."
::=
match exp
case ecr as CREF(componentRef=WILD(__)) then
  ''
case CREF(ty= t as DAE.T_ARRAY(__)) then
  let lhsStr = scalarLhsCref(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match context
  case SIMULATION(__) then
    <<
    copy_<%expTypeShort(t)%>_array_data_mem(&<%rhsStr%>, &<%lhsStr%>);
    >>
  else
    '<%lhsStr%> = <%rhsStr%>;'
case UNARY(exp = e as CREF(ty= t as DAE.T_ARRAY(__))) then
  let lhsStr = scalarLhsCref(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  match context
  case SIMULATION(__) then
    <<
    usub_<%expTypeShort(t)%>_array(&<%rhsStr%>);<%\n%>
    copy_<%expTypeShort(t)%>_array_data_mem(&<%rhsStr%>, &<%lhsStr%>);
    >>
  else
    '<%lhsStr%> = -<%rhsStr%>;'
case CREF(__) then
  let lhsStr = scalarLhsCref(exp, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  <<
  <%lhsStr%> = <%rhsStr%>;
  >>   
case UNARY(exp = e as CREF(__)) then
  let lhsStr = scalarLhsCref(e, context, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
  <<
  <%lhsStr%> = -<%rhsStr%>;
  >>
case _ then 
  <<
  /* SimCodeC.tpl template: writeLhsCref: UNHANDLED LHwwS 
   * <%ExpressionDump.printExpStr(exp)%> = <%rhsStr%> 
   */
  >>
end writeLhsCref;


template scalarLhsCref(Exp ecr, Context context, Text &preExp,Text &varDecls, SimCode simCode) ::=
match ecr
case ecr as CREF(componentRef=CREF_IDENT(subscriptLst=subs)) then
  if crefNoSub(ecr.componentRef) then
    contextCref(ecr.componentRef, context,simCode)
  else
    daeExpCrefRhs(ecr, context, &preExp, &varDecls, simCode)
case ecr as CREF(componentRef=CREF_QUAL(__)) then
    contextCref(ecr.componentRef, context,simCode)
else
    "ONLY_IDENT_OR_QUAL_CREF_SUPPORTED_SLHS"
end scalarLhsCref;






template elseExpr(DAE.Else it, Context context, Text &preExp, Text &varDecls,SimCode simCode) ::= 
  match it
  case NOELSE(__) then ""
  case ELSEIF(__) then
    let &preExp = buffer ""
    let condExp = daeExp(exp, context, &preExp, &varDecls,simCode)
    <<
    else {
    <%preExp%>
    if (<%condExp%>) {
      <%statementLst |> it => algStatement(it, context, &varDecls,simCode) 
      ;separator="\n"%>
    }
    <%elseExpr(else_, context, &preExp, &varDecls,simCode)%>
    }
    >>
  case ELSE(__) then
    <<
    else {
      <%statementLst |> it => algStatement(it, context, &varDecls,simCode) 
      ;separator="\n"%>
    }
    >>
end elseExpr;


template expType(DAE.Type ty, Boolean isArray)
 "Generate type helper."
::=
  if isArray 
  then 'expType_<%expTypeArray1(ty,0)%>_NOT_YET'
  else expTypeShort(ty)
end expType;


template expTypeArrayIf(DAE.Type ty)
 "Generate type helper."
::=
  expTypeFlag(ty, 4)
end expTypeArrayIf;

template expTypeArray1(DAE.Type ty, Integer dims) ::=
<<
 SimArray<%dims%><<%expTypeShort(ty)%>>
>>
end expTypeArray1;


template functionOnlyZeroCrossing(list<ZeroCrossing> zeroCrossings,Text& varDecls,SimCode simCode)
  "Generates function in simulation file."
::=
  
  let zeroCrossingsCode = zeroCrossingsTpl2(zeroCrossings, &varDecls /*BUFD*/, simCode)
  <<
    
    <%zeroCrossingsCode%> 
  >>
end functionOnlyZeroCrossing;


template zeroCrossingsTpl2(list<ZeroCrossing> zeroCrossings, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for zero crossings."
::=

  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    zeroCrossingTpl2(i0, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end zeroCrossingsTpl2;

/*
template zeroCrossingTpl2(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a zero crossing."
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    let &preExp = buffer "" /*BUFD*/
    let e1 = daeExp(exp1, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let op = zeroCrossingOpFunc(operator)
    let e2 = daeExp(exp2, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let res = tempDecl("bool", &varDecls /*BUFC*/)
    <<
    <%preExp%>
    <%res%>=(<%e1%><%op%><%e2%>);
    _condition<%zerocrossingIndex%>=<%res%>;
    >>
end zeroCrossingTpl2;
*/


template zeroCrossingTpl2(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates code for a zero crossing."
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    <<
    checkConditions(<%zerocrossingIndex%>,false);
    >>
end zeroCrossingTpl2;


template timeEventCondition(list<SampleCondition> sampleConditions,Text &varDecls, SimCode simCode)
::=
  
  let timeEventConditionCode = timeEventcondition(sampleConditions, &varDecls /*BUFD*/, simCode)
  <<
  
    <%timeEventConditionCode%>
  >>
end timeEventCondition;


template timeEventcondition(list<SampleCondition> sampleConditions,Text &varDecls /*BUFP*/,SimCode simCode)
::=
  (sampleConditions |> (relation_,index)  =>
    timeEventcondition1(index, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end timeEventcondition;

template timeEventcondition1(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case CALL(path=IDENT(name="sample"), expLst={start, interval,index}) then
    let &preExp = buffer "" /*BUFD*/
    let eStart = daeExp(start, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
    let eInterval = daeExp(interval, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
    let eIndex = daeExp(index, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
    let res = tempDecl("bool", &varDecls /*BUFC*/)
     <<
     <%preExp%>
     <%res%>= false;
     _condition<%eIndex%> = <%res%>;
      event_times_type sample<%eIndex%> = _event_handling.makePeriodeEvents(<%eStart%>,te,<%eInterval%>,<%eIndex%>);
     _event_handling.addTimeEvents(sample<%eIndex%>);
    >>
end timeEventcondition1;

template resetTimeEvents(list<SampleCondition> sampleConditions,list<SimWhenClause> whenClauses,SimCode simCode)
::=
match simCode
case SIMCODE(sampleConditions=sam) then
  << 
     <%resetTimeEvent(sampleConditions,simCode)%>
     double h[<%helpvarlength(simCode)%>];
     <%helpvarvector(whenClauses,simCode)%>
      _event_handling.setHelpVars(h);
  >> 
end resetTimeEvents;

template resetTimeEvent(list<SampleCondition> sampleConditions,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  (sampleConditions |> (relation_,index)  =>
    resetTimeEvent1(index, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end resetTimeEvent;

template resetTimeEvent1(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case CALL(path=IDENT(name="sample"), expLst={start, interval,index}) then
    let &preExp = buffer "" /*BUFD*/
    let eIndex = daeExp(index, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
     <<
     _condition<%eIndex%> = false;
    >>
end resetTimeEvent1;

template handleEvent(list<SampleCondition> sampleConditions,SimCode simCode)
::=
match simCode
case SIMCODE(__) then
  << 
     switch(index)    
    {
     <%handleEvent1(sampleConditions,simCode)%>
    }
  >>
 
end handleEvent;

template handleEvent1(list<SampleCondition> sampleConditions,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  (sampleConditions |> (relation_,index)  =>
    handleEvent2(index, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end handleEvent1;

template handleEvent2(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case CALL(path=IDENT(name="sample"), expLst={start, interval,index}) then
    let &preExp = buffer "" /*BUFD*/
    let eIndex = daeExp(index, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
     <<
     case <%eIndex%>:
      {
       _condition<%eIndex%> = true;
       break;
      }
    >>
end handleEvent2;

template checkConditions(list<ZeroCrossing> zeroCrossings,list<SimWhenClause> whenClauses,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let zeroCrossingsCode = checkConditions1(zeroCrossings, &varDecls /*BUFD*/, simCode)
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then 
<<
   void <%lastIdentOfPath(modelInfo.name)%>::checkConditions(unsigned int index, bool all)
   { 
      <%varDecls%>
      <%zeroCrossingsCode%>      
   }
>>
end checkConditions;

template checkConditions1(list<ZeroCrossing> zeroCrossings, Text &varDecls /*BUFP*/,SimCode simCode)
::=

  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    checkConditions2(i0, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end checkConditions1;

template checkConditions2(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    let &preExp = buffer "" /*BUFD*/
    let e1 = daeExp(exp1, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let op = zeroCrossingOpFunc(operator)
    let e2 = daeExp(exp2, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    let res = tempDecl("bool", &varDecls /*BUFC*/)
    <<
    <%preExp%>
    if(index==<%zerocrossingIndex%> || all)
    {
      <%res%>=(<%e1%><%op%><%e2%>);
      _conditions1[<%zerocrossingIndex%>]=<%res%>;
    }
   >>
end checkConditions2;

template handleSystemEvents(list<ZeroCrossing> zeroCrossings,list<SimWhenClause> whenClauses,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let zeroCrossingsCode = handleSystemEvents1(zeroCrossings, &varDecls /*BUFD*/, simCode)
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then 
  <<
   void <%lastIdentOfPath(modelInfo.name)%>::handleSystemEvents(const bool* events)
   { 
    <%varDecls%>
    bool restart=true;
    int iter=0;
    <%zeroCrossingsCode%>
     while(restart && !(iter++ > _dimZeroFunc))
    {
        double h[<%helpvarlength(simCode)%>];
        <%helpvarvector(whenClauses,simCode)%>
             _event_handling.setHelpVars(h);
            //iterate and handle all events inside the eventqueue
            restart=_event_handling.IterateEventQueue(_conditions1);
           
     }
     saveConditions();
     resetTimeEvents();
    if(iter>_dimZeroFunc && restart ){
     throw std::runtime_error("Number of event iteration steps exceeded. ");}
   }
  >>
end handleSystemEvents;

template handleSystemEvents1(list<ZeroCrossing> zeroCrossings, Text &varDecls /*BUFP*/,SimCode simCode)
::=

  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    handleSystemEvents2(i0, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end handleSystemEvents1;

template handleSystemEvents2(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    let &preExp = buffer "" /*BUFD*/
    <<
    <%preExp%>
    if(events[<%zerocrossingIndex%>])
    {
      checkConditions(<%zerocrossingIndex%>,false);
      _event_handling.addEvent(<%zerocrossingIndex%>);
      handleEvent(<%zerocrossingIndex%>);
    }
   >>
 end handleSystemEvents2;

template zeroCrossingOpFunc(Operator op)
 "Generates zero crossing function name for operator."
::=
  match op
  case LESS(__)      then "<"
  case GREATER(__)   then ">"
  case LESSEQ(__)    then "<="
  case GREATEREQ(__) then ">="
  case EQUAL(__)     then "=="
  case NEQUAL(__)    then "!="
end zeroCrossingOpFunc;

template giveZeroFunc1(list<ZeroCrossing> zeroCrossings,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let zeroCrossingsCode = giveZeroFunc2(zeroCrossings, &varDecls /*BUFD*/, simCode)
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
<<
 void <%lastIdentOfPath(modelInfo.name)%>::giveZeroFunc(double* f)
  {
   <%varDecls%>
   <%zeroCrossingsCode%>
  }   
>>
end giveZeroFunc1;


template giveConditions(SimCode simCode)
::=
 match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
<<
 void <%lastIdentOfPath(modelInfo.name)%>::giveConditions(bool* c)
  {
    memcpy(c,_conditions0,_dimZeroFunc*sizeof(bool));
  }   
>>
end giveConditions;

template setConditions(SimCode simCode)
::=
 match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
<<
 void <%lastIdentOfPath(modelInfo.name)%>::setConditions(bool* c)
  {
    memcpy(_conditions0,c,_dimZeroFunc*sizeof(bool));
  }   
>>
end setConditions;

template saveConditions(SimCode simCode)
::=
 match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
<<
 void <%lastIdentOfPath(modelInfo.name)%>::saveConditions()
  {
    SystemDefaultImplementation::saveConditions();
  }   
>>
end saveConditions;

template giveZeroFunc2(list<ZeroCrossing> zeroCrossings, Text &varDecls /*BUFP*/,SimCode simCode)
::=

  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    giveZeroFunc3(i0, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end giveZeroFunc2;

template giveZeroFunc3(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
 let &preExp = buffer "" /*BUFD*/
  match relation
  case rel as  RELATION(index=zerocrossingIndex) then
       let e1 = daeExp(exp1, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let e2 = daeExp(exp2, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
       match rel.operator
        
        case LESS(__) 
        case LESSEQ(__) then
       <<
         if(_conditions0[<%zerocrossingIndex%>])
                f[<%index1%>]=(<%e1%>-EPSILON-<%e2%>);
           else
                f[<%index1%>]=(<%e2%>-<%e1%>-EPSILON);
      >>
      case GREATER(__)
      case GREATEREQ(__) then
        <<
         if(_conditions0[<%zerocrossingIndex%>])
                f[<%index1%>]=(<%e2%>-<%e1%>-EPSILON);
           else
                f[<%index1%>]=(<%e1%>-EPSILON-<%e2%>);
         >>
     end match  
end giveZeroFunc3;

/*
template giveZeroFunc3(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
 let &preExp = buffer "" /*BUFD*/
  match relation
  case rel as  RELATION(index=zerocrossingIndex) then
       let e1 = daeExp(exp1, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let e2 = daeExp(exp2, contextOther, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
       match rel.operator
        
        case LESS(__) 
        case LESSEQ(__) then
       <<
         if(_event_handling.pre(_condition<%zerocrossingIndex%>,"_condition<%zerocrossingIndex%>"))
                f[<%index1%>]=(<%e1%>-EPSILON-<%e2%>);
           else
                f[<%index1%>]=(<%e2%>-<%e1%>-EPSILON);
      >>
      case GREATER(__)
      case GREATEREQ(__) then
        <<
         if(_event_handling.pre(_condition<%zerocrossingIndex%>,"_condition<%zerocrossingIndex%>"))
                f[<%index1%>]=(<%e2%>-<%e1%>-EPSILON);
           else
                f[<%index1%>]=(<%e1%>-EPSILON-<%e2%>);
         >>
     end match  
end giveZeroFunc3;
*/

template conditionvarZero(list<ZeroCrossing> zeroCrossings,SimCode simCode)
::=
  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    conditionvarZero1(i0, relation_, simCode)
  ;separator="\n")
end conditionvarZero;

template conditionvarZero1(Integer index1, Exp relation,SimCode simCode)
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    <<
     bool _condition<%zerocrossingIndex%>;
    >>
end conditionvarZero1;

template saveconditionvar(list<ZeroCrossing> zeroCrossings,SimCode simCode)
::=
  (zeroCrossings |> ZERO_CROSSING(__) hasindex i0 =>
    saveconditionvar1(i0, relation_, simCode)
  ;separator="\n")
end saveconditionvar;

template saveconditionvar1(Integer index1, Exp relation,SimCode simCode)
::=
  match relation
  case RELATION(index=zerocrossingIndex) then
    <<
     _event_handling.save(_condition<%zerocrossingIndex%>,"_condition<%zerocrossingIndex%>");
    >>
end saveconditionvar1;



template conditionvarSample(list<SampleCondition> sampleConditions,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  (sampleConditions |> (relation_,index)  =>
    conditionvarSample1(index, relation_, &varDecls /*BUFD*/,simCode)
  ;separator="\n")
end conditionvarSample;

template conditionvarSample1(Integer index1, Exp relation, Text &varDecls /*BUFP*/,SimCode simCode)
::=
  match relation
  case CALL(path=IDENT(name="sample"), expLst={start, interval,index}) then
    let &preExp = buffer "" /*BUFD*/
    let eIndex = daeExp(index, contextOther, &preExp, &varDecls /*BUFD*/,simCode)
     <<
     bool _condition<%eIndex%>;
    >>
end conditionvarSample1;

template conditionvariable(list<ZeroCrossing> zeroCrossings,list<SampleCondition> sampleConditions,SimCode simCode) 
::=
  let conditionvariable = conditionvarZero(zeroCrossings,simCode)
  let conditionvarsample = conditionvarSample(sampleConditions,simCode)
  /*
  <<
   <%conditionvariable%>
   <%conditionvarsample%>
  >>
  */
  <<
    <%conditionvarsample%>
  >>
end conditionvariable;

template expTypeFromExpFlag(Exp exp, Integer flag)
::=
  match exp
  case ICONST(__)        then match flag case 8 then "int" case 1 then "int" else "int"
  case RCONST(__)        then match flag case 1 then "double" else "double"
  case SCONST(__)        then if acceptMetaModelicaGrammar() then
                                (match flag case 1 then "metatype" else "modelica_metatype")
                              else
                                (match flag case 1 then "string" else "modelica_string")
  case BCONST(__)        then match flag case 1 then "bool" else "modelica_boolean"
  case ENUM_LITERAL(__)  then match flag case 8 then "int" case 1 then "int" else "int"
  case e as BINARY(__)
  case e as UNARY(__)
  case e as LBINARY(__)
  case e as LUNARY(__)
  case e as RELATION(__) then expTypeFromOpFlag(e.operator, flag)
  case IFEXP(__)         then expTypeFromExpFlag(expThen, flag)
  case CALL(attr=CALL_ATTR(__))          then expTypeFlag(attr.ty, flag)
  case c as ARRAY(__)
  case c as MATRIX(__)
  case c as RANGE(__)
  case c as CAST(__)
  case c as CREF(__)
  case c as CODE(__)     then expTypeFlag(c.ty, flag)
  case ASUB(__)          then expTypeFromExpFlag(exp, flag)
  case REDUCTION(__)     then expTypeFlag(typeof(exp), flag)
  case BOX(__)
  case CONS(__)
  case LIST(__)

  case META_TUPLE(__)
  case META_OPTION(__)
  case MATCHEXPRESSION(__)
  case METARECORDCALL(__)
  case BOX(__)           then match flag case 1 then "metatype" else "modelica_metatype"
  case c as UNBOX(__)    then expTypeFlag(c.ty, flag)
  case c as SHARED_LITERAL(__) then expTypeFlag(c.ty, flag)
  else ""
end expTypeFromExpFlag;

template expTypeFromOpFlag(Operator op, Integer flag)
 "Generate type helper."
::=
  match op
  case o as ADD(__)
  case o as SUB(__)
  case o as MUL(__)
  case o as DIV(__)
  case o as POW(__)

  case o as UMINUS(__)
  case o as UMINUS_ARR(__)
  case o as ADD_ARR(__)
  case o as SUB_ARR(__)
  case o as MUL_ARR(__)
  case o as DIV_ARR(__)
  case o as MUL_ARRAY_SCALAR(__)
  case o as ADD_ARRAY_SCALAR(__)
  case o as SUB_SCALAR_ARRAY(__)
  case o as MUL_SCALAR_PRODUCT(__)
  case o as MUL_MATRIX_PRODUCT(__)
  case o as DIV_ARRAY_SCALAR(__)
  case o as DIV_SCALAR_ARRAY(__)
  case o as POW_ARRAY_SCALAR(__)
  case o as POW_SCALAR_ARRAY(__)
  case o as POW_ARR(__)
  case o as POW_ARR2(__)
  case o as LESS(__)
  case o as LESSEQ(__)
  case o as GREATER(__)
  case o as GREATEREQ(__)
  case o as EQUAL(__)
  case o as NEQUAL(__) then
    expTypeFlag(o.ty, flag)
  case o as AND(__)
  case o as OR(__)
  case o as NOT(__) then
    match flag case 1 then "bool" else "modelica_boolean"
  else "expTypeFromOpFlag:ERROR"
end expTypeFromOpFlag;

template checkForDiscreteEvents(list<ComponentRef> discreteModelVars,SimCode simCode)
::=

  let changediscreteVars = (discreteModelVars |> var => match var case CREF_QUAL(__) case CREF_IDENT(__) then
       'if (_event_handling.change(<%cref(var)%>,"<%cref(var)%>")) {  restart=true; }'
       ;separator="\n")
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then
  <<
  bool <%lastIdentOfPath(modelInfo.name)%>::checkForDiscreteEvents()
  {
    bool restart = false;
    <%changediscreteVars%>
    return restart;
  }
  >>
end checkForDiscreteEvents;
/*
template update(list<list<SimEqSystem>> continousEquations,list<SimEqSystem> discreteEquations,list<SimWhenClause> whenClauses,list<SimEqSystem> parameterEquations,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let continous = (continousEquations |> eqs => (eqs |> eq =>
      equation_(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator="\n")
  let paraEquations = (parameterEquations |> eq =>
      equation_(eq, contextOther, &varDecls /*BUFD*/,simCode)
    ;separator="\n") 
  let discrete = (discreteEquations |> eq =>
      equation_(eq, contextSimulationDiscrete, &varDecls /*BUFC*/,simCode)
    ;separator="\n")
  let reinit = (whenClauses |> when hasindex i0 =>
         genreinits(when, &varDecls,i0,simCode)
    ;separator="\n") 
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
  <<
  void <%lastIdentOfPath(modelInfo.name)%>::update(const UPDATE command)
  {
    <%varDecls%>
   if(command & CONTINOUS)
  {
    <%paraEquations%>
    <%continous%>
  }
   if (command & DISCRETE)
  {
    <%discrete%>
    <%reinit%>
  }
  }
  >>
end update;
*/
template update( list<SimEqSystem> allEquationsPlusWhen,list<SimWhenClause> whenClauses,SimCode simCode)
::=
  let &varDecls = buffer "" /*BUFD*/
  let all_equations = (allEquationsPlusWhen |> eqs => (eqs |> eq =>
      equation_(eq, contextOther, &varDecls /*BUFC*/,simCode))
    ;separator="\n")
  
  let reinit = (whenClauses |> when hasindex i0 =>
         genreinits(when, &varDecls,i0,simCode)
    ;separator="\n") 
  match simCode
  case SIMCODE(modelInfo = MODELINFO(__)) then    
  <<
  void <%lastIdentOfPath(modelInfo.name)%>::update(const UPDATE command)
  { 
    <%varDecls%>
     saveAll();
    if(IContinous::RANKING) checkConditions(0,true);
      <%all_equations%>
    <%reinit%>
  }
  >>
end update;


template genreinits(SimWhenClause whenClauses, Text &varDecls, Integer int,SimCode simCode)
::=

match whenClauses
case SIM_WHEN_CLAUSE(__) then
  let &preExp = buffer "" /*BUFD*/
  let &helpInits = buffer "" /*BUFD*/
  let helpIf = (conditions |> (e, hidx) =>
      let helpInit = daeExp(e, contextSimulationDiscrete, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
      let &helpInits +=""
      '_event_handling.edge(_event_handling[<%hidx%>],"h<%hidx%>")'
   ;separator="||")
  let ifthen = functionWhenReinitStatementThen(reinits, &varDecls /*BUFP*/,simCode)                     

if reinits then  
<<

  //For whenclause index: <%int%>
  <%preExp%>
  <%helpInits%>
  if (<%helpIf%>) { 
    <%ifthen%>
  }
>>
end genreinits;

template functionWhenReinitStatementThen(list<WhenOperator> reinits, Text &varDecls /*BUFP*/,SimCode simCode)
 "Generates re-init statement for when equation."
::=
  let body = (reinits |> reinit =>
    match reinit
    case REINIT(__) then 
      let &preExp = buffer "" /*BUFD*/
      let val = daeExp(value, contextSimulationDiscrete,
                   &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
     <<
      <%preExp%>
                <%cref1(stateVar,simCode)%> = <%val%>;
              
                >>
    case TERMINATE(__) then 
      let &preExp = buffer "" /*BUFD*/
    let msgVar = daeExp(message, contextSimulationDiscrete, &preExp /*BUFC*/, &varDecls /*BUFD*/,simCode)
    <<  
                <%preExp%> 
                MODELICA_TERMINATE(<%msgVar%>);
                >>
  case ASSERT(source=SOURCE(info=info)) then 
    assertCommon(condition, message, contextSimulationDiscrete, &varDecls, info,simCode)
  ;separator="\n")
  <<
   <%body%>  
  >>
end functionWhenReinitStatementThen;

template LabeledDAE(list<String> labels, SimCode simCode) ::=
if Flags.isSet(Flags.WRITE_TO_BUFFER) then match simCode
case SIMCODE(modelInfo = MODELINFO(__))
then
<<

IHistory* <%lastIdentOfPath(modelInfo.name)%>::getHistory()
{
   return _historyImpl;
}

<%if labels then
<<
label_list_type <%lastIdentOfPath(modelInfo.name)%>::getLabels()
{
   label_list_type labels = tuple_list_of 
   <%(labels |> label hasindex index0 => '(<%index0%>,&_<%label%>_1,&_<%label%>_2)') ;separator=" "%>;
   return labels;
}
>>
else
<<
label_list_type <%lastIdentOfPath(modelInfo.name)%>::getLabels()
{
   return label_list_type();
}
>>%>
 
void <%lastIdentOfPath(modelInfo.name)%>::setVariables(const ublas::vector<double>& variables,const ublas::vector<double>& variables2)
{
   <%setVariables(modelInfo)%>
}
>>
end LabeledDAE; 

template setVariables(ModelInfo modelInfo)
::= 
match modelInfo
case MODELINFO(vars = vars as SIMVARS(__))
then 
<<
 <%{(vars.algVars |> SIMVAR(__) hasindex myindex =>
       '<%cref(name)%>=variables(<%myindex%>);'
      ;separator="\n"),
      (vars.intAlgVars |> SIMVAR(__) hasindex myindex =>
       '<%cref(name)%>=variables(<%numAlgvar(modelInfo)%>+<%myindex%>);'
      ;separator="\n"),
      (vars.boolAlgVars |> SIMVAR(__) hasindex myindex =>
       '<%cref(name)%>=variables(<%numAlgvar(modelInfo)%>+<%numIntAlgvar(modelInfo)%>+<%myindex%>);'
      ;separator="\n"),
      (vars.stateVars |> SIMVAR(__) hasindex myindex =>
       '_z[<%index%>]=variables(<%numAlgvars(modelInfo)%>+<%myindex%>);'
      ;separator="\n"),
      (vars.derivativeVars |> SIMVAR(__) hasindex myindex =>
      '_zDot[<%index%>]=variables2(<%myindex%>);'
      ;separator="\n")}
     ;separator="\n"%>     
>>
end setVariables;

end CodegenCpp;
