package Interfaces "Connectors and partial models for input/output blocks"
   import Modelica.SIunits;
   extends Modelica.Icons.Library;
   annotation (preferedView="info",Documentation(info="
 <HTML>
 <p>
 This package contains interface definitions for
 <b>continuous</b> input/output blocks. In particular it
 contains the following <b>connector</b> classes:
 </p>
 <pre>
   <b>RealInput</b>       Connector with input  signals of type Real.
   <b>RealOutput</b>      Connector with output signals of type Real.
   <b>BooleanInput</b>    Connector with input  signals of type Boolean.
   <b>BooleanOutput</b>   Connector with output signals of type Boolean.
   <b>IntegerInput</b>    Connector with input  signals of type Integer.
   <b>IntegerOutput</b>   Connector with output signals of type Integer.
   <b>RealSignal</b>      Connector with input/output signals of type Real.
   <b>BooleanSignal</b>   Connector with input/output signals of type Boolean.
   <b>IntegerSignal</b>   Connector with input/output signals of type Integer.
 </pre>
 <p>
 The following <b>partial</b> block classes are provided
 to model <b>continuous</b> control blocks:
 </p>
 <pre>
   <b>BlockIcon</b>     Basic graphical layout of continuous block
   <b>SO</b>            Single Output continuous control block
   <b>MO</b>            Multiple Output continuous control block
   <b>SISO</b>          Single Input Single Output continuous control block
   <b>SI2SO</b>         2 Single Input / 1 Single Output continuous control block
   <b>SIMO</b>          Single Input Multiple Output continuous control block
   <b>MISO</b>          Multiple Input Single Output continuous control block
   <b>MIMO</b>          Multiple Input Multiple Output continuous control block
   <b>MIMOs</b>         Multiple Input Multiple Output continuous control block
                 with same number of inputs and outputs
   <b>MI2MO</b>         2 Multiple Input / Multiple Output continuous
                 control block
   <b>SignalSource</b>  Base class for continuous signal sources
   <b>SVcontrol</b>     Single-Variable continuous controller
   <b>MVcontrol</b>     Multi-Variable continuous controller
 </pre>
 <p>
 The following <b>partial</b> block classes are provided
 to model <b>discrete</b> control blocks:
 </p>
 <pre>
   <b>DiscreteBlockIcon</b> Basic graphical layout of discrete block
   <b>DiscreteBlock</b>     Base class of discrete control blocks
   <b>DiscreteSISO</b>      Single Input Single Output discrete control block
   <b>DiscreteMIMO</b>      Multiple Input Multiple Output discrete control block
   <b>DiscreteMIMOs</b>     Multiple Input Multiple Output discrete control block
   <b>SVdiscrete</b>        Discrete Single-Variable controller
   <b>MVdiscrete</b>        Discrete Multi-Variable controllerk
 </pre>
 <p>
 The following <b>partial</b> block classes are provided
 to model <b>Boolean</b> control blocks:
 </p>
 <pre>
   <b>BooleanBlockIcon</b>     Basic graphical layout of Boolean block
   <b>BooleanSISO</b>          Single Input Single Output control block
                        with signals of type Boolean
   <b>BooleanMIMOs</b>         Multiple Input Multiple Output control block
                        with same number of inputs and outputs
   <b>MI2BooleanMOs</b>        2 Multiple Input / Boolean Multiple Output
                        block with same signal lengths
   <b>BooleanSignalSource</b>  Base class for Boolean signal sources
   <b>IntegerMIBooleanMOs</b>  Multiple Integer Input Multiple Boolean Output control block
                        with same number of inputs and outputs
 </pre>
 <p>
 The following <b>partial</b> block classes are provided
 to model <b>Integer</b> control blocks:
 </p>
 <pre>
   <b>IntegerBlockIcon</b>     Basic graphical layout of Integer block
   <b>IntegerMO</b>            Multiple Output control block
   <b>IntegerSignalSource</b>  Base class for Integer signal sources
 </pre>
 <p>In addition, a subpackage <b>BusAdaptors</b> is temporarily provided
 in order to make a signal bus concept available. It will be removed,
 when the package Block is revised exploiting new Modelica features.</p>
 </HTML>
 ",revisions="<html>
 <ul>
 <li><i>Oct. 21, 2002</i>
        by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
        and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
        Added several new interfaces. <a href=\"../Documentation/ChangeNotes1.5.html\">Detailed description</a> available.
 <li><i>Oct. 24, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        RealInputSignal renamed to RealInput. RealOutputSignal renamed to
        output RealOutput. GraphBlock renamed to BlockIcon. SISOreal renamed to
        SISO. SOreal renamed to SO. I2SOreal renamed to M2SO.
        SignalGenerator renamed to SignalSource. Introduced the following
        new models: MIMO, MIMOs, SVcontrol, MVcontrol, DiscreteBlockIcon,
        DiscreteBlock, DiscreteSISO, DiscreteMIMO, DiscreteMIMOs,
        BooleanBlockIcon, BooleanSISO, BooleanSignalSource, MI2BooleanMOs.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized a first version, based on an existing Dymola library
        of Dieter Moormann and Hilding Elmqvist.</li>
 </ul>
 </html>
 "));
   connector RealSignal "Real port (both input/output possible)"
      replaceable type SignalType = Real ;
      extends SignalType;
   end RealSignal;
   connector BooleanSignal = Boolean "Boolean port (both input/output possible)";
   connector IntegerSignal = Integer "Integer port (both input/output possible)";
   connector RealInput =input  RealSignal "'input Real' as connector" annotation (defaultComponentName="u",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={0,0,191},fillColor={0,0,191},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,50.},{100.,0.},{0.,-50.},{0.,50.}},lineColor={0,0,191},fillColor={0,0,191},fillPattern=FillPattern.Solid),Text(extent={{-140.,120.},{100.,60.}},textString="%name",fillColor={0,0,191})}));
   connector RealOutput =output  RealSignal "'output Real' as connector" annotation (defaultComponentName="y",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,50.},{0.,0.},{-100.,-50.},{-100.,50.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,120.},{140.,60.}},textString="%name",fillColor={0,0,191})}));
   connector BooleanInput =input  BooleanSignal "'input Boolean' as connector" annotation (defaultComponentName="u",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={255,0,255},fillColor={255,0,255},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,50.},{100.,0.},{0.,-50.},{0.,50.}},lineColor={255,0,255},fillColor={255,0,255},fillPattern=FillPattern.Solid),Text(extent={{-140.,120.},{100.,60.}},textString="%name",fillColor={255,0,255})}));
   connector BooleanOutput =output  BooleanSignal "'output Boolean' as connector" annotation (defaultComponentName="y",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,50.},{0.,0.},{-100.,-50.},{-100.,50.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,120.},{140.,60.}},textString="%name",fillColor={255,0,255})}));
   connector IntegerInput =input  IntegerSignal "'input Integer' as connector" annotation (defaultComponentName="u",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={255,127,0},fillColor={255,127,0},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,50.},{100.,0.},{0.,-50.},{0.,50.}},lineColor={255,127,0},fillColor={255,127,0},fillPattern=FillPattern.Solid),Text(extent={{-140.,120.},{100.,60.}},fillColor={255,127,0},textString="%name")}));
   connector IntegerOutput =output  IntegerSignal "'output Integer' as connector" annotation (defaultComponentName="y",Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,100.},{100.,0.},{-100.,-100.},{-100.,100.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,50.},{0.,0.},{-100.,-50.},{-100.,50.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,120.},{140.,60.}},fillColor={255,127,0},textString="%name")}));
   partial block BlockIcon "Basic graphical layout of continuous block"
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}));
   end BlockIcon;
   partial block SO "Single Output continuous control block"
      extends BlockIcon;
      RealOutput y "Connector of Real output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end SO;
   partial block MO "Multiple Output continuous control block"
      extends BlockIcon;
      parameter Integer nout(min=1)=1 "Number of outputs";
      RealOutput y[nout] "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Documentation(info="
 "));
   end MO;
   partial block SISO "Single Input Single Output continuous control block"
      extends BlockIcon;
      RealInput u "Connector of Real input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealOutput y "Connector of Real output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end SISO;
   partial block SI2SO "2 Single Input / 1 Single Output continuous control block"
      extends BlockIcon;
      RealInput u1 "Connector of Real input signal 1" annotation (Placement(transformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.)));
      RealInput u2 "Connector of Real input signal 2" annotation (Placement(transformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.)));
      RealOutput y "Connector of Real output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Documentation(info="
 Block has two input vectors u1 and u2 and one
 output vector y. All signals are scalar.
 "),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end SI2SO;
   partial block SIMO "Single Input Multiple Output continuous control block"
      extends BlockIcon;
      parameter Integer nout=1 "Number of outputs";
      RealInput u "Connector of Real input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealOutput y[nout] "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<HTML>
 <p>Block has one continuous input signal and a vector of continuous output
 signals.</p>
 </HTML>
 "));
   end SIMO;
   partial block MISO "Multiple Input Single Output continuous control block"
      extends BlockIcon;
      parameter Integer nin=1 "Number of inputs";
      RealInput u[nin] "Connector of Real input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealOutput y "Connector of Real output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="
 <HTML>
 <p>
 Block has a vector of continuous input signals and
 one continuous output signal.
 </p>
 </HTML>
 "));
   end MISO;
   partial block MIMO "Multiple Input Multiple Output continuous control block"
      extends BlockIcon;
      parameter Integer nin=1 "Number of inputs";
      parameter Integer nout=1 "Number of outputs";
      RealInput u[nin] "Connector of Real input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealOutput y[nout] "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="
 <HTML>
 <p>
 Block has a continuous input and a continuous output signal vector.
 The signal sizes of the input and output vector may be different.
 </p>
 </HTML>
 "));
   end MIMO;
   partial block MIMOs "Multiple Input Multiple Output continuous control block with same number of inputs and outputs"
      extends BlockIcon;
      parameter Integer n=1 "Number of inputs (= number of outputs)";
      RealInput u[n] "Connector of Real input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealOutput y[n] "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Documentation(info="
 <HTML>
 <p>
 Block has a continuous input and a continuous output signal vector
 where the signal sizes of the input and output vector are identical.
 </p>
 </HTML>
 "));
   end MIMOs;
   partial block MI2MO "2 Multiple Input / Multiple Output continuous control block"
      extends BlockIcon;
      parameter Integer n=1 "Dimension of input and output vectors.";
      RealInput u1[n] "Connector 1 of Real input signals" annotation (Placement(transformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.)));
      RealInput u2[n] "Connector 2 of Real input signals" annotation (Placement(transformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.)));
      RealOutput y[n] "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Documentation(info="
 Block has two Input vectors u1 and u2 and one
 output vector y. All vectors have the same number of elements.
 "),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end MI2MO;
   partial block SignalSource "Base class for continuous signal source"
      extends SO;
      parameter Real offset=0 "offset of output signal";
      parameter SIunits.Time startTime=0 "output = offset for time < startTime";
   end SignalSource;
   partial block SVcontrol "Single-Variable continuous controller"
      extends BlockIcon;
      RealInput u_s "Connector of setpoint input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealInput u_m "Connector of measurement input signal" annotation (Placement(transformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true),iconTransformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true)));
      RealOutput y "Connector of actuator output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-102.,34.},{-142.,24.}},textString="(setpoint)",fillColor={0,0,255}),Text(extent={{100.,24.},{140.,14.}},textString="(actuator)",fillColor={0,0,255}),Text(extent={{-83.,-112.},{-33.,-102.}},textString=" (measurement)",fillColor={0,0,255})}));
   end SVcontrol;
   partial block MVcontrol "Multi-Variable continuous controller"
      extends BlockIcon;
      parameter Integer nu_s=1 "Number of setpoint inputs";
      parameter Integer nu_m=1 "Number of measurement inputs";
      parameter Integer ny=1 "Number of actuator outputs";
      RealInput u_s[nu_s] "Connector of setpoint input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      RealInput u_m[nu_m] "Connector of measurement input signals" annotation (Placement(transformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true),iconTransformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true)));
      RealOutput y[ny] "Connector of actuator output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-100.,36.},{-140.,26.}},textString="(setpoint)",fillColor={0,0,255}),Text(extent={{102.,24.},{142.,14.}},textString="(actuator)",fillColor={0,0,255}),Text(extent={{-75.,-108.},{-25.,-98.}},textString=" (measurement)",fillColor={0,0,255})}));
   end MVcontrol;
   partial block DiscreteBlockIcon "Graphical layout of discrete block component icon"
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={223,223,159},fillPattern=FillPattern.Solid),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}));
   end DiscreteBlockIcon;
   partial block DiscreteBlock "Base class of discrete control blocks"
      extends DiscreteBlockIcon;
      parameter SI.Time samplePeriod(min=100*Modelica.Constants.eps)=0.1 "Sample period of component";
      parameter SI.Time startTime=0 "First sample time instant";
      
   protected
      output Boolean sampleTrigger "True, if sample time instant";
      output Boolean firstTrigger "Rising edge signals first sample instant";
      
   equation
      sampleTrigger = sample(startTime,samplePeriod);
      when sampleTrigger then
         firstTrigger = time <= startTime+samplePeriod/2;
      end when;
   end DiscreteBlock;
   partial block DiscreteSISO "Single Input Single Output discrete control block"
      extends DiscreteBlock;
      Modelica.Blocks.Interfaces.RealInput u "Continuous input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Modelica.Blocks.Interfaces.RealOutput y "Continuous output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
   end DiscreteSISO;
   partial block DiscreteMIMO "Multiple Input Multiple Output discrete control block"
      extends DiscreteBlock;
      parameter Integer nin=1 "Number of inputs";
      parameter Integer nout=1 "Number of outputs";
      Modelica.Blocks.Interfaces.RealInput u[nin] "Continuous input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Modelica.Blocks.Interfaces.RealOutput y[nout] "Continuous output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<html>
 <p>
 Block has a continuous input and a continuous output signal vector
 which are sampled due to the defined <b>samplePeriod</b> parameter.
 </p>
 </HTML>
 "));
   end DiscreteMIMO;
   partial block DiscreteMIMOs "Multiple Input Multiple Output discrete control block"
      parameter Integer n=1 "Number of inputs (= number of outputs)";
      extends DiscreteBlock;
      Modelica.Blocks.Interfaces.RealInput u[n] "Continuous input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Modelica.Blocks.Interfaces.RealOutput y[n] "Continuous output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<html>
 <p>
 Block has a continuous input and a continuous output signal vector
 where the signal sizes of the input and output vector are identical.
 These signals are sampled due to the defined <b>samplePeriod</b> parameter.
 </p>
 </HTML>
 "));
   end DiscreteMIMOs;
   partial block SVdiscrete "Discrete Single-Variable controller"
      extends DiscreteBlock;
      Discrete.Sampler sampler_s(final samplePeriod=samplePeriod,final startTime=startTime) annotation (Placement(transformation(x=-90.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-90.,y=0.,scale=0.1,aspectRatio=1.)));
      Discrete.Sampler sampler_m(final samplePeriod=samplePeriod,final startTime=startTime) annotation (Placement(transformation(x=0.,y=-90.,scale=0.1,aspectRatio=1.,rotation=90),iconTransformation(x=0.,y=-90.,scale=0.1,aspectRatio=1.,rotation=90)));
      Modelica.Blocks.Interfaces.RealInput u_s "Continuous scalar setpoint input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Modelica.Blocks.Interfaces.RealInput u_m "Continuous scalar measurement input signal" annotation (Placement(transformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true),iconTransformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true)));
      Modelica.Blocks.Interfaces.RealOutput y "Continuous scalar actuator output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-100.,34.},{-140.,24.}},textString="(setpoint)",fillColor={0,0,0}),Text(extent={{100.,22.},{130.,14.}},textString="(actuator)",fillColor={0,0,0}),Text(extent={{-70.,-112.},{-20.,-102.}},textString=" (measurement)",fillColor={0,0,0})}));
      
   equation
      connect(u_s, sampler_s.u) annotation (Line(points={{-120.,0.},{-102.,0.}},color={0,0,255}));
      connect(u_m, sampler_m.u) annotation (Line(points={{0.,-120.},{0.,-111.},{0.,-102.},{-7.34764e-16,-102.}},color={0,0,255}));
   end SVdiscrete;
   partial block MVdiscrete "Discrete Multi-Variable controller"
      extends DiscreteBlock;
      parameter Integer nu_s=1 "Number of setpoint inputs";
      parameter Integer nu_m=1 "Number of measurement inputs";
      parameter Integer ny=1 "Number of actuator outputs";
      Discrete.Sampler sampler_s[nu_s](each final samplePeriod=samplePeriod,each final startTime=startTime) annotation (Placement(transformation(x=-80.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-80.,y=0.,scale=0.1,aspectRatio=1.)));
      Discrete.Sampler sampler_m[nu_m](each final samplePeriod=samplePeriod,each final startTime=startTime) annotation (Placement(transformation(x=0.,y=-80.,scale=0.1,aspectRatio=1.,rotation=90),iconTransformation(x=0.,y=-80.,scale=0.1,aspectRatio=1.,rotation=90)));
      Modelica.Blocks.Interfaces.RealInput u_s[nu_s] "Continuous setpoint input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Modelica.Blocks.Interfaces.RealInput u_m[nu_m] "Continuous measurement input signals" annotation (Placement(transformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true),iconTransformation(x=0.,y=-120.,scale=0.2,aspectRatio=1.,rotation=-90,flipHorizontal=true,flipVertical=true)));
      Modelica.Blocks.Interfaces.RealOutput y[ny] "Continuous actuator output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-100.,-10.},{-80.,-30.}},textString="u_s",fillColor={0,0,255}),Text(extent={{-98.,34.},{-138.,24.}},textString="(setpoint)",fillColor={0,0,0}),Text(extent={{98.,24.},{138.,14.}},textString="(actuator)",fillColor={0,0,0}),Text(extent={{-62.,-110.},{-12.,-100.}},textString=" (measurement)",fillColor={0,0,0})}));
      
   equation
      connect(u_s, sampler_s.u) annotation (Line(points={{-120.,0.},{-92.,0.}},color={0,0,255}));
      connect(u_m, sampler_m.u) annotation (Line(points={{0.,-120.},{0.,-106.},{0.,-92.},{-7.34764e-16,-92.}},color={0,0,255}));
   end MVdiscrete;
   partial block BooleanBlockIcon "Basic graphical layout of Boolean block"
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}));
   end BooleanBlockIcon;
   partial block BooleanSISO "Single Input Single Output control block with signals of type Boolean"
      extends BooleanBlockIcon;
      
   public
      BooleanInput u "Connector of Boolean input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      BooleanOutput y "Connector of Boolean output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end BooleanSISO;
   partial block BooleanMIMOs "Multiple Input Multiple Output continuous control block with same number of inputs and outputs of boolean type"
      extends BooleanBlockIcon;
      parameter Integer n=1 "Number of inputs (= number of outputs)";
      BooleanInput u[n] "Connector of Boolean input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      BooleanOutput y[n] "Connector of Boolean output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<HTML>
 <p>
 Block has a continuous input and a continuous output signal vector
 where the signal sizes of the input and output vector are identical
 and of type Boolean.
 </p>
 </HTML>
 "));
   end BooleanMIMOs;
   partial block MI2BooleanMOs "2 Multiple Input / Boolean Multiple Output block with same signal lengths"
      extends BooleanBlockIcon;
      parameter Integer n=1 "Dimension of input and output vectors.";
      RealInput u1[n] "Connector 1 of Boolean input signals" annotation (Placement(transformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.)));
      RealInput u2[n] "Connector 2 of Boolean input signals" annotation (Placement(transformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.)));
      BooleanOutput y[n] "Connector of Boolean output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<html>
 <p>Block has two Boolean input vectors u1 and u2 and one Boolean output
 vector y. All vectors have the same number of elements.</p>
 </html>
 "));
   end MI2BooleanMOs;
   partial block SI2BooleanSO "2 Single Input / Boolean Single Output block"
      extends BooleanBlockIcon;
      input RealInput u1 "Connector 1 of Boolean input signals" annotation (Placement(transformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=60.,scale=0.2,aspectRatio=1.)));
      input RealInput u2 "Connector 2 of Boolean input signals" annotation (Placement(transformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-60.,scale=0.2,aspectRatio=1.)));
      output BooleanOutput y "Connector of Boolean output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Documentation(info="
 Block has two Boolean inputs u1 and u2 and one Boolean output y.
 "));
   end SI2BooleanSO;
   partial block BooleanSignalSource "Base class for Boolean signal sources"
      extends BooleanBlockIcon;
      BooleanOutput y "Connector of Boolean output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,68.},{-80.,-80.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-70.},{68.,-70.}},color={192,192,192}),Polygon(points={{90.,-70.},{68.,-62.},{68.,-78.},{90.,-70.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end BooleanSignalSource;
   partial block IntegerBlockIcon "Basic graphical layout of Integer block"
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}));
   end IntegerBlockIcon;
   partial block IntegerSO "Single Integer Output continuous control block"
      extends IntegerBlockIcon;
      IntegerOutput y "Connector of Integer output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
   end IntegerSO;
   partial block IntegerMO "Multiple Integer Output continuous control block"
      extends IntegerBlockIcon;
      parameter Integer nout(min=1)=1 "Number of outputs";
      IntegerOutput y[nout] "Connector of Integer output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
   end IntegerMO;
   partial block IntegerSignalSource "Base class for continuous Integer signal source"
      extends IntegerSO;
      parameter Integer offset=0 "offset of output signal";
      parameter SI.Time startTime=0 "y = offset for time < startTime";
   end IntegerSignalSource;
   partial block IntegerSIBooleanSO "Integer Input Boolean Output continuous control block"
      extends BooleanBlockIcon;
      IntegerInput u "Connector of Integer input signal" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      BooleanOutput y "Connector of Boolean output signal" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<HTML>
 <p>
 Block has a continuous Integer input and a continuous Boolean output signal.
 </p>
 </HTML>
 "));
   end IntegerSIBooleanSO;
   partial block IntegerMIBooleanMOs "Multiple Integer Input Multiple Boolean Output continuous control block with same number of inputs and outputs"
      extends BooleanBlockIcon;
      parameter Integer n=1 "Number of inputs (= number of outputs)";
      IntegerInput u[n] "Connector of Integer input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      BooleanOutput y[n] "Connector of Boolean output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="<HTML>
 <p>
 Block has a continuous integer input and a continuous boolean output signal vector
 where the signal sizes of the input and output vector are identical.
 </p>
 </HTML>
 "));
   end IntegerMIBooleanMOs;
   partial block partialBooleanBlockIcon "Basic graphical layout of logical block"
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,100.},{100.,-100.}},lineColor={0,0,0},fillColor={235,235,235},fillPattern=FillPattern.Solid),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}));
   end partialBooleanBlockIcon;
   partial block partialBooleanSISO "partialSingle Input / Single Output Boolean block"
      extends partialBooleanBlockIcon;
      Blocks.Interfaces.BooleanInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-71.,7.},{-85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}));
   end partialBooleanSISO;
   partial block partialBooleanSI2SO "partial2 Single Input / 1 Single Output Boolean block"
      extends partialBooleanBlockIcon;
      Blocks.Interfaces.BooleanInput u1 annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanInput u2 annotation (Placement(transformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-71.,7.},{-85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{-71.,-74.},{-85.,-88.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}));
   end partialBooleanSI2SO;
   partial block partialBooleanSI3SO "partial2 Single Input / 1 Single Output Boolean block"
      extends partialBooleanBlockIcon;
      Blocks.Interfaces.BooleanInput u1 annotation (Placement(transformation(x=-120.,y=80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=80.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanInput u2 annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanInput u3 annotation (Placement(transformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.)));
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-71.,74.},{-85.,88.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{-71.,7.},{-85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{-71.,-74.},{-85.,-88.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}));
   end partialBooleanSI3SO;
   partial block partialBooleanSI "partialSingle Input Boolean block"
      extends partialBooleanBlockIcon;
      Blocks.Interfaces.BooleanInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-71.,7.},{-85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}));
   end partialBooleanSI;
   partial block partialBooleanSO "partialSingle Output Boolean block"
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      extends partialBooleanBlockIcon;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}));
   end partialBooleanSO;
   partial block partialBooleanSource "partialBoolean source block"
      extends partialBooleanBlockIcon;
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-80.,88.},{-88.,66.},{-72.,66.},{-80.,88.}},lineColor={255,0,255},fillColor={255,0,255},fillPattern=FillPattern.Solid),Line(points={{-80.,66.},{-80.,-82.}},color={255,0,255}),Line(points={{-90.,-70.},{72.,-70.}},color={255,0,255}),Polygon(points={{90.,-70.},{68.,-62.},{68.,-78.},{90.,-70.}},lineColor={255,0,255},fillColor={255,0,255},fillPattern=FillPattern.Solid),Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-80.,80.},{-88.,58.},{-72.,58.},{-80.,80.}},lineColor={0,0,0},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,58.},{-80.,-90.}},color={0,0,0}),Line(points={{-90.,-70.},{68.,-70.}},color={0,0,0}),Polygon(points={{90.,-70.},{68.,-62.},{68.,-78.},{90.,-70.}},lineColor={0,0,0},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{54.,-84.},{106.,-96.}},textString="time",fillColor={0,0,0}),Text(extent={{-108.,80.},{-92.,64.}},fillColor={0,0,0},textString="y")}));
   end partialBooleanSource;
   partial block partialBooleanThresholdComparison "partialBoolean block to compare the Real input u with a threshold"
      parameter Real threshold=0 "Comparison with respect to threshold";
      Blocks.Interfaces.RealInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.,Hide=true),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.,Hide=true)));
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,100.},{100.,-100.}},lineColor={0,0,0},fillColor={235,235,235},fillPattern=FillPattern.Solid),Text(extent={{-150.,-140.},{150.,-110.}},fillColor={0,0,0},textString="%threshold"),Ellipse(extent={{71.,7.},{85.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end partialBooleanThresholdComparison;
   partial block partialBooleanComparison "partialBoolean block to compare two Real inputs u1 and u2"
      Blocks.Interfaces.RealInput u1 annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.,Hide=true),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.,Hide=true)));
      Blocks.Interfaces.RealInput u2 annotation (Placement(transformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.,Hide=true),iconTransformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.,Hide=true)));
      Blocks.Interfaces.BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true)));
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,100.},{100.,-100.}},lineColor={0,0,0},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{73.,7.},{87.,-7.}},lineColor={235,235,235},fillColor={235,235,235},fillPattern=FillPattern.Solid),Ellipse(extent={{32.,10.},{52.,-10.}},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-100.,-80.},{42.,-80.},{42.,0.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
   end partialBooleanComparison;
   package Adaptors "Components to send signals to the bus or receive signals from the bus"
      extends Modelica.Icons.Library;
      block SendReal "Send Real signal to bus"
         RealOutput toBus annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         RealInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0}),Text(extent={{-100.,30.},{100.,-30.}},textString="send",fillColor={0,0,255})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         toBus = u;
      end SendReal;
      block SendBoolean "Send Boolean signal to bus"
         BooleanOutput toBus annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         BooleanInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0}),Text(extent={{-100.,30.},{100.,-30.}},textString="send",fillColor={255,0,255})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         toBus = u;
      end SendBoolean;
      block SendInteger "Send Integer signal to bus"
         IntegerOutput toBus annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         IntegerInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0}),Text(extent={{-100.,30.},{100.,-30.}},textString="send",fillColor={255,127,0})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         toBus = u;
      end SendInteger;
      block ReceiveReal "Receive Real signal from bus"
         RealInput fromBus annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         RealOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,30.},{100.,-30.}},textString="receive",fillColor={0,0,255}),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         y = fromBus;
      end ReceiveReal;
      block ReceiveBoolean "Receive Boolean signal from bus"
         BooleanInput fromBus annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         BooleanOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,30.},{100.,-30.}},textString="receive",fillColor={255,0,255}),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         y = fromBus;
      end ReceiveBoolean;
      block ReceiveInteger "Receive Integer signal from bus"
         IntegerInput fromBus annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         IntegerOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,30.},{100.,-30.}},textString="receive",fillColor={255,127,0}),Text(extent={{-144.,96.},{144.,46.}},textString="%name",fillColor={0,0,0})}),Documentation(info="<HTML>
 </HTML>
 "));
         
      equation
         y = fromBus;
      end ReceiveInteger;
      model AdaptorReal "Adaptor between 'old' and 'new' Real signal connectors"
         RealSignal newReal annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true)));
         RealPort oldReal(final n=1) annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (structurallyIncomplete(),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},fillColor={0,0,0},textString=""),Text(extent={{-88.,22.},{88.,-24.}},fillColor={0,0,255},textString="adaptor"),Text(extent={{-216.,-58.},{36.,-80.}},fillColor={0,0,0},fillPattern=FillPattern.Solid,textString="port.signal")}),Documentation(info=""));
         
      protected
         connector RealPort "Connector with signals of type Real"
            parameter Integer n=1 "Dimension of signal vector" annotation (Hide=true);
            replaceable type SignalType = Real "type of signal" ;
            SignalType signal[n] "Real signals" annotation (Hide=true);
         end RealPort;
         
      equation
         newReal = oldReal.signal[1];
      end AdaptorReal;
      model AdaptorBoolean "Adaptor between 'old' and 'new' Boolean signal connectors"
         BooleanSignal newBoolean annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true)));
         BooleanPort oldBoolean(final n=1) annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (structurallyIncomplete(),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},fillColor={0,0,0},textString=""),Text(extent={{-88.,22.},{88.,-24.}},fillColor={255,0,255},textString="adaptor"),Text(extent={{-216.,-58.},{36.,-80.}},fillColor={0,0,0},fillPattern=FillPattern.Solid,textString="port.signal")}),Documentation(info=""));
         
      protected
         connector BooleanPort "Connector with signals of type Boolean"
            parameter Integer n=1 "Dimension of signal vector" annotation (Hide=true);
            replaceable type SignalType = Boolean "type of signal" ;
            SignalType signal[n] "Boolean signals" annotation (Hide=true);
         end BooleanPort;
         
      equation
         newBoolean = oldBoolean.signal[1];
      end AdaptorBoolean;
      model AdaptorInteger "Adaptor between 'old' and 'new' Integer signal connectors"
         IntegerSignal newInteger annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.,Hide=true)));
         IntegerPort oldInteger(final n=1) annotation (Placement(transformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-110.,y=0.,scale=0.1,aspectRatio=1.)));
         annotation (structurallyIncomplete(),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,40.},{100.,-40.}},lineColor={255,127,0},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-144.,96.},{144.,46.}},fillColor={0,0,0},textString=""),Text(extent={{-88.,22.},{88.,-24.}},fillColor={255,127,0},textString="adaptor"),Text(extent={{-216.,-58.},{36.,-80.}},fillColor={0,0,0},fillPattern=FillPattern.Solid,textString="port.signal")}),Documentation(info="
 "));
         
      protected
         connector IntegerPort "Connector with signals of type Integer"
            parameter Integer n=1 "Dimension of signal vector" annotation (Hide=true);
            replaceable type SignalType = Integer "type of signal" ;
            SignalType signal[n] "Integer signals" annotation (Hide=true);
         end IntegerPort;
         
      equation
         newInteger = oldInteger.signal[1];
      end AdaptorInteger;
   end Adaptors;
end Interfaces;
