/*
------------------------------------------------------------------------------------
This file is part of OpenModelica.

Copyright (c) 1998-2005, Linkpings universitet,
Department of Computer and Information Science, PELAB
See also: www.ida.liu.se/projects/OpenModelica

All rights reserved.

(The new BSD license, see also
http://www.opensource.org/licenses/bsd-license.php)


Redistribution and use in source and binary forms, with or without
modification,
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    
	* Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.

    * Neither the name of Linkpings universitet nor the names of its contributors
      may be used to endorse or promote products derived from this software without
      specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

For more information about the Qt-library visit TrollTech:s webpage regarding
licence: http://www.trolltech.com/products/qt/licensing.html

------------------------------------------------------------------------------------
*/

#include <exception>
#include <stdexcept>

#include <qlabel.h>
#include <qapplication.h>

#include "cellcursor.h"
#include "visitor.h"

namespace IAEX
{

   /*! \class CellCursor 
    * \brief Implements a special type of cell that
    * is used as a cursor within the document.
    *
    * The cellcursor class acts like a ordinary cell. It extends a
    * cell with the functionality to move around in the celltree. See
    * moveUp and moveDown members.
    *
    * This class should be inherited with a lot of precaution. It has
    * a lot of responsibility and dependency within the
    * application. So change it with caution. 
    *
    * To change the cursors look reimplement the CursorWidget to have
    * the desired look.
    *
    */
   CellCursor::CellCursor(QWidget *parent, const char *name)
      : Cell(parent, name)
   {
      setHeight(3);
      QWidget *content = new CursorWidget(this);
      
      setMainWidget(content);
      hideTreeView(true);
      setBackgroundMode(Qt::PaletteBase);
      setBackgroundColor(QColor(100,100,100));
   }
   
   CellCursor::~CellCursor()
   {
      
   }   
   
   /*!
    * \bug This does not work correctly.
    */
   void CellCursor::accept(Visitor &v)
   {
      //Does not have any childs!
      v.visitCellCursorNodeBefore(this);
      v.visitCellCursorNodeAfter(this);  
      
      if(hasNext())
	 next()->accept(v);
   }

   
   void CellCursor::addBefore(Cell *newCell)
   {
      if(parentCell()->child() == this)
      { //first in line.
	 newCell->setParentCell(parentCell());
	 newCell->setNext(this);
	 newCell->setPrevious(0);
	 parentCell()->setChild(newCell);
	 setPrevious(newCell);
      }
      else
      {
	 newCell->setParentCell(parentCell());
	 newCell->setPrevious(previous());
	 previous()->setNext(newCell);
	 setPrevious(newCell);
	 newCell->setNext(this);
      }

      parentCell()->addCellWidget(newCell);
      //emit changedPosition();
   }

   /*! \brief Replaces current cell with a new cell. 
    *
    * \todo create a cellcopy operation.
    * \todo test!
    *
    */
   void CellCursor::replaceCurrentWith(Cell *newCell)
   {
//       newCell->setParent(currentCell()->parentCell());
//       newCell->setChild(currentCell()->child());
//       newCell->setLast(currentCell()->last());
//       newCell->setPrevious(currentCell()->previous());
//       newCell->setNext(currentCell()->next());
      qDebug("replaceWithCurrent");
      
      newCell->setText(currentCell()->text());

      //Replace cell.
      deleteCurrentCell();
      addBefore(newCell);
      qDebug("End replaceWithCurrent");
   }

   void CellCursor::removeFromCurrentPosition()
   {
      //remove all widgets from parents layout.
      Cell *par = parentCell();
      par->removeCellWidgets();
	  
      if(parentCell()->child() == this)
		 parentCell()->setChild(next());
      
      if(parentCell()->last() == this)
		 parentCell()->setLast(previous());
      
      if(hasNext())
		 next()->setPrevious(previous());
      
      if(hasPrevious())
		 previous()->setNext(next());
      
      //Insert all widgets again.
      par->addCellWidgets();
   } 

   /*!
    * Removes a cell and all its subcells from the tree.
    *
    * This should work for all cells. But it will leave an empty
    * cellgroup if last cell is deleted in cellgroup.
    *
    * This does not delete the cell, it just removes the cell from the
    * celltree.
    */
   void CellCursor::removeCurrentCell()
   {
      if(hasPrevious()) //If cursor has previous
      {	 
		 Cell *current = previous();
		 
		 removeFromCurrentPosition();
		 
		 if(current->hasPrevious())
			current->previous()->setNext(this);
		 else
			parentCell()->setChild(this);
		 
		 setPrevious(current->previous());
		 
		 current->setParentCell(0);
		 current->setPrevious(0);
		 current->setNext(0);
		 current->setChild(0);
		 current->setLast(0);
	
		 current->hide();
		 parentCell()->addCellWidgets();
      }
   }
  
   /*! \bug Segfault in cellgroups. Probably a parent, child or last.
    * \bug Deletion of last cell in cellgroup should be taken care of.
    */
   void CellCursor::deleteCurrentCell()
   {
      if(hasPrevious()) //If cursor has previous
      {	 
		 //removeCurrentCell();
		 
		 //OLD CODE
		 //Remove currentCell. 
		 Cell *current = previous(); //Save a pointer to the cell being deleted.
		 
		 removeCurrentCell();
		 // removeFromCurrentPosition();
		 
// 	 if(current->hasPrevious())
// 	    current->previous()->setNext(this);
// 	 else
// 	    parentCell()->setChild(this);
		 
// 	 setPrevious(current->previous());
		 
// 	 current->setParentCell(0);
// 	 current->setPrevious(0);
// 	 current->setNext(0);
// 	 current->setChild(0);
// 	 current->setLast(0);
		 
		 //Segfault on delete.
		 delete current;	 
		 
		 //parentCell()->addCellWidgets();
      }
      //emit changedPosition();
   }

	/*! Returns current cell. 
	 */
	Cell *CellCursor::currentCell()
	{
		if(!hasPrevious()) //First in group. 
			return parentCell(); //Will always work.
		else
			return previous();
	}

   void CellCursor::moveUp()
   {      

      if(!hasPrevious())
      {
	 if(parentCell()->hasParentCell())
	    moveBefore(parentCell());
	 else
	 {
	    //Do nothing. Can not move before root.
	 }
      }
      else //previous() exists. 
      {  
	 if(previous()->hasChilds())
	 {
	    if(!previous()->isClosed())
	       moveToLastChild(previous());
	    else
	       moveBefore(previous());
	 }
	 else
	 {  
	    moveBefore(previous());
	 }
      }
      emit positionChanged(x(), y(), 5,5);
      //emit changedPosition();
   }

   /*!
    * \bug Segmentationfault when last cell.  
    * 
    * \todo It is better that Commands take care of how to change
    * state of cells.
    */
   void CellCursor::moveDown()
   {
      if(!hasNext())
      {
	 if(parentCell()->hasParentCell())
	    moveAfter(parentCell());
	 else
	 { 
	    //Do nothing. Can not move before root.
	 }
      }
      else //Has next.
      {
	 if(next()->hasChilds())
	 {  
	    if(!next()->isClosed())
	       moveToFirstChild(next());
	    else
	       moveAfter(next());
	 }
	 else
	 {
	    moveAfter(next());
	 }
      }
      //emit changedPosition();
      emit positionChanged(x(), y(), 5,5);
   }

   /*! Insert this cell as first child of parent.
    *
    * \bug This does not seem to work correctly.
    */
   void CellCursor::moveToFirstChild(Cell *parent)
   {
      if(parent->hasChilds())
      {
	 parent->removeCellWidgets();
	 moveBefore(parent->child());
	 parent->addCellWidgets();
      }
      else //No child. 
      {
	 //Become first child.
	 parent->removeCellWidgets();
	 parent->setChild(this);
	 parent->setLast(this);
	 parent->addCellWidgets();
      }
      
      //emit changedPosition();
   }
   
   /*!
    * \bug This does not seem to work correctly.
    */
   void CellCursor::moveToLastChild(Cell *parent)
   {
      if(parent->hasChilds())
      {
	 parent->removeCellWidgets();
	 moveAfter(parent->last());
	 parent->addCellWidgets();      
      }
      else
      {
	 throw runtime_error("LAST CHILD: Tried to move to a child that did not exist.");
      }

      //emit changedPosition();
   }

   /*!
    * \bug Fel vid flytt s cursor hamnar som sista barn.
    */
   void CellCursor::moveAfter(Cell *current)
   {
      removeFromCurrentPosition();
      
      //if(!current->hasParentCell())
      //  throw runtime_error("Could not insert after root");
      
      if(current->hasParentCell())
      {
	 current->parentCell()->removeCellWidgets();
	 
	 if(current->hasNext() == 0)
	 {
	    current->parentCell()->setLast(this);
	 }
	 else
	    current->next()->setPrevious(this);
	 
	 setParentCell(current->parentCell());
	 setNext(current->next());
	 current->setNext(this);
	 setPrevious(current);
	 
	 //insert widgets to parents layout.
	 parentCell()->addCellWidgets();      
      }
      else
      {
	 //If current does not have a parent. That is current is not
	 //in the celltree at all or that current is the root of the
	 //tree. It should not be possible to move after the root of
	 //the tree. Do nothing!
      }
      // emit changedPosition();
//      emit positionChanged(x(), y(), 5,5);
   }
   

   /*
    * \bug fel  om vi flyttas till att vara frsta barn..
    */
   void CellCursor::moveBefore(Cell *current)
   {
      removeFromCurrentPosition();
      
      //Remove all widgets from currents parent.
      current->parentCell()->removeCellWidgets();

      //Move to new position.
      if(current->hasParentCell())
      {
	 setParentCell(current->parentCell());
	 if(!current->hasPrevious())
	    current->parentCell()->setChild(this);
	 else
	    current->previous()->setNext(this);

      } 
      else
	 throw runtime_error("Could not insert before root");
 
      setPrevious(current->previous());
      current->setPrevious(this);
      setNext(current);

      //Insert widgets to parents layout.
      parentCell()->addCellWidgets();
      //emit changedPosition();
//      emit positionChanged(x(), y(), 5, 5);
   }

   
   /*! \class CursorWidget 
    *
    * \brief CursorWidget describes how the cursor should be painted.
    */
   void CursorWidget::paintEvent(QPaintEvent *event)
   {
      QPainter painter(this);
      
      this->setPaletteBackgroundColor(QColor(0,0,0));
      painter.setPen(QPen(black,1, SolidLine));
      
      QWidget::paintEvent(event);
      
      painter.drawRect(0, 0, width(), height());
   }
}
