package Modelica "Modelica Standard Library (Version 3.1)"
extends Modelica.Icons.Library;

  package Media "Library of media property models"
  extends Modelica.Icons.Library;
  import SI = Modelica.SIunits;

  package Examples
    "Demonstrate usage of property models (currently: simple tests)"
    extends Modelica.Icons.Library;

  model MoistAir "Ideal gas flue gas  model"
      extends Modelica.Icons.Example;
      package Medium = Air.MoistAir "Medium model";
      Medium.BaseProperties medium(
         T(start = 274.0),
         X(start = {0.95,0.05}),
         p(start = 1.0e5));
    //  Medium.SpecificEntropy s=Medium.specificEntropy(medium);
    //  Medium.SpecificEnthalpy h_is = Medium.isentropicEnthalpyApproximation(medium, 2.0e5);
      parameter Medium.MolarMass[2] MMx = {Medium.dryair.MM,Medium.steam.MM}
      "Vector of molar masses (consisting of dry air and of steam)";
      Medium.MolarMass MM = 1/((1-medium.X[1])/MMx[1]+medium.X[1]/MMx[2])
      "Molar mass of gas part of mixture";
    //  Real[4] dddX=Medium.density_derX(medium,MM);

    Medium.ThermodynamicState state1;
    Medium.ThermodynamicState state2;
    Medium.ThermodynamicState smoothState;
    Real m_flow_ext;
    Real der_p;
    Real der_T;
  equation
      der(medium.p) = 0.0;
      der(medium.T) = 90;
      medium.X[Medium.Air] = 0.95;
      //    medium.X[Medium.Water] = 0.05;
      // one simple assumption only for quick testing:
    //  medium.X_liquidWater = if medium.X_sat < medium.X[2] then medium.X[2] - medium.X_sat else 0.0;

     // Smooth state
     m_flow_ext = time - 0.5;
     state1.p = 1.e5*(1+time);
     state1.T = 300 + 10*time;
     state1.X = {time, 1-time};
     state2.p = 1.e5*(1+time/2);
     state2.T = 340 - 20*time;
     state2.X = {0.5*time, 1-0.5*time};
     smoothState = Medium.setSmoothState(m_flow_ext, state1, state2, 0.2);
     der_p = der(smoothState.p);
     der_T = der(smoothState.T);
  end MoistAir;
  end Examples;

  package Interfaces "Interfaces for media models"
    extends Modelica.Icons.Library;
    import SI = Modelica.SIunits;

    partial package PartialMedium
    "Partial medium properties (base package of all media packages)"

      import SI = Modelica.SIunits;
      extends Modelica.Icons.Library;

      // Constants to be set in Medium
      constant
      Modelica.Media.Interfaces.PartialMedium.Choices.IndependentVariables
      ThermoStates "Enumeration type for independent variables";
      constant String mediumName = "unusablePartialMedium" "Name of the medium";
      constant String substanceNames[:]={mediumName}
      "Names of the mixture substances. Set substanceNames={mediumName} if only one substance.";
      constant String extraPropertiesNames[:]=fill("", 0)
      "Names of the additional (extra) transported properties. Set extraPropertiesNames=fill(\"\",0) if unused";
      constant Boolean singleState
      "= true, if u and d are not a function of pressure";
      constant Boolean reducedX=true
      "= true if medium contains the equation sum(X) = 1.0; set reducedX=true if only one substance (see docu for details)";
      constant Boolean fixedX=false
      "= true if medium contains the equation X = reference_X";
      constant AbsolutePressure reference_p=101325
      "Reference pressure of Medium: default 1 atmosphere";
      constant Temperature reference_T=298.15
      "Reference temperature of Medium: default 25 deg Celsius";
      constant MassFraction reference_X[nX]= fill(1/nX, nX)
      "Default mass fractions of medium";
      constant AbsolutePressure p_default=101325
      "Default value for pressure of medium (for initialization)";
      constant Temperature T_default = Modelica.SIunits.Conversions.from_degC(20)
      "Default value for temperature of medium (for initialization)";
      constant SpecificEnthalpy h_default = specificEnthalpy_pTX(p_default, T_default, X_default)
      "Default value for specific enthalpy of medium (for initialization)";
      constant MassFraction X_default[nX]=reference_X
      "Default value for mass fractions of medium (for initialization)";

      final constant Integer nS=size(substanceNames, 1) "Number of substances";
      constant Integer nX = nS "Number of mass fractions";
      constant Integer nXi=if fixedX then 0 else if reducedX then nS - 1 else nS
      "Number of structurally independent mass fractions (see docu for details)";

      final constant Integer nC=size(extraPropertiesNames, 1)
      "Number of extra (outside of standard mass-balance) transported properties";

      replaceable record FluidConstants
      "critical, triple, molecular and other standard data of fluid"
        extends Modelica.Icons.Record;
        String iupacName
        "complete IUPAC name (or common name, if non-existent)";
        String casRegistryNumber
        "chemical abstracts sequencing number (if it exists)";
        String chemicalFormula
        "Chemical formula, (brutto, nomenclature according to Hill";
        String structureFormula "Chemical structure formula";
        MolarMass molarMass "molar mass";
      end FluidConstants;

      replaceable record ThermodynamicState
      "Minimal variable set that is available as input argument to every medium function"
        extends Modelica.Icons.Record;
      end ThermodynamicState;

      replaceable partial model BaseProperties
      "Base properties (p, d, T, h, u, R, MM and, if applicable, X and Xi) of a medium"
        InputAbsolutePressure p "Absolute pressure of medium";
        InputMassFraction[nXi] Xi(start=reference_X[1:nXi])
        "Structurally independent mass fractions";
        InputSpecificEnthalpy h "Specific enthalpy of medium";
        Density d "Density of medium";
        Temperature T "Temperature of medium";
        MassFraction[nX] X(start=reference_X)
        "Mass fractions (= (component mass)/total mass  m_i/m)";
        SpecificInternalEnergy u "Specific internal energy of medium";
        SpecificHeatCapacity R "Gas constant (of mixture if applicable)";
        MolarMass MM "Molar mass (of mixture or single fluid)";
        ThermodynamicState state
        "thermodynamic state record for optional functions";
        parameter Boolean preferredMediumStates=false
        "= true if StateSelect.prefer shall be used for the independent property variables of the medium";
        parameter Boolean standardOrderComponents = true
        "if true, and reducedX = true, the last element of X will be computed from the other ones";
        SI.Conversions.NonSIunits.Temperature_degC T_degC=
            Modelica.SIunits.Conversions.to_degC(T)
        "Temperature of medium in [degC]";
        SI.Conversions.NonSIunits.Pressure_bar p_bar=
         Modelica.SIunits.Conversions.to_bar(p)
        "Absolute pressure of medium in [bar]";

        // Local connector definition, used for equation balancing check
        connector InputAbsolutePressure = input SI.AbsolutePressure
        "Pressure as input signal connector";
        connector InputSpecificEnthalpy = input SI.SpecificEnthalpy
        "Specific enthalpy as input signal connector";
        connector InputMassFraction = input SI.MassFraction
        "Mass fraction as input signal connector";

      equation
        if standardOrderComponents then
          Xi = X[1:nXi];

            if fixedX then
              X = reference_X;
            end if;
            if reducedX and not fixedX then
              X[nX] = 1 - sum(Xi);
            end if;
            for i in 1:nX loop
              assert(X[i] >= -1.e-5 and X[i] <= 1 + 1.e-5, "Mass fraction X[" +
                     String(i) + "] = " + String(X[i]) + "of substance "
                     + substanceNames[i] + "\nof medium " + mediumName + " is not in the range 0..1");
            end for;

        end if;

        assert(p >= 0.0, "Pressure (= " + String(p) + " Pa) of medium \"" +
          mediumName + "\" is negative\n(Temperature = " + String(T) + " K)");
      end BaseProperties;

      replaceable partial function setState_pTX
      "Return thermodynamic state as function of p, T and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "thermodynamic state record";
      end setState_pTX;

      replaceable partial function setState_phX
      "Return thermodynamic state as function of p, h and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "thermodynamic state record";
      end setState_phX;

      replaceable partial function setState_psX
      "Return thermodynamic state as function of p, s and composition X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "thermodynamic state record";
      end setState_psX;

      replaceable partial function setState_dTX
      "Return thermodynamic state as function of d, T and composition X or Xi"
        extends Modelica.Icons.Function;
        input Density d "density";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output ThermodynamicState state "thermodynamic state record";
      end setState_dTX;

      replaceable partial function setSmoothState
      "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
        extends Modelica.Icons.Function;
        input Real x "m_flow or dp";
        input ThermodynamicState state_a "Thermodynamic state if x > 0";
        input ThermodynamicState state_b "Thermodynamic state if x < 0";
        input Real x_small(min=0)
        "Smooth transition in the region -x_small < x < x_small";
        output ThermodynamicState state
        "Smooth thermodynamic state for all x (continuous and differentiable)";
      end setSmoothState;

      replaceable partial function dynamicViscosity "Return dynamic viscosity"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output DynamicViscosity eta "Dynamic viscosity";
      end dynamicViscosity;

      replaceable partial function thermalConductivity
      "Return thermal conductivity"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output ThermalConductivity lambda "Thermal conductivity";
      end thermalConductivity;

      replaceable function prandtlNumber "Return the Prandtl number"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output PrandtlNumber Pr "Prandtl number";
      algorithm
        Pr := dynamicViscosity(state)*specificHeatCapacityCp(state)/thermalConductivity(
          state);
      end prandtlNumber;

      replaceable partial function pressure "Return pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output AbsolutePressure p "Pressure";
      end pressure;

      replaceable partial function temperature "Return temperature"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output Temperature T "Temperature";
      end temperature;

      replaceable partial function density "Return density"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output Density d "Density";
      end density;

      replaceable partial function specificEnthalpy "Return specific enthalpy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificEnthalpy h "Specific enthalpy";
      end specificEnthalpy;

      replaceable partial function specificInternalEnergy
      "Return specific internal energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificEnergy u "Specific internal energy";
      end specificInternalEnergy;

      replaceable partial function specificEntropy "Return specific entropy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificEntropy s "Specific entropy";
      end specificEntropy;

      replaceable partial function specificGibbsEnergy
      "Return specific Gibbs energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificEnergy g "Specific Gibbs energy";
      end specificGibbsEnergy;

      replaceable partial function specificHelmholtzEnergy
      "Return specific Helmholtz energy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificEnergy f "Specific Helmholtz energy";
      end specificHelmholtzEnergy;

      replaceable partial function specificHeatCapacityCp
      "Return specific heat capacity at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificHeatCapacity cp
        "Specific heat capacity at constant pressure";
      end specificHeatCapacityCp;

      function heatCapacity_cp = specificHeatCapacityCp
      "alias for deprecated name";

      replaceable partial function specificHeatCapacityCv
      "Return specific heat capacity at constant volume"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SpecificHeatCapacity cv
        "Specific heat capacity at constant volume";
      end specificHeatCapacityCv;

      function heatCapacity_cv = specificHeatCapacityCv
      "alias for deprecated name";

      replaceable partial function isentropicExponent
      "Return isentropic exponent"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output IsentropicExponent gamma "Isentropic exponent";
      end isentropicExponent;

      replaceable partial function isentropicEnthalpy
      "Return isentropic enthalpy"
        extends Modelica.Icons.Function;
        input AbsolutePressure p_downstream "downstream pressure";
        input ThermodynamicState refState "reference state for entropy";
        output SpecificEnthalpy h_is "Isentropic enthalpy";
      end isentropicEnthalpy;

      replaceable partial function velocityOfSound "Return velocity of sound"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output VelocityOfSound a "Velocity of sound";
      end velocityOfSound;

      replaceable partial function isobaricExpansionCoefficient
      "Return overall the isobaric expansion coefficient beta"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output IsobaricExpansionCoefficient beta
        "Isobaric expansion coefficient";
      end isobaricExpansionCoefficient;

      function beta = isobaricExpansionCoefficient
      "alias for isobaricExpansionCoefficient for user convenience";

      replaceable partial function isothermalCompressibility
      "Return overall the isothermal compressibility factor"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output SI.IsothermalCompressibility kappa "Isothermal compressibility";
      end isothermalCompressibility;

      function kappa = isothermalCompressibility
      "alias of isothermalCompressibility for user convenience";

      // explicit derivative functions for finite element models
      replaceable partial function density_derp_h
      "Return density derivative wrt pressure at const specific enthalpy"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output DerDensityByPressure ddph "Density derivative wrt pressure";
      end density_derp_h;

      replaceable partial function density_derh_p
      "Return density derivative wrt specific enthalpy at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output DerDensityByEnthalpy ddhp
        "Density derivative wrt specific enthalpy";
      end density_derh_p;

      replaceable partial function density_derp_T
      "Return density derivative wrt pressure at const temperature"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output DerDensityByPressure ddpT "Density derivative wrt pressure";
      end density_derp_T;

      replaceable partial function density_derT_p
      "Return density derivative wrt temperature at constant pressure"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output DerDensityByTemperature ddTp
        "Density derivative wrt temperature";
      end density_derT_p;

      replaceable partial function density_derX
      "Return density derivative wrt mass fraction"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output Density[nX] dddX "Derivative of density wrt mass fraction";
      end density_derX;

      replaceable partial function molarMass
      "Return the molar mass of the medium"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state record";
        output MolarMass MM "Mixture molar mass";
      end molarMass;

      replaceable function specificEnthalpy_pTX
      "Return specific enthalpy from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:]=reference_X "Mass fractions";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_pTX(p,T,X));
      end specificEnthalpy_pTX;

      replaceable function density_pTX "Return density from p, T, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        input MassFraction X[:] "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_pTX(p,T,X));
      end density_pTX;

      replaceable function temperature_phX
      "Return temperature from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_phX(p,h,X));
      end temperature_phX;

      replaceable function density_phX "Return density from p, h, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_phX(p,h,X));
      end density_phX;

      replaceable function temperature_psX
      "Return temperature from p,s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Temperature T "Temperature";
      algorithm
        T := temperature(setState_psX(p,s,X));
      end temperature_psX;

      replaceable function density_psX "Return density from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output Density d "Density";
      algorithm
        d := density(setState_psX(p,s,X));
      end density_psX;

      replaceable function specificEnthalpy_psX
      "Return specific enthalpy from p, s, and X or Xi"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        input MassFraction X[:]=reference_X "Mass fractions";
        output SpecificEnthalpy h "Specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_psX(p,s,X));
      end specificEnthalpy_psX;

      type AbsolutePressure = SI.AbsolutePressure (
          min=0,
          max=1.e8,
          nominal=1.e5,
          start=1.e5)
      "Type for absolute pressure with medium specific attributes";

      type Density = SI.Density (
          min=0,
          max=1.e5,
          nominal=1,
          start=1) "Type for density with medium specific attributes";
      type DynamicViscosity = SI.DynamicViscosity (
          min=0,
          max=1.e8,
          nominal=1.e-3,
          start=1.e-3)
      "Type for dynamic viscosity with medium specific attributes";
      type EnthalpyFlowRate = SI.EnthalpyFlowRate (
          nominal=1000.0,
          min=-1.0e8,
          max=1.e8)
      "Type for enthalpy flow rate with medium specific attributes";
      type MassFlowRate = SI.MassFlowRate (
          quantity="MassFlowRate." + mediumName,
          min=-1.0e5,
          max=1.e5) "Type for mass flow rate with medium specific attributes";
      type MassFraction = Real (
          quantity="MassFraction",
          final unit="kg/kg",
          min=0,
          max=1,
          nominal=0.1) "Type for mass fraction with medium specific attributes";
      type MoleFraction = Real (
          quantity="MoleFraction",
          final unit="mol/mol",
          min=0,
          max=1,
          nominal=0.1) "Type for mole fraction with medium specific attributes";
      type MolarMass = SI.MolarMass (
          min=0.001,
          max=0.25,
          nominal=0.032) "Type for molar mass with medium specific attributes";
      type MolarVolume = SI.MolarVolume (
          min=1e-6,
          max=1.0e6,
          nominal=1.0) "Type for molar volume with medium specific attributes";
      type IsentropicExponent = SI.RatioOfSpecificHeatCapacities (
          min=1,
          max=500000,
          nominal=1.2,
          start=1.2)
      "Type for isentropic exponent with medium specific attributes";
      type SpecificEnergy = SI.SpecificEnergy (
          min=-1.0e8,
          max=1.e8,
          nominal=1.e6)
      "Type for specific energy with medium specific attributes";
      type SpecificInternalEnergy = SpecificEnergy
      "Type for specific internal energy with medium specific attributes";
      type SpecificEnthalpy = SI.SpecificEnthalpy (
          min=-1.0e8,
          max=1.e8,
          nominal=1.e6)
      "Type for specific enthalpy with medium specific attributes";
      type SpecificEntropy = SI.SpecificEntropy (
          min=-1.e6,
          max=1.e6,
          nominal=1.e3)
      "Type for specific entropy with medium specific attributes";
      type SpecificHeatCapacity = SI.SpecificHeatCapacity (
          min=0,
          max=1.e6,
          nominal=1.e3,
          start=1.e3)
      "Type for specific heat capacity with medium specific attributes";
      type SurfaceTension = SI.SurfaceTension
      "Type for surface tension with medium specific attributes";
      type Temperature = SI.Temperature (
          min=1,
          max=1.e4,
          nominal=300,
          start=300) "Type for temperature with medium specific attributes";
      type ThermalConductivity = SI.ThermalConductivity (
          min=0,
          max=500,
          nominal=1,
          start=1)
      "Type for thermal conductivity with medium specific attributes";
      type PrandtlNumber = SI.PrandtlNumber (
          min=1e-3,
          max=1e5,
          nominal=1.0)
      "Type for Prandtl number with medium specific attributes";
      type VelocityOfSound = SI.Velocity (
          min=0,
          max=1.e5,
          nominal=1000,
          start=1000)
      "Type for velocity of sound with medium specific attributes";
      type ExtraProperty = Real (min=0.0, start=1.0)
      "Type for unspecified, mass-specific property transported by flow";
      type CumulativeExtraProperty = Real (min=0.0, start=1.0)
      "Type for conserved integral of unspecified, mass specific property";
      type ExtraPropertyFlowRate = Real(unit="kg/s")
      "Type for flow rate of unspecified, mass-specific property";
      type IsobaricExpansionCoefficient = Real (
          min=0,
          max=1.0e8,
          unit="1/K")
      "Type for isobaric expansion coefficient with medium specific attributes";
      type DipoleMoment = Real (
          min=0.0,
          max=2.0,
          unit="debye",
          quantity="ElectricDipoleMoment")
      "Type for dipole moment with medium specific attributes";

      type DerDensityByPressure = SI.DerDensityByPressure
      "Type for partial derivative of density with resect to pressure with medium specific attributes";
      type DerDensityByEnthalpy = SI.DerDensityByEnthalpy
      "Type for partial derivative of density with resect to enthalpy with medium specific attributes";
      type DerEnthalpyByPressure = SI.DerEnthalpyByPressure
      "Type for partial derivative of enthalpy with resect to pressure with medium specific attributes";
      type DerDensityByTemperature = SI.DerDensityByTemperature
      "Type for partial derivative of density with resect to temperature with medium specific attributes";

      package Choices "Types, constants to define menu choices"

        type IndependentVariables = enumeration(
          T "Temperature",
          pT "Pressure, Temperature",
          ph "Pressure, Specific Enthalpy",
          phX "Pressure, Specific Enthalpy, Mass Fraction",
          pTX "Pressure, Temperature, Mass Fractions",
          dTX "Density, Temperature, Mass Fractions")
        "Enumeration defining the independent variables of a medium";

        type Init = enumeration(
          NoInit "NoInit (no initialization)",
          InitialStates "InitialStates (initialize medium states)",
          SteadyState "SteadyState (initialize in steady state)",
          SteadyMass
            "SteadyMass (initialize density or pressure in steady state)")
        "Enumeration defining initialization for fluid flow";

        type ReferenceEnthalpy = enumeration(
          ZeroAt0K
            "The enthalpy is 0 at 0 K (default), if the enthalpy of formation is excluded", 

          ZeroAt25C
            "The enthalpy is 0 at 25 degC, if the enthalpy of formation is excluded", 

          UserDefined
            "The user-defined reference enthalpy is used at 293.15 K (25 degC)")
        "Enumeration defining the reference enthalpy of a medium";

        type ReferenceEntropy = enumeration(
          ZeroAt0K "The entropy is 0 at 0 K (default)",
          ZeroAt0C "The entropy is 0 at 0 degC",
          UserDefined
            "The user-defined reference entropy is used at 293.15 K (25 degC)")
        "Enumeration defining the reference entropy of a medium";

        type pd = enumeration(
          default "Default (no boundary condition for p or d)",
          p_known "p_known (pressure p is known)",
          d_known "d_known (density d is known)")
        "Enumeration defining whether p or d are known for the boundary condition";

        type Th = enumeration(
          default "Default (no boundary condition for T or h)",
          T_known "T_known (temperature T is known)",
          h_known "h_known (specific enthalpy h is known)")
        "Enumeration defining whether T or h are known as boundary condition";

      end Choices;

    end PartialMedium;

    partial package PartialPureSubstance
    "base class for pure substances of one chemical substance"
      extends PartialMedium(final reducedX = true, final fixedX=true);

     replaceable function setState_pT "Return thermodynamic state from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output ThermodynamicState state "thermodynamic state record";
     algorithm
        state := setState_pTX(p,T,fill(0,0));
     end setState_pT;

      replaceable function setState_ph
      "Return thermodynamic state from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output ThermodynamicState state "thermodynamic state record";
      algorithm
        state := setState_phX(p,h,fill(0, 0));
      end setState_ph;

      replaceable function setState_ps
      "Return thermodynamic state from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output ThermodynamicState state "thermodynamic state record";
      algorithm
        state := setState_psX(p,s,fill(0,0));
      end setState_ps;

      replaceable function setState_dT
      "Return thermodynamic state from d and T"
        extends Modelica.Icons.Function;
        input Density d "density";
        input Temperature T "Temperature";
        output ThermodynamicState state "thermodynamic state record";
      algorithm
        state := setState_dTX(d,T,fill(0,0));
      end setState_dT;

      replaceable function density_ph "Return density from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output Density d "Density";
      algorithm
        d := density_phX(p, h, fill(0,0));
      end density_ph;

      replaceable function temperature_ph "Return temperature from p and h"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        output Temperature T "Temperature";
      algorithm
        T := temperature_phX(p, h, fill(0,0));
      end temperature_ph;

      replaceable function pressure_dT "Return pressure from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        output AbsolutePressure p "Pressure";
      algorithm
        p := pressure(setState_dTX(d, T, fill(0,0)));
      end pressure_dT;

      replaceable function specificEnthalpy_dT
      "Return specific enthalpy from d and T"
        extends Modelica.Icons.Function;
        input Density d "Density";
        input Temperature T "Temperature";
        output SpecificEnthalpy h "specific enthalpy";
      algorithm
        h := specificEnthalpy(setState_dTX(d, T, fill(0,0)));
      end specificEnthalpy_dT;

      replaceable function specificEnthalpy_ps
      "Return specific enthalpy from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output SpecificEnthalpy h "specific enthalpy";
      algorithm
        h := specificEnthalpy_psX(p,s,fill(0,0));
      end specificEnthalpy_ps;

      replaceable function temperature_ps "Return temperature from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output Temperature T "Temperature";
      algorithm
        T := temperature_psX(p,s,fill(0,0));
      end temperature_ps;

      replaceable function density_ps "Return density from p and s"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input SpecificEntropy s "Specific entropy";
        output Density d "Density";
      algorithm
        d := density_psX(p, s, fill(0,0));
      end density_ps;

      replaceable function specificEnthalpy_pT
      "Return specific enthalpy from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output SpecificEnthalpy h "specific enthalpy";
      algorithm
        h := specificEnthalpy_pTX(p, T, fill(0,0));
      end specificEnthalpy_pT;

      replaceable function density_pT "Return density from p and T"
        extends Modelica.Icons.Function;
        input AbsolutePressure p "Pressure";
        input Temperature T "Temperature";
        output Density d "Density";
      algorithm
        d := density(setState_pTX(p, T, fill(0,0)));
      end density_pT;

      redeclare replaceable partial model extends BaseProperties(
        final standardOrderComponents=true)
      end BaseProperties;
    end PartialPureSubstance;

  partial package PartialMixtureMedium
    "Base class for pure substances of several chemical substances"
      extends PartialMedium;

      redeclare replaceable record extends ThermodynamicState
      "thermodynamic state variables"
        AbsolutePressure p "Absolute pressure of medium";
        Temperature T "Temperature of medium";
        MassFraction X[nX]
        "Mass fractions (= (component mass)/total mass  m_i/m)";
      end ThermodynamicState;

      redeclare replaceable record extends FluidConstants
      "extended fluid constants"
        Temperature criticalTemperature "critical temperature";
        AbsolutePressure criticalPressure "critical pressure";
        MolarVolume criticalMolarVolume "critical molar Volume";
        Real acentricFactor "Pitzer acentric factor";
        Temperature triplePointTemperature "triple point temperature";
        AbsolutePressure triplePointPressure "triple point pressure";
        Temperature meltingPoint "melting point at 101325 Pa";
        Temperature normalBoilingPoint "normal boiling point (at 101325 Pa)";
        DipoleMoment dipoleMoment
        "dipole moment of molecule in Debye (1 debye = 3.33564e10-30 C.m)";
        Boolean hasIdealGasHeatCapacity=false
        "true if ideal gas heat capacity is available";
        Boolean hasCriticalData=false "true if critical data are known";
        Boolean hasDipoleMoment=false "true if a dipole moment known";
        Boolean hasFundamentalEquation=false "true if a fundamental equation";
        Boolean hasLiquidHeatCapacity=false
        "true if liquid heat capacity is available";
        Boolean hasSolidHeatCapacity=false
        "true if solid heat capacity is available";
        Boolean hasAccurateViscosityData=false
        "true if accurate data for a viscosity function is available";
        Boolean hasAccurateConductivityData=false
        "true if accurate data for thermal conductivity is available";
        Boolean hasVapourPressureCurve=false
        "true if vapour pressure data, e.g. Antoine coefficents are known";
        Boolean hasAcentricFactor=false
        "true if Pitzer accentric factor is known";
        SpecificEnthalpy HCRIT0=0.0
        "Critical specific enthalpy of the fundamental equation";
        SpecificEntropy SCRIT0=0.0
        "Critical specific entropy of the fundamental equation";
        SpecificEnthalpy deltah=0.0
        "Difference between specific enthalpy model (h_m) and f.eq. (h_f) (h_m - h_f)";
        SpecificEntropy deltas=0.0
        "Difference between specific enthalpy model (s_m) and f.eq. (s_f) (s_m - s_f)";
      end FluidConstants;

    constant FluidConstants[nS] fluidConstants "constant data for the fluid";

    replaceable function gasConstant
      "Return the gas constant of the mixture (also for liquids)"
        extends Modelica.Icons.Function;
        input ThermodynamicState state "thermodynamic state";
        output SI.SpecificHeatCapacity R "mixture gas constant";
    end gasConstant;

      function moleToMassFractions
      "Return mass fractions X from mole fractions"
        extends Modelica.Icons.Function;
        input SI.MoleFraction moleFractions[:] "Mole fractions of mixture";
        input MolarMass[:] MMX "molar masses of components";
        output SI.MassFraction X[size(moleFractions, 1)]
        "Mass fractions of gas mixture";
    protected
        MolarMass Mmix =  moleFractions*MMX "molar mass of mixture";
      algorithm
        for i in 1:size(moleFractions, 1) loop
          X[i] := moleFractions[i]*MMX[i] /Mmix;
        end for;
      end moleToMassFractions;

      function massToMoleFractions
      "Return mole fractions from mass fractions X"
        extends Modelica.Icons.Function;
        input SI.MassFraction X[:] "Mass fractions of mixture";
        input SI.MolarMass[:] MMX "molar masses of components";
        output SI.MoleFraction moleFractions[size(X, 1)]
        "Mole fractions of gas mixture";
    protected
        Real invMMX[size(X, 1)] "inverses of molar weights";
        SI.MolarMass Mmix "molar mass of mixture";
      algorithm
        for i in 1:size(X, 1) loop
          invMMX[i] := 1/MMX[i];
        end for;
        Mmix := 1/(X*invMMX);
        for i in 1:size(X, 1) loop
          moleFractions[i] := Mmix*X[i]/MMX[i];
        end for;
      end massToMoleFractions;

  end PartialMixtureMedium;

    partial package PartialCondensingGases
    "Base class for mixtures of condensing and non-condensing gases"
      extends PartialMixtureMedium(
           ThermoStates = Choices.IndependentVariables.pTX);

    replaceable partial function saturationPressure
      "Return saturation pressure of condensing fluid"
      extends Modelica.Icons.Function;
      input Temperature Tsat "saturation temperature";
      output AbsolutePressure psat "saturation pressure";
    end saturationPressure;

    replaceable partial function enthalpyOfVaporization
      "Return vaporization enthalpy of condensing fluid"
      extends Modelica.Icons.Function;
      input Temperature T "temperature";
      output SpecificEnthalpy r0 "vaporization enthalpy";
    end enthalpyOfVaporization;

    replaceable partial function enthalpyOfLiquid
      "Return liquid enthalpy of condensing fluid"
      extends Modelica.Icons.Function;
      input Temperature T "temperature";
      output SpecificEnthalpy h "liquid enthalpy";
    end enthalpyOfLiquid;

    replaceable partial function enthalpyOfGas
      "Return enthalpy of non-condensing gas mixture"
      extends Modelica.Icons.Function;
      input Temperature T "temperature";
      input MassFraction[:] X "vector of mass fractions";
      output SpecificEnthalpy h "specific enthalpy";
    end enthalpyOfGas;

    replaceable partial function enthalpyOfCondensingGas
      "Return enthalpy of condensing gas (most often steam)"
      extends Modelica.Icons.Function;
      input Temperature T "temperature";
      output SpecificEnthalpy h "specific enthalpy";
    end enthalpyOfCondensingGas;

    replaceable partial function enthalpyOfNonCondensingGas
      "Return enthalpy of the non-condensing species"
      extends Modelica.Icons.Function;
      input Temperature T "temperature";
      output SpecificEnthalpy h "specific enthalpy";
    end enthalpyOfNonCondensingGas;
    end PartialCondensingGases;
  end Interfaces;

  package Common
    "data structures and fundamental functions for fluid properties"
    extends Modelica.Icons.Library;

    function smoothStep
    "Approximation of a general step, such that the characteristic is continuous and differentiable"
      extends Modelica.Icons.Function;
      input Real x "Abszissa value";
      input Real y1 "Ordinate value for x > 0";
      input Real y2 "Ordinate value for x < 0";
      input Real x_small(min=0) = 1e-5
      "Approximation of step for -x_small <= x <= x_small; x_small > 0 required";
      output Real y
      "Ordinate value to approximate y = if x > 0 then y1 else y2";
    algorithm
      y := smooth(1, if x >  x_small then y1 else 
                     if x < -x_small then y2 else 
                     if abs(x_small)>0 then (x/x_small)*((x/x_small)^2 - 3)*(y2-y1)/4 + (y1+y2)/2 else (y1+y2)/2);

    end smoothStep;

   package OneNonLinearEquation
    "Determine solution of a non-linear algebraic equation in one unknown without derivatives in a reliable and efficient way"
     extends Modelica.Icons.Library;

      replaceable record f_nonlinear_Data
      "Data specific for function f_nonlinear"
        extends Modelica.Icons.Record;
      end f_nonlinear_Data;

      replaceable partial function f_nonlinear
      "Nonlinear algebraic equation in one unknown: y = f_nonlinear(x,p,X)"
        extends Modelica.Icons.Function;
        input Real x "Independent variable of function";
        input Real p = 0.0
        "disregaded variables (here always used for pressure)";
        input Real[:] X = fill(0,0)
        "disregaded variables (her always used for composition)";
        input f_nonlinear_Data f_nonlinear_data
        "Additional data for the function";
        output Real y "= f_nonlinear(x)";
        // annotation(derivative(zeroDerivative=y)); // this must hold for all replaced functions
      end f_nonlinear;

      replaceable function solve
      "Solve f_nonlinear(x_zero)=y_zero; f_nonlinear(x_min) - y_zero and f_nonlinear(x_max)-y_zero must have different sign"
        import Modelica.Utilities.Streams.error;
        extends Modelica.Icons.Function;
        input Real y_zero
        "Determine x_zero, such that f_nonlinear(x_zero) = y_zero";
        input Real x_min "Minimum value of x";
        input Real x_max "Maximum value of x";
        input Real pressure = 0.0
        "disregaded variables (here always used for pressure)";
        input Real[:] X = fill(0,0)
        "disregaded variables (here always used for composition)";
         input f_nonlinear_Data f_nonlinear_data
        "Additional data for function f_nonlinear";
         input Real x_tol =  100*Modelica.Constants.eps
        "Relative tolerance of the result";
         output Real x_zero "f_nonlinear(x_zero) = y_zero";
    protected
         constant Real eps = Modelica.Constants.eps "machine epsilon";
         Real a = x_min "Current best minimum interval value";
         Real b = x_max "Current best maximum interval value";
         Real c "Intermediate point a <= c <= b";
         Real d;
         Real e "b - a";
         Real m;
         Real s;
         Real p;
         Real q;
         Real r;
         Real tol;
         Real fa "= f_nonlinear(a) - y_zero";
         Real fb "= f_nonlinear(b) - y_zero";
         Real fc;
         Boolean found = false;
      algorithm
         // Check that f(x_min) and f(x_max) have different sign
         fa :=f_nonlinear(x_min, pressure, X, f_nonlinear_data) - y_zero;
         fb :=f_nonlinear(x_max, pressure, X, f_nonlinear_data) - y_zero;
         fc := fb;
         if fa > 0.0 and fb > 0.0 or 
            fa < 0.0 and fb < 0.0 then
            error("The arguments x_min and x_max to OneNonLinearEquation.solve(..)\n" +
                  "do not bracket the root of the single non-linear equation:\n" +
                  "  x_min  = " + String(x_min) + "\n" +
                  "  x_max  = " + String(x_max) + "\n" +
                  "  y_zero = " + String(y_zero) + "\n" +
                  "  fa = f(x_min) - y_zero = " + String(fa) + "\n" +
                  "  fb = f(x_max) - y_zero = " + String(fb) + "\n" +
                  "fa and fb must have opposite sign which is not the case");
         end if;

         // Initialize variables
         c :=a;
         fc :=fa;
         e :=b - a;
         d :=e;

         // Search loop
         while not found loop
            if abs(fc) < abs(fb) then
               a :=b;
               b :=c;
               c :=a;
               fa :=fb;
               fb :=fc;
               fc :=fa;
            end if;

            tol :=2*eps*abs(b) + x_tol;
            m :=(c - b)/2;

            if abs(m) <= tol or fb == 0.0 then
               // root found (interval is small enough)
               found :=true;
               x_zero :=b;
            else
               // Determine if a bisection is needed
               if abs(e) < tol or abs(fa) <= abs(fb) then
                  e :=m;
                  d :=e;
               else
                  s :=fb/fa;
                  if a == c then
                     // linear interpolation
                     p :=2*m*s;
                     q :=1 - s;
                  else
                     // inverse quadratic interpolation
                     q :=fa/fc;
                     r :=fb/fc;
                     p :=s*(2*m*q*(q - r) - (b - a)*(r - 1));
                     q :=(q - 1)*(r - 1)*(s - 1);
                  end if;

                  if p > 0 then
                     q :=-q;
                  else
                     p :=-p;
                  end if;

                  s :=e;
                  e :=d;
                  if 2*p < 3*m*q-abs(tol*q) and p < abs(0.5*s*q) then
                     // interpolation successful
                     d :=p/q;
                  else
                     // use bi-section
                     e :=m;
                     d :=e;
                  end if;
               end if;

               // Best guess value is defined as "a"
               a :=b;
               fa :=fb;
               b :=b + (if abs(d) > tol then d else if m > 0 then tol else -tol);
               fb :=f_nonlinear(b, pressure, X, f_nonlinear_data) - y_zero;

               if fb > 0 and fc > 0 or 
                  fb < 0 and fc < 0 then
                  // initialize variables
                  c :=a;
                  fc :=fa;
                  e :=b - a;
                  d :=e;
               end if;
            end if;
         end while;
      end solve;

   end OneNonLinearEquation;
  end Common;

    package Air "Medium models for air"

      package MoistAir "Air: Moist air model (240 ... 400 K)"
        extends Interfaces.PartialCondensingGases(
           mediumName="Moist air",
           substanceNames={"water", "air"},
           final reducedX=true,
           final singleState=false,
           reference_X={0.01,0.99},
           fluidConstants = {IdealGases.Common.FluidData.H2O,IdealGases.Common.FluidData.N2});

        constant Integer Water=1
        "Index of water (in substanceNames, massFractions X, etc.)";

        constant Integer Air=2
        "Index of air (in substanceNames, massFractions X, etc.)";

        constant Real k_mair =  steam.MM/dryair.MM "ratio of molar weights";

        constant IdealGases.Common.DataRecord dryair = IdealGases.Common.SingleGasesData.Air;

        constant IdealGases.Common.DataRecord steam = IdealGases.Common.SingleGasesData.H2O;

        constant SI.MolarMass[2] MMX = {steam.MM,dryair.MM}
        "Molar masses of components";
        import Modelica.Media.Interfaces;
        import Modelica.Math;
        import SI = Modelica.SIunits;
        import Cv = Modelica.SIunits.Conversions;
        import Modelica.Constants;
        import Modelica.Media.IdealGases.Common.SingleGasNasa;

        redeclare record extends ThermodynamicState
        "ThermodynamicState record for moist air"
        end ThermodynamicState;

        redeclare replaceable model extends BaseProperties(
          T(stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default),
          p(stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default),
          Xi(stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default),
          redeclare final constant Boolean standardOrderComponents=true)
        "Moist air base properties record"

          /* p, T, X = X[Water] are used as preferred states, since only then all
     other quantities can be computed in a recursive sequence.
     If other variables are selected as states, static state selection
     is no longer possible and non-linear algebraic equations occur.
      */
          MassFraction x_water "Mass of total water/mass of dry air";
          Real phi "Relative humidity";

      protected
          MassFraction X_liquid "Mass fraction of liquid or solid water";
          MassFraction X_steam "Mass fraction of steam water";
          MassFraction X_air "Mass fraction of air";
          MassFraction X_sat
          "Steam water mass fraction of saturation boundary in kg_water/kg_moistair";
          MassFraction x_sat
          "Steam water mass content of saturation boundary in kg_water/kg_dryair";
          AbsolutePressure p_steam_sat "Partial saturation pressure of steam";
        equation
          assert(T >= 200.0 and T <= 423.15, "
Temperature T is not in the allowed range
200.0 K <= (T ="     + String(T) + " K) <= 423.15 K
required from medium model \""           + mediumName + "\".");
          MM = 1/(Xi[Water]/MMX[Water]+(1.0-Xi[Water])/MMX[Air]);

          p_steam_sat = min(saturationPressure(T),0.999*p);
          X_sat = min(p_steam_sat * k_mair/max(100*Constants.eps, p - p_steam_sat)*(1 - Xi[Water]), 1.0)
          "Water content at saturation with respect to actual water content";
          X_liquid = max(Xi[Water] - X_sat, 0.0);
          X_steam  = Xi[Water]-X_liquid;
          X_air    = 1-Xi[Water];

          h = specificEnthalpy_pTX(p,T,Xi);
          R = dryair.R*(X_air/(1 - X_liquid)) + steam.R*X_steam/(1 - X_liquid);
          //
          u = h - R*T;
          d = p/(R*T);
          /* Note, u and d are computed under the assumption that the volume of the liquid
         water is neglible with respect to the volume of air and of steam
      */
          state.p = p;
          state.T = T;
          state.X = X;

          // these x are per unit mass of DRY air!
          x_sat    = k_mair*p_steam_sat/max(100*Constants.eps,p - p_steam_sat);
          x_water = Xi[Water]/max(X_air,100*Constants.eps);
          phi = p/p_steam_sat*Xi[Water]/(Xi[Water] + k_mair*X_air);
        end BaseProperties;

        redeclare function setState_pTX
        "Return thermodynamic state as function of pressure p, temperature T and composition X"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input Temperature T "Temperature";
          input MassFraction X[:]=reference_X "Mass fractions";
          output ThermodynamicState state "Thermodynamic state";
        algorithm
          state := if size(X,1) == nX then ThermodynamicState(p=p,T=T, X=X) else 
                 ThermodynamicState(p=p,T=T, X=cat(1,X,{1-sum(X)}));
        end setState_pTX;

        redeclare function setState_phX
        "Return thermodynamic state as function of pressure p, specific enthalpy h and composition X"
          extends Modelica.Icons.Function;
          input AbsolutePressure p "Pressure";
          input SpecificEnthalpy h "Specific enthalpy";
          input MassFraction X[:]=reference_X "Mass fractions";
          output ThermodynamicState state "Thermodynamic state";
        algorithm
          state := if size(X,1) == nX then ThermodynamicState(p=p,T=T_phX(p,h,X),X=X) else 
                 ThermodynamicState(p=p,T=T_phX(p,h,X), X=cat(1,X,{1-sum(X)}));
        end setState_phX;

        redeclare function setState_dTX
        "Return thermodynamic state as function of density d, temperature T and composition X"
          extends Modelica.Icons.Function;
          input Density d "density";
          input Temperature T "Temperature";
          input MassFraction X[:]=reference_X "Mass fractions";
          output ThermodynamicState state "Thermodynamic state";
        algorithm
          state := if size(X,1) == nX then ThermodynamicState(p=d*({steam.R,dryair.R}*X)*T,T=T,X=X) else 
                 ThermodynamicState(p=d*({steam.R,dryair.R}*cat(1,X,{1-sum(X)}))*T,T=T, X=cat(1,X,{1-sum(X)}));
        end setState_dTX;

      redeclare function extends setSmoothState
        "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
      algorithm
        state := ThermodynamicState(p=Media.Common.smoothStep(x, state_a.p, state_b.p, x_small),
                                    T=Media.Common.smoothStep(x, state_a.T, state_b.T, x_small),
                                    X=Media.Common.smoothStep(x, state_a.X, state_b.X, x_small));
      end setSmoothState;

        redeclare function extends gasConstant
        "Return ideal gas constant as a function from thermodynamic state, only valid for phi<1"

        algorithm
          R := dryair.R*(1-state.X[Water]) + steam.R*state.X[Water];
        end gasConstant;

        function saturationPressureLiquid
        "Return saturation pressure of water as a function of temperature T in the range of 273.16 to 373.16 K"

          extends Modelica.Icons.Function;
          input SI.Temperature Tsat "saturation temperature";
          output SI.AbsolutePressure psat "saturation pressure";
        algorithm
          psat := 611.657*Math.exp(17.2799 - 4102.99/(Tsat - 35.719));
        end saturationPressureLiquid;

        function saturationPressureLiquid_der
        "Time derivative of saturationPressureLiquid"

          extends Modelica.Icons.Function;
          input SI.Temperature Tsat "Saturation temperature";
          input Real dTsat(unit="K/s") "Saturation temperature derivative";
          output Real psat_der(unit="Pa/s") "Saturation pressure";
        algorithm
        /*psat := 611.657*Math.exp(17.2799 - 4102.99/(Tsat - 35.719));*/
          psat_der:=611.657*Math.exp(17.2799 - 4102.99/(Tsat - 35.719))*4102.99*dTsat/(Tsat - 35.719)/(Tsat - 35.719);

        end saturationPressureLiquid_der;

        function sublimationPressureIce
        "Return sublimation pressure of water as a function of temperature T between 223.16 and 273.16 K"

          extends Modelica.Icons.Function;
          input SI.Temperature Tsat "sublimation temperature";
          output SI.AbsolutePressure psat "sublimation pressure";
        algorithm
          psat := 611.657*Math.exp(22.5159*(1.0 - 273.16/Tsat));
        end sublimationPressureIce;

        function sublimationPressureIce_der
        "Derivative function for 'sublimationPressureIce'"

          extends Modelica.Icons.Function;
          input SI.Temperature Tsat "Sublimation temperature";
          input Real dTsat(unit="K/s")
          "Time derivative of sublimation temperature";
          output Real psat_der(unit="Pa/s") "Sublimation pressure";
        algorithm
          /*psat := 611.657*Math.exp(22.5159*(1.0 - 273.16/Tsat));*/
          psat_der:=611.657*Math.exp(22.5159*(1.0 - 273.16/Tsat))*22.5159*273.16*dTsat/Tsat/Tsat;
        end sublimationPressureIce_der;

        redeclare function extends saturationPressure
        "Return saturation pressure of water as a function of temperature T between 223.16 and 373.16 K"

        algorithm
          psat := Utilities.spliceFunction(saturationPressureLiquid(Tsat),sublimationPressureIce(Tsat),Tsat-273.16,1.0);
        end saturationPressure;

        function saturationPressure_der
        "Derivative function for 'saturationPressure'"
          input Temperature Tsat "Saturation temperature";
          input Real dTsat(unit="K/s")
          "Time derivative of saturation temperature";
          output Real psat_der(unit="Pa/s") "Saturation pressure";

        algorithm
          /*psat := Utilities.spliceFunction(saturationPressureLiquid(Tsat),sublimationPressureIce(Tsat),Tsat-273.16,1.0);*/
          psat_der := Utilities.spliceFunction_der(
            saturationPressureLiquid(Tsat),
            sublimationPressureIce(Tsat),
            Tsat - 273.16,
            1.0,
            saturationPressureLiquid_der(Tsat=Tsat, dTsat=dTsat),
            sublimationPressureIce_der(Tsat=Tsat, dTsat=dTsat),
            dTsat,
            0);
        end saturationPressure_der;

       redeclare function extends enthalpyOfVaporization
        "Return enthalpy of vaporization of water as a function of temperature T, 0 - 130 degC"

       algorithm
        /*r0 := 1e3*(2501.0145 - (T - 273.15)*(2.3853 + (T - 273.15)*(0.002969 - (T
      - 273.15)*(7.5293e-5 + (T - 273.15)*4.6084e-7))));*/
       //katrin: replaced by linear correlation, simpler and more accurate in the entire region
       //source VDI-Waermeatlas, linear inter- and extrapolation between values for 0.01°C and 40°C.
       r0:=(2405900-2500500)/(40-0)*(T-273.16)+2500500;
       end enthalpyOfVaporization;

       redeclare function extends enthalpyOfLiquid
        "Return enthalpy of liquid water as a function of temperature T(use enthalpyOfWater instead)"

       algorithm
         h := (T - 273.15)*1e3*(4.2166 - 0.5*(T - 273.15)*(0.0033166 + 0.333333*(T - 273.15)*(0.00010295
            - 0.25*(T - 273.15)*(1.3819e-6 + 0.2*(T - 273.15)*7.3221e-9))));
       end enthalpyOfLiquid;

       redeclare function extends enthalpyOfGas
        "Return specific enthalpy of gas (air and steam) as a function of temperature T and composition X"

       algorithm
         h := SingleGasNasa.h_Tlow(data=steam, T=T, refChoice=3, h_off=46479.819+2501014.5)*X[Water]
              + SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684)*(1.0-X[Water]);
       end enthalpyOfGas;

       redeclare function extends enthalpyOfCondensingGas
        "Return specific enthalpy of steam as a function of temperature T"

       algorithm
         h := SingleGasNasa.h_Tlow(data=steam, T=T, refChoice=3, h_off=46479.819+2501014.5);
       end enthalpyOfCondensingGas;

       redeclare function extends enthalpyOfNonCondensingGas
        "Return specific enthalpy of dry air as a function of temperature T"

       algorithm
         h := SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684);
       end enthalpyOfNonCondensingGas;

      function enthalpyOfWater
        "Computes specific enthalpy of water (solid/liquid) near atmospheric pressure from temperature T"
        input SIunits.Temperature T "Temperature";
        output SIunits.SpecificEnthalpy h "Specific enthalpy of water";
      algorithm
      /*simple model assuming constant properties:
  heat capacity of liquid water:4200 J/kg
  heat capacity of solid water: 2050 J/kg
  enthalpy of fusion (liquid=>solid): 333000 J/kg*/

        h:=Utilities.spliceFunction(4200*(T-273.15),2050*(T-273.15)-333000,T-273.16,0.1);
      end enthalpyOfWater;

      function enthalpyOfWater_der "Derivative function of enthalpyOfWater"
        input SIunits.Temperature T "Temperature";
        input Real dT(unit="K/s") "Time derivative of temperature";
        output Real dh(unit="J/(kg.s)") "Time derivative of specific enthalpy";
      algorithm
      /*simple model assuming constant properties:
  heat capacity of liquid water:4200 J/kg
  heat capacity of solid water: 2050 J/kg
  enthalpy of fusion (liquid=>solid): 333000 J/kg*/

        //h:=Utilities.spliceFunction(4200*(T-273.15),2050*(T-273.15)-333000,T-273.16,0.1);
        dh:=Utilities.spliceFunction_der(4200*(T-273.15),2050*(T-273.15)-333000,T-273.16,0.1,4200*dT,2050*dT,dT,0);
      end enthalpyOfWater_der;

       redeclare function extends pressure
        "Returns pressure of ideal gas as a function of the thermodynamic state record"

       algorithm
        p := state.p;
       end pressure;

       redeclare function extends temperature
        "Return temperature of ideal gas as a function of the thermodynamic state record"

       algorithm
         T := state.T;
       end temperature;

      function T_phX
        "Return temperature as a function of pressure p, specific enthalpy h and composition X"
        input AbsolutePressure p "Pressure";
        input SpecificEnthalpy h "Specific enthalpy";
        input MassFraction[:] X "Mass fractions of composition";
        output Temperature T "Temperature";

      protected
      package Internal
          "Solve h(data,T) for T with given h (use only indirectly via temperature_phX)"
        extends Modelica.Media.Common.OneNonLinearEquation;
        redeclare record extends f_nonlinear_Data
            "Data to be passed to non-linear function"
          extends Modelica.Media.IdealGases.Common.DataRecord;
        end f_nonlinear_Data;

        redeclare function extends f_nonlinear
        algorithm
            y := h_pTX(p,x,X);
        end f_nonlinear;

        // Dummy definition has to be added for current Dymola
        redeclare function extends solve
        end solve;
      end Internal;

      algorithm
        T := Internal.solve(h, 240, 400, p, X[1:nXi], steam);
      end T_phX;

       redeclare function extends density
        "Returns density of ideal gas as a function of the thermodynamic state record"

       algorithm
         d := state.p/(gasConstant(state)*state.T);
       end density;

      redeclare function extends specificEnthalpy
        "Return specific enthalpy of moist air as a function of the thermodynamic state record"

      algorithm
        h := h_pTX(state.p, state.T, state.X);
      end specificEnthalpy;

      function h_pTX
        "Return specific enthalpy of moist air as a function of pressure p, temperature T and composition X"
        extends Modelica.Icons.Function;
        input SI.Pressure p "Pressure";
        input SI.Temperature T "Temperature";
        input SI.MassFraction X[:] "Mass fractions of moist air";
        output SI.SpecificEnthalpy h "Specific enthalpy at p, T, X";
      protected
        SI.AbsolutePressure p_steam_sat "Partial saturation pressure of steam";
        SI.MassFraction X_sat "Absolute humidity per unit mass of moist air";
        SI.MassFraction X_liquid "mass fraction of liquid water";
        SI.MassFraction X_steam "mass fraction of steam water";
        SI.MassFraction X_air "mass fraction of air";
      algorithm
        p_steam_sat :=saturationPressure(T);
        //p_steam_sat :=min(saturationPressure(T), 0.999*p);
        X_sat:=min(p_steam_sat*k_mair/max(100*Constants.eps, p - p_steam_sat)*(1 - X[
          Water]), 1.0);
        X_liquid :=max(X[Water] - X_sat, 0.0);
        X_steam  :=X[Water] - X_liquid;
        X_air    :=1 - X[Water];
       /* h        := {SingleGasNasa.h_Tlow(data=steam,  T=T, refChoice=3, h_off=46479.819+2501014.5),
               SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684)}*
    {X_steam, X_air} + enthalpyOfLiquid(T)*X_liquid;*/
         h        := {SingleGasNasa.h_Tlow(data=steam,  T=T, refChoice=3, h_off=46479.819+2501014.5),
                     SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684)}*
          {X_steam, X_air} + enthalpyOfWater(T)*X_liquid;
      end h_pTX;

      function h_pTX_der "Derivative function of h_pTX"
        extends Modelica.Icons.Function;
        input SI.Pressure p "Pressure";
        input SI.Temperature T "Temperature";
        input SI.MassFraction X[:] "Mass fractions of moist air";
        input Real dp(unit="Pa/s") "Pressure derivative";
        input Real dT(unit="K/s") "Temperature derivative";
        input Real dX[:](each unit="1/s") "Composition derivative";
        output Real h_der(unit="J/(kg.s)")
          "Time derivative of specific enthalpy";
      protected
        SI.AbsolutePressure p_steam_sat "Partial saturation pressure of steam";
        SI.MassFraction X_sat "Absolute humidity per unit mass of moist air";
        SI.MassFraction X_liquid "Mass fraction of liquid water";
        SI.MassFraction X_steam "Mass fraction of steam water";
        SI.MassFraction X_air "Mass fraction of air";
        SI.MassFraction x_sat
          "Absolute humidity per unit mass of dry air at saturation";
        Real dX_steam(unit="1/s") "Time deriveative of steam mass fraction";
        Real dX_air(unit="1/s") "Time derivative of dry air mass fraction";
        Real dX_liq(unit="1/s")
          "Time derivative of liquid/solid water mass fraction";
        Real dps(unit="Pa/s") "Time derivative of saturation pressure";
        Real dx_sat(unit="1/s")
          "Time derivative of abolute humidity per unit mass of dry air";
      algorithm
        p_steam_sat :=saturationPressure(T);
        x_sat:=p_steam_sat*k_mair/max(100*Modelica.Constants.eps, p - p_steam_sat);
        X_sat:=min(x_sat*(1 - X[Water]), 1.0);
        X_liquid :=Utilities.spliceFunction(X[Water] - X_sat, 0.0, X[Water] - X_sat,1e-6);
        X_steam  :=X[Water] - X_liquid;
        X_air    :=1 - X[Water];

        dX_air:=-dX[Water];
        dps:=saturationPressure_der(Tsat=T, dTsat=dT);
        dx_sat:=k_mair*(dps*(p-p_steam_sat)-p_steam_sat*(dp-dps))/(p-p_steam_sat)/(p-p_steam_sat);
        dX_liq:=Utilities.spliceFunction_der(X[Water] - X_sat, 0.0, X[Water] - X_sat,1e-6,(1+x_sat)*dX[Water]-(1-X[Water])*dx_sat,0.0,(1+x_sat)*dX[Water]-(1-X[Water])*dx_sat,0.0);
        //dX_liq:=if X[Water]>=X_sat then (1+x_sat)*dX[Water]-(1-X[Water])*dx_sat else 0;
        dX_steam:=dX[Water]-dX_liq;

        h_der:= X_steam*Modelica.Media.IdealGases.Common.SingleGasNasa.h_Tlow_der(data=steam, T=T, refChoice=3, h_off=46479.819+2501014.5, dT=dT)+
                dX_steam*Modelica.Media.IdealGases.Common.SingleGasNasa.h_Tlow(data=steam,  T=T, refChoice=3, h_off=46479.819+2501014.5) +
                X_air*Modelica.Media.IdealGases.Common.SingleGasNasa.h_Tlow_der(data=dryair, T=T, refChoice=3, h_off=25104.684, dT=dT) +
                dX_air*Modelica.Media.IdealGases.Common.SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684) +
                X_liquid*enthalpyOfWater_der(T=T, dT=dT) +
                dX_liq*enthalpyOfWater(T);

      end h_pTX_der;

      redeclare function extends isentropicExponent
        "Return isentropic exponent (only for gas fraction!)"
      algorithm
        gamma := specificHeatCapacityCp(state)/specificHeatCapacityCv(state);
      end isentropicExponent;

      redeclare function extends specificInternalEnergy
        "Return specific internal energy of moist air as a function of the thermodynamic state record"
        extends Modelica.Icons.Function;
        output SI.SpecificInternalEnergy u "Specific internal energy";
      algorithm
         u := specificInternalEnergy_pTX(state.p,state.T,state.X);

      end specificInternalEnergy;

      function specificInternalEnergy_pTX
        "Return specific internal energy of moist air as a function of pressure p, temperature T and composition X"
        input SI.Pressure p "Pressure";
        input SI.Temperature T "Temperature";
        input SI.MassFraction X[:] "Mass fractions of moist air";
        output SI.SpecificInternalEnergy u "Specific internal energy";
      protected
        SI.AbsolutePressure p_steam_sat "Partial saturation pressure of steam";
        SI.MassFraction X_liquid "Mass fraction of liquid water";
        SI.MassFraction X_steam "Mass fraction of steam water";
        SI.MassFraction X_air "Mass fraction of air";
        SI.MassFraction X_sat "Absolute humidity per unit mass of moist air";
        Real R_gas "Ideal gas constant";
      algorithm
        p_steam_sat :=saturationPressure(T);
        X_sat:=min(p_steam_sat*k_mair/max(100*Constants.eps, p - p_steam_sat)*(1 - X[
          Water]), 1.0);
        X_liquid :=max(X[Water] - X_sat, 0.0);
        X_steam  :=X[Water] - X_liquid;
        X_air    :=1 - X[Water];
        R_gas:= dryair.R*X_air/(1-X_liquid)+steam.R* X_steam/(1-X_liquid);
        u       := X_steam*SingleGasNasa.h_Tlow(data=steam,  T=T, refChoice=3, h_off=46479.819+2501014.5)+
                   X_air*SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684)
                   + enthalpyOfWater(T)*X_liquid-R_gas*T;

      end specificInternalEnergy_pTX;

      function specificInternalEnergy_pTX_der
        "Derivative function for specificInternalEnergy_pTX"
        input SI.Pressure p "Pressure";
        input SI.Temperature T "Temperature";
        input SI.MassFraction X[:] "Mass fractions of moist air";
        input Real dp(unit="Pa/s") "Pressure derivative";
        input Real dT(unit="K/s") "Temperature derivative";
        input Real dX[:](each unit="1/s") "Mass fraction derivatives";
        output Real u_der(unit="J/(kg.s)")
          "Specific internal energy derivative";
      protected
        SI.AbsolutePressure p_steam_sat "Partial saturation pressure of steam";
        SI.MassFraction X_liquid "Mass fraction of liquid water";
        SI.MassFraction X_steam "Mass fraction of steam water";
        SI.MassFraction X_air "Mass fraction of air";
        SI.MassFraction X_sat "Absolute humidity per unit mass of moist air";
        SI.SpecificHeatCapacity R_gas "Ideal gas constant";

        SI.MassFraction x_sat
          "Absolute humidity per unit mass of dry air at saturation";
        Real dX_steam(unit="1/s") "Time deriveative of steam mass fraction";
        Real dX_air(unit="1/s") "Time derivative of dry air mass fraction";
        Real dX_liq(unit="1/s")
          "Time derivative of liquid/solid water mass fraction";
        Real dps(unit="Pa/s") "Time derivative of saturation pressure";
        Real dx_sat(unit="1/s")
          "Time derivative of abolute humidity per unit mass of dry air";
        Real dR_gas(unit="J/(kg.K.s)") "Time derivative of ideal gas constant";
      algorithm
        p_steam_sat :=saturationPressure(T);
        x_sat:=p_steam_sat*k_mair/max(100*Modelica.Constants.eps, p - p_steam_sat);
        X_sat:=min(x_sat*(1 - X[Water]), 1.0);
        X_liquid :=Utilities.spliceFunction(X[Water] - X_sat, 0.0, X[Water] - X_sat,1e-6);
        X_steam  :=X[Water] - X_liquid;
        X_air    :=1 - X[Water];
        R_gas:= steam.R*X_steam/(1-X_liquid)+dryair.R* X_air/(1-X_liquid);

        dX_air:=-dX[Water];
        dps:=saturationPressure_der(Tsat=T, dTsat=dT);
        dx_sat:=k_mair*(dps*(p-p_steam_sat)-p_steam_sat*(dp-dps))/(p-p_steam_sat)/(p-p_steam_sat);
        dX_liq:=Utilities.spliceFunction_der(X[Water] - X_sat, 0.0, X[Water] - X_sat,1e-6,(1+x_sat)*dX[Water]-(1-X[Water])*dx_sat,0.0,(1+x_sat)*dX[Water]-(1-X[Water])*dx_sat,0.0);
        dX_steam:=dX[Water]-dX_liq;
        dR_gas:=(steam.R*(dX_steam*(1-X_liquid)+dX_liq*X_steam)+dryair.R*(dX_air*(1-X_liquid)+dX_liq*X_air))/(1-X_liquid)/(1-X_liquid);

        u_der:=X_steam*SingleGasNasa.h_Tlow_der(data=steam, T=T, refChoice=3, h_off=46479.819+2501014.5, dT=dT)+
               dX_steam*SingleGasNasa.h_Tlow(data=steam,  T=T, refChoice=3, h_off=46479.819+2501014.5) +
               X_air*SingleGasNasa.h_Tlow_der(data=dryair, T=T, refChoice=3, h_off=25104.684, dT=dT) +
               dX_air*SingleGasNasa.h_Tlow(data=dryair, T=T, refChoice=3, h_off=25104.684) +
               X_liquid*enthalpyOfWater_der(T=T, dT=dT) +
               dX_liq*enthalpyOfWater(T) - dR_gas*T-R_gas*dT;
      end specificInternalEnergy_pTX_der;

       redeclare function extends specificEntropy
        "Return specific entropy from thermodynamic state record, only valid for phi<1"

      protected
          MoleFraction[2] Y = massToMoleFractions(state.X,{steam.MM,dryair.MM})
          "molar fraction";
       algorithm
         s:=SingleGasNasa.s0_Tlow(dryair, state.T)*(1-state.X[Water])
           + SingleGasNasa.s0_Tlow(steam, state.T)*state.X[Water]
           - (state.X[Water]*Modelica.Constants.R/MMX[Water]*(if state.X[Water]<Modelica.Constants.eps then state.X[Water] else Modelica.Math.log(Y[Water]*state.p/reference_p))
             + (1-state.X[Water])*Modelica.Constants.R/MMX[Air]*(if (1-state.X[Water])<Modelica.Constants.eps then (1-state.X[Water]) else Modelica.Math.log(Y[Air]*state.p/reference_p)));
       end specificEntropy;

      redeclare function extends specificGibbsEnergy
        "Return specific Gibbs energy as a function of the thermodynamic state record, only valid for phi<1"
        extends Modelica.Icons.Function;
      algorithm
        g := h_pTX(state.p,state.T,state.X) - state.T*specificEntropy(state);
      end specificGibbsEnergy;

      redeclare function extends specificHelmholtzEnergy
        "Return specific Helmholtz energy as a function of the thermodynamic state record, only valid for phi<1"
        extends Modelica.Icons.Function;
      algorithm
        f := h_pTX(state.p,state.T,state.X) - gasConstant(state)*state.T - state.T*specificEntropy(state);
      end specificHelmholtzEnergy;

       redeclare function extends specificHeatCapacityCp
        "Return specific heat capacity at constant pressure as a function of the thermodynamic state record"

      protected
         Real dT(unit="s/K") = 1.0;
       algorithm
         cp := h_pTX_der(state.p,state.T,state.X, 0.0, 1.0, zeros(size(state.X,1)))*dT
          "Definition of cp: dh/dT @ constant p";
         //      cp:= SingleGasNasa.cp_Tlow(dryair, state.T)*(1-state.X[Water])
         //        + SingleGasNasa.cp_Tlow(steam, state.T)*state.X[Water];
       end specificHeatCapacityCp;

      redeclare function extends specificHeatCapacityCv
        "Return specific heat capacity at constant volume as a function of the thermodynamic state record"

      algorithm
        cv:= SingleGasNasa.cp_Tlow(dryair, state.T)*(1-state.X[Water]) +
          SingleGasNasa.cp_Tlow(steam, state.T)*state.X[Water]
          - gasConstant(state);
      end specificHeatCapacityCv;

      redeclare function extends dynamicViscosity
        "Return dynamic viscosity as a function of the thermodynamic state record, valid from 73.15 K to 373.15 K"

          import Modelica.Media.Incompressible.TableBased.Polynomials_Temp;
      algorithm
        eta := Polynomials_Temp.evaluate({(-4.96717436974791E-011), 5.06626785714286E-008, 1.72937731092437E-005},
             Cv.to_degC(state.T));
      end dynamicViscosity;

      redeclare function extends thermalConductivity
        "Return thermal conductivity as a function of the thermodynamic state record, valid from 73.15 K to 373.15 K"

          import Modelica.Media.Incompressible.TableBased.Polynomials_Temp;
      algorithm
        lambda := Polynomials_Temp.evaluate({(-4.8737307422969E-008), 7.67803133753502E-005, 0.0241814385504202},
         Cv.to_degC(state.T));
      end thermalConductivity;

        package Utilities "utility functions"

          function spliceFunction "Spline interpolation of two functions"
              input Real pos "Returned value for x-deltax >= 0";
              input Real neg "Returned value for x+deltax <= 0";
              input Real x "Function argument";
              input Real deltax=1 "Region around x with spline interpolation";
              output Real out;
        protected
              Real scaledX;
              Real scaledX1;
              Real y;
          algorithm
              scaledX1 := x/deltax;
              scaledX := scaledX1*Modelica.Math.asin(1);
              if scaledX1 <= -0.999999999 then
                y := 0;
              elseif scaledX1 >= 0.999999999 then
                y := 1;
              else
                y := (Modelica.Math.tanh(Modelica.Math.tan(scaledX)) + 1)/2;
              end if;
              out := pos*y + (1 - y)*neg;
          end spliceFunction;

          function spliceFunction_der "Derivative of spliceFunction"
              input Real pos;
              input Real neg;
              input Real x;
              input Real deltax=1;
              input Real dpos;
              input Real dneg;
              input Real dx;
              input Real ddeltax=0;
              output Real out;
        protected
              Real scaledX;
              Real scaledX1;
              Real dscaledX1;
              Real y;
          algorithm
              scaledX1 := x/deltax;
              scaledX := scaledX1*Modelica.Math.asin(1);
              dscaledX1 := (dx - scaledX1*ddeltax)/deltax;
              if scaledX1 <= -0.99999999999 then
                y := 0;
              elseif scaledX1 >= 0.9999999999 then
                y := 1;
              else
                y := (Modelica.Math.tanh(Modelica.Math.tan(scaledX)) + 1)/2;
              end if;
              out := dpos*y + (1 - y)*dneg;
              if (abs(scaledX1) < 1) then
                out := out + (pos - neg)*dscaledX1*Modelica.Math.asin(1)/2/(
                  Modelica.Math.cosh(Modelica.Math.tan(scaledX))*Modelica.Math.cos(
                  scaledX))^2;
              end if;
          end spliceFunction_der;
        end Utilities;
      end MoistAir;
    end Air;

    package IdealGases
    "Data and models of ideal gases (single, fixed and dynamic mixtures) from NASA source"
    extends Modelica.Icons.Library;

      package Common "Common packages and data for the ideal gas models"
      extends Modelica.Icons.Library;

      record DataRecord
        "Coefficient data record for properties of ideal gases based on NASA source"
        extends Modelica.Icons.Record;
        String name "Name of ideal gas";
        SI.MolarMass MM "Molar mass";
        SI.SpecificEnthalpy Hf "Enthalpy of formation at 298.15K";
        SI.SpecificEnthalpy H0 "H0(298.15K) - H0(0K)";
        SI.Temperature Tlimit
          "Temperature limit between low and high data sets";
        Real alow[7] "Low temperature coefficients a";
        Real blow[2] "Low temperature constants b";
        Real ahigh[7] "High temperature coefficients a";
        Real bhigh[2] "High temperature constants b";
        SI.SpecificHeatCapacity R "Gas constant";
      end DataRecord;

      partial package SingleGasNasa
        "Medium model of an ideal gas based on NASA source"

        extends Interfaces.PartialPureSubstance(
           ThermoStates = Choices.IndependentVariables.pT,
           redeclare final record FluidConstants = 
              Modelica.Media.IdealGases.Common.FluidData.FluidConstants,
           mediumName=data.name,
           substanceNames={data.name},
           singleState=false,
           Temperature(min=200, max=6000, start=500, nominal=500),
           SpecificEnthalpy(start=if referenceChoice==ReferenceEnthalpy.ZeroAt0K then data.H0 else 
              if referenceChoice==ReferenceEnthalpy.UserDefined then h_offset else 0, nominal=1.0e5),
           Density(start=10, nominal=10),
           AbsolutePressure(start=10e5, nominal=10e5));

        redeclare record extends ThermodynamicState
          "thermodynamic state variables for ideal gases"
          AbsolutePressure p "Absolute pressure of medium";
          Temperature T "Temperature of medium";
        end ThermodynamicState;

        import SI = Modelica.SIunits;
        import Modelica.Math;
        import
          Modelica.Media.Interfaces.PartialMedium.Choices.ReferenceEnthalpy;

        constant Boolean excludeEnthalpyOfFormation=true
          "If true, enthalpy of formation Hf is not included in specific enthalpy h";
        constant ReferenceEnthalpy referenceChoice=Choices.
              ReferenceEnthalpy.ZeroAt0K "Choice of reference enthalpy";
        constant SpecificEnthalpy h_offset=0.0
          "User defined offset for reference enthalpy, if referenceChoice = UserDefined";

        constant IdealGases.Common.DataRecord data
          "Data record of ideal gas substance";

        constant FluidConstants[nS] fluidConstants
          "constant data for the fluid";

        redeclare model extends BaseProperties(
         T(stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default),
         p(stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default))
          "Base properties of ideal gas medium"
        equation
          assert(T >= 200 and T <= 6000, "
Temperature T (= "       + String(T) + " K) is not in the allowed range
200 K <= T <= 6000 K required from medium model \""       + mediumName + "\".
");
          MM = data.MM;
          R = data.R;
          h = h_T(data, T, excludeEnthalpyOfFormation, referenceChoice, h_offset);
          u = h - R*T;

          // Has to be written in the form d=f(p,T) in order that static
          // state selection for p and T is possible
          d = p/(R*T);
          // connect state with BaseProperties
          state.T = T;
          state.p = p;
        end BaseProperties;

          redeclare function setState_pTX
          "Return thermodynamic state as function of p, T and composition X"
            extends Modelica.Icons.Function;
            input AbsolutePressure p "Pressure";
            input Temperature T "Temperature";
            input MassFraction X[:]=reference_X "Mass fractions";
            output ThermodynamicState state;
          algorithm
            state := ThermodynamicState(p=p,T=T);
          end setState_pTX;

          redeclare function setState_phX
          "Return thermodynamic state as function of p, h and composition X"
            extends Modelica.Icons.Function;
            input AbsolutePressure p "Pressure";
            input SpecificEnthalpy h "Specific enthalpy";
            input MassFraction X[:]=reference_X "Mass fractions";
            output ThermodynamicState state;
          algorithm
            state := ThermodynamicState(p=p,T=T_h(h));
          end setState_phX;

          redeclare function setState_psX
          "Return thermodynamic state as function of p, s and composition X"
            extends Modelica.Icons.Function;
            input AbsolutePressure p "Pressure";
            input SpecificEntropy s "Specific entropy";
            input MassFraction X[:]=reference_X "Mass fractions";
            output ThermodynamicState state;
          algorithm
            state := ThermodynamicState(p=p,T=T_ps(p,s));
          end setState_psX;

          redeclare function setState_dTX
          "Return thermodynamic state as function of d, T and composition X"
            extends Modelica.Icons.Function;
            input Density d "density";
            input Temperature T "Temperature";
            input MassFraction X[:]=reference_X "Mass fractions";
            output ThermodynamicState state;
          algorithm
            state := ThermodynamicState(p=d*data.R*T,T=T);
          end setState_dTX;

            redeclare function extends setSmoothState
          "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
            algorithm
              state := ThermodynamicState(p=Media.Common.smoothStep(x, state_a.p, state_b.p, x_small),
                                          T=Media.Common.smoothStep(x, state_a.T, state_b.T, x_small));
            end setSmoothState;

        redeclare function extends pressure "return pressure of ideal gas"
        algorithm
          p := state.p;
        end pressure;

        redeclare function extends temperature
          "return temperature of ideal gas"
        algorithm
          T := state.T;
        end temperature;

        redeclare function extends density "return density of ideal gas"
        algorithm
          d := state.p/(data.R*state.T);
        end density;

        redeclare function extends specificEnthalpy "Return specific enthalpy"
          extends Modelica.Icons.Function;
        algorithm
          h := h_T(data,state.T);
        end specificEnthalpy;

        redeclare function extends specificInternalEnergy
          "Return specific internal energy"
          extends Modelica.Icons.Function;
        algorithm
          u := h_T(data,state.T) - data.R*state.T;
        end specificInternalEnergy;

        redeclare function extends specificEntropy "Return specific entropy"
          extends Modelica.Icons.Function;
        algorithm
          s := s0_T(data, state.T) - data.R*Modelica.Math.log(state.p/reference_p);
        end specificEntropy;

        redeclare function extends specificGibbsEnergy
          "Return specific Gibbs energy"
          extends Modelica.Icons.Function;
        algorithm
          g := h_T(data,state.T) - state.T*specificEntropy(state);
        end specificGibbsEnergy;

        redeclare function extends specificHelmholtzEnergy
          "Return specific Helmholtz energy"
          extends Modelica.Icons.Function;
        algorithm
          f := h_T(data,state.T) - data.R*state.T - state.T*specificEntropy(state);
        end specificHelmholtzEnergy;

        redeclare function extends specificHeatCapacityCp
          "Return specific heat capacity at constant pressure"
        algorithm
          cp := cp_T(data, state.T);
        end specificHeatCapacityCp;

        redeclare function extends specificHeatCapacityCv
          "Compute specific heat capacity at constant volume from temperature and gas data"
        algorithm
          cv := cp_T(data, state.T) - data.R;
        end specificHeatCapacityCv;

        redeclare function extends isentropicExponent
          "Return isentropic exponent"
        algorithm
          gamma := specificHeatCapacityCp(state)/specificHeatCapacityCv(state);
        end isentropicExponent;

        redeclare function extends velocityOfSound "Return velocity of sound"
          extends Modelica.Icons.Function;
        algorithm
          a := sqrt(max(0,data.R*state.T*cp_T(data, state.T)/specificHeatCapacityCv(state)));
        end velocityOfSound;

        function isentropicEnthalpyApproximation
          "approximate method of calculating h_is from upstream properties and downstream pressure"
          extends Modelica.Icons.Function;
          input SI.Pressure p2 "downstream pressure";
          input ThermodynamicState state "properties at upstream location";
          input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
          input ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
          input SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
          output SI.SpecificEnthalpy h_is "isentropic enthalpy";
        protected
          IsentropicExponent gamma =  isentropicExponent(state)
            "Isentropic exponent";
        algorithm
          h_is := h_T(data,state.T,exclEnthForm,refChoice,h_off) +
            gamma/(gamma - 1.0)*state.p/density(state)*((p2/state.p)^((gamma - 1)/gamma) - 1.0);
        end isentropicEnthalpyApproximation;

        redeclare function extends isentropicEnthalpy
          "Return isentropic enthalpy"
        input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
        input ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
        input SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
        algorithm
          h_is := isentropicEnthalpyApproximation(p_downstream,refState,exclEnthForm,refChoice,h_off);
        end isentropicEnthalpy;

        redeclare function extends isobaricExpansionCoefficient
          "Returns overall the isobaric expansion coefficient beta"
        algorithm
          beta := 1/state.T;
        end isobaricExpansionCoefficient;

        redeclare function extends isothermalCompressibility
          "Returns overall the isothermal compressibility factor"
        algorithm
          kappa := 1.0/state.p;
        end isothermalCompressibility;

        redeclare function extends density_derp_T
          "Returns the partial derivative of density with respect to pressure at constant temperature"
        algorithm
          ddpT := 1/(state.T*data.R);
        end density_derp_T;

        redeclare function extends density_derT_p
          "Returns the partial derivative of density with respect to temperature at constant pressure"
        algorithm
          ddTp := -state.p/(state.T*state.T*data.R);
        end density_derT_p;

        redeclare function extends density_derX
          "Returns the partial derivative of density with respect to mass fractions at constant pressure and temperature"
        algorithm
          dddX := fill(0,nX);
        end density_derX;

        function cp_T
          "Compute specific heat capacity at constant pressure from temperature and gas data"
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          output SI.SpecificHeatCapacity cp
            "Specific heat capacity at temperature T";
        algorithm
          cp := smooth(0,if T < data.Tlimit then data.R*(1/(T*T)*(data.alow[1] + T*(
            data.alow[2] + T*(1.*data.alow[3] + T*(data.alow[4] + T*(data.alow[5] + T
            *(data.alow[6] + data.alow[7]*T))))))) else data.R*(1/(T*T)*(data.ahigh[1]
             + T*(data.ahigh[2] + T*(1.*data.ahigh[3] + T*(data.ahigh[4] + T*(data.
            ahigh[5] + T*(data.ahigh[6] + data.ahigh[7]*T))))))));
        end cp_T;

        function cp_Tlow
          "Compute specific heat capacity at constant pressure, low T region"
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          output SI.SpecificHeatCapacity cp
            "Specific heat capacity at temperature T";
        algorithm
          cp := data.R*(1/(T*T)*(data.alow[1] + T*(
            data.alow[2] + T*(1.*data.alow[3] + T*(data.alow[4] + T*(data.alow[5] + T
            *(data.alow[6] + data.alow[7]*T)))))));
        end cp_Tlow;

        function cp_Tlow_der
          "Compute specific heat capacity at constant pressure, low T region"
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          input Real dT "Temperature derivative";
          output Real cp_der "Derivative of specific heat capacity";
        algorithm
          cp_der := dT*data.R/(T*T*T)*(-2*data.alow[1] + T*(
            -data.alow[2] + T*T*(data.alow[4] + T*(2.*data.alow[5] + T
            *(3.*data.alow[6] + 4.*data.alow[7]*T)))));
        end cp_Tlow_der;

        function h_T "Compute specific enthalpy from temperature and gas data; reference is decided by the
    refChoice input, or by the referenceChoice package constant by default"
          import Modelica.Media.Interfaces.PartialMedium.Choices;
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
          input Choices.ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
          input SI.SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
          output SI.SpecificEnthalpy h "Specific enthalpy at temperature T";
            //     annotation (InlineNoEvent=false, Inline=false,
            //                 derivative(zeroDerivative=data,
            //                            zeroDerivative=exclEnthForm,
            //                            zeroDerivative=refChoice,
            //                            zeroDerivative=h_off) = h_T_der);
        algorithm
          h := smooth(0,(if T < data.Tlimit then data.R*((-data.alow[1] + T*(data.
            blow[1] + data.alow[2]*Math.log(T) + T*(1.*data.alow[3] + T*(0.5*data.
            alow[4] + T*(1/3*data.alow[5] + T*(0.25*data.alow[6] + 0.2*data.alow[7]*T))))))
            /T) else data.R*((-data.ahigh[1] + T*(data.bhigh[1] + data.ahigh[2]*
            Math.log(T) + T*(1.*data.ahigh[3] + T*(0.5*data.ahigh[4] + T*(1/3*data.
            ahigh[5] + T*(0.25*data.ahigh[6] + 0.2*data.ahigh[7]*T))))))/T)) + (if 
            exclEnthForm then -data.Hf else 0.0) + (if (refChoice
             == Choices.ReferenceEnthalpy.ZeroAt0K) then data.H0 else 0.0) + (if 
            refChoice == Choices.ReferenceEnthalpy.UserDefined then h_off else 
                  0.0));
        end h_T;

        function h_T_der "derivative function for h_T"
          import Modelica.Media.Interfaces.PartialMedium.Choices;
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
          input Choices.ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
          input SI.SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
          input Real dT "Temperature derivative";
          output Real h_der "Specific enthalpy at temperature T";
        algorithm
          h_der := dT*cp_T(data,T);
        end h_T_der;

        function h_Tlow "Compute specific enthalpy, low T region; reference is decided by the
    refChoice input, or by the referenceChoice package constant by default"
          import Modelica.Media.Interfaces.PartialMedium.Choices;
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
          input Choices.ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
          input SI.SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
          output SI.SpecificEnthalpy h "Specific enthalpy at temperature T";
            //     annotation (Inline=false,InlineNoEvent=false, derivative(zeroDerivative=data,
            //                                zeroDerivative=exclEnthForm,
            //                                zeroDerivative=refChoice,
            //                                zeroDerivative=h_off) = h_Tlow_der);
        algorithm
          h := data.R*((-data.alow[1] + T*(data.
            blow[1] + data.alow[2]*Math.log(T) + T*(1.*data.alow[3] + T*(0.5*data.
            alow[4] + T*(1/3*data.alow[5] + T*(0.25*data.alow[6] + 0.2*data.alow[7]*T))))))
            /T) + (if 
            exclEnthForm then -data.Hf else 0.0) + (if (refChoice
             == Choices.ReferenceEnthalpy.ZeroAt0K) then data.H0 else 0.0) + (if 
            refChoice == Choices.ReferenceEnthalpy.UserDefined then h_off else 
                  0.0);
        end h_Tlow;

        function h_Tlow_der "Compute specific enthalpy, low T region; reference is decided by the
    refChoice input, or by the referenceChoice package constant by default"
          import Modelica.Media.Interfaces.PartialMedium.Choices;
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          input Boolean exclEnthForm=excludeEnthalpyOfFormation
            "If true, enthalpy of formation Hf is not included in specific enthalpy h";
          input Choices.ReferenceEnthalpy refChoice=referenceChoice
            "Choice of reference enthalpy";
          input SI.SpecificEnthalpy h_off=h_offset
            "User defined offset for reference enthalpy, if referenceChoice = UserDefined";
          input Real dT(unit="K/s") "Temperature derivative";
          output Real h_der(unit="J/(kg.s)")
            "Derivative of specific enthalpy at temperature T";
        algorithm
          h_der := dT*cp_Tlow(data,T);
        end h_Tlow_der;

        function s0_T "Compute specific entropy from temperature and gas data"
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          output SI.SpecificEntropy s "Specific entropy at temperature T";
        algorithm
          s := noEvent(if T < data.Tlimit then data.R*(data.blow[2] - 0.5*data.alow[
            1]/(T*T) - data.alow[2]/T + data.alow[3]*Math.log(T) + T*(
            data.alow[4] + T*(0.5*data.alow[5] + T*(1/3*data.alow[6] + 0.25*data.alow[
            7]*T)))) else data.R*(data.bhigh[2] - 0.5*data.ahigh[1]/(T*T) - data.
            ahigh[2]/T + data.ahigh[3]*Math.log(T) + T*(data.ahigh[4]
             + T*(0.5*data.ahigh[5] + T*(1/3*data.ahigh[6] + 0.25*data.ahigh[7]*T)))));
        end s0_T;

        function s0_Tlow "Compute specific entropy, low T region"
          extends Modelica.Icons.Function;
          input IdealGases.Common.DataRecord data "Ideal gas data";
          input SI.Temperature T "Temperature";
          output SI.SpecificEntropy s "Specific entropy at temperature T";
        algorithm
          s := data.R*(data.blow[2] - 0.5*data.alow[
            1]/(T*T) - data.alow[2]/T + data.alow[3]*Math.log(T) + T*(
            data.alow[4] + T*(0.5*data.alow[5] + T*(1/3*data.alow[6] + 0.25*data.alow[
            7]*T))));
        end s0_Tlow;

        function dynamicViscosityLowPressure
          "Dynamic viscosity of low pressure gases"
          extends Modelica.Icons.Function;
          input SI.Temp_K T "Gas temperature";
          input SI.Temp_K Tc "Critical temperature of gas";
          input SI.MolarMass M "Molar mass of gas";
          input SI.MolarVolume Vc "Critical molar volume of gas";
          input Real w "Acentric factor of gas";
          input DipoleMoment mu "Dipole moment of gas molecule";
          input Real k =  0.0 "Special correction for highly polar substances";
          output SI.DynamicViscosity eta "Dynamic viscosity of gas";
        protected
          parameter Real Const1_SI=40.785*10^(-9.5)
            "Constant in formula for eta converted to SI units";
          parameter Real Const2_SI=131.3/1000.0
            "Constant in formula for mur converted to SI units";
          Real mur=Const2_SI*mu/sqrt(Vc*Tc)
            "Dimensionless dipole moment of gas molecule";
          Real Fc=1 - 0.2756*w + 0.059035*mur^4 + k
            "Factor to account for molecular shape and polarities of gas";
          Real Tstar "Dimensionless temperature defined by equation below";
          Real Ov "Viscosity collision integral for the gas";

        algorithm
          Tstar := 1.2593*T/Tc;
          Ov := 1.16145*Tstar^(-0.14874) + 0.52487*exp(-0.7732*Tstar) + 2.16178*exp(-2.43787
            *Tstar);
          eta := Const1_SI*Fc*sqrt(M*T)/(Vc^(2/3)*Ov);
        end dynamicViscosityLowPressure;

        redeclare replaceable function extends dynamicViscosity
          "dynamic viscosity"
        algorithm
          assert(fluidConstants[1].hasCriticalData,
          "Failed to compute dynamicViscosity: For the species \"" + mediumName + "\" no critical data is available.");
          assert(fluidConstants[1].hasDipoleMoment,
          "Failed to compute dynamicViscosity: For the species \"" + mediumName + "\" no critical data is available.");
          eta := dynamicViscosityLowPressure(state.T,
                             fluidConstants[1].criticalTemperature,
                             fluidConstants[1].molarMass,
                             fluidConstants[1].criticalMolarVolume,
                             fluidConstants[1].acentricFactor,
                             fluidConstants[1].dipoleMoment);
        end dynamicViscosity;

        function thermalConductivityEstimate
          "Thermal conductivity of polyatomic gases(Eucken and Modified Eucken correlation)"
          extends Modelica.Icons.Function;
          input SpecificHeatCapacity Cp "Constant pressure heat capacity";
          input DynamicViscosity eta "Dynamic viscosity";
          input Integer method(min=1,max=2)=1
            "1: Eucken Method, 2: Modified Eucken Method";
          output ThermalConductivity lambda "Thermal conductivity [W/(m.k)]";
        algorithm
          lambda := if method == 1 then eta*(Cp - data.R + (9/4)*data.R) else eta*(Cp
             - data.R)*(1.32 + 1.77/((Cp/Modelica.Constants.R) - 1.0));
        end thermalConductivityEstimate;

        redeclare replaceable function extends thermalConductivity
          "thermal conductivity of gas"
          input Integer method=1 "1: Eucken Method, 2: Modified Eucken Method";
        algorithm
          assert(fluidConstants[1].hasCriticalData,
          "Failed to compute thermalConductivity: For the species \"" + mediumName + "\" no critical data is available.");
          lambda := thermalConductivityEstimate(specificHeatCapacityCp(state),
            dynamicViscosity(state), method=method);
        end thermalConductivity;

        redeclare function extends molarMass
          "return the molar mass of the medium"
        algorithm
          MM := data.MM;
        end molarMass;

        function T_h "Compute temperature from specific enthalpy"
          input SpecificEnthalpy h "Specific enthalpy";
          output Temperature T "Temperature";

        protected
        package Internal
            "Solve h(data,T) for T with given h (use only indirectly via temperature_phX)"
          extends Modelica.Media.Common.OneNonLinearEquation;
          redeclare record extends f_nonlinear_Data
              "Data to be passed to non-linear function"
            extends Modelica.Media.IdealGases.Common.DataRecord;
          end f_nonlinear_Data;

          redeclare function extends f_nonlinear
          algorithm
              y := h_T(f_nonlinear_data,x);
          end f_nonlinear;

          // Dummy definition has to be added for current Dymola
          redeclare function extends solve
          end solve;
        end Internal;

        algorithm
          T := Internal.solve(h, 200, 6000, 1.0e5, {1}, data);
        end T_h;

        function T_ps "Compute temperature from pressure and specific entropy"
          input AbsolutePressure p "Pressure";
          input SpecificEntropy s "Specific entropy";
          output Temperature T "Temperature";

        protected
        package Internal
            "Solve h(data,T) for T with given h (use only indirectly via temperature_phX)"
          extends Modelica.Media.Common.OneNonLinearEquation;
          redeclare record extends f_nonlinear_Data
              "Data to be passed to non-linear function"
            extends Modelica.Media.IdealGases.Common.DataRecord;
          end f_nonlinear_Data;

          redeclare function extends f_nonlinear
          algorithm
              y := s0_T(f_nonlinear_data,x)- data.R*Modelica.Math.log(p/reference_p);
          end f_nonlinear;

          // Dummy definition has to be added for current Dymola
          redeclare function extends solve
          end solve;
        end Internal;

        algorithm
          T := Internal.solve(s, 200, 6000, p, {1}, data);
        end T_ps;

      end SingleGasNasa;

        package FluidData "Critical data, dipole moments and related data"

          record FluidConstants "Extended fluid constants"
            extends Modelica.Media.Interfaces.PartialMedium.FluidConstants;
            PartialMedium.Temperature criticalTemperature
            "critical temperature";
            PartialMedium.AbsolutePressure criticalPressure "critical pressure";
            PartialMedium.MolarVolume criticalMolarVolume
            "critical molar Volume";
            Real acentricFactor "Pitzer acentric factor";
            PartialMedium.Temperature triplePointTemperature
            "triple point temperature";
            PartialMedium.AbsolutePressure triplePointPressure
            "triple point pressure";
            PartialMedium.Temperature meltingPoint "melting point at 101325 Pa";
            PartialMedium.Temperature normalBoilingPoint
            "normal boiling point (at 101325 Pa)";
            PartialMedium.DipoleMoment dipoleMoment
            "dipole moment of molecule in Debye (1 debye = 3.33564e10-30 C.m)";
            Boolean hasIdealGasHeatCapacity=false
            "true if ideal gas heat capacity is available";
            Boolean hasCriticalData=false "true if critical data are known";
            Boolean hasDipoleMoment=false "true if a dipole moment known";
            Boolean hasFundamentalEquation=false
            "true if a fundamental equation";
            Boolean hasLiquidHeatCapacity=false
            "true if liquid heat capacity is available";
            Boolean hasSolidHeatCapacity=false
            "true if solid heat capacity is available";
            Boolean hasAccurateViscosityData=false
            "true if accurate data for a viscosity function is available";
            Boolean hasAccurateConductivityData=false
            "true if accurate data for thermal conductivity is available";
            Boolean hasVapourPressureCurve=false
            "true if vapour pressure data, e.g. Antoine coefficents are known";
            Boolean hasAcentricFactor=false
            "true if Pitzer accentric factor is known";
            PartialMedium.SpecificEnthalpy HCRIT0=0.0
            "Critical specific enthalpy of the fundamental equation";
            PartialMedium.SpecificEntropy SCRIT0=0.0
            "Critical specific entropy of the fundamental equation";
            PartialMedium.SpecificEnthalpy deltah=0.0
            "Difference between specific enthalpy model (h_m) and f.eq. (h_f) (h_m - h_f)";
            PartialMedium.SpecificEntropy deltas=0.0
            "Difference between specific enthalpy model (s_m) and f.eq. (s_f) (s_m - s_f)";
          end FluidConstants;
          extends Modelica.Icons.Library;
          import Modelica.Media.Interfaces.PartialMedium;
          import Modelica.Media.IdealGases.Common.SingleGasesData;

          constant FluidConstants N2(
                               chemicalFormula =        "N2",
                               iupacName =              "unknown",
                               structureFormula =       "unknown",
                               casRegistryNumber =      "7727-37-9",
                               meltingPoint =            63.15,
                               normalBoilingPoint =      77.35,
                               criticalTemperature =    126.20,
                               criticalPressure =        33.98e5,
                               criticalMolarVolume =     90.10e-6,
                               acentricFactor =           0.037,
                               dipoleMoment =             0.0,
                               molarMass =              SingleGasesData.N2.MM,
                               hasDipoleMoment =       true,
                               hasIdealGasHeatCapacity=true,
                               hasCriticalData =       true,
                               hasAcentricFactor =     true);

          constant FluidConstants H2O(
                               chemicalFormula =        "H2O",
                               iupacName =              "unknown",
                               structureFormula =       "unknown",
                               casRegistryNumber =      "7732-18-5",
                               meltingPoint =           273.15,
                               normalBoilingPoint =     373.15,
                               criticalTemperature =    647.14,
                               criticalPressure =       220.64e5,
                               criticalMolarVolume =     55.95e-6,
                               acentricFactor =           0.344,
                               dipoleMoment =             1.8,
                               molarMass =              SingleGasesData.H2O.MM,
                               hasDipoleMoment =       true,
                               hasIdealGasHeatCapacity=true,
                               hasCriticalData =       true,
                               hasAcentricFactor =     true);
        end FluidData;

        package SingleGasesData
        "Ideal gas data based on the NASA Glenn coefficients"
          extends Modelica.Icons.Library;

          constant IdealGases.Common.DataRecord Air(
            name="Air",
            MM=0.0289651159,
            Hf=-4333.833858403446,
            H0=298609.6803431054,
            Tlimit=1000,
            alow={10099.5016,-196.827561,5.00915511,-0.00576101373,1.06685993e-005,-7.94029797e-009,
                2.18523191e-012},
            blow={-176.796731,-3.921504225},
            ahigh={241521.443,-1257.8746,5.14455867,-0.000213854179,7.06522784e-008,-1.07148349e-011,
                6.57780015e-016},
            bhigh={6462.26319,-8.147411905},
            R=287.0512249529787);

          constant IdealGases.Common.DataRecord H2O(
            name="H2O",
            MM=0.01801528,
            Hf=-13423382.81725291,
            H0=549760.6476280135,
            Tlimit=1000,
            alow={-39479.6083,575.573102,0.931782653,0.00722271286,-7.34255737e-006,
                4.95504349e-009,-1.336933246e-012},
            blow={-33039.7431,17.24205775},
            ahigh={1034972.096,-2412.698562,4.64611078,0.002291998307,-6.836830479999999e-007,
                9.426468930000001e-011,-4.82238053e-015},
            bhigh={-13842.86509,-7.97814851},
            R=461.5233290850878);

          constant IdealGases.Common.DataRecord N2(
            name="N2",
            MM=0.0280134,
            Hf=0,
            H0=309498.4543111511,
            Tlimit=1000,
            alow={22103.71497,-381.846182,6.08273836,-0.00853091441,1.384646189e-005,-9.62579362e-009,
                2.519705809e-012},
            blow={710.846086,-10.76003744},
            ahigh={587712.406,-2239.249073,6.06694922,-0.00061396855,1.491806679e-007,-1.923105485e-011,
                1.061954386e-015},
            bhigh={12832.10415,-15.86640027},
            R=296.8033869505308);
        end SingleGasesData;
      end Common;
    end IdealGases;

    package Incompressible
    "Medium model for T-dependent properties, defined by tables or polynomials"
      import SI = Modelica.SIunits;
      import Cv = Modelica.SIunits.Conversions;
      import Modelica.Constants;
      import Modelica.Math;

      package Common "Common data structures"

        record BaseProps_Tpoly "Fluid state record"
          extends Modelica.Icons.Record;
          SI.Temperature T "temperature";
          SI.Pressure p "pressure";
          //    SI.Density d "density";
        end BaseProps_Tpoly;
      end Common;

      package TableBased "Incompressible medium properties based on tables"
        import Poly = Modelica.Media.Incompressible.TableBased.Polynomials_Temp;
        extends Modelica.Media.Interfaces.PartialMedium(
           ThermoStates = if enthalpyOfT then Choices.IndependentVariables.T else Choices.IndependentVariables.pT,
           final reducedX=true,
           final fixedX = true,
           mediumName="tableMedium",
           redeclare record ThermodynamicState=Common.BaseProps_Tpoly,
           singleState=true);

        constant Boolean enthalpyOfT=true
        "true if enthalpy is approximated as a function of T only, (p-dependence neglected)";

        constant Boolean densityOfT = size(tableDensity,1) > 1
        "true if density is a function of temperature";

        constant Temperature T_min "Minimum temperature valid for medium model";

        constant Temperature T_max "Maximum temperature valid for medium model";

        constant Temperature T0=273.15 "reference Temperature";

        constant SpecificEnthalpy h0=0 "reference enthalpy at T0, reference_p";

        constant SpecificEntropy s0=0 "reference entropy at T0, reference_p";

        constant MolarMass MM_const=0.1 "Molar mass";

        constant Integer npol=2 "degree of polynomial used for fitting";

        constant Integer neta=size(tableViscosity,1)
        "number of data points for viscosity";

        constant Real[:,:] tableDensity "Table for rho(T)";

        constant Real[:,:] tableHeatCapacity "Table for Cp(T)";

        constant Real[:,:] tableViscosity "Table for eta(T)";

        constant Real[:,:] tableConductivity "Table for lambda(T)";

        constant Boolean TinK "true if T[K],Kelvin used for table temperatures";

        constant Boolean hasDensity = not (size(tableDensity,1)==0)
        "true if table tableDensity is present";

        constant Boolean hasHeatCapacity = not (size(tableHeatCapacity,1)==0)
        "true if table tableHeatCapacity is present";

        constant Boolean hasViscosity = not (size(tableViscosity,1)==0)
        "true if table tableViscosity is present";

        final constant Real invTK[neta] = if size(tableViscosity,1) > 0 then 
            invertTemp(tableViscosity[:,1],TinK) else fill(0,0);

        final constant Real poly_rho[:] = if hasDensity then 
                                             Poly.fitting(tableDensity[:,1],tableDensity[:,2],npol) else 
                                               zeros(npol+1);

        final constant Real poly_Cp[:] = if hasHeatCapacity then 
                                             Poly.fitting(tableHeatCapacity[:,1],tableHeatCapacity[:,2],npol) else 
                                               zeros(npol+1);

        final constant Real poly_eta[:] = if hasViscosity then 
                                             Poly.fitting(invTK, Math.log(tableViscosity[:,2]),npol) else 
                                               zeros(npol+1);

        final constant Real poly_lam[:] = if size(tableConductivity,1)>0 then 
                                             Poly.fitting(tableConductivity[:,1],tableConductivity[:,2],npol) else 
                                               zeros(npol+1);

        function invertTemp "function to invert temperatures"
          input Real[:] table "table temperature data";
          input Boolean Tink "flag for Celsius or Kelvin";
          output Real invTable[size(table,1)] "inverted temperatures";
        algorithm
          for i in 1:size(table,1) loop
            invTable[i] := if TinK then 1/table[i] else 1/Cv.from_degC(table[i]);
          end for;
        end invertTemp;

        redeclare model extends BaseProperties(
          final standardOrderComponents=true,
          p_bar=Cv.to_bar(p),
          T_degC(start = T_start-273.15)=Cv.to_degC(T),
          T(start = T_start,
            stateSelect=if preferredMediumStates then StateSelect.prefer else StateSelect.default))
        "Base properties of T dependent medium"
        //  redeclare parameter SpecificHeatCapacity R=Modelica.Constants.R,

          SI.SpecificHeatCapacity cp "specific heat capacity";
          parameter SI.Temperature T_start = 298.15 "initial temperature";
        equation
          assert(hasDensity,"Medium " + mediumName +
                            " can not be used without assigning tableDensity.");
          assert(T >= T_min and T <= T_max, "Temperature T (= " + String(T) +
                 " K) is not in the allowed range (" + String(T_min) +
                 " K <= T <= " + String(T_max) + " K) required from medium model \""
                 + mediumName + "\".");
          R = Modelica.Constants.R;
          cp = Poly.evaluate(poly_Cp,if TinK then T else T_degC);
          h = if enthalpyOfT then h_T(T) else  h_pT(p,T,densityOfT);
          if singleState then
            u = h_T(T) - reference_p/d;
          else
            u = h - p/d;
          end if;
          d = Poly.evaluate(poly_rho,if TinK then T else T_degC);
          state.T = T;
          state.p = p;
          MM = MM_const;
        end BaseProperties;

        redeclare function extends setState_pTX
        "Returns state record, given pressure and temperature"
        algorithm
          state := ThermodynamicState(p=p,T=T);
        end setState_pTX;

        redeclare function extends setState_dTX
        "Returns state record, given pressure and temperature"
        algorithm
          assert(false, "for incompressible media with d(T) only, state can not be set from density and temperature");
        end setState_dTX;

        redeclare function extends setState_phX
        "Returns state record, given pressure and specific enthalpy"
        algorithm
          state :=ThermodynamicState(p=p,T=T_ph(p,h));
        end setState_phX;

        redeclare function extends setState_psX
        "Returns state record, given pressure and specific entropy"
        algorithm
          state :=ThermodynamicState(p=p,T=T_ps(p,s));
        end setState_psX;

            redeclare function extends setSmoothState
        "Return thermodynamic state so that it smoothly approximates: if x > 0 then state_a else state_b"
            algorithm
              state :=ThermodynamicState(p=Media.Common.smoothStep(x, state_a.p, state_b.p, x_small),
                                         T=Media.Common.smoothStep(x, state_a.T, state_b.T, x_small));
            end setSmoothState;

        redeclare function extends specificHeatCapacityCv
        "Specific heat capacity at constant volume (or pressure) of medium"

        algorithm
          assert(hasHeatCapacity,"Specific Heat Capacity, Cv, is not defined for medium "
                                                 + mediumName + ".");
          cv := Poly.evaluate(poly_Cp,if TinK then state.T else state.T - 273.15);
        end specificHeatCapacityCv;

        redeclare function extends specificHeatCapacityCp
        "Specific heat capacity at constant volume (or pressure) of medium"

        algorithm
          assert(hasHeatCapacity,"Specific Heat Capacity, Cv, is not defined for medium "
                                                 + mediumName + ".");
          cp := Poly.evaluate(poly_Cp,if TinK then state.T else state.T - 273.15);
        end specificHeatCapacityCp;

        redeclare function extends dynamicViscosity
        "Return dynamic viscosity as a function of the thermodynamic state record"

        algorithm
          assert(size(tableViscosity,1)>0,"DynamicViscosity, eta, is not defined for medium "
                                                 + mediumName + ".");
          eta := Math.exp(Poly.evaluate(poly_eta, 1/state.T));
        end dynamicViscosity;

        redeclare function extends thermalConductivity
        "Return thermal conductivity as a function of the thermodynamic state record"

        algorithm
          assert(size(tableConductivity,1)>0,"ThermalConductivity, lambda, is not defined for medium "
                                                 + mediumName + ".");
          lambda := Poly.evaluate(poly_lam,if TinK then state.T else Cv.to_degC(state.T));
        end thermalConductivity;

        function s_T "compute specific entropy"
          input Temperature T "temperature";
          output SpecificEntropy s "specific entropy";
        algorithm
          s := s0 + (if TinK then 
            Poly.integralValue(poly_Cp[1:npol],T, T0) else 
            Poly.integralValue(poly_Cp[1:npol],Cv.to_degC(T),Cv.to_degC(T0)))
            + Modelica.Math.log(T/T0)*
            Poly.evaluate(poly_Cp,if TinK then 0 else Modelica.Constants.T_zero);
        end s_T;

        redeclare function extends specificEntropy "Return specific entropy
 as a function of the thermodynamic state record"

      protected
          Integer npol=size(poly_Cp,1)-1;
        algorithm
          assert(hasHeatCapacity,"Specific Entropy, s(T), is not defined for medium "
                                                 + mediumName + ".");
          s := s_T(state.T);
        end specificEntropy;

        function h_T "Compute specific enthalpy from temperature"
          import Modelica.SIunits.Conversions.to_degC;
          extends Modelica.Icons.Function;
          input SI.Temperature T "Temperature";
          output SI.SpecificEnthalpy h "Specific enthalpy at p, T";
        algorithm
          h :=h0 + Poly.integralValue(poly_Cp, if TinK then T else Cv.to_degC(T), if TinK then 
          T0 else Cv.to_degC(T0));
        end h_T;

        function h_T_der "Compute specific enthalpy from temperature"
          import Modelica.SIunits.Conversions.to_degC;
          extends Modelica.Icons.Function;
          input SI.Temperature T "Temperature";
          input Real dT "temperature derivative";
          output Real dh "derivative of Specific enthalpy at T";
        algorithm
          dh :=Poly.evaluate(poly_Cp, if TinK then T else Cv.to_degC(T))*dT;
        end h_T_der;

        function h_pT "Compute specific enthalpy from pressure and temperature"
          import Modelica.SIunits.Conversions.to_degC;
          extends Modelica.Icons.Function;
          input SI.Pressure p "Pressure";
          input SI.Temperature T "Temperature";
          input Boolean densityOfT = false
          "include or neglect density derivative dependence of enthalpy";
          output SI.SpecificEnthalpy h "Specific enthalpy at p, T";
        algorithm
          h :=h0 + Poly.integralValue(poly_Cp, if TinK then T else Cv.to_degC(T), if TinK then 
          T0 else Cv.to_degC(T0)) + (p - reference_p)/Poly.evaluate(poly_rho, if TinK then 
                  T else Cv.to_degC(T))
            *(if densityOfT then (1 + T/Poly.evaluate(poly_rho, if TinK then T else Cv.to_degC(T))
          *Poly.derivativeValue(poly_rho,if TinK then T else Cv.to_degC(T))) else 1.0);
        end h_pT;

        redeclare function extends temperature
        "Return temperature as a function of the thermodynamic state record"
        algorithm
         T := state.T;
        end temperature;

        redeclare function extends pressure
        "Return pressure as a function of the thermodynamic state record"
        algorithm
         p := state.p;
        end pressure;

        redeclare function extends density
        "Return density as a function of the thermodynamic state record"
        algorithm
          d := Poly.evaluate(poly_rho,if TinK then state.T else Cv.to_degC(state.T));
        end density;

        redeclare function extends specificEnthalpy
        "Return specific enthalpy as a function of the thermodynamic state record"
        algorithm
          h := if enthalpyOfT then h_T(state.T) else h_pT(state.p,state.T);
        end specificEnthalpy;

        redeclare function extends specificInternalEnergy
        "Return specific internal energy as a function of the thermodynamic state record"
        algorithm
          u := if enthalpyOfT then h_T(state.T) else h_pT(state.p,state.T)
            - (if singleState then  reference_p/density(state) else state.p/density(state));
        end specificInternalEnergy;

        function T_ph "Compute temperature from pressure and specific enthalpy"
          input AbsolutePressure p "pressure";
          input SpecificEnthalpy h "specific enthalpy";
          output Temperature T "temperature";
      protected
          package Internal
          "Solve h(T) for T with given h (use only indirectly via temperature_phX)"
            extends Modelica.Media.Common.OneNonLinearEquation;

            redeclare record extends f_nonlinear_Data
            "superfluous record, fix later when better structure of inverse functions exists"
                constant Real[5] dummy = {1,2,3,4,5};
            end f_nonlinear_Data;

            redeclare function extends f_nonlinear
            "p is smuggled in via vector"
            algorithm
              y := if singleState then h_T(x) else h_pT(p,x);
            end f_nonlinear;

            // Dummy definition has to be added for current Dymola
            redeclare function extends solve
            end solve;
          end Internal;
        algorithm
         T := Internal.solve(h, T_min, T_max, p, {1}, Internal.f_nonlinear_Data());
        end T_ph;

        function T_ps "Compute temperature from pressure and specific enthalpy"
          input AbsolutePressure p "pressure";
          input SpecificEntropy s "specific entropy";
          output Temperature T "temperature";
      protected
          package Internal
          "Solve h(T) for T with given h (use only indirectly via temperature_phX)"
            extends Modelica.Media.Common.OneNonLinearEquation;

            redeclare record extends f_nonlinear_Data
            "superfluous record, fix later when better structure of inverse functions exists"
                constant Real[5] dummy = {1,2,3,4,5};
            end f_nonlinear_Data;

            redeclare function extends f_nonlinear
            "p is smuggled in via vector"
            algorithm
              y := s_T(x);
            end f_nonlinear;

            // Dummy definition has to be added for current Dymola
            redeclare function extends solve
            end solve;
          end Internal;
        algorithm
         T := Internal.solve(s, T_min, T_max, p, {1}, Internal.f_nonlinear_Data());
        end T_ps;

        package Polynomials_Temp
        "Temporary Functions operating on polynomials (including polynomial fitting); only to be used in Modelica.Media.Incompressible.TableBased"
          extends Modelica.Icons.Library;

          function evaluate "Evaluate polynomial at a given abszissa value"
            extends Modelica.Icons.Function;
            input Real p[:]
            "Polynomial coefficients (p[1] is coefficient of highest power)";
            input Real u "Abszissa value";
            output Real y "Value of polynomial at u";
          algorithm
            y := p[1];
            for j in 2:size(p, 1) loop
              y := p[j] + u*y;
            end for;
          end evaluate;

          function derivativeValue
          "Value of derivative of polynomial at abszissa value u"
            extends Modelica.Icons.Function;
            input Real p[:]
            "Polynomial coefficients (p[1] is coefficient of highest power)";
            input Real u "Abszissa value";
            output Real y "Value of derivative of polynomial at u";
        protected
            Integer n=size(p, 1);
          algorithm
            y := p[1]*(n - 1);
            for j in 2:size(p, 1)-1 loop
              y := p[j]*(n - j) + u*y;
            end for;
          end derivativeValue;

          function secondDerivativeValue
          "Value of 2nd derivative of polynomial at abszissa value u"
            extends Modelica.Icons.Function;
            input Real p[:]
            "Polynomial coefficients (p[1] is coefficient of highest power)";
            input Real u "Abszissa value";
            output Real y "Value of 2nd derivative of polynomial at u";
        protected
            Integer n=size(p, 1);
          algorithm
            y := p[1]*(n - 1)*(n - 2);
            for j in 2:size(p, 1)-2 loop
              y := p[j]*(n - j)*(n - j - 1) + u*y;
            end for;
          end secondDerivativeValue;

          function integralValue
          "Integral of polynomial p(u) from u_low to u_high"
            extends Modelica.Icons.Function;
            input Real p[:] "Polynomial coefficients";
            input Real u_high "High integrand value";
            input Real u_low=0 "Low integrand value, default 0";
            output Real integral=0.0
            "Integral of polynomial p from u_low to u_high";
        protected
            Integer n=size(p, 1) "degree of integrated polynomial";
            Real y_low=0 "value at lower integrand";
          algorithm
            for j in 1:n loop
              integral := u_high*(p[j]/(n - j + 1) + integral);
              y_low := u_low*(p[j]/(n - j + 1) + y_low);
            end for;
            integral := integral - y_low;
          end integralValue;

          function fitting
          "Computes the coefficients of a polynomial that fits a set of data points in a least-squares sense"
            extends Modelica.Icons.Function;
            input Real u[:] "Abscissa data values";
            input Real y[size(u, 1)] "Ordinate data values";
            input Integer n(min=1)
            "Order of desired polynomial that fits the data points (u,y)";
            output Real p[n + 1]
            "Polynomial coefficients of polynomial that fits the date points";
        protected
            Real V[size(u, 1), n + 1] "Vandermonde matrix";
          algorithm
            // Construct Vandermonde matrix
            V[:, n + 1] := ones(size(u, 1));
            for j in n:-1:1 loop
              V[:, j] := {u[i] * V[i, j + 1] for i in 1:size(u,1)};
            end for;

            // Solve least squares problem
            p :=Modelica.Math.Matrices.leastSquares(V, y);
          end fitting;

          function evaluate_der "Evaluate polynomial at a given abszissa value"
            extends Modelica.Icons.Function;
            input Real p[:]
            "Polynomial coefficients (p[1] is coefficient of highest power)";
            input Real u "Abszissa value";
            input Real du "Abszissa value";
            output Real dy "Value of polynomial at u";
        protected
            Integer n=size(p, 1);
          algorithm
            dy := p[1]*(n - 1);
            for j in 2:size(p, 1)-1 loop
              dy := p[j]*(n - j) + u*dy;
            end for;
            dy := dy*du;
          end evaluate_der;

          function integralValue_der
          "Time derivative of integral of polynomial p(u) from u_low to u_high, assuming only u_high as time-dependent (Leibnitz rule)"
            extends Modelica.Icons.Function;
            input Real p[:] "Polynomial coefficients";
            input Real u_high "High integrand value";
            input Real u_low=0 "Low integrand value, default 0";
            input Real du_high "High integrand value";
            input Real du_low=0 "Low integrand value, default 0";
            output Real dintegral=0.0
            "Integral of polynomial p from u_low to u_high";
          algorithm
            dintegral := evaluate(p,u_high)*du_high;
          end integralValue_der;

          function derivativeValue_der
          "time derivative of derivative of polynomial"
            extends Modelica.Icons.Function;
            input Real p[:]
            "Polynomial coefficients (p[1] is coefficient of highest power)";
            input Real u "Abszissa value";
            input Real du "delta of abszissa value";
            output Real dy
            "time-derivative of derivative of polynomial w.r.t. input variable at u";
        protected
            Integer n=size(p, 1);
          algorithm
            dy := secondDerivativeValue(p,u)*du;
          end derivativeValue_der;
        end Polynomials_Temp;
      end TableBased;
    end Incompressible;
  end Media;

  package Math
  "Library of mathematical functions (e.g., sin, cos) and of functions operating on vectors and matrices"
  import SI = Modelica.SIunits;
  extends Modelica.Icons.Library2;

  package Matrices "Library of functions operating on matrices"
    extends Modelica.Icons.Library;

    function leastSquares
    "Solve overdetermined or underdetermined real system of linear equations A*x=b in a least squares sense (A may be rank deficient)"
      extends Modelica.Icons.Function;
      input Real A[:, :] "Matrix A";
      input Real b[size(A, 1)] "Vector b";
      output Real x[size(A, 2)]
      "Vector x such that min|A*x-b|^2 if size(A,1) >= size(A,2) or min|x|^2 and A*x=b, if size(A,1) < size(A,2)";

  protected
      Integer info;
      Integer rank;
      Real xx[max(size(A,1),size(A,2))];
    algorithm
      (xx,info,rank) := LAPACK.dgelsx_vec(A, b, 100*Modelica.Constants.eps);
      x := xx[1:size(A,2)];
      assert(info == 0, "Solving an overdetermined or underdetermined linear system of
equations with function \"Matrices.leastSquares\" failed.");
    end leastSquares;

    package LAPACK
    "Interface to LAPACK library (should usually not directly be used but only indirectly via Modelica.Math.Matrices)"
      extends Modelica.Icons.Library;

      function dgelsx_vec
      "Computes the minimum-norm solution to a real linear least squares problem with rank deficient A"

        extends Modelica.Icons.Function;
        input Real A[:, :];
        input Real b[size(A,1)];
        input Real rcond=0.0 "Reciprocal condition number to estimate rank";
        output Real x[max(nrow,ncol)]= cat(1,b,zeros(max(nrow,ncol)-nrow))
        "solution is in first size(A,2) rows";
        output Integer info;
        output Integer rank "Effective rank of A";
    protected
        Integer nrow=size(A,1);
        Integer ncol=size(A,2);
        Integer nx=max(nrow,ncol);
        Integer lwork=max( min(nrow,ncol)+3*ncol, 2*min(nrow,ncol)+1);
        Real work[lwork];
        Real Awork[nrow,ncol]=A;
        Integer jpvt[ncol]=zeros(ncol);
        external "FORTRAN 77" dgelsx(nrow, ncol, 1, Awork, nrow, x, nx, jpvt,
                                    rcond, rank, work, lwork, info);


      end dgelsx_vec;
    end LAPACK;
  end Matrices;

  function cos "Cosine"
    extends baseIcon1;
    input SI.Angle u;
    output Real y;

  external "C" y=  cos(u);
  end cos;

  function tan "Tangent (u shall not be -pi/2, pi/2, 3*pi/2, ...)"
    extends baseIcon2;
    input SI.Angle u;
    output Real y;

  external "C" y=  tan(u);
  end tan;

  function asin "Inverse sine (-1 <= u <= 1)"
    extends baseIcon2;
    input Real u;
    output SI.Angle y;

  external "C" y=  asin(u);
  end asin;

  function cosh "Hyperbolic cosine"
    extends baseIcon2;
    input Real u;
    output Real y;

  external "C" y=  cosh(u);
  end cosh;

  function tanh "Hyperbolic tangent"
    extends baseIcon2;
    input Real u;
    output Real y;

  external "C" y=  tanh(u);
  end tanh;

  function exp "Exponential, base e"
    extends baseIcon2;
    input Real u;
    output Real y;

  external "C" y=  exp(u);
  end exp;

  function log "Natural (base e) logarithm (u shall be > 0)"
    extends baseIcon1;
    input Real u;
    output Real y;

  external "C" y=  log(u);

  end log;

  partial function baseIcon1
    "Basic icon for mathematical function with y-axis on left side"

  end baseIcon1;

  partial function baseIcon2
    "Basic icon for mathematical function with y-axis in middle"

  end baseIcon2;
  end Math;

  package Utilities
  "Library of utility functions dedicated to scripting (operating on files, streams, strings, system)"
    extends Modelica.Icons.Library;

    package Streams "Read from files and write to files"
      extends Modelica.Icons.Library;

      function error "Print error message and cancel all actions"
        extends Modelica.Icons.Function;
        input String string "String to be printed to error message window";
        external "C" ModelicaError(string);
      end error;
    end Streams;
  end Utilities;

  package Constants
  "Library of mathematical constants and constants of nature (e.g., pi, eps, R, sigma)"
    import SI = Modelica.SIunits;
    import NonSI = Modelica.SIunits.Conversions.NonSIunits;
    extends Modelica.Icons.Library2;

    final constant Real eps=1.e-15 "Biggest number such that 1.0 + eps = 1.0";

    final constant Real R(final unit="J/(mol.K)") = 8.314472
    "Molar gas constant";

    final constant NonSI.Temperature_degC T_zero=-273.15
    "Absolute zero temperature";
  end Constants;

  package Icons "Library of icons"

    partial package Library "Icon for library"

    end Library;

    partial package Library2
    "Icon for library where additional icon elements shall be added"

    end Library2;

    partial model Example "Icon for an example model"

    end Example;

    partial function Function "Icon for a function"

    end Function;

    partial record Record "Icon for a record"

    end Record;
  end Icons;

  package SIunits
  "Library of type and unit definitions based on SI units according to ISO 31-1992"
    extends Modelica.Icons.Library2;

    package Conversions
    "Conversion functions to/from non SI units and type definitions of non SI units"
      extends Modelica.Icons.Library2;

      package NonSIunits "Type definitions of non SI units"
        extends Modelica.Icons.Library2;

        type Temperature_degC = Real (final quantity="ThermodynamicTemperature",
              final unit="degC")
        "Absolute temperature in degree Celsius (for relative temperature use SIunits.TemperatureDifference)";

        type Pressure_bar = Real (final quantity="Pressure", final unit="bar")
        "Absolute pressure in bar";
      end NonSIunits;

      function to_degC "Convert from Kelvin to °Celsius"
        extends ConversionIcon;
        input Temperature Kelvin "Kelvin value";
        output NonSIunits.Temperature_degC Celsius "Celsius value";
      algorithm
        Celsius := Kelvin + Modelica.Constants.T_zero;
      end to_degC;

      function from_degC "Convert from °Celsius to Kelvin"
        extends ConversionIcon;
        input NonSIunits.Temperature_degC Celsius "Celsius value";
        output Temperature Kelvin "Kelvin value";
      algorithm
        Kelvin := Celsius - Modelica.Constants.T_zero;
      end from_degC;

      function to_bar "Convert from Pascal to bar"
        extends ConversionIcon;
        input Pressure Pa "Pascal value";
        output NonSIunits.Pressure_bar bar "bar value";
      algorithm
        bar := Pa/1e5;
      end to_bar;

      partial function ConversionIcon "Base icon for conversion functions"
      end ConversionIcon;
    end Conversions;

    type Angle = Real (
        final quantity="Angle",
        final unit="rad",
        displayUnit="deg");

    type Velocity = Real (final quantity="Velocity", final unit="m/s");

    type Density = Real (
        final quantity="Density",
        final unit="kg/m3",
        displayUnit="g/cm3",
        min=0);

    type Pressure = Real (
        final quantity="Pressure",
        final unit="Pa",
        displayUnit="bar");

    type AbsolutePressure = Pressure (min=0);

    type DynamicViscosity = Real (
        final quantity="DynamicViscosity",
        final unit="Pa.s",
        min=0);

    type SurfaceTension = Real (final quantity="SurfaceTension", final unit="N/m");

    type EnthalpyFlowRate = Real (final quantity="EnthalpyFlowRate", final unit
        =   "W");

    type MassFlowRate = Real (quantity="MassFlowRate", final unit="kg/s");

    type ThermodynamicTemperature = Real (
        final quantity="ThermodynamicTemperature",
        final unit="K",
        min = 0,
        displayUnit="degC")
    "Absolute temperature (use type TemperatureDifference for relative temperatures)";

    type Temp_K = ThermodynamicTemperature;

    type Temperature = ThermodynamicTemperature;

    type Compressibility = Real (final quantity="Compressibility", final unit=
            "1/Pa");

    type IsothermalCompressibility = Compressibility;

    type ThermalConductivity = Real (final quantity="ThermalConductivity", final unit
        =      "W/(m.K)");

    type SpecificHeatCapacity = Real (final quantity="SpecificHeatCapacity",
          final unit="J/(kg.K)");

    type RatioOfSpecificHeatCapacities = Real (final quantity=
            "RatioOfSpecificHeatCapacities", final unit="1");

    type SpecificEntropy = Real (final quantity="SpecificEntropy", final unit=
            "J/(kg.K)");

    type SpecificEnergy = Real (final quantity="SpecificEnergy", final unit=
            "J/kg");

    type SpecificInternalEnergy = SpecificEnergy;

    type SpecificEnthalpy = SpecificEnergy;

    type DerDensityByEnthalpy = Real (final unit="kg.s2/m5");

    type DerDensityByPressure = Real (final unit="s2/m2");

    type DerDensityByTemperature = Real (final unit="kg/(m3.K)");

    type DerEnthalpyByPressure = Real (final unit="J.m.s2/kg2");

    type MolarMass = Real (final quantity="MolarMass", final unit="kg/mol",min=0);

    type MolarVolume = Real (final quantity="MolarVolume", final unit="m3/mol", min=0);

    type MassFraction = Real (final quantity="MassFraction", final unit="1");

    type MoleFraction = Real (final quantity="MoleFraction", final unit="1");

    type PrandtlNumber = Real (final quantity="PrandtlNumber", final unit="1");
  end SIunits;
end Modelica;
model Modelica_Media_Examples_MoistAir
 extends Modelica.Media.Examples.MoistAir;
  annotation(experiment(Tolerance=1e-005),uses(Modelica(version="3.1")));
end Modelica_Media_Examples_MoistAir;
