package Basic
   extends Modelica.Icons.Library;
   annotation (Documentation(info="<HTML>
 <p>
 This package contains basic analog electrical components:
 <ul>
 <li>Ground</li>
 <li>Resistor</li>
 <li>HeatingResistor</li>
 <li>Conductor</li>
 <li>Capacitor</li>
 <li>Inductor</li>
 <li>Transformer</li>
 <li>Gyrator</li>
 <li>EMF (Electroc-Motoric-Force)</li>
 <li>Linear controlled sources (VCV, VCC, CCV, CCC)</li>
 <li>OpAmp</li>
 </ul>
 </p>
 
 <dl>
 <dt>
 <b>Main Authors:</b>
 <dd>
 <a href=\"http://people.eas.iis.fhg.de/Christoph.Clauss/\">Christoph Clau&szlig;</a>
     &lt;<a href=\"mailto:clauss@eas.iis.fhg.de\">clauss@eas.iis.fhg.de</a>&gt;<br>
     <a href=\"http://people.eas.iis.fhg.de/Andre.Schneider/\">Andr&eacute; Schneider</a>
     &lt;<a href=\"mailto:schneider@eas.iis.fhg.de\">schneider@eas.iis.fhg.de</a>&gt;<br>
     Fraunhofer Institute for Integrated Circuits<br>
     Design Automation Department<br>
     Zeunerstra&szlig;e 38<br>
     D-01069 Dresden<br>
 
 <p>
 <dt>
 <b>Version:</b>
 <dd>
 $Id: Basic.mo,v 1.14 2003/10/31 11:11:34 SvenErik Exp $<br>
 <p>
 <dt>
 <b>Copyright:</b>
 <dd>
 Copyright &copy; 1998-2002, Modelica Association and Fraunhofer-Gesellschaft.<br>
 <i>The Modelica package is <b>free</b> software; it can be redistributed and/or modified
 under the terms of the <b>Modelica license</b>, see the license conditions
 and the accompanying <b>disclaimer</b> in the documentation of package
 Modelica in file \"Modelica/package.mo\".</i><br>
 <p>
 </dl>
 </HTML>
 "));
   model Ground "Ground node"
      annotation (Documentation(info="<HTML>
 <P>
 Ground of an electrical circuit. The potential at the
 ground node is zero. Every electrical circuit has to contain
 at least one ground object.
 </P>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-60.,50.},{60.,50.}},color={0,0,255}),Line(points={{-40.,30.},{40.,30.}},color={0,0,255}),Line(points={{-20.,10.},{20.,10.}},color={0,0,255}),Line(points={{0.,90.},{0.,50.}},color={0,0,255}),Text(extent={{-100.,-68.},{100.,-10.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-60.,50.},{60.,50.}},thickness=0.5,color={0,0,255}),Line(points={{-40.,30.},{40.,30.}},thickness=0.5,color={0,0,255}),Line(points={{-20.,10.},{20.,10.}},thickness=0.5,color={0,0,255}),Line(points={{0.,90.},{0.,50.}},thickness=0.5,color={0,0,255}),Text(extent={{-24.,-38.},{22.,-6.}},textString="p.v=0",fillColor={0,0,255})}));
      Interfaces.Pin p annotation (Placement(transformation(x=0.,y=100.,scale=0.1,aspectRatio=1.,rotation=-90,flipVertical=true),iconTransformation(x=0.,y=100.,scale=0.1,aspectRatio=1.,rotation=-90,flipVertical=true)));
      
   equation
      p.v = 0;
   end Ground;
   model Resistor "Ideal linear electrical resistor"
      extends Interfaces.OnePort;
      parameter SI.Resistance R=1 "Resistance";
      annotation (Documentation(info="<HTML>
 <P>
 The linear resistor connects the branch voltage <i>v</i> with the
 branch current <i>i</i> by <i>i*R = v</i>.
 The Resistance <i>R</i> is allowed to be positive, zero, or negative.
 </P>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,30.},{70.,-30.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255}),Text(extent={{-98.,-58.},{102.,-120.}},textString="R=%R",fillColor={0,0,0}),Text(extent={{-100.,40.},{100.,100.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,30.},{70.,-30.}},lineColor={0,0,255}),Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255})}));
      
   equation
      R*i = v;
   end Resistor;
   model HeatingResistor "Temperature dependent electrical resistor"
      extends Interfaces.OnePort;
      parameter SI.Resistance R_ref "Resistance at temperature T_ref";
      parameter SI.Temperature T_ref=300 "Reference temperature";
      parameter Real alpha(unit="1/K")=0 "Temperature coefficient of resistance";
      SI.Resistance R "Resistance = R_ref*(1 + alpha*(heatPort.T - T_ref));";
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-110.,20.},{-85.,20.}},color={160,160,160}),Polygon(points={{-95.,23.},{-85.,20.},{-95.,17.},{-95.,23.}},lineColor={160,160,160},fillColor={160,160,160},fillPattern=FillPattern.Solid),Line(points={{90.,20.},{115.,20.}},color={160,160,160}),Line(points={{-125.,0.},{-115.,0.}},color={160,160,160}),Line(points={{-120.,-5.},{-120.,5.}},color={160,160,160}),Text(extent={{-110.,25.},{-90.,45.}},textString="i",fillColor={160,160,160}),Polygon(points={{105.,23.},{115.,20.},{105.,17.},{105.,23.}},lineColor={160,160,160},fillColor={160,160,160},fillPattern=FillPattern.Solid),Line(points={{115.,0.},{125.,0.}},color={160,160,160}),Text(extent={{90.,45.},{110.,25.}},textString="i",fillColor={160,160,160}),Rectangle(extent={{-70.,30.},{70.,-30.}},lineColor={0,0,255}),Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255}),Line(points={{0.,-30.},{0.,-90.}},color={191,0,0}),Line(points={{-52.,-50.},{48.,50.}},color={0,0,255}),Polygon(points={{40.,52.},{50.,42.},{54.,56.},{40.,52.}},lineColor={0,0,255},fillColor={0,0,255},fillPattern=FillPattern.Solid)}),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-143.,60.},{143.,98.}},textString="%name",fillColor={0,0,255}),Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255}),Rectangle(extent={{-70.,30.},{70.,-30.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{0.,-30.},{0.,-91.}},color={191,0,0}),Line(points={{-52.,-50.},{48.,50.}},color={0,0,255}),Polygon(points={{40.,52.},{50.,42.},{54.,56.},{40.,52.}},lineColor={0,0,255},fillColor={0,0,255},fillPattern=FillPattern.Solid)}),Documentation(info="<HTML>
 <p>This is a model for an electrical resistor where the generated heat
 is dissipated to the environment via connector <b>heatPort</b> and where
 the resistance R is temperature dependent according to the following
 equation:</p>
 
 <pre>    R = R_ref*(1 + alpha*(heatPort.T - T_ref))
 </pre>
 
 <p><b>alpha</b> is the <b>temperature coefficient of resistance</b>, which
 is often abbreviated as <b>TCR</b>. In resistor catalogues, it is usually
 defined as <b>X [ppm/K]</b> (parts per million, similarly to per centage)
 meaning <b>X*1.e-6 [1/K]</b>. Resistors are available for 1 .. 7000 ppm/K,
 i.e., alpha = 1e-6 .. 7e-3 1/K;</p>
 
 <p>When connector <b>heatPort</b> is <b>not</b> connected, the temperature
 dependent behaviour is switched off by setting heatPort.T = T_ref.
 Additionally, the equation <tt>heatPort.Q_dot = 0</tt> is implicitly present
 due to a special rule in Modelica that flow variables of not connected
 connectors are set to zero.</p>
 </HTML>
 "));
      Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_a heatPort annotation (Placement(transformation(x=0.,y=-100.,scale=0.1,aspectRatio=1.,rotation=-90,flipVertical=true),iconTransformation(x=0.,y=-100.,scale=0.1,aspectRatio=1.,rotation=-90,flipVertical=true)));
      
   equation
      v = R*i;
      if cardinality(heatPort) > 0 then
         R = R_ref*(1+alpha*(heatPort.T-T_ref));
         heatPort.Q_dot = -v*i;
      else
         R = R_ref;
         heatPort.T = T_ref;
      end if;
   end HeatingResistor;
   model Conductor "Ideal linear electrical conductor"
      extends Interfaces.OnePort;
      parameter SI.Conductance G=1 "Conductance";
      annotation (Documentation(info="<HTML>
 <P>
 The linear conductor connects the branch voltage <i>v</i> with the
 branch current <i>i</i> by <i>i = v*G</i>.
 The Conductance <i>G</i> is allowed to be positive, zero, or negative.
 </P>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,30.},{70.,-30.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Rectangle(extent={{-70.,30.},{70.,-30.}},lineColor={0,0,255}),Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255}),Text(extent={{-100.,-60.},{100.,-122.}},textString="G=%G",fillColor={0,0,0},pattern=LinePattern.None),Text(extent={{-100.,40.},{100.,98.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,0.},{-70.,0.}},color={0,0,255}),Line(points={{70.,0.},{90.,0.}},color={0,0,255})}));
      
   equation
      i = G*v;
   end Conductor;
   model Capacitor "Ideal linear electrical capacitor"
      extends Interfaces.OnePort;
      parameter SI.Capacitance C=1 "Capacitance";
      annotation (Documentation(info="<HTML>
 <p>
 The linear capacitor connects the branch voltage <i>v</i> with the
 branch current <i>i</i> by <i>i = C * dv/dt</i>.
 The Capacitance <i>C</i> is allowed to be positive, zero, or negative.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-14.,28.},{-14.,-28.}},thickness=0.5,color={0,0,255}),Line(points={{14.,28.},{14.,-28.}},thickness=0.5,color={0,0,255}),Line(points={{-90.,0.},{-14.,0.}},color={0,0,255}),Line(points={{14.,0.},{90.,0.}},color={0,0,255}),Text(extent={{-100.,-60.},{100.,-120.}},textString="C=%C",fillColor={0,0,0}),Text(extent={{-100.,40.},{100.,100.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-20.,40.},{-20.,-40.}},thickness=0.5,color={0,0,255}),Line(points={{20.,40.},{20.,-40.}},thickness=0.5,color={0,0,255}),Line(points={{-90.,0.},{-20.,0.}},color={0,0,255}),Line(points={{20.,0.},{90.,0.}},color={0,0,255})}));
      
   equation
      i = C*der(v);
   end Capacitor;
   model Inductor "Ideal linear electrical inductor"
      extends Interfaces.OnePort;
      parameter SI.Inductance L=1 "Inductance";
      annotation (Documentation(info="<HTML>
 <P>
 The linear inductor connects the branch voltage <i>v</i> with the
 branch current <i>i</i> by  <i>v = L * di/dt</i>.
 The Inductance <i>L</i> is allowed to be positive, zero, or negative.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-60.,-15.},{-30.,15.}},lineColor={0,0,255}),Ellipse(extent={{-30.,-15.},{0.,15.}},lineColor={0,0,255}),Ellipse(extent={{0.,-15.},{30.,15.}},lineColor={0,0,255}),Ellipse(extent={{30.,-15.},{60.,15.}},lineColor={0,0,255}),Rectangle(extent={{-60.,-30.},{60.,0.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{60.,0.},{90.,0.}},color={0,0,255}),Line(points={{-90.,0.},{-60.,0.}},color={0,0,255}),Text(extent={{-100.,-60.},{100.,-120.}},textString="L=%L",fillColor={0,0,0}),Text(extent={{-100.,40.},{100.,100.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-60.,-15.},{-30.,15.}},lineColor={0,0,255}),Ellipse(extent={{-30.,-15.},{0.,15.}},lineColor={0,0,255}),Ellipse(extent={{0.,-15.},{30.,15.}},lineColor={0,0,255}),Ellipse(extent={{30.,-15.},{60.,15.}},lineColor={0,0,255}),Rectangle(extent={{-60.,-30.},{60.,0.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{60.,0.},{90.,0.}},color={0,0,255}),Line(points={{-90.,0.},{-60.,0.}},color={0,0,255})}));
      
   equation
      L*der(i) = v;
   end Inductor;
   model Transformer "Transformer with two ports"
      extends Interfaces.TwoPort;
      parameter SI.Inductance L1=1 "Primary inductance";
      parameter SI.Inductance L2=1 "Secondary inductance";
      parameter SI.Inductance M=1 "Coupling inductance";
      annotation (Documentation(info="<HTML>
 <p>The transformer is a two port. The left port voltage <i>v1</i>, left port current <i>i1</i>,
 right port voltage <i>v2</i> and right port current <i>i2</i> are connected by the following
 relation:</p>
 <pre>         | v1 |         | L1   M  |  | i1' |
          |    |    =    |         |  |     |
          | v2 |         | M    L2 |  | i2' |</pre>
 <p><i>L1</i>, <i>L2</i>, and <i>M</i> are the primary, secondary, and coupling inductances respectively.</p>
 </html>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-100.,128.},{100.,70.}},textString="%name",fillColor={0,0,255}),Ellipse(extent={{-45.,-50.},{-20.,-25.}},lineColor={0,0,255}),Ellipse(extent={{-45.,-25.},{-20.,0.}},lineColor={0,0,255}),Ellipse(extent={{-45.,0.},{-20.,25.}},lineColor={0,0,255}),Ellipse(extent={{-45.,25.},{-20.,50.}},lineColor={0,0,255}),Rectangle(extent={{-72.,-60.},{-33.,60.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{-90.,50.},{-32.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-32.,-50.}},color={0,0,255}),Ellipse(extent={{20.,-50.},{45.,-25.}},lineColor={0,0,255}),Ellipse(extent={{20.,-25.},{45.,0.}},lineColor={0,0,255}),Ellipse(extent={{20.,0.},{45.,25.}},lineColor={0,0,255}),Ellipse(extent={{20.,25.},{45.,50.}},lineColor={0,0,255}),Rectangle(extent={{33.,-60.},{72.,60.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{32.,50.},{90.,50.}},color={0,0,255}),Line(points={{32.,-50.},{90.,-50.}},color={0,0,255}),Text(extent={{-89.,18.},{-60.,-10.}},textString="L1",fillColor={0,0,255}),Text(extent={{64.,18.},{90.,-10.}},textString="L2",fillColor={0,0,255}),Text(extent={{-18.,-70.},{20.,-98.}},textString="M",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-45.,-50.},{-20.,-25.}},lineColor={0,0,255}),Ellipse(extent={{-45.,-25.},{-20.,0.}},lineColor={0,0,255}),Ellipse(extent={{-45.,0.},{-20.,25.}},lineColor={0,0,255}),Ellipse(extent={{-45.,25.},{-20.,50.}},lineColor={0,0,255}),Rectangle(extent={{-72.,-60.},{-33.,60.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{-90.,50.},{-32.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-32.,-50.}},color={0,0,255}),Ellipse(extent={{20.,-50.},{45.,-25.}},lineColor={0,0,255}),Ellipse(extent={{20.,-25.},{45.,0.}},lineColor={0,0,255}),Ellipse(extent={{20.,0.},{45.,25.}},lineColor={0,0,255}),Ellipse(extent={{20.,25.},{45.,50.}},lineColor={0,0,255}),Rectangle(extent={{33.,-60.},{72.,60.}},lineColor={255,255,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{32.,50.},{90.,50.}},color={0,0,255}),Line(points={{32.,-50.},{90.,-50.}},color={0,0,255})}));
      
   equation
      v1 = L1*der(i1)+M*der(i2);
      v2 = M*der(i1)+L2*der(i2);
   end Transformer;
   model Gyrator "Gyrator"
      extends Interfaces.TwoPort;
      parameter SI.Conductance G1=1 "Gyration conductance";
      parameter SI.Conductance G2=1 "Gyration conductance";
      annotation (Documentation(info="<HTML>
 <P>
 A gyrator is a two-port element defined by the following equations:
 </P>
 <PRE>
     i1 =  G2 * v2
     i2 = -G1 * v1
 </PRE>
 <P>
 where the constants <i>G1</i>, <i>G2</i> are called the gyration conductance.
 </P>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-90.,50.},{-60.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-60.,-50.}},color={0,0,255}),Line(points={{60.,50.},{90.,50.}},color={0,0,255}),Line(points={{60.,-50.},{90.,-50.}},color={0,0,255}),Line(points={{-40.,30.},{40.,30.}},color={0,0,255}),Line(points={{-20.,-20.},{20.,-20.}},color={0,0,255}),Polygon(points={{30.,34.},{40.,30.},{30.,26.},{30.,34.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Polygon(points={{-26.,-19.},{-16.,-15.},{-16.,-23.},{-26.,-19.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-5.,10.},{-10.,-10.}},thickness=0.5,color={0,0,255}),Line(points={{9.,10.},{4.,-9.}},thickness=0.5,color={0,0,255}),Line(points={{-14.,9.},{16.,10.}},thickness=0.5,color={0,0,255}),Text(extent={{-100.,130.},{100.,69.}},textString="%name",fillColor={0,0,255}),Text(extent={{-29.,59.},{30.,30.}},textString="G1",fillColor={0,0,255}),Text(extent={{-29.,-29.},{29.,-58.}},textString="G2",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Line(points={{-90.,50.},{-60.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-60.,-50.}},color={0,0,255}),Line(points={{60.,50.},{90.,50.}},color={0,0,255}),Line(points={{60.,-50.},{90.,-50.}},color={0,0,255}),Line(points={{-40.,30.},{40.,30.}},color={0,0,255}),Line(points={{-21.,-22.},{19.,-22.}},color={0,0,255}),Polygon(points={{30.,34.},{40.,30.},{30.,26.},{30.,34.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Polygon(points={{-20.,-30.},{-10.,-26.},{-10.,-34.},{-20.,-30.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-4.,5.},{-6.,-5.}},thickness=1.,color={0,0,255}),Line(points={{3.,5.},{1.,-5.}},thickness=1.,color={0,0,255}),Line(points={{-8.,5.},{7.,5.}},thickness=1.,color={0,0,255}),Text(extent={{-20.,50.},{20.,35.}},textString="G1",fillColor={0,0,255}),Text(extent={{-20.,-35.},{20.,-50.}},textString="G2",fillColor={0,0,255})}));
      
   equation
      i1 = G2*v2;
      i2 = -G1*v1;
   end Gyrator;
   model EMF "Electromotoric force (electric/mechanic transformer)"
      parameter Real k(final unit="N.m/A")=1 "Transformation coefficient";
      SI.Voltage v "Voltage drop between the two pins";
      SI.Current i "Current flowing from positive to negative pin";
      SI.AngularVelocity w "Angular velocity of flange_b";
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,90.},{0.,40.}},color={0,0,255}),Rectangle(extent={{40.,10.},{90.,-10.}},fillColor={192,192,192},lineColor={0,0,0},pattern=LinePattern.None,fillPattern=FillPattern.HorizontalCylinder),Ellipse(extent={{-40.,40.},{40.,-40.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{0.,-90.},{0.,-40.}},color={0,0,255}),Text(extent={{20.,-40.},{100.,-100.}},textString="%name",fillColor={0,0,255}),Text(extent={{30.,100.},{119.,40.}},textString="k=%k",fillColor={160,160,160})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-17.,95.},{-20.,85.},{-23.,95.},{-17.,95.}},lineColor={160,160,160},fillColor={160,160,160},fillPattern=FillPattern.Solid),Line(points={{-20.,110.},{-20.,85.}},color={160,160,160}),Text(extent={{-40.,110.},{-30.,90.}},textString="i",fillColor={160,160,160}),Line(points={{9.,75.},{19.,75.}},color={192,192,192}),Line(points={{0.,90.},{0.,40.}},color={0,0,255}),Ellipse(extent={{-40.,40.},{40.,-40.}},lineColor={0,0,255}),Rectangle(extent={{40.,10.},{90.,-10.}},fillColor={192,192,192},lineColor={0,0,0},pattern=LinePattern.None,fillPattern=FillPattern.HorizontalCylinder),Line(points={{-20.,-110.},{-20.,-85.}},color={160,160,160}),Polygon(points={{-17.,-100.},{-20.,-110.},{-23.,-100.},{-17.,-100.}},lineColor={160,160,160},fillColor={160,160,160},fillPattern=FillPattern.Solid),Text(extent={{-40.,-110.},{-30.,-90.}},textString="i",fillColor={160,160,160}),Line(points={{8.,-79.},{18.,-79.}},color={192,192,192}),Line(points={{0.,-90.},{0.,-40.}},color={0,0,255}),Line(points={{14.,80.},{14.,70.}},color={192,192,192}),Line(points={{140.,0.},{110.,0.}},color={0,0,0}),Text(extent={{114.,-4.},{148.,-14.}},textString="flange_b.phi",fillColor={0,0,255}),Polygon(points={{140.,3.},{150.,0.},{140.,-3.},{140.,3.},{140.,3.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Text(extent={{112.,16.},{148.,6.}},textString="flange_b.tau",fillColor={0,0,255}),Polygon(points={{120.,35.},{100.,40.},{100.,30.},{120.,35.}},lineColor={128,128,128},fillColor={128,128,128},fillPattern=FillPattern.Solid),Line(points={{30.,35.},{101.,35.}},color={128,128,128}),Text(extent={{37.,59.},{105.,46.}},textString="rotation axis",fillColor={128,128,128})}),Documentation(info="<HTML>
 <p>
 EMF transforms electrical energy into rotational mechanical energy.
 It is used as basic building block of an electrical motor. The mechanical
 connector flange_b can be connected to elements of the
 Modelica.Mechanics.Rotational library. flange_b.tau is the cut-torque,
 flange_b.phi is the angle at the rotational connection.
 </p>
 </HTML>
 "));
      Interfaces.PositivePin p annotation (Placement(transformation(x=0.,y=100.,scale=0.1,aspectRatio=1.,rotation=90),iconTransformation(x=0.,y=100.,scale=0.1,aspectRatio=1.,rotation=90)));
      Interfaces.NegativePin n annotation (Placement(transformation(x=0.,y=-100.,scale=0.1,aspectRatio=1.,rotation=90),iconTransformation(x=0.,y=-100.,scale=0.1,aspectRatio=1.,rotation=90)));
      Modelica.Mechanics.Rotational.Interfaces.Flange_b flange_b annotation (Placement(transformation(x=100.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=100.,y=0.,scale=0.1,aspectRatio=1.)));
      
   equation
      v = p.v-n.v;
      0 = p.i+n.i;
      i = p.i;
      w = der(flange_b.phi);
      k*w = v;
      flange_b.tau = -k*i;
   end EMF;
   model VCV "Linear voltage-controlled voltage source"
      extends Interfaces.TwoPort;
      parameter Real gain=1 "Voltage gain";
      annotation (Documentation(info="<HTML>
 <p>
 The linear voltage-controlled voltage source is a TwoPort.
 The right port voltage v2 is controlled by the left port voltage v1
 via
 </p>
 
 <pre>
     v2 = v1 * gain.
 </pre>
 
 <p>
 The left port current is zero. Any voltage gain can be chosen.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,70.},{70.,-70.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Text(extent={{-99.,-79.},{100.,-129.}},textString="%name",fillColor={0,0,255}),Line(points={{-90.,50.},{-30.,50.}},color={0,0,255}),Line(points={{-30.,-50.},{-90.,-50.}},color={0,0,255}),Line(points={{100.,50.},{30.,50.},{30.,-50.},{100.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,50.},{-30.,50.}},color={0,0,255}),Line(points={{-30.,-50.},{-90.,-50.}},color={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,-50.},{90.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Rectangle(extent={{-70.,70.},{70.,-70.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255})}));
      
   equation
      v2 = v1*gain;
      i1 = 0;
   end VCV;
   model VCC "Linear voltage-controlled current source"
      extends Interfaces.TwoPort;
      parameter SI.Conductance transConductance=1 "Transconductance";
      annotation (Documentation(info="<HTML>
 <p>
 The linear voltage-controlled current source is a TwoPort.
 The right port current i2 is controlled by the left port voltage v1
 via
 </p>
 <pre>
     i2 = v1 * transConductance.
 </pre>
 <p>
 The left port current is zero. Any transConductance can be chosen.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,70.},{70.,-70.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Text(extent={{-99.,-80.},{100.,-129.}},textString="%name",fillColor={0,0,255}),Line(points={{-90.,50.},{-30.,50.}},color={0,0,255}),Line(points={{-30.,-50.},{-90.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,20.}},color={0,0,255}),Line(points={{91.,-50.},{30.,-50.},{30.,-20.}},color={0,0,255}),Line(points={{10.,0.},{50.,0.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,50.},{-30.,50.}},color={0,0,255}),Line(points={{-30.,-50.},{-90.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Rectangle(extent={{-70.,70.},{70.,-70.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,20.}},color={0,0,255}),Line(points={{91.,-50.},{30.,-50.},{30.,-20.}},color={0,0,255}),Line(points={{10.,0.},{50.,0.}},color={0,0,255})}));
      
   equation
      i2 = v1*transConductance;
      i1 = 0;
   end VCC;
   model CCV "Linear current-controlled voltage source"
      extends Interfaces.TwoPort;
      parameter SI.Resistance transResistance=1 "Transresistance";
      annotation (Documentation(info="<HTML>
 <p>
 The linear current-controlled voltage source is a TwoPort.
 The right port voltage v2 is controlled by the left port current i1
 via
 </p>
 <pre>
     v2 = i1 * transResistance.
 </pre>
 <p>
 The left port voltage is zero. Any transResistance can be chosen.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,70.},{70.,-70.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Text(extent={{-99.,-80.},{100.,-130.}},textString="%name",fillColor={0,0,255}),Line(points={{100.,50.},{30.,50.},{30.,-50.},{100.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-90.,50.},{-20.,50.},{-20.,-50.},{-90.,-50.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Rectangle(extent={{-70.,70.},{70.,-70.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,-50.},{90.,-50.}},color={0,0,255}),Line(points={{-90.,50.},{-30.,50.},{-30.,-50.},{-90.,-50.}},color={0,0,255})}));
      
   equation
      v2 = i1*transResistance;
      v1 = 0;
   end CCV;
   model CCC "Linear current-controlled current source"
      extends Interfaces.TwoPort;
      parameter Real gain=1 "Current gain";
      annotation (Documentation(info="<HTML>
 <p>
 The linear current-controlled current source is a TwoPort.
 The right port current i2 is controlled by the left port current i1
 via
 </p>
 
 <pre>
     i2 = i1 * gain.
 </pre>
 
 <p>
 The left port voltage is zero. Any current gain can be chosen.
 </p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-70.,70.},{70.,-70.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Text(extent={{-104.,-76.},{97.,-127.}},textString="%name",fillColor={0,0,255}),Line(points={{-100.,50.},{-30.,50.},{-30.,-50.},{-100.,-50.}},color={0,0,255}),Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,20.}},color={0,0,255}),Line(points={{91.,-50.},{30.,-50.},{30.,-20.}},color={0,0,255}),Line(points={{10.,0.},{50.,0.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{10.,20.},{50.,-20.}},lineColor={0,0,255}),Rectangle(extent={{-70.,70.},{70.,-70.}},lineColor={0,0,255}),Line(points={{-20.,60.},{20.,60.}},color={0,0,255}),Polygon(points={{20.,60.},{10.,63.},{10.,57.},{20.,60.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{90.,50.},{30.,50.},{30.,20.}},color={0,0,255}),Line(points={{91.,-50.},{30.,-50.},{30.,-20.}},color={0,0,255}),Line(points={{10.,0.},{50.,0.}},color={0,0,255}),Line(points={{-90.,50.},{-30.,50.},{-30.,-50.},{-91.,-50.}},color={0,0,255})}));
      
   equation
      i2 = i1*gain;
      v1 = 0;
   end CCC;
   model OpAmp "Simple nonideal model of an OpAmp with limitation"
      annotation (Documentation(info="<HTML>
 <P>
 The OpAmp is a simle nonideal model with a smooth out.v = f(vin) characteristic,
 where \"vin = in_p.v - in_n.v\".
 The characteristic is limited by VMax.v and VMin.v. Its slope at vin=0
 is the parameter Slope, which must be positive. (Therefore, the absolute 
 value of Slope is taken into calculation.)
 </P>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-95.,88.},{115.,146.}},textString="%name",fillColor={0,0,255}),Polygon(points={{60.,0.},{-60.,70.},{-60.,-70.},{60.,0.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-45.,-10.},{-20.,-10.},{-14.,-9.},{-11.,-7.},{-9.,7.},{-6.,9.},{0.,10.},{20.,10.}},color={0,0,255}),Line(points={{0.,35.},{0.,80.}},color={0,0,255}),Line(points={{0.,-35.},{0.,-80.}},color={0,0,255}),Line(points={{-90.,50.},{-60.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-60.,-50.}},color={0,0,255}),Line(points={{60.,0.},{90.,0.}},color={0,0,255}),Line(points={{-48.,32.},{-28.,32.}},color={0,0,255}),Line(points={{-39.,-20.},{-39.,-41.}},color={0,0,255}),Line(points={{-50.,-31.},{-28.,-31.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{60.,0.},{-60.,70.},{-60.,-70.},{60.,0.}},fillColor={255,255,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-45.,-10.},{-20.,-10.},{-14.,-9.},{-11.,-7.},{-9.,7.},{-6.,9.},{0.,10.},{20.,10.}},color={0,0,255}),Line(points={{0.,35.},{0.,80.}},color={0,0,255}),Line(points={{0.,-35.},{0.,-80.}},color={0,0,255}),Line(points={{-90.,50.},{-60.,50.}},color={0,0,255}),Line(points={{-90.,-50.},{-60.,-50.}},color={0,0,255}),Line(points={{60.,0.},{90.,0.}},color={0,0,255}),Line(points={{-55.,50.},{-45.,50.}},color={0,0,255}),Line(points={{-50.,-45.},{-50.,-55.}},color={0,0,255}),Line(points={{-55.,-50.},{-45.,-50.}},color={0,0,255}),Text(extent={{-112.,-93.},{-91.,-73.}},fillColor={160,160,160},textString="in_p.i=0"),Polygon(points={{120.,3.},{110.,0.},{120.,-3.},{120.,3.}},lineColor={0,0,0},fillColor={160,160,160},pattern=LinePattern.None,fillPattern=FillPattern.HorizontalCylinder),Line(points={{111.,0.},{136.,0.}},color={160,160,160}),Text(extent={{118.,2.},{135.,17.}},textString="i2",fillColor={0,0,0},pattern=LinePattern.None,fillPattern=FillPattern.HorizontalCylinder),Text(extent={{-111.,60.},{-85.,80.}},fillColor={160,160,160},textString="in_n.i=0"),Line(points={{-100.,-35.},{-100.,23.},{-100.,24.}},color={160,160,160},arrow={Arrow.None,Arrow.Filled}),Text(extent={{-97.,-16.},{-74.,4.}},fillColor={160,160,160},textString="vin")}));
      parameter Real Slope=1 "Slope of the out.v/vin characteristic at vin=0";
      Modelica.Electrical.Analog.Interfaces.PositivePin in_p "Positive pin of the input port" annotation (Placement(transformation(x=-100.,y=-50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-100.,y=-50.,scale=0.1,aspectRatio=1.)));
      Modelica.Electrical.Analog.Interfaces.NegativePin in_n "Negative pin of the input port" annotation (Placement(transformation(x=-100.,y=50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-100.,y=50.,scale=0.1,aspectRatio=1.)));
      Modelica.Electrical.Analog.Interfaces.PositivePin out "Output pin" annotation (Placement(transformation(x=100.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=100.,y=0.,scale=0.1,aspectRatio=1.)));
      Modelica.Electrical.Analog.Interfaces.PositivePin VMax "Positive output voltage limitation" annotation (Placement(transformation(x=0.,y=70.,scale=0.1,aspectRatio=1.),iconTransformation(x=0.,y=70.,scale=0.1,aspectRatio=1.)));
      Modelica.Electrical.Analog.Interfaces.NegativePin VMin "Negative output voltage limitation" annotation (Placement(transformation(x=0.,y=-70.,scale=0.1,aspectRatio=1.),iconTransformation(x=0.,y=-70.,scale=0.1,aspectRatio=1.)));
      SI.Voltage vin "input voltagae";
      
   protected
      Real f "auxiliary variable";
      Real absSlope;
      
   equation
      in_p.i = 0;
      in_n.i = 0;
      VMax.i = 0;
      VMin.i = 0;
      vin = in_p.v-in_n.v;
      f = 2/(VMax.v-VMin.v);
      absSlope = if Slope < 0 then
         -Slope
      else
         Slope;
      out.v = (VMax.v+VMin.v)/2+(absSlope*vin)/(1+absSlope*noEvent(if f*vin < 0 then
         -f*vin
      else
         f*vin));
   end OpAmp;
end Basic;