List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Terminate", "Title"], Cell[TextData[List["\nThe most common usage of ", StyleBox["terminate", Rule[FontFamily, "Courier"]], " is to give a more appropriate stopping criteria for terminating a simulation than a fixed point in time. The ", StyleBox["terminate", Rule[FontFamily, "Courier"]], " function takes a single argument of type ", StyleBox["String", Rule[FontFamily, "Courier"]], ", which is used to indicate the reason for a successful termination. This is exemplified in the ", StyleBox["terminate", Rule[FontFamily, "Courier"]], " statement within the ", StyleBox["when", Rule[FontFamily, "Courier"]], "-statement in ", StyleBox["MoonLanding", Rule[FontFamily, "Courier"]], "."]], "Text"], Cell[CellGroupData[List[Cell["Celestial Body", "Section"], Cell["class CelestialBody\n  constant Real g = 6.672e-11;\n  parameter Real radius;\n  parameter String name;\n  Real mass;\nend CelestialBody;", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Rocket", "Section"], Cell["class Rocket\n  parameter String name;\n  Real mass(start = 1038.358);\n  Real altitude(start = 59404);\n  Real velocity(start = -2003);\n  Real acceleration;\n  Real thrust;\t\t\t\t\t\t\t\t\t\t// Thrust force on the rocket\n  Real gravity;\t\t\t\t\t\t\t\t\t\t// Gravity forcefield\n  parameter Real massLossRate = 0.000277;\nequation\n  (thrust - mass*gravity)/mass = acceleration;\n  der(mass) = -massLossRate * abs(thrust);\n  der(altitude) = velocity;\n  der(velocity) = acceleration;\nend Rocket;  ", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["MoonLanding", "Section"], Cell["When using this terminate condition the simulation stops when the moon lander touches the ground on the moon.", "Text"], Cell[TextData[List["class MoonLanding\n  parameter Real force1 = 36350;\t\t\t\t\t\n  parameter Real force2 = 1308;\nprotected\t\t\t\t\t\t\t\t\t\t\t\n  parameter Real thrustEndTime = 210;\n  parameter Real thrustDecreaseTime = 43.2;\npublic\t\t\t\t\t\t\n  Rocket apollo(name = \"Apollo13\");\n  CelestialBody moon(name = \"moon\", mass = 7.382e22, radius = 1.738e6);\n  \nequation\n  apollo.thrust = if (time < thrustDecreaseTime) then force1\n  \t\t\t\t  else if (time < thrustEndTime) then force2\n  \t\t\t\t  else 0;\n  apollo.gravity = moon.g*moon.mass/(apollo.altitude + moon.radius) ^2;\n  \nalgorithm\n  when apollo.altitude < 0 then\t\t// Termination condition\n  ", StyleBox["  terminate(\"The moon lander touches the ground of the moon\");", Rule[FontColor, RGBColor[0, 0, 1]]], "\n  end when;  \nend MoonLanding;  "]], "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["MoonLanding Simulation", "Section"], Cell["When we simulate the MoonLanding class from 0 to 230 we get a message that the simulation ended before reaching 230. In the dslog.txt file we see that the simulation is terminated at T = 218.", "Text"], Cell["simulate( MoonLanding, stopTime=230 );", "ModelicaInput"], Cell[BoxData[RowBox[List[RowBox[List["Simulate", "::", "smft"]], RowBox[List[":", " "]], "\"Simulate ended before reaching \\!\\(230\\).\""]]], "Message"], Cell[BoxData["\"<SimulationData: \\!\\(MoonLanding\\) : 2004-1-30 14:52:7.2697736 : \\!\\({0.`, 218.4767608642578`}\\) : 481 data points : 3 events : 15 variables>\\n\\!\\({\\*TagBox[\\(apollo . acceleration\\), HoldForm], \\*TagBox[\\(apollo . altitude\\), HoldForm], \\*TagBox[\\(apollo . gravity\\), HoldForm], \\*TagBox[\\(apollo . mass\\), HoldForm], \\*TagBox[\\(apollo . massLossRate\\), HoldForm], \\*TagBox[\\(apollo . thrust\\), HoldForm], \\*TagBox[\\(apollo . velocity\\), HoldForm], \\*TagBox[\\\"force1\\\", HoldForm], \\*TagBox[\\\"force2\\\", HoldForm], \\*TagBox[\\(moon . g\\), HoldForm], \\*TagBox[\\(moon . mass\\), HoldForm], \\*TagBox[\\(moon . radius\\), HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . altitude)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . mass)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . velocity)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm]}\\)\""], "Output"], Cell[BoxData[RowBox[List["!!", " ", RowBox[List["dslog", ".", "txt"]]]]], "Input"], Cell["plot( { apollo.velocity, apollo.altitude }, xrange={0,208} );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0045704 0.0339144 9.58529e-006 [\n[.25233 .02141 -6 -9 ]\n[.25233 .02141 6 0 ]\n[.48085 .02141 -9 -9 ]\n[.48085 .02141 9 0 ]\n[.70937 .02141 -9 -9 ]\n[.70937 .02141 9 0 ]\n[.93789 .02141 -9 -9 ]\n[.93789 .02141 9 0 ]\n[1.025 .03391 0 -6.28125 ]\n[1.025 .03391 10 6.28125 ]\n[.01131 .12977 -30 -4.5 ]\n[.01131 .12977 0 4.5 ]\n[.01131 .22562 -30 -4.5 ]\n[.01131 .22562 0 4.5 ]\n[.01131 .32147 -30 -4.5 ]\n[.01131 .32147 0 4.5 ]\n[.01131 .41733 -30 -4.5 ]\n[.01131 .41733 0 4.5 ]\n[.01131 .51318 -30 -4.5 ]\n[.01131 .51318 0 4.5 ]\n[.01131 .60903 -30 -4.5 ]\n[.01131 .60903 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.25233 .03391 m\n.25233 .04016 L\ns\n[(50)] .25233 .02141 0 1 Mshowa\n.48085 .03391 m\n.48085 .04016 L\ns\n[(100)] .48085 .02141 0 1 Mshowa\n.70937 .03391 m\n.70937 .04016 L\ns\n[(150)] .70937 .02141 0 1 Mshowa\n.93789 .03391 m\n.93789 .04016 L\ns\n[(200)] .93789 .02141 0 1 Mshowa\n.125 Mabswid\n.06951 .03391 m\n.06951 .03766 L\ns\n.11522 .03391 m\n.11522 .03766 L\ns\n.16092 .03391 m\n.16092 .03766 L\ns\n.20663 .03391 m\n.20663 .03766 L\ns\n.29803 .03391 m\n.29803 .03766 L\ns\n.34374 .03391 m\n.34374 .03766 L\ns\n.38944 .03391 m\n.38944 .03766 L\ns\n.43515 .03391 m\n.43515 .03766 L\ns\n.52655 .03391 m\n.52655 .03766 L\ns\n.57226 .03391 m\n.57226 .03766 L\ns\n.61796 .03391 m\n.61796 .03766 L\ns\n.66367 .03391 m\n.66367 .03766 L\ns\n.75507 .03391 m\n.75507 .03766 L\ns\n.80078 .03391 m\n.80078 .03766 L\ns\n.84648 .03391 m\n.84648 .03766 L\ns\n.89219 .03391 m\n.89219 .03766 L\ns\n.98359 .03391 m\n.98359 .03766 L\ns\n.25 Mabswid\n0 .03391 m\n1 .03391 L\ns\ngsave\n1.025 .03391 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .12977 m\n.03006 .12977 L\ns\n[(10000)] .01131 .12977 1 0 Mshowa\n.02381 .22562 m\n.03006 .22562 L\ns\n[(20000)] .01131 .22562 1 0 Mshowa\n.02381 .32147 m\n.03006 .32147 L\ns\n[(30000)] .01131 .32147 1 0 Mshowa\n.02381 .41733 m\n.03006 .41733 L\ns\n[(40000)] .01131 .41733 1 0 Mshowa\n.02381 .51318 m\n.03006 .51318 L\ns\n[(50000)] .01131 .51318 1 0 Mshowa\n.02381 .60903 m\n.03006 .60903 L\ns\n[(60000)] .01131 .60903 1 0 Mshowa\n.125 Mabswid\n.02381 .05309 m\n.02756 .05309 L\ns\n.02381 .07226 m\n.02756 .07226 L\ns\n.02381 .09143 m\n.02756 .09143 L\ns\n.02381 .1106 m\n.02756 .1106 L\ns\n.02381 .14894 m\n.02756 .14894 L\ns\n.02381 .16811 m\n.02756 .16811 L\ns\n.02381 .18728 m\n.02756 .18728 L\ns\n.02381 .20645 m\n.02756 .20645 L\ns\n.02381 .24479 m\n.02756 .24479 L\ns\n.02381 .26396 m\n.02756 .26396 L\ns\n.02381 .28313 m\n.02756 .28313 L\ns\n.02381 .3023 m\n.02756 .3023 L\ns\n.02381 .34064 m\n.02756 .34064 L\ns\n.02381 .35981 m\n.02756 .35981 L\ns\n.02381 .37898 m\n.02756 .37898 L\ns\n.02381 .39816 m\n.02756 .39816 L\ns\n.02381 .4365 m\n.02756 .4365 L\ns\n.02381 .45567 m\n.02756 .45567 L\ns\n.02381 .47484 m\n.02756 .47484 L\ns\n.02381 .49401 m\n.02756 .49401 L\ns\n.02381 .53235 m\n.02756 .53235 L\ns\n.02381 .55152 m\n.02756 .55152 L\ns\n.02381 .57069 m\n.02756 .57069 L\ns\n.02381 .58986 m\n.02756 .58986 L\ns\n.02381 .01474 m\n.02756 .01474 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02591 .01486 L\n.02801 .01501 L\n.03012 .01516 L\n.03222 .01531 L\n.03432 .01546 L\n.03642 .01561 L\n.03853 .01577 L\n.04063 .01592 L\n.04273 .01607 L\n.04483 .01623 L\n.04694 .01638 L\n.04904 .01654 L\n.05114 .01669 L\n.05324 .01685 L\n.05535 .01701 L\n.05745 .01717 L\n.05955 .01733 L\n.06165 .01749 L\n.06375 .01765 L\n.06586 .01781 L\n.06796 .01798 L\n.07006 .01814 L\n.07216 .01831 L\n.07427 .01847 L\n.07637 .01864 L\n.07847 .0188 L\n.08057 .01897 L\n.08268 .01914 L\n.08478 .01931 L\n.08688 .01948 L\n.08898 .01965 L\n.09109 .01983 L\n.09319 .02 L\n.09529 .02018 L\n.09739 .02035 L\n.0995 .02053 L\n.1016 .0207 L\n.1037 .02088 L\n.1058 .02106 L\n.1079 .02124 L\n.11001 .02142 L\n.11211 .02161 L\n.11421 .02179 L\n.11631 .02198 L\n.11842 .02216 L\n.12052 .02235 L\n.12262 .02254 L\n.12472 .02272 L\n.12683 .02291 L\nMistroke\n.12893 .02311 L\n.13103 .0233 L\n.13313 .02349 L\n.13524 .02369 L\n.13734 .02388 L\n.13944 .02408 L\n.14154 .02428 L\n.14365 .02448 L\n.14575 .02468 L\n.14785 .02488 L\n.14995 .02508 L\n.15206 .02529 L\n.15416 .02549 L\n.15626 .0257 L\n.15836 .02591 L\n.16046 .02612 L\n.16257 .02633 L\n.16467 .02654 L\n.16677 .02675 L\n.16887 .02697 L\n.17098 .02719 L\n.17308 .0274 L\n.17518 .02762 L\n.17728 .02784 L\n.17939 .02807 L\n.18149 .02829 L\n.18359 .02852 L\n.18569 .02874 L\n.1878 .02897 L\n.1899 .0292 L\n.192 .02944 L\n.1941 .02967 L\n.19621 .0299 L\n.19831 .03014 L\n.20041 .03038 L\n.20251 .03062 L\n.20461 .03086 L\n.20672 .03111 L\n.20882 .03135 L\n.21092 .0316 L\n.21302 .03185 L\n.21513 .0321 L\n.21723 .03236 L\n.21933 .03261 L\n.22125 .03285 L\n.22125 .03285 L\n.22143 .03285 L\n.22354 .03285 L\n.22564 .03285 L\n.22774 .03286 L\nMistroke\n.22984 .03286 L\n.23195 .03286 L\n.23405 .03286 L\n.23615 .03287 L\n.23825 .03287 L\n.24036 .03287 L\n.24246 .03287 L\n.24456 .03288 L\n.24666 .03288 L\n.24876 .03288 L\n.25087 .03288 L\n.25297 .03289 L\n.25507 .03289 L\n.25717 .03289 L\n.25928 .03289 L\n.26138 .0329 L\n.26348 .0329 L\n.26558 .0329 L\n.26769 .0329 L\n.26979 .03291 L\n.27189 .03291 L\n.27399 .03291 L\n.2761 .03291 L\n.2782 .03292 L\n.2803 .03292 L\n.2824 .03292 L\n.28451 .03292 L\n.28661 .03293 L\n.28871 .03293 L\n.29081 .03293 L\n.29291 .03293 L\n.29502 .03294 L\n.29712 .03294 L\n.29922 .03294 L\n.30132 .03294 L\n.30343 .03295 L\n.30553 .03295 L\n.30763 .03295 L\n.30973 .03295 L\n.31184 .03296 L\n.31394 .03296 L\n.31604 .03296 L\n.31814 .03296 L\n.32025 .03297 L\n.32235 .03297 L\n.32445 .03297 L\n.32655 .03297 L\n.32866 .03298 L\n.33076 .03298 L\n.33286 .03298 L\nMistroke\n.33496 .03298 L\n.33707 .03299 L\n.33917 .03299 L\n.34127 .03299 L\n.34337 .03299 L\n.34547 .033 L\n.34758 .033 L\n.34968 .033 L\n.35178 .033 L\n.35388 .03301 L\n.35599 .03301 L\n.35809 .03301 L\n.36019 .03302 L\n.36229 .03302 L\n.3644 .03302 L\n.3665 .03302 L\n.3686 .03303 L\n.3707 .03303 L\n.37281 .03303 L\n.37491 .03303 L\n.37701 .03304 L\n.37911 .03304 L\n.38122 .03304 L\n.38332 .03304 L\n.38542 .03305 L\n.38752 .03305 L\n.38962 .03305 L\n.39173 .03305 L\n.39383 .03306 L\n.39593 .03306 L\n.39803 .03306 L\n.40014 .03307 L\n.40224 .03307 L\n.40434 .03307 L\n.40644 .03307 L\n.40855 .03308 L\n.41065 .03308 L\n.41275 .03308 L\n.41485 .03308 L\n.41696 .03309 L\n.41906 .03309 L\n.42116 .03309 L\n.42326 .03309 L\n.42537 .0331 L\n.42747 .0331 L\n.42957 .0331 L\n.43167 .03311 L\n.43377 .03311 L\n.43588 .03311 L\n.43798 .03311 L\nMistroke\n.44008 .03312 L\n.44218 .03312 L\n.44429 .03312 L\n.44639 .03312 L\n.44849 .03313 L\n.45059 .03313 L\n.4527 .03313 L\n.4548 .03313 L\n.4569 .03314 L\n.459 .03314 L\n.46111 .03314 L\n.46321 .03315 L\n.46531 .03315 L\n.46741 .03315 L\n.46952 .03315 L\n.47162 .03316 L\n.47372 .03316 L\n.47582 .03316 L\n.47792 .03316 L\n.48003 .03317 L\n.48213 .03317 L\n.48423 .03317 L\n.48633 .03318 L\n.48844 .03318 L\n.49054 .03318 L\n.49264 .03318 L\n.49474 .03319 L\n.49685 .03319 L\n.49895 .03319 L\n.50105 .0332 L\n.50315 .0332 L\n.50526 .0332 L\n.50736 .0332 L\n.50946 .03321 L\n.51156 .03321 L\n.51367 .03321 L\n.51577 .03321 L\n.51787 .03322 L\n.51997 .03322 L\n.52208 .03322 L\n.52418 .03323 L\n.52628 .03323 L\n.52838 .03323 L\n.53048 .03323 L\n.53259 .03324 L\n.53469 .03324 L\n.53679 .03324 L\n.53889 .03325 L\n.541 .03325 L\n.5431 .03325 L\nMistroke\n.5452 .03325 L\n.5473 .03326 L\n.54941 .03326 L\n.55151 .03326 L\n.55361 .03327 L\n.55571 .03327 L\n.55782 .03327 L\n.55992 .03327 L\n.56202 .03328 L\n.56412 .03328 L\n.56623 .03328 L\n.56833 .03329 L\n.57043 .03329 L\n.57253 .03329 L\n.57463 .03329 L\n.57674 .0333 L\n.57884 .0333 L\n.58094 .0333 L\n.58304 .03331 L\n.58515 .03331 L\n.58725 .03331 L\n.58935 .03331 L\n.59145 .03332 L\n.59356 .03332 L\n.59566 .03332 L\n.59776 .03333 L\n.59986 .03333 L\n.60197 .03333 L\n.60407 .03333 L\n.60617 .03334 L\n.60827 .03334 L\n.61038 .03334 L\n.61248 .03335 L\n.61458 .03335 L\n.61668 .03335 L\n.61878 .03335 L\n.62089 .03336 L\n.62299 .03336 L\n.62509 .03336 L\n.62719 .03337 L\n.6293 .03337 L\n.6314 .03337 L\n.6335 .03337 L\n.6356 .03338 L\n.63771 .03338 L\n.63981 .03338 L\n.64191 .03339 L\n.64401 .03339 L\n.64612 .03339 L\n.64822 .0334 L\nMistroke\n.65032 .0334 L\n.65242 .0334 L\n.65453 .0334 L\n.65663 .03341 L\n.65873 .03341 L\n.66083 .03341 L\n.66293 .03342 L\n.66504 .03342 L\n.66714 .03342 L\n.66924 .03343 L\n.67134 .03343 L\n.67345 .03343 L\n.67555 .03343 L\n.67765 .03344 L\n.67975 .03344 L\n.68186 .03344 L\n.68396 .03345 L\n.68606 .03345 L\n.68816 .03345 L\n.69027 .03346 L\n.69237 .03346 L\n.69447 .03346 L\n.69657 .03346 L\n.69868 .03347 L\n.70078 .03347 L\n.70288 .03347 L\n.70498 .03348 L\n.70709 .03348 L\n.70919 .03348 L\n.71129 .03349 L\n.71339 .03349 L\n.71549 .03349 L\n.7176 .03349 L\n.7197 .0335 L\n.7218 .0335 L\n.7239 .0335 L\n.72601 .03351 L\n.72811 .03351 L\n.73021 .03351 L\n.73231 .03352 L\n.73442 .03352 L\n.73652 .03352 L\n.73862 .03353 L\n.74072 .03353 L\n.74283 .03353 L\n.74493 .03353 L\n.74703 .03354 L\n.74913 .03354 L\n.75124 .03354 L\n.75334 .03355 L\nMistroke\n.75544 .03355 L\n.75754 .03355 L\n.75964 .03356 L\n.76175 .03356 L\n.76385 .03356 L\n.76595 .03357 L\n.76805 .03357 L\n.77016 .03357 L\n.77226 .03357 L\n.77436 .03358 L\n.77646 .03358 L\n.77857 .03358 L\n.78067 .03359 L\n.78277 .03359 L\n.78487 .03359 L\n.78698 .0336 L\n.78908 .0336 L\n.79118 .0336 L\n.79328 .03361 L\n.79539 .03361 L\n.79749 .03361 L\n.79959 .03362 L\n.80169 .03362 L\n.80379 .03362 L\n.8059 .03362 L\n.808 .03363 L\n.8101 .03363 L\n.8122 .03363 L\n.81431 .03364 L\n.81641 .03364 L\n.81851 .03364 L\n.82061 .03365 L\n.82272 .03365 L\n.82482 .03365 L\n.82692 .03366 L\n.82902 .03366 L\n.83113 .03366 L\n.83323 .03367 L\n.83533 .03367 L\n.83743 .03367 L\n.83954 .03368 L\n.84164 .03368 L\n.84374 .03368 L\n.84584 .03369 L\n.84794 .03369 L\n.85005 .03369 L\n.85215 .0337 L\n.85425 .0337 L\n.85635 .0337 L\n.85846 .03371 L\nMistroke\n.86056 .03371 L\n.86266 .03371 L\n.86476 .03371 L\n.86687 .03372 L\n.86897 .03372 L\n.87107 .03372 L\n.87317 .03373 L\n.87528 .03373 L\n.87738 .03373 L\n.87948 .03374 L\n.88158 .03374 L\n.88369 .03374 L\n.88579 .03375 L\n.88789 .03375 L\n.88999 .03375 L\n.89209 .03376 L\n.8942 .03376 L\n.8963 .03376 L\n.8984 .03377 L\n.9005 .03377 L\n.90261 .03377 L\n.90471 .03378 L\n.90681 .03378 L\n.90891 .03378 L\n.91102 .03379 L\n.91312 .03379 L\n.91522 .03379 L\n.91732 .0338 L\n.91943 .0338 L\n.92153 .0338 L\n.92363 .03381 L\n.92573 .03381 L\n.92784 .03381 L\n.92994 .03382 L\n.93204 .03382 L\n.93414 .03382 L\n.93625 .03383 L\n.93835 .03383 L\n.94045 .03383 L\n.94255 .03384 L\n.94465 .03384 L\n.94676 .03384 L\n.94886 .03385 L\n.95096 .03385 L\n.95306 .03385 L\n.95517 .03386 L\n.95727 .03386 L\n.95937 .03386 L\n.96147 .03387 L\n.96358 .03387 L\nMistroke\n.96568 .03387 L\n.96778 .03388 L\n.96988 .03388 L\n.97199 .03388 L\n.97409 .03389 L\n.97619 .03389 L\nMfstroke\n0 .5 0 r\n.02381 .60332 m\n.02591 .59452 L\n.02801 .58579 L\n.03012 .57713 L\n.03222 .56854 L\n.03432 .56002 L\n.03642 .55157 L\n.03853 .54318 L\n.04063 .53487 L\n.04273 .52663 L\n.04483 .51846 L\n.04694 .51036 L\n.04904 .50233 L\n.05114 .49438 L\n.05324 .4865 L\n.05535 .47869 L\n.05745 .47095 L\n.05955 .46328 L\n.06165 .45569 L\n.06375 .44818 L\n.06586 .44074 L\n.06796 .43337 L\n.07006 .42608 L\n.07216 .41886 L\n.07427 .41172 L\n.07637 .40465 L\n.07847 .39767 L\n.08057 .39076 L\n.08268 .38392 L\n.08478 .37717 L\n.08688 .37049 L\n.08898 .36389 L\n.09109 .35737 L\n.09319 .35093 L\n.09529 .34457 L\n.09739 .33829 L\n.0995 .3321 L\n.1016 .32598 L\n.1037 .31994 L\n.1058 .31399 L\n.1079 .30812 L\n.11001 .30234 L\n.11211 .29663 L\n.11421 .29102 L\n.11631 .28548 L\n.11842 .28003 L\n.12052 .27467 L\n.12262 .26939 L\n.12472 .2642 L\n.12683 .2591 L\nMistroke\n.12893 .25408 L\n.13103 .24916 L\n.13313 .24432 L\n.13524 .23957 L\n.13734 .23491 L\n.13944 .23034 L\n.14154 .22586 L\n.14365 .22148 L\n.14575 .21718 L\n.14785 .21298 L\n.14995 .20887 L\n.15206 .20486 L\n.15416 .20093 L\n.15626 .19711 L\n.15836 .19338 L\n.16046 .18974 L\n.16257 .18621 L\n.16467 .18277 L\n.16677 .17942 L\n.16887 .17618 L\n.17098 .17304 L\n.17308 .16999 L\n.17518 .16705 L\n.17728 .16421 L\n.17939 .16147 L\n.18149 .15883 L\n.18359 .15629 L\n.18569 .15387 L\n.1878 .15154 L\n.1899 .14932 L\n.192 .14721 L\n.1941 .1452 L\n.19621 .1433 L\n.19831 .14151 L\n.20041 .13983 L\n.20251 .13826 L\n.20461 .1368 L\n.20672 .13546 L\n.20882 .13422 L\n.21092 .1331 L\n.21302 .1321 L\n.21513 .13121 L\n.21723 .13043 L\n.21933 .12977 L\n.22125 .12928 L\n.22125 .12928 L\n.22143 .12923 L\n.22354 .12874 L\n.22564 .12826 L\n.22774 .12777 L\nMistroke\n.22984 .12728 L\n.23195 .1268 L\n.23405 .12631 L\n.23615 .12583 L\n.23825 .12535 L\n.24036 .12487 L\n.24246 .12439 L\n.24456 .12391 L\n.24666 .12343 L\n.24876 .12296 L\n.25087 .12248 L\n.25297 .12201 L\n.25507 .12153 L\n.25717 .12106 L\n.25928 .12059 L\n.26138 .12012 L\n.26348 .11965 L\n.26558 .11919 L\n.26769 .11872 L\n.26979 .11826 L\n.27189 .11779 L\n.27399 .11733 L\n.2761 .11687 L\n.2782 .11641 L\n.2803 .11595 L\n.2824 .11549 L\n.28451 .11504 L\n.28661 .11458 L\n.28871 .11413 L\n.29081 .11368 L\n.29291 .11322 L\n.29502 .11277 L\n.29712 .11232 L\n.29922 .11188 L\n.30132 .11143 L\n.30343 .11098 L\n.30553 .11054 L\n.30763 .11009 L\n.30973 .10965 L\n.31184 .10921 L\n.31394 .10877 L\n.31604 .10833 L\n.31814 .10789 L\n.32025 .10746 L\n.32235 .10702 L\n.32445 .10659 L\n.32655 .10615 L\n.32866 .10572 L\n.33076 .10529 L\n.33286 .10486 L\nMistroke\n.33496 .10443 L\n.33707 .10401 L\n.33917 .10358 L\n.34127 .10316 L\n.34337 .10273 L\n.34547 .10231 L\n.34758 .10189 L\n.34968 .10147 L\n.35178 .10105 L\n.35388 .10063 L\n.35599 .10022 L\n.35809 .0998 L\n.36019 .09939 L\n.36229 .09897 L\n.3644 .09856 L\n.3665 .09815 L\n.3686 .09774 L\n.3707 .09733 L\n.37281 .09693 L\n.37491 .09652 L\n.37701 .09612 L\n.37911 .09571 L\n.38122 .09531 L\n.38332 .09491 L\n.38542 .09451 L\n.38752 .09411 L\n.38962 .09372 L\n.39173 .09332 L\n.39383 .09293 L\n.39593 .09253 L\n.39803 .09214 L\n.40014 .09175 L\n.40224 .09136 L\n.40434 .09097 L\n.40644 .09058 L\n.40855 .0902 L\n.41065 .08981 L\n.41275 .08943 L\n.41485 .08904 L\n.41696 .08866 L\n.41906 .08828 L\n.42116 .0879 L\n.42326 .08753 L\n.42537 .08715 L\n.42747 .08678 L\n.42957 .0864 L\n.43167 .08603 L\n.43377 .08566 L\n.43588 .08529 L\n.43798 .08492 L\nMistroke\n.44008 .08455 L\n.44218 .08418 L\n.44429 .08382 L\n.44639 .08346 L\n.44849 .08309 L\n.45059 .08273 L\n.4527 .08237 L\n.4548 .08201 L\n.4569 .08165 L\n.459 .0813 L\n.46111 .08094 L\n.46321 .08059 L\n.46531 .08023 L\n.46741 .07988 L\n.46952 .07953 L\n.47162 .07918 L\n.47372 .07884 L\n.47582 .07849 L\n.47792 .07814 L\n.48003 .0778 L\n.48213 .07746 L\n.48423 .07712 L\n.48633 .07678 L\n.48844 .07644 L\n.49054 .0761 L\n.49264 .07576 L\n.49474 .07543 L\n.49685 .07509 L\n.49895 .07476 L\n.50105 .07443 L\n.50315 .0741 L\n.50526 .07377 L\n.50736 .07344 L\n.50946 .07312 L\n.51156 .07279 L\n.51367 .07247 L\n.51577 .07215 L\n.51787 .07182 L\n.51997 .0715 L\n.52208 .07119 L\n.52418 .07087 L\n.52628 .07055 L\n.52838 .07024 L\n.53048 .06992 L\n.53259 .06961 L\n.53469 .0693 L\n.53679 .06899 L\n.53889 .06868 L\n.541 .06838 L\n.5431 .06807 L\nMistroke\n.5452 .06777 L\n.5473 .06746 L\n.54941 .06716 L\n.55151 .06686 L\n.55361 .06656 L\n.55571 .06626 L\n.55782 .06597 L\n.55992 .06567 L\n.56202 .06538 L\n.56412 .06508 L\n.56623 .06479 L\n.56833 .0645 L\n.57043 .06421 L\n.57253 .06393 L\n.57463 .06364 L\n.57674 .06336 L\n.57884 .06307 L\n.58094 .06279 L\n.58304 .06251 L\n.58515 .06223 L\n.58725 .06195 L\n.58935 .06167 L\n.59145 .0614 L\n.59356 .06112 L\n.59566 .06085 L\n.59776 .06058 L\n.59986 .06031 L\n.60197 .06004 L\n.60407 .05977 L\n.60617 .05951 L\n.60827 .05924 L\n.61038 .05898 L\n.61248 .05872 L\n.61458 .05845 L\n.61668 .05819 L\n.61878 .05794 L\n.62089 .05768 L\n.62299 .05742 L\n.62509 .05717 L\n.62719 .05692 L\n.6293 .05667 L\n.6314 .05641 L\n.6335 .05617 L\n.6356 .05592 L\n.63771 .05567 L\n.63981 .05543 L\n.64191 .05518 L\n.64401 .05494 L\n.64612 .0547 L\n.64822 .05446 L\nMistroke\n.65032 .05422 L\n.65242 .05399 L\n.65453 .05375 L\n.65663 .05352 L\n.65873 .05328 L\n.66083 .05305 L\n.66293 .05282 L\n.66504 .0526 L\n.66714 .05237 L\n.66924 .05214 L\n.67134 .05192 L\n.67345 .05169 L\n.67555 .05147 L\n.67765 .05125 L\n.67975 .05103 L\n.68186 .05082 L\n.68396 .0506 L\n.68606 .05039 L\n.68816 .05017 L\n.69027 .04996 L\n.69237 .04975 L\n.69447 .04954 L\n.69657 .04933 L\n.69868 .04913 L\n.70078 .04892 L\n.70288 .04872 L\n.70498 .04852 L\n.70709 .04831 L\n.70919 .04811 L\n.71129 .04792 L\n.71339 .04772 L\n.71549 .04752 L\n.7176 .04733 L\n.7197 .04714 L\n.7218 .04695 L\n.7239 .04676 L\n.72601 .04657 L\n.72811 .04638 L\n.73021 .0462 L\n.73231 .04601 L\n.73442 .04583 L\n.73652 .04565 L\n.73862 .04547 L\n.74072 .04529 L\n.74283 .04511 L\n.74493 .04494 L\n.74703 .04476 L\n.74913 .04459 L\n.75124 .04442 L\n.75334 .04425 L\nMistroke\n.75544 .04408 L\n.75754 .04391 L\n.75964 .04375 L\n.76175 .04358 L\n.76385 .04342 L\n.76595 .04326 L\n.76805 .0431 L\n.77016 .04294 L\n.77226 .04279 L\n.77436 .04263 L\n.77646 .04248 L\n.77857 .04232 L\n.78067 .04217 L\n.78277 .04202 L\n.78487 .04187 L\n.78698 .04173 L\n.78908 .04158 L\n.79118 .04144 L\n.79328 .04129 L\n.79539 .04115 L\n.79749 .04101 L\n.79959 .04088 L\n.80169 .04074 L\n.80379 .0406 L\n.8059 .04047 L\n.808 .04034 L\n.8101 .04021 L\n.8122 .04008 L\n.81431 .03995 L\n.81641 .03982 L\n.81851 .0397 L\n.82061 .03957 L\n.82272 .03945 L\n.82482 .03933 L\n.82692 .03921 L\n.82902 .03909 L\n.83113 .03898 L\n.83323 .03886 L\n.83533 .03875 L\n.83743 .03864 L\n.83954 .03853 L\n.84164 .03842 L\n.84374 .03831 L\n.84584 .0382 L\n.84794 .0381 L\n.85005 .038 L\n.85215 .0379 L\n.85425 .0378 L\n.85635 .0377 L\n.85846 .0376 L\nMistroke\n.86056 .0375 L\n.86266 .03741 L\n.86476 .03732 L\n.86687 .03723 L\n.86897 .03714 L\n.87107 .03705 L\n.87317 .03696 L\n.87528 .03688 L\n.87738 .03679 L\n.87948 .03671 L\n.88158 .03663 L\n.88369 .03655 L\n.88579 .03648 L\n.88789 .0364 L\n.88999 .03633 L\n.89209 .03625 L\n.8942 .03618 L\n.8963 .03611 L\n.8984 .03604 L\n.9005 .03598 L\n.90261 .03591 L\n.90471 .03585 L\n.90681 .03578 L\n.90891 .03572 L\n.91102 .03566 L\n.91312 .03561 L\n.91522 .03555 L\n.91732 .0355 L\n.91943 .03544 L\n.92153 .03539 L\n.92363 .03534 L\n.92573 .03529 L\n.92784 .03525 L\n.92994 .0352 L\n.93204 .03516 L\n.93414 .03511 L\n.93625 .03507 L\n.93835 .03503 L\n.94045 .035 L\n.94255 .03496 L\n.94465 .03492 L\n.94676 .03489 L\n.94886 .03486 L\n.95096 .03483 L\n.95306 .0348 L\n.95517 .03477 L\n.95727 .03475 L\n.95937 .03473 L\n.96147 .0347 L\n.96358 .03468 L\nMistroke\n.96568 .03466 L\n.96778 .03465 L\n.96988 .03463 L\n.97199 .03462 L\n.97409 .0346 L\n.97619 .03459 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0119048 0.119048 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0214286 0.0721271 0.140476 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 .5 0 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .08095 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(altitude) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n159.000 12.813 moveto\n(L) show\n165.000 12.813 moveto\n(@) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n177.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\n% Start of sub-graphic\np\n0.0110024 0.159524 0.0721271 0.278571 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .21905 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(velocity) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n159.000 12.813 moveto\n(L) show\n165.000 12.813 moveto\n(@) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n177.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00F03IfMT2000000@0fMWI0P00000X0=WIf@D000000P3IfMT2000000@0fMWI0P00000V0=WI\nf@D000000P3IfMT2000000@0fMWI0P00000U0=WIf@@000000`3IfMT2000000@0fMWI0P00000P0=WI\nf@00E`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000002T0fMWI00D00000\n0=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0\n9@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`00\n0000fMWI0=WIf@0R0=WIf@030000003IfMT0fMWI00<0fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@00000O0=WIf@00FP3IfMT010000000fMWI0=WIf@0000020=WIf@030000003I\nfMT0fMWI02L0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00<000000=WIf@3IfMT09@3IfMT00`000000fMWI0=WIf@040=WIf@040000003IfMT0fMWI0000\n0080fMWI00<000000=WIf@3IfMT08`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000007`3IfMT005P0fMWI0P0000030=WIf@040000003IfMT0\nfMWI000002T0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00<000000=WIf@3IfMT09@3IfMT00`000000fMWI0=WIf@020=WIf@8000000`3IfMT010000000\nfMWI0=WIf@00000V0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@030000003IfMT0fMWI01d0fMWI000W0=WIf@030000003IfMT0fMWI02h0fMWI00<00000\n0=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000003IfMT0fMWI0=WIf@000000\n0P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02D0fMWI00<000000=WIf@3I\nfMT00P3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000002@0fMWI00@000000=WI\nf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000001l0fMWI\n000W0=WIf@030000003IfMT0fMWI02h0fMWI0`0000030=WIf@800000:@3IfMT2000000@0fMWI0P00\n00040=WIf@8000009`3IfMT2000000@0fMWI0`0000030=WIf@8000009P3IfMT2000000@0fMWI0P00\n00040=WIf@800000803IfMT002L0fMWI00<000000=WIf@3IfMT0mP3IfMT002L0fMWI2@000?o`0=WI\nf@009`3IfMT00`000000fMWI0=WIf@060=WIf@d0003oh`3IfMT002L0fMWI00<000000=WIf@3IfMT0\n4P3IfMT;0000omT0fMWI000W0=WIf@030000003IfMT0fMWI01d0fMWI2@000?o10=WIf@8000003@3I\nfMT00280fMWI;P00002`0000o`<007l01P0000070=WIf@040000003IfMT0fMWI000000`0fMWI000W\n0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WIf@08\n0=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@07\n0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WIf@08\n0=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@07\n0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WIf@08\n0=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@04\n0=WIfAl007l0103IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00d0fMWI000W0=WI\nf@030000003IfMT0fMWI0380fMWI00<000000=WIf@3IfMT0<@3IfMT00`000000fMWI0=WIf@0b0=WI\nf@030000003IfMT0fMWI00X0fMWI3`00O`0H0=WIf@030000003IfMT0fMWI01<0fMWI1@00000<0=WI\nf@009`3IfMT00`000000fMWI0=WIf@2L0=WIf@`007l0?P3IfMT00`000000fMWI0=WIf@0=0=WIf@00\n9`3IfMT00`000000fMWI0=WIf@2B0=WIf@X007l0FP3IfMT002L0fMWI0P00002:0=WIf@X007l0H`3I\nfMT002L0fMWI00<000000=WIf@3IfMT0P@3IfMT9001o06`0fMWI000W0=WIf@030000003IfMT0fMWI\n07X0fMWI2000O`1d0=WIf@009`3IfMT00`000000fMWI0=WIf@1c0=WIf@L007l0O03IfMT002L0fMWI\n0P00001]0=WIf@L007l0P`3IfMT002L0fMWI00<000000=WIf@3IfMT0IP3IfMT7001o08T0fMWI000W\n0=WIf@030000003IfMT0fMWI0600fMWI1P00O`2@0=WIf@009`3IfMT00`000000fMWI0=WIf@1J0=WI\nf@H007l0UP3IfMT002L0fMWI00<000000=WIf@3IfMT0E@3IfMT6001o09/0fMWI000W0=WIf@800000\nD@3IfMT5001o0:40fMWI000W0=WIf@030000003IfMT0fMWI04X0fMWI1P00O`2V0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@150=WIf@H007l0Z`3IfMT002L0fMWI00<000000=WIf@3IfMT0@@3IfMT5001o\n0;00fMWI000W0=WIf@800000?@3IfMT5001o0;D0fMWI000W0=WIf@030000003IfMT0fMWI03L0fMWI\n1@00O`2j0=WIf@00203IfMT500000080fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P00\n00040=WIf@030000003IfMT0fMWI03<0fMWI1000O`2o0=WIf@002P3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000`0=WIf@D007l0``3IfMT0\n00X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0=WI\nf@<00000:P3IfMT5001o0<L0fMWI000:0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT01@000000fMWI0=WIf@3IfMT0000002X0fMWI0`00O`3;0=WIf@002P3IfMT01@000000fMWI\n0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000X0=WIf@8007l0\ncP3IfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103I\nfMT00`000000fMWI0=WIf@0U0=WIf@8007l0c`3IfMT002L0fMWI00<000000=WIf@3IfMT0903IfMT2\n001o0=00fMWI000W0=WIf@800000903IfMT2001o0=40fMWI000W0=WIf@030000003IfMT0fMWI0280\nfMWI0P00O`3B0=WIf@009`3IfMT00`000000fMWI0=WIf@0R0=WIf@03001o003IfMT0fMWI0=40fMWI\n000W0=WIf@030000003IfMT0fMWI0240fMWI00<007l00=WIf@3IfMT0dP3IfMT002L0fMWI0P00000Q\n0=WIf@03001o003IfMT0fMWI0=<0fMWI000W0=WIf@030000003IfMT0fMWI01l0fMWI0P00O`3E0=WI\nf@009`3IfMT00`000000fMWI0=WIf@0O0=WIf@03001o003IfMT0fMWI0=@0fMWI000W0=WIf@030000\n003IfMT0fMWI01h0fMWI00<007l00=WIf@3IfMT0e@3IfMT002L0fMWI00<000000=WIf@3IfMT07@3I\nfMT2001o0=L0fMWI000W0=WIf@8000007P3IfMT00`00O`00fMWI0=WIf@3F0=WIf@009`3IfMT00`00\n0000fMWI0=WIf@0L0=WIf@03001o003IfMT0fMWI0=L0fMWI000W0=WIf@030000003IfMT0fMWI01`0\nfMWI00<007l00=WIf@3IfMT0e`3IfMT002L0fMWI00<000000=WIf@3IfMT06`3IfMT00`00O`00fMWI\n0=WIf@3H0=WIf@009`3IfMT2000001`0fMWI00<007l00=WIf@3IfMT0f03IfMT002L0fMWI00<00000\n0=WIf@3IfMT06P3IfMT00`00O`00fMWI0=WIf@3I0=WIf@00203IfMT4000000<0fMWI0P0000040=WI\nf@800000103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0fMWI01T0fMWI00<007l00=WI\nf@3IfMT0fP3IfMT000P0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WI\nf@0000030=WIf@030000003IfMT0fMWI01T0fMWI00<007l00=WIf@3IfMT0fP3IfMT000T0fMWI00<0\n00000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@<00000603IfMT0\n0`00O`00fMWI0=WIf@3K0=WIf@002P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080\nfMWI00D000000=WIf@3IfMT0fMWI0000000J0=WIf@03001o003IfMT0fMWI0=/0fMWI00080=WIf@04\n0000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000\nfMWI0=WIf@0G0=WIf@03001o003IfMT0fMWI0=`0fMWI00090=WIf@800000103IfMT2000000@0fMWI\n0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3IfMT05`3IfMT00`00O`00fMWI\n0=WIf@3L0=WIf@009`3IfMT00`000000fMWI0=WIf@0G0=WIf@03001o003IfMT0fMWI0=`0fMWI000W\n0=WIf@8000005`3IfMT00`00O`00fMWI0=WIf@3M0=WIf@009`3IfMT00`000000fMWI0=WIf@0F0=WI\nf@03001o003IfMT0fMWI0=d0fMWI000W0=WIf@030000003IfMT0fMWI01H0fMWI00<007l00=WIf@3I\nfMT0g@3IfMT002L0fMWI00<000000=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@3N0=WIf@009`3I\nfMT2000001H0fMWI00<007l00=WIf@3IfMT0gP3IfMT002L0fMWI00<000000=WIf@3IfMT0503IfMT0\n0`00O`00fMWI0=WIf@3O0=WIf@009`3IfMT00`000000fMWI0=WIf@0D0=WIf@03001o003IfMT0fMWI\n0=l0fMWI000W0=WIf@030000003IfMT0fMWI01@0fMWI00<007l00=WIf@3IfMT0g`3IfMT002L0fMWI\n00<000000=WIf@3IfMT04`3IfMT00`00O`00fMWI0=WIf@3P0=WIf@009`3IfMT2000001@0fMWI00<0\n07l00=WIf@3IfMT0h03IfMT002L0fMWI00<000000=WIf@3IfMT04P3IfMT00`00O`00fMWI0=WIf@3Q\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0B0=WIf@03001o003IfMT0fMWI0>40fMWI000W0=WIf@03\n0000003IfMT0fMWI0180fMWI00<007l00=WIf@3IfMT0h@3IfMT002L0fMWI0P00000B0=WIf@03001o\n003IfMT0fMWI0>80fMWI000W0=WIf@030000003IfMT0fMWI0140fMWI00<007l00=WIf@3IfMT0hP3I\nfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT0\n0`000000fMWI0=WIf@0@0=WIf@03001o003IfMT0fMWI0><0fMWI00080=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@0@0=WI\nf@03001o003IfMT0fMWI0><0fMWI000:0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT010000000fMWI0=WIf@3IfMT300000100fMWI00<007l00=WIf@3IfMT0h`3IfMT000/0fMWI\n00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT000000140fMWI\n00<007l00=WIf@3IfMT0i03IfMT000P0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00l0fMWI00<007l00=WIf@3IfMT0i03I\nfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT0\n0`000000fMWI0=WIf@0?0=WIf@03001o003IfMT0fMWI0>@0fMWI000W0=WIf@030000003IfMT0fMWI\n00h0fMWI00<007l00=WIf@3IfMT0i@3IfMT002L0fMWI0P00000?0=WIf@03001o003IfMT0fMWI0>D0\nfMWI000W0=WIf@030000003IfMT0fMWI00h0fMWI00<007l00=WIf@3IfMT0i@3IfMT002L0fMWI00<0\n00000=WIf@3IfMT03@3IfMT00`00O`00fMWI0=WIf@3V0=WIf@009`3IfMT00`000000fMWI0=WIf@0=\n0=WIf@03001o003IfMT0fMWI0>H0fMWI000W0=WIf@8000003P3IfMT00`00O`00fMWI0=WIf@3V0=WI\nf@009`3IfMT00`000000fMWI0=WIf@0<0=WIf@03001o003IfMT0fMWI0>L0fMWI000W0=WIf@030000\n003IfMT0fMWI00`0fMWI00<007l00=WIf@3IfMT0i`3IfMT002L0fMWI00<000000=WIf@3IfMT0303I\nfMT00`00O`00fMWI0=WIf@3W0=WIf@009`3IfMT00`000000fMWI0=WIf@0;0=WIf@03001o003IfMT0\nfMWI0>P0fMWI000W0=WIf@800000303IfMT00`00O`00fMWI0=WIf@3X0=WIf@009`3IfMT00`000000\nfMWI0=WIf@0;0=WIf@03001o003IfMT0fMWI0>P0fMWI000W0=WIf@030000003IfMT0fMWI00X0fMWI\n00<007l00=WIf@3IfMT0j@3IfMT002L0fMWI00<000000=WIf@3IfMT02P3IfMT00`00O`00fMWI0=WI\nf@3Y0=WIf@009`3IfMT2000000/0fMWI00<007l00=WIf@3IfMT0j@3IfMT002L0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@3Z0=WIf@002P3IfMT300000080fMWI0P0000040=WIf@80\n0000103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0fMWI00T0fMWI00<007l00=WIf@3I\nfMT0jP3IfMT000/0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WI\nf@3IfMT0000000/0fMWI00<007l00=WIf@3IfMT0jP3IfMT000P0fMWI1@0000001@3IfMT000000=WI\nf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI000000<0fMWI0`0000080=WIf@03001o003IfMT0fMWI0>/0fMWI00080=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`00\n0000fMWI0=WIf@080=WIf@03001o003IfMT0fMWI0>/0fMWI00090=WIf@030000003IfMT000000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@080=WIf@03001o\n003IfMT0fMWI0>/0fMWI000:0=WIf@8000000`3IfMT2000000@0fMWI0P0000040=WIf@800000103I\nfMT2000000@0fMWI00<000000=WIf@3IfMT0203IfMT00`00O`00fMWI0=WIf@3[0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@080=WIf@03001o003IfMT0fMWI0>/0fMWI000W0=WIf@800000203IfMT00`00\nO`00fMWI0=WIf@3/0=WIf@009`3IfMT00`000000fMWI0=WIf@070=WIf@03001o003IfMT0fMWI0>`0\nfMWI000W0=WIf@030000003IfMT0fMWI00L0fMWI00<007l00=WIf@3IfMT0k03IfMT002L0fMWI00<0\n00000=WIf@3IfMT01`3IfMT00`00O`00fMWI0=WIf@3/0=WIf@009`3IfMT2000000L0fMWI00<007l0\n0=WIf@3IfMT0k@3IfMT002L0fMWI00<000000=WIf@3IfMT01P3IfMT00`00O`00fMWI0=WIf@3]0=WI\nf@009`3IfMT00`000000fMWI0=WIf@060=WIf@03001o003IfMT0fMWI0>d0fMWI000W0=WIf@030000\n003IfMT0fMWI00D0fMWI00<007l00=WIf@3IfMT0kP3IfMT002L0fMWI00<000000=WIf@3IfMT01@3I\nfMT00`00O`00fMWI0=WIf@3^0=WIf@009`3IfMT2000000H0fMWI00<007l00=WIf@3IfMT0kP3IfMT0\n02L0fMWI00<000000=WIf@3IfMT01@3IfMT00`00O`00fMWI0=WIf@3^0=WIf@009`3IfMT00`000000\nfMWI0=WIf@040=WIf@03001o003IfMT0fMWI0>l0fMWI000W0=WIf@030000003IfMT0fMWI00@0fMWI\n00<007l00=WIf@3IfMT0k`3IfMT002L0fMWI0P0000050=WIf@03001o003IfMT0fMWI0>l0fMWI000W\n0=WIf@030000003IfMT0fMWI00<0fMWI00<007l00=WIf@3IfMT0l03IfMT000T0fMWI0P0000040=WI\nf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@030=WI\nf@03001o003IfMT0fMWI0?00fMWI00080=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI\n00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@030=WIf@03001o003IfMT0fMWI0?00\nfMWI000;0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI\n0`0000030=WIf@03001o003IfMT0fMWI0?00fMWI00090=WIf@8000000`3IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI0080fMWI00<007l00=WIf@3IfMT0l@3IfMT0\n00T0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@03\n0000003IfMT0fMWI0080fMWI00<007l00=WIf@3IfMT0l@3IfMT000T0fMWI0`0000030=WIf@800000\n103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@020=WIf@03001o\n003IfMT0fMWI0?40fMWI000W0=WIf@030000003IfMT0fMWI0080fMWI00<007l00=WIf@3IfMT0l@3I\nfMT002L0fMWI0P0000020=WIf@03001o003IfMT0fMWI0?80fMWI000W0=WIf@050000003IfMT0fMWI\n0=WIf@00O`00m03IfMT002L0fMWI00D000000=WIf@3IfMT0fMWI001o003d0=WIf@009`3IfMT01000\n0000fMWI0=WIf@00O`3e0=WIf@009`3IfMT2000000030=WIf@00O`00fMWI0?@0fMWI000W0=WIf@04\n0000003IfMT0fMWI001o0?D0fMWI000W0=WIf@040000003IfMT0fMWI001o0?D0fMWI000W0=WIf@03\n0000003IfMT007l00?H0fMWI000W0=WIf@030000003IfMT007l00?H0fMWI000W0=WIf@80000000<0\n07l00=WIf@3IfMT0m03IfMT002L0fMWI00<000000=WIf@00O`00mP3IfMT002L0fMWI00<00000001o\n003IfMT0mP3IfMT002L0fMWI00<00000001o003IfMT0mP3IfMT002L0fMWI00<00000001o003IfMT0\nmP3IfMT002L0fMWI00<00000001o003IfMT0mP3IfMT000P0fMWI0`0000040=WIf@800000103IfMT2\n000000@0fMWI0P0000040=WIf@800000103IfMT00`00O`00fMWI0=WIf@3f0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<007l00=WI\nf@3IfMT0mP3IfMT000P0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI\n0=WIf@0000030=WIf@<00000mP3IfMT000P0fMWI0`0000030=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI000000<0fMWI00<000000=WIf@3IfMT0mP3IfMT000T0fMWI00<000000=WIf@3IfMT00P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI0?H0fMWI00090=WIf@<0\n00000`3IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3I\nfMT0mP3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0\nfMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0\nfMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT002D0fMWI00<000000=WIf@3IfMT01@3IfMT3\n00000580fMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002:0=WIf@00903IfMT00`00\n0000fMWI0=WIf@070=WIf@030000003IfMT0fMWI0580fMWI00<000000=WIf@3IfMT0103IfMT00`00\n0000fMWI0=WIf@080=WIf@030000003IfMT0fMWI08P0fMWI000S0=WIf@030000003IfMT0fMWI0080\nfMWI1@000000103IfMT000000000000000040=WIf@8000000`3IfMT3000000<0fMWI0`0000040=WI\nf@800000103IfMT2000000<0fMWI1@000000103IfMT000000000000000040=WIf@800000103IfMT3\n000000<0fMWI0P0000040=WIf@@000000P3IfMT400000080fMWI0`0000040=WIf@030000003IfMT0\nfMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0R03IfMT0\n02<0fMWI00<000000=WIf@3IfMT00P3IfMT400000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3I\nfMT010000000fMWI0=WIf@0000080=WIf@@000000`3IfMT00`000000fMWI0=WIf@020=WIf@040000\n003IfMT0fMWI000000@0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000070=WIf@030000\n003IfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@050000003IfMT0fMWI0=WI\nf@000000RP3IfMT001H0fMWI2000O`050=WIf@030000003IfMT0fMWI00D0fMWI00@000000=WIf@3I\nfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@050000003IfMT0fMWI0=WIf@0000001@3I\nfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000000/0fMWI00D000000=WIf@3IfMT0\nfMWI000000040=WIf@030000003IfMT0fMWI00D0fMWI00D000000=WIf@3IfMT0fMWI000000050=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT4000000@0fMWI00<0\n00000=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0\nR03IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT3000000030=WIf@000000000000800000103I\nfMT2000000@0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@030=WIf@8000002P3I\nfMT3000000<0fMWI00@000000=WIf@3IfMT0fMWI1@0000030=WIf@8000000P3IfMT5000000040=WI\nf@00000000000=WIf@8000000`3IfMT3000000<0fMWI0P0000050=WIf@030000003IfMT0fMWI00<0\nfMWI00<000000=WIf@3IfMT01@0000030=WIf@030000003IfMT0fMWI08P0fMWI000T0=WIf@030000\n003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0E0=WIf@030000\n003IfMT0fMWI0080fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@0<0=WIf@030000\n003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0103IfMT01@000000fMWI0=WIf@3IfMT0000000H0\nfMWI00<000000=WIf@3IfMT0R03IfMT002D0fMWI00<000000=WIf@3IfMT04P3IfMT2000000@0fMWI\n0P00000F0=WIf@800000303IfMT00`000000fMWI0=WIf@0?0=WIf@800000203IfMT00`000000fMWI\n0=WIf@050=WIf@800000203IfMT2000008X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0\nfMWI00D0fMWI0`00001<0=WIf@800000103IfMT00`000000fMWI0=WIf@050=WIf@800000203IfMT2\n000008X0fMWI000T0=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0C03IfMT00`00\n0000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@280=WIf@008`3IfMT00`000000fMWI0=WIf@020=WIf@D0000000@0fMWI00000000\n00000000103IfMT2000000<0fMWI0`0000030=WIf@<00000103IfMT2000000@0fMWI0P0000040=WI\nf@800000103IfMT300000080fMWI0`0000040=WIf@800000103IfMT3000000<0fMWI0`0000030=WI\nf@8000001@3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WI\nf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0R03IfMT001H0fMWI20000?l50=WIf@03\n0000003IfMT0fMWI0080fMWI100000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000`3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0080fMWI00@00000\n0=WIf@3IfMT000002@3IfMT2000000<0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WI\nf@020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT01@3IfMT01@000000fMWI\n0=WIf@3IfMT000000080fMWI00L000000=WIf@3IfMT0fMWI0000003IfMT0000000@0fMWI00<00000\n0=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00D000000=WIf@3IfMT0fMWI\n0000002:0=WIf@008`3IfMT00`000000fMWI0=WIf@050=WIf@040000003IfMT0fMWI00000080fMWI\n00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00<000000=WI\nf@3IfMT00P3IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI00000080fMWI1000\n00030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI\n0=WIf@050=WIf@050000003IfMT0fMWI0=WIf@0000001@3IfMT010000000fMWI0=WIf@0000040=WI\nf@030000003IfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0\nfMWI08P0fMWI000S0=WIf@030000003IfMT0fMWI00<0fMWI0`0000000`3IfMT00000000000020000\n00@0fMWI0P0000040=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00`3IfMT20000\n00P0fMWI0P0000020=WIf@8000000P3IfMT2000000@0fMWI00<000000=WIf@3IfMT00`3IfMT20000\n00@0fMWI0`0000030=WIf@8000000P3IfMT5000000050=WIf@00000000000000003IfMT00P000003\n0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT01@0000030=WIf@030000003IfMT0\nfMWI08P0fMWI000T0=WIf@030000003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT00`3IfMT00`00\n0000fMWI0=WIf@0K0=WIf@030000003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT02P3IfMT00`00\n0000fMWI0=WIf@040=WIf@050000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@28\n0=WIf@009@3IfMT00`000000fMWI0=WIf@0B0=WIf@800000103IfMT2000001`0fMWI0P00000B0=WI\nf@030000003IfMT0fMWI00d0fMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002:0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-37.9664`, -12293.5`, 0.952173`, 454.009`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -10.3955`, 0.119891`, 0.0615577`]], Rule[List[List[22.125`, 30.4375`], List[43.0625`, 26.8125`]], List[-6.32337`, -42.9027`, 0.240604`, 0.240604`]], Rule[List[List[22.125`, 30.4375`], List[24.1875`, 7.9375`]], List[-6.32337`, -47.4441`, 0.240604`, 0.240604`]]]]], Cell[TextData[List["Vertical ", StyleBox["velocity", Rule[FontFamily, "Courier"]], " relative to the lunar surface and ", StyleBox["Altitude", Rule[FontFamily, "Courier"]], " of the Apollo rocket over the lunar surface."]], "NumberedFigure"]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[798, 907]], Rule[WindowMargins, List[List[3, Automatic], List[Automatic, 0]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]