List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Design Optimization", "Title"], Cell[TextData[List["\nAn important use for modeling and simulation is to improve a system design, usually before it is physically realized and manufactured. In this process it is customary to perform a number of simulations for different values of the design parameters, until a design has been obtained that best fulfills a given set of design criteria. To automate this process it is very useful if the simulation environment has programmable scripting facilities that provide the glue between optimization algorithms, executing simulations and various computational functions.\n\nThe following example demonstrates the solution of a small but non-trivial design optimization problem containing dynamic system simulations. First we define a Modelica model of a ", StyleBox["linear actuator", Rule[FontSlant, "Italic"]], " with a spring damped stopping, as well as a first order system that will be used to generate a kind of \"reference\" response. The goal is to find a damping for the translational spring-damper that minimizes the deviation between the step response of the linear actuator system and the step response from our \"reference\" first order system. "]], "Text"], Cell[CellGroupData[List[Cell["LinearActuator Model", "Section"], Cell[TextData[List["In this example we use the MathModelica environment with Mathematica as the scripting language providing the built-in optimization function ", StyleBox["FindMinimum", Rule[FontFamily, "Courier"]], "."]], "Text"], Cell[GraphicsData["Metafile", "CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000BZ80@0006`000000000000006D100090@00\n0000000000358000=aX00215CDH00040R2X102X7000;0000000000000000000000@00005003`0000\n@0400000000000000000082Y0`00hP@0AP0001R7000<Q`00AdA9@`4008000`00Q>SjU000003dQP00\n0@090003NT<000`0<P000000500002H63`0N0?ooool401@0001GKg9T3P1=JF=bKg=_IW@PEfmbI0D0\n000;0P000005000030;n03d170000?/240070000002l0P0000010P8RDgUcM6E]000000X000040000\n003ooooo0@000000<0040000;@4000D000020@40000500002@80000270000?/2l?l00000002@0@00\n0004@00BE6U]IG<PCVEg859_KF5^0;iRmGO7H_Eg0@000000<0040000;@4100<0000N00L0000F1?h0\n?@400000400002H63`0F0?ooool00?oooooooooo?`400?l0000300007P050000;P4000001@0000X2\n000000D000090P00000500000@;oool01`000?`20@00000000040000;@4200P0003j0PD00@000000\n00040000;@4300@000030@P01`000184oP3n03d1?P4500002`8000001@0000`2n`:l0a@0000V1Pl0\n7P3ooooo100D0000EfmbI0h0CFUSLVmcKfId85M_LV@5000030;k0[`370000?/250090000002l0P00\n00040P8RDgUcM6E]07M000001`L:UKiRmGO7H_Eg0@000000<0040000;@4400D000020@40000L0000\nn`:Lo`0000000901000000A0019DJFeULb1>IGLPDVm]HFh0_V;eMlMRmGL10000000`00@0000]0@D0\n1@0000T200000PD0000;0P00000=0000<PX000000@040000002l0o/286T]00L0003l0P000;l00000\n100002d11P080000nP8000400000_`02100002d11`0700006`@c09419`290@L0003l0P00oooo0000\n100002d120040000l04600P0003j0P000000000000040000;@4600@0003`0@L0100002d10P080000\nnP8000400000_`02100002d11`0700006`@c0?l19`3b0@@0000]0@P0100002d11P040000l0470100\n000V1Pl05P3ooooo002<0@00;@000:D1000a000020000?X2000100000;l000@0000]0@L0100002d1\n0P0800009@<208d1;P2S0Bl0100002d11P040000l04700@0000]0@P0200002H63`060?ooool10100\n000V1Pl05P3ooooo003O0@00;@000?`1000a000020000?X2000100000;l000@0000]0@L0100002d1\n0P0800009@<20>01;P3j0Bl0100002d11P040000l04700@0000]0@P0200002H63`060?ooool10100\n000V1Pl05P3ooooo002S0@007`000>41000n0000100002d10`070000o0800000000000@0000]0@L0\n7P0003P50P0600H0X`4O0>017`3P0Cd0X`4m0:<17`2S0Al0Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0\n100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002W0@00\n7`000>41000n0000100002d10`070000o08000L71`0000@0000]0@L07P0003P50P0600H0Y`4O0>01\n7`3P0Cd0Y`4m0:L17`2W0Al0Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0100002d11P040000;@4800@0\n003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002W0@007`000>41000n0000100002d1\n0`070000o08000l?3`0000@0000]0@L07P0003P50P0600H0Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0\nY`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0100002d11P040000;@4800@0003`0@L0200002H63`060?oo\nool10100000V1Pl05P3ooooo002W0@007`000>41000n0000100002d10`070000o08001LG5`0000@0\n000]0@L07P0003P50P0600H0Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0Y`4O0=/17`3K0Cd0Y`4m0:L1\n7`2W0Al0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo\n002W0@007`000=`1000n0000100002d10`070000o080020P800000@0000]0@L07P0003P50P0600H0\nY`4O0=/17`3K0Cd0Y`4m0:L17`2W0Al0Z`4O0=/17`3K0CT0Z`4i0:/17`2[0Al0100002d11P040000\n;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002[0@007`000=`1000j0000\n100002d10`070000o08002TY:@0000@0000]0@L07P0003P50P0600H0Z`4O0=/17`3K0CT0Z`4i0:/1\n7`2[0Al0Z`4O0=/17`3K0CT0Z`4i0:/17`2[0Al0100002d11P040000;@4800@0003`0@L0200002H6\n3`060?ooool10100000V1Pl05P3ooooo002[0@007`000=`1000j0000100002d10`070000o080038b\n<P0000@0000]0@L07P0003P50P0600H0Z`4O0=/17`3K0CT0Z`4i0:/17`2[0Al0Z`4O0=/17`3K0CT0\nZ`4i0:/17`2[0Al0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl0\n5P3ooooo002[0@007`000=`1000j0000100002d10`070000o08003`l?00000@0000]0@L07P0003P5\n0P0600H0Z`4O0=/17`3K0CT0Z`4i0:/17`2[0Al0Z`4S0=P18`3H0CT0Z`4i0:/18`2[0B<0100002d1\n1P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002[0@008`000=T1\n000j0000100002d10`070000o08004I6AP0000@0000]0@L07P0003P50P0600H0Z`4S0=P18`3H0CT0\nZ`4i0:/18`2[0B<0[P4S0=P18`3H0CT0[P4i0:h18`2^0B<0100002d11P040000;@4800@0003`0@L0\n200002H63`060?ooool10100000V1Pl05P3ooooo002^0@008`000=T1000j0000100002d10`070000\no080055AD@0000@0000]0@L07P0003P50P0600H0[P4S0=P18`3H0CT0[P4i0:h18`2^0B<0[P4S0=P1\n8`3H0CT0[P4i0:h18`2^0B<0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100\n000V1Pl05P3ooooo002^0@008`000=T1000j0000100002d10`070000o08005aLG00000@0000]0@L0\n7P0003P50P0600H0[P4S0=P18`3H0CT0[P4i0:h18`2^0B<0[P4S0=P18`3H0CT0[P4i0:h18`2^0B<0\n100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002^0@00\n8`000=T1000j0000100002d10`070000o08006QXJ00000@0000]0@L07P0003P50P0600H0[P4S0=P1\n8`3H0CT0[P4i0:h18`2^0B<0[P4S0=@18`3D0CT0[P4i0:h18`2^0B<0100002d11P040000;@4800@0\n003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002^0@008`000=D1000j0000100002d1\n0`070000o08007AdM00000@0000]0@L07P0003P50P0600H0[P4S0=@18`3D0CT0[P4i0:h18`2^0B<0\n[P4S0=@18`3D0CH0[P4f0:h18`2^0B<0100002d11P040000;@4800@0003`0@L0200002H63`060?oo\nool10100000V1Pl05P3ooooo002^0@008`000=D1000g0000100002d10`070000o0800861P@0000@0\n000]0@L07P0003P50P0600H0[P4S0=@18`3D0CH0[P4f0:h18`2^0B<0/P4S0=@18`3D0CH0/P4f0;81\n8`2b0B<0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo\n002b0@008`000=D1000g0000100002d10`070000o08008n?S`0000@0000]0@L07P0003P50P0600H0\n/P4S0=@18`3D0CH0/P4f0;818`2b0B<0/P4S0=@18`3D0CH0/P4f0;818`2b0B<0100002d11P040000\n;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002b0@008`000=D1000g0000\n100002d10`070000o08009bLW00000@0000]0@L07P0003P50P0600H0/P4S0=@18`3D0CH0/P4f0;81\n8`2b0B<0/P4W0=019`3@0CH0/P4f0;819`2b0BL0100002d11P040000;@4800@0003`0@L0200002H6\n3`060?ooool10100000V1Pl05P3ooooo002b0@009`000=41000g0000100002d10`070000o0800:VY\nZ@0000@0000]0@L07P0003P50P0600H0/P4W0=019`3@0CH0/P4f0;819`2b0BL0/P4W0=019`3@0CH0\n/P4f0;819`2b0BL0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl0\n5P3ooooo002b0@009`000=41000g0000100002d10`070000o0800;Bd]00000@0000]0@L07P0003P5\n0P0600H0/P4W0=019`3@0CH0/P4f0;819`2b0BL0]P4W0=019`3@0CH0]P4f0;H19`2f0BL0100002d1\n1P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002f0@009`000=41\n000g0000100002d10`070000o0800;jn_P0000@0000]0@L07P0003P50P0600H0]P4W0=019`3@0CH0\n]P4f0;H19`2f0BL0]P4W0=019`3@0CH0]P4f0;H19`2f0BL0100002d11P040000;@4800@0003`0@L0\n200002H63`060?ooool10100000V1Pl05P3ooooo002f0@009`000=41000g0000100002d10`070000\no0800<O7a`0000@0000]0@L07P0003P50P0600H0]P4W0=019`3@0CH0]P4f0;H19`2f0BL0]P4W0=01\n9`3@0CH0]P4f0;H19`2f0BL0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100\n000V1Pl05P3ooooo002f0@009`000=41000g0000100002d10`070000o0800<o?c`0000@0000]0@L0\n7P0003P50P0600H0]P4W0=019`3@0CH0]P4f0;H19`2f0BL0]P4W0<`19`3<0C80]P4b0;H19`2f0BL0\n100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002f0@00\n9`000<d1000c0000100002d10`070000o0800=KFeP0000@0000]0@L07P0003P50P0600H0]P4W0<`1\n9`3<0C80]P4b0;H19`2f0BL0^P4W0<`19`3<0C80^P4b0;X19`2j0BL0100002d11P040000;@4800@0\n003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002j0@009`000<d1000c0000100002d1\n0`070000o0800=gMg@0000@0000]0@L07P0003P50P0600H0^P4W0<`19`3<0C80^P4b0;X19`2j0BL0\n^P4Z0<`1:P3<0C80^P4b0;X1:P2j0BX0100002d11P040000;@4800@0003`0@L0200002H63`060?oo\nool10100000V1Pl05P3ooooo002j0@00:P000<d1000c0000100002d10`070000o0800>?Sh`0000@0\n000]0@L07P0003P50P0600H0^P4Z0<`1:P3<0C80^P4b0;X1:P2j0BX0^P4Z0<`1:P3<0C80^P4b0;X1\n:P2j0BX0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo\n002j0@00:P000<d1000c0000100002d10`070000o0800>SXj00000@0000]0@L07P0003P50P0600H0\n^P4Z0<`1:P3<0C80^P4b0;X1:P2j0BX0^P4Z0<T1:P390C80^P4b0;X1:P2j0BX0100002d11P040000\n;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002j0@00:P000<X1000c0000\n100002d10`070000o0800>k^kP0000@0000]0@L07P0003P50P0600H0^P4Z0<T1:P390C80^P4b0;X1\n:P2j0BX0_@4Z0<T1:P390C80_@4b0;d1:P2m0BX0100002d11P040000;@4800@0003`0@L0200002H6\n3`060?ooool10100000V1Pl05P3ooooo002m0@00:P000<X1000c0000100002d10`070000o0800?;b\nlP0000@0000]0@L07P0003P50P0600H0_@4Z0<T1:P390C80_@4b0;d1:P2m0BX0_@4Z0<T1:P390C80\n_@4b0;d1:P2m0BX0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl0\n5P3ooooo002m0@00:P000<X1000c0000100002d10`070000o0800?Ogm`0000@0000]0@L07P0003P5\n0P0600H0_@4Z0<T1:P390C80_@4b0;d1:P2m0BX0_@4Z0<T1:P390Bh0_@4^0;d1:P2m0BX0100002d1\n1P040000;@4800@0003`0@L0200002H63`060?ooool10100000V1Pl05P3ooooo002m0@00:P000<X1\n000_0000100002d10`070000o0800?_kn`0000@0000]0@L07P0003P50P0600H0_@4Z0<T1:P390Bh0\n_@4^0;d1:P2m0BX0_@4Z0<D1:P350Bh0_@4^0;d1:P2m0BX0100002d11P040000;@4800@0003`0@L0\n200002H63`060?ooool10100000V1Pl05P3ooooo002m0@00:P000<H1000_0000100002d10`070000\no0800?ooo`0000@0000]0@L07P0003P50P0600H0_@4Z0<D1:P350Bh0_@4^0;d1:P2m0BX0`@4Z0<D1\n:P350Bh0`@4^0<41:P310BX0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100\n000V1Pl05P3ooooo00310@00:P000<H1000_0000100002d10`070000o0800?ooo`0000@0000]0@L0\n6P0003P50P0600@0`@4Z0<D1:P350Bh0`@4^0<41:P310BX0`@4^0<D1;P310Bh0`@4^00@0000]0@H0\n100002d120040000l04700P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00`@4002h000360@00\n;`0000@0000]0@<01`000?`2003oool000040000;@4701H0000h1@8010040<41;P350Bh0`@4^0<41\n;P310Bh0a@4^0<41;P310Bh0100002d11P040000;@4800@0003`0@L0200002H63`060?ooool10100\n000V1Pl05P3ooooo00310@00;P000<H1000_0000100002d10`070000o0800?ooo`0000@0000]0@L0\n500003P50P0400<0`@4^0<D1;P310Bh0`@4^0<41;P310Bh0`@4^00@0000]0@H0100002d120040000\nl04700P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00`@4002h000320@00;`0000@0000]0@<0\n1`000?`2003oool000040000;@470180000h1@800`030<41;P310Bh0`@4^0<41;P310Bh0`@4^00@0\n000]0@H0100002d120040000l04700P0000V1Pl01P3ooooo0@040000;@4200P0003j0P000@000000\n00840000;@4700L0000K13h0h@4O0:<1100002d120040000;@4600@0003`0@L0400002H63`0F0?oo\nool00</1001F0000g@4006D000080000nP800040001[KF/0100002d11`070000o08006]]J`0000@0\n000]0@T0300002@3103K0El0c05G0<`1H`3K0El0100002d11P040000l04700@0000]0@P010000?01\n2@0800009PH?00H0ooooo`40400002H63`0F0?ooool00:81001N0000cP40064000080000nP800040\n00000000100002d11`040000;@4200P0000U0`80X`5O0<`1G`040000;@4600@0003`0@L0100002d1\n200800009PH?00H0ooooo`40100002d10P040000;@4300L0000K19`0AP9S06/1100002d120040000\n;@4600<0000N00L0000F19/0A@9S06/11@0000810@0001`0003k0]oo00000000T040000014008T5b\nJF5/0?Eg@0000>862SRnH_Egaf;eM`4000000300100002d11`0M0000<PYS06/1300406/1H`140Y/0\nLfaYI6U^IdeQLg<a4@0700L04P0801<04P0L01404@0B0180100002d11@0=0000<PXX02/20@0406/1\nH`140Y/0800^00@0000W0Ooo1`000?`20000_`0000040000;@4900P0003j0P000@00002o00840000\n;@4:00L0000K13<0OP8W0782100002d120040000l04900@0000]0@H010000?012P040000;@4200P0\n003j0P000@00002o00840000;@4900L0000K13<0j`8W0=l2100002d120040000;@4600@0003`0@T0\n400002H63`0F0?ooool008@200080000f@8002T000080000nP80004000000000100002d12@040000\n;@4201@0000U0`P0Q@8H08`2602H0RL0Y`890;H29`350PT0d08H0=L260040000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool008@2000G0000Q`8005H000080000\nnP80004000000000100002d12@040000;@4200P0000U0`80Q@8H08D2E0040000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool008@2001C0000UP8005H000080000\nnP80004000000000100002d12@040000;@4200P0000U0`80Q@9D09@2E0040000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`401`000?`20036`lH000040000;@4900P0003j0P000@000000\n00840000;@4:00L0000K1600`P9909@2100002d120040000l04900@0000]0@H010000?012P0@0000\n9PH?01H0ooooo`00T`8004L0003F0P00B`0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P2D0TP0e09900@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00T`8005h0003F0P00H@0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P2D0Ul0e09O00@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00`08005<0003I0P00EP0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P310U@0e`9D00@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00eP8001L0003J0P00EP0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P3G0QP0f09D00@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00O08002d000270P00<@0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P1m0Rh0Q@8_00@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00eP8002d0003Q0P00<@0000P0003j0P000@00000000040000;@4900@0000]0@80\n200002D30P3G0Rh0g`8_00@0000]0@H010000?012@040000;@4800P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00^0800340003>0P00?`0000P0003j0P000@0006]]J`040000;@4900L0003l0P00\nJfe[0000100002d12P0<000090<40<`2=P2i0S80^@8m0<`2=P040000;@4600@0003`0@T0100002d1\n20040000l04:00P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00S`8003D0002k0P00>00000P0\n003j0P000@00000000040000;@4900@0000]0@80200002D30P2@0SH0^@8f00@0000]0@H010000?01\n2@040000;@4800P0000V1Pl01P3ooooo0@040000;@4200@0000]0@<01`0001/4W0160f<0A@840000\n;@4800@0000]0@H00`0001h01`0001H4V`150f<0A@8500000P410000100002d11`0500002@800002\n4`00038:H`150PD010150V<0A0>K07=`LVU^01404P0;00L04`0G0000<PYS08d2200404D2H`140i/0\nIdAQKG1ULS4B01T04P0L01<04@0<0180100002d11@0=0000<PXX02P30@0404D2H`140i/0800^00@0\n000W0Ooo400002H63`0F0?ooool004@300140000W@<004L000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80A@=509/3A@040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40400002H63`0F0?ooool008<300140000W@<005T000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80V`=508@3E`040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40400002H63`0F0?ooool006d300140000QP<005T000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80Q0=506h3E`040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40400002H63`0F0?ooool005X300140000L0<005T000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80KP=505/3E`040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40400002H63`0F0?ooool004@300140000G@<005T000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80F`=504D3E`040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40400002H63`0F0?ooool006d3000a0000L0<004L000080000nP80004000000000100002d1\n2@040000;@4200P0000U0`80KP=506h3<P040000;@4600@0003`0@T0100002d1200800009PH?00H0\nooooo`40100002d10P080000nP8000400000_`02100002d12@0700006`@c07H39`1Z0`@0000]0@P0\n100002d11P040000l04900@0000]0@80100002d10`0700006`BL0;h3H`150`@0000]0@P0100002d1\n1P0300007P0700005PBK0;h3H`150`D000020@4000040000;@4700D000090P00008D0000<PYS04D3\n1P0404D3H`2m0i/0IVUhIF@a2P0701004P0C0180100002d11@0=0000<PXX09d30@0404D3H`2m0i/0\n800^00@0000W0Ooo400002H63`0F0?ooool00;T0000G0000i@0004<000080000nP80004000000000\n100002d12@070000o0800?ooo`0000@0000]0@X0<P0002@35`3S02h0h`0b0>00>@3L03d0f0110=40\n@@3=0440a@110<80?@2n03T0^P0b0;X0;P2j02L0_P0S0<807`3501/0c@0H0=40603H01/0g00O0>00\n8`3S02L0h`0^00@0000]0@H010000?012@040000;@4800@0003`0@X0200002H63`060?ooool10100\n000V1Pl05P3ooooo003800009P000=H0000d000020000?X200010000000000@0000]0@T01`000?`2\n0000000000040000;@4:01X0000T0`/0e00^0=@0<P3A0380c@0b0<T0;P3902X0c@0Z0<d09`3A02X0\ne00Z0=@0;P040000;@4600@0003`0@T0100002d120040000l04:00P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00[`0002L0002k0000<`0000@0000]0@<01`000?`20000000000040000;@4901h0\n000h1@801P060:l09`2j02L0^P0b0:l0<P2_02L0[`0W0:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z00@0\n000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00[`0002X0\n002k0000<`0000@0000]0@<01`000?`2000Q8B4000040000;@4901h0000h1@801P060:l0:P2j02X0\n^P0b0:l0<P2_02X0[`0Z0:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z00@0000]0@H0100002d120040000\nl04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00[`0002X0002k0000<`0000@0000]0@<0\n1`000?`20017ATL000040000;@4901h0000h1@801P060:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z0:l0\n:P2j02X0^P0b0:l0<P2_02X0[`0Z00@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo\n0@0@00009PH?01H0ooooo`00[`0002X0002k0000<`0000@0000]0@<01`000?`2001fM7H000040000\n;@4901h0000h1@801P060:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z0:l0:P2j02X0^P0^0:l0;P2_02X0\n[`0Z00@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00\n[`0002X0002k0000;`0000@0000]0@<01`000?`2002MVYd000040000;@4901X0000h1@801P040:l0\n:P2j02X0^P0^0:l0;P2_02X0[`0Z0:l0;P2j02h0[`0^0:l0;P040000;@4600@0000]0@P010000?01\n2@0800009PH?00H0ooooo`40400002H63`0F0?ooool00:l0000^0000^`0002l000040000;@4300L0\n003l0P00]K:e0000100002d12@0F0000>0D200@0102_02h0^P0^0:l0;P2_02h0[`0^0;X0;P2_02h0\n[`0^00@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00\n[`0002h0002k0000;`0000@0000]0@<01`000?`20036`lH000040000;@4901H0000h1@8010040:l0\n;P2j02h0[`0^0:l0;P2_02h0^P0^0:l0;P2_02h0100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool100@0000]0@8020000?X200010000003o0P@0000]0@T01`0001/4<`2k02L0[`040000\n;@4800@0000]0@H010000?012@0@00009PH?01H0ooooo`00]@0004@0000?0@00CP0000P0003j0P00\n0@00000000040000;@4900L0003l0P00Ngek0000100002d12P0H000090<:0;H0C03204D0c@1<0=P0\nA@3S04`0k`150?H0C0010DD03@5<0;H0C0040000;@4600@0003`0@T0100002d120040000l04:00P0\n000V1Pl01P3ooooo0@070000o08007]mN`0000@0000]0@T020000?X20001000000000P@0000]0@X0\n1`0001/4F@0I0D`0]`040000;@4800@0003`0@T0100002d11P040000l04:00L0003l0P00Ngek0000\n100002d12@080000nP80004000000002100002d12P0700006`A=01T1<P0A0@@0000]0@P010000?01\n2@040000;@4600@0003`0@X01`000?`2001[KF/000040000;@4900P0003j0P000@00000000840000\n;@4:00L0000K13<0/00W0:@0100002d120040000l04900@0000]0@H010000?012P040000;@4200P0\n003j0P000@00002o00840000;@4900L0000K13<07@4W0141100002d120040000;@4600@0003`0@T0\n100002d10P040000;@4300L0000K19`0K`5S07H0100002d120040000;@4600<0000N00L0000F19/0\nK`5S07H01@0000810@0000@0000]0@L01@0000T200000Ql0000b2V<0MP0=00@0MP1S06h1V`1YI6EQ\nK4MUHG9B<U@a;@L04`0B01801`0K01804P0<01P04P0E0180100002d11@0=0000<PXX05L10@0407H0\nH`1^0I/0800^00@0000W0Ooo1`000?`2001[KF/000040000;@4900P0003j0P000@00000000840000\n;@4:00L0000K10D1<03i02@0100002d120040000l04900@0000]0@H010000?012P070000o0800?oo\no`0000@0000]0@T020000?X20001000000000P@0000]0@X01`0001/4V00`08`090040000;@4800@0\n003`0@T0100002d11P040000l04:0100000V1Pl05P3ooooo000f0000j@0003T0003g000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<203L0m@0g0>X0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000f0000WP0003T0002X000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<203L0YP0g09l0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000f0000k00005<0003`000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<203L0k@1A0>h0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000f0000X@0005<0002U000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<203L0XP1A0:<0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000K0000j@0001l0003g000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<201`0m@0M0>X0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000K0000WP0001l0002X000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<201`0YP0M09l0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000@0000k00001h0003`000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<20140k@0L0>h0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000@0000X@0001h0002U000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<20140XP0L0:<0100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo001@0000X@0005<0003_000020000?X2\n00010000000000@0000]0@T0100002d10P0800009@<20540k@1A0:80100002d11P040000l04900@0\n000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000A0000[P0004L0003S0000100002d1\n0`070000o0800000000000@0000]0@T07P0003P50P0600H04@3R0140[P160:h0AP3R0140hP0A0>80\n4@3R0140[P160:h0AP3R0140hP0A0>80100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000A0000[P0004L0003S0000100002d10`070000o08000H61P0000@0\n000]0@T07P0003P50P0600H04@3R0140[P160:h0AP3R0140hP0A0>804@3R0140[P160:h0AP3R0140\nhP0A0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000A0000[P0004L0003S0000100002d10`070000o08000d=3@0000@0000]0@T07P0003P50P0600H0\n4@3R0140[P160:h0AP3R0140hP0A0>804@3R0140[P120:h0@P3R0140hP0A0>80100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000A0000[P0004<0003S0000\n100002d10`070000o08001@D500000@0000]0@T07P0003P50P0600H04@3R0140[P120:h0@P3R0140\nhP0A0>805@3R01D0[P120:h0@P3R01D0hP0E0>80100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000E0000[P0004<0003S0000100002d10`070000o08001/K\n6`0000@0000]0@T07P0003P50P0600H05@3R01D0[P120:h0@P3R01D0hP0E0>805@3R01D0[P120:h0\n@P3R01D0hP0E0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000E0000[P0004<0003S0000100002d10`070000o08002<S8`0000@0000]0@T07P0003P5\n0P0600H05@3R01D0[P120:h0@P3R01D0hP0E0>805@3R01D0[P120:h0@P3R01D0hP0E0>80100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000E0000[P0004<0\n003S0000100002d10`070000o08002/[:`0000@0000]0@T07P0003P50P0600H05@3R01D0[P120:h0\n@P3R01D0hP0E0>805@3R01D0[P0n0:h0?P3R01D0hP0E0>80100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000E0000[P0003l0003S0000100002d10`070000\no08003@c=00000@0000]0@T07P0003P50P0600H05@3R01D0[P0n0:h0?P3R01D0hP0E0>806@3R01T0\n[P0n0:h0?P3R01T0hP0I0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000I0000[P0003l0003S0000100002d10`070000o08003dl?@0000@0000]0@T0\n7P0003P50P0600H06@3R01T0[P0n0:h0?P3R01T0hP0I0>806@3R01T0[P0n0:h0?P3R01T0hP0I0>80\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000I0000\n[P0003l0003S0000100002d10`070000o08004I5AP0000@0000]0@T07P0003P50P0600H06@3R01T0\n[P0n0:h0?P3R01T0hP0I0>806@3R01T0[P0n0:h0?P3R01T0hP0I0>80100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000I0000[P0003l0003S0000100002d1\n0`070000o080051?D00000@0000]0@T07P0003P50P0600H06@3R01T0[P0n0:h0?P3R01T0hP0I0>80\n6@3R01T0[P0j0:h0>P3R01T0hP0I0>80100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000I0000[P0003/0003S0000100002d10`070000o08005]IF`0000@0\n000]0@T07P0003P50P0600H06@3R01T0[P0j0:h0>P3R01T0hP0I0>80703R01`0[P0j0:h0>P3R01`0\nhP0L0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000L0000[P0003/0003S0000100002d10`070000o08006ITIP0000@0000]0@T07P0003P50P0600H0\n703R01`0[P0j0:h0>P3R01`0hP0L0>80703R01`0[P0j0:h0>P3R01`0hP0L0>80100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000L0000[P0003/0003S0000\n100002d10`070000o080075`L@0000@0000]0@T07P0003P50P0600H0703R01`0[P0j0:h0>P3R01`0\nhP0L0>80703R01`0[P0j0:h0>P3R01`0hP0L0>80100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000L0000[P0003/0003S0000100002d10`070000o08007ek\nO@0000@0000]0@T07P0003P50P0600H0703R01`0[P0j0:h0>P3R01`0hP0L0>80703R01`0[P0g0:h0\n=`3R01`0hP0L0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000L0000[P0003P0003S0000100002d10`070000o08008N5Q`0000@0000]0@T07P0003P5\n0P0600H0703R01`0[P0g0:h0=`3R01`0hP0L0>80703R01`0[P0g0:h0=`3R01`0hP0L0>80100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000L0000[P0003P0\n003S0000100002d10`070000o080092>T00000@0000]0@T07P0003P50P0600H0703R01`0[P0g0:h0\n=`3R01`0hP0L0>80803R0200[P0g0:h0=`3R0200hP0P0>80100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000P0000[P0003P0003S0000100002d10`070000\no08009RFV00000@0000]0@T07P0003P50P0600H0803R0200[P0g0:h0=`3R0200hP0P0>80803R0200\n[P0g0:h0=`3R0200hP0P0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000P0000[P0003P0003S0000100002d10`070000o08009nLW`0000@0000]0@T0\n7P0003P50P0600H0803R0200[P0g0:h0=`3R0200hP0P0>80803R0200[P0c0:h0<`3R0200hP0P0>80\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000P0000\n[P0003@0003S0000100002d10`070000o0800:FSY@0000@0000]0@T07P0003P50P0600H0803R0200\n[P0c0:h0<`3R0200hP0P0>80803R0200[P0c0:h0<`3R0200hP0P0>80100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000P0000[P0003@0003S0000100002d1\n0`070000o0800:^XZ`0000@0000]0@T07P0003P50P0600H0803R0200[P0c0:h0<`3R0200hP0P0>80\n903R02@0[P0c0:h0<`3R02@0hP0T0>80100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000T0000[P0003@0003S0000100002d10`070000o0800;2]/00000@0\n000]0@T07P0003P50P0600H0903R02@0[P0c0:h0<`3R02@0hP0T0>80903R02@0[P0c0:h0<`3R02@0\nhP0T0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000T0000[P0003@0003S0000100002d10`070000o0800;Fb]@0000@0000]0@T07P0003P50P0600H0\n903R02@0[P0c0:h0<`3R02@0hP0T0>80903R02@0[P0_0:h0;`3R02@0hP0T0>80100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000T0000[P000300003S0000\n100002d10`070000o0800;Vf^@0000@0000]0@T07P0003P50P0600H0903R02@0[P0_0:h0;`3R02@0\nhP0T0>80903R02@0[P0_0:h0;`3R02@0hP0T0>80100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000T0000[P000300003S0000100002d10`070000o0800;fj\n_@0000@0000]0@T07P0003P50P0600H0903R02@0[P0_0:h0;`3R02@0hP0T0>80:03R02P0[P0_0:h0\n;`3R02P0hP0X0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000X0000[P000300003S0000100002d10`070000o0800<6n`@0000@0000]0@T07P0003P5\n0P0600H0:03R02P0[P0_0:h0;`3R02P0hP0X0>80:03R02P0[P0_0:h0;`3R02P0hP0X0>80100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000[P000300\n003S0000100002d10`070000o0800<C1a00000@0000]0@T07P0003P50P0600H0:03R02P0[P0_0:h0\n;`3R02P0hP0X0>80:03R02P0[P0_0:h0;`3R02P0hP0X0>80100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000X0000[P000300003S0000100002d10`070000\no0800<K3aP0000@0000]0@T07P0003P50P0600H0:03R02P0[P0_0:h0;`3R02P0hP0X0>80:03R02P0\n[P0[0:h0:`3R02P0hP0X0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000X0000[P0002`0003S0000100002d10`070000o0800<K3aP0000@0000]0@T0\n6P0003P50P0600@0:03R02P0[P0[0:h0:`3R02P0hP0X0>80:`3R02/0[P0[0>80:`3R00@0000]0@H0\n100002d120040000l04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00:`000:h0000/0000\nh`0000@0000]0@<01`000?`20036`lH000040000;@4901H0000h1@80100402/0hP0[0:h0:`3R02/0\nhP0[0>80:`2^02/0hP0[0>80100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000[0000[P0002`0003S0000100002d10`070000o0800<K3aP0000@0000]0@T0\n5P0003P50P0400@0:`3R02/0[P0[0>80:`3R02/0hP0[0:h0:`3R02/0hP040000;@4600@0000]0@P0\n10000?012@0800009PH?00H0ooooo`40100002d10P080000nP80004000000002100002d12@070000\n6`CS04L0[P0A00@0000]0@P0100002d11P040000l0490100000V1Pl05P3ooooo000T0000hP000300\n003j0000100002d10`070000o0800000000000@0000]0@T07P0003P50P0600H0903i02@0hP0_0>80\n;`3i02@0n@0T0?T0:03i02P0hP0_0>80;`3i02P0n@0X0?T0100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000X0000hP000300003j0000100002d10`070000\no080024Q8@0000@0000]0@T07P0003P50P0600H0:03i02P0hP0_0>80;`3i02P0n@0X0?T0:03i02P0\nhP0_0>80;`3i02P0n@0X0?T0100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000X0000hP000300003j0000100002d10`070000o08004M6A`0000@0000]0@T0\n7P0003P50P0600H0:03i02P0hP0_0>80;`3i02P0n@0X0?T0:03i02P0hP0_0>80;`3i02P0n@0X0?T0\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nhP000300003j0000100002d10`070000o08007IdMP0000@0000]0@T07P0003P50P0600H0:03i02P0\nhP0_0>80;`3i02P0n@0X0?T0:03i02P0hP0[0>80:`3i02P0n@0X0?T0100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000hP0002`0003j0000100002d1\n0`070000o08009fJW@0000@0000]0@T06P0003P50P0600@0:03i02P0hP0[0>80:`3i02P0n@0X0?T0\n:`3i02/0hP0[0?T0:`3i00@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00:`000>80000/0000nP0000@0000]0@<01`000?`2002e/[D000040000;@4901H0\n000h1@80100402/0n@0[0>80:`3i02/0n@0[0?T0:`3R02/0n@0[0?T0100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000[0000hP0002`0003j0000100002d1\n0`070000o0800<K3aP0000@0000]0@T05P0003P50P0400@0:`3i02/0hP0[0?T0:`3i02/0n@0[0>80\n:`3i02/0n@040000;@4600@0000]0@P010000?012@0800009PH?00H0ooooo`40100002d10P080000\nnP80004000000002100002d12@0700006`Cj0300hP0T00@0000]0@P0100002d11P040000l0490100\n000V1Pl05P3ooooo000T0000U`000300002_0000100002d10`070000o0800000000000@0000]0@T0\n7P0003P50P0600H0902^02@0U`0_09L0;`2^02@0[P0T0:h0:02^02P0U`0_09L0;`2^02P0[P0X0:h0\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nU`000300002_0000100002d10`070000o080024Q8@0000@0000]0@T07P0003P50P0600H0:02^02P0\nU`0_09L0;`2^02P0[P0X0:h0:02^02P0U`0_09L0;`2^02P0[P0X0:h0100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000U`000300002_0000100002d1\n0`070000o08004M6A`0000@0000]0@T07P0003P50P0600H0:02^02P0U`0_09L0;`2^02P0[P0X0:h0\n:02^02P0U`0_09L0;`2^02P0[P0X0:h0100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000X0000U`000300002_0000100002d10`070000o08007IdMP0000@0\n000]0@T07P0003P50P0600H0:02^02P0U`0_09L0;`2^02P0[P0X0:h0:02^02P0U`0[09L0:`2^02P0\n[P0X0:h0100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000X0000U`0002`0002_0000100002d10`070000o08009fJW@0000@0000]0@T06P0003P50P0600@0\n:02^02P0U`0[09L0:`2^02P0[P0X0:h0:`2^02/0U`0[0:h0:`2^00@0000]0@H0100002d120040000\nl04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00:`0009L0000/0000[`0000@0000]0@<0\n1`000?`2002e/[D000040000;@4901H0000h1@80100402/0[P0[09L0:`2^02/0[P0[0:h0:`2G02/0\n[P0[0:h0100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000[0000U`0002`0002_0000100002d10`070000o0800<K3aP0000@0000]0@T05P0003P50P0400@0\n:`2^02/0U`0[0:h0:`2^02/0[P0[09L0:`2^02/0[P040000;@4600@0000]0@P010000?012@080000\n9PH?00H0ooooo`40100002d10P080000nP80004000000002100002d12@0700006`B_0300U`0T00@0\n000]0@P0100002d11P040000l04900@0000]0@80100002d10`0700006`Cn0?@0a01_00@0000]0@P0\n100002d11P0300007P0700005PCl0?<0a01_00D000020@4000040000;@4700D000090P00008G0000\n<P[406l0200406l0a03b0?`0JFiULWAYHC4701<04P0;00T0200B0180100002d11@0=0000<PZ90=/0\n0@0406l0a03b0?`0800^00@0000W0Ooo1`000?`2001[KF/000040000;@4900P0003j0P000@000000\n00840000;@4:00L0000K1:H1<02J0B@0100002d120040000l04900@0000]0@H010000?012P070000\no0800?ooo`0000@0000]0@T020000?X20001000000000P@0000]0@X01`0001/4>P4`02d190040000\n;@4800@0003`0@T0100002d11P040000l04:0100000V1Pl05P3ooooo00050000?`4002H0002D0@00\n20000?X200010000000000@0000]0@T0100002d10P0D00009@<801D0TP4E08/190200@H0L@4T0681\n1P1C0AD0A`4E0401100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000D0000T@4005<0002E0@0020000?X200010000000000@0000]0@T0100002d10P080000\n9@<201D0TP5A09<1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl0\n5P3ooooo001@0000PP4005<0002D0@0020000?X200010000000000@0000]0@T0100002d10P080000\n9@<20540TP5A08<1100002d11P040000l04900@0000]0@P0200002H63`060?ooool100L0003l0P00\na/?60000100002d12@080000nP80004000000002100002d12P0700006`B50F40E`5200@0000]0@P0\n10000?012@040000;@4600@0003`0@X0400002H63`0F0?ooool0044000130@00A00008D100080000\nnP80004000000000100002d12@040000;@4200P0000U0`80@P230D80A0440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool005l000130@00HP0008D100080000\nnP80004000000000100002d12@040000;@4200P0000U0`80H0230F00A0440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool00500000o0@00D`0005P100080000\nnP80004000000000100002d12@040000;@4200P0000U0`80D@1F0E40@0440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool001@0000o0@00D`00048100080000\nnP80004000000000100002d12@040000;@4200P0000U0`805@100E40@0440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool002X0002A0@00;P0009`100080000\nnP80004000000000100002d12@040000;@4200P0000U0`80:`2J0B`0TP440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40400002H63`0F0?ooool002X0000g0@00;P00048100080000\nnP80004000000000100002d12@040000;@4200P0000U0`80:`100B`0>0440000;@4600@0003`0@T0\n100002d1200800009PH?00H0ooooo`40100002d10P040000;@4300L0000K19@1N`5J0GX0100002d1\n20040000;@4600<0000N00L0000F1981N`5J0GX01@0000810@0000@0000]0@L01@0000T200000Ql0\n000b2UX1NP0=00@0NP1J0GX1TP5cL79YKVM4HFe`IG8b;A404P0;00L04`0B01T04P0L01<04@0<0180\n100002d11@0=0000<PXO0Ed10@0407X0FP5j0I81800^00@0000W0Ooo1`000?`2001[KF/000040000\n;@4900P0003j0P000@00000000840000;@4:00L0000K14l2<0130R@0100002d120040000l04900@0\n000]0@H010000?012P070000o0800?ooo`0000@0000]0@T020000?X20001000000000P@0000]0@X0\n1`0001/4hP4`0=H190040000;@4800@0003`0@T0100002d11P040000l04:0100000V1Pl05P3ooooo\n000f0000;`8003T0000n0P0020000?X200010000000000@0000]0@T0100002d10P0800009@<203L0\n?08g0302100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000f0000i04003T0003b0@0020000?X200010000000000@0000]0@T0100002d10P0800009@<203L0\nl04g0>D1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000f0000=`8005<0000j0P0020000?X200010000000000@0000]0@T0100002d10P0800009@<203L0\n>09A03P2100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000f0000j`4005<0003_0@0020000?X200010000000000@0000]0@T0100002d10P0800009@<203L0\nk05A0>d1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000K0000;`8001l0000n0P0020000?X200010000000000@0000]0@T0100002d10P0800009@<201`0\n?08M0302100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000K0000i04001l0003b0@0020000?X200010000000000@0000]0@T0100002d10P0800009@<201`0\nl04M0>D1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000@0000=`8001h0000j0P0020000?X200010000000000@0000]0@T0100002d10P0800009@<20140\n>08L03P2100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000@0000j`4001h0003_0@0020000?X200010000000000@0000]0@T0100002d10P0800009@<20140\nk04L0>d1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n001@0000j`4005<0000j0P0020000?X200010000000000@0000]0@T0100002d10P0800009@<20540\n>09A0>`1100002d11P040000l04900@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo\n000A0000n04004L0000^0P00100002d10`070000o0800000000000@0000]0@T07P0003P50P0600H0\n4@0]0Q40n0560?P1AP0]0Q40;@8A02d24@0]0Q40n0560?P1AP0]0Q40;@8A02d2100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000A0000n04004L0000^0P00\n100002d10`070000o08000H61P0000@0000]0@T07P0003P50P0600H04@0]0Q40n0560?P1AP0]0Q40\n;@8A02d24@0]0Q40n0560?P1AP0]0Q40;@8A02d2100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000A0000n04004L0000^0P00100002d10`070000o08000d=\n3@0000@0000]0@T07P0003P50P0600H04@0]0Q40n0560?P1AP0]0Q40;@8A02d24@0]0Q40n0520?P1\n@P0]0Q40;@8A02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000A0000n04004<0000^0P00100002d10`070000o08001@D500000@0000]0@T07P0003P5\n0P0600H04@0]0Q40n0520?P1@P0]0Q40;@8A02d25@0]0QD0n0520?P1@P0]0QD0;@8E02d2100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000E0000n04004<0\n000^0P00100002d10`070000o08001/K6`0000@0000]0@T07P0003P50P0600H05@0]0QD0n0520?P1\n@P0]0QD0;@8E02d25@0]0QD0n0520?P1@P0]0QD0;@8E02d2100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000E0000n04004<0000^0P00100002d10`070000\no08002<S8`0000@0000]0@T07P0003P50P0600H05@0]0QD0n0520?P1@P0]0QD0;@8E02d25@0]0QD0\nn0520?P1@P0]0QD0;@8E02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000E0000n04004<0000^0P00100002d10`070000o08002/[:`0000@0000]0@T0\n7P0003P50P0600H05@0]0QD0n0520?P1@P0]0QD0;@8E02d25@0]0QD0n04n0?P1?P0]0QD0;@8E02d2\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000E0000\nn04003l0000^0P00100002d10`070000o08003@c=00000@0000]0@T07P0003P50P0600H05@0]0QD0\nn04n0?P1?P0]0QD0;@8E02d26@0]0QT0n04n0?P1?P0]0QT0;@8I02d2100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000I0000n04003l0000^0P00100002d1\n0`070000o08003dl?@0000@0000]0@T07P0003P50P0600H06@0]0QT0n04n0?P1?P0]0QT0;@8I02d2\n6@0]0QT0n04n0?P1?P0]0QT0;@8I02d2100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000I0000n04003l0000^0P00100002d10`070000o08004I5AP0000@0\n000]0@T07P0003P50P0600H06@0]0QT0n04n0?P1?P0]0QT0;@8I02d26@0]0QT0n04n0?P1?P0]0QT0\n;@8I02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000I0000n04003l0000^0P00100002d10`070000o080051?D00000@0000]0@T07P0003P50P0600H0\n6@0]0QT0n04n0?P1?P0]0QT0;@8I02d26@0]0QT0n04j0?P1>P0]0QT0;@8I02d2100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000I0000n04003/0000^0P00\n100002d10`070000o08005]IF`0000@0000]0@T07P0003P50P0600H06@0]0QT0n04j0?P1>P0]0QT0\n;@8I02d2700]0Q`0n04j0?P1>P0]0Q`0;@8L02d2100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000L0000n04003/0000^0P00100002d10`070000o08006IT\nIP0000@0000]0@T07P0003P50P0600H0700]0Q`0n04j0?P1>P0]0Q`0;@8L02d2700]0Q`0n04j0?P1\n>P0]0Q`0;@8L02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000L0000n04003/0000^0P00100002d10`070000o080075`L@0000@0000]0@T07P0003P5\n0P0600H0700]0Q`0n04j0?P1>P0]0Q`0;@8L02d2700]0Q`0n04j0?P1>P0]0Q`0;@8L02d2100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000L0000n04003/0\n000^0P00100002d10`070000o08007ekO@0000@0000]0@T07P0003P50P0600H0700]0Q`0n04j0?P1\n>P0]0Q`0;@8L02d2700]0Q`0n04g0?P1=`0]0Q`0;@8L02d2100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000L0000n04003P0000^0P00100002d10`070000\no08008N5Q`0000@0000]0@T07P0003P50P0600H0700]0Q`0n04g0?P1=`0]0Q`0;@8L02d2700]0Q`0\nn04g0?P1=`0]0Q`0;@8L02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000L0000n04003P0000^0P00100002d10`070000o080092>T00000@0000]0@T0\n7P0003P50P0600H0700]0Q`0n04g0?P1=`0]0Q`0;@8L02d2800]0R00n04g0?P1=`0]0R00;@8P02d2\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000P0000\nn04003P0000^0P00100002d10`070000o08009RFV00000@0000]0@T07P0003P50P0600H0800]0R00\nn04g0?P1=`0]0R00;@8P02d2800]0R00n04g0?P1=`0]0R00;@8P02d2100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000P0000n04003P0000^0P00100002d1\n0`070000o08009nLW`0000@0000]0@T07P0003P50P0600H0800]0R00n04g0?P1=`0]0R00;@8P02d2\n800]0R00n04c0?P1<`0]0R00;@8P02d2100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000P0000n04003@0000^0P00100002d10`070000o0800:FSY@0000@0\n000]0@T07P0003P50P0600H0800]0R00n04c0?P1<`0]0R00;@8P02d2800]0R00n04c0?P1<`0]0R00\n;@8P02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000P0000n04003@0000^0P00100002d10`070000o0800:^XZ`0000@0000]0@T07P0003P50P0600H0\n800]0R00n04c0?P1<`0]0R00;@8P02d2900]0R@0n04c0?P1<`0]0R@0;@8T02d2100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000T0000n04003@0000^0P00\n100002d10`070000o0800;2]/00000@0000]0@T07P0003P50P0600H0900]0R@0n04c0?P1<`0]0R@0\n;@8T02d2900]0R@0n04c0?P1<`0]0R@0;@8T02d2100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000T0000n04003@0000^0P00100002d10`070000o0800;Fb\n]@0000@0000]0@T07P0003P50P0600H0900]0R@0n04c0?P1<`0]0R@0;@8T02d2900]0R@0n04_0?P1\n;`0]0R@0;@8T02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl0\n5P3ooooo000T0000n0400300000^0P00100002d10`070000o0800;Vf^@0000@0000]0@T07P0003P5\n0P0600H0900]0R@0n04_0?P1;`0]0R@0;@8T02d2900]0R@0n04_0?P1;`0]0R@0;@8T02d2100002d1\n1P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000T0000n0400300\n000^0P00100002d10`070000o0800;fj_@0000@0000]0@T07P0003P50P0600H0900]0R@0n04_0?P1\n;`0]0R@0;@8T02d2:00]0RP0n04_0?P1;`0]0RP0;@8X02d2100002d11P040000;@4800@0003`0@T0\n200002H63`060?ooool10100000V1Pl05P3ooooo000X0000n0400300000^0P00100002d10`070000\no0800<6n`@0000@0000]0@T07P0003P50P0600H0:00]0RP0n04_0?P1;`0]0RP0;@8X02d2:00]0RP0\nn04_0?P1;`0]0RP0;@8X02d2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100\n000V1Pl05P3ooooo000X0000n0400300000^0P00100002d10`070000o0800<C1a00000@0000]0@T0\n7P0003P50P0600H0:00]0RP0n04_0?P1;`0]0RP0;@8X02d2:00]0RP0n04_0?P1;`0]0RP0;@8X02d2\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nn0400300000^0P00100002d10`070000o0800<K3aP0000@0000]0@T07P0003P50P0600H0:00]0RP0\nn04_0?P1;`0]0RP0;@8X02d2:00]0RP0n04[0?P1:`0]0RP0;@8X02d2100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000n04002`0000^0P00100002d1\n0`070000o0800<K3aP0000@0000]0@T06P0003P50P0600@0:00]0RP0n04[0?P1:`0]0RP0;@8X02d2\n:`0]0R/0n04[02d2:`0]0P@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@0000\n9PH?01H0ooooo`00:`000?P1000/0000;P8000@0000]0@<01`000?`20036`lH000040000;@4901H0\n000h1@80100402/0;@8[0?P1:`0]0R/0;@8[02d2:`3h0B/0;@8[02d2100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000[0000n04002`0000^0P00100002d1\n0`070000o0800<K3aP0000@0000]0@T05P0003P50P0400@0:`0]0R/0n04[02d2:`0]0R/0;@8[0?P1\n:`0]0R/0;@840000;@4600@0000]0@P010000?012@0800009PH?00H0ooooo`40100002d10P080000\nnP80004000000002100002d12@0700006`@^0TL0n04A00@0000]0@P0100002d11P040000l0490100\n000V1Pl05P3ooooo000T0000;@80030000140P00100002d10`070000o0800000000000@0000]0@T0\n7P0003P50P0600H090130R@0;@8_02d2;`130R@0@`8T04<2:0130RP0;@8_02d2;`130RP0@`8X04<2\n100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\n;@80030000140P00100002d10`070000o080024Q8@0000@0000]0@T07P0003P50P0600H0:0130RP0\n;@8_02d2;`130RP0@`8X04<2:0130RP0;@8_02d2;`130RP0@`8X04<2100002d11P040000;@4800@0\n003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000;@80030000140P00100002d1\n0`070000o08004M6A`0000@0000]0@T07P0003P50P0600H0:0130RP0;@8_02d2;`130RP0@`8X04<2\n:0130RP0;@8_02d2;`130RP0@`8X04<2100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000X0000;@80030000140P00100002d10`070000o08007IdMP0000@0\n000]0@T07P0003P50P0600H0:0130RP0;@8_02d2;`130RP0@`8X04<2:0130RP0;@8[02d2:`130RP0\n@`8X04<2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000X0000;@8002`000140P00100002d10`070000o08009fJW@0000@0000]0@T06P0003P50P0600@0\n:0130RP0;@8[02d2:`130RP0@`8X04<2:`130R/0;@8[04<2:`130P@0000]0@H0100002d120040000\nl04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00:`0002d2000/0000A08000@0000]0@<0\n1`000?`2002e/[D000040000;@4901H0000h1@80100402/0@`8[02d2:`130R/0@`8[04<2:`0]0R/0\n@`8[04<2100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000[0000;@8002`000140P00100002d10`070000o0800<K3aP0000@0000]0@T05P0003P50P0400@0\n:`130R/0;@8[04<2:`130R/0@`8[02d2:`130R/0@`840000;@4600@0000]0@P010000?012@080000\n9PH?00H0ooooo`40100002d10P080000nP80004000000002100002d12@0700006`A40S00;@8T00@0\n000]0@P0100002d11P040000l0490100000V1Pl05P3ooooo000T0000h@400300003i0@00100002d1\n0`070000o0800000000000@0000]0@T07P0003P50P0600H0903h0B@0h@4_0>41;`3h0B@0n04T0?P1\n:03h0BP0h@4_0>41;`3h0BP0n04X0?P1100002d11P040000;@4800@0003`0@T0200002H63`060?oo\nool10100000V1Pl05P3ooooo000X0000h@400300003i0@00100002d10`070000o080024Q8@0000@0\n000]0@T07P0003P50P0600H0:03h0BP0h@4_0>41;`3h0BP0n04X0?P1:03h0BP0h@4_0>41;`3h0BP0\nn04X0?P1100002d11P040000;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo\n000X0000h@400300003i0@00100002d10`070000o08004M6A`0000@0000]0@T07P0003P50P0600H0\n:03h0BP0h@4_0>41;`3h0BP0n04X0?P1:03h0BP0h@4_0>41;`3h0BP0n04X0?P1100002d11P040000\n;@4800@0003`0@T0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000h@400300003i0@00\n100002d10`070000o08007IdMP0000@0000]0@T07P0003P50P0600H0:03h0BP0h@4_0>41;`3h0BP0\nn04X0?P1:03h0BP0h@4[0>41:`3h0BP0n04X0?P1100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000X0000h@4002`0003i0@00100002d10`070000o08009fJ\nW@0000@0000]0@T06P0003P50P0600@0:03h0BP0h@4[0>41:`3h0BP0n04X0?P1:`3h0B/0h@4[0?P1\n:`3h0@@0000]0@H0100002d120040000l04900P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00\n:`000>41000/0000n@4000@0000]0@<01`000?`2002e/[D000040000;@4901H0000h1@80100402/0\nn04[0>41:`3h0B/0n04[0?P1:`3Q0B/0n04[0?P1100002d11P040000;@4800@0003`0@T0200002H6\n3`060?ooool10100000V1Pl05P3ooooo000[0000h@4002`0003i0@00100002d10`070000o0800<K3\naP0000@0000]0@T05P0003P50P0400@0:`3h0B/0h@4[0?P1:`3h0B/0n04[0>41:`3h0B/0n0440000\n;@4600@0000]0@P010000?012@0800009PH?00H0ooooo`40100002d10P080000nP80004000000002\n100002d12@0700006`Ci0C00h@4T00@0000]0@P0100002d11P040000l04900@0000]0@80100002d1\n0`0700006`@Z0PH1l@6200@0000]0@P0100002d11P0300007P0700005P@Y0PH1l@6200D000020@40\n00040000;@4700D000090P00008G0000<P[a0H8020040880l@450BT2JFiULWAYHC8701<04P0;00T0\n200B0180100002d11@0=0000<PZf0Nh00@040880l@450BT2800^00@0000W0Ooo100002d10P040000\n;@4300L0000K1=H2:06M0^L0100002d120040000;@4600<0000N00L0000F1=D2:06M0^L01@000081\n0@0001`0003k0^Oo00000000T040000014008T5bJF5/0?Eg@00009X62Q^nH_Egaf;eM`4000000300\n100002d12@0500002@8000024000038:W@;W00<0103W09d29`7E0WAQM@0700h03P040000;@4500d0\n000b2UX22P4100@0i`2M0RL1e@8P02d0100002L1ool@00009PH?01H0ooooo`00/P0006P2000B0@00\nS08000P0003j0P000@00000000040000;@4:00@0000]0@805P0002D32@0@0HX20@5k0_80L0;S06T2\nd@1/0/D0M0:n07/2]P230[<0RP840000;@4600@0003`0@X0100002d1200800009PH?00H0ooooo`40\n400002H63`0F0?ooool00;80001[0P00c`0008`200080000nP80004000000000100002d12P070000\no0800000000000@0000]0@/0300002@3102c08X2_P1/0/d0N`:c08X2100002d11P040000l04:00@0\n000]0@P010000?012`0800009PH?00H0ooooo`40400002H63`0F0?ooool001L1001n0P00;@4009P2\n00080000nP80004000000?l0100002d12P070000o0800000o`0000@0000]0@/0300002@3100[0Gl2\n606:0R/1UP8[0Gl2100002d11P040000l04:00@0000]0@P010000?012`0800009PH?00H0ooooo`40\n1`000?`2003oool000040000;@4:00P0003j0P000@00000000840000;@4;00L0000K18l2/0270Z@0\n100002d120040000l04:00@0000]0@H010000?012`040000;@4200@0000]0@<01`0001/4n08d0Kl2\nZ0040000;@4800@0000]0@H00`0001h01`0001H4m`8c0Kl2Z00500000P410000100002d11`050000\n2@8000025P00038:_`:X00L0102X0;l2<P7g0WA_LW5eIC402P0A00/04`0C01804P040000;@4500d0\n000b2X@2604100@0Z02o0S81m`8P02h0100002L1ool70000o0800?ooo`0000@0000]0@X020000?X2\n00010000003o0P@0000]0@/01`0001/4`0820UH2V0440000;@4800@0003`0@X0100002d11P040000\nl04;0100000V1Pl05P3ooooo002<0@00QP8009X1002@0P0020000?X200010000003o00@0000]0@X0\n1`000?`2003oool000040000;@4;00`0000T0`@0V0670Xd1RP:H0Hh2V0670P@0000]0@H010000?01\n2P040000;@4800@0003`0@/0200002H63`060?ooool10100000V1Pl05P3ooooo003e0@00I0800?P1\n002f0P0020000?X200010000a/?600@0000]0@X0100002d10P0800009@<20?H1I@;f0K@2100002d1\n1P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo003a0@00F@800002\n001W0P0020000?X200010000a/?600@0000]0@X01`000?`20036`lH000040000;@4;00`0000T0`@0\nmP5J0_h1I@;b0FD2mP5J0P@0000]0@H010000?012P040000;@4800@0003`0@/0200002H63`060?oo\nool10100000V1Pl05P3ooooo002N0@00[`800002002b0P0020000?X200010000a/?600@0000]0@X0\n100002d10P0800009@<20?h1/0:O0K02100002d11P040000l04:00@0000]0@P0200002H63`060?oo\nool10100000V1Pl05P3ooooo002K0@00Z`800:T1002f0P0020000?X200010000a/?600@0000]0@X0\n1`000?`20036`lH000040000;@4;00`0000T0`@0W06`0ZL1[0:W0K@2W06`0P@0000]0@H010000?01\n2P040000;@4800@0003`0@/0200002H63`060?ooool10100000V1Pl05P3ooooo002N0@00K`800?P1\n002b0P0020000?X200010000000000@0000]0@X0100002d10P0<00009@<40?H1/0;<0K02c05`0Yl1\nL0840000;@4600@0003`0@X0100002d1200800009PH?00H0ooooo`40100002d10P040000;@4300L0\n000K1?P24@:o0ZL1100002d120040000;@4600<0000N00L0000F1?L24@:o0ZL11@0000810@0000@0\n000]0@L01@0000T200000Q<0000b2[l2Y`4500@0Y`6o0Q02m`9cM6E`<@8A00T04P0C0180100002d1\n1@0=0000<PZ40_P10@040:L1_`8@0_L2800^00@0000W0Ooo400002H63`0F0?ooool002P0000[0000\nZ`0009L000080000nP8000@000000000100002d12P040000;@4200X0000U0`<0:`2C02/0;P2W02h0\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000A0@00\n:`000941000c000020000?X200040000000000@0000]0@X0100002d10P0800009@<201@1;P2=0Bl0\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo003g0@00\n:`0007d2000c000020000?X200040000000000@0000]0@X0100002d10P0800009@<20?X1;P1i0Rl0\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo003P0P00\n:`000783000c000020000?X200040000000000@0000]0@X0100002d10P0800009@<20><2;P1^0bl0\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nn@000300000i0@0020000?X200040000000000@0000]0@X0100002d10P0800009@<202/0o00/03D1\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nV`400300003N0@0020000?X200040000000000@0000]0@X0100002d10P0800009@<202/0WP4/0=X1\n100002d11P040000l04:00@0000]0@P0200002H63`060?ooool10100000V1Pl05P3ooooo000X0000\nA0800:l0002>0P0020000?X200040000000000@0000]0@X0100002d10P0:00009@<302/0A`8[08X2\nZ`2:0P@0000]0@H010000?012P040000;@4800P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00\n704008L2002D0@00S`8000P0003j0P001000000000040000;@4:00@0000]0@80200002D30P0O0HX2\nT06;0P@0000]0@H010000?012P040000;@4800P0000V1Pl01P3ooooo0@040000;@4400@0003`0@D0\n10000?011`040000l04900L0003l0P00oooo0000100002d11@080000nP80000000000000100002d1\n1`0400009`7oo`P0000V1Pl01P3ooooo0@0400009`7oo`@0000]0@000`0000004@0000`000080000\n2`000100001V0@00304000T0000@0000IP4000`1000:000040000000000000002@000100000m0@00\noP000580001`0@000@0001000007000000000000002l0P00000000420R9C07T0L`1d06D0K@000000\n0000000000000000000000000000000000000000000000000000000000000000000B00020001:000\n:>4K00014`3a0@00000C004800000A<0l6lN01R16`01@040000C02Sc4P000000??DB09D[n7LH=_Qg\nooooodce4P00V_Qg000C0040003RV?Qg0@000;3j60020000dPDQ`oooool001<0884K07P14`000?ag\n000C05Sc6`0001<0@0HC000@0018G1/0X24K04QL6`1h0A<0@5`K013c6`000000000C07P14`3P4000\n001I00004`1HNQ`0N04C0;D2001@]_ag5[GlMbfeo7L00000000000T0002`nQP0Z?@B0:Aj7032`?Ag\n0@0006Af00P000009@0000`0000100004P0000`000010000600000`000000002DP00070100020000\nl?ooo`0000000000000009010000000014004U@0J@1]06D0L`0P04h0I@1g0200DP1_06d0H@1^0000\n000000000000000000000000000000000000000000000000001I00004`1HNQ`0N04C0;D2001@]_ag\n5[GlMbfeo7L00000000000T0002`nQP0Z?@B0:Aj7032`?Ag0@0006Af00P00000C?DB002Jn7L001<0\n0@000>:Hn7L10000/?XH0080003B1B73ooooo`004`0PPA/0N04C0000o7L001<0F?<K00004`101Q<0\n010004QL6`2P8A/0B5`K07P14`10G1/04?<K0000000001<0N04C0>0@000005T0000C05Qj701h0A<0\n]@80052fo7LF]Oag;KGlMbD0002`nQP0R?@B00L7001D`?Ag0@0001T0/0471`ZFI7H02000000U0000\n30000080000Q0000200001h0000H000000000000000m0@00oP000240000800005P0000`000000000\nN000010000000000000001P0000<0000000001T0000<0000oooo02D0000<00001@00P2D0000<0000\n2000P1l0000H0000?P4003d1003n0000oP0000X0000@0000000000000009000040000;`3003k0P00\n2@000100002l0`00n`800580001`0@000`0001@00009000000000000002l0P00000000@20R9C07T0\nL`1d06D0K@0000000000000000000000000000000000000000000000000000000000000000000000\n00000000F@0001<0F7XL07P14`2e0P00D;KlMaJeo7L]]Oag9@000;3j6028m1801`L005C0m7L10000\n6@2`0@L72YITMP08000004ce4P00V_Qg000C0040003RV?Qg0@000;3j60020000dPDQ`oooool001<0\n884K07P14`000?ag000C05Sc6`0001<0@0HC000@0018G1/0X24K04QL6`1h0A<0@5`K013c6`0@8_Ag\na?3mO`3PoGl1000000000000000000004P<00010000B0`Cd0@0006Cd4P0U0000/?XH00T0002`nQP0\nZ?@B02Al7032`?Ag0@0006Af00P000009@0000`0000300004P0000`000010000DP00070100040000\nW?ooo`0000000000000009010000000014004U@0J@1]06D0L`0P04h0I@1g0200DP1_06d0H@1^0000\n00000000000000000000000000000000000000000000000000000000000000004P<00010000B0`Cd\n0@0006Cd4P0U0000/?XH00T0002`nQP0Z?@B02Al7032`?Ag0@0006Af00P00000C?DB002Jn7L001<0\n0@000>:Hn7L10000/?XH0080003B1B73ooooo`004`0PPA/0N04C0000o7L001<0F?<K00004`101Q<0\n010004QL6`2P8A/0B5`K07P14`10G1/04?<K010Rm7O4l?eo0>3mO`400000000000000000000B0`00\n040001831?@10000I?@B02D0002`nQP0R?@B0>@5001D`?Ag0@0001T0/07T1@Z?I7H02000000U0000\n300000@0000H000030000000008:00004000000000000000E00005@00000000000000100000W0000\n0@0003GB345oO`a10000000000010000C00000@00000000000000;`3003k0P00D0000200000]0000\n9`0001P0000500000000002o000000009@0000`0000500009P0001`0000600000000004000000000\n0;l00RD0000<00001P0002/0000H0000R@4002L0002>0@00<00002L0000H00001`000000003oool0\n000002D0000<00001`0002P0000<00001@0002H0000L00001@0000000000000000000000000U0000\n300000D0000X0000300000H0000U0000300000D0080V0000700000H0000000000@0000000000_`02\n9@0000`000060000:`0001P0003b0@009`000?`1000`00009@0000`0000700009@0000`000050000\n:00000`0000600009P0001`00006000000000040000000000;l002D0000<00001P0002D0000<0000\n1@00P5L0000T0000U@000100002M00004@000080002=0Bh0X`4_02D0000<00001@0002P0000<0000\n1P0002D0000<00001`0002H0000L00001P000000000100000000002o000U0000300000H0000U0000\n300000D0081G000090000;@0000@0000_P00014000020000h04^0?X1;`0U0000300000D0000X0000\n300000H0000U0000300000L0000U0000300000P0080W0000600000H00000000000000000000U0000\n300000H0001K0000F00009d0000;0000]00001D000020000300000H000060000X`4O0>017`3P0Cd0\nX`4m0:<17`2S0Al0Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al09@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P000060000000000L71`0000009@0000`000060000\nF`0005P0002O00002`000;@0000E00000P0000`0000600001P000:L17`3P0Al0h04m0:L1?@2W0Al0\nY`4O0:L17`3P0Al0h04m0:L1?@2W0Al0Y`4O02D0000<00001@0002D0000<00001`0002P0000<0000\n1P0002D0000<00002000P2L0000H00001P000000000?3`l0000002D0000<00001P0005/0001H0000\nW`0000/0002d00005@000080000<00001P0000H0002W0Al0h04O0>01?@2W0Cd0Y`4O0:L17`2W0Al0\nh04O0>01?@2W0Cd0Y`4O0:L17`0U0000300000D0000U0000300000L0000X0000300000H0000U0000\n300000P0080W0000600000H0000000005aLG0000000U0000300000H0001K0000F00009l0000;0000\n]00001D000020000300000H000060000Y`4O0>017`3P0Cd0Y`4m0:L17`2W0Al0Y`4O0=/17`3K0Cd0\nY`4m0:L17`2W0Al09@0000`0000500009@0000`000070000:00000`0000600009@0000`000080020\n9`0001P0000600000000020P800000009@0000`000060000F`0005P0002O00002`000;80000E0000\n0P0000`0000600001P000:L17`3K0Al0f`4m0:L1?@2W0Al0Y`4O0:/17`3K0Al0f`4i0:/1>@2[0Al0\nZ`4O02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H0000\n1P000000000Y:BT0000002D0000<00001P0005/0001H0000X00000/0002b000050000080000<0000\n1P0000H0002[0Al0f`4O0=/1>@2[0CT0Z`4O0:/17`2[0Al0f`4O0=/1>@2[0CT0Z`4O0:/17`0U0000\n300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000\n<S8b0000000U0000300000H0001K0000F0000:00000;0000/P0001@000020000300000H000060000\nZ`4O0=/17`3K0CT0Z`4i0:/17`2[0Al0Z`4O0=/17`3K0CT0Z`4i0:/17`2[0Al09@0000`000050000\n9@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000000003`l?0000000\n9@0000`000060000F`0005P0002P00002`000;80000D00000P0000`0000600001P000:/17`3K0Al0\nf`4i0:/1>@2[0Al0Z`4O0:/18`3H0B<0f04i0:/1>@2[0B<0Z`4S02D0000<00001@0002D0000<0000\n1`0002P0000<00001P0002D0000<00002000P2L0000H00001P0000000016ATH0000002D0000<0000\n1P0005/0001H0000X00000`0002a000050000080000<00001P0000H0002[0B<0f04S0=P1>@2[0CT0\nZ`4S0:/18`2^0B<0f04S0=P1>@2^0CT0[P4S0:h18`0U0000300000D0000U0000300000L0000X0000\n300000H0000U0000300000P0080W0000600000H000000000DE5A0000000U0000300000H0001K0000\nF0000:80000<0000/@0001@000020000300000H000060000[P4S0=P18`3H0CT0[P4i0:h18`2^0B<0\n[P4S0=P18`3H0CT0[P4i0:h18`2^0B<09@0000`0000500009@0000`000070000:00000`000060000\n9@0000`0000800209`0001P000060000000005aLG00000009@0000`000060000F`0005P0002R0000\n30000;40000D00000P0000`0000600001P000:h18`3H0B<0f04i0:h1>@2^0B<0[P4S0:h18`3H0B<0\nf04i0:h1>@2^0B<0[P4S02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n2000P2L0000H00001P000000001XJ6P0000002D0000<00001P0005/0001H0000XP0000`0002a0000\n50000080000<00001P0000H0002^0B<0f04S0=P1>@2^0CT0[P4S0:h18`2^0B<0e04S0=@1>@2^0CT0\n[P4S0:h18`0U0000300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000\n600000H000000000M7Ad0000000U0000300000H0001K0000F0000:80000<0000/00001@000020000\n300000H000060000[P4S0=@18`3D0CT0[P4i0:h18`2^0B<0[P4S0=@18`3D0CH0[P4f0:h18`2^0B<0\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n00000861P@0000009@0000`000060000F`0005P0002R000030000;00000C00000P0000`000060000\n1P000:h18`3D0B<0e04f0:h1=P2^0B<0[P4S0;818`3D0B<0e04f0;81=P2b0B<0/P4S02D0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000002?Shl0\n000002D0000<00001P0005/0001H0000X`0000`0002`00004`000080000<00001P0000H0002b0B<0\ne04S0=@1=P2b0CH0/P4S0;818`2b0B<0e04S0=@1=P2b0CH0/P4S0;818`0U0000300000D0000U0000\n300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000W9bL0000000U0000\n300000H0001K0000F0000:<0000<0000/00001<000020000300000H000060000/P4S0=@18`3D0CH0\n/P4f0;818`2b0B<0/P4W0=019`3@0CH0/P4f0;819`2b0BL09@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P00006000000000:VYZ@0000009@0000`000060000\nF`0005P0002S00003P000:h0000C00000P0000`0000600001P000;819`3@0BL0d04f0;81=P2b0BL0\n/P4W0;819`3@0BL0d04f0;81=P2b0BL0/P4W02D0000<00001@0002D0000<00001`0002P0000<0000\n1P0002D0000<00002000P2L0000H00001P000000002d];@0000002D0000<00001P0005/0001H0000\nX`0000h0002^00004`000080000<00001P0000H0002b0BL0d04W0=01=P2b0CH0/P4W0;819`2f0BL0\nd04W0=01=P2f0CH0]P4W0;H19`0U0000300000D0000U0000300000L0000X0000300000H0000U0000\n300000P0080W0000600000H000000000_[jn0000000U0000300000H0001K0000F0000:D0000>0000\n[P0001<000020000300000H000060000]P4W0=019`3@0CH0]P4f0;H19`2f0BL0]P4W0=019`3@0CH0\n]P4f0;H19`2f0BL09@0000`0000500009@0000`000070000:00000`0000600009@0000`000080020\n9`0001P00006000000000<O7a`0000009@0000`000060000F`0005P0002U00003P000:h0000C0000\n0P0000`0000600001P000;H19`3@0BL0d04f0;H1=P2f0BL0]P4W0;H19`3@0BL0d04f0;H1=P2f0BL0\n]P4W02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H0000\n1P000000003?cll0000002D0000<00001P0005/0001H0000Y@0000h0002^00004`000080000<0000\n1P0000H0002f0BL0d04W0=01=P2f0CH0]P4W0;H19`2f0BL0c04W0<`1<P2f0C80]P4W0;H19`0U0000\n300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000\ne]KF0000000U0000300000H0001K0000F0000:D0000>0000[@00018000020000300000H000060000\n]P4W0<`19`3<0C80]P4b0;H19`2f0BL0^P4W0<`19`3<0C80^P4b0;X19`2j0BL09@0000`000050000\n9@0000`000070000:00000`0000600009@0000`0000800209`0001P00006000000000=gMg@000000\n9@0000`000060000F`0005P0002V00003P000:d0000B00000P0000`0000600001P000;X19`3<0BL0\nc04b0;X1<P2j0BL0^P4W0;X1:P3<0BX0c04b0;X1<P2j0BX0^P4Z02D0000<00001@0002D0000<0000\n1`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000003Shn<0000002D0000<0000\n1P0005/0001H0000YP0000l0002]00004P000080000<00001P0000H0002j0BX0c04Z0<`1<P2j0C80\n^P4Z0;X1:P2j0BX0c04Z0<`1<P2j0C80^P4Z0;X1:P0U0000300000D0000U0000300000L0000X0000\n300000H0000U0000300000P0080W0000600000H000000000j>SX0000000U0000300000H0001K0000\nF0000:H0000?0000[@00018000020000300000H000060000^P4Z0<`1:P3<0C80^P4b0;X1:P2j0BX0\n^P4Z0<T1:P390C80^P4b0;X1:P2j0BX09@0000`0000500009@0000`000070000:00000`000060000\n9@0000`0000800209`0001P00006000000000>k^kP0000009@0000`000060000F`0005P0002V0000\n3`000:`0000B00000P0000`0000600001P000;X1:P390BX0b@4b0;X1<P2j0BX0^P4Z0;d1:P390BX0\nb@4b0;d1<P2m0BX0_@4Z02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n2000P2L0000H00001P000000003bl_80000002D0000<00001P0005/0001H0000Y`0000l0002/0000\n4P000080000<00001P0000H0002m0BX0b@4Z0<T1<P2m0C80_@4Z0;d1:P2m0BX0b@4Z0<T1<P2m0C80\n_@4Z0;d1:P0U0000300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000\n600000H000000000moOg0000000U0000300000H0001K0000F0000:L0000?0000[000018000020000\n300000H000060000_@4Z0<T1:P390C80_@4b0;d1:P2m0BX0_@4Z0<T1:P390Bh0_@4^0;d1:P2m0BX0\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n00000?_kn`0000009@0000`000060000F`0005P0002W00003`000:`0000@00000P0000`000060000\n1P000;d1:P390BX0b@4^0;d1;P2m0BX0_@4Z0;d1:P350BX0a@4^0;d1;P2m0BX0_@4Z02D0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000003oool0\n000002D0000<00001P0005/0001H0000Y`0000l0002Z000040000080000<00001P0000H0002m0BX0\na@4Z0<D1;P2m0Bh0_@4Z0;d1:P310BX0a@4Z0<D1;P310Bh0`@4Z0<41:P0U0000300000D0000U0000\n300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000oooo0000000U0000\n300000H0001K0000D0000:T0000?0000ZP000100000200002P0000H000040000`@4Z0<D1:P350Bh0\n`@4^0<41:P310BX0`@4^0<D1;P310Bh0`@4^02D0000<00001@0002D0000<00001`0002P0000<0000\n1P0002D0000<00002000P2L0000H00001P000000003oool0000002D0000<00001P0005/000180000\nZ@000100002Z00004000008000080000100000@000310Bh0a@4^0<41;P310Bh0`@4^0<D1;P310Bh0\n`@4^02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H0000\n1P000000003oool0000002D0000<00001P0005/000140000Z@000100002Z00004000008000070000\n100000<000310Bh0a@4^0<41;P310Bh0`@4^0<41;P310Bh09@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P00006000000000?ooo`0000009@0000`000060000\nF`000400002Y000040000:T0000@00000P0000H0000300000`000<41;P310Bh0`@4^0<41;P310Bh0\n`@4^02D0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00001@00P2H0000L0000\n1P0000000001000000000000008U0000300000H0000[000060000:<1000O0000gP4003/0000U0000\n300000L0000U0000300000D0000X0000300000H0000V0000700000H0000000000@000000001[KF/0\n9@0000`0000600009`0001P000080000000006]]J`0000009@0000`000080000EP0002`0002]0000\n7`000;80000S000010000=/1G`3<0EL0c05S0=/1G`0U0000300000D0000X0000300000H0000U0000\n300000L0000X0000300000P0000V0000700000P0000000000@000000000000009@0000`000080000\n9@0000`000050020E`0002@0002M00008@000:d0000Q00000P000:<1G`3<0El09@0000`000050000\n:00000`0000800009@0000`0000700009@0000`0000500209@0000`000080020:`0001P0001[0@00\nH`0004<2002I00009@0000`0000700009@0000`0000500008@0000P0000N0000600006/1001S0000\nA@8009/0000B000030000040001B0000L04000P0003Ooooo0000000000000000T04000000004@00R\n@@1b06T0H@1/00000000000000000000000000000000000000000000000000000000000000000000\n000000006@301AT000000000000B0`00040001831?@10000I?@B02D0002`nQP0R?@B0<cc4P000000\niCShM`004`10F1h000000:Sc4P281Q<0G?@B09D[n7L001<01000003d4P10]oag60LC005H7P020000\nJ?<B0000000`m180003hMaPfn7Oooooo@?@B089cj7L001<000000<056@2e1S1F`0DI0220mgL1P000\n42;dMlC`oGl0h?eo0@00000000000000000001830000@0004P<4mP40001Tm180l37dMa831?KB1B73\n4P<4mP000004mP00P?@B093Bm7MTMP08000002D0000<0000200005@0002D0000R00002<0003@0000\n<@000040000edPa1Ogl<@F/1001S0000300004`000040000J`4006<000140P00V`0006@0001c06`0\nJ@1T06T0KP1W04d0H@1c07<0<@0A00001`0000L0000B0000200001<0000B000070000140000A0000\n4P000180000U0000300000@0001D0000E0000=00000S0000f@0003D000010000=M8<@Gmo344[0P00\n:0000040001<0000100006/1001S0000A08009/0001@0000800002h0000R000030000?oooolW0000\n600000H0000000000;l00000000U0000300000H0000V0000700000T0000000000@0000000000_`02\n9@0000`000090000:`0001P0001b0P009`0007/2000`00009@0000`000070000:00000`000060000\n9@0000`000050000:00000`0000900009@0000`0000500209P0001`0000900000000004000000000\n0;l00RD0000<00002@0002/0000H0000g`8002L0003X0P00<00002D0000<00001`0002D0000<0000\n1@0002P0000<00002@0002H0000L00002@0000000001000000000000000U0000300000T0000U0000\n300000D0081G0000?0000?80000300004@4000h000080000Q@8H08`2602H0RL0Y`890;H29`350PT0\nd08H0=L2600U0000300000D0000X0000300000T0000U0000300000L0000V0000700000T000000000\n0@000000000000009@0000`0000900009@0000`000050020E`0002@0003b000020000?80000N0000\n0P0008D260250U@09@0000`000050000:00000`0000900009@0000`0000700009P0001`000090000\n0000004000000000000002D0000<00002@0002D0000<00001@00P5L0000T0000lP0001h0003h0000\n7P00008000250U@0U09D02D0000<00001@0002P0000<00002@0002D0000<00001`0002L0000H0000\n2@0000000036`lH0000002D0000<00002@0002H0000L00001P0000000001000000000000008U0000\n300000H0000[0000600009@200190000_`8005d0000U0000300000L0000X0000300000T0000U0000\n300000D0000X0000300000H0000V0000700000H0000000000@000000000000009@0000`000060000\n9@0000`000050020E`0002@0003h00006@000101000J00000P0009@2B03D0TT09@0000`000050000\n:00000`0000600009@0000`0000700009P0001`0000600000000004000000000000002D0000<0000\n1P0002D0000<00001@00P5L0000T0000n0000240000@0@008@000080002D0Ul0e09O02D0000<0000\n1@0002P0000<00001P0002D0000<00001`0002H0000L00001P0000000001000000000000000U0000\n300000H0000U0000300000D0081G0000900000T1000N00004@4001h000020000`@9D0=L2E00U0000\n300000D0000X0000300000H0000U0000300000L0000V0000700000H0000000000@00000000000000\n9@0000`0000600009@0000`000050020E`0002@0000A0@0020000141000N00000P000=L2603H0U@0\n9@0000`000050000:00000`0000600009@0000`0000700009P0001`0000600000000004000000000\n000002D0000<00001P0002D0000<00001@00P5L0000T0000k`000100003b00004@000080001m0Rh0\nQ@8_02D0000<00001@0002P0000<00001P0002D0000<00001`0002H0000L00001P00000000010000\n00000000000U0000300000H0000U0000300000D0081G000090000141000@00005040014000020000\ne`8^0=l2;`0U0000300000D0000X0000300000H0000U0000300000L0000V0000700000H000000000\n0@000000001[KF/09@0000`0000600009`0001P000090000000006]]J`0000009@0000`000090000\nEP0002`000060@004P0000d1000E000010000<`2=P2i0S80^@8m0<`2=P0U0000300000D0000X0000\n300000H0000U0000300000L0000X0000300000T0000V0000700000T0000000000@00000000000000\n9@0000`0000900009@0000`000050020E`0002@0003f00004`0000H1000C00000P000902=P2i0SH0\n9@0000`000050000:00000`0000900009@0000`0000700009@0000`0000500209@0000`000080020\n:`0001P000150P00H`0004<3002I00009@0000`0000700009@0000`0000500008@0000P0000N0000\n600004D2001S0000A@<009/0000B000030000040000U0000300000P0000H000030000000009D0000\nK0000=X0000S0000m@00034000010000=M8<@Gmo34550P00H`0000D0001<0000100004D2001S0000\nA0<009/0001H0000L`1`0780J@1^00004@000180000;00001`0001<0001D0000O0000?D0000S0000\n;`40034000010000=M8<@Gmo346=0P00H`0000P0001<0000100004D2001S0000A0<009/0001L0000\nI`140640K@1`06D0LP0a0180000I00004P0001`0000C00004@0000`0000B00009@0000`000040000\nE00005@0000_0@008`0003T1000e00000@0003GB345oO`a1:0<002P000010000C00000@000150P00\nH`0004@3002K0000D0000200000^00008P0000`0003ooooo9P0001`0000900000000004000000000\n000002D0000<00002@0002D0000<00001@00P5L0000T0000>P4001P0001K0@006000008000150dD0\nV`=502D0000<00001@0002P0000<00002@0002D0000<00001`0002H0000L00002@00000000010000\n00000000000U0000300000T0000U0000300000D0081G000090000581000H0000F`4001l000020000\nV`=508@3E`0U0000300000D0000X0000300000T0000U0000300000L0000V0000700000T000000000\n0@000000000000009@0000`0000900009@0000`000050020E`0002@0001:0@0060000581000O0000\n0P0008@3A@1^0eL09@0000`000050000:00000`0000900009@0000`0000700009P0001`000090000\n0000004000000000000002D0000<00002@0002D0000<00001@00P5L0000T0000@`4001P0001:0@00\n7`000080001^0dD0F`=G02D0000<00001@0002P0000<00002@0002D0000<00001`0002H0000L0000\n2@0000000001000000000000000U0000300000T0000U0000300000D0081G0000900003X1000H0000\n@`4001l000020000F`=504D3E`0U0000300000D0000X0000300000T0000U0000300000L0000V0000\n700000T0000000000@000000000000009@0000`0000900009@0000`000050020E`0002@0001:0@00\n4P0004X1000H00000P0006h3A@1^0c809@0000`000050000:00000`0000900009@0000`000070000\n9@0000`0000500209P0001`00009000000000040000000000;l00RD0000<00002@0002/0000H0000\nJP<002L0001c0`00<00002D0000<00001`0002D0000<00001@0002P0000<00002@0002D0000<0000\n1@00P2D0000<00002000P2/0000H0000A@<006<0002k0`00V@0002D0000<00001`0002D0000<0000\n1@000240000800007P0001P000150`00H`000;h3002K00004P0000`0000100009@0000`000080000\n600000`000000002E0000700000j0@008`0005/1000a00000@0003GB345oO`a1A@<006<000060000\nC00000@000150`00H`000;d3002K0000F00006H0J@1h06D0I00a00X00007000040000180000C0000\n4P0002D0000<0000100005@0001D0000F`4002<0001U0@00=@000040000edPa1Ogl<@Id3000X0000\n0@0004`000040000A@<006<0002m0`00V`000500000P0000;P000280000<0000ooooobH0000L0000\n2@0000000001000000000000000U0000300000T0000W0000600000H000000000oooo0000000U0000\n300000H0001F0000N00004H000080000E@0001L0000G0000h`0^0><0<P3P03T0g00m0=P0@@3A0440\nc@110<D0@@3203d0_P0i0;X0<P2j02h0^P0W0;h08`3201l0a@0K0<d0603A01P0f00K0=`07`3P02<0\nh`0W0><0;P0U0000300000D0000X0000300000T0000U0000300000L0000X0000300000H0000V0000\n700000H0000000000@000000000000009@0000`0000600009`0001P0000900000000000000000000\n9@0000`000090000EP0004P0001<00003P000500000B00002`000=@0;P3D0380d@0b0<d0<P3902h0\nb@0Z0<d0:P3=02L0d@0Z0=@0:P3D02h09@0000`000050000:00000`0000600009@0000`000070000\n:00000`0000900009@0000`0000800209`0001P00009000000000000000000009@0000`000090000\nF`0005P0001200003P0004H0000B00000P0000`0000600001P000:l09`2j02L0^P0b0:l0<P2_02L0\n[`0W0:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z02D0000<00001@0002D0000<00001`0002P0000<0000\n2@0002D0000<00002000P2L0000H00002@000000000Q8B40000002D0000<00002@0005/0001H0000\n@P0000l0001600004P000080000<00001P0000H0002_02X0^P0Z0;X0<P2_0380[`0Z0:l0:P2_02X0\n^P0Z0;X0<P2_0380[`0Z0:l0:P0U0000300000D0000U0000300000L0000X0000300000T0000U0000\n300000P0080W0000600000T000000000AdI70000000U0000300000T0001K0000F0000480000?0000\nAP00018000020000300000H000060000[`0Z0;X0:P2j0380[`0b0:l0:P2_02X0[`0Z0;X0:P2j0380\n[`0b0:l0:P2_02X09@0000`0000500009@0000`000070000:00000`0000900009@0000`000080020\n9`0001P000090000000007IdMP0000009@0000`000090000F`0005P0001200003`0004H0000B0000\n0P0000`0000600001P000:l0:P2j02X0^P0b0:l0<P2_02X0[`0Z0:l0:P2j02X0^P0^0:l0;P2_02X0\n[`0Z02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H0000\n2@000000002MVYd0000002D0000<00002@0005/0001@0000@P0000l00016000040000080000:0000\n1P0000@0002_02X0^P0Z0;X0;P2_02h0[`0Z0:l0:P2_02h0^P0^0:l0;P2_02h09@0000`000050000\n9@0000`000070000:00000`0000900009@0000`0000800209`0001P00009000000000;Fb]@000000\n9@0000`000090000F`0004P000120000400004H0000@00000P0000P00004000010000:l0;P2j02h0\n[`0^0:l0;P2_02h0^P0^0:l0;P2_02h09@0000`0000500009@0000`000070000:00000`000090000\n9@0000`0000800209`0001P00009000000000<K3aP0000009@0000`000090000F`0004P000120000\n400004H0000@00000P0000P00004000010000:l0;P2j02h0[`0^0:l0;P2_02h0^P0^0:l0;P2_02h0\n9@0000`0000500009@0000`000070000:00000`0000900009@0000`0000500209P0001`000090000\n0000004000000000003o0RD0000<00002@0002/0000H0000[`0002L0002h0000<00002D0000<0000\n1`0002D0000<00001@0002P0000<00002@0002H0000L00002@0000000001000000000000000U0000\n300000T0000W0000600000H000000000Ngek0000000U0000300000H0001F0000A00004@0000H0000\nI@0001/0000:0000]P1<0<80A@3=04`0f0150><0C03_04D0mP1<0041A@0=0D`0]P1<02D0000<0000\n1@0002P0000<00002@0002D0000<00001`0002P0000<00001P0002L0000H00001P000000001kOG/0\n000002D0000<00001P0002H0000L00002@0000000001000000000000008U0000300000T0000[0000\n60000;L0001<00005P4005H0000U0000300000L0000X0000300000H0000U0000300000D0000X0000\n300000T0000W0000600000T000000000Ngek0000000U0000300000T0000V0000700000H000000000\n0@000000000000029@0000`000060000:`0001P0000A0@00<P0001H1001:00009@0000`000070000\n:00000`0000900009@0000`000050000:00000`0000600009`0001P000060000000006]]J`000000\n9@0000`0000600009P0001`000090000000000400000000000000RD0000<00002@0002/0000H0000\nY00002L0002]0000<00002D0000<00001`0002P0000<00001P0002D0000<00001@0002P0000<0000\n2@0002D0000<00001@00P2H0000L00002@000000000100000000002o008U0000300000T0000[0000\n60000141000W00006P400300000U0000300000L0000U0000300000D0000X0000300000T0000U0000\n300000D0080U0000300000P0080[0000600007H0001S0000K04009T0000U0000300000L0000U0000\n300000D0000Q0000200001h0000H0000MP0006<0001_0@00V`000180000<00000@0002D0000<0000\n200001P0000<000000000U@0002L0000;00002<000200000<@000040000edPa1Ogl<@GH0001S0000\n3@0004`000040000MP0006<0001^0@00V`0006P0001Y06@0I@1Q06`0A`1U0640LP1B0380E00a0000\n1`0001<0000B00004P0000L0000K00004P000180000<000060000180000E00004P0002D0000<0000\n100005@0001D0000P@0002<000280000=@000040000edPa1Ogl<@EL1000X00000@0004`000040000\nMP0006<0001^0@00V`000500000P0000;P000280000<0000ooooobL0000H00002@000000001[KF/0\n000002D0000<00002@0002H0000L00001P0000000001000000000000008U0000300000H0000[0000\n600002@0003i0000;@000081000U0000300000L0000X0000300000T0000U0000300000D0000X0000\n300000H0000W0000600000H000000000oooo0000000U0000300000H0000V0000700000T000000000\n0@000000000000029@0000`000090000:`0001P0000T0000S00002d0002E00009@0000`000070000\n:00000`0000600009@0000`000050000:00000`0000900009P0001`0000900000000004000000000\n000002D0000<00002@0002D0000<00001@00P5L0000T00005@000580000E0000EP000080000g0?D0\n=`3Z02D0000<00001@0002P0000<00002@0002D0000<00001`0002H0000L00002@00000000010000\n00000000000U0000300000T0000U0000300000D0081G0000900001D0000h00005@0003X000020000\n=`2V03L0W`0U0000300000D0000X0000300000T0000U0000300000L0000V0000700000T000000000\n0@000000000000009@0000`0000900009@0000`000050020E`0002@0000E0000D`0001h0001D0000\n0P0003L0k@1A0>h09@0000`000050000:00000`0000900009@0000`0000700009P0001`000090000\n0000004000000000000002D0000<00002@0002D0000<00001@00P5L0000T00005@0003T0000N0000\n>@000080000g0:80D@2S02D0000<00001@0002P0000<00002@0002D0000<00001`0002H0000L0000\n2@0000000001000000000000000U0000300000T0000U0000300000D0081G0000900000/0001B0000\n2`0005H000020000703e01d0jP0U0000300000D0000X0000300000T0000U0000300000L0000V0000\n700000T0000000000@000000000000009@0000`0000900009@0000`000050020E`0002@0000;0000\n>00000/0000j00000P0001`0YP0M09l09@0000`000050000:00000`0000900009@0000`000070000\n9P0001`0000900000000004000000000000002D0000<00002@0002D0000<00001@00P5L0000T0000\n1P0005<0000;0000E0000080000A0>d0703^02D0000<00001@0002P0000<00002@0002D0000<0000\n1`0002H0000L00002@0000000001000000000000000U0000300000T0000U0000300000D0081G0000\n900000H0000i00002`0003T0000200004@2R01`0X`0U0000300000D0000X0000300000T0000U0000\n300000L0000V0000700000T0000000000@000000000000009@0000`0000900009@0000`000050020\nE`0002@0000N0000>@0001h0001C00000P000540k@1A0:809@0000`000050000:00000`000090000\n9@0000`0000700009@0000`0000800209`0001P00009000000000000000000009@0000`000090000\nF`0005P000060000?@0001X0001?00000P0000`0000600001P000140hP0A0:h0AP2^04H0hP0A0>80\n4@3R0140hP0A0:h0AP2^04H0hP0A0>804@3R02D0000<00001@0002D0000<00001`0002P0000<0000\n2@0002D0000<00002000P2L0000H00002@00000000061PH0000002D0000<00002@0005/0001H0000\n1P0003d0000J0000C`000080000<00001P0000H0000A0>804@2^04H0[P160>804@3R0140hP0A0>80\n4@2^04H0[P160>804@3R0140hP0U0000300000D0000U0000300000L0000X0000300000T0000U0000\n300000P0080W0000600000T0000000003@d=0000000U0000300000T0001K0000F00000H0000m0000\n6P0004l000020000300000H0000600004@3R0140[P160:h0AP3R0140hP0A0>804@3R0140[P120:h0\n@P3R0140hP0A0>809@0000`0000500009@0000`000070000:00000`0000900009@0000`000080020\n9`0001P000090000000001@D500000009@0000`000090000F`0005P000060000?@0001T0001?0000\n0P0000`0000600001P000140hP0A0:h0@P2^0480hP0A0>804@3R01D0hP0E0:h0@P2^0480hP0E0>80\n5@3R02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H0000\n2@000000000K6a/0000002D0000<00002@0005/0001H0000200003d0000I0000C`000080000<0000\n1P0000H0000E0>805@2^0480[P120>805@3R01D0hP0E0>805@2^0480[P120>805@3R01D0hP0U0000\n300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000600000T000000000\n8b<S0000000U0000300000T0001K0000F00000P0000m00006@0004l000020000300000H000060000\n5@3R01D0[P120:h0@P3R01D0hP0E0>805@3R01D0[P120:h0@P3R01D0hP0E0>809@0000`000050000\n9@0000`000070000:00000`0000900009@0000`0000800209`0001P000090000000002/[:`000000\n9@0000`000090000F`0005P000080000?@0001T0001?00000P0000`0000600001P0001D0hP0E0:h0\n@P2^0480hP0E0>805@3R01D0hP0E0:h0?P2^03h0hP0E0>805@3R02D0000<00001@0002D0000<0000\n1`0002P0000<00002@0002D0000<00002000P2L0000H00002@000000000d<c@0000002D0000<0000\n2@0005/0001H0000200003d0000G0000C`000080000<00001P0000H0000E0>805@2^03h0[P0n0>80\n5@3R01D0hP0I0>806@2^03h0[P0n0>806@3R01T0hP0U0000300000D0000U0000300000L0000X0000\n300000T0000U0000300000P0080W0000600000T000000000?C`m0000000U0000300000T0001K0000\nF00000T0000m00005`0004l000020000300000H0000600006@3R01T0[P0n0:h0?P3R01T0hP0I0>80\n6@3R01T0[P0n0:h0?P3R01T0hP0I0>809@0000`0000500009@0000`000070000:00000`000090000\n9@0000`0000800209`0001P000090000000004I5AP0000009@0000`000090000F`0005P000090000\n?@0001L0001?00000P0000`0000600001P0001T0hP0I0:h0?P2^03h0hP0I0>806@3R01T0hP0I0:h0\n?P2^03h0hP0I0>806@3R02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<0000\n2000P2L0000H00002@000000001@Ce00000002D0000<00002@0005/0001H00002@0003d0000G0000\nC`000080000<00001P0000H0000I0>806@2^03h0[P0n0>806@3R01T0hP0I0>806@2^03X0[P0j0>80\n6@3R01T0hP0U0000300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000\n600000T000000000FeUK0000000U0000300000T0001K0000F00000T0000m00005P0004l000020000\n300000H0000600006@3R01T0[P0j0:h0>P3R01T0hP0I0>80703R01`0[P0j0:h0>P3R01`0hP0L0>80\n9@0000`0000500009@0000`000070000:00000`0000900009@0000`0000800209`0001P000090000\n000006ITIP0000009@0000`000090000F`0005P0000;0000?@0001H0001?00000P0000`000060000\n1P0001`0hP0L0:h0>P2^03X0hP0L0>80703R01`0hP0L0:h0>P2^03X0hP0L0>80703R02D0000<0000\n1@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H00002@000000001aL740\n000002D0000<00002@0005/0001H00002`0003d0000F0000C`000080000<00001P0000H0000L0>80\n702^03X0[P0j0>80703R01`0hP0L0>80702^03X0[P0j0>80703R01`0hP0U0000300000D0000U0000\n300000L0000X0000300000T0000U0000300000P0080W0000600000T000000000OG]m0000000U0000\n300000T0001K0000F00000/0000m00005P0004l000020000300000H000060000703R01`0[P0j0:h0\n>P3R01`0hP0L0>80703R01`0[P0g0:h0=`3R01`0hP0L0>809@0000`0000500009@0000`000070000\n:00000`0000900009@0000`0000800209`0001P000090000000008N5Q`0000009@0000`000090000\nF`0005P0000;0000?@0001D0001?00000P0000`0000600001P0001`0hP0L0:h0=`2^03L0hP0L0>80\n703R01`0hP0L0:h0=`2^03L0hP0L0>80703R02D0000<00001@0002D0000<00001`0002P0000<0000\n2@0002D0000<00002000P2L0000H00002@000000002@SY00000002D0000<00002@0005/0001H0000\n2`0003d0000E0000C`000080000<00001P0000H0000L0>80702^03L0[P0g0>80703R01`0hP0P0>80\n802^03L0[P0g0>80803R0200hP0U0000300000D0000U0000300000L0000X0000300000T0000U0000\n300000P0080W0000600000T000000000V9JH0000000U0000300000T0001K0000F00000`0000m0000\n5@0004l000020000300000H000060000803R0200[P0g0:h0=`3R0200hP0P0>80803R0200[P0g0:h0\n=`3R0200hP0P0>809@0000`0000500009@0000`000070000:00000`0000900009@0000`000080020\n9`0001P000090000000009nLW`0000009@0000`000090000F`0005P0000<0000?@0001D0001?0000\n0P0000`0000600001P000200hP0P0:h0=`2^03L0hP0P0>80803R0200hP0P0:h0<`2^03<0hP0P0>80\n803R02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H0000\n2@000000002UXjD0000002D0000<00002@0005/0001H0000300003d0000C0000C`000080000<0000\n1P0000H0000P0>80802^03<0[P0c0>80803R0200hP0P0>80802^03<0[P0c0>80803R0200hP0U0000\n300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000600000T000000000\nZjR[0000000U0000300000T0001K0000F00000`0000m00004`0004l000020000300000H000060000\n803R0200[P0c0:h0<`3R0200hP0P0>80903R02@0[P0c0:h0<`3R02@0hP0T0>809@0000`000050000\n9@0000`000070000:00000`0000900009@0000`0000800209`0001P00009000000000;2]/0000000\n9@0000`000090000F`0005P0000>0000?@0001<0001?00000P0000`0000600001P0002@0hP0T0:h0\n<`2^03<0hP0T0>80903R02@0hP0T0:h0<`2^03<0hP0T0>80903R02D0000<00001@0002D0000<0000\n1`0002P0000<00002@0002D0000<00002000P2L0000H00002@000000002e/[D0000002D0000<0000\n2@0005/0001H00003P0003d0000C0000C`000080000<00001P0000H0000T0>80902^03<0[P0c0>80\n903R02@0hP0T0>80902^02l0[P0_0>80903R02@0hP0U0000300000D0000U0000300000L0000X0000\n300000T0000U0000300000P0080W0000600000T000000000^KJi0000000U0000300000T0001K0000\nF00000h0000m00004P0004l000020000300000H000060000903R02@0[P0_0:h0;`3R02@0hP0T0>80\n903R02@0[P0_0:h0;`3R02@0hP0T0>809@0000`0000500009@0000`000070000:00000`000090000\n9@0000`0000800209`0001P00009000000000;fj_@0000009@0000`000090000F`0005P0000>0000\n?@000180001?00000P0000`0000600001P0002@0hP0T0:h0;`2^02l0hP0T0>80903R02P0hP0X0:h0\n;`2^02l0hP0X0>80:03R02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<0000\n2000P2L0000H00002@0000000031_/40000002D0000<00002@0005/0001H00003`0003d0000B0000\nC`000080000<00001P0000H0000X0>80:02^02l0[P0_0>80:03R02P0hP0X0>80:02^02l0[P0_0>80\n:03R02P0hP0U0000300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000\n600000T000000000a<740000000U0000300000T0001K0000F00000l0000m00004P0004l000020000\n300000H000060000:03R02P0[P0_0:h0;`3R02P0hP0X0>80:03R02P0[P0_0:h0;`3R02P0hP0X0>80\n9@0000`0000500009@0000`000070000:00000`0000900009@0000`0000800209`0001P000090000\n00000<K3aP0000009@0000`000090000F`0005P0000?0000?@000180001?00000P0000`000060000\n1P0002P0hP0X0:h0;`2^02l0hP0X0>80:03R02P0hP0X0:h0:`2^02/0hP0X0>80:03R02D0000<0000\n1@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H00002@0000000036`lH0\n000002D0000<00002@0005/0001@00003`0003d0000@0000C`000080000:00001P0000@0000X0>80\n:02^02/0[P0[0>80:03R02P0hP0[0>80:`2^02/0hP0[0>809@0000`0000500009@0000`000070000\n:00000`0000900009@0000`0000800209`0001P00009000000000<K3aP0000009@0000`000090000\nF`0004P0000@0000?@000100001?00000P0000P000040000100002/0hP0[0:h0:`3R02/0hP0[0>80\n:`2^02/0hP0[0>809@0000`0000500009@0000`000070000:00000`0000900009@0000`000080020\n9`0001P00009000000000<K3aP0000009@0000`000090000F`0004P0000@0000?@000100001?0000\n0P0000P000040000100002/0hP0[0:h0:`3R02/0hP0[0>80:`2^02/0hP0[0>809@0000`000050000\n9@0000`000070000:00000`0000900009@0000`0000500209P0001`0000900000000004000000000\n00000RD0000<00002@0002/0000H00004@000:h000140000h00002D0000<00001`0002D0000<0000\n1@0002P0000<00002@0002D0000<00002000P2L0000H00002@00000000000000000002D0000<0000\n2@0005/0001H00003P0004l0000B0000E`000080000<00001P0000H0000T0?T0903R02l0hP0_0?T0\n903i02@0n@0X0?T0:03R02l0hP0_0?T0:03i02P0n@0U0000300000D0000U0000300000L0000X0000\n300000T0000U0000300000P0080W0000600000T0000000008B4Q0000000U0000300000T0001K0000\nF00000l0001?00004P0005L000020000300000H000060000:03i02P0hP0_0>80;`3i02P0n@0X0?T0\n:03i02P0hP0_0>80;`3i02P0n@0X0?T09@0000`0000500009@0000`000070000:00000`000090000\n9@0000`0000800209`0001P000090000000004M6A`0000009@0000`000090000F`0005P0000?0000\nC`000180001G00000P0000`0000600001P0002P0n@0X0>80;`3R02l0n@0X0?T0:03i02P0n@0X0>80\n;`3R02l0n@0X0?T0:03i02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<0000\n2000P2L0000H00002@000000001fM7H0000002D0000<00002@0005/0001H00003`0004l0000B0000\nE`000080000<00001P0000H0000X0?T0:03R02l0hP0_0?T0:03i02P0n@0X0?T0:03R02/0hP0[0?T0\n:03i02P0n@0U0000300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000\n600000T000000000WIZM0000000U0000300000T0001K0000D00000l0001?0000400005L000020000\n2P0000H000040000:03i02P0hP0[0>80:`3i02P0n@0X0?T0:`3i02/0hP0[0?T0:`3i02D0000<0000\n1@0002D0000<00001`0002P0000<00002@0002D0000<00002000P2L0000H00002@000000002e/[D0\n000002D0000<00002@0005/000180000400004l0000@0000E`00008000080000100000@0000[0?T0\n:`3R02/0n@0[0?T0:`3i02/0hP0[0?T0:`3i02D0000<00001@0002D0000<00001`0002P0000<0000\n2@0002D0000<00002000P2L0000H00002@0000000036`lH0000002D0000<00002@0005/000180000\n400004l0000@0000E`00008000080000100000@0000[0?T0:`3R02/0n@0[0?T0:`3i02/0hP0[0?T0\n:`3i02D0000<00001@0002D0000<00001`0002P0000<00002@0002D0000<00001@00P2H0000L0000\n2@0000000001000000000000008U0000300000T0000[0000600002@0003R0000;@000?L0000U0000\n300000L0000U0000300000D0000X0000300000T0000U0000300000P0080W0000600000T000000000\n00000000000U0000300000T0001K0000F00000h0000e00004P0003d000020000300000H000060000\n902^02@0U`0_09L0;`2^02@0[P0T0:h0:02^02P0U`0_09L0;`2^02P0[P0X0:h09@0000`000050000\n9@0000`000070000:00000`0000900009@0000`0000800209`0001P0000900000000024Q8@000000\n9@0000`000090000F`0005P0000?0000=@000180000m00000P0000`0000600001P0002P0[P0X09L0\n;`2G02l0[P0X0:h0:02^02P0[P0X09L0;`2G02l0[P0X0:h0:02^02D0000<00001@0002D0000<0000\n1`0002P0000<00002@0002D0000<00002000P2L0000H00002@0000000017ATL0000002D0000<0000\n2@0005/0001H00003`0003D0000B0000?@000080000<00001P0000H0000X0:h0:02G02l0U`0_0:h0\n:02^02P0[P0X0:h0:02G02l0U`0_0:h0:02^02P0[P0U0000300000D0000U0000300000L0000X0000\n300000T0000U0000300000P0080W0000600000T000000000MWAf0000000U0000300000T0001K0000\nF00000l0000e00004P0003d000020000300000H000060000:02^02P0U`0_09L0;`2^02P0[P0X0:h0\n:02^02P0U`0[09L0:`2^02P0[P0X0:h09@0000`0000500009@0000`000070000:00000`000090000\n9@0000`0000800209`0001P000090000000009fJW@0000009@0000`000090000F`000500000?0000\n=@000100000m00000P0000X000060000100002P0[P0X09L0:`2G02/0[P0X0:h0:02^02/0[P0[09L0\n:`2^02/0[P0U0000300000D0000U0000300000L0000X0000300000T0000U0000300000P0080W0000\n600000T000000000]K:e0000000U0000300000T0001K0000B0000100000e0000400003d000020000\n200000@000040000:`2^02/0U`0[0:h0:`2^02/0[P0[09L0:`2^02/0[P0U0000300000D0000U0000\n300000L0000X0000300000T0000U0000300000P0080W0000600000T000000000a/?60000000U0000\n300000T0001K0000B0000100000e0000400003d000020000200000@000040000:`2^02/0U`0[0:h0\n:`2^02/0[P0[09L0:`2^02/0[P0U0000300000D0000U0000300000L0000X0000300000T0000U0000\n300000D0080V0000700000T0000000000@000000000000029@0000`000090000:`0001P0000T0000\nU`0002d0002/00009@0000`0000700009@0000`000050000:00000`0000900009@0000`000050020\n9@0000`000080020:`0001P0001_0000a0000?40003k00009@0000`0000700009@0000`000050000\n8@0000P0000N0000600006l000340000l`000?`0000B000030000040000U0000300000P0000H0000\n30000000009D0000O00002X000150000DP0005<000010000=M8<@Gmo345_0000a00000P0001<0000\n100006l000340000lP000?`0001L0000J@1^06D0LP1d06T0H@0a00L0000C00004P0000/000090000\n20000180000B00009@0000`000040000E00005@0001B0000A@0005X0001G00000@0003GB345oO`a1\nf`0008T000010000C00000@0001_0000a0000?80003l0000D0000200000^00008P0000`0003ooooo\n9`0001P000090000000006]]J`0000009@0000`0000900009P0001`0000600000000004000000000\n00000RD0000<00001P0002/0000H0000900009X1000]0000X`4002D0000<00001`0002P0000<0000\n2@0002D0000<00001@0002P0000<00001P0002L0000H00001P000000003oool0000002D0000<0000\n1P0002H0000L00002@0000000001000000000000008U0000300000T0000[0000600002@0000]0@00\n;@0003L1000U0000300000L0000X0000300000H0000U0000300000D0000X0000300000T0000V0000\n700000T0000000000@000000000000009@0000`0000900009@0000`000050020E`0003`000020000\nL00000h0002=0000200001D0TP4E08/190200@H0L@4T06811P1C0AD0A`4E04019@0000`000050000\n:00000`0000900009@0000`0000700009P0001`0000900000000004000000000000002D0000<0000\n2@0002D0000<00001@00P5L0000T0000200008d0000N0000SP000080000E0981D@2C0BD0000<0000\n1@0002P0000<00002@0002D0000<00001`0002H0000L00002@0000000001000000000000000U0000\n300000T0000U0000300000D0081G0000900001h0002800007P0008d000020000D@2B0E40P`4U0000\n300000D0000X0000300000T0000U0000300000L0000W0000600000T000000000a/?60000000U0000\n300000T0000V0000700000H0000000000@000000000000029@0000`000060000:`0001P000120000\nE`4005h000220@009@0000`000070000:00000`0000900009@0000`000050000:00000`000060000\n9P0001`0000600000000004000000000000002D0000<00001P0002D0000<00001@00P5L0000T0000\n6@000780000I0000R0000080001208<1@P140BD0000<00001@0002P0000<00001P0002D0000<0000\n1`0002H0000L00001P0000000001000000000000000U0000300000H0000U0000300000D0081G0000\n900002@0001b0000900008P000020000H0230F00A04U0000300000D0000X0000300000H0000U0000\n300000L0000V0000700000H0000000000@000000000000009@0000`0000600009@0000`000050020\nE`0002@0000N0000L00001h0001h00000P000540EP5A04019@0000`000050000:00000`000060000\n9@0000`0000700009P0001`0000600000000004000000000000002D0000<00001P0002D0000<0000\n1@00P5L0000T000020000700000N0000L0000080000E0401D@100BD0000<00001@0002P0000<0000\n1P0002D0000<00001`0002H0000L00001P0000000001000000000000000U0000300000H0000U0000\n300000D0081G000090000100002=00004@00090000020000:`2J0B`0TP4U0000300000D0000X0000\n300000H0000U0000300000L0000V0000700000H0000000000@000000000000009@0000`000060000\n9@0000`000050020E`0002@0000@0000KP000140001`00000P0002/0@04/03P19@0000`000050000\n:00000`0000600009@0000`0000700009@0000`0000500209@0000`000080020:`0001P0001j0000\nFP4007P1002A0@009@0000`0000700009@0000`0000500008@0000P0000N0000600007X0001J0@00\nN`400981000B000030000040000U0000300000P0000H000030000000009D0000W00002h0001j0000\nP`0008P000010000=M8<@Gmo345j0000FP4000d0001<0000100007X0001J0@00NP400981001X0000\nL`1`0780J@1^06L0A01Q06d0L01U0780<P000140000B00002`0000L0000C00004P0001T0000B0000\n700001<0000A000030000180000U0000300000@0001D0000E00008<0001j0000S@0008`000010000\n=M8<@Gmo345M0@007`400040001<0000100007X0001J0@00NP400981001@0000800002h0000R0000\n30000?oooolW0000600000H000000000Jfe[0000000U0000300000H0000V0000700000T000000000\n0@000000000000029@0000`000090000:`0001P0000T0000@`8002d0001<0P009@0000`000070000\n:00000`0000600009@0000`000050000:00000`0000900009`0001P00009000000000?ooo`000000\n9@0000`0000900009P0001`000060000000000400000000000000RD0000<00001P0002/0000H0000\n90000=H1000]0000g`4002D0000<00001`0002P0000<00002@0002D0000<00001@0002P0000<0000\n1P0002H0000L00001P0000000001000000000000000U0000300000H0000U0000300000D0081G0000\n900001D0003500005@000<T000020000=`0l0SL0<08U0000300000D0000X0000300000H0000U0000\n300000L0000V0000700000H0000000000@000000000000009@0000`0000600009@0000`000050020\nE`0002@0000E0000ZP0001D0002^00000P0003L0l04g0>D19@0000`000050000:00000`000060000\n9@0000`0000700009P0001`0000600000000004000000000000002D0000<00001P0002D0000<0000\n1@00P5L0000T00005@000<P0000N0000b0000080000g03P2D@0h0RD0000<00001@0002P0000<0000\n1P0002D0000<00001`0002H0000L00001P0000000001000000000000000U0000300000H0000U0000\n300000D0081G0000900001D0002]00007P000:d000020000=`3/0E40k@4U0000300000D0000X0000\n300000H0000U0000300000L0000V0000700000H0000000000@000000000000009@0000`000060000\n9@0000`000050020E`0002@0000;0000a@0000/0003900000P0001`0?08M03029@0000`000050000\n:00000`0000600009@0000`0000700009P0001`0000600000000004000000000000002D0000<0000\n1P0002D0000<00001@00P5L0000T00002`000:X0000;0000[P000080000L0?017@3U0BD0000<0000\n1@0002P0000<00001P0002D0000<00001`0002H0000L00001P0000000001000000000000000U0000\n300000H0000U0000300000D0081G0000900000H0003800002`000<P0000200004@0h0Q`0>08U0000\n300000D0000X0000300000H0000U0000300000L0000V0000700000H0000000000@00000000000000\n9@0000`0000600009@0000`000050020E`0002@000060000[@0000/0002]00000P000140k04L0>d1\n9@0000`000050000:00000`0000600009@0000`0000700009P0001`0000600000000004000000000\n000002D0000<00001P0002D0000<00001@00P5L0000T00007P000:d0000N0000b0000080001A03P2\nD@3/0BD0000<00001@0002P0000<00001P0002D0000<00001`0002D0000<00002000P2L0000H0000\n1P00000000000000000002D0000<00001P0005/0001H00001P000;40000J0000a0000080000<0000\n1P0000H0000A02d24@3h0DH0n05602d24@0]0Q40;@8A02d24@3h0DH0n05602d24@0]0Q40;@8U0000\n300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000\n1PH60000000U0000300000H0001K0000F00000H0002a00006P000<@000020000300000H000060000\n4@0]0Q40n0560?P1AP0]0Q40;@8A02d24@0]0Q40n0560?P1AP0]0Q40;@8A02d29@0000`000050000\n9@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000000000d=3@000000\n9@0000`000060000F`0005P000060000/@0001X0003400000P0000`0000600001P000140;@8A0?P1\nAP3h0DH0;@8A02d24@0]0Q40;@8A0?P1@P3h0D80;@8A02d24@0]0RD0000<00001@0002D0000<0000\n1`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000000D51@0000002D0000<0000\n1P0005/0001H00001P000;40000I0000a0000080000<00001P0000H0000A02d24@3h0D80n05202d2\n4@0]0Q40;@8E02d25@3h0D80n05202d25@0]0QD0;@8U0000300000D0000U0000300000L0000X0000\n300000H0000U0000300000P0080W0000600000H0000000006a/K0000000U0000300000H0001K0000\nF00000P0002a00006@000<@000020000300000H0000600005@0]0QD0n0520?P1@P0]0QD0;@8E02d2\n5@0]0QD0n0520?P1@P0]0QD0;@8E02d29@0000`0000500009@0000`000070000:00000`000060000\n9@0000`0000800209`0001P000060000000002<S8`0000009@0000`000060000F`0005P000080000\n/@0001T0003400000P0000`0000600001P0001D0;@8E0?P1@P3h0D80;@8E02d25@0]0QD0;@8E0?P1\n@P3h0D80;@8E02d25@0]0RD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n2000P2L0000H00001P000000000[:b/0000002D0000<00001P0005/0001H000020000;40000I0000\na0000080000<00001P0000H0000E02d25@3h0D80n05202d25@0]0QD0;@8E02d25@3h0Ch0n04n02d2\n5@0]0QD0;@8U0000300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000\n600000H000000000=3<d0000000U0000300000H0001K0000F00000P0002a00005`000<@000020000\n300000H0000600005@0]0QD0n04n0?P1?P0]0QD0;@8E02d26@0]0QT0n04n0?P1?P0]0QT0;@8I02d2\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n000003dl?@0000009@0000`000060000F`0005P000090000/@0001L0003400000P0000`000060000\n1P0001T0;@8I0?P1?P3h0Ch0;@8I02d26@0]0QT0;@8I0?P1?P3h0Ch0;@8I02d26@0]0RD0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P0000000016ADH0\n000002D0000<00001P0005/0001H00002@000;40000G0000a0000080000<00001P0000H0000I02d2\n6@3h0Ch0n04n02d26@0]0QT0;@8I02d26@3h0Ch0n04n02d26@0]0QT0;@8U0000300000D0000U0000\n300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000D4m@0000000U0000\n300000H0001K0000F00000T0002a00005`000<@000020000300000H0000600006@0]0QT0n04n0?P1\n?P0]0QT0;@8I02d26@0]0QT0n04j0?P1>P0]0QT0;@8I02d29@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P000060000000005]IF`0000009@0000`000060000\nF`0005P000090000/@0001H0003400000P0000`0000600001P0001T0;@8I0?P1>P3h0CX0;@8I02d2\n6@0]0Q`0;@8L0?P1>P3h0CX0;@8L02d2700]0RD0000<00001@0002D0000<00001`0002P0000<0000\n1P0002D0000<00002000P2L0000H00001P000000001VI6H0000002D0000<00001P0005/0001H0000\n2`000;40000F0000a0000080000<00001P0000H0000L02d2703h0CX0n04j02d2700]0Q`0;@8L02d2\n703h0CX0n04j02d2700]0Q`0;@8U0000300000D0000U0000300000L0000X0000300000H0000U0000\n300000P0080W0000600000H000000000LG1a0000000U0000300000H0001K0000F00000/0002a0000\n5P000<@000020000300000H000060000700]0Q`0n04j0?P1>P0]0Q`0;@8L02d2700]0Q`0n04j0?P1\n>P0]0Q`0;@8L02d29@0000`0000500009@0000`000070000:00000`0000600009@0000`000080020\n9`0001P000060000000007ekO@0000009@0000`000060000F`0005P0000;0000/@0001H000340000\n0P0000`0000600001P0001`0;@8L0?P1>P3h0CX0;@8L02d2700]0Q`0;@8L0?P1=`3h0CL0;@8L02d2\n700]0RD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H0000\n1P0000000027QHL0000002D0000<00001P0005/0001H00002`000;40000E0000a0000080000<0000\n1P0000H0000L02d2703h0CL0n04g02d2700]0Q`0;@8L02d2703h0CL0n04g02d2700]0Q`0;@8U0000\n300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000\nT8j@0000000U0000300000H0001K0000F00000/0002a00005@000<@000020000300000H000060000\n700]0Q`0n04g0?P1=`0]0Q`0;@8L02d2800]0R00n04g0?P1=`0]0R00;@8P02d29@0000`000050000\n9@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000000009RFV0000000\n9@0000`000060000F`0005P0000<0000/@0001D0003400000P0000`0000600001P000200;@8P0?P1\n=`3h0CL0;@8P02d2800]0R00;@8P0?P1=`3h0CL0;@8P02d2800]0RD0000<00001@0002D0000<0000\n1`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000002OW9l0000002D0000<0000\n1P0005/0001H000030000;40000E0000a0000080000<00001P0000H0000P02d2803h0CL0n04g02d2\n800]0R00;@8P02d2803h0C<0n04c02d2800]0R00;@8U0000300000D0000U0000300000L0000X0000\n300000H0000U0000300000P0080W0000600000H000000000YJ>U0000000U0000300000H0001K0000\nF00000`0002a00004`000<@000020000300000H000060000800]0R00n04c0?P1<`0]0R00;@8P02d2\n800]0R00n04c0?P1<`0]0R00;@8P02d29@0000`0000500009@0000`000070000:00000`000060000\n9@0000`0000800209`0001P00006000000000:^XZ`0000009@0000`000060000F`0005P0000<0000\n/@0001<0003400000P0000`0000600001P000200;@8P0?P1<`3h0C<0;@8P02d2800]0R@0;@8T0?P1\n<`3h0C<0;@8T02d2900]0RD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n2000P2L0000H00001P000000002`[K00000002D0000<00001P0005/0001H00003P000;40000C0000\na0000080000<00001P0000H0000T02d2903h0C<0n04c02d2900]0R@0;@8T02d2903h0C<0n04c02d2\n900]0R@0;@8U0000300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000\n600000H000000000]K:e0000000U0000300000H0001K0000F00000h0002a00004`000<@000020000\n300000H000060000900]0R@0n04c0?P1<`0]0R@0;@8T02d2900]0R@0n04_0?P1;`0]0R@0;@8T02d2\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n00000;Vf^@0000009@0000`000060000F`0005P0000>0000/@000180003400000P0000`000060000\n1P0002@0;@8T0?P1;`3h0Bl0;@8T02d2900]0R@0;@8T0?P1;`3h0Bl0;@8T02d2900]0RD0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000002m^[d0\n000002D0000<00001P0005/0001H00003P000;40000B0000a0000080000<00001P0000H0000T02d2\n903h0Bl0n04_02d2900]0R@0;@8X02d2:03h0Bl0n04_02d2:00]0RP0;@8U0000300000D0000U0000\n300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000`Kk10000000U0000\n300000H0001K0000F00000l0002a00004P000<@000020000300000H000060000:00]0RP0n04_0?P1\n;`0]0RP0;@8X02d2:00]0RP0n04_0?P1;`0]0RP0;@8X02d29@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P00006000000000<C1a00000009@0000`000060000\nF`0005P0000?0000/@000180003400000P0000`0000600001P0002P0;@8X0?P1;`3h0Bl0;@8X02d2\n:00]0RP0;@8X0?P1;`3h0Bl0;@8X02d2:00]0RD0000<00001@0002D0000<00001`0002P0000<0000\n1P0002D0000<00002000P2L0000H00001P0000000036`lH0000002D0000<00001P0005/0001H0000\n3`000;40000B0000a0000080000<00001P0000H0000X02d2:03h0Bl0n04_02d2:00]0RP0;@8X02d2\n:03h0B/0n04[02d2:00]0RP0;@8U0000300000D0000U0000300000L0000X0000300000H0000U0000\n300000P0080W0000600000H000000000a/?60000000U0000300000H0001K0000D00000l0002a0000\n40000<@0000200002P0000H000040000:00]0RP0n04[0?P1:`0]0RP0;@8X02d2:`0]0R/0n04[02d2\n:`0]0RD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H0000\n1P0000000036`lH0000002D0000<00001P0005/00018000040000;40000@0000a000008000080000\n100000@0000[02d2:`3h0B/0;@8[02d2:`0]0R/0n04[02d2:`0]0RD0000<00001@0002D0000<0000\n1`0002P0000<00001P0002D0000<00002000P2L0000H00001P0000000036`lH0000002D0000<0000\n1P0005/00018000040000;40000@0000a000008000080000100000@0000[02d2:`3h0B/0;@8[02d2\n:`0]0R/0n04[02d2:`0]0RD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n1@00P2H0000L00001P0000000001000000000000008U0000300000H0000[000060000140003h0@00\nA00002/2000U0000300000L0000U0000300000D0000X0000300000H0000U0000300000P0080W0000\n600000H00000000000000000000U0000300000H0001K0000F00000h0003400004P000</000020000\n300000H00006000090130R@0;@8_02d2;`130R@0@`8T04<2:0130RP0;@8_02d2;`130RP0@`8X04<2\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n0000024Q8@0000009@0000`000060000F`0005P0000?0000a0000180003;00000P0000`000060000\n1P0002P0@`8X02d2;`0]0Rl0@`8X04<2:0130RP0@`8X02d2;`0]0Rl0@`8X04<2:0130RD0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P0000000017ATL0\n000002D0000<00001P0005/0001H00003`000<@0000B0000b`000080000<00001P0000H0000X04<2\n:00]0Rl0;@8_04<2:0130RP0@`8X04<2:00]0Rl0;@8_04<2:0130RP0@`8U0000300000D0000U0000\n300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000MWAf0000000U0000\n300000H0001K0000F00000l0003400004P000</000020000300000H000060000:0130RP0;@8_02d2\n;`130RP0@`8X04<2:0130RP0;@8[02d2:`130RP0@`8X04<29@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P000060000000009fJW@0000009@0000`000060000\nF`000500000?0000a0000100003;00000P0000X000060000100002P0@`8X02d2:`0]0R/0@`8X04<2\n:0130R/0@`8[02d2:`130R/0@`8U0000300000D0000U0000300000L0000X0000300000H0000U0000\n300000P0080W0000600000H000000000]K:e0000000U0000300000H0001K0000B000010000340000\n40000</000020000200000@000040000:`130R/0;@8[04<2:`130R/0@`8[02d2:`130R/0@`8U0000\n300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000600000H000000000\na/?60000000U0000300000H0001K0000B00001000034000040000</000020000200000@000040000\n:`130R/0;@8[04<2:`130R/0@`8[02d2:`130R/0@`8U0000300000D0000U0000300000L0000X0000\n300000H0000U0000300000D0080V0000700000H0000000000@000000000000029@0000`000060000\n:`0001P0000T0000;@8002d000110P009@0000`0000700009@0000`000050000:00000`000060000\n9@0000`0000800209`0001P00006000000000000000000009@0000`000060000F`0005P0000>0000\nZ@000180002a00000P0000`0000600001P0002@0n04T0>41;`3Q0Bl0n04T0?P1903h0BP0n04X0>41\n;`3Q0Bl0n04X0?P1:03h0BD0000<00001@0002D0000<00001`0002P0000<00001P0002D0000<0000\n2000P2L0000H00001P000000000Q8B40000002D0000<00001P0005/0001H00003`000:T0000B0000\n/@000080000<00001P0000H0000X0?P1:03Q0Bl0h@4_0?P1:03h0BP0n04X0?P1:03Q0Bl0h@4_0?P1\n:03h0BP0n04U0000300000D0000U0000300000L0000X0000300000H0000U0000300000P0080W0000\n600000H000000000AdI70000000U0000300000H0001K0000F00000l0002Y00004P000;4000020000\n300000H000060000:03h0BP0h@4_0>41;`3h0BP0n04X0?P1:03h0BP0h@4_0>41;`3h0BP0n04X0?P1\n9@0000`0000500009@0000`000070000:00000`0000600009@0000`0000800209`0001P000060000\n000007IdMP0000009@0000`000060000F`0005P0000?0000Z@000180002a00000P0000`000060000\n1P0002P0n04X0>41;`3Q0Bl0n04X0?P1:03h0BP0n04X0>41:`3Q0B/0n04X0?P1:03h0BD0000<0000\n1@0002D0000<00001`0002P0000<00001P0002D0000<00002000P2L0000H00001P000000002MVYd0\n000002D0000<00001P0005/0001@00003`000:T0000@0000/@000080000:00001P0000@0000X0?P1\n:03Q0B/0h@4[0?P1:03h0BP0n04[0?P1:`3Q0B/0n04[0?P19@0000`0000500009@0000`000070000\n:00000`0000600009@0000`0000800209`0001P00006000000000;Fb]@0000009@0000`000060000\nF`0004P0000@0000Z@000100002a00000P0000P000040000100002/0n04[0>41:`3h0B/0n04[0?P1\n:`3Q0B/0n04[0?P19@0000`0000500009@0000`000070000:00000`0000600009@0000`000080020\n9`0001P00006000000000<K3aP0000009@0000`000060000F`0004P0000@0000Z@000100002a0000\n0P0000P000040000100002/0n04[0>41:`3h0B/0n04[0?P1:`3Q0B/0n04[0?P19@0000`000050000\n9@0000`000070000:00000`0000600009@0000`0000500209P0001`0000600000000004000000000\n00000RD0000<00001P0002/0000H000090000>41000]0000mP4002D0000<00001`0002D0000<0000\n1@0002P0000<00001P0002D0000<00001@00P2D0000<00002000P2/0000H0000PP000?4100030@00\n9`8002D0000<00001`0002D0000<00001@000240000800007P0001P000220000l@4000H1000Y0P00\n4P0000`0000100009@0000`000080000600000`000000002E00007`0000a0000[`0005T0002m0000\n0@0003GB345oO`a1PP000?4100080000C00000@000220000l@4000D1000Y0P00G00006T0KP1U0780\nM01Y0640<P0700004`000180000;00002@0000P0000B00004P0002D0000<0000100005@0001D0000\nF@000:l0001Q0000`@000040000edPa1Ogl<@Nh0002f0@000@0004`000040000PP000?4100050@00\n:@800500000P0000;P000280000<0000ooooobD0000<00001@00P2D0000<00002000P2/0000H0000\ni`0009d2000U0@00d`8002D0000<00001`0002D0000<00001@000240000800007P0001P0003W0000\nW@8002P1003E0P004P0000`000010000DP00070100060000ioooo`00000000000000090100000000\n14008T40LP1Y0640K000000000000000000000000000000000000000000000000000000000000000\n0000000000000380J9lG06D0LP0b00004P<00010000B0`Cf0@0006Cd4P3`<OAg4P<4m]858L?<la80\n00000>Dhn7L001<0V10H0000002Xla80R0HC05cd4P2E:oQg83WhMooooom/m180@;OlMnP64`2P41P0\nX10H01[e702P41P000000000000B0`000400018310T100007?@B0?0am7L001<0D8PO0040000B0`@9\n10T0010Rm7O4l?eo0>3mO`400000000000000000000B0`000400018310X10000I?@B0?0am7LB0`@:\ndPDQ`a8310X0000010X0083d4P2@d_AgI7H02000000U0000300000H0000H000030000000009D0000\nH00005L0003[0000I0000?H000010000=M8<@Gmo347W0000W@8000<0001<000010000>L0002M0P00\n9`400=D2001D0000M01Q07D0000700003P0000h0000U0000300000@0001D0000E00006@0003[0000\nKP000?X000010000=M8<@Gmo344:0@00FP800040001<000010000>L0002M0P009`400=D2001@0000\n800002d0000R000030000?oooolV0000700000T0000000000@000000000000009@0000`000090000\n9@0000`000050020E`00040000130000f@0006H0003T00002@000101RP810G/2lP1`0^<0J@;A06`2\na@1d0[h0N`:f08<2/`2:0RD0000<00001@0002P0000<00002@0002D0000<00001`0002H0000L0000\n2@0000000001000000000000000U0000300000T0000W0000600000X00000000000000000000U0000\n300000X0001F0000;00004<0003J0000C@000>@000040000/`2:0[h0K0;=07/2/`2:0RD0000<0000\n1@0002P0000<00002@0002D0000<00001`0002P0000<00002P0002H0000L00002P00000000010000\n00000000o`0U0000300000X0000W0000600000T000000000003o0000000U0000300000T0001F0000\n;00006T0003P0000L0000>T000040000:`5o0QP1RP8[0IH2:`5o0RD0000<00001@0002P0000<0000\n2P0002D0000<00001`0002P0000<00002@0002L0000H00002@000000003oool0000002D0000<0000\n2@0002H0000L00002P0000000001000000000000008U0000300000X0000[000060000:@000270P00\n[@0008`2000U0000300000L0000X0000300000T0000U0000300000D0000X0000300000X0000U0000\n300000D0080U0000300000P0080[000060000:P0002o0P00<@400?D2000U0000300000L0000U0000\n300000D0000Q0000200001h0000H0000Z0000;l2000c0@00m`800180000<00000@0002D0000<0000\n200001P0000<000000000U@0001h0000?`000?L0001Y00001@400040000edPa1Ogl<@JP0002o0P00\n1`0004`000040000Z0000;l2000b0@00m`8005`0001d06l0LP1a07D0I@0a00002P000140000;0000\n4`0001<0000B00004P0002D0000<0000100005@0001D0000J@000?L0001b00002@400040000edPa1\nOgl<@AP100240P000@0004`000040000Z0000;l2000b0@00m`800500000P0000;P000280000<0000\nooooobL0000H00002P000000003oool0000002D0000<00002P0002H0000L00002@00000000010000\n00000000o`8U0000300000T0000[0000600009P1001F0P00o`400;d2000U0000300000L0000X0000\n300000X0000U0000300000D0000X0000300000T0000V0000700000T0000000000@00000000000?l0\n9@0000`0000900009`0001P0000:000000000?ooo`0000009@0000`0000:0000EP0002`0002E0000\nh`0009T0003V0000100009P1Q`:=0HX2V06>0YP1Q`8U0000300000D0000X0000300000T0000U0000\n300000L0000X0000300000X0000V0000700000X0000000000@0000000036`lH09@0000`0000:0000\n9@0000`000050020E`0002@0002m0000e`000;d0003c00000P000?H1I@;f0K@29@0000`000050000\n:00000`0000:00009@0000`0000700009P0001`0000:00000000004000000000a/?602D0000<0000\n2P0002L0000H00002@0000000036`lH0000002D0000<00002@0005H0000/0000^`000=<000300000\ne`0000@0003f0EX2oP5U0_81I@;f0EX29@0000`000050000:00000`0000:00009@0000`000070000\n:00000`0000900009P0001`0000900000000004000000000a/?602D0000<00002@0002D0000<0000\n1@00P5L0000T0000W0000?8000300000lP000080003n0K02W`6`0RD0000<00001@0002P0000<0000\n2@0002D0000<00001`0002H0000L00002@0000000001000000000<K3aP0U0000300000T0000W0000\n600000X000000000a/?60000000U0000300000X0001F0000;00009/0003`0000W`000?<000040000\nW06`0ZL1[0:W0K@2W06`0RD0000<00001@0002P0000<00002@0002D0000<00001`0002P0000<0000\n2P0002H0000L00002P0000000001000000000000000U0000300000X0000U0000300000D0081G0000\n;00009`0003K0000_@000?8000040000mP6`0/`1/0;<0G02W`5`0RD0000<00001@0002P0000<0000\n2P0002D0000<00001`0002D0000<00001@00P2D0000<00002000P2/0000H0000Y`400;l2000>0P00\nm@8002D0000<00001`0002D0000<00001@000240000800007P0001P0002W0@00_`800142003g0P00\n4P0000`0000100009@0000`000080000600000`000000002E00006`0002O0000m`000;d000050@00\n0@0003GB345oO`a1Y`400;l200050000C00000@0002W0@00_`800102003g0P00F00007<0M01U0700\n<@000140000900004P0001<0000B00009@0000`000040000E00005@0002m0000m`000<D000090@00\n0@0003GB345oO`a1n04008@200010000C00000@0002W0@00_`800102003g0P00D0000200000^0000\n8P0000`0003ooooo9P0001`0000:0000000000@000000000000002D0000<00002P0002D0000<0000\n1@00P5L0000X00003P0000h000110000=P0000<0000[09<0:`0^0:L0;P0U0000300000D0000X0000\n300000X0000U0000300000L0000V0000700000X00000000010000000000000009@0000`0000:0000\n9@0000`000050020E`0002@0001V00003P0009L0000C00000P0001@1;P2=0Bl09@0000`000050000\n:00000`0000:00009@0000`0000700009P0001`0000:0000000000@000000000000002D0000<0000\n2P0002D0000<00001@00P5L0000T0000_00000h0003`00004`000080003j0Bh0N@8_02D0000<0000\n1@0002P0000<00002P0002D0000<00001`0002H0000L00002P0000000004000000000000000U0000\n300000X0000U0000300000D0081G0000900001@1000>0000C04001<000020000h`8^06h3;`0U0000\n300000D0000X0000300000X0000U0000300000L0000V0000700000X0000000001000000000000000\n9@0000`0000:00009@0000`000050020E`0002@0000>0000E`0001<0001_00000P0002/0o00/03D1\n9@0000`000050000:00000`0000:00009@0000`0000700009P0001`0000:0000000000@000000000\n000002D0000<00002P0002D0000<00001@00P5L0000T00003P0008l0000C0000Z@000080000[09h1\n;03J0BD0000<00001@0002P0000<00002P0002D0000<00001`0002H0000L00002P00000000040000\n00000000000U0000300000X0000U0000300000D0081G0000:00000h0003;0000@P000>H000030000\n:`170R/0RP:[08X29@0000`000050000:00000`0000:00009@0000`0000700009P0001`0000:0000\n000000@000000000000002D0000<00002P0002D0000<00001@00P5L0000T0000JP000>80002H0000\ni`000080000O0HX2T06;0RD0000<00001@0002P0000<00002P0002D0000<00001`0002D0000<0000\n0`0002P0000<0000100002P0000<0000200002P0000<00001P0002L0000H00001P000000003oool0\n000002D0000<00001P0002H0000L0000200000000000000000000000000U0000300000P0000R0000\n30000?oooolR000030000?oooolU000030000040000U0000300000L0080U000030000000080`0000\n300000l0080U0000300000d0081;00004000000000050000:00000`000010000:00000`000020000\n:00000`000030000:00000`000060000:00000`000050000:00000`000080000:00000`000070000\n3P0001@000000000400001@0\n"], "Text", Rule[ImageSize, List[357, 265]], Rule[ImageMargins, List[List[0, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]]], Cell[BoxData[RowBox[List["A", " ", "LinearActuator", " ", "model", " ", "containing", " ", "a", " ", "spring", " ", "damped", " ", "connection", " ", "to", " ", "an", " ", "anchoring", " ", RowBox[List["point", "."]]]]], "NumberedFigure"], Cell["\nConsider the following model of a linear actuator, depicted in Figure 1, with a spring-damped connection to an anchoring point:", "Text"], Cell["model LinearActuator\n  import Modelica.Mechanics.Translational;\n  import Modelica.Mechanics.Rotational;\n  import Modelica.Blocks.Sources;\n  Translational.SlidingMass \tslidingMass1(m = 0.5);\n  Translational.SpringDamper \tspringDamper1(d = 3, c = 20);\n  Translational.Fixed  \t\t  \tfixed1;\n  Rotational.IdealGearR2T \t  \tidealGearR2T1;\n  Rotational.Inertia \t\t    inertia1(J = 0.1);\n  Rotational.SpringDamper   \tspringDamper2(c = 15, d = 2);\n  Rotational.Inertia   \t    \tinertia2(J = 0.1);\n  Rotational.Torque          \ttorque1;\n  Sources.Step               \tstep1;\nequation\n  connect(inertia1.flange_b,      idealGearR2T1.flange_a);\n  connect(idealGearR2T1.flange_b, slidingMass1.flange_a);\n  connect(slidingMass1.flange_b,  springDamper1.flange_a);\n  connect(springDamper1.flange_b, fixed1.flange_b);\n  connect(inertia1.flange_a,      springDamper2.flange_b);\n  connect(springDamper2.flange_a, inertia2.flange_b);\n  connect(inertia2.flange_a,      torque1.flange_b);\n  connect(torque1.inPort,         step1.outPort);\nend LinearActuator;", "ModelicaInput"], Cell[TextData[List["\nWe first make a simulation of the step response for the above linear actuator with damping ", StyleBox["d=2", Rule[FontFamily, "Courier"]], " and store the result in the scripting language variable called ", StyleBox["resLinActFirst", Rule[FontFamily, "Courier"]], ". Then we plot the simulated position of the sliding mass as a function of time to take a closer look at the response."]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Simulation of LinearActuator", "Section"], Cell["simulate( LinearActuator, stopTime=5 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(LinearActuator\\) : 2004-2-2 16:7:31.9963506 : \\!\\({0.`, 5.`}\\) : 502 data points : 1 events : 78 variables>\\n\\!\\({\\*TagBox[\\(fixed1 . flange\[UnderBracket]b . f\\), HoldForm], \\*TagBox[\\(fixed1 . flange\[UnderBracket]b . s\\), HoldForm], \\*TagBox[\\(fixed1 . s0\\), HoldForm], \\*TagBox[\\(idealGearR2T1 . flange\[UnderBracket]a . \[Phi]\\), HoldForm], \\*TagBox[\\(idealGearR2T1 . flange\[UnderBracket]a . \[Tau]\\), HoldForm], \\*TagBox[\\(idealGearR2T1 . flange\[UnderBracket]b . f\\), HoldForm], \\*TagBox[\\(idealGearR2T1 . flange\[UnderBracket]b . s\\), HoldForm], \\*TagBox[\\(idealGearR2T1 . ratio\\), HoldForm], \\*TagBox[\\(inertia1 . a\\), HoldForm], \\*TagBox[\\(inertia1 . flange\[UnderBracket]a . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia1 . flange\[UnderBracket]a . \[Tau]\\), HoldForm], \\*TagBox[\\(inertia1 . flange\[UnderBracket]b . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia1 . flange\[UnderBracket]b . \[Tau]\\), HoldForm], \\*TagBox[\\(inertia1 . J\\), HoldForm], \\*TagBox[\\(inertia1 . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia1 . w\\), HoldForm], \\*TagBox[\\(inertia2 . a\\), HoldForm], \\*TagBox[\\(inertia2 . flange\[UnderBracket]a . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia2 . flange\[UnderBracket]a . \[Tau]\\), HoldForm], \\*TagBox[\\(inertia2 . flange\[UnderBracket]b . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia2 . flange\[UnderBracket]b . \[Tau]\\), HoldForm], \\*TagBox[\\(inertia2 . J\\), HoldForm], \\*TagBox[\\(inertia2 . \[Phi]\\), HoldForm], \\*TagBox[\\(inertia2 . w\\), HoldForm], \\*TagBox[\\(slidingMass1 . a\\), HoldForm], \\*TagBox[\\(slidingMass1 . flange\[UnderBracket]a . f\\), HoldForm], \\*TagBox[\\(slidingMass1 . flange\[UnderBracket]a . s\\), HoldForm], \\*TagBox[\\(slidingMass1 . flange\[UnderBracket]b . f\\), HoldForm], \\*TagBox[\\(slidingMass1 . flange\[UnderBracket]b . s\\), HoldForm], \\*TagBox[\\(slidingMass1 . L\\), HoldForm], \\*TagBox[\\(slidingMass1 . m\\), HoldForm], \\*TagBox[\\(slidingMass1 . s\\), HoldForm], \\*TagBox[\\(slidingMass1 . v\\), HoldForm], \\*TagBox[\\(springDamper1 . c\\), HoldForm], \\*TagBox[\\(springDamper1 . d\\), HoldForm], \\*TagBox[\\(springDamper1 . f\\), HoldForm], \\*TagBox[\\(springDamper1 . flange\[UnderBracket]a . f\\), HoldForm], \\*TagBox[\\(springDamper1 . flange\[UnderBracket]a . s\\), HoldForm], \\*TagBox[\\(springDamper1 . flange\[UnderBracket]b . f\\), HoldForm], \\*TagBox[\\(springDamper1 . flange\[UnderBracket]b . s\\), HoldForm], \\*TagBox[\\(springDamper1 . s\[UnderBracket]rel\\), HoldForm], \\*TagBox[\\(springDamper1 . s\[UnderBracket]rel0\\), HoldForm], \\*TagBox[\\(springDamper1 . v\[UnderBracket]rel\\), HoldForm], \\*TagBox[\\(springDamper2 . c\\), HoldForm], \\*TagBox[\\(springDamper2 . d\\), HoldForm], \\*TagBox[\\(springDamper2 . flange\[UnderBracket]a . \[Phi]\\), HoldForm], \\*TagBox[\\(springDamper2 . flange\[UnderBracket]a . \[Tau]\\), HoldForm], \\*TagBox[\\(springDamper2 . flange\[UnderBracket]b . \[Phi]\\), HoldForm], \\*TagBox[\\(springDamper2 . flange\[UnderBracket]b . \[Tau]\\), HoldForm], \\*TagBox[\\(springDamper2 . phi\[UnderBracket]rel\\), HoldForm], \\*TagBox[\\(springDamper2 . phi\[UnderBracket]rel0\\), HoldForm], \\*TagBox[\\(springDamper2 . \[Tau]\\), HoldForm], \\*TagBox[\\(springDamper2 . w\[UnderBracket]rel\\), HoldForm], \\*TagBox[\\(\\(\\(step1 . height\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(step1 . nout\\), HoldForm], \\*TagBox[\\(\\(\\(step1 . offset\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(step1 . outPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(step1 . outPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(step1 . startTime\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(step1 . y\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(torque1 . flange\[UnderBracket]b . \[Phi]\\), HoldForm], \\*TagBox[\\(torque1 . flange\[UnderBracket]b . \[Tau]\\), HoldForm], \\*TagBox[\\(torque1 . inPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(torque1 . inPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(torque1 . \[Tau]\\), HoldForm], \\*TagBox[SuperscriptBox[\\((inertia1 . \[Phi])\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((inertia1 . \[Phi])\\), \\\"\[Prime]\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((inertia1 . w)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((inertia2 . \[Phi])\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((inertia2 . w)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((slidingMass1 . s)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((slidingMass1 . s)\\), \\\"\[Prime]\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((slidingMass1 . v)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((slidingMass1 . flange\[UnderBracket]a . s)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((slidingMass1 . flange\[UnderBracket]a . s)\\), \\\"\[Prime]\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((springDamper1 . s\[UnderBracket]rel)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((springDamper1 . flange\[UnderBracket]a . s)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((springDamper2 . phi\[UnderBracket]rel)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm]}\\)\""], "Output"], Cell["", "Text"], Cell["plot( slidingMass1.s );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.190476 0.0147151 8.97992 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -3 -9 ]\n[.97619 .00222 3 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .10451 -24 -4.5 ]\n[.01131 .10451 0 4.5 ]\n[.01131 .19431 -24 -4.5 ]\n[.01131 .19431 0 4.5 ]\n[.01131 .28411 -24 -4.5 ]\n[.01131 .28411 0 4.5 ]\n[.01131 .37391 -24 -4.5 ]\n[.01131 .37391 0 4.5 ]\n[.01131 .46371 -24 -4.5 ]\n[.01131 .46371 0 4.5 ]\n[.01131 .55351 -24 -4.5 ]\n[.01131 .55351 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(1)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(2)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(3)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(4)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(5)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.0619 .01472 m\n.0619 .01847 L\ns\n.1 .01472 m\n.1 .01847 L\ns\n.1381 .01472 m\n.1381 .01847 L\ns\n.17619 .01472 m\n.17619 .01847 L\ns\n.25238 .01472 m\n.25238 .01847 L\ns\n.29048 .01472 m\n.29048 .01847 L\ns\n.32857 .01472 m\n.32857 .01847 L\ns\n.36667 .01472 m\n.36667 .01847 L\ns\n.44286 .01472 m\n.44286 .01847 L\ns\n.48095 .01472 m\n.48095 .01847 L\ns\n.51905 .01472 m\n.51905 .01847 L\ns\n.55714 .01472 m\n.55714 .01847 L\ns\n.63333 .01472 m\n.63333 .01847 L\ns\n.67143 .01472 m\n.67143 .01847 L\ns\n.70952 .01472 m\n.70952 .01847 L\ns\n.74762 .01472 m\n.74762 .01847 L\ns\n.82381 .01472 m\n.82381 .01847 L\ns\n.8619 .01472 m\n.8619 .01847 L\ns\n.9 .01472 m\n.9 .01847 L\ns\n.9381 .01472 m\n.9381 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .10451 m\n.03006 .10451 L\ns\n[(0.01)] .01131 .10451 1 0 Mshowa\n.02381 .19431 m\n.03006 .19431 L\ns\n[(0.02)] .01131 .19431 1 0 Mshowa\n.02381 .28411 m\n.03006 .28411 L\ns\n[(0.03)] .01131 .28411 1 0 Mshowa\n.02381 .37391 m\n.03006 .37391 L\ns\n[(0.04)] .01131 .37391 1 0 Mshowa\n.02381 .46371 m\n.03006 .46371 L\ns\n[(0.05)] .01131 .46371 1 0 Mshowa\n.02381 .55351 m\n.03006 .55351 L\ns\n[(0.06)] .01131 .55351 1 0 Mshowa\n.125 Mabswid\n.02381 .03267 m\n.02756 .03267 L\ns\n.02381 .05063 m\n.02756 .05063 L\ns\n.02381 .06859 m\n.02756 .06859 L\ns\n.02381 .08655 m\n.02756 .08655 L\ns\n.02381 .12247 m\n.02756 .12247 L\ns\n.02381 .14043 m\n.02756 .14043 L\ns\n.02381 .15839 m\n.02756 .15839 L\ns\n.02381 .17635 m\n.02756 .17635 L\ns\n.02381 .21227 m\n.02756 .21227 L\ns\n.02381 .23023 m\n.02756 .23023 L\ns\n.02381 .24819 m\n.02756 .24819 L\ns\n.02381 .26615 m\n.02756 .26615 L\ns\n.02381 .30207 m\n.02756 .30207 L\ns\n.02381 .32003 m\n.02756 .32003 L\ns\n.02381 .33799 m\n.02756 .33799 L\ns\n.02381 .35595 m\n.02756 .35595 L\ns\n.02381 .39187 m\n.02756 .39187 L\ns\n.02381 .40983 m\n.02756 .40983 L\ns\n.02381 .42779 m\n.02756 .42779 L\ns\n.02381 .44575 m\n.02756 .44575 L\ns\n.02381 .48167 m\n.02756 .48167 L\ns\n.02381 .49963 m\n.02756 .49963 L\ns\n.02381 .51759 m\n.02756 .51759 L\ns\n.02381 .53555 m\n.02756 .53555 L\ns\n.02381 .57147 m\n.02756 .57147 L\ns\n.02381 .58943 m\n.02756 .58943 L\ns\n.02381 .60739 m\n.02756 .60739 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .01477 L\n.02762 .01508 L\n.02952 .01587 L\n.03143 .0173 L\n.03333 .01951 L\n.03524 .0226 L\n.03714 .02661 L\n.03905 .0316 L\n.04095 .03757 L\n.04286 .04452 L\n.04476 .05244 L\n.04667 .06129 L\n.04857 .07103 L\n.05048 .08162 L\n.05238 .09299 L\n.05429 .10509 L\n.05619 .11785 L\n.0581 .13122 L\n.06 .14513 L\n.0619 .15951 L\n.06381 .17429 L\n.06571 .18941 L\n.06762 .2048 L\n.06952 .22041 L\n.07143 .23616 L\n.07333 .25201 L\n.07524 .26789 L\n.07714 .28376 L\n.07905 .29956 L\n.08095 .31524 L\n.08286 .33077 L\n.08476 .34609 L\n.08667 .36116 L\n.08857 .37596 L\n.09048 .39044 L\n.09238 .40457 L\n.09429 .41832 L\n.09619 .43167 L\n.0981 .4446 L\n.1 .45707 L\n.1019 .46908 L\n.10381 .48061 L\n.10571 .49164 L\n.10762 .50216 L\n.10952 .51216 L\n.11143 .52163 L\n.11333 .53057 L\n.11524 .53898 L\n.11714 .54684 L\nMistroke\n.11905 .55417 L\n.12095 .56096 L\n.12286 .56722 L\n.12476 .57294 L\n.12667 .57814 L\n.12857 .58282 L\n.13048 .58699 L\n.13238 .59065 L\n.13429 .59383 L\n.13619 .59652 L\n.1381 .59875 L\n.14 .60052 L\n.1419 .60185 L\n.14381 .60275 L\n.14571 .60323 L\n.14762 .60332 L\n.14952 .60302 L\n.15143 .60236 L\n.15333 .60135 L\n.15524 .60001 L\n.15714 .59835 L\n.15905 .59639 L\n.16095 .59416 L\n.16286 .59166 L\n.16476 .58891 L\n.16667 .58594 L\n.16857 .58275 L\n.17048 .57937 L\n.17238 .57582 L\n.17429 .5721 L\n.17619 .56824 L\n.1781 .56426 L\n.18 .56016 L\n.1819 .55596 L\n.18381 .55169 L\n.18571 .54735 L\n.18762 .54296 L\n.18952 .53854 L\n.19143 .53409 L\n.19333 .52963 L\n.19524 .52517 L\n.19714 .52072 L\n.19905 .5163 L\n.20095 .51192 L\n.20286 .50758 L\n.20476 .50331 L\n.20667 .49909 L\n.20857 .49495 L\n.21048 .4909 L\n.21238 .48693 L\nMistroke\n.21429 .48306 L\n.21619 .4793 L\n.2181 .47565 L\n.22 .47211 L\n.2219 .46869 L\n.22381 .4654 L\n.22571 .46224 L\n.22762 .45921 L\n.22952 .45631 L\n.23143 .45355 L\n.23333 .45093 L\n.23524 .44845 L\n.23714 .44611 L\n.23905 .44392 L\n.24095 .44187 L\n.24286 .43995 L\n.24476 .43819 L\n.24667 .43656 L\n.24857 .43507 L\n.25048 .43372 L\n.25238 .43251 L\n.25429 .43142 L\n.25619 .43048 L\n.2581 .42965 L\n.26 .42896 L\n.2619 .42839 L\n.26381 .42793 L\n.26571 .42759 L\n.26762 .42737 L\n.26952 .42725 L\n.27143 .42723 L\n.27333 .42731 L\n.27524 .42749 L\n.27714 .42776 L\n.27905 .42812 L\n.28095 .42856 L\n.28286 .42907 L\n.28476 .42966 L\n.28667 .43032 L\n.28857 .43104 L\n.29048 .43182 L\n.29238 .43265 L\n.29429 .43354 L\n.29619 .43447 L\n.2981 .43544 L\n.3 .43645 L\n.3019 .4375 L\n.30381 .43857 L\n.30571 .43967 L\n.30762 .44078 L\nMistroke\n.30952 .44192 L\n.31143 .44306 L\n.31333 .44422 L\n.31524 .44538 L\n.31714 .44655 L\n.31905 .44771 L\n.32095 .44887 L\n.32286 .45002 L\n.32476 .45117 L\n.32667 .45229 L\n.32857 .45341 L\n.33048 .4545 L\n.33238 .45558 L\n.33429 .45663 L\n.33619 .45766 L\n.3381 .45866 L\n.34 .45963 L\n.3419 .46058 L\n.34381 .46149 L\n.34571 .46237 L\n.34762 .46322 L\n.34952 .46404 L\n.35143 .46482 L\n.35333 .46557 L\n.35524 .46629 L\n.35714 .46697 L\n.35905 .46761 L\n.36095 .46821 L\n.36286 .46877 L\n.36476 .4693 L\n.36667 .46979 L\n.36857 .47024 L\n.37048 .47066 L\n.37238 .47104 L\n.37429 .47139 L\n.37619 .47171 L\n.3781 .47198 L\n.38 .47223 L\n.3819 .47244 L\n.38381 .47262 L\n.38571 .47277 L\n.38762 .47288 L\n.38952 .47297 L\n.39143 .47304 L\n.39333 .47307 L\n.39524 .47308 L\n.39714 .47307 L\n.39905 .47303 L\n.40095 .47296 L\n.40286 .47288 L\nMistroke\n.40476 .47277 L\n.40667 .47265 L\n.40857 .4725 L\n.41048 .47234 L\n.41238 .47217 L\n.41429 .47197 L\n.41619 .47177 L\n.4181 .47155 L\n.42 .47132 L\n.4219 .47108 L\n.42381 .47083 L\n.42571 .47058 L\n.42762 .47031 L\n.42952 .47004 L\n.43143 .46976 L\n.43333 .46948 L\n.43524 .46919 L\n.43714 .4689 L\n.43905 .46861 L\n.44095 .46832 L\n.44286 .46802 L\n.44476 .46773 L\n.44667 .46744 L\n.44857 .46716 L\n.45048 .46687 L\n.45238 .46659 L\n.45429 .46631 L\n.45619 .46603 L\n.4581 .46576 L\n.46 .46549 L\n.4619 .46524 L\n.46381 .46499 L\n.46571 .46474 L\n.46762 .4645 L\n.46952 .46427 L\n.47143 .46405 L\n.47333 .46384 L\n.47524 .46363 L\n.47714 .46343 L\n.47905 .46324 L\n.48095 .46306 L\n.48286 .46289 L\n.48476 .46273 L\n.48667 .46258 L\n.48857 .46243 L\n.49048 .4623 L\n.49238 .46217 L\n.49429 .46206 L\n.49619 .46195 L\n.4981 .46185 L\nMistroke\n.5 .46176 L\n.5019 .46168 L\n.50381 .46161 L\n.50571 .46155 L\n.50762 .46149 L\n.50952 .46145 L\n.51143 .46141 L\n.51333 .46138 L\n.51524 .46135 L\n.51714 .46134 L\n.51905 .46133 L\n.52095 .46133 L\n.52286 .46133 L\n.52476 .46134 L\n.52667 .46136 L\n.52857 .46138 L\n.53048 .46141 L\n.53238 .46144 L\n.53429 .46148 L\n.53619 .46152 L\n.5381 .46157 L\n.54 .46162 L\n.5419 .46167 L\n.54381 .46173 L\n.54571 .46178 L\n.54762 .46185 L\n.54952 .46191 L\n.55143 .46198 L\n.55333 .46205 L\n.55524 .46212 L\n.55714 .46219 L\n.55905 .46226 L\n.56095 .46233 L\n.56286 .46241 L\n.56476 .46248 L\n.56667 .46256 L\n.56857 .46263 L\n.57048 .46271 L\n.57238 .46278 L\n.57429 .46285 L\n.57619 .46293 L\n.5781 .463 L\n.58 .46307 L\n.5819 .46314 L\n.58381 .46321 L\n.58571 .46327 L\n.58762 .46334 L\n.58952 .4634 L\n.59143 .46346 L\n.59333 .46352 L\nMistroke\n.59524 .46358 L\n.59714 .46363 L\n.59905 .46369 L\n.60095 .46374 L\n.60286 .46379 L\n.60476 .46384 L\n.60667 .46388 L\n.60857 .46392 L\n.61048 .46396 L\n.61238 .464 L\n.61429 .46404 L\n.61619 .46407 L\n.6181 .4641 L\n.62 .46413 L\n.6219 .46416 L\n.62381 .46418 L\n.62571 .4642 L\n.62762 .46422 L\n.62952 .46424 L\n.63143 .46426 L\n.63333 .46427 L\n.63524 .46428 L\n.63714 .46429 L\n.63905 .4643 L\n.64095 .4643 L\n.64286 .46431 L\n.64476 .46431 L\n.64667 .46431 L\n.64857 .46431 L\n.65048 .46431 L\n.65238 .4643 L\n.65429 .4643 L\n.65619 .46429 L\n.6581 .46428 L\n.66 .46427 L\n.6619 .46426 L\n.66381 .46425 L\n.66571 .46424 L\n.66762 .46423 L\n.66952 .46421 L\n.67143 .4642 L\n.67333 .46418 L\n.67524 .46417 L\n.67714 .46415 L\n.67905 .46413 L\n.68095 .46411 L\n.68286 .4641 L\n.68476 .46408 L\n.68667 .46406 L\n.68857 .46404 L\nMistroke\n.69048 .46402 L\n.69238 .464 L\n.69429 .46398 L\n.69619 .46397 L\n.6981 .46395 L\n.7 .46393 L\n.7019 .46391 L\n.70381 .46389 L\n.70571 .46387 L\n.70762 .46386 L\n.70952 .46384 L\n.71143 .46382 L\n.71333 .4638 L\n.71524 .46379 L\n.71714 .46377 L\n.71905 .46376 L\n.72095 .46374 L\n.72286 .46373 L\n.72476 .46371 L\n.72667 .4637 L\n.72857 .46369 L\n.73048 .46368 L\n.73238 .46367 L\n.73429 .46366 L\n.73619 .46365 L\n.7381 .46364 L\n.74 .46363 L\n.7419 .46362 L\n.74381 .46362 L\n.74571 .46361 L\n.74762 .4636 L\n.74952 .4636 L\n.75143 .4636 L\n.75333 .46359 L\n.75524 .46359 L\n.75714 .46359 L\n.75905 .46359 L\n.76095 .46359 L\n.76286 .46358 L\n.76476 .46359 L\n.76667 .46359 L\n.76857 .46359 L\n.77048 .46359 L\n.77238 .46359 L\n.77429 .46359 L\n.77619 .4636 L\n.7781 .4636 L\n.78 .46361 L\n.7819 .46361 L\n.78381 .46361 L\nMistroke\n.78571 .46362 L\n.78762 .46362 L\n.78952 .46363 L\n.79143 .46363 L\n.79333 .46364 L\n.79524 .46364 L\n.79714 .46364 L\n.79905 .46365 L\n.80095 .46365 L\n.80286 .46365 L\n.80476 .46366 L\n.80667 .46366 L\n.80857 .46366 L\n.81048 .46367 L\n.81238 .46367 L\n.81429 .46368 L\n.81619 .46368 L\n.8181 .46368 L\n.82 .46369 L\n.8219 .46369 L\n.82381 .46369 L\n.82571 .4637 L\n.82762 .4637 L\n.82952 .4637 L\n.83143 .4637 L\n.83333 .46371 L\n.83524 .46371 L\n.83714 .46371 L\n.83905 .46371 L\n.84095 .46371 L\n.84286 .46371 L\n.84476 .46371 L\n.84667 .46371 L\n.84857 .46371 L\n.85048 .46372 L\n.85238 .46372 L\n.85429 .46372 L\n.85619 .46372 L\n.8581 .46371 L\n.86 .46371 L\n.8619 .46371 L\n.86381 .46371 L\n.86571 .46371 L\n.86762 .46371 L\n.86952 .46371 L\n.87143 .46371 L\n.87333 .46371 L\n.87524 .4637 L\n.87714 .4637 L\n.87905 .4637 L\nMistroke\n.88095 .4637 L\n.88286 .4637 L\n.88476 .46369 L\n.88667 .46369 L\n.88857 .46369 L\n.89048 .46369 L\n.89238 .46368 L\n.89429 .46368 L\n.89619 .46368 L\n.8981 .46368 L\n.9 .46368 L\n.9019 .46367 L\n.90381 .46367 L\n.90571 .46367 L\n.90762 .46367 L\n.90952 .46367 L\n.91143 .46367 L\n.91333 .46366 L\n.91524 .46366 L\n.91714 .46366 L\n.91905 .46366 L\n.92095 .46366 L\n.92286 .46366 L\n.92476 .46366 L\n.92667 .46366 L\n.92857 .46366 L\n.93048 .46366 L\n.93238 .46366 L\n.93429 .46365 L\n.93619 .46365 L\n.9381 .46365 L\n.94 .46365 L\n.9419 .46365 L\n.94381 .46365 L\n.94571 .46365 L\n.94762 .46365 L\n.94952 .46365 L\n.95143 .46365 L\n.95333 .46365 L\n.95524 .46365 L\n.95714 .46366 L\n.95905 .46366 L\n.96095 .46366 L\n.96286 .46366 L\n.96476 .46366 L\n.96667 .46366 L\n.96857 .46366 L\n.97048 .46366 L\n.97238 .46366 L\n.97429 .46366 L\nMistroke\n.97619 .46366 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(slidingMass1) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n147.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(s) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n153.000 12.813 moveto\n(L) show\n159.000 12.813 moveto\n(@) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n171.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00C03IfMT5000002P0fMWI1000\n000Z0=WIf@800000;03IfMT3000002T0fMWI0P00000M0=WIf@00CP3IfMT00`000000fMWI0=WIf@0X\n0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WIf@3IfMT00000;03IfMT00`000000fMWI0=WI\nf@0W0=WIf@040000003IfMT0fMWI000001`0fMWI001>0=WIf@030000003IfMT0fMWI02T0fMWI00<0\n00000=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0X0=WIf@D00000:`3IfMT00`000000fMWI0=WI\nf@0J0=WIf@00CP3IfMT00`000000fMWI0=WIf@0Z0=WIf@030000003IfMT0fMWI02/0fMWI00<00000\n0=WIf@3IfMT09`3IfMT010000000fMWI0=WIf@00000Z0=WIf@8000007@3IfMT004h0fMWI00<00000\n0=WIf@3IfMT0:03IfMT010000000fMWI0=WIf@00000Y0=WIf@040000003IfMT0fMWI000002X0fMWI\n00<000000=WIf@000000:P3IfMT00`000000fMWI0=WIf@0L0=WIf@00C@3IfMT2000002/0fMWI0P00\n000[0=WIf@800000;03IfMT2000002X0fMWI0`00000L0=WIf@00o`3IfMTQ0=WIf@008@3IfMT00`00\n0000fMWI0=WIf@3l0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@008@3IfMT00`000000fMWI\n0=WIf@3]0=WIf@8000003@3IfMT00240fMWI00<000000=WIf@3IfMT0k03IfMT010000000fMWI0=WI\nf@00000<0=WIf@00703IfMT5000000<0003oi@0000070=WIf@030000003IfMT0fMWI00d0fMWI000Q\n0=WIf@040000003IfMT0fMWI0000obT0fMWI00<000000=WIf@3IfMT0:P3IfMT00`000000fMWI0=WI\nf@0Z0=WIf@030000003IfMT0fMWI02X0fMWI00<000000=WIf@3IfMT0:P3IfMT00`000000fMWI0=WI\nf@0:0=WIf@D00000303IfMT00240fMWI00D000000=WIf@3IfMT0fMWI0000o`3Z0=WIf@030000003I\nfMT0fMWI00d0fMWI000Q0=WIf@050000003IfMT0fMWI0=WIf@000?l0nP3IfMT00240fMWI0P000002\n0=WIf@030000o`3IfMT0fMWI0?P0fMWI000Q0=WIf@050000003IfMT0fMWI0=WIf@000?l0nP3IfMT0\n0240fMWI00D000000=WIf@3IfMT0fMWI0000o`3j0=WIf@008@3IfMT00`000000fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI0?L0fMWI000Q0=WIf@8000000`3IfMT00`000?l0fMWI0=WIf@3g0=WIf@00\n8@3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI0?L0fMWI000Q0=WIf@030000003I\nfMT0fMWI00<0fMWI00<0003o0=WIf@3IfMT0mP3IfMT00240fMWI00<000000=WIf@3IfMT00`3IfMT0\n0`000?l0fMWI0=WIf@3f0=WIf@008@3IfMT2000000@0fMWI00<0003o0=WIf@3IfMT0mP3IfMT00240\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000?l0fMWI0=WIf@3f0=WIf@008@3IfMT00`000000fMWI\n0=WIf@030=WIf@030000o`3IfMT0fMWI0?H0fMWI000Q0=WIf@030000003IfMT0fMWI00@0fMWI00<0\n003o0=WIf@3IfMT0m@3IfMT00240fMWI00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@3e\n0=WIf@008@3IfMT2000000D0fMWI00<0003o0=WIf@3IfMT0m@3IfMT00240fMWI00<000000=WIf@3I\nfMT0103IfMT00`000?l0fMWI0=WIf@3e0=WIf@002@3IfMT2000000@0fMWI0P0000040=WIf@800000\n0`3IfMT500000080fMWI00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@3e0=WIf@00203I\nfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<000000=WIf@3I\nfMT00P3IfMT00`000000fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0?@0fMWI00080=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000000fMWI0=WIf@020=WI\nf@<000001@3IfMT00`000?l0fMWI0=WIf@3d0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WI\nf@040000003IfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WIf@05\n0=WIf@030000o`3IfMT0fMWI0?@0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@00000\n0=WIf@3IfMT00000103IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI00D0fMWI00<0\n003o0=WIf@3IfMT0m03IfMT000T0fMWI0P00000:0=WIf@800000103IfMT2000000@0fMWI00<00000\n0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@3d0=WIf@008@3IfMT2000000H0fMWI00<0003o0=WI\nf@3IfMT0m03IfMT00240fMWI00<000000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3c0=WIf@00\n8@3IfMT00`000000fMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q0=WIf@030000003I\nfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT0l`3IfMT00240fMWI0P0000070=WIf@030000o`3IfMT0\nfMWI0?<0fMWI000Q0=WIf@030000003IfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT0l`3IfMT00240\nfMWI00<000000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3c0=WIf@008@3IfMT00`000000fMWI\n0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q0=WIf@8000001`3IfMT00`000?l0fMWI0=WI\nf@3c0=WIf@008@3IfMT00`000000fMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q0=WI\nf@030000003IfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT0l`3IfMT00240fMWI00<000000=WIf@3I\nfMT01`3IfMT00`000?l0fMWI0=WIf@3b0=WIf@008@3IfMT00`000000fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0?80fMWI000Q0=WIf@800000203IfMT00`000?l0fMWI0=WIf@3b0=WIf@008@3IfMT0\n0`000000fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0?80fMWI00090=WIf@800000103IfMT20000\n00@0fMWI0P0000030=WIf@@000000`3IfMT00`000000fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI\n0?80fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000000P3IfMT0\n0`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0lP3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000030=WIf@030000003I\nfMT0fMWI00<0fMWI0`0000070=WIf@030000o`3IfMT0fMWI0?80fMWI00080=WIf@040000003IfMT0\nfMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000000fMWI0=WIf@020=WIf@030000\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0l@3IfMT000P0fMWI00@000000=WIf@3IfMT00000\n203IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@3a0=WIf@002@3IfMT2000000X0fMWI0P0000040=WIf@80\n0000103IfMT00`000000fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI0?40fMWI000Q0=WIf@800000\n2@3IfMT00`000?l0fMWI0=WIf@3a0=WIf@008@3IfMT00`000000fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI0?40fMWI000Q0=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0l@3IfMT0\n0240fMWI00<000000=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@3a0=WIf@008@3IfMT2000000T0\nfMWI00<0003o0=WIf@3IfMT0l@3IfMT00240fMWI00<000000=WIf@3IfMT02@3IfMT00`000?l0fMWI\n0=WIf@3`0=WIf@008@3IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0?00fMWI000Q\n0=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0l03IfMT00240fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@3`0=WIf@008@3IfMT2000000X0fMWI00<0003o0=WIf@3I\nfMT0l03IfMT00240fMWI00<000000=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@3`0=WIf@008@3I\nfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0?00fMWI000Q0=WIf@030000003IfMT0\nfMWI00X0fMWI00<0003o0=WIf@3IfMT0k`3IfMT00240fMWI0P00000;0=WIf@030000o`3IfMT0fMWI\n0>l0fMWI000Q0=WIf@030000003IfMT0fMWI00X0fMWI00<0003o0=WIf@3IfMT0k`3IfMT000T0fMWI\n0P0000040=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0fMWI00X0fMWI\n00<0003o0=WIf@3IfMT0k`3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT02P3IfMT00`00\n0?l0fMWI0=WIf@3_0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI\n000000@0fMWI00<000000=WIf@3IfMT00P3IfMT3000000X0fMWI00<0003o0=WIf@3IfMT0k`3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000050=WIf@050000003I\nfMT0fMWI0=WIf@000000303IfMT00`000?l0fMWI0=WIf@3_0=WIf@00203IfMT010000000fMWI0=WI\nf@0000080=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`00\n0000fMWI0=WIf@0:0=WIf@030000o`3IfMT0fMWI0>l0fMWI00090=WIf@8000002P3IfMT2000000@0\nfMWI0P0000040=WIf@030000003IfMT0fMWI00X0fMWI00<0003o0=WIf@3IfMT0k`3IfMT00240fMWI\n0P00000;0=WIf@030000o`3IfMT0fMWI0>l0fMWI000Q0=WIf@030000003IfMT0fMWI00X0fMWI00<0\n003o0=WIf@3IfMT0k`3IfMT00240fMWI00<000000=WIf@3IfMT02`3IfMT00`000?l0fMWI0=WIf@3^\n0=WIf@008@3IfMT00`000000fMWI0=WIf@0;0=WIf@030000o`3IfMT0fMWI0>h0fMWI000Q0=WIf@03\n0000003IfMT0fMWI00/0fMWI00<0003o0=WIf@3IfMT0kP3IfMT00240fMWI0P00000<0=WIf@030000\no`3IfMT0fMWI0>h0fMWI000Q0=WIf@030000003IfMT0fMWI00/0fMWI00<0003o0=WIf@3IfMT0kP3I\nfMT00240fMWI00<000000=WIf@3IfMT02`3IfMT00`000?l0fMWI0=WIf@3^0=WIf@008@3IfMT00`00\n0000fMWI0=WIf@0;0=WIf@030000o`3IfMT0fMWI0>h0fMWI000Q0=WIf@800000303IfMT00`000?l0\nfMWI0=WIf@3^0=WIf@008@3IfMT00`000000fMWI0=WIf@0<0=WIf@030000o`3IfMT0fMWI0>d0fMWI\n000Q0=WIf@030000003IfMT0fMWI00`0fMWI00<0003o0=WIf@3IfMT0k@3IfMT00240fMWI00<00000\n0=WIf@3IfMT0303IfMT00`000?l0fMWI0=WIf@3]0=WIf@008@3IfMT2000000d0fMWI00<0003o0=WI\nf@3IfMT0k@3IfMT00240fMWI00<000000=WIf@3IfMT0303IfMT00`000?l0fMWI0=WIf@3]0=WIf@00\n2@3IfMT2000000@0fMWI0P0000040=WIf@8000001@3IfMT300000080fMWI00<000000=WIf@3IfMT0\n303IfMT00`000?l0fMWI0=WIf@3]0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000\n003IfMT0fMWI000000D0fMWI00D000000=WIf@3IfMT0fMWI0000000>0=WIf@030000o`3IfMT0fMWI\n0>d0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000000P3IfMT5\n00000080fMWI0`00000=0=WIf@030000o`3IfMT0fMWI0>`0fMWI00080=WIf@040000003IfMT0fMWI\n000000P0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000\n003IfMT0fMWI00d0fMWI00<0003o0=WIf@3IfMT0k03IfMT000P0fMWI00@000000=WIf@3IfMT00000\n203IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0000000<0fMWI00<000000=WIf@3I\nfMT03@3IfMT00`000?l0fMWI0=WIf@3/0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@800000\n0`3IfMT00`000000fMWI0=WIf@0=0=WIf@030000o`3IfMT0fMWI0>`0fMWI000Q0=WIf@8000003P3I\nfMT00`000?l0fMWI0=WIf@3/0=WIf@008@3IfMT00`000000fMWI0=WIf@0=0=WIf@030000o`3IfMT0\nfMWI0>`0fMWI000Q0=WIf@030000003IfMT0fMWI00h0fMWI00<0003o0=WIf@3IfMT0j`3IfMT00240\nfMWI00<000000=WIf@3IfMT03P3IfMT00`000?l0fMWI0=WIf@3[0=WIf@008@3IfMT00`000000fMWI\n0=WIf@0>0=WIf@030000o`3IfMT0fMWI0>/0fMWI000Q0=WIf@8000003`3IfMT00`000?l0fMWI0=WI\nf@3[0=WIf@008@3IfMT00`000000fMWI0=WIf@0>0=WIf@030000o`3IfMT0fMWI0>/0fMWI000Q0=WI\nf@030000003IfMT0fMWI00h0fMWI00<0003o0=WIf@3IfMT0j`3IfMT00240fMWI00<000000=WIf@3I\nfMT03P3IfMT00`000?l0fMWI0=WIf@3[0=WIf@008@3IfMT200000100fMWI00<0003o0=WIf@3IfMT0\n8`3IfMT70000ol00fMWI000Q0=WIf@030000003IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT08@3I\nfMT20000o`H0fMWI0`000?nn0=WIf@008@3IfMT00`000000fMWI0=WIf@0?0=WIf@030000o`3IfMT0\nfMWI0200fMWI00<0003o0=WIf@3IfMT02@3IfMT20000ok`0fMWI000Q0=WIf@030000003IfMT0fMWI\n00l0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0<0=WIf@80003o^P3IfMT00240\nfMWI0P00000@0=WIf@030000o`3IfMT0fMWI01h0fMWI00<0003o0=WIf@3IfMT03P3IfMT30000okP0\nfMWI000Q0=WIf@030000003IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT07@3IfMT00`000?l0fMWI\n0=WIf@0A0=WIf@80003o]`3IfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P00\n00040=WIf@030000003IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT0703IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@80003o]@3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT03`3IfMT00`000?l0\nfMWI0=WIf@0K0=WIf@80003o5`3IfMT30000ok<0fMWI00080=WIf@040000003IfMT0fMWI000000P0\nfMWI00@000000=WIf@3IfMT000001@3IfMT010000000fMWI0=WIf@3IfMT3000000l0fMWI00<0003o\n0=WIf@3IfMT06`3IfMT00`000?l0fMWI0=WIf@0H0=WIf@<0003o703IfMTS0000oa00fMWIA@000?lM\n0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000<0fMWI0P00\n00040=WIf@030000003IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT06P3IfMT00`000?l0fMWI0=WI\nf@0K0=WIf@<0003o4P3IfMT80000ob40fMWI4P000?mQ0=WIf@00203IfMT010000000fMWI0=WIf@00\n00080=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI\n0=WIf@0@0=WIf@030000o`3IfMT0fMWI01T0fMWI00<0003o0=WIf@3IfMT07P3IfMT60000o`D0fMWI\n2@000?nK0=WIf@002@3IfMT2000000X0fMWI0P0000040=WIf@<000000`3IfMT00`000000fMWI0=WI\nf@0@0=WIf@030000o`3IfMT0fMWI01P0fMWI00<0003o0=WIf@3IfMT0903IfMT70000oj<0fMWI000Q\n0=WIf@030000003IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0603IfMT00`000?l0fMWI0=WIf@3>\n0=WIf@008@3IfMT200000140fMWI00<0003o0=WIf@3IfMT05`3IfMT00`000?l0fMWI0=WIf@3?0=WI\nf@008@3IfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI01H0fMWI00<0003o0=WIf@3I\nfMT0c`3IfMT00240fMWI00<000000=WIf@3IfMT04@3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030000\no`3IfMT0fMWI0=00fMWI000Q0=WIf@030000003IfMT0fMWI0140fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`000?l0fMWI0=WIf@3@0=WIf@008@3IfMT200000180fMWI00<0003o0=WIf@3IfMT05@3IfMT0\n0`000?l0fMWI0=WIf@3@0=WIf@008@3IfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0d@3IfMT00240fMWI00<000000=WIf@3IfMT04P3IfMT00`000?l0\nfMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI0=40fMWI000Q0=WIf@030000003IfMT0fMWI0180fMWI\n00<0003o0=WIf@3IfMT04P3IfMT00`000?l0fMWI0=WIf@3B0=WIf@008@3IfMT2000001<0fMWI00<0\n003o0=WIf@3IfMT04P3IfMT00`000?l0fMWI0=WIf@3B0=WIf@008@3IfMT00`000000fMWI0=WIf@0B\n0=WIf@030000o`3IfMT0fMWI0140fMWI00<0003o0=WIf@3IfMT0d`3IfMT00240fMWI00<000000=WI\nf@3IfMT04P3IfMT00`000?l0fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI0=<0fMWI000Q0=WIf@03\n0000003IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@3D0=WIf@00\n8@3IfMT2000001@0fMWI00<0003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@3D0=WIf@008@3I\nfMT00`000000fMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT0e03I\nfMT00240fMWI00<000000=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@0>0=WIf@030000o`3IfMT0\nfMWI0=D0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000030=WIf@<00000103IfMT00`00\n0000fMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI00h0fMWI00<0003o0=WIf@3IfMT0e@3IfMT000P0\nfMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI000000<0fMWI00<000000=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@0=0=WIf@030000o`3I\nfMT0fMWI0=H0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000\n0P3IfMT010000000fMWI0=WIf@0000030=WIf@<00000503IfMT00`000?l0fMWI0=WIf@0<0=WIf@03\n0000o`3IfMT0fMWI0=H0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3I\nfMT000000P3IfMT3000000@0fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0;0=WI\nf@030000o`3IfMT0fMWI0=L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WI\nf@3IfMT000000`3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT02`3IfMT00`000?l0fMWI0=WIf@3G0=WIf@002@3IfMT2000000X0fMWI0P0000040=WI\nf@<000000`3IfMT00`000000fMWI0=WIf@0E0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT0f03IfMT00240fMWI0P00000F0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WIf@3I\nfMT0f03IfMT00240fMWI00<000000=WIf@3IfMT05P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0=T0fMWI000Q0=WIf@030000003IfMT0fMWI01H0fMWI00<0003o0=WIf@3IfMT01`3I\nfMT00`000?l0fMWI0=WIf@3I0=WIf@008@3IfMT00`000000fMWI0=WIf@0G0=WIf@030000o`3IfMT0\nfMWI00D0fMWI00<0003o0=WIf@3IfMT0fP3IfMT00240fMWI0P00000H0=WIf@030000o`3IfMT0fMWI\n00D0fMWI00<0003o0=WIf@3IfMT0fP3IfMT00240fMWI00<000000=WIf@3IfMT05`3IfMT00`000?l0\nfMWI0=WIf@040=WIf@030000o`3IfMT0fMWI0=/0fMWI000Q0=WIf@030000003IfMT0fMWI01P0fMWI\n00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@3L0=WIf@008@3IfMT00`000000fMWI0=WI\nf@0I0=WIf@@0003og`3IfMT00240fMWI0P00003m0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WI\nf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@009@3IfMT00`000000fMWI0=WIf@0T0=WIf@<00000;@3IfMT00`000000fMWI0=WIf@05\n0=WIf@800000203IfMT200000900fMWI000T0=WIf@030000003IfMT0fMWI02P0fMWI00<000000=WI\nf@3IfMT0:`3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT0SP3IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT3000000<0fMWI0`0000040=WIf@<0\n00000`3IfMT400000080fMWI0`0000000`3IfMT00000000000020=WIf@8000000P3IfMT300000004\n0=WIf@00000000000=WIf@<0000000<0fMWI0000000000000`000000103IfMT00000000000000003\n0=WIf@<000000`3IfMT500000080fMWI0P0000030=WIf@<000001@3IfMT00`000000fMWI0=WIf@03\n0=WIf@030000003IfMT0fMWI0080fMWI0P0000040=WIf@030000003IfMT0fMWI08h0fMWI000S0=WI\nf@030000003IfMT0fMWI00D0fMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@050000003IfMT0\nfMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@000000\n0P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0000000<0fMWI00<000000=WIf@3I\nfMT0100000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000\nfMWI0=WIf@0:0=WIf@030000003IfMT0fMWI0080fMWI00<000000=WIf@3IfMT00`3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00D000000=WIf@3IfMT0fMWI0000002@0=WIf@005P3IfMT80000\no`D0fMWI00<000000=WIf@3IfMT00P3IfMT3000000@0fMWI00<000000=WIf@3IfMT0103IfMT01@00\n0000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT01`000000fMWI0000003IfMT000000=WI\nf@0000001@3IfMT00`000000fMWI0=WIf@03000000<0fMWI0`0000050=WIf@030000003IfMT0fMWI\n00L0fMWI0`0000050=WIf@030000003IfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI00000006\n0=WIf@030000003IfMT0fMWI08h0fMWI000S0=WIf@030000003IfMT0fMWI0080fMWI100000030=WI\nf@030000003IfMT0fMWI00<0fMWI0P0000040=WIf@<000000`3IfMT200000080fMWI100000040=WI\nf@H0000000<0fMWI0000000000000`3IfMT300000080fMWI100000020=WIf@@00000103IfMT00`00\n0000fMWI0=WIf@070=WIf@@00000103IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI\n00D000000`3IfMT00`000000fMWI0=WIf@2>0=WIf@00903IfMT00`000000fMWI0=WIf@080=WIf@03\n0000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT04@3IfMT2000000030=WIf@000000000001H0\nfMWI00<000000=WIf@3IfMT03P3IfMT00`000000fMWI0=WIf@040=WIf@050000003IfMT0fMWI0=WI\nf@0000001P3IfMT00`000000fMWI0=WIf@2>0=WIf@009@3IfMT00`000000fMWI0=WIf@060=WIf@80\n00001P3IfMT00`000000fMWI0=WIf@030=WIf@800000103IfMT00`000000fMWI0=WIf@0<0=WIf@80\n000000@0fMWI0000000000000000503IfMT2000000l0fMWI00<000000=WIf@3IfMT01@3IfMT20000\n00P0fMWI0P00002@0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.756122`, -0.0114985`, 0.0222733`, 0.000472447`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[BoxData[RowBox[List["Plot", " ", "of", " ", "step", " ", "response", " ", "from", " ", "the", " ", "linear", " ", RowBox[List["actuator", "."]]]]], "NumberedFigure"], Cell[CellGroupData[List[Cell["Different behavior with different values of d.", "Subsection"], Cell[TextData[List["Assume that we have some freedom in choosing the damping in the translational spring-damper. A number of simulation runs show what kind of behavior we have for different values of the damping parameter ", StyleBox["d", Rule[FontFamily, "Courier"]], ". The Mathematica ", StyleBox["Table[]", Rule[FontFamily, "Courier"]], " function is used to collect the results from calls to ", StyleBox["Simulate[]", Rule[FontFamily, "Courier"]], " into an array ", StyleBox["res", Rule[FontFamily, "Courier"]], ". The array then contains the results from simulations of ", StyleBox["LinearActuator", Rule[FontFamily, "Courier"]], " with a damping of 2 and 14 in increments of 2, i.e. seven simulations are performed.\n"]], "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List["res", " ", "=", " ", RowBox[List["Table", "[", RowBox[List[RowBox[List["Simulate", "[", RowBox[List["LinearActuator", ",", RowBox[List["{", RowBox[List["t", ",", "0", ",", "4"]], "}"]], ",", "\n", " ", " ", " ", " ", " ", " ", RowBox[List["ParameterValues", " ", "->", RowBox[List["{", RowBox[List[RowBox[List["springDamper1", ".", "d"]], " ", "==", " ", "s"]], "}"]]]]]], "]"]], ",", " ", RowBox[List["{", RowBox[List["s", ",", "2", ",", "15", ",", "2"]], "}"]]]], "]"]]]], ";"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]]], "Input"], Cell["", "Text"], Cell[BoxData[RowBox[List[StyleBox["PlotSimulation", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["[", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], RowBox[List[StyleBox[RowBox[List["slidingMass1", ".", RowBox[List["s", "[", "t", "]"]]]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[RowBox[List["{", RowBox[List["t", ",", " ", "0", ",", " ", "4"]], "}"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["\n", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["      ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[RowBox[List["SimulationResult", " ", "->", " ", "res"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[RowBox[List["Legend", " ", "->", " ", "False"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]]], StyleBox["]", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]]], "Input"], Cell[BoxData[RowBox[List["Plots", " ", "of", " ", "step", " ", "responses", " ", "from", " ", "seven", " ", "simulations", " ", "of", " ", "the", " ", "linear", " ", "actuator", " ", "with", " ", "different", " ", "camping", " ", RowBox[List["coefficients", "."]]]]], "NumberedFigure"], Cell[TextData[List["\nNow assume that we would like to choose the damping parameter d so that the resulting linear actuator system behaves as closely as possible to the following \"reference\" first order system response, obtained by solving a first order ODE using the Mathematica differential equation solver ", StyleBox["NDSolve", Rule[FontFamily, "Courier"]], ":\n"]], "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List["resReference", " ", "=", " ", RowBox[List["NDSolve", "[", RowBox[List[RowBox[List["{", RowBox[List[RowBox[List[RowBox[List[RowBox[List["0.2", "*", RowBox[List[RowBox[List["y", "'"]], "[", "t", "]"]]]], "+", RowBox[List["y", "[", "t", "]"]], " "]], "==", " ", "0.05"]], ",", " ", RowBox[List[RowBox[List["y", "[", "0", "]"]], " ", "==", " ", "0"]]]], "}"]], ",", " ", RowBox[List["{", "y", "}"]], ",", " ", RowBox[List["{", RowBox[List["t", ",", "0", ",", "4"]], "}"]]]], "]"]]]], ";"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]], "Input"], Cell["\nWe make a plot to compare the step response we simulated first for a damping value d=2 and the response from the \[OpenCurlyDoubleQuote]reference\[CloseCurlyDoubleQuote] first order system.\n", "Text"], Cell[BoxData[RowBox[List[StyleBox["PlotSimulation", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["[", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], RowBox[List[StyleBox[RowBox[List["{", RowBox[List[RowBox[List["slidingMass1", ".", RowBox[List["s", "[", "t", "]"]]]], ",", " ", RowBox[List[RowBox[List["y", "[", "t", "]"]], " ", "/.", " ", "resReference"]]]], "}"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[RowBox[List["{", RowBox[List["t", ",", "0", ",", "4"]], "}"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["\n", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["    ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], RowBox[List[StyleBox["SimulationResult", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox["->", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox["resLinActFirst", Rule[FormatType, StandardForm], Rule[FontSize, 10.5`]]]]]], StyleBox["]", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]]], "Input"], Cell[BoxData[RowBox[List[RowBox[List["Comparison", " ", "plot", " ", "between", " ", "the", " ", "step", " ", "response", " ", "of", " ", "the", " ", "linear", " ", "actuator", " ", "and", " ", "the", " ", "\"reference\"", " ", "first"]], "-", RowBox[List["order", " ", RowBox[List["system", "."]]]]]]], "NumberedFigure"]], Open]], Cell[CellGroupData[List[Cell["Simulation of an integral", "Subsection"], Cell["Now, let us make things a little more automatic. Simulate and compute the integral of the square deviation from t = 0 to t = 4. This integral is a measure of the deviation from the \"reference\" response and is thus suitable as a goal function to be minimized.\n", "Text"], Cell[BoxData[RowBox[List[RowBox[List[StyleBox["Simulate", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["[", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], RowBox[List[StyleBox["LinearActuator", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[RowBox[List["{", RowBox[List["t", ",", "0", ",", "4"]], "}"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["\n", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["       ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[RowBox[List["ParameterValues", " ", "->", " ", RowBox[List["{", RowBox[List[RowBox[List["springDamper1", ".", "d"]], " ", "==", " ", "3"]], "}"]]]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]]], StyleBox["]", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]], StyleBox[";", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]]], "Input"], Cell[BoxData[StyleBox[RowBox[List["NIntegrate", "[", RowBox[List[RowBox[List[RowBox[List["First", "[", RowBox[List[RowBox[List["(", RowBox[List[RowBox[List["y", "[", "t", "]"]], " ", "/.", " ", "resReference"]], ")"]], " ", "-", " ", RowBox[List["slidingMass1", ".", RowBox[List["s", "[", "t", "]"]]]]]], "]"]], "^", "2"]], ",", " ", RowBox[List["{", RowBox[List["t", ",", " ", "0", ",", " ", "4"]], "}"]]]], "]"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]]], "Input"], Cell[TextData[List["We define a Mathematica function, ", StyleBox["f[a_]", Rule[FontFamily, "Courier"]], ", doing the same thing as above, but for different values of the spring-damper design parameter ", StyleBox["d", Rule[FontFamily, "Courier"]], ". The function contains two variables ", StyleBox["res", Rule[FontFamily, "Courier"]], " and ", StyleBox["t", Rule[FontFamily, "Courier"]], ". The expression ", StyleBox["(y[t]/.resReference)", Rule[FontFamily, "Courier"]], " retrieves the interpolating function representing the variable ", StyleBox["y[t]", Rule[FontFamily, "Courier"]], ", i.e. the reference response.\n"]], "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List[RowBox[List["f", "[", "a_", "]"]], " ", ":=", " ", RowBox[List["Module", "[", RowBox[List[RowBox[List["{", RowBox[List["res", ",", " ", "t"]], "}"]], ",", " ", "\n", "  ", RowBox[List[RowBox[List["res", "=", RowBox[List["Simulate", "[", RowBox[List["LinearActuator", ",", " ", RowBox[List["{", RowBox[List["t", ",", " ", "0", ",", " ", "4"]], "}"]], ",", RowBox[List["ParameterValues", " ", "->", " ", RowBox[List["{", RowBox[List[RowBox[List["springDamper1", ".", "d"]], " ", "==", " ", "a"]], "}"]]]]]], "]"]]]], ";", "\[IndentingNewLine]", RowBox[List["NIntegrate", "[", RowBox[List[SuperscriptBox[RowBox[List["First", "[", RowBox[List[RowBox[List["(", RowBox[List[RowBox[List["y", "[", "t", "]"]], "/.", StyleBox["resReference", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]]]], ")"]], "-", RowBox[List[StyleBox[RowBox[List[StyleBox["s", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], "lidingMass1"]]], ".", RowBox[List["s", "[", "t", "]"]]]]]], "]"]], "2"], ",", RowBox[List["{", RowBox[List["t", ",", "0", ",", "4"]], "}"]]]], "]"]]]]]], "]"]]]], ";"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]]], "Input"], Cell["", "Text"], Cell[TextData[List["We tabulate some results for ", StyleBox["2 \[LessEqual] d \[LessEqual] 10", Rule[FontSlant, "Italic"]], " into an array ", StyleBox["resDeviations", Rule[FontFamily, "Courier"]], ":"]], "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List["resDeviations", " ", "=", " ", RowBox[List["Table", "[", RowBox[List[RowBox[List["{", RowBox[List["d", ",", " ", RowBox[List["f", "[", "d", "]"]]]], "}"]], ",", " ", RowBox[List["{", RowBox[List["d", ",", " ", "2", ",", " ", "10", ",", " ", "0.5"]], "}"]]]], "]"]]]], ";"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]], "Input"], Cell["The tabulated values are interpolated into a Mathematica interpolating function. The default interpolation order is 3.", "Text"], Cell[BoxData[RowBox[List["fpre", "=", StyleBox[RowBox[List["Interpolation", "[", "resDeviations", "]"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]]]], "Input"], Cell["A plot of the interpolationg function fpre:", "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List["Plot", "[", RowBox[List[RowBox[List["fpre", "[", "d", "]"]], ",", " ", RowBox[List["{", RowBox[List["d", ",", " ", "2", ",", " ", "10"]], "}"]]]], "]"]], ";"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]], "Input"], Cell[BoxData[RowBox[List["Plot", " ", "of", " ", "the", " ", "deviation", " ", "function", " ", "for", " ", "finding", " ", "the", " ", "value", " ", "of", " ", "the", " ", "damping", " ", "parameter", " ", "that", " ", "gives", " ", "a", " ", "minimum", " ", "deviation", " ", "from", " ", "the", " ", "reference", " ", RowBox[List["response", "."]]]]], "NumberedFigure"], Cell[TextData[List["\nThe minimizing value of ", StyleBox["d", Rule[FontFamily, "Courier"]], " can be computed using ", StyleBox["FindMinimum", Rule[FontFamily, "Courier"]], "."]], "Text"], Cell[BoxData[StyleBox[RowBox[List["FindMinimum", "[", RowBox[List[RowBox[List["fpre", "[", "d", "]"]], ",", " ", RowBox[List["{", RowBox[List["d", ",", " ", "4"]], "}"]]]], "]"]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]], "Input"], Cell["A simulation with the optimal parameter value:", "Text"], Cell[BoxData[RowBox[List[RowBox[List[StyleBox["Simulate", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["[", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], RowBox[List[StyleBox["LinearActuator", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[RowBox[List["{", RowBox[List["t", ",", " ", "0", ",", " ", "4"]], "}"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[",", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox[" ", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["\n", Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]], StyleBox["           ", Rule[FormatType, StandardForm], Rule[FontSize, 11]], StyleBox[RowBox[List["ParameterValues", " ", "->", " ", RowBox[List["{", RowBox[List[RowBox[List["springDamper1", ".", "d"]], " ", "==", " ", "5.28642"]], "}"]]]], Rule[FormatType, StandardForm], Rule[FontSize, 11]]]], StyleBox["]", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]], StyleBox[";", Rule[FormatType, StandardForm], Rule[FontSize, 11]]]]], "Input"], Cell["\nA plot comparing the \[OpenCurlyDoubleQuote]reference\[CloseCurlyDoubleQuote] first order system and linear actuator system responses together with a plot of the squared deviation amplified by a factor 100.\n", "Text"], Cell[BoxData[StyleBox[RowBox[List[RowBox[List["PlotSimulation", "[", RowBox[List[RowBox[List["{", RowBox[List[RowBox[List["slidingMass1", ".", RowBox[List["s", "[", "t", "]"]]]], ",", " ", "\n", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", RowBox[List[RowBox[List["y", "[", "t", "]"]], " ", "/.", " ", "resReference"]], ",", " ", "\n", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", "100", "*", RowBox[List[RowBox[List["(", RowBox[List[RowBox[List["slidingMass1", ".", RowBox[List["s", "[", "t", "]"]]]], " ", "-", " ", RowBox[List["(", RowBox[List[RowBox[List["y", "[", "t", "]"]], " ", "/.", " ", "resReference"]], ")"]]]], ")"]], "^", "2"]]]], "}"]], ",", "\n", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", RowBox[List["{", RowBox[List["t", ",", " ", "0", ",", " ", "4"]], "}"]], ",", " ", RowBox[List["Legend", " ", "->", " ", "False"]]]], "]"]], ";"]], Rule[FormatType, StandardForm], Rule[FontFamily, "Courier New"], Rule[FontSize, 10.5`]]], "Input"], Cell[BoxData[RowBox[List["Comparison", " ", "plot", " ", "of", " ", "the", " ", "first", " ", "and", " ", "second", " ", "order", " ", "system", " ", "step", " ", "responses", " ", "together", " ", "with", " ", "the", " ", "squared", " ", RowBox[List["error", "."]]]]], "NumberedFigure"]], Open]]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[1014, 1009]], Rule[WindowMargins, List[List[45, Automatic], List[Automatic, 0]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]