List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Classes for Connectors, Places and Transitions", "Title"], Cell[CellGroupData[List[Cell["Petri Nets", "Section"], Cell[TextData[List[StyleBox["Petri Nets", Rule[FontSlant, "Italic"]], " ar", "e bipartite graphs that only allow connection of different kinds of nodes, places to transitions or transitions to places. The Modelica type system can ensure that only legal connections are made if we define two different connector classes, one class ", StyleBox["PTPort", Rule[FontFamily, "Courier"]], " for for places to transitions-with class prefix and variable suffixes ", StyleBox["PT", Rule[FontFamily, "Courier"]], ", and another class TP", StyleBox["Port", Rule[FontFamily, "Courier"]], " for transitions for places - with class prefix and variable suffixes TP. Since variable names are different, the connector types are different in the Modelica type system, even though the connector classes in both cases represent the same kind of variables:state and firing information. \n\nIn order to have slightly different connector icons for the input version (filled triangle) and the output version (unfilled triangle), we also define otherwise identical input and output versions of these connector classes, see below. There is also a ", StyleBox["ConditionPort", Rule[FontFamily, "Courier"]], " connector class for specifying an optional input condition to a ", StyleBox["Transition", Rule[FontFamily, "Courier"]], ". "]], "Text"], Cell["connector PTPort \t\"Connecting Places to Transitions\"\n  Boolean sPT    \t\"state information\";\n  Boolean fPT    \t\"firing information\";\nend PTPort;", "ModelicaInput"], Cell["", "Text"], Cell["connector PTPortIn  = PTPort;  \t\t\t// Input version of PTPort", "ModelicaInput"], Cell["", "Text"], Cell["connector PTPortOut = PTPort;  \t\t\t// Output version of PTPort", "ModelicaInput"], Cell["", "Text"], Cell["connector TPPort \t\t\"Connecting Transitions to Places\"\n  Boolean sTP   \t\t\"state information\";\n  Boolean fTP  \t\t    \"firing information\";\nend TPPort;", "ModelicaInput"], Cell["", "Text"], Cell["connector TPPortIn  = TPPort; \t\t\t // Input version of PTPort", "ModelicaInput"], Cell["", "Text"], Cell["connector TPPortOut = TPPort; \t\t\t // Output version of PTPort", "ModelicaInput"], Cell["", "Text"], Cell["connector ConditionPort \t\t\"Optional condition to a Transition\"\n  Boolean c(start=true);\nend ConditionPort;", "ModelicaInput"], Cell[TextData[List["\nWe now define ", StyleBox["Place", Rule[FontFamily, "Courier"]], " and ", StyleBox["Transition", Rule[FontFamily, "Courier"]], " classes.\n"]], "Text"], Cell["class Place\n  parameter Integer nIn  = 1 \t\t \"Number of input transition ports\";\n  parameter Integer nOut = 1 \t\t \"Number of output transition ports\";\n  TPPortIn  inTrans[nIn]   \t   \t\t \"Vector of input transition connectors\";\n  PTPortOut outTrans[nOut]\t\t     \"Vector of output transition connectors\";\n  Boolean   state;\n  Boolean   ostate;\n  Boolean   d1, d2;  \t\t\t\t // Dummy variables\n equation\n  ostate = pre(state);  \t\t   \t // ostate is the current state\n      \t\t\t\t\t\t\t\t // Report state to input transitions, vector equation\n  cat(1,inTrans.sTP,{d1}) = cat(1,{ostate},inTrans.sTP or inTrans.fTP);\n      \t\t\t\t\t\t\t\t // Report state to output transitions, column matrix equation\n  [outTrans.sPT,{d2}] = [{ostate}; outTrans.sPT and not outTrans.fPT];\n      \t\t\t\t\t\t\t\t // Compute new state for next Petri net iteration\n  state = (ostate and not anyTrue(outTrans.sPT)) or (anyTrue(inTrans.sTP));\nend Place;", "ModelicaInput"], Cell["", "Text"], Cell["class Transition\n  ConditionPort condition( start = true ) \t\t\"Optional condition\";\n  parameter Integer nIn  = 1;\n  parameter Integer nOut = 1; \n  PTPortIn  inPlaces[nIn]   \t\t\t\t\t\"Vector of input place connectors\";\n  TPPortOut outPlaces[nOut] \t\t\t\t\t\"Vector of output place connectors\";\n  Boolean   fire;\n equation\n  fire = condition and allTrue(inPlaces.sPT) and not anyTrue(outPlaces.sTP);\n      \t\t\t\t\t\t\t\t\t\t\t// Report firing info to input places, vector equation\n  inPlaces.fPT = fill(fire,nIn);\n     \t\t\t\t\t\t\t\t\t\t    // Report firing info to output places, vector equation\n  outPlaces.fTP = fill(fire,nOut);\nend Transition;", "ModelicaInput"], Cell[TextData[List["\nWe also need utility ", StyleBox["Boolean", Rule[FontFamily, "Courier"]], " reduction functions ", StyleBox["anyTrue", Rule[FontFamily, "Courier"]], " and ", StyleBox["allTrue", Rule[FontFamily, "Courier"]], ".\n"]], "Text"], Cell["function anyTrue  \"True, if at least one element is true in a vector\"\n  input  Boolean inp[:];\n  output Boolean result;\n algorithm\n  result := false;\n  for i in 1:size(inp,1) loop\n    result := result or inp[i];\n  end for;\nend anyTrue;", "ModelicaInput"], Cell["", "Text"], Cell["function allTrue  \"True, if all elements are true in a vector\"\n  input  Boolean inp[:];\n  output Boolean result;\n algorithm\n  result := true;\n  for i in 1:size(inp,1) loop\n    result := result and inp[i];\n  end for;\nend allTrue;", "ModelicaInput"], Cell[TextData[List["\nAll classes and functions are collected into a package called ", StyleBox["NormalPetriNet", Rule[FontFamily, "Courier"]], ":\n"]], "Text"], Cell["encapsulated package NormalPetriNet\n  connector PTPort ...;\n  connector PTPortIn ...;\n  connector PTPortOut ...;\n  connector TPPort ...;\n  connector TPPortIn ...;\n  connector TPPortOut ...;\n  connector ConditionPort ...;\n  class Place ...;\n  class Transition ...;\n  function anyTrue ...;\n  function allTrue ...; \nend NormalPetriNet;", "Program"]], Open]], Cell[CellGroupData[List[Cell["A Modelica Petri Net Model and Simulation of a Job Shop System", "Section"], Cell["We now return to our job shop system example. Its Petri net graph is shown once more below in Figure 1, with the difference that the token has advanced, representing the condition C3 that a job is currently being processed.\n", "Text"], Cell[GraphicsData["Metafile", "CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@0004cd0@0006`0000000000P000302003F0000\n00000000001X<`00na@00215CDH00040m4`00:@1000;0000000000000000000000@00005003`0000\n@0400000000000000000082Y0`00hP@0AP0000@J003h6@00AdA9@`4008000`007Lb7JP00003P6@00\n0@090003l0`000`0J0000000500002H63`0N0?ooool401@0001GKg9T3P1=JF=bKg=_IW@PEfmbI0D0\n000;0P000005000030;;0?4170000?/240070000002l0P0000010P8RDgUcM6E]000000X000040000\n003ooooo0@000000<0040000;@4000D000020@40000500002@80000270000?/2l?l00000002@0@00\n0004@00BE6U]IG<PCVEg859_KF5^0;iRmGO7H_Eg0@000000<0040000;@4100<0000N00L0000F1</0\nl@400000400002H63`0F0?ooool00?ooooooooool`400<`0000300007P050000;P4000001@0000X2\n000000D000090P00000500000@;oool01`000?`20@00000000040000;@4200P0003j0PD00@000000\n00040000;@4300@000030@P01`000184b`3;0?41lP4500002`8000001@0000`2m@@Y31@0000V1Pl0\n7P3ooooo100D0000EfmbI0h0CFUSLVmcKfId85M_LV@5000030;e12T<70000?/250090000002l0P00\n00040P8RDgUcM6E]07M00000?`D:>KiRmGO7H_Eg0@000000<0040000;@4400D000020@40000L0000\nn`:]o`0000000901000000A00291LVUQK03eMd00001F20X2_V;eMlMRmGL10000000`00@0000]0@D0\n1@0000T200000PD0000;0P00000=0000<P[611@60@040000000Y3?D486D]00@0000]0@80100002d1\n0`0700006`BX1400H`@200L0003l0P00oooo0000100002d11P080000nP80000000000000100002d1\n1`0300007P0700005PBW1400H`@200@0000]0@D01@0000810@0000@0000]0@D04000038:G@@200<0\n100206<4?`2W120P804G01L05`0L0000n`:eo`0000000901000000A0019DJFeULb1>IGLPDVm]HFh0\n_V;eMlMRmGL10000000`00@0000]0@P03@00038:I0A70040100206<4?`2W12008P0400009`7oo`@0\n000]0@80100002d10`0700006`@71B<<`PCT2`@0000]0@H0100002d11`0300007P0700005P@61B8<\n`PCT2`@0000]0@D01@0000810@0000@0000]0@D01@0000T200000Q00000b2[`4i0/300@0i0_2124<\n1PDP82015`0G01L0100002d1200=0000<P[312T<0@040>@;`P@Q30H5804R00@0000W0Ooo100002d1\n0P040000;@4300L0000K10h5@0391040100002d11P040000;@4700<0000N00L0000F10d5?`391040\n100002d11@0500000P410000100002d11@0500002@8000024000038:``@100<010010<T4?P0=1B0P\n804G01L05`040000;@4800d0000b2/X4AP0100@00@3913h03@DP0280100002L1ool40000;@4200@0\n000]0@<01`0001/4^`230A80?`040000;@4600@0000]0@L00`0001h01`0001H4^P230A80?`050000\n0P41000070000?/2]Ol00000002@0@000004@00R@G9YHF`0mGM00000n@L:aKiRmGO7H_Eg0@000000\n<0040000;@4900D000090P00008I0000<PXE03l02@0403l04P220KX0AC4j844PJVmR0380:P0E01D0\n<P0E0140:@0Z00d0000b2QD0L04100@0?`0B0881^P0P0BL01@0000810@0000@0000]0@T05P00038:\nJ`0o00L0100o0180PP6j065bLVUfIG<0:P0I01T04@0U02X09P040000;@4800d0000b2V/08@4100@0\n?`0B0881^P0P0B40100002L1ool40000;@4200@0000]0@<01`0001/4SP;e0]L1Y`440000;@4600@0\n000]0@L00`0001h01`0001H4S0;e0]L1Y`4500000P410000100002d12@0500002@8000026@00038:\nfP6W0@T0102W0ML1m0:<0T<b>R1186Y_HP0f02X05@0E03805@0A02X0:P0=0000<P[J0Md20@040:L1\ne`7d0X`2804W00D000020@4000040000;@4901X0000b2Rh2Y`4:00@0Y`7G0O@2S09YLb1gHFUdJFiW\n4@0V01D0=@0Z01405@0A02T0:P040000;@4800d0000b2Rh2g08100@0Y`7G0O@2S08P0B40100002L1\nool40000;@4200@0000]0@<01`0001/4b01D1A80E@<40000;@4600@0000]0@L00`0001h01`0001H4\na`1D1A80E@<500000P410000100002d12@0500002@8000026@00038:5@1E0`T0101E0a80D`G704Db\n>R1CM65bM6Lb02X05@0E03805@0Z01T05@0=0000<PXE07X40@0405D34P1C1LL0804W00D000020@40\n00040000;@490200000b2VT0E@<>00@0E@<B05<5a`1`LVmSIG=cJFiW86Y_HRX06@0Z02H0:@0V02D0\n4@0Z02X0500A02X0:P040000;@4800d0000b2VT0>PD100@0E@<B05<5a`0P0B40100002L1ool40000\n;@4200@0000]0@<01`0001/4a@1]2180c0H40000;@4600@0000]0@L00`0001h01`0001H4a01]2180\nc0H500000P410000100002d12@0500002@8000027@00038:5@3<1P`0103<1Q80K0S404Dc>R1186Y_\nHR1YLc80:P0E01D0<P0E0140:@0Z01D0400V00d0000b2QD0B0P100@0c0HB06`8a00P0BL01@000081\n0@0000@0000]0@T06@00038:J@3<1PT0103<1Q80K0S406=_KG1/IGAUI00V02X0?P0Z0140:P0D02X0\n:P040000;@4800d0000b2VT09`P100@0c0HB06`8a00P0B40100002L1ool40000;@4200@0000]0@<0\n1`0001/4RP;d2]L1L0P40000;@4600@0000]0@L00`0001h01`0001H4R@;d2]L1L0P500000P410000\n100002d12@0500002@800002:@00038:fP5`21@0101`2=L1l`Z90T<d>R1186Y_HR1YLb1gHFUdJFiW\n=P0Z01D05@0b01D04@0Z02T05@0A02D05@0e02X04@0E0140:P0Z00d0000b2]X1k`X100@0L0SG0O<:\nR@8P0BL01@0000810@0000@0000]0@T07@00038:;P9`20`0101`2=L1l`Z90WA_869U86meM71eM1D0\n:P0E02X0:P0D02X0:P0E02T0:P0E00@0000]0@P03@00038:;P;m2@40101`2=L1l`Z90R018@040000\n9`7oo`@0000]0@80100002d10`0700006`BA0Pl7d07>10@0000]0@H0100002d11`0300007P070000\n5PB@0Pl7d07>10D000020@4000040000;@4900D000090P00008P0000<P[C0Lh43P040<h4d04>1i02\n@c<j85AXIB1ZKf8PJG<f02X05@0E02d0:P0Z01D04@0Z02X0500A02H03@00038:d`6N1P40103>1=01\n3PN@0R019`0500000P410000100002d12@0R0000<PXX0/h43`040<h4d04>1i02HVEYKVLPL79_HfEc\nLfETM2X0:P0A02X0:@0E02X06@0Y02H0:@0V02H0:@0Z00@0000]0@P03@00038::0;e1P40103>1=01\n3PN@0R018@0400009`7oo`@0000]0@80100002d10`0700006`C30<d;2@2j2@@0000]0@H0100002d1\n1`0300007P0700005PC20<`;2@2j2@D000020@4000040000;@4900D000090P00008R0000<PX<0;X9\n3`040;X92@3;2l80AC@j844PIVU^JG=XIF@PM380:P0E01D0<P0E01D04@0Y01409@0Z02X0:P0E00d0\n000b2P`0W`/100@0^PT90</;`P0P0BL01@0000810@0000@0000]0@T08`00038:H@2j2A00102j2@T0\nb`_206Y_HR1YLb1TIFaYMVEbIF@A02X0:P0E01009P0E02T0:P0A01409@0Z01T0:@0Z00@0000]0@P0\n3@00038:H@2_2`40102j2@T0b`_202018@0400009`7oo`@0000]0@80100002d10`0700006`C414D7\n2PB410@0000]0@H0100002d11`0300007P0700005PC314@72PB410D000020@4000040000;@4900D0\n00090P00008O0000<PX=18@43@0408@42PA31l<4@c4j85AXIB1ZKf8PLbdf02X05@0E02d0:P0Z01D0\n4@0Z02X0500V01@0000b2Pd4@`H600@0Q0@:14<7``AXKg0PJG<Z02T0:P0E01009P0=0000<PX=10/7\n0@0408@42PA31l<4800W00D000020@4000040000;@4902`0000b2V84Q0@F00@0Q0@:14<7``AYI6aU\n;21gHFUdJFiW86I_LR1gKg9[4@0Z0140:P0D01D0=@0Z01405@0A02X0:P0E01@0:P0I01D0=@0Z01T0\n9P040000;@4800d0000b2V84;0L100@0Q0@:14<7``@P0240100002L1ool@00009PH?01H0ooooo`00\n``0002d1000W0P00GP4000P0003j0P001000000000040000;@4:00@0000]0@80200002D30P3604D1\nn`550@@0000]0@<010000?012P070000o0800000000000@0000]0@X02P0002@30`3i0E/190960OT1\n<0440000;@4700@0000]0@H010000?012P0800009PH?00H0ooooo`40400002H63`0F0?ooool00<P2\n000]0@00;0@005h100080000nP8000@000000000100002d12P040000;@4200P0000U0`80b`950@04\nA@440000;@4300@0003`0@X01`000?`20000000000040000;@4:00X0000T0`<0oP=K0BP4AP7n0c01\n100002d11`040000;@4600@0003`0@X0200002H63`060?ooool10100000V1Pl05P3ooooo000U1000\n;@4008T5001N0@0020000?X200040000000000@0000]0@X0100002d10P0800009@<202P4A@5M1DD1\n100002d10`040000l04:00L0003l0P0000000000100002d12P0:000090<305/5F`661DH1F`D`0@@0\n000]0@L0100002d11P040000l04:00P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00=@H002d1\n002I1`00GP4000P0003j0P001000000000040000;@4:00@0000]0@80200002D30P0h1TD1K@M50@@0\n000]0@<010000?012P070000o0800000000000@0000]0@X02P0002@30`1[1e/1UPM60F/7<0440000\n;@4700@0000]0@H010000?012P0800009PH?00H0ooooo`40400002H63`0F0?ooool009P7000]0@00\no@P005h100080000nP8000@000000000100002d12P040000;@4200P0000U0`80V`M50M48A@440000\n;@4300@0003`0@X01`000?`20000000000040000;@4:00X0000T0`<0c`QK0OX8AP7?2301100002d1\n1`040000;@4600@0003`0@X0200002H63`060?ooool10100000V1Pl05P3ooooo002N2@00;@40008;\n001N0@0020000?X200040000000000@0000]0@X0100002d10P0800009@<20:49A@7F2TD1100002d1\n0`040000l04:00L0003l0P0000000000100002d12P0:000090<30=@:F`7n2TH1e0X`0@@0000]0@L0\n100002d11P040000l04:00P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00aP000>T0003?0000\n[@4000P0003j0P001P00000000040000;@4:00@0000]0@80200002D30P3:0>d0bP2X0@@0000]0@L0\n10000?012P040000;@4600P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00:P@00>T0000c1000\n[@4000P0003j0P001P00000000040000;@4:00@0000]0@80200002D30P0^1>d0;PBX0@@0000]0@L0\n10000?012P040000;@4600P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00U0L00>T0002M1`00\n[@4000P0003j0P001P00000000040000;@4:00@0000]0@80200002D30P2H1nd0V0NX0@@0000]0@L0\n10000?012P040000;@4600P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`001@/00>T0000>2`00\n[@4000P0003j0P001P00000000040000;@4:00@0000]0@80200002D30P092nd02@^X0@@0000]0@L0\n10000?012P040000;@4600P0000V1Pl01P3ooooo0@070000o0800?ooo`0000@0000]0@X020000?X2\n0006000000000P@0000]0@/01`0001P4W0790_/0:0840000;@4600@0003`0@X0100002d11`040000\nl04;00L0003l0P00oooo0000100002d12P080000nP8000H000000002100002d12`07000060CZ0cD6\nB@>D1@@0000]0@H010000?012P040000;@4700@0003`0@/01`000?`2003oool000040000;@4:00P0\n003j0P001P00000000840000;@4;00L0000H19`1=@Kk09@5100002d11P040000l04:00@0000]0@L0\n10000?012`070000o0800?ooo`0000@0000]0@X020000?X20006000000000P@0000]0@/01`0001P4\nW06O2O/0oPP40000;@4600@0003`0@X0100002d11`040000l04;0100000V1Pl05P3ooooo000m1P00\nO@400348002]0`0020000?X200040000000000@0000]0@X0100002d10P1R00009@<_09X7P06[1h01\n_0N20L@7Q07=1hL1e@N;0Md7T@7^1ih1m`NU0@08[@492;P140S40AL8dP4L2>@17PS^0B48nP4S20H2\n9@PD0RT8<@8/2502;@Q_0Rd8SP8[2:X2:@Rg0RH8``8O2=X25@S_0PT80`?k1aH3k0LW0m`7=`?<1dH3\n_0MC0j`7GP>K1fP3RPM_0gT7MP=V1g/3DPN00cd7Q0<V1hP34PN;0o`6SP?U1Y03cPJA0id6T`=Z1Y@3\n100002d10`040000l04:00L0003l0P0000000000100002d12P0:000090<306/6O`=01YH3K0JZ0`@0\n000]0@L0100002d11P040000l04:00P0000V1Pl01P3ooooo0@0@00009PH?01H0ooooo`00L@<00781\n002I1@00VP<000P0003j0P001000000000040000;@4:00@0000]0@80J00002D3<P2D1ID3J`FE0d05\nU0<D1I83o0BA0n@4S`?:18d3[@B<0hl4RP=`18P3DPB60cH4PP<K17d33`Aj0`@4M`?_0g03g@=X0l/3\nGP>l0e@3[P=80j43>`>F0bd3S0<M0hD330=o0oP2N`?R0WT3b`9f0i`2MP>50WD3L09e0e/2M@=50WL3\n6P9i0`H2O0?c0Gl3hP620m<1QP?70HT3_@6=0kD1T@>_0IH3Z@6L0j@1Y0>P0Jh3V`6f0iP1_`>D0M<3\nS`7Z0h`10`B90@@0000]0@<010000?012P070000o0800000000000@0000]0@X02P0002@30`031:01\n;0B80@04M@440000;@4700@0000]0@H010000?012P0800009PH?00H0ooooo`401`000?`200000000\n00040000;@4:00P0003j0P001P00000000840000;@4;00L0000H16<1o0Dd0Ld5100002d11P040000\nl04:00@0000]0@L010000?012`040000;@4400@0003`0@D010000?0120040000l04900L0003l0P00\noooo0000100002d11@080000nP80000000000000100002d1200400009`7oo`P0000V1Pl01P3ooooo\n0@0400009`7oo`@0000]0@000`0000004@0000`0000800002`000100000a0P00e`0000T0000@0000\n<@800=L0000:000040000000000000002@000100003a0@00b`000580001`0@000@00010000070000\n00000000002l0P00000000420R9C07T0L`1d06D0K@00000000000000000000000000000000000000\n0000000000000000000000000000000000000002000001<0:?PJ07P14`3k1000D;KlMaJeo7L]]Oag\n00000800000@JQX0000C02P<4`0N0000j84G09cc4P1@C1X0N04C07P14`2PI1X0ooooofSe4P101Q<0\n:0`C000000000000EPPQ0oooool001<0D4`J07P14`000?ag000C002`5P0001<0@0HC000300101Q<0\n=?@B03cd4P1c_Oag00@007P34`1h0a<020000040001h0a<0000C051<6P1h0A<0N04C04Q<6P1@]_ag\n5[GlMbfeo7L00000000000T0000@JQX0Z?@B07an6P32`?Ag0@0006Af00P000009@0000`000010000\n4P0000`000010000600000`000000002DP00070100020000l?ooo`00000000000000090100000000\n14004U@0J@1]06D0L`0P04h0I@1g0200DP1_06d0H@1^000000000000000000000000000000000000\n00000000000001<0000C051<6P1h0A<0N04C04Q<6P1@]_ag5[GlMbfeo7L00000000000T0000@JQX0\nZ?@B07an6P32`?Ag0@0006Af00P00000N04C0:1T6P3oooooJ?DB04064`0X31<000000000001F2243\nooooo`004`1@C1X0N04C0000o7L001<00;0F00004`101Q<000<004064`0dm180??@B07>mo7L01000\nN0<C07P34`0800000@0007P34`0001<0D4`J07P14`1h0A<0B4`J052fo7LF]Oag;KGlMbD0000@JQX0\nR?@B03l5001D`?Ag0@0001T0/04o1@XjI7H02000000U000030000080000Q0000200001h0000H0000\n00000000003a0@00b`000240000800005P0000`000000000N000010000000000000001P0000<0000\n000001T0000<0000oooo02D0000<00001@00P2D0000<00002000P1l0000H0000lP400?41003;0000\nb`0000X0000@00000000000000090000400002T<003e10002@000100000Y3000m@@00580001`0@00\n0`0001@00009000000000000002l0P00000000@20R9C07T0L`1d06D0K@0000000000000000000000\n000000000000000000000000000000000000000000000000000C00004`1@C1X0N04C07P14`18C1X0\nD;KlMaJeo7L]]Oag9@00011Z6P28m180?`D005C0m7L100006@2`0Cl52SYTMP08000007P14`2PI1X0\nooooofSe4P101Q<0:0`C000000000000EPPQ0oooool001<0D4`J07P14`000?ag000C002`5P0001<0\n@0HC000300101Q<0=?@B03cd4P1c_Oag00@007P34`0@8_Aga?3mO`3PoGl100000000000000000000\nW0<00010002L0`AH0@0006Cd4P0U000046XJ00T0000@JQX0Z?@B0?ao6P32`?Ag0@0006Af00P00000\n9@0000`0000300004P0000`000010000DP00070100040000[Oooo`00000000000000090100000000\n14008T40LP1Y0640K000000000000000000000000000000000000000000000000000000000000000\n00000000000000000000000000000000W0<00010002L0`AH0@0006Cd4P0U000046XJ00T0000@JQX0\nZ?@B0?ao6P32`?Ag0@0006Af00P00000N04C0:1T6P3oooooJ?DB04064`0X31<000000000001F2243\nooooo`004`1@C1X0N04C0000o7L001<00;0F00004`101Q<000<004064`0dm180??@B07>mo7L01000\nN0<C010Rm7O4l?eo0>3mO`400000000000000000002L0`00040009`315P10000I?@B02D0000@JQX0\nR?@B00<8001D`?Ag0@0001T0/04320YhI7H02000000U0000300000@0000H000030000000008:0000\n4000000000000000E00005@0000I0@00c`000241003F00000@00069VQd0`RXM050H00<H400010000\nC00000@000000000000002T<003e1000D0000200000]00009@0000`0000500209@0000`000080020\n:`0001P000020000H`@003/0002R10009`0001P00005000000000?ooo`0000009@0000`000050000\n9P0001`0000600000000000000000000000002D0000<00001P000240000800007P0001P000020000\nH`@00400002W10009@0000`0000400004P0000`0000100009@0000`000040000E000060000000000\n_P0000X0003900000@00069VQd0`RXM00P0005d400030000C00000@000020000H`@003l0002W1000\nE0000200800P00005`0001L0000G0000DP00070100070000]Oooo`00000000000000090100000000\n14004U@0J@1]06D0L`0P04h0I@1g0200DP1_06d0H@1^000000000000000000000000000000000000\n00000000000000000000000000000000W0<00010002L0`AH0@0006Cd4P0U000046XJ08Sd4P3<la80\n00000>Dhn7L001<0B:8F0000002Xla80R0HC05cd4P2E:oQg83WhMooooom/m180@;OlMaP74`1@XQH0\nD:8F05:16P1@XQH000000000002L0`00040009`315T100007?@B0?0am7L001<0d;TJ0040002L0`AI\n15T0000Rm7L300000>3mO`400000000000000000000001<00`00002R5P0H0000I?@B004000030000\ng?<B0600002Tm180UB_hMaPfn7OoooooI7H02000000U0000300000L0001D0000E000000000000000\nooooooooool10000HVJ7@32:Qd170000I0@00040001<000010000080001S1000?`000:L4001@0000\n80000280000R000030000?oooolU0000300000D0080U0000300000P0080[000060000>@;00321000\n7P`00045000U0000300000D0000U0000300000H0000Q0000200001h0000H0000i0/00<84000R3000\n1PD002D0000<000010000180000<00000@0002D0000<0000100001P0000<000000000U@0001P0000\n9P800<h0000`0P00eP000040001RIXM0<8Z7@>@;002l10000`0004`000040000i0/00<84000Q3000\n1PD005@0000P0200800001L0000G00005`0002D0000<00001`0005@0001D000000000000003ooooo\nooooo`40001RIXM0<8Z7@2T<003310000@0004`000040000i0/00<84000Q30001PD00500000P0000\n8P000280000<0000ooooobD0000<00001@00P2D0000<00002000P2/0000H00000@000<T4000k0000\n20D002D0000<00001@0002D0000<00001P000240000800007P0001P000010000b@@003l0000=1@00\n9@0000`0000400004P0000`0000100009@0000`000040000600000`000000002E000060000000000\nd00000X0003F00000@00069VQd0`RXM00@000<<400030000C00000@000010000b@@003h0000=1@00\nE0000200800P00005`0001L0000G00009@0000`000070000E00005@00000000000000?oooooooooo\n0@00069VQd0`RXM0AP000<X400010000C00000@000010000b@@003h0000=1@00D0000200000R0000\n8P0000`0003ooooo9@0000`0000500209@0000`000080020:`0001P0000o00004P0007h1002e0000\n9@0000`0000500009@0000`0000600008@0000P0000N0000600003l0000B0000P`400;X0000B0000\n30000040001B0000L04000P0002eoooo0000000000000000T04000000004@00R@@1b06T0H@1/0000\n00000000000000000000000000000000000000000000000000000000000000000000000000000000\n00000000000001<00`00002R5P0H0000I?@B004000030000g?<B0<cc4P000000iCShM`004`2heQD0\n00000:Sc4P281Q<0G?@B09D[n7LP>OQgooooofcd4P10]oagj0HC0<3F5@30eQD048<J0<3F5@000000\n00000?d30000@000o@<4m@40000Lm180l37dM`004`0L_aX00@000?d31?D4m@0042;dMlC`oGl0h?eo\n0@00000000000000000009`30000@000W0<4H040001Tm180l37dMi`3161F2243W0<4H0000004H000\nP?@B093Bm7MTMP08000002D0000<0000200001P0000<000000000U@0002400002`0000@000120000\n4`000040001RIXM0<8Z7@3l0000E00002@0004`000040000?`00018000220@00^P00060000150340\n>P0P0440801Z06l0HP000380000Z00005@0001D0000b00005@000140000Y0000:P0005@0001D0000\n@P0000@0001500004`000040001RIXM0<8Z7@701000E00000@0004`000040000?`00018000220@00\n^P000500000P00009`000180000<00000@0002D0000<0000200005@0001h00002`000180000c0000\n7`000040001RIXM0<8Z7@3l0001[00001`0004`000040000?`00018000220@00^P0005`0001Q0780\nLP1Y07H0I@1c0000:P0001T0000I00004@0002D0000Z00009P0002D0000<00001`0005@0001D0000\n=0000180000i00007`000040001RIXM0<8Z7@241001[00000@0004`000040000?`00018000220@00\n^P000500000P00008@000280000<0000ooooobD0000<00001@00P2D0000<00002000P2/0000H0000\nY`400=L1003`0P00R08002D0000<00001@0002D0000<00001P000240000800007P0001P0002W0@00\ne`400?D2002<0P004P0000`0000100009@0000`000080000600000`000000002E00008@0001<0000\nD00008<0001O00000@00069VQd0`RXM0Y`400=X100090000C00000@0002W0@00e`400?@2002<0P00\nH00004<0<P0j0200@@0P06X0K`1R0000=P0002X0000E00005@000380000E00004@0002X0000Z0000\nE00005@000240000D00008P0001O00000@00069VQd0`RXM0g@800=X100010000C00000@0002W0@00\ne`400?@2002<0P00D0000200000W00004P0000`0000100009@0000`000080000E00008P0001<0000\nG`0008<0001]00000@00069VQd0`RXM0Y`4002h2000:0000C00000@0002W0@00e`400?@2002<0P00\nH00006T0L`0P07L0H@1Y07@0J@1^06L04@0002H0000E0000=@0002X0000A00005@000140000Y0000\n:P0002D0000<00001`0005@0001D0000Q00005l000280000K@000040001RIXM0<8Z7@=`2000^0P00\n0@0004`000040000Y`400=L1003d0P00S0800500000P00008@000280000<0000ooooobD0000<0000\n1@00P2D0000<00002000P2/0000H0000E@<00180001>1@00`P0002D0000<00001@0002D0000<0000\n1P000240000800007P0001P0001E0`004P0005@5003700004P0000`0000100009@0000`000080000\n600000`000000002E00008@0002J000010000<h0000C00000@00069VQd0`RXM0E@<001D000090000\nC00000@0001E0`004P0005<500370000H00004D0<P0j0200D`1d0640LP1d0000<P0002X0000E0000\n5@000380000E0000:P0001T0000E0000E00005@0003?000010000=H0000C00000@00069VQd0`RXM0\nNP@001D000010000C00000@0001E0`004P0005<500370000D0000200000W00004P0000`000010000\n9@0000`000080000E0000:00002J00004P000?40000Q00000@00069VQd0`RXM0E@<006T0000>0000\nC00000@0001E0`004P0005<500370000J0000700LP1_06<0I@1c07<0J@1^06L0801Z06l0HP0Z0000\n6@0002X0000V0000:@0002H0000U00004@0002X0000Z000050000140000Z0000:P0002D0000<0000\n1`0005@0001D0000lP000180003e000080000040001RIXM0<8Z7@3X5001Y00000@0004`000040000\nE@<00180001C1@00a`000500000P00008@000280000<0000ooooobD0000<00001@00P2D0000<0000\n2000P2/0000H0000c0H00180001X2000_`0002D0000<00001@0002D0000<00001P00024000080000\n7P0001P0003<1P004P0006d8003400004P0000`0000100009@0000`000080000600000`000000002\nE00009@0000j0@00100007h1000C00000@00069VQd0`RXM0c0H001D0000<0000C00000@0003<1P00\n4P0006`800340000I00004D0<`0j0200@@0P06X0K`1R0200J@1c0380000Z00005@0001D0000b0000\n5@000140000Y0000:P0001D0000@00009P0005@0001D0000O`4000@000240@004`000040001RIXM0\n<8Z7@4P8000E00000@0004`000040000c0H00180001/2000a0000500000P00009`000180000<0000\n0@0002D0000<0000200005@000240000>P400180001h0@0080000040001RIXM0<8Z7@<`6001Y0000\n2@0004`000040000c0H00180001/2000a0000600001S06l0K@1`06`0I@1d06D0I00002H0000Z0000\n?P0002X0000A0000:P0001@0000Z0000:P0002D0000<00001`0005@0001D0000N@400180001n0@00\n80000040001RIXM0<8Z7@2L8001Y00000@0004`000040000c0H00180001/2000a0000500000P0000\n8@000280000<0000ooooobD0000<00001@00P2D0000<00002000P2/0000H0000L0P00=L1003^2P00\nQ08002D0000<00001@0002D0000<00001P000240000800007P0001P0001`2000e`400?@:00290P00\n4P0000`0000100009@0000`000080000600000`000000002E0000<@000260@00D0000?T1001O0000\n0@00069VQd0`RXM0L0P00=X1000D0000C00000@0001`2000e`400?<:00290P00M00004<0=00j0200\n@@0P06X0K`1R0200J@1c0200M`1Q06T0M01Y06h0I`0f0000:P0001D0000E0000<P0001D0000A0000\n:P0002T0000E00004@0002D0000E0000=@0002X0000A00005@000140000Z0000:P0005@0001D0000\nn@400500003i0@00G`000040001RIXM0<8Z7@>l:003J0@000@0004`000040000L0P00=L1003c2P00\nR@800500000P00009`000180000<00000@0002D0000<0000200005@0002D0000QP4005l0003=0@00\nK@000040001RIXM0<8Z7@708000^0P00300004`000040000L0P00=L1003c2P00R@8006@0001d06l0\n801R06D0801_07D0M01`07D0M00E0000:P0001D0000Z0000:P0001@0000Z0000:P0001D0000Y0000\n:P0001D0000U0000300000L0001D0000E0000<h1001O0000d`4006d000010000HVJ7@32:Qd3m2@00\n;P800040001<000010000708003G0@00l`X008T2001@000080000240000R000030000?oooolU0000\n300000D0080U0000300000P0080[000060000<h4003@0@002PL008/2000U0000300000D0000U0000\n300000H0000Q0000200001h0000H0000cP@00=01000?1`00T0800180000<00000@0002D0000<0000\n200001P0000<000000000U@0002P0000gP0004l0000a0@00GP000040001RIXM0<8Z7@<h4003C0@00\n3P0004`000040000cP@00=01000>1`00T08006P0001303<0>P0P05@0J01U0200JP1_0680801Y07<0\n=P0002X0000E00005@0002d0000Z0000:P0001D0000A0000:P0002X0000D00004@0002H0001D0000\nE0000381001?0000>@4005h000010000HVJ7@32:Qd2N1P00d`400040001<000010000<h4003@0@00\n3PL00902001@0000800002L0000B000030000040000U0000300000P0001D0000Z0000=h0001N0000\n@@4006d000010000HVJ7@32:Qd3>1000:08000l0001<000010000<h4003@0@003PL00902001/0000\nHP1U06T0KP1W0200L01b06l0H`1U07<0L`1U06@0000Z0000:P000140000Z0000:@0001D0000Z0000\n6@0002T0000V0000:@0002H0000V0000:@0002X0000U0000300000L0001D0000E0000481001N0000\nA@4006`000010000HVJ7@32:Qd3e1P00:0800040001<000010000<h4003@0@003PL00902001@0000\n80000240000R000030000?oooolU0000300000D0080U0000300000P0080[000060000;X900090000\na`/00;d0000U0000300000D0000U0000300000H0000Q0000200001h0000H0000^PT000T0003<2`00\n`P000180000<00000@0002D0000<0000200001P0000<000000000U@0002X0000`P400080000I0P00\n4@000040001RIXM0<8Z7@;X9000<00003`0004`000040000^PT000T0003;2`00`P0006`0001503@0\n>P0P0440801V06T0KP1Y07<0J01U06@080000380000Z00005@0001D0000b00005@0001D0000A0000\n:@000140000U0000:P0002X0000Z00005@0005@0001D00006@800080000P0P004@000040001RIXM0\n<8Z7@9l;000<00000@0004`000040000^PT000T0003;2`00`P000500000P00009`000180000<0000\n0@0002D0000<0000200005@0002/0000`P400100000L0P007`000040001RIXM0<8Z7@;X9001Q0000\n400004`000040000^PT000T0003;2`00`P0006`0001Z06l0HP0P06T0L`0P06@0I@1/06T0MP1U0780\nI@1T0140000Z0000:P0001D0000@00009P0001D0000Y0000:P000140000A00009@0002X0000I0000\n:@0002X0000U0000300000L0001D0000E00001`2000@0000808001h000010000HVJ7@32:Qd2_2`00\nH@000040001<000010000;X900090000b`/00<80001@000080000240000R000030000?oooolU0000\n300000D0080U0000300000P0080[0000600008@4000:1000@0L00;h4000U0000300000D0000U0000\n300000H0000Q0000200001h0000H0000Q0@000X400141`00``@00180000<00000@0002D0000<0000\n200001P0000<000000000U@0002L0000d@000;00000Q0@00_`000040001RIXM0<8Z7@8@4000=1000\n3@0004`000040000Q0@000X400131`00``@006P000130340>P0P05@0J01U0200JP1_0680801c0000\n=P0002X0000E00005@0002d0000Z0000:P0001D0000A0000:P0002X0000D00009P0005@0001`0000\n8@400;0000150@00_`000040001RIXM0<8Z7@4<6000=10001P0004`000040000Q0@000X400131`00\n``@005P0001X06l0L00P06T0L`0Z0000:@0002X0000E0000400002H0001D0000E00004H1002`0000\nC@400;l000010000HVJ7@32:Qd0;1`003@@00040001<0000100008@4000:1000@`L00<<4001@0000\n800002L0000B000030000040000U0000300000P0001D0000d0000=40002n0000B`400<d000010000\nHVJ7@32:Qd241000HP@001H0001<0000100008@4000:1000@`L00<<4001h0000J@1T06`0I@0/0200\nM`1Q06T0M01Y06h0I`0P06H0K`1b0200M`1_0780J`0A0000:P000140000Z0000500001D0000e0000\n:P000140000E00004@0002X0000Z00005@0001@0000Z00006@0001D0000e0000:P0001T0000V0000\n9@0000`000070000E00005@0001;0@00_P0004l1003<00000@00069VQd0`RXM0;0L0068400010000\nC00000@0002410002P@004<700331000D0000200000Q00008P0000`0003ooooo9P0001`000090000\n000000@000000000000002D0000<00002@0002D0000<00001@00P5L0000T0000900003L0001L0000\n=`000080003604D1n`550BD0000<00002000P2P0000<00002@0002L0000H00002@00000000000000\n000002D0000<00002@0005H0000X0000F`0003@0001S0000>`0000<0003i0E/190960OT1<04U0000\n300000H0000U0000300000D0000X0000300000T0000V0000700000T0000000001000000000000000\n9@0000`0000900009@0000`000050020E`0002@000210000=`000;T0000g00000P000</2A@4014D1\n9@0000`000080020:00000`0000900009`0001P00009000000000000000000009@0000`000090000\nEP0002P0002i0000=0000<00000k00000`000?h3F`4X14H1oP<`0BD0000<00001P0002D0000<0000\n1@0002P0000<00002@0002H0000L00002@0000000004000000000000000U0000300000T0000U0000\n300000D0081G000090000<00000g0000n00003L000020000:0A50Ed5A@4U0000300000P0080X0000\n300000T0000W0000600000T00000000000000000000U0000300000T0001F0000:0000?P0000d0000\no`0003/000030000F`EK0HH5AP5K1C019@0000`0000600009@0000`000050000:00000`000090000\n9P0001`000090000000000@000000000000002D0000<00002@0002D0000<00001@00P5L0000T0000\n7`4003L0001G0@00=`000080000h1TD1K@M50BD0000<00002000P2P0000<00002@0002L0000H0000\n2@00000000000000000002D0000<00002@0005H0000X0000E`4003@0001O0@00>`0000<0001[1e/1\nUPM60F/7<04U0000300000H0000U0000300000D0000X0000300000T0000V0000700000T000000000\n10000000000000009@0000`0000900009@0000`000050020E`0002@0001P0@00=`0009L1000g0000\n0P0009/7A@7A24D19@0000`000080020:00000`0000900009`0001P0000900000000000000000000\n9@0000`000090000EP0002P0002G0@00=00009l1000k00000`000<l8F`7j24H1c`P`0BD0000<0000\n1P0002D0000<00001@0002P0000<00002@0002H0000L00002@0000000004000000000000000U0000\n300000T0000U0000300000D0081G000090000;d1000g0000m@4003L000020000X@U50MH:A@4U0000\n300000P0080X0000300000T0000W0000600000T00000000000000000000U0000300000T0001F0000\n:0000?@1000d0000o04003/000030000e0YK0Oh:AP7D2S019@0000`0000600009@0000`000050000\n:00000`0000900009P0001`000090000000000H000000000000002D0000<00002@0002D0000<0000\n1@00P5L0000T0000900002P0000T0000B0000080003:0>d0bP2X0BD0000<00001P0002P0000<0000\n2@0002D0000<00001@0002H0000L00002@0000000006000000000000000U0000300000T0000U0000\n300000D0081G000090000<40000X0000`@0004P000020000;PC]02h4Z04U0000300000H0000X0000\n300000T0000U0000300000D0000V0000700000T0000000001P000000000000009@0000`000090000\n9@0000`000050020E`0002@0001O0@00:00005l1001800000P0009P7k@2H1jP19@0000`000060000\n:00000`0000900009@0000`0000500009P0001`000090000000000H000000000000002D0000<0000\n2@0002D0000<00001@00P5L0000T0000oP4002P0003n0@00B000008000092nd02@^X0BD0000<0000\n1P0002P0000<00002@0002D0000<00001@0002L0000H00002@000000003oool0000002D0000<0000\n2@0002H0000L00002P0000000006000000000000008U0000300000X0000Z0000600002P2003k0000\na08009H1000U0000300000D0000X0000300000T0000U0000300000H0000X0000300000X0000W0000\n600000X000000000oooo0000000U0000300000X0000V0000700000T0000000001P00000000000002\n9@0000`000090000:P0001P0002D1@00B@<00306003T0`009@0000`000050000:00000`0000:0000\n9@0000`000060000:00000`0000900009`0001P00009000000000?ooo`0000009@0000`000090000\n9P0001`0000:0000000000H00000000000000RD0000<00002P0002X0000H0000U0D00?/0000`1P00\nUP4002D0000<00001@0002P0000<00002@0002D0000<00001P0002P0000<00002P0002L0000H0000\n2P000000003oool0000002D0000<00002P0002H0000L00002@0000000006000000000000008U0000\n300000T0000Z000060000?h8003k0000V@T009H1000U0000300000D0000X0000300000X0000U0000\n300000H0000X0000300000T0000V0000700000T00000000010000000000000009@0000`000090000\n9@0000`000050020E`000=P0000X0@00@@0007X1002K0000;`0009X7P06[1h01_0N20L@7Q07=1hL1\ne@N;0Md7T@7^1ih1m`NU0@08[@492;P140S40AL8dP4L2>@17PS^0B48nP4S20H29@PD0RT8<@8/2502\n;@Q_0Rd8SP8[2:X2:@Rg0RH8``8O2=X25@S_0PT80`?k1aH3k0LW0m`7=`?<1dH3_0MC0j`7GP>K1fP3\nRPM_0gT7MP=V1g/3DPN00cd7Q0<V1hP34PN;0o`6SP?U1Y03cPJA0id6T`=Z1Y@39@0000`000080020\n:00000`0000900009`0001P00009000000000000000000009@0000`000090000EP0002P0000Q0@00\nV00002T1002O00000`0006/6O`=01YH3K0JZ0bD0000<00001P0002D0000<00001@0002P0000<0000\n2@0002H0000L00002@0000000004000000000000000U0000300000T0000U0000300000D0081G0000\ni0000:00001300000P4009/0000b0000U0FE0f/5U@=01I@350FB0o`4T@?T18l3bPB=0jd4S0>?18X3\nL0B80e84QP<f18836`Am0`l4NP<417L3k`=`0md3J0?;0eh3_0=D0jh3B0>Q0c/3UP<]0h`37@>50``3\nO`?h0W/3hP9i0l/2MP>L0WH3Q@9e0g02M@=K0WD3A@9g0aX2N@<60W`3l`5o0n81PP?C0HH3a`690kd1\nS@>e0I43[`6F0jT1W0>T0J@3X06^0i/1]P>H0Kl3U07C0hl1jP><0@<4R@4U0000300000P0080X0000\n300000T0000W0000600000T00000000000000000000U0000300000T0001F0000:0000;T0000o0000\n`@0004L0000300000`BP0B`4R04017D19@0000`0000600009@0000`000050000:00000`000090000\n9`0001P00009000000000000000000009@0000`0000900009P0001`0000:0000000000H000000000\n00000RD0000<00002P0002X0000H0000c@D003@1003f1@00G@4002D0000<00001@0002P0000<0000\n2@0002D0000<00001P0002P0000<00002P0002D0000<00000`0002P0000<0000100002P0000<0000\n1`0002P0000<0000200002L0000H000020000000003oool0000002D0000<0000200002H0000L0000\n1`0000000000000000000000000U0000300000L0000R000030000?oooolR000030000?oooolU0000\n30000040000U0000300000L0080U000030000000080`0000300000l0080U0000300000d0081;0000\n4000000000050000:00000`000010000:00000`000020000:00000`000030000:00000`000080000\n:00000`000050000:00000`000060000:00000`0000700003P0001@000000000400001@0\n"], "Text", Rule[ImageSize, List[560, 212]], Rule[ImageMargins, List[List[0, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]]], Cell[TextData[List[StyleBox["Figure 1:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" The job shop system Petri Net graph once more. The job is in the stage of being processed.\n", Rule[FontFamily, "Courier New"]]]], "Text"], Cell[TextData[List["We would now like to build a Modelica Petri net model of the job shop system, using our recently developed ", StyleBox["NormalPetriNet", Rule[FontFamily, "Courier"]], " package. We need modified instances of the general ", StyleBox["Place", Rule[FontFamily, "Courier"]], " and ", StyleBox["Transition", Rule[FontFamily, "Courier"]], " classes in order to reflect the different numbers of input and output ports.The ", StyleBox["JobShopSystem", Rule[FontFamily, "Courier"]], " model follows below, containing these modified instances of places and transitions, as well as connection equations.\n"]], "Text"], Cell["model JobShopSystem\n  import ModelicaAdditions.PetriNets.*;\n  \n  Transition arrival(nIn = 0, nOut = 1); \t\t    // E1\n  Place      inwait(nIn = 1, nOut = 1);    \t\t    // C2\n  Transition synchronize(nIn = 2, nOut = 1); \t\t// E2\n  Place      processing(nIn = 1, nOut = 1);\t\t    // C3\n  Transition parallel(nIn = 1, nOut = 2);    \t\t// E3\n  Place      idle(nIn = 1, nOut = 1);        \t\t// C1\n  Place      outwait(nIn = 1, nOut = 1);  \t\t    // C4\n  Transition delivery(nIn = 1, nOut = 0);    \t\t// E4\n equation\n  connect(arrival.outPlaces, inwait.inTrans);\n  connect(inwait.outTrans[1], synchronize.inPlaces[1]);\n  connect(idle.outTrans[1], synchronize.inPlaces[2]);\n  connect(synchronize.outPlaces[1], processing.inTrans[1]);\n  connect(processing.outTrans[1], parallel.inPlaces[1]);\n  connect(parallel.outPlaces[1], outwait.inTrans[1]);\n  connect(parallel.outPlaces[2], idle.inTrans[1]);\n  connect(outwait.outTrans[1], delivery.inPlaces[1]);\nend JobShopSystem;", "ModelicaInput"], Cell[CellGroupData[List[Cell["Simulation of JobShopSystem", "Subsection"], Cell["Finally we simulate the job shop system model using the above default settings, for a job that passes through the system and is processed.", "Text"], Cell["simulate( JobShopSystem, stopTime = 1 );", "ModelicaInput"], Cell["", "Text"], Cell["plot( inWait.state );", "ModelicaInput"], Cell["", "Text"], Cell["plot( process.state );", "ModelicaInput"], Cell["", "Text"], Cell["plot( idle.state );", "ModelicaInput"], Cell["", "Text"], Cell["plot( outWait.state );", "ModelicaInput"]], Open]]], Open]], Cell[CellGroupData[List[Cell["Job Shop System Using ModelicaAdditions.PetriNets", "Section"], Cell["model JobShopSystemWorking\nimport ModelicaAdditions.PetriNets.*;\nimport Modelica.Blocks.Sources.*;\n\nSampleTrigger trig2(period={0.2},startTime={0.5});\nSampleTrigger trig1(period={0.2},startTime={0.4});\nPlace01 arrivalPlace(initialState=true);\nPlace10 deliveryPlace;\n\nTransition arrival;\nPlace11 inWait;\nSynchronize synchronize;\nPlace11 process;\nPlace11 idle(initialState=true);\nParallel parallell;\nPlace11 outWait;\nTransition delivery;\n\nequation\nconnect(parallell.inTransition,process.outTransition);\nconnect(synchronize.outTransition,process.inTransition);\nconnect(parallell.outTransition1,idle.inTransition);\nconnect(outWait.inTransition,parallell.outTransition2);\nconnect(outWait.outTransition,delivery.inTransition);\nconnect(synchronize.inTransition1,idle.outTransition);\nconnect(synchronize.inTransition2,inWait.outTransition);\nconnect(inWait.inTransition,arrival.outTransition);\nconnect(deliveryPlace.inTransition,delivery.outTransition);\nconnect(arrival.inTransition,arrivalPlace.outTransition);\nconnect(trig2.outPort,synchronize.conditionPort);\nconnect(trig2.outPort,delivery.conditionPort);\nconnect(trig1.outPort,arrival.conditionPort);\nconnect(parallell.conditionPort,trig1.outPort);\n\nend JobShopSystemWorking;", "ModelicaInput"], Cell[CellGroupData[List[Cell["Simulation of JobShopSystemWorking", "Subsection"], Cell["Finally we simulate the job shop system model using the above default settings, for a job that passes through the system and is processed.", "Text"], Cell["simulate( JobShopSystemWorking, stopTime = 1 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(JobShopSystemWorking\\) : 2004-2-4 9:12:20.4103987 : \\!\\({0.`, 1.`}\\) : 506 data points : 5 events : 80 variables>\\n\\!\\({\\*TagBox[\\(arrival . condition\\), HoldForm], \\*TagBox[\\(arrival . conditionPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(arrival . conditionPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(arrival . fire\\), HoldForm], \\*TagBox[\\(arrival . inTransition . fire\\), HoldForm], \\*TagBox[\\(arrival . inTransition . state\\), HoldForm], \\*TagBox[\\(arrival . outTransition . set\\), HoldForm], \\*TagBox[\\(arrival . outTransition . state\\), HoldForm], \\*TagBox[\\(arrivalPlace . initialState\\), HoldForm], \\*TagBox[\\(arrivalPlace . outTransition . fire\\), HoldForm], \\*TagBox[\\(arrivalPlace . outTransition . state\\), HoldForm], \\*TagBox[\\(arrivalPlace . state\\), HoldForm], \\*TagBox[\\(delivery . condition\\), HoldForm], \\*TagBox[\\(delivery . conditionPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(delivery . conditionPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(delivery . fire\\), HoldForm], \\*TagBox[\\(delivery . inTransition . fire\\), HoldForm], \\*TagBox[\\(delivery . inTransition . state\\), HoldForm], \\*TagBox[\\(delivery . outTransition . set\\), HoldForm], \\*TagBox[\\(delivery . outTransition . state\\), HoldForm], \\*TagBox[\\(deliveryPlace . initialState\\), HoldForm], \\*TagBox[\\(deliveryPlace . inTransition . set\\), HoldForm], \\*TagBox[\\(deliveryPlace . inTransition . state\\), HoldForm], \\*TagBox[\\(deliveryPlace . state\\), HoldForm], \\*TagBox[\\\"\[UnderBracket]derdummy\\\", HoldForm], \\*TagBox[\\\"\[UnderBracket]dummy\\\", HoldForm], \\*TagBox[\\(idle . initialState\\), HoldForm], \\*TagBox[\\(idle . inTransition . set\\), HoldForm], \\*TagBox[\\(idle . inTransition . state\\), HoldForm], \\*TagBox[\\(idle . outTransition . fire\\), HoldForm], \\*TagBox[\\(idle . outTransition . state\\), HoldForm], \\*TagBox[\\(idle . state\\), HoldForm], \\*TagBox[\\(inWait . initialState\\), HoldForm], \\*TagBox[\\(inWait . inTransition . set\\), HoldForm], \\*TagBox[\\(inWait . inTransition . state\\), HoldForm], \\*TagBox[\\(inWait . outTransition . fire\\), HoldForm], \\*TagBox[\\(inWait . outTransition . state\\), HoldForm], \\*TagBox[\\(inWait . state\\), HoldForm], \\*TagBox[\\(outWait . initialState\\), HoldForm], \\*TagBox[\\(outWait . inTransition . set\\), HoldForm], \\*TagBox[\\(outWait . inTransition . state\\), HoldForm], \\*TagBox[\\(outWait . outTransition . fire\\), HoldForm], \\*TagBox[\\(outWait . outTransition . state\\), HoldForm], \\*TagBox[\\(outWait . state\\), HoldForm], \\*TagBox[\\(parallell . condition\\), HoldForm], \\*TagBox[\\(parallell . conditionPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(parallell . conditionPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(parallell . fire\\), HoldForm], \\*TagBox[\\(parallell . inTransition . fire\\), HoldForm], \\*TagBox[\\(parallell . inTransition . state\\), HoldForm], \\*TagBox[\\(parallell . outTransition1 . set\\), HoldForm], \\*TagBox[\\(parallell . outTransition1 . state\\), HoldForm], \\*TagBox[\\(parallell . outTransition2 . set\\), HoldForm], \\*TagBox[\\(parallell . outTransition2 . state\\), HoldForm], \\*TagBox[\\(process . initialState\\), HoldForm], \\*TagBox[\\(process . inTransition . set\\), HoldForm], \\*TagBox[\\(process . inTransition . state\\), HoldForm], \\*TagBox[\\(process . outTransition . fire\\), HoldForm], \\*TagBox[\\(process . outTransition . state\\), HoldForm], \\*TagBox[\\(process . state\\), HoldForm], \\*TagBox[\\(synchronize . condition\\), HoldForm], \\*TagBox[\\(synchronize . conditionPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(synchronize . conditionPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(synchronize . fire\\), HoldForm], \\*TagBox[\\(synchronize . inTransition1 . fire\\), HoldForm], \\*TagBox[\\(synchronize . inTransition1 . state\\), HoldForm], \\*TagBox[\\(synchronize . inTransition2 . fire\\), HoldForm], \\*TagBox[\\(synchronize . inTransition2 . state\\), HoldForm], \\*TagBox[\\(synchronize . outTransition . set\\), HoldForm], \\*TagBox[\\(synchronize . outTransition . state\\), HoldForm], \\*TagBox[\\(trig1 . nout\\), HoldForm], \\*TagBox[\\(trig1 . outPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(trig1 . outPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(trig1 . period\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(trig1 . startTime\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(trig2 . nout\\), HoldForm], \\*TagBox[\\(trig2 . outPort . n\\), HoldForm], \\*TagBox[\\(\\(\\(trig2 . outPort . signal\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(trig2 . period\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(trig2 . startTime\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm]}\\)\""], "Output"], Cell["plot( inWait.state );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.952381 0.0147151 0.588604 [\n[.21429 .00222 -9 -9 ]\n[.21429 .00222 9 0 ]\n[.40476 .00222 -9 -9 ]\n[.40476 .00222 9 0 ]\n[.59524 .00222 -9 -9 ]\n[.59524 .00222 9 0 ]\n[.78571 .00222 -9 -9 ]\n[.78571 .00222 9 0 ]\n[.97619 .00222 -3 -9 ]\n[.97619 .00222 3 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(0.2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(0.4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(0.6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(0.8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(1)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\nMistroke\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\nMistroke\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\nMistroke\n.30952 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\nMistroke\n.40476 .01472 L\n.40476 .60332 L\n.40667 .60332 L\n.40857 .60332 L\n.41048 .60332 L\n.41238 .60332 L\n.41429 .60332 L\n.41619 .60332 L\n.4181 .60332 L\n.42 .60332 L\n.4219 .60332 L\n.42381 .60332 L\n.42571 .60332 L\n.42762 .60332 L\n.42952 .60332 L\n.43143 .60332 L\n.43333 .60332 L\n.43524 .60332 L\n.43714 .60332 L\n.43905 .60332 L\n.44095 .60332 L\n.44286 .60332 L\n.44476 .60332 L\n.44667 .60332 L\n.44857 .60332 L\n.45048 .60332 L\n.45238 .60332 L\n.45429 .60332 L\n.45619 .60332 L\n.4581 .60332 L\n.46 .60332 L\n.4619 .60332 L\n.46381 .60332 L\n.46571 .60332 L\n.46762 .60332 L\n.46952 .60332 L\n.47143 .60332 L\n.47333 .60332 L\n.47524 .60332 L\n.47714 .60332 L\n.47905 .60332 L\n.48095 .60332 L\n.48286 .60332 L\n.48476 .60332 L\n.48667 .60332 L\n.48857 .60332 L\n.49048 .60332 L\n.49238 .60332 L\n.49429 .60332 L\n.49619 .60332 L\nMistroke\n.4981 .60332 L\n.5 .60332 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\nMistroke\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\nMistroke\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\nMistroke\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\nMistroke\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\nMistroke\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(inWait) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(state) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n141.000 12.813 moveto\n(L) show\n147.000 12.813 moveto\n(@) show\n153.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n159.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00@`3IfMT2000000@0fMWI0P0000030=WIf@@000007`3IfMT2000000@0fMWI0P0000050=WI\nf@<000007P3IfMT2000000@0fMWI0P0000030=WIf@<00000803IfMT2000000@0fMWI0P0000040=WI\nf@8000009@3IfMT5000001/0fMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WI\nf@3IfMT07`3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI01`0fMWI00@00000\n0=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0\nfMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@P3IfMT010000000\nfMWI0=WIf@0000090=WIf@030000003IfMT0fMWI01h0fMWI00@000000=WIf@3IfMT00000203IfMT5\n000001d0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WI\nf@00@P3IfMT010000000fMWI0=WIf@00000:0=WIf@030000003IfMT0fMWI01d0fMWI00@000000=WI\nf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0fMWI\n0`00000O0=WIf@040000003IfMT0fMWI000000T0fMWI0P00000W0=WIf@030000003IfMT0fMWI01/0\nfMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000007P3IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0000001h0fMWI00@000000=WIf@3IfMT000002@3I\nfMT00`000000fMWI0=WIf@0N0=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n00009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@`3IfMT2000000X0fMWI0P00000P0=WIf@800000\n2`3IfMT2000001l0fMWI0P00000:0=WIf@<000007`3IfMT2000000X0fMWI0P00000V0=WIf@800000\n7@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WI\nf@00703IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000003IfMT0fMWI0?80fMWI\n0P00000=0=WIf@005P3IfMT6000005d0003o5P00001d0000o`D00000203IfMT010000000fMWI0=WI\nf@00000<0=WIf@00703IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000000fMWI0=WIf@0[\n0=WIf@030000003IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000003IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0:\n0=WIf@D00000303IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07h0fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<000000=WIf@3I\nfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000\n003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703I\nfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3I\nfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WI\nf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@03\n0000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00\n703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nSP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WI\nf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WI\nf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WI\nf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@2>0=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT0\n0`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0\nfMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI0P00001J0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000P0fMWI00@000000=WIf@3IfMT000002P3IfMT0\n0`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@2>0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0\nfMWI000000@0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI08h0fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<000000=WIf@3IfMT0F@3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI\n05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000\nfMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI\n00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI\n000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0\nfMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`00\n0000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0\nfMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0\nfMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0\nF@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003I\nfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT0\n0`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@2>0=WIf@002@3IfMT2000000@0fMWI0P0000050=WIf@<000000`3IfMT00`000000fMWI\n0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000P0fMWI00@0\n00000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI05T0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00203IfMT010000000fMWI0=WI\nf@0000080=WIf@D000000`3IfMT2000005X0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@2>0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0\nfMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0\nfMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@000000103IfMT00`000000\nfMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000T0fMWI\n0P00000;0=WIf@800000103IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3I\nfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000\n003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703I\nfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3I\nfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WI\nf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@03\n0000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00\n703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nSP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WI\nf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WI\nf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WI\nf@002@3IfMT2000000@0fMWI0P0000030=WIf@<000001@3IfMT00`000000fMWI0=WIf@1I0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WI\nf@040000003IfMT0fMWI000000@0fMWI0P00001J0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0SP3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0fMWI00<00000\n0=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI00080=WI\nf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@1I\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000T0fMWI0P00000:0=WI\nf@<00000103IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI\n05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000\nfMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI\n00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI\n000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0\nfMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`00\n0000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0\nfMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0\nfMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0\nF@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI00090=WIf@800000103I\nfMT2000000@0fMWI0P0000050=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@2>0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003I\nfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI08h0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n0000103IfMT2000005X0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00\n203IfMT010000000fMWI0=WIf@0000090=WIf@8000001@3IfMT00`000000fMWI0=WIf@1I0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@03\n0000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00\n703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nSP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WI\nf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WI\nf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WI\nf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI\n0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000L\n0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@2>\n0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI08h0fMWI000L0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@2>0=WIf@00703IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001`0fMWI00<000000=WIf@3IfMT0F@3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI08h0fMWI000D0=WIf@D000000`3IfMT00`00\n0000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0SP3IfMT001H0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0SP3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT2000005X0fMWI\n60000?n@0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3I\nfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WI\nf@8000001@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI04P0\nfMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002L0=WIf@00903IfMT00`000000fMWI\n0=WIf@1:0=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@2J0=WIf@008`3IfMT00`000000fMWI0=WIf@030=WIf@<0000000<0fMWI0000000000000P3I\nfMT200000080fMWI00<000000=WIf@0000000P3IfMT500000080fMWI0`0000030=WIf@800000103I\nfMT2000000<0fMWI0`0000040=WIf@8000000`3IfMT500000080fMWI0P0000040=WIf@<00000103I\nfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0080fMWI0P0000040=WIf@030000003I\nfMT0fMWI09X0fMWI000S0=WIf@030000003IfMT0fMWI00@0fMWI00D000000=WIf@3IfMT0fMWI0000\n00020=WIf@090000003IfMT0fMWI0000003IfMT000000=WIf@000000fMWI00@00000103IfMT01@00\n0000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT02@3IfMT010000000fMWI0=WIf@00\n00020=WIf@030000003IfMT0fMWI00@000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000\n003IfMT0fMWI00D0fMWI00<000000=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT000000080\nfMWI00D000000=WIf@3IfMT0fMWI0000002L0=WIf@005P3IfMT80000o`D0fMWI00<000000=WIf@3I\nfMT0103IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00P000000=WIf@3IfMT000000=WIf@00\n0000fMWI000000@0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WIf@3IfMT0000000/0\nfMWI0`0000030=WIf@030000003IfMT0fMWI00H0fMWI00@000000=WIf@3IfMT000001@3IfMT40000\n00@0fMWI00<000000=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<00000\n0=WIf@3IfMT0VP3IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT200000080fMWI100000030=WI\nf@050000003IfMT000000=WIf@0000000P3IfMT3000000<0fMWI0P0000020=WIf@D00000203IfMT4\n000000030=WIf@000000000000<000000`3IfMT3000000030=WIf@000000000000<000000`3IfMT2\n000000D0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@05000000<0fMWI00<00000\n0=WIf@3IfMT0VP3IfMT002@0fMWI00<000000=WIf@3IfMT03@3IfMT01@000000fMWI0=WIf@3IfMT0\n000000d0fMWI00<000000=WIf@3IfMT03`3IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0\nfMWI00X0fMWI00<000000=WIf@3IfMT0103IfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<0\n00000=WIf@3IfMT0VP3IfMT002D0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WIf@06\n0=WIf@<0000000<0fMWI0000000000002@3IfMT00`000000fMWI0=WIf@0[0=WIf@030000003IfMT0\nfMWI00D0fMWI0P0000080=WIf@800000W03IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00001"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.121768`, -0.155509`, 0.00434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell["plot( process.state );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.952381 0.0147151 0.588604 [\n[.21429 .00222 -9 -9 ]\n[.21429 .00222 9 0 ]\n[.40476 .00222 -9 -9 ]\n[.40476 .00222 9 0 ]\n[.59524 .00222 -9 -9 ]\n[.59524 .00222 9 0 ]\n[.78571 .00222 -9 -9 ]\n[.78571 .00222 9 0 ]\n[.97619 .00222 -3 -9 ]\n[.97619 .00222 3 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(0.2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(0.4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(0.6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(0.8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(1)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\nMistroke\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\nMistroke\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\nMistroke\n.30952 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\nMistroke\n.40476 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\nMistroke\n.4981 .01472 L\n.5 .01472 L\n.5 .60332 L\n.5019 .60332 L\n.50381 .60332 L\n.50571 .60332 L\n.50762 .60332 L\n.50952 .60332 L\n.51143 .60332 L\n.51333 .60332 L\n.51524 .60332 L\n.51714 .60332 L\n.51905 .60332 L\n.52095 .60332 L\n.52286 .60332 L\n.52476 .60332 L\n.52667 .60332 L\n.52857 .60332 L\n.53048 .60332 L\n.53238 .60332 L\n.53429 .60332 L\n.53619 .60332 L\n.5381 .60332 L\n.54 .60332 L\n.5419 .60332 L\n.54381 .60332 L\n.54571 .60332 L\n.54762 .60332 L\n.54952 .60332 L\n.55143 .60332 L\n.55333 .60332 L\n.55524 .60332 L\n.55714 .60332 L\n.55905 .60332 L\n.56095 .60332 L\n.56286 .60332 L\n.56476 .60332 L\n.56667 .60332 L\n.56857 .60332 L\n.57048 .60332 L\n.57238 .60332 L\n.57429 .60332 L\n.57619 .60332 L\n.5781 .60332 L\n.58 .60332 L\n.5819 .60332 L\n.58381 .60332 L\n.58571 .60332 L\n.58762 .60332 L\n.58952 .60332 L\nMistroke\n.59143 .60332 L\n.59333 .60332 L\n.59524 .60332 L\n.59524 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\nMistroke\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\nMistroke\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\nMistroke\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\nMistroke\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(process) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n117.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(state) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n147.000 12.813 moveto\n(L) show\n153.000 12.813 moveto\n(@) show\n159.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n165.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00@`3IfMT2000000@0fMWI0P0000030=WIf@@000007`3IfMT2000000@0fMWI0P0000050=WI\nf@<000007P3IfMT2000000@0fMWI0P0000030=WIf@<00000803IfMT2000000@0fMWI0P0000040=WI\nf@8000009@3IfMT5000001/0fMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WI\nf@3IfMT07`3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI01`0fMWI00@00000\n0=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0\nfMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@P3IfMT010000000\nfMWI0=WIf@0000090=WIf@030000003IfMT0fMWI01h0fMWI00@000000=WIf@3IfMT00000203IfMT5\n000001d0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WI\nf@00@P3IfMT010000000fMWI0=WIf@00000:0=WIf@030000003IfMT0fMWI01d0fMWI00@000000=WI\nf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0fMWI\n0`00000O0=WIf@040000003IfMT0fMWI000000T0fMWI0P00000W0=WIf@030000003IfMT0fMWI01/0\nfMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000007P3IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0000001h0fMWI00@000000=WIf@3IfMT000002@3I\nfMT00`000000fMWI0=WIf@0N0=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n00009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@`3IfMT2000000X0fMWI0P00000P0=WIf@800000\n2`3IfMT2000001l0fMWI0P00000:0=WIf@<000007`3IfMT2000000X0fMWI0P00000V0=WIf@800000\n7@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WI\nf@00703IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000003IfMT0fMWI0?80fMWI\n0P00000=0=WIf@005P3IfMT6000007@0003o5P00001M0000o`D00000203IfMT010000000fMWI0=WI\nf@00000<0=WIf@00703IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000000fMWI0=WIf@0[\n0=WIf@030000003IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0:\n0=WIf@D00000303IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI06L0fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<000000=WIf@3I\nfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000\n003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WI\nf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@03\n0000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WI\nf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WI\nf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WI\nf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1g0=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT0\n0`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0\nfMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI0P00001a0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI00@000000=WIf@3IfMT000002P3IfMT0\n0`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0\nfMWI000000@0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI\n0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI\n00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI\n000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0\nfMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`00\n0000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0\nfMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0\nfMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0\nL03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003I\nfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT0\n0`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@1g0=WIf@002@3IfMT2000000@0fMWI0P0000050=WIf@<000000`3IfMT00`000000fMWI\n0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI00@0\n00000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WI\nf@0000080=WIf@D000000`3IfMT200000740fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0\nfMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0\nfMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@000000103IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000T0fMWI\n0P00000;0=WIf@800000103IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3I\nfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000\n003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WI\nf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@03\n0000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WI\nf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WI\nf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WI\nf@002@3IfMT2000000@0fMWI0P0000030=WIf@<000001@3IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WI\nf@040000003IfMT0fMWI000000@0fMWI0P00001a0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0M`3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0fMWI00<00000\n0=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00080=WI\nf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000T0fMWI0P00000:0=WI\nf@<00000103IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI\n0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI\n00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI\n000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0\nfMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`00\n0000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0\nfMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0\nfMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0\nL03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00090=WIf@800000103I\nfMT2000000@0fMWI0P0000050=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003I\nfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI07L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n0000103IfMT200000740fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n203IfMT010000000fMWI0=WIf@0000090=WIf@8000001@3IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@03\n0000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WI\nf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WI\nf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WI\nf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI\n0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L\n0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g\n0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000D0=WIf@D000000`3IfMT00`00\n0000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001H0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT200000740fMWI\n60000?mi0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3I\nfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WI\nf@8000001@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT000000080\n0000C03IfMT00`000000fMWI0=WIf@050=WIf@800000203IfMT2000009H0fMWI000T0=WIf@050000\n003IfMT0fMWI0=WIf@000000CP3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00P0\nfMWI00<000000=WIf@3IfMT0U03IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT3000000<0fMWI\n100000030=WIf@800000103IfMT3000000<0fMWI0`0000020=WIf@<000000`3IfMT3000000@0fMWI\n0P0000030=WIf@<00000103IfMT2000000<0fMWI1@0000020=WIf@800000103IfMT3000000@0fMWI\n00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@020=WIf@800000103IfMT00`000000fMWI\n0=WIf@2D0=WIf@008`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000000<0fMWI\n00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI00<0\nfMWI00<000000=WIf@3IfMT01P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00T0\nfMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0400000080fMWI00@000000=WI\nf@3IfMT000000P3IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0fMWI00<0fMWI00D00000\n0=WIf@3IfMT0fMWI000000020=WIf@050000003IfMT0fMWI0=WIf@000000UP3IfMT001H0fMWI2000\n0?l50=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI\n0=WIf@020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT00`3IfMT400000080\nfMWI0`0000030=WIf@<000002@3IfMT3000000<0fMWI00<000000=WIf@3IfMT01P3IfMT010000000\nfMWI0=WIf@0000050=WIf@@00000103IfMT00`000000fMWI0=WIf@030=WIf@050000003IfMT0fMWI\n0=WIf@0000001P3IfMT00`000000fMWI0=WIf@2D0=WIf@008`3IfMT010000000fMWI0=WIf@3IfMT4\n000000<0fMWI100000030=WIf@800000103IfMT3000000<0fMWI0P0000030=WIf@@000000P3IfMT4\n000000P0fMWI100000000`3IfMT0000000000003000000<0fMWI0`0000000`3IfMT0000000000003\n000000<0fMWI0P0000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT01@000003\n0=WIf@030000003IfMT0fMWI09@0fMWI000T0=WIf@030000003IfMT0fMWI03L0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@0:0=WIf@030000003IfMT0fMWI00@0fMWI00D000000=WI\nf@3IfMT0fMWI000000060=WIf@030000003IfMT0fMWI09@0fMWI000U0=WIf@030000003IfMT0fMWI\n04h0fMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002F0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.121768`, -0.155509`, 0.00434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell["plot( idle.state );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.952381 0.0147151 0.588604 [\n[.21429 .00222 -9 -9 ]\n[.21429 .00222 9 0 ]\n[.40476 .00222 -9 -9 ]\n[.40476 .00222 9 0 ]\n[.59524 .00222 -9 -9 ]\n[.59524 .00222 9 0 ]\n[.78571 .00222 -9 -9 ]\n[.78571 .00222 9 0 ]\n[.97619 .00222 -3 -9 ]\n[.97619 .00222 3 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(0.2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(0.4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(0.6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(0.8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(1)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02571 .60332 L\n.02762 .60332 L\n.02952 .60332 L\n.03143 .60332 L\n.03333 .60332 L\n.03524 .60332 L\n.03714 .60332 L\n.03905 .60332 L\n.04095 .60332 L\n.04286 .60332 L\n.04476 .60332 L\n.04667 .60332 L\n.04857 .60332 L\n.05048 .60332 L\n.05238 .60332 L\n.05429 .60332 L\n.05619 .60332 L\n.0581 .60332 L\n.06 .60332 L\n.0619 .60332 L\n.06381 .60332 L\n.06571 .60332 L\n.06762 .60332 L\n.06952 .60332 L\n.07143 .60332 L\n.07333 .60332 L\n.07524 .60332 L\n.07714 .60332 L\n.07905 .60332 L\n.08095 .60332 L\n.08286 .60332 L\n.08476 .60332 L\n.08667 .60332 L\n.08857 .60332 L\n.09048 .60332 L\n.09238 .60332 L\n.09429 .60332 L\n.09619 .60332 L\n.0981 .60332 L\n.1 .60332 L\n.1019 .60332 L\n.10381 .60332 L\n.10571 .60332 L\n.10762 .60332 L\n.10952 .60332 L\n.11143 .60332 L\n.11333 .60332 L\n.11524 .60332 L\n.11714 .60332 L\nMistroke\n.11905 .60332 L\n.12095 .60332 L\n.12286 .60332 L\n.12476 .60332 L\n.12667 .60332 L\n.12857 .60332 L\n.13048 .60332 L\n.13238 .60332 L\n.13429 .60332 L\n.13619 .60332 L\n.1381 .60332 L\n.14 .60332 L\n.1419 .60332 L\n.14381 .60332 L\n.14571 .60332 L\n.14762 .60332 L\n.14952 .60332 L\n.15143 .60332 L\n.15333 .60332 L\n.15524 .60332 L\n.15714 .60332 L\n.15905 .60332 L\n.16095 .60332 L\n.16286 .60332 L\n.16476 .60332 L\n.16667 .60332 L\n.16857 .60332 L\n.17048 .60332 L\n.17238 .60332 L\n.17429 .60332 L\n.17619 .60332 L\n.1781 .60332 L\n.18 .60332 L\n.1819 .60332 L\n.18381 .60332 L\n.18571 .60332 L\n.18762 .60332 L\n.18952 .60332 L\n.19143 .60332 L\n.19333 .60332 L\n.19524 .60332 L\n.19714 .60332 L\n.19905 .60332 L\n.20095 .60332 L\n.20286 .60332 L\n.20476 .60332 L\n.20667 .60332 L\n.20857 .60332 L\n.21048 .60332 L\n.21238 .60332 L\nMistroke\n.21429 .60332 L\n.21619 .60332 L\n.2181 .60332 L\n.22 .60332 L\n.2219 .60332 L\n.22381 .60332 L\n.22571 .60332 L\n.22762 .60332 L\n.22952 .60332 L\n.23143 .60332 L\n.23333 .60332 L\n.23524 .60332 L\n.23714 .60332 L\n.23905 .60332 L\n.24095 .60332 L\n.24286 .60332 L\n.24476 .60332 L\n.24667 .60332 L\n.24857 .60332 L\n.25048 .60332 L\n.25238 .60332 L\n.25429 .60332 L\n.25619 .60332 L\n.2581 .60332 L\n.26 .60332 L\n.2619 .60332 L\n.26381 .60332 L\n.26571 .60332 L\n.26762 .60332 L\n.26952 .60332 L\n.27143 .60332 L\n.27333 .60332 L\n.27524 .60332 L\n.27714 .60332 L\n.27905 .60332 L\n.28095 .60332 L\n.28286 .60332 L\n.28476 .60332 L\n.28667 .60332 L\n.28857 .60332 L\n.29048 .60332 L\n.29238 .60332 L\n.29429 .60332 L\n.29619 .60332 L\n.2981 .60332 L\n.3 .60332 L\n.3019 .60332 L\n.30381 .60332 L\n.30571 .60332 L\n.30762 .60332 L\nMistroke\n.30952 .60332 L\n.31143 .60332 L\n.31333 .60332 L\n.31524 .60332 L\n.31714 .60332 L\n.31905 .60332 L\n.32095 .60332 L\n.32286 .60332 L\n.32476 .60332 L\n.32667 .60332 L\n.32857 .60332 L\n.33048 .60332 L\n.33238 .60332 L\n.33429 .60332 L\n.33619 .60332 L\n.3381 .60332 L\n.34 .60332 L\n.3419 .60332 L\n.34381 .60332 L\n.34571 .60332 L\n.34762 .60332 L\n.34952 .60332 L\n.35143 .60332 L\n.35333 .60332 L\n.35524 .60332 L\n.35714 .60332 L\n.35905 .60332 L\n.36095 .60332 L\n.36286 .60332 L\n.36476 .60332 L\n.36667 .60332 L\n.36857 .60332 L\n.37048 .60332 L\n.37238 .60332 L\n.37429 .60332 L\n.37619 .60332 L\n.3781 .60332 L\n.38 .60332 L\n.3819 .60332 L\n.38381 .60332 L\n.38571 .60332 L\n.38762 .60332 L\n.38952 .60332 L\n.39143 .60332 L\n.39333 .60332 L\n.39524 .60332 L\n.39714 .60332 L\n.39905 .60332 L\n.40095 .60332 L\n.40286 .60332 L\nMistroke\n.40476 .60332 L\n.40476 .60332 L\n.40667 .60332 L\n.40857 .60332 L\n.41048 .60332 L\n.41238 .60332 L\n.41429 .60332 L\n.41619 .60332 L\n.4181 .60332 L\n.42 .60332 L\n.4219 .60332 L\n.42381 .60332 L\n.42571 .60332 L\n.42762 .60332 L\n.42952 .60332 L\n.43143 .60332 L\n.43333 .60332 L\n.43524 .60332 L\n.43714 .60332 L\n.43905 .60332 L\n.44095 .60332 L\n.44286 .60332 L\n.44476 .60332 L\n.44667 .60332 L\n.44857 .60332 L\n.45048 .60332 L\n.45238 .60332 L\n.45429 .60332 L\n.45619 .60332 L\n.4581 .60332 L\n.46 .60332 L\n.4619 .60332 L\n.46381 .60332 L\n.46571 .60332 L\n.46762 .60332 L\n.46952 .60332 L\n.47143 .60332 L\n.47333 .60332 L\n.47524 .60332 L\n.47714 .60332 L\n.47905 .60332 L\n.48095 .60332 L\n.48286 .60332 L\n.48476 .60332 L\n.48667 .60332 L\n.48857 .60332 L\n.49048 .60332 L\n.49238 .60332 L\n.49429 .60332 L\n.49619 .60332 L\nMistroke\n.4981 .60332 L\n.5 .60332 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\nMistroke\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59714 .60332 L\n.59905 .60332 L\n.60095 .60332 L\n.60286 .60332 L\n.60476 .60332 L\n.60667 .60332 L\n.60857 .60332 L\n.61048 .60332 L\n.61238 .60332 L\n.61429 .60332 L\n.61619 .60332 L\n.6181 .60332 L\n.62 .60332 L\n.6219 .60332 L\n.62381 .60332 L\n.62571 .60332 L\n.62762 .60332 L\n.62952 .60332 L\n.63143 .60332 L\n.63333 .60332 L\n.63524 .60332 L\n.63714 .60332 L\n.63905 .60332 L\n.64095 .60332 L\n.64286 .60332 L\n.64476 .60332 L\n.64667 .60332 L\n.64857 .60332 L\n.65048 .60332 L\n.65238 .60332 L\n.65429 .60332 L\n.65619 .60332 L\n.6581 .60332 L\n.66 .60332 L\n.6619 .60332 L\n.66381 .60332 L\n.66571 .60332 L\n.66762 .60332 L\n.66952 .60332 L\n.67143 .60332 L\n.67333 .60332 L\n.67524 .60332 L\n.67714 .60332 L\n.67905 .60332 L\n.68095 .60332 L\n.68286 .60332 L\nMistroke\n.68476 .60332 L\n.68667 .60332 L\n.68857 .60332 L\n.69048 .60332 L\n.69048 .60332 L\n.69238 .60332 L\n.69429 .60332 L\n.69619 .60332 L\n.6981 .60332 L\n.7 .60332 L\n.7019 .60332 L\n.70381 .60332 L\n.70571 .60332 L\n.70762 .60332 L\n.70952 .60332 L\n.71143 .60332 L\n.71333 .60332 L\n.71524 .60332 L\n.71714 .60332 L\n.71905 .60332 L\n.72095 .60332 L\n.72286 .60332 L\n.72476 .60332 L\n.72667 .60332 L\n.72857 .60332 L\n.73048 .60332 L\n.73238 .60332 L\n.73429 .60332 L\n.73619 .60332 L\n.7381 .60332 L\n.74 .60332 L\n.7419 .60332 L\n.74381 .60332 L\n.74571 .60332 L\n.74762 .60332 L\n.74952 .60332 L\n.75143 .60332 L\n.75333 .60332 L\n.75524 .60332 L\n.75714 .60332 L\n.75905 .60332 L\n.76095 .60332 L\n.76286 .60332 L\n.76476 .60332 L\n.76667 .60332 L\n.76857 .60332 L\n.77048 .60332 L\n.77238 .60332 L\n.77429 .60332 L\n.77619 .60332 L\nMistroke\n.7781 .60332 L\n.78 .60332 L\n.7819 .60332 L\n.78381 .60332 L\n.78571 .60332 L\n.78762 .60332 L\n.78952 .60332 L\n.79143 .60332 L\n.79333 .60332 L\n.79524 .60332 L\n.79714 .60332 L\n.79905 .60332 L\n.80095 .60332 L\n.80286 .60332 L\n.80476 .60332 L\n.80667 .60332 L\n.80857 .60332 L\n.81048 .60332 L\n.81238 .60332 L\n.81429 .60332 L\n.81619 .60332 L\n.8181 .60332 L\n.82 .60332 L\n.8219 .60332 L\n.82381 .60332 L\n.82571 .60332 L\n.82762 .60332 L\n.82952 .60332 L\n.83143 .60332 L\n.83333 .60332 L\n.83524 .60332 L\n.83714 .60332 L\n.83905 .60332 L\n.84095 .60332 L\n.84286 .60332 L\n.84476 .60332 L\n.84667 .60332 L\n.84857 .60332 L\n.85048 .60332 L\n.85238 .60332 L\n.85429 .60332 L\n.85619 .60332 L\n.8581 .60332 L\n.86 .60332 L\n.8619 .60332 L\n.86381 .60332 L\n.86571 .60332 L\n.86762 .60332 L\n.86952 .60332 L\n.87143 .60332 L\nMistroke\n.87333 .60332 L\n.87524 .60332 L\n.87714 .60332 L\n.87905 .60332 L\n.88095 .60332 L\n.88286 .60332 L\n.88476 .60332 L\n.88667 .60332 L\n.88857 .60332 L\n.89048 .60332 L\n.89238 .60332 L\n.89429 .60332 L\n.89619 .60332 L\n.8981 .60332 L\n.9 .60332 L\n.9019 .60332 L\n.90381 .60332 L\n.90571 .60332 L\n.90762 .60332 L\n.90952 .60332 L\n.91143 .60332 L\n.91333 .60332 L\n.91524 .60332 L\n.91714 .60332 L\n.91905 .60332 L\n.92095 .60332 L\n.92286 .60332 L\n.92476 .60332 L\n.92667 .60332 L\n.92857 .60332 L\n.93048 .60332 L\n.93238 .60332 L\n.93429 .60332 L\n.93619 .60332 L\n.9381 .60332 L\n.94 .60332 L\n.9419 .60332 L\n.94381 .60332 L\n.94571 .60332 L\n.94762 .60332 L\n.94952 .60332 L\n.95143 .60332 L\n.95333 .60332 L\n.95524 .60332 L\n.95714 .60332 L\n.95905 .60332 L\n.96095 .60332 L\n.96286 .60332 L\n.96476 .60332 L\n.96667 .60332 L\nMistroke\n.96857 .60332 L\n.97048 .60332 L\n.97238 .60332 L\n.97429 .60332 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(idle) show\n93.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n99.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(state) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n129.000 12.813 moveto\n(L) show\n135.000 12.813 moveto\n(@) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n147.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00@`3IfMT2000000@0fMWI0P0000030=WIf@@000007`3IfMT2000000@0fMWI0P0000050=WI\nf@<000007P3IfMT2000000@0fMWI0P0000030=WIf@<00000803IfMT2000000@0fMWI0P0000040=WI\nf@8000009@3IfMT5000001/0fMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WI\nf@3IfMT07`3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI01`0fMWI00@00000\n0=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0\nfMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@P3IfMT010000000\nfMWI0=WIf@0000090=WIf@030000003IfMT0fMWI01h0fMWI00@000000=WIf@3IfMT00000203IfMT5\n000001d0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WI\nf@00@P3IfMT010000000fMWI0=WIf@00000:0=WIf@030000003IfMT0fMWI01d0fMWI00@000000=WI\nf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0fMWI\n0`00000O0=WIf@040000003IfMT0fMWI000000T0fMWI0P00000W0=WIf@030000003IfMT0fMWI01/0\nfMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000007P3IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0000001h0fMWI00@000000=WIf@3IfMT000002@3I\nfMT00`000000fMWI0=WIf@0N0=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n00009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@`3IfMT2000000X0fMWI0P00000P0=WIf@800000\n2`3IfMT2000001l0fMWI0P00000:0=WIf@<000007`3IfMT2000000X0fMWI0P00000V0=WIf@800000\n7@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WI\nf@00703IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000003IfMT0fMWI0?80fMWI\n0P00000=0=WIf@005P3IfMUi000001P0003oH@0000080=WIf@040000003IfMT0fMWI000000`0fMWI\n000L0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WI\nf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WI\nf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WI\nf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000000fMWI0=WI\nf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WI\nf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WI\nf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WI\nf@0;0=WIf@030000003IfMT0fMWI00d0fMWI000L0=WIf@030000003IfMT0fMWI02/0fMWI00<00000\n0=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0[0=WIf@030000003IfMT0fMWI00X0fMWI1@00000<\n0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0I`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0\nL03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003I\nfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT0\n0`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3I\nfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000\n003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WI\nf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@03\n0000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI00090=WIf@800000103IfMT2000000<0fMWI100000040=WIf@030000003IfMT0\nfMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000080=WIf@030000003IfMT0fMWI00D0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00080=WIf@040000003IfMT0fMWI\n000000T0fMWI00<000000=WIf@3IfMT0103IfMT200000740fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@00000:0=WIf@030000003IfMT0\nfMWI00<0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT000T0fMWI0P00000:0=WIf@8000001@3IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI\n0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L\n0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g\n0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI\n0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI\n00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI\n000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0\nfMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`00\n0000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0\nfMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0\nfMWI00090=WIf@800000103IfMT2000000D0fMWI0`0000030=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WI\nf@00000;0=WIf@030000003IfMT0fMWI0080fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI\n1@0000030=WIf@800000L@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI\n00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI\n00@000000=WIf@3IfMT000002@3IfMT00`000000fMWI000000040=WIf@030000003IfMT0fMWI0700\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@002@3IfMT2000000/0fMWI\n0P0000040=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0\nL03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003I\nfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT0\n0`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3I\nfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000\n003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WI\nf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00090=WIf@80\n0000103IfMT2000000<0fMWI0`0000050=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@04\n0000003IfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WI\nf@3IfMT00000103IfMT200000740fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g\n0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@<000001@3IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000P0fMWI00@000000=WI\nf@3IfMT000002@3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI0700fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@002@3IfMT2000000X0fMWI0`0000040=WI\nf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WI\nf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI\n0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L\n0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g\n0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI\n0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000\nfMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI\n00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI\n000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT000T0fMWI0P0000040=WIf@800000103I\nfMT2000000D0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000\n103IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@80\n0000L@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI00080=WIf@040000\n003IfMT0fMWI000000T0fMWI0P0000050=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@04\n0000003IfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI07L0fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<000000=WIf@3I\nfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000\n003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703I\nfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3I\nfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WI\nf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@03\n0000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00\n703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nM`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WI\nf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI07L0fMWI000L0=WI\nf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WI\nf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0M`3IfMT001`0fMWI00<000000=WIf@3IfMT0L03IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI07L0fMWI000L0=WIf@030000003IfMT0fMWI0700fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1g0=WIf@00703IfMT00`000000fMWI0=WIf@1`0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0M`3IfMT001@0fMWI1@0000030=WIf@030000003IfMT0fMWI\n0700fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1g0=WIf@005P3IfMT00`000000\nfMWI0=WIf@030=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001H0fMWI\n00<000000=WIf@3IfMT00`3IfMUd0000oaH0fMWIG@000?lM0=WIf@005P3IfMT00`000000fMWI0=WI\nf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT0\n0`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WIf@8000001@3IfMT00`000000fMWI0=WIf@3o0=WI\nf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI000U0=WIf@030000003IfMT0fMWI03`0fMWI00<000000=WIf@3IfMT01@3IfMT2000000P0\nfMWI0P00002X0=WIf@00903IfMT00`000000fMWI0=WIf@0n0=WIf@030000003IfMT0fMWI00@0fMWI\n00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@2V0=WIf@008`3IfMT00`000000fMWI0=WI\nf@030=WIf@<000000`3IfMT4000000040=WIf@0000000000000000@0fMWI0`0000030=WIf@800000\n0`3IfMT3000000@0fMWI0P0000030=WIf@D000000P3IfMT2000000@0fMWI0`0000040=WIf@030000\n003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0\nYP3IfMT002<0fMWI00<000000=WIf@3IfMT0103IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI\n00D000000=WIf@3IfMT0fMWI000000040=WIf@030000003IfMT0fMWI00`0fMWI00@000000=WIf@3I\nfMT000000P3IfMT00`000000fMWI0=WIf@0400000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n0`000000fMWI0=WIf@050=WIf@030000003IfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI0000\n00020=WIf@050000003IfMT0fMWI0=WIf@000000Z03IfMT001H0fMWI20000?l50=WIf@030000003I\nfMT0fMWI00@0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@050000003IfMT0fMWI0=WIf@00\n0000103IfMT4000000P0fMWI0`0000030=WIf@030000003IfMT0fMWI00H0fMWI00@000000=WIf@3I\nfMT000001@3IfMT4000000@0fMWI00<000000=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT0\n000000H0fMWI00<000000=WIf@3IfMT0YP3IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT20000\n00@0fMWI0`0000030=WIf@030000003IfMT0fMWI00<0fMWI0P0000090=WIf@@0000000<0fMWI0000\n000000000`0000030=WIf@<0000000<0fMWI0000000000000`0000030=WIf@8000001@3IfMT00`00\n0000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00D000000`3IfMT00`000000fMWI0=WIf@2V0=WI\nf@00903IfMT00`000000fMWI0=WIf@0:0=WIf@050000003IfMT0fMWI0=WIf@0000005P3IfMT00`00\n0000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00X0fMWI00<000000=WIf@3IfMT0103IfMT01@00\n0000fMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0YP3IfMT002D0fMWI00<000000=WI\nf@3IfMT00P3IfMT00`000000fMWI0=WIf@030=WIf@8000000P3IfMT2000002h0fMWI00<000000=WI\nf@3IfMT01@3IfMT2000000P0fMWI0P00002X0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.121768`, -0.155509`, 0.00434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell["plot( outWait.state );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.952381 0.0147151 0.588604 [\n[.21429 .00222 -9 -9 ]\n[.21429 .00222 9 0 ]\n[.40476 .00222 -9 -9 ]\n[.40476 .00222 9 0 ]\n[.59524 .00222 -9 -9 ]\n[.59524 .00222 9 0 ]\n[.78571 .00222 -9 -9 ]\n[.78571 .00222 9 0 ]\n[.97619 .00222 -3 -9 ]\n[.97619 .00222 3 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(0.2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(0.4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(0.6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(0.8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(1)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\nMistroke\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\nMistroke\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\nMistroke\n.30952 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\nMistroke\n.40476 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\nMistroke\n.4981 .01472 L\n.5 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\nMistroke\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59714 .60332 L\n.59905 .60332 L\n.60095 .60332 L\n.60286 .60332 L\n.60476 .60332 L\n.60667 .60332 L\n.60857 .60332 L\n.61048 .60332 L\n.61238 .60332 L\n.61429 .60332 L\n.61619 .60332 L\n.6181 .60332 L\n.62 .60332 L\n.6219 .60332 L\n.62381 .60332 L\n.62571 .60332 L\n.62762 .60332 L\n.62952 .60332 L\n.63143 .60332 L\n.63333 .60332 L\n.63524 .60332 L\n.63714 .60332 L\n.63905 .60332 L\n.64095 .60332 L\n.64286 .60332 L\n.64476 .60332 L\n.64667 .60332 L\n.64857 .60332 L\n.65048 .60332 L\n.65238 .60332 L\n.65429 .60332 L\n.65619 .60332 L\n.6581 .60332 L\n.66 .60332 L\n.6619 .60332 L\n.66381 .60332 L\n.66571 .60332 L\n.66762 .60332 L\n.66952 .60332 L\n.67143 .60332 L\n.67333 .60332 L\n.67524 .60332 L\n.67714 .60332 L\n.67905 .60332 L\n.68095 .60332 L\n.68286 .60332 L\nMistroke\n.68476 .60332 L\n.68667 .60332 L\n.68857 .60332 L\n.69048 .60332 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\nMistroke\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\nMistroke\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\nMistroke\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outWait) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n117.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(state) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n147.000 12.813 moveto\n(L) show\n153.000 12.813 moveto\n(@) show\n159.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n165.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00@`3IfMT2000000@0fMWI0P0000030=WIf@@000007`3IfMT2000000@0fMWI0P0000050=WI\nf@<000007P3IfMT2000000@0fMWI0P0000030=WIf@<00000803IfMT2000000@0fMWI0P0000040=WI\nf@8000009@3IfMT5000001/0fMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WI\nf@3IfMT07`3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI01`0fMWI00@00000\n0=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0\nfMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@P3IfMT010000000\nfMWI0=WIf@0000090=WIf@030000003IfMT0fMWI01h0fMWI00@000000=WIf@3IfMT00000203IfMT5\n000001d0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000009P3IfMT00`000000fMWI0=WIf@0K0=WI\nf@00@P3IfMT010000000fMWI0=WIf@00000:0=WIf@030000003IfMT0fMWI01d0fMWI00@000000=WI\nf@3IfMT00000203IfMT010000000fMWI0=WIf@00000N0=WIf@040000003IfMT0fMWI000000P0fMWI\n0`00000O0=WIf@040000003IfMT0fMWI000000T0fMWI0P00000W0=WIf@030000003IfMT0fMWI01/0\nfMWI00120=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT000007P3IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0000001h0fMWI00@000000=WIf@3IfMT000002@3I\nfMT00`000000fMWI0=WIf@0N0=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n00009P3IfMT00`000000fMWI0=WIf@0K0=WIf@00@`3IfMT2000000X0fMWI0P00000P0=WIf@800000\n2`3IfMT2000001l0fMWI0P00000:0=WIf@<000007`3IfMT2000000X0fMWI0P00000V0=WIf@800000\n7@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WI\nf@00703IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000003IfMT0fMWI0?80fMWI\n0P00000=0=WIf@005P3IfMT6000008/0003o5P0000160000o`D00000203IfMT010000000fMWI0=WI\nf@00000<0=WIf@00703IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0\n00000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0\n00000=WIf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000000fMWI0=WIf@0[\n0=WIf@030000003IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0:\n0=WIf@D00000303IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0500fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<000000=WIf@3I\nfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000\n003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703I\nfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03I\nfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@27\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WI\nf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@03\n0000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00\n703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nH03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WI\nf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<00000\n0=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WI\nf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WI\nf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1P0=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT0\n0`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0\nfMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI0P0000280=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000P0fMWI00@000000=WIf@3IfMT000002P3IfMT0\n0`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1P0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0\nfMWI000000@0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0600fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<000000=WIf@3IfMT0Q`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI\n08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000\nfMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI\n00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI\n000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0\nfMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`00\n0000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0\nfMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600\nfMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0\nQ`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003I\nfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT0\n0`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT0\n01`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@1P0=WIf@002@3IfMT2000000@0fMWI0P0000050=WIf@<000000`3IfMT00`000000fMWI\n0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000P0fMWI00@0\n00000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI08L0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00203IfMT010000000fMWI0=WI\nf@0000080=WIf@D000000`3IfMT2000008P0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1P0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0\nfMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600\nfMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@000000103IfMT00`000000\nfMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000T0fMWI\n0P00000;0=WIf@800000103IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3I\nfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000\n003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703I\nfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03I\nfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@27\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WI\nf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@03\n0000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00\n703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nH03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WI\nf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<00000\n0=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WI\nf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WI\nf@002@3IfMT2000000@0fMWI0P0000030=WIf@<000001@3IfMT00`000000fMWI0=WIf@270=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WI\nf@040000003IfMT0fMWI000000@0fMWI0P0000280=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0H03IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0fMWI00<00000\n0=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI00080=WI\nf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@27\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000T0fMWI0P00000:0=WI\nf@<00000103IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI\n08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000\nfMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI\n00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI\n000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0\nfMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`00\n0000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0\nfMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600\nfMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0\nQ`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI00090=WIf@800000103I\nfMT2000000@0fMWI0P0000050=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1P0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003I\nfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0600fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT0\n0000103IfMT2000008P0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00\n203IfMT010000000fMWI0=WIf@0000090=WIf@8000001@3IfMT00`000000fMWI0=WIf@270=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@03\n0000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00\n703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\nH03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WI\nf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<00000\n0=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WI\nf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WI\nf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI\n0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<0\n00000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000L\n0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@1P\n0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0600fMWI000L0=WIf@030000003IfMT0fMWI08L0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@1P0=WIf@00703IfMT00`000000fMWI0=WIf@270=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001`0fMWI00<000000=WIf@3IfMT0Q`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0600fMWI000D0=WIf@D000000`3IfMT00`00\n0000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0H03IfMT001H0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@270=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0H03IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT2000008P0fMWI\n60000?mR0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3I\nfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WI\nf@8000001@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI04h0\nfMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002F0=WIf@00903IfMT00`000000fMWI\n0=WIf@1@0=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@2D0=WIf@008`3IfMT00`000000fMWI0=WIf@030=WIf@800000103IfMT400000080fMWI0P00\n00040=WIf@030000003IfMT000000080fMWI1@0000020=WIf@<000000`3IfMT2000000@0fMWI0P00\n00030=WIf@<00000103IfMT2000000<0fMWI1@0000020=WIf@800000103IfMT3000000@0fMWI00<0\n00000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@020=WIf@800000103IfMT00`000000fMWI0=WI\nf@2D0=WIf@008`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT000000=WI\nf@000000fMWI100000040=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WI\nf@090=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0100000020=WIf@040000\n003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT01@3IfMT00`000000fMWI0=WIf@030=WIf@05\n0000003IfMT0fMWI0=WIf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT0000009H0fMWI000F0=WI\nf@P0003o1@3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@050000003IfMT000000=WIf@000000\n103IfMT00`000000fMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@0000002`3IfMT3000000<0\nfMWI00<000000=WIf@3IfMT01P3IfMT010000000fMWI0=WIf@0000050=WIf@@00000103IfMT00`00\n0000fMWI0=WIf@030=WIf@050000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@2D\n0=WIf@008`3IfMT00`000000fMWI0=WIf@030=WIf@8000000P3IfMT2000000040=WIf@0000000000\n0=WIf@D000000P3IfMT01@000000fMWI0000003IfMT000000080fMWI0`0000030=WIf@8000000P3I\nfMT5000000P0fMWI100000000`3IfMT0000000000003000000<0fMWI0`0000000`3IfMT000000000\n0003000000<0fMWI0P0000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT01@00\n00030=WIf@030000003IfMT0fMWI09@0fMWI000T0=WIf@030000003IfMT0fMWI00d0fMWI00<00000\n0=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT0000000d0fMWI00<000000=WIf@3IfMT03`3I\nfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00X0fMWI00<000000=WIf@3IfMT0103I\nfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0U03IfMT002D0fMWI00<0\n00000=WIf@3IfMT04@3IfMT3000000030=WIf@000000000000T0fMWI00<000000=WIf@3IfMT0:`3I\nfMT00`000000fMWI0=WIf@050=WIf@800000203IfMT2000009H0fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.121768`, -0.155509`, 0.00434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"]], Open]]], Closed]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[928, 907]], Rule[WindowMargins, List[List[146, Automatic], List[Automatic, 0]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]