List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Discrete Event System Specification (DEVS)", "Title"], Cell[CellGroupData[List[Cell["The Random Package", "Section"], Cell["package Random\nimport Modelica.Math;\nconstant Real NV_MAGICCONST = 4 * exp( -0.5 ) / sqrt( 2.0 );\ntype Seed=Integer[3];\n\nfunction random \"input random number generator with external storage of the seed\"\n  input Seed si \"input random seed\";\n  output Real x \"uniform random variate between 0 and 1\";\n  output Seed so \"output random seed\";\nalgorithm\n  so[1] := rem((171 * si[1]),30269);\n  so[2] := rem((172 * si[2]),30307);\n  so[3] := rem((170 * si[3]),30323);\n  // zero is a poor Seed, therfore substitute 1;\n  if so[1] == 0 then\n    so[1] := 1;\n  end if;\n  if so[2] == 0 then\n    so[2] := 1;\n  end if;\n  if so[3] == 0 then\n    so[3] := 1;\n  end if;\n  x := rem((so[1]/30269.0 +so[2]/30307.0 + so[3]/30323.0),1.0);\nend random;\n\nfunction normalvariate \"normally distributed random variable\"\n  input Real mu \"mean value\";\n  input Real sigma \"standard deviation\";\n  input Seed si \"input random seed\";\n  output Real x \"gaussian random variate\";\n  output Seed so \"output random seed\";\nprotected\n  Seed s1, s2;\n  Real z, zz, u1, u2;\n  Boolean break=false;\nalgorithm\n  s1 := si;\n  u2 := 1;\n  while not break loop\n    (u1,s2) := random(s1);\n    (u2,s1) := random(s2);\n    z := NV_MAGICCONST*(u1-0.5)/u2;\n    zz := z*z/4.0;\n    break := zz <= (- Math.log(u2));\n  end while;\n  x := mu + z*sigma;\n  so := s1; \nend normalvariate;\n\nconnector discreteConnector\n   discrete Boolean dcon;\nend discreteConnector;\n\nend Random;", "ModelicaInput"], Cell["\nDymolaBUG: for some strange  reason Dymola crashes when the following statement is initialized\ntype Seed=Integer[3](min=zeros(3),max=ones(3)*255);\nthis will also cause MathModelica to loop forever in the evaluation of the model.\nThe follwing statement should be used instead\ntype Seed;\n", "Text"]], Open]], Cell[CellGroupData[List[Cell["DEVS Model", "Section"], Cell["We will now describe a fairly comprehensive example on how DEVS models can be represented in Modelica. One difference in Modelica is that we have a built-in time variable for absolute time rather than a local timer tlocal that is reset at the most recent event. This local timer is not really needed for modeling purposes since time is available, but is included to show the relation to traditional DEVS models. The facilities for model structuring and reuse are in general much greater in Modelica due to its object-oriented properties compared to traditional DEVS. In order to simplify comparisons this example connects model components such as servers or generators in a way that is analogous to traditional DEVS.\n", "Text"], Cell["connector EventPort\n\tBoolean signal;\nend EventPort;", "ModelicaInput"], Cell["", "Text"], Cell["partial model BasicDEVSTwoPort\n\n  import Modelica.Constants.inf; \t\t\t// import constant inf\n  input  EventPort  inp;         \t\t\t// Input port for external events\n  output EventPort  outp;        \t\t\t// Output port for emitted events\n  Boolean   DEVSactive(start=false);\t\t// Start in non-active mode (phase)\n  Real      tResidual(start=inf);    \t\t// Time until internal event (sigma)\n  Real      tlocal(start=0);         \t\t// Local timer reset at events (e)\n  discrete Real tNextEvent(start=0.0001); \t\t// Time of next internal event\n  discrete Real tLastEvent(start=0);   \t\t// Time of most recent event\nequation\n  tResidual = if tNextEvent<time // tNextEvent==inf\n              then  inf  else  tNextEvent-time;\n\ttlocal = if inp.signal or tNextEvent<=time // extern or intern ev\n           then 0 else time - tLastEvent;\n  when {inp.signal, pre(tNextEvent)<=time} then\n    tLastEvent = time;                     // Note the time of this event\n  end when;\nend BasicDEVSTwoPort;\n", "ModelicaInput"], Cell["\nA simple job shop model illustrates DEVS based modeling. A generator for arriving jobs is connected to a simple DEVS based server.\n", "Text"], Cell[GraphicsData["Metafile", "CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@0002Q00@0006`00000000000000;L100290000\n00000000001O:000L0d00215CDH00040@2P00>H0000;0000000000000000000000@00005003`0000\n@0400000000000000000082Y0`00hP@0AP0000P<003j2`00AdA9@`4008000`00NJ5MAP00003R2`00\n0@090003l@D000`09@000000500002H63`0N0?ooool401@0001GKg9T3P1=JF=bKg=_IW@PEfmbI0D0\n000;0P000005000030:208L170000?/2l?l00000002@0@000004@00BE6U]IG<PCVEg859_KF5^0;iR\nmGO7H_Eg0@000000<0040000;@4000D000020@40000500002@8000020`0001h01`0001H4PP270@00\n000@00009PH?01H0ooooo`00oooooooooon80@00P`0000<0000N00D0000^0@00000500002P800000\n1@0000T2000000D000010_ooo`070000o0810000000000@0000]0@4020000?X21@010000000000@0\n000]0@80100000<1200500002`8000001@0000`2;@><2A@0000V1Pl07P3ooooo100D0000EfmbI0h0\nCFUSLVmcKfId85M_LV@50000308]0h`970000?/250090000002l0P0000040P8RDgUcM6E]07M00000\n/PP:_;iRmGO7H_Eg0@000000<0040000;@4300D000020@40000L0000n`:]o`0000000901000000A0\n019DJFeULb1>IGLPDVm]HFh0_V;eMlMRmGL10000000`00@0000]0@@01@0000T200000PD0000;0P00\n000=0000<PX100000@040000002<2Bd386TU00@0000]0@40100002d10P0700006`A_0>/8:@2/20L0\n003l0P00oooo0000100002d11@080000nP80000000000000100002d11P0300007P0700005PA]0>X8\n:@2/20D000020@40000L0000n`:]o`0000000901000000A00291LVUQK03eMd00002d20Xg_V;eMlMR\nmGL10000000`00@0000]0@L04000038:8`2/20<0102/22T0j@Q]020P804G01L05`0L0000n`:eo`00\n00000901000000A0019DJFeULb1>IGLPDVm]HFh0_V;eMlMRmGL10000000`00@0000]0@P03@00038:\n:P3a2040102/22T0j@Q]02008P0400009`7oo`L0003l0P00oooo0000100002d12@080000nP8000H0\n00000002100002d12P0700006`A70_87?@7b10@0000]0@D010000?012@040000;@4600@0003`0@X0\n100002d10@040000;@4200L0000K1?@1`0NA0Bl5100002d11@040000;@4600<0000N00L0000F1?81\n`0NA0Bl51@0000810@0001`0003k0[Go00000000T040000014008T5bJF5/0?Eg@0000;T82QnnH_Eg\naf;eM`4000000300100002d12@0500002@8000028`00038:TP4l1A00100_1I41_`Ob0G=YKG1/IDA5\nEU=cIG9fIG8V0140?P0Z0140:P0f0380<P0a02H0:P0I02D0:P0I00@0000]0@P03@00038:TP6b1`40\n100_1I41_`Ob0B008@0400009`7oo`@0000]0@40100002d10P0700006`@[0d00iP8200@0000]0@D0\n100002d11P0300007P0700005P@Z0d00iP8200D000020@4000040000;@4700D000090P00008@0000\n<P[P0P800`040080iP8o02X3820P0AL05`0G00@0000]0@P03@00038:i`97004010020>H2?`0Z0b00\n8P0400009`7oo`L0003l0P00oooo0000100002d12P080000nP8000H000000002100002d12`070000\n6`CP0A05X`7D10@0000]0@D010000?012P040000;@4600@0003`0@/0400002H63`0F0?ooool003L4\n002Y0@00eP@00=`100080000nP8000H000000000100002d12P040000;@4100P0000U0`80>`C20JT4\n`P440000;@4200@0003`0@X01`000?`20000000000040000;@4:00X0000T0`<0Y`CH0M84`P6W1:d1\n100002d11P040000;@4500@0003`0@X0200002H63`060?ooool100@0000]0@40100002d10P070000\n6`A408/9oom<2@@0000]0@D0100002d11P0300007P0700005PA308X9oom<2@D000020@4000040000\n;@4700D000090P00008@0000<P[iod`90`0404`9oon92D<0820P01L05`0G00@0000]0@P03@00038:\n002A2@40101<2OooR@U302008P0400009`7oo`@0000]0@40100002d10P0700006`@>0Md5YP0F0`@0\n000]0@D0100002d11P0300007P0700005P@=0Md5YP0G0`D000020@4000040000;@4700D000090P00\n008P0000<PZX04/33P0401L3YP3L1@d1M6EcM4A5EU=CIG9fIG8G02h0:P0G03`0>00g03P0=`0^01d0\n:@0^01`0100002d1100=0000<PZX0:T50@0401L3YP3L1@d1800V00@0000W0Ooo1`000?`2003oool0\n00040000;@4:00P0003j0P001P00000000840000;@4;00L0000K12D25PAV0C81100002d11@040000\nl04:00@0000]0@H010000?012`040000;@4100@0000]0@801`0001/4n07J0h`1D`440000;@4500@0\n000]0@H00`0001h01`0001H4mP7J0h`1D`4500000P410000100002d1100500002@8000029@00038:\nS`5N0A40101C0H`1f@?f0FMUKVEbHGAUBVmRAGIUKWAc02T09@0Y02D0700U01L09@0Q02X0:P0c02T0\n9@0Y01P080040000;@4800d0000b2YH1cP<100@0D`6<0MT3mP4P02<0100002L1ool70000o0800?oo\no`0000@0000]0@X020000?X20006000000000P@0000]0@/01`0001/4h04g1:<1lP<40000;@4500@0\n003`0@X0100002d11P040000l04;00@0000]0@4020000?X20006000000000P@0000]0@X01`0001/4\n`@;A2780`@040000;@4500@0000]0@H010000?012P070000o0800?ooo`0000@0000]0@X020000?X2\n0006000000000P@0000]0@/01`0001/4h04@2:<1d`L40000;@4500@0003`0@X0100002d11P040000\nl04;0100000V1Pl05P3ooooo000>2000Z0400:l8003K0@0020000?X200060000000000@0000]0@X0\n100002d10@0800009@<20188`P622<41100002d10P040000l04:00L0003l0P0000000000100002d1\n2P0:000090<30808e`6Z2<01O`R/0@@0000]0@H0100002d11@040000l04:00P0000V1Pl01P3ooooo\n0@070000o0800?ooo`0000@0000]0@X020000?X20006000000000P@0000]0@/01`0001/4hP7/2:H1\n[`P40000;@4500@0003`0@X0100002d11P040000l04;00@0000]0@<010000?0110040000l04700@0\n003`0@P010000?012@070000o0800?ooo`0000@0000]0@@020000?X200000000000000@0000]0@L0\n100002L1ool800009PH?00H0ooooo`40100002L1oolL0000n`8@00L000000;`2000000420R9CNG=d\nIFd000002P0000@0000000P000010000000`00@0000]0@P00`000000000A0000300000P0000;0000\n40000;T1002:00002@000100002i0@00RP0000X0000@00000000000000090000400008L100220000\nDP00070100010000l?ooo`0000000000000009010000000014004U@0J@1]06D0L`0P04h0I@1g0200\nDP1_06d0H@1^0000000000000000000000000000000000000000000000000180008000004`10K1T0\nN04C07P2001@]_ag5[GlMbfeo7L00000P0000:056@0001<0:0`C01h00010hAH0W?<B0002001HmA80\n@8/H07P14`3oooooJ?DB04064`0X31<000000000003k224E03Y20R@00000gHKnY1J<0@00600h31T0\n1?@B000000249i@1L0aJ09SKF@300000oMSooo3Hooodf?oo0020?`@0003bP_oo0000020K0`0001<0\n@8/H07P14`1h0A<0>8/H052fo7LF]Oag;KGlMe044`1@11<02@000:056@2Xm180A;LH0<;0m7L10000\nI7H02000000U000030000040000B000030000040000H000030000000008Q0000200001h0000H0000\n0000000000270@00PP000240000800005P0000`000000000N000010000000000000001P0000<0000\n000001T0000<0000oooo02D0000<00001@00P2D0000<00002000P0X0000@00000000000000090000\n400008`9000]0`002@000100002<2@00;@<00580001`0@000P0001@00009000000000000002l0P00\n000000@20R9C07T0L`1d06D0K@000000000000000000000000000000000000000000000000000000\n0000000000000000000300004`10RaP0N04C07P14`0hRaP0D;KlMaJeo7L]]OagD0@C05044`090000\nX0DI0:Sd4P14]aP0`/3dM`40001TMP080000042;601h0A<0ooooofSe4P101Q<0:0`C000000000000\nn`PQ5@0j@P8T00000=f6oZ@FS04001P0>0`I00Cd4P000000Q2ND0G0<FP2HfeT0`0000?gHooo`f?oo\nm=Soo`00P3l@8_Aga?3mO`3PoGl100000000000000000000MPD00010001f1@B;0@0006Cd4P0U0000\nX0DI00T0002P1AT0Z?@B07Jh6032`?Ag0@0006Af00P000009@0000`0000200004P0000`000010000\nDP00070100030000[Oooo`0000000000000009010000000014004U@0J@1]06D0L`0P04h0I@1g0200\nDP1_06d0H@1^00000000000000000000000000000000000000000000000000000000000000000000\nMPD00010001f1@B;0@0006Cd4P0U0000X0DI00T0002P1AT0Z?@B07Jh6032`?Ag0@0006Af00P00000\n@8/H07P14`3oooooJ?DB04064`0X31<000000000003k224E03Y20R@00000gHKnY1J<0@00600h31T0\n1?@B000000249i@1L0aJ09SKF@300000oMSooo3Hooodf?oo0020?a0Rm7O4l?eo0>3mO`4000000000\n00000000001f1@00040007H518/10000I?@B02D0002P1AT0R?@B0;@8001D`?Ag0@0001T0/06d20Xl\nI7H02000000U0000300000<0000H000030000000008:00004000000000000000E00005@000000000\n000000H0000?00000@000=QDQd0IbhM00000004000010000C00000@000000000000008`9000]0`00\nD0000200000U00009@0000`0000500209@0000`000080020:`0001P0002/2000:@000>D8001Y0000\n9`0001P00004000000000?ooo`0000009@0000`0000400009P0001`0000500000000000000000000\n000002D0000<00001@000240000800007P0001P0002/2000:@000>X8001]00004P0000`000010000\nDP00070100060000[Oooo`0000000000000009010000000014008T40LP1Y0640K000000000000000\n00000000000000000000000000000000000000000000000000000000000000000000000000000000\nMPD00010001f1@B;0@0006Cd4P0U0000X0DI08Sd4P3<la8000000>Dhn7L001<0<14H0000002Xla80\nR0HC05cd4P2E:oQg83WhMooooom/m180@;OlMnP64`0@8_Aga?3mO`3PoGl1000000000000001f1@00\n040007H518`100007?@B0?0am7Mf1@B<F1hI0040001f1@B<18`0010Rm7O4l?eo0>3mO`4000000000\n00000000001f1@00040007H518d10000I?@B0?0am7Mf1@B=n`PQ5GH518d0000018d0083d4P2@d_Ag\nI7H02000000U0000300000H0001D0000H000094100070000V`40018000010000f5B7@1W;Qd2/2000\n8`0000<0001<000010000:`8000Y0000j@P006d0001D0000800P0200000G00005`0001L0001B0000\nL04000L0002eoooo0000000000000000T04000000004@00BE01Y06d0I@1c0200CP1U07L0801B06l0\nK@1Q06h00000000000000000000000000000000000000000000000000000000000000000001f1@00\n040007H518d10000I?@B0?0am7Mf1@B=n`PQ5Lcc4P000000iCShM`004`00QaL000000:Sc4P281Q<0\nG?@B09D[n7LP>OQgooooofcd4P10]oag60LC00R75`08QaL0o;TH00R75`000000000007H50000@000\nMPD4S040000Lm180l37dM`004`1H7QT00@0007H518`4S00002;dM`<00000h?eo0@00000000000000\n000000004`03000008LG01P0001Tm1800@0000<0003Lla80H0000:Cd4P2E:oQg63KhMooooomTMP08\n000002D0000<00001`0005@0001D000000000000003oooooooooo`40003HE8M06L^7@?48000Z0000\n0@0004`000040000[0P002T0003Y2000K@000500000P00008P000280000<0000ooooobL0000H0000\n20000000003oool0000002D0000<0000200002H0000L00002@0000000006000000000000008U0000\n300000T0000[000060000?84000m0@00k0L00442000U0000300000@0000X0000300000P0000U0000\n300000D0000X0000300000T0000U0000300000D0080U0000300000P0080[0000600002l5002A0@00\n^PL00>h1000U0000300000@0000U0000300000D0000Q0000200001h0000H0000;`D0094100301`00\nlP400180000<00000@000580001`0@002@000;Goool0000000000000002@0@00000000A002910780\nJ@1Q06`0000000000000000000000000000000000000000000000000000000000000000000000000\n000G01Qn5`000000000000004`03000008LG01P0001Tm1800@0000<0003Lla80c?<B0000003U>?Qg\n000C030A60000000Z?<B08P64`1Lm180UB_hM`004`0400000?@B042go7OX1Q<00A4H010Rm7L@8_Ag\na?3mO`3PoGl0000000000000000l2@00040003`919<100009?@B0?0am7Ll2@BCn`PQ5@40000@8_Ag\na?3mO`3PoGl100000000000000000000MPD00010001f1@B@0@0006Cd4P3`<OAgMPD4T?/88AEf1@B@\n000000B@0020m180T=;dMfAf00P000009@0000`000090000600000`000000002E0000:`0003b0000\nA00006<1001C00000@000=QDQd0IbhM0?0D00981000@0000C00000@0000_1@00T@400;l7003b0@00\nK00007<0J@1]0700K01U04@0A@1F05<0L`1U0780MP1U07809P000140000n0000:P000140000Z0000\n=P000380000b0000<@0002H0000Z00006@0002D0000Z00006@0002D0000<00001`0005@0001D0000\nI04004@0001U0@00DP000040003HE8M06L^7@;87002B0@000@0004`000040000;`D00941002o1`00\nlP400500000P00008@000280000<0000ooooobD0000<00001@00P2D0000<00002000P2/0000H0000\n0P000>H2000k00009@<002D0000<0000100002D0000<00001@000240000800007P0001P000020000\niP800400000Z0`004P0000`0000100009@0000`000060000600000`000000002E000060000000000\nOP0000X0002900000@000=QDQd0IbhM00P000>0200030000C00000@000020000iP8003l0000Z0`00\nE0000200800P00005`0001L0000G00009@0000`000070000E00005@00000000000000?oooooooooo\n0@000=QDQd0IbhM0A`000>L200010000C00000@000020000iP8003l0000Z0`00D0000200000R0000\n8P0000`0003ooooo9`0001P00008000000000?ooo`0000009@0000`0000800009P0001`0000:0000\n000000H00000000000000RD0000<00002P0002/0000H0000e0@00:<1000;1@00fP4002D0000<0000\n100002P0000<0000200002D0000<00001@0002P0000<00002P0002H0000L00002P00000000060000\n00000000000U0000300000X0000U0000300000D0081G000090000<<0001<0000e`0004`000020000\n>`C20JT4`P4U0000300000P0080X0000300000X0000W0000600000X00000000000000000000U0000\n300000X0001F0000:0000=L000190000g`00050000030000Y`CH0M84`P6W1:d19@0000`000050000\n9@0000`000040000:00000`0000:00009@0000`0000500209@0000`000080020:`0001P0001<2@00\nooooohD9000n00009@0000`0000400009@0000`0000500008@0000P0000N0000600004`9003ooooo\nRPT004<0000B000030000040000U0000300000H0000H000030000000009D0000H0000:d100000000\n]`4000X000010000f5B7@1W;Qd1<2@00nOooo`<0001<0000100004`9003oooooR@T004<0001D0000\n800P0200000G00005`0001L0000U0000300000L0001D0000E000000000000000ooooooooool10000\nf5B7@1W;Qd2A2@0000000040001<0000100004`9003oooooR@T004<0001@000080000280000R0000\n30000?oooolU0000300000D0080U0000300000P0080[0000600001H3002V0000e`D000P1000U0000\n300000@0000U0000300000D0000Q0000200001h0000H00005`<00:H0003M1@003@400180000<0000\n0@0002D0000<00001P0001P0000<000000000U@0002P0000V00001d000050@00;0000040003HE8M0\n6L^7@4/3002X00003P0004`0000400005`<00:H0003L1@003@4006P0001d06D0L`1d04@0A@1F05<0\nD`1U0780MP1U07805`0002h0000Z00005`0003`0000h0000=`0003P0000g0000;P0001d0000Y0000\n;P0001`0000U0000300000<0001D0000E00000D1000M00002`4002`000010000f5B7@1W;Qd2Y1@00\nZ0000040001<0000100001L3002V0000g0D000d1001@0000800022H0000R000030000?oooolW0000\n600000X000000000oooo0000000U0000300000X0000V0000700000P0000000001P00000000000002\n9@0000`000080000:`0001P0000b0@00IP400144000O0P009@0000`000040000:00000`0000:0000\n9@0000`000050000:00000`0000800009@0000`0000500209@0000`000080020:`0001P0001C0@00\nS0400=D3003b0@009@0000`0000400009@0000`0000500008@0000P0000N0000600005<1002<0@00\nfP<00?H1000B000030000040000U0000300000<0000H000030000000009D0000]00003l000140000\n[`0005<000010000f5B7@1W;Qd1N0@00S`400140001<0000100005<1002<0@00f@<00?H1001`0000\nI`1U06h0I@1b0640M01U04X0K`1R04D0MP1U06h0M01c0000:@0002D0000Y00009@0001`0000U0000\n5`0002D0000Q0000:P0002X0000c0000:@0002D0000Y000060000200000U0000300000L0001D0000\nE0000;0000150000/@0005<000010000f5B7@1W;Qd3>0`00UP400040001<0000100005<1002<0@00\nf@<00?H1001@0000800002<0000R000030000?oooolW0000600000P000000000oooo0000000U0000\n300000P0000V0000700000X0000000001P000000000000029@0000`0000:0000:`0001P0003b0`00\nX`400344003J0@009@0000`000040000:00000`0000800009@0000`000050000:00000`0000:0000\n9@0000`0000500209P0001`0000:0000000000H00000000000000RD0000<00002P0002/0000H0000\n`@000780003;2000^`8002D0000<0000100002D0000<00001@0002P0000<00002P0002L0000H0000\n2P000000003oool0000002D0000<00002P0002H0000L0000200000000006000000000000008U0000\n300000P0000[000060000=<7002S0@002PP00=X1000U0000300000@0000X0000300000X0000U0000\n300000D0000X0000300000P0000V0000700000P0000000001P000000000000009@0000`000080000\n9@0000`000050020E`0002@0001e0@00C00008T1001<00000P000188`P622<419@0000`000080020\n:00000`0000800009`0001P00008000000000000000000009@0000`000080000EP0002P000280@00\nB@000901001@00000`000808e`6Z2<01O`R/0BD0000<00001@0002D0000<0000100002P0000<0000\n200002L0000H000020000000003oool0000002D0000<0000200002H0000L00002P00000000060000\n00000000008U0000300000X0000[000060000:l8002V0@00iPP00=`1000U0000300000@0000X0000\n300000P0000U0000300000D0000X0000300000X0000U000030000080000X0000300000<0000X0000\n300000H0000X0000300000L0000X0000300000T0000W0000600000T000000000oooo0000000U0000\n300000T0000V0000700000L00000000000000000000000009@0000`0000700008P0000`0003ooooo\n8P0000`0003oooooDP00070100060000400000L00000000000000;`2000000000@828U<0N@1c07@0\nI@1]00000000000000000000000000000000000000000000000000000000000000000000000001L0\n67hG0680A@1f06D0KP1d07<0001f1@B@0@0006Cd4P3`<OAgMPD4T?/88AG<la8000000>Dhn7L001<0\nR4@G0000002Xla80R0HC05cd4P2E:oQg000C00@00000m180@;OlMdP74`01A1L00`0006Sc4P000000\n<?@B0000n7LH=_Qgoooood3d4P22LnQg000C0000000HOQL0l`P:AAQn5`0PP?Mg19h0003d4P040000\nfe;dM`cd4P000000`?DB05HHjWN`MNQg00000?ooo`00000067hG0;g6m7Ok224EL?@B0060001Pa_Ag\nn`PQ5G3d4P2`P?MgI7H02000000U0000300000H0000U0000300000L0080U000030000000080`0000\n300000l0080U0000300000d0081;00004000000000050000:00000`000010000:00000`000020000\n:00000`000090000:00000`000040000:00000`000050000:00000`000070000:00000`000060000\n3P0001@000000000400001@0\n"], "Text", Rule[ImageSize, List[439, 137]], Rule[ImageMargins, List[List[0, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]]], Cell[TextData[StyleBox["The testDEVSServer model connects a job generator to a simple DEVS-based server.", Rule[FontFamily, "Courier New"]]], "NumberedFigure"], Cell[CellGroupData[List[Cell["testDEVSServer Model", "Subsection"], Cell[TextData[List["The ", StyleBox["testDEVSServer", Rule[FontFamily, "Courier"]], " model contains two main items: a ", StyleBox["generateJobEvents", Rule[FontFamily, "Courier"]], " generator component that generates arriving jobs at regular intervals, and a ", StyleBox["simpleDEVSServer", Rule[FontFamily, "Courier"]], " server component being a very simple model of a job shop system that services arriving jobs with a normally distributed service time. The ", StyleBox["SignalView", Rule[FontFamily, "Courier"]], " class instances are needed to widen sharp peaks for plotting.\n"]], "Text"], Cell["model testDEVSServer\n  GenerateJobEvents  generateJobEvents(period=1.5,startTime=3);  \n  SimpleDEVSServer   simpleDEVSServer;   \n  SignalView         viewSignal1;   // Not shown, just for plotting\n  SignalView         viewSignal2;   // Not shown, just for plotting\n equation\n  connect(generateJobEvents.outp, simpleDEVSServer.inp);\n  connect(simpleDEVSServer.outp, viewSignal2.inp);\n  connect(generateJobEvents.outp, viewSignal1.inp);  // Just for plotting\nend testDEVSServer;", "ModelicaInput"], Cell["", "Text"], Cell["model GenerateJobEvents  \"Generates Job events periodically\"\n\toutput EventPort outp; // Output port\n  parameter Real startTime = 0;\n  parameter Real period = 2;\nequation\n  outp.signal = sample(startTime,period);\nend GenerateJobEvents;", "ModelicaInput"], Cell["", "Text"]], Open]], Cell[CellGroupData[List[Cell["SimpleDEVSServer", "Subsection"], Cell[TextData[List["The ", StyleBox["SimpleDEVSServer", Rule[FontFamily, "Courier"]], " class is a fairly straightforward extension of the basic DEVS class BasicDEVSTwoPort. A randomly distributed service time variable ", StyleBox["servTime", Rule[FontFamily, "Courier"]], " has been added with a normal distribution determined by parameters ", StyleBox["mean", Rule[FontFamily, "Courier"]], " and ", StyleBox["stdev", Rule[FontFamily, "Courier"]], ". The discrete ", StyleBox["randomSeed", Rule[FontFamily, "Courier"]], " variable is needed to store the seed between calls to ", StyleBox["normalvariate", Rule[FontFamily, "Courier"]], ", which as all Modelica functions is a mathematical function whose value is completely determined by its inputs. The absolute value of ", StyleBox["servTime", Rule[FontFamily, "Courier"]], " is used to guarantee positive service times.\n"]], "Text"], Cell["model SimpleDEVSServer\n  import Modelica.Constants.inf; // import constant inf\n  extends BasicDEVSTwoPort;\n  Boolean   internalEvent;\n  Boolean   externalEvent;\n  parameter Real  mean  = 2.0  \"mean of job service time\";\n  parameter Real  stdev = 0.5;\n  discrete  Real  servTime;  // time needed to service current job\n discrete  Random.Seed randomSeed(start={23,87,187});\n \nequation\n  internalEvent = pre(tNextEvent)<=time;\n  externalEvent = inp.signal;\n  //outp.signal   = internalEvent; \n  when {externalEvent,internalEvent} then\n    (servTime,randomSeed) = Random.normalvariate(mean,stdev, pre(randomSeed));\n    if edge(externalEvent) and not pre(DEVSactive) then    \t\t\t\n    \tDEVSactive = true;\n    \ttNextEvent  = time + abs(servTime); \n    elseif edge(externalEvent) and pre(DEVSactive) then\n      DEVSactive = true;\n      tNextEvent  = pre(tNextEvent); \n    else\n      DEVSactive = false;\n      tNextEvent = pre(tNextEvent); \n    end if;\n  end when;\n  \n  outp.signal = if pre(DEVSactive) and edge(internalEvent) then true else false;\nend SimpleDEVSServer;", "ModelicaInput"], Cell["", "Text"], Cell["model SignalView  \"Increase width of sample trigger signals\"\n\tEventPort    inp;\n\tEventPort    outp;\n\tparameter Real width=0.001;\n\tdiscrete  Real T0;\nequation\n\twhen inp.signal then\n    T0=time;\n\tend when;\n\toutp.signal=(time>=T0) and (time<T0+width);\nend TriggerSignalView;", "ModelicaInput"], Cell[TextData[List["\nRemember the following two connect equations from the testDEVSServer model. When using the currently available Modelica tools it is necessary to employ some kind of ", StyleBox["SignalView", Rule[FontFamily, "Courier"]], " peak widening facility in order to plot sharp peaks that have zero width, i.e. have no time duration. Otherwise these peaks would not be visible in the diagram.\n"]], "Text"], Cell["class Conn1\nequation\n  connect(simpleDEVSServer.outp, viewSignal2.inp);\nend Conn1;", "ModelicaInput"], Cell["", "Text"], Cell["class Conn2\nequation\n  connect(generateJobEvents.outp, viewSignal1.inp);\nend Conn2;", "ModelicaInput"], Cell["", "Text"]], Open]], Cell[CellGroupData[List[Cell["Simulation of testDEVSServer", "Subsection"], Cell["We simulate the testDEVSServer model to be able to view some of the input and output signals. The plot agrees with the specification of the generateJobEvents instance, where the first event is generated at time = 3 and subsequent events are generated with a periodicity 1.5 seconds.\n", "Text"], Cell["simulate( testDEVSServer, stopTime=10 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(testDEVSServer\\) : 2004-2-3 13:7:10.8734826 : \\!\\({0.`, 10.`}\\) : 529 data points : 9 events : 28 variables>\\n\\!\\({\\*TagBox[\\\"\[UnderBracket]derdummy\\\", HoldForm], \\*TagBox[\\\"\[UnderBracket]dummy\\\", HoldForm], \\*TagBox[\\(generateJobEvents . outp . signal\\), HoldForm], \\*TagBox[\\(generateJobEvents . period\\), HoldForm], \\*TagBox[\\(generateJobEvents . startTime\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . DEVSactive\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . externalEvent\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . inp . signal\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . internalEvent\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . mean\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . outp . signal\\), HoldForm], \\*TagBox[\\(\\(\\(simpleDEVSServer . randomSeed\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(simpleDEVSServer . randomSeed\\)\\) \[LeftDoubleBracket] \\(\\(2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(simpleDEVSServer . randomSeed\\)\\) \[LeftDoubleBracket] \\(\\(3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . servTime\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . stdev\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . tLastEvent\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . tlocal\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . tNextEvent\\), HoldForm], \\*TagBox[\\(simpleDEVSServer . tResidual\\), HoldForm], \\*TagBox[\\(viewSignal1 . inp . signal\\), HoldForm], \\*TagBox[\\(viewSignal1 . outp . signal\\), HoldForm], \\*TagBox[\\(viewSignal1 . T0\\), HoldForm], \\*TagBox[\\(viewSignal1 . width\\), HoldForm], \\*TagBox[\\(viewSignal2 . inp . signal\\), HoldForm], \\*TagBox[\\(viewSignal2 . outp . signal\\), HoldForm], \\*TagBox[\\(viewSignal2 . T0\\), HoldForm], \\*TagBox[\\(viewSignal2 . width\\), HoldForm]}\\)\""], "Output"], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( { viewSignal1.outp.signal, viewSignal2.outp.signal, simpleDEVSServer.DEVSactive } );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .60332 L\n.30962 .60332 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .60332 L\n.45248 .60332 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .01472 L\n.5511 .01472 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59533 .60332 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .60332 L\n.73819 .60332 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .01472 L\n.78894 .01472 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 .5 0 r\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .01472 L\n.30962 .01472 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .01472 L\n.45248 .01472 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .60332 L\n.5511 .60332 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .01472 L\n.59533 .01472 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .01472 L\n.73819 .01472 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .60332 L\n.78894 .60332 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .01472 L\n.88105 .01472 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n1 0 0 r\n.02381 .01472 m\n.02382 .01472 L\n.02382 .01472 L\n.0239 .01472 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .60332 L\n.30962 .60332 L\n.30962 .60332 L\n.31143 .60332 L\n.31333 .60332 L\n.31524 .60332 L\n.31714 .60332 L\n.31905 .60332 L\n.32095 .60332 L\n.32286 .60332 L\n.32476 .60332 L\n.32667 .60332 L\n.32857 .60332 L\n.33048 .60332 L\n.33238 .60332 L\n.33429 .60332 L\n.33619 .60332 L\n.3381 .60332 L\n.34 .60332 L\n.3419 .60332 L\n.34381 .60332 L\n.34571 .60332 L\n.34762 .60332 L\n.34952 .60332 L\n.35143 .60332 L\n.35333 .60332 L\n.35524 .60332 L\n.35714 .60332 L\n.35905 .60332 L\n.36095 .60332 L\n.36286 .60332 L\n.36476 .60332 L\n.36667 .60332 L\n.36857 .60332 L\n.37048 .60332 L\n.37238 .60332 L\n.37429 .60332 L\n.37619 .60332 L\n.3781 .60332 L\n.38 .60332 L\n.3819 .60332 L\n.38381 .60332 L\n.38571 .60332 L\n.38762 .60332 L\n.38952 .60332 L\nMistroke\n.39143 .60332 L\n.39333 .60332 L\n.39524 .60332 L\n.39714 .60332 L\n.39905 .60332 L\n.40095 .60332 L\n.40286 .60332 L\n.40476 .60332 L\n.40667 .60332 L\n.40857 .60332 L\n.41048 .60332 L\n.41238 .60332 L\n.41429 .60332 L\n.41619 .60332 L\n.4181 .60332 L\n.42 .60332 L\n.4219 .60332 L\n.42381 .60332 L\n.42571 .60332 L\n.42762 .60332 L\n.42952 .60332 L\n.43143 .60332 L\n.43333 .60332 L\n.43524 .60332 L\n.43714 .60332 L\n.43905 .60332 L\n.44095 .60332 L\n.44286 .60332 L\n.44476 .60332 L\n.44667 .60332 L\n.44857 .60332 L\n.45048 .60332 L\n.45238 .60332 L\n.45238 .60332 L\n.45248 .60332 L\n.45248 .60332 L\n.45429 .60332 L\n.45619 .60332 L\n.4581 .60332 L\n.46 .60332 L\n.4619 .60332 L\n.46381 .60332 L\n.46571 .60332 L\n.46762 .60332 L\n.46952 .60332 L\n.47143 .60332 L\n.47333 .60332 L\n.47524 .60332 L\n.47714 .60332 L\n.47905 .60332 L\nMistroke\n.48095 .60332 L\n.48286 .60332 L\n.48476 .60332 L\n.48667 .60332 L\n.48857 .60332 L\n.49048 .60332 L\n.49238 .60332 L\n.49429 .60332 L\n.49619 .60332 L\n.4981 .60332 L\n.5 .60332 L\n.5019 .60332 L\n.50381 .60332 L\n.50571 .60332 L\n.50762 .60332 L\n.50952 .60332 L\n.51143 .60332 L\n.51333 .60332 L\n.51524 .60332 L\n.51714 .60332 L\n.51905 .60332 L\n.52095 .60332 L\n.52286 .60332 L\n.52476 .60332 L\n.52667 .60332 L\n.52857 .60332 L\n.53048 .60332 L\n.53238 .60332 L\n.53429 .60332 L\n.53619 .60332 L\n.5381 .60332 L\n.54 .60332 L\n.5419 .60332 L\n.54381 .60332 L\n.54571 .60332 L\n.54762 .60332 L\n.54952 .60332 L\n.55101 .60332 L\n.55101 .01472 L\n.5511 .01472 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59533 .60332 L\n.59533 .60332 L\n.59714 .60332 L\n.59905 .60332 L\n.60095 .60332 L\n.60286 .60332 L\n.60476 .60332 L\n.60667 .60332 L\n.60857 .60332 L\n.61048 .60332 L\n.61238 .60332 L\n.61429 .60332 L\n.61619 .60332 L\n.6181 .60332 L\n.62 .60332 L\n.6219 .60332 L\n.62381 .60332 L\n.62571 .60332 L\n.62762 .60332 L\n.62952 .60332 L\n.63143 .60332 L\n.63333 .60332 L\n.63524 .60332 L\n.63714 .60332 L\n.63905 .60332 L\n.64095 .60332 L\n.64286 .60332 L\n.64476 .60332 L\n.64667 .60332 L\n.64857 .60332 L\n.65048 .60332 L\n.65238 .60332 L\n.65429 .60332 L\n.65619 .60332 L\nMistroke\n.6581 .60332 L\n.66 .60332 L\n.6619 .60332 L\n.66381 .60332 L\n.66571 .60332 L\n.66762 .60332 L\n.66952 .60332 L\n.67143 .60332 L\n.67333 .60332 L\n.67524 .60332 L\n.67714 .60332 L\n.67905 .60332 L\n.68095 .60332 L\n.68286 .60332 L\n.68476 .60332 L\n.68667 .60332 L\n.68857 .60332 L\n.69048 .60332 L\n.69238 .60332 L\n.69429 .60332 L\n.69619 .60332 L\n.6981 .60332 L\n.7 .60332 L\n.7019 .60332 L\n.70381 .60332 L\n.70571 .60332 L\n.70762 .60332 L\n.70952 .60332 L\n.71143 .60332 L\n.71333 .60332 L\n.71524 .60332 L\n.71714 .60332 L\n.71905 .60332 L\n.72095 .60332 L\n.72286 .60332 L\n.72476 .60332 L\n.72667 .60332 L\n.72857 .60332 L\n.73048 .60332 L\n.73238 .60332 L\n.73429 .60332 L\n.73619 .60332 L\n.7381 .60332 L\n.7381 .60332 L\n.73819 .60332 L\n.73819 .60332 L\n.74 .60332 L\n.7419 .60332 L\n.74381 .60332 L\n.74571 .60332 L\nMistroke\n.74762 .60332 L\n.74952 .60332 L\n.75143 .60332 L\n.75333 .60332 L\n.75524 .60332 L\n.75714 .60332 L\n.75905 .60332 L\n.76095 .60332 L\n.76286 .60332 L\n.76476 .60332 L\n.76667 .60332 L\n.76857 .60332 L\n.77048 .60332 L\n.77238 .60332 L\n.77429 .60332 L\n.77619 .60332 L\n.7781 .60332 L\n.78 .60332 L\n.7819 .60332 L\n.78381 .60332 L\n.78571 .60332 L\n.78762 .60332 L\n.78885 .60332 L\n.78885 .01472 L\n.78894 .01472 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .60332 L\n.88286 .60332 L\n.88476 .60332 L\n.88667 .60332 L\n.88857 .60332 L\n.89048 .60332 L\n.89238 .60332 L\n.89429 .60332 L\n.89619 .60332 L\n.8981 .60332 L\n.9 .60332 L\n.9019 .60332 L\n.90381 .60332 L\n.90571 .60332 L\n.90762 .60332 L\n.90952 .60332 L\n.91143 .60332 L\n.91333 .60332 L\n.91524 .60332 L\n.91714 .60332 L\n.91905 .60332 L\n.92095 .60332 L\n.92286 .60332 L\nMistroke\n.92476 .60332 L\n.92667 .60332 L\n.92857 .60332 L\n.93048 .60332 L\n.93238 .60332 L\n.93429 .60332 L\n.93619 .60332 L\n.9381 .60332 L\n.94 .60332 L\n.9419 .60332 L\n.94381 .60332 L\n.94571 .60332 L\n.94762 .60332 L\n.94952 .60332 L\n.95143 .60332 L\n.95333 .60332 L\n.95524 .60332 L\n.95714 .60332 L\n.95905 .60332 L\n.96095 .60332 L\n.96286 .60332 L\n.96476 .60332 L\n.96667 .60332 L\n.96857 .60332 L\n.97048 .60332 L\n.97238 .60332 L\n.97429 .60332 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.00815217 0.0815217 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0146739 0.0721271 0.0961957 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n1 0 0 r\n.00561 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .05543 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(simpleDEVSServer) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(DEVSactive) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n231.000 12.813 moveto\n(L) show\n237.000 12.813 moveto\n(@) show\n243.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n249.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\n% Start of sub-graphic\np\n0.0110024 0.109239 0.0721271 0.190761 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 .5 0 r\n.00561 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal2) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\n% Start of sub-graphic\np\n0.0110024 0.203804 0.0721271 0.285326 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00561 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .24457 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal1) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P0000160?l003T007l0303o000]001o01P0o`005`00O`050000\n00P0fMWI00@000000=WIf@3IfMT00000303IfMT001`0fMWI00<007l00=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT00`00\n0000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00\n0000fMWI0=WIf@080=WIf@030000003o0000fMWI00T0fMWI00<0o`000=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o\n0000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT03@3IfMT001`0\nfMWI00<007l00=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<000000=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<0o`000=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@0:0=WIf@D00000303IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0R0=WIf@030000003IfMT0fMWI00d0fMWI\n000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`00\n0=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`00\n0=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WI\nf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3I\nfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703I\nfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3I\nfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0\nfMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0\nfMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0\nfMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o\n0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o\n0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o\n0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380\nfMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0\no`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0\no`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@08\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D\n0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WI\nf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00\n703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n5@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n2@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003I\nfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003I\nfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003I\nfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT0\n0`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT0\n0`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT0\n0`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI\n0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WI\nf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WI\nf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WI\nf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`00\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`00\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`00\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WI\nf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WIf@3I\nfMT01@3IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI00080=WIf@040000\n003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`00O`0000000=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000X0fMWI00<000000=WIf@3I\nfMT00`3IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI00080=WIf@040000\n003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`00O`00fMWI0=WIf@120=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WI\nf@030?l0003IfMT0fMWI0380fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<007l00=WIf@3I\nfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3I\nfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3I\nfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0\n003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3I\nfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00\nO`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o\n0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o\n0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI\n0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L\n0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E\n0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@09\n0=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WI\nf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WI\nf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WI\nf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0\n<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT0\n0`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT0\n0`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT0\n0`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI\n0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI\n00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI\n01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000\nfMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI\n000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`00\n0=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`00\n0=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WI\nf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3I\nfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703I\nfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3I\nfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI00090=WIf@800000103IfMT2\n000000D0fMWI0`0000030=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT0\n00P0fMWI00@000000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@03001o003IfMT0\nfMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0\nfMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0\nfMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT5\n000000<0fMWI00<007l00000003IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00203IfMT01000\n0000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<007l00=WIf@3IfMT0@P3I\nfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203I\nfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503I\nfMT00`3o0000fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000090=WIf@030000003I\nfMT0000000@0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@002@3IfMT2\n000000/0fMWI0P0000040=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00\nfMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000\nfMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000\nfMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI\n00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI\n00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI\n00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WI\nf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WI\nf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WI\nf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WI\nf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3I\nfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3I\nfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3I\nfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0\n003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3I\nfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00\nO`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o\n0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o\n0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI\n0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L\n0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E\n0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@09\n0=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WI\nf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WI\nf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WI\nf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0\n<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT0\n0`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT0\n0`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT0\n0`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI\n0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI\n00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI\n01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000\nfMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI\n00090=WIf@800000103IfMT2000000<0fMWI0`0000050=WIf@03001o003IfMT0fMWI0480fMWI00<0\no`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0\no`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0\no`000=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WI\nf@0000040=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT000P0fMWI00@0\n00000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@03001o00000000fMWI0480\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<0o`000=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0\nfMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0\nfMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0\nfMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI\n0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000\nfMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0`0000040=WIf@03001o003IfMT0fMWI0480fMWI\n00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI\n00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI\n00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WI\nf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WI\nf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WI\nf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WI\nf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3I\nfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3I\nfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3I\nfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0\n003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0\n003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3I\nfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00\nO`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o\n0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o\n0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0\nfMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0\nfMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI\n0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L\n0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E\n0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@09\n0=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WI\nf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WI\nf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WI\nf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0\n<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT0\n0`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT0\n0`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT0\n0`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI\n0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI\n00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI\n01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000\nfMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI\n000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`00\n0=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`00\n0=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@002@3IfMT2000000@0fMWI0P0000040=WI\nf@8000001@3IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI00080=WIf@04\n0000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@08\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D\n0=WIf@030?l0003IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@00000\n0=WIf@3IfMT00000103IfMT00`00O`0000000=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI0008\n0=WIf@040000003IfMT0fMWI000000T0fMWI0P0000050=WIf@03001o003IfMT0fMWI0480fMWI00<0\no`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0\no`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0\no`000=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WI\nf@0000040=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT000T0fMWI0P00\n000:0=WIf@8000001@3IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L\n0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E\n0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@09\n0=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WI\nf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WI\nf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WI\nf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@03\n0?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@03\n0?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0\n<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT0\n0`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT0\n0`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT0\n0`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI\n0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI\n00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI\n01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000\nfMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI\n000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`00\n0=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`00\n0=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WI\nf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WI\nf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3I\nfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703I\nfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3I\nfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0\nfMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0\nfMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0\nfMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o\n0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o\n0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o\n0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030?l0003IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@080=WIf@030?l0003IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D0=WIf@030?l0003IfMT0fMWI0380\nfMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0\no`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0\no`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`3o0000fMWI0=WIf@08\n0=WIf@030?l0003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`3o0000fMWI0=WIf@0D\n0=WIf@030?l0003IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@030?l0003IfMT0fMWI01@0fMWI00<0o`000=WI\nf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@00\n503IfMT5000000<0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<0o`000=WIf@3IfMT0203IfMT00`3o0000fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<0o`000=WIf@3IfMT0503IfMT00`3o0000fMWI0=WIf@0b0=WIf@005P3I\nfMT00`000000fMWI0=WIf@030=WIf@03001o003IfMT0fMWI0480fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0o`000=WIf@0E0=WIf@030?l0003IfMT0fMWI00P0fMWI00<0o`000=WIf@3IfMT07`3I\nfMT00`000?l0o`000=WIf@090=WIf@030?l00000O`00fMWI01@0fMWI00<0o`000=WIf@3IfMT0<P3I\nfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`00O`0000000=WIf@120=WIfC/0o`002P3IfMT_\n0?l001H0fMWI603o000M0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI\n0?l0fMWI0P3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80\nfMWI000E0=WIf@8000001@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o\n0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI0140fMWI0`00003T0=WIf@00\n903IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI0><0fMWI000S0=WIf@030000003I\nfMT0fMWI0080fMWI0`0000040=WIf@<0000000P0fMWI000000000000fMWI0000003IfMT000000=WI\nf@<000000`3IfMT3000000@0fMWI0`0000000`3IfMT000000000000200000080fMWI1@0000030=WI\nf@8000000`3IfMT3000000<0fMWI0`0000040=WIf@<000000P3IfMT4000000<0fMWI0P0000040=WI\nf@<000000P3IfMT4000000<0fMWI0P0000020=WIf@@000000P3IfMT5000000<0fMWI0P0000230=WI\nf@005P3IfMT80?l000D0fMWI00<000000=WIf@3IfMT01@3IfMT00`000000fMWI0=WIf@020=WIf@0;\n0000003IfMT0fMWI0=WIf@000000fMWI0000003IfMT000000=WIf@0000000P3IfMT01@000000fMWI\n0=WIf@3IfMT0000000@0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0103IfMT00`000000\nfMWI0=WIf@030=WIf@8000000`3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00P0\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0\nfMWI08<0fMWI000S0=WIf@030000003IfMT0fMWI0080fMWI0`0000050=WIf@0;0000003IfMT0fMWI\n0=WIf@000000fMWI0000003IfMT000000=WIf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT00000\n00@0fMWI100000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT00`3IfMT0\n10000000fMWI0=WIf@0000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT01000\n00030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT4000000<0fMWI\n00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI00<0\nfMWI00<000000=WIf@3IfMT0P`3IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT4000000<0fMWI\n0P0000020=WIf@<0000000<0fMWI0000003IfMT0100000040=WIf@030000003IfMT0fMWI00<0fMWI\n0P0000030=WIf@040000003IfMT0fMWI00000080fMWI0P0000040=WIf@040000003IfMT0fMWI0000\n00<0fMWI0P0000040=WIf@800000103IfMT2000000<0fMWI100000000`3IfMT00000000000020=WI\nf@8000000P3IfMT2000000<0fMWI100000080=WIf@040000003IfMT0fMWI00000080fMWI0P000004\n0=WIf@030000003IfMT0fMWI08<0fMWI000T0=WIf@030000003IfMT0fMWI01X0fMWI00<000000=WI\nf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0fMWI0000003IfMT00000\n103IfMT00`000000fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3I\nfMT0903IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0fMWI0000003IfMT00000QP3I\nfMT002D0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@0=0=WIf@8000002@3IfMT4\n00000080fMWI1@0000000`3IfMT00000000000020=WIf@80000000<0fMWI0000000000000P000002\n0=WIf@@000009@3IfMT400000080fMWI1@0000000`3IfMT00000000000250=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009@3IfMT00`000000fMWI0=WIf@0T\n0=WIf@<00000<P3IfMT3000001H0fMWI0`0000230=WIf@00903IfMT00`000000fMWI0=WIf@0X0=WI\nf@030000003IfMT0fMWI0300fMWI00<000000=WIf@3IfMT0V`3IfMT002<0fMWI00<000000=WIf@3I\nfMT00`3IfMT2000000@0fMWI0`0000030=WIf@<000000`3IfMT00`000000fMWI000000020=WIf@<0\n0000103IfMT3000000<0fMWI0`0000000`3IfMT00000000000020=WIf@80000000<0fMWI00000000\n00000`000000103IfMT000000000000000030=WIf@@000000`3IfMT2000000@0fMWI0P0000040=WI\nf@@000000P3IfMT2000000<0fMWI0`0000040=WIf@8000000`3IfMT3000000@0fMWI0`0000030=WI\nf@800000P`3IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000D0fMWI00D000000=WIf@3I\nfMT0fMWI000000050=WIf@070000003IfMT000000=WIf@000000fMWI000000020=WIf@030000003I\nfMT0fMWI0080fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00<000000=WIf@3IfMT0100000030=WIf@030000003IfMT0fMWI0080fMWI00<000000=WIf@3I\nfMT02@3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI0080fMWI\n00D000000=WIf@3IfMT0fMWI000000250=WIf@005P3IfMT8001o00D0fMWI00<000000=WIf@3IfMT0\n0P3IfMT010000000fMWI0=WIf@0000040=WIf@040000003IfMT0fMWI0=WIf@@000000P3IfMT01@00\n0000fMWI0000003IfMT0000000@0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@050000\n003IfMT0fMWI0=WIf@0000001@3IfMT00`000000fMWI0=WIf@080=WIf@040000003IfMT0fMWI0000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@040000003IfMT0\nfMWI000000P0fMWI0`0000050=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT002<0fMWI00@0\n00000=WIf@3IfMT0fMWI0P0000020=WIf@8000000P3IfMT2000000@0fMWI0P0000020=WIf@<00000\n00<0fMWI0000000000000P3IfMT2000000@0fMWI0P0000040=WIf@P00000103IfMT3000000<0fMWI\n00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WIf@8000000P3IfMT2000000040=WI\nf@00000000000=WIf@D0000000<0fMWI0000000000000P0000090=WIf@@000000`3IfMT2000000@0\nfMWI0P0000230=WIf@00903IfMT00`000000fMWI0=WIf@0I0=WIf@040000003IfMT0fMWI000001/0\nfMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@00000D0=WIf@030000003IfMT0fMWI\n0:40fMWI000U0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT03@3IfMT4000000@0\nfMWI00<000000=WIf@3IfMT04`3IfMT2000000D0fMWI0P00000_0=WIf@030000003IfMT0fMWI08L0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI000U0=WIf@030000003IfMT0fMWI02@0fMWI0`00000b0=WIf@<000005P3IfMT3000008<0fMWI\n000T0=WIf@030000003IfMT0fMWI02P0fMWI00<000000=WIf@3IfMT0<03IfMT00`000000fMWI0=WI\nf@2K0=WIf@008`3IfMT00`000000fMWI0=WIf@030=WIf@800000103IfMT3000000<0fMWI0`000003\n0=WIf@030000003IfMT000000080fMWI0`0000040=WIf@<000000`3IfMT3000000030=WIf@000000\n00000080fMWI0P0000000`3IfMT0000000000003000000040=WIf@0000000000000000<0fMWI1@00\n00020=WIf@800000103IfMT2000000@0fMWI100000020=WIf@8000000`3IfMT3000000@0fMWI0P00\n00030=WIf@<00000103IfMT3000000<0fMWI0P0000230=WIf@008`3IfMT00`000000fMWI0=WIf@03\n0=WIf@8000001@3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00L000000=WIf@000000fMWI\n0000003IfMT000000080fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WIf@3IfMT00000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@04000000<0fMWI00<00000\n0=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@070=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n00/0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WIf@3IfMT0000008D0fMWI000F0=WI\nf@P0003o1@3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000000@0fMWI00@00000\n0=WIf@3IfMT0fMWI100000020=WIf@050000003IfMT000000=WIf@000000103IfMT00`000000fMWI\n0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@030000003IfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0\nfMWI00L0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000\n003IfMT0fMWI00<0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0fMWI00D000000=WIf@3I\nfMT0fMWI000000250=WIf@008`3IfMT010000000fMWI0=WIf@3IfMT200000080fMWI0P0000020=WI\nf@800000103IfMT200000080fMWI0`0000000`3IfMT00000000000020=WIf@800000103IfMT20000\n00@0fMWI200000040=WIf@<000000`3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI\n00P0fMWI0P0000020=WIf@80000000@0fMWI000000000000fMWI1@0000000`3IfMT0000000000002\n000000T0fMWI100000030=WIf@800000103IfMT2000008<0fMWI000T0=WIf@030000003IfMT0fMWI\n01T0fMWI00@000000=WIf@3IfMT000006`3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0\nfMWI01<0fMWI00<000000=WIf@3IfMT0X@3IfMT002D0fMWI00<000000=WIf@3IfMT0203IfMT00`00\n0000fMWI0=WIf@0=0=WIf@@00000103IfMT00`000000fMWI0=WIf@0C0=WIf@8000001@3IfMT20000\n02l0fMWI00<000000=WIf@3IfMT0Q`3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00001"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -15.1348`, 0.119891`, 0.0898941`]], Rule[List[List[22.125`, 30.4375`], List[43.9375`, 32.875`]], List[-6.32337`, -42.0681`, 0.240604`, 0.240604`]], Rule[List[List[22.125`, 30.4375`], List[31.0625`, 19.9375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]], Rule[List[List[22.125`, 30.4375`], List[18.1875`, 7.0625`]], List[-6.32337`, -48.2712`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 1:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" viewSignal1.outp.signal, viewSignal2.outp.signal and DEVSactive", Rule[FontFamily, "Courier New"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( { viewSignal1.outp.signal, viewSignal2.outp.signal } );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .60332 L\n.30962 .60332 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .60332 L\n.45248 .60332 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .01472 L\n.5511 .01472 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59533 .60332 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .60332 L\n.73819 .60332 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .01472 L\n.78894 .01472 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 .5 0 r\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .01472 L\n.30962 .01472 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .01472 L\n.45248 .01472 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .60332 L\n.5511 .60332 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .01472 L\n.59533 .01472 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .01472 L\n.73819 .01472 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .60332 L\n.78894 .60332 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .01472 L\n.88105 .01472 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0119048 0.119048 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0214286 0.0721271 0.140476 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 .5 0 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .08095 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal2) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\n% Start of sub-graphic\np\n0.0110024 0.159524 0.0721271 0.278571 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .21905 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal1) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P00003W001o00D00000203IfMT010000000fMWI0=WIf@00000<\n0=WIf@00703IfMT00`00O`00fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<00000001o\n003IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`00O`00fMWI0=WIf@0[0=WIf@03\n0000003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000003IfMT0fMWI00X0fMWI1@00000<0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0280fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0\nfMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0\nfMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03\n001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03\n001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI\n00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WI\nf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WI\nf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\nO`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00\nO`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00\nO`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@800000103IfMT2000000<0\nfMWI100000040=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI\n00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@03001o003IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3IfMT00`000000\nfMWI0=WIf@040=WIf@03001o00000000fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI\n00@000000=WIf@3IfMT000002P3IfMT00`000000fMWI0=WIf@030=WIf@03001o003IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000\nfMWI0=WIf@0000040=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0\nfMWI0P00000:0=WIf@8000001@3IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0\n07l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0\n07l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n5@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n2@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o\n003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0\nfMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0\nfMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P0000040=WIf@8000001@3IfMT3000000<0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l0\n0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l0\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@00000;\n0=WIf@030000003IfMT0fMWI0080fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@D000000`3IfMT00`00O`0000000=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@00000\n0=WIf@3IfMT00000103IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI0008\n0=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@000000103IfMT00`00O`00fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@8000002`3IfMT2000000@0fMWI00<007l0\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l0\n0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l0\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3I\nfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0\n07l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0\n07l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n5@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n2@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o\n003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P0000040=WIf@8000000`3I\nfMT3000000D0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT0\n10000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<007l00=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000\n003IfMT0fMWI000000@0fMWI00<007l00000003IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n203IfMT010000000fMWI0=WIf@0000080=WIf@<000001@3IfMT00`00O`00fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3I\nfMT0103IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@800000\n2P3IfMT3000000@0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3I\nfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3I\nfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0\n07l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<0\n07l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n5@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n2@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o\n003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000050=WIf@03001o003IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT01000\n0000fMWI0=WIf@0000040=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@03001o0000\n0000fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3I\nfMT2000000D0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT0\n10000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<007l00=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@03001o003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI\n00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI\n00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o\n003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0\nfMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0\nfMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0E0=WIf@03\n001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03\n001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<007l00=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI01D0fMWI00<007l00=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00T0fMWI00<007l00=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@03001o003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0E0=WIf@03001o003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@090=WIf@03001o003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<007l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n01D0fMWI00<007l00=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n00T0fMWI00<007l00=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00O`00\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000D0=WIf@D000000`3IfMT00`00O`00fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<0\n07l00=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@80003o5P3IfMT00`00O`00fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT2001o01D0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`00O`0000000=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00O`00fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`00O`00fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WI\nf@3IfMT0o`3IfMT20=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0\nfMWI0P3IfMT001D0fMWI0P0000050=WIf@030000003IfMT0fMWI0?l0fMWI0P3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT002D0fMWI00<000000=WIf@3IfMT0903IfMT300000380fMWI0`00000F0=WIf@<00000P`3IfMT0\n02@0fMWI00<000000=WIf@3IfMT0:03IfMT00`000000fMWI0=WIf@0`0=WIf@030000003IfMT0fMWI\n09/0fMWI000S0=WIf@030000003IfMT0fMWI00<0fMWI0P0000040=WIf@<000000`3IfMT3000000<0\nfMWI00<000000=WIf@0000000P3IfMT3000000@0fMWI0`0000030=WIf@<0000000<0fMWI00000000\n00000P3IfMT2000000030=WIf@000000000000<0000000@0fMWI00000000000000000`3IfMT40000\n00<0fMWI0P0000040=WIf@800000103IfMT400000080fMWI0P0000030=WIf@<00000103IfMT20000\n00<0fMWI0`0000040=WIf@<000000`3IfMT2000008<0fMWI000S0=WIf@030000003IfMT0fMWI00<0\nfMWI0P0000050=WIf@050000003IfMT0fMWI0=WIf@0000001@3IfMT01`000000fMWI0000003IfMT0\n00000=WIf@0000000P3IfMT00`000000fMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@000000\n0P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI00@000000`3IfMT00`000000\nfMWI0=WIf@020=WIf@030000003IfMT0fMWI00T0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n2`3IfMT00`000000fMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT001H0fMWI\n2000O`050=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT00000103IfMT010000000\nfMWI0=WIf@3IfMT400000080fMWI00D000000=WIf@000000fMWI000000040=WIf@030000003IfMT0\nfMWI0080fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI\n00<000000=WIf@3IfMT00`3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00<000000=WIf@3I\nfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00<00000\n0=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000080=WIf@<000001@3IfMT01@000000fMWI0=WI\nf@3IfMT0000008D0fMWI000S0=WIf@040000003IfMT0fMWI0=WIf@8000000P3IfMT200000080fMWI\n0P0000040=WIf@8000000P3IfMT3000000030=WIf@00000000000080fMWI0P0000040=WIf@800000\n103IfMT8000000@0fMWI0`0000030=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0\n203IfMT200000080fMWI0P000000103IfMT000000000003IfMT5000000030=WIf@00000000000080\n00002@3IfMT4000000<0fMWI0P0000040=WIf@800000P`3IfMT002@0fMWI00<000000=WIf@3IfMT0\n6@3IfMT010000000fMWI0=WIf@00000K0=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3I\nfMT00000503IfMT00`000000fMWI0=WIf@2Q0=WIf@009@3IfMT00`000000fMWI0=WIf@080=WIf@03\n0000003IfMT0fMWI00d0fMWI100000040=WIf@030000003IfMT0fMWI01<0fMWI0P0000050=WIf@80\n0000;`3IfMT00`000000fMWI0=WIf@270=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009@3IfMT00`000000fMWI0=WI\nf@0T0=WIf@<00000<P3IfMT3000001H0fMWI0`0000230=WIf@00903IfMT00`000000fMWI0=WIf@0X\n0=WIf@030000003IfMT0fMWI0300fMWI00<000000=WIf@3IfMT0V`3IfMT002<0fMWI00<000000=WI\nf@3IfMT00`3IfMT2000000@0fMWI0`0000030=WIf@<000000`3IfMT00`000000fMWI000000020=WI\nf@<00000103IfMT3000000<0fMWI0`0000000`3IfMT00000000000020=WIf@80000000<0fMWI0000\n000000000`000000103IfMT000000000000000030=WIf@D000000P3IfMT2000000@0fMWI0P000004\n0=WIf@@000000P3IfMT2000000<0fMWI0`0000040=WIf@8000000`3IfMT3000000@0fMWI0`000003\n0=WIf@800000P`3IfMT001H0fMWI20000?l50=WIf@030000003IfMT0fMWI00<0fMWI0P0000050=WI\nf@050000003IfMT0fMWI0=WIf@0000001@3IfMT01`000000fMWI0000003IfMT000000=WIf@000000\n0P3IfMT00`000000fMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@030000003IfMT0fMWI00@000000`3IfMT00`000000fMWI0=WIf@040=WI\nf@030000003IfMT0fMWI00L0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000002`3IfMT00`000000\nfMWI0=WIf@020=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT002<0fMWI00<000000=WIf@3I\nfMT00P3IfMT010000000fMWI0=WIf@0000040=WIf@040000003IfMT0fMWI0=WIf@@000000P3IfMT0\n1@000000fMWI0000003IfMT0000000@0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@05\n0000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@070=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@040000003I\nfMT0fMWI000000P0fMWI0`0000050=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT002<0fMWI\n00@000000=WIf@3IfMT0fMWI0P0000020=WIf@8000000P3IfMT2000000@0fMWI0P0000020=WIf@<0\n000000<0fMWI0000000000000P3IfMT2000000@0fMWI0P0000040=WIf@P00000103IfMT3000000<0\nfMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WIf@8000000P3IfMT200000004\n0=WIf@00000000000=WIf@D0000000<0fMWI0000000000000P0000090=WIf@@000000`3IfMT20000\n00@0fMWI0P0000230=WIf@00903IfMT00`000000fMWI0=WIf@0I0=WIf@040000003IfMT0fMWI0000\n01/0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI\n0:40fMWI000U0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT03@3IfMT4000000@0\nfMWI00<000000=WIf@3IfMT04`3IfMT2000000D0fMWI0P00000_0=WIf@030000003IfMT0fMWI08L0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -10.3955`, 0.119891`, 0.0615577`]], Rule[List[List[22.125`, 30.4375`], List[43.0625`, 26.8125`]], List[-6.32337`, -42.9027`, 0.240604`, 0.240604`]], Rule[List[List[22.125`, 30.4375`], List[24.1875`, 7.9375`]], List[-6.32337`, -47.4441`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 2:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - viewSignal1.outp.signal and viewSignal2.outp.signal", Rule[FontFamily, "Courier New"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( viewSignal1.outp.signal );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .60332 L\n.30962 .60332 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .60332 L\n.45248 .60332 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .01472 L\n.5511 .01472 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59533 .60332 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .60332 L\n.73819 .60332 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .01472 L\n.78894 .01472 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal1) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P00003W0000o`D00000203IfMT010000000fMWI0=WIf@00000<\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@090=WIf@03\n0000003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@0:0=WIf@D0\n0000303IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI0280fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT00`000?l0\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI\n00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI\n0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI00<0003o0000003IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000X0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000\nfMWI0=WIf@0000040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@030000o`3IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L\n0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@03\n0000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI\n00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI\n000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WI\nf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P0000040=WIf@8000001@3IfMT3000000<0fMWI00<0003o\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WI\nf@040000003IfMT0fMWI000000/0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WI\nf@3IfMT00000203IfMT5000000<0fMWI00<0003o0000003IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P\n0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@00000\n0=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3IfMT00`000000fMWI0000\n00040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@002@3IfMT2000000/0fMWI0P0000040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0\nfMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI0380fMWI00090=WIf@800000103IfMT2000000<0fMWI0`0000050=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT0\n10000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0003o0=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003I\nfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l000000=WIf@120=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT3000000D0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@03\n0000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3I\nfMT0103IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT000T0fMWI0P00000:0=WIf@<00000103IfMT00`000?l0fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI\n00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI0380fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000050=WIf@030000o`3IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000\nfMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0003o0=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI\n000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l000000=WIf@120=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3I\nfMT2000000D0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000\n103IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0\n803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT000T0fMWI0P00000:0=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3I\nfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT0\n0`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI\n00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI\n000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WI\nf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WI\nf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WI\nf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3I\nfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`00\n0?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O\n0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@03\n0000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0\n7`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT0\n0`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI\n0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00503IfMT5000000<0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI0380fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<0\n003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0O0=WIf@80003o8@3IfMT00`000?l0fMWI0=WI\nf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0<P3IfMT001H0fMWI00<00000\n0=WIf@3IfMT00`3IfMT00`000?l000000=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WI\nf@80fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00\n5@3IfMT2000000D0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009@3IfMT00`000000fMWI0=WI\nf@0T0=WIf@<00000<P3IfMT3000001H0fMWI0`0000230=WIf@00903IfMT00`000000fMWI0=WIf@0X\n0=WIf@030000003IfMT0fMWI0300fMWI00<000000=WIf@3IfMT0V`3IfMT002<0fMWI00<000000=WI\nf@3IfMT00`3IfMT2000000@0fMWI0`0000030=WIf@<000000`3IfMT00`000000fMWI000000020=WI\nf@<00000103IfMT3000000<0fMWI0`0000000`3IfMT00000000000020=WIf@80000000<0fMWI0000\n000000000`000000103IfMT000000000000000030=WIf@D000000P3IfMT2000000@0fMWI0P000004\n0=WIf@@000000P3IfMT2000000<0fMWI0`0000040=WIf@8000000`3IfMT3000000@0fMWI0`000003\n0=WIf@800000P`3IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000D0fMWI00D000000=WI\nf@3IfMT0fMWI000000050=WIf@070000003IfMT000000=WIf@000000fMWI000000020=WIf@030000\n003IfMT0fMWI0080fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00<000000=WIf@3IfMT0100000030=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WI\nf@3IfMT01`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI0080\nfMWI00D000000=WIf@3IfMT0fMWI000000250=WIf@005P3IfMT80000o`D0fMWI00<000000=WIf@3I\nfMT00P3IfMT010000000fMWI0=WIf@0000040=WIf@040000003IfMT0fMWI0=WIf@@000000P3IfMT0\n1@000000fMWI0000003IfMT0000000@0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@05\n0000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@070=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@040000003I\nfMT0fMWI000000P0fMWI0`0000050=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT002<0fMWI\n00@000000=WIf@3IfMT0fMWI0P0000020=WIf@8000000P3IfMT2000000@0fMWI0P0000020=WIf@<0\n000000<0fMWI0000000000000P3IfMT2000000@0fMWI0P0000040=WIf@P00000103IfMT3000000<0\nfMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WIf@8000000P3IfMT200000004\n0=WIf@00000000000=WIf@D0000000<0fMWI0000000000000P0000090=WIf@@000000`3IfMT20000\n00@0fMWI0P0000230=WIf@00903IfMT00`000000fMWI0=WIf@0I0=WIf@040000003IfMT0fMWI0000\n01/0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI\n0:40fMWI000U0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT03@3IfMT4000000@0\nfMWI00<000000=WIf@3IfMT04`3IfMT2000000D0fMWI0P00000_0=WIf@030000003IfMT0fMWI08L0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 3:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - The output signal outp of the generateJobEvents generator component in testDEVSServer.", Rule[FontFamily, "Courier New"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( simpleDEVSServer.tNextEvent );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147099 0.0516781 [\n[.21429 .00221 -3 -9 ]\n[.21429 .00221 3 0 ]\n[.40476 .00221 -3 -9 ]\n[.40476 .00221 3 0 ]\n[.59524 .00221 -3 -9 ]\n[.59524 .00221 3 0 ]\n[.78571 .00221 -3 -9 ]\n[.78571 .00221 3 0 ]\n[.97619 .00221 -6 -9 ]\n[.97619 .00221 6 0 ]\n[1.025 .01471 0 -6.28125 ]\n[1.025 .01471 10 6.28125 ]\n[.01131 .11807 -6 -4.5 ]\n[.01131 .11807 0 4.5 ]\n[.01131 .22142 -6 -4.5 ]\n[.01131 .22142 0 4.5 ]\n[.01131 .32478 -6 -4.5 ]\n[.01131 .32478 0 4.5 ]\n[.01131 .42813 -6 -4.5 ]\n[.01131 .42813 0 4.5 ]\n[.01131 .53149 -12 -4.5 ]\n[.01131 .53149 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01471 m\n.21429 .02096 L\ns\n[(2)] .21429 .00221 0 1 Mshowa\n.40476 .01471 m\n.40476 .02096 L\ns\n[(4)] .40476 .00221 0 1 Mshowa\n.59524 .01471 m\n.59524 .02096 L\ns\n[(6)] .59524 .00221 0 1 Mshowa\n.78571 .01471 m\n.78571 .02096 L\ns\n[(8)] .78571 .00221 0 1 Mshowa\n.97619 .01471 m\n.97619 .02096 L\ns\n[(10)] .97619 .00221 0 1 Mshowa\n.125 Mabswid\n.07143 .01471 m\n.07143 .01846 L\ns\n.11905 .01471 m\n.11905 .01846 L\ns\n.16667 .01471 m\n.16667 .01846 L\ns\n.2619 .01471 m\n.2619 .01846 L\ns\n.30952 .01471 m\n.30952 .01846 L\ns\n.35714 .01471 m\n.35714 .01846 L\ns\n.45238 .01471 m\n.45238 .01846 L\ns\n.5 .01471 m\n.5 .01846 L\ns\n.54762 .01471 m\n.54762 .01846 L\ns\n.64286 .01471 m\n.64286 .01846 L\ns\n.69048 .01471 m\n.69048 .01846 L\ns\n.7381 .01471 m\n.7381 .01846 L\ns\n.83333 .01471 m\n.83333 .01846 L\ns\n.88095 .01471 m\n.88095 .01846 L\ns\n.92857 .01471 m\n.92857 .01846 L\ns\n.25 Mabswid\n0 .01471 m\n1 .01471 L\ns\ngsave\n1.025 .01471 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .11807 m\n.03006 .11807 L\ns\n[(2)] .01131 .11807 1 0 Mshowa\n.02381 .22142 m\n.03006 .22142 L\ns\n[(4)] .01131 .22142 1 0 Mshowa\n.02381 .32478 m\n.03006 .32478 L\ns\n[(6)] .01131 .32478 1 0 Mshowa\n.02381 .42813 m\n.03006 .42813 L\ns\n[(8)] .01131 .42813 1 0 Mshowa\n.02381 .53149 m\n.03006 .53149 L\ns\n[(10)] .01131 .53149 1 0 Mshowa\n.125 Mabswid\n.02381 .04055 m\n.02756 .04055 L\ns\n.02381 .06639 m\n.02756 .06639 L\ns\n.02381 .09223 m\n.02756 .09223 L\ns\n.02381 .14391 m\n.02756 .14391 L\ns\n.02381 .16974 m\n.02756 .16974 L\ns\n.02381 .19558 m\n.02756 .19558 L\ns\n.02381 .24726 m\n.02756 .24726 L\ns\n.02381 .2731 m\n.02756 .2731 L\ns\n.02381 .29894 m\n.02756 .29894 L\ns\n.02381 .35062 m\n.02756 .35062 L\ns\n.02381 .37646 m\n.02756 .37646 L\ns\n.02381 .4023 m\n.02756 .4023 L\ns\n.02381 .45397 m\n.02756 .45397 L\ns\n.02381 .47981 m\n.02756 .47981 L\ns\n.02381 .50565 m\n.02756 .50565 L\ns\n.02381 .55733 m\n.02756 .55733 L\ns\n.02381 .58317 m\n.02756 .58317 L\ns\n.02381 .60901 m\n.02756 .60901 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02382 .01472 L\n.02382 .01472 L\n.0239 .01472 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .30078 L\n.30962 .30078 L\n.30962 .30078 L\n.31143 .30078 L\n.31333 .30078 L\n.31524 .30078 L\n.31714 .30078 L\n.31905 .30078 L\n.32095 .30078 L\n.32286 .30078 L\n.32476 .30078 L\n.32667 .30078 L\n.32857 .30078 L\n.33048 .30078 L\n.33238 .30078 L\n.33429 .30078 L\n.33619 .30078 L\n.3381 .30078 L\n.34 .30078 L\n.3419 .30078 L\n.34381 .30078 L\n.34571 .30078 L\n.34762 .30078 L\n.34952 .30078 L\n.35143 .30078 L\n.35333 .30078 L\n.35524 .30078 L\n.35714 .30078 L\n.35905 .30078 L\n.36095 .30078 L\n.36286 .30078 L\n.36476 .30078 L\n.36667 .30078 L\n.36857 .30078 L\n.37048 .30078 L\n.37238 .30078 L\n.37429 .30078 L\n.37619 .30078 L\n.3781 .30078 L\n.38 .30078 L\n.3819 .30078 L\n.38381 .30078 L\n.38571 .30078 L\n.38762 .30078 L\n.38952 .30078 L\nMistroke\n.39143 .30078 L\n.39333 .30078 L\n.39524 .30078 L\n.39714 .30078 L\n.39905 .30078 L\n.40095 .30078 L\n.40286 .30078 L\n.40476 .30078 L\n.40667 .30078 L\n.40857 .30078 L\n.41048 .30078 L\n.41238 .30078 L\n.41429 .30078 L\n.41619 .30078 L\n.4181 .30078 L\n.42 .30078 L\n.4219 .30078 L\n.42381 .30078 L\n.42571 .30078 L\n.42762 .30078 L\n.42952 .30078 L\n.43143 .30078 L\n.43333 .30078 L\n.43524 .30078 L\n.43714 .30078 L\n.43905 .30078 L\n.44095 .30078 L\n.44286 .30078 L\n.44476 .30078 L\n.44667 .30078 L\n.44857 .30078 L\n.45048 .30078 L\n.45238 .30078 L\n.45238 .30078 L\n.45248 .30078 L\n.45248 .30078 L\n.45429 .30078 L\n.45619 .30078 L\n.4581 .30078 L\n.46 .30078 L\n.4619 .30078 L\n.46381 .30078 L\n.46571 .30078 L\n.46762 .30078 L\n.46952 .30078 L\n.47143 .30078 L\n.47333 .30078 L\n.47524 .30078 L\n.47714 .30078 L\n.47905 .30078 L\nMistroke\n.48095 .30078 L\n.48286 .30078 L\n.48476 .30078 L\n.48667 .30078 L\n.48857 .30078 L\n.49048 .30078 L\n.49238 .30078 L\n.49429 .30078 L\n.49619 .30078 L\n.4981 .30078 L\n.5 .30078 L\n.5019 .30078 L\n.50381 .30078 L\n.50571 .30078 L\n.50762 .30078 L\n.50952 .30078 L\n.51143 .30078 L\n.51333 .30078 L\n.51524 .30078 L\n.51714 .30078 L\n.51905 .30078 L\n.52095 .30078 L\n.52286 .30078 L\n.52476 .30078 L\n.52667 .30078 L\n.52857 .30078 L\n.53048 .30078 L\n.53238 .30078 L\n.53429 .30078 L\n.53619 .30078 L\n.5381 .30078 L\n.54 .30078 L\n.5419 .30078 L\n.54381 .30078 L\n.54571 .30078 L\n.54762 .30078 L\n.54952 .30078 L\n.55101 .30078 L\n.55101 .30078 L\n.5511 .30078 L\n.5511 .30078 L\n.55143 .30078 L\n.55333 .30078 L\n.55524 .30078 L\n.55714 .30078 L\n.55905 .30078 L\n.56095 .30078 L\n.56286 .30078 L\n.56476 .30078 L\n.56667 .30078 L\nMistroke\n.56857 .30078 L\n.57048 .30078 L\n.57238 .30078 L\n.57429 .30078 L\n.57619 .30078 L\n.5781 .30078 L\n.58 .30078 L\n.5819 .30078 L\n.58381 .30078 L\n.58571 .30078 L\n.58762 .30078 L\n.58952 .30078 L\n.59143 .30078 L\n.59333 .30078 L\n.59524 .30078 L\n.59524 .42983 L\n.59533 .42983 L\n.59533 .42983 L\n.59714 .42983 L\n.59905 .42983 L\n.60095 .42983 L\n.60286 .42983 L\n.60476 .42983 L\n.60667 .42983 L\n.60857 .42983 L\n.61048 .42983 L\n.61238 .42983 L\n.61429 .42983 L\n.61619 .42983 L\n.6181 .42983 L\n.62 .42983 L\n.6219 .42983 L\n.62381 .42983 L\n.62571 .42983 L\n.62762 .42983 L\n.62952 .42983 L\n.63143 .42983 L\n.63333 .42983 L\n.63524 .42983 L\n.63714 .42983 L\n.63905 .42983 L\n.64095 .42983 L\n.64286 .42983 L\n.64476 .42983 L\n.64667 .42983 L\n.64857 .42983 L\n.65048 .42983 L\n.65238 .42983 L\n.65429 .42983 L\n.65619 .42983 L\nMistroke\n.6581 .42983 L\n.66 .42983 L\n.6619 .42983 L\n.66381 .42983 L\n.66571 .42983 L\n.66762 .42983 L\n.66952 .42983 L\n.67143 .42983 L\n.67333 .42983 L\n.67524 .42983 L\n.67714 .42983 L\n.67905 .42983 L\n.68095 .42983 L\n.68286 .42983 L\n.68476 .42983 L\n.68667 .42983 L\n.68857 .42983 L\n.69048 .42983 L\n.69238 .42983 L\n.69429 .42983 L\n.69619 .42983 L\n.6981 .42983 L\n.7 .42983 L\n.7019 .42983 L\n.70381 .42983 L\n.70571 .42983 L\n.70762 .42983 L\n.70952 .42983 L\n.71143 .42983 L\n.71333 .42983 L\n.71524 .42983 L\n.71714 .42983 L\n.71905 .42983 L\n.72095 .42983 L\n.72286 .42983 L\n.72476 .42983 L\n.72667 .42983 L\n.72857 .42983 L\n.73048 .42983 L\n.73238 .42983 L\n.73429 .42983 L\n.73619 .42983 L\n.7381 .42983 L\n.7381 .42983 L\n.73819 .42983 L\n.73819 .42983 L\n.74 .42983 L\n.7419 .42983 L\n.74381 .42983 L\n.74571 .42983 L\nMistroke\n.74762 .42983 L\n.74952 .42983 L\n.75143 .42983 L\n.75333 .42983 L\n.75524 .42983 L\n.75714 .42983 L\n.75905 .42983 L\n.76095 .42983 L\n.76286 .42983 L\n.76476 .42983 L\n.76667 .42983 L\n.76857 .42983 L\n.77048 .42983 L\n.77238 .42983 L\n.77429 .42983 L\n.77619 .42983 L\n.7781 .42983 L\n.78 .42983 L\n.7819 .42983 L\n.78381 .42983 L\n.78571 .42983 L\n.78762 .42983 L\n.78885 .42983 L\n.78885 .42983 L\n.78894 .42983 L\n.78894 .42983 L\n.78952 .42983 L\n.79143 .42983 L\n.79333 .42983 L\n.79524 .42983 L\n.79714 .42983 L\n.79905 .42983 L\n.80095 .42983 L\n.80286 .42983 L\n.80476 .42983 L\n.80667 .42983 L\n.80857 .42983 L\n.81048 .42983 L\n.81238 .42983 L\n.81429 .42983 L\n.81619 .42983 L\n.8181 .42983 L\n.82 .42983 L\n.8219 .42983 L\n.82381 .42983 L\n.82571 .42983 L\n.82762 .42983 L\n.82952 .42983 L\n.83143 .42983 L\n.83333 .42983 L\nMistroke\n.83524 .42983 L\n.83714 .42983 L\n.83905 .42983 L\n.84095 .42983 L\n.84286 .42983 L\n.84476 .42983 L\n.84667 .42983 L\n.84857 .42983 L\n.85048 .42983 L\n.85238 .42983 L\n.85429 .42983 L\n.85619 .42983 L\n.8581 .42983 L\n.86 .42983 L\n.8619 .42983 L\n.86381 .42983 L\n.86571 .42983 L\n.86762 .42983 L\n.86952 .42983 L\n.87143 .42983 L\n.87333 .42983 L\n.87524 .42983 L\n.87714 .42983 L\n.87905 .42983 L\n.88095 .42983 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .60332 L\n.88286 .60332 L\n.88476 .60332 L\n.88667 .60332 L\n.88857 .60332 L\n.89048 .60332 L\n.89238 .60332 L\n.89429 .60332 L\n.89619 .60332 L\n.8981 .60332 L\n.9 .60332 L\n.9019 .60332 L\n.90381 .60332 L\n.90571 .60332 L\n.90762 .60332 L\n.90952 .60332 L\n.91143 .60332 L\n.91333 .60332 L\n.91524 .60332 L\n.91714 .60332 L\n.91905 .60332 L\n.92095 .60332 L\n.92286 .60332 L\nMistroke\n.92476 .60332 L\n.92667 .60332 L\n.92857 .60332 L\n.93048 .60332 L\n.93238 .60332 L\n.93429 .60332 L\n.93619 .60332 L\n.9381 .60332 L\n.94 .60332 L\n.9419 .60332 L\n.94381 .60332 L\n.94571 .60332 L\n.94762 .60332 L\n.94952 .60332 L\n.95143 .60332 L\n.95333 .60332 L\n.95524 .60332 L\n.95714 .60332 L\n.95905 .60332 L\n.96095 .60332 L\n.96286 .60332 L\n.96476 .60332 L\n.96667 .60332 L\n.96857 .60332 L\n.97048 .60332 L\n.97238 .60332 L\n.97429 .60332 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(simpleDEVSServer) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(tNextEvent) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n231.000 12.813 moveto\n(L) show\n237.000 12.813 moveto\n(@) show\n243.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n249.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00@`3IfMT4\n000002d0fMWI0`00000Z0=WIf@<00000;@3IfMT2000002T0fMWI1@0000020=WIf@8000006`3IfMT0\n04<0fMWI00<000000=WIf@3IfMT0;`3IfMT00`000000fMWI0=WIf@0Y0=WIf@040000003IfMT0fMWI\n000002/0fMWI00@000000=WIf@3IfMT00000:P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI\n00<000000=WIf@3IfMT0603IfMT004@0fMWI00<000000=WIf@3IfMT0:`3IfMT5000002X0fMWI00@0\n00000=WIf@3IfMT00000:`3IfMT010000000fMWI0=WIf@00000Z0=WIf@050000003IfMT0fMWI0=WI\nf@0000000P3IfMT00`000000fMWI0=WIf@0H0=WIf@00A@3IfMT00`000000fMWI0=WIf@0Z0=WIf@04\n0000003IfMT0fMWI000002/0fMWI0`00000]0=WIf@800000:`3IfMT01@000000fMWI0=WIf@3IfMT0\n00000080fMWI00<000000=WIf@3IfMT0603IfMT004<0fMWI00@000000=WIf@3IfMT00000;03IfMT0\n0`000000fMWI0000000/0=WIf@030000003IfMT0fMWI02/0fMWI00@000000=WIf@3IfMT00000:P3I\nfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT0603IfMT004@0fMWI0P00\n000^0=WIf@800000;03IfMT3000002`0fMWI0P00000Z0=WIf@800000103IfMT2000001/0fMWI003o\n0=WIfB40fMWI000F0=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT001H0fMWI00<000000=WIf@3I\nfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003I\nfMT0fMWI0?P0fMWI0P00000=0=WIf@00403IfMT6000004L0003oZ`0000080=WIf@040000003IfMT0\nfMWI000000`0fMWI000F0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT02@3IfMT0\n0`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3IfMT0203IfMT0\n0`000000fMWI0=WIf@0<0=WIf@030000003IfMT0fMWI00d0fMWI000F0=WIf@030000003IfMT0fMWI\n02`0fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0E0=WIf@030000003IfMT0fMWI\n02`0fMWI00<000000=WIf@3IfMT0;03IfMT00`000000fMWI0=WIf@0/0=WIf@030000003IfMT0fMWI\n00/0fMWI1@00000<0=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0;40\nfMWI00<000000=WIf@3IfMT03@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI\n0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F\n0=WIf@800000A03IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WI\nf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3I\nfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3I\nfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0\nfMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`00\n0?l0fMWI0=WIf@310=WIf@005P3IfMT2000004@0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI\n00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WI\nf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o\n0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WI\nf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@800000\nA03IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3I\nfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT0\n01H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000\nfMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000>0=WIf@@00000103IfMT00`000000fMWI\n0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000>0=WIf@030000003IfMT0fMWI00D0fMWI00<0\n00000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@003`3IfMT00`000000fMWI0=WIf@04\n0=WIf@800000A03IfMT00`000?l0fMWI0=WIf@310=WIf@00403IfMT00`000000fMWI0=WIf@030=WI\nf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT000h0fMWI00@000000=WIf@3I\nfMT00000103IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000?0=WIf@80\n00001@3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000\n003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3I\nfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT2000004@0fMWI00<0003o0=WIf@3IfMT0`@3IfMT0\n01H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000\nfMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI\n00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WI\nf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WI\nf@800000A03IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@03\n0000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0\n`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT0\n0`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI\n04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0\nfMWI0=WIf@310=WIf@005P3IfMT2000004@0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<0\n00000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@13\n0=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WI\nf@3IfMT0`@3IfMT00100fMWI0`0000030=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3I\nfMT0`@3IfMT00140fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WIf@130=WIf@030000\no`3IfMT0fMWI0<40fMWI000>0=WIf@D000000`3IfMT2000004@0fMWI00<0003o0=WIf@3IfMT0`@3I\nfMT000h0fMWI00@000000=WIf@3IfMT00000103IfMT00`000000fMWI0=WIf@130=WIf@030000o`3I\nfMT0fMWI0<40fMWI000?0=WIf@030000003IfMT0000000@0fMWI00<000000=WIf@3IfMT0@`3IfMT0\n0`000?l0fMWI0=WIf@310=WIf@00403IfMT2000000@0fMWI00<000000=WIf@3IfMT0@`3IfMT00`00\n0?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40\nfMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<0\n00000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT2000004@0fMWI00<0003o\n0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WIf@310=WI\nf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000\n003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3I\nfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0\nfMWI0<40fMWI000F0=WIf@800000A03IfMT00`000?l0fMWI0=WIf@310=WIf@005P3IfMT00`000000\nfMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WIf@030000003IfMT0fMWI04<0fMWI\n00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI00<000000=WIf@3IfMT0@`3IfMT00`000?l0fMWI0=WI\nf@310=WIf@005P3IfMT00`000000fMWI0=WIf@130=WIf@030000o`3IfMT0fMWI0<40fMWI000F0=WI\nf@030000003IfMT0fMWI04<0fMWI00<0003o0=WIf@3IfMT0`@3IfMT001H0fMWI0P0000140=WIf@80\n003o8P3IfMT00`000?l0fMWI0=WIf@0F0=WIf@030000o`3IfMT0fMWI08@0fMWI000F0=WIf@030000\n003IfMT0fMWI04<0fMWIB0000?ml0=WIf@005P3IfMT00`000000fMWI0=WIf@2:0=WIf@030000o`3I\nfMT0fMWI07X0fMWI000F0=WIf@030000003IfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT0\n01H0fMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@003P3IfMT3000000D0\nfMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@003P3IfMT010000000fMWI\n0=WIf@0000040=WIf@030000003IfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT000h0fMWI\n00@000000=WIf@3IfMT00000103IfMT2000008/0fMWI00<0003o0=WIf@3IfMT0NP3IfMT000h0fMWI\n0`0000050=WIf@030000003IfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT000l0fMWI00<0\n00000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI000?\n0=WIf@<00000103IfMT00`000000fMWI0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI000F0=WI\nf@030000003IfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT001H0fMWI00<000000=WIf@3I\nfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@005P3IfMT2000008/0fMWI00<0003o0=WIf@3IfMT0\nNP3IfMT001H0fMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI000F0=WIf@030000003IfMT0fMWI\n08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT001H0fMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0\nfMWI0=WIf@1j0=WIf@005P3IfMT00`000000fMWI0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI\n000F0=WIf@030000003IfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT001H0fMWI0P00002;\n0=WIf@030000o`3IfMT0fMWI07X0fMWI000F0=WIf@030000003IfMT0fMWI08X0fMWI00<0003o0=WI\nf@3IfMT0NP3IfMT001H0fMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@00\n5P3IfMT00`000000fMWI0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI000F0=WIf@030000003I\nfMT0fMWI08X0fMWI00<0003o0=WIf@3IfMT0NP3IfMT001H0fMWI00<000000=WIf@3IfMT0RP3IfMT0\n0`000?l0fMWI0=WIf@1j0=WIf@005P3IfMT2000008/0fMWI00<0003o0=WIf@3IfMT0NP3IfMT001H0\nfMWI00<000000=WIf@3IfMT0RP3IfMT00`000?l0fMWI0=WIf@1j0=WIf@005P3IfMT00`000000fMWI\n0=WIf@2:0=WIf@030000o`3IfMT0fMWI07X0fMWI000F0=WIf@030000003IfMT0fMWI08X0fMWI00<0\n003o0=WIf@3IfMT0NP3IfMT000l0fMWI0P0000050=WIf@030000003IfMT0fMWI08X0fMWI00<0003o\n0=WIf@3IfMT0NP3IfMT000h0fMWI00@000000=WIf@3IfMT00000103IfMT00`000000fMWI0=WIf@2:\n0=WIf@030000o`3IfMT0fMWI07X0fMWI000>0=WIf@040000003IfMT0fMWI000000@0fMWI0P00002;\n0=WIf@80003o8P3IfMT00`000?l0fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI04X0fMWI000?0=WI\nf@8000001@3IfMT00`000000fMWI0=WIf@2:0=WIfDP0003o=@3IfMT000h0fMWI00@000000=WIf@3I\nfMT00000103IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000?0=WIf@80\n00001@3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000\n003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3I\nfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0\nfMWI03<0fMWI000F0=WIf@800000dP3IfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT00`000000\nfMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI0=40fMWI\n00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WI\nf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WI\nf@030000003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI0P00003B0=WIf@03\n0000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0\n<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI\n0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0\nfMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI\n000F0=WIf@800000dP3IfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A\n0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI0=40fMWI00<0003o0=WI\nf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WIf@0c0=WIf@00\n203IfMT500000080fMWI0P0000050=WIf@030000003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0\n<`3IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@030000003IfMT0fMWI0080fMWI\n00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WIf@0c0=WIf@002P3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00<000000=WIf@3IfMT00P3IfMT200000=80fMWI00<0003o0=WIf@3IfMT0\n<`3IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@030000003IfMT0fMWI0080fMWI\n00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WIf@0c0=WIf@002P3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000\no`3IfMT0fMWI03<0fMWI00090=WIf@800000103IfMT2000000D0fMWI00<000000=WIf@3IfMT0d@3I\nfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0\nfMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0\nfMWI0P00003B0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI0=40fMWI\n00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0fMWI0=WI\nf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WI\nf@030000003IfMT0fMWI0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3I\nfMT0d@3IfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000\no`3IfMT0fMWI03<0fMWI000F0=WIf@800000dP3IfMT00`000?l0fMWI0=WIf@0c0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI000F0=WIf@030000003IfMT0fMWI\n0=40fMWI00<0003o0=WIf@3IfMT0<`3IfMT001H0fMWI00<000000=WIf@3IfMT0d@3IfMT00`000?l0\nfMWI0=WIf@0c0=WIf@005P3IfMT00`000000fMWI0=WIf@3A0=WIf@030000o`3IfMT0fMWI03<0fMWI\n000F0=WIf@030000003IfMT0fMWI0=40fMWI60000?lN0=WIf@005P3IfMT200000?l0fMWI2@3IfMT0\n01H0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI\n0140fMWI0`00003T0=WIf@00903IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI0><0\nfMWI000S0=WIf@030000003IfMT0fMWI0080fMWI0`0000040=WIf@<0000000P0fMWI000000000000\nfMWI0000003IfMT000000=WIf@<000000`3IfMT3000000@0fMWI0`0000000`3IfMT0000000000002\n00000080fMWI1@0000030=WIf@8000000`3IfMT3000000<0fMWI0`0000040=WIf@<000000P3IfMT4\n000000<0fMWI0P0000040=WIf@<000000P3IfMT4000000<0fMWI0P0000040=WIf@8000000P3IfMT3\n000000030=WIf@000000fMWI0080fMWI0P0000230=WIf@008`3IfMT00`000000fMWI0=WIf@050=WI\nf@030000003IfMT0fMWI0080fMWI00/000000=WIf@3IfMT0fMWI0000003IfMT000000=WIf@000000\nfMWI000000020=WIf@050000003IfMT0fMWI0=WIf@000000103IfMT00`000000fMWI0=WIf@030=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`00\n0000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00<0fMWI0P0000030=WIf@030000003IfMT0fMWI\n00@0fMWI00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\n000000000080fMWI00<000000=WIf@3IfMT0P`3IfMT001H0fMWI20000?l50=WIf@030000003IfMT0\nfMWI0080fMWI0`0000050=WIf@0;0000003IfMT0fMWI0=WIf@000000fMWI0000003IfMT000000=WI\nf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000@0fMWI100000020=WIf@040000003IfMT0\nfMWI00000080fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000050=WIf@030000\n003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0100000030=WIf@030000003IfMT0fMWI0080fMWI\n00@000000=WIf@3IfMT000000P3IfMT4000000<0fMWI00<000000=WIf@3IfMT0203IfMT00`000000\nfMWI0=WIf@030=WIf@040000003IfMT0000000000080fMWI0`0000230=WIf@008`3IfMT00`000000\nfMWI0=WIf@020=WIf@@000000`3IfMT200000080fMWI0`0000000`3IfMT000000=WIf@04000000@0\nfMWI00<000000=WIf@3IfMT00`3IfMT2000000<0fMWI00@000000=WIf@3IfMT000000P3IfMT20000\n00@0fMWI00@000000=WIf@3IfMT000000`3IfMT2000000@0fMWI0P0000040=WIf@8000000`3IfMT4\n000000030=WIf@00000000000080fMWI0P0000020=WIf@8000000`3IfMT4000000L0fMWI1@000002\n0=WIf@80000000<0fMWI0000003IfMT00P3IfMT2000008<0fMWI000T0=WIf@030000003IfMT0fMWI\n01X0fMWI00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0\nfMWI0000003IfMT00000103IfMT00`000000fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00<000000=WIf@3IfMT0903IfMT00`000000fMWI0=WIf@030=WIf@80000000<0fMWI0000003I\nfMT0Q`3IfMT002D0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@0=0=WIf@800000\n2@3IfMT400000080fMWI1@0000000`3IfMT00000000000020=WIf@80000000<0fMWI000000000000\n0P0000020=WIf@@00000:`3IfMT2000000040=WIf@0000000000000008L0fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-0.937208`, -1.62914`, 0.0424163`, 0.0781695`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 4:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - ", Rule[FontFamily, "Courier New"]], StyleBox["The output signal simpleDEVSServer.tNextEvent.", Rule[FontFamily, "Courier"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( viewSignal2.outp.signal );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02382 .60332 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .01472 L\n.30962 .01472 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .01472 L\n.45248 .01472 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .60332 L\n.5511 .60332 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .01472 L\n.59533 .01472 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .01472 L\n.73819 .01472 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .60332 L\n.78894 .60332 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .01472 L\n.88105 .01472 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(viewSignal2) show\n135.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n141.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(outp) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(signal) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n207.000 12.813 moveto\n(L) show\n213.000 12.813 moveto\n(@) show\n219.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n225.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P00003W0000o`D00000203IfMT010000000fMWI0=WIf@00000<\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000000\no`3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT02`3IfMT00`000000fMWI0=WIf@0=0=WIf@00703IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000003IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000003IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0:0=WIf@D0\n0000303IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000\no`3IfMT0fMWI03T0fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03I\nfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0\nfMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0\nfMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI\n0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0\n003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f\n0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WI\nf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@03\n0000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\nO03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3I\nfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI\n04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0\nfMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI\n00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WI\nf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o\n0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WI\nf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000\no`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703I\nfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0\nfMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`00\n0?l0fMWI0=WIf@190=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT00`000?l0\nfMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT000P0fMWI\n00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI07`0\nfMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00203IfMT010000000fMWI\n0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI00<0003o0000003IfMT0O03IfMT00`000?l0\nfMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI00080=WIf@040000003IfMT0fMWI000000X0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0\nfMWI00<0003o0=WIf@3IfMT0B@3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000\nfMWI0=WIf@0000040=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`00\n0?l0fMWI0=WIf@190=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@030000o`3IfMT0fMWI07`0\nfMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI\n0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0\n003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L\n0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@19\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WI\nf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@03\n0000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0\n=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3I\nfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT0\n0`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI\n07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0\nfMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI\n000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WI\nf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o\n0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WI\nf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3I\nfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000\no`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03I\nfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0\nfMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0\nfMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI\n0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0\n003o0=WIf@3IfMT0B@3IfMT000T0fMWI0P0000040=WIf@8000001@3IfMT3000000<0fMWI00<0003o\n0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI00080=WI\nf@040000003IfMT0fMWI000000/0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@1l\n0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT000P0fMWI00@000000=WI\nf@3IfMT00000203IfMT5000000<0fMWI00<0003o0000003IfMT0O03IfMT00`000?l0fMWI0=WIf@0f\n0=WIf@030000o`3IfMT0fMWI04T0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@00000\n0=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0\n003o0=WIf@3IfMT0B@3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3IfMT00`000000fMWI0000\n00040=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WI\nf@190=WIf@002@3IfMT2000000/0fMWI0P0000040=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o\n0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WI\nf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000\no`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703I\nfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0\nfMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`00\n0?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0\nfMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI\n0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0\n003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l\n0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@03\n0000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00\n703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0\nB@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3I\nfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT0\n0`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI\n03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0\nfMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI\n00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WI\nf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WI\nf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WI\nf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3I\nfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000\no`3IfMT0fMWI04T0fMWI00090=WIf@800000103IfMT2000000<0fMWI0`0000050=WIf@030000o`3I\nfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00203IfMT0\n10000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0003o0=WIf@3IfMT0\nO03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI00080=WIf@040000003I\nfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l000000=WIf@1l0=WIf@03\n0000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT000P0fMWI00@000000=WIf@3IfMT0\n0000203IfMT3000000D0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@03\n0000o`3IfMT0fMWI04T0fMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3I\nfMT0103IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3I\nfMT0B@3IfMT000T0fMWI0P00000:0=WIf@<00000103IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000\no`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03I\nfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0\nfMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0\nfMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI\n0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0\n003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f\n0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WI\nf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@03\n0000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\nO03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3I\nfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI\n04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0\nfMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI\n00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WI\nf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o\n0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WI\nf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000\no`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703I\nfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0\nfMWI04T0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000050=WIf@030000o`3IfMT0fMWI\n07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00203IfMT010000000\nfMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0003o0=WIf@3IfMT0O03IfMT0\n0`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI00080=WIf@040000003IfMT0fMWI\n000000P0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l000000=WIf@1l0=WIf@030000o`3I\nfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3I\nfMT2000000D0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3I\nfMT0fMWI04T0fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000\n103IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0\nB@3IfMT000T0fMWI0P00000:0=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3I\nfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT0\n0`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI\n07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0\nfMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI\n000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WI\nf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o\n0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WI\nf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3I\nfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000\no`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03I\nfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0\nfMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0\nfMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI\n0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0\n003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f\n0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WI\nf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1l0=WIf@03\n0000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\nO03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3I\nfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o0=WIf@3IfMT0B@3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI\n04T0fMWI000L0=WIf@030000o`3IfMT0fMWI07`0fMWI00<0003o0=WIf@3IfMT0=P3IfMT00`000?l0\nfMWI0=WIf@190=WIf@00503IfMT5000000<0fMWI00<0003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI\n0=WIf@0f0=WIf@030000o`3IfMT0fMWI04T0fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<0\n003o0=WIf@3IfMT0O03IfMT00`000?l0fMWI0=WIf@0f0=WIf@80003oBP3IfMT001H0fMWI00<00000\n0=WIf@3IfMT00`3IfMT00`000?l000000=WIf@1l0=WIf@030000o`3IfMT0fMWI03H0fMWI00<0003o\n0=WIf@3IfMT0B@3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WI\nf@80fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00\n5@3IfMT2000000D0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009@3IfMT00`000000fMWI0=WI\nf@0T0=WIf@<00000<P3IfMT3000001H0fMWI0`0000230=WIf@00903IfMT00`000000fMWI0=WIf@0X\n0=WIf@030000003IfMT0fMWI0300fMWI00<000000=WIf@3IfMT0V`3IfMT002<0fMWI00<000000=WI\nf@3IfMT00`3IfMT2000000@0fMWI0`0000030=WIf@<000000`3IfMT00`000000fMWI000000020=WI\nf@<00000103IfMT3000000<0fMWI0`0000000`3IfMT00000000000020=WIf@80000000<0fMWI0000\n000000000`000000103IfMT000000000000000030=WIf@@000000`3IfMT2000000@0fMWI0P000004\n0=WIf@@000000P3IfMT2000000<0fMWI0`0000040=WIf@8000000`3IfMT3000000@0fMWI0`000003\n0=WIf@800000P`3IfMT002<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000D0fMWI00D000000=WI\nf@3IfMT0fMWI000000050=WIf@070000003IfMT000000=WIf@000000fMWI000000020=WIf@030000\n003IfMT0fMWI0080fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00<000000=WIf@3IfMT0100000030=WIf@030000003IfMT0fMWI0080fMWI00<000000=WI\nf@3IfMT02@3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000;0=WIf@030000003IfMT0fMWI0080\nfMWI00D000000=WIf@3IfMT0fMWI000000250=WIf@005P3IfMT80000o`D0fMWI00<000000=WIf@3I\nfMT00P3IfMT010000000fMWI0=WIf@0000040=WIf@040000003IfMT0fMWI0=WIf@@000000P3IfMT0\n1@000000fMWI0000003IfMT0000000@0fMWI00<000000=WIf@3IfMT00P3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@05\n0000003IfMT0fMWI0=WIf@0000001@3IfMT00`000000fMWI0=WIf@080=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@030=WIf@040000003I\nfMT0fMWI000000P0fMWI0`0000050=WIf@050000003IfMT0fMWI0=WIf@000000Q@3IfMT002<0fMWI\n00@000000=WIf@3IfMT0fMWI0P0000020=WIf@8000000P3IfMT2000000@0fMWI0P0000020=WIf@<0\n000000<0fMWI0000000000000P3IfMT2000000@0fMWI0P0000040=WIf@P00000103IfMT3000000<0\nfMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WIf@8000000P3IfMT200000004\n0=WIf@00000000000=WIf@D0000000<0fMWI0000000000000P0000090=WIf@@000000`3IfMT20000\n00@0fMWI0P0000230=WIf@00903IfMT00`000000fMWI0=WIf@0I0=WIf@040000003IfMT0fMWI0000\n01/0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@00000D0=WIf@030000003IfMT0\nfMWI0:40fMWI000U0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT03@3IfMT40000\n00@0fMWI00<000000=WIf@3IfMT04`3IfMT2000000D0fMWI0P00000_0=WIf@030000003IfMT0fMWI\n08L0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 5:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - A signal is emitted through simpleDEVSServer.outp when completing \\ service of a job, here Job 1 and Job 3.", Rule[FontFamily, "Courier New"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( simpleDEVSServer.DEVSactive );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02382 .01472 L\n.02382 .01472 L\n.0239 .01472 L\n.0239 .01472 L\n.02571 .01472 L\n.02762 .01472 L\n.02952 .01472 L\n.03143 .01472 L\n.03333 .01472 L\n.03524 .01472 L\n.03714 .01472 L\n.03905 .01472 L\n.04095 .01472 L\n.04286 .01472 L\n.04476 .01472 L\n.04667 .01472 L\n.04857 .01472 L\n.05048 .01472 L\n.05238 .01472 L\n.05429 .01472 L\n.05619 .01472 L\n.0581 .01472 L\n.06 .01472 L\n.0619 .01472 L\n.06381 .01472 L\n.06571 .01472 L\n.06762 .01472 L\n.06952 .01472 L\n.07143 .01472 L\n.07333 .01472 L\n.07524 .01472 L\n.07714 .01472 L\n.07905 .01472 L\n.08095 .01472 L\n.08286 .01472 L\n.08476 .01472 L\n.08667 .01472 L\n.08857 .01472 L\n.09048 .01472 L\n.09238 .01472 L\n.09429 .01472 L\n.09619 .01472 L\n.0981 .01472 L\n.1 .01472 L\n.1019 .01472 L\n.10381 .01472 L\n.10571 .01472 L\n.10762 .01472 L\n.10952 .01472 L\nMistroke\n.11143 .01472 L\n.11333 .01472 L\n.11524 .01472 L\n.11714 .01472 L\n.11905 .01472 L\n.12095 .01472 L\n.12286 .01472 L\n.12476 .01472 L\n.12667 .01472 L\n.12857 .01472 L\n.13048 .01472 L\n.13238 .01472 L\n.13429 .01472 L\n.13619 .01472 L\n.1381 .01472 L\n.14 .01472 L\n.1419 .01472 L\n.14381 .01472 L\n.14571 .01472 L\n.14762 .01472 L\n.14952 .01472 L\n.15143 .01472 L\n.15333 .01472 L\n.15524 .01472 L\n.15714 .01472 L\n.15905 .01472 L\n.16095 .01472 L\n.16286 .01472 L\n.16476 .01472 L\n.16667 .01472 L\n.16857 .01472 L\n.17048 .01472 L\n.17238 .01472 L\n.17429 .01472 L\n.17619 .01472 L\n.1781 .01472 L\n.18 .01472 L\n.1819 .01472 L\n.18381 .01472 L\n.18571 .01472 L\n.18762 .01472 L\n.18952 .01472 L\n.19143 .01472 L\n.19333 .01472 L\n.19524 .01472 L\n.19714 .01472 L\n.19905 .01472 L\n.20095 .01472 L\n.20286 .01472 L\n.20476 .01472 L\nMistroke\n.20667 .01472 L\n.20857 .01472 L\n.21048 .01472 L\n.21238 .01472 L\n.21429 .01472 L\n.21619 .01472 L\n.2181 .01472 L\n.22 .01472 L\n.2219 .01472 L\n.22381 .01472 L\n.22571 .01472 L\n.22762 .01472 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\nMistroke\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\n.30762 .01472 L\n.30952 .01472 L\n.30952 .60332 L\n.30962 .60332 L\n.30962 .60332 L\n.31143 .60332 L\n.31333 .60332 L\n.31524 .60332 L\n.31714 .60332 L\n.31905 .60332 L\n.32095 .60332 L\n.32286 .60332 L\n.32476 .60332 L\n.32667 .60332 L\n.32857 .60332 L\n.33048 .60332 L\n.33238 .60332 L\n.33429 .60332 L\n.33619 .60332 L\n.3381 .60332 L\n.34 .60332 L\n.3419 .60332 L\n.34381 .60332 L\n.34571 .60332 L\n.34762 .60332 L\n.34952 .60332 L\n.35143 .60332 L\n.35333 .60332 L\n.35524 .60332 L\n.35714 .60332 L\n.35905 .60332 L\n.36095 .60332 L\n.36286 .60332 L\n.36476 .60332 L\n.36667 .60332 L\n.36857 .60332 L\n.37048 .60332 L\n.37238 .60332 L\n.37429 .60332 L\n.37619 .60332 L\n.3781 .60332 L\n.38 .60332 L\n.3819 .60332 L\n.38381 .60332 L\n.38571 .60332 L\n.38762 .60332 L\n.38952 .60332 L\nMistroke\n.39143 .60332 L\n.39333 .60332 L\n.39524 .60332 L\n.39714 .60332 L\n.39905 .60332 L\n.40095 .60332 L\n.40286 .60332 L\n.40476 .60332 L\n.40667 .60332 L\n.40857 .60332 L\n.41048 .60332 L\n.41238 .60332 L\n.41429 .60332 L\n.41619 .60332 L\n.4181 .60332 L\n.42 .60332 L\n.4219 .60332 L\n.42381 .60332 L\n.42571 .60332 L\n.42762 .60332 L\n.42952 .60332 L\n.43143 .60332 L\n.43333 .60332 L\n.43524 .60332 L\n.43714 .60332 L\n.43905 .60332 L\n.44095 .60332 L\n.44286 .60332 L\n.44476 .60332 L\n.44667 .60332 L\n.44857 .60332 L\n.45048 .60332 L\n.45238 .60332 L\n.45238 .60332 L\n.45248 .60332 L\n.45248 .60332 L\n.45429 .60332 L\n.45619 .60332 L\n.4581 .60332 L\n.46 .60332 L\n.4619 .60332 L\n.46381 .60332 L\n.46571 .60332 L\n.46762 .60332 L\n.46952 .60332 L\n.47143 .60332 L\n.47333 .60332 L\n.47524 .60332 L\n.47714 .60332 L\n.47905 .60332 L\nMistroke\n.48095 .60332 L\n.48286 .60332 L\n.48476 .60332 L\n.48667 .60332 L\n.48857 .60332 L\n.49048 .60332 L\n.49238 .60332 L\n.49429 .60332 L\n.49619 .60332 L\n.4981 .60332 L\n.5 .60332 L\n.5019 .60332 L\n.50381 .60332 L\n.50571 .60332 L\n.50762 .60332 L\n.50952 .60332 L\n.51143 .60332 L\n.51333 .60332 L\n.51524 .60332 L\n.51714 .60332 L\n.51905 .60332 L\n.52095 .60332 L\n.52286 .60332 L\n.52476 .60332 L\n.52667 .60332 L\n.52857 .60332 L\n.53048 .60332 L\n.53238 .60332 L\n.53429 .60332 L\n.53619 .60332 L\n.5381 .60332 L\n.54 .60332 L\n.5419 .60332 L\n.54381 .60332 L\n.54571 .60332 L\n.54762 .60332 L\n.54952 .60332 L\n.55101 .60332 L\n.55101 .01472 L\n.5511 .01472 L\n.5511 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\nMistroke\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\n.59333 .01472 L\n.59524 .01472 L\n.59524 .60332 L\n.59533 .60332 L\n.59533 .60332 L\n.59714 .60332 L\n.59905 .60332 L\n.60095 .60332 L\n.60286 .60332 L\n.60476 .60332 L\n.60667 .60332 L\n.60857 .60332 L\n.61048 .60332 L\n.61238 .60332 L\n.61429 .60332 L\n.61619 .60332 L\n.6181 .60332 L\n.62 .60332 L\n.6219 .60332 L\n.62381 .60332 L\n.62571 .60332 L\n.62762 .60332 L\n.62952 .60332 L\n.63143 .60332 L\n.63333 .60332 L\n.63524 .60332 L\n.63714 .60332 L\n.63905 .60332 L\n.64095 .60332 L\n.64286 .60332 L\n.64476 .60332 L\n.64667 .60332 L\n.64857 .60332 L\n.65048 .60332 L\n.65238 .60332 L\n.65429 .60332 L\n.65619 .60332 L\nMistroke\n.6581 .60332 L\n.66 .60332 L\n.6619 .60332 L\n.66381 .60332 L\n.66571 .60332 L\n.66762 .60332 L\n.66952 .60332 L\n.67143 .60332 L\n.67333 .60332 L\n.67524 .60332 L\n.67714 .60332 L\n.67905 .60332 L\n.68095 .60332 L\n.68286 .60332 L\n.68476 .60332 L\n.68667 .60332 L\n.68857 .60332 L\n.69048 .60332 L\n.69238 .60332 L\n.69429 .60332 L\n.69619 .60332 L\n.6981 .60332 L\n.7 .60332 L\n.7019 .60332 L\n.70381 .60332 L\n.70571 .60332 L\n.70762 .60332 L\n.70952 .60332 L\n.71143 .60332 L\n.71333 .60332 L\n.71524 .60332 L\n.71714 .60332 L\n.71905 .60332 L\n.72095 .60332 L\n.72286 .60332 L\n.72476 .60332 L\n.72667 .60332 L\n.72857 .60332 L\n.73048 .60332 L\n.73238 .60332 L\n.73429 .60332 L\n.73619 .60332 L\n.7381 .60332 L\n.7381 .60332 L\n.73819 .60332 L\n.73819 .60332 L\n.74 .60332 L\n.7419 .60332 L\n.74381 .60332 L\n.74571 .60332 L\nMistroke\n.74762 .60332 L\n.74952 .60332 L\n.75143 .60332 L\n.75333 .60332 L\n.75524 .60332 L\n.75714 .60332 L\n.75905 .60332 L\n.76095 .60332 L\n.76286 .60332 L\n.76476 .60332 L\n.76667 .60332 L\n.76857 .60332 L\n.77048 .60332 L\n.77238 .60332 L\n.77429 .60332 L\n.77619 .60332 L\n.7781 .60332 L\n.78 .60332 L\n.7819 .60332 L\n.78381 .60332 L\n.78571 .60332 L\n.78762 .60332 L\n.78885 .60332 L\n.78885 .01472 L\n.78894 .01472 L\n.78894 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\nMistroke\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\n.87905 .01472 L\n.88095 .01472 L\n.88095 .60332 L\n.88105 .60332 L\n.88105 .60332 L\n.88286 .60332 L\n.88476 .60332 L\n.88667 .60332 L\n.88857 .60332 L\n.89048 .60332 L\n.89238 .60332 L\n.89429 .60332 L\n.89619 .60332 L\n.8981 .60332 L\n.9 .60332 L\n.9019 .60332 L\n.90381 .60332 L\n.90571 .60332 L\n.90762 .60332 L\n.90952 .60332 L\n.91143 .60332 L\n.91333 .60332 L\n.91524 .60332 L\n.91714 .60332 L\n.91905 .60332 L\n.92095 .60332 L\n.92286 .60332 L\nMistroke\n.92476 .60332 L\n.92667 .60332 L\n.92857 .60332 L\n.93048 .60332 L\n.93238 .60332 L\n.93429 .60332 L\n.93619 .60332 L\n.9381 .60332 L\n.94 .60332 L\n.9419 .60332 L\n.94381 .60332 L\n.94571 .60332 L\n.94762 .60332 L\n.94952 .60332 L\n.95143 .60332 L\n.95333 .60332 L\n.95524 .60332 L\n.95714 .60332 L\n.95905 .60332 L\n.96095 .60332 L\n.96286 .60332 L\n.96476 .60332 L\n.96667 .60332 L\n.96857 .60332 L\n.97048 .60332 L\n.97238 .60332 L\n.97429 .60332 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(simpleDEVSServer) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(DEVSactive) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n231.000 12.813 moveto\n(L) show\n237.000 12.813 moveto\n(@) show\n243.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n249.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P0000160000ocT0000030000?l]000001P0003o700000080=WI\nf@040000003IfMT0fMWI000000`0fMWI000L0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WI\nf@3IfMT0203IfMT00`000000003o0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@0;0=WIf@030000003IfMT0fMWI00d0fMWI000L0=WIf@03\n0000003IfMT0fMWI02/0fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000003IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@0:0=WIf@D0\n0000303IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0280fMWI00<000000=WIf@3IfMT03@3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`00\n0?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g\n0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0\n:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI\n00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000@0fMWI0P0000030=WIf@@00000103IfMT00`000000\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0\nfMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI\n00@000000=WIf@3IfMT00000203IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI\n0=WIf@0000090=WIf@030000003IfMT0fMWI00@0fMWI0P0000130=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002P3IfMT00`000000\nfMWI0=WIf@030=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI0000\n00@0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI00090=WIf@8000002P3IfMT2000000D0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3I\nfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L\n0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@03\n0000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0\nfMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI\n00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI\n000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@002@3IfMT2000000@0fMWI0P0000050=WIf@<000000`3IfMT00`000000fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WI\nf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@000008\n0=WIf@D000000`3IfMT2000004<0fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0\n00000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI0008\n0=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@000000103IfMT00`000000fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P00000;\n0=WIf@800000103IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o\n0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WI\nf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3I\nfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0\nfMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0\nfMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`00\n0000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`00\n0?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0\nfMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g\n0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0\n:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI\n00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3I\nfMT2000000@0fMWI0P0000030=WIf@<000001@3IfMT00`000000fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203I\nfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000\n003IfMT0fMWI000000@0fMWI0P0000130=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT3000000D0fMWI00<000000=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000\n003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P00000:0=WIf@<00000\n103IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT0\n0`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<00000\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3I\nfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n00000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L\n0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@03\n0000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@800000103IfMT20000\n00@0fMWI0P0000050=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI\n000000@0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103I\nfMT2000004<0fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT0\n10000000fMWI0=WIf@0000090=WIf@8000001@3IfMT00`000000fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203I\nfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@030000003I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI\n00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\n003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WI\nf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT0\n0`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000D0=WIf@D000000`3IfMT00`000000fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001H0fMWI00<0\n00000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI0200fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000F\n0=WIf@030000003IfMT0fMWI00<0fMWI0P0000130=WIfC/0003o2P3IfMT_0000oaH0fMWI60000?lM\n0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3IfMT001H0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WIf@800000\n1@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI0140fMWI0`00\n003T0=WIf@00903IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI0><0fMWI000S0=WI\nf@030000003IfMT0fMWI0080fMWI0`0000040=WIf@<0000000P0fMWI000000000000fMWI0000003I\nfMT000000=WIf@<000000`3IfMT3000000@0fMWI0`0000000`3IfMT000000000000200000080fMWI\n1@0000030=WIf@8000000`3IfMT3000000<0fMWI0`0000040=WIf@<000000P3IfMT4000000<0fMWI\n0P0000040=WIf@<000000P3IfMT4000000<0fMWI0P0000020=WIf@@000000P3IfMT5000000<0fMWI\n0P0000230=WIf@008`3IfMT00`000000fMWI0=WIf@050=WIf@030000003IfMT0fMWI0080fMWI00/0\n00000=WIf@3IfMT0fMWI0000003IfMT000000=WIf@000000fMWI000000020=WIf@050000003IfMT0\nfMWI0=WIf@000000103IfMT00`000000fMWI0=WIf@030=WIf@040000003IfMT0fMWI00000080fMWI\n00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@040=WIf@030000003I\nfMT0fMWI00<0fMWI0P0000030=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203I\nfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3I\nfMT0P`3IfMT001H0fMWI20000?l50=WIf@030000003IfMT0fMWI0080fMWI0`0000050=WIf@0;0000\n003IfMT0fMWI0=WIf@000000fMWI0000003IfMT000000=WIf@0000000P3IfMT01@000000fMWI0=WI\nf@3IfMT0000000@0fMWI100000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3I\nfMT00`3IfMT010000000fMWI0=WIf@0000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WI\nf@3IfMT0100000030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT4\n000000<0fMWI00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@030000003I\nfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0P`3IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT4\n000000<0fMWI0P0000020=WIf@<0000000<0fMWI0000003IfMT0100000040=WIf@030000003IfMT0\nfMWI00<0fMWI0P0000030=WIf@040000003IfMT0fMWI00000080fMWI0P0000040=WIf@040000003I\nfMT0fMWI000000<0fMWI0P0000040=WIf@800000103IfMT2000000<0fMWI100000000`3IfMT00000\n000000020=WIf@8000000P3IfMT2000000<0fMWI100000080=WIf@040000003IfMT0fMWI00000080\nfMWI0P0000040=WIf@030000003IfMT0fMWI08<0fMWI000T0=WIf@030000003IfMT0fMWI01X0fMWI\n00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0fMWI0000\n003IfMT00000103IfMT00`000000fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<0\n00000=WIf@3IfMT0903IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0fMWI0000003I\nfMT00000QP3IfMT002D0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@0=0=WIf@80\n00002@3IfMT400000080fMWI1@0000000`3IfMT00000000000020=WIf@80000000<0fMWI00000000\n00000P0000020=WIf@@000009@3IfMT400000080fMWI1@0000000`3IfMT00000000000250=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 6:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - The variable simpleDEVSServer.DEVSactive is true while a job is being serviced.", Rule[FontFamily, "Courier New"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( simpleDEVSServer.servTime );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.208022 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .11873 -18 -4.5 ]\n[.01131 .11873 0 4.5 ]\n[.01131 .22274 -6 -4.5 ]\n[.01131 .22274 0 4.5 ]\n[.01131 .32675 -18 -4.5 ]\n[.01131 .32675 0 4.5 ]\n[.01131 .43076 -6 -4.5 ]\n[.01131 .43076 0 4.5 ]\n[.01131 .53477 -18 -4.5 ]\n[.01131 .53477 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .11873 m\n.03006 .11873 L\ns\n[(0.5)] .01131 .11873 1 0 Mshowa\n.02381 .22274 m\n.03006 .22274 L\ns\n[(1)] .01131 .22274 1 0 Mshowa\n.02381 .32675 m\n.03006 .32675 L\ns\n[(1.5)] .01131 .32675 1 0 Mshowa\n.02381 .43076 m\n.03006 .43076 L\ns\n[(2)] .01131 .43076 1 0 Mshowa\n.02381 .53477 m\n.03006 .53477 L\ns\n[(2.5)] .01131 .53477 1 0 Mshowa\n.125 Mabswid\n.02381 .03552 m\n.02756 .03552 L\ns\n.02381 .05632 m\n.02756 .05632 L\ns\n.02381 .07712 m\n.02756 .07712 L\ns\n.02381 .09792 m\n.02756 .09792 L\ns\n.02381 .13953 m\n.02756 .13953 L\ns\n.02381 .16033 m\n.02756 .16033 L\ns\n.02381 .18113 m\n.02756 .18113 L\ns\n.02381 .20193 m\n.02756 .20193 L\ns\n.02381 .24354 m\n.02756 .24354 L\ns\n.02381 .26434 m\n.02756 .26434 L\ns\n.02381 .28514 m\n.02756 .28514 L\ns\n.02381 .30595 m\n.02756 .30595 L\ns\n.02381 .34755 m\n.02756 .34755 L\ns\n.02381 .36835 m\n.02756 .36835 L\ns\n.02381 .38915 m\n.02756 .38915 L\ns\n.02381 .40996 m\n.02756 .40996 L\ns\n.02381 .45156 m\n.02756 .45156 L\ns\n.02381 .47236 m\n.02756 .47236 L\ns\n.02381 .49316 m\n.02756 .49316 L\ns\n.02381 .51397 m\n.02756 .51397 L\ns\n.02381 .55557 m\n.02756 .55557 L\ns\n.02381 .57637 m\n.02756 .57637 L\ns\n.02381 .59718 m\n.02756 .59718 L\ns\n.02381 .61798 m\n.02756 .61798 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02382 .01472 L\n.02382 .51422 L\n.0239 .51422 L\n.0239 .51422 L\n.02571 .51422 L\n.02762 .51422 L\n.02952 .51422 L\n.03143 .51422 L\n.03333 .51422 L\n.03524 .51422 L\n.03714 .51422 L\n.03905 .51422 L\n.04095 .51422 L\n.04286 .51422 L\n.04476 .51422 L\n.04667 .51422 L\n.04857 .51422 L\n.05048 .51422 L\n.05238 .51422 L\n.05429 .51422 L\n.05619 .51422 L\n.0581 .51422 L\n.06 .51422 L\n.0619 .51422 L\n.06381 .51422 L\n.06571 .51422 L\n.06762 .51422 L\n.06952 .51422 L\n.07143 .51422 L\n.07333 .51422 L\n.07524 .51422 L\n.07714 .51422 L\n.07905 .51422 L\n.08095 .51422 L\n.08286 .51422 L\n.08476 .51422 L\n.08667 .51422 L\n.08857 .51422 L\n.09048 .51422 L\n.09238 .51422 L\n.09429 .51422 L\n.09619 .51422 L\n.0981 .51422 L\n.1 .51422 L\n.1019 .51422 L\n.10381 .51422 L\n.10571 .51422 L\n.10762 .51422 L\n.10952 .51422 L\nMistroke\n.11143 .51422 L\n.11333 .51422 L\n.11524 .51422 L\n.11714 .51422 L\n.11905 .51422 L\n.12095 .51422 L\n.12286 .51422 L\n.12476 .51422 L\n.12667 .51422 L\n.12857 .51422 L\n.13048 .51422 L\n.13238 .51422 L\n.13429 .51422 L\n.13619 .51422 L\n.1381 .51422 L\n.14 .51422 L\n.1419 .51422 L\n.14381 .51422 L\n.14571 .51422 L\n.14762 .51422 L\n.14952 .51422 L\n.15143 .51422 L\n.15333 .51422 L\n.15524 .51422 L\n.15714 .51422 L\n.15905 .51422 L\n.16095 .51422 L\n.16286 .51422 L\n.16476 .51422 L\n.16667 .51422 L\n.16857 .51422 L\n.17048 .51422 L\n.17238 .51422 L\n.17429 .51422 L\n.17619 .51422 L\n.1781 .51422 L\n.18 .51422 L\n.1819 .51422 L\n.18381 .51422 L\n.18571 .51422 L\n.18762 .51422 L\n.18952 .51422 L\n.19143 .51422 L\n.19333 .51422 L\n.19524 .51422 L\n.19714 .51422 L\n.19905 .51422 L\n.20095 .51422 L\n.20286 .51422 L\n.20476 .51422 L\nMistroke\n.20667 .51422 L\n.20857 .51422 L\n.21048 .51422 L\n.21238 .51422 L\n.21429 .51422 L\n.21619 .51422 L\n.2181 .51422 L\n.22 .51422 L\n.2219 .51422 L\n.22381 .51422 L\n.22571 .51422 L\n.22762 .51422 L\n.22952 .51422 L\n.23143 .51422 L\n.23333 .51422 L\n.23524 .51422 L\n.23714 .51422 L\n.23905 .51422 L\n.24095 .51422 L\n.24286 .51422 L\n.24476 .51422 L\n.24667 .51422 L\n.24857 .51422 L\n.25048 .51422 L\n.25238 .51422 L\n.25429 .51422 L\n.25619 .51422 L\n.2581 .51422 L\n.26 .51422 L\n.2619 .51422 L\n.26381 .51422 L\n.26571 .51422 L\n.26762 .51422 L\n.26952 .51422 L\n.27143 .51422 L\n.27333 .51422 L\n.27524 .51422 L\n.27714 .51422 L\n.27905 .51422 L\n.28095 .51422 L\n.28286 .51422 L\n.28476 .51422 L\n.28667 .51422 L\n.28857 .51422 L\n.29048 .51422 L\n.29238 .51422 L\n.29429 .51422 L\n.29619 .51422 L\n.2981 .51422 L\n.3 .51422 L\nMistroke\n.3019 .51422 L\n.30381 .51422 L\n.30571 .51422 L\n.30762 .51422 L\n.30952 .51422 L\n.30952 .54217 L\n.30962 .54217 L\n.30962 .54217 L\n.31143 .54217 L\n.31333 .54217 L\n.31524 .54217 L\n.31714 .54217 L\n.31905 .54217 L\n.32095 .54217 L\n.32286 .54217 L\n.32476 .54217 L\n.32667 .54217 L\n.32857 .54217 L\n.33048 .54217 L\n.33238 .54217 L\n.33429 .54217 L\n.33619 .54217 L\n.3381 .54217 L\n.34 .54217 L\n.3419 .54217 L\n.34381 .54217 L\n.34571 .54217 L\n.34762 .54217 L\n.34952 .54217 L\n.35143 .54217 L\n.35333 .54217 L\n.35524 .54217 L\n.35714 .54217 L\n.35905 .54217 L\n.36095 .54217 L\n.36286 .54217 L\n.36476 .54217 L\n.36667 .54217 L\n.36857 .54217 L\n.37048 .54217 L\n.37238 .54217 L\n.37429 .54217 L\n.37619 .54217 L\n.3781 .54217 L\n.38 .54217 L\n.3819 .54217 L\n.38381 .54217 L\n.38571 .54217 L\n.38762 .54217 L\n.38952 .54217 L\nMistroke\n.39143 .54217 L\n.39333 .54217 L\n.39524 .54217 L\n.39714 .54217 L\n.39905 .54217 L\n.40095 .54217 L\n.40286 .54217 L\n.40476 .54217 L\n.40667 .54217 L\n.40857 .54217 L\n.41048 .54217 L\n.41238 .54217 L\n.41429 .54217 L\n.41619 .54217 L\n.4181 .54217 L\n.42 .54217 L\n.4219 .54217 L\n.42381 .54217 L\n.42571 .54217 L\n.42762 .54217 L\n.42952 .54217 L\n.43143 .54217 L\n.43333 .54217 L\n.43524 .54217 L\n.43714 .54217 L\n.43905 .54217 L\n.44095 .54217 L\n.44286 .54217 L\n.44476 .54217 L\n.44667 .54217 L\n.44857 .54217 L\n.45048 .54217 L\n.45238 .54217 L\n.45238 .42364 L\n.45248 .42364 L\n.45248 .42364 L\n.45429 .42364 L\n.45619 .42364 L\n.4581 .42364 L\n.46 .42364 L\n.4619 .42364 L\n.46381 .42364 L\n.46571 .42364 L\n.46762 .42364 L\n.46952 .42364 L\n.47143 .42364 L\n.47333 .42364 L\n.47524 .42364 L\n.47714 .42364 L\n.47905 .42364 L\nMistroke\n.48095 .42364 L\n.48286 .42364 L\n.48476 .42364 L\n.48667 .42364 L\n.48857 .42364 L\n.49048 .42364 L\n.49238 .42364 L\n.49429 .42364 L\n.49619 .42364 L\n.4981 .42364 L\n.5 .42364 L\n.5019 .42364 L\n.50381 .42364 L\n.50571 .42364 L\n.50762 .42364 L\n.50952 .42364 L\n.51143 .42364 L\n.51333 .42364 L\n.51524 .42364 L\n.51714 .42364 L\n.51905 .42364 L\n.52095 .42364 L\n.52286 .42364 L\n.52476 .42364 L\n.52667 .42364 L\n.52857 .42364 L\n.53048 .42364 L\n.53238 .42364 L\n.53429 .42364 L\n.53619 .42364 L\n.5381 .42364 L\n.54 .42364 L\n.5419 .42364 L\n.54381 .42364 L\n.54571 .42364 L\n.54762 .42364 L\n.54952 .42364 L\n.55101 .42364 L\n.55101 .47688 L\n.5511 .47688 L\n.5511 .47688 L\n.55143 .47688 L\n.55333 .47688 L\n.55524 .47688 L\n.55714 .47688 L\n.55905 .47688 L\n.56095 .47688 L\n.56286 .47688 L\n.56476 .47688 L\n.56667 .47688 L\nMistroke\n.56857 .47688 L\n.57048 .47688 L\n.57238 .47688 L\n.57429 .47688 L\n.57619 .47688 L\n.5781 .47688 L\n.58 .47688 L\n.5819 .47688 L\n.58381 .47688 L\n.58571 .47688 L\n.58762 .47688 L\n.58952 .47688 L\n.59143 .47688 L\n.59333 .47688 L\n.59524 .47688 L\n.59524 .4376 L\n.59533 .4376 L\n.59533 .4376 L\n.59714 .4376 L\n.59905 .4376 L\n.60095 .4376 L\n.60286 .4376 L\n.60476 .4376 L\n.60667 .4376 L\n.60857 .4376 L\n.61048 .4376 L\n.61238 .4376 L\n.61429 .4376 L\n.61619 .4376 L\n.6181 .4376 L\n.62 .4376 L\n.6219 .4376 L\n.62381 .4376 L\n.62571 .4376 L\n.62762 .4376 L\n.62952 .4376 L\n.63143 .4376 L\n.63333 .4376 L\n.63524 .4376 L\n.63714 .4376 L\n.63905 .4376 L\n.64095 .4376 L\n.64286 .4376 L\n.64476 .4376 L\n.64667 .4376 L\n.64857 .4376 L\n.65048 .4376 L\n.65238 .4376 L\n.65429 .4376 L\n.65619 .4376 L\nMistroke\n.6581 .4376 L\n.66 .4376 L\n.6619 .4376 L\n.66381 .4376 L\n.66571 .4376 L\n.66762 .4376 L\n.66952 .4376 L\n.67143 .4376 L\n.67333 .4376 L\n.67524 .4376 L\n.67714 .4376 L\n.67905 .4376 L\n.68095 .4376 L\n.68286 .4376 L\n.68476 .4376 L\n.68667 .4376 L\n.68857 .4376 L\n.69048 .4376 L\n.69238 .4376 L\n.69429 .4376 L\n.69619 .4376 L\n.6981 .4376 L\n.7 .4376 L\n.7019 .4376 L\n.70381 .4376 L\n.70571 .4376 L\n.70762 .4376 L\n.70952 .4376 L\n.71143 .4376 L\n.71333 .4376 L\n.71524 .4376 L\n.71714 .4376 L\n.71905 .4376 L\n.72095 .4376 L\n.72286 .4376 L\n.72476 .4376 L\n.72667 .4376 L\n.72857 .4376 L\n.73048 .4376 L\n.73238 .4376 L\n.73429 .4376 L\n.73619 .4376 L\n.7381 .4376 L\n.7381 .47638 L\n.73819 .47638 L\n.73819 .47638 L\n.74 .47638 L\n.7419 .47638 L\n.74381 .47638 L\n.74571 .47638 L\nMistroke\n.74762 .47638 L\n.74952 .47638 L\n.75143 .47638 L\n.75333 .47638 L\n.75524 .47638 L\n.75714 .47638 L\n.75905 .47638 L\n.76095 .47638 L\n.76286 .47638 L\n.76476 .47638 L\n.76667 .47638 L\n.76857 .47638 L\n.77048 .47638 L\n.77238 .47638 L\n.77429 .47638 L\n.77619 .47638 L\n.7781 .47638 L\n.78 .47638 L\n.7819 .47638 L\n.78381 .47638 L\n.78571 .47638 L\n.78762 .47638 L\n.78885 .47638 L\n.78885 .60332 L\n.78894 .60332 L\n.78894 .60332 L\n.78952 .60332 L\n.79143 .60332 L\n.79333 .60332 L\n.79524 .60332 L\n.79714 .60332 L\n.79905 .60332 L\n.80095 .60332 L\n.80286 .60332 L\n.80476 .60332 L\n.80667 .60332 L\n.80857 .60332 L\n.81048 .60332 L\n.81238 .60332 L\n.81429 .60332 L\n.81619 .60332 L\n.8181 .60332 L\n.82 .60332 L\n.8219 .60332 L\n.82381 .60332 L\n.82571 .60332 L\n.82762 .60332 L\n.82952 .60332 L\n.83143 .60332 L\n.83333 .60332 L\nMistroke\n.83524 .60332 L\n.83714 .60332 L\n.83905 .60332 L\n.84095 .60332 L\n.84286 .60332 L\n.84476 .60332 L\n.84667 .60332 L\n.84857 .60332 L\n.85048 .60332 L\n.85238 .60332 L\n.85429 .60332 L\n.85619 .60332 L\n.8581 .60332 L\n.86 .60332 L\n.8619 .60332 L\n.86381 .60332 L\n.86571 .60332 L\n.86762 .60332 L\n.86952 .60332 L\n.87143 .60332 L\n.87333 .60332 L\n.87524 .60332 L\n.87714 .60332 L\n.87905 .60332 L\n.88095 .60332 L\n.88095 .51187 L\n.88105 .51187 L\n.88105 .51187 L\n.88286 .51187 L\n.88476 .51187 L\n.88667 .51187 L\n.88857 .51187 L\n.89048 .51187 L\n.89238 .51187 L\n.89429 .51187 L\n.89619 .51187 L\n.8981 .51187 L\n.9 .51187 L\n.9019 .51187 L\n.90381 .51187 L\n.90571 .51187 L\n.90762 .51187 L\n.90952 .51187 L\n.91143 .51187 L\n.91333 .51187 L\n.91524 .51187 L\n.91714 .51187 L\n.91905 .51187 L\n.92095 .51187 L\n.92286 .51187 L\nMistroke\n.92476 .51187 L\n.92667 .51187 L\n.92857 .51187 L\n.93048 .51187 L\n.93238 .51187 L\n.93429 .51187 L\n.93619 .51187 L\n.9381 .51187 L\n.94 .51187 L\n.9419 .51187 L\n.94381 .51187 L\n.94571 .51187 L\n.94762 .51187 L\n.94952 .51187 L\n.95143 .51187 L\n.95333 .51187 L\n.95524 .51187 L\n.95714 .51187 L\n.95905 .51187 L\n.96095 .51187 L\n.96286 .51187 L\n.96476 .51187 L\n.96667 .51187 L\n.96857 .51187 L\n.97048 .51187 L\n.97238 .51187 L\n.97429 .51187 L\n.97619 .51187 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(simpleDEVSServer) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(servTime) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n219.000 12.813 moveto\n(L) show\n225.000 12.813 moveto\n(@) show\n231.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n237.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0\nfMWI1@0000020=WIf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI\n0=WIf@0X0=WIf@040000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@00\n0000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WI\nf@3IfMT0:P3IfMT5000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00\n000Y0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3I\nfMT00`000000fMWI0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000\n:P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI\n00@000000=WIf@3IfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0\nfMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<00000\n0=WIf@3IfMT05`3IfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WI\nf@800000103IfMT2000001X0fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0l@3IfMT010000000fMWI0=WIf@00000<\n0=WIf@005P3IfMT6000000030000o`00000000000>T00000203IfMT00`000000fMWI0=WIf@0=0=WI\nf@00703IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3I\nfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3I\nfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3I\nfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3I\nfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3I\nfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3I\nfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<000000=WIf@3I\nfMT02P3IfMT5000000`0fMWI000L0=WIf@030000o`3IfMT0fMWI0?40fMWI00<000000=WIf@3IfMT0\n3@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o\n0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3I\nfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`00\n0?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L\n0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WI\nf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o\n0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI00090=WIf@800000103I\nfMT2000000@0fMWI0P0000050=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT000P0fMWI00@00000\n0=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@030000o`3IfMT0fMWI0?l0fMWI\n0P3IfMT000P0fMWI00@000000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@020=WIf@030000\no`000000fMWI0?l0fMWI0P3IfMT000P0fMWI00@000000=WIf@3IfMT000002@3IfMT2000000D0fMWI\n00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00203IfMT010000000fMWI0=WIf@0000090=WIf@030000\n003IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@002@3IfMT2000000X0fMWI0`00\n00040=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT2\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0\nfMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WI\nf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000\no`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI\n000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT2\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0\nfMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00503IfMT5000000<0fMWI00<0\n003o0=WIf@3IfMT0o`3IfMT20=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000o`3IfMT0\nfMWI0?l0fMWI0P3IfMT001H0fMWI00<000000=WIf@3IfMT00`3IfMT00`000?l000000=WIf@3o0=WI\nf@80fMWI000F0=WIf@030000003IfMT0fMWI00<0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00\n5P3IfMT00`000000fMWI0=WIf@030=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001D0fMWI0P00\n00050=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT2\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0\nfMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WI\nf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000\no`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI\n000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT2\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0\nfMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00203IfMT500000080fMWI0P00\n00040=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000:0=WIf@030000003IfMT0\nfMWI00L0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000:\n0=WIf@030000003IfMT0fMWI00X0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l000000=WIf@3o\n0=WIf@80fMWI000:0=WIf@030000003IfMT0fMWI00P0fMWI0P0000050=WIf@030000o`3IfMT0fMWI\n0?l0fMWI0P3IfMT000X0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@040=WIf@03\n0000o`3IfMT0fMWI0?l0fMWI0P3IfMT000T0fMWI0P00000:0=WIf@<00000103IfMT00`000?l0fMWI\n0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703I\nfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80\nfMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3I\nfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI\n0?l0fMWI0P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI\n0=WIf@3o0=WIf@80fMWI000L0=WIf@030000o`3IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000?l0fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0?l0fMWI0P3IfMT001@0fMWI100000040=WIf@030000o`3IfMT0fMWI06D0fMWI\n00<0003o0=WIf@3IfMT0V@3IfMT001@0fMWI00<000000=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WI\nf@1T0=WIfAT0003oQ03IfMT001D0fMWI00<000000=WIf@3IfMT0103IfMT00`000?l000000=WIf@1T\n0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0PP3IfMT001H0fMWI00<000000=WI\nf@3IfMT00`3IfMT00`000?l0fMWI0=WIf@1T0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WI\nf@3IfMT0PP3IfMT001@0fMWI00@000000=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@1T0=WI\nf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0203IfMTS0000oeL0fMWI000E0=WIf@80\n00001@3IfMT00`000?l0fMWI0=WIf@1T0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI05D0fMWI000L0=WIf@030000\no`3IfMT0fMWI06@0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0E@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0I03I\nfMT00`000?l0fMWI0=WIf@0E0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3I\nfMT00`000?l0fMWI0=WIf@1E0=WIf@00703IfMT00`000?l0fMWI0=WIf@1T0=WIf@030000o`3IfMT0\nfMWI01D0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0\nfMWI05D0fMWI000L0=WIf@030000o`3IfMT0fMWI06@0fMWI00<0003o0=WIf@3IfMT05@3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0E@3IfMT001`0\nfMWI00<0003o0=WIf@3IfMT0I03IfMT00`000?l0fMWI0=WIf@0E0=WIf@030000o`3IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@1E0=WIf@00703IfMT00`000?l0fMWI\n0=WIf@1T0=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI05D0fMWI000L0=WIf@030000o`3IfMT0fMWI06@0fMWI00<0\n003o0=WIf@3IfMT05@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0E@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0I03IfMT00`000?l0fMWI0=WIf@0E\n0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT07`3IfMT=0000od/0fMWI000L0=WI\nf@030000o`3IfMT0fMWI06@0fMWI00<0003o0=WIf@3IfMT05@3IfMT<0000obd0fMWI00<0003o0=WI\nf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0I03IfMT00`000?l0fMWI0=WIf@1>0=WIf@03\n0000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI06@0fMWI00<0003o0=WIf@3IfMT0\nCP3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0fMWI0=WIf@1T0=WIf@030000o`3I\nfMT0fMWI04h0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0I03IfMT0\n0`000?l0fMWI0=WIf@1>0=WIf@030000o`3IfMT0fMWI04T0fMWI000L0=WIf@030000o`3IfMT0fMWI\n06@0fMWI00<0003o0=WIf@3IfMT0CP3IfMT00`000?l0fMWI0=WIf@190=WIf@00703IfMT00`000?l0\nfMWI0=WIf@1T0=WIf@030000o`3IfMT0fMWI04h0fMWI00<0003o0=WIf@3IfMT0B@3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0I03IfMT00`000?l0fMWI0=WIf@1>0=WIf@030000o`3IfMT0fMWI04T0fMWI\n000L0=WIf@030000o`3IfMT0fMWI06@0fMWI00<0003o0=WIf@3IfMT0CP3IfMT00`000?l0fMWI0=WI\nf@0D0=WIfAP0003o7@3IfMT001`0fMWIAP000?lQ0=WIf@030000o`3IfMT0fMWI04h0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0CP3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0380fMWI00080=WIf@@000000`3IfMT2000000@0fMWI0P0000050=WIf@03\n0000003IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT07`3IfMT00`000?l0fMWI0=WIf@1>0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00<000000=WIf@3IfMT0\n2@3IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT07`3IfMT00`000?l0fMWI0=WIf@1>0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT000T0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI0=WIf@020=WIf@800000\n@`3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI04h0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002P3IfMT00`000000fMWI0=WIf@080=WIf@8000001@3I\nfMT00`000000fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0CP3I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00080=WIf@040000003IfMT0\nfMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@120=WIfB<0003oD03I\nfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@8000002P3IfMT3\n000000@0fMWI00<000000=WIf@3IfMT0]@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0;D0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@2e0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0]@3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0;D0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@2e\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WI\nf@3IfMT0]@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000003IfMT0fMWI0;D0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000000fMWI0=WIf@2e0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<000000=WIf@3IfMT0]@3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0;D0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WIf@2e0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<000000=WIf@3IfMT0]@3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000003IfMT0fMWI0;D0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000000fMWI0=WI\nf@2e0=WIfAP0003o=03IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`00\n0000fMWI0=WIf@3o0=WIf@80fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI0P3IfMT001`0fMWI\n00<000000=WIf@3IfMT0o`3IfMT20=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@009@3IfMT00`000000fMWI0=WIf@0A0=WIf@<00000i03IfMT002@0fMWI00<0\n00000=WIf@3IfMT04`3IfMT00`000000fMWI0=WIf@3S0=WIf@008`3IfMT00`000000fMWI0=WIf@02\n0=WIf@<00000103IfMT3000000080=WIf@00000000000=WIf@000000fMWI0000003IfMT3000000<0\nfMWI0`0000040=WIf@<0000000<0fMWI0000000000000P0000020=WIf@D000000`3IfMT2000000<0\nfMWI0`0000030=WIf@<00000103IfMT300000080fMWI100000030=WIf@800000103IfMT300000080\nfMWI100000030=WIf@8000000`3IfMT3000000@0fMWI0`0000020=WIf@<00000P`3IfMT002<0fMWI\n00<000000=WIf@3IfMT01@3IfMT00`000000fMWI0=WIf@020=WIf@0;0000003IfMT0fMWI0=WIf@00\n0000fMWI0000003IfMT000000=WIf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000@0fMWI\n00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003I\nfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@030=WIf@800000\n0`3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00/0fMWI00@000000=WIf@3IfMT0\n00001P3IfMT00`000000fMWI0=WIf@220=WIf@005P3IfMT80000o`D0fMWI00<000000=WIf@3IfMT0\n0P3IfMT3000000D0fMWI00/000000=WIf@3IfMT0fMWI0000003IfMT000000=WIf@000000fMWI0000\n00020=WIf@050000003IfMT0fMWI0=WIf@000000103IfMT400000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT00`000000fMWI0=WIf@030=WIf@040000003IfMT0fMWI000000D0fMWI00<000000=WI\nf@3IfMT00`3IfMT00`000000fMWI0=WIf@04000000<0fMWI00<000000=WIf@3IfMT00P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@@000000`3IfMT00`000000fMWI0=WIf@080=WIf@<000000`3IfMT4\n000000<0fMWI00<000000=WIf@3IfMT0PP3IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT40000\n00<0fMWI0P0000020=WIf@<0000000<0fMWI0000003IfMT0100000040=WIf@030000003IfMT0fMWI\n00<0fMWI0P0000030=WIf@040000003IfMT0fMWI00000080fMWI0P0000040=WIf@040000003IfMT0\nfMWI000000<0fMWI0P0000040=WIf@800000103IfMT2000000<0fMWI100000000`3IfMT000000000\n00020=WIf@8000000P3IfMT2000000<0fMWI100000080=WIf@@000000`3IfMT2000000<0fMWI0`00\n00230=WIf@00903IfMT00`000000fMWI0=WIf@0J0=WIf@030000003IfMT0fMWI00P0fMWI00@00000\n0=WIf@3IfMT000000P3IfMT01P000000fMWI0=WIf@000000fMWI000000@0fMWI00<000000=WIf@00\n00000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI0;H0fMWI000U0=WIf@03\n0000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT03@3IfMT2000000T0fMWI100000020=WIf@D0\n000000<0fMWI0000000000000P3IfMT2000000030=WIf@0000000000008000000P3IfMT400000;P0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.449423`, 0.0434554`, 0.019895`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 7:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - ", Rule[FontFamily, "Courier New"]], StyleBox["The service time variable simpleDEVSServer.servTime obtains new values at events.", Rule[FontFamily, "Courier"]]]], "Text"]], Open]], Cell[CellGroupData[List[Cell["Plot", "Subsubsection"], Cell["plot( simpleDEVSServer.internalEvent );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0952381 0.0147151 0.588604 [\n[.21429 .00222 -3 -9 ]\n[.21429 .00222 3 0 ]\n[.40476 .00222 -3 -9 ]\n[.40476 .00222 3 0 ]\n[.59524 .00222 -3 -9 ]\n[.59524 .00222 3 0 ]\n[.78571 .00222 -3 -9 ]\n[.78571 .00222 3 0 ]\n[.97619 .00222 -6 -9 ]\n[.97619 .00222 6 0 ]\n[1.025 .01472 0 -6.28125 ]\n[1.025 .01472 10 6.28125 ]\n[.01131 .13244 -18 -4.5 ]\n[.01131 .13244 0 4.5 ]\n[.01131 .25016 -18 -4.5 ]\n[.01131 .25016 0 4.5 ]\n[.01131 .36788 -18 -4.5 ]\n[.01131 .36788 0 4.5 ]\n[.01131 .4856 -18 -4.5 ]\n[.01131 .4856 0 4.5 ]\n[.01131 .60332 -6 -4.5 ]\n[.01131 .60332 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.21429 .01472 m\n.21429 .02097 L\ns\n[(2)] .21429 .00222 0 1 Mshowa\n.40476 .01472 m\n.40476 .02097 L\ns\n[(4)] .40476 .00222 0 1 Mshowa\n.59524 .01472 m\n.59524 .02097 L\ns\n[(6)] .59524 .00222 0 1 Mshowa\n.78571 .01472 m\n.78571 .02097 L\ns\n[(8)] .78571 .00222 0 1 Mshowa\n.97619 .01472 m\n.97619 .02097 L\ns\n[(10)] .97619 .00222 0 1 Mshowa\n.125 Mabswid\n.07143 .01472 m\n.07143 .01847 L\ns\n.11905 .01472 m\n.11905 .01847 L\ns\n.16667 .01472 m\n.16667 .01847 L\ns\n.2619 .01472 m\n.2619 .01847 L\ns\n.30952 .01472 m\n.30952 .01847 L\ns\n.35714 .01472 m\n.35714 .01847 L\ns\n.45238 .01472 m\n.45238 .01847 L\ns\n.5 .01472 m\n.5 .01847 L\ns\n.54762 .01472 m\n.54762 .01847 L\ns\n.64286 .01472 m\n.64286 .01847 L\ns\n.69048 .01472 m\n.69048 .01847 L\ns\n.7381 .01472 m\n.7381 .01847 L\ns\n.83333 .01472 m\n.83333 .01847 L\ns\n.88095 .01472 m\n.88095 .01847 L\ns\n.92857 .01472 m\n.92857 .01847 L\ns\n.25 Mabswid\n0 .01472 m\n1 .01472 L\ns\ngsave\n1.025 .01472 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .13244 m\n.03006 .13244 L\ns\n[(0.2)] .01131 .13244 1 0 Mshowa\n.02381 .25016 m\n.03006 .25016 L\ns\n[(0.4)] .01131 .25016 1 0 Mshowa\n.02381 .36788 m\n.03006 .36788 L\ns\n[(0.6)] .01131 .36788 1 0 Mshowa\n.02381 .4856 m\n.03006 .4856 L\ns\n[(0.8)] .01131 .4856 1 0 Mshowa\n.02381 .60332 m\n.03006 .60332 L\ns\n[(1)] .01131 .60332 1 0 Mshowa\n.125 Mabswid\n.02381 .04415 m\n.02756 .04415 L\ns\n.02381 .07358 m\n.02756 .07358 L\ns\n.02381 .10301 m\n.02756 .10301 L\ns\n.02381 .16187 m\n.02756 .16187 L\ns\n.02381 .1913 m\n.02756 .1913 L\ns\n.02381 .22073 m\n.02756 .22073 L\ns\n.02381 .27959 m\n.02756 .27959 L\ns\n.02381 .30902 m\n.02756 .30902 L\ns\n.02381 .33845 m\n.02756 .33845 L\ns\n.02381 .39731 m\n.02756 .39731 L\ns\n.02381 .42674 m\n.02756 .42674 L\ns\n.02381 .45617 m\n.02756 .45617 L\ns\n.02381 .51503 m\n.02756 .51503 L\ns\n.02381 .54446 m\n.02756 .54446 L\ns\n.02381 .57389 m\n.02756 .57389 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02382 .01472 L\n.02382 .60332 L\n.0239 .60332 L\n.0239 .60332 L\n.02571 .60332 L\n.02762 .60332 L\n.02952 .60332 L\n.03143 .60332 L\n.03333 .60332 L\n.03524 .60332 L\n.03714 .60332 L\n.03905 .60332 L\n.04095 .60332 L\n.04286 .60332 L\n.04476 .60332 L\n.04667 .60332 L\n.04857 .60332 L\n.05048 .60332 L\n.05238 .60332 L\n.05429 .60332 L\n.05619 .60332 L\n.0581 .60332 L\n.06 .60332 L\n.0619 .60332 L\n.06381 .60332 L\n.06571 .60332 L\n.06762 .60332 L\n.06952 .60332 L\n.07143 .60332 L\n.07333 .60332 L\n.07524 .60332 L\n.07714 .60332 L\n.07905 .60332 L\n.08095 .60332 L\n.08286 .60332 L\n.08476 .60332 L\n.08667 .60332 L\n.08857 .60332 L\n.09048 .60332 L\n.09238 .60332 L\n.09429 .60332 L\n.09619 .60332 L\n.0981 .60332 L\n.1 .60332 L\n.1019 .60332 L\n.10381 .60332 L\n.10571 .60332 L\n.10762 .60332 L\n.10952 .60332 L\nMistroke\n.11143 .60332 L\n.11333 .60332 L\n.11524 .60332 L\n.11714 .60332 L\n.11905 .60332 L\n.12095 .60332 L\n.12286 .60332 L\n.12476 .60332 L\n.12667 .60332 L\n.12857 .60332 L\n.13048 .60332 L\n.13238 .60332 L\n.13429 .60332 L\n.13619 .60332 L\n.1381 .60332 L\n.14 .60332 L\n.1419 .60332 L\n.14381 .60332 L\n.14571 .60332 L\n.14762 .60332 L\n.14952 .60332 L\n.15143 .60332 L\n.15333 .60332 L\n.15524 .60332 L\n.15714 .60332 L\n.15905 .60332 L\n.16095 .60332 L\n.16286 .60332 L\n.16476 .60332 L\n.16667 .60332 L\n.16857 .60332 L\n.17048 .60332 L\n.17238 .60332 L\n.17429 .60332 L\n.17619 .60332 L\n.1781 .60332 L\n.18 .60332 L\n.1819 .60332 L\n.18381 .60332 L\n.18571 .60332 L\n.18762 .60332 L\n.18952 .60332 L\n.19143 .60332 L\n.19333 .60332 L\n.19524 .60332 L\n.19714 .60332 L\n.19905 .60332 L\n.20095 .60332 L\n.20286 .60332 L\n.20476 .60332 L\nMistroke\n.20667 .60332 L\n.20857 .60332 L\n.21048 .60332 L\n.21238 .60332 L\n.21429 .60332 L\n.21619 .60332 L\n.2181 .60332 L\n.22 .60332 L\n.2219 .60332 L\n.22381 .60332 L\n.22571 .60332 L\n.22762 .60332 L\n.22952 .60332 L\n.23143 .60332 L\n.23333 .60332 L\n.23524 .60332 L\n.23714 .60332 L\n.23905 .60332 L\n.24095 .60332 L\n.24286 .60332 L\n.24476 .60332 L\n.24667 .60332 L\n.24857 .60332 L\n.25048 .60332 L\n.25238 .60332 L\n.25429 .60332 L\n.25619 .60332 L\n.2581 .60332 L\n.26 .60332 L\n.2619 .60332 L\n.26381 .60332 L\n.26571 .60332 L\n.26762 .60332 L\n.26952 .60332 L\n.27143 .60332 L\n.27333 .60332 L\n.27524 .60332 L\n.27714 .60332 L\n.27905 .60332 L\n.28095 .60332 L\n.28286 .60332 L\n.28476 .60332 L\n.28667 .60332 L\n.28857 .60332 L\n.29048 .60332 L\n.29238 .60332 L\n.29429 .60332 L\n.29619 .60332 L\n.2981 .60332 L\n.3 .60332 L\nMistroke\n.3019 .60332 L\n.30381 .60332 L\n.30571 .60332 L\n.30762 .60332 L\n.30952 .60332 L\n.30952 .01472 L\n.30962 .01472 L\n.30962 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\nMistroke\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45238 .01472 L\n.45248 .01472 L\n.45248 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\nMistroke\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55101 .01472 L\n.55101 .60332 L\n.5511 .60332 L\n.5511 .60332 L\n.55143 .60332 L\n.55333 .60332 L\n.55524 .60332 L\n.55714 .60332 L\n.55905 .60332 L\n.56095 .60332 L\n.56286 .60332 L\n.56476 .60332 L\n.56667 .60332 L\nMistroke\n.56857 .60332 L\n.57048 .60332 L\n.57238 .60332 L\n.57429 .60332 L\n.57619 .60332 L\n.5781 .60332 L\n.58 .60332 L\n.5819 .60332 L\n.58381 .60332 L\n.58571 .60332 L\n.58762 .60332 L\n.58952 .60332 L\n.59143 .60332 L\n.59333 .60332 L\n.59524 .60332 L\n.59524 .01472 L\n.59533 .01472 L\n.59533 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\nMistroke\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.7381 .01472 L\n.73819 .01472 L\n.73819 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\nMistroke\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78885 .01472 L\n.78885 .60332 L\n.78894 .60332 L\n.78894 .60332 L\n.78952 .60332 L\n.79143 .60332 L\n.79333 .60332 L\n.79524 .60332 L\n.79714 .60332 L\n.79905 .60332 L\n.80095 .60332 L\n.80286 .60332 L\n.80476 .60332 L\n.80667 .60332 L\n.80857 .60332 L\n.81048 .60332 L\n.81238 .60332 L\n.81429 .60332 L\n.81619 .60332 L\n.8181 .60332 L\n.82 .60332 L\n.8219 .60332 L\n.82381 .60332 L\n.82571 .60332 L\n.82762 .60332 L\n.82952 .60332 L\n.83143 .60332 L\n.83333 .60332 L\nMistroke\n.83524 .60332 L\n.83714 .60332 L\n.83905 .60332 L\n.84095 .60332 L\n.84286 .60332 L\n.84476 .60332 L\n.84667 .60332 L\n.84857 .60332 L\n.85048 .60332 L\n.85238 .60332 L\n.85429 .60332 L\n.85619 .60332 L\n.8581 .60332 L\n.86 .60332 L\n.8619 .60332 L\n.86381 .60332 L\n.86571 .60332 L\n.86762 .60332 L\n.86952 .60332 L\n.87143 .60332 L\n.87333 .60332 L\n.87524 .60332 L\n.87714 .60332 L\n.87905 .60332 L\n.88095 .60332 L\n.88095 .01472 L\n.88105 .01472 L\n.88105 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\nMistroke\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(simpleDEVSServer) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(internalEvent) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n249.000 12.813 moveto\n(L) show\n255.000 12.813 moveto\n(@) show\n261.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n267.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00B03IfMT4000002`0fMWI0`00000Y0=WIf@<00000;03IfMT2000002P0fMWI1@0000020=WI\nf@8000006P3IfMT004P0fMWI00<000000=WIf@3IfMT0;P3IfMT00`000000fMWI0=WIf@0X0=WIf@04\n0000003IfMT0fMWI000002X0fMWI00@000000=WIf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004T0fMWI00<000000=WIf@3IfMT0:P3IfMT5\n000002T0fMWI00@000000=WIf@3IfMT00000:P3IfMT010000000fMWI0=WIf@00000Y0=WIf@050000\n003IfMT0fMWI0=WIf@0000000P3IfMT00`000000fMWI0=WIf@0G0=WIf@00BP3IfMT00`000000fMWI\n0=WIf@0Y0=WIf@040000003IfMT0fMWI000002X0fMWI0`00000/0=WIf@800000:P3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3IfMT004P0fMWI00@000000=WIf@3I\nfMT00000:`3IfMT00`000000fMWI0000000[0=WIf@030000003IfMT0fMWI02X0fMWI00@000000=WI\nf@3IfMT00000:@3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00<000000=WIf@3IfMT05`3I\nfMT004T0fMWI0P00000]0=WIf@800000:`3IfMT3000002/0fMWI0P00000Y0=WIf@800000103IfMT2\n000001X0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000L0=WIf@030000003IfMT0fMWI0?l0fMWI\n0P3IfMT001`0fMWI00<000000=WIf@3IfMT0o`3IfMT20=WIf@00703IfMT00`000000fMWI0=WIf@3b\n0=WIf@8000003@3IfMT001H0fMWI1P0000000`000?l0000000000012000003/0003o2P00000_0000\noaH0000060000?l5000000P0fMWI00@000000=WIf@3IfMT00000303IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@03000000000?l0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WI\nf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@080=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WI\nf@3IfMT03@3IfMT001`0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000000fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000003IfMT0fMWI00X0fMWI1@00000<0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT08P3IfMT00`000000fMWI0=WIf@0=0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT0\n0`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3I\nfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@800000103IfMT2000000<0\nfMWI100000040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@030000003IfMT0fMWI00D0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI00080=WIf@040000003IfMT0fMWI000000T0fMWI00<000000=WIf@3IfMT0103IfMT00`000?l0\n00000=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0\nfMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI\n00@000000=WIf@3IfMT000002P3IfMT00`000000fMWI0=WIf@030=WIf@030000o`3IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI\n0=WIf@0000080=WIf@040000003IfMT0fMWI000000@0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@8000002P3IfMT2000000D0\nfMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0\nfMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380\nfMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI\n0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI\n0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0\n003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g\n0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D\n0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WI\nf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WI\nf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0\n@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0\n:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI\n00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000@0fMWI0P000005\n0=WIf@<000000`3IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o\n0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT000002`3IfMT00`000000fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00203IfMT010000000fMWI0=WIf@0000080=WIf@D000000`3IfMT00`000?l000000=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WI\nf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT010000000fMWI0=WIf@00\n00090=WIf@030000003IfMT0000000@0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI00090=WIf@8000002`3IfMT2000000@0fMWI00<0003o\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3I\nfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`00\n0?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480\nfMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0\nfMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0\n003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L\n0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b\n0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WI\nf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WI\nf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@03\n0000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@03\n0000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT0\n0`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT0\n0`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI\n0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI\n02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0\nfMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0\nfMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI\n00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI\n00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI\n000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WI\nf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o\n0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o\n0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WI\nf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WI\nf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000\no`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000\no`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P0000040=WIf@8000000`3I\nfMT3000000D0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000\n103IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@03\n0000o`000000fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n203IfMT010000000fMWI0=WIf@0000080=WIf@<000001@3IfMT00`000?l0fMWI0=WIf@120=WIf@03\n0000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@03\n0000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT0\n00002@3IfMT00`000000fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3I\nfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3I\nfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0`0000040=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT0\n0`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI\n00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI\n00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WI\nf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WI\nf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o\n0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o\n0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WI\nf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WI\nf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WI\nf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3I\nfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3I\nfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000\no`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000\no`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3I\nfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503I\nfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0\nfMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0\nfMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000T0fMWI0P0000040=WIf@800000103IfMT20000\n00D0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI00080=WIf@040000003IfMT0fMWI000000P0fMWI00@000000=WIf@3IfMT00000103IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n00P0fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000040=WIf@030000o`00\n0000fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00203IfMT0\n10000000fMWI0=WIf@0000090=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3I\nfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3I\nfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT000P0fMWI00@000000=WIf@3IfMT00000203I\nfMT010000000fMWI0=WIf@0000040=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0\n=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@0b0=WIf@002@3IfMT2000000X0fMWI0P0000050=WIf@030000o`3I\nfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT0\n0`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT0\n0`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT0\n01`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI\n00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI\n0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0\nfMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI\n00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI\n00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WI\nf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WI\nf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o\n0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o\n0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WI\nf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WI\nf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3I\nfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3I\nfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000\no`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000\no`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703I\nfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203I\nfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3I\nfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0\nfMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`00\n0?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`00\n0?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0\nfMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0\nfMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI\n0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI\n0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0\n003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0\n003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@12\n0=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[\n0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WI\nf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000L0=WIf@03\n0000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT00`000?l0fMWI0=WIf@080=WIf@03\n0000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@0b0=WIf@00\n703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0\n203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0\n<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0fMWI0=WIf@0g0=WIf@030000o`3I\nfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0380fMWI000L0=WIf@030000o`3IfMT0fMWI0480fMWI00<0003o0=WIf@3IfMT0=`3IfMT0\n0`000?l0fMWI0=WIf@080=WIf@030000o`3IfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0503IfMT0\n0`000?l0fMWI0=WIf@0b0=WIf@00703IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI\n03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001`0fMWI00<0003o0=WIf@3IfMT0@P3IfMT00`000?l0\nfMWI0=WIf@0g0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0:`3IfMT00`000?l0\nfMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0380fMWI000D0=WIf@D000000`3IfMT00`000?l0fMWI\n0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0003o0=WIf@3IfMT0203IfMT00`000?l0fMWI\n0=WIf@0[0=WIf@030000o`3IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0<P3IfMT001H0fMWI00<0\n00000=WIf@3IfMT00`3IfMT00`000?l0fMWI0=WIf@120=WIf@030000o`3IfMT0fMWI03L0fMWI00<0\n003o0=WIf@3IfMT0203IfMT00`000?l0fMWI0=WIf@0[0=WIf@80003o5@3IfMT00`000?l0fMWI0=WI\nf@0b0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIfDH0003o>@3IfMT<0000obd0fMWI60000?ld\n0=WIf@005P3IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0?l0fMWI0P3IfMT001H0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI000E0=WIf@800000\n1@3IfMT00`000000fMWI0=WIf@3o0=WIf@80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI0140fMWI0`00\n003T0=WIf@00903IfMT00`000000fMWI0=WIf@0C0=WIf@030000003IfMT0fMWI0><0fMWI000S0=WI\nf@030000003IfMT0fMWI0080fMWI0`0000040=WIf@<0000000P0fMWI000000000000fMWI0000003I\nfMT000000=WIf@<000000`3IfMT3000000@0fMWI0`0000000`3IfMT000000000000200000080fMWI\n1@0000030=WIf@8000000`3IfMT3000000<0fMWI0`0000040=WIf@<000000P3IfMT4000000<0fMWI\n0P0000040=WIf@<000000P3IfMT4000000<0fMWI0P0000040=WIf@<0000000<0fMWI000000000000\n0P3IfMT200000080fMWI0P0000230=WIf@008`3IfMT00`000000fMWI0=WIf@050=WIf@030000003I\nfMT0fMWI0080fMWI00/000000=WIf@3IfMT0fMWI0000003IfMT000000=WIf@000000fMWI00000002\n0=WIf@050000003IfMT0fMWI0=WIf@000000103IfMT00`000000fMWI0=WIf@030=WIf@040000003I\nfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WI\nf@040=WIf@030000003IfMT0fMWI00<0fMWI0P0000030=WIf@030000003IfMT0fMWI00@0fMWI00<0\n00000=WIf@3IfMT02P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT0\n0000Q@3IfMT001H0fMWI20000?l50=WIf@030000003IfMT0fMWI0080fMWI0`0000050=WIf@0;0000\n003IfMT0fMWI0=WIf@000000fMWI0000003IfMT000000=WIf@0000000P3IfMT01@000000fMWI0=WI\nf@3IfMT0000000@0fMWI100000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3I\nfMT00`3IfMT010000000fMWI0=WIf@0000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WI\nf@3IfMT0100000030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT4\n000000<0fMWI00<000000=WIf@3IfMT02P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@0\n00000=WIf@3IfMT00000Q@3IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT4000000<0fMWI0P00\n00020=WIf@<0000000<0fMWI0000003IfMT0100000040=WIf@030000003IfMT0fMWI00<0fMWI0P00\n00030=WIf@040000003IfMT0fMWI00000080fMWI0P0000040=WIf@040000003IfMT0fMWI000000<0\nfMWI0P0000040=WIf@800000103IfMT2000000<0fMWI100000000`3IfMT00000000000020=WIf@80\n00000P3IfMT2000000<0fMWI100000090=WIf@8000000P3IfMT400000080fMWI100000230=WIf@00\n903IfMT00`000000fMWI0=WIf@0J0=WIf@030000003IfMT0fMWI00P0fMWI00@000000=WIf@3IfMT0\n00000P3IfMT01P000000fMWI0=WIf@000000fMWI000000@0fMWI00<000000=WIf@0000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI0300fMWI00<000000=WIf@3IfMT0P`3I\nfMT002D0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@0=0=WIf@8000002@3IfMT4\n00000080fMWI1@0000000`3IfMT00000000000020=WIf@80000000<0fMWI0000000000000P000002\n0=WIf@@00000:03IfMT00`000000fMWI0=WIf@2=0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\no`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00\n"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-1.21768`, -0.155509`, 0.0434554`, 0.00703123`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[BoxData[TagBox[RowBox[List[RowBox[List["\[SkeletonIndicator]", "Graphics"]], "\[SkeletonIndicator]"]], False, Rule[Editable, False]]], "Output"], Cell[TextData[List[StyleBox["Diagram 8:", Rule[FontFamily, "Courier New"], Rule[FontWeight, "Bold"]], StyleBox[" - ", Rule[FontFamily, "Courier New"]], StyleBox["The output signal simpleDEVSServer.internalEvent.", Rule[FontFamily, "Courier"]]]], "Text"]], Open]]], Open]]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[839, 867]], Rule[WindowMargins, List[List[324, Automatic], List[100, Automatic]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]