List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Sampled Systems", "Title"], Cell[CellGroupData[List[Cell["Simple Periodic Sampler", "Section"], Cell[TextData[List["The following model is a ", StyleBox["simple periodic sampler", Rule[FontSlant, "Italic"]], " with a sampling period ", StyleBox["T", Rule[FontFamily, "Courier"]], " that is constant and defined as a parameter that can be changed by the model user. This model has only one kind of event, the sampling event. We use the built-in function ", StyleBox["sample", Rule[FontFamily, "Courier"]], " in the ", StyleBox["when", Rule[FontFamily, "Courier"]], "-condition ", StyleBox["sample(0, T)", Rule[FontFamily, "Courier"]], " to periodically generate the sampling events with a period time ", StyleBox["T", Rule[FontFamily, "Courier"]], ". This is a simple model using the state space equations.\n"]], "Text"], Cell["model SimplePeriodicSampler\n  parameter Real T = 1\t\t\t\t\"Sample period\";\n  input Real u\t\t\t\t\t\t\"Input used at sample events\";\n  discrete output Real y\t\t\t\"Output computed at sample events\";\n  replaceable function f = Modelica.Math.atan2;\n  replaceable function h = Modelica.Math.atan2;\nprotected\n  discrete Real x;\t\t\t\t\t// discrete state variable\nequation\n  when sample(0, T) then\n    x = f(pre(x), u);\t\t\t\t// state update expression\n    y = h(pre(x), u);\t\t\t\t// output expression\n  end when;\nend SimplePeriodicSampler;", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Base Class for Sampling Models", "Section"], Cell[TextData[List["It is possible to formulate a base class for scalar sampling models using state-space equations.The class ", StyleBox["BaseSampler", Rule[FontFamily, "Courier"]], " below can for example be specialized to either a periodic sampler or an aperiodic sampler.\n"]], "Text"], Cell["partial class BaseSampler\n  input Real u;\n  discrete output Real y;\n  Boolean doSample;\n  replaceable function f = Modelica.Math.atan2;\n  replaceable function h = Modelica.Math.atan2;\nprotected\n  discrete Real x;\nequation\n  when doSample then\n    x = f(pre(x), u);\t\t\t\t\t\t\t\t\t// state update expression\n    x = h(pre(x), u);\t\t\t\t\t\t\t\t\t// output expression\n    end when;\nend BaseSampler;", "ModelicaInput"], Cell[TextData[List["\nBelow the base model is extended to a periodic sampling model ", StyleBox["PeriodicSampler", Rule[FontFamily, "Courier"]], " with a fixed period time T.\n"]], "Text"], Cell["model PeriodicSampler\n  extend BaseSampler;\n  parameter Real T = 1\t\t\t\t\"sample period\";\nequation\n  doSample = sample(0, T);\nend PeriodicSampler;", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Aperiodic Sampler", "Section"], Cell[TextData[List["Besides the above periodic sampler, an aperiodic sampling model with a time-varying sample period can be obtained by specializing the base class. A variable ", StyleBox["nextSampling", Rule[FontFamily, "Courier"]], " is used for the next sampling time, which is current time ", StyleBox["+ periodTime", Rule[FontFamily, "Courier"]], " when the current time has reached the previous value of ", StyleBox["nextSampling", Rule[FontFamily, "Courier"]], ". In this example we increase the period time by the logarithm of the current time just to have a time varying period time.\n"]], "Text"], Cell["model APeriodicSampler\n  extend BaseSampler;\n  discrete Real nextSampling(start = 0);\n  discrete Real periodTime(start = 1)\t\t\"time-varying sample period\";\nequation\n  doSample = time > nextSampling;\n  when doSample then\n    nextSampling = pre(nextSampling) + periodTime;\n    periodTime = pre(periodTime) + log(pre(periodTime));\n  end when;\nend APeriodicSampler;", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Discrete Scalar State Space\[NonBreakingSpace]Sampling Model", "Section"], Cell["A discrete first order scalar state space model with periodic sampling can be formulated as follows, essentially being a special case of the above BaseSampler model even though we do not extend that class in this particular case.\n", "Text"], Cell["model DiscreteScalarStateSpace\n  parameter Real a, b, c, d;\n  parameter Real T = 1;\n  input Real u;\n  discrete output Real y;\nprotected\n  discrete Real x;\nequation\n  when sample(0, T) then\n    x = a * pre(x) + b * u;\n    y = c * pre(x) + d * u;\n  end when;\nend DiscreteScalarStateSpace;", "ModelicaInput"], Cell[TextData[List["\nAlternatively, the discrete scalar state space model can be expressed by extending the ", StyleBox["BaseSampler", Rule[FontFamily, "Courier"]], " model for periodic sampling and redeclaring the functions ", StyleBox["f", Rule[FontFamily, "Courier"]], " and ", StyleBox["g", Rule[FontFamily, "Courier"]], " as below. However, in this case the result is not shorter or more readable since the syntax for expressing the local functions ", StyleBox["f", Rule[FontFamily, "Courier"]], "1 and ", StyleBox["h1", Rule[FontFamily, "Courier"]], " is not very concise. On the other hand, if the functions ", StyleBox["f", Rule[FontFamily, "Courier"]], "1 and ", StyleBox["h1", Rule[FontFamily, "Courier"]], " instead would be used in several places, the modeling style based on replaceable functions would be preferable.\n"]], "Text"], Cell["model DiscreteScalarStateSpace2\n  extends BaseSampler(redeclare function f = f1, redeclare function h = h1);\n  parameter Real a, b, c, d;\n  parameter Real T = 1\t\t\t\t\"sample period\";\n  function f1\n    input Real x1;\n    input Real x2;\n    input Real y1;\n  algorithm\n    y1 := a * x1 + b * x2;\n  end f1;\n  function h1\n    input Real x1;\n    input Real x2;\n    input Real y1;\n  algorithm\n    y1 := c * x1 + d * x2;\n  end h1;\nequation\n  doSample = sample(0, T);\nend DiscreteScalarStateSpace2;", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Discrete Vector State Space Sampling Model", "Section"], Cell[TextData[List["Usually state space models are expressed using the more general vector form, as in the ", StyleBox["DiscreteVectorStateSpace", Rule[FontFamily, "Courier"]], " periodic sampling model below:"]], "Text"], Cell["model DiscreteVectorStateSpace\n  parameter Integer n = 5, m = 4, p = 2;\n  parameter Real A[n, n] = fill(1, n, n);\n  parameter Real B[n, m] = fill(2, n, m);\n  parameter Real C[p, n] = fill(3, p, n);\n  parameter Real D[p, m] = fill(4, p, m);\n  parameter Real T = 1;\n  input Real u[m];\n  discrete output Real y[p];\nprotected\n  discrete Real x[n];// = fill(2, n);\nequation\n  when sample(0, T) then\n    x = A * pre(x) + B * u;\n    y = C * pre(x) + D * u;\n  end when;\nend DiscreteVectorStateSpace;", "ModelicaInput"], Cell["", "Text"], Cell["model DVSSTest\n  DiscreteVectorStateSpace dvss;\nequation\n  dvss.u= fill(time,dvss.m);\nend DVSSTest;", "ModelicaInput"], Cell[CellGroupData[List[Cell["Simulation of DVSSTest", "Subsection"], Cell["simulate( DVSSTest, stopTime=10 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(DVSSTest\\) : 2004-1-30 15:54:1.1532098 : \\!\\({0.`, 10.`}\\) : 511 data points : 10 events : 75 variables>\\n\\!\\({\\*TagBox[\\\"\[UnderBracket]derdummy\\\", HoldForm], \\*TagBox[\\\"\[UnderBracket]dummy\\\", HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(1, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(1, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(1, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(1, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(1, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(2, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(2, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(2, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(2, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(2, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(3, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(3, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(3, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(3, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(3, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(4, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(4, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(4, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(4, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(4, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(5, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(5, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(5, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(5, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . A\\)\\) \[LeftDoubleBracket] \\(\\(5, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(1, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(1, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(1, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(1, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(2, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(2, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(2, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(2, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(3, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(3, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(3, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(3, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(4, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(4, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(4, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(4, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(5, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(5, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(5, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . B\\)\\) \[LeftDoubleBracket] \\(\\(5, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(1, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(1, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(1, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(1, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(1, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(2, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(2, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(2, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(2, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . C\\)\\) \[LeftDoubleBracket] \\(\\(2, 5\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(1, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(1, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(1, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(1, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(2, 1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(2, 2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(2, 3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . D\\)\\) \[LeftDoubleBracket] \\(\\(2, 4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(dvss . m\\), HoldForm], \\*TagBox[\\(dvss . n\\), HoldForm], \\*TagBox[\\(dvss . p\\), HoldForm], \\*TagBox[\\(dvss . T\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . u\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . u\\)\\) \[LeftDoubleBracket] \\(\\(2\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . u\\)\\) \[LeftDoubleBracket] \\(\\(3\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . u\\)\\) \[LeftDoubleBracket] \\(\\(4\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . y\\)\\) \[LeftDoubleBracket] \\(\\(1\\)\\) \[RightDoubleBracket]\\), HoldForm], \\*TagBox[\\(\\(\\(dvss . y\\)\\) \[LeftDoubleBracket] \\(\\(2\\)\\) \[RightDoubleBracket]\\), HoldForm]}\\)\""], "Output"]], Open]]], Open]], Cell[CellGroupData[List[Cell["Two Rate Sampling Model", "Section"], Cell[TextData[List["In the ", StyleBox["TwoRateSampler", Rule[FontFamily, "Courier"]], " model below the slow sampler is five times slower than the fast sampler, i.e. the ", StyleBox["slowSample", Rule[FontFamily, "Courier"]], " when-equation is evaluated at every 5th activation of the ", StyleBox["fastSample when", Rule[FontFamily, "Courier"]], "-equation.The two samplers are synchronized via equations involving the ", StyleBox["cyCounter", Rule[FontFamily, "Courier"]], " variable. \n"]], "Text"], Cell["model TwoRateSampler\n  discrete Real x,y;\n  Boolean fastSample;\n  Boolean slowSample;\n  Integer cyCounter(start=0);  \t\t\t\t// Cyclic count 0,1,2,3,4, 0,1,2,3,4,... \n equation \n  fastSample = sample(0,1);    \t\t\t// Define the fast clock\n  when fastSample then   \n    cyCounter  = if pre(cyCounter) < 5 then pre(cyCounter)+1 else 0;\n    slowSample = pre(cyCounter) == 0; \t    // Define the slow clock\n\n  end when;\n equation\n  when fastSample then      \t\t\t\t// fast sampling\n    x = sin(time); \t\t\t\t\t\t\t\n  end when;\n equation\n  when slowSample then    \t\t\t\t    // slow sampling (5-times slower)\n    y = log(time); \t\t\t\t\t\t\t\n  end when;\nend TwoRateSampler;", "ModelicaInput"], Cell[CellGroupData[List[Cell["Simulation of TwoRateSampler", "Subsection"], Cell["simulate( TwoRateSampler, stopTime=100 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(TwoRateSampler\\) : 2004-1-30 15:54:10.2936008 : \\!\\({0.`, 100.`}\\) : 601 data points : 100 events : 7 variables>\\n\\!\\({\\*TagBox[\\\"cyCounter\\\", HoldForm], \\*TagBox[\\\"\[UnderBracket]derdummy\\\", HoldForm], \\*TagBox[\\\"\[UnderBracket]dummy\\\", HoldForm], \\*TagBox[\\\"fastSample\\\", HoldForm], \\*TagBox[\\\"slowSample\\\", HoldForm], \\*TagBox[\\\"x\\\", HoldForm], \\*TagBox[\\\"y\\\", HoldForm]}\\)\""], "Output"], Cell["plot( { x, y }, xrange={7,100} );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n-0.0478751 0.0102407 0.120496 0.105781 [\n[.15694 .108 -6 -9 ]\n[.15694 .108 6 0 ]\n[.36175 .108 -6 -9 ]\n[.36175 .108 6 0 ]\n[.56656 .108 -6 -9 ]\n[.56656 .108 6 0 ]\n[.77138 .108 -6 -9 ]\n[.77138 .108 6 0 ]\n[.97619 .108 -9 -9 ]\n[.97619 .108 9 0 ]\n[1.025 .1205 0 -6.28125 ]\n[1.025 .1205 10 6.28125 ]\n[-0.00917 .01471 -12 -4.5 ]\n[-0.00917 .01471 0 4.5 ]\n[-0.00917 .22628 -6 -4.5 ]\n[-0.00917 .22628 0 4.5 ]\n[-0.00917 .33206 -6 -4.5 ]\n[-0.00917 .33206 0 4.5 ]\n[-0.00917 .43784 -6 -4.5 ]\n[-0.00917 .43784 0 4.5 ]\n[-0.00917 .54362 -6 -4.5 ]\n[-0.00917 .54362 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.15694 .1205 m\n.15694 .12675 L\ns\n[(20)] .15694 .108 0 1 Mshowa\n.36175 .1205 m\n.36175 .12675 L\ns\n[(40)] .36175 .108 0 1 Mshowa\n.56656 .1205 m\n.56656 .12675 L\ns\n[(60)] .56656 .108 0 1 Mshowa\n.77138 .1205 m\n.77138 .12675 L\ns\n[(80)] .77138 .108 0 1 Mshowa\n.97619 .1205 m\n.97619 .12675 L\ns\n[(100)] .97619 .108 0 1 Mshowa\n.125 Mabswid\n.20814 .1205 m\n.20814 .12425 L\ns\n.25934 .1205 m\n.25934 .12425 L\ns\n.31055 .1205 m\n.31055 .12425 L\ns\n.41295 .1205 m\n.41295 .12425 L\ns\n.46416 .1205 m\n.46416 .12425 L\ns\n.51536 .1205 m\n.51536 .12425 L\ns\n.61777 .1205 m\n.61777 .12425 L\ns\n.66897 .1205 m\n.66897 .12425 L\ns\n.72017 .1205 m\n.72017 .12425 L\ns\n.82258 .1205 m\n.82258 .12425 L\ns\n.87378 .1205 m\n.87378 .12425 L\ns\n.92499 .1205 m\n.92499 .12425 L\ns\n.10573 .1205 m\n.10573 .12425 L\ns\n.05453 .1205 m\n.05453 .12425 L\ns\n.25 Mabswid\n0 .1205 m\n1 .1205 L\ns\ngsave\n1.025 .1205 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.00333 .01471 m\n.00958 .01471 L\ns\n[(-1)] -0.00917 .01471 1 0 Mshowa\n.00333 .22628 m\n.00958 .22628 L\ns\n[(1)] -0.00917 .22628 1 0 Mshowa\n.00333 .33206 m\n.00958 .33206 L\ns\n[(2)] -0.00917 .33206 1 0 Mshowa\n.00333 .43784 m\n.00958 .43784 L\ns\n[(3)] -0.00917 .43784 1 0 Mshowa\n.00333 .54362 m\n.00958 .54362 L\ns\n[(4)] -0.00917 .54362 1 0 Mshowa\n.125 Mabswid\n.00333 .03587 m\n.00708 .03587 L\ns\n.00333 .05703 m\n.00708 .05703 L\ns\n.00333 .07818 m\n.00708 .07818 L\ns\n.00333 .09934 m\n.00708 .09934 L\ns\n.00333 .14165 m\n.00708 .14165 L\ns\n.00333 .16281 m\n.00708 .16281 L\ns\n.00333 .18396 m\n.00708 .18396 L\ns\n.00333 .20512 m\n.00708 .20512 L\ns\n.00333 .24743 m\n.00708 .24743 L\ns\n.00333 .26859 m\n.00708 .26859 L\ns\n.00333 .28975 m\n.00708 .28975 L\ns\n.00333 .3109 m\n.00708 .3109 L\ns\n.00333 .35321 m\n.00708 .35321 L\ns\n.00333 .37437 m\n.00708 .37437 L\ns\n.00333 .39553 m\n.00708 .39553 L\ns\n.00333 .41668 m\n.00708 .41668 L\ns\n.00333 .459 m\n.00708 .459 L\ns\n.00333 .48015 m\n.00708 .48015 L\ns\n.00333 .50131 m\n.00708 .50131 L\ns\n.00333 .52247 m\n.00708 .52247 L\ns\n.00333 .56478 m\n.00708 .56478 L\ns\n.00333 .58593 m\n.00708 .58593 L\ns\n.00333 .60709 m\n.00708 .60709 L\ns\n.25 Mabswid\n.00333 0 m\n.00333 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .09094 m\n.02381 .18999 L\n.02586 .18999 L\n.02791 .18999 L\n.02995 .18999 L\n.032 .18999 L\n.03405 .18999 L\n.03405 .22515 L\n.0361 .22515 L\n.03815 .22515 L\n.04019 .22515 L\n.04224 .22515 L\n.04429 .22515 L\n.04429 .16409 L\n.04634 .16409 L\n.04839 .16409 L\n.05044 .16409 L\n.05248 .16409 L\n.05453 .16409 L\n.05453 .06295 L\n.05658 .06295 L\n.05863 .06295 L\n.06068 .06295 L\n.06272 .06295 L\n.06477 .06295 L\n.06477 .01472 L\n.06682 .01472 L\n.06887 .01472 L\n.07092 .01472 L\n.07296 .01472 L\n.07501 .01472 L\n.07501 .06374 L\n.07706 .06374 L\n.07911 .06374 L\n.08116 .06374 L\n.08321 .06374 L\n.08525 .06374 L\n.08525 .16494 L\n.0873 .16494 L\n.08935 .16494 L\n.0914 .16494 L\n.09345 .16494 L\n.09549 .16494 L\n.09549 .22528 L\n.09754 .22528 L\n.09959 .22528 L\n.10164 .22528 L\n.10369 .22528 L\n.10573 .22528 L\n.10573 .18928 L\nMistroke\n.10778 .18928 L\n.10983 .18928 L\n.11188 .18928 L\n.11393 .18928 L\n.11598 .18928 L\n.11598 .09004 L\n.11802 .09004 L\n.12007 .09004 L\n.12212 .09004 L\n.12417 .09004 L\n.12622 .09004 L\n.12622 .0188 L\n.12826 .0188 L\n.13031 .0188 L\n.13236 .0188 L\n.13441 .0188 L\n.13646 .0188 L\n.13646 .04105 L\n.1385 .04105 L\n.14055 .04105 L\n.1426 .04105 L\n.14465 .04105 L\n.1467 .04105 L\n.1467 .13635 L\n.14875 .13635 L\n.15079 .13635 L\n.15284 .13635 L\n.15489 .13635 L\n.15694 .13635 L\n.15694 .21707 L\n.15899 .21707 L\n.16103 .21707 L\n.16308 .21707 L\n.16513 .21707 L\n.16718 .21707 L\n.16718 .209 L\n.16923 .209 L\n.17127 .209 L\n.17332 .209 L\n.17537 .209 L\n.17742 .209 L\n.17742 .11956 L\n.17947 .11956 L\n.18152 .11956 L\n.18356 .11956 L\n.18561 .11956 L\n.18766 .11956 L\n.18766 .03098 L\n.18971 .03098 L\n.19176 .03098 L\nMistroke\n.1938 .03098 L\n.19585 .03098 L\n.1979 .03098 L\n.1979 .0247 L\n.19995 .0247 L\n.202 .0247 L\n.20405 .0247 L\n.20609 .0247 L\n.20814 .0247 L\n.20814 .1065 L\n.21019 .1065 L\n.21224 .1065 L\n.21429 .1065 L\n.21633 .1065 L\n.21838 .1065 L\n.21838 .20116 L\n.22043 .20116 L\n.22248 .20116 L\n.22453 .20116 L\n.22657 .20116 L\n.22862 .20116 L\n.22862 .22166 L\n.23067 .22166 L\n.23272 .22166 L\n.23477 .22166 L\n.23682 .22166 L\n.23886 .22166 L\n.23886 .14915 L\n.24091 .14915 L\n.24296 .14915 L\n.24501 .14915 L\n.24706 .14915 L\n.2491 .14915 L\n.2491 .0503 L\n.25115 .0503 L\n.2532 .0503 L\n.25525 .0503 L\n.2573 .0503 L\n.25934 .0503 L\n.25934 .01598 L\n.26139 .01598 L\n.26344 .01598 L\n.26549 .01598 L\n.26754 .01598 L\n.26959 .01598 L\n.26959 .07776 L\n.27163 .07776 L\n.27368 .07776 L\n.27573 .07776 L\n.27778 .07776 L\nMistroke\n.27983 .07776 L\n.27983 .17883 L\n.28187 .17883 L\n.28392 .17883 L\n.28597 .17883 L\n.28802 .17883 L\n.29007 .17883 L\n.29007 .22627 L\n.29211 .22627 L\n.29416 .22627 L\n.29621 .22627 L\n.29826 .22627 L\n.30031 .22627 L\n.30031 .17646 L\n.30236 .17646 L\n.3044 .17646 L\n.30645 .17646 L\n.3085 .17646 L\n.31055 .17646 L\n.31055 .0752 L\n.3126 .0752 L\n.31464 .0752 L\n.31669 .0752 L\n.31874 .0752 L\n.32079 .0752 L\n.32079 .01558 L\n.32284 .01558 L\n.32488 .01558 L\n.32693 .01558 L\n.32898 .01558 L\n.33103 .01558 L\n.33103 .05242 L\n.33308 .05242 L\n.33513 .05242 L\n.33717 .05242 L\n.33922 .05242 L\n.34127 .05242 L\n.34127 .15185 L\n.34332 .15185 L\n.34537 .15185 L\n.34741 .15185 L\n.34946 .15185 L\n.35151 .15185 L\n.35151 .22245 L\n.35356 .22245 L\n.35561 .22245 L\n.35765 .22245 L\n.3597 .22245 L\n.36175 .22245 L\n.36175 .19931 L\nMistroke\n.3638 .19931 L\n.36585 .19931 L\n.3679 .19931 L\n.36994 .19931 L\n.37199 .19931 L\n.37199 .10372 L\n.37404 .10372 L\n.37609 .10372 L\n.37814 .10372 L\n.38018 .10372 L\n.38223 .10372 L\n.38223 .02354 L\n.38428 .02354 L\n.38633 .02354 L\n.38838 .02354 L\n.39042 .02354 L\n.39247 .02354 L\n.39247 .03251 L\n.39452 .03251 L\n.39657 .03251 L\n.39862 .03251 L\n.40067 .03251 L\n.40271 .03251 L\n.40271 .12237 L\n.40476 .12237 L\n.40681 .12237 L\n.40886 .12237 L\n.41091 .12237 L\n.41295 .12237 L\n.41295 .21051 L\n.415 .21051 L\n.41705 .21051 L\n.4191 .21051 L\n.42115 .21051 L\n.4232 .21051 L\n.4232 .21589 L\n.42524 .21589 L\n.42729 .21589 L\n.42934 .21589 L\n.43139 .21589 L\n.43344 .21589 L\n.43344 .13357 L\n.43548 .13357 L\n.43753 .13357 L\n.43958 .13357 L\n.44163 .13357 L\n.44368 .13357 L\n.44368 .03923 L\n.44572 .03923 L\n.44777 .03923 L\nMistroke\n.44982 .03923 L\n.45187 .03923 L\n.45392 .03923 L\n.45392 .01961 L\n.45597 .01961 L\n.45801 .01961 L\n.46006 .01961 L\n.46211 .01961 L\n.46416 .01961 L\n.46416 .09274 L\n.46621 .09274 L\n.46825 .09274 L\n.4703 .09274 L\n.47235 .09274 L\n.4744 .09274 L\n.4744 .19139 L\n.47645 .19139 L\n.47849 .19139 L\n.48054 .19139 L\n.48259 .19139 L\n.48464 .19139 L\n.48464 .22486 L\n.48669 .22486 L\n.48874 .22486 L\n.49078 .22486 L\n.49283 .22486 L\n.49488 .22486 L\n.49488 .16238 L\n.49693 .16238 L\n.49898 .16238 L\n.50102 .16238 L\n.50307 .16238 L\n.50512 .16238 L\n.50512 .06139 L\n.50717 .06139 L\n.50922 .06139 L\n.51126 .06139 L\n.51331 .06139 L\n.51536 .06139 L\n.51536 .01474 L\n.51741 .01474 L\n.51946 .01474 L\n.52151 .01474 L\n.52355 .01474 L\n.5256 .01474 L\n.5256 .06533 L\n.52765 .06533 L\n.5297 .06533 L\n.53175 .06533 L\n.53379 .06533 L\nMistroke\n.53584 .06533 L\n.53584 .16663 L\n.53789 .16663 L\n.53994 .16663 L\n.54199 .16663 L\n.54403 .16663 L\n.54608 .16663 L\n.54608 .22552 L\n.54813 .22552 L\n.55018 .22552 L\n.55223 .22552 L\n.55428 .22552 L\n.55632 .22552 L\n.55632 .18785 L\n.55837 .18785 L\n.56042 .18785 L\n.56247 .18785 L\n.56452 .18785 L\n.56656 .18785 L\n.56656 .08825 L\n.56861 .08825 L\n.57066 .08825 L\n.57271 .08825 L\n.57476 .08825 L\n.5768 .08825 L\n.5768 .0183 L\n.57885 .0183 L\n.5809 .0183 L\n.58295 .0183 L\n.585 .0183 L\n.58705 .0183 L\n.58705 .0423 L\n.58909 .0423 L\n.59114 .0423 L\n.59319 .0423 L\n.59524 .0423 L\n.59729 .0423 L\n.59729 .1382 L\n.59933 .1382 L\n.60138 .1382 L\n.60343 .1382 L\n.60548 .1382 L\n.60753 .1382 L\n.60753 .21782 L\n.60957 .21782 L\n.61162 .21782 L\n.61367 .21782 L\n.61572 .21782 L\n.61777 .21782 L\n.61777 .20796 L\nMistroke\n.61982 .20796 L\n.62186 .20796 L\n.62391 .20796 L\n.62596 .20796 L\n.62801 .20796 L\n.62801 .11769 L\n.63006 .11769 L\n.6321 .11769 L\n.63415 .11769 L\n.6362 .11769 L\n.63825 .11769 L\n.63825 .03 L\n.6403 .03 L\n.64235 .03 L\n.64439 .03 L\n.64644 .03 L\n.64849 .03 L\n.64849 .02551 L\n.65054 .02551 L\n.65259 .02551 L\n.65463 .02551 L\n.65668 .02551 L\n.65873 .02551 L\n.65873 .10835 L\n.66078 .10835 L\n.66283 .10835 L\n.66487 .10835 L\n.66692 .10835 L\n.66897 .10835 L\n.66897 .20236 L\n.67102 .20236 L\n.67307 .20236 L\n.67512 .20236 L\n.67716 .20236 L\n.67921 .20236 L\n.67921 .2211 L\n.68126 .2211 L\n.68331 .2211 L\n.68536 .2211 L\n.6874 .2211 L\n.68945 .2211 L\n.68945 .14735 L\n.6915 .14735 L\n.69355 .14735 L\n.6956 .14735 L\n.69764 .14735 L\n.69969 .14735 L\n.69969 .04891 L\n.70174 .04891 L\n.70379 .04891 L\nMistroke\n.70584 .04891 L\n.70789 .04891 L\n.70993 .04891 L\n.70993 .01629 L\n.71198 .01629 L\n.71403 .01629 L\n.71608 .01629 L\n.71813 .01629 L\n.72017 .01629 L\n.72017 .07948 L\n.72222 .07948 L\n.72427 .07948 L\n.72632 .07948 L\n.72837 .07948 L\n.73041 .07948 L\n.73041 .18038 L\n.73246 .18038 L\n.73451 .18038 L\n.73656 .18038 L\n.73861 .18038 L\n.74066 .18038 L\n.74066 .22623 L\n.7427 .22623 L\n.74475 .22623 L\n.7468 .22623 L\n.74885 .22623 L\n.7509 .22623 L\n.7509 .17486 L\n.75294 .17486 L\n.75499 .17486 L\n.75704 .17486 L\n.75909 .17486 L\n.76114 .17486 L\n.76114 .07352 L\n.76318 .07352 L\n.76523 .07352 L\n.76728 .07352 L\n.76933 .07352 L\n.77138 .07352 L\n.77138 .01536 L\n.77343 .01536 L\n.77547 .01536 L\n.77752 .01536 L\n.77957 .01536 L\n.78162 .01536 L\n.78162 .05387 L\n.78367 .05387 L\n.78571 .05387 L\n.78776 .05387 L\n.78981 .05387 L\nMistroke\n.79186 .05387 L\n.79186 .15363 L\n.79391 .15363 L\n.79595 .15363 L\n.798 .15363 L\n.80005 .15363 L\n.8021 .15363 L\n.8021 .22293 L\n.80415 .22293 L\n.8062 .22293 L\n.80824 .22293 L\n.81029 .22293 L\n.81234 .22293 L\n.81234 .19805 L\n.81439 .19805 L\n.81644 .19805 L\n.81848 .19805 L\n.82053 .19805 L\n.82258 .19805 L\n.82258 .10187 L\n.82463 .10187 L\n.82668 .10187 L\n.82873 .10187 L\n.83077 .10187 L\n.83282 .10187 L\n.83282 .02281 L\n.83487 .02281 L\n.83692 .02281 L\n.83897 .02281 L\n.84101 .02281 L\n.84306 .02281 L\n.84306 .03356 L\n.84511 .03356 L\n.84716 .03356 L\n.84921 .03356 L\n.85125 .03356 L\n.8533 .03356 L\n.8533 .12424 L\n.85535 .12424 L\n.8574 .12424 L\n.85945 .12424 L\n.8615 .12424 L\n.86354 .12424 L\n.86354 .21147 L\n.86559 .21147 L\n.86764 .21147 L\n.86969 .21147 L\n.87174 .21147 L\n.87378 .21147 L\n.87378 .21506 L\nMistroke\n.87583 .21506 L\n.87788 .21506 L\n.87993 .21506 L\n.88198 .21506 L\n.88402 .21506 L\n.88402 .13171 L\n.88607 .13171 L\n.88812 .13171 L\n.89017 .13171 L\n.89222 .13171 L\n.89427 .13171 L\n.89427 .03804 L\n.89631 .03804 L\n.89836 .03804 L\n.90041 .03804 L\n.90246 .03804 L\n.90451 .03804 L\n.90451 .02018 L\n.90655 .02018 L\n.9086 .02018 L\n.91065 .02018 L\n.9127 .02018 L\n.91475 .02018 L\n.91475 .09455 L\n.91679 .09455 L\n.91884 .09455 L\n.92089 .09455 L\n.92294 .09455 L\n.92499 .09455 L\n.92499 .19277 L\n.92704 .19277 L\n.92908 .19277 L\n.93113 .19277 L\n.93318 .19277 L\n.93523 .19277 L\n.93523 .22454 L\n.93728 .22454 L\n.93932 .22454 L\n.94137 .22454 L\n.94342 .22454 L\n.94547 .22454 L\n.94547 .16065 L\n.94752 .16065 L\n.94956 .16065 L\n.95161 .16065 L\n.95366 .16065 L\n.95571 .16065 L\n.95571 .05984 L\n.95776 .05984 L\n.95981 .05984 L\nMistroke\n.96185 .05984 L\n.9639 .05984 L\n.96595 .05984 L\n.96595 .0148 L\n.968 .0148 L\n.97005 .0148 L\n.97209 .0148 L\n.97414 .0148 L\n.97619 .0148 L\nMfstroke\n0 .5 0 r\n.02381 .31003 m\n.02381 .31003 L\n.02586 .31003 L\n.02791 .31003 L\n.02995 .31003 L\n.032 .31003 L\n.03405 .31003 L\n.03405 .31003 L\n.0361 .31003 L\n.03815 .31003 L\n.04019 .31003 L\n.04224 .31003 L\n.04429 .31003 L\n.04429 .31003 L\n.04634 .31003 L\n.04839 .31003 L\n.05044 .31003 L\n.05248 .31003 L\n.05453 .31003 L\n.05453 .31003 L\n.05658 .31003 L\n.05863 .31003 L\n.06068 .31003 L\n.06272 .31003 L\n.06477 .31003 L\n.06477 .31003 L\n.06682 .31003 L\n.06887 .31003 L\n.07092 .31003 L\n.07296 .31003 L\n.07501 .31003 L\n.07501 .38335 L\n.07706 .38335 L\n.07911 .38335 L\n.08116 .38335 L\n.08321 .38335 L\n.08525 .38335 L\n.08525 .38335 L\n.0873 .38335 L\n.08935 .38335 L\n.0914 .38335 L\n.09345 .38335 L\n.09549 .38335 L\n.09549 .38335 L\n.09754 .38335 L\n.09959 .38335 L\n.10164 .38335 L\n.10369 .38335 L\n.10573 .38335 L\n.10573 .38335 L\nMistroke\n.10778 .38335 L\n.10983 .38335 L\n.11188 .38335 L\n.11393 .38335 L\n.11598 .38335 L\n.11598 .38335 L\n.11802 .38335 L\n.12007 .38335 L\n.12212 .38335 L\n.12417 .38335 L\n.12622 .38335 L\n.12622 .38335 L\n.12826 .38335 L\n.13031 .38335 L\n.13236 .38335 L\n.13441 .38335 L\n.13646 .38335 L\n.13646 .42624 L\n.1385 .42624 L\n.14055 .42624 L\n.1426 .42624 L\n.14465 .42624 L\n.1467 .42624 L\n.1467 .42624 L\n.14875 .42624 L\n.15079 .42624 L\n.15284 .42624 L\n.15489 .42624 L\n.15694 .42624 L\n.15694 .42624 L\n.15899 .42624 L\n.16103 .42624 L\n.16308 .42624 L\n.16513 .42624 L\n.16718 .42624 L\n.16718 .42624 L\n.16923 .42624 L\n.17127 .42624 L\n.17332 .42624 L\n.17537 .42624 L\n.17742 .42624 L\n.17742 .42624 L\n.17947 .42624 L\n.18152 .42624 L\n.18356 .42624 L\n.18561 .42624 L\n.18766 .42624 L\n.18766 .42624 L\n.18971 .42624 L\n.19176 .42624 L\nMistroke\n.1938 .42624 L\n.19585 .42624 L\n.1979 .42624 L\n.1979 .45667 L\n.19995 .45667 L\n.202 .45667 L\n.20405 .45667 L\n.20609 .45667 L\n.20814 .45667 L\n.20814 .45667 L\n.21019 .45667 L\n.21224 .45667 L\n.21429 .45667 L\n.21633 .45667 L\n.21838 .45667 L\n.21838 .45667 L\n.22043 .45667 L\n.22248 .45667 L\n.22453 .45667 L\n.22657 .45667 L\n.22862 .45667 L\n.22862 .45667 L\n.23067 .45667 L\n.23272 .45667 L\n.23477 .45667 L\n.23682 .45667 L\n.23886 .45667 L\n.23886 .45667 L\n.24091 .45667 L\n.24296 .45667 L\n.24501 .45667 L\n.24706 .45667 L\n.2491 .45667 L\n.2491 .45667 L\n.25115 .45667 L\n.2532 .45667 L\n.25525 .45667 L\n.2573 .45667 L\n.25934 .45667 L\n.25934 .48028 L\n.26139 .48028 L\n.26344 .48028 L\n.26549 .48028 L\n.26754 .48028 L\n.26959 .48028 L\n.26959 .48028 L\n.27163 .48028 L\n.27368 .48028 L\n.27573 .48028 L\n.27778 .48028 L\nMistroke\n.27983 .48028 L\n.27983 .48028 L\n.28187 .48028 L\n.28392 .48028 L\n.28597 .48028 L\n.28802 .48028 L\n.29007 .48028 L\n.29007 .48028 L\n.29211 .48028 L\n.29416 .48028 L\n.29621 .48028 L\n.29826 .48028 L\n.30031 .48028 L\n.30031 .48028 L\n.30236 .48028 L\n.3044 .48028 L\n.30645 .48028 L\n.3085 .48028 L\n.31055 .48028 L\n.31055 .48028 L\n.3126 .48028 L\n.31464 .48028 L\n.31669 .48028 L\n.31874 .48028 L\n.32079 .48028 L\n.32079 .49957 L\n.32284 .49957 L\n.32488 .49957 L\n.32693 .49957 L\n.32898 .49957 L\n.33103 .49957 L\n.33103 .49957 L\n.33308 .49957 L\n.33513 .49957 L\n.33717 .49957 L\n.33922 .49957 L\n.34127 .49957 L\n.34127 .49957 L\n.34332 .49957 L\n.34537 .49957 L\n.34741 .49957 L\n.34946 .49957 L\n.35151 .49957 L\n.35151 .49957 L\n.35356 .49957 L\n.35561 .49957 L\n.35765 .49957 L\n.3597 .49957 L\n.36175 .49957 L\n.36175 .49957 L\nMistroke\n.3638 .49957 L\n.36585 .49957 L\n.3679 .49957 L\n.36994 .49957 L\n.37199 .49957 L\n.37199 .49957 L\n.37404 .49957 L\n.37609 .49957 L\n.37814 .49957 L\n.38018 .49957 L\n.38223 .49957 L\n.38223 .51587 L\n.38428 .51587 L\n.38633 .51587 L\n.38838 .51587 L\n.39042 .51587 L\n.39247 .51587 L\n.39247 .51587 L\n.39452 .51587 L\n.39657 .51587 L\n.39862 .51587 L\n.40067 .51587 L\n.40271 .51587 L\n.40271 .51587 L\n.40476 .51587 L\n.40681 .51587 L\n.40886 .51587 L\n.41091 .51587 L\n.41295 .51587 L\n.41295 .51587 L\n.415 .51587 L\n.41705 .51587 L\n.4191 .51587 L\n.42115 .51587 L\n.4232 .51587 L\n.4232 .51587 L\n.42524 .51587 L\n.42729 .51587 L\n.42934 .51587 L\n.43139 .51587 L\n.43344 .51587 L\n.43344 .51587 L\n.43548 .51587 L\n.43753 .51587 L\n.43958 .51587 L\n.44163 .51587 L\n.44368 .51587 L\n.44368 .53 L\n.44572 .53 L\n.44777 .53 L\nMistroke\n.44982 .53 L\n.45187 .53 L\n.45392 .53 L\n.45392 .53 L\n.45597 .53 L\n.45801 .53 L\n.46006 .53 L\n.46211 .53 L\n.46416 .53 L\n.46416 .53 L\n.46621 .53 L\n.46825 .53 L\n.4703 .53 L\n.47235 .53 L\n.4744 .53 L\n.4744 .53 L\n.47645 .53 L\n.47849 .53 L\n.48054 .53 L\n.48259 .53 L\n.48464 .53 L\n.48464 .53 L\n.48669 .53 L\n.48874 .53 L\n.49078 .53 L\n.49283 .53 L\n.49488 .53 L\n.49488 .53 L\n.49693 .53 L\n.49898 .53 L\n.50102 .53 L\n.50307 .53 L\n.50512 .53 L\n.50512 .54246 L\n.50717 .54246 L\n.50922 .54246 L\n.51126 .54246 L\n.51331 .54246 L\n.51536 .54246 L\n.51536 .54246 L\n.51741 .54246 L\n.51946 .54246 L\n.52151 .54246 L\n.52355 .54246 L\n.5256 .54246 L\n.5256 .54246 L\n.52765 .54246 L\n.5297 .54246 L\n.53175 .54246 L\n.53379 .54246 L\nMistroke\n.53584 .54246 L\n.53584 .54246 L\n.53789 .54246 L\n.53994 .54246 L\n.54199 .54246 L\n.54403 .54246 L\n.54608 .54246 L\n.54608 .54246 L\n.54813 .54246 L\n.55018 .54246 L\n.55223 .54246 L\n.55428 .54246 L\n.55632 .54246 L\n.55632 .54246 L\n.55837 .54246 L\n.56042 .54246 L\n.56247 .54246 L\n.56452 .54246 L\n.56656 .54246 L\n.56656 .5536 L\n.56861 .5536 L\n.57066 .5536 L\n.57271 .5536 L\n.57476 .5536 L\n.5768 .5536 L\n.5768 .5536 L\n.57885 .5536 L\n.5809 .5536 L\n.58295 .5536 L\n.585 .5536 L\n.58705 .5536 L\n.58705 .5536 L\n.58909 .5536 L\n.59114 .5536 L\n.59319 .5536 L\n.59524 .5536 L\n.59729 .5536 L\n.59729 .5536 L\n.59933 .5536 L\n.60138 .5536 L\n.60343 .5536 L\n.60548 .5536 L\n.60753 .5536 L\n.60753 .5536 L\n.60957 .5536 L\n.61162 .5536 L\n.61367 .5536 L\n.61572 .5536 L\n.61777 .5536 L\n.61777 .5536 L\nMistroke\n.61982 .5536 L\n.62186 .5536 L\n.62391 .5536 L\n.62596 .5536 L\n.62801 .5536 L\n.62801 .56368 L\n.63006 .56368 L\n.6321 .56368 L\n.63415 .56368 L\n.6362 .56368 L\n.63825 .56368 L\n.63825 .56368 L\n.6403 .56368 L\n.64235 .56368 L\n.64439 .56368 L\n.64644 .56368 L\n.64849 .56368 L\n.64849 .56368 L\n.65054 .56368 L\n.65259 .56368 L\n.65463 .56368 L\n.65668 .56368 L\n.65873 .56368 L\n.65873 .56368 L\n.66078 .56368 L\n.66283 .56368 L\n.66487 .56368 L\n.66692 .56368 L\n.66897 .56368 L\n.66897 .56368 L\n.67102 .56368 L\n.67307 .56368 L\n.67512 .56368 L\n.67716 .56368 L\n.67921 .56368 L\n.67921 .56368 L\n.68126 .56368 L\n.68331 .56368 L\n.68536 .56368 L\n.6874 .56368 L\n.68945 .56368 L\n.68945 .57289 L\n.6915 .57289 L\n.69355 .57289 L\n.6956 .57289 L\n.69764 .57289 L\n.69969 .57289 L\n.69969 .57289 L\n.70174 .57289 L\n.70379 .57289 L\nMistroke\n.70584 .57289 L\n.70789 .57289 L\n.70993 .57289 L\n.70993 .57289 L\n.71198 .57289 L\n.71403 .57289 L\n.71608 .57289 L\n.71813 .57289 L\n.72017 .57289 L\n.72017 .57289 L\n.72222 .57289 L\n.72427 .57289 L\n.72632 .57289 L\n.72837 .57289 L\n.73041 .57289 L\n.73041 .57289 L\n.73246 .57289 L\n.73451 .57289 L\n.73656 .57289 L\n.73861 .57289 L\n.74066 .57289 L\n.74066 .57289 L\n.7427 .57289 L\n.74475 .57289 L\n.7468 .57289 L\n.74885 .57289 L\n.7509 .57289 L\n.7509 .58135 L\n.75294 .58135 L\n.75499 .58135 L\n.75704 .58135 L\n.75909 .58135 L\n.76114 .58135 L\n.76114 .58135 L\n.76318 .58135 L\n.76523 .58135 L\n.76728 .58135 L\n.76933 .58135 L\n.77138 .58135 L\n.77138 .58135 L\n.77343 .58135 L\n.77547 .58135 L\n.77752 .58135 L\n.77957 .58135 L\n.78162 .58135 L\n.78162 .58135 L\n.78367 .58135 L\n.78571 .58135 L\n.78776 .58135 L\n.78981 .58135 L\nMistroke\n.79186 .58135 L\n.79186 .58135 L\n.79391 .58135 L\n.79595 .58135 L\n.798 .58135 L\n.80005 .58135 L\n.8021 .58135 L\n.8021 .58135 L\n.80415 .58135 L\n.8062 .58135 L\n.80824 .58135 L\n.81029 .58135 L\n.81234 .58135 L\n.81234 .58919 L\n.81439 .58919 L\n.81644 .58919 L\n.81848 .58919 L\n.82053 .58919 L\n.82258 .58919 L\n.82258 .58919 L\n.82463 .58919 L\n.82668 .58919 L\n.82873 .58919 L\n.83077 .58919 L\n.83282 .58919 L\n.83282 .58919 L\n.83487 .58919 L\n.83692 .58919 L\n.83897 .58919 L\n.84101 .58919 L\n.84306 .58919 L\n.84306 .58919 L\n.84511 .58919 L\n.84716 .58919 L\n.84921 .58919 L\n.85125 .58919 L\n.8533 .58919 L\n.8533 .58919 L\n.85535 .58919 L\n.8574 .58919 L\n.85945 .58919 L\n.8615 .58919 L\n.86354 .58919 L\n.86354 .58919 L\n.86559 .58919 L\n.86764 .58919 L\n.86969 .58919 L\n.87174 .58919 L\n.87378 .58919 L\n.87378 .59649 L\nMistroke\n.87583 .59649 L\n.87788 .59649 L\n.87993 .59649 L\n.88198 .59649 L\n.88402 .59649 L\n.88402 .59649 L\n.88607 .59649 L\n.88812 .59649 L\n.89017 .59649 L\n.89222 .59649 L\n.89427 .59649 L\n.89427 .59649 L\n.89631 .59649 L\n.89836 .59649 L\n.90041 .59649 L\n.90246 .59649 L\n.90451 .59649 L\n.90451 .59649 L\n.90655 .59649 L\n.9086 .59649 L\n.91065 .59649 L\n.9127 .59649 L\n.91475 .59649 L\n.91475 .59649 L\n.91679 .59649 L\n.91884 .59649 L\n.92089 .59649 L\n.92294 .59649 L\n.92499 .59649 L\n.92499 .59649 L\n.92704 .59649 L\n.92908 .59649 L\n.93113 .59649 L\n.93318 .59649 L\n.93523 .59649 L\n.93523 .60332 L\n.93728 .60332 L\n.93932 .60332 L\n.94137 .60332 L\n.94342 .60332 L\n.94547 .60332 L\n.94547 .60332 L\n.94752 .60332 L\n.94956 .60332 L\n.95161 .60332 L\n.95366 .60332 L\n.95571 .60332 L\n.95571 .60332 L\n.95776 .60332 L\n.95981 .60332 L\nMistroke\n.96185 .60332 L\n.9639 .60332 L\n.96595 .60332 L\n.96595 .60332 L\n.968 .60332 L\n.97005 .60332 L\n.97209 .60332 L\n.97414 .60332 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0119048 0.119048 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0214286 0.0721271 0.140476 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 .5 0 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .08095 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(y) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n69.000 12.813 moveto\n(@) show\n75.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n81.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\n% Start of sub-graphic\np\n0.0110024 0.159524 0.0721271 0.278571 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00374 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .21905 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(x) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n69.000 12.813 moveto\n(@) show\n75.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n81.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT000h0\nfMWI1@0000030=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT00100fMWI00<000000=WIf@3IfMT0\n0`3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000@0=WIf@030000003IfMT0fMWI00<0fMWI0P00\n000<0=WIf@@0003o;03IfMT30000o``0fMWI0`000?l/0=WIf@<0003o;03IfMT30000o``0fMWI0`00\n0?l/0=WIf@@0003o7@3IfMT000P0fMWI100000040=WIf@030000003IfMT0fMWI00<0fMWI00<00000\n0=WIf@3IfMT02`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@@0003o7@3IfMT00`000?l0fMWI0000\no`0<0=WIf@030000o`3IfMT0003o01d0fMWI0`000?l<0=WIf@030000o`3IfMT0003o00`0fMWI0`00\n0?lM0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l07@3IfMT40000o`/0fMWI00<0\n003o0=WIf@3IfMT07P3IfMT00100fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0;\n0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o3P3IfMT30000o``0fMWI\n00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0;0=WIf@@0003o3P3IfMT00`000?l0fMWI\n0000o`0<0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l03P3IfMT40000o`/0fMWI\n00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@<0003o3P3IfMT010000?l0fMWI\n0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI01h0fMWI000?0=WIf@8000001@3IfMT00`000000fMWI\n0=WIf@0;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o3P3IfMT00`00\n0?l0fMWI0000o`0<0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l02`3IfMT01000\n0?l0fMWI0=WIf@000?l>0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l0303IfMT0\n0`000?l0fMWI0000o`0<0=WIf@<0003o0P3IfMT00`000?l0fMWI0=WIf@090=WIf@030000o`3IfMT0\n003o00`0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0>0=WIf@040000o`3IfMT0\nfMWI0000o`/0fMWI00<0003o0=WIf@3IfMT07P3IfMT001H0fMWI00<000000=WIf@3IfMT02`3IfMT0\n10000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI10000?l00`3IfMT0003o\n0=WIf@0;0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l02`3IfMT00`000?l0fMWI\n0=WIf@030000o``0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@030000\no`3IfMT0003o00`0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WIf@030000\no`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`030000o`/0fMWI\n00@0003o0=WIf@3IfMT0003o2`3IfMT00`000?l0fMWI0=WIf@0N0=WIf@005P3IfMT00`000000fMWI\n0=WIf@0;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00X0fMWI00<0003o0=WIf@000?l0303IfMT00`00\n0?l0fMWI0000o`0;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02P3IfMT00`00\n0?l0fMWI0000o`0<0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l0303IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WIf@000?l0303IfMT00`00\n0?l0fMWI0000o`0<0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT30000\no`80fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@0N0=WIf@005P3IfMT00`000000\nfMWI0=WIf@0;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00<0003o0=WIf@3IfMT00`000?l90=WI\nf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02P3IfMT00`000?l0fMWI0000o`0<0=WI\nf@030000o`3IfMT0003o00/0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WI\nf@@0003o00<0fMWI0000o`3IfMT02`3IfMT00`000?l0fMWI0000o`0<0=WIf@030000o`3IfMT0003o\n00<0003o2@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WI\nf@000?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WI\nf@3IfMT07P3IfMT001H0fMWI00<000000=WIf@3IfMT02`3IfMT010000?l0fMWI0=WIf@000?l;0=WI\nf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00X0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0;0=WI\nf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00X0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WI\nf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WI\nf@030000o`3IfMT0fMWI01h0fMWI000F0=WIf@030000003IfMT0fMWI00/0fMWI00@0003o0=WIf@3I\nfMT0003o2`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@@0003o00<0fMWI0000o`3IfMT02`3IfMT00`00\n0?l0fMWI0000o`0;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00X0fMWI00<0003o0=WIf@000?l0303IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`00\n0?l0fMWI0=WIf@070=WIf@<0003o00<0fMWI0000o`3IfMT02`3IfMT00`000?l0fMWI0000o`0<0=WI\nf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT07P3IfMT001H0fMWI00<000000=WIf@3I\nfMT02`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0003o00<0003o203IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`00\n0?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`00\n0?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT01@00\n0?l0fMWI0=WIf@3IfMT0003o00`0fMWI00<0003o0=WIf@000?l00`000?l90=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00T0fMWI00<0003o0=WIf@3IfMT07P3IfMT001H0fMWI00<000000=WIf@3IfMT02`3IfMT01000\n0?l0fMWI0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01P3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n1@000?l0fMWI0=WIf@3IfMT0003o00`0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WI\nf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WI\nf@020=WIf@030000o`3IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT07P3IfMT001H0fMWI00<00000\n0=WIf@3IfMT02@3IfMT30000o`80fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02P3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT40000o`030=WIf@000?l0\nfMWI00/0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT01@000?l0fMWI0=WIf@3IfMT0003o00`0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI0`00\n0?lP0=WIf@005P3IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0080fMWI0`000?l9\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0:\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI10000?l90=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00D0003o\n0=WIf@3IfMT0fMWI0000o`0<0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3I\nfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3I\nfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0200fMWI000F0=WIf@030000003IfMT0\nfMWI00T0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI00D0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@050000o`3IfMT0\nfMWI0=WIf@000?l0303IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@005P3IfMT00`000000fMWI0=WIf@09\n0=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n000000030000o`000000fMWI0080fMWI0P0000030=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI10000?l4\n0=WIf@030000o`3IfMT00000008000000P3IfMT2000000030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@3IfMT01@3IfMT00`000?l0fMWI00000002000000@0fMWI00<000000000o`3IfMT00`3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n1@000?l0fMWI0=WIf@3IfMT0003o00P0fMWI00<000000000o`000?l00P000?l01@3IfMT000000000\n003IfMT0003o00T0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0103IfMT3000000040000o`000000fMWI0=WIf@80\n0000103IfMT2000001L0fMWI000F0=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0\n103IfMT00`000?l0fMWI0=WIf@070=WIf@060000o`3IfMT0fMWI0000003IfMT0003o0`3IfMT01000\n0000fMWI0=WIf@0000020=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@@0003o1P3IfMT00`000?l0fMWI0=WIf@050=WIf@040000o`3IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT0003o103IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT000000080fMWI00L000000=WIf@3IfMT000000=WIf@000?l0000000<0fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00@0003o0=WI\nf@3IfMT0fMWI10000?l40=WIf@040000003IfMT0003o00000080fMWI00D000000=WIf@3IfMT00000\n0000o`090=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI\n0=WIf@020=WIf@030000o`3IfMT0fMWI00H0fMWI00D000000000o`3IfMT0fMWI000000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0503IfMT001H0fMWI00<000000=WIf@3I\nfMT02@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00H0003o0=WIf@3I\nfMT0fMWI000000000?l30=WIf@040000003IfMT0fMWI00000080fMWI00<0003o0=WIf@3IfMT00P3I\nfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103I\nfMT00`000?l0fMWI0=WIf@050=WIf@030000o`00000000000080000000D0fMWI0000003IfMT0fMWI\n0000o`040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI\n0=WIf@070=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@0000000P3IfMT01`000000fMWI\n0=WIf@000000fMWI0000o`0000000`3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI\n0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI\n0080fMWI00@000000=WIf@000?l000000P3IfMT01@000000fMWI0=WIf@000000003o00T0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01P3IfMT01@000000003o0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT00`000000fMWI0=WIf@0D0=WIf@005P3IfMT00`000000fMWI0=WIf@090=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@050000\no`3IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@000?l40=WIf@030000o`3IfMT0fMWI\n00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI\n00D0fMWI00@0003o0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000?l40=WIf@030000o`3I\nfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@0000000P0000030=WIf@040000003IfMT0003o000000<0fMWI\n00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI\n00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@030=WIf@03000000000?l0fMWI0080fMWI\n00D000000=WIf@3IfMT000000000o`090=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3I\nfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00H0fMWI00D000000000o`3I\nfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0503IfMT0\n01H0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI\n00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT30000o`80fMWI00@000000=WIf@000?l000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3I\nfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3I\nfMT010000?l000000=WIf@0000020=WIf@040000003IfMT0fMWI0000o`@0fMWI00<0003o0=WIf@3I\nfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3I\nfMT01@3IfMT010000?l0fMWI0=WIf@0000030=WIf@@0003o00D000000=WIf@3IfMT0fMWI0000o`09\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@04\n0=WIf@030000o`3IfMT0fMWI0080fMWI00@000000=WIf@000?l000000P3IfMT01@000000fMWI0=WI\nf@000000003o00T0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01P3IfMT01@000000003o0=WIf@3IfMT000000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0D0=WIf@005P3IfMT00`000000\nfMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0\nfMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00<0fMWI00<000000000o`3IfMT00`3IfMT2000000<0\nfMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00D0\nfMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@040000o`3IfMT000000000\n00<0fMWI0P0000000`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT40000o`H0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@030000\n00030=WIf@000?l0fMWI00800000103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI\n0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI\n00<0fMWI00<000000000o`3IfMT00`3IfMT2000000030=WIf@000?l0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI0`000?l50=WI\nf@80000000@0003o0=WIf@3IfMT0fMWI0P0000040=WIf@8000005`3IfMT001H0fMWI00<000000=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WI\nf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WI\nf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WI\nf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WI\nf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WI\nf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WI\nf@3IfMT0803IfMT001H0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@040=WIf@03\n0000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@03\n0000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@03\n0000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@03\n0000o`3IfMT0fMWI00@0fMWI0`000?l40=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3I\nfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3I\nfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3I\nfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3I\nfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI10000?l40=WIf@03\n0000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@03\n0000o`3IfMT0fMWI0200fMWI000F0=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0\n103IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0\n103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI10000?l60=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0080fMWI0`000?l70=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0\n103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0\n1@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0\n1@3IfMT00`000?l0fMWI0=WIf@0P0=WIf@005P3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@0P0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@0A0=WIf@8000003@3IfMT001D0fMWI1P0000000`000?l00000000000040000\n00030000o`000000000000@0000000<0003o0000000000001@0000000`000?l00000000000040000\n00030000o`000000000000D000000`000?l7000000030000o`000000000000@0000000<0003o0000\n000000001@0000000`000?l0000000000004000000030000o`000000000000D0000000<0003o0000\n00000000100000000`000?l0000000000004000000030000o`000000000000L0000000<0003o0000\n000000001@0000000`000?l0000000000004000000030000o`000000000000D0000000<0003o0000\n00000000100000000`000?l0000000000005000000030000o`000000000000@0000010000?l60000\n00030000o`000000000000D0000000<0003o000000000000100000000`000?l00000000000040000\n00030000o`000000000000D0000000<0003o000000000000100000000`000?l00000000000050000\n00030000o`000000000000L0000000<0003o000000000000100000000`000?l00000000000050000\n00030000o`000000000000P00000203IfMT010000000fMWI0=WIf@00000<0=WIf@005P3IfMT00`00\n0000fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`00\n0?l0fMWI0=WIf@020=WIf@040000003IfMT0fMWI0000o`H0fMWI00@0003o0=WIf@3IfMT00000103I\nfMT00`000?l0fMWI0=WIf@040=WIf@040000003IfMT0fMWI0000o`H0fMWI00@0003o0=WIf@3IfMT0\n0000103IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3I\nfMT00P3IfMT00`000000fMWI0000o`060=WIf@@0003o1P3IfMT00`000?l0fMWI0=WIf@020=WIf@04\n0000003IfMT0fMWI0000o`H0fMWI00@0003o0=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@04\n0=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000000fMWI0000o`09\n0=WIf@030000o`3IfMT0fMWI00D0fMWI00D0003o0=WIf@3IfMT0fMWI000000020=WIf@030000o`3I\nfMT0fMWI00@0fMWI00@0003o0=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@040=WIf@030000\no`3IfMT0fMWI00D0fMWI0`000?l20=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0\n103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0080fMWI00<000000=WIf@3IfMT0\n2`3IfMT00`000000fMWI0=WIf@0=0=WIf@005P3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI00@0fMWI00@0003o0=WIf@3IfMT00000103IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000000fMWI0000o`090=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT010000?l0fMWI0=WIf@0000040=WIf@03\n0000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@03\n0000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@020=WIf@03\n0000003IfMT0fMWI00X0fMWI1@00000<0=WIf@005P3IfMT00`000000fMWI0=WIf@020=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000\no`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@@0003o\n1P3IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0100fMWI00<000000=WIf@3IfMT0\n3@3IfMT001H0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@@0003o103I\nfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3I\nfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103I\nfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI0`000?l70=WIf@030000o`3I\nfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3I\nfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3I\nfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI0200fMWI000F0=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT0\n0`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI0`000?l40=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0\nfMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI00D0fMWI00<0003o0=WIf@3IfMT0803IfMT001H0fMWI00<000000=WIf@3IfMT00P3IfMT00`00\n0?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`00\n0?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`00\n0?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`00\n0?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`00\n0?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`00\n0?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`00\n0?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`00\n0?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0803IfMT001H0\nfMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00@0\nfMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080\nfMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@<0003o1`3IfMT00`000?l0\nfMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0\nfMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0\nfMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0\nfMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0\nfMWI0=WIf@020=WIf@@0003o1P3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0\nfMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080\nfMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0200\nfMWI000F0=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI\n0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI\n0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI\n0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT01@3IfMT30000o`@0fMWI\n00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI\n00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00@0fMWI\n00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI\n00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI00@0fMWI\n00<0003o0=WIf@3IfMT01@3IfMT30000o`@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI\n0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI\n0=WIf@0P0=WIf@005P3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00@0fMWI00<0\n003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00D0fMWI00<0\n003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@0P0=WIf@005P3IfMT00`000000fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@05\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@04\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@04\n0=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT40000ob80fMWI000F0=WI\nf@030000003IfMT0fMWI0080fMWI00@0003o0=WIf@3IfMT0fMWI10000?l60=WIf@@0003o103IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT3\n0000o`L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0S0=WIf@005P3IfMT00`00\n0000fMWI0=WIf@020=WIf@050000o`3IfMT0fMWI0=WIf@000?l0303IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@04\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI0`000?l40=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI02<0fMWI000F0=WI\nf@030000003IfMT0fMWI0080fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0<0=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI00@0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@040=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI02<0fMWI000F0=WIf@030000003IfMT0fMWI0080fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0<\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@07\n0=WIf@030000o`3IfMT0fMWI0080fMWI0`000?l90=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI0`000?l9\n0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@02\n0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0S\n0=WIf@005P3IfMT00`000000fMWI0=WIf@020=WIf@050000o`3IfMT0fMWI0=WIf@000?l0303IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT3\n0000o`80fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT30000o`80fMWI00<0003o0=WIf@3IfMT02@3IfMT0\n0`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT0\n0`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT08`3IfMT0\n01H0fMWI00<000000=WIf@3IfMT00P3IfMT01@000?l0fMWI0=WIf@3IfMT0003o00`0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00T0fMWI00@0003o\n0=WIf@3IfMT0003o2`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI0080fMWI\n00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI\n00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0S0=WIf@005P3IfMT00`000000fMWI0=WI\nf@020=WIf@<0003o00<0fMWI0000o`3IfMT02`3IfMT00`000?l0fMWI0000o`030000o`T0fMWI00<0\n003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0\n003o0=WIf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI0080fMWI\n00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI\n00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI00<0003o\n2@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0\n0P3IfMT00`000?l0fMWI0=WIf@090=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00<0003o0=WIf@3I\nfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3I\nfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI02<0fMWI000F0=WIf@030000\n003IfMT0fMWI00@0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@030000\no`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000\no`3IfMT0fMWI00T0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT00`000?l0fMWI0=WIf@020=WIf@03\n0000o`3IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@<0\n003o0P3IfMT00`000?l0fMWI0=WIf@090=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00<0003o0=WI\nf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WI\nf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00L0fMWI10000?l0\n0`3IfMT0003o0=WIf@0T0=WIf@005P3IfMT00`000000fMWI0=WIf@040=WIf@030000o`3IfMT0003o\n00`0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI\n00L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@090=WIf@040000o`3IfMT0fMWI\n0000o`/0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000o`3I\nfMT0fMWI0000o`/0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@03\n0000o`3IfMT0fMWI00<0003o2@3IfMT00`000?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00X0\nfMWI00<0003o0=WIf@000?l09@3IfMT001H0fMWI00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI\n0000o`0<0=WIf@030000o`3IfMT0003o00`0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI\n0=WIf@070=WIf@<0003o0P3IfMT00`000?l0fMWI0=WIf@090=WIf@040000o`3IfMT0fMWI0000o`/0\nfMWI00<0003o0=WIf@3IfMT00`000?l90=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3I\nfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00<0003o\n0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@070=WIf@@0003o00<0fMWI0000o`3IfMT02P3IfMT0\n10000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00<0003o0=WIf@3IfMT0\n0P3IfMT00`000?l0fMWI0=WIf@0:0=WIf@030000o`3IfMT0003o02D0fMWI000F0=WIf@030000003I\nfMT0fMWI00@0fMWI00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@030000o`3I\nfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02@3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000\no`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT00`000?l0fMWI0=WIf@020=WI\nf@030000o`3IfMT0fMWI00T0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT010000?l0fMWI0=WIf@00\n0?l;0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT02P3IfMT00`000?l0fMWI0000\no`0;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT00`000?l0\nfMWI0=WIf@020=WIf@030000o`3IfMT0fMWI00X0fMWI00<0003o0=WIf@000?l09@3IfMT001H0fMWI\n00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI0000o`0<0=WIf@030000o`3IfMT0003o00`0fMWI\n00<0003o0=WIf@000?l00`000?l;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3I\nfMT0003o2`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@<0003o0P3IfMT00`000?l0fMWI0=WIf@09\n0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT00`000?l0fMWI\n0=WIf@030000o``0fMWI00<0003o0=WIf@000?l02`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@04\n0000o`3IfMT0fMWI0000o`/0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0:0=WI\nf@030000o`3IfMT0003o02D0fMWI000F0=WIf@030000003IfMT0fMWI00@0fMWI00<0003o0=WIf@00\n0?l0303IfMT00`000?l0fMWI0000o`0<0=WIf@030000o`3IfMT0003o00h0fMWI00@0003o0=WIf@3I\nfMT0003o2`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI0000o`d0fMWI00@0\n003o0=WIf@3IfMT0003o2`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI0000\no`/0fMWI00@0003o0=WIf@3IfMT0003o3P3IfMT00`000?l0fMWI0000o`0;0=WIf@040000o`3IfMT0\nfMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o2`3IfMT60000o``0fMWI00<0003o0=WIf@000?l0\n9@3IfMT000h0fMWI1@0000030=WIf@030000003IfMT0fMWI00@0fMWI00<0003o0=WIf@000?l0303I\nfMT00`000?l0fMWI0000o`0<0=WIf@<0003o3P3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000\no`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o3@3IfMT40000o`/0fMWI00@0003o0=WI\nf@3IfMT0003o2`3IfMT010000?l0fMWI0=WIf@000?l;0=WIf@@0003o3P3IfMT00`000?l0fMWI0000\no`0;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI00@0003o0=WIf@3IfMT0003o7@3IfMT00`000?l0\nfMWI0000o`0U0=WIf@00403IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00@0fMWI\n00<0003o0=WIf@000?l0303IfMT00`000?l0fMWI0000o`0M0=WIf@@0003o2`3IfMT010000?l0fMWI\n0=WIf@000?l;0=WIf@@0003o703IfMT010000?l0fMWI0=WIf@000?l;0=WIf@040000o`3IfMT0fMWI\n0000oad0fMWI0`000?l;0=WIf@040000o`3IfMT0fMWI0000o`/0fMWI10000?lM0=WIf@030000o`3I\nfMT0003o02D0fMWI000@0=WIf@030000003IfMT0fMWI00<0fMWI0P0000050=WIf@<0003o303IfMT3\n0000ob`0fMWI10000?l[0=WIf@@0003o2`3IfMT40000ob/0fMWI10000?l/0=WIf@<0003o9@3IfMT0\n0100fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000@0=WIf@03\n0000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WIf@003`3IfMT2000000D0fMWI\n00<000000=WIf@3IfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F\n0=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT001H0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WI\nf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003IfMT0fMWI0?l0fMWI\n203IfMT001H0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o\n0=WIf@P0fMWI000F0=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT001H0fMWI00<000000=WIf@3I\nfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003I\nfMT0fMWI0?l0fMWI203IfMT001H0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WIf@005P3IfMT00`00\n0000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003IfMT0fMWI0?l0fMWI203IfMT001H0fMWI\n00<000000=WIf@3IfMT0o`3IfMT80=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F\n0=WIf@030000003IfMT0fMWI0080fMWI3@00O`3h0=WIf@005P3IfMT00`000000fMWI0=WIf@0>0=WI\nf@03001o003IfMT0fMWI0?H0fMWI000F0=WIf@030000003IfMT0fMWI00h0fMWI00<007l00=WIf@3I\nfMT0mP3IfMT001H0fMWI00<000000=WIf@3IfMT03P3IfMT00`00O`00fMWI0=WIf@3f0=WIf@003P3I\nfMT4000000@0fMWI00<000000=WIf@3IfMT03P3IfMT00`00O`00fMWI0=WIf@3f0=WIf@003P3IfMT0\n0`000000fMWI0=WIf@050=WIf@030000003IfMT0fMWI00h0fMWI00<007l00=WIf@3IfMT0mP3IfMT0\n00l0fMWI00<000000=WIf@3IfMT0103IfMT2000000l0fMWI00<007l00=WIf@3IfMT0mP3IfMT00100\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0>0=WIf@03001o003IfMT0fMWI0?H0\nfMWI000>0=WIf@040000003IfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT03P3IfMT00`00O`00\nfMWI0=WIf@3f0=WIf@003`3IfMT2000000D0fMWI00<000000=WIf@3IfMT03P3IfMT00`00O`00fMWI\n0=WIf@3f0=WIf@005P3IfMT00`000000fMWI0=WIf@0>0=WIf@03001o003IfMT0fMWI0?H0fMWI000F\n0=WIf@030000003IfMT0fMWI00h0fMWI00<007l00=WIf@3IfMT0mP3IfMT001H0fMWI00<000000=WI\nf@3IfMT03P3IfMT00`00O`00fMWI0=WIf@3f0=WIf@005P3IfMT00`000000fMWI0=WIf@0>0=WIf@03\n001o003IfMT0fMWI0?H0fMWI000F0=WIf@030000003IfMT0fMWI00h0fMWI00<007l00=WIf@3IfMT0\nmP3IfMT001H0fMWI00<000000=WIf@3IfMT03P3IfMT00`00O`00fMWI0=WIf@3f0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@0>0=WIf@03001o003IfMT0fMWI0?H0fMWI000F0=WIf@030000003IfMT0fMWI\n00h0fMWI00<007l00=WIf@3IfMT0mP3IfMT001H0fMWI00<000000=WIf@3IfMT03P3IfMT@001o0>T0\nfMWI000F0=WIf@030000003IfMT0fMWI01d0fMWI00<007l00=WIf@3IfMT0i`3IfMT001H0fMWI00<0\n00000=WIf@3IfMT07@3IfMT00`00O`00fMWI0=WIf@3W0=WIf@005P3IfMT00`000000fMWI0=WIf@0M\n0=WIf@03001o003IfMT0fMWI0>L0fMWI000F0=WIf@030000003IfMT0fMWI01d0fMWI00<007l00=WI\nf@3IfMT0i`3IfMT001H0fMWI00<000000=WIf@3IfMT07@3IfMT00`00O`00fMWI0=WIf@3W0=WIf@00\n5P3IfMT00`000000fMWI0=WIf@0M0=WIf@03001o003IfMT0fMWI0>L0fMWI000F0=WIf@030000003I\nfMT0fMWI01d0fMWI00<007l00=WIf@3IfMT0i`3IfMT001H0fMWI00<000000=WIf@3IfMT07@3IfMT0\n0`00O`00fMWI0=WIf@3W0=WIf@005P3IfMT00`000000fMWI0=WIf@0M0=WIf@03001o003IfMT0fMWI\n0>L0fMWI000F0=WIf@030000003IfMT0fMWI01d0fMWI00H007l00=WIf@3IfMT007l00=WIf@00O`02\n0=WIf@03001o003IfMT007l00080fMWI00<007l00=WIf@3IfMT0fP3IfMT000l0fMWI0P0000050=WI\nf@030000003IfMT0fMWI01d0fMWI4000O`3J0=WIf@003P3IfMT010000000fMWI0=WIf@0000040=WI\nf@030000003IfMT0fMWI02`0fMWI00<007l00=WIf@3IfMT0f03IfMT00100fMWI00<000000=WIf@3I\nfMT00`3IfMT2000002d0fMWI00<007l00=WIf@3IfMT0f03IfMT00140fMWI00<000000=WIf@3IfMT0\n0P3IfMT00`000000fMWI0=WIf@0/0=WIf@03001o003IfMT0fMWI0=P0fMWI000>0=WIf@040000003I\nfMT0fMWI000000@0fMWI00<000000=WIf@3IfMT0;03IfMT00`00O`00fMWI0=WIf@3H0=WIf@003`3I\nfMT2000000D0fMWI00<000000=WIf@3IfMT0;03IfMT00`00O`00fMWI0=WIf@3H0=WIf@005P3IfMT0\n0`000000fMWI0=WIf@0/0=WIf@03001o003IfMT0fMWI0=P0fMWI000F0=WIf@030000003IfMT0fMWI\n02`0fMWI4000O`3;0=WIf@005P3IfMT00`000000fMWI0=WIf@0k0=WIf@03001o003IfMT0fMWI0<T0\nfMWI000F0=WIf@030000003IfMT0fMWI03/0fMWI00<007l00=WIf@3IfMT0b@3IfMT001H0fMWI00<0\n00000=WIf@3IfMT0>`3IfMT00`00O`00fMWI0=WIf@390=WIf@005P3IfMT00`000000fMWI0=WIf@0k\n0=WIf@03001o003IfMT0fMWI0<T0fMWI000F0=WIf@030000003IfMT0fMWI03/0fMWI00H007l00=WI\nf@3IfMT007l00=WIf@00O`020=WIf@03001o003IfMT007l00080fMWI00<007l00=WIf@3IfMT0_03I\nfMT001H0fMWI00<000000=WIf@3IfMT0>`3IfMT@001o0;`0fMWI000F0=WIf@030000003IfMT0fMWI\n04X0fMWI00<007l00=WIf@3IfMT0^P3IfMT001H0fMWI00<000000=WIf@3IfMT0BP3IfMT00`00O`00\nfMWI0=WIf@2j0=WIf@005P3IfMT00`000000fMWI0=WIf@1:0=WIf@03001o003IfMT0fMWI0;X0fMWI\n000F0=WIf@030000003IfMT0fMWI04X0fMWI3`00O`2^0=WIf@005P3IfMT00`000000fMWI0=WIf@1H\n0=WIf@03001o003IfMT0fMWI0:`0fMWI000F0=WIf@030000003IfMT0fMWI05P0fMWI00<007l00=WI\nf@3IfMT0[03IfMT001H0fMWI00<000000=WIf@3IfMT0F03IfMT00`00O`00fMWI0=WIf@2/0=WIf@00\n5P3IfMT00`000000fMWI0=WIf@1H0=WIfA0007l0W`3IfMT001H0fMWI00<000000=WIf@3IfMT0I`3I\nfMT00`00O`00fMWI0=WIf@2M0=WIf@005P3IfMT00`000000fMWI0=WIf@1W0=WIf@03001o003IfMT0\nfMWI09d0fMWI000F0=WIf@030000003IfMT0fMWI06L0fMWI00@007l00=WIf@3IfMT007l00P3IfMT0\n0`00O`00fMWI001o00020=WIf@03001o003IfMT007l00980fMWI000F0=WIf@030000003IfMT0fMWI\n06L0fMWI4000O`2@0=WIf@00403IfMT3000000<0fMWI00<000000=WIf@3IfMT0MP3IfMT00`00O`00\nfMWI0=WIf@2>0=WIf@004@3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI07H0fMWI\n00@007l00=WIf@3IfMT007l00P3IfMT00`00O`00fMWI001o00020=WIf@03001o003IfMT007l008<0\nfMWI000>0=WIf@D000000`3IfMT2000007L0fMWI4000O`210=WIf@003P3IfMT010000000fMWI0=WI\nf@0000040=WIf@030000003IfMT0fMWI08D0fMWI00<007l00=WIf@3IfMT0O`3IfMT000l0fMWI00<0\n00000=WIf@000000103IfMT00`000000fMWI0=WIf@250=WIf@04001o003IfMT0fMWI001o0080fMWI\n00<007l00=WIf@00O`000P3IfMT00`00O`00fMWI001o001d0=WIf@00403IfMT2000000@0fMWI00<0\n00000=WIf@3IfMT0Q@3IfMT@001o0780fMWI000F0=WIf@030000003IfMT0fMWI09@0fMWI00<007l0\n0=WIf@3IfMT0L03IfMT001H0fMWI00<000000=WIf@3IfMT0U03IfMT@001o06<0fMWI000F0=WIf@03\n0000003IfMT0fMWI0:<0fMWI00<007l00=WIf@3IfMT0H@3IfMT001H0fMWI00<000000=WIf@3IfMT0\nX`3IfMT@001o05@0fMWI000F0=WIf@030000003IfMT0fMWI0;80fMWI00<007l00=WIf@3IfMT0DP3I\nfMT001H0fMWI00<000000=WIf@3IfMT0/P3IfMT@001o04D0fMWI000F0=WIf@030000003IfMT0fMWI\n0<40fMWI00<007l00=WIf@3IfMT0@`3IfMT001H0fMWI00<000000=WIf@3IfMT0`@3IfMT@001o03H0\nfMWI000F0=WIf@030000003IfMT0fMWI0=00fMWI00H007l00=WIf@3IfMT007l00=WIf@00O`020=WI\nf@03001o003IfMT007l00080fMWI00<007l00=WIf@3IfMT09`3IfMT001H0fMWI00<000000=WIf@3I\nfMT0d03IfMT@001o02L0fMWI000F0=WIf@030000003IfMT0fMWI0=l0fMWI00H007l00=WIf@3IfMT0\n07l00=WIf@00O`020=WIf@03001o003IfMT0fMWI01d0fMWI000F0=WIf@030000003IfMT0fMWI0=l0\nfMWI2`00O`0M0=WIf@005P3IfMT00`000000fMWI0=WIf@3o0=WIf@P0fMWI000F0=WIf@030000003I\nfMT0fMWI0?l0fMWI203IfMT001H0fMWI00<000000=WIf@3IfMT0o`3IfMT80=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WI\nf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@008P3IfMT2000000H0fMWI0P000008\n0=WIf@800000jP3IfMT002<0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WI\nf@030000003IfMT0fMWI0>P0fMWI000T0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3I\nfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0j03IfMT002<0fMWI00<000000=WIf@000000\n103IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00D000000=WIf@3IfMT0fMWI0000003Z0=WI\nf@005P3IfMT8001o00@0fMWI00@000000=WIf@3IfMT00000103IfMT01@000000fMWI0=WIf@3IfMT0\n000000H0fMWI00<000000=WIf@3IfMT0j03IfMT00240fMWI0`0000000`3IfMT00000000000030=WI\nf@030000003IfMT0fMWI00D000000`3IfMT00`000000fMWI0=WIf@3X0=WIf@00:P3IfMT01@000000\nfMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0j03IfMT002X0fMWI0P0000080=WIf@80\n0000jP3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0\nfMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0\nfMWI8@3IfMT00?l0fMWI8@3IfMT002X0fMWI0P0000080=WIf@800000jP3IfMT002X0fMWI00<00000\n0=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@3X0=WIf@008@3IfMT2000000030=WIf@0000000000\n00@0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0j03IfMT001H0\nfMWI20000?l40=WIf@030000003IfMT0000000D0fMWI00D000000=WIf@3IfMT0fMWI000000020=WI\nf@050000003IfMT0fMWI0=WIf@000000jP3IfMT002<0fMWI00<000000=WIf@3IfMT0103IfMT01@00\n0000fMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0j03IfMT00240fMWI0P0000000`3I\nfMT00000000000040=WIf@030000003IfMT0fMWI00D000000`3IfMT00`000000fMWI0=WIf@3X0=WI\nf@00:P3IfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<000000=WIf@3IfMT0j03IfMT002X0\nfMWI0P0000080=WIf@800000jP3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT0\n0?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT0\n0?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00001"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-3.86733`, -1.7217`, 0.40244`, 0.0389604`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -10.3955`, 0.119891`, 0.0615577`]], Rule[List[List[22.125`, 30.4375`], List[43.0625`, 26.8125`]], List[-6.32337`, -42.9027`, 0.240604`, 0.240604`]], Rule[List[List[22.125`, 30.4375`], List[24.1875`, 7.9375`]], List[-6.32337`, -47.4441`, 0.240604`, 0.240604`]]]]]], Open]]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[853, 834]], Rule[WindowMargins, List[List[8, Automatic], List[Automatic, 7]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]