List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Terminate", "Title"], Cell["\nThe terminate equation successfully terminates the current simulation, i.e. no error condition is indicated. A string argument is used to indicate the reason for successful termination, which can be reported back to the user in some appropriate way by the simulation tool. Terminate is most commonly used to give a more appropriate stopping criteria for terminating a simulation than a fixed point in time.", "Text"], Cell[CellGroupData[List[Cell["Celestial Body", "Section"], Cell[TextData[List["This example illustrates a Rocket landing on the moon. The class ", StyleBox["CelestialBody", Rule[FontFamily, "Courier"]], " can be used to store information about celestial bodies such as the earth and the moon. "]], "Text"], Cell["class CelestialBody\n  constant Real g = 6.672e-11;\n  parameter Real radius;\n  parameter String name;\n  Real mass;\nend CelestialBody;", "ModelicaInput"]], Closed]], Cell[CellGroupData[List[Cell["Rocket", "Section"], Cell[TextData[List["The class ", StyleBox["Rocket", Rule[FontFamily, "Courier"]], " embodies the equations of vertical motion for a rocket which is influenced by an external gravitational force field ", StyleBox["gravity", Rule[FontFamily, "Courier"]], ", and the force ", StyleBox["thrust", Rule[FontFamily, "Courier"]], " from the rocket motor, acting in the opposite direction to the gravitational force.\n"]], "Text"], Cell["class Rocket\n  parameter String name;\n  Real mass(start = 1038.358);\n  Real altitude(start = 59404);\n  Real velocity(start = -2003);\n  Real acceleration;\n  Real thrust;\t\t\t\t\t\t\t\t\t\t// Thrust force on the rocket\n  Real gravity;\t\t\t\t\t\t\t\t\t\t// Gravity forcefield\n  parameter Real massLossRate = 0.000277;\nequation\n  (thrust - mass*gravity)/mass = acceleration;\n  der(mass) \t= -massLossRate * abs(thrust);\n  der(altitude) = velocity;\n  der(velocity) = acceleration;\nend Rocket;  ", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["MoonLanding", "Section"], Cell[TextData[List["We can use ", StyleBox["terminate", Rule[FontFamily, "Courier"]], " to stop the simulation when the lander touches the surface in order to avoid the lander descending into the interior of the moon. This is the case for the version of the ", StyleBox["MoonLanding", Rule[FontFamily, "Courier"]], " class below.\n"]], "Text"], Cell["class MoonLanding\n  parameter Real force1 = 36350;\n  parameter Real force2 = 1308;\n  parameter Real thrustEndTime = 210;\n  parameter Real thrustDecreaseTime = 43.2;\n  Rocket apollo(name = \"Apollo13\");\n  CelestialBody moon(name = \"moon\", mass = 7.382e22, radius = 1.738e6);\nequation\n  apollo.thrust = if (time < thrustDecreaseTime) then force1\n  \t\t\t\t  else if (time < thrustEndTime) then force2\n  \t\t\t\t  else 0;\n  apollo.gravity = moon.g*moon.mass/(apollo.altitude + moon.radius) ^2;\n  \n  when apollo.altitude < 0 then\t\t\t\t\t\t\t\t\t\t// Termination condition\n    terminate(\"The moon lander touches the ground of moon\");\n  end when;  \nend MoonLanding;  ", "ModelicaInput"]], Open]], Cell[CellGroupData[List[Cell["Simulation of MoonLanding", "Section"], Cell[TextData[List["Now we simulate the ", StyleBox["MoonLanding", Rule[FontFamily, "Courier"]], " example during the time interval {0, 230}, since the ", StyleBox["Rocket", Rule[FontFamily, "Courier"]], " reaches the surface of the moon at time 200, the simulation is terminated."]], "Text"], Cell["simulate( MoonLanding, stopTime=200 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(MoonLanding\\) : 2004-1-30 14:10:35.9447540 : \\!\\({0.`, 200.`}\\) : 503 data points : 2 events : 17 variables>\\n\\!\\({\\*TagBox[\\(apollo . acceleration\\), HoldForm], \\*TagBox[\\(apollo . altitude\\), HoldForm], \\*TagBox[\\(apollo . gravity\\), HoldForm], \\*TagBox[\\(apollo . mass\\), HoldForm], \\*TagBox[\\(apollo . massLossRate\\), HoldForm], \\*TagBox[\\(apollo . thrust\\), HoldForm], \\*TagBox[\\(apollo . velocity\\), HoldForm], \\*TagBox[\\\"force1\\\", HoldForm], \\*TagBox[\\\"force2\\\", HoldForm], \\*TagBox[\\(moon . g\\), HoldForm], \\*TagBox[\\(moon . mass\\), HoldForm], \\*TagBox[\\(moon . radius\\), HoldForm], \\*TagBox[\\\"thrustDecreaseTime\\\", HoldForm], \\*TagBox[\\\"thrustEndTime\\\", HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . altitude)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . mass)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\((apollo . velocity)\\), \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm]}\\)\""], "Output"], Cell["Then we plot the simulation. It is possible to plot all the different variables of Apollo", "Text"], Cell["plot( apollo.altitude );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0047619 0.0124566 9.94651e-006 [\n[.2619 -4e-005 -6 -9 ]\n[.2619 -4e-005 6 0 ]\n[.5 -4e-005 -9 -9 ]\n[.5 -4e-005 9 0 ]\n[.7381 -4e-005 -9 -9 ]\n[.7381 -4e-005 9 0 ]\n[.97619 -4e-005 -9 -9 ]\n[.97619 -4e-005 9 0 ]\n[1.025 .01246 0 -6.28125 ]\n[1.025 .01246 10 6.28125 ]\n[.01131 .11192 -30 -4.5 ]\n[.01131 .11192 0 4.5 ]\n[.01131 .21139 -30 -4.5 ]\n[.01131 .21139 0 4.5 ]\n[.01131 .31085 -30 -4.5 ]\n[.01131 .31085 0 4.5 ]\n[.01131 .41032 -30 -4.5 ]\n[.01131 .41032 0 4.5 ]\n[.01131 .50978 -30 -4.5 ]\n[.01131 .50978 0 4.5 ]\n[.01131 .60925 -30 -4.5 ]\n[.01131 .60925 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.2619 .01246 m\n.2619 .01871 L\ns\n[(50)] .2619 -4e-005 0 1 Mshowa\n.5 .01246 m\n.5 .01871 L\ns\n[(100)] .5 -4e-005 0 1 Mshowa\n.7381 .01246 m\n.7381 .01871 L\ns\n[(150)] .7381 -4e-005 0 1 Mshowa\n.97619 .01246 m\n.97619 .01871 L\ns\n[(200)] .97619 -4e-005 0 1 Mshowa\n.125 Mabswid\n.07143 .01246 m\n.07143 .01621 L\ns\n.11905 .01246 m\n.11905 .01621 L\ns\n.16667 .01246 m\n.16667 .01621 L\ns\n.21429 .01246 m\n.21429 .01621 L\ns\n.30952 .01246 m\n.30952 .01621 L\ns\n.35714 .01246 m\n.35714 .01621 L\ns\n.40476 .01246 m\n.40476 .01621 L\ns\n.45238 .01246 m\n.45238 .01621 L\ns\n.54762 .01246 m\n.54762 .01621 L\ns\n.59524 .01246 m\n.59524 .01621 L\ns\n.64286 .01246 m\n.64286 .01621 L\ns\n.69048 .01246 m\n.69048 .01621 L\ns\n.78571 .01246 m\n.78571 .01621 L\ns\n.83333 .01246 m\n.83333 .01621 L\ns\n.88095 .01246 m\n.88095 .01621 L\ns\n.92857 .01246 m\n.92857 .01621 L\ns\n.25 Mabswid\n0 .01246 m\n1 .01246 L\ns\ngsave\n1.025 .01246 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .11192 m\n.03006 .11192 L\ns\n[(10000)] .01131 .11192 1 0 Mshowa\n.02381 .21139 m\n.03006 .21139 L\ns\n[(20000)] .01131 .21139 1 0 Mshowa\n.02381 .31085 m\n.03006 .31085 L\ns\n[(30000)] .01131 .31085 1 0 Mshowa\n.02381 .41032 m\n.03006 .41032 L\ns\n[(40000)] .01131 .41032 1 0 Mshowa\n.02381 .50978 m\n.03006 .50978 L\ns\n[(50000)] .01131 .50978 1 0 Mshowa\n.02381 .60925 m\n.03006 .60925 L\ns\n[(60000)] .01131 .60925 1 0 Mshowa\n.125 Mabswid\n.02381 .03235 m\n.02756 .03235 L\ns\n.02381 .05224 m\n.02756 .05224 L\ns\n.02381 .07214 m\n.02756 .07214 L\ns\n.02381 .09203 m\n.02756 .09203 L\ns\n.02381 .13181 m\n.02756 .13181 L\ns\n.02381 .15171 m\n.02756 .15171 L\ns\n.02381 .1716 m\n.02756 .1716 L\ns\n.02381 .19149 m\n.02756 .19149 L\ns\n.02381 .23128 m\n.02756 .23128 L\ns\n.02381 .25117 m\n.02756 .25117 L\ns\n.02381 .27107 m\n.02756 .27107 L\ns\n.02381 .29096 m\n.02756 .29096 L\ns\n.02381 .33074 m\n.02756 .33074 L\ns\n.02381 .35064 m\n.02756 .35064 L\ns\n.02381 .37053 m\n.02756 .37053 L\ns\n.02381 .39042 m\n.02756 .39042 L\ns\n.02381 .43021 m\n.02756 .43021 L\ns\n.02381 .4501 m\n.02756 .4501 L\ns\n.02381 .47 m\n.02756 .47 L\ns\n.02381 .48989 m\n.02756 .48989 L\ns\n.02381 .52967 m\n.02756 .52967 L\ns\n.02381 .54957 m\n.02756 .54957 L\ns\n.02381 .56946 m\n.02756 .56946 L\ns\n.02381 .58935 m\n.02756 .58935 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02571 .59538 L\n.02762 .58749 L\n.02952 .57965 L\n.03143 .57187 L\n.03333 .56415 L\n.03524 .55647 L\n.03714 .54886 L\n.03905 .54129 L\n.04095 .53379 L\n.04286 .52634 L\n.04476 .51894 L\n.04667 .5116 L\n.04857 .50431 L\n.05048 .49709 L\n.05238 .48992 L\n.05429 .48281 L\n.05619 .47575 L\n.0581 .46876 L\n.06 .46182 L\n.0619 .45493 L\n.06381 .44811 L\n.06571 .44134 L\n.06762 .43464 L\n.06952 .42799 L\n.07143 .4214 L\n.07333 .41487 L\n.07524 .40839 L\n.07714 .40198 L\n.07905 .39563 L\n.08095 .38934 L\n.08286 .38311 L\n.08476 .37694 L\n.08667 .37083 L\n.08857 .36478 L\n.09048 .3588 L\n.09238 .35287 L\n.09429 .34701 L\n.09619 .34122 L\n.0981 .33548 L\n.1 .32981 L\n.1019 .32419 L\n.10381 .31865 L\n.10571 .31317 L\n.10762 .30775 L\n.10952 .3024 L\n.11143 .29711 L\n.11333 .29189 L\n.11524 .28673 L\n.11714 .28164 L\nMistroke\n.11905 .27662 L\n.12095 .27166 L\n.12286 .26677 L\n.12476 .26194 L\n.12667 .25718 L\n.12857 .2525 L\n.13048 .24788 L\n.13238 .24333 L\n.13429 .23884 L\n.13619 .23443 L\n.1381 .23009 L\n.14 .22581 L\n.1419 .22161 L\n.14381 .21748 L\n.14571 .21342 L\n.14762 .20943 L\n.14952 .20551 L\n.15143 .20167 L\n.15333 .19789 L\n.15524 .19419 L\n.15714 .19057 L\n.15905 .18702 L\n.16095 .18354 L\n.16286 .18014 L\n.16476 .17681 L\n.16667 .17356 L\n.16857 .17038 L\n.17048 .16728 L\n.17238 .16426 L\n.17429 .16131 L\n.17619 .15845 L\n.1781 .15566 L\n.18 .15295 L\n.1819 .15032 L\n.18381 .14777 L\n.18571 .14529 L\n.18762 .1429 L\n.18952 .14059 L\n.19143 .13836 L\n.19333 .13622 L\n.19524 .13415 L\n.19714 .13217 L\n.19905 .13028 L\n.20095 .12846 L\n.20286 .12673 L\n.20476 .12509 L\n.20667 .12353 L\n.20857 .12206 L\n.21048 .12068 L\n.21238 .11938 L\nMistroke\n.21429 .11817 L\n.21619 .11705 L\n.2181 .11602 L\n.22 .11507 L\n.2219 .11422 L\n.22381 .11346 L\n.22571 .11279 L\n.22762 .11221 L\n.22952 .11172 L\n.22952 .11172 L\n.23143 .11128 L\n.23333 .11084 L\n.23524 .1104 L\n.23714 .10997 L\n.23905 .10953 L\n.24095 .10909 L\n.24286 .10866 L\n.24476 .10823 L\n.24667 .10779 L\n.24857 .10736 L\n.25048 .10693 L\n.25238 .1065 L\n.25429 .10607 L\n.25619 .10564 L\n.2581 .10521 L\n.26 .10479 L\n.2619 .10436 L\n.26381 .10394 L\n.26571 .10351 L\n.26762 .10309 L\n.26952 .10267 L\n.27143 .10224 L\n.27333 .10182 L\n.27524 .1014 L\n.27714 .10099 L\n.27905 .10057 L\n.28095 .10015 L\n.28286 .09974 L\n.28476 .09932 L\n.28667 .09891 L\n.28857 .09849 L\n.29048 .09808 L\n.29238 .09767 L\n.29429 .09726 L\n.29619 .09685 L\n.2981 .09644 L\n.3 .09603 L\n.3019 .09563 L\n.30381 .09522 L\n.30571 .09481 L\nMistroke\n.30762 .09441 L\n.30952 .09401 L\n.31143 .0936 L\n.31333 .0932 L\n.31524 .0928 L\n.31714 .0924 L\n.31905 .092 L\n.32095 .09161 L\n.32286 .09121 L\n.32476 .09081 L\n.32667 .09042 L\n.32857 .09003 L\n.33048 .08963 L\n.33238 .08924 L\n.33429 .08885 L\n.33619 .08846 L\n.3381 .08807 L\n.34 .08768 L\n.3419 .08729 L\n.34381 .08691 L\n.34571 .08652 L\n.34762 .08614 L\n.34952 .08575 L\n.35143 .08537 L\n.35333 .08499 L\n.35524 .08461 L\n.35714 .08423 L\n.35905 .08385 L\n.36095 .08347 L\n.36286 .08309 L\n.36476 .08272 L\n.36667 .08234 L\n.36857 .08197 L\n.37048 .08159 L\n.37238 .08122 L\n.37429 .08085 L\n.37619 .08048 L\n.3781 .08011 L\n.38 .07974 L\n.3819 .07937 L\n.38381 .07901 L\n.38571 .07864 L\n.38762 .07828 L\n.38952 .07791 L\n.39143 .07755 L\n.39333 .07719 L\n.39524 .07683 L\n.39714 .07647 L\n.39905 .07611 L\n.40095 .07575 L\nMistroke\n.40286 .07539 L\n.40476 .07503 L\n.40667 .07468 L\n.40857 .07432 L\n.41048 .07397 L\n.41238 .07362 L\n.41429 .07327 L\n.41619 .07292 L\n.4181 .07257 L\n.42 .07222 L\n.4219 .07187 L\n.42381 .07152 L\n.42571 .07118 L\n.42762 .07083 L\n.42952 .07049 L\n.43143 .07014 L\n.43333 .0698 L\n.43524 .06946 L\n.43714 .06912 L\n.43905 .06878 L\n.44095 .06844 L\n.44286 .06811 L\n.44476 .06777 L\n.44667 .06743 L\n.44857 .0671 L\n.45048 .06677 L\n.45238 .06643 L\n.45429 .0661 L\n.45619 .06577 L\n.4581 .06544 L\n.46 .06511 L\n.4619 .06479 L\n.46381 .06446 L\n.46571 .06413 L\n.46762 .06381 L\n.46952 .06349 L\n.47143 .06316 L\n.47333 .06284 L\n.47524 .06252 L\n.47714 .0622 L\n.47905 .06188 L\n.48095 .06156 L\n.48286 .06125 L\n.48476 .06093 L\n.48667 .06062 L\n.48857 .0603 L\n.49048 .05999 L\n.49238 .05968 L\n.49429 .05937 L\n.49619 .05906 L\nMistroke\n.4981 .05875 L\n.5 .05844 L\n.5019 .05813 L\n.50381 .05783 L\n.50571 .05752 L\n.50762 .05722 L\n.50952 .05692 L\n.51143 .05661 L\n.51333 .05631 L\n.51524 .05601 L\n.51714 .05571 L\n.51905 .05542 L\n.52095 .05512 L\n.52286 .05482 L\n.52476 .05453 L\n.52667 .05423 L\n.52857 .05394 L\n.53048 .05365 L\n.53238 .05336 L\n.53429 .05307 L\n.53619 .05278 L\n.5381 .05249 L\n.54 .0522 L\n.5419 .05192 L\n.54381 .05163 L\n.54571 .05135 L\n.54762 .05107 L\n.54952 .05078 L\n.55143 .0505 L\n.55333 .05022 L\n.55524 .04994 L\n.55714 .04967 L\n.55905 .04939 L\n.56095 .04911 L\n.56286 .04884 L\n.56476 .04856 L\n.56667 .04829 L\n.56857 .04802 L\n.57048 .04775 L\n.57238 .04748 L\n.57429 .04721 L\n.57619 .04694 L\n.5781 .04668 L\n.58 .04641 L\n.5819 .04615 L\n.58381 .04588 L\n.58571 .04562 L\n.58762 .04536 L\n.58952 .0451 L\n.59143 .04484 L\nMistroke\n.59333 .04458 L\n.59524 .04432 L\n.59714 .04407 L\n.59905 .04381 L\n.60095 .04356 L\n.60286 .0433 L\n.60476 .04305 L\n.60667 .0428 L\n.60857 .04255 L\n.61048 .0423 L\n.61238 .04205 L\n.61429 .0418 L\n.61619 .04156 L\n.6181 .04131 L\n.62 .04107 L\n.6219 .04083 L\n.62381 .04058 L\n.62571 .04034 L\n.62762 .0401 L\n.62952 .03986 L\n.63143 .03963 L\n.63333 .03939 L\n.63524 .03915 L\n.63714 .03892 L\n.63905 .03869 L\n.64095 .03845 L\n.64286 .03822 L\n.64476 .03799 L\n.64667 .03776 L\n.64857 .03753 L\n.65048 .03731 L\n.65238 .03708 L\n.65429 .03686 L\n.65619 .03663 L\n.6581 .03641 L\n.66 .03619 L\n.6619 .03597 L\n.66381 .03575 L\n.66571 .03553 L\n.66762 .03531 L\n.66952 .03509 L\n.67143 .03488 L\n.67333 .03466 L\n.67524 .03445 L\n.67714 .03424 L\n.67905 .03403 L\n.68095 .03382 L\n.68286 .03361 L\n.68476 .0334 L\n.68667 .03319 L\nMistroke\n.68857 .03299 L\n.69048 .03278 L\n.69238 .03258 L\n.69429 .03237 L\n.69619 .03217 L\n.6981 .03197 L\n.7 .03177 L\n.7019 .03157 L\n.70381 .03138 L\n.70571 .03118 L\n.70762 .03098 L\n.70952 .03079 L\n.71143 .0306 L\n.71333 .03041 L\n.71524 .03022 L\n.71714 .03003 L\n.71905 .02984 L\n.72095 .02965 L\n.72286 .02946 L\n.72476 .02928 L\n.72667 .02909 L\n.72857 .02891 L\n.73048 .02873 L\n.73238 .02855 L\n.73429 .02837 L\n.73619 .02819 L\n.7381 .02801 L\n.74 .02783 L\n.7419 .02766 L\n.74381 .02748 L\n.74571 .02731 L\n.74762 .02714 L\n.74952 .02697 L\n.75143 .0268 L\n.75333 .02663 L\n.75524 .02646 L\n.75714 .02629 L\n.75905 .02613 L\n.76095 .02596 L\n.76286 .0258 L\n.76476 .02564 L\n.76667 .02548 L\n.76857 .02531 L\n.77048 .02516 L\n.77238 .025 L\n.77429 .02484 L\n.77619 .02469 L\n.7781 .02453 L\n.78 .02438 L\n.7819 .02423 L\nMistroke\n.78381 .02407 L\n.78571 .02392 L\n.78762 .02377 L\n.78952 .02363 L\n.79143 .02348 L\n.79333 .02333 L\n.79524 .02319 L\n.79714 .02305 L\n.79905 .0229 L\n.80095 .02276 L\n.80286 .02262 L\n.80476 .02248 L\n.80667 .02235 L\n.80857 .02221 L\n.81048 .02207 L\n.81238 .02194 L\n.81429 .02181 L\n.81619 .02167 L\n.8181 .02154 L\n.82 .02141 L\n.8219 .02129 L\n.82381 .02116 L\n.82571 .02103 L\n.82762 .02091 L\n.82952 .02078 L\n.83143 .02066 L\n.83333 .02054 L\n.83524 .02042 L\n.83714 .0203 L\n.83905 .02018 L\n.84095 .02006 L\n.84286 .01994 L\n.84476 .01983 L\n.84667 .01972 L\n.84857 .0196 L\n.85048 .01949 L\n.85238 .01938 L\n.85429 .01927 L\n.85619 .01916 L\n.8581 .01906 L\n.86 .01895 L\n.8619 .01885 L\n.86381 .01874 L\n.86571 .01864 L\n.86762 .01854 L\n.86952 .01844 L\n.87143 .01834 L\n.87333 .01824 L\n.87524 .01815 L\n.87714 .01805 L\nMistroke\n.87905 .01796 L\n.88095 .01786 L\n.88286 .01777 L\n.88476 .01768 L\n.88667 .01759 L\n.88857 .0175 L\n.89048 .01742 L\n.89238 .01733 L\n.89429 .01724 L\n.89619 .01716 L\n.8981 .01708 L\n.9 .017 L\n.9019 .01692 L\n.90381 .01684 L\n.90571 .01676 L\n.90762 .01668 L\n.90952 .01661 L\n.91143 .01653 L\n.91333 .01646 L\n.91524 .01639 L\n.91714 .01632 L\n.91905 .01625 L\n.92095 .01618 L\n.92286 .01611 L\n.92476 .01605 L\n.92667 .01598 L\n.92857 .01592 L\n.93048 .01586 L\n.93238 .01579 L\n.93429 .01573 L\n.93619 .01567 L\n.9381 .01562 L\n.94 .01556 L\n.9419 .01551 L\n.94381 .01545 L\n.94571 .0154 L\n.94762 .01535 L\n.94952 .0153 L\n.95143 .01525 L\n.95333 .0152 L\n.95524 .01515 L\n.95714 .01511 L\n.95905 .01506 L\n.96095 .01502 L\n.96286 .01498 L\n.96476 .01494 L\n.96667 .0149 L\n.96857 .01486 L\n.97048 .01482 L\n.97238 .01478 L\nMistroke\n.97429 .01475 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(altitude) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n159.000 12.813 moveto\n(L) show\n165.000 12.813 moveto\n(@) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n177.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00FP3IfMT2000000@0fMWI0P00\n000[0=WIf@D000000P3IfMT2000000@0fMWI0P00000X0=WIf@D000000P3IfMT2000000@0fMWI0P00\n000W0=WIf@@000000`3IfMT2000000@0fMWI0P00000G0=WIf@00F@3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI000002`0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09`3IfMT01@000000fMWI0=WIf@3IfMT0\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0T0=WIf@030000003I\nfMT0fMWI00<0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000F0=WIf@00\nG03IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02X0fMWI00D000000=WIf@3I\nfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09`3IfMT0\n0`000000fMWI0=WIf@040=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09@3I\nfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00005P3IfMT005X0fMWI0P0000030=WIf@040000003IfMT0fMWI000002`0fMWI00D000000=WIf@3I\nfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09`3IfMT0\n0`000000fMWI0=WIf@020=WIf@8000000`3IfMT010000000fMWI0=WIf@00000X0=WIf@050000003I\nfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI01@0\nfMWI001J0=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT00000;03IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0W\n0=WIf@030000003IfMT0fMWI0080fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@00\n000V0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000\nfMWI0=WIf@00000F0=WIf@00FP3IfMT3000000<0fMWI0P00000/0=WIf@800000103IfMT2000000@0\nfMWI0P00000Y0=WIf@800000103IfMT3000000<0fMWI0P00000X0=WIf@800000103IfMT2000000@0\nfMWI0P00000G0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009`3IfMT00`000000fMWI0=WIf@3f\n0=WIf@009`3IfMT00`000000fMWI0=WIf@3W0=WIf@8000003@3IfMT002L0fMWI00<000000=WIf@3I\nfMT0iP3IfMT010000000fMWI0=WIf@00000<0=WIf@008P3IfMWJ000000L0003o1P0000070=WIf@03\n0000003IfMT0fMWI00d0fMWI000W0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0\n203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0\n203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0\n203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0\n203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0\n203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0\n203IfMT010000000fMWI0=WIf@3IfMTG0000o`H0fMWI00<000000=WIf@3IfMT02P3IfMT5000000`0\nfMWI000W0=WIf@030000003IfMT0fMWI0:`0fMWI3`000?l[0=WIf@030000003IfMT0fMWI00d0fMWI\n000W0=WIf@030000003IfMT0fMWI0:40fMWI30000?m90=WIf@009`3IfMT2000009P0fMWI2P000?mE\n0=WIf@009`3IfMT00`000000fMWI0=WIf@2>0=WIf@X0003oGP3IfMT002L0fMWI00<000000=WIf@3I\nfMT0QP3IfMT80000ofP0fMWI000W0=WIf@030000003IfMT0fMWI07h0fMWI20000?m`0=WIf@009`3I\nfMT00`000000fMWI0=WIf@1g0=WIf@P0003oM`3IfMT002L0fMWI0P00001b0=WIf@L0003oOP3IfMT0\n02L0fMWI00<000000=WIf@3IfMT0JP3IfMT70000ohD0fMWI000W0=WIf@030000003IfMT0fMWI06@0\nfMWI1`000?n;0=WIf@009`3IfMT00`000000fMWI0=WIf@1O0=WIf@H0003oT@3IfMT002L0fMWI00<0\n00000=WIf@3IfMT0F@3IfMT60000oiL0fMWI000W0=WIf@800000E@3IfMT50000oid0fMWI000W0=WI\nf@030000003IfMT0fMWI04h0fMWI1P000?nR0=WIf@009`3IfMT00`000000fMWI0=WIf@1:0=WIf@D0\n003oY`3IfMT002L0fMWI00<000000=WIf@3IfMT0A03IfMT60000oj`0fMWI000W0=WIf@800000@03I\nfMT60000ok40fMWI000W0=WIf@030000003IfMT0fMWI03/0fMWI1@000?nf0=WIf@009`3IfMT00`00\n0000fMWI0=WIf@0f0=WIf@H0003o^P3IfMT000P0fMWI1@0000020=WIf@800000103IfMT2000000@0\nfMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0b0=WIf@D0003o_`3IfMT000X0fMWI\n00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@050000003IfMT0fMWI0=WIf@000000\n;`3IfMT50000ol@0fMWI000:0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@3IfMT3000002X0fMWI10000?o80=WIf@002P3IfMT01@000000fMWI0=WIf@3I\nfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000[0=WIf@80003ob`3IfMT0\n00X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@050000003IfMT0fMWI0=WI\nf@000000:@3IfMT20000old0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@80\n0000103IfMT2000000@0fMWI00<000000=WIf@3IfMT09P3IfMT20000olh0fMWI000W0=WIf@800000\n9P3IfMT20000oll0fMWI000W0=WIf@030000003IfMT0fMWI02@0fMWI0P000?o@0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@0S0=WIf@80003od@3IfMT002L0fMWI00<000000=WIf@3IfMT08`3IfMT00`00\n0?l0fMWI0=WIf@3@0=WIf@009`3IfMT00`000000fMWI0=WIf@0R0=WIf@030000o`3IfMT0fMWI0=40\nfMWI000W0=WIf@8000008P3IfMT20000om<0fMWI000W0=WIf@030000003IfMT0fMWI0200fMWI0P00\n0?oD0=WIf@009`3IfMT00`000000fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI0=<0fMWI000W0=WI\nf@030000003IfMT0fMWI01l0fMWI0P000?oE0=WIf@009`3IfMT200000200fMWI00<0003o0=WIf@3I\nfMT0e03IfMT002L0fMWI00<000000=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@3E0=WIf@009`3I\nfMT00`000000fMWI0=WIf@0M0=WIf@80003oe`3IfMT002L0fMWI00<000000=WIf@3IfMT07@3IfMT0\n0`000?l0fMWI0=WIf@3F0=WIf@009`3IfMT00`000000fMWI0=WIf@0L0=WIf@030000o`3IfMT0fMWI\n0=L0fMWI000W0=WIf@8000007@3IfMT00`000?l0fMWI0=WIf@3G0=WIf@009`3IfMT00`000000fMWI\n0=WIf@0K0=WIf@80003of@3IfMT002L0fMWI00<000000=WIf@3IfMT06`3IfMT00`000?l0fMWI0=WI\nf@3H0=WIf@00203IfMT4000000<0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P000004\n0=WIf@030000003IfMT0fMWI01X0fMWI00<0003o0=WIf@3IfMT0f@3IfMT000P0fMWI00<000000=WI\nf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI01X0\nfMWI00<0003o0=WIf@3IfMT0f@3IfMT000T0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000030=WIf@<000006@3IfMT00`000?l0fMWI0=WIf@3J0=WIf@002P3IfMT0\n1@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000K\n0=WIf@030000o`3IfMT0fMWI0=X0fMWI00080=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@0I0=WIf@030000o`3IfMT0fMWI\n0=X0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0\nfMWI00<000000=WIf@3IfMT0603IfMT00`000?l0fMWI0=WIf@3K0=WIf@009`3IfMT2000001T0fMWI\n00<0003o0=WIf@3IfMT0f`3IfMT002L0fMWI00<000000=WIf@3IfMT05`3IfMT00`000?l0fMWI0=WI\nf@3L0=WIf@009`3IfMT00`000000fMWI0=WIf@0G0=WIf@030000o`3IfMT0fMWI0=`0fMWI000W0=WI\nf@030000003IfMT0fMWI01H0fMWI00<0003o0=WIf@3IfMT0g@3IfMT002L0fMWI00<000000=WIf@3I\nfMT05P3IfMT00`000?l0fMWI0=WIf@3M0=WIf@009`3IfMT2000001H0fMWI00<0003o0=WIf@3IfMT0\ngP3IfMT002L0fMWI00<000000=WIf@3IfMT05@3IfMT00`000?l0fMWI0=WIf@3N0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@0E0=WIf@030000o`3IfMT0fMWI0=h0fMWI000W0=WIf@030000003IfMT0fMWI\n01@0fMWI00<0003o0=WIf@3IfMT0g`3IfMT002L0fMWI0P00000E0=WIf@030000o`3IfMT0fMWI0=l0\nfMWI000W0=WIf@030000003IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT0h03IfMT002L0fMWI00<0\n00000=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@3P0=WIf@009`3IfMT00`000000fMWI0=WIf@0C\n0=WIf@030000o`3IfMT0fMWI0>00fMWI000W0=WIf@030000003IfMT0fMWI0180fMWI00<0003o0=WI\nf@3IfMT0h@3IfMT002L0fMWI0P00000C0=WIf@030000o`3IfMT0fMWI0>40fMWI000W0=WIf@030000\n003IfMT0fMWI0180fMWI00<0003o0=WIf@3IfMT0h@3IfMT000T0fMWI0P0000040=WIf@800000103I\nfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3I\nfMT0fMWI0>80fMWI00080=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000`3IfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI0>80fMWI000:0=WI\nf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003I\nfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@3IfMT30000\n0140fMWI00<0003o0=WIf@3IfMT0hP3IfMT000/0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT01@000000fMWI0=WIf@3IfMT000000180fMWI00<0003o0=WIf@3IfMT0h`3IfMT000P0fMWI\n00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000\n003IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0h`3IfMT000T0fMWI0P0000040=WIf@800000103I\nfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0?0=WIf@030000o`3I\nfMT0fMWI0>@0fMWI000W0=WIf@030000003IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT0i03IfMT0\n02L0fMWI0P00000@0=WIf@030000o`3IfMT0fMWI0>@0fMWI000W0=WIf@030000003IfMT0fMWI00h0\nfMWI00<0003o0=WIf@3IfMT0i@3IfMT002L0fMWI00<000000=WIf@3IfMT03P3IfMT00`000?l0fMWI\n0=WIf@3U0=WIf@009`3IfMT00`000000fMWI0=WIf@0>0=WIf@030000o`3IfMT0fMWI0>D0fMWI000W\n0=WIf@030000003IfMT0fMWI00h0fMWI00<0003o0=WIf@3IfMT0i@3IfMT002L0fMWI0P00000>0=WI\nf@030000o`3IfMT0fMWI0>H0fMWI000W0=WIf@030000003IfMT0fMWI00d0fMWI00<0003o0=WIf@3I\nfMT0iP3IfMT002L0fMWI00<000000=WIf@3IfMT0303IfMT00`000?l0fMWI0=WIf@3W0=WIf@009`3I\nfMT00`000000fMWI0=WIf@0<0=WIf@030000o`3IfMT0fMWI0>L0fMWI000W0=WIf@8000003@3IfMT0\n0`000?l0fMWI0=WIf@3W0=WIf@009`3IfMT00`000000fMWI0=WIf@0;0=WIf@030000o`3IfMT0fMWI\n0>P0fMWI000W0=WIf@030000003IfMT0fMWI00/0fMWI00<0003o0=WIf@3IfMT0j03IfMT002L0fMWI\n00<000000=WIf@3IfMT02`3IfMT00`000?l0fMWI0=WIf@3X0=WIf@009`3IfMT00`000000fMWI0=WI\nf@0;0=WIf@030000o`3IfMT0fMWI0>P0fMWI000W0=WIf@8000002`3IfMT00`000?l0fMWI0=WIf@3Y\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0:0=WIf@030000o`3IfMT0fMWI0>T0fMWI000:0=WIf@<0\n00000P3IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3I\nfMT02P3IfMT00`000?l0fMWI0=WIf@3Y0=WIf@002`3IfMT010000000fMWI0=WIf@0000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@050000003IfMT0fMWI0=WIf@0000002`3IfMT00`000?l0fMWI0=WIf@3Z0=WIf@00203I\nfMT5000000050=WIf@000000fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT3000000T0fMWI00<0003o\n0=WIf@3IfMT0jP3IfMT000P0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000\nfMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0jP3IfMT000T0\nfMWI00<000000=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000\n003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0j`3IfMT000X0fMWI0P0000030=WIf@800000103I\nfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@080=WIf@030000o`3I\nfMT0fMWI0>/0fMWI000W0=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0j`3IfMT0\n02L0fMWI0P0000080=WIf@030000o`3IfMT0fMWI0>`0fMWI000W0=WIf@030000003IfMT0fMWI00L0\nfMWI00<0003o0=WIf@3IfMT0k03IfMT002L0fMWI00<000000=WIf@3IfMT01`3IfMT00`000?l0fMWI\n0=WIf@3/0=WIf@009`3IfMT00`000000fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0>`0fMWI000W\n0=WIf@800000203IfMT00`000?l0fMWI0=WIf@3/0=WIf@009`3IfMT00`000000fMWI0=WIf@060=WI\nf@030000o`3IfMT0fMWI0>d0fMWI000W0=WIf@030000003IfMT0fMWI00H0fMWI00<0003o0=WIf@3I\nfMT0k@3IfMT002L0fMWI00<000000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3]0=WIf@009`3I\nfMT00`000000fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI0>h0fMWI000W0=WIf@8000001P3IfMT0\n0`000?l0fMWI0=WIf@3^0=WIf@009`3IfMT00`000000fMWI0=WIf@050=WIf@030000o`3IfMT0fMWI\n0>h0fMWI000W0=WIf@030000003IfMT0fMWI00@0fMWI00<0003o0=WIf@3IfMT0k`3IfMT002L0fMWI\n00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@3_0=WIf@009`3IfMT00`000000fMWI0=WI\nf@040=WIf@030000o`3IfMT0fMWI0>l0fMWI000W0=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@3_\n0=WIf@009`3IfMT00`000000fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI0>l0fMWI00090=WIf@80\n0000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3I\nfMT00`3IfMT00`000?l0fMWI0=WIf@3`0=WIf@00203IfMT010000000fMWI0=WIf@0000020=WIf@04\n0000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT00`3IfMT00`000?l0fMWI\n0=WIf@3`0=WIf@002`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI0=WIf@<000000`3IfMT00`000?l0fMWI0=WIf@3`0=WIf@002@3IfMT2000000<0fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@030=WIf@030000o`3IfMT0fMWI\n0?00fMWI00090=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0`3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI0?40fMWI00090=WIf@<000000`3I\nfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3IfMT00P3I\nfMT00`000?l0fMWI0=WIf@3a0=WIf@009`3IfMT00`000000fMWI0=WIf@020=WIf@030000o`3IfMT0\nfMWI0?40fMWI000W0=WIf@8000000P3IfMT00`000?l0fMWI0=WIf@3b0=WIf@009`3IfMT01@000000\nfMWI0=WIf@3IfMT0003o0?@0fMWI000W0=WIf@050000003IfMT0fMWI0=WIf@000?l0m03IfMT002L0\nfMWI00@000000=WIf@3IfMT0003om@3IfMT002L0fMWI0P0000000`3IfMT0003o0=WIf@3d0=WIf@00\n9`3IfMT010000000fMWI0=WIf@000?oe0=WIf@009`3IfMT010000000fMWI0=WIf@000?oe0=WIf@00\n9`3IfMT010000000fMWI0=WIf@000?oe0=WIf@009`3IfMT010000000fMWI0=WIf@000?oe0=WIf@00\n9`3IfMT2000000030000o`3IfMT0fMWI0?@0fMWI000W0=WIf@030000003IfMT0003o0?H0fMWI000W\n0=WIf@030000003IfMT0003o0?H0fMWI000W0=WIf@03000000000?l0fMWI0?H0fMWI000W0=WIf@03\n000000000?l0fMWI0?H0fMWI000W0=WIf@03000000000?l0fMWI0?H0fMWI000W0=WIf@0300000000\n0?l0fMWI0?H0fMWI00080=WIf@<00000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2\n000000@0fMWI00<0003o0=WIf@3IfMT0mP3IfMT000P0fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000o`3IfMT0fMWI0?H0fMWI00080=WIf@04\n0000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT300000?H0\nfMWI00080=WIf@<000000`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003I\nfMT0fMWI0?H0fMWI00090=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3I\nfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3I\nfMT000000`3IfMT00`000000fMWI0=WIf@3f0=WIf@002@3IfMT3000000<0fMWI0P0000040=WIf@80\n0000103IfMT2000000@0fMWI0P00003m0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@009@3IfMT00`000000fMWI0=WIf@050=WIf@<00000DP3IfMT00`000000fMWI0=WIf@050=WI\nf@800000203IfMT2000008X0fMWI000T0=WIf@030000003IfMT0fMWI00L0fMWI00<000000=WIf@3I\nfMT0DP3IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3I\nfMT0R03IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT5000000040=WIf@0000000000000000@0\nfMWI0P0000030=WIf@<000000`3IfMT3000000@0fMWI0P0000040=WIf@8000000`3IfMT500000004\n0=WIf@0000000000000000@0fMWI0P0000040=WIf@<000000`3IfMT2000000@0fMWI100000020=WI\nf@@000000P3IfMT3000000@0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@020=WI\nf@800000103IfMT00`000000fMWI0=WIf@280=WIf@008`3IfMT00`000000fMWI0=WIf@020=WIf@@0\n00000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<00000\n0=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000000P0fMWI1000\n00030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT00000103IfMT01@000000fMWI\n0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI000000L0fMWI00<000000=WIf@3IfMT00`3IfMT01@000000fMWI0=WI\nf@3IfMT000000080fMWI00D000000=WIf@3IfMT0fMWI0000002:0=WIf@005P3IfMT80000o`D0fMWI\n00<000000=WIf@3IfMT01@3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00D000000=WIf@3IfMT0fMWI000000050=WIf@030000003IfMT0fMWI0080fMWI00@00000\n0=WIf@3IfMT000002`3IfMT01@000000fMWI0=WIf@3IfMT0000000@0fMWI00<000000=WIf@3IfMT0\n1@3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@@00000103IfMT00`000000fMWI0=WIf@030=WIf@050000003IfMT0\nfMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@280=WIf@008`3IfMT00`000000fMWI0=WIf@03\n0=WIf@<0000000<0fMWI0000000000000P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@03\n0=WIf@030000003IfMT0fMWI00<0fMWI0P00000:0=WIf@<000000`3IfMT010000000fMWI0=WIf@3I\nfMT5000000<0fMWI0P0000020=WIf@D0000000@0fMWI000000000000fMWI0P0000030=WIf@<00000\n0`3IfMT2000000D0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@05000000<0fMWI\n00<000000=WIf@3IfMT0R03IfMT002@0fMWI00<000000=WIf@3IfMT0503IfMT00`000000fMWI0=WI\nf@030=WIf@030000003IfMT0fMWI01D0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WI\nf@090=WIf@030000003IfMT0fMWI00`0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WI\nf@040=WIf@050000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@280=WIf@009@3I\nfMT00`000000fMWI0=WIf@0B0=WIf@800000103IfMT2000001H0fMWI0P00000<0=WIf@030000003I\nfMT0fMWI00l0fMWI0P0000080=WIf@030000003IfMT0fMWI00D0fMWI0P0000080=WIf@800000RP3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3I\nfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00001"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-36.4396`, -10757, 0.913881`, 437.521`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[TextData[List["\t\t\t\t\t\t", StyleBox["Diagram 1:", Rule[FontWeight, "Bold"]], " ", StyleBox["Altitude", Rule[FontFamily, "Courier"]], " of the Apollo rocket over the lunar surface\n"]], "Text"], Cell["plot( apollo.thrust );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0047619 -0.00725551 1.67971e-005 [\n[.2619 .06423 -6 -9 ]\n[.2619 .06423 6 0 ]\n[.5 .06423 -9 -9 ]\n[.5 .06423 9 0 ]\n[.7381 .06423 -9 -9 ]\n[.7381 .06423 9 0 ]\n[.97619 .06423 -9 -9 ]\n[.97619 .06423 9 0 ]\n[1.025 .07673 0 -6.28125 ]\n[1.025 .07673 10 6.28125 ]\n[.01131 .16072 -30 -4.5 ]\n[.01131 .16072 0 4.5 ]\n[.01131 .2447 -30 -4.5 ]\n[.01131 .2447 0 4.5 ]\n[.01131 .32869 -30 -4.5 ]\n[.01131 .32869 0 4.5 ]\n[.01131 .41267 -30 -4.5 ]\n[.01131 .41267 0 4.5 ]\n[.01131 .49666 -30 -4.5 ]\n[.01131 .49666 0 4.5 ]\n[.01131 .58064 -30 -4.5 ]\n[.01131 .58064 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.2619 .07673 m\n.2619 .08298 L\ns\n[(50)] .2619 .06423 0 1 Mshowa\n.5 .07673 m\n.5 .08298 L\ns\n[(100)] .5 .06423 0 1 Mshowa\n.7381 .07673 m\n.7381 .08298 L\ns\n[(150)] .7381 .06423 0 1 Mshowa\n.97619 .07673 m\n.97619 .08298 L\ns\n[(200)] .97619 .06423 0 1 Mshowa\n.125 Mabswid\n.07143 .07673 m\n.07143 .08048 L\ns\n.11905 .07673 m\n.11905 .08048 L\ns\n.16667 .07673 m\n.16667 .08048 L\ns\n.21429 .07673 m\n.21429 .08048 L\ns\n.30952 .07673 m\n.30952 .08048 L\ns\n.35714 .07673 m\n.35714 .08048 L\ns\n.40476 .07673 m\n.40476 .08048 L\ns\n.45238 .07673 m\n.45238 .08048 L\ns\n.54762 .07673 m\n.54762 .08048 L\ns\n.59524 .07673 m\n.59524 .08048 L\ns\n.64286 .07673 m\n.64286 .08048 L\ns\n.69048 .07673 m\n.69048 .08048 L\ns\n.78571 .07673 m\n.78571 .08048 L\ns\n.83333 .07673 m\n.83333 .08048 L\ns\n.88095 .07673 m\n.88095 .08048 L\ns\n.92857 .07673 m\n.92857 .08048 L\ns\n.25 Mabswid\n0 .07673 m\n1 .07673 L\ns\ngsave\n1.025 .07673 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .16072 m\n.03006 .16072 L\ns\n[(10000)] .01131 .16072 1 0 Mshowa\n.02381 .2447 m\n.03006 .2447 L\ns\n[(15000)] .01131 .2447 1 0 Mshowa\n.02381 .32869 m\n.03006 .32869 L\ns\n[(20000)] .01131 .32869 1 0 Mshowa\n.02381 .41267 m\n.03006 .41267 L\ns\n[(25000)] .01131 .41267 1 0 Mshowa\n.02381 .49666 m\n.03006 .49666 L\ns\n[(30000)] .01131 .49666 1 0 Mshowa\n.02381 .58064 m\n.03006 .58064 L\ns\n[(35000)] .01131 .58064 1 0 Mshowa\n.125 Mabswid\n.02381 .09353 m\n.02756 .09353 L\ns\n.02381 .11032 m\n.02756 .11032 L\ns\n.02381 .12712 m\n.02756 .12712 L\ns\n.02381 .14392 m\n.02756 .14392 L\ns\n.02381 .17751 m\n.02756 .17751 L\ns\n.02381 .19431 m\n.02756 .19431 L\ns\n.02381 .21111 m\n.02756 .21111 L\ns\n.02381 .2279 m\n.02756 .2279 L\ns\n.02381 .2615 m\n.02756 .2615 L\ns\n.02381 .2783 m\n.02756 .2783 L\ns\n.02381 .29509 m\n.02756 .29509 L\ns\n.02381 .31189 m\n.02756 .31189 L\ns\n.02381 .34548 m\n.02756 .34548 L\ns\n.02381 .36228 m\n.02756 .36228 L\ns\n.02381 .37908 m\n.02756 .37908 L\ns\n.02381 .39587 m\n.02756 .39587 L\ns\n.02381 .42947 m\n.02756 .42947 L\ns\n.02381 .44627 m\n.02756 .44627 L\ns\n.02381 .46306 m\n.02756 .46306 L\ns\n.02381 .47986 m\n.02756 .47986 L\ns\n.02381 .51345 m\n.02756 .51345 L\ns\n.02381 .53025 m\n.02756 .53025 L\ns\n.02381 .54705 m\n.02756 .54705 L\ns\n.02381 .56385 m\n.02756 .56385 L\ns\n.02381 .05993 m\n.02756 .05993 L\ns\n.02381 .04314 m\n.02756 .04314 L\ns\n.02381 .02634 m\n.02756 .02634 L\ns\n.02381 .00954 m\n.02756 .00954 L\ns\n.02381 .59744 m\n.02756 .59744 L\ns\n.02381 .61424 m\n.02756 .61424 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02571 .60332 L\n.02762 .60332 L\n.02952 .60332 L\n.03143 .60332 L\n.03333 .60332 L\n.03524 .60332 L\n.03714 .60332 L\n.03905 .60332 L\n.04095 .60332 L\n.04286 .60332 L\n.04476 .60332 L\n.04667 .60332 L\n.04857 .60332 L\n.05048 .60332 L\n.05238 .60332 L\n.05429 .60332 L\n.05619 .60332 L\n.0581 .60332 L\n.06 .60332 L\n.0619 .60332 L\n.06381 .60332 L\n.06571 .60332 L\n.06762 .60332 L\n.06952 .60332 L\n.07143 .60332 L\n.07333 .60332 L\n.07524 .60332 L\n.07714 .60332 L\n.07905 .60332 L\n.08095 .60332 L\n.08286 .60332 L\n.08476 .60332 L\n.08667 .60332 L\n.08857 .60332 L\n.09048 .60332 L\n.09238 .60332 L\n.09429 .60332 L\n.09619 .60332 L\n.0981 .60332 L\n.1 .60332 L\n.1019 .60332 L\n.10381 .60332 L\n.10571 .60332 L\n.10762 .60332 L\n.10952 .60332 L\n.11143 .60332 L\n.11333 .60332 L\n.11524 .60332 L\n.11714 .60332 L\nMistroke\n.11905 .60332 L\n.12095 .60332 L\n.12286 .60332 L\n.12476 .60332 L\n.12667 .60332 L\n.12857 .60332 L\n.13048 .60332 L\n.13238 .60332 L\n.13429 .60332 L\n.13619 .60332 L\n.1381 .60332 L\n.14 .60332 L\n.1419 .60332 L\n.14381 .60332 L\n.14571 .60332 L\n.14762 .60332 L\n.14952 .60332 L\n.15143 .60332 L\n.15333 .60332 L\n.15524 .60332 L\n.15714 .60332 L\n.15905 .60332 L\n.16095 .60332 L\n.16286 .60332 L\n.16476 .60332 L\n.16667 .60332 L\n.16857 .60332 L\n.17048 .60332 L\n.17238 .60332 L\n.17429 .60332 L\n.17619 .60332 L\n.1781 .60332 L\n.18 .60332 L\n.1819 .60332 L\n.18381 .60332 L\n.18571 .60332 L\n.18762 .60332 L\n.18952 .60332 L\n.19143 .60332 L\n.19333 .60332 L\n.19524 .60332 L\n.19714 .60332 L\n.19905 .60332 L\n.20095 .60332 L\n.20286 .60332 L\n.20476 .60332 L\n.20667 .60332 L\n.20857 .60332 L\n.21048 .60332 L\n.21238 .60332 L\nMistroke\n.21429 .60332 L\n.21619 .60332 L\n.2181 .60332 L\n.22 .60332 L\n.2219 .60332 L\n.22381 .60332 L\n.22571 .60332 L\n.22762 .60332 L\n.22952 .60332 L\n.22952 .01472 L\n.23143 .01472 L\n.23333 .01472 L\n.23524 .01472 L\n.23714 .01472 L\n.23905 .01472 L\n.24095 .01472 L\n.24286 .01472 L\n.24476 .01472 L\n.24667 .01472 L\n.24857 .01472 L\n.25048 .01472 L\n.25238 .01472 L\n.25429 .01472 L\n.25619 .01472 L\n.2581 .01472 L\n.26 .01472 L\n.2619 .01472 L\n.26381 .01472 L\n.26571 .01472 L\n.26762 .01472 L\n.26952 .01472 L\n.27143 .01472 L\n.27333 .01472 L\n.27524 .01472 L\n.27714 .01472 L\n.27905 .01472 L\n.28095 .01472 L\n.28286 .01472 L\n.28476 .01472 L\n.28667 .01472 L\n.28857 .01472 L\n.29048 .01472 L\n.29238 .01472 L\n.29429 .01472 L\n.29619 .01472 L\n.2981 .01472 L\n.3 .01472 L\n.3019 .01472 L\n.30381 .01472 L\n.30571 .01472 L\nMistroke\n.30762 .01472 L\n.30952 .01472 L\n.31143 .01472 L\n.31333 .01472 L\n.31524 .01472 L\n.31714 .01472 L\n.31905 .01472 L\n.32095 .01472 L\n.32286 .01472 L\n.32476 .01472 L\n.32667 .01472 L\n.32857 .01472 L\n.33048 .01472 L\n.33238 .01472 L\n.33429 .01472 L\n.33619 .01472 L\n.3381 .01472 L\n.34 .01472 L\n.3419 .01472 L\n.34381 .01472 L\n.34571 .01472 L\n.34762 .01472 L\n.34952 .01472 L\n.35143 .01472 L\n.35333 .01472 L\n.35524 .01472 L\n.35714 .01472 L\n.35905 .01472 L\n.36095 .01472 L\n.36286 .01472 L\n.36476 .01472 L\n.36667 .01472 L\n.36857 .01472 L\n.37048 .01472 L\n.37238 .01472 L\n.37429 .01472 L\n.37619 .01472 L\n.3781 .01472 L\n.38 .01472 L\n.3819 .01472 L\n.38381 .01472 L\n.38571 .01472 L\n.38762 .01472 L\n.38952 .01472 L\n.39143 .01472 L\n.39333 .01472 L\n.39524 .01472 L\n.39714 .01472 L\n.39905 .01472 L\n.40095 .01472 L\nMistroke\n.40286 .01472 L\n.40476 .01472 L\n.40667 .01472 L\n.40857 .01472 L\n.41048 .01472 L\n.41238 .01472 L\n.41429 .01472 L\n.41619 .01472 L\n.4181 .01472 L\n.42 .01472 L\n.4219 .01472 L\n.42381 .01472 L\n.42571 .01472 L\n.42762 .01472 L\n.42952 .01472 L\n.43143 .01472 L\n.43333 .01472 L\n.43524 .01472 L\n.43714 .01472 L\n.43905 .01472 L\n.44095 .01472 L\n.44286 .01472 L\n.44476 .01472 L\n.44667 .01472 L\n.44857 .01472 L\n.45048 .01472 L\n.45238 .01472 L\n.45429 .01472 L\n.45619 .01472 L\n.4581 .01472 L\n.46 .01472 L\n.4619 .01472 L\n.46381 .01472 L\n.46571 .01472 L\n.46762 .01472 L\n.46952 .01472 L\n.47143 .01472 L\n.47333 .01472 L\n.47524 .01472 L\n.47714 .01472 L\n.47905 .01472 L\n.48095 .01472 L\n.48286 .01472 L\n.48476 .01472 L\n.48667 .01472 L\n.48857 .01472 L\n.49048 .01472 L\n.49238 .01472 L\n.49429 .01472 L\n.49619 .01472 L\nMistroke\n.4981 .01472 L\n.5 .01472 L\n.5019 .01472 L\n.50381 .01472 L\n.50571 .01472 L\n.50762 .01472 L\n.50952 .01472 L\n.51143 .01472 L\n.51333 .01472 L\n.51524 .01472 L\n.51714 .01472 L\n.51905 .01472 L\n.52095 .01472 L\n.52286 .01472 L\n.52476 .01472 L\n.52667 .01472 L\n.52857 .01472 L\n.53048 .01472 L\n.53238 .01472 L\n.53429 .01472 L\n.53619 .01472 L\n.5381 .01472 L\n.54 .01472 L\n.5419 .01472 L\n.54381 .01472 L\n.54571 .01472 L\n.54762 .01472 L\n.54952 .01472 L\n.55143 .01472 L\n.55333 .01472 L\n.55524 .01472 L\n.55714 .01472 L\n.55905 .01472 L\n.56095 .01472 L\n.56286 .01472 L\n.56476 .01472 L\n.56667 .01472 L\n.56857 .01472 L\n.57048 .01472 L\n.57238 .01472 L\n.57429 .01472 L\n.57619 .01472 L\n.5781 .01472 L\n.58 .01472 L\n.5819 .01472 L\n.58381 .01472 L\n.58571 .01472 L\n.58762 .01472 L\n.58952 .01472 L\n.59143 .01472 L\nMistroke\n.59333 .01472 L\n.59524 .01472 L\n.59714 .01472 L\n.59905 .01472 L\n.60095 .01472 L\n.60286 .01472 L\n.60476 .01472 L\n.60667 .01472 L\n.60857 .01472 L\n.61048 .01472 L\n.61238 .01472 L\n.61429 .01472 L\n.61619 .01472 L\n.6181 .01472 L\n.62 .01472 L\n.6219 .01472 L\n.62381 .01472 L\n.62571 .01472 L\n.62762 .01472 L\n.62952 .01472 L\n.63143 .01472 L\n.63333 .01472 L\n.63524 .01472 L\n.63714 .01472 L\n.63905 .01472 L\n.64095 .01472 L\n.64286 .01472 L\n.64476 .01472 L\n.64667 .01472 L\n.64857 .01472 L\n.65048 .01472 L\n.65238 .01472 L\n.65429 .01472 L\n.65619 .01472 L\n.6581 .01472 L\n.66 .01472 L\n.6619 .01472 L\n.66381 .01472 L\n.66571 .01472 L\n.66762 .01472 L\n.66952 .01472 L\n.67143 .01472 L\n.67333 .01472 L\n.67524 .01472 L\n.67714 .01472 L\n.67905 .01472 L\n.68095 .01472 L\n.68286 .01472 L\n.68476 .01472 L\n.68667 .01472 L\nMistroke\n.68857 .01472 L\n.69048 .01472 L\n.69238 .01472 L\n.69429 .01472 L\n.69619 .01472 L\n.6981 .01472 L\n.7 .01472 L\n.7019 .01472 L\n.70381 .01472 L\n.70571 .01472 L\n.70762 .01472 L\n.70952 .01472 L\n.71143 .01472 L\n.71333 .01472 L\n.71524 .01472 L\n.71714 .01472 L\n.71905 .01472 L\n.72095 .01472 L\n.72286 .01472 L\n.72476 .01472 L\n.72667 .01472 L\n.72857 .01472 L\n.73048 .01472 L\n.73238 .01472 L\n.73429 .01472 L\n.73619 .01472 L\n.7381 .01472 L\n.74 .01472 L\n.7419 .01472 L\n.74381 .01472 L\n.74571 .01472 L\n.74762 .01472 L\n.74952 .01472 L\n.75143 .01472 L\n.75333 .01472 L\n.75524 .01472 L\n.75714 .01472 L\n.75905 .01472 L\n.76095 .01472 L\n.76286 .01472 L\n.76476 .01472 L\n.76667 .01472 L\n.76857 .01472 L\n.77048 .01472 L\n.77238 .01472 L\n.77429 .01472 L\n.77619 .01472 L\n.7781 .01472 L\n.78 .01472 L\n.7819 .01472 L\nMistroke\n.78381 .01472 L\n.78571 .01472 L\n.78762 .01472 L\n.78952 .01472 L\n.79143 .01472 L\n.79333 .01472 L\n.79524 .01472 L\n.79714 .01472 L\n.79905 .01472 L\n.80095 .01472 L\n.80286 .01472 L\n.80476 .01472 L\n.80667 .01472 L\n.80857 .01472 L\n.81048 .01472 L\n.81238 .01472 L\n.81429 .01472 L\n.81619 .01472 L\n.8181 .01472 L\n.82 .01472 L\n.8219 .01472 L\n.82381 .01472 L\n.82571 .01472 L\n.82762 .01472 L\n.82952 .01472 L\n.83143 .01472 L\n.83333 .01472 L\n.83524 .01472 L\n.83714 .01472 L\n.83905 .01472 L\n.84095 .01472 L\n.84286 .01472 L\n.84476 .01472 L\n.84667 .01472 L\n.84857 .01472 L\n.85048 .01472 L\n.85238 .01472 L\n.85429 .01472 L\n.85619 .01472 L\n.8581 .01472 L\n.86 .01472 L\n.8619 .01472 L\n.86381 .01472 L\n.86571 .01472 L\n.86762 .01472 L\n.86952 .01472 L\n.87143 .01472 L\n.87333 .01472 L\n.87524 .01472 L\n.87714 .01472 L\nMistroke\n.87905 .01472 L\n.88095 .01472 L\n.88286 .01472 L\n.88476 .01472 L\n.88667 .01472 L\n.88857 .01472 L\n.89048 .01472 L\n.89238 .01472 L\n.89429 .01472 L\n.89619 .01472 L\n.8981 .01472 L\n.9 .01472 L\n.9019 .01472 L\n.90381 .01472 L\n.90571 .01472 L\n.90762 .01472 L\n.90952 .01472 L\n.91143 .01472 L\n.91333 .01472 L\n.91524 .01472 L\n.91714 .01472 L\n.91905 .01472 L\n.92095 .01472 L\n.92286 .01472 L\n.92476 .01472 L\n.92667 .01472 L\n.92857 .01472 L\n.93048 .01472 L\n.93238 .01472 L\n.93429 .01472 L\n.93619 .01472 L\n.9381 .01472 L\n.94 .01472 L\n.9419 .01472 L\n.94381 .01472 L\n.94571 .01472 L\n.94762 .01472 L\n.94952 .01472 L\n.95143 .01472 L\n.95333 .01472 L\n.95524 .01472 L\n.95714 .01472 L\n.95905 .01472 L\n.96095 .01472 L\n.96286 .01472 L\n.96476 .01472 L\n.96667 .01472 L\n.96857 .01472 L\n.97048 .01472 L\n.97238 .01472 L\nMistroke\n.97429 .01472 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(thrust) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n147.000 12.813 moveto\n(L) show\n153.000 12.813 moveto\n(@) show\n159.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n165.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@009`3IfMT00`000000fMWI0=WI\nf@3f0=WIf@009`3IfMT00`000000fMWI0=WIf@3f0=WIf@009`3IfMT200000?L0fMWI000W0=WIf@03\n0000003IfMT0fMWI02d0fMWI[0000?lM0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000\no`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3I\nfMT002L0fMWI0P00000^0=WIf@030000o`3IfMT0fMWI00800000103IfMT2000002/0fMWI1@000002\n0=WIf@800000103IfMT2000002P0fMWI1@0000020=WIf@800000103IfMT2000002L0fMWI10000003\n0=WIf@800000103IfMT2000001L0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WI\nf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02X0fMWI00D00000\n0=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0\n9`3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`00\n0000fMWI0=WIf@0T0=WIf@030000003IfMT0fMWI00<0fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@00000F0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0\nfMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0Z0=WIf@050000003I\nfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02L0\nfMWI00<000000=WIf@3IfMT0103IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI\n02D0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI000001H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT00P000003\n0=WIf@040000003IfMT0fMWI000002`0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000\n003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09`3IfMT00`000000fMWI0=WIf@020=WIf@80\n00000`3IfMT010000000fMWI0=WIf@00000X0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI01@0fMWI000W0=WIf@800000;P3IfMT0\n10000?l0fMWI0=WIf@0000040=WIf@040000003IfMT0fMWI000002`0fMWI00D000000=WIf@3IfMT0\nfMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT09`3IfMT00`00\n0000fMWI0=WIf@020=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000009P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00005P3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@03000000<0fMWI\n0P00000/0=WIf@800000103IfMT2000000@0fMWI0P00000Y0=WIf@800000103IfMT3000000<0fMWI\n0P00000X0=WIf@800000103IfMT2000000@0fMWI0P00000G0=WIf@009`3IfMT00`000000fMWI0=WI\nf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@36\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@03\n0000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0\n;@3IfMT00`000?l0fMWI0=WIf@2g0=WIf@8000003@3IfMT00280fMWI=@0000000`000?l000000000\n002_000000L0fMWI00@000000=WIf@3IfMT00000303IfMT002L0fMWI00<000000=WIf@3IfMT0203I\nfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203I\nfMT01@000000fMWI0=WIf@3IfMT0003o00H0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@070=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI\n0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI\n0=WIf@0=0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI00@0fMWI00<0\n00000=WIf@3IfMT0=03IfMT00`000000fMWI0=WIf@0d0=WIf@030000003IfMT0fMWI03<0fMWI00<0\n00000=WIf@3IfMT02P3IfMT5000000`0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o\n0=WIf@3IfMT0]P3IfMT00`000000fMWI0=WIf@0=0=WIf@009`3IfMT2000002h0fMWI00<0003o0=WI\nf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00\n9`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003I\nfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI0P00000^0=WIf@030000o`3IfMT0\nfMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0\nfMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI\n0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@800000;P3IfMT00`000?l0fMWI0=WI\nf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WI\nf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3I\nfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0\naP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00203IfMT5\n00000080fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0\nfMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI00000002\n0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI\n0=WIf@0000020=WIf@050000003IfMT0fMWI0=WIf@000000;`3IfMT00`000?l0fMWI0=WIf@360=WI\nf@002P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\nfMWI0`00000]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000:0=WIf@050000003IfMT0fMWI0=WIf@00\n00000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000002l0fMWI00<0003o0=WIf@3IfMT0\naP3IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@050000003IfMT0\nfMWI0=WIf@000000;`3IfMT00`000?l0fMWI0=WIf@360=WIf@002@3IfMT2000000@0fMWI0P000004\n0=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@36\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@03\n0000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0\n;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3I\nfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`00\n0000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0\nfMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI0P00000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI\n000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<00000\n0=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WI\nf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@360=WIf@00\n9`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI00080=WIf@D000000P3I\nfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3IfMT0;@3I\nfMT00`000?l0fMWI0=WIf@360=WIf@002P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00D000000=WIf@3IfMT0fMWI0000000_0=WIf@030000o`3IfMT0fMWI0<H0fMWI000:0=WI\nf@030000003IfMT0fMWI00@0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI0`00000]0=WIf@03\n0000o`3IfMT0fMWI0<H0fMWI000:0=WIf@030000003IfMT0fMWI0080fMWI0P0000030=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000003\n0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT000X0fMWI00<000000=WI\nf@3IfMT00P3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI02d0fMWI\n00<0003o0=WIf@3IfMT0aP3IfMT000T0fMWI0P0000040=WIf@<000000`3IfMT2000000@0fMWI0P00\n00040=WIf@800000103IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W\n0=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WI\nf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3I\nfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3I\nfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT0\n0`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI\n0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI\n0P00000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0\n003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@36\n0=WIf@009`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3I\nfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00203IfMT4000000<0fMWI0P0000040=WIf@800000\n103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0\naP3IfMT000P0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000003\n0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT000T0fMWI00<000000=WI\nf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@<00000;@3IfMT00`000?l0\nfMWI0=WIf@360=WIf@002P3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3I\nfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00D0\n00000=WIf@3IfMT0fMWI0000000_0=WIf@030000o`3IfMT0fMWI0<H0fMWI00080=WIf@040000003I\nfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WI\nf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P000004\n0=WIf@800000103IfMT2000000@0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@36\n0=WIf@009`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3I\nfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000\no`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3I\nfMT002L0fMWI0P00000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI\n02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0\nfMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI\n000W0=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]\n0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WI\nf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00\n9`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3I\nfMT00`000?l0fMWI0=WIf@360=WIf@00203IfMT4000000<0fMWI0P0000040=WIf@800000103IfMT2\n000000@0fMWI0P0000040=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT0\n00P0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@03\n0000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT000T0fMWI00<000000=WIf@3IfMT0\n1@3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000P3IfMT010000000fMWI0=WIf@3IfMT3000002d0fMWI00<0003o0=WIf@3IfMT0aP3I\nfMT000X0fMWI00<000000=WIf@3IfMT00P3IfMT2000000<0fMWI00@000000=WIf@3IfMT000000P3I\nfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3I\nfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00203IfMT010000000fMWI0=WIf@0000030=WIf@03\n0000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WI\nf@360=WIf@002@3IfMT2000000@0fMWI0`0000030=WIf@800000103IfMT2000000@0fMWI0P000004\n0=WIf@800000;P3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WI\nf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3I\nfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3I\nfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT0\n0`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI\n0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI\n0P00000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0\n003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@36\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@80\n0000;P3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000\no`3IfMT0fMWI0<H0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103I\nfMT2000000@0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@00203IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<00000\n0=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@002P3IfMT01@000000fMWI0=WIf@3IfMT0\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI0`00000]0=WIf@030000o`3IfMT0fMWI\n0<H0fMWI000;0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0\nfMWI0000000_0=WIf@030000o`3IfMT0fMWI0<H0fMWI00080=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000\no`3IfMT0fMWI0<H0fMWI00090=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@800000103I\nfMT2000000@0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT2\n000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`00\n0?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0\nfMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI0P00\n000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o\n0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WI\nf@009`3IfMT2000002h0fMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI00<000000=WIf@3IfMT0\n;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3I\nfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0fMWI00<0003o0=WIf@3IfMT0aP3IfMT0\n02L0fMWI0P00000^0=WIf@030000o`3IfMT0fMWI0<H0fMWI000W0=WIf@030000003IfMT0fMWI02d0\nfMWI00<0003o0=WIf@3IfMT0aP3IfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI\n0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI\n00080=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3I\nfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI000:0=WIf@030000003IfMT0\nfMWI00@0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI0`00000]0=WIf@030000o`3IfMT0fMWI\n0<H0fMWI000;0=WIf@040000003IfMT0fMWI0=WIf@8000000`3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI\n0=WIf@0]0=WIf@030000o`3IfMT0fMWI0<H0fMWI00080=WIf@040000003IfMT0fMWI000000<0fMWI\n00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@0]0=WIf@030000o`3IfMT0\nfMWI0<H0fMWI00090=WIf@800000103IfMT3000000<0fMWI0P0000040=WIf@800000103IfMT20000\n00@0fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@360=WIf@009`3IfMT2000002h0\nfMWI00<0003o0=WIf@3IfMT0aP3IfMT002L0fMWI<@000?o80=WIf@009`3IfMT00`000000fMWI0=WI\nf@3f0=WIf@009`3IfMT00`000000fMWI0=WIf@3f0=WIf@009`3IfMT200000?L0fMWI000W0=WIf@03\n0000003IfMT0fMWI0?H0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0\nfMWI00D0fMWI0`0000160=WIf@030000003IfMT0fMWI00D0fMWI0P0000080=WIf@800000UP3IfMT0\n02@0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WIf@160=WIf@030000003IfMT0fMWI\n00@0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@2D0=WIf@008`3IfMT00`000000\nfMWI0=WIf@020=WIf@D0000000@0fMWI0000000000000000103IfMT2000000<0fMWI0`0000030=WI\nf@<00000103IfMT2000000@0fMWI0P0000040=WIf@8000000P3IfMT3000000040=WIf@0000000000\n0=WIf@@000000`3IfMT4000000040=WIf@0000000000000000@0fMWI0P0000050=WIf@030000003I\nfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3IfMT0U03I\nfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT400000080fMWI00@000000=WIf@3IfMT000000P3I\nfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0\n0P3IfMT010000000fMWI0=WIf@0000080=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WI\nf@3IfMT000000`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI000000D0fMWI00@0\n00000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0fMWI00<0fMWI\n00D000000=WIf@3IfMT0fMWI000000020=WIf@050000003IfMT0fMWI0=WIf@000000UP3IfMT001H0\nfMWI20000?l50=WIf@030000003IfMT0fMWI00D0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@050000003IfMT0fMWI0=WIf@0000001@3IfMT00`000000fMWI0=WI\nf@020=WIf@040000003IfMT0fMWI000000P0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI\n0=WIf@0000030=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT30000\n00<0fMWI00<000000=WIf@3IfMT01@3IfMT00`000000fMWI0=WIf@030=WIf@050000003IfMT0fMWI\n0=WIf@0000001P3IfMT00`000000fMWI0=WIf@2D0=WIf@008`3IfMT00`000000fMWI0=WIf@030=WI\nf@<0000000<0fMWI0000000000000P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@030=WI\nf@030000003IfMT0fMWI00<0fMWI0P0000080=WIf@D000000P3IfMT3000000<0fMWI10000000103I\nfMT000000000003IfMT200000080fMWI100000000`3IfMT0000000000003000000@0fMWI00<00000\n0=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@05000000<0fMWI00<000000=WIf@3IfMT0U03IfMT0\n02@0fMWI00<000000=WIf@3IfMT0503IfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI\n00h0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0E0=WIf@030000003IfMT0fMWI\n00@0fMWI00<000000=WIf@3IfMT0103IfMT01@000000fMWI0=WIf@3IfMT0000000H0fMWI00<00000\n0=WIf@3IfMT0U03IfMT002D0fMWI00<000000=WIf@3IfMT04P3IfMT2000000@0fMWI0P00000E0=WI\nf@8000007@3IfMT00`000000fMWI0=WIf@050=WIf@800000203IfMT2000009H0fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-36.4396`, -4350.79`, 0.913881`, 259.081`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[TextData[List[StyleBox["\t\t\t\t\tDiagram 2:", Rule[FontWeight, "Bold"]], " ", StyleBox["Thrust", Rule[FontFamily, "Courier"]], " from the rocket motor, with an initial high thrust f1 followed by a lower thrust f2.\n"]], "Text"], Cell["plot( apollo.mass );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0047619 -0.639451 0.00119686 [\n[.2619 .06617 -6 -9 ]\n[.2619 .06617 6 0 ]\n[.5 .06617 -9 -9 ]\n[.5 .06617 9 0 ]\n[.7381 .06617 -9 -9 ]\n[.7381 .06617 9 0 ]\n[.97619 .06617 -9 -9 ]\n[.97619 .06617 9 0 ]\n[1.025 .07867 0 -6.28125 ]\n[1.025 .07867 10 6.28125 ]\n[.01131 .19835 -18 -4.5 ]\n[.01131 .19835 0 4.5 ]\n[.01131 .31804 -18 -4.5 ]\n[.01131 .31804 0 4.5 ]\n[.01131 .43772 -18 -4.5 ]\n[.01131 .43772 0 4.5 ]\n[.01131 .55741 -24 -4.5 ]\n[.01131 .55741 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.2619 .07867 m\n.2619 .08492 L\ns\n[(50)] .2619 .06617 0 1 Mshowa\n.5 .07867 m\n.5 .08492 L\ns\n[(100)] .5 .06617 0 1 Mshowa\n.7381 .07867 m\n.7381 .08492 L\ns\n[(150)] .7381 .06617 0 1 Mshowa\n.97619 .07867 m\n.97619 .08492 L\ns\n[(200)] .97619 .06617 0 1 Mshowa\n.125 Mabswid\n.07143 .07867 m\n.07143 .08242 L\ns\n.11905 .07867 m\n.11905 .08242 L\ns\n.16667 .07867 m\n.16667 .08242 L\ns\n.21429 .07867 m\n.21429 .08242 L\ns\n.30952 .07867 m\n.30952 .08242 L\ns\n.35714 .07867 m\n.35714 .08242 L\ns\n.40476 .07867 m\n.40476 .08242 L\ns\n.45238 .07867 m\n.45238 .08242 L\ns\n.54762 .07867 m\n.54762 .08242 L\ns\n.59524 .07867 m\n.59524 .08242 L\ns\n.64286 .07867 m\n.64286 .08242 L\ns\n.69048 .07867 m\n.69048 .08242 L\ns\n.78571 .07867 m\n.78571 .08242 L\ns\n.83333 .07867 m\n.83333 .08242 L\ns\n.88095 .07867 m\n.88095 .08242 L\ns\n.92857 .07867 m\n.92857 .08242 L\ns\n.25 Mabswid\n0 .07867 m\n1 .07867 L\ns\ngsave\n1.025 .07867 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .19835 m\n.03006 .19835 L\ns\n[(700)] .01131 .19835 1 0 Mshowa\n.02381 .31804 m\n.03006 .31804 L\ns\n[(800)] .01131 .31804 1 0 Mshowa\n.02381 .43772 m\n.03006 .43772 L\ns\n[(900)] .01131 .43772 1 0 Mshowa\n.02381 .55741 m\n.03006 .55741 L\ns\n[(1000)] .01131 .55741 1 0 Mshowa\n.125 Mabswid\n.02381 .1026 m\n.02756 .1026 L\ns\n.02381 .12654 m\n.02756 .12654 L\ns\n.02381 .15048 m\n.02756 .15048 L\ns\n.02381 .17441 m\n.02756 .17441 L\ns\n.02381 .22229 m\n.02756 .22229 L\ns\n.02381 .24623 m\n.02756 .24623 L\ns\n.02381 .27016 m\n.02756 .27016 L\ns\n.02381 .2941 m\n.02756 .2941 L\ns\n.02381 .34197 m\n.02756 .34197 L\ns\n.02381 .36591 m\n.02756 .36591 L\ns\n.02381 .38985 m\n.02756 .38985 L\ns\n.02381 .41379 m\n.02756 .41379 L\ns\n.02381 .46166 m\n.02756 .46166 L\ns\n.02381 .4856 m\n.02756 .4856 L\ns\n.02381 .50954 m\n.02756 .50954 L\ns\n.02381 .53347 m\n.02756 .53347 L\ns\n.02381 .05473 m\n.02756 .05473 L\ns\n.02381 .03079 m\n.02756 .03079 L\ns\n.02381 .00685 m\n.02756 .00685 L\ns\n.02381 .58135 m\n.02756 .58135 L\ns\n.02381 .60528 m\n.02756 .60528 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .60332 m\n.02571 .5985 L\n.02762 .59368 L\n.02952 .58886 L\n.03143 .58404 L\n.03333 .57922 L\n.03524 .5744 L\n.03714 .56958 L\n.03905 .56476 L\n.04095 .55993 L\n.04286 .55511 L\n.04476 .55029 L\n.04667 .54547 L\n.04857 .54065 L\n.05048 .53583 L\n.05238 .53101 L\n.05429 .52619 L\n.05619 .52137 L\n.0581 .51655 L\n.06 .51173 L\n.0619 .50691 L\n.06381 .50209 L\n.06571 .49727 L\n.06762 .49245 L\n.06952 .48763 L\n.07143 .48281 L\n.07333 .47799 L\n.07524 .47317 L\n.07714 .46835 L\n.07905 .46353 L\n.08095 .45871 L\n.08286 .45388 L\n.08476 .44906 L\n.08667 .44424 L\n.08857 .43942 L\n.09048 .4346 L\n.09238 .42978 L\n.09429 .42496 L\n.09619 .42014 L\n.0981 .41532 L\n.1 .4105 L\n.1019 .40568 L\n.10381 .40086 L\n.10571 .39604 L\n.10762 .39122 L\n.10952 .3864 L\n.11143 .38158 L\n.11333 .37676 L\n.11524 .37194 L\n.11714 .36712 L\nMistroke\n.11905 .3623 L\n.12095 .35748 L\n.12286 .35266 L\n.12476 .34783 L\n.12667 .34301 L\n.12857 .33819 L\n.13048 .33337 L\n.13238 .32855 L\n.13429 .32373 L\n.13619 .31891 L\n.1381 .31409 L\n.14 .30927 L\n.1419 .30445 L\n.14381 .29963 L\n.14571 .29481 L\n.14762 .28999 L\n.14952 .28517 L\n.15143 .28035 L\n.15333 .27553 L\n.15524 .27071 L\n.15714 .26589 L\n.15905 .26107 L\n.16095 .25625 L\n.16286 .25143 L\n.16476 .24661 L\n.16667 .24178 L\n.16857 .23696 L\n.17048 .23214 L\n.17238 .22732 L\n.17429 .2225 L\n.17619 .21768 L\n.1781 .21286 L\n.18 .20804 L\n.1819 .20322 L\n.18381 .1984 L\n.18571 .19358 L\n.18762 .18876 L\n.18952 .18394 L\n.19143 .17912 L\n.19333 .1743 L\n.19524 .16948 L\n.19714 .16466 L\n.19905 .15984 L\n.20095 .15502 L\n.20286 .1502 L\n.20476 .14538 L\n.20667 .14056 L\n.20857 .13574 L\n.21048 .13091 L\n.21238 .12609 L\nMistroke\n.21429 .12127 L\n.21619 .11645 L\n.2181 .11163 L\n.22 .10681 L\n.2219 .10199 L\n.22381 .09717 L\n.22571 .09235 L\n.22762 .08753 L\n.22952 .08271 L\n.22952 .08271 L\n.23143 .08254 L\n.23333 .08236 L\n.23524 .08219 L\n.23714 .08202 L\n.23905 .08184 L\n.24095 .08167 L\n.24286 .0815 L\n.24476 .08132 L\n.24667 .08115 L\n.24857 .08098 L\n.25048 .0808 L\n.25238 .08063 L\n.25429 .08046 L\n.25619 .08028 L\n.2581 .08011 L\n.26 .07993 L\n.2619 .07976 L\n.26381 .07959 L\n.26571 .07941 L\n.26762 .07924 L\n.26952 .07907 L\n.27143 .07889 L\n.27333 .07872 L\n.27524 .07855 L\n.27714 .07837 L\n.27905 .0782 L\n.28095 .07803 L\n.28286 .07785 L\n.28476 .07768 L\n.28667 .07751 L\n.28857 .07733 L\n.29048 .07716 L\n.29238 .07699 L\n.29429 .07681 L\n.29619 .07664 L\n.2981 .07647 L\n.3 .07629 L\n.3019 .07612 L\n.30381 .07595 L\n.30571 .07577 L\nMistroke\n.30762 .0756 L\n.30952 .07542 L\n.31143 .07525 L\n.31333 .07508 L\n.31524 .0749 L\n.31714 .07473 L\n.31905 .07456 L\n.32095 .07438 L\n.32286 .07421 L\n.32476 .07404 L\n.32667 .07386 L\n.32857 .07369 L\n.33048 .07352 L\n.33238 .07334 L\n.33429 .07317 L\n.33619 .073 L\n.3381 .07282 L\n.34 .07265 L\n.3419 .07248 L\n.34381 .0723 L\n.34571 .07213 L\n.34762 .07196 L\n.34952 .07178 L\n.35143 .07161 L\n.35333 .07144 L\n.35524 .07126 L\n.35714 .07109 L\n.35905 .07092 L\n.36095 .07074 L\n.36286 .07057 L\n.36476 .07039 L\n.36667 .07022 L\n.36857 .07005 L\n.37048 .06987 L\n.37238 .0697 L\n.37429 .06953 L\n.37619 .06935 L\n.3781 .06918 L\n.38 .06901 L\n.3819 .06883 L\n.38381 .06866 L\n.38571 .06849 L\n.38762 .06831 L\n.38952 .06814 L\n.39143 .06797 L\n.39333 .06779 L\n.39524 .06762 L\n.39714 .06745 L\n.39905 .06727 L\n.40095 .0671 L\nMistroke\n.40286 .06693 L\n.40476 .06675 L\n.40667 .06658 L\n.40857 .06641 L\n.41048 .06623 L\n.41238 .06606 L\n.41429 .06588 L\n.41619 .06571 L\n.4181 .06554 L\n.42 .06536 L\n.4219 .06519 L\n.42381 .06502 L\n.42571 .06484 L\n.42762 .06467 L\n.42952 .0645 L\n.43143 .06432 L\n.43333 .06415 L\n.43524 .06398 L\n.43714 .0638 L\n.43905 .06363 L\n.44095 .06346 L\n.44286 .06328 L\n.44476 .06311 L\n.44667 .06294 L\n.44857 .06276 L\n.45048 .06259 L\n.45238 .06242 L\n.45429 .06224 L\n.45619 .06207 L\n.4581 .0619 L\n.46 .06172 L\n.4619 .06155 L\n.46381 .06137 L\n.46571 .0612 L\n.46762 .06103 L\n.46952 .06085 L\n.47143 .06068 L\n.47333 .06051 L\n.47524 .06033 L\n.47714 .06016 L\n.47905 .05999 L\n.48095 .05981 L\n.48286 .05964 L\n.48476 .05947 L\n.48667 .05929 L\n.48857 .05912 L\n.49048 .05895 L\n.49238 .05877 L\n.49429 .0586 L\n.49619 .05843 L\nMistroke\n.4981 .05825 L\n.5 .05808 L\n.5019 .05791 L\n.50381 .05773 L\n.50571 .05756 L\n.50762 .05739 L\n.50952 .05721 L\n.51143 .05704 L\n.51333 .05687 L\n.51524 .05669 L\n.51714 .05652 L\n.51905 .05634 L\n.52095 .05617 L\n.52286 .056 L\n.52476 .05582 L\n.52667 .05565 L\n.52857 .05548 L\n.53048 .0553 L\n.53238 .05513 L\n.53429 .05496 L\n.53619 .05478 L\n.5381 .05461 L\n.54 .05444 L\n.5419 .05426 L\n.54381 .05409 L\n.54571 .05392 L\n.54762 .05374 L\n.54952 .05357 L\n.55143 .0534 L\n.55333 .05322 L\n.55524 .05305 L\n.55714 .05288 L\n.55905 .0527 L\n.56095 .05253 L\n.56286 .05236 L\n.56476 .05218 L\n.56667 .05201 L\n.56857 .05183 L\n.57048 .05166 L\n.57238 .05149 L\n.57429 .05131 L\n.57619 .05114 L\n.5781 .05097 L\n.58 .05079 L\n.5819 .05062 L\n.58381 .05045 L\n.58571 .05027 L\n.58762 .0501 L\n.58952 .04993 L\n.59143 .04975 L\nMistroke\n.59333 .04958 L\n.59524 .04941 L\n.59714 .04923 L\n.59905 .04906 L\n.60095 .04889 L\n.60286 .04871 L\n.60476 .04854 L\n.60667 .04837 L\n.60857 .04819 L\n.61048 .04802 L\n.61238 .04785 L\n.61429 .04767 L\n.61619 .0475 L\n.6181 .04732 L\n.62 .04715 L\n.6219 .04698 L\n.62381 .0468 L\n.62571 .04663 L\n.62762 .04646 L\n.62952 .04628 L\n.63143 .04611 L\n.63333 .04594 L\n.63524 .04576 L\n.63714 .04559 L\n.63905 .04542 L\n.64095 .04524 L\n.64286 .04507 L\n.64476 .0449 L\n.64667 .04472 L\n.64857 .04455 L\n.65048 .04438 L\n.65238 .0442 L\n.65429 .04403 L\n.65619 .04386 L\n.6581 .04368 L\n.66 .04351 L\n.6619 .04334 L\n.66381 .04316 L\n.66571 .04299 L\n.66762 .04282 L\n.66952 .04264 L\n.67143 .04247 L\n.67333 .04229 L\n.67524 .04212 L\n.67714 .04195 L\n.67905 .04177 L\n.68095 .0416 L\n.68286 .04143 L\n.68476 .04125 L\n.68667 .04108 L\nMistroke\n.68857 .04091 L\n.69048 .04073 L\n.69238 .04056 L\n.69429 .04039 L\n.69619 .04021 L\n.6981 .04004 L\n.7 .03987 L\n.7019 .03969 L\n.70381 .03952 L\n.70571 .03935 L\n.70762 .03917 L\n.70952 .039 L\n.71143 .03883 L\n.71333 .03865 L\n.71524 .03848 L\n.71714 .03831 L\n.71905 .03813 L\n.72095 .03796 L\n.72286 .03778 L\n.72476 .03761 L\n.72667 .03744 L\n.72857 .03726 L\n.73048 .03709 L\n.73238 .03692 L\n.73429 .03674 L\n.73619 .03657 L\n.7381 .0364 L\n.74 .03622 L\n.7419 .03605 L\n.74381 .03588 L\n.74571 .0357 L\n.74762 .03553 L\n.74952 .03536 L\n.75143 .03518 L\n.75333 .03501 L\n.75524 .03484 L\n.75714 .03466 L\n.75905 .03449 L\n.76095 .03432 L\n.76286 .03414 L\n.76476 .03397 L\n.76667 .0338 L\n.76857 .03362 L\n.77048 .03345 L\n.77238 .03327 L\n.77429 .0331 L\n.77619 .03293 L\n.7781 .03275 L\n.78 .03258 L\n.7819 .03241 L\nMistroke\n.78381 .03223 L\n.78571 .03206 L\n.78762 .03189 L\n.78952 .03171 L\n.79143 .03154 L\n.79333 .03137 L\n.79524 .03119 L\n.79714 .03102 L\n.79905 .03085 L\n.80095 .03067 L\n.80286 .0305 L\n.80476 .03033 L\n.80667 .03015 L\n.80857 .02998 L\n.81048 .02981 L\n.81238 .02963 L\n.81429 .02946 L\n.81619 .02929 L\n.8181 .02911 L\n.82 .02894 L\n.8219 .02877 L\n.82381 .02859 L\n.82571 .02842 L\n.82762 .02824 L\n.82952 .02807 L\n.83143 .0279 L\n.83333 .02772 L\n.83524 .02755 L\n.83714 .02738 L\n.83905 .0272 L\n.84095 .02703 L\n.84286 .02686 L\n.84476 .02668 L\n.84667 .02651 L\n.84857 .02634 L\n.85048 .02616 L\n.85238 .02599 L\n.85429 .02582 L\n.85619 .02564 L\n.8581 .02547 L\n.86 .0253 L\n.8619 .02512 L\n.86381 .02495 L\n.86571 .02478 L\n.86762 .0246 L\n.86952 .02443 L\n.87143 .02426 L\n.87333 .02408 L\n.87524 .02391 L\n.87714 .02373 L\nMistroke\n.87905 .02356 L\n.88095 .02339 L\n.88286 .02321 L\n.88476 .02304 L\n.88667 .02287 L\n.88857 .02269 L\n.89048 .02252 L\n.89238 .02235 L\n.89429 .02217 L\n.89619 .022 L\n.8981 .02183 L\n.9 .02165 L\n.9019 .02148 L\n.90381 .02131 L\n.90571 .02113 L\n.90762 .02096 L\n.90952 .02079 L\n.91143 .02061 L\n.91333 .02044 L\n.91524 .02027 L\n.91714 .02009 L\n.91905 .01992 L\n.92095 .01975 L\n.92286 .01957 L\n.92476 .0194 L\n.92667 .01922 L\n.92857 .01905 L\n.93048 .01888 L\n.93238 .0187 L\n.93429 .01853 L\n.93619 .01836 L\n.9381 .01818 L\n.94 .01801 L\n.9419 .01784 L\n.94381 .01766 L\n.94571 .01749 L\n.94762 .01732 L\n.94952 .01714 L\n.95143 .01697 L\n.95333 .0168 L\n.95524 .01662 L\n.95714 .01645 L\n.95905 .01628 L\n.96095 .0161 L\n.96286 .01593 L\n.96476 .01576 L\n.96667 .01558 L\n.96857 .01541 L\n.97048 .01524 L\n.97238 .01506 L\nMistroke\n.97429 .01489 L\n.97619 .01472 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(mass) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n135.000 12.813 moveto\n(L) show\n141.000 12.813 moveto\n(@) show\n147.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n153.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@008@3IfMT200000?d0fMWI\n000Q0=WIf@030000003IfMT0fMWI0?`0fMWI000Q0=WIf@030000003IfMT0fMWI0=d0fMWI0P000?lM\n0=WIf@008@3IfMT00`000000fMWI0=WIf@3A0=WIf@`0003o7`3IfMT00240fMWI00<000000=WIf@3I\nfMT0a`3IfMT;0000obX0fMWI000Q0=WIf@030000003IfMT0fMWI0;/0fMWI30000?le0=WIf@008@3I\nfMT2000003<0fMWI0P0000040=WIf@800000;03IfMT500000080fMWI0P0000040=WIf@800000:@3I\nfMT500000080fMWI0P0000040=WIf@800000103IfMT;0000oaX0fMWI100000030=WIf@800000103I\nfMT2000001L0fMWI000Q0=WIf@030000003IfMT0fMWI0340fMWI00@000000=WIf@3IfMT000000P3I\nfMT010000000fMWI0=WIf@00000]0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02P0fMWI00D000000=WIf@3IfMT0fMWI0000000<\n0000obD0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003I\nfMT0fMWI000001H0fMWI000Q0=WIf@030000003IfMT0fMWI03@0fMWI00@000000=WIf@3IfMT00000\n0P3IfMT00`000000fMWI0=WIf@0[0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02<0fMWI2`000?l01@3IfMT000000=WIf@3IfMT0\n00000080fMWI00<000000=WIf@3IfMT09`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000005P3IfMT00240fMWI00<000000=WIf@3IfMT0<P3I\nfMT2000000<0fMWI00@000000=WIf@3IfMT00000;@3IfMT01@000000fMWI0=WIf@3IfMT000000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0H0=WIf@/0003o1@3IfMT00`00\n0000fMWI0=WIf@020=WIf@8000000`3IfMT010000000fMWI0=WIf@00000Z0=WIf@050000003IfMT0\nfMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI01@0fMWI\n000Q0=WIf@030000003IfMT0fMWI0380fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WI\nf@00000]0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WI\nf@030000003IfMT0fMWI00d0fMWI2`000?l@0=WIf@030000003IfMT0fMWI0080fMWI00<000000=WI\nf@3IfMT00P3IfMT010000000fMWI0=WIf@00000X0=WIf@040000003IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000F0=WIf@008@3IfMT00`000000fMWI\n0=WIf@0b0=WIf@<000000`3IfMT2000002d0fMWI0P0000040=WIf@800000103IfMT2000000D0fMWI\n2`000?lJ0=WIf@800000103IfMT3000000<0fMWI0P00000Z0=WIf@800000103IfMT2000000@0fMWI\n0P00000G0=WIf@008@3IfMT200000700fMWI2`000?n20=WIf@008@3IfMT00`000000fMWI0=WIf@1T\n0=WIf@/0003oS@3IfMT00240fMWI00<000000=WIf@3IfMT0F@3IfMT;0000oiP0fMWI000Q0=WIf@03\n0000003IfMT0fMWI04h0fMWI2`000?nS0=WIf@008@3IfMT00`000000fMWI0=WIf@130=WIf@/0003o\nW`3IfMT2000000d0fMWI000L0=WIfD0000002`000?nR000000L0fMWI00@000000=WIf@3IfMT00000\n303IfMT00240fMWI00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003I\nfMT0fMWI00P0fMWI00<000000=WIf@3IfMT0203IfMT010000000fMWI0=WIf@3IfMT:0000o`T0fMWI\n00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI\n00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI\n00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@090=WIf@030000003IfMT0fMWI00P0fMWI\n00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00P0fMWI\n00<000000=WIf@3IfMT02@3IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI00/0fMWI\n00<000000=WIf@3IfMT03@3IfMT00240fMWI00<000000=WIf@3IfMT0;@3IfMT00`000?l0fMWI0=WI\nf@060=WIf@030000003IfMT0fMWI03D0fMWI00<000000=WIf@3IfMT0=@3IfMT00`000000fMWI0=WI\nf@0e0=WIf@030000003IfMT0fMWI00X0fMWI1@00000<0=WIf@008@3IfMT00`000000fMWI0=WIf@0]\n0=WIf@030000o`3IfMT0fMWI0;`0fMWI00<000000=WIf@3IfMT03@3IfMT00240fMWI00<000000=WI\nf@3IfMT0;@3IfMT00`000?l0fMWI0=WIf@3<0=WIf@008@3IfMT00`000000fMWI0=WIf@0]0=WIf@03\n0000o`3IfMT0fMWI0<`0fMWI000Q0=WIf@800000;@3IfMT00`000?l0fMWI0=WIf@3=0=WIf@008@3I\nfMT00`000000fMWI0=WIf@0/0=WIf@030000o`3IfMT0fMWI0<d0fMWI000Q0=WIf@030000003IfMT0\nfMWI02/0fMWI00<0003o0=WIf@3IfMT0cP3IfMT00240fMWI00<000000=WIf@3IfMT0:`3IfMT00`00\n0?l0fMWI0=WIf@3>0=WIf@008@3IfMT00`000000fMWI0=WIf@0Z0=WIf@030000o`3IfMT0fMWI0<l0\nfMWI000Q0=WIf@030000003IfMT0fMWI02X0fMWI00<0003o0=WIf@3IfMT0c`3IfMT00240fMWI0P00\n000[0=WIf@030000o`3IfMT0fMWI0<l0fMWI000Q0=WIf@030000003IfMT0fMWI02T0fMWI00<0003o\n0=WIf@3IfMT0d03IfMT00240fMWI00<000000=WIf@3IfMT0:@3IfMT00`000?l0fMWI0=WIf@3@0=WI\nf@008@3IfMT00`000000fMWI0=WIf@0Y0=WIf@030000o`3IfMT0fMWI0=00fMWI000Q0=WIf@030000\n003IfMT0fMWI02P0fMWI00<0003o0=WIf@3IfMT0d@3IfMT00240fMWI0P00000Y0=WIf@030000o`3I\nfMT0fMWI0=40fMWI000Q0=WIf@030000003IfMT0fMWI02L0fMWI00<0003o0=WIf@3IfMT0dP3IfMT0\n0240fMWI00<000000=WIf@3IfMT09`3IfMT00`000?l0fMWI0=WIf@3B0=WIf@008@3IfMT00`000000\nfMWI0=WIf@0W0=WIf@030000o`3IfMT0fMWI0=80fMWI000Q0=WIf@030000003IfMT0fMWI02H0fMWI\n00<0003o0=WIf@3IfMT0d`3IfMT00240fMWI00<000000=WIf@3IfMT09P3IfMT00`000?l0fMWI0=WI\nf@3C0=WIf@008@3IfMT2000002H0fMWI00<0003o0=WIf@3IfMT0e03IfMT00240fMWI00<000000=WI\nf@3IfMT09@3IfMT00`000?l0fMWI0=WIf@3D0=WIf@008@3IfMT00`000000fMWI0=WIf@0U0=WIf@03\n0000o`3IfMT0fMWI0=@0fMWI000Q0=WIf@030000003IfMT0fMWI02@0fMWI00<0003o0=WIf@3IfMT0\ne@3IfMT00100fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI0P0000040=WIf@030000003I\nfMT0fMWI02@0fMWI00<0003o0=WIf@3IfMT0e@3IfMT00100fMWI00D000000=WIf@3IfMT0fMWI0000\n00020=WIf@040000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000V0=WIf@03\n0000o`3IfMT0fMWI0=D0fMWI000@0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0=WIf@<000008`3IfMT00`000?l0fMWI0=WIf@3F\n0=WIf@004@3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00D0\n00000=WIf@3IfMT0fMWI0000000U0=WIf@030000o`3IfMT0fMWI0=H0fMWI000>0=WIf@040000003I\nfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WI\nf@030000003IfMT0fMWI0280fMWI00<0003o0=WIf@3IfMT0e`3IfMT000h0fMWI100000030=WIf@80\n0000103IfMT2000000@0fMWI00<000000=WIf@3IfMT08P3IfMT00`000?l0fMWI0=WIf@3G0=WIf@00\n8@3IfMT00`000000fMWI0=WIf@0Q0=WIf@030000o`3IfMT0fMWI0=P0fMWI000Q0=WIf@8000008P3I\nfMT00`000?l0fMWI0=WIf@3H0=WIf@008@3IfMT00`000000fMWI0=WIf@0P0=WIf@030000o`3IfMT0\nfMWI0=T0fMWI000Q0=WIf@030000003IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT0f@3IfMT00240\nfMWI00<000000=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@3I0=WIf@008@3IfMT00`000000fMWI\n0=WIf@0P0=WIf@030000o`3IfMT0fMWI0=T0fMWI000Q0=WIf@030000003IfMT0fMWI01l0fMWI00<0\n003o0=WIf@3IfMT0fP3IfMT00240fMWI0P00000P0=WIf@030000o`3IfMT0fMWI0=X0fMWI000Q0=WI\nf@030000003IfMT0fMWI01h0fMWI00<0003o0=WIf@3IfMT0f`3IfMT00240fMWI00<000000=WIf@3I\nfMT07P3IfMT00`000?l0fMWI0=WIf@3K0=WIf@008@3IfMT00`000000fMWI0=WIf@0M0=WIf@030000\no`3IfMT0fMWI0=`0fMWI000Q0=WIf@030000003IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT0g03I\nfMT00240fMWI00<000000=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@3L0=WIf@008@3IfMT20000\n01d0fMWI00<0003o0=WIf@3IfMT0g@3IfMT00240fMWI00<000000=WIf@3IfMT0703IfMT00`000?l0\nfMWI0=WIf@3M0=WIf@008@3IfMT00`000000fMWI0=WIf@0L0=WIf@030000o`3IfMT0fMWI0=d0fMWI\n000Q0=WIf@030000003IfMT0fMWI01/0fMWI00<0003o0=WIf@3IfMT0gP3IfMT00240fMWI00<00000\n0=WIf@3IfMT06`3IfMT00`000?l0fMWI0=WIf@3N0=WIf@008@3IfMT2000001/0fMWI00<0003o0=WI\nf@3IfMT0g`3IfMT00240fMWI00<000000=WIf@3IfMT06P3IfMT00`000?l0fMWI0=WIf@3O0=WIf@00\n8@3IfMT00`000000fMWI0=WIf@0J0=WIf@030000o`3IfMT0fMWI0=l0fMWI000Q0=WIf@030000003I\nfMT0fMWI01T0fMWI00<0003o0=WIf@3IfMT0h03IfMT000l0fMWI0P0000040=WIf@800000103IfMT2\n000000@0fMWI00<000000=WIf@3IfMT06@3IfMT00`000?l0fMWI0=WIf@3P0=WIf@003P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0`3IfMT00`000000fMWI0=WIf@0H0=WIf@030000o`3IfMT0fMWI0>40fMWI000>0=WIf@040000003I\nfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WI\nf@<00000603IfMT00`000?l0fMWI0=WIf@3Q0=WIf@003`3IfMT2000000<0fMWI00@000000=WIf@3I\nfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI01L0fMWI00<0003o\n0=WIf@3IfMT0hP3IfMT000h0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT05`3IfMT00`000?l0fMWI\n0=WIf@3R0=WIf@003`3IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WI\nf@0G0=WIf@030000o`3IfMT0fMWI0>80fMWI000Q0=WIf@030000003IfMT0fMWI01H0fMWI00<0003o\n0=WIf@3IfMT0h`3IfMT00240fMWI0P00000G0=WIf@030000o`3IfMT0fMWI0><0fMWI000Q0=WIf@03\n0000003IfMT0fMWI01H0fMWI00<0003o0=WIf@3IfMT0h`3IfMT00240fMWI00<000000=WIf@3IfMT0\n5@3IfMT00`000?l0fMWI0=WIf@3T0=WIf@008@3IfMT00`000000fMWI0=WIf@0E0=WIf@030000o`3I\nfMT0fMWI0>@0fMWI000Q0=WIf@030000003IfMT0fMWI01@0fMWI00<0003o0=WIf@3IfMT0i@3IfMT0\n0240fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@3U0=WIf@008@3IfMT2000001@0\nfMWI00<0003o0=WIf@3IfMT0iP3IfMT00240fMWI00<000000=WIf@3IfMT04`3IfMT00`000?l0fMWI\n0=WIf@3V0=WIf@008@3IfMT00`000000fMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI0>H0fMWI000Q\n0=WIf@030000003IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT0iP3IfMT00240fMWI00<000000=WI\nf@3IfMT04P3IfMT00`000?l0fMWI0=WIf@3W0=WIf@008@3IfMT00`000000fMWI0=WIf@0B0=WIf@03\n0000o`3IfMT0fMWI0>L0fMWI000Q0=WIf@8000004P3IfMT00`000?l0fMWI0=WIf@3X0=WIf@008@3I\nfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI0>P0fMWI000Q0=WIf@030000003IfMT0\nfMWI0100fMWI00<0003o0=WIf@3IfMT0j@3IfMT00240fMWI00<000000=WIf@3IfMT0403IfMT00`00\n0?l0fMWI0=WIf@3Y0=WIf@008@3IfMT00`000000fMWI0=WIf@0@0=WIf@030000o`3IfMT0fMWI0>T0\nfMWI000Q0=WIf@800000403IfMT00`000?l0fMWI0=WIf@3Z0=WIf@008@3IfMT00`000000fMWI0=WI\nf@0?0=WIf@030000o`3IfMT0fMWI0>X0fMWI000Q0=WIf@030000003IfMT0fMWI00l0fMWI00<0003o\n0=WIf@3IfMT0jP3IfMT00240fMWI00<000000=WIf@3IfMT03P3IfMT00`000?l0fMWI0=WIf@3[0=WI\nf@003P3IfMT3000000@0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0>0=WIf@03\n0000o`3IfMT0fMWI0>/0fMWI000@0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000\nfMWI0=WIf@0000020=WIf@050000003IfMT0fMWI0=WIf@0000003`3IfMT00`000?l0fMWI0=WIf@3/\n0=WIf@003`3IfMT300000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00030=WIf@<000003@3IfMT00`000?l0fMWI0=WIf@3/0=WIf@003P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000\nfMWI0=WIf@0<0=WIf@030000o`3IfMT0fMWI0>d0fMWI000>0=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0\nfMWI00`0fMWI00<0003o0=WIf@3IfMT0k@3IfMT000l0fMWI0`0000030=WIf@800000103IfMT20000\n00@0fMWI00<000000=WIf@3IfMT0303IfMT00`000?l0fMWI0=WIf@3]0=WIf@008@3IfMT00`000000\nfMWI0=WIf@0;0=WIf@030000o`3IfMT0fMWI0>h0fMWI000Q0=WIf@030000003IfMT0fMWI00/0fMWI\n00<0003o0=WIf@3IfMT0kP3IfMT00240fMWI0P00000;0=WIf@030000o`3IfMT0fMWI0>l0fMWI000Q\n0=WIf@030000003IfMT0fMWI00X0fMWI00<0003o0=WIf@3IfMT0k`3IfMT00240fMWI00<000000=WI\nf@3IfMT02P3IfMT00`000?l0fMWI0=WIf@3_0=WIf@008@3IfMT00`000000fMWI0=WIf@090=WIf@03\n0000o`3IfMT0fMWI0?00fMWI000Q0=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0\nl03IfMT00240fMWI0P00000:0=WIf@030000o`3IfMT0fMWI0?00fMWI000Q0=WIf@030000003IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0l@3IfMT00240fMWI00<000000=WIf@3IfMT0203IfMT00`00\n0?l0fMWI0=WIf@3a0=WIf@008@3IfMT00`000000fMWI0=WIf@070=WIf@030000o`3IfMT0fMWI0?80\nfMWI000Q0=WIf@030000003IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0lP3IfMT00240fMWI00<0\n00000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3c0=WIf@008@3IfMT2000000L0fMWI00<0003o\n0=WIf@3IfMT0l`3IfMT00240fMWI00<000000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3c0=WI\nf@008@3IfMT00`000000fMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q0=WIf@030000\n003IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0m03IfMT00240fMWI00<000000=WIf@3IfMT01@3I\nfMT00`000?l0fMWI0=WIf@3d0=WIf@008@3IfMT00`000000fMWI0=WIf@040=WIf@030000o`3IfMT0\nfMWI0?D0fMWI000Q0=WIf@8000001@3IfMT00`000?l0fMWI0=WIf@3e0=WIf@008@3IfMT00`000000\nfMWI0=WIf@030=WIf@030000o`3IfMT0fMWI0?H0fMWI000Q0=WIf@030000003IfMT0fMWI00<0fMWI\n00<0003o0=WIf@3IfMT0mP3IfMT000P0fMWI1@0000020=WIf@800000103IfMT2000000@0fMWI0P00\n00040=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0m`3IfMT000X0fMWI00D00000\n0=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT01@000000fMWI0=WIf@3IfMT0000000@0fMWI00<0003o0=WIf@3IfMT0m`3IfMT000X0\nfMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@3IfMT300000080fMWI00<0003o0=WIf@3IfMT0\nm`3IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000<0fMWI00<0003o\n0=WIf@3IfMT0n03IfMT000X0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000<0\nfMWI00<0003o0=WIf@3IfMT0n03IfMT000T0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI\n0P0000040=WIf@050000003IfMT0fMWI0=WIf@000?l0nP3IfMT00240fMWI00@000000=WIf@3IfMT0\n003on`3IfMT00240fMWI00@000000=WIf@3IfMT0003on`3IfMT00240fMWI0P0000000`000?l0fMWI\n0=WIf@3j0=WIf@008@3IfMT00`000000fMWI0000o`3l0=WIf@008@3IfMT00`000000fMWI0000o`3l\n0=WIf@008@3IfMT00`000000003o0=WIf@3l0=WIf@008@3IfMT00`000000003o0=WIf@3l0=WIf@00\n8@3IfMT00`000?l0fMWI0=WIf@3l0=WIf@008@3IfMT200000?d0fMWI000Q0=WIf@030000003IfMT0\nfMWI0?`0fMWI000Q0=WIf@030000003IfMT0fMWI0?`0fMWI000Q0=WIf@030000003IfMT0fMWI0?`0\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40\nfMWI000U0=WIf@030000003IfMT0fMWI00D0fMWI0`00000j0=WIf@030000003IfMT0fMWI00D0fMWI\n0P0000080=WIf@800000XP3IfMT002@0fMWI00<000000=WIf@3IfMT01`3IfMT00`000000fMWI0=WI\nf@0j0=WIf@030000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WI\nf@2P0=WIf@008`3IfMT00`000000fMWI0=WIf@020=WIf@D0000000@0fMWI0000000000000000103I\nfMT2000000<0fMWI0`0000030=WIf@<00000103IfMT2000000@0fMWI0P0000020=WIf@80000000D0\nfMWI0000003IfMT000000=WIf@05000000040=WIf@0000000000000000<0fMWI0`0000050=WIf@03\n0000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0fMWI00<000000=WIf@3I\nfMT0X03IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT400000080fMWI00@000000=WIf@3IfMT0\n00000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WI\nf@3IfMT00P3IfMT010000000fMWI0=WIf@0000080=WIf@060000003IfMT000000=WIf@000000fMWI\n100000050=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI\n0=WIf@030=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT00000\n0:80fMWI000F0=WIf@P0003o1@3IfMT00`000000fMWI0=WIf@050=WIf@040000003IfMT0fMWI0000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00<0\n00000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000080=WIf@050000003IfMT000000=WIf@00\n0000103IfMT00`000000fMWI0=WIf@03000000<0fMWI0`0000050=WIf@030000003IfMT0fMWI00<0\nfMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0fMWI0:00fMWI000S0=WIf@03\n0000003IfMT0fMWI00<0fMWI0`0000000`3IfMT0000000000002000000@0fMWI0P0000040=WIf@03\n0000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000P0fMWI0`0000000`3IfMT0\n00000=WIf@020=WIf@<000000P3IfMT400000080fMWI100000040=WIf@030000003IfMT0fMWI00<0\nfMWI00<000000=WIf@3IfMT01@0000030=WIf@030000003IfMT0fMWI0:00fMWI000T0=WIf@030000\n003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0W0=WIf@030000\n003IfMT0fMWI00@0fMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0fMWI0:00\nfMWI000U0=WIf@030000003IfMT0fMWI0180fMWI0P0000040=WIf@800000:03IfMT00`000000fMWI\n0=WIf@050=WIf@800000203IfMT200000:80fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WI\nfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-30.2449`, 475.32`, 0.890933`, 3.54472`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[TextData[List["\t\t\t\t\t\t", StyleBox["Diagram 3:", Rule[FontWeight, "Bold"]], " Rocket mass decreases when the fuel is consumed\n"]], "Text"], Cell["plot( apollo.gravity );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0047619 -8.48524 5.57542 [\n[.2619 .42294 -6 -9 ]\n[.2619 .42294 6 0 ]\n[.5 .42294 -9 -9 ]\n[.5 .42294 9 0 ]\n[.7381 .42294 -9 -9 ]\n[.7381 .42294 9 0 ]\n[.97619 .42294 -9 -9 ]\n[.97619 .42294 9 0 ]\n[1.025 .43544 0 -6.28125 ]\n[1.025 .43544 10 6.28125 ]\n[.01131 .10091 -24 -4.5 ]\n[.01131 .10091 0 4.5 ]\n[.01131 .21242 -24 -4.5 ]\n[.01131 .21242 0 4.5 ]\n[.01131 .32393 -24 -4.5 ]\n[.01131 .32393 0 4.5 ]\n[.01131 .54695 -24 -4.5 ]\n[.01131 .54695 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.2619 .43544 m\n.2619 .44169 L\ns\n[(50)] .2619 .42294 0 1 Mshowa\n.5 .43544 m\n.5 .44169 L\ns\n[(100)] .5 .42294 0 1 Mshowa\n.7381 .43544 m\n.7381 .44169 L\ns\n[(150)] .7381 .42294 0 1 Mshowa\n.97619 .43544 m\n.97619 .44169 L\ns\n[(200)] .97619 .42294 0 1 Mshowa\n.125 Mabswid\n.07143 .43544 m\n.07143 .43919 L\ns\n.11905 .43544 m\n.11905 .43919 L\ns\n.16667 .43544 m\n.16667 .43919 L\ns\n.21429 .43544 m\n.21429 .43919 L\ns\n.30952 .43544 m\n.30952 .43919 L\ns\n.35714 .43544 m\n.35714 .43919 L\ns\n.40476 .43544 m\n.40476 .43919 L\ns\n.45238 .43544 m\n.45238 .43919 L\ns\n.54762 .43544 m\n.54762 .43919 L\ns\n.59524 .43544 m\n.59524 .43919 L\ns\n.64286 .43544 m\n.64286 .43919 L\ns\n.69048 .43544 m\n.69048 .43919 L\ns\n.78571 .43544 m\n.78571 .43919 L\ns\n.83333 .43544 m\n.83333 .43919 L\ns\n.88095 .43544 m\n.88095 .43919 L\ns\n.92857 .43544 m\n.92857 .43919 L\ns\n.25 Mabswid\n0 .43544 m\n1 .43544 L\ns\ngsave\n1.025 .43544 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .10091 m\n.03006 .10091 L\ns\n[(1.54)] .01131 .10091 1 0 Mshowa\n.02381 .21242 m\n.03006 .21242 L\ns\n[(1.56)] .01131 .21242 1 0 Mshowa\n.02381 .32393 m\n.03006 .32393 L\ns\n[(1.58)] .01131 .32393 1 0 Mshowa\n.02381 .54695 m\n.03006 .54695 L\ns\n[(1.62)] .01131 .54695 1 0 Mshowa\n.125 Mabswid\n.02381 .12879 m\n.02756 .12879 L\ns\n.02381 .15667 m\n.02756 .15667 L\ns\n.02381 .18455 m\n.02756 .18455 L\ns\n.02381 .2403 m\n.02756 .2403 L\ns\n.02381 .26818 m\n.02756 .26818 L\ns\n.02381 .29605 m\n.02756 .29605 L\ns\n.02381 .35181 m\n.02756 .35181 L\ns\n.02381 .37969 m\n.02756 .37969 L\ns\n.02381 .40756 m\n.02756 .40756 L\ns\n.02381 .46332 m\n.02756 .46332 L\ns\n.02381 .49119 m\n.02756 .49119 L\ns\n.02381 .51907 m\n.02756 .51907 L\ns\n.02381 .07304 m\n.02756 .07304 L\ns\n.02381 .04516 m\n.02756 .04516 L\ns\n.02381 .01728 m\n.02756 .01728 L\ns\n.02381 .57483 m\n.02756 .57483 L\ns\n.02381 .6027 m\n.02756 .6027 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .02227 L\n.02762 .02979 L\n.02952 .03726 L\n.03143 .0447 L\n.03333 .05209 L\n.03524 .05944 L\n.03714 .06674 L\n.03905 .074 L\n.04095 .08122 L\n.04286 .08839 L\n.04476 .09552 L\n.04667 .10261 L\n.04857 .10965 L\n.05048 .11664 L\n.05238 .12358 L\n.05429 .13048 L\n.05619 .13732 L\n.0581 .14413 L\n.06 .15088 L\n.0619 .15759 L\n.06381 .16425 L\n.06571 .17085 L\n.06762 .17741 L\n.06952 .18392 L\n.07143 .19038 L\n.07333 .19679 L\n.07524 .20315 L\n.07714 .20945 L\n.07905 .2157 L\n.08095 .2219 L\n.08286 .22804 L\n.08476 .23413 L\n.08667 .24017 L\n.08857 .24616 L\n.09048 .25209 L\n.09238 .25796 L\n.09429 .26378 L\n.09619 .26954 L\n.0981 .27524 L\n.1 .28089 L\n.1019 .28648 L\n.10381 .29201 L\n.10571 .29748 L\n.10762 .30289 L\n.10952 .30825 L\n.11143 .31354 L\n.11333 .31877 L\n.11524 .32395 L\n.11714 .32905 L\nMistroke\n.11905 .3341 L\n.12095 .33908 L\n.12286 .34401 L\n.12476 .34887 L\n.12667 .35366 L\n.12857 .35839 L\n.13048 .36305 L\n.13238 .36765 L\n.13429 .37218 L\n.13619 .37665 L\n.1381 .38105 L\n.14 .38538 L\n.1419 .38964 L\n.14381 .39383 L\n.14571 .39796 L\n.14762 .40202 L\n.14952 .406 L\n.15143 .40991 L\n.15333 .41375 L\n.15524 .41752 L\n.15714 .42122 L\n.15905 .42484 L\n.16095 .42839 L\n.16286 .43187 L\n.16476 .43527 L\n.16667 .43859 L\n.16857 .44184 L\n.17048 .44502 L\n.17238 .44811 L\n.17429 .45113 L\n.17619 .45407 L\n.1781 .45693 L\n.18 .45971 L\n.1819 .46241 L\n.18381 .46503 L\n.18571 .46757 L\n.18762 .47003 L\n.18952 .47241 L\n.19143 .4747 L\n.19333 .47691 L\n.19524 .47904 L\n.19714 .48108 L\n.19905 .48303 L\n.20095 .4849 L\n.20286 .48668 L\n.20476 .48838 L\n.20667 .48998 L\n.20857 .4915 L\n.21048 .49293 L\n.21238 .49427 L\nMistroke\n.21429 .49552 L\n.21619 .49668 L\n.2181 .49774 L\n.22 .49872 L\n.2219 .4996 L\n.22381 .50039 L\n.22571 .50108 L\n.22762 .50168 L\n.22952 .50218 L\n.22952 .50218 L\n.23143 .50264 L\n.23333 .50309 L\n.23524 .50354 L\n.23714 .504 L\n.23905 .50445 L\n.24095 .5049 L\n.24286 .50535 L\n.24476 .5058 L\n.24667 .50625 L\n.24857 .50669 L\n.25048 .50714 L\n.25238 .50758 L\n.25429 .50803 L\n.25619 .50847 L\n.2581 .50892 L\n.26 .50936 L\n.2619 .5098 L\n.26381 .51024 L\n.26571 .51068 L\n.26762 .51112 L\n.26952 .51155 L\n.27143 .51199 L\n.27333 .51242 L\n.27524 .51286 L\n.27714 .51329 L\n.27905 .51373 L\n.28095 .51416 L\n.28286 .51459 L\n.28476 .51502 L\n.28667 .51545 L\n.28857 .51588 L\n.29048 .5163 L\n.29238 .51673 L\n.29429 .51715 L\n.29619 .51758 L\n.2981 .518 L\n.3 .51842 L\n.3019 .51885 L\n.30381 .51927 L\n.30571 .51969 L\nMistroke\n.30762 .52011 L\n.30952 .52052 L\n.31143 .52094 L\n.31333 .52136 L\n.31524 .52177 L\n.31714 .52219 L\n.31905 .5226 L\n.32095 .52301 L\n.32286 .52343 L\n.32476 .52384 L\n.32667 .52425 L\n.32857 .52466 L\n.33048 .52506 L\n.33238 .52547 L\n.33429 .52588 L\n.33619 .52628 L\n.3381 .52668 L\n.34 .52709 L\n.3419 .52749 L\n.34381 .52789 L\n.34571 .52829 L\n.34762 .52869 L\n.34952 .52909 L\n.35143 .52949 L\n.35333 .52988 L\n.35524 .53028 L\n.35714 .53067 L\n.35905 .53107 L\n.36095 .53146 L\n.36286 .53185 L\n.36476 .53224 L\n.36667 .53263 L\n.36857 .53302 L\n.37048 .53341 L\n.37238 .5338 L\n.37429 .53418 L\n.37619 .53457 L\n.3781 .53495 L\n.38 .53534 L\n.3819 .53572 L\n.38381 .5361 L\n.38571 .53648 L\n.38762 .53686 L\n.38952 .53724 L\n.39143 .53762 L\n.39333 .53799 L\n.39524 .53837 L\n.39714 .53874 L\n.39905 .53912 L\n.40095 .53949 L\nMistroke\n.40286 .53986 L\n.40476 .54023 L\n.40667 .5406 L\n.40857 .54097 L\n.41048 .54134 L\n.41238 .54171 L\n.41429 .54207 L\n.41619 .54244 L\n.4181 .5428 L\n.42 .54316 L\n.4219 .54353 L\n.42381 .54389 L\n.42571 .54425 L\n.42762 .54461 L\n.42952 .54497 L\n.43143 .54532 L\n.43333 .54568 L\n.43524 .54603 L\n.43714 .54639 L\n.43905 .54674 L\n.44095 .54709 L\n.44286 .54745 L\n.44476 .5478 L\n.44667 .54815 L\n.44857 .54849 L\n.45048 .54884 L\n.45238 .54919 L\n.45429 .54953 L\n.45619 .54988 L\n.4581 .55022 L\n.46 .55056 L\n.4619 .5509 L\n.46381 .55124 L\n.46571 .55158 L\n.46762 .55192 L\n.46952 .55226 L\n.47143 .5526 L\n.47333 .55293 L\n.47524 .55327 L\n.47714 .5536 L\n.47905 .55393 L\n.48095 .55426 L\n.48286 .55459 L\n.48476 .55492 L\n.48667 .55525 L\n.48857 .55558 L\n.49048 .55591 L\n.49238 .55623 L\n.49429 .55656 L\n.49619 .55688 L\nMistroke\n.4981 .5572 L\n.5 .55752 L\n.5019 .55784 L\n.50381 .55816 L\n.50571 .55848 L\n.50762 .5588 L\n.50952 .55911 L\n.51143 .55943 L\n.51333 .55974 L\n.51524 .56006 L\n.51714 .56037 L\n.51905 .56068 L\n.52095 .56099 L\n.52286 .5613 L\n.52476 .56161 L\n.52667 .56191 L\n.52857 .56222 L\n.53048 .56252 L\n.53238 .56283 L\n.53429 .56313 L\n.53619 .56343 L\n.5381 .56373 L\n.54 .56403 L\n.5419 .56433 L\n.54381 .56463 L\n.54571 .56493 L\n.54762 .56522 L\n.54952 .56552 L\n.55143 .56581 L\n.55333 .5661 L\n.55524 .56639 L\n.55714 .56668 L\n.55905 .56697 L\n.56095 .56726 L\n.56286 .56755 L\n.56476 .56783 L\n.56667 .56812 L\n.56857 .5684 L\n.57048 .56869 L\n.57238 .56897 L\n.57429 .56925 L\n.57619 .56953 L\n.5781 .56981 L\n.58 .57009 L\n.5819 .57036 L\n.58381 .57064 L\n.58571 .57091 L\n.58762 .57119 L\n.58952 .57146 L\n.59143 .57173 L\nMistroke\n.59333 .572 L\n.59524 .57227 L\n.59714 .57254 L\n.59905 .57281 L\n.60095 .57307 L\n.60286 .57334 L\n.60476 .5736 L\n.60667 .57386 L\n.60857 .57413 L\n.61048 .57439 L\n.61238 .57465 L\n.61429 .5749 L\n.61619 .57516 L\n.6181 .57542 L\n.62 .57567 L\n.6219 .57593 L\n.62381 .57618 L\n.62571 .57643 L\n.62762 .57669 L\n.62952 .57694 L\n.63143 .57718 L\n.63333 .57743 L\n.63524 .57768 L\n.63714 .57792 L\n.63905 .57817 L\n.64095 .57841 L\n.64286 .57865 L\n.64476 .5789 L\n.64667 .57914 L\n.64857 .57937 L\n.65048 .57961 L\n.65238 .57985 L\n.65429 .58008 L\n.65619 .58032 L\n.6581 .58055 L\n.66 .58079 L\n.6619 .58102 L\n.66381 .58125 L\n.66571 .58148 L\n.66762 .58171 L\n.66952 .58193 L\n.67143 .58216 L\n.67333 .58238 L\n.67524 .58261 L\n.67714 .58283 L\n.67905 .58305 L\n.68095 .58327 L\n.68286 .58349 L\n.68476 .58371 L\n.68667 .58392 L\nMistroke\n.68857 .58414 L\n.69048 .58435 L\n.69238 .58457 L\n.69429 .58478 L\n.69619 .58499 L\n.6981 .5852 L\n.7 .58541 L\n.7019 .58562 L\n.70381 .58583 L\n.70571 .58603 L\n.70762 .58624 L\n.70952 .58644 L\n.71143 .58664 L\n.71333 .58684 L\n.71524 .58704 L\n.71714 .58724 L\n.71905 .58744 L\n.72095 .58764 L\n.72286 .58783 L\n.72476 .58803 L\n.72667 .58822 L\n.72857 .58841 L\n.73048 .5886 L\n.73238 .58879 L\n.73429 .58898 L\n.73619 .58917 L\n.7381 .58936 L\n.74 .58954 L\n.7419 .58973 L\n.74381 .58991 L\n.74571 .59009 L\n.74762 .59027 L\n.74952 .59045 L\n.75143 .59063 L\n.75333 .59081 L\n.75524 .59098 L\n.75714 .59116 L\n.75905 .59133 L\n.76095 .59151 L\n.76286 .59168 L\n.76476 .59185 L\n.76667 .59202 L\n.76857 .59218 L\n.77048 .59235 L\n.77238 .59252 L\n.77429 .59268 L\n.77619 .59285 L\n.7781 .59301 L\n.78 .59317 L\n.7819 .59333 L\nMistroke\n.78381 .59349 L\n.78571 .59364 L\n.78762 .5938 L\n.78952 .59396 L\n.79143 .59411 L\n.79333 .59426 L\n.79524 .59441 L\n.79714 .59457 L\n.79905 .59472 L\n.80095 .59486 L\n.80286 .59501 L\n.80476 .59516 L\n.80667 .5953 L\n.80857 .59544 L\n.81048 .59559 L\n.81238 .59573 L\n.81429 .59587 L\n.81619 .59601 L\n.8181 .59614 L\n.82 .59628 L\n.8219 .59641 L\n.82381 .59655 L\n.82571 .59668 L\n.82762 .59681 L\n.82952 .59694 L\n.83143 .59707 L\n.83333 .5972 L\n.83524 .59733 L\n.83714 .59745 L\n.83905 .59758 L\n.84095 .5977 L\n.84286 .59782 L\n.84476 .59794 L\n.84667 .59806 L\n.84857 .59818 L\n.85048 .5983 L\n.85238 .59842 L\n.85429 .59853 L\n.85619 .59864 L\n.8581 .59876 L\n.86 .59887 L\n.8619 .59898 L\n.86381 .59909 L\n.86571 .59919 L\n.86762 .5993 L\n.86952 .5994 L\n.87143 .59951 L\n.87333 .59961 L\n.87524 .59971 L\n.87714 .59981 L\nMistroke\n.87905 .59991 L\n.88095 .60001 L\n.88286 .60011 L\n.88476 .6002 L\n.88667 .6003 L\n.88857 .60039 L\n.89048 .60048 L\n.89238 .60057 L\n.89429 .60066 L\n.89619 .60075 L\n.8981 .60083 L\n.9 .60092 L\n.9019 .601 L\n.90381 .60109 L\n.90571 .60117 L\n.90762 .60125 L\n.90952 .60133 L\n.91143 .60141 L\n.91333 .60148 L\n.91524 .60156 L\n.91714 .60163 L\n.91905 .60171 L\n.92095 .60178 L\n.92286 .60185 L\n.92476 .60192 L\n.92667 .60199 L\n.92857 .60205 L\n.93048 .60212 L\n.93238 .60218 L\n.93429 .60225 L\n.93619 .60231 L\n.9381 .60237 L\n.94 .60243 L\n.9419 .60249 L\n.94381 .60254 L\n.94571 .6026 L\n.94762 .60265 L\n.94952 .60271 L\n.95143 .60276 L\n.95333 .60281 L\n.95524 .60286 L\n.95714 .60291 L\n.95905 .60295 L\n.96095 .603 L\n.96286 .60304 L\n.96476 .60309 L\n.96667 .60313 L\n.96857 .60317 L\n.97048 .60321 L\n.97238 .60325 L\nMistroke\n.97429 .60328 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(gravity) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n153.000 12.813 moveto\n(L) show\n159.000 12.813 moveto\n(@) show\n165.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n171.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@008@3IfMT00`000000fMWI\n0=WIf@3l0=WIf@008@3IfMT00`000000fMWI0=WIf@3l0=WIf@008@3IfMT00`000?l0fMWI0=WIf@3l\n0=WIf@008@3IfMT00`000?l000000=WIf@3l0=WIf@008@3IfMT00`000000003o0=WIf@3l0=WIf@00\n8@3IfMT00`000000003o0=WIf@3l0=WIf@008@3IfMT00`000000003o0=WIf@3l0=WIf@008@3IfMT0\n0`000000003o0=WIf@3l0=WIf@008@3IfMT00`000000fMWI0000o`3l0=WIf@008@3IfMT00`000000\nfMWI0000o`3l0=WIf@008@3IfMT2000000030000o`3IfMT0fMWI0?X0fMWI000Q0=WIf@040000003I\nfMT0fMWI0000oo/0fMWI000Q0=WIf@040000003IfMT0fMWI0000oo/0fMWI000Q0=WIf@040000003I\nfMT0fMWI0000oo/0fMWI000Q0=WIf@040000003IfMT0fMWI0000oo/0fMWI000Q0=WIf@050000003I\nfMT0fMWI0=WIf@000?l0nP3IfMT00240fMWI0P0000020=WIf@030000o`3IfMT0fMWI0?P0fMWI000Q\n0=WIf@050000003IfMT0fMWI0=WIf@000?l0nP3IfMT00240fMWI00D000000=WIf@3IfMT0fMWI0000\no`3j0=WIf@008@3IfMT01@000000fMWI0=WIf@3IfMT0003o0?X0fMWI000Q0=WIf@030000003IfMT0\nfMWI0080fMWI00<0003o0=WIf@3IfMT0m`3IfMT000P0fMWI1@0000020=WIf@800000103IfMT20000\n00D0fMWI0`0000020=WIf@030000003IfMT0fMWI0080fMWI00<0003o0=WIf@3IfMT0m`3IfMT000X0\nfMWI00<000000=WIf@3IfMT01`3IfMT010000000fMWI0=WIf@0000050=WIf@050000003IfMT0fMWI\n0=WIf@000000103IfMT00`000?l0fMWI0=WIf@3g0=WIf@002P3IfMT00`000000fMWI0=WIf@0:0=WI\nf@030000003IfMT0fMWI00D000000P3IfMT3000000<0fMWI00<0003o0=WIf@3IfMT0mP3IfMT000X0\nfMWI00<000000=WIf@3IfMT0203IfMT2000000<0fMWI00@000000=WIf@3IfMT000000`3IfMT00`00\n0000fMWI0=WIf@030=WIf@030000o`3IfMT0fMWI0?H0fMWI000:0=WIf@030000003IfMT0fMWI00P0\nfMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI000000030=WIf@030000003IfMT0fMWI00<0\nfMWI00<0003o0=WIf@3IfMT0mP3IfMT000T0fMWI0P00000:0=WIf@<00000103IfMT2000000<0fMWI\n00<000000=WIf@3IfMT0103IfMT00`000?l0fMWI0=WIf@3e0=WIf@008@3IfMT00`000000fMWI0=WI\nf@040=WIf@030000o`3IfMT0fMWI0?D0fMWI000Q0=WIf@030000003IfMT0fMWI00@0fMWI00<0003o\n0=WIf@3IfMT0m@3IfMT00240fMWI0P0000050=WIf@030000o`3IfMT0fMWI0?D0fMWI000Q0=WIf@03\n0000003IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0m03IfMT00240fMWI00<000000=WIf@3IfMT0\n1@3IfMT00`000?l0fMWI0=WIf@3d0=WIf@008@3IfMT00`000000fMWI0=WIf@050=WIf@030000o`3I\nfMT0fMWI0?@0fMWI000Q0=WIf@030000003IfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT0l`3IfMT0\n0240fMWI00<000000=WIf@3IfMT01P3IfMT00`000?l0fMWI0=WIf@3c0=WIf@008@3IfMT00`000000\nfMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q0=WIf@8000001`3IfMT00`000?l0fMWI\n0=WIf@3c0=WIf@008@3IfMT00`000000fMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0?<0fMWI000Q\n0=WIf@030000003IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0lP3IfMT00240fMWI00<000000=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@3b0=WIf@008@3IfMT00`000000fMWI0=WIf@070=WIf@03\n0000o`3IfMT0fMWI0?80fMWI000Q0=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WIf@3IfMT0\nl@3IfMT00240fMWI0P0000090=WIf@030000o`3IfMT0fMWI0?40fMWI000Q0=WIf@030000003IfMT0\nfMWI00P0fMWI00<0003o0=WIf@3IfMT0l@3IfMT00240fMWI00<000000=WIf@3IfMT02@3IfMT00`00\n0?l0fMWI0=WIf@3`0=WIf@008@3IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0?00\nfMWI000Q0=WIf@030000003IfMT0fMWI00T0fMWI00<0003o0=WIf@3IfMT0l03IfMT000P0fMWI1@00\n00020=WIf@800000103IfMT2000000<0fMWI0`0000040=WIf@030000003IfMT0fMWI00X0fMWI00<0\n003o0=WIf@3IfMT0k`3IfMT000X0fMWI00<000000=WIf@3IfMT01`3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT02P3IfMT00`000?l0fMWI\n0=WIf@3_0=WIf@002P3IfMT00`000000fMWI0=WIf@0:0=WIf@040000003IfMT0fMWI00000080fMWI\n00@000000=WIf@3IfMT0fMWI0`00000:0=WIf@030000o`3IfMT0fMWI0>l0fMWI000:0=WIf@030000\n003IfMT0fMWI00P0fMWI0P0000030=WIf@<00000103IfMT00`000000fMWI0=WIf@0:0=WIf@030000\no`3IfMT0fMWI0>l0fMWI000:0=WIf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT00`3I\nfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI00/0fMWI00<0003o0=WIf@3IfMT0kP3I\nfMT000T0fMWI0P00000:0=WIf@<000000`3IfMT3000000<0fMWI00<000000=WIf@3IfMT02`3IfMT0\n0`000?l0fMWI0=WIf@3^0=WIf@008@3IfMT00`000000fMWI0=WIf@0;0=WIf@030000o`3IfMT0fMWI\n0>h0fMWI000Q0=WIf@030000003IfMT0fMWI00`0fMWI00<0003o0=WIf@3IfMT0k@3IfMT00240fMWI\n00<000000=WIf@3IfMT0303IfMT00`000?l0fMWI0=WIf@3]0=WIf@008@3IfMT2000000d0fMWI00<0\n003o0=WIf@3IfMT0k@3IfMT00240fMWI00<000000=WIf@3IfMT03@3IfMT00`000?l0fMWI0=WIf@3/\n0=WIf@008@3IfMT00`000000fMWI0=WIf@0=0=WIf@030000o`3IfMT0fMWI0>`0fMWI000Q0=WIf@03\n0000003IfMT0fMWI00d0fMWI00<0003o0=WIf@3IfMT0k03IfMT00240fMWI00<000000=WIf@3IfMT0\n3P3IfMT00`000?l0fMWI0=WIf@3[0=WIf@008@3IfMT00`000000fMWI0=WIf@0>0=WIf@030000o`3I\nfMT0fMWI0>/0fMWI000Q0=WIf@800000403IfMT00`000?l0fMWI0=WIf@3Z0=WIf@008@3IfMT00`00\n0000fMWI0=WIf@0?0=WIf@030000o`3IfMT0fMWI0>X0fMWI000Q0=WIf@030000003IfMT0fMWI00l0\nfMWI00<0003o0=WIf@3IfMT0jP3IfMT00240fMWI00<000000=WIf@3IfMT03`3IfMT00`000?l0fMWI\n0=WIf@3Z0=WIf@008@3IfMT00`000000fMWI0=WIf@0@0=WIf@030000o`3IfMT0fMWI0>T0fMWI000Q\n0=WIf@030000003IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0j@3IfMT00240fMWI00<000000=WI\nf@3IfMT0403IfMT00`000?l0fMWI0=WIf@3Y0=WIf@008@3IfMT200000180fMWI00<0003o0=WIf@3I\nfMT0j03IfMT00240fMWI00<000000=WIf@3IfMT04@3IfMT00`000?l0fMWI0=WIf@3X0=WIf@008@3I\nfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI0>P0fMWI000Q0=WIf@030000003IfMT0\nfMWI0180fMWI00<0003o0=WIf@3IfMT0i`3IfMT000P0fMWI1@0000020=WIf@800000103IfMT20000\n00@0fMWI0P0000040=WIf@030000003IfMT0fMWI0180fMWI00<0003o0=WIf@3IfMT0i`3IfMT000X0\nfMWI00<000000=WIf@3IfMT01`3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI\n000000<0fMWI00<000000=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@3V0=WIf@002P3IfMT00`00\n0000fMWI0=WIf@0:0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI0`00\n000C0=WIf@030000o`3IfMT0fMWI0>H0fMWI000:0=WIf@030000003IfMT0fMWI00P0fMWI0P000004\n0=WIf@800000103IfMT00`000000fMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI0>H0fMWI000:0=WI\nf@030000003IfMT0fMWI00P0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@000003\n0=WIf@030000003IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT0iP3IfMT000T0fMWI0P00000:0=WI\nf@<000000`3IfMT2000000@0fMWI00<000000=WIf@3IfMT0503IfMT00`000?l0fMWI0=WIf@3U0=WI\nf@008@3IfMT00`000000fMWI0=WIf@0D0=WIf@030000o`3IfMT0fMWI0>D0fMWI000Q0=WIf@030000\n003IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0i03IfMT00240fMWI00<000000=WIf@3IfMT05@3I\nfMT00`000?l0fMWI0=WIf@3T0=WIf@008@3IfMT2000001L0fMWI00<0003o0=WIf@3IfMT0h`3IfMT0\n0240fMWI00<000000=WIf@3IfMT05P3IfMT00`000?l0fMWI0=WIf@3S0=WIf@008@3IfMT00`000000\nfMWI0=WIf@0G0=WIf@030000o`3IfMT0fMWI0>80fMWI000Q0=WIf@030000003IfMT0fMWI01L0fMWI\n00<0003o0=WIf@3IfMT0hP3IfMT00240fMWI00<000000=WIf@3IfMT05`3IfMT00`000?l0fMWI0=WI\nf@3R0=WIf@008@3IfMT00`000000fMWI0=WIf@0G0=WIf@030000o`3IfMT0fMWI0>80fMWI000Q0=WI\nf@8000006@3IfMT00`000?l0fMWI0=WIf@3Q0=WIf@008@3IfMT00`000000fMWI0=WIf@0H0=WIf@03\n0000o`3IfMT0fMWI0>40fMWI000Q0=WIf@030000003IfMT0fMWI01T0fMWI00<0003o0=WIf@3IfMT0\nh03IfMT00240fMWI00<000000=WIf@3IfMT06@3IfMT00`000?l0fMWI0=WIf@0F0=WIf@800000103I\nfMT2000002`0fMWI1@0000020=WIf@800000103IfMT2000002T0fMWI1@0000020=WIf@800000103I\nfMT2000002T0fMWI100000030=WIf@800000103IfMT2000001L0fMWI000Q0=WIf@030000003IfMT0\nfMWI01X0fMWI00<0003o0=WIf@3IfMT0503IfMT010000000fMWI0=WIf@0000020=WIf@040000003I\nfMT0fMWI000002d0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI0000\n0080fMWI00<000000=WIf@3IfMT0:03IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0V0=WIf@030000003IfMT0fMWI00<0fMWI00@0\n00000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000F0=WIf@008@3IfMT00`000000fMWI\n0=WIf@0J0=WIf@030000o`3IfMT0fMWI01L0fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000\nfMWI0=WIf@0[0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@030000003IfMT0fMWI02P0fMWI00<000000=WIf@3IfMT0103IfMT010000000fMWI0=WIf@00\n00020=WIf@030000003IfMT0fMWI02L0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WI\nf@0000020=WIf@040000003IfMT0fMWI000001H0fMWI000Q0=WIf@030000003IfMT0fMWI01/0fMWI\n00<0003o0=WIf@3IfMT0503IfMT2000000<0fMWI00@000000=WIf@3IfMT00000;@3IfMT01@000000\nfMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0X\n0=WIf@030000003IfMT0fMWI0080fMWI0P0000030=WIf@040000003IfMT0fMWI000002X0fMWI00D0\n00000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3I\nfMT0503IfMT00240fMWI0P00000L0=WIf@80003o5@3IfMT00`000000fMWI0=WIf@020=WIf@040000\n003IfMT0fMWI000002d0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI\n00000080fMWI00<000000=WIf@3IfMT0:03IfMT00`000000fMWI0=WIf@020=WIf@030000003IfMT0\nfMWI0080fMWI00@000000=WIf@3IfMT00000:03IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000005P3IfMT00240fMWI00<000000=WIf@3I\nfMT0703IfMT00`000?l0fMWI0=WIf@0C0=WIf@<000000`3IfMT2000002d0fMWI0P0000040=WIf@80\n0000103IfMT2000002X0fMWI0P0000040=WIf@<000000`3IfMT2000002X0fMWI0P0000040=WIf@80\n0000103IfMT2000001L0fMWI000Q0=WIf@030000003IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT0\ng@3IfMT00240fMWI00<000000=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@3L0=WIf@008@3IfMT0\n0`000000fMWI0=WIf@0M0=WIf@030000o`3IfMT0fMWI0=`0fMWI000Q0=WIf@030000003IfMT0fMWI\n01h0fMWI00<0003o0=WIf@3IfMT0c03IfMT2000000d0fMWI000Q0=WIf@030000003IfMT0fMWI01h0\nfMWI0P000?o<0=WIf@040000003IfMT0fMWI000000`0fMWI000L0=WIfBL0000000<0003o00000000\n0000``0000070=WIf@030000003IfMT0fMWI00d0fMWI000Q0=WIf@030000003IfMT0fMWI0200fMWI\n00<0003o0=WIf@3IfMT04`3IfMT00`000000fMWI0=WIf@0e0=WIf@030000003IfMT0fMWI03D0fMWI\n00<000000=WIf@3IfMT0=@3IfMT00`000000fMWI0=WIf@0:0=WIf@D00000303IfMT00240fMWI00<0\n00000=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@390=WIf@030000003IfMT0fMWI00d0fMWI000Q\n0=WIf@030000003IfMT0fMWI0200fMWI0P000?oJ0=WIf@008@3IfMT00`000000fMWI0=WIf@0Q0=WI\nf@030000o`3IfMT0fMWI0=P0fMWI000Q0=WIf@030000003IfMT0fMWI0280fMWI00<0003o0=WIf@3I\nfMT0e`3IfMT00240fMWI0P00000T0=WIf@030000o`3IfMT0fMWI0=H0fMWI000Q0=WIf@030000003I\nfMT0fMWI02@0fMWI00<0003o0=WIf@3IfMT0e@3IfMT00240fMWI00<000000=WIf@3IfMT0903IfMT2\n0000omH0fMWI000Q0=WIf@030000003IfMT0fMWI02D0fMWI00<0003o0=WIf@3IfMT0e03IfMT00240\nfMWI00<000000=WIf@3IfMT09P3IfMT00`000?l0fMWI0=WIf@3C0=WIf@008@3IfMT00`000000fMWI\n0=WIf@0W0=WIf@030000o`3IfMT0fMWI0=80fMWI000Q0=WIf@030000003IfMT0fMWI02P0fMWI0P00\n0?oB0=WIf@008@3IfMT2000002X0fMWI0P000?oA0=WIf@008@3IfMT00`000000fMWI0=WIf@0Z0=WI\nf@<0003oc`3IfMT00240fMWI00<000000=WIf@3IfMT0;03IfMT40000ol`0fMWI000Q0=WIf@030000\n003IfMT0fMWI02l0fMWI1@000?o80=WIf@008@3IfMT00`000000fMWI0=WIf@0c0=WIf@D0003oa03I\nfMT00240fMWI00<000000=WIf@3IfMT0>03IfMT50000okl0fMWI000Q0=WIf@800000?@3IfMT50000\nok/0fMWI000Q0=WIf@030000003IfMT0fMWI0440fMWI1@000?nf0=WIf@008@3IfMT00`000000fMWI\n0=WIf@150=WIf@H0003o/@3IfMT00240fMWI00<000000=WIf@3IfMT0BP3IfMT50000ojd0fMWI000Q\n0=WIf@030000003IfMT0fMWI04l0fMWI1@000?nX0=WIf@00203IfMT500000080fMWI0P0000030=WI\nf@<000000`3IfMT4000000<0fMWI00<000000=WIf@3IfMT0E03IfMT60000oj80fMWI000:0=WIf@03\n0000003IfMT0fMWI00L0fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@040=WI\nf@030000003IfMT0fMWI05T0fMWI1P000?nM0=WIf@002P3IfMT00`000000fMWI0=WIf@070=WIf@04\n0000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT00`3IfMT3000005h0fMWI1P000?nH0=WI\nf@002P3IfMT00`000000fMWI0=WIf@070=WIf@<000001@3IfMT00`000000fMWI0=WIf@020=WIf@03\n0000003IfMT0fMWI06@0fMWI1P000?nB0=WIf@002P3IfMT00`000000fMWI0=WIf@080=WIf@030000\n003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@1Y0=WIf@L0\n003oS03IfMT000T0fMWI0P00000:0=WIf@<000000`3IfMT2000000@0fMWI00<000000=WIf@3IfMT0\nK`3IfMT70000ohH0fMWI000Q0=WIf@030000003IfMT0fMWI07D0fMWI1`000?n00=WIf@008@3IfMT0\n0`000000fMWI0=WIf@1l0=WIf@L0003oN@3IfMT00240fMWI0P0000230=WIf@P0003oLP3IfMT00240\nfMWI00<000000=WIf@3IfMT0RP3IfMT80000ofX0fMWI000Q0=WIf@030000003IfMT0fMWI0980fMWI\n20000?mR0=WIf@008@3IfMT00`000000fMWI0=WIf@2J0=WIf@X0003oF03IfMT00240fMWI00<00000\n0=WIf@3IfMT0X`3IfMT;0000odh0fMWI000Q0=WIf@030000003IfMT0fMWI0:d0fMWI3P000?m10=WI\nf@008@3IfMT00`000000fMWI0=WIf@2j0=WIfA80003o<03IfMT00240fMWI0P00003=0=WIfA<0003o\n7@3IfMT00240fMWI00<000000=WIf@3IfMT0o03IfMT00240fMWI00<000000=WIf@3IfMT0o03IfMT0\n0240fMWI00<000000=WIf@3IfMT0o03IfMT00240fMWI00<000000=WIf@3IfMT0o03IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT002D0fMWI\n00<000000=WIf@3IfMT01@3IfMT300000280fMWI0`00000Q0=WIf@800000103IfMT00`000000fMWI\n0=WIf@050=WIf@800000203IfMT200000900fMWI000T0=WIf@030000003IfMT0fMWI00L0fMWI00<0\n00000=WIf@3IfMT0903IfMT00`000000fMWI0=WIf@0O0=WIf@030000003IfMT0fMWI00<0fMWI00<0\n00000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@080=WIf@030000003IfMT0fMWI08h0fMWI000S\n0=WIf@030000003IfMT0fMWI0080fMWI1@000000103IfMT000000000000000040=WIf@8000000`3I\nfMT3000000<0fMWI0`0000040=WIf@800000103IfMT2000000@0fMWI0`0000020=WIf@@000000P3I\nfMT500000080fMWI0P0000040=WIf@<000000`3IfMT2000000D0fMWI00<000000=WIf@3IfMT00`3I\nfMT00`000000fMWI0=WIf@030=WIf@030000003IfMT0fMWI0080fMWI0P0000040=WIf@030000003I\nfMT0fMWI08h0fMWI000S0=WIf@030000003IfMT0fMWI0080fMWI100000020=WIf@040000003IfMT0\nfMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000fMWI0=WIf@030=WIf@030000\n003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000030=WI\nf@030000003IfMT0fMWI0080fMWI100000030=WIf@8000001@3IfMT01@000000fMWI0=WIf@3IfMT0\n00000080fMWI00L000000=WIf@3IfMT0fMWI0000003IfMT0000000@0fMWI00<000000=WIf@3IfMT0\n0`3IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00D000000=WIf@3IfMT0fMWI0000002@0=WI\nf@005P3IfMT80000o`D0fMWI00<000000=WIf@3IfMT01@3IfMT010000000fMWI0=WIf@0000020=WI\nf@040000003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI000000050=WIf@030000003I\nfMT0fMWI0080fMWI00@000000=WIf@3IfMT00000203IfMT010000000fMWI0=WIf@0000030=WIf@03\n0000003IfMT0fMWI00D0fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@020=WI\nf@050000003IfMT0fMWI0=WIf@0000001@3IfMT010000000fMWI0=WIf@0000040=WIf@030000003I\nfMT0fMWI00<0fMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0fMWI08h0fMWI\n000S0=WIf@030000003IfMT0fMWI00<0fMWI0`0000000`3IfMT0000000000002000000@0fMWI0P00\n00040=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000X0fMWI1000\n00000`3IfMT0000000000002000000<0fMWI0`0000000`3IfMT00000000000020=WIf@8000000P3I\nfMT200000080fMWI1@0000001@3IfMT00000000000000000fMWI008000000`3IfMT00`000000fMWI\n0=WIf@030=WIf@030000003IfMT0fMWI00D000000`3IfMT00`000000fMWI0=WIf@2>0=WIf@00903I\nfMT00`000000fMWI0=WIf@0D0=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT0;03I\nfMT00`000000fMWI0=WIf@0:0=WIf@030000003IfMT0fMWI00@0fMWI00D000000=WIf@3IfMT0fMWI\n000000060=WIf@030000003IfMT0fMWI08h0fMWI000U0=WIf@030000003IfMT0fMWI0180fMWI0P00\n00040=WIf@800000:P3IfMT00`000000fMWI0=WIf@0=0=WIf@030000003IfMT0fMWI00D0fMWI0P00\n00080=WIf@800000T03IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI\n8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00?l0fMWI8@3IfMT00001"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-30.2449`, 1.50925`, 0.890933`, 0.000760935`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[TextData[List["\t\t\t\t\t\t", StyleBox["Diagram 4:", Rule[FontWeight, "Bold"]], " Gradually increasing ", StyleBox["gravity", Rule[FontFamily, "Courier"]], " when the rocket approaches the lunar surface\n"]], "Text"], Cell["plot( apollo.velocity );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .74402 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.47619 0.312017 0.47619 [\n[ 0 0 0 0 ]\n[ 1 .74402 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n% Start of sub-graphic\np\n0.0238095 0.0177147 0.97619 0.606318 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.0238095 0.0047619 0.60577 0.000295085 [\n[.2619 .59327 -6 -9 ]\n[.2619 .59327 6 0 ]\n[.5 .59327 -9 -9 ]\n[.5 .59327 9 0 ]\n[.7381 .59327 -9 -9 ]\n[.7381 .59327 9 0 ]\n[.97619 .59327 -9 -9 ]\n[.97619 .59327 9 0 ]\n[1.025 .60577 0 -6.28125 ]\n[1.025 .60577 10 6.28125 ]\n[.01131 .0156 -30 -4.5 ]\n[.01131 .0156 0 4.5 ]\n[.01131 .16314 -30 -4.5 ]\n[.01131 .16314 0 4.5 ]\n[.01131 .31069 -30 -4.5 ]\n[.01131 .31069 0 4.5 ]\n[.01131 .45823 -24 -4.5 ]\n[.01131 .45823 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.2619 .60577 m\n.2619 .61202 L\ns\n[(50)] .2619 .59327 0 1 Mshowa\n.5 .60577 m\n.5 .61202 L\ns\n[(100)] .5 .59327 0 1 Mshowa\n.7381 .60577 m\n.7381 .61202 L\ns\n[(150)] .7381 .59327 0 1 Mshowa\n.97619 .60577 m\n.97619 .61202 L\ns\n[(200)] .97619 .59327 0 1 Mshowa\n.125 Mabswid\n.07143 .60577 m\n.07143 .60952 L\ns\n.11905 .60577 m\n.11905 .60952 L\ns\n.16667 .60577 m\n.16667 .60952 L\ns\n.21429 .60577 m\n.21429 .60952 L\ns\n.30952 .60577 m\n.30952 .60952 L\ns\n.35714 .60577 m\n.35714 .60952 L\ns\n.40476 .60577 m\n.40476 .60952 L\ns\n.45238 .60577 m\n.45238 .60952 L\ns\n.54762 .60577 m\n.54762 .60952 L\ns\n.59524 .60577 m\n.59524 .60952 L\ns\n.64286 .60577 m\n.64286 .60952 L\ns\n.69048 .60577 m\n.69048 .60952 L\ns\n.78571 .60577 m\n.78571 .60952 L\ns\n.83333 .60577 m\n.83333 .60952 L\ns\n.88095 .60577 m\n.88095 .60952 L\ns\n.92857 .60577 m\n.92857 .60952 L\ns\n.25 Mabswid\n0 .60577 m\n1 .60577 L\ns\ngsave\n1.025 .60577 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n63.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n0.000 0.000 rmoveto\n1.000 setlinewidth\ngrestore\n.02381 .0156 m\n.03006 .0156 L\ns\n[(-2000)] .01131 .0156 1 0 Mshowa\n.02381 .16314 m\n.03006 .16314 L\ns\n[(-1500)] .01131 .16314 1 0 Mshowa\n.02381 .31069 m\n.03006 .31069 L\ns\n[(-1000)] .01131 .31069 1 0 Mshowa\n.02381 .45823 m\n.03006 .45823 L\ns\n[(-500)] .01131 .45823 1 0 Mshowa\n.125 Mabswid\n.02381 .04511 m\n.02756 .04511 L\ns\n.02381 .07462 m\n.02756 .07462 L\ns\n.02381 .10413 m\n.02756 .10413 L\ns\n.02381 .13363 m\n.02756 .13363 L\ns\n.02381 .19265 m\n.02756 .19265 L\ns\n.02381 .22216 m\n.02756 .22216 L\ns\n.02381 .25167 m\n.02756 .25167 L\ns\n.02381 .28118 m\n.02756 .28118 L\ns\n.02381 .34019 m\n.02756 .34019 L\ns\n.02381 .3697 m\n.02756 .3697 L\ns\n.02381 .39921 m\n.02756 .39921 L\ns\n.02381 .42872 m\n.02756 .42872 L\ns\n.02381 .48774 m\n.02756 .48774 L\ns\n.02381 .51724 m\n.02756 .51724 L\ns\n.02381 .54675 m\n.02756 .54675 L\ns\n.02381 .57626 m\n.02756 .57626 L\ns\n.25 Mabswid\n.02381 0 m\n.02381 .61803 L\ns\n0 0 1 r\n.001 w\n.02381 .01472 m\n.02571 .01868 L\n.02762 .02265 L\n.02952 .02664 L\n.03143 .03065 L\n.03333 .03468 L\n.03524 .03872 L\n.03714 .04278 L\n.03905 .04685 L\n.04095 .05095 L\n.04286 .05505 L\n.04476 .05918 L\n.04667 .06333 L\n.04857 .06749 L\n.05048 .07167 L\n.05238 .07587 L\n.05429 .08008 L\n.05619 .08432 L\n.0581 .08857 L\n.06 .09284 L\n.0619 .09713 L\n.06381 .10144 L\n.06571 .10577 L\n.06762 .11012 L\n.06952 .11448 L\n.07143 .11887 L\n.07333 .12327 L\n.07524 .12769 L\n.07714 .13214 L\n.07905 .1366 L\n.08095 .14109 L\n.08286 .14559 L\n.08476 .15012 L\n.08667 .15467 L\n.08857 .15923 L\n.09048 .16382 L\n.09238 .16843 L\n.09429 .17306 L\n.09619 .17772 L\n.0981 .18239 L\n.1 .18709 L\n.1019 .1918 L\n.10381 .19655 L\n.10571 .20131 L\n.10762 .2061 L\n.10952 .21091 L\n.11143 .21574 L\n.11333 .2206 L\n.11524 .22548 L\n.11714 .23039 L\nMistroke\n.11905 .23532 L\n.12095 .24027 L\n.12286 .24525 L\n.12476 .25025 L\n.12667 .25528 L\n.12857 .26034 L\n.13048 .26542 L\n.13238 .27053 L\n.13429 .27566 L\n.13619 .28082 L\n.1381 .286 L\n.14 .29122 L\n.1419 .29646 L\n.14381 .30173 L\n.14571 .30702 L\n.14762 .31235 L\n.14952 .3177 L\n.15143 .32308 L\n.15333 .3285 L\n.15524 .33394 L\n.15714 .33941 L\n.15905 .34491 L\n.16095 .35044 L\n.16286 .356 L\n.16476 .36159 L\n.16667 .36722 L\n.16857 .37288 L\n.17048 .37856 L\n.17238 .38428 L\n.17429 .39004 L\n.17619 .39583 L\n.1781 .40165 L\n.18 .4075 L\n.1819 .41339 L\n.18381 .41931 L\n.18571 .42527 L\n.18762 .43127 L\n.18952 .4373 L\n.19143 .44337 L\n.19333 .44947 L\n.19524 .45561 L\n.19714 .46179 L\n.19905 .46801 L\n.20095 .47427 L\n.20286 .48056 L\n.20476 .4869 L\n.20667 .49327 L\n.20857 .49969 L\n.21048 .50614 L\n.21238 .51264 L\nMistroke\n.21429 .51918 L\n.21619 .52577 L\n.2181 .53239 L\n.22 .53906 L\n.2219 .54578 L\n.22381 .55254 L\n.22571 .55934 L\n.22762 .56619 L\n.22952 .57309 L\n.22952 .57309 L\n.23143 .57316 L\n.23333 .57322 L\n.23524 .57329 L\n.23714 .57335 L\n.23905 .57342 L\n.24095 .57349 L\n.24286 .57355 L\n.24476 .57362 L\n.24667 .57368 L\n.24857 .57375 L\n.25048 .57382 L\n.25238 .57388 L\n.25429 .57395 L\n.25619 .57401 L\n.2581 .57408 L\n.26 .57415 L\n.2619 .57421 L\n.26381 .57428 L\n.26571 .57435 L\n.26762 .57441 L\n.26952 .57448 L\n.27143 .57455 L\n.27333 .57461 L\n.27524 .57468 L\n.27714 .57475 L\n.27905 .57481 L\n.28095 .57488 L\n.28286 .57495 L\n.28476 .57502 L\n.28667 .57508 L\n.28857 .57515 L\n.29048 .57522 L\n.29238 .57528 L\n.29429 .57535 L\n.29619 .57542 L\n.2981 .57549 L\n.3 .57555 L\n.3019 .57562 L\n.30381 .57569 L\n.30571 .57576 L\nMistroke\n.30762 .57583 L\n.30952 .57589 L\n.31143 .57596 L\n.31333 .57603 L\n.31524 .5761 L\n.31714 .57616 L\n.31905 .57623 L\n.32095 .5763 L\n.32286 .57637 L\n.32476 .57644 L\n.32667 .57651 L\n.32857 .57657 L\n.33048 .57664 L\n.33238 .57671 L\n.33429 .57678 L\n.33619 .57685 L\n.3381 .57692 L\n.34 .57699 L\n.3419 .57705 L\n.34381 .57712 L\n.34571 .57719 L\n.34762 .57726 L\n.34952 .57733 L\n.35143 .5774 L\n.35333 .57747 L\n.35524 .57754 L\n.35714 .57761 L\n.35905 .57768 L\n.36095 .57774 L\n.36286 .57781 L\n.36476 .57788 L\n.36667 .57795 L\n.36857 .57802 L\n.37048 .57809 L\n.37238 .57816 L\n.37429 .57823 L\n.37619 .5783 L\n.3781 .57837 L\n.38 .57844 L\n.3819 .57851 L\n.38381 .57858 L\n.38571 .57865 L\n.38762 .57872 L\n.38952 .57879 L\n.39143 .57886 L\n.39333 .57893 L\n.39524 .579 L\n.39714 .57907 L\n.39905 .57914 L\n.40095 .57921 L\nMistroke\n.40286 .57928 L\n.40476 .57935 L\n.40667 .57942 L\n.40857 .57949 L\n.41048 .57957 L\n.41238 .57964 L\n.41429 .57971 L\n.41619 .57978 L\n.4181 .57985 L\n.42 .57992 L\n.4219 .57999 L\n.42381 .58006 L\n.42571 .58013 L\n.42762 .5802 L\n.42952 .58028 L\n.43143 .58035 L\n.43333 .58042 L\n.43524 .58049 L\n.43714 .58056 L\n.43905 .58063 L\n.44095 .5807 L\n.44286 .58078 L\n.44476 .58085 L\n.44667 .58092 L\n.44857 .58099 L\n.45048 .58106 L\n.45238 .58114 L\n.45429 .58121 L\n.45619 .58128 L\n.4581 .58135 L\n.46 .58142 L\n.4619 .5815 L\n.46381 .58157 L\n.46571 .58164 L\n.46762 .58171 L\n.46952 .58179 L\n.47143 .58186 L\n.47333 .58193 L\n.47524 .582 L\n.47714 .58208 L\n.47905 .58215 L\n.48095 .58222 L\n.48286 .5823 L\n.48476 .58237 L\n.48667 .58244 L\n.48857 .58251 L\n.49048 .58259 L\n.49238 .58266 L\n.49429 .58273 L\n.49619 .58281 L\nMistroke\n.4981 .58288 L\n.5 .58295 L\n.5019 .58303 L\n.50381 .5831 L\n.50571 .58318 L\n.50762 .58325 L\n.50952 .58332 L\n.51143 .5834 L\n.51333 .58347 L\n.51524 .58354 L\n.51714 .58362 L\n.51905 .58369 L\n.52095 .58377 L\n.52286 .58384 L\n.52476 .58392 L\n.52667 .58399 L\n.52857 .58406 L\n.53048 .58414 L\n.53238 .58421 L\n.53429 .58429 L\n.53619 .58436 L\n.5381 .58444 L\n.54 .58451 L\n.5419 .58459 L\n.54381 .58466 L\n.54571 .58474 L\n.54762 .58481 L\n.54952 .58489 L\n.55143 .58496 L\n.55333 .58504 L\n.55524 .58511 L\n.55714 .58519 L\n.55905 .58526 L\n.56095 .58534 L\n.56286 .58541 L\n.56476 .58549 L\n.56667 .58556 L\n.56857 .58564 L\n.57048 .58571 L\n.57238 .58579 L\n.57429 .58587 L\n.57619 .58594 L\n.5781 .58602 L\n.58 .58609 L\n.5819 .58617 L\n.58381 .58625 L\n.58571 .58632 L\n.58762 .5864 L\n.58952 .58647 L\n.59143 .58655 L\nMistroke\n.59333 .58663 L\n.59524 .5867 L\n.59714 .58678 L\n.59905 .58686 L\n.60095 .58693 L\n.60286 .58701 L\n.60476 .58709 L\n.60667 .58716 L\n.60857 .58724 L\n.61048 .58732 L\n.61238 .58739 L\n.61429 .58747 L\n.61619 .58755 L\n.6181 .58763 L\n.62 .5877 L\n.6219 .58778 L\n.62381 .58786 L\n.62571 .58794 L\n.62762 .58801 L\n.62952 .58809 L\n.63143 .58817 L\n.63333 .58825 L\n.63524 .58832 L\n.63714 .5884 L\n.63905 .58848 L\n.64095 .58856 L\n.64286 .58864 L\n.64476 .58871 L\n.64667 .58879 L\n.64857 .58887 L\n.65048 .58895 L\n.65238 .58903 L\n.65429 .58911 L\n.65619 .58918 L\n.6581 .58926 L\n.66 .58934 L\n.6619 .58942 L\n.66381 .5895 L\n.66571 .58958 L\n.66762 .58966 L\n.66952 .58973 L\n.67143 .58981 L\n.67333 .58989 L\n.67524 .58997 L\n.67714 .59005 L\n.67905 .59013 L\n.68095 .59021 L\n.68286 .59029 L\n.68476 .59037 L\n.68667 .59045 L\nMistroke\n.68857 .59053 L\n.69048 .59061 L\n.69238 .59069 L\n.69429 .59077 L\n.69619 .59085 L\n.6981 .59093 L\n.7 .59101 L\n.7019 .59109 L\n.70381 .59117 L\n.70571 .59125 L\n.70762 .59133 L\n.70952 .59141 L\n.71143 .59149 L\n.71333 .59157 L\n.71524 .59165 L\n.71714 .59173 L\n.71905 .59181 L\n.72095 .59189 L\n.72286 .59197 L\n.72476 .59205 L\n.72667 .59213 L\n.72857 .59221 L\n.73048 .59229 L\n.73238 .59238 L\n.73429 .59246 L\n.73619 .59254 L\n.7381 .59262 L\n.74 .5927 L\n.7419 .59278 L\n.74381 .59286 L\n.74571 .59294 L\n.74762 .59303 L\n.74952 .59311 L\n.75143 .59319 L\n.75333 .59327 L\n.75524 .59335 L\n.75714 .59344 L\n.75905 .59352 L\n.76095 .5936 L\n.76286 .59368 L\n.76476 .59376 L\n.76667 .59385 L\n.76857 .59393 L\n.77048 .59401 L\n.77238 .59409 L\n.77429 .59418 L\n.77619 .59426 L\n.7781 .59434 L\n.78 .59442 L\n.7819 .59451 L\nMistroke\n.78381 .59459 L\n.78571 .59467 L\n.78762 .59476 L\n.78952 .59484 L\n.79143 .59492 L\n.79333 .595 L\n.79524 .59509 L\n.79714 .59517 L\n.79905 .59525 L\n.80095 .59534 L\n.80286 .59542 L\n.80476 .59551 L\n.80667 .59559 L\n.80857 .59567 L\n.81048 .59576 L\n.81238 .59584 L\n.81429 .59592 L\n.81619 .59601 L\n.8181 .59609 L\n.82 .59618 L\n.8219 .59626 L\n.82381 .59635 L\n.82571 .59643 L\n.82762 .59651 L\n.82952 .5966 L\n.83143 .59668 L\n.83333 .59677 L\n.83524 .59685 L\n.83714 .59694 L\n.83905 .59702 L\n.84095 .59711 L\n.84286 .59719 L\n.84476 .59728 L\n.84667 .59736 L\n.84857 .59745 L\n.85048 .59753 L\n.85238 .59762 L\n.85429 .5977 L\n.85619 .59779 L\n.8581 .59787 L\n.86 .59796 L\n.8619 .59805 L\n.86381 .59813 L\n.86571 .59822 L\n.86762 .5983 L\n.86952 .59839 L\n.87143 .59848 L\n.87333 .59856 L\n.87524 .59865 L\n.87714 .59873 L\nMistroke\n.87905 .59882 L\n.88095 .59891 L\n.88286 .59899 L\n.88476 .59908 L\n.88667 .59917 L\n.88857 .59925 L\n.89048 .59934 L\n.89238 .59943 L\n.89429 .59951 L\n.89619 .5996 L\n.8981 .59969 L\n.9 .59978 L\n.9019 .59986 L\n.90381 .59995 L\n.90571 .60004 L\n.90762 .60012 L\n.90952 .60021 L\n.91143 .6003 L\n.91333 .60039 L\n.91524 .60047 L\n.91714 .60056 L\n.91905 .60065 L\n.92095 .60074 L\n.92286 .60083 L\n.92476 .60091 L\n.92667 .601 L\n.92857 .60109 L\n.93048 .60118 L\n.93238 .60127 L\n.93429 .60136 L\n.93619 .60144 L\n.9381 .60153 L\n.94 .60162 L\n.9419 .60171 L\n.94381 .6018 L\n.94571 .60189 L\n.94762 .60198 L\n.94952 .60207 L\n.95143 .60215 L\n.95333 .60224 L\n.95524 .60233 L\n.95714 .60242 L\n.95905 .60251 L\n.96095 .6026 L\n.96286 .60269 L\n.96476 .60278 L\n.96667 .60287 L\n.96857 .60296 L\n.97048 .60305 L\n.97238 .60314 L\nMistroke\n.97429 .60323 L\n.97619 .60332 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\nMathSubEnd\nP\n% End of sub-graphic\n% Start of sub-graphic\np\n0.0714286 0.583445 0.547619 0.726302 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.00611247 0.0611247 0.0220588 0.220588 [\n[ 0 0 0 0 ]\n[ 1 .3 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 .3 L\n0 .3 L\nclosepath\nclip\nnewpath\n% Start of sub-graphic\np\n0.0110024 0.0397059 0.0721271 0.260294 MathSubStart\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.5 0.5 0.5 0.5 [\n[ 0 0 0 0 ]\n[ 1 1 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 0 m\n1 0 L\n1 1 L\n0 1 L\nclosepath\nclip\nnewpath\n0 0 1 r\n.00187 w\n[ ] 0 setdash\n0 .5 m\n1 .5 L\ns\nMathSubEnd\nP\n% End of sub-graphic\n0 g\ngsave\n.08007 .15 -61 -10.2813 Mabsadd m\n1 1 Mabs scale\ncurrentpoint translate\n0 20.5625 translate 1 -1 scale\n/g { setgray} bind def\n/k { setcmykcolor} bind def\n/p { gsave} bind def\n/r { setrgbcolor} bind def\n/w { setlinewidth} bind def\n/C { curveto} bind def\n/F { fill} bind def\n/L { lineto} bind def\n/rL { rlineto} bind def\n/P { grestore} bind def\n/s { stroke} bind def\n/S { show} bind def\n/N {currentpoint 3 -1 roll show moveto} bind def\n/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def\n/m { moveto} bind def\n/Mr { rmoveto} bind def\n/Mx {currentpoint exch pop moveto} bind def\n/My {currentpoint pop exch moveto} bind def\n/X {0 rmoveto} bind def\n/Y {0 exch rmoveto} bind def\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n63.000 12.813 moveto\n(H) show\n69.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(apollo) show\n105.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(.) show\n111.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(velocity) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n159.000 12.813 moveto\n(L) show\n165.000 12.813 moveto\n(@) show\n171.000 12.813 moveto\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n(t) show\n%%IncludeResource: font Mathematica2Mono\n%%IncludeFont: Mathematica2Mono\n/Mathematica2Mono findfont 10.000 scalefont\n[1 0 0 -1 0 0 ] makefont setfont\n0.000 0.000 0.000 setrgbcolor\n177.000 12.813 moveto\n(D) show\n1.000 setlinewidth\ngrestore\nMathSubEnd\nP\n% End of sub-graphic\n0 0 m\n1 0 L\n1 .74402 L\n0 .74402 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 214.25`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000eR000`400?l00000o`00003oo`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ\n0=WIf@00o`3IfMTQ0=WIf@009`3IfMT00`000000fMWI0=WIf@3f0=WIf@009`3IfMT00`000000fMWI\n0=WIf@3f0=WIf@003P3IfMT4000000<0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<0\n00000=WIf@3IfMT0mP3IfMT000h0fMWI00<000000=WIf@3IfMT00`3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000\nfMWI0=WIf@3f0=WIf@003`3IfMT00`000000fMWI0=WIf@020=WIf@040000003IfMT0fMWI00000080\nfMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000o`000000\n00000?H0fMWI00080=WIf@@00000103IfMT01@000000fMWI0=WIf@3IfMT000000080fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000020=WIf@060000003IfMT0fMWI0=WIf@00\n0000003om`3IfMT000h0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000003o\n0=WIf@3f0=WIf@003`3IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT2000000@0fMWI00<0\n00000=WIf@000?l0mP3IfMT002L0fMWI00@000000=WIf@3IfMT0003om@3IfMT002L0fMWI00@00000\n0=WIf@3IfMT0003om@3IfMT002L0fMWI00@000000=WIf@3IfMT0003om@3IfMT002L0fMWI0P000002\n0=WIf@030000o`3IfMT0fMWI0?80fMWI000W0=WIf@050000003IfMT0fMWI0=WIf@000?l0m03IfMT0\n02L0fMWI00<000000=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@3a0=WIf@009`3IfMT00`000000\nfMWI0=WIf@020=WIf@030000o`3IfMT0fMWI0?40fMWI000W0=WIf@030000003IfMT0fMWI00<0fMWI\n00<0003o0=WIf@3IfMT0l03IfMT002L0fMWI00<000000=WIf@3IfMT00`3IfMT00`000?l0fMWI0=WI\nf@3`0=WIf@009`3IfMT00`000000fMWI0=WIf@040=WIf@030000o`3IfMT0fMWI0>l0fMWI000W0=WI\nf@8000001@3IfMT00`000?l0fMWI0=WIf@3_0=WIf@009`3IfMT00`000000fMWI0=WIf@040=WIf@03\n0000o`3IfMT0fMWI0>l0fMWI000W0=WIf@030000003IfMT0fMWI00D0fMWI00<0003o0=WIf@3IfMT0\nkP3IfMT002L0fMWI00<000000=WIf@3IfMT01@3IfMT00`000?l0fMWI0=WIf@3^0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@060=WIf@030000o`3IfMT0fMWI0>d0fMWI000W0=WIf@030000003IfMT0fMWI\n00H0fMWI00<0003o0=WIf@3IfMT0k@3IfMT002L0fMWI0P0000080=WIf@030000o`3IfMT0fMWI0>`0\nfMWI000W0=WIf@030000003IfMT0fMWI00L0fMWI00<0003o0=WIf@3IfMT0k03IfMT002L0fMWI00<0\n00000=WIf@3IfMT01`3IfMT00`000?l0fMWI0=WIf@3/0=WIf@009`3IfMT00`000000fMWI0=WIf@08\n0=WIf@030000o`3IfMT0fMWI0>/0fMWI000W0=WIf@030000003IfMT0fMWI00P0fMWI00<0003o0=WI\nf@3IfMT0j`3IfMT002L0fMWI00<000000=WIf@3IfMT02@3IfMT00`000?l0fMWI0=WIf@3Z0=WIf@00\n9`3IfMT00`000000fMWI0=WIf@090=WIf@030000o`3IfMT0fMWI0>X0fMWI000W0=WIf@8000002`3I\nfMT00`000?l0fMWI0=WIf@3Y0=WIf@009`3IfMT00`000000fMWI0=WIf@0:0=WIf@030000o`3IfMT0\nfMWI0>T0fMWI000W0=WIf@030000003IfMT0fMWI00/0fMWI00<0003o0=WIf@3IfMT0j03IfMT002L0\nfMWI00<000000=WIf@3IfMT02`3IfMT00`000?l0fMWI0=WIf@3X0=WIf@009`3IfMT00`000000fMWI\n0=WIf@0;0=WIf@030000o`3IfMT0fMWI0>P0fMWI000>0=WIf@D000000P3IfMT2000000@0fMWI0P00\n00040=WIf@800000103IfMT00`000000fMWI0=WIf@0<0=WIf@030000o`3IfMT0fMWI0>L0fMWI000@\n0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00D000000=WIf@3IfMT0fMWI0000000>0=WIf@030000o`3IfMT0fMWI\n0>L0fMWI000@0=WIf@030000003IfMT0fMWI00@0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI0=WIf@<000003@3IfMT00`000?l0fMWI0=WI\nf@3V0=WIf@00203IfMT4000000@0fMWI00<000000=WIf@3IfMT00P3IfMT2000000<0fMWI00@00000\n0=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI00d0fMWI\n00<0003o0=WIf@3IfMT0iP3IfMT00100fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WI\nf@020=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000`3IfMT00`000000\nfMWI0=WIf@0>0=WIf@030000o`3IfMT0fMWI0>D0fMWI000?0=WIf@800000103IfMT3000000<0fMWI\n0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0>0=WIf@030000o`3IfMT0fMWI0>D0fMWI\n000W0=WIf@030000003IfMT0fMWI00h0fMWI00<0003o0=WIf@3IfMT0i@3IfMT002L0fMWI00<00000\n0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@3T0=WIf@009`3IfMT00`000000fMWI0=WIf@0?0=WI\nf@030000o`3IfMT0fMWI0>@0fMWI000W0=WIf@800000403IfMT00`000?l0fMWI0=WIf@3T0=WIf@00\n9`3IfMT00`000000fMWI0=WIf@0@0=WIf@030000o`3IfMT0fMWI0><0fMWI000W0=WIf@030000003I\nfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0h`3IfMT002L0fMWI00<000000=WIf@3IfMT04@3IfMT0\n0`000?l0fMWI0=WIf@3R0=WIf@009`3IfMT00`000000fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI\n0>80fMWI000W0=WIf@030000003IfMT0fMWI0180fMWI00<0003o0=WIf@3IfMT0h@3IfMT002L0fMWI\n0P00000C0=WIf@030000o`3IfMT0fMWI0>40fMWI000W0=WIf@030000003IfMT0fMWI0180fMWI00<0\n003o0=WIf@3IfMT0h@3IfMT002L0fMWI00<000000=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@3P\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0C0=WIf@030000o`3IfMT0fMWI0>00fMWI000W0=WIf@03\n0000003IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT0h03IfMT002L0fMWI00<000000=WIf@3IfMT0\n503IfMT00`000?l0fMWI0=WIf@3O0=WIf@009`3IfMT00`000000fMWI0=WIf@0D0=WIf@030000o`3I\nfMT0fMWI0=l0fMWI000W0=WIf@8000005P3IfMT00`000?l0fMWI0=WIf@3N0=WIf@009`3IfMT00`00\n0000fMWI0=WIf@0E0=WIf@030000o`3IfMT0fMWI0=h0fMWI000W0=WIf@030000003IfMT0fMWI01D0\nfMWI00<0003o0=WIf@3IfMT0gP3IfMT002L0fMWI00<000000=WIf@3IfMT05P3IfMT00`000?l0fMWI\n0=WIf@3M0=WIf@009`3IfMT00`000000fMWI0=WIf@0F0=WIf@030000o`3IfMT0fMWI0=d0fMWI000W\n0=WIf@030000003IfMT0fMWI01H0fMWI00<0003o0=WIf@3IfMT0g@3IfMT002L0fMWI00<000000=WI\nf@3IfMT05`3IfMT00`000?l0fMWI0=WIf@3L0=WIf@009`3IfMT2000001P0fMWI00<0003o0=WIf@3I\nfMT0g03IfMT002L0fMWI00<000000=WIf@3IfMT0603IfMT00`000?l0fMWI0=WIf@3K0=WIf@009`3I\nfMT00`000000fMWI0=WIf@0H0=WIf@030000o`3IfMT0fMWI0=/0fMWI000W0=WIf@030000003IfMT0\nfMWI01P0fMWI00<0003o0=WIf@3IfMT0f`3IfMT002L0fMWI00<000000=WIf@3IfMT06@3IfMT00`00\n0?l0fMWI0=WIf@3J0=WIf@003P3IfMT500000080fMWI0P0000040=WIf@800000103IfMT2000000@0\nfMWI00<000000=WIf@3IfMT06@3IfMT00`000?l0fMWI0=WIf@3J0=WIf@00403IfMT01@000000fMWI\n0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@000002\n0=WIf@050000003IfMT0fMWI0=WIf@0000006`3IfMT00`000?l0fMWI0=WIf@3J0=WIf@00403IfMT0\n1@000000fMWI0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI\n0=WIf@0000020=WIf@040000003IfMT0fMWI0=WIf@<000006P3IfMT00`000?l0fMWI0=WIf@3I0=WI\nf@00203IfMT4000000@0fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI\n00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0000001`0fMWI\n00<0003o0=WIf@3IfMT0f@3IfMT00100fMWI00D000000=WIf@3IfMT0fMWI000000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT000000P3IfMT01@000000fMWI0=WIf@3IfMT0\n000001`0fMWI00<0003o0=WIf@3IfMT0f@3IfMT000l0fMWI0P0000040=WIf@800000103IfMT20000\n00@0fMWI0P0000040=WIf@030000003IfMT0fMWI01/0fMWI00<0003o0=WIf@3IfMT0f03IfMT002L0\nfMWI00<000000=WIf@3IfMT06`3IfMT00`000?l0fMWI0=WIf@3H0=WIf@009`3IfMT00`000000fMWI\n0=WIf@0L0=WIf@030000o`3IfMT0fMWI0=L0fMWI000W0=WIf@030000003IfMT0fMWI01`0fMWI00<0\n003o0=WIf@3IfMT0e`3IfMT002L0fMWI0P00000M0=WIf@030000o`3IfMT0fMWI0=L0fMWI000W0=WI\nf@030000003IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT0e`3IfMT002L0fMWI00<000000=WIf@3I\nfMT07@3IfMT00`000?l0fMWI0=WIf@3F0=WIf@009`3IfMT00`000000fMWI0=WIf@0M0=WIf@030000\no`3IfMT0fMWI0=H0fMWI000W0=WIf@030000003IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT0eP3I\nfMT002L0fMWI00<000000=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@3E0=WIf@009`3IfMT20000\n01l0fMWI00<0003o0=WIf@3IfMT0e@3IfMT002L0fMWI00<000000=WIf@3IfMT07`3IfMT00`000?l0\nfMWI0=WIf@3D0=WIf@009`3IfMT00`000000fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI0=@0fMWI\n000W0=WIf@030000003IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT0e03IfMT002L0fMWI00<00000\n0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@3C0=WIf@009`3IfMT00`000000fMWI0=WIf@0P0=WI\nf@030000o`3IfMT0fMWI0=<0fMWI000W0=WIf@030000003IfMT0fMWI0200fMWI00<0003o0=WIf@3I\nfMT0d`3IfMT002L0fMWI0P00000Q0=WIf@030000o`3IfMT0fMWI0=<0fMWI000W0=WIf@030000003I\nfMT0fMWI0240fMWI00<0003o0=WIf@3IfMT0dP3IfMT002L0fMWI00<000000=WIf@3IfMT08@3IfMT0\n0`000?l0fMWI0=WIf@3B0=WIf@009`3IfMT00`000000fMWI0=WIf@0R0=WIf@030000o`3IfMT0fMWI\n0=40fMWI000W0=WIf@030000003IfMT0fMWI0280fMWI00<0003o0=WIf@3IfMT0d@3IfMT002L0fMWI\n00<000000=WIf@3IfMT08P3IfMT00`000?l0fMWI0=WIf@3A0=WIf@009`3IfMT00`000000fMWI0=WI\nf@0S0=WIf@030000o`3IfMT0fMWI0=00fMWI000W0=WIf@800000903IfMT00`000?l0fMWI0=WIf@3@\n0=WIf@009`3IfMT00`000000fMWI0=WIf@0S0=WIf@030000o`3IfMT0fMWI0=00fMWI000W0=WIf@03\n0000003IfMT0fMWI02<0fMWI00<0003o0=WIf@3IfMT0d03IfMT002L0fMWI00<000000=WIf@3IfMT0\n903IfMT00`000?l0fMWI0=WIf@3?0=WIf@009`3IfMT00`000000fMWI0=WIf@0T0=WIf@030000o`3I\nfMT0fMWI0<l0fMWI000E0=WIf@800000103IfMT2000000@0fMWI0P0000040=WIf@030000003IfMT0\nfMWI02@0fMWI00<0003o0=WIf@3IfMT0c`3IfMT001@0fMWI00@000000=WIf@3IfMT000000P3IfMT0\n10000000fMWI0=WIf@0000020=WIf@040000003IfMT0fMWI000000<0fMWI00<000000=WIf@3IfMT0\n9@3IfMT00`000?l0fMWI0=WIf@3>0=WIf@005`3IfMT010000000fMWI0=WIf@0000020=WIf@040000\n003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0fMWI0`00000U0=WIf@030000o`3IfMT0fMWI\n0<h0fMWI000>0=WIf@@000000`3IfMT2000000<0fMWI00@000000=WIf@3IfMT000000P3IfMT01000\n0000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI02D0fMWI00<0003o0=WIf@3IfMT0cP3IfMT0\n01D0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000020=WIf@040000003IfMT0\nfMWI000000<0fMWI00<000000=WIf@3IfMT09P3IfMT00`000?l0fMWI0=WIf@3=0=WIf@005@3IfMT3\n000000<0fMWI0P0000040=WIf@800000103IfMT00`000000fMWI0=WIf@0V0=WIf@030000o`3IfMT0\nfMWI0<d0fMWI000W0=WIf@030000003IfMT0fMWI02H0fMWI00<0003o0=WIf@3IfMT0c@3IfMT002L0\nfMWI00<000000=WIf@3IfMT09`3IfMT00`000?l0fMWI0=WIf@3<0=WIf@009`3IfMT00`000000fMWI\n0=WIf@0W0=WIf@030000o`3IfMT0fMWI0<`0fMWI000W0=WIf@800000:03IfMT00`000?l0fMWI0=WI\nf@3<0=WIf@009`3IfMT00`000000fMWI0=WIf@0W0=WIf@030000o`3IfMT0fMWI0<`0fMWI000W0=WI\nf@030000003IfMT0fMWI02P0fMWI00<0003o0=WIf@3IfMT0b`3IfMT002L0fMWI00<000000=WIf@3I\nfMT0:03IfMT00`000?l0fMWI0=WIf@3;0=WIf@009`3IfMT00`000000fMWI0=WIf@0X0=WIf@030000\no`3IfMT0fMWI0</0fMWI000W0=WIf@030000003IfMT0fMWI02T0fMWI00<0003o0=WIf@3IfMT0bP3I\nfMT002L0fMWI0P00000Z0=WIf@030000o`3IfMT0fMWI0<X0fMWI000W0=WIf@030000003IfMT0fMWI\n02T0fMWI00<0003o0=WIf@3IfMT0bP3IfMT002L0fMWI00<000000=WIf@3IfMT0:P3IfMT00`000?l0\nfMWI0=WIf@390=WIf@009`3IfMT00`000000fMWI0=WIf@0Z0=WIf@030000o`3IfMT0fMWI0<T0fMWI\n000W0=WIf@030000003IfMT0fMWI02X0fMWI00<0003o0=WIf@3IfMT0b@3IfMT002L0fMWI00<00000\n0=WIf@3IfMT0:P3IfMT00`000?l0fMWI0=WIf@390=WIf@009`3IfMT00`000000fMWI0=WIf@0Z0=WI\nf@030000o`3IfMT0fMWI0<T0fMWI000W0=WIf@800000;03IfMT00`000?l0fMWI0=WIf@380=WIf@00\n9`3IfMT00`000000fMWI0=WIf@0[0=WIf@030000o`3IfMT0fMWI0<P0fMWI000W0=WIf@030000003I\nfMT0fMWI02/0fMWI00<0003o0=WIf@3IfMT0b03IfMT002L0fMWI00<000000=WIf@3IfMT0;03IfMT0\n10000?l0fMWI0=WIf@3IfMT2000000@0fMWI0P00000[0=WIf@D000000P3IfMT2000000@0fMWI0P00\n000X0=WIf@D000000P3IfMT2000000@0fMWI0P00000W0=WIf@@000000`3IfMT2000000@0fMWI0P00\n000G0=WIf@009`3IfMT00`000000fMWI0=WIf@0/0=WIf@040000o`3IfMT0fMWI00000080fMWI00@0\n00000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0Z0=WIf@050000003IfMT0fMWI0=WIf@00\n00000P3IfMT010000000fMWI0=WIf@0000020=WIf@030000003IfMT0fMWI02L0fMWI00D000000=WI\nf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0903I\nfMT00`000000fMWI0=WIf@030=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT0\n00005P3IfMT002L0fMWI00<000000=WIf@3IfMT0;03IfMT00`000?l0fMWI0=WIf@030=WIf@040000\n003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0:P3IfMT01@000000fMWI0=WIf@3IfMT00000\n0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0W0=WIf@030000003IfMT0\nfMWI00@0fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0U0=WIf@030000003I\nfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00000F0=WIf@00\n9`3IfMT00`000000fMWI0=WIf@0]0=WIf@/0003o00<000000=WIf@3IfMT0:P3IfMT01@000000fMWI\n0=WIf@3IfMT000000080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@0W0=WI\nf@030000003IfMT0fMWI0080fMWI0P0000030=WIf@040000003IfMT0fMWI000002P0fMWI00D00000\n0=WIf@3IfMT0fMWI000000020=WIf@040000003IfMT0fMWI00000080fMWI00<000000=WIf@3IfMT0\n503IfMT002L0fMWI0P00000a0=WIf@030000003IfMT0fMWI0080fMWI00<000000=WIf@3IfMT07000\n0?lA0=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT010000000fMWI0=WIf@0000020=WIf@03\n0000003IfMT0fMWI02L0fMWI00<000000=WIf@3IfMT00P3IfMT00`000000fMWI0=WIf@020=WIf@04\n0000003IfMT0fMWI000002H0fMWI00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@00\n00020=WIf@040000003IfMT0fMWI000001H0fMWI000W0=WIf@030000003IfMT0fMWI0300fMWI0`00\n00030=WIf@800000703IfMTJ0000o`80fMWI0P00000Y0=WIf@800000103IfMT3000000<0fMWI0P00\n000X0=WIf@800000103IfMT2000000@0fMWI0P00000G0=WIf@009`3IfMT00`000000fMWI0=WIf@1^\n0=WIfAX0003oKP3IfMT002L0fMWI00<000000=WIf@3IfMT0Q`3IfMTI0000oeH0fMWI000W0=WIf@03\n0000003IfMT0fMWI09l0fMWI60000?lo0=WIf@009`3IfMT00`000000fMWI0=WIf@2f0=WIfAL0003o\n6P3IfMT2000000d0fMWI000W0=WIf@030000003IfMT0fMWI0<`0fMWI3@000?l=0=WIf@040000003I\nfMT0fMWI000000`0fMWI000R0=WIfNL000001`3IfMT00`000000fMWI0=WIf@0=0=WIf@009`3IfMT0\n0`000000fMWI0=WIf@0d0=WIf@030000003IfMT0fMWI03@0fMWI00<000000=WIf@3IfMT0=03IfMT0\n0`000000fMWI0=WIf@0c0=WIf@030000003IfMT0fMWI00X0fMWI1@00000<0=WIf@009`3IfMT00`00\n0000fMWI0=WIf@3V0=WIf@030000003IfMT0fMWI00d0fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI003o0=WIfB40fMWI\n003o0=WIfB40fMWI003o0=WIfB40fMWI000U0=WIf@030000003IfMT0fMWI00D0fMWI0`00001<0=WI\nf@800000103IfMT00`000000fMWI0=WIf@050=WIf@800000203IfMT2000008X0fMWI000T0=WIf@03\n0000003IfMT0fMWI00L0fMWI00<000000=WIf@3IfMT0C03IfMT00`000000fMWI0=WIf@030=WIf@03\n0000003IfMT0fMWI00@0fMWI00<000000=WIf@3IfMT0203IfMT00`000000fMWI0=WIf@280=WIf@00\n8`3IfMT00`000000fMWI0=WIf@020=WIf@D0000000@0fMWI0000000000000000103IfMT2000000<0\nfMWI0`0000030=WIf@<00000103IfMT2000000@0fMWI0P0000040=WIf@800000103IfMT300000080\nfMWI0`0000040=WIf@800000103IfMT3000000<0fMWI0`0000030=WIf@8000001@3IfMT00`000000\nfMWI0=WIf@030=WIf@030000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00P3IfMT2000000@0\nfMWI00<000000=WIf@3IfMT0R03IfMT002<0fMWI00<000000=WIf@3IfMT00P3IfMT400000080fMWI\n00@000000=WIf@3IfMT000000P3IfMT010000000fMWI0=WIf@0000030=WIf@030000003IfMT0fMWI\n00<0fMWI00<000000=WIf@3IfMT00P3IfMT010000000fMWI0=WIf@0000090=WIf@8000000`3IfMT0\n0`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI0080fMWI00@000000=WIf@3IfMT000000P3I\nfMT00`000000fMWI0=WIf@050=WIf@050000003IfMT0fMWI0=WIf@0000000P3IfMT01`000000fMWI\n0=WIf@3IfMT000000=WIf@000000103IfMT00`000000fMWI0=WIf@030=WIf@050000003IfMT0fMWI\n0=WIf@0000000P3IfMT01@000000fMWI0=WIf@3IfMT0000008X0fMWI000F0=WIf@P0003o1@3IfMT0\n0`000000fMWI0=WIf@050=WIf@040000003IfMT0fMWI00000080fMWI00@000000=WIf@3IfMT00000\n0P3IfMT01@000000fMWI0=WIf@3IfMT0000000D0fMWI00<000000=WIf@3IfMT00P3IfMT010000000\nfMWI0=WIf@0000080=WIf@040000003IfMT0fMWI00000080fMWI100000030=WIf@030000003IfMT0\nfMWI0080fMWI00@000000=WIf@3IfMT000000P3IfMT00`000000fMWI0=WIf@050=WIf@050000003I\nfMT0fMWI0=WIf@0000001@3IfMT010000000fMWI0=WIf@0000040=WIf@030000003IfMT0fMWI00<0\nfMWI00D000000=WIf@3IfMT0fMWI000000060=WIf@030000003IfMT0fMWI08P0fMWI000S0=WIf@03\n0000003IfMT0fMWI00<0fMWI0`0000000`3IfMT0000000000002000000@0fMWI0P0000040=WIf@03\n0000003IfMT0fMWI00<0fMWI00<000000=WIf@3IfMT00`3IfMT2000000P0fMWI0P0000020=WIf@80\n00000P3IfMT2000000@0fMWI00<000000=WIf@3IfMT00`3IfMT2000000@0fMWI0`0000030=WIf@80\n00000P3IfMT5000000050=WIf@00000000000000003IfMT00P0000030=WIf@030000003IfMT0fMWI\n00<0fMWI00<000000=WIf@3IfMT01@0000030=WIf@030000003IfMT0fMWI08P0fMWI000T0=WIf@03\n0000003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT00`3IfMT00`000000fMWI0=WIf@0K0=WIf@03\n0000003IfMT0fMWI01@0fMWI00<000000=WIf@3IfMT02P3IfMT00`000000fMWI0=WIf@040=WIf@05\n0000003IfMT0fMWI0=WIf@0000001P3IfMT00`000000fMWI0=WIf@280=WIf@009@3IfMT00`000000\nfMWI0=WIf@0B0=WIf@800000103IfMT2000001`0fMWI0P00000B0=WIf@030000003IfMT0fMWI00d0\nfMWI00<000000=WIf@3IfMT01@3IfMT2000000P0fMWI0P00002:0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3IfMTQ0=WIf@00o`3I\nfMTQ0=WIf@00"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[213.25`, 0]], List[-1.05141`, -0.655244`, 0.00732689`, 0.00732689`]], Rule[List[List[7, 279.938`], List[208.125`, 39.4375`]], List[-36.4396`, -2306.32`, 0.913881`, 14.7477`]], Rule[List[List[20.625`, 157.125`], List[46, 5.0625`]], List[-2.57534`, -5.65634`, 0.119891`, 0.0332217`]], Rule[List[List[22.125`, 30.4375`], List[40.5625`, 10.4375`]], List[-6.32337`, -45.1734`, 0.240604`, 0.240604`]]]]], Cell[TextData[List["\t\t\t\t\t\t", StyleBox["Diagram 5:", Rule[FontWeight, "Bold"]], " Vertical ", StyleBox["velocity", Rule[FontFamily, "Courier"]], " relative to the lunar surface"]], "Text"]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[908, 848]], Rule[WindowMargins, List[List[0, Automatic], List[Automatic, 0]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]