List[Null, Null, Null, Null, Null, Null, Null, Notebook[List[Cell[CellGroupData[List[Cell["Van der Pol Model", "Title"], Cell[TextData[List["\nThis example describes a Van der Pol oscillator. Notice that here the keyword ", StyleBox["model", Rule[FontFamily, "Courier"]], " is used instead of ", StyleBox["class", Rule[FontFamily, "Courier"]], " with the same meaning. This example contains declarations of two state variables x and y, both of type Real and a parameter constant lambda, which is a so-called simulation parameter. The keyword parameter specifies that the variable is constant during a simulation run, but can have its value initialized before a run, or between runs. Finally, there is an equation section starting with the keyword equation, containing two mutually dependent equations that define the dynamics of the model.\n"]], "Text"], Cell[TextData[List[StyleBox["model", Rule[FontColor, RGBColor[0, 0, 1]], Rule[Background, GrayLevel[1]]], " VanDerPol\t\t\t\t\t\t\t\t\t\t\"Van der Pol oscillator model\"\n  Real x(start = 1);\t\t\t\t\t\t\t\t\t\t\t\n  Real y(start = 1);\t\t\t\t\t\t\t\t\t\t\t\t\t\t\n  ", StyleBox["parameter Real lambda = 0.3;", Rule[FontColor, RGBColor[0, 0, 1]]], "\nequation\n  ", StyleBox["der(x) = y;", Rule[FontColor, RGBColor[0, 0, 1]]], "\t\t\t\t\t\t\t\t\t\t \n  ", StyleBox["der (y) = - x + lambda*(1 - x*x)*y;", Rule[FontColor, RGBColor[0, 0, 1]]], "\t\nend VanDerPol;"]], "ModelicaInput"], Cell[CellGroupData[List[Cell["Simulation of Van der Pol ", "Section"], Cell["To illustrate the behavior of the model, we give a command to simulate the Van der Pol oscillator during 25 seconds starting at time 0.", "Text"], Cell["simulate(VanDerPol, startTime=0, stopTime=25 );", "ModelicaInput"], Cell[BoxData["\"<SimulationData: \\!\\(VanDerPol\\) : 2004-1-29 17:34:45.0683315 : \\!\\({0.`, 25.`}\\) : 502 data points : 1 events : 5 variables>\\n\\!\\({\\*TagBox[\\\"\[Lambda]\\\", HoldForm], \\*TagBox[\\\"x\\\", HoldForm], \\*TagBox[\\\"y\\\", HoldForm], \\*TagBox[SuperscriptBox[\\\"x\\\", \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm], \\*TagBox[SuperscriptBox[\\\"y\\\", \\\"\[Prime]\\\", Rule[MultilineFunction, None]], HoldForm]}\\)\""], "Output"], Cell["plotParametric( x, y );", "ModelicaInput"], Cell[GraphicsData["PostScript", "%!\n%%Creator: Mathematica\n%%AspectRatio: .61803 \nMathPictureStart\n/Mabs {\nMgmatrix idtransform\nMtmatrix dtransform\n} bind def\n/Mabsadd { Mabs\n3 -1 roll add\n3 1 roll add\nexch } bind def\n%% Graphics\n%%IncludeResource: font Courier\n%%IncludeFont: Courier\n/Courier findfont 10  scalefont  setfont\n% Scaling calculations\n0.500252 0.238346 0.308881 0.140685 [\n[.02356 .29638 -6 -9 ]\n[.02356 .29638 6 0 ]\n[.26191 .29638 -6 -9 ]\n[.26191 .29638 6 0 ]\n[.7386 .29638 -3 -9 ]\n[.7386 .29638 3 0 ]\n[.97694 .29638 -3 -9 ]\n[.97694 .29638 3 0 ]\n[.48775 .02751 -12 -4.5 ]\n[.48775 .02751 0 4.5 ]\n[.48775 .1682 -12 -4.5 ]\n[.48775 .1682 0 4.5 ]\n[.48775 .44957 -6 -4.5 ]\n[.48775 .44957 0 4.5 ]\n[.48775 .59025 -6 -4.5 ]\n[.48775 .59025 0 4.5 ]\n[ 0 0 0 0 ]\n[ 1 .61803 0 0 ]\n] MathScale\n% Start of Graphics\n1 setlinecap\n1 setlinejoin\nnewpath\n0 g\n.25 Mabswid\n[ ] 0 setdash\n.02356 .30888 m\n.02356 .31513 L\ns\n[(-2)] .02356 .29638 0 1 Mshowa\n.26191 .30888 m\n.26191 .31513 L\ns\n[(-1)] .26191 .29638 0 1 Mshowa\n.7386 .30888 m\n.7386 .31513 L\ns\n[(1)] .7386 .29638 0 1 Mshowa\n.97694 .30888 m\n.97694 .31513 L\ns\n[(2)] .97694 .29638 0 1 Mshowa\n.125 Mabswid\n.07123 .30888 m\n.07123 .31263 L\ns\n.1189 .30888 m\n.1189 .31263 L\ns\n.16657 .30888 m\n.16657 .31263 L\ns\n.21424 .30888 m\n.21424 .31263 L\ns\n.30958 .30888 m\n.30958 .31263 L\ns\n.35724 .30888 m\n.35724 .31263 L\ns\n.40491 .30888 m\n.40491 .31263 L\ns\n.45258 .30888 m\n.45258 .31263 L\ns\n.54792 .30888 m\n.54792 .31263 L\ns\n.59559 .30888 m\n.59559 .31263 L\ns\n.64326 .30888 m\n.64326 .31263 L\ns\n.69093 .30888 m\n.69093 .31263 L\ns\n.78627 .30888 m\n.78627 .31263 L\ns\n.83394 .30888 m\n.83394 .31263 L\ns\n.88161 .30888 m\n.88161 .31263 L\ns\n.92927 .30888 m\n.92927 .31263 L\ns\n.25 Mabswid\n0 .30888 m\n1 .30888 L\ns\n.50025 .02751 m\n.5065 .02751 L\ns\n[(-2)] .48775 .02751 1 0 Mshowa\n.50025 .1682 m\n.5065 .1682 L\ns\n[(-1)] .48775 .1682 1 0 Mshowa\n.50025 .44957 m\n.5065 .44957 L\ns\n[(1)] .48775 .44957 1 0 Mshowa\n.50025 .59025 m\n.5065 .59025 L\ns\n[(2)] .48775 .59025 1 0 Mshowa\n.125 Mabswid\n.50025 .05565 m\n.504 .05565 L\ns\n.50025 .08379 m\n.504 .08379 L\ns\n.50025 .11192 m\n.504 .11192 L\ns\n.50025 .14006 m\n.504 .14006 L\ns\n.50025 .19633 m\n.504 .19633 L\ns\n.50025 .22447 m\n.504 .22447 L\ns\n.50025 .25261 m\n.504 .25261 L\ns\n.50025 .28074 m\n.504 .28074 L\ns\n.50025 .33702 m\n.504 .33702 L\ns\n.50025 .36515 m\n.504 .36515 L\ns\n.50025 .39329 m\n.504 .39329 L\ns\n.50025 .42143 m\n.504 .42143 L\ns\n.50025 .4777 m\n.504 .4777 L\ns\n.50025 .50584 m\n.504 .50584 L\ns\n.50025 .53398 m\n.504 .53398 L\ns\n.50025 .56211 m\n.504 .56211 L\ns\n.25 Mabswid\n.50025 0 m\n.50025 .61803 L\ns\n0 0 1 r\n.001 w\n.7386 .44957 m\n.75021 .44226 L\n.76117 .43444 L\n.77145 .42615 L\n.781 .41744 L\n.78981 .40836 L\n.79783 .39898 L\n.80506 .38933 L\n.81146 .37949 L\n.81702 .36949 L\n.82173 .35941 L\n.82559 .34929 L\n.82859 .33917 L\n.83074 .32909 L\n.83203 .3191 L\n.83248 .30922 L\n.83209 .29948 L\n.83089 .28989 L\n.82889 .28048 L\n.82609 .27126 L\n.82252 .26223 L\n.81819 .25339 L\n.81312 .24475 L\n.80733 .23631 L\n.80083 .22806 L\n.79365 .22 L\n.78578 .21211 L\n.77726 .2044 L\n.76808 .19686 L\n.75828 .18948 L\n.74786 .18225 L\n.73683 .17517 L\n.72521 .16824 L\n.71301 .16144 L\n.70024 .15479 L\n.68691 .14828 L\n.67304 .14192 L\n.65863 .13572 L\n.64371 .12968 L\n.62828 .12383 L\n.61237 .11819 L\n.59598 .11277 L\n.57915 .10762 L\n.5619 .10276 L\n.54425 .09823 L\n.52623 .09407 L\n.50787 .09034 L\n.48922 .08709 L\n.47032 .08437 L\n.45121 .08224 L\nMistroke\n.43194 .08077 L\n.41258 .08001 L\n.39318 .08002 L\n.37382 .08086 L\n.35457 .08257 L\n.33551 .08521 L\n.3167 .08879 L\n.29824 .09336 L\n.28021 .0989 L\n.26269 .10541 L\n.24576 .11288 L\n.22951 .12125 L\n.214 .13048 L\n.1993 .14051 L\n.18549 .15124 L\n.17261 .16259 L\n.16072 .17447 L\n.14985 .18677 L\n.14004 .19938 L\n.13131 .21221 L\n.12367 .22517 L\n.11713 .23816 L\n.11169 .2511 L\n.10734 .26393 L\n.10407 .27658 L\n.10186 .289 L\n.10069 .30115 L\n.10054 .313 L\n.10138 .32452 L\n.10318 .3357 L\n.10592 .34653 L\n.10955 .35701 L\n.11406 .36716 L\n.11942 .37697 L\n.12559 .38647 L\n.13255 .39567 L\n.14028 .40459 L\n.14876 .41325 L\n.15796 .42167 L\n.16786 .42986 L\n.17845 .43786 L\n.18971 .44568 L\n.20162 .45334 L\n.21418 .46084 L\n.22736 .46822 L\n.24117 .47547 L\n.25558 .4826 L\n.2706 .48963 L\n.2862 .49655 L\n.30239 .50335 L\nMistroke\n.31914 .51004 L\n.33646 .51659 L\n.35432 .52298 L\n.37272 .5292 L\n.39164 .53521 L\n.41105 .54098 L\n.43094 .54645 L\n.45128 .55157 L\n.47204 .5563 L\n.49317 .56054 L\n.51465 .56424 L\n.53641 .56731 L\n.55841 .56968 L\n.58057 .57125 L\n.60283 .57192 L\n.62511 .57162 L\n.64732 .57027 L\n.66937 .56778 L\n.69116 .56412 L\n.71258 .55924 L\n.73354 .55312 L\n.75393 .54576 L\n.77363 .53719 L\n.79257 .52745 L\n.81063 .51663 L\n.82773 .50481 L\n.84379 .4921 L\n.85875 .47864 L\n.87254 .46455 L\n.88511 .44998 L\n.89644 .43508 L\n.90649 .41998 L\n.91526 .40481 L\n.92274 .38969 L\n.92895 .37473 L\n.93391 .36001 L\n.93763 .34562 L\n.94014 .33161 L\n.94149 .31804 L\n.94171 .30492 L\n.94083 .29228 L\n.93891 .28012 L\n.93597 .26845 L\n.93207 .25724 L\n.92724 .24649 L\n.92151 .23616 L\n.91493 .22624 L\n.90752 .2167 L\n.89932 .2075 L\n.89036 .19862 L\nMistroke\n.88065 .19002 L\n.87023 .18167 L\n.85911 .17355 L\n.84731 .16562 L\n.83484 .15787 L\n.82173 .15026 L\n.80798 .14279 L\n.79359 .13542 L\n.77859 .12815 L\n.76298 .12096 L\n.74676 .11385 L\n.72994 .10682 L\n.71253 .09986 L\n.69453 .093 L\n.67596 .08623 L\n.65682 .07959 L\n.63713 .0731 L\n.61689 .06679 L\n.59613 .06071 L\n.57486 .05491 L\n.55312 .04946 L\n.53094 .04442 L\n.50834 .03987 L\n.48538 .0359 L\n.46212 .03261 L\n.4386 .03008 L\n.41491 .02843 L\n.39112 .02778 L\n.36731 .0282 L\n.34359 .02982 L\n.32006 .03271 L\n.29683 .03692 L\n.27401 .04251 L\n.25172 .04949 L\n.23009 .05785 L\n.20922 .06757 L\n.18924 .07858 L\n.17024 .09077 L\n.15232 .10404 L\n.13557 .11824 L\n.12005 .13321 L\n.10583 .14879 L\n.09295 .1648 L\n.08143 .18108 L\n.0713 .19748 L\n.06256 .21383 L\n.0552 .23003 L\n.0492 .24594 L\n.04453 .26149 L\n.04116 .27659 L\nMistroke\n.03904 .29119 L\n.03814 .30525 L\n.03841 .31876 L\n.0398 .33169 L\n.04226 .34406 L\n.04574 .35589 L\n.05021 .36718 L\n.05561 .37797 L\n.0619 .3883 L\n.06905 .39819 L\n.07702 .40767 L\n.08577 .4168 L\n.09529 .4256 L\n.10554 .43411 L\n.1165 .44236 L\n.12815 .4504 L\n.14047 .45824 L\n.15344 .46591 L\n.16707 .47345 L\n.18132 .48088 L\n.1962 .4882 L\n.2117 .49545 L\n.22781 .50263 L\n.24452 .50975 L\n.26184 .51681 L\n.27975 .52382 L\n.29825 .53075 L\n.31733 .5376 L\n.33699 .54435 L\n.35721 .55097 L\n.37799 .55742 L\n.3993 .56366 L\n.42113 .56962 L\n.44346 .57526 L\n.46624 .58048 L\n.48945 .58522 L\n.51303 .58937 L\n.53694 .59284 L\n.56111 .59552 L\n.58548 .59729 L\n.60995 .59803 L\n.63443 .59765 L\n.65883 .59601 L\n.68304 .59306 L\n.70694 .5887 L\n.73041 .58289 L\n.75332 .57563 L\n.77556 .56692 L\n.797 .55679 L\n.81753 .54532 L\nMistroke\n.83703 .53261 L\n.8554 .51879 L\n.87256 .504 L\n.88843 .48842 L\n.90296 .47222 L\n.91609 .45559 L\n.9278 .43872 L\n.93807 .42176 L\n.94692 .40489 L\n.95434 .38822 L\n.96037 .37187 L\n.96502 .35595 L\n.96835 .34052 L\n.9704 .32563 L\n.97121 .31133 L\n.97083 .29763 L\n.96932 .28453 L\n.96672 .27203 L\n.96309 .26012 L\n.95848 .24876 L\n.95292 .23792 L\n.94647 .22758 L\n.93916 .21768 L\n.93103 .2082 L\n.92211 .19909 L\n.91244 .19031 L\n.90203 .18182 L\n.89092 .1736 L\n.87912 .1656 L\n.86665 .15779 L\n.85353 .15014 L\n.83976 .14263 L\n.82537 .13523 L\n.81035 .12791 L\n.79471 .12068 L\n.77846 .1135 L\n.76161 .10638 L\n.74416 .0993 L\n.72611 .09228 L\n.70747 .08531 L\n.68824 .07842 L\n.66843 .07162 L\n.64805 .06493 L\n.62711 .05841 L\n.60562 .05208 L\n.58361 .04601 L\n.5611 .04026 L\n.53812 .03491 L\n.5147 .03004 L\n.49089 .02574 L\nMistroke\n.46675 .02212 L\n.44234 .01929 L\n.41772 .01736 L\n.39297 .01647 L\n.3682 .01671 L\n.34351 .01821 L\n.31899 .02104 L\n.29477 .02529 L\n.27098 .03101 L\n.24773 .03822 L\n.22516 .04692 L\n.20339 .05708 L\n.18253 .06861 L\n.16271 .08141 L\n.14403 .09537 L\n.12657 .11031 L\n.11041 .12608 L\n.09562 .14248 L\n.08224 .15932 L\n.0703 .17642 L\n.05981 .1936 L\n.05077 .21071 L\n.04318 .22761 L\n.037 .24417 L\n.0322 .2603 L\n.02876 .27592 L\n.02661 .29098 L\n.02571 .30545 L\n.02601 .31929 L\n.02745 .33252 L\n.02999 .34513 L\n.03358 .35714 L\n.03815 .36858 L\n.04367 .37948 L\n.0501 .38989 L\n.05738 .39983 L\n.06549 .40935 L\n.07439 .41849 L\n.08405 .42729 L\n.09444 .43579 L\n.10555 .44403 L\n.11733 .45204 L\n.12979 .45985 L\n.14291 .4675 L\n.15666 .47502 L\n.17105 .48242 L\n.18606 .48973 L\n.20169 .49697 L\n.21792 .50414 L\n.23477 .51127 L\nMistroke\n.25221 .51835 L\n.27025 .52539 L\n.28888 .53237 L\n.30811 .53928 L\n.32791 .54611 L\n.34829 .55283 L\n.36923 .5594 L\n.39072 .56578 L\n.41274 .57192 L\n.43526 .57775 L\n.45827 .58319 L\n.48172 .58817 L\n.50556 .59258 L\n.52976 .59634 L\n.55424 .59933 L\n.57894 .60141 L\n.60377 .60249 L\n.62865 .60244 L\n.65347 .60114 L\n.67813 .59852 L\n.70251 .59448 L\n.72649 .58897 L\n.74994 .58196 L\n.77272 .57344 L\n.79472 .56344 L\n.81581 .55204 L\n.83588 .53933 L\n.85482 .52545 L\n.87253 .51055 L\n.88896 .49482 L\n.90401 .47841 L\n.91766 .46154 L\n.92986 .44438 L\n.94061 .4271 L\n.94989 .40988 L\n.95772 .39286 L\n.96412 .37616 L\n.96913 .35989 L\n.97278 .34412 L\n.97512 .32893 L\n.97619 .31433 L\n.97606 .30036 L\n.97477 .28702 L\n.97237 .27429 L\n.96892 .26218 L\n.96448 .25064 L\n.95907 .23964 L\n.95276 .22916 L\n.94558 .21915 L\n.93757 .20956 L\nMistroke\n.92877 .20037 L\n.9192 .19152 L\n.90889 .18298 L\n.89788 .17471 L\n.88617 .16668 L\n.87379 .15884 L\n.86076 .15117 L\n.84708 .14365 L\n.83277 .13623 L\n.81783 .12891 L\n.80228 .12167 L\n.78612 .11448 L\n.76935 .10735 L\n.75198 .10026 L\n.73401 .09322 L\n.71545 .08622 L\n.69629 .07929 L\n.67656 .07244 L\n.65625 .06569 L\n.63537 .05909 L\n.61394 .05266 L\n.59198 .04647 L\n.5695 .04057 L\n.54654 .03504 L\n.52313 .02996 L\n.49931 .02542 L\n.47513 .02153 L\n.45065 .0184 L\n.42594 .01613 L\n.40108 .01486 L\n.37616 .01472 L\n.35127 .01577 L\n.32653 .01816 L\n.30206 .02196 L\n.27796 .02722 L\n.25437 .03399 L\n.23142 .04227 L\n.20923 .05203 L\n.18794 .06323 L\n.16765 .07575 L\n.14848 .08947 L\n.13052 .10425 L\n.11384 .11989 L\n.09853 .13623 L\n.08461 .15307 L\n.07214 .17023 L\n.06113 .18754 L\n.05159 .20481 L\n.0435 .22191 L\n.03685 .2387 L\nMistroke\n.0316 .25508 L\n.02772 .27095 L\n.02516 .28627 L\n.02388 .30099 L\n.02381 .31509 L\n.02491 .32855 L\n.02712 .34138 L\n.0304 .35361 L\n.03468 .36525 L\n.03993 .37634 L\n.04609 .38691 L\n.05313 .39701 L\n.06101 .40666 L\n.06969 .41592 L\n.07913 .42482 L\n.08932 .43341 L\n.10022 .44172 L\n.11182 .4498 L\n.12409 .45766 L\n.13702 .46536 L\n.1506 .4729 L\n.1648 .48033 L\n.17964 .48767 L\n.19509 .49492 L\n.21116 .50212 L\n.22783 .50926 L\n.2451 .51635 L\n.26297 .52341 L\n.28144 .53041 L\n.3005 .53735 L\n.32014 .54422 L\n.34036 .551 L\n.36115 .55763 L\n.38249 .5641 L\n.40437 .57034 L\n.42676 .5763 L\n.44965 .58191 L\n.47299 .58708 L\n.49676 .59172 L\n.52089 .59575 L\n.54533 .59904 L\n.57002 .60147 L\n.59487 .60294 L\n.61981 .60332 L\n.64472 .60247 L\n.66951 .60033 L\n.69406 .59679 L\n.71825 .59178 L\n.74195 .58527 L\n.76504 .57725 L\nMistroke\n.78738 .56775 L\nMfstroke\n0 0 m\n1 0 L\n1 .61803 L\n0 .61803 L\nclosepath\nclip\nnewpath\n% End of Graphics\nMathPictureEnd\n"], "Graphics", Rule[ImageSize, List[288, 177.938`]], Rule[ImageMargins, List[List[35, 0], List[0, 0]]], Rule[ImageRegion, List[List[0, 1], List[0, 1]]], Rule[ImageCache, GraphicsData["Bitmap", "CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oS`3IfMT0\n0`000000fMWI0=WIf@2>0=WIf@00S`3IfMT00`000000fMWI0=WIf@2>0=WIf@00S`3IfMT00`000000\nfMWI0=WIf@2>0=WIf@00S`3IfMT00`000000fMWI0=WIf@2>0=WIf@00H@3IfMTM0000oa40fMWI00<0\n00000=WIf@3IfMT0SP3IfMT005X0fMWI30000?l>0=WIfA40003o00<0fMWI0000000000000P000005\n0=WIf@030000003IfMT0fMWI08h0fMWI001C0=WIf@`0003o8`3IfMT:0000o`<0fMWI00<000000=WI\nf@3IfMT0SP3IfMT004h0fMWI2P000?l_0=WIf@030000003IfMT0003o00T0003oS@3IfMT004/0fMWI\n1P000?l>0=WIfB80003o0`0000040=WIf@030000003IfMT0fMWI00@0fMWI00<000000000o`000?l0\n1P000?n80=WIf@00B03IfMT50000o``0fMWI1P000?lR0=WIf@L0003o00<0fMWI0000003IfMT0103I\nfMT00`000000fMWI0=WIf@030=WIf@H0003oQ@3IfMT004H0fMWI0P000?l00`3IfMT0003o0000o`09\n0=WIf@D0003o;P3IfMT00`000000003o0000o`040000o`030=WIf@000000fMWI00P0fMWI1`000?n0\n0=WIf@00A03IfMT50000o`L0fMWI10000?lj0=WIf@L0003o203IfMT80000og/0fMWI00110=WIf@D0\n003o1`3IfMT30000ocl0fMWI00<000000=WIf@3IfMT00`3IfMT50000o`P0fMWI1P000?mh0=WIf@00\n?P3IfMT40000o`P0fMWI0`000?m20=WIf@030000003IfMT0fMWI00P0fMWI0`000?l90=WIf@D0003o\nM@3IfMT003`0fMWI10000?l70=WIf@<0003oA@3IfMT00`000000fMWI0=WIf@0;0=WIf@@0003o2@3I\nfMT50000og40fMWI000j0=WIf@@0003o1P3IfMT30000odP0fMWI0P00000@0=WIf@D0003o2@3IfMT4\n0000ofd0fMWI000i0=WIf@<0003o1P3IfMT20000od/0fMWI00<000000=WIf@3IfMT0503IfMT40000\no`P0fMWI1P000?mX0=WIf@00=`3IfMT30000o`H0fMWI0P000?m=0=WIf@030000003IfMT0fMWI01P0\nfMWI0`000?l80=WIf@H0003oI@3IfMT003H0fMWI0P000?l60=WIf@80003oC`3IfMT00`000000fMWI\n0=WIf@0K0=WIf@80003o2@3IfMT60000of80fMWI000d0=WIf@80003o1P3IfMT20000oe40fMWI00<0\n00000=WIf@3IfMT07@3IfMT40000o`P0fMWI1P000?mO0=WIf@00<P3IfMT20000o`H0fMWI0P000?mC\n0=WIf@030000003IfMT0fMWI0240fMWI1@000?l60=WIf@H0003oG03IfMT00300fMWI0P000?l60=WI\nf@80003oE@3IfMT00`000000fMWI0=WIf@0V0=WIf@<0003o1P3IfMT50000oeX0fMWI000^0=WIf@<0\n003o1@3IfMT20000ocH0fMWI2`000?lF0=WIf@030000003IfMT0fMWI02T0fMWI0`000?l60=WIf@D0\n003oE`3IfMT002d0fMWI0P000?l60=WIf@030000o`3IfMT0fMWI02/0fMWI2`000?l;0=WIf@/0003o\n2`3IfMT00`000000fMWI0=WIf@0/0=WIf@80003o1`3IfMT50000oe@0fMWI000[0=WIf@<0003o1@3I\nfMT20000obT0fMWI1@000?lQ0=WIf@/0003o0P00000_0=WIf@<0003o1P3IfMT60000oe40fMWI000Z\n0=WIf@<0003o1@3IfMT00`000?l0fMWI0=WIf@0T0=WIf@D0003o<@3IfMT50000obl0fMWI0`000?l6\n0=WIf@D0003oC`3IfMT002T0fMWI0P000?l60=WIf@030000o`3IfMT0fMWI0200fMWI1@000?lf0=WI\nf@030000003IfMT0fMWI0080fMWI1@000?l]0=WIf@<0003o1P3IfMT50000od`0fMWI000W0=WIf@<0\n003o1@3IfMT20000ob40fMWI0P000?lk0=WIf@030000003IfMT0fMWI00L0fMWI0P000?l^0=WIf@80\n003o1`3IfMT40000odX0fMWI000V0=WIf@030000o`3IfMT0003o00@0fMWI0P000?lP0=WIf@<0003o\n?@3IfMT00`000000fMWI0=WIf@090=WIf@@0003o;03IfMT30000o`H0fMWI1@000?m70=WIf@009@3I\nfMT30000o`<0fMWI0P000?lP0=WIf@80003o@03IfMT00`000000fMWI0=WIf@0=0=WIf@D0003o:P3I\nfMT20000o`L0fMWI10000?m50=WIf@00903IfMT20000o`@0fMWI00<0003o0=WIf@3IfMT07@3IfMT3\n0000od80fMWI00<000000=WIf@3IfMT04P3IfMT30000obT0fMWI0`000?l60=WIf@D0003o@P3IfMT0\n02<0fMWI0P000?l40=WIf@030000o`3IfMT0fMWI01`0fMWI0P000?m50=WIf@030000003IfMT0fMWI\n01D0fMWI0P000?lZ0=WIf@80003o1`3IfMT40000od00fMWI000R0=WIf@80003o103IfMT00`000?l0\nfMWI0=WIf@0K0=WIf@80003oA`3IfMT2000001P0fMWI10000?lX0=WIf@80003o1`3IfMT50000ocd0\nfMWI000Q0=WIf@80003o103IfMT00`000?l0fMWI0=WIf@0J0=WIf@80003oB@3IfMT00`000000fMWI\n0=WIf@0K0=WIf@@0003o9P3IfMT20000o`L0fMWI0P000?l00`3IfMT0003o0000o`0k0=WIf@00803I\nfMT20000o`@0fMWI00<0003o0=WIf@3IfMT06@3IfMT20000od/0fMWI00<000000=WIf@3IfMT07`3I\nfMT20000obH0fMWI0`000?l60=WIf@D0003o>@3IfMT001l0fMWI0P000?l40=WIf@030000o`3IfMT0\nfMWI01P0fMWI0P000?m=0=WIf@030000003IfMT0fMWI0240fMWI0P000?lW0=WIf@80003o1`3IfMT4\n0000ocL0fMWI000N0=WIf@80003o103IfMT00`000?l0fMWI0=WIf@0G0=WIf@80003oC`3IfMT00`00\n0000fMWI0=WIf@0S0=WIf@80003o9`3IfMT20000o`L0fMWI10000?le0=WIf@007P3IfMT00`000?l0\nfMWI0=WIf@020=WIf@030000o`3IfMT0fMWI01L0fMWI00<0003o0=WIf@3IfMT0C`3IfMT00`000000\nfMWI0=WIf@0U0=WIf@<0003o9P3IfMT20000o`L0fMWI0`000?ld0=WIf@007@3IfMT00`000?l0fMWI\n0=WIf@020=WIf@030000o`3IfMT0fMWI01L0fMWI00<0003o0=WIf@3IfMT0D03IfMT00`000000fMWI\n0=WIf@0X0=WIf@@0003o903IfMT20000o`L0fMWI0`000?lb0=WIf@00703IfMT00`000?l0fMWI0=WI\nf@020=WIf@030000o`3IfMT0fMWI01H0fMWI0P000?mC0=WIf@030000003IfMT0fMWI02`0fMWI0P00\n0?lT0=WIf@80003o1`3IfMT20000oc40fMWI000K0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT05P3IfMT00`000?l0fMWI0=WIf@1C0=WIf@800000;`3IfMT20000ob@0fMWI0P000?l7\n0=WIf@80003o;`3IfMT001X0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0F0=WI\nf@030000o`3IfMT0fMWI05@0fMWI00<000000=WIf@3IfMT0<03IfMT20000ob@0fMWI00<0003o0=WI\nf@3IfMT01P3IfMT20000obd0fMWI000I0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o0=WIf@3I\nfMT05@3IfMT20000oeL0fMWI00<000000=WIf@3IfMT0<P3IfMT20000ob<0fMWI0P000?l70=WIf@<0\n003o:`3IfMT001P0fMWI0P000?l30=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0\nE`3IfMT00`000000fMWI0=WIf@0d0=WIf@80003o8`3IfMT00`000?l0fMWI0=WIf@060=WIf@<0003o\n:@3IfMT001L0fMWI0P000?l30=WIf@030000o`3IfMT0fMWI01D0fMWI00<0003o0=WIf@3IfMT0F03I\nfMT00`000000fMWI0=WIf@0f0=WIf@030000o`3IfMT0fMWI0240fMWI0P000?l70=WIf@@0003o9`3I\nfMT001L0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0C0=WIf@80003oF`3IfMT0\n0`000000fMWI0=WIf@0g0=WIf@80003o8`3IfMT20000o`L0fMWI10000?lU0=WIf@005P3IfMT00`00\n0?l0fMWI0=WIf@020=WIf@030000o`3IfMT0fMWI01<0fMWI00<0003o0=WIf@3IfMT0DP3IfMT50000\n00@0fMWI00<000000=WIf@3IfMT0>@3IfMT20000ob<0fMWI0P000?l70=WIf@@0003o8`3IfMT001H0\nfMWI00D0003o0=WIf@3IfMT0fMWI0000o`0E0=WIf@030000o`3IfMT0fMWI05D0fMWI00<000000=WI\nf@3IfMT0103IfMT00`000000fMWI0=WIf@0k0=WIf@80003o8`3IfMT00`000?l0fMWI0=WIf@060=WI\nf@@0003o8@3IfMT001D0fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0E0=WIf@030000o`3IfMT0fMWI\n05H0fMWI00<000000=WIf@3IfMT0103IfMT3000003d0fMWI00<0003o0=WIf@3IfMT08@3IfMT20000\no`P0fMWI0`000?lP0=WIf@00503IfMT20000o`80fMWI00<0003o0=WIf@3IfMT0503IfMT00`000?l0\nfMWI0=WIf@1>0=WIf@@00000103IfMT00`000000fMWI0=WIf@040=WIf@030000003IfMT0fMWI03h0\nfMWI0P000?lS0=WIf@80003o203IfMT20000oal0fMWI000C0=WIf@80003o0`3IfMT00`000?l0fMWI\n0=WIf@0C0=WIf@030000o`3IfMT0fMWI05L0fMWI00<000000=WIf@3IfMT0103IfMT00`000000fMWI\n0=WIf@100=WIf@80003o8`3IfMT00`000?l0fMWI0=WIf@060=WIf@80003o7P3IfMT001<0fMWI00D0\n003o0=WIf@3IfMT0fMWI0000o`0E0=WIf@030000o`3IfMT0fMWI05L0fMWI0P0000060=WIf@030000\n003IfMT0fMWI0480fMWI0P000?lR0=WIf@030000o`3IfMT0fMWI00H0fMWI0`000?lL0=WIf@004P3I\nfMT20000o`80fMWI00<0003o0=WIf@3IfMT04`3IfMT00`000?l0fMWI0=WIf@1P0=WIf@030000003I\nfMT0fMWI04@0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@060=WIf@030000o`3I\nfMT0003o01/0fMWI000A0=WIf@80003o0`3IfMT00`000?l0fMWI0=WIf@0B0=WIf@030000o`3IfMT0\nfMWI0640fMWI00<000000=WIf@3IfMT0A@3IfMT20000ob40fMWI0P000?l70=WIf@@0003o6@3IfMT0\n0140fMWI0P000?l20=WIf@030000o`3IfMT0fMWI0180fMWI00<0003o0=WIf@3IfMT0HP3IfMT00`00\n0000fMWI0=WIf@170=WIf@030000o`3IfMT0fMWI0200fMWI00<0003o0=WIf@3IfMT01P3IfMT00`00\n0?l0fMWI0000o`0H0=WIf@00403IfMT20000o`<0fMWI00<0003o0=WIf@3IfMT04@3IfMT00`000?l0\nfMWI0=WIf@1S0=WIf@030000003IfMT0fMWI04P0fMWI0P000?lQ0=WIf@80003o1`3IfMT00`000?l0\nfMWI0000o`0G0=WIf@00403IfMT01@000?l0fMWI0=WIf@3IfMT0003o01@0fMWI00<0003o0=WIf@3I\nfMT0H`3IfMT2000004/0fMWI00<0003o0=WIf@3IfMT0803IfMT00`000?l0fMWI0=WIf@050=WIf@03\n0000o`3IfMT0003o01H0fMWI000?0=WIf@80003o0`3IfMT00`000?l0fMWI0=WIf@0A0=WIf@030000\no`3IfMT0fMWI06@0fMWI00<000000=WIf@3IfMT0B`3IfMT00`000?l0fMWI0=WIf@0P0=WIf@030000\no`3IfMT0fMWI00D0fMWI00<0003o0=WIf@000?l05@3IfMT000l0fMWI00D0003o0=WIf@3IfMT0fMWI\n0000o`0C0=WIf@030000o`3IfMT0fMWI06D0fMWI00<000000=WIf@3IfMT0C03IfMT00`000?l0fMWI\n0=WIf@0O0=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o0=WIf@000?l0503IfMT000h0fMWI0P00\n0?l30=WIf@030000o`3IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0IP3IfMT00`000000fMWI0=WI\nf@1=0=WIf@80003o803IfMT00`000?l0fMWI0=WIf@060=WIf@80003o503IfMT000h0fMWI00D0003o\n0=WIf@3IfMT0fMWI0000o`0B0=WIf@030000o`3IfMT0fMWI06L0fMWI00<000000=WIf@3IfMT0C`3I\nfMT00`000?l0fMWI0=WIf@0N0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?lC0=WIf@003@3IfMT2\n0000o`<0fMWI00<0003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@1X0=WIf@030000003IfMT0\nfMWI0500fMWI00<0003o0=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@060=WIf@<0003o4@3IfMT0\n00d0fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0B0=WIf@030000o`3IfMT0fMWI06P0fMWI00<00000\n0=WIf@3IfMT0D03IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o\n0=WIf@000?l0403IfMT000`0fMWI0P000?l30=WIf@030000o`3IfMT0fMWI00l0fMWI00<0003o0=WI\nf@3IfMT0J@3IfMT00`000000fMWI0=WIf@1A0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WI\nf@3IfMT01P3IfMT00`000?l0fMWI0000o`0?0=WIf@00303IfMT01@000?l0fMWI0=WIf@3IfMT0003o\n0180fMWI00<0003o0=WIf@3IfMT0J@3IfMT2000005<0fMWI00<0003o0=WIf@3IfMT07`3IfMT00`00\n0?l0fMWI0=WIf@050=WIf@030000o`3IfMT0003o00l0fMWI000;0=WIf@80003o0`3IfMT00`000?l0\nfMWI0=WIf@0?0=WIf@030000o`3IfMT0fMWI06X0fMWI00<000000=WIf@3IfMT0D`3IfMT00`000?l0\nfMWI0=WIf@0O0=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@000?l03P3IfMT000/0fMWI\n00<0003o0=WIf@3IfMT00P3IfMT00`000?l0fMWI0=WIf@0?0=WIf@030000o`3IfMT0fMWI06X0fMWI\n00<000000=WIf@3IfMT0E03IfMT00`000?l0fMWI0=WIf@0N0=WIf@030000o`3IfMT0fMWI00H0fMWI\n0P000?l>0=WIf@002P3IfMT20000o`<0fMWI00<0003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WI\nf@1[0=WIf@030000003IfMT0fMWI05D0fMWI00<0003o0=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WI\nf@050=WIf@030000o`3IfMT0003o00d0fMWI000:0=WIf@030000o`3IfMT0fMWI0080fMWI00<0003o\n0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@1[0=WIf@030000003IfMT0fMWI05H0fMWI00<0003o\n0=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@060=WIf@80003o3@3IfMT000X0fMWI00D0003o0=WI\nf@3IfMT0fMWI0000o`0B0=WIf@030000o`3IfMT0fMWI06/0fMWI00<000000=WIf@3IfMT0EP3IfMT0\n0`000?l0fMWI0=WIf@0N0=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@000?l0303IfMT0\n00X0fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0A0=WIf@030000o`3IfMT0fMWI06`0fMWI00<00000\n0=WIf@3IfMT0E`3IfMT00`000?l0fMWI0=WIf@0M0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?l<\n0=WIf@002@3IfMT20000o`<0fMWI00<0003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@1/0=WI\nf@030000003IfMT0fMWI05L0fMWI00<0003o0=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@050=WI\nf@030000o`3IfMT0003o00/0fMWI00090=WIf@050000o`3IfMT0fMWI0=WIf@000?l04P3IfMT00`00\n0?l0fMWI0=WIf@1/0=WIf@800000F@3IfMT00`000?l0fMWI0=WIf@0M0=WIf@030000o`3IfMT0fMWI\n00H0fMWI0P000?l;0=WIf@002@3IfMT01@000?l0fMWI0=WIf@3IfMT0003o0140fMWI00<0003o0=WI\nf@3IfMT0K@3IfMT00`000000fMWI0=WIf@1H0=WIf@030000o`3IfMT0fMWI01h0fMWI00<0003o0=WI\nf@3IfMT01@3IfMT20000o`/0fMWI00090=WIf@050000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`00\n0?l0fMWI0=WIf@1]0=WIf@030000003IfMT0fMWI05T0fMWI00<0003o0=WIf@3IfMT07@3IfMT00`00\n0?l0fMWI0=WIf@060=WIf@80003o2P3IfMT000P0fMWI0P000?l30=WIf@030000o`3IfMT0fMWI00l0\nfMWI00<0003o0=WIf@3IfMT0K@3IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0fMWI01h0\nfMWI00<0003o0=WIf@3IfMT01@3IfMT20000o`X0fMWI00080=WIf@80003o0P0000000`3IfMT0003o\n0=WIf@0@0=WIf@030000o`3IfMT0fMWI02X0fMWI1@00000n0=WIf@030000003IfMT0fMWI03l0fMWI\n1@00000E0=WIf@030000o`3IfMT0fMWI01h0fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l00000\n00000002000000H0fMWI00080=WIf@050000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`000?l0fMWI\n0=WIf@0]0=WIf@030000003IfMT0fMWI03h0fMWI00<000000=WIf@3IfMT0@@3IfMT00`000000fMWI\n0=WIf@0F0=WIf@030000o`3IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`T0fMWI\n00080=WIf@050000o`000000fMWI0=WIf@000?l04@3IfMT00`000?l0fMWI0=WIf@0]0=WIf@030000\n003IfMT0fMWI03h0fMWI00<000000=WIf@3IfMT0@@3IfMT00`000000fMWI0=WIf@0F0=WIf@030000\no`3IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`030000003IfMT0fMWI00H0fMWI\n00020=WIf@@000000P3IfMT01@000?l0fMWI0000003IfMT0003o0140fMWI00<0003o0=WIf@3IfMT0\n9@3IfMT4000000@0fMWI00<000000=WIf@3IfMT0?P3IfMT00`000000fMWI0=WIf@110=WIf@030000\n003IfMT0fMWI01L0fMWI00<0003o0=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@050=WIf@80003o\n00<0fMWI0000003IfMT01P3IfMT000L0fMWI0P000?l20=WIf@03000000000?l0fMWI0100fMWI00<0\n003o0=WIf@3IfMT0;@3IfMT00`000000fMWI0=WIf@0n0=WIf@800000@P3IfMT00`000000fMWI0=WI\nf@0G0=WIf@030000o`3IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT01@3IfMT20000o`80fMWI00<0\n00000=WIf@3IfMT0103IfMT000L0fMWI00D0003o0=WIf@00000000000000o`0A0=WIf@030000o`3I\nfMT0fMWI02d0fMWI0P0000100=WIf@030000003IfMT0fMWI0400fMWI0P00000I0=WIf@030000o`3I\nfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT01@3IfMT010000?l0fMWI0000o`0000070=WIf@001`3I\nfMT01@000?l0fMWI0=WIf@3IfMT0003o0140fMWI00<0003o0=WIf@3IfMT0K`3IfMT00`000000fMWI\n0=WIf@1L0=WIf@030000o`3IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT01@3IfMT00`000?l0fMWI\n0000o`080=WIf@001`3IfMT01@000?l0fMWI0=WIf@3IfMT0003o0140fMWI00<0003o0=WIf@3IfMT0\nK`3IfMT00`000000fMWI0=WIf@1L0=WIf@030000o`3IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT0\n1@3IfMT00`000?l0fMWI0000o`080=WIf@001`3IfMT01@000?l0fMWI0=WIf@3IfMT0003o0140fMWI\n00<0003o0=WIf@3IfMT0K`3IfMT00`000000fMWI0=WIf@1L0=WIf@030000o`3IfMT0fMWI01`0fMWI\n00<0003o0=WIf@3IfMT01P3IfMT20000o`P0fMWI00070=WIf@050000o`3IfMT0fMWI0=WIf@000?l0\n4@3IfMT00`000?l0fMWI0=WIf@1_0=WIf@030000003IfMT0fMWI05`0fMWI00<0003o0=WIf@3IfMT0\n703IfMT00`000?l0fMWI0=WIf@060=WIf@80003o203IfMT000L0fMWI00D0003o0=WIf@3IfMT0fMWI\n0000o`0A0=WIf@030000o`3IfMT0fMWI06l0fMWI00<000000=WIf@3IfMT0G03IfMT00`000?l0fMWI\n0=WIf@0L0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?l80=WIf@001`3IfMT01@000?l0fMWI0=WI\nf@3IfMT0003o0140fMWI00<0003o0=WIf@3IfMT0K`3IfMT00`000000fMWI0=WIf@1L0=WIf@030000\no`3IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`P0fMWI0007000000050000o`00\n00000000000000000?l04@0000000`000?l000000000003>000000030000o`000000000001`00000\n00<0003o0000000000001P0000020000o`L000000@3IfMT000L0fMWI00D0003o0=WIf@3IfMT0fMWI\n0000o`080=WIf@030000003IfMT0fMWI00H0fMWI00<0003o0=WIf@3IfMT00P3IfMT00`000000fMWI\n0=WIf@0;0=WIf@030000003IfMT0fMWI00X0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI\n0=WIf@0:0=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI\n0=WIf@0:0=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI\n0=WIf@0:0=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT02`3IfMT00`000000fMWI\n0=WIf@0:0=WIf@030000003IfMT0fMWI00/0fMWI00<000000=WIf@3IfMT02P3IfMT00`000?l00000\n0=WIf@0;0=WIf@030000003IfMT0fMWI00/0fMWI00@000000=WIf@3IfMT0003o203IfMT20000o`03\n0000003IfMT0fMWI00D0fMWI00070=WIf@050000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`000?l0\nfMWI0=WIf@0[0=WIf@030000003IfMT0fMWI0440fMWI00<000000=WIf@3IfMT0@@3IfMT00`000000\nfMWI0=WIf@0H0=WIf@030000o`3IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`03\n0000003IfMT0fMWI00D0fMWI00070=WIf@050000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`000?l0\nfMWI0=WIf@1_0=WIf@030000003IfMT0fMWI05`0fMWI00<0003o0=WIf@3IfMT0703IfMT00`000?l0\nfMWI0=WIf@060=WIf@80003o203IfMT000L0fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0A0=WIf@03\n0000o`3IfMT0fMWI06l0fMWI00<000000=WIf@3IfMT0G03IfMT00`000?l0fMWI0=WIf@0L0=WIf@03\n0000o`3IfMT0fMWI00H0fMWI0P000?l80=WIf@001`3IfMT01@000?l0fMWI0=WIf@3IfMT0003o0140\nfMWI00<0003o0=WIf@3IfMT0K`3IfMT00`000000fMWI0=WIf@1L0=WIf@030000o`3IfMT0fMWI01`0\nfMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`P0fMWI00070=WIf@80003o0P3IfMT00`000?l0fMWI\n0=WIf@0?0=WIf@030000o`3IfMT0fMWI06l0fMWI00<000000=WIf@3IfMT0G03IfMT00`000?l0fMWI\n0=WIf@0L0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?l80=WIf@001`3IfMT20000o`80fMWI00<0\n003o0=WIf@3IfMT03`3IfMT00`000?l0fMWI0=WIf@1_0=WIf@030000003IfMT0fMWI05`0fMWI00<0\n003o0=WIf@3IfMT0703IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3IfMT0003o00P0fMWI0008\n0=WIf@040000o`3IfMT0fMWI0000oa40fMWI00<0003o0=WIf@3IfMT0K`3IfMT2000005`0fMWI00<0\n003o0=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@050=WIf@80003o2@3IfMT000P0fMWI00D0003o\n0=WIf@3IfMT0fMWI0000o`0@0=WIf@030000o`3IfMT0fMWI06l0fMWI00<000000=WIf@3IfMT0F`3I\nfMT00`000?l0fMWI0=WIf@0L0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?l90=WIf@00203IfMT0\n1@000?l0fMWI0=WIf@3IfMT0003o0140fMWI00<0003o0=WIf@3IfMT0KP3IfMT00`000000fMWI0=WI\nf@1K0=WIf@030000o`3IfMT0fMWI01`0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`T0fMWI0008\n0=WIf@050000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`000?l0fMWI0=WIf@1^0=WIf@030000003I\nfMT0fMWI05X0fMWI00<0003o0=WIf@3IfMT07@3IfMT00`000?l0fMWI0=WIf@050=WIf@030000o`3I\nfMT0003o00T0fMWI00080=WIf@80003o0`3IfMT00`000?l0fMWI0=WIf@0>0=WIf@030000o`3IfMT0\nfMWI06h0fMWI00<000000=WIf@3IfMT0FP3IfMT00`000?l0fMWI0=WIf@0M0=WIf@030000o`3IfMT0\nfMWI00D0fMWI00<0003o0=WIf@000?l02@3IfMT000T0fMWI00D0003o0=WIf@3IfMT0fMWI0000o`0A\n0=WIf@030000o`3IfMT0fMWI06d0fMWI00<000000=WIf@3IfMT0F@3IfMT00`000?l0fMWI0=WIf@0M\n0=WIf@030000o`3IfMT0fMWI00H0fMWI0P000?l:0=WIf@002@3IfMT01@000?l0fMWI0=WIf@3IfMT0\n003o0140fMWI00<0003o0=WIf@3IfMT0K@3IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0\nfMWI01d0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`X0fMWI000:0=WIf@040000o`3IfMT0fMWI\n0000oa80fMWI00<0003o0=WIf@3IfMT0K03IfMT00`000000fMWI0=WIf@1I0=WIf@030000o`3IfMT0\nfMWI01d0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`X0fMWI000:0=WIf@050000o`3IfMT0fMWI\n0=WIf@000?l04@3IfMT00`000?l0fMWI0=WIf@1/0=WIf@800000F@3IfMT00`000?l0fMWI0=WIf@0N\n0=WIf@030000o`3IfMT0fMWI00D0fMWI00<0003o0=WIf@000?l02P3IfMT000X0fMWI0P000?l20=WI\nf@030000o`3IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0J`3IfMT00`000000fMWI0=WIf@1H0=WI\nf@030000o`3IfMT0fMWI01d0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`/0fMWI000;0=WIf@05\n0000o`3IfMT0fMWI0=WIf@000?l04@3IfMT00`000?l0fMWI0=WIf@1[0=WIf@030000003IfMT0fMWI\n05L0fMWI00<0003o0=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@060=WIf@80003o2`3IfMT000/0\nfMWI0P000?l20=WIf@030000o`3IfMT0fMWI0100fMWI00<0003o0=WIf@3IfMT0JP3IfMT00`000000\nfMWI0=WIf@1G0=WIf@030000o`3IfMT0fMWI01h0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`/0\nfMWI000<0=WIf@050000o`3IfMT0fMWI0=WIf@000?l04P3IfMT00`000?l0fMWI0=WIf@1Y0=WIf@03\n0000003IfMT0fMWI05H0fMWI00<0003o0=WIf@3IfMT07P3IfMT00`000?l0fMWI0=WIf@060=WIf@03\n0000o`3IfMT0003o00/0fMWI000<0=WIf@80003o0`3IfMT00`000?l0fMWI0=WIf@0?0=WIf@030000\no`3IfMT0fMWI06T0fMWI00<000000=WIf@3IfMT0E@3IfMT00`000?l0fMWI0=WIf@0O0=WIf@030000\no`3IfMT0fMWI00H0fMWI0P000?l<0=WIf@003@3IfMT01@000?l0fMWI0=WIf@3IfMT0003o0180fMWI\n00<0003o0=WIf@3IfMT0J03IfMT00`000000fMWI0=WIf@1E0=WIf@030000o`3IfMT0fMWI01h0fMWI\n00<0003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0000o`0<0=WIf@003P3IfMT01@000?l0fMWI0=WI\nf@3IfMT0003o0180fMWI00<0003o0=WIf@3IfMT0I`3IfMT00`000000fMWI0=WIf@1D0=WIf@030000\no`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT01P3IfMT20000o`d0fMWI000>0=WIf@80003o0`3I\nfMT00`000?l0fMWI0=WIf@0?0=WIf@030000o`3IfMT0fMWI06L0fMWI0P00001D0=WIf@030000o`3I\nfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0000o`0=0=WIf@003`3IfMT0\n1@000?l0fMWI0=WIf@3IfMT0003o0180fMWI00<0003o0=WIf@3IfMT0IP3IfMT00`000000fMWI0=WI\nf@1C0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0000\no`0=0=WIf@00403IfMT01@000?l0fMWI0=WIf@3IfMT0003o0180fMWI00<0003o0=WIf@3IfMT0I@3I\nfMT00`000000fMWI0=WIf@1B0=WIf@030000o`3IfMT0fMWI01l0fMWI00<0003o0=WIf@3IfMT01P3I\nfMT00`000?l0fMWI0000o`0>0=WIf@00403IfMT20000o`<0fMWI00<0003o0=WIf@3IfMT0403IfMT0\n0`000?l0fMWI0=WIf@1T0=WIf@030000003IfMT0fMWI0540fMWI00<0003o0=WIf@3IfMT0803IfMT0\n0`000?l0fMWI0=WIf@060=WIf@030000o`3IfMT0003o00h0fMWI000A0=WIf@80003o0`3IfMT00`00\n0?l0fMWI0=WIf@0@0=WIf@030000o`3IfMT0fMWI06<0fMWI00<000000=WIf@3IfMT0D03IfMT00`00\n0?l0fMWI0=WIf@0P0=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o0=WIf@000?l03`3IfMT00180\nfMWI0P000?l30=WIf@030000o`3IfMT0fMWI00l0fMWI00<0003o0=WIf@3IfMT0H`3IfMT00`000000\nfMWI0=WIf@1?0=WIf@030000o`3IfMT0fMWI0240fMWI00<0003o0=WIf@3IfMT01P3IfMT00`000?l0\nfMWI0000o`0?0=WIf@004`3IfMT20000o`80fMWI00<0003o0=WIf@3IfMT0403IfMT00`000?l0fMWI\n0=WIf@1R0=WIf@030000003IfMT0fMWI04h0fMWI00<0003o0=WIf@3IfMT08@3IfMT00`000?l0fMWI\n0=WIf@060=WIf@030000o`3IfMT0003o0100fMWI000D0=WIf@80003o0P3IfMT00`000?l0fMWI0=WI\nf@0@0=WIf@80003oHP3IfMT00`000000fMWI0=WIf@1=0=WIf@030000o`3IfMT0fMWI0280fMWI00<0\n003o0=WIf@3IfMT01P3IfMT00`000?l0fMWI0000o`0@0=WIf@005@3IfMT20000o`80fMWI0P000?lB\n0=WIf@030000o`3IfMT0fMWI05l0fMWI0P00001=0=WIf@030000o`3IfMT0fMWI0280fMWI00<0003o\n0=WIf@3IfMT01P3IfMT00`000?l0fMWI0000o`0A0=WIf@005`3IfMT01@000?l0fMWI0=WIf@3IfMT0\n003o0180fMWI00<0003o0=WIf@3IfMT0GP3IfMT00`000000fMWI0=WIf@1<0=WIf@030000o`3IfMT0\nfMWI0240fMWI00<0003o0=WIf@3IfMT01P3IfMT010000?l0fMWI0=WIf@000?lA0=WIf@00603IfMT0\n1@000?l0fMWI0=WIf@3IfMT0003o0180fMWI00<0003o0=WIf@3IfMT0G@3IfMT00`000000fMWI0=WI\nf@1;0=WIf@030000o`3IfMT0fMWI0240fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0000\no`0B0=WIf@00603IfMT30000o`80fMWI00<0003o0=WIf@3IfMT0403IfMT00`000?l0fMWI0=WIf@1L\n0=WIf@030000003IfMT0fMWI04T0fMWI0P000?lT0=WIf@030000o`3IfMT0fMWI00H0fMWI00<0003o\n0=WIf@000?l04`3IfMT001T0fMWI00<0003o0=WIf@000?l00P3IfMT00`000?l0fMWI0=WIf@0@0=WI\nf@80003oG03IfMT00`000000fMWI0=WIf@180=WIf@030000o`3IfMT0fMWI02<0fMWI00<0003o0=WI\nf@3IfMT01`3IfMT00`000?l0fMWI0000o`0C0=WIf@006P3IfMT30000o`80fMWI0P000?lB0=WIf@03\n0000o`3IfMT0fMWI05T0fMWI00<000000=WIf@3IfMT0AP3IfMT20000obD0fMWI00<0003o0=WIf@3I\nfMT01`3IfMT00`000?l0fMWI0000o`0D0=WIf@00703IfMT20000o`<0fMWI00<0003o0=WIf@3IfMT0\n403IfMT20000oe00fMWI1@0000040=WIf@030000003IfMT0fMWI04D0fMWI00<0003o0=WIf@3IfMT0\n903IfMT00`000?l0fMWI0=WIf@070=WIf@040000o`3IfMT0fMWI0000oa@0fMWI000N0=WIf@80003o\n0P3IfMT00`000?l0fMWI0=WIf@0A0=WIf@030000o`3IfMT0fMWI04l0fMWI00<000000=WIf@3IfMT0\n103IfMT00`000000fMWI0=WIf@130=WIf@80003o9`3IfMT00`000?l0fMWI0=WIf@070=WIf@030000\no`3IfMT0003o01D0fMWI000O0=WIf@80003o0P3IfMT20000oa80fMWI0P000?m?0=WIf@030000003I\nfMT0fMWI00@0fMWI0`0000110=WIf@80003o:03IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3I\nfMT0003o01H0fMWI000Q0=WIf@050000o`3IfMT0fMWI0=WIf@000?l04`3IfMT20000odd0fMWI00<0\n00000=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@1Z0=WIf@030000o`3IfMT0fMWI00P0fMWI00<0\n003o0=WIf@000?l05P3IfMT00280fMWI0P000?l20=WIf@80003o4`3IfMT20000od/0fMWI00<00000\n0=WIf@3IfMT0103IfMT00`000000fMWI0=WIf@1Z0=WIf@030000o`3IfMT0fMWI00L0fMWI00<0003o\n0=WIf@000?l05`3IfMT002<0fMWI0`000?l20=WIf@030000o`3IfMT0fMWI0180fMWI00<0003o0=WI\nf@3IfMT0A`3IfMT2000000H0fMWI00<000000=WIf@3IfMT0J@3IfMT00`000?l0fMWI0=WIf@070=WI\nf@040000o`3IfMT0fMWI0000oaL0fMWI000T0=WIf@@0003o00<0fMWI0000o`000?l04`3IfMT00`00\n0?l0fMWI0=WIf@1>0=WIf@030000003IfMT0fMWI06T0fMWI00<0003o0=WIf@3IfMT01P3IfMT01000\n0?l0fMWI0=WIf@000?lH0=WIf@009P3IfMT40000o`030=WIf@000?l0003o0180fMWI0P000?m>0=WI\nf@030000003IfMT0fMWI06P0fMWI00<0003o0=WIf@3IfMT01`3IfMT00`000?l0fMWI0000o`0I0=WI\nf@00:03IfMT30000o`80fMWI0P000?lB0=WIf@030000o`3IfMT0fMWI04/0fMWI00<000000=WIf@3I\nfMT0I`3IfMT00`000?l0fMWI0=WIf@070=WIf@030000o`3IfMT0003o01X0fMWI000Z0=WIf@<0003o\n0P3IfMT20000oa40fMWI0P000?m;0=WIf@030000003IfMT0fMWI06H0fMWI00<0003o0=WIf@3IfMT0\n1`3IfMT010000?l0fMWI0=WIf@000?lJ0=WIf@00;03IfMT20000o`<0fMWI0P000?lA0=WIf@80003o\nB@3IfMT2000006H0fMWI00<0003o0=WIf@3IfMT01`3IfMT010000?l0fMWI0=WIf@000?lK0=WIf@00\n;P3IfMT20000o`<0fMWI0P000?lA0=WIf@80003oA`3IfMT00`000000fMWI0=WIf@1T0=WIf@030000\no`3IfMT0fMWI00L0fMWI00@0003o0=WIf@3IfMT0003o703IfMT00300fMWI0P000?l30=WIf@80003o\n4@3IfMT20000odD0fMWI00<000000=WIf@3IfMT0HP3IfMT20000o`T0fMWI00@0003o0=WIf@3IfMT0\n003o7@3IfMT00380fMWI0`000?l20=WIf@80003o4@3IfMT20000od<0fMWI00<000000=WIf@3IfMT0\nH03IfMT20000o`X0fMWI00@0003o0=WIf@000?l0003o7P3IfMT003@0fMWI0`000?l20=WIf@80003o\n4@3IfMT30000od00fMWI00<000000=WIf@3IfMT0G`3IfMT00`000?l0fMWI0=WIf@090=WIf@030000\no`3IfMT0003o0200fMWI000g0=WIf@80003o0P3IfMT30000oa40fMWI0P000?ln0=WIf@030000003I\nfMT0fMWI05h0fMWI00<0003o0=WIf@3IfMT02@3IfMT00`000?l0fMWI0000o`0Q0=WIf@00>@3IfMT2\n0000o`<0fMWI0P000?lA0=WIf@80003o?03IfMT00`000000fMWI0=WIf@1M0=WIf@030000o`3IfMT0\nfMWI00X0fMWI0P000?lR0=WIf@00>`3IfMT30000o`80fMWI0`000?l@0=WIf@80003o>P3IfMT00`00\n0000fMWI0=WIf@1L0=WIf@030000o`3IfMT0fMWI00X0fMWI0P000?lS0=WIf@00?@3IfMT30000o`<0\nfMWI0P000?l@0=WIf@<0003o=`3IfMT2000005`0fMWI00<0003o0=WIf@3IfMT02P3IfMT00`000?l0\nfMWI0000o`0S0=WIf@00@03IfMT30000o`80fMWI0P000?lA0=WIf@80003o=@3IfMT00`000000fMWI\n0=WIf@1I0=WIf@80003o303IfMT00`000?l0fMWI0000o`0T0=WIf@00@P3IfMT30000o`80fMWI0P00\n0?lA0=WIf@@0003o<@3IfMT00`000000fMWI0=WIf@1G0=WIf@80003o303IfMT20000o`030=WIf@00\n0?l0fMWI02@0fMWI00150=WIf@<0003o00<0fMWI0000o`000?l04`3IfMT40000obd0fMWI00<00000\n0=WIf@3IfMT0EP3IfMT00`000?l0fMWI0=WIf@0;0=WIf@040000o`3IfMT0fMWI0000obH0fMWI0017\n0=WIf@<0003o00<0fMWI0000o`000?l05@3IfMT20000ob/0fMWI00<000000=WIf@3IfMT0E@3IfMT0\n0`000?l0fMWI0=WIf@0;0=WIf@040000o`3IfMT0003o0000obL0fMWI001:0=WIf@D0003o5@3IfMT3\n0000obP0fMWI00<000000=WIf@3IfMT0D`3IfMT20000o``0fMWI0P000?l00`3IfMT0003o0=WIf@0X\n0=WIf@00C03IfMT60000oaD0fMWI0`000?lU0=WIf@030000003IfMT0fMWI0580fMWI00<0003o0=WI\nf@3IfMT02`3IfMT010000?l0fMWI0=WIf@000?lZ0=WIf@00C`3IfMT50000oaH0fMWI0`000?lR0=WI\nf@030000003IfMT0fMWI0500fMWI0P000?l<0=WIf@80003o00<0fMWI0000o`000?l0:`3IfMT00580\nfMWI1P000?lE0=WIf@80003o803IfMT2000004h0fMWI0`000?l=0=WIf@040000o`3IfMT0fMWI0000\nobd0fMWI001E0=WIf@P0003o4P3IfMT40000oa`0fMWI00<000000=WIf@3IfMT0B`3IfMT20000o`l0\nfMWI00@0003o0=WIf@000?l0003o;P3IfMT005L0fMWI10000?l20=WIf@80003o503IfMT50000oaL0\nfMWI00<000000=WIf@3IfMT0B@3IfMT20000o`l0fMWI0P000?l00`3IfMT0003o0=WIf@0_0=WIf@00\nFP3IfMT40000o`040=WIf@000?l0003o0000oaH0fMWI0P000?lE0=WIf@030000003IfMT0fMWI04L0\nfMWI0P000?l@0=WIf@040000o`3IfMT0003o0000oc40fMWI001M0=WIf@<0003o0P3IfMT30000oaD0\nfMWI10000?lA0=WIf@030000003IfMT0fMWI04D0fMWI0P000?lA0=WIf@030000o`3IfMT0003o03<0\nfMWI001P0=WIf@<0003o0P3IfMT30000oaH0fMWI1@000?l<0=WIf@030000003IfMT0fMWI0480fMWI\n0`000?lA0=WIf@@0003o=03IfMT006<0fMWI0`000?l20=WIf@<0003o603IfMT30000o`T0fMWI00<0\n00000=WIf@3IfMT0?`3IfMT30000oa80fMWI0P000?l00`3IfMT0003o0=WIf@0e0=WIf@00IP3IfMT3\n0000o`80fMWI0`000?lH0=WIf@@0003o1@3IfMT00`000000fMWI0=WIf@0l0=WIf@<0003o4`3IfMT2\n0000o`030=WIf@000?l0003o03L0fMWI001Y0=WIf@<0003o0P3IfMT30000oaT0fMWI1P000?l00`00\n0000fMWI0=WIf@0g0=WIf@@0003o503IfMT20000o`030=WIf@000?l0003o03T0fMWI001/0=WIf@<0\n003o0P3IfMT30000oa/0fMWI00<000000000o`000?l010000?l]0=WIf@L0003o5P3IfMT20000o`03\n0=WIf@000?l0003o03/0fMWI001_0=WIf@@0003o00@0fMWI0000o`000?l0003o603IfMT00`000000\nfMWI0=WIf@040=WIf@L0003o803IfMT60000oaX0fMWI0`000?l00`3IfMT0003o0000o`0m0=WIf@00\nL`3IfMT90000oa<0fMWI00<000000=WIf@3IfMT02`3IfMTP0000oad0fMWI0`000?l20=WIf@80003o\n?`3IfMT007H0fMWI1@000?l00`3IfMT0003o0000o`030000o`h0fMWI00<000000=WIf@3IfMT0A@3I\nfMT30000o`80fMWI0`000?m10=WIf@00N@3IfMT50000o`<0fMWI0`000?l;0=WIf@030000003IfMT0\nfMWI0440fMWI10000?l20=WIf@@0003o@`3IfMT007h0fMWI1@000?l00`3IfMT0003o0000o`030000\no`H0fMWI00<000000=WIf@3IfMT0?P3IfMT30000o`<0fMWI10000?m60=WIf@00PP3IfMT50000o`80\n00001`000?ll0=WIf@@0003o0P3IfMT40000odX0fMWI00260=WIf@P0003o00<0fMWI000000000?l0\n1P000?l`0=WIf@D0003o0`3IfMT40000odd0fMWI00270=WIf@040000003IfMT0fMWI0=WIf@X0003o\n0P3IfMT60000ob<0fMWI1`000?l40=WIf@@0003oD@3IfMT008P0fMWI00<000000=WIf@3IfMT0103I\nfMT00`000000fMWI0=WIf@0:0000o`030=WIf@000?l0003o00D0003o5@3IfMT70000o`@0fMWI1`00\n0?mE0=WIf@00QP3IfMT010000000fMWI0=WIf@0000050=WIf@030000003IfMT0fMWI00L0fMWI2P00\n0?l00`3IfMT0003o0000o`0C0000o`@0fMWI2@000?mJ0=WIf@00Q`3IfMT2000000H0fMWI00<00000\n0=WIf@3IfMT03P3IfMTO0000of40fMWI002?0=WIf@030000003IfMT0fMWI08h0fMWI002?0=WIf@03\n0000003IfMT0fMWI08h0fMWI002?0=WIf@030000003IfMT0fMWI08h0fMWI002?0=WIf@030000003I\nfMT0fMWI08h0fMWI0000"]], Rule[ImageRangeCache, List[Rule[List[List[0, 287], List[176.938`, 0]], List[-2.10405`, -2.19557`, 0.014655`, 0.0248283`]]]]]], Open]]], Open]]], Rule[FrontEndVersion, "5.1 for Microsoft Windows"], Rule[ScreenRectangle, List[List[0, 1280], List[0, 951]]], Rule[WindowSize, List[914, 924]], Rule[WindowMargins, List[List[1, Automatic], List[Automatic, 0]]], Rule[StyleDefinitions, "MathModelica.nb"]], Null, Null, Null, Null, Null]