package Continuous "Continuous control blocks with internal states"
   import Modelica.Blocks.Interfaces;
   import Modelica.SIunits;
   extends Modelica.Icons.Library;
   annotation (preferedView="info",Documentation(info="
 <HTML>
 <p>
 This package contains basic <b>continuous</b> input/output blocks
 described by differential equations.
 </p>
 </HTML>
 "));
   block Integrator "Output the integral of the input signals"
      parameter Real k=1 "Integrator gains";
      parameter Real y_start=0 "Start values of integrators";
      extends Interfaces.SISO(y(start=y_start,fixed=true));
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output <b>y</b> (element-wise) as
 <i>integral</i> of the input <b>u</b> multiplied with
 the gain <i>k</i>:
 </p>
 <pre>
          k
      y = - u
          s
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{0.,-10.},{60.,-70.}},textString="I",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="k=%k",fillColor={0,0,0}),Line(points={{-80.,-80.},{80.,80.}},color={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255}),Text(extent={{-36.,60.},{32.,2.}},textString="k",fillColor={0,0,0}),Text(extent={{-32.,0.},{36.,-58.}},textString="s",fillColor={0,0,0}),Line(points={{-46.,0.},{46.,0.}},color={0,0,0})}));
      
   equation
      der(y) = k*u;
   end Integrator;
   block LimIntegrator "Integrator with limited values of the outputs"
      parameter Real k=1 "Integrator gains";
      parameter Real outMax=1 "Upper limits of outputs";
      parameter Real outMin=-outMax "Lower limits of outputs";
      parameter Real y_start=0 "Start values of integrators";
      extends Interfaces.SISO(y(start=y_start,fixed=true));
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes <b>y</b> (element-wise) as <i>integral</i>
 of the input <b>u</b> multiplied with the gain <i>k</i>. If the
 integral reaches a given upper or lower <i>limit</i> and the
 input will drive the integral outside of this bound, the
 integration is halted and only restarted if the input drives
 the integral away from the bounds.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 <p><b>Copyright (C) 1999-2000, Modelica Association and DLR.</b></p>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{20.,20.},{80.,20.}},color={0,0,255}),Text(extent={{0.,-10.},{60.,-70.}},textString="I",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="k=%k",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Text(extent={{-54.,46.},{-4.,-48.}},textString="lim",fillColor={0,0,0}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255}),Text(extent={{-8.,60.},{60.,2.}},textString="k",fillColor={0,0,0}),Text(extent={{-8.,-2.},{60.,-60.}},textString="s",fillColor={0,0,0}),Line(points={{4.,0.},{46.,0.}},color={0,0,0})}));
      
   equation
      der(y) = if y < outMin and u < 0 or y > outMax and u > 0 then
         0
      else
         k*u;
   end LimIntegrator;
   block Derivative "Approximated derivative block"
      parameter Real k=1 "Gains";
      parameter SIunits.Time T(min=Modelica.Constants.small)=0.01 "Time constants (T>0 required; T=0 is ideal derivative block)";
      extends Interfaces.SISO;
      output Real x "State of block";
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks defines the transfer function between the
 input u and the output y
 (element-wise) as <i>approximated derivative</i>:
 </p>
 <pre>
              k * s
      y = ------------ * u
             T * s + 1
 </pre>
 <p>
 If you would like to be able to change easily between different
 transfer functions (FirstOrder, SecondOrder, ... ) by changing
 parameters, use the general block <b>TransferFunction</b> instead
 and model a derivative block with parameters<br>
 b = {k,0}, a = {T, 1}.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 15, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Special handling, if k is zero. Introduced, in order that
        the D-part of the PID controllers can be set to zero without
        introducing numerical problems.</li>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,60.},{-70.,17.95},{-60.,-11.46},{-50.,-32.05},{-40.,-46.45},{-30.,-56.53},{-20.,-63.58},{-10.,-68.51},{0.,-71.96},{10.,-74.37},{20.,-76.06},{30.,-77.25},{40.,-78.07},{50.,-78.65},{60.,-79.06}},color={0,0,255}),Text(extent={{0.,0.},{60.,60.}},textString="DT1",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="k=%k",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-54.,52.},{50.,10.}},textString="k s",fillColor={0,0,0}),Text(extent={{-54.,-6.},{52.,-52.}},textString="T s + 1",fillColor={0,0,0}),Line(points={{-50.,0.},{50.,0.}},color={0,0,0}),Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255})}));
      
   equation
      der(x) = if noEvent(abs(k) >= Modelica.Constants.eps) then
         (u-x)/T
      else
         0;
      y = if noEvent(abs(k) >= Modelica.Constants.eps) then
         k/T*(u-x)
      else
         0;
   end Derivative;

   block FirstOrder "First order transfer function block (= 1 pole)"
      parameter Real k=1 "Gain";
      parameter SIunits.Time T=1 "Time Constant";
      extends Interfaces.SISO;
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks defines the transfer function between the input u
 and the output y (element-wise) as <i>first order</i> system:
 </p>
 <pre>
                k
      y = ------------ * u
             T * s + 1
 </pre>
 <p>
 If you would like to be able to change easily between different
 transfer functions (FirstOrder, SecondOrder, ... ) by changing
 parameters, use the general block <b>TransferFunction</b> instead
 and model a first order SISO system with parameters<br>
 b = {k}, a = {T, 1}.
 </p>
 <pre>
 Example:
    parameter: k = 0.3, T = 0.4
    results in:
              0.3
       y = ----------- * u
           0.4 s + 1.0
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,88.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-70.,-45.11},{-60.,-19.58},{-50.,-0.9087},{-40.,12.75},{-30.,22.75},{-20.,30.06},{-10.,35.41},{0.,39.33},{10.,42.19},{20.,44.29},{30.,45.82},{40.,46.94},{50.,47.76},{60.,48.36},{70.,48.8},{80.,49.12}},color={0,0,255}),Text(extent={{0.,0.},{60.,-60.}},textString="PT1",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="T=%T",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-48.,52.},{50.,8.}},textString="k",fillColor={0,0,0}),Text(extent={{-54.,-6.},{56.,-56.}},textString="T s + 1",fillColor={0,0,0}),Line(points={{-50.,0.},{50.,0.}},color={0,0,0}),Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255})}));
      
   equation
      der(y) = (k*u-y)/T;
   end FirstOrder;
   block SecondOrder "Second order transfer function block (= 2 poles)"
      parameter Real k=1 "Gain";
      parameter Real w=1 "Angular frequency";
      parameter Real D=1 "Damping";
      extends Interfaces.SISO;
      output Real yd "Derivative of y";
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks defines the transfer function between the input u and
 the output y (element-wise) as <i>second order</i> system:
 </p>
 <pre>
                              k
      y = ---------------------------------------- * u
             ( s / w )^2 + 2*D*( s / w ) + 1
 </pre>
 <p>
 If you would like to be able to change easily between different
 transfer functions (FirstOrder, SecondOrder, ... ) by changing
 parameters, use the general model class <b>TransferFunction</b>
 instead and model a second order SISO system with parameters<br>
 b = {k}, a = {1/w^2, 2*D/w, 1}.
 </p>
 <pre>
 Example:
 
    parameter: k =  0.3,  w = 0.5,  D = 0.4
    results in:
                   0.3
       y = ------------------- * u
           4.0 s^2 + 1.6 s + 1
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized and state space representation changed, such
        that the output and its derivative are used as state.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,88.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-72.,-68.53},{-64.,-39.5},{-56.,-2.522},{-48.,32.75},{-40.,58.8},{-32.,71.51},{-24.,70.49},{-16.,58.45},{-8.,40.06},{0.,20.55},{8.,4.459},{16.,-5.271},{24.,-7.629},{32.,-3.428},{40.,5.21},{48.,15.56},{56.,25.03},{64.,31.66},{72.,34.5},{80.,33.61}},color={0,0,255}),Text(extent={{0.,-10.},{60.,-70.}},textString="PT2",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="w=%w",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Text(extent={{-60.,60.},{60.,14.}},textString="k",fillColor={0,0,0}),Text(extent={{-60.,8.},{-32.,-20.}},textString="s",fillColor={0,0,0}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255}),Line(points={{-50.,14.},{50.,14.}},color={0,0,0}),Line(points={{-54.,-20.},{-38.,-20.}},color={0,0,0}),Text(extent={{-52.,-26.},{-36.,-48.}},textString="w",fillColor={0,0,0}),Line(points={{-50.,2.},{-56.,-8.},{-56.,-28.},{-52.,-46.}},color={0,0,0}),Line(points={{-40.,2.},{-34.,-10.},{-34.,-30.},{-38.,-46.}},color={0,0,0}),Text(extent={{-34.,8.},{-22.,-10.}},textString="2",fillColor={0,0,0}),Text(extent={{-34.,-6.},{6.,-36.}},textString="+2D",fillColor={0,0,0}),Text(extent={{2.,8.},{30.,-20.}},textString="s",fillColor={0,0,0}),Line(points={{8.,-20.},{24.,-20.}},color={0,0,0}),Text(extent={{10.,-26.},{26.,-48.}},textString="w",fillColor={0,0,0}),Line(points={{12.,2.},{6.,-8.},{6.,-28.},{10.,-46.}},color={0,0,0}),Line(points={{22.,2.},{28.,-10.},{28.,-30.},{24.,-46.}},color={0,0,0}),Text(extent={{30.,2.},{58.,-42.}},textString="+1",fillColor={0,0,0})}));
      
   equation
      der(y) = yd;
      der(yd) = w*(w*(k*u-y)-(2*D)*yd);
   end SecondOrder;
   block PI "Proportional-Integral controller"
      parameter Real k=1 "Gain";
      parameter SIunits.Time T=1 "Time Constant (T>0 required)";
      extends Interfaces.SISO;
      output Real x "State of block";
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks defines the transfer function between the input u and
 the output y (element-wise) as <i>PI</i> system:
 </p>
 <pre>
                  1
    y = k * (1 + ---) * u
                 T*s
            T*s + 1
      = k * ------- * u
              T*s
 </pre>
 <p>
 If you would like to be able to change easily between different
 transfer functions (FirstOrder, SecondOrder, ... ) by changing
 parameters, use the general model class <b>TransferFunction</b>
 instead and model a PI SISO system with parameters<br>
 b = {k*T, k}, a = {T, 0}.
 </p>
 <pre>
 Example:
 
    parameter: k = 0.3,  T = 0.4
 
    results in:
                0.4 s + 1
       y = 0.3 ----------- * u
                  0.4 s
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Nov. 4, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Vectorized.</li>
 <li><i>June 30, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,88.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,-20.},{60.,80.}},thickness=0.25,color={0,0,255}),Text(extent={{0.,6.},{60.,-56.}},textString="PI",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="T=%T",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Text(extent={{-68.,24.},{-24.,-18.}},textString="k",fillColor={0,0,0}),Text(extent={{-32.,48.},{60.,0.}},textString="T s + 1",fillColor={0,0,0}),Text(extent={{-30.,-8.},{52.,-40.}},textString="T s",fillColor={0,0,0}),Line(points={{-24.,0.},{54.,0.}},color={0,0,0}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{62.,0.},{100.,0.}},color={0,0,255})}));
      
   equation
      der(x) = u/T;
      y = k*(x+u);
   end PI;
   block PID "PID-controller in additive description form"
      extends Interfaces.SISO;
      parameter Real k=1 "Gain";
      parameter SIunits.Time Ti(min=Modelica.Constants.small)=0.5 "Time Constant of Integrator";
      parameter SIunits.Time Td(min=0)=0.1 "Time Constant of Derivative block";
      parameter Real Nd(min=Modelica.Constants.small)=10 "The higher Nd, the more ideal the derivative block";
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,50.},{-80.,-20.},{60.,80.}},color={0,0,255}),Text(extent={{-20.,-20.},{80.,-60.}},textString="PID",fillColor={192,192,192}),Text(extent={{-150.,-150.},{150.,-110.}},textString="Ti=%Ti",fillColor={0,0,0})}),Documentation(info="<HTML>
 <p>
 This is the text-book version of a PID-controller.
 For a more practically useful PID-controller, use
 block LimPID.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Aug. 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "));
      Blocks.Math.Gain P "Proportional part of PID controller" annotation (Placement(transformation(x=-40.,y=80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-40.,y=80.,scale=0.2,aspectRatio=1.)));
      Blocks.Continuous.Integrator I(k=1/Ti) "Integral part of PID controller" annotation (Placement(transformation(x=-40.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-40.,y=0.,scale=0.2,aspectRatio=1.)));
      Blocks.Continuous.Derivative D(k=Td,T=max([Td/Nd,100*Modelica.Constants.eps])) "Derivative part of PID controller" annotation (Placement(transformation(x=-40.,y=-80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-40.,y=-80.,scale=0.2,aspectRatio=1.)));
      Blocks.Math.Gain Gain(k=k) "Gain of PID controller" annotation (Placement(transformation(x=70.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=70.,y=0.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Add3 Add annotation (Placement(transformation(x=30.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=30.,y=0.,scale=0.1,aspectRatio=1.)));
      
   equation
      connect(P.y, Add.u1) annotation (Line(points={{-20.,80.},{0.,80.},{0.,8.},{20.,8.}},color={0,0,255}));
      connect(I.y, Add.u2) annotation (Line(points={{-20.,0.},{20.,0.}},color={0,0,255}));
      connect(D.y, Add.u3) annotation (Line(points={{-20.,-80.},{0.,-80.},{0.,-8.},{20.,-8.}},color={0,0,255}));
      connect(Add.y, Gain.u) annotation (Line(points={{40.,0.},{60.,0.}},color={0,0,255}));
      connect(y, Gain.y) annotation (Line(points={{100.,0.},{80.,0.}},color={0,0,255}));
      connect(u, I.u) annotation (Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}));
      connect(u, P.u) annotation (Line(points={{-100.,0.},{-80.,0.},{-80.,80.},{-60.,80.}},color={0,0,255}));
      connect(u, D.u) annotation (Line(points={{-100.,0.},{-80.,0.},{-80.,-80.},{-60.,-80.}},color={0,0,255}));
   end PID;
   block LimPID "PID controller with limited output, anti-windup compensation and setpoint weighting"
      extends Interfaces.SVcontrol;
      parameter Real k(min=0)=1 "Gain of PID block";
      parameter SIunits.Time Ti(min=Modelica.Constants.small)=0.5 "Time constant of Integrator block";
      parameter SIunits.Time Td(min=0)=0.1 "Time constant of Derivative block";
      parameter Real yMax=1 "Upper limit of output";
      parameter Real yMin=-yMax "Lower limit of output";
      parameter Real wp(min=0)=1 "Set-point weight for Proportional block (0..1)";
      parameter Real wd(min=0)=0 "Set-point weight for Derivative block (0..1)";
      parameter Real Ni(min=100*Modelica.Constants.eps)=0.9 "Ni*Ti is time constant of anti-windup compensation";
      parameter Real Nd(min=100*Modelica.Constants.eps)=10 "The higher Nd, the more ideal the derivative block";
      Blocks.Nonlinear.Limiter limiter(uMax=yMax,uMin=yMin) annotation (Placement(transformation(x=80.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=80.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,78.},{-80.,-90.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,-80.},{82.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,50.},{-80.,-20.},{30.,60.},{80.,60.}},color={0,0,255}),Text(extent={{-20.,-20.},{80.,-60.}},textString="PID",fillColor={192,192,192})}),Documentation(info="<HTML>
 <p>
 This is a PID controller incorporating several practical aspects.
 It is designed according to chapter 3 of the book
 </p>
 <pre>
    K. Astroem, T. Haegglund: PID Controllers: Theory, Design, and Tuning.
                              2nd edition, 1995.
 </pre>
 <p>
 Besides the additive <b>proportional, integral</b> and <b>derivative</b>
 part of this controller, the following practical aspects are included:
 </p>
 <ul>
 <li> The output of this controller is limited. If the controller is
      in its limits, anti-windup compensation is activated to drive
      the integrator state to zero. </li>
 <li> The high-frequency gain of the derivative part is limited
      to avoid excessive amplification of measurement noise.</li>
 <li> Setpoint weighting is present, which allows to weight
      the setpoint in the proportional and the derivative part
      independantly from the measurement. The controller will respond
      to load disturbances and measurement noise independantly of this setting
      (parameters wp, wd). However, setpoint changes will depend on this
      setting. For example, it is useful to set the setpoint weight wd
      for the derivative part to zero, if steps may occur in the
      setpoint signal.</li>
 </ul>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Aug. 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.</li>
 </ul>
 </HTML>
 "));
      Blocks.Math.Add addP(k1=wp,k2=-1) annotation (Placement(transformation(x=-70.,y=50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-70.,y=50.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Add addD(k1=wd,k2=-1) annotation (Placement(transformation(x=-70.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-70.,y=0.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Gain P annotation (Placement(transformation(x=-30.,y=50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-30.,y=50.,scale=0.1,aspectRatio=1.)));
      Blocks.Continuous.Integrator I(k=1/Ti) annotation (Placement(transformation(x=-30.,y=-50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-30.,y=-50.,scale=0.1,aspectRatio=1.)));
      Blocks.Continuous.Derivative D(k=Td,T=max([Td/Nd,1.e-14])) annotation (Placement(transformation(x=-30.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=-30.,y=0.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Gain gainPID(k=k) annotation (Placement(transformation(x=40.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=40.,y=0.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Add3 addPID annotation (Placement(transformation(x=10.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=10.,y=0.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Add3 addI(k2=-1) annotation (Placement(transformation(x=-70.,y=-50.,scale=0.1,aspectRatio=1.),iconTransformation(x=-70.,y=-50.,scale=0.1,aspectRatio=1.)));
      Blocks.Math.Add addSat(k2=-1) annotation (Placement(transformation(x=80.,y=-50.,scale=0.1,aspectRatio=1.,rotation=-90),iconTransformation(x=80.,y=-50.,scale=0.1,aspectRatio=1.,rotation=-90)));
      Blocks.Math.Gain gainTrack(k=1/(k*Ni)) annotation (Placement(transformation(x=30.,y=-70.,scale=0.1,aspectRatio=1.,flipHorizontal=true),iconTransformation(x=30.,y=-70.,scale=0.1,aspectRatio=1.,flipHorizontal=true)));
      
   equation
      assert(yMax >= yMin,"PID: Limits must be consistent");
      connect(u_s, addP.u1) annotation (Line(points={{-102.,0.},{-96.,0.},{-96.,56.},{-80.,56.}},color={0,0,255}));
      connect(u_m, addP.u2) annotation (Line(points={{0.,-100.},{0.,-92.},{-92.,-92.},{-92.,44.},{-80.,44.}},thickness=0.5,color={0,0,255}));
      connect(u_s, addD.u1) annotation (Line(points={{-102.,0.},{-96.,0.},{-96.,6.},{-82.,6.}},color={0,0,255}));
      connect(u_m, addD.u2) annotation (Line(points={{0.,-100.},{0.,-92.},{-92.,-92.},{-92.,-6.},{-82.,-6.},{-82.,-6.}},thickness=0.5,color={0,0,255}));
      connect(u_s, addI.u1) annotation (Line(points={{-100.,0.},{-96.,0.},{-96.,-42.},{-82.,-42.}},color={0,0,255}));
      connect(u_m, addI.u2) annotation (Line(points={{0.,-104.},{0.,-92.},{-92.,-92.},{-92.,-50.},{-80.,-50.}},thickness=0.5,color={0,0,255}));
      connect(gainTrack.y, addI.u3) annotation (Line(points={{20.,-70.},{-88.,-70.},{-88.,-58.},{-80.,-58.}},color={0,0,255}));
      connect(addP.y, P.u) annotation (Line(points={{-60.,50.},{-40.,50.},{-40.,50.}},color={0,0,255}));
      connect(addD.y, D.u) annotation (Line(points={{-60.,0.},{-42.,0.}},color={0,0,255}));
      connect(addI.y, I.u) annotation (Line(points={{-58.,-50.},{-40.,-50.}},color={0,0,255}));
      connect(P.y, addPID.u1) annotation (Line(points={{-18.,50.},{-10.,50.},{-10.,8.},{0.,8.}},color={0,0,255}));
      connect(D.y, addPID.u2) annotation (Line(points={{-20.,0.},{-2.,0.},{-2.,0.}},color={0,0,255}));
      connect(I.y, addPID.u3) annotation (Line(points={{-18.,-50.},{-10.,-50.},{-10.,-8.},{-2.,-8.},{-2.,-8.}},color={0,0,255}));
      connect(addPID.y, gainPID.u) annotation (Line(points={{21.,0.},{28.,0.}},color={0,0,255}));
      connect(gainPID.y, addSat.u2) annotation (Line(points={{50.,0.},{60.,0.},{60.,-30.},{74.,-30.},{74.,-40.}},color={0,0,255}));
      connect(addSat.y, gainTrack.u) annotation (Line(points={{80.,-62.},{80.,-70.},{42.,-70.}},color={0,0,255}));
      connect(gainPID.y, limiter.u) annotation (Line(points={{50.,0.},{70.,0.}},color={0,0,255}));
      connect(limiter.y, y) annotation (Line(points={{90.,0.},{100.,0.}},color={0,0,255}));
      connect(limiter.y, addSat.u1) annotation (Line(points={{90.,0.},{94.,0.},{94.,-20.},{86.,-20.},{86.,-40.}},color={0,0,255}));
   end LimPID;
   block TransferFunction "Linear transfer function"
      extends Interfaces.SISO;
      parameter Real b[:]={1} "Numerator coefficients of transfer function.";
      parameter Real a[:]={1,1} "Denominator coefficients of transfer function.";
      output Real x[size(a,1)-1] "State of transfer function from controller canonical form";
      
   protected
      parameter Integer na=size(a,1) "Size of Denominator of transfer function.";
      parameter Integer nb(max=na)=size(b,1) "Size of Numerator of transfer function.";
      parameter Integer nx=size(a,1)-1;
      Real x1dot "Derivative of first state of TransferFcn";
      Real xn "Highest order state of TransferFcn";
      annotation (Documentation(info="
 <HTML>
 <p>
 This block defines the transfer function between the input
 u and the output y
 as (nb = dimension of b, na = dimension of a):
 </p>
 <pre>
            b[1]*s^[nb-1] + b[2]*s^[nb-2] + ... + b[nb]
    y(s) = --------------------------------------------- * u(s)
            a[1]*s^[na-1] + a[2]*s^[na-2] + ... + a[na]
 </pre>
 <p>
 State variables <b>x</b> are defined according to <b>controller canonical</b>
 form. Initial values of the states can be set as start values of <b>x</b>.
 <p>
 <p>
 Example:
 </p>
 <pre>
      TransferFunction g(b = {2,4}, a = {1,3});
 </pre>
 <p>
 results in the following transfer function:
 </p>
 <pre>
         2*s + 4
    y = --------- * u
          s + 3
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized based on a realization of the corresponding
        Dymola library of Hilding Elmqvist.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,0.},{80.,0.}},color={0,0,255}),Text(extent={{-90.,10.},{90.,90.}},textString="b(s)",fillColor={0,0,255}),Text(extent={{-90.,-10.},{90.,-90.}},textString="a(s)",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{40.,0.},{-40.,0.}},color={0,0,0}),Text(extent={{-55.,55.},{55.,5.}},textString="b(s)",fillColor={0,0,0}),Text(extent={{-55.,-5.},{55.,-55.}},textString="a(s)",fillColor={0,0,0}),Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255})}));
      
   equation
      [der(x); xn] = [x1dot; x];
      [u] = transpose([a])*[x1dot; x];
      [y] = transpose([zeros(na-nb,1); b])*[x1dot; x];
   end TransferFunction;
   block StateSpace "Linear state space system"
      parameter Real A[:,size(A,1)]={{1,0},{0,1}} "Matrix A of state space model";
      parameter Real B[size(A,1),:]={{1},{1}} "Matrix B of state space model";
      parameter Real C[:,size(A,1)]={{1,1}} "Matrix C of state space model";
      parameter Real D[size(C,1),size(B,2)]=zeros(size(C,1),size(B,2)) "Matrix D of state space model";
      extends Interfaces.MIMO(final nin=size(B,2),final nout=size(C,1));
      output Real x[size(A,1)] "State vector";
      annotation (Documentation(info="
 <HTML>
 <p>
 The State Space block defines the relation
 between the input u and the output
 y in state space form:
 </p>
 <pre>
 
     der(x) = A * x + B * u
         y  = C * x + D * u
 </pre>
 <p>
 The input is a vector of length nu, the output is a vector
 of length ny and nx is the number of states. Accordingly
 </p>
 <pre>
         A has the dimension: A(nx,nx),
         B has the dimension: B(nx,nu),
         C has the dimension: C(ny,nx),
         D has the dimension: D(ny,nu)
 </pre>
 <p>
 Example:
 </p>
 <pre>
      parameter: A = [0.12, 2;3, 1.5]
      parameter: B = [2, 7;3, 1]
      parameter: C = [0.1, 2]
      parameter: D = zeros(ny,nu)
 results in the following equations:
   [der(x[1])]   [0.12  2.00] [x[1]]   [2.0  7.0] [u[1]]
   [         ] = [          ]*[    ] + [        ]*[    ]
   [der(x[2])]   [3.00  1.50] [x[2]]   [0.1  2.0] [u[2]]
                              [x[1]]            [u[1]]
        y[1]   = [0.1  2.0] * [    ] + [0  0] * [    ]
                              [x[2]]            [u[2]]
 </pre>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,10.},{-10.,90.}},textString="A",fillColor={0,0,255}),Text(extent={{10.,10.},{90.,90.}},textString="B",fillColor={0,0,255}),Text(extent={{-90.,-10.},{-10.,-90.}},textString="C",fillColor={0,0,255}),Text(extent={{10.,-10.},{90.,-90.}},textString="D",fillColor={0,0,255}),Line(points={{0.,-90.},{0.,90.}},color={192,192,192}),Line(points={{-90.,0.},{90.,0.}},color={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-60.,60.},{60.,-60.}},lineColor={0,0,255}),Text(extent={{-60.,40.},{60.,0.}},textString="sx=Ax+Bu",fillColor={0,0,0}),Text(extent={{-60.,0.},{60.,-40.}},textString=" y=Cx+Du",fillColor={0,0,0}),Line(points={{-100.,0.},{-60.,0.}},color={0,0,255}),Line(points={{60.,0.},{100.,0.}},color={0,0,255})}));
      
   equation
      der(x) = A*x+B*u;
      y = C*x+D*u;
   end StateSpace;
   block Der "Derivative of input (= analytic differentations)"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-80.,76.},{80.,-82.}},textString="der()",fillColor={0,0,255})}),Documentation(info="<HTML>
 <p>
 Defines that the output y is the <i>derivative</i>
 of the input u. Note, that Modelica.Blocks.Continuous.Derivative
 computes the derivative in an approximate sense, where as this block computes
 the derivative exactly. This requires that the input u is differentiated
 by the Modelica translator, if this derivative is not yet present in
 the model.
 </p>
 </HTML>"));
      
   equation
      y = der(u);
   end Der;


end Continuous;