package Math "Mathematical functions as input/output blocks"
   import Modelica.SIunits;
   import Modelica.Blocks.Interfaces;
   extends Modelica.Icons.Library;
   annotation (preferedView="info",Documentation(info="
 <HTML>
 <p>
 This package contains basic <b>mathematical operations</b>,
 such as summation and multiplication, and basic <b>mathematical
 functions</b>, such as <b>sqrt</b> and <b>sin</b>, as
 input/output blocks. All blocks of this library can be either
 connected with continuous blocks or with sampled-data blocks.
 In particular the following operations and functions are
 provided:
 </p>
 <pre>
    <b>TwoInputs</b>        Change causality of input signals.
    <b>TwoOutputs</b>       Change causality of output signals.
    <b>Gain</b>             Output the input multiplied by a scalar gain
    <b>MatrixGain</b>       Output the product of a gain matrix with the input
    <b>Sum</b>              Output the sum of the elements of the input vector
    <b>Feedback</b>         Output difference between commanded and feedback input
    <b>Add</b>              Output the sum of the two inputs
    <b>Add3</b>             Output the sum of the three inputs
    <b>Product</b>          Output product of the two inputs
    <b>Division</b>         Output first input divided by second input
    <b>Abs</b>              Output the absolute value of the input
    <b>Sign</b>             Output the sign of the input
    <b>Sqrt</b>             Output the square root of the input
    <b>Sin</b>              Output the sine of the input
    <b>Cos</b>              Output the cosine of the input
    <b>Tan</b>              Output the tangent of the input
    <b>Asin</b>             Output the arc sine of the input
    <b>Acos</b>             Output the arc cosine of the input
    <b>Atan</b>             Output the arc tangent of the input
    <b>Atan2</b>            Output atan(u1/u2) of the inputs u1 and u2
    <b>Sinh</b>             Output the hyperbolic sine of the input
    <b>Cosh</b>             Output the hyperbolic cosine of the input
    <b>Tanh</b>             Output the hyperbolic tangent of the input
    <b>Exp</b>              Output the exponential (base e) of the input
    <b>Log</b>              Output the natural (base e) logarithm of the input
    <b>Log10</b>            Output the base 10 logarithm of the input
    <b>RealToInteger</b>    Output the nearest Integer value to the input
    <b>IntegerToReal</b>    Output the input as Real value
    <b>RealToBoolean</b>    Convert Real to Boolean
    <b>BooleanToReal</b>    Convert Boolean to Real
    <b>IntegerToBoolean</b> Convert Integer to Boolean
    <b>BooleanToInteger</b> Convert Boolean to Integer
    <b>Min</b>              Output the minimum of the two inputs
    <b>Edge</b>             Set output to true at rising edge of the input
    <b>BooleanChange</b>    Set output to true when Boolean input changes
    <b>IntegerChange</b>    Set output to true when Integer input changes
 </pre>
 </HTML>
 ",revisions="<html>
 <ul>
 <li><i>October 21, 2002</i>
        by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>
        and <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
        New blocks added: RealToInteger, IntegerToReal, Max, Min, Edge, BooleanChange, IntegerChange.
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
        Realized (partly based on an existing Dymola library
        of Dieter Moormann and Hilding Elmqvist).
 </li>
 </ul>
 </html"));

   block Gain "Output the product of a gain value with the input signal"
      parameter Real k=1 "Gain value multiplied with input signal";
      
   public
      Interfaces.RealInput u "Input signal connector" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      Interfaces.RealOutput y "Output signal connector" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="
 <HTML>
 <p>
 This block computes output <i>y</i> as
 <i>product</i> of gain <i>k</i> with the
 input <i>u</i>:
 </p>
 <pre>
     y = k * u;
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>October 20, 2004</i>
        by <a href=\"http://www.robotic.dlr.de/Christian.Schweiger/\">Christian Schweiger</a>:<br>
        Icon changed. Documentation corrected to scalar implementation.
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.robotic.dlr.de/Martin.Otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,-100.},{-100.,100.},{100.,0.},{-100.,-100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-150.,-140.},{150.,-100.}},textString="k=%k",fillColor={0,0,0}),Text(extent={{-150.,140.},{150.,100.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-100.,-100.},{-100.,100.},{100.,0.},{-100.,-100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-76.,38.},{0.,-34.}},textString="k",fillColor={0,0,255})}));
      
   equation
      y = k*u;
   end Gain;
   block MatrixGain "Output the product of a gain matrix with the input signal vector"
      parameter Real K[:,:]={{1,0},{0,1}} "Gain matrix which is multiplied with the input";
      extends Interfaces.MIMO(final nin=size(K,2),final nout=size(K,1));
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output vector <b>y</b> as <i>product</i> of the
 gain matrix <b>K</b> with the input signal vector <b>u</b>:
 </p>
 <pre>
     <b>y</b> = <b>K</b> * <b>u</b>;
 </pre>
 <p>
 Example:
 </p>
 <pre>
    parameter: <b>K</b> = [0.12 2; 3 1.5]
    results in the following equations:
      | y[1] |     | 0.12  2.00 |   | u[1] |
      |      |  =  |            | * |      |
      | y[2] |     | 3.00  1.50 |   | u[2] |
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,-60.},{90.,60.}},textString="*K",fillColor={160,160,160}),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-90.,-60.},{90.,60.}},textString="*K",fillColor={160,160,160})}));
      
   equation
      y = K*u;
   end MatrixGain;
   block Sum "Output the sum of the elements of the input vector"
      extends Interfaces.MISO;
      parameter Real k[nin]=ones(nin) "Optional: sum coefficients";
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output <b>y</b> as
 <i>sum</i> of the elements of the input signal vector
 <b>u</b>:
 </p>
 <pre>
     <b>y</b> = <b>u</b>[1] + <b>u</b>[2] + ...;
 </pre>
 <p>
 Example:
 </p>
 <pre>
      parameter:   nin = 3;
   results in the following equations:
      y = u[1] + u[2] + u[3];
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>Oct. 24, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{26.,42.},{-34.,42.},{6.,2.},{-34.,-38.},{26.,-38.}},color={0,0,0},thickness=0.25),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{26.,42.},{-34.,42.},{6.,2.},{-34.,-38.},{26.,-38.}},color={0,0,0},thickness=0.25)}));
      
   equation
      y = k*u;
   end Sum;
   block Feedback "Output difference between commanded and feedback input"
      input Interfaces.RealInput u1 annotation (Placement(transformation(x=-80.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-80.,y=0.,scale=0.2,aspectRatio=1.)));
      input Interfaces.RealInput u2 annotation (Placement(transformation(x=0.,y=-80.,scale=0.2,aspectRatio=1.,rotation=90),iconTransformation(x=0.,y=-80.,scale=0.2,aspectRatio=1.,rotation=90)));
      output Interfaces.RealOutput y annotation (Placement(transformation(x=90.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=90.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output <b>y</b> as <i>difference</i> of the
 commanded input <b>u1</b> and the feedback
 input <b>u2</b>:
 </p>
 <pre>
     <b>y</b> = <b>u1</b> - <b>u2</b>;
 </pre>
 <p>
 Example:
 </p>
 <pre>
      parameter:   n = 2
   results in the following equations:
      y = u1 - u2
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-20.,20.},{20.,-20.}},lineColor={0,0,191},fillColor={235,235,235},fillPattern=FillPattern.Solid),Line(points={{-60.,0.},{-20.,0.}},color={0,0,191}),Line(points={{20.,0.},{80.,0.}},color={0,0,191}),Line(points={{0.,-20.},{0.,-60.}},color={0,0,191}),Text(extent={{-14.,0.},{82.,-94.}},textString="-",fillColor={0,0,0}),Text(extent={{-100.,110.},{100.,60.}},textString="%name",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Ellipse(extent={{-20.,20.},{20.,-20.}},pattern=LinePattern.Solid,lineThickness=0.25,fillColor={235,235,235},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Line(points={{-60.,0.},{-20.,0.}},color={0,0,255}),Line(points={{20.,0.},{80.,0.}},color={0,0,255}),Line(points={{0.,-20.},{0.,-60.}},color={0,0,255}),Text(extent={{-12.,10.},{84.,-84.}},textString="-",fillColor={0,0,0})}));
      
   equation
      y = u1-u2;
   end Feedback;
   block Add "Output the sum of the two inputs"
      extends Interfaces.SI2SO;
      parameter Real k1=1 "Gain of upper input";
      parameter Real k2=1 "Gain of lower input";
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output <b>y</b> as <i>sum</i> of the
 two input signals <b>u1</b> and <b>u2</b>:
 </p>
 <pre>
     <b>y</b> = k1*<b>u1</b> + k2*<b>u2</b>;
 </pre>
 <p>
 Example:
 </p>
 <pre>
      parameter:   k1= +2, k2= -3
   results in the following equations:
      y = 2 * u1 - 3 * u2
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-98.,-52.},{7.,-92.}},textString="%k2",fillColor={0,0,0}),Text(extent={{-100.,90.},{5.,50.}},textString="%k1",fillColor={0,0,0}),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255}),Line(points={{-100.,60.},{-40.,60.},{-30.,40.}},color={0,0,255}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,255}),Line(points={{-100.,-60.},{-40.,-60.},{-30.,-40.}},color={0,0,255}),Line(points={{-15.,-25.99},{15.,25.99}},color={0,0,0}),Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{50.,0.},{100.,0.}},color={0,0,255}),Line(points={{-100.,60.},{-74.,24.},{-44.,24.}},color={0,0,191}),Line(points={{-100.,-60.},{-74.,-28.},{-42.,-28.}},color={0,0,191}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,191}),Line(points={{50.,0.},{100.,0.}},color={0,0,191}),Text(extent={{-38.,34.},{38.,-34.}},textString="+",fillColor={0,0,0}),Text(extent={{-100.,52.},{5.,92.}},textString="%k1",fillColor={0,0,0}),Text(extent={{-100.,-52.},{5.,-92.}},textString="%k2",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-98.,-52.},{7.,-92.}},textString="%k2",fillColor={0,0,0}),Text(extent={{-100.,90.},{5.,50.}},textString="%k1",fillColor={0,0,0}),Line(points={{-100.,60.},{-40.,60.},{-30.,40.}},color={0,0,255}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,255}),Line(points={{-100.,-60.},{-40.,-60.},{-30.,-40.}},color={0,0,255}),Line(points={{-15.,-25.99},{15.,25.99}},color={0,0,0}),Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,191},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{50.,0.},{100.,0.}},color={0,0,255}),Line(points={{-100.,60.},{-74.,24.},{-44.,24.}},color={0,0,191}),Line(points={{-100.,-60.},{-74.,-28.},{-42.,-28.}},color={0,0,191}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,191}),Line(points={{50.,0.},{100.,0.}},color={0,0,191}),Text(extent={{-38.,34.},{38.,-34.}},textString="+",fillColor={0,0,0}),Text(extent={{-100.,52.},{5.,92.}},textString="k1",fillColor={0,0,0}),Text(extent={{-100.,-52.},{5.,-92.}},textString="k2",fillColor={0,0,0})}));
      
   equation
      y = k1*u1+k2*u2;
   end Add;
   block Add3 "Output the sum of the three inputs"
      extends Interfaces.BlockIcon;
      parameter Real k1=1 "Gain of upper input";
      parameter Real k2=1 "Gain of middle input";
      parameter Real k3=1 "Gain of lower input";
      input Interfaces.RealInput u1 "Connector 1 of Real input signals" annotation (Placement(transformation(x=-120.,y=80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=80.,scale=0.2,aspectRatio=1.)));
      input Interfaces.RealInput u2 "Connector 2 of Real input signals" annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      input Interfaces.RealInput u3 "Connector 3 of Real input signals" annotation (Placement(transformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=-80.,scale=0.2,aspectRatio=1.)));
      output Interfaces.RealOutput y "Connector of Real output signals" annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes output <b>y</b> as <i>sum</i> of the
 three input signals <b>u1</b>, <b>u2</b> and <b>u3</b>:
 </p>
 <pre>
     <b>y</b> = k1*<b>u1</b> + k2*<b>u2</b> + k3*<b>u3</b>;
 </pre>
 <p>
 Example:
 </p>
 <pre>
      parameter:   k1= +2, k2= -3, k3=1;
   results in the following equations:
      y = 2 * u1 - 3 * u2 + u3;
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-100.,50.},{5.,90.}},textString="%k1",fillColor={0,0,0}),Text(extent={{-100.,-20.},{5.,20.}},textString="%k2",fillColor={0,0,0}),Text(extent={{-100.,-50.},{5.,-90.}},textString="%k3",fillColor={0,0,0}),Text(extent={{2.,36.},{100.,-44.}},textString="+",fillColor={0,0,0})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,50.},{5.,90.}},textString="%k1",fillColor={0,0,0}),Text(extent={{-100.,-20.},{5.,20.}},textString="%k2",fillColor={0,0,0}),Text(extent={{-100.,-50.},{5.,-90.}},textString="%k3",fillColor={0,0,0}),Text(extent={{2.,36.},{100.,-44.}},textString="+",fillColor={0,0,0}),Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Text(extent={{-100.,50.},{5.,90.}},textString="k1",fillColor={0,0,0}),Text(extent={{-100.,-20.},{5.,20.}},textString="k2",fillColor={0,0,0}),Text(extent={{-100.,-50.},{5.,-90.}},textString="k3",fillColor={0,0,0}),Text(extent={{2.,36.},{100.,-44.}},textString="+",fillColor={0,0,0})}));
      
   equation
      y = (k1*u1+k2*u2)+k3*u3;
   end Add3;
   block Product "Output product of the two inputs"
      extends Interfaces.SI2SO;
      annotation (Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <i>product</i> of the corresponding elements of
 the two inputs <b>u1</b> and <b>u2</b>:
 </p>
 <pre>
     y = u1 * u2;
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-100.,60.},{-40.,60.},{-30.,40.}},color={0,0,191}),Line(points={{-100.,-60.},{-40.,-60.},{-30.,-40.}},color={0,0,191}),Line(points={{50.,0.},{100.,0.}},color={0,0,191}),Line(points={{-30.,0.},{30.,0.}},color={0,0,0}),Line(points={{-15.,25.99},{15.,-25.99}},color={0,0,0}),Line(points={{-15.,-25.99},{15.,25.99}},color={0,0,0}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,191})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{-100.,60.},{-40.,60.},{-30.,40.}},color={0,0,255}),Line(points={{-100.,-60.},{-40.,-60.},{-30.,-40.}},color={0,0,255}),Line(points={{50.,0.},{100.,0.}},color={0,0,255}),Line(points={{-30.,0.},{30.,0.}},color={0,0,0}),Line(points={{-15.,25.99},{15.,-25.99}},color={0,0,0}),Line(points={{-15.,-25.99},{15.,25.99}},color={0,0,0}),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,255})}));
      
   equation
      y = u1*u2;
   end Product;
   block Division "Output first input divided by second input"
      extends Interfaces.SI2SO;
      annotation (Documentation(info="
 <HTML>
 <p>
 This block computes the output <b>y</b> (element-wise)
 by <i>dividing</i> the corresponding elements of
 the two inputs <b>u1</b> and <b>u2</b>:
 </p>
 <pre>
     y = u1 / u2;
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{50.,0.},{100.,0.}},color={0,0,191}),Line(points={{-30.,0.},{30.,0.}},color={0,0,0}),Ellipse(extent={{-5.,20.},{5.,30.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Ellipse(extent={{-5.,-20.},{5.,-30.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,191}),Text(extent={{-150.,150.},{150.,110.}},textString="%name",fillColor={0,0,255}),Line(points={{-100.,60.},{-66.,60.},{-40.,30.}},color={0,0,191}),Line(points={{-100.,-60.},{0.,-60.},{0.,-50.}},color={0,0,191})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Rectangle(extent={{-100.,-100.},{100.,100.}},lineColor={0,0,255},fillColor={255,255,255},fillPattern=FillPattern.Solid),Line(points={{50.,0.},{100.,0.}},color={0,0,255}),Line(points={{-30.,0.},{30.,0.}},color={0,0,0}),Ellipse(extent={{-5.,20.},{5.,30.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Ellipse(extent={{-5.,-20.},{5.,-30.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Ellipse(extent={{-50.,50.},{50.,-50.}},lineColor={0,0,255}),Line(points={{-100.,60.},{-66.,60.},{-40.,30.}},color={0,0,255}),Line(points={{-100.,-60.},{0.,-60.},{0.,-50.}},color={0,0,255})}));
      
   equation
      y = u1/u2;
   end Division;
   block Abs "Output the absolute value of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{92.,0.},{70.,8.},{70.,-8.},{92.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{0.,0.},{80.,80.}},color={0,0,0}),Line(points={{0.,-14.},{0.,68.}},color={192,192,192}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-34.,-28.},{38.,-76.}},textString="abs",fillColor={192,192,192}),Line(points={{-88.,0.},{76.,0.}},color={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-100.,0.},{76.,0.}},color={192,192,192}),Polygon(points={{92.,0.},{76.,6.},{76.,-6.},{92.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{0.,0.},{80.,80.}},color={0,0,0}),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{7.,98.},{34.,78.}},textString="u",fillColor={160,160,160}),Text(extent={{74.,-8.},{96.,-28.}},textString="y",fillColor={160,160,160}),Text(extent={{52.,-3.},{72.,-23.}},textString="1",fillColor={0,0,255}),Text(extent={{-86.,-1.},{-66.,-21.}},textString="-1",fillColor={0,0,255}),Text(extent={{-28.,79.},{-8.,59.}},textString="1",fillColor={0,0,255})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <i>absolute value</i> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>abs</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = abs(u);
   end Abs;
   block Sign "Output the sign of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{0.,-80.}},color={0,0,0}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Text(extent={{-90.,72.},{-18.,24.}},textString="sign",fillColor={192,192,192}),Line(points={{0.,80.},{80.,80.}},color={0,0,0}),Rectangle(extent={{-2.,2.},{2.,-4.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{0.,-80.}},color={0,0,0}),Line(points={{-0.01,0.},{0.01,0.}},color={0,0,0}),Line(points={{0.,80.},{80.,80.}},color={0,0,0}),Rectangle(extent={{-2.,2.},{2.,-4.}},lineColor={0,0,0},fillColor={0,0,0},fillPattern=FillPattern.Solid),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{7.,102.},{32.,82.}},textString="y",fillColor={160,160,160}),Text(extent={{70.,-6.},{94.,-26.}},textString="u",fillColor={160,160,160}),Text(extent={{-25.,86.},{-5.,70.}},textString="1",fillColor={0,0,255}),Text(extent={{5.,-72.},{25.,-88.}},textString="-1",fillColor={0,0,255})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <b>sign</b> of the input <b>u</b>:
 </p>
 <pre>
          1  <b>if</b> u > 0
     y =  0  <b>if</b> u == 0
         -1  <b>if</b> u < 0
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = sign(u);
   end Sign;
   block Sqrt "Output the square root of the input (input >= 0 required)"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-90.,-80.},{68.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-79.2,-68.7},{-78.4,-64.},{-76.8,-57.3},{-73.6,-47.9},{-67.9,-36.1},{-59.1,-22.2},{-46.2,-6.49},{-28.5,10.7},{-4.42,30.},{27.7,51.3},{69.5,74.7},{80.,80.}},color={0,0,0}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-88.},{-80.,68.}},color={192,192,192}),Text(extent={{-8.,-4.},{64.,-52.}},textString="sqrt",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-92.,-80.},{84.,-80.}},color={192,192,192}),Polygon(points={{100.,-80.},{84.,-74.},{84.,-86.},{100.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-79.2,-68.7},{-78.4,-64.},{-76.8,-57.3},{-73.6,-47.9},{-67.9,-36.1},{-59.1,-22.2},{-46.2,-6.49},{-28.5,10.7},{-4.42,30.},{27.7,51.3},{69.5,74.7},{80.,80.}},color={0,0,0}),Polygon(points={{-80.,98.},{-86.,82.},{-74.,82.},{-80.,98.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-90.},{-80.,84.}},color={192,192,192}),Text(extent={{-71.,98.},{-44.,78.}},textString="y",fillColor={160,160,160}),Text(extent={{60.,-52.},{84.,-72.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <i>square root</i> of the input <b>u</b>:
 </p>
 <pre>
     y = sqrt( u );
 </pre>
 <p>
 All elements of the input vector shall be zero or positive.
 Otherwise an error occurs.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = sqrt(u);
   end Sqrt;
   block Sin "Output the sine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,68.}},color={192,192,192}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Line(points={{-80.,0.},{-68.7,34.2},{-61.5,53.1},{-55.1,66.4},{-49.4,74.6},{-43.8,79.1},{-38.2,79.8},{-32.6,76.6},{-26.9,69.7},{-21.3,59.4},{-14.9,44.1},{-6.83,21.2},{10.1,-30.8},{17.3,-50.2},{23.7,-64.2},{29.3,-73.1},{35.,-78.4},{40.6,-80.},{46.2,-77.6},{51.9,-71.5},{57.5,-61.9},{63.9,-47.2},{72.,-24.8},{80.,0.}},color={0,0,0}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{12.,84.},{84.,36.}},textString="sin",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,80.},{-88.,80.}},color={192,192,192}),Line(points={{-80.,-80.},{-88.,-80.}},color={192,192,192}),Line(points={{-80.,-90.},{-80.,84.}},color={192,192,192}),Text(extent={{-75.,98.},{-46.,78.}},textString="y",fillColor={160,160,160}),Polygon(points={{-80.,96.},{-86.,80.},{-74.,80.},{-80.,96.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,0.},{-68.7,34.2},{-61.5,53.1},{-55.1,66.4},{-49.4,74.6},{-43.8,79.1},{-38.2,79.8},{-32.6,76.6},{-26.9,69.7},{-21.3,59.4},{-14.9,44.1},{-6.83,21.2},{10.1,-30.8},{17.3,-50.2},{23.7,-64.2},{29.3,-73.1},{35.,-78.4},{40.6,-80.},{46.2,-77.6},{51.9,-71.5},{57.5,-61.9},{63.9,-47.2},{72.,-24.8},{80.,0.}},color={0,0,0}),Text(extent={{-105.,72.},{-85.,88.}},textString="1",fillColor={0,0,255}),Text(extent={{70.,25.},{90.,5.}},textString="2*pi",fillColor={0,0,255}),Text(extent={{-105.,-72.},{-85.,-88.}},textString="-1",fillColor={0,0,255}),Text(extent={{76.,-10.},{98.,-30.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <b>sine</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>sin</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.sin(u);
   end Sin;
   block Cos "Output the cosine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,68.}},color={192,192,192}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{-74.4,78.1},{-68.7,72.3},{-63.1,63.},{-56.7,48.7},{-48.6,26.6},{-29.3,-32.5},{-22.1,-51.7},{-15.7,-65.3},{-10.1,-73.8},{-4.42,-78.8},{1.21,-79.9},{6.83,-77.1},{12.5,-70.6},{18.1,-60.6},{24.5,-45.7},{32.6,-23.},{50.3,31.3},{57.5,50.7},{63.9,64.6},{69.5,73.4},{75.2,78.6},{80.,80.}},color={0,0,0}),Text(extent={{-36.,82.},{36.,34.}},textString="cos",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,80.},{-88.,80.}},color={192,192,192}),Line(points={{-80.,-80.},{-88.,-80.}},color={192,192,192}),Line(points={{-80.,-90.},{-80.,84.}},color={192,192,192}),Text(extent={{-75.,102.},{-48.,82.}},textString="y",fillColor={160,160,160}),Polygon(points={{-80.,100.},{-86.,84.},{-74.,84.},{-80.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-103.,72.},{-83.,88.}},textString="1",fillColor={0,0,255}),Text(extent={{-79.,-72.},{-59.,-88.}},textString="-1",fillColor={0,0,255}),Text(extent={{70.,25.},{90.,5.}},textString="2*pi",fillColor={0,0,255}),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{-74.4,78.1},{-68.7,72.3},{-63.1,63.},{-56.7,48.7},{-48.6,26.6},{-29.3,-32.5},{-22.1,-51.7},{-15.7,-65.3},{-10.1,-73.8},{-4.42,-78.8},{1.21,-79.9},{6.83,-77.1},{12.5,-70.6},{18.1,-60.6},{24.5,-45.7},{32.6,-23.},{50.3,31.3},{57.5,50.7},{63.9,64.6},{69.5,73.4},{75.2,78.6},{80.,80.}},color={0,0,0}),Text(extent={{74.,-4.},{98.,-24.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <b>cos</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>cos</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.cos(u);
   end Cos;
   block Tan "Output the tangent of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Line(points={{-80.,-80.},{-78.4,-68.4},{-76.8,-59.7},{-74.4,-50.},{-71.2,-40.9},{-67.1,-33.},{-60.7,-24.8},{-51.1,-17.2},{-35.8,-9.98},{-4.42,-1.07},{33.4,9.12},{49.4,16.2},{59.1,23.2},{65.5,30.6},{70.4,39.1},{73.6,47.4},{76.,56.1},{77.6,63.8},{80.,80.}},color={0,0,0}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-90.,72.},{-18.,24.}},textString="tan",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-88.},{0.,86.}},color={192,192,192}),Text(extent={{11.,100.},{38.,80.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,102.},{-6.,86.},{6.,86.},{0.,102.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-37.,-72.},{-17.,-88.}},textString="-5.8",fillColor={0,0,255}),Text(extent={{-33.,86.},{-13.,70.}},textString=" 5.8",fillColor={0,0,255}),Text(extent={{70.,25.},{90.,5.}},textString="1.4",fillColor={0,0,255}),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-78.4,-68.4},{-76.8,-59.7},{-74.4,-50.},{-71.2,-40.9},{-67.1,-33.},{-60.7,-24.8},{-51.1,-17.2},{-35.8,-9.98},{-4.42,-1.07},{33.4,9.12},{49.4,16.2},{59.1,23.2},{65.5,30.6},{70.4,39.1},{73.6,47.4},{76.,56.1},{77.6,63.8},{80.,80.}},color={0,0,0}),Text(extent={{70.,-6.},{94.,-26.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise)
 as <b>tan</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>tan</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.tan(u);
   end Tan;
   block Asin "Output the arc sine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Line(points={{-80.,-80.},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,-49.8},{-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,52.7},{75.2,62.2},{77.6,67.5},{80.,80.}},color={0,0,0}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-88.,78.},{-16.,30.}},textString="asin",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{9.,102.},{36.,82.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-40.,-72.},{-15.,-88.}},textString="-pi/2",fillColor={0,0,255}),Text(extent={{-38.,88.},{-13.,72.}},textString=" pi/2",fillColor={0,0,255}),Text(extent={{70.,25.},{90.,5.}},textString="+1",fillColor={0,0,255}),Text(extent={{-90.,21.},{-70.,1.}},textString="-1",fillColor={0,0,255}),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-79.2,-72.8},{-77.6,-67.5},{-73.6,-59.4},{-66.3,-49.8},{-53.5,-37.3},{-30.2,-19.7},{37.4,24.8},{57.5,40.8},{68.7,52.7},{75.2,62.2},{77.6,67.5},{80.,80.}},color={0,0,0}),Text(extent={{74.,-4.},{100.,-26.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>sine-inverse</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>asin</b>( u );
 </pre>
 <p>
 The absolute values of the elements of the input <b>u</b> need to
 be less or equal to one (<b>abs</b>( u ) <= 1).
 Otherwise an error occurs.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.asin(u);
   end Asin;
   block Acos "Output the arc cosine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{-79.2,72.8},{-77.6,67.5},{-73.6,59.4},{-66.3,49.8},{-53.5,37.3},{-30.2,19.7},{37.4,-24.8},{57.5,-40.8},{68.7,-52.7},{75.2,-62.2},{77.6,-67.5},{80.,-80.}},color={0,0,0}),Line(points={{0.,-88.},{0.,68.}},color={192,192,192}),Line(points={{-90.,-80.},{68.,-80.}},color={192,192,192}),Polygon(points={{90.,-80.},{68.,-72.},{68.,-88.},{90.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-86.,-14.},{-14.,-62.}},textString="acos",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{13.,100.},{42.,82.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,-80.},{84.,-80.}},color={192,192,192}),Polygon(points={{100.,-80.},{84.,-74.},{84.,-86.},{100.,-80.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{-79.2,72.8},{-77.6,67.5},{-73.6,59.4},{-66.3,49.8},{-53.5,37.3},{-30.2,19.7},{37.4,-24.8},{57.5,-40.8},{68.7,-52.7},{75.2,-62.2},{77.6,-67.5},{80.,-80.}},color={0,0,0}),Text(extent={{-30.,88.},{-5.,72.}},textString=" pi",fillColor={0,0,255}),Text(extent={{-94.,-57.},{-74.,-77.}},textString="-1",fillColor={0,0,255}),Text(extent={{80.,-45.},{100.,-65.}},textString="+1",fillColor={0,0,255}),Text(extent={{76.,-84.},{102.,-102.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>cosine-inverse</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>acos</b>( u );
 </pre>
 <p>
 The absolute values of the elements of the input <b>u</b> need to
 be less or equal to one (<b>abs</b>( u ) <= 1).
 Otherwise an error occurs.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.acos(u);
   end Acos;
   block Atan "Output the arc tangent of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Line(points={{-80.,-80.},{-52.7,-75.2},{-37.4,-69.7},{-26.9,-63.},{-19.7,-55.2},{-14.1,-45.8},{-10.1,-36.4},{-6.03,-23.9},{-1.21,-5.06},{5.23,21.},{9.25,34.1},{13.3,44.2},{18.1,52.9},{24.5,60.8},{33.4,67.6},{47.,73.6},{69.5,78.6},{80.,80.}},color={0,0,0}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-86.,68.},{-14.,20.}},textString="atan",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{13.,102.},{42.,82.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-52.7,-75.2},{-37.4,-69.7},{-26.9,-63.},{-19.7,-55.2},{-14.1,-45.8},{-10.1,-36.4},{-6.03,-23.9},{-1.21,-5.06},{5.23,21.},{9.25,34.1},{13.3,44.2},{18.1,52.9},{24.5,60.8},{33.4,67.6},{47.,73.6},{69.5,78.6},{80.,80.}},color={0,0,0}),Text(extent={{-32.,91.},{-12.,71.}},textString="1.4",fillColor={0,0,255}),Text(extent={{-32.,-71.},{-12.,-91.}},textString="-1.4",fillColor={0,0,255}),Text(extent={{73.,26.},{93.,10.}},textString=" 5.8",fillColor={0,0,255}),Text(extent={{-103.,20.},{-83.,4.}},textString="-5.8",fillColor={0,0,255}),Text(extent={{66.,-8.},{94.,-28.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>tangent-inverse</b> of the input <b>u</b>:
 </p>
 <pre>
     y= <b>atan</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.atan(u);
   end Atan;

   block Sinh "Output the hyperbolic sine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-86.,80.},{-14.,32.}},textString="sinh",fillColor={192,192,192}),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Line(points={{-80.,-80.},{-76.,-65.4},{-71.2,-51.4},{-65.5,-38.8},{-59.1,-28.1},{-51.1,-18.7},{-41.4,-11.4},{-27.7,-5.5},{-4.42,-0.653},{24.5,4.57},{39.,10.1},{49.4,17.2},{57.5,25.9},{63.9,35.8},{69.5,47.4},{74.4,60.4},{78.4,73.8},{80.,80.}},color={0,0,0}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{9.,98.},{34.,78.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-76.,-65.4},{-71.2,-51.4},{-65.5,-38.8},{-59.1,-28.1},{-51.1,-18.7},{-41.4,-11.4},{-27.7,-5.5},{-4.42,-0.653},{24.5,4.57},{39.,10.1},{49.4,17.2},{57.5,25.9},{63.9,35.8},{69.5,47.4},{74.4,60.4},{78.4,73.8},{80.,80.}},color={0,0,0}),Text(extent={{-31.,72.},{-11.,88.}},textString="27",fillColor={0,0,255}),Text(extent={{-35.,-88.},{-15.,-72.}},textString="-27",fillColor={0,0,255}),Text(extent={{70.,25.},{90.,5.}},textString="4",fillColor={0,0,255}),Text(extent={{-98.,21.},{-78.,1.}},textString="-4",fillColor={0,0,255}),Text(extent={{72.,-6.},{96.,-26.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>hyperbolic sine</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>sinh</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.sinh(u);
   end Sinh;
   block Cosh "Output the hyperbolic cosine of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Text(extent={{4.,66.},{66.,20.}},textString="cosh",fillColor={192,192,192}),Line(points={{-80.,80.},{-77.6,61.1},{-74.4,39.3},{-71.2,20.7},{-67.1,1.29},{-63.1,-14.6},{-58.3,-29.8},{-52.7,-43.5},{-46.2,-55.1},{-39.,-64.3},{-30.2,-71.7},{-18.9,-77.1},{-4.42,-79.9},{10.9,-79.1},{23.7,-75.2},{34.2,-68.7},{42.2,-60.6},{48.6,-51.2},{54.3,-40.},{59.1,-27.5},{63.1,-14.6},{67.1,1.29},{71.2,20.7},{74.4,39.3},{77.6,61.1},{80.,80.}},color={0,0,0}),Line(points={{-90.,-86.083},{68.,-86.083}},color={192,192,192}),Polygon(points={{90.,-86.083},{68.,-78.083},{68.,-94.083},{90.,-86.083}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{11.,98.},{38.,78.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,-86.083},{84.,-86.083}},color={192,192,192}),Polygon(points={{100.,-86.083},{84.,-80.083},{84.,-92.083},{100.,-86.083}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,80.},{-77.6,61.1},{-74.4,39.3},{-71.2,20.7},{-67.1,1.29},{-63.1,-14.6},{-58.3,-29.8},{-52.7,-43.5},{-46.2,-55.1},{-39.,-64.3},{-30.2,-71.7},{-18.9,-77.1},{-4.42,-79.9},{10.9,-79.1},{23.7,-75.2},{34.2,-68.7},{42.2,-60.6},{48.6,-51.2},{54.3,-40.},{59.1,-27.5},{63.1,-14.6},{67.1,1.29},{71.2,20.7},{74.4,39.3},{77.6,61.1},{80.,80.}},color={0,0,0}),Text(extent={{-31.,72.},{-11.,88.}},textString="27",fillColor={0,0,255}),Text(extent={{62.,-67.},{82.,-87.}},textString="4",fillColor={0,0,255}),Text(extent={{-104.,-63.},{-84.,-83.}},textString="-4",fillColor={0,0,255}),Text(extent={{78.,-54.},{100.,-74.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>hyperbolic cosine</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>cosh</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.cosh(u);
   end Cosh;
   block Tanh "Output the hyperbolic tangent of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Line(points={{-80.,-80.},{-47.8,-78.7},{-35.8,-75.7},{-27.7,-70.6},{-22.1,-64.2},{-17.3,-55.9},{-12.5,-44.3},{-7.64,-29.2},{-1.21,-4.82},{6.83,26.3},{11.7,42.},{16.5,54.2},{21.3,63.1},{26.9,69.9},{34.2,75.},{45.4,78.4},{72.,79.9},{80.,80.}},color={0,0,0}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-88.,72.},{-16.,24.}},textString="tanh",fillColor={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{9.,98.},{42.,80.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-47.8,-78.7},{-35.8,-75.7},{-27.7,-70.6},{-22.1,-64.2},{-17.3,-55.9},{-12.5,-44.3},{-7.64,-29.2},{-1.21,-4.82},{6.83,26.3},{11.7,42.},{16.5,54.2},{21.3,63.1},{26.9,69.9},{34.2,75.},{45.4,78.4},{72.,79.9},{80.,80.}},color={0,0,0}),Text(extent={{70.,25.},{90.,5.}},textString="4",fillColor={0,0,255}),Text(extent={{-106.,21.},{-86.,1.}},textString="-4",fillColor={0,0,255}),Text(extent={{-29.,72.},{-9.,88.}},textString="1",fillColor={0,0,255}),Text(extent={{3.,-72.},{23.,-88.}},textString="-1",fillColor={0,0,255}),Text(extent={{72.,-8.},{98.,-26.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>hyperbolic tangent</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>tanh</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.tanh(u);
   end Tanh;
   block Exp "Output the exponential (base e) of the input"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,-80.},{0.,68.}},color={192,192,192}),Polygon(points={{0.,90.},{-8.,68.},{8.,68.},{0.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-86.,50.},{-14.,2.}},textString="exp",fillColor={192,192,192}),Line(points={{-80.,-80.},{-31.,-77.9},{-6.03,-74.},{10.9,-68.4},{23.7,-61.},{34.2,-51.6},{43.,-40.3},{50.3,-27.8},{56.7,-13.5},{62.3,2.23},{67.1,18.6},{72.,38.2},{76.,57.6},{80.,80.}},color={0,0,0}),Line(points={{-90.,-80.3976},{68.,-80.3976}},color={192,192,192}),Polygon(points={{90.,-80.3976},{68.,-72.3976},{68.,-88.3976},{90.,-80.3976}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid)}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{0.,80.},{-8.,80.}},color={192,192,192}),Line(points={{0.,-80.},{-8.,-80.}},color={192,192,192}),Line(points={{0.,-90.},{0.,84.}},color={192,192,192}),Text(extent={{9.,100.},{40.,80.}},textString="y",fillColor={160,160,160}),Polygon(points={{0.,100.},{-6.,84.},{6.,84.},{0.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,-80.3976},{84.,-80.3976}},color={192,192,192}),Polygon(points={{100.,-80.3976},{84.,-74.3976},{84.,-86.3976},{100.,-80.3976}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-31.,-77.9},{-6.03,-74.},{10.9,-68.4},{23.7,-61.},{34.2,-51.6},{43.,-40.3},{50.3,-27.8},{56.7,-13.5},{62.3,2.23},{67.1,18.6},{72.,38.2},{76.,57.6},{80.,80.}},color={0,0,0}),Text(extent={{-31.,72.},{-11.,88.}},textString="20",fillColor={0,0,255}),Text(extent={{-92.,-83.},{-72.,-103.}},textString="-3",fillColor={0,0,255}),Text(extent={{70.,-83.},{90.,-103.}},textString="3",fillColor={0,0,255}),Text(extent={{-18.,-53.},{2.,-73.}},textString="1",fillColor={0,0,255}),Text(extent={{66.,-52.},{96.,-72.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>exponential</b> (of base e) of the input <b>u</b>:
 </p>
 <pre>
     y = <b>exp</b>( u );
 </pre>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.exp(u);
   end Exp;
   block Log "Output the natural (base e) logarithm of the input (input > 0 required)"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,-80.},{-80.,68.}},color={192,192,192}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-79.2,-50.6},{-78.4,-37.},{-77.6,-28.},{-76.8,-21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28.},{-47.,38.1},{-31.8,48.1},{-10.1,58.},{22.1,68.},{68.7,78.1},{80.,80.}},color={0,0,0}),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Text(extent={{-6.,-24.},{66.,-72.}},textString="log",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,80.},{-88.,80.}},color={192,192,192}),Line(points={{-80.,-80.},{-88.,-80.}},color={192,192,192}),Line(points={{-80.,-90.},{-80.,84.}},color={192,192,192}),Text(extent={{-67.,98.},{-40.,80.}},textString="y",fillColor={160,160,160}),Polygon(points={{-80.,100.},{-86.,84.},{-74.,84.},{-80.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-79.2,-50.6},{-78.4,-37.},{-77.6,-28.},{-76.8,-21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28.},{-47.,38.1},{-31.8,48.1},{-10.1,58.},{22.1,68.},{68.7,78.1},{80.,80.}},color={0,0,0}),Text(extent={{-105.,72.},{-85.,88.}},textString="3",fillColor={0,0,255}),Text(extent={{-109.,-88.},{-89.,-72.}},textString="-3",fillColor={0,0,255}),Text(extent={{70.,-3.},{90.,-23.}},textString="20",fillColor={0,0,255}),Text(extent={{-78.,-1.},{-58.,-21.}},textString="1",fillColor={0,0,255}),Text(extent={{68.,28.},{94.,8.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>natural (base e) logarithm</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>log</b>( u );
 </pre>
 <p>
 An error occurs if the elements of the input <b>u</b> are
 zero or negative.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.log(u);
   end Log;
   block Log10 "Output the base 10 logarithm of the input (input > 0 required)"
      extends Interfaces.SISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Polygon(points={{90.,0.},{68.,8.},{68.,-8.},{90.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-90.,0.},{68.,0.}},color={192,192,192}),Line(points={{-79.8,-80.},{-79.2,-50.6},{-78.4,-37.},{-77.6,-28.},{-76.8,-21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28.},{-47.,38.1},{-31.8,48.1},{-10.1,58.},{22.1,68.},{68.7,78.1},{80.,80.}},color={0,0,0}),Polygon(points={{-80.,90.},{-88.,68.},{-72.,68.},{-80.,90.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-80.,-80.},{-80.,68.}},color={192,192,192}),Text(extent={{-30.,-22.},{60.,-70.}},textString="log10",fillColor={192,192,192})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Line(points={{-80.,80.},{-88.,80.}},color={192,192,192}),Line(points={{-80.,-80.},{-88.,-80.}},color={192,192,192}),Line(points={{-80.,-90.},{-80.,84.}},color={192,192,192}),Text(extent={{-65.,96.},{-38.,78.}},textString="y",fillColor={160,160,160}),Polygon(points={{-80.,100.},{-86.,84.},{-74.,84.},{-80.,100.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-100.,0.},{84.,0.}},color={192,192,192}),Polygon(points={{100.,0.},{84.,6.},{84.,-6.},{100.,0.}},lineColor={192,192,192},fillColor={192,192,192},fillPattern=FillPattern.Solid),Line(points={{-79.8,-80.},{-79.2,-50.6},{-78.4,-37.},{-77.6,-28.},{-76.8,-21.3},{-75.2,-11.4},{-72.8,-1.31},{-69.5,8.08},{-64.7,17.9},{-57.5,28.},{-47.,38.1},{-31.8,48.1},{-10.1,58.},{22.1,68.},{68.7,78.1},{80.,80.}},color={0,0,0}),Text(extent={{70.,-3.},{90.,-23.}},textString="20",fillColor={0,0,255}),Text(extent={{-78.,-1.},{-58.,-21.}},textString="1",fillColor={0,0,255}),Text(extent={{-109.,72.},{-89.,88.}},textString=" 1.3",fillColor={0,0,255}),Text(extent={{-109.,-88.},{-89.,-72.}},textString="-1.3",fillColor={0,0,255}),Text(extent={{62.,30.},{90.,10.}},textString="u",fillColor={160,160,160})}),Documentation(info="
 <HTML>
 <p>
 This blocks computes the output <b>y</b> (element-wise) as the
 <i>base 10 logarithm</b> of the input <b>u</b>:
 </p>
 <pre>
     y = <b>log10</b>( u );
 </pre>
 <p>
 An error occurs if the elements of the input <b>u</b> are
 zero or negative.
 </p>
 <p><b>Release Notes:</b></p>
 <ul>
 <li><i>August 7, 1999</i>
        by <a href=\"http://www.op.dlr.de/~otter/\">Martin Otter</a>:<br>
        Realized.
 </li>
 </ul>
 </HTML>
 "));
      
   equation
      y = Modelica.Math.log10(u);
   end Log10;


   block BooleanToReal "Convert Boolean to Real signal"
      extends Interfaces.partialBooleanSI;
      parameter Real realTrue=1. "Output signal for true Boolean input";
      parameter Real realFalse=0. "Output signal for false Boolean input";
      Blocks.Interfaces.RealOutput y annotation (Placement(transformation(x=110.,y=0.,scale=0.1,aspectRatio=1.),iconTransformation(x=110.,y=0.,scale=0.1,aspectRatio=1.)));
      annotation (Documentation(info=""),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-86.,92.},{-6.,10.}},textString="B",fillColor={255,0,255}),Polygon(points={{-12.,-46.},{-32.,-26.},{-32.,-36.},{-64.,-36.},{-64.,-56.},{-32.,-56.},{-32.,-66.},{-12.,-46.}},fillColor={0,0,255},fillPattern=FillPattern.Solid,lineColor={0,0,255}),Text(extent={{8.,-4.},{92.,-94.}},textString="R",fillColor={0,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
      
   equation
      y = if u then
         realTrue
      else
         realFalse;
   end BooleanToReal;

   block RealToBoolean "Convert Real to Boolean signal"
      extends Interfaces.partialBooleanSO;
      parameter Real threshold=0.5 "Output signal y is true, if input u >= threshold";
      Blocks.Interfaces.RealInput u annotation (Placement(transformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.),iconTransformation(x=-120.,y=0.,scale=0.2,aspectRatio=1.)));
      annotation (Documentation(info=""),Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-86.,92.},{-6.,10.}},textString="R",fillColor={0,0,255}),Polygon(points={{-12.,-46.},{-32.,-26.},{-32.,-36.},{-64.,-36.},{-64.,-56.},{-32.,-56.},{-32.,-66.},{-12.,-46.}},lineColor={255,0,255},fillColor={255,0,255},fillPattern=FillPattern.Solid),Text(extent={{8.,-4.},{92.,-94.}},textString="B",fillColor={255,0,255})}),Diagram(coordinateSystem(extent={{-100.,-100.},{100.,100.}})));
      
   equation
      y = u >= threshold;
   end RealToBoolean;

   block Max "Pass through the largest signal"
      extends Interfaces.SI2SO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,36.},{90.,-36.}},fillColor={160,160,160},textString="max()")}),Documentation(info="<html>
 <p>
 This block computes the output <b>y</b> as <i>maximum</i>
 of the two inputs <b>u1</b> and <b>u2</b>:
 </p>
 <pre>    y = <b>max</b> ( u1 , u2 );
 </pre>
 </html>
 "));
      
   equation
      y = max(u1,u2);
   end Max;
   block Min "Pass through the smallest signal"
      extends Interfaces.SI2SO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,36.},{90.,-36.}},fillColor={160,160,160},textString="min()")}),Documentation(info="<html>
 <p>
 This block computes the output <b>y</b> as <i>minimum</i> of
 the two inputs <b>u1</b> and <b>u2</b>:
 </p>
 <pre>    y = <b>min</b> ( u1 , u2 );
 </pre>
 </html>
 "));
      
   equation
      y = min(u1,u2);
   end Min;
   block Edge "Indicates rising edge of boolean signal"
      extends Interfaces.BooleanSISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,36.},{90.,-36.}},fillColor={160,160,160},textString="edge()")}),Documentation(info="<html>
 <p>
 This block sets the Boolean output <b>y</b> to true,
 when the Boolean input <b>u</b> shows a <i>rising edge</i>:
 </p>
 <pre>    y = <b>edge</b>( u );
 </pre>
 </html>
 "));
      
   equation
      y = edge(u);
   end Edge;
   block BooleanChange "Indicates boolean signal changing"
      extends Interfaces.BooleanSISO;
      annotation (Icon(coordinateSystem(extent={{-100.,-100.},{100.,100.}}),graphics={Text(extent={{-90.,36.},{90.,-36.}},fillColor={160,160,160},textString="change()")}),Documentation(info="<html>
 <p>
 This block sets the Boolean output <b>y</b> to true, when the
 Boolean input <b>u</b> shows a <i>rising or falling edge</i>,
 i.e., when the signal changes:
 </p>
 <pre>    y = <b>change</b>( u );
 </pre>
 </html>
 "));
      
   equation
      y = change(u);
   end BooleanChange;

end Math;
